/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _ALBUM_ART_PVN1_H_
#define _ALBUM_ART_PVN1_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "channel_art_interface.h"
#include "dataservice_base.h"

  /************************/
 /** CONSTANTS / MACROS **/
/************************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

#define ALBUM_ART_HIGHBAND_OBJECT_NAME "AlbArt:HB"

// A slightly more descriptive NULL pointer for calls to
// bReadNext to show we don't care about the total #
// of bits read.
#define TOTAL_BITS_DONTCARE ((size_t*)0)

#define ALBUM_ART_HIGHBAND_PVN   (1)

// Album art uses DSI 315; DMI 285 is for high-priority images
// (live art), DMI 286 is for low-priority images (default image
// updates), and DMI 287 is for image association messages.
#define ALBUM_ART_DSI (315)

// Album art SID-associated durations are given in increments
// of 15 seconds
#define ALBUM_ART_DURATION_SECS  (15)

// A simple byte-to-bits constant, which we need during our
// stuffing calculation
#define ALBUM_BITS_PER_BYTE (8)

// Per SX-9845-0231, section 4.2.1, Album Art has 3 carousels:
// High-priority (ephemeral) images, low-priority (default) images,
// and the default assignment specifier.
#define ALBUM_ART_HIGHBAND_NUM_CAROUSELS (3)

// Per SX-9845-0231, section 4.2., our max AU size is 5120 bytes
#define ALBUM_ART_HIGHBAND_MAX_AU_SIZE (5120)

// However, album art messages (where a message is defined as a completed
// association and corresponding reassembled image) can be comprised of up to
// four individual AUs (per SX-9845-0231, 5.2, which defines the max image
// size for album art as 16k.)
#define ALBUM_ART_MAX_MESSAGES_PER_IMAGE (4)
#define ALBUM_ART_HIGHBAND_MAX_MESSAGE_SIZE (ALBUM_ART_MAX_MESSAGES_PER_IMAGE\
        * ALBUM_ART_HIGHBAND_MAX_AU_SIZE)

// ... and our buffer size is just the maximum message size
// times the number of carousels.
#define ALBUM_ART_HIGHBAND_BUFFER_BYTESIZE (ALBUM_ART_HIGHBAND_NUM_CAROUSELS\
        * ALBUM_ART_HIGHBAND_MAX_MESSAGE_SIZE)

// Used by the album art interface to provide .tStartServiceId /
// .tEndServiceId
#define ALBUM_ART_HIGHBAND_INITIAL_SERVICE_ID (1)
#define ALBUM_ART_HIGHBAND_FINAL_SERVICE_ID   (999)

// We use -1 as the initial ID so we can tell when
// we're not in a sequence. The actual start-of-sequence
// number will be 0, however.
#define ALBUM_ART_LAST_SEQ_NO_INITIAL_VAL (-1)
#define ALBUM_ART_FIRST_MULTI_AU_SEQ_NO (0)

// Here we define our allowed range of ephemeral IDs. The
// lower bound is determined by the "last" service ID that can
// make use of album art (999, see above), while the upper bound
// is determined by the maximum number we can cram into an album
// art filename (9999).
#define ALBUM_ART_FIRST_EPHEMERAL_ID ((UN16)1000)
#define ALBUM_ART_LAST_EPHEMERAL_ID ((UN16)9999)

// The following are default behavior specifiers; see
// SX-9845-0231 Section 6.2.2; when a behavior is specified,
// it basically amounts to using the service default, or using
// the provided image (sometimes for a particle CDO eType only)
#define ALBUM_ART_USE_DEFAULT ((UN8)0x0)
#define ALBUM_ART_USE_PROVIDED ((UN8)0x1)
#define ALBUM_ART_USE_PROVIDED_MUSIC_ONLY ((UN8)0x2)
#define ALBUM_ART_USE_PROVIDED_SPORTS_ONLY ((UN8)0x3)

// The default behavior comes in as 4 bits, but only the bottom two
// are valid (see above.) The following masks clips out the bits
// we care about.
#define ALBUM_ART_HIGHBAND_DEFAUT_BEHAVIOR_BITMASK  (0x03)

// The default service image is associated with
// SID 0x0.
#define ALBUM_ART_SERVICE_DEFAULT_ID ((UN16)0x0)

// Image File AU Field Length Definitions
#define ALBUM_ART_HIGHBAND_PRESENCE_BITLEN                      (1)

// Header bit lengths
#define ALBUM_ART_HIGHBAND_PVN_BITLEN                           (4)
#define ALBUM_ART_HIGHBAND_CARID_BITLEN                         (3)
#define ALBUM_ART_HIGHBAND_ACTION_BITLEN                        (4)
#define ALBUM_ART_HIGHBAND_COMPRESSION_BITLEN                   (3)

// Tigger-related bit lengths
#define ALBUM_ART_HIGHBAND_SID_PID_TRIGGER_BITLEN               (42)
#define ALBUM_ART_HIGHBAND_SID_TRIGGER_BITLEN                   (18)
#define ALBUM_ART_HIGHBAND_DEFAULT_SID_TRIGGER_BITLEN           (10)
#define ALBUM_ART_HIGHBAND_BLANK_SID_TRIGGER_BITLEN             (10)
#define ALBUM_ART_HIGHBAND_UPDATE_DEFAULT_TRGIGER_BITLEN        (18)

// Association-related bit lengths
#define ALBUM_ART_HIGHBAND_SID_BITLEN                           (10)
#define ALBUM_ART_HIGHBAND_DURATION_BITLEN                      (8)
#define ALBUM_ART_HIGHBAND_PID_BITLEN                           (32)
#define ALBUM_ART_HIGHBAND_IMAGEID_BITLEN                       (10)
#define ALBUM_ART_HIGHBAND_IMAGEVER_BITLEN                      (8)

// Extdata-related bit lengths
#define ALBUM_ART_HIGHBAND_EXTDATA_COUNT_BITLEN                 (8)
#define ALBUM_ART_HIGHBAND_EXTDATA_BLOCK_BITLEN                 (8)

// Caption-releated bit lengths
#define ALBUM_ART_HIGHBAND_MAX_CAPTION_SYMBOL_COUNT             (180)

// Payload count-related bit lengths
#define ALBUM_ART_HIGHBAND_AU_COUNT_BITLEN                      (4)
#define ALBUM_ART_HIGHBAND_SID_COUNT_BITLEN                     (10)

// Default behavior-related bit lengths
#define ALBUM_ART_HIGHBAND_ASSIGNMENT_VERSION_BITLEN            (8)
#define ALBUM_ART_HIGHBAND_DEFUALT_BEHAVIOR_BITLEN              (4)
#define ALBUM_ART_HIGHBAND_DEFUALT_ID_BITLEN                    (10)

// Definitions for image actions
// Tie the image to a PID/SID combo
#define ALBUM_ART_ACTION_PID_ASSOCIATE                          ((UN8)0)
// Tie the image to a SID for a period of time
#define ALBUM_ART_ACTION_TIMED_SID_ASSOCIATE                    ((UN8)1)
// No image, just use the default for the given SID
#define ALBUM_ART_ACTION_SID_DEFAULT                            ((UN8)2)
// No image, just use the blank image for the given SID
#define ALBUM_ART_ACTION_SID_BLANK                              ((UN8)3)
// The given image is the new default for a SID
#define ALBUM_ART_ACTION_DEFAULT_UPDATE                         ((UN8)4)
// Actions beyond this are unknown / undefined
#define ALBUM_ART_ACTION_UNKNOWN                                ((UN8)5)

  /**************/
 /** TYPEDEFS **/
/**************/

// Enumeration of the message types supported
// by the high-band channel art service; the image message
// does not have an actual message number, but is instead
// determined by the carousel ID.

typedef enum album_art_highband_carid_enum
{
    ALBUM_ART_HIGHBAND_CARID_IMAGE_HIGH_PRIO = 0x0,
    ALBUM_ART_HIGHBAND_CARID_IMAGE_LOW_PRIO = 0x1,
    ALBUM_ART_HIGHBAND_CARID_DEFAULT_ASSOC = 0x2,
    ALBUM_ART_HIGHBAND_CARID_UNKNOWN
} ALBUM_ART_HIGHBAND_CARID_ENUM;

// Enumeration for album art compression types
// Album Art only uses JPEG, but this is here
// for future expansion.

typedef enum album_art_compression_enum
{
    ALBUM_ART_COMPRESSION_JPEG,
    ALBUM_ART_COMPRESSION_UNKNOWN
} ALBUM_ART_COMPRESSION_ENUM;

// Enumeration which specifies the result of
// processing for an association message
typedef enum album_art_highband_process_result_enum
{
    ALBUM_ART_HIGHBAND_PROCESS_RESULT_COMPLETE,
    ALBUM_ART_HIGHBAND_PROCESS_RESULT_INCOMPLETE,
    ALBUM_ART_HIGHBAND_PROCESS_RESULT_ERROR

} ALBUM_ART_HIGHBAND_PROCESS_RESULT_ENUM;

// An image control struct to keep track of the image
// that we're currently reassembling. Both the high
// and low priority carousels each have one of these.

typedef struct album_art_highband_image_ctrl_struct
{
    // Image attributes and handle
    ALBUM_ART_ASSOC_ROW_STRUCT sImageInProcess;
    OSAL_BUFFER_HDL hImageData;

    // AU attributes
    N8 n8LastSeqNum;
    UN8 un8NumMessagesProcessed;

} ALBUM_ART_HIGHBAND_IMAGE_CTRL_STRUCT;

// The master 'control' structure for the album art
// processor object.

typedef struct album_art_highband_object_struct
{
    // Central Channel art service handle
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService;

    // CRC handle (used to compute SDTP AU ISO 3309 CRC32 based CRCs)
    OSAL_OBJECT_HDL hCRC;

    // Structure used to track image progress
    ALBUM_ART_HIGHBAND_IMAGE_CTRL_STRUCT sLowImageCtrl;

    ALBUM_ART_HIGHBAND_IMAGE_CTRL_STRUCT sHighImageCtrl;

} ALBUM_ART_HIGHBAND_OBJECT_STRUCT;

  /****************/
 /** PROTOTYPES **/
/****************/

static CHANNEL_ART_INTERFACE_OBJECT hInit (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    SMS_OBJECT hParent
        );

static void vUnInit (
    ALBUM_ART_INTERFACE_OBJECT hInterface
        );

static BOOLEAN bProcessMessage (
    ALBUM_ART_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        );

/* Object Private Prototypes */

/* Message Processors */

static ALBUM_ART_HIGHBAND_CARID_ENUM eParseMessageHeader (
    ALBUM_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    size_t *ptBitsRead
        );

static BOOLEAN bProcessImageMessage (
    ALBUM_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    size_t tHeaderBitsRead,
    ALBUM_ART_HIGHBAND_IMAGE_CTRL_STRUCT *psImageCtrl
        );

static void vResetImageTracking (
    ALBUM_ART_HIGHBAND_IMAGE_CTRL_STRUCT *psImageCtrl
        );

static BOOLEAN bImageReassembler (
    ALBUM_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    UN8 un8AUTotal,
    UN8 un8AUCount,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    STRING_OBJECT *phCaption,
    OSAL_BUFFER_HDL *phPayload,
    ALBUM_ART_HIGHBAND_IMAGE_CTRL_STRUCT *psImageCtrl,
    UN8 un8Action
        );

static BOOLEAN bProcessAssociationMessage (
	ALBUM_ART_HIGHBAND_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        );

static BOOLEAN bReadNext(
    OSAL_BUFFER_HDL hPayload,
    size_t tBitCount,
    size_t *ptTotalBitsRead,
    void *pvVariable,
    char *pacName
        );

  /***************/
 /** VARIABLES **/
/***************/

const ALBUM_ART_PLUGIN_INTERFACE_STRUCT GsAlbumArtIntf =
{
    /*.tDSI = */ALBUM_ART_DSI,
    /*.tOTABufferByteSize = */ALBUM_ART_HIGHBAND_BUFFER_BYTESIZE,
    /*.tStartServiceID = */ALBUM_ART_HIGHBAND_INITIAL_SERVICE_ID,
    /*.tEndServiceID = */ALBUM_ART_HIGHBAND_FINAL_SERVICE_ID,
    /*.hInit = */hInit,
    /*.vUnInit = */vUnInit,
    /*.bProcessMessage = */bProcessMessage
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _ALBUM_ART_PVN1_H_
