/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*                 Licensed Materials - Sirius XM Radio Inc.                  */
/******************************************************************************/
/*******************************************************************************
*
*   DESCRIPTION
*
*   A simple SDTP re-assembler module that relies upon messages received
*   in full by some other entity
*
*******************************************************************************/

#include "osal.h"
#include "sms_obj.h"
#include "sdtp.h"
#include "_sdtp.h"

static const char *gpacThisFile = __FILE__;
#include "sms_api_debug.h"

/*****************************************************************************
*
*   Friend Functions
*
*****************************************************************************/

/*****************************************************************************
*
*   SDTP_hConnect
*
*****************************************************************************/
SDTP_HDL SDTP_hConnect(
    SMS_OBJECT hParent
        )
{
    SDTP_CONNECTION_STRUCT *psConnection =
        (SDTP_CONNECTION_STRUCT *)NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    // Verify object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hParent);
    if (bOwner == FALSE)
    {
        return SDTP_INVALID_HDL;
    }

    // Allocate memory for this connection
    psConnection = (SDTP_CONNECTION_STRUCT *)
        SMSO_hCreate(
            SDTP_CONNECTION_NAME,
            sizeof(SDTP_CONNECTION_STRUCT),
            hParent, FALSE);
    if (psConnection == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SDTP_OUTPUT_PREFIX": Unable to allocate memory for connection");
        return SDTP_INVALID_HDL;
    }

    // Create the linked list of Access Units
    eReturnCode = OSAL.eLinkedListCreate(
        &psConnection->hAccessUnits,
        SDTP_CONNECTION_NAME":"SDTP_AULIST_NAME,
        (OSAL_LL_COMPARE_HANDLER)n16CompareAU,
        OSAL_LL_OPTION_NONE );

    if (eReturnCode != OSAL_SUCCESS)
    {
        // Make sure the ll has been marked invalid
        psConnection->hAccessUnits = OSAL_INVALID_OBJECT_HDL;

        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SDTP_OUTPUT_PREFIX": Error: Unable to allocate memory for AU list.");
        SDTP_vDisconnect( psConnection );

        return SDTP_INVALID_HDL;
    }

    // Initialize statistics
    OSAL.bMemSet(&psConnection->sStatistics, 0,
        sizeof(psConnection->sStatistics));

    return (SDTP_HDL)psConnection;
}

/*****************************************************************************
*
*   SDTP_vDisconnect
*
*****************************************************************************/
void SDTP_vDisconnect (
    SDTP_HDL hSDTP
        )
{
    BOOLEAN bOwner;

    // Check object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hSDTP);
    if (bOwner == TRUE)
    {
        SDTP_CONNECTION_STRUCT *psConnection =
            (SDTP_CONNECTION_STRUCT *)hSDTP;

        // Free access unit list
        if (psConnection->hAccessUnits != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Free any memory associated with the ll before the deletion
            eReturnCode = OSAL.eLinkedListRemoveAll( psConnection->hAccessUnits, vReleasePSINode );

            if (eReturnCode == OSAL_SUCCESS)
            {
                // Free the memory associated with the ll
                OSAL.eLinkedListDelete( psConnection->hAccessUnits );
            }
        }

        SMSO_vDestroy((SMS_OBJECT)psConnection);
    }

    return;
}

/*****************************************************************************
*
*   SDTP_hReAssemble
*
*****************************************************************************/
OSAL_BUFFER_HDL SDTP_hReAssemble (
    SDTP_HDL hSDTP,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bOwner;
    OSAL_BUFFER_HDL hAU = OSAL_INVALID_BUFFER_HDL;

    // Verify object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hSDTP);
    if (bOwner == FALSE)
    {
        return OSAL_INVALID_BUFFER_HDL;
    }

    do
    {
        BOOLEAN bOk;
        SDTP_CONNECTION_STRUCT *psConnection =
            (SDTP_CONNECTION_STRUCT *)hSDTP;
        SDTP_ACCESS_UNIT_STRUCT *psAU =
            (SDTP_ACCESS_UNIT_STRUCT *)NULL;
        UN8 aun8RawHdr[SDTPRx_HDR_LEN];
        SDTP_FRAME_HDR_STRUCT sFrameHdr;
        SDTPRx_SUM tReportedSum;
        SDTPRx_SYNC tSync = 0;
        size_t tBytesRead;

        if (*phPayload == OSAL_INVALID_BUFFER_HDL)
        {
            // Error! Something is wrong.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SDTP_OUTPUT_PREFIX": Empty Message.");

            // Increment Statistics
            psConnection->sStatistics.un32Errors++;
            break;
        }

        // Read the sync bit
        tBytesRead = OSAL.tBufferReadHead(*phPayload, &tSync, sizeof(SDTPRx_SYNC));
        if (tBytesRead != sizeof(SDTPRx_SYNC))
        {
            psConnection->sStatistics.un32Errors++;
            break;
        }

        if (tSync != SDTPRx_SOF)
        {
            psConnection->sStatistics.un32OutOfSync++;
            break;
        }

        // Read out the checksum from the frame w/o check assuming
        // the module provides us with already valid and checked packets.
        tBytesRead = OSAL.tBufferReadTail(*phPayload, &tReportedSum,
                            sizeof(SDTPRx_SUM));
        if (tBytesRead != sizeof(SDTPRx_SUM))
        {
            psConnection->sStatistics.un32Errors++;
            break;
        }

        // Read the header
        tBytesRead = OSAL.tBufferReadHead(*phPayload, &aun8RawHdr[0], SDTPRx_HDR_LEN);
        if (tBytesRead != SDTPRx_HDR_LEN)
        {
            psConnection->sStatistics.un32Errors++;
            break;
        }

        // Extract the header from the payload
        vExtractHeader(&aun8RawHdr[0], &sFrameHdr);

        // Find the relevant AU descriptor
        psAU = psFindAU(psConnection, sFrameHdr.tPSI);
        if (psAU == NULL)
        {
            psConnection->sStatistics.un32UnknownStream++;

            break;
        }

        // Is this stream in sync with us?
        bOk = bSyncStream(
            &psConnection->sStatistics,
            psConnection, psAU, &sFrameHdr, phPayload);
        if (bOk == FALSE)
        {
            // Clean up the AU
            if (psAU->hPayload != OSAL_INVALID_BUFFER_HDL)
            {
                OSAL.eBufferFree(psAU->hPayload);
                psAU->hPayload = OSAL_INVALID_BUFFER_HDL;
            }

            psAU->hPayload = OSAL_INVALID_BUFFER_HDL;
            psAU->tPayloadLen = 0;
            psAU->tNumPacketsLeft = 0;
            break;
        }

        // We have another valid frame now
        psConnection->sStatistics.un32Frames++;

        if (*phPayload != OSAL_INVALID_BUFFER_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Copy data from the input frame to the access unit payload buffer
            eReturnCode = OSAL.eBufferAppend(psAU->hPayload, *phPayload);
            if (eReturnCode != OSAL_SUCCESS)
            {
                if (psAU->hPayload != OSAL_INVALID_BUFFER_HDL)
                {
                    OSAL.eBufferFree(psAU->hPayload);
                    psAU->hPayload = OSAL_INVALID_BUFFER_HDL;
                }

                psAU->hPayload = OSAL_INVALID_BUFFER_HDL;
                psAU->tPayloadLen = 0;
                psAU->tNumPacketsLeft = 0;
                psConnection->sStatistics.un32BufferOverflow++;
                break;
            }
        }

        // Are we done with this AU?
        if (sFrameHdr.tEndAu)
        {
            // Yeah, we're all done now!

            // Send the AU to the caller
            hAU = psAU->hPayload;
            psAU->hPayload = OSAL_INVALID_BUFFER_HDL;

            psConnection->sStatistics.un32AUs++;
        }

    } while (FALSE);

    // It's now time to free the frame if is valid
    if (*phPayload != OSAL_INVALID_BUFFER_HDL)
    {
        OSAL.eBufferFree(*phPayload);
        *phPayload = OSAL_INVALID_BUFFER_HDL;
    }

    return hAU;
}

/*****************************************************************************
*
*   SDTP_vResetConnection
*
*   Used when an SDTP connection has undergone a PSI list change.  Forces
*   this connection to re-discover the list of active PSIs.
*
*****************************************************************************/
void SDTP_vResetConnection (
    SDTP_HDL hSDTP
        )
{
    BOOLEAN bOwner;

    // Check object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hSDTP);
    if (bOwner == TRUE)
    {
        SDTP_CONNECTION_STRUCT *psConnection =
            (SDTP_CONNECTION_STRUCT *)hSDTP;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Empty the list
        eReturnCode = OSAL.eLinkedListRemoveAll( psConnection->hAccessUnits, vReleasePSINode );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SDTP_OUTPUT_PREFIX": Unable to clear AU list");
        }
    }

    return;
}

/*****************************************************************************
*
*   SDTP_vPrintToLog
*
*****************************************************************************/
void SDTP_vPrintToLog (
    SDTP_HDL hSDTP,
    SDTP_LOGGING_CALLBACK vPrintLog
        )
{
    BOOLEAN bOwner;

    // Verify input
    if (vPrintLog == NULL)
    {
        return;
    }

    // Check object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hSDTP);
    if (bOwner == TRUE)
    {
        SDTP_CONNECTION_STRUCT *psConnection =
            (SDTP_CONNECTION_STRUCT *)hSDTP;

        // Write to the log
        vPrintLog(
            SDTP_LOG_STATS,
            psConnection->sStatistics.un32Frames,
            psConnection->sStatistics.un32AUs,
            psConnection->sStatistics.un32NoMoreBuffers,
            psConnection->sStatistics.un32BufferOverflow,
            psConnection->sStatistics.un32OutOfSync,
            psConnection->sStatistics.un32SyncEmulation,
            psConnection->sStatistics.un32UnknownStream,
            psConnection->sStatistics.un32UnexpectedSOA,
            psConnection->sStatistics.un32UnexpectedPLPC,
            psConnection->sStatistics.un32DroppedFrames,
            psConnection->sStatistics.un32Errors
                );

        // Write CSV as well
        vPrintLog (
            SDTP_STATS_CSV,
            psConnection->sStatistics.un32Frames,
            psConnection->sStatistics.un32AUs,
            psConnection->sStatistics.un32NoMoreBuffers,
            psConnection->sStatistics.un32BufferOverflow,
            psConnection->sStatistics.un32OutOfSync,
            psConnection->sStatistics.un32SyncEmulation,
            psConnection->sStatistics.un32UnknownStream,
            psConnection->sStatistics.un32UnexpectedSOA,
            psConnection->sStatistics.un32UnexpectedPLPC,
            psConnection->sStatistics.un32DroppedFrames,
            psConnection->sStatistics.un32Errors
                );
    }
    return;
}

/*****************************************************************************
*
*   Private Functions
*
*****************************************************************************/

/*****************************************************************************
*
*   bSyncStream
*
*****************************************************************************/
static BOOLEAN bSyncStream (
    SDTP_STATISTICS_STRUCT *psStats,
    SDTP_CONNECTION_STRUCT *psConnection,
    SDTP_ACCESS_UNIT_STRUCT *psAU,
    SDTP_FRAME_HDR_STRUCT *psFrameHdr,
    OSAL_BUFFER_HDL *phFrame
        )
{
    BOOLEAN bSyncOk = FALSE;

    do
    {
        // If the packet has the start-of-access-unit (SOA) bit set
        if (psFrameHdr->tStartAu)
        {
            // If we aren't done with a previous access unit,
            // clean it up and start this one
            if (psAU->tNumPacketsLeft != 0)
            {
                puts(SDTP_OUTPUT_PREFIX": Unexpected AU segment(SOA=1)");

                // Increment Statistics
                psStats->un32UnexpectedSOA++;
            }

            // Initialize the access unit data
            psAU->tNumPacketsLeft = 0;
            psAU->tPayloadLen = 0;

            // Ensure any previous access unit is gone
            if (psAU->hPayload != OSAL_INVALID_BUFFER_HDL)
            {
                OSAL.eBufferFree( psAU->hPayload );
            }

            // This is now the start of the AU
            psAU->hPayload = *phFrame;
            *phFrame = OSAL_INVALID_BUFFER_HDL;

            // Verify we received a frame from the block pool
            if (psAU->hPayload == OSAL_INVALID_BUFFER_HDL)
            {
                // Increment Statistics
                psStats->un32NoMoreBuffers++;
                break;
            }
        }
        // SOA not set
        else
        {
            // If we think a new access unit should be starting, fail
            if (psAU->tNumPacketsLeft == 0)
            {
                puts(SDTP_OUTPUT_PREFIX": Unexpected Access Unit segment(SOA=0)");

                // Increment Statistics
                psStats->un32UnexpectedPLPC++;
                break;
            }
            else
            {
                // We have verified that part of an access unit is
                // here, so decrement the number of packets left in
                // this access unit
                psAU->tNumPacketsLeft--;
            }
        }

        // If the packet has the end-of-access-unit (EOA) bit set
        if (psFrameHdr->tEndAu)
        {
            // If we aren't done with the current access unit, fail
            if (psAU->tNumPacketsLeft != 0)
            {
                printf(SDTP_OUTPUT_PREFIX": Unexpected Access Unit segment(EOA=1)");

                // Increment Statistics
                psStats->un32UnexpectedPLPC++;

                break;
            }
        }
        // If the EOA bit isn't set
        else
        {
            // This is the first section of an access unit in a
            // series.  We don't know how many SDTP packets will
            // make up this access unit yet -- but this packet
            // will tell us
            if ( psFrameHdr->tStartAu )
            {
                psAU->tNumPacketsLeft =
                    psFrameHdr->tLenOrCount;
            }
            // We are in the process of collecting access units from a
            // series of SDTP packets.
            else
            {
                // If we do not agree on how many more SDTP packets
                // constitute this access unit, fail
                if (    ( psAU->tNumPacketsLeft )
                     != ( psFrameHdr->tLenOrCount ) )
                {
                    puts(SDTP_OUTPUT_PREFIX": Unexpected Access Unit segment(PLPC mismatch)");

                    // Increment Statistics
                    psStats->un32UnexpectedPLPC++;
                    break;
                }
            }
        }

        bSyncOk = TRUE;

    } while (FALSE);

    return bSyncOk;
}

/*****************************************************************************
*
*   vExtractHeader
*
*****************************************************************************/
static void vExtractHeader (
    UN8 *pun8RawHdr,
    SDTP_FRAME_HDR_STRUCT *psHdr
        )
{
    // Clear the header structure
    OSAL.bMemSet(psHdr, 0, sizeof(SDTP_FRAME_HDR_STRUCT));

    // Extract the bit fields
    psHdr->tStartAu =
        SDTPRx_GET_SOA( pun8RawHdr[SDTPRx_BITFIELDS_OFFSET] );
    psHdr->tEndAu =
        SDTPRx_GET_EOA( pun8RawHdr[SDTPRx_BITFIELDS_OFFSET] );
    psHdr->tRFU =
        SDTPRx_GET_RFU( pun8RawHdr[SDTPRx_BITFIELDS_OFFSET] );

    // Extract the PSI field

    // Get the 4 least significant bits of this byte...
    psHdr->tPSI = (pun8RawHdr[PSI_OFFSET] & 0xF);
    psHdr->tPSI <<= 6;

    // And the 6 most significant bits of this byte
    psHdr->tPSI |=
        ( pun8RawHdr[PSI_OFFSET + PSI_BYTE_LEN] >> 2 );

    // Extract the Length / Count field

    // We want the 2 least significant bits of this byte...
    psHdr->tLenOrCount = ( pun8RawHdr[SDTPRx_COUNT_OFFSET] & 0x3 );
    psHdr->tLenOrCount <<= 8;

    // And the 8 bits that make up this byte
    psHdr->tLenOrCount |=
        pun8RawHdr[SDTPRx_COUNT_OFFSET + SDTPRx_COUNT_OFFSET_LEN];

    return;
}

/*****************************************************************************
*
*   psAddAU
*
*****************************************************************************/
static SDTP_ACCESS_UNIT_STRUCT *psAddAU (
    SDTP_CONNECTION_STRUCT *psConnection,
    PSI tPSI
        )
{
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    SDTP_ACCESS_UNIT_STRUCT *psAU;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    do
    {
        // Construct a unique name for the memory of this node
        snprintf( &acName[0], sizeof(acName),
            SDTP_CONNECTION_NAME":AU Node %i", tPSI );

        // Allocate space for our pointer, which will be placed in the LL
        psAU = (SDTP_ACCESS_UNIT_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(SDTP_ACCESS_UNIT_STRUCT),
                (SMS_OBJECT)psConnection, FALSE );

        if(psAU == NULL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SDTP_OUTPUT_PREFIX": Error: Can't add AU - out of memory.");

            break;
        }

        // Set the PSI value
        psAU->tPSI = tPSI;

        // Init fields...
        psAU->hPayload = OSAL_INVALID_BUFFER_HDL;
        psAU->tNumPacketsLeft = 0;
        psAU->tPayloadLen = 0;
        psAU->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Attempt to insert into linked list
        eReturnCode = OSAL.eLinkedListAdd(
            psConnection->hAccessUnits,
            &psAU->hEntry,
            psAU );

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SDTP_OUTPUT_PREFIX": Error: Unable to insert node to AU list.");

            // Have to free this here, 'cause it'll be lost memory if we don't
            SMSO_vDestroy((SMS_OBJECT)psAU);
            break;
        }

        return psAU;

    } while (FALSE);

    return NULL;
}

/*****************************************************************************
*
*   psFindAU
*
*****************************************************************************/
static SDTP_ACCESS_UNIT_STRUCT *psFindAU (
    SDTP_CONNECTION_STRUCT *psConnection,
    PSI tPSI
        )
{
    SDTP_ACCESS_UNIT_STRUCT *psAU =
        (SDTP_ACCESS_UNIT_STRUCT *)NULL;
    SDTP_ACCESS_UNIT_STRUCT sSearch;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;

    // Search for this PSI
    sSearch.tPSI = tPSI;

    // Get the PSI node from the connection
    eReturnCode = OSAL.eLinkedListSearch(
        psConnection->hAccessUnits,
        &hEntry,
        (void *)&sSearch );
    if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        psAU = psAddAU(psConnection, tPSI);
    }
    else if (eReturnCode == OSAL_SUCCESS)
    {
        // Get the actual structure from within the node
        psAU = (SDTP_ACCESS_UNIT_STRUCT *)
            OSAL.pvLinkedListThis( hEntry );
    }

    return psAU;
}

/*****************************************************************************
*
*   n16CompareAU
*
*   Utility function called by the OSAL Linked List API to compare two
*   Access Unit structures
*
*   Inputs:
*       *psAU1 - Pointer to an access unit structure
*           in the list
*       *psAU2 - An access unit to compare against
*
*   Outputs:
*       -1 if the PSI is greater than the Access Unit's PSI
*        0 if the PSI is equal to the Access Unit's PSI
*        1 if the PSI is less than the Access Unit's PSI
*
*****************************************************************************/
static N16 n16CompareAU (
    SDTP_ACCESS_UNIT_STRUCT *psAU1,
    SDTP_ACCESS_UNIT_STRUCT *psAU2
        )
{
    N16 n16Result;

    if (psAU1->tPSI > psAU2->tPSI)
    {
        n16Result = 1;
    }
    else if (psAU1->tPSI < psAU2->tPSI)
    {
        n16Result = -1;
    }
    else
    {
        n16Result = 0;
    }

    return n16Result;
}

/*****************************************************************************
*
*   vReleasePSINode
*
*   Utility function called by the OSAL Linked List API to delete
*   access unit nodes appropriately
*
*   Inputs:
*       *pvData - Pointer to an access unit structure
*
*   Outputs:
*       None
*
*****************************************************************************/
static void vReleasePSINode( void *pvData )
{
    SDTP_ACCESS_UNIT_STRUCT *psAU =
        (SDTP_ACCESS_UNIT_STRUCT *)pvData;

    // If the node is valid
    if (psAU != NULL)
    {
        // If the OSAL buffer handle is valid
        if (psAU->hPayload != NULL)
        {
            // Free it!
            OSAL.eBufferFree( psAU->hPayload );
            psAU->hPayload = OSAL_INVALID_BUFFER_HDL;
        }

        // Free the node
        SMSO_vDestroy((SMS_OBJECT)psAU);
    }

    return;
}
