/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the radio interface header file.
 *
 ******************************************************************************/

#ifndef RADIO_H_
#define RADIO_H_

#include "standard.h"

#include "osal.h"
#include "sms_api.h"

#include "sms_event_types.h"
#include "sti_api.h"
#include "srh.h"

#include "channel_obj.h"
#include "dataservice_base.h"

// If radio logging hasn't been configured already
// then only enable when we're in debug
#ifndef SMS_LOGGING
#if SMS_DEBUG == 1
#define SMS_LOGGING 1
#else
#define SMS_LOGGING 0
#endif
#endif

/**********************/

#define RADIO_MODULE_FEATURE_NONE                   (0x0000)

#define RADIO_MODULE_FEATURE_IR                     (0x0001)
#define RADIO_MODULE_FEATURE_AUDIO_RECORDING        (0x0002)
#define RADIO_MODULE_FEATURE_OVERLAY                (0x0004)
#define RADIO_MODULE_FEATURE_I2S_SLAVE              (0x0008)
#define RADIO_MODULE_FEATURE_ADVANCED_IR            (0x0010)
#define RADIO_MODULE_FEATURE_EXPANDED_SMART_FAVS    (0x0020)

#define RADIO_MODULE_FEATURE_MASK_ALL               (   \
       (RADIO_MODULE_FEATURE_MASK)                      \
        RADIO_MODULE_FEATURE_IR                 |       \
        RADIO_MODULE_FEATURE_AUDIO_RECORDING    |       \
        RADIO_MODULE_FEATURE_OVERLAY            |       \
        RADIO_MODULE_FEATURE_I2S_SLAVE          |       \
        RADIO_MODULE_FEATURE_ADVANCED_IR        |       \
        RADIO_MODULE_FEATURE_EXPANDED_SMART_FAVS            )

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/
typedef UN32 RADIO_MODULE_FEATURE_MASK;

// This is a structured used to store various RADIO specific
// constants related to category handling.
typedef struct radio_category_struct
{
    UN16 un16ShortNameMaxLen;
    UN16 un16MediumNameMaxLen;
    UN16 un16LongNameMaxLen;

    UN16 un16BroadcastIdMax;
    UN16 un16BroadcastIdMin;
    UN16 un16BroadcastIdNotAssigned;

    UN16 un16DerivedMax;
    UN16 un16DerivedMin;
    UN16 un16DerivedGroupSize;

} RADIO_CATEGORY_STRUCT;

// This is a structured used to store various RADIO specific
// constants related to channel handling.
typedef struct radio_channel_struct
{
    UN16 un16ShortNameMaxLen;
    UN16 un16MediumNameMaxLen;
    UN16 un16LongNameMaxLen;

    UN16 un16ArtistMaxLen;
    UN16 un16TitleMaxLen;
    UN16 un16ComposerMaxLen;
    UN16 un16LongDescMaxLen;
    UN16 un16ShortDescMaxLen;

    SERVICE_ID tDefaultServiceID;
    SERVICE_ID tSafeServiceID;

} RADIO_CHANNEL_STRUCT;

// This is a structured used to store various RADIO specific
// constants related to playback handling.
typedef struct radio_playback_struct
{
    UN32 un32DefaultWarningOffset;

} RADIO_PLAYBACK_STRUCT;

// This is a structured used to store various RADIO specific
// constants
typedef struct radio_misc_struct
{
    BOOLEAN bDstObserved;
    DSI     tDsiMin;
    DSI     tDsiMax;

} RADIO_MISC_STRUCT;

// This is the entire structure of RADIO specific constants
// which any entity outside of RADIO can access as needed.
typedef struct radio_constants_struct
{
    RADIO_CATEGORY_STRUCT sCategory;
    RADIO_CHANNEL_STRUCT sChannel;
    RADIO_PLAYBACK_STRUCT sPlayback;
    RADIO_MISC_STRUCT sMisc;

} RADIO_CONSTANTS_STRUCT;

// Common structure used by all RADIO implementations
// for SRM-specific data
typedef struct radio_srm_object_struct
{
    // This SRM object
    SRM_OBJECT hSRM;

    // This SRM's name
    const char *pacName;

    // Associated device handle
    FILE *psDevice;

    // SRM Capabilities
    SRH_DEVICE_CAPABILITIES_MASK tCapabilities;

    // Event handler for parent object
    void *hEventHdlr;

} RADIO_SRM_OBJECT_STRUCT;

// The protocol version number type
typedef UN8 PVN;

// Object handle for private radio data
typedef void * RADIO_PRIVATE_DATA_OBJECT;
#define RADIO_PRIVATE_DATA_INVALID_OBJECT (RADIO_PRIVATE_DATA_OBJECT)NULL

// Seek (playback) types
typedef enum radio_seek_type
{
    RADIO_SEEK_TYPE_TIME,
    RADIO_SEEK_TYPE_SONG,
    RADIO_SEEK_TYPE_PREVIOUS,
    RADIO_SEEK_TYPE_NEXT

} RADIO_SEEK_TYPE;

// Types of Seek Monitor, new Seek Monitors are to be added here.
// Maximum 8 Seek Monitors are allowed.
typedef enum seek_monitor_type_enum
{
    SEEK_MONITOR_TYPE_SPORTS_FLASH = 0,
    SEEK_MONITOR_TYPE_INVALID
} SEEK_MONITOR_TYPE_ENUM;

// Types of Bulletin Monitor, new Bulletin Monitors are to be added here.
typedef enum bulletin_monitor_type_enum
{
    BULLETIN_MONITOR_TYPE_TW_NOW = 0,
    BULLETIN_MONITOR_TYPE_INVALID
} BULLETIN_MONITOR_TYPE_ENUM;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

extern const char GacUnSubscribedTextDefault[];
extern const RADIO_CONSTANTS_STRUCT GsRadio;

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/* Object Friend Prototypes */

BOOLEAN RADIO_bInitializeSMS(void);

void RADIO_vUninitializeSMS(void);

BOOLEAN RADIO_bInitializeSrm(
    SRM_OBJECT hSRM,
    const char *pacName,
    const char *pacDriverName,
    SRH_DEVICE_CAPABILITIES_MASK *ptCapabilities,
    void *pvEventHdlr
        );

void RADIO_vUninitializeSrm(
    SRM_OBJECT hSRM
        );

BOOLEAN RADIO_bPostSrmEvent(SMS_EVENT_HDL hEvent, void *pvArg);

BOOLEAN RADIO_bSrmEventHandler(
    SRM_OBJECT hSRM,
    SMS_EVENT_TYPE_ENUM eEventType,
    const void *pvEventData
        );

MODULE_ID RADIO_tGetUniqueModuleId(
    SRM_OBJECT hSRM,
    const char *pacDecoderName
        );

BOOLEAN RADIO_bInitializeModule(
    MODULE_OBJECT hModule,
    MODULE_ID tId,
    SRM_OBJECT hSRM,
    const char *pacSRMName,
    FILE *psDevice,
    void *hEventHdlr
        );

void RADIO_vUninitializeModule(
    MODULE_OBJECT hModule,
    BOOLEAN bReset
        );

BOOLEAN RADIO_bModuleEventHandler(
    MODULE_OBJECT hModule,
    SMS_EVENT_TYPE_ENUM eEventType,
    const void *pvEventData
        );

SRH_DEVICE_CAPABILITIES_MASK RADIO_tDecoderCapabilities (
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder
        );

BOOLEAN RADIO_bInitializeDecoder(
    DECODER_OBJECT hDecoder,
    SRH_DEVICE_CAPABILITIES_MASK tCapabilities,
    STI_HDL hSTI,
    MODULE_SUBSTATUS_ENUM eSubStatus,
    MODULE_SUBSTATUS_REASON_CODE tSubReasonCode,
    UN32 un32SubSuspendDate,
    void *pvEventHdlr
        );

void RADIO_vUninitializeDecoder(
    DECODER_OBJECT hDecoder
        );

BOOLEAN RADIO_bDecoderEventHandler(
    DECODER_OBJECT hDecoder,
    SMS_EVENT_TYPE_ENUM eEventType,
    const void *pvEventData
        );

CHANNEL_ID RADIO_tChannelIdFromOffset(
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    N16 n16Offset);

N16 RADIO_n16ChannelOffsetFromId(
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId);

CATEGORY_ID RADIO_tCategoryIdFromOffset(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    N16 n16Offset);

N16 RADIO_n16CategoryOffsetFromId(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId);

BOOLEAN RADIO_bTuneServiceId(
    DECODER_OBJECT hDecoder,
    SERVICE_ID tServiceId,
    BOOLEAN bMature,
    BOOLEAN bLocked,
    BOOLEAN bSkipped,
    BOOLEAN bPlayUnrestricted
        );

BOOLEAN RADIO_bScanContent (
    DECODER_OBJECT hDecoder,
    BOOLEAN bMusicOnly
        );

BOOLEAN RADIO_bScanBack (
    DECODER_OBJECT hDecoder,
    BOOLEAN bReverse
        );

BOOLEAN RADIO_bScanForward (
    DECODER_OBJECT hDecoder
        );

BOOLEAN RADIO_bScanTerminate (
    DECODER_OBJECT hDecoder,
    BOOLEAN bAbort
        );

BOOLEAN RADIO_bScanSelectCfg (
    DECODER_OBJECT hDecoder,
    BOOLEAN bMusicOnly,
    BOOLEAN bScanMature,
    BOOLEAN bScanLocked,
    BOOLEAN bScanSkipped
        );

BOOLEAN RADIO_bScanItemsMon (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        );

BOOLEAN RADIO_bSignalMonitorSwitch (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        );

BOOLEAN RADIO_bAntennaAimingMonitorSwitch (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        );

BOOLEAN RADIO_bAudioPresenceMonitorSwitch (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        );

BOOLEAN RADIO_bRequestChannel(
    DECODER_OBJECT hDecoder,
    SERVICE_ID tServiceId
        );

PLAYBACK_ERROR_CODE_ENUM RADIO_ePlay(
    DECODER_OBJECT hDecoder
        );

PLAYBACK_ERROR_CODE_ENUM RADIO_ePause(
    DECODER_OBJECT hDecoder
        );

PLAYBACK_ERROR_CODE_ENUM RADIO_eSeek(
    DECODER_OBJECT hDecoder,
    RADIO_SEEK_TYPE eType,
    N32 n32Offset,
    BOOLEAN bPauseAfterSeek);

PLAYBACK_ERROR_CODE_ENUM RADIO_eSetPlaybackParams(
    DECODER_OBJECT hDecoder,
    UN32 un32WarningOffset
        );

BOOLEAN RADIO_bIsChannelSubscribed(
    DECODER_OBJECT hDecoder,
    SERVICE_ID tServiceId);

CATEGORY_OBJECT RADIO_hCreateCategory(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        );

BOOLEAN RADIO_bGetESN(
    MODULE_OBJECT hModule,
    STRING_OBJECT *phESN
        );

MODULE_SUBSTATUS_ENUM RADIO_eSubStatus(
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder
        );

MODULE_SUBSTATUS_REASON_CODE RADIO_tReasonCode(
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder
        );

UN32 RADIO_un32SuspendDate(
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder
        );

STRING_OBJECT RADIO_hPhoneNumber(
    MODULE_OBJECT hModule
        );

BOOLEAN RADIO_bIsAudioSubscribed(
    MODULE_OBJECT hModule
        );

STRING_OBJECT RADIO_hReasonText(
    MODULE_OBJECT hModule
        );

BOOLEAN RADIO_bDaylightSavingsTimeSupported(
    MODULE_OBJECT hModule
        );

BOOLEAN RADIO_bSetTimeParameters(
    MODULE_OBJECT hModule,
    N16 n16GmtOffset,
    BOOLEAN bDaylightSavingsTimeObserved
        );

SMSAPI_RETURN_CODE_ENUM RADIO_eFirmwareUpdate(
    MODULE_OBJECT hModule,
    const char *pcFirmwareFileName
        );

void RADIO_vAbortFirmwareUpdate(
    MODULE_OBJECT hModule
        );

N8 RADIO_n8GetNumAntennas(
    DECODER_OBJECT hDecoder
        );

ANTENNA_STATE_ENUM RADIO_eGetAntennaState(
    DECODER_OBJECT hDecoder,
    N8 n8Antenna
        );

BOOLEAN RADIO_bFindCapableDevices(
    SRM_OBJECT hSRM,
    MODULE_OBJECT *phModule,
    DECODER_OBJECT *phDecoder,
    const char **ppacDecoderName,
    SRH_DEVICE_CAPABILITIES_MASK tRequestedCapabilities
        );

BOOLEAN RADIO_bFreeDataPayload (
    OSAL_BUFFER_HDL hPayload
        );

BOOLEAN RADIO_bFeaturedFavoritesEnable (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        );

#if SMS_LOGGING == 1

const char *RADIO_pacEventTypeText(
    SMS_EVENT_TYPE_ENUM eType
        );

#endif /* SMS_LOGGING == 1 */

SPORTS_ENUM RADIO_eMapSport(
    char const *pacSportType
        );

LEAGUE_ENUM RADIO_eMapLeague(
    UN32 un32LeagueId
        );

BOOLEAN RADIO_bConfirmDiscoveredMarkets(void);

SMSAPI_RETURN_CODE_ENUM RADIO_ePackage (
    MODULE_OBJECT hModule,
    MODULE_PACKGE_CMD_ENUM eCmd,
    UN8 un8PackageIndex
        );

BOOLEAN RADIO_bDefaultDst(void);

SMSAPI_RETURN_CODE_ENUM RADIO_eToneGenerationStart(
    DECODER_OBJECT hDecoder,
    UN32 un32ToneFreqHz,
    N8 n8Volume,
    DECODER_TONE_GENERATION_BALANCE_ENUM eBalance,
    BOOLEAN bAlert
        );

SMSAPI_RETURN_CODE_ENUM RADIO_eToneGenerationStop(
    DECODER_OBJECT hDecoder
        );

SMSAPI_RETURN_CODE_ENUM RADIO_eAdjustVolume(
    DECODER_OBJECT hDecoder,
    N16 n16Level
        );

SMSAPI_RETURN_CODE_ENUM RADIO_eModifyEngineeringData(
    MODULE_OBJECT hModule,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eEnable
        );

SMSAPI_RETURN_CODE_ENUM RADIO_eGetFirmwareFileVersion(
    FILE *psFile,
    UN32 *pun32FileVersion,
    UN32 *pun32FileEarliestVersion,
    UN32 *pun32FileLatestVersion,
    UN32 *pun32FileTypeID
        );

SMSAPI_RETURN_CODE_ENUM RADIO_eGetIRDurationOfBuffer(
    DECODER_OBJECT hDecoder,
    UN32 *pun32IRDurationOfBuffer
        );

BOOLEAN RADIO_bSmartSelect(
    DECODER_OBJECT hDecoder,
    SERVICE_ID atServiceId[],
    UN16 un16NumServiceIds
        );

BOOLEAN RADIO_bSetSmartFavsCfg(
    DECODER_OBJECT hDecoder,
    SMART_FAVORITES_PLAY_POINT_CTRL_ENUM ePlayPoint
        );

BOOLEAN RADIO_bSetPlaySeconds (
    DECODER_OBJECT hDecoder,
    UN8 un8PlaySeconds
        );

BOOLEAN RADIO_bLockChannel (
    DECODER_OBJECT hDecoder,
    SERVICE_ID atServiceId[],
    UN16 un16NumServiceIds
        );

BOOLEAN RADIO_bSkipChannel (
    DECODER_OBJECT hDecoder,
    SERVICE_ID atServiceId[],
    UN16 un16NumServiceIds
        );

BOOLEAN RADIO_bAudioVolume(
    DECODER_OBJECT hDecoder,
    N8 n8Volume
        );

BOOLEAN RADIO_bPostModuleEvent (
    MODULE_OBJECT hModule,
    SMS_EVENT_TYPE_ENUM eEvent,
    EVENT_OPTIONS_TYPE tOptions,
    const void *pvData
        );

BOOLEAN RADIO_bUpdateFeaturedFavoritesState (
    DECODER_OBJECT hDecoder,
    BOOLEAN bUpdateComplete
        );

BOOLEAN RADIO_bGetFFTableStatus (
    DECODER_OBJECT hDecoder
        );

BOOLEAN RADIO_bTuneMixConfigure (
    DECODER_OBJECT hDecoder,
    SERVICE_ID *ptServiceId,
    UN16 un16NumServiceIds,
    UN8 un8TuneMixChannelIndex
        );

void RADIO_vSelfTuneComplete(
    DECODER_OBJECT hDecoder
        );

BOOLEAN RADIO_bSeekMonitorEnable (
    DECODER_OBJECT hDecoder,
    SEEK_MONITOR_TYPE_ENUM eType,
    UN8 un8ValueCnt,
    UN8 un8ValueLen,
    const void *pvValues
        );

BOOLEAN RADIO_bSeekMonitorDisable (
    DECODER_OBJECT hDecoder,
    SEEK_MONITOR_TYPE_ENUM eType
        );

BOOLEAN RADIO_bSeekMonitorDisableAll (
    DECODER_OBJECT hDecoder
        );

BOOLEAN RADIO_bSetGamesMonitor(
    DECODER_OBJECT hDecoder,
    SERVICE_ID atServiceId[],
    UN16 un16NumServiceIds
        );

BOOLEAN RADIO_bPlayFlashEvent(
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tEventId
        );

BOOLEAN RADIO_bAbortFlashEvent(
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tEventId
        );

BOOLEAN RADIO_bRemainFlashEventChannel(
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tEventId
        );

BOOLEAN RADIO_bBulletinMonitor (
    DECODER_OBJECT hDecoder,
    BULLETIN_MONITOR_TYPE_ENUM eType,
    UN8 un8ParamCnt,
    const UN16 *pun16ParamList
        );

BOOLEAN RADIO_bPlayBulletin(
    DECODER_OBJECT hDecoder,
    TW_NOW_BULLETIN_ID tBulletinId
        );

BOOLEAN RADIO_bAbortBulletin(
    DECODER_OBJECT hDecoder,
    TW_NOW_BULLETIN_ID tBulletinId
        );

#endif /* RADIO_H_ */
