/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
#include <stdlib.h>
#include <stdio.h>

#include "standard.h"
#include "osal.h"
#include "sxi.h"
#include "sti_api.h"
#include "sxi_rw.h"
#include "sxiapi.h"

/*****************************************************************************
*
*       SXIAPI_eUARTLinkStartCmd
*
*       This function assembles the serial message that will start the UART
*       Link Layer
*
*       Inputs:
*           STI_HDL hSTI,
*           UN8 un8UARTControl,
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eUARTLinkStartCmd (
    STI_HDL hSTI,
    SXIAPI_UARTCONTROL_ENUM eValue
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Check input parameters
    if( eValue >= SXIAPI_UARTCONTROL_UNKNOWN)
    {
        // There are input parameter errors!
        return SXIAPI_STATUSCODE_ERROR;
    }

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_UARTLINKSTARTCMD);

        // Populate message
        // See RX224, sec 3.2
        SXIAPI_bWriteUint8(hPayload, (UN8)eValue);
        SXIAPI_bWriteUint8(hPayload, 0x00);    // RESERVED

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eUARTLinkStatsCmd
*
*       This function assembles the serial message that will query the UART
*       Link Layer stats
*
*       Inputs:
*           STI_HDL hSTI,
*           UN8 un8ResetStats,
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eUARTLinkStatsCmd (
    STI_HDL hSTI,
    UN8 un8ResetStats
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_UARTLINKSTATSCMD);

        // Populate message
        SXIAPI_bWriteUint8(hPayload, un8ResetStats);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_DEFAULT_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}
