/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*DESCRIPTION
*
*	This file contains all the functions that are common to the SXIAPI
*   Read/Write Methods when the STI is used.
*
*******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sxi_rw.h"

#if  !(defined(OSAL_CPU_BIG_ENDIAN)) && !(defined(OSAL_CPU_LITTLE_ENDIAN))
    #error Error! You must define OSAL_CPU_LITTLE_ENDIAN or OSAL_CPU_BIG_ENDIAN
#endif

    // Endian-less inline functions...

/*****************************************************************************
*
*   SXIAPI_bWriteBoolean
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bWriteBoolean(OSAL_BUFFER_HDL hPayload, BOOLEAN bData)
{
    UN8 un8Data = (UN8)bData;
    size_t tNum;
    // Condition input
    if(un8Data != 0) un8Data = 1;
    tNum = OSAL.tBufferWriteTail(hPayload, &un8Data, sizeof(UN8));
    return (tNum == sizeof(UN8));
}

/*****************************************************************************
*
*   SXIAPI_bWriteUint8
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bWriteUint8(OSAL_BUFFER_HDL hPayload, UN8 un8Data)
{
    size_t tNum = OSAL.tBufferWriteTail(hPayload, &un8Data, sizeof(UN8));
    return (tNum == sizeof(UN8));
}

/*****************************************************************************
*
*   SXIAPI_bWriteSint8
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bWriteSint8(OSAL_BUFFER_HDL hPayload, N8 n8Data)
{
    size_t tNum = OSAL.tBufferWriteTail(hPayload, &n8Data, sizeof(N8));
    return (tNum == sizeof(N8));
}

/*****************************************************************************
*
*   SXIAPI_bReadBoolean
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bReadBoolean(OSAL_BUFFER_HDL hPayload, BOOLEAN *pbData)
{
    UN8 un8Data;
    size_t tNum = OSAL.tBufferReadHead(hPayload, &un8Data, sizeof(UN8));

    // Condition output
    *pbData = (un8Data != 0) ? 1 : 0;
    return (tNum == sizeof(UN8));
}

/*****************************************************************************
*
*   SXIAPI_bReadUint8
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bReadUint8(OSAL_BUFFER_HDL hPayload, UN8 *pun8Data)
{
    size_t tNum = OSAL.tBufferReadHead(hPayload, pun8Data, sizeof(UN8));
    return (tNum == sizeof(UN8));
}

/*****************************************************************************
*
*   SXIAPI_bReadSint8
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bReadSint8(OSAL_BUFFER_HDL hPayload, N8 *pn8Data)
{
    size_t tNum = OSAL.tBufferReadHead(hPayload, pn8Data, sizeof(N8));
    return (tNum == sizeof(N8));
}

/*****************************************************************************
*
*   SXIAPI_bPeekUint8
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bPeekUint8(OSAL_BUFFER_HDL hPayload, UN8 *pun8Data, size_t tOffset)
{
    size_t
    tNum = OSAL.tBufferPeek(hPayload, pun8Data, sizeof(UN8), tOffset);
    return (tNum == sizeof(UN8));
}

/*****************************************************************************
*
*   SXIAPI_bWriteUint16
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bWriteUint16(OSAL_BUFFER_HDL hPayload, UN16 un16Data)
{
    size_t tNum;

#ifdef OSAL_CPU_LITTLE_ENDIAN
    un16Data = SWAPUN16(un16Data);
#endif

    tNum = OSAL.tBufferWriteTail(hPayload, (void *)&un16Data, sizeof(UN16));

    return (tNum == sizeof(UN16));
}

/*****************************************************************************
*
*   SXIAPI_bWriteSint16
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bWriteSint16(OSAL_BUFFER_HDL hPayload, N16 n16Data)
{
    size_t tNum;

#ifdef OSAL_CPU_LITTLE_ENDIAN
    n16Data = (N16)SWAPUN16(n16Data);
#endif

    tNum = OSAL.tBufferWriteTail(hPayload, (void *)&n16Data, sizeof(N16));

    return (tNum == sizeof(N16));
}

/*****************************************************************************
*
*   SXIAPI_bWriteUint32
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bWriteUint32(OSAL_BUFFER_HDL hPayload, UN32 un32Data)
{
    size_t tNum;

#ifdef OSAL_CPU_LITTLE_ENDIAN
    un32Data = SWAPUN32(un32Data);
#endif

    tNum = OSAL.tBufferWriteTail(hPayload, (void *)&un32Data, sizeof(UN32));

    return (tNum == sizeof(UN32));
}

/*****************************************************************************
*
*   SXIAPI_bWriteSint32
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bWriteSint32(OSAL_BUFFER_HDL hPayload, N32 n32Data)
{
    size_t tNum;

#ifdef OSAL_CPU_LITTLE_ENDIAN
    n32Data = (N32)SWAPUN32(n32Data);
#endif

    tNum = OSAL.tBufferWriteTail(hPayload, (void *)&n32Data, sizeof(UN32));

    return (tNum == sizeof(N32));
}

/*****************************************************************************
*
*   SXIAPI_bReadUint16
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bReadUint16(OSAL_BUFFER_HDL hPayload, UN16 *pun16Data)
{
    size_t tNum;
#ifdef OSAL_CPU_LITTLE_ENDIAN
    UN16 un16Data;
    tNum = OSAL.tBufferReadHead(hPayload, (void*)&un16Data, sizeof(UN16));
    *pun16Data = SWAPUN16(un16Data);
#else
    tNum = OSAL.tBufferReadHead(hPayload, (void*)pun16Data, sizeof(UN16));
#endif

    return (tNum == sizeof(UN16));
}

/*****************************************************************************
*
*   SXIAPI_bPeekUint16
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bPeekUint16(OSAL_BUFFER_HDL hPayload, UN16 *pun16Data, size_t tOffset)
{
    size_t tNum;

#ifdef OSAL_CPU_LITTLE_ENDIAN
    UN16 un16Data;
    tNum = OSAL.tBufferPeek(hPayload, (void *)&un16Data, sizeof(UN16), tOffset);
    *pun16Data = SWAPUN16(un16Data);
#else
    tNum = OSAL.tBufferPeek(hPayload, (void *)pun16Data, sizeof(UN16), tOffset);
#endif

    return (tNum == sizeof(UN16));
}

/*****************************************************************************
*
*   SXIAPI_bReadSint16
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bReadSint16(OSAL_BUFFER_HDL hPayload, N16 *pn16Data)
{
    size_t tNum;
#ifdef OSAL_CPU_LITTLE_ENDIAN
    N16 n16Data;
    tNum = OSAL.tBufferReadHead(hPayload, (void*)&n16Data, sizeof(N16));
    *pn16Data = (N16)SWAPUN16(n16Data);
#else
    tNum = OSAL.tBufferReadHead(hPayload, (void*)pn16Data, sizeof(N16));
#endif

    return (tNum == sizeof(N16));
}

/*****************************************************************************
*
*   SXIAPI_bReadUint32
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bReadUint32(OSAL_BUFFER_HDL hPayload, UN32 *pun32Data)
{
    size_t tNum;
#ifdef OSAL_CPU_LITTLE_ENDIAN
    UN32 un32Data;
    tNum = OSAL.tBufferReadHead(hPayload, (void *)&un32Data, sizeof(UN32));
    *pun32Data = SWAPUN32(un32Data);
#else
    tNum = OSAL.tBufferReadHead(hPayload, (void *)pun32Data, sizeof(UN32));
#endif
    return (tNum == sizeof(UN32));
}

/*****************************************************************************
*
*   SXIAPI_bReadSint32
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bReadSint32(OSAL_BUFFER_HDL hPayload, N32 *pn32Data)
{
    size_t tNum;
#ifdef OSAL_CPU_LITTLE_ENDIAN
    N32 n32Data;
    tNum = OSAL.tBufferReadHead(hPayload, (void *)&n32Data, sizeof(N32));
    *pn32Data = (N32)SWAPUN32(n32Data);
#else
    tNum = OSAL.tBufferReadHead(hPayload, (void *)pn32Data, sizeof(N32));
#endif
    return (tNum == sizeof(N32));
}

/*****************************************************************************
*
*       SXIAPI_bWriteString
*
*       This function writes a NULL terminated ANSI-C string to an SXI
*       payload buffer according to RX223.
*
*       Inputs:
*               hPayload - A handle to the SSP Payload to write to
*               pacString - A NULL terminated string to write
*
*       Outputs:
*               BOOLEAN TRUE if entire string was written, otherwise FALSE
*
*****************************************************************************/
BOOLEAN SXIAPI_bWriteString(
    OSAL_BUFFER_HDL hPayload,
    const char *pacString
        )
{
    size_t tNum, tLen;

    // String length including null terminator
    tLen = strlen(pacString) + 1;

    // Write string
    tNum = OSAL.tBufferWriteTail(hPayload, (UN8*)pacString, tLen);

    // Verify length written
    return (tNum == tLen);
}

/*****************************************************************************
*
*       SXIAPI_bReadString
*
*       This function read a string from a provided SXI payload into
*       a provided character array as an ANSI-C NULL terminated string.
*
*       Inputs:
*               hPayload - A handle to the SSP Payload to read from.
*               pacString - A pointer to a character array to write the string
*                   into.
*               tSize - The maximum number of characters to write, including
*                   the NULL terminator. You must allocate at least tSize bytes.
*
*       Outputs:
*               BOOLEAN TRUE if entire string was copied, otherwise FALSE
*
*****************************************************************************/
BOOLEAN SXIAPI_bReadString(
    OSAL_BUFFER_HDL hPayload,
    char *pacString,
    size_t tSize
        )
{
    size_t tReadSize;

    // SXI strings are just normal old null-terminated C Strings.
    // So read until we hit a null or we have no more bytes to read.
    tReadSize = OSAL.tBufferReadString(hPayload, pacString, tSize);

    return (tReadSize != 0) ? TRUE : FALSE;
}

/*****************************************************************************
*
*       SXIAPI_bWriteBlob
*
*       This function a binary data array to an SXI payload buffer
*
*       Inputs:
*               hPayload - A handle to the SXI Payload to write to
*               pvBlob - A data array to write
*               tSize - size of data to write
*
*       Outputs:
*               BOOLEAN TRUE if entire data array was written, 
*                       otherwise FALSE
*
*****************************************************************************/
__INLINE__ BOOLEAN SXIAPI_bWriteBlob(
    OSAL_BUFFER_HDL hPayload,
    const void *pvBlob,
    size_t tSize
        )
{
    size_t tNum;

    // Write data array
    tNum = OSAL.tBufferWriteTail(hPayload, pvBlob, tSize);

    // Verify length written
    return (tNum == tSize);
}
