/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
#include <time.h>

#include "standard.h"

#include "osal.h"
#include "sms_obj.h"

#include "sti_api.h"

#include "sms_event.h"
#include "radio_event_types.h"

#include "sxill.h"

#include "sxi_rw.h"
#include "sxi_fsm.h"
#include "sxi_gmd.h"
#include "sxi.h"
#include "_sxi.h"

#include "sms_fcsxm.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

#ifndef MAX
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#endif
// Calculates waiting time for SXI total command sending with all retries
// as r * t + max(t, m).
#define SXI_TOTAL_DELAY(r,t,m)  ( (r) * (t) + MAX( (m), (t) ) )

/*****************************************************************************
*
*   SXI_hGMD
*
*****************************************************************************/
SXI_GMD_HDL SXI_hGMD (
    STI_HDL hControlCxn
        )
{
    SXI_GMD_HDL hGMD = SXI_GMD_INVALID_HDL;
    SXI_CONNECTION_ARG_STRUCT *psConnectionArg;

    psConnectionArg = (SXI_CONNECTION_ARG_STRUCT *)
        STI_pvGetConnectionArgument(hControlCxn);
    if(psConnectionArg != SXI_CONNECTION_ARG_INVALID_HANDLE)
    {
        // This is only available from control connections
        if(psConnectionArg->tPayloadType == SXILL_PAYLOAD_TYPE_CONTROL)
        {
            hGMD = psConnectionArg->uConnection.sControl.hGMD;
        }
    }

    return hGMD;
}

/*****************************************************************************
*
*   SXI_hFSM
*
*****************************************************************************/
SXI_FSM_HDL SXI_hFSM (
    STI_HDL hControlCxn
        )
{
    SXI_FSM_HDL hFSM = SXI_FSM_INVALID_HDL;
    SXI_CONNECTION_ARG_STRUCT *psConnectionArg;

    psConnectionArg =  (SXI_CONNECTION_ARG_STRUCT *)
        STI_pvGetConnectionArgument(hControlCxn);
    if(psConnectionArg != SXI_CONNECTION_ARG_INVALID_HANDLE)
    {
        hFSM = psConnectionArg->hFSM;
    }

    return hFSM;
}

/*****************************************************************************
*
*   SXI_hInitConnectionArg
*
*****************************************************************************/
SXI_CONNECTION_ARG_HANDLE SXI_hInitConnectionArg(
    SXI_FSM_HDL hFSM,
    SXILL_PAYLOAD_TYPE tPayloadType,
    UN32 un32ResponseQueueSize,
    OSAL_OBJECT_HDL hDataPayloadDestinations
        )
{
    SXI_CONNECTION_ARG_STRUCT *psObj;
    BOOLEAN bSuccess = FALSE;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Construct a unique name for this object.
    snprintf( &acName[0], sizeof(acName), "SXI:ConnArg:%u", tPayloadType );

    // Create an instance of this object
    psObj = (SXI_CONNECTION_ARG_STRUCT *)
        SMSO_hCreate(
            &acName[0],
            sizeof(SXI_CONNECTION_ARG_STRUCT),
            SMS_INVALID_OBJECT, // Parent
            TRUE); // Lock it up

    if(psObj == NULL)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SXI_LOG_BASE_NAME": Unable to create SXI Conn Arg type: %d",
            tPayloadType);
        return SXI_CONNECTION_ARG_INVALID_HANDLE;
    }

    // Save off FSM
    psObj->hFSM = hFSM;

    // Save off type
    psObj->tPayloadType = tPayloadType;

    // Initialize ReTx in case it is not needed
    psObj->psReTx = (SXI_RETX_STRUCT *)NULL;

    switch(tPayloadType)
    {
        case SXILL_PAYLOAD_TYPE_LINK:
        {
            // Initialize name
            psObj->pacName = "Link";

            // Initialize counters
            psObj->uConnection.sLink.un32KeepAlivesReceived = 0;
            psObj->uConnection.sLink.un32LinkErrors = 0;
            psObj->uConnection.sLink.un32UnsupportedPayloadType = 0;
            psObj->uConnection.sLink.un32UnknownOpCode = 0;
            psObj->uConnection.sLink.un32UnknownError = 0;

            // Create Re-Tx
            psObj->uConnection.sLink.psReTx = psCreateReTx(psObj->pacName);
            if(NULL != psObj->uConnection.sLink.psReTx)
            {
                psObj->psReTx = psObj->uConnection.sLink.psReTx;
                bSuccess = TRUE;
            }
        }
        break;

        case SXILL_PAYLOAD_TYPE_CONTROL:
        {
            SXI_CONTROL_CONNECTION_STRUCT *psControl =
                &psObj->uConnection.sControl;
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Initialize name
            psObj->pacName = "Control";

            // Initialize interface
            psControl->tTransactionID = 0;

            // Store data demux list
            psControl->hDataPayloadDestinations = hDataPayloadDestinations;

            // Create a response queue, of size requested
            if (un32ResponseQueueSize > 0)
            {
                psControl->psRespQueue =
                    psCreateResponseQueue(psObj->pacName,
                        un32ResponseQueueSize);

                if (psControl->psRespQueue == NULL)
                {
                    break;
                }
            }

            // Create Re-Tx
            psControl->psReTx = psCreateReTx("Control");
            if(NULL == psControl->psReTx)
            {
                // Error!
                break;
            }

            psObj->psReTx = psControl->psReTx;

            // Create a GMD object
            psControl->hGMD = SXI_GMD_hInit();
            if(SXI_GMD_INVALID_HDL == psControl->hGMD)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SXI_LOG_BASE_NAME": Can't create GMD object.");
                break;
            }

            // Create the data block pool list
            eReturnCode = OSAL.eLinkedListCreate(
                &psControl->hDataBlockPools,
                "SXI:BlockPoolList",
                (OSAL_LL_COMPARE_HANDLER)n16CompareBlockPoolEntry,
                OSAL_LL_OPTION_NONE);
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SXI_LOG_BASE_NAME": Can't create OSAL list.");
                break;
            }

            bSuccess = TRUE;
        }
        break;

        case SXILL_PAYLOAD_TYPE_DATA:
        {
            SXI_DATA_CONNECTION_STRUCT *psData =
                &psObj->uConnection.sData;

            // Initialize name
            psObj->pacName = "Data";

            // Store data demux list
            psData->hDataPayloadDestinations = hDataPayloadDestinations;

            bSuccess = TRUE;
        }
        break;

        case SXILL_PAYLOAD_TYPE_AUDIO:
        {
            // Initialize name
            psObj->pacName = "Audio";

            // TODO:
            bSuccess = FALSE;
        }
        break;

        default:
        break;
    }

#if SMS_LOGGING == 1
    if (psObj != NULL)
    {
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Construct a unique name for this object.
        snprintf( &acName[0], sizeof(acName),
                  SXI_LOG_BASE_NAME":%s",
                  psObj->pacName);

        // Initialize statistics
        OSAL.bMemSet(&psObj->sLog.sStatistics, 0, 
            sizeof(psObj->sLog.sStatistics));

        // Open our stats log
        eReturnCode = OSAL.eLogCreate(&psObj->sLog.hLog, acName,
            SXI_LOG_SIZE, OSAL_LOG_OPTION_FORMAT);

        if (OSAL_SUCCESS == eReturnCode)
        {
            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "Log created.\n");

            // Write the header
            OSAL.n32LogWrite(psObj->sLog.hLog, SXI_LOG_HEADER, acName);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME": Unable to create stats log.\n");
        }
    }
#endif

    if(bSuccess == TRUE)
    {
        SMSO_vUnlock((SMS_OBJECT)psObj);
    }
    else
    {
        SXI_vFreeConnectionArg(
            (SXI_CONNECTION_ARG_HANDLE)psObj);
        psObj = (SXI_CONNECTION_ARG_STRUCT *)NULL;
    }

    return (SXI_CONNECTION_ARG_HANDLE)psObj;
}

/*****************************************************************************
*
*   SXI_eSendPayload
*
*****************************************************************************/
STI_PROTOCOL_RESULT_CODE_ENUM SXI_eSendPayload(
    STI_HDL hSTI,
    OSAL_BUFFER_HDL hPayload,
    UN8 un8Retries,
    N32 n32Timeout,
    SXIAPI_STATUSCODE_ENUM *peStatusCode
        )
{
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    STI_PROTOCOL_RESULT_CODE_ENUM eResultCode = STI_RESULT_CODE_ERROR;
    SXI_CONNECTION_ARG_STRUCT *psConnectionArg;
    void *pvArg;

    if(peStatusCode == NULL)
    {
        // Fine! we'll use our own
        peStatusCode = &eStatusCode;
    }

    // Initialize return value
    *peStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Bosch ID#26: Avoid sending any commands to the module
    // during low-voltage scenario
    if(fc_sxm_bIsSMSLIB_CvOn())
    {
    	return eResultCode;
    }

    // Extract connection specific info
    pvArg = pvGetConnectionSpecificInfo(hSTI, &psConnectionArg);
    if(pvArg != NULL)
    {
        // If we expect to send and re-try we must acquire
        // the retx synchronization mutex first.
        if( (0 < un8Retries) && // has re-tries
            (OSAL_OBJ_TIMEOUT_NONE < n32Timeout) && // has timeout
            (psConnectionArg->psReTx != NULL) // has Re-Tx control
                )
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Wait to acquire re-tx synchronization mutex
            eReturnCode = OSAL.eSemTake(
                psConnectionArg->psReTx->hMutex, OSAL_OBJ_TIMEOUT_INFINITE);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Make sure resource is taken
                OSAL.eSemTake(psConnectionArg->psReTx->hResponse,
                    OSAL_OBJ_TIMEOUT_NONE);

                // Initialize re-tx structure
                psConnectionArg->psReTx->sMsg.eStatusCode =
                    SXIAPI_STATUSCODE_MSG_UNKNOWN;
                psConnectionArg->psReTx->sMsg.hPayload =
                    OSAL_INVALID_BUFFER_HDL;
                psConnectionArg->psReTx->sMsg.n32Timeout = 0;
                psConnectionArg->psReTx->sMsg.un8Retries = 0;
                psConnectionArg->psReTx->sMsg.sHdr.tOpType = 0;
                psConnectionArg->psReTx->sMsg.sHdr.tTransactionID = 0;

#if (OBJECT_DEBUG == 1)
                psConnectionArg->psReTx->un32TimeoutCounter = 0;
#endif

                // Send payload to STI for tx, no blocking, with retry
                eResultCode = STI_eSendPayload(
                    hSTI, FALSE, hPayload, un8Retries, n32Timeout);
                if(eResultCode == STI_RESULT_CODE_OK)
                {
                    // Wait for timeout/response mutex, when obtained
                    // this means we have either received a response or
                    // a timeout happened (no response received).
                    printf(SXI_LOG_BASE_NAME": Setting timer for %d millisecond(s)\n", SXI_TOTAL_DELAY(un8Retries, n32Timeout, 1000));
                    eReturnCode = OSAL.eSemTake(
                        psConnectionArg->psReTx->hResponse,
                        /* finite timeout to avoid infinite lock */
                        SXI_TOTAL_DELAY(un8Retries, n32Timeout, 1000)
                            );
                    if(eReturnCode == OSAL_SUCCESS)
                    {
                        // Go grab status code from message
                        *peStatusCode = psConnectionArg->psReTx->sMsg.eStatusCode;

                        // Retrieve response status code
                        if((*peStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED) ||
                            (*peStatusCode ==  SXIAPI_STATUSCODE_MSG_COMPLETE))
                        {
                            // Command was received/complete (OK)
                        }
                        else if(*peStatusCode == SXIAPI_STATUSCODE_MSG_FAIL)
                        {
                            // Command was received but failed (ERROR)
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                SXI_LOG_BASE_NAME": SXIAPI_STATUSCODE_MSG_FAIL");
                            eResultCode = STI_RESULT_CODE_ERROR;
                        }
                        else if(*peStatusCode == SXIAPI_STATUSCODE_MSG_UNKNOWN)
                        {
                            // Command was received but is unknown (ERROR)
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                SXI_LOG_BASE_NAME": SXIAPI_STATUSCODE_MSG_UNKNOWN");
                            eResultCode = STI_RESULT_CODE_ERROR;
                        }
                        else
                        {
                            // Command was not responded to (TIMEOUT)
#if (DEBUG_OBJECT == 1)
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                SXI_LOG_BASE_NAME
                                ": SXIAPI_STATUSCODE_ERROR (timeout).");

                            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                                "Timeout: [%u:%u]->[%u:%u]=%u/%u msec,"
                                " %u timeouts, opcode=0x%x, tid=%u\n",
                                psConnectionArg->psReTx->un32StartSeconds,
                                psConnectionArg->psReTx->un16StartMsec,
                                psConnectionArg->psReTx->un32EndSeconds,
                                psConnectionArg->psReTx->un16EndMsec,
                                psConnectionArg->psReTx->un32DurationMsec,
                                psConnectionArg->psReTx->sMsg.n32Timeout,
                                psConnectionArg->psReTx->un32TimeoutCounter,
                                psConnectionArg->psReTx->sMsg.sHdr.tOpType,
                                psConnectionArg->psReTx->sMsg.sHdr.tTransactionID
                                    );
#endif
                            eResultCode = STI_RESULT_CODE_TIMEOUT;
                        }
                    }
                    else
                    {
                        // Command was not responded to and no timeout
                        // happened after this command was sent. Something
                        // is really wrong here. (BIG_PROBLEM)
                        printf(SXI_LOG_BASE_NAME": [%d] BIG_PROBLEM (%s)!\n",
                            __LINE__, OSAL.pacGetReturnCodeName(eReturnCode));

                        // Give up(release) re-tx msg ctrl
                        vReleaseReTx(psConnectionArg->psReTx);
                        eResultCode = STI_RESULT_CODE_ERROR;
                    }
                }
                else
                {
                    // SXi Send Payload error
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SXI_LOG_BASE_NAME": SXI Send Payload Error (%u)!",
                        eResultCode);
                }

                // Release synchronization mutex
                OSAL.eSemGive(psConnectionArg->psReTx->hMutex);
            }
            else
            {
                // SXi Synchronization Mutex error
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SXI_LOG_BASE_NAME": SXI hMutex Error! (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
        }
        else // No re-try, just send and pray
        {
            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 5,
                "No re-try, send without waiting for response.\n");

            // Send payload to STI for tx, no blocking, no re-try, no timeout
            eResultCode = STI_eSendPayload(
                hSTI, FALSE, hPayload, 0, OSAL_OBJ_TIMEOUT_NONE);
            *peStatusCode = SXIAPI_STATUSCODE_MSG_RECEIVED;
        }
    }
    else
    {
        // SXi connection error
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SXI_LOG_BASE_NAME": SXI No Connection!");
        eResultCode = STI_RESULT_CODE_ERROR;
    }

    return eResultCode;
}

/*****************************************************************************
*
*   SXI_tGetNextTransIDForConnection
*
*****************************************************************************/
SXIAPI_TRANID SXI_tGetNextTransIDForConnection (
    STI_HDL hSTI
        )
{
    SXIAPI_TRANID tTransactionID = 0;
    SXI_CONNECTION_ARG_STRUCT *psConnectionArg;

    psConnectionArg =  (SXI_CONNECTION_ARG_STRUCT *)
        STI_pvGetConnectionArgument(hSTI);
    if(psConnectionArg != SXI_CONNECTION_ARG_INVALID_HANDLE)
    {
        if(psConnectionArg->tPayloadType == SXILL_PAYLOAD_TYPE_CONTROL)
        {
            BOOLEAN bLocked;

            bLocked = SMSO_bLock(
                (SMS_OBJECT)psConnectionArg, OSAL_OBJ_TIMEOUT_INFINITE);
            if (bLocked == TRUE)
            {
                tTransactionID =
                    psConnectionArg->uConnection.sControl.tTransactionID++;

                SMSO_vUnlock((SMS_OBJECT)psConnectionArg);
            }
        }
    }

    return tTransactionID;
}

/*****************************************************************************
*
*   SXI_tGetTypeForConnection
*
*****************************************************************************/
SXILL_PAYLOAD_TYPE SXI_tGetTypeForConnection (
    STI_HDL hSTI
        )
{
    SXILL_PAYLOAD_TYPE tPayloadType = SXILL_PAYLOAD_TYPE_UNKNOWN;
    SXI_CONNECTION_ARG_STRUCT *psConnectionArg;

    psConnectionArg =  (SXI_CONNECTION_ARG_STRUCT *)
        STI_pvGetConnectionArgument(hSTI);
    if(psConnectionArg != SXI_CONNECTION_ARG_INVALID_HANDLE)
    {
        tPayloadType = psConnectionArg->tPayloadType;
    }

    return tPayloadType;
}

/*****************************************************************************
*
*   SXI_vFreeConnectionArg
*
*****************************************************************************/
void SXI_vFreeConnectionArg(
    SXI_CONNECTION_ARG_HANDLE hSXIConnectionArg
        )
{
    BOOLEAN bLocked;

    bLocked = SMSO_bLock((SMS_OBJECT)hSXIConnectionArg,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        SXI_CONNECTION_ARG_STRUCT *psObj =
            (SXI_CONNECTION_ARG_STRUCT *)hSXIConnectionArg;

        switch(psObj->tPayloadType)
        {
            case SXILL_PAYLOAD_TYPE_LINK:
            {
                SXI_LINK_CONNECTION_STRUCT *psLink =
                    &psObj->uConnection.sLink;

                if(NULL != psLink->psReTx)
                {
                    vDestroyReTx(psLink->psReTx);
                }
            }
            break;

            case SXILL_PAYLOAD_TYPE_CONTROL:
            {
                SXI_CONTROL_CONNECTION_STRUCT *psControl =
                    &psObj->uConnection.sControl;
                OSAL_RETURN_CODE_ENUM eReturnCode;

                // Free ReTx
                if(NULL != psControl->psReTx)
                {
                    vDestroyReTx(psControl->psReTx);
                }

                // Free the response queue
                if (psControl->psRespQueue != NULL)
                {
                    vDestroyResponseQueue(psControl->psRespQueue);
                    psControl->psRespQueue = (SXI_RESPONSE_QUEUE_STRUCT *)NULL;
                }

                // Free GMD
                if(SXI_GMD_INVALID_HDL != psControl->hGMD)
                {
                    SXI_GMD_vUninit(psControl->hGMD);
                    psControl->hGMD = SXI_GMD_INVALID_HDL;
                }

                // Free data payload list
                if (OSAL_INVALID_OBJECT_HDL != psControl->hDataPayloadDestinations)
                {
                    // Clear out the list
                    eReturnCode = OSAL.eLinkedListRemoveAll(
                        psControl->hDataPayloadDestinations,
                        (OSAL_LL_RELEASE_HANDLER)SXI_vFreeDataDest);
                    if (OSAL_SUCCESS == eReturnCode)
                    {
                        // Delete the list
                        OSAL.eLinkedListDelete(psControl->hDataPayloadDestinations);

                        // Clear the handle
                        psControl->hDataPayloadDestinations = OSAL_INVALID_OBJECT_HDL;
                    }
                }

                // Free block pool list
                if (OSAL_INVALID_OBJECT_HDL != psControl->hDataBlockPools)
                {
                    // Clear out the list
                    eReturnCode = OSAL.eLinkedListRemoveAll(
                        psControl->hDataBlockPools,
                        (OSAL_LL_RELEASE_HANDLER)vReleaseBlockPoolEntry);
                    if (OSAL_SUCCESS == eReturnCode)
                    {
                        // Delete the list
                        OSAL.eLinkedListDelete(psControl->hDataBlockPools);

                        // Clear the handle
                        psControl->hDataBlockPools = OSAL_INVALID_OBJECT_HDL;
                    }
                }
            }
            break;

            case SXILL_PAYLOAD_TYPE_DATA:
            {
                SXI_DATA_CONNECTION_STRUCT *psData =
                    &psObj->uConnection.sData;

                // Clear payload list handle
                psData->hDataPayloadDestinations = OSAL_INVALID_OBJECT_HDL;
            }
            break;

            case SXILL_PAYLOAD_TYPE_AUDIO:
            {
                // TODO:
            }
            break;

            default:
            break;
        }

#if SMS_LOGGING == 1
        // close our stats log
        if(psObj->sLog.hLog != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eLogDelete(psObj->sLog.hLog);
            psObj->sLog.hLog = OSAL_INVALID_OBJECT_HDL;
        }
#endif

        // Destroy the object
        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   SXI_bLinkConnectionCallback
*
*****************************************************************************/
BOOLEAN SXI_bLinkConnectionCallback(
    STI_HDL hSTI,
    STI_PROTOCOL_HDL hProtocol,
    STI_PROTOCOL_CALLBACK_MSG_STRUCT const *psCallback
        )
{
    BOOLEAN bResponseHandled = FALSE;
    SXI_LINK_CONNECTION_STRUCT *psLink;

    // Extract connection specific info
    psLink = (SXI_LINK_CONNECTION_STRUCT *)
        pvGetConnectionSpecificInfo(hSTI, (SXI_CONNECTION_ARG_STRUCT **)NULL);

    if (psLink == NULL)
    {
        return FALSE;
    }

    // Based on response type, process response.
    switch(psCallback->eType)
    {
        case STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_PAYLOAD:
        {
            // Decode SXI Message
            bResponseHandled =
                bDecodeLink(hProtocol, hSTI,
                    psLink, psCallback->sMsg.psRx);
        }
        break;

        case STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_PAYLOAD:
        {
            // Payload has been transmitted
            bResponseHandled =
                bHandleTxPayload(hSTI, SXILL_PAYLOAD_TYPE_LINK,
                    psCallback->sMsg.psTx, psLink->psReTx);
        }
        break;

        case STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_TIMEOUT:
        {
            vSignalTimeout(hProtocol, psLink->psReTx);
        }
        break;

        case STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_OOB:
        default:
            // Drop/Ignore it
        break;
    }

    return bResponseHandled;
}

/*****************************************************************************
*
*   SXI_bControlConnectionCallback
*
*****************************************************************************/
BOOLEAN SXI_bControlConnectionCallback(
    STI_HDL hSTI,
    STI_PROTOCOL_HDL hProtocol,
    STI_PROTOCOL_CALLBACK_MSG_STRUCT const *psCallback
        )
{
    BOOLEAN bResponseHandled = FALSE;
    SXI_CONTROL_CONNECTION_STRUCT *psControl;

    // Extract connection specific info
    psControl = (SXI_CONTROL_CONNECTION_STRUCT *)
        pvGetConnectionSpecificInfo(hSTI, (SXI_CONNECTION_ARG_STRUCT **)NULL);

    if (psControl != NULL)
    {
        // Based on response type, process response.
        switch(psCallback->eType)
        {
            case STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_PAYLOAD:
            {
                // The callback argument is
                // Decode SXI Message
                bResponseHandled =
                    bDecodeControl(hProtocol, hSTI,
                    psControl, psCallback->sMsg.psRx, psCallback->pvArg);

            } // case
            break;

            case STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_PAYLOAD:
            {
                // Payload has been transmitted
                bResponseHandled =
                    bHandleTxPayload(
                        hSTI, SXILL_PAYLOAD_TYPE_CONTROL,
                        psCallback->sMsg.psTx, psControl->psReTx);
            }
            break;

            case STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_TIMEOUT:
            {
                vSignalTimeout(hProtocol, psControl->psReTx);
            }
            break;

            case STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_OOB:
            {
                STI_OOB_TYPE_ENUM eType = psCallback->sMsg.psOOB->eType;

                if (eType == STI_OOB_TYPE_LINK_ERROR)
                {

                    SXI_FSM_HDL hFSM = SXI_hFSM(hSTI);
#if SMS_DEBUG == 1
                    SXILL_RX_ERROR_ENUM eError = (SXILL_RX_ERROR_ENUM)
                        (size_t)psCallback->sMsg.psOOB->pvData;

                    SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                        "Received OOB Response %u (eError = %u).\n",
                        eType, eError);
#endif
                    bResponseHandled = SXI_FSM_bPostOOB(
                        hFSM, psCallback->sMsg.psOOB);
                    if(bResponseHandled == FALSE)
                    {
                        SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                            "SXI_bControlConnectionCallback: OOB msg post failed.");
                    }
                }
                else if (eType == STI_OOB_TYPE_PROTOCOL_SPECIFIC)
                {
                    // We have new information on data payloads.
                    SXI_ADD_DATA_PAYLOAD_DEST_STRUCT *psAddDest =
                        (SXI_ADD_DATA_PAYLOAD_DEST_STRUCT *)psCallback->sMsg.psOOB->pvData;

                    // Process this event now
                    bResponseHandled = bHandleNewDataDestination(psControl, psAddDest);
                }
            }
            break;

            default:
                // Ignore it
            break;

        } // switch
    }

    return bResponseHandled;
}

/*****************************************************************************
*
*   SXI_bDataConnectionCallback
*
*****************************************************************************/
BOOLEAN SXI_bDataConnectionCallback(
    STI_HDL hSTI,
    STI_PROTOCOL_HDL hProtocol,
    STI_PROTOCOL_CALLBACK_MSG_STRUCT const *psCallback
        )
{
    BOOLEAN bResponseHandled = FALSE;
    SXI_DATA_CONNECTION_STRUCT *psData;

    // Extract connection specific info
    psData = (SXI_DATA_CONNECTION_STRUCT *)
        pvGetConnectionSpecificInfo(hSTI, (SXI_CONNECTION_ARG_STRUCT **)NULL);

    // Ensure info is valid
    if ((psCallback != NULL) && (psData != NULL))
    {
        // Based on response type, process response.
        switch(psCallback->eType)
        {
            case STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_PAYLOAD:
            {
                UN16 un16Opcode;
                UN8 un8TransactionId;
                SXIAPI_RX_PAYLOAD_STRUCT sRxPayload;
                BOOLEAN bSuccess;
                SXIAPI_DMI tDMI;
                SXIAPI_DATA_PACKET_TYPE_ENUM ePayloadType;

                sRxPayload.hPayload = psCallback->sMsg.psRx->hPayload;

                n32SXiLogWritePayload(hSTI, sRxPayload.hPayload, "OK");

                do
                {
                    OSAL_RETURN_CODE_ENUM eReturnCode;
                    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                    SXI_DATA_PAYLOAD_DEST_STRUCT sSearch, *psDest;
                    SXI_BLOCKPOOL_ITERATOR_STRUCT sIteratorData;

                    // Read payload enough to get the opcode so we can figure
                    // out what to do with it.
                    bSuccess = SXIAPI_bReadUint16(sRxPayload.hPayload, &un16Opcode);
                    if (bSuccess == FALSE)
                    {
                        // Error! Unable to get opcode
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            SXI_LOG_BASE_NAME":SXI_bDataConnectionCallback:"
                            " Unable to get opcode.");
                        break;
                    }

                    sRxPayload.sHdr.tOpType = (SXIAPI_OPTYPE)((SXIAPI_OPCODE)un16Opcode &
                        SXIAPI_MESSOP_TYPE_MASK);

                    // Read the transaction ID which is right past the opcode
                    bSuccess = SXIAPI_bReadUint8(sRxPayload.hPayload, &un8TransactionId);
                    if (bSuccess == FALSE)
                    {
                        // Error! Unable to get transaction id
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            SXI_LOG_BASE_NAME":SXI_bDataConnectionCallback:"
                            " Unable to get transaction id.");
                        break;
                    }

                    // Use extracted transaction id
                    sRxPayload.sHdr.tTransactionID =
                        (SXIAPI_TRANID)un8TransactionId;

                    // Decode to get the payload type & DMI
                    bSuccess = SXI_bDecodeDataPacket(
                        sRxPayload.hPayload, &ePayloadType, &tDMI);
                    if (bSuccess == FALSE)
                    {
                        // Error! Unable to get payload type and/or DMI
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            SXI_LOG_BASE_NAME":SXI_bDataConnectionCallback:"
                            " Unable to get payload type and/or DMI.");
                        break;
                    }

                    // Locate the destination for this data
                    sSearch.tDMI = tDMI;
                    eReturnCode = OSAL.eLinkedListSearch(
                        psData->hDataPayloadDestinations,
                        &hEntry,
                        &sSearch);
                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        // It's okay for this to happen!
                        break;
                    }

                    // Extract the entry now
                    psDest = (SXI_DATA_PAYLOAD_DEST_STRUCT *)
                        OSAL.pvLinkedListThis(hEntry);

                    eReturnCode = OSAL.eLinkedListItems(
                        psDest->hDestPools, &sIteratorData.un32Items);

                    if ((OSAL_SUCCESS != eReturnCode) ||
                        (0 == sIteratorData.un32Items))
                    {
                        // Something wrong with destination pools list
                        break;
                    }

                    sIteratorData.ePayloadType = ePayloadType;
                    sIteratorData.hPayload = sRxPayload.hPayload;
                    sIteratorData.bHandled = FALSE;

                    // Transfer payload to destination pool(s)
                    OSAL.eLinkedListIterate(
                        psDest->hDestPools,
                        (OSAL_LL_ITERATOR_HANDLER)bTransferPayloadIterator,
                        &sIteratorData);

                    bResponseHandled = sIteratorData.bHandled;

                } while(FALSE);

                // Free payloads if they still exist
                RADIO_bFreeDataPayload(psCallback->sMsg.psRx->hPayload);
                psCallback->sMsg.psRx->hPayload = OSAL_INVALID_BUFFER_HDL;
            }
            break;

            case STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_PAYLOAD:
            case STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_TIMEOUT:
            case STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_OOB:

            default:
                // Ignore it
            break;
        }
    }
    return bResponseHandled;
}

/*****************************************************************************
*
*   SXI_bAudioConnectionCallback
*
*****************************************************************************/
BOOLEAN SXI_bAudioConnectionCallback(
    STI_HDL hSTI,
    STI_PROTOCOL_HDL hProtocol,
    STI_PROTOCOL_CALLBACK_MSG_STRUCT const *psCallback
        )
{
    BOOLEAN bResponseHandled = FALSE;

#if 0

    // The following is commented for reference purposes, as it
    // may be used when this function is fleshed out.

    SXI_AUDIO_CONNECTION_STRUCT *psControl;

    // Extract connection specific info
    psControl = (SXI_AUDIO_CONNECTION_STRUCT *)
    pvGetConnectionSpecificInfo(hSTI, (SXI_CONNECTION_ARG_STRUCT **)NULL);

#endif

    // Based on response type, process response.
    switch(psCallback->eType)
    {
        // TODO: Audio
        case STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_PAYLOAD:
        case STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_PAYLOAD:
        case STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_TIMEOUT:
        case STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_OOB:
        default:
            // Ignore it
        break;

    } // switch

    return bResponseHandled;
}

/*****************************************************************************
*
*   psAllocateResponse
*
*****************************************************************************/
static SXIAPI_RX_STRUCT *psAllocateResponse(
    SXI_RESPONSE_QUEUE_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SXIAPI_RX_STRUCT *psResp = (SXIAPI_RX_STRUCT *)NULL;

    eReturnCode =
        OSAL.eMessageAllocate(
            psObj->hQueue,
            (void *)((void *)&psResp),
            OSAL_QUEUE_FLAG_NONBLOCK
                );
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Initialize response structure
        vInitInstance(&psResp->sInstance, psObj->hMutex);
    }

    return psResp;
}

/*****************************************************************************
*
*   SXI_vReAllocateResponse
*
*****************************************************************************/
void SXI_vReAllocateResponse(
    SXIAPI_RX_STRUCT *psResp
        )
{
    // Check input
    if(psResp != NULL)
    {
        vAddInstance(&psResp->sInstance);
    }

    return;
}

/*****************************************************************************
*
*   SXI_vFreeResponse
*
*****************************************************************************/
void SXI_vFreeResponse(
    SXIAPI_RX_STRUCT *psResp
        )
{
    // Check input
    if(psResp != NULL)
    {
        BOOLEAN bRemove;

        bRemove = bRemoveInstance(&psResp->sInstance);
        if(TRUE == bRemove)
        {
            // Release msg
            if (NULL != psResp->sInstance.tvReleaseFunction)
            {
                psResp->sInstance.tvReleaseFunction(&psResp->uData);
            }
            // Free msg.
            OSAL.eMessageFree((void**)psResp);
        }
    }

    return;
}

/*****************************************************************************
*
*   vInitInstance
*
*****************************************************************************/
static void vInitInstance (
    SXI_INSTANCE_STRUCT *psInstance,
    OSAL_OBJECT_HDL hMutex
        )
{
    // Initialize response structure
    psInstance->hMutex = hMutex;
    psInstance->tAllocationCount = 1;
    psInstance->tvReleaseFunction = NULL;

    return;
}

/*****************************************************************************
*
*   vAddInstance
*
*****************************************************************************/
static void vAddInstance (
    SXI_INSTANCE_STRUCT *psInstance
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eSemTake(
        psInstance->hMutex, OSAL_OBJ_TIMEOUT_INFINITE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Increment instance count
        psInstance->tAllocationCount++;

        OSAL.eSemGive(psInstance->hMutex);
    }

    return;
}

/*****************************************************************************
*
*   bRemoveInstance
*
*****************************************************************************/
static BOOLEAN bRemoveInstance (
    SXI_INSTANCE_STRUCT *psInstance
        )
{
    BOOLEAN bRemove = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eSemTake(
        psInstance->hMutex, OSAL_OBJ_TIMEOUT_INFINITE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Check if we have an outstanding instance
        if(psInstance->tAllocationCount > 0)
        {
            // Decrement instance count
            psInstance->tAllocationCount--;

            // If and only if the instance count is zero can this be freed
            if(psInstance->tAllocationCount == 0)
            {
                // No instances remain
                bRemove = TRUE;
            }
        }

        OSAL.eSemGive(psInstance->hMutex);
    }

    return bRemove;
}

/*****************************************************************************
*
*   bDecodeControl
*
*****************************************************************************/
static BOOLEAN bDecodeControl(
    STI_PROTOCOL_HDL hProtocol,
    STI_HDL hSTI,
    SXI_CONTROL_CONNECTION_STRUCT *psControlConnection,
    STI_PROTOCOL_RX_PAYLOAD_STRUCT *psRx,
    void *pvArg
        )
{
    BOOLEAN bResponseHandled = FALSE;
    MODULE_OBJECT hModule = (MODULE_OBJECT)pvArg;

    do
    {
        UN16 un16Opcode;
        UN8 un8TransactionId;
        BOOLEAN bSuccess;
        SXIAPI_OPCODE tMessop;
        SXI_MSG_HDR_STRUCT sSxiHdr;

        n32SXiLogWritePayload(hSTI, psRx->hPayload, "OK");

        // Read payload enough to get the opcode so we can figure
        // out what to do with it.
        bSuccess = SXIAPI_bReadUint16(psRx->hPayload, &un16Opcode);
        if (bSuccess == FALSE)
        {
            // Error! Unable to get opcode
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME": Unable to get opcode.");
            break;
        }

        // Read the transaction ID which is right past the opcode
        bSuccess = SXIAPI_bReadUint8(psRx->hPayload, &un8TransactionId);
        if (bSuccess == FALSE)
        {
            // Error! Unable to get transaction id
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME": Unable to get transaction id.");
            break;
        }

        // Separate the wheat from the chaff
        sSxiHdr.tOpType = (SXIAPI_OPTYPE)((SXIAPI_OPCODE)un16Opcode &
            SXIAPI_MESSOP_TYPE_MASK);
        sSxiHdr.tTransactionID =
            (SXIAPI_TRANID)un8TransactionId;
        tMessop = ((SXIAPI_OPCODE)un16Opcode &
            ~SXIAPI_MESSOP_TYPE_MASK);

        // Based on the messop, process response
        if (tMessop == SXIAPI_MESSOP_IND)
        {
            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                "Control: OpType = %#04.4x, tId=%u (IND - %#04.4x)\n",
                sSxiHdr.tOpType, sSxiHdr.tTransactionID, tMessop);

            // Allocate an rx data structure, used to decode the IND
            psControlConnection->psSxiInd =
                psAllocateResponse(psControlConnection->psRespQueue);
            if(psControlConnection->psSxiInd == NULL)
            {
                // Error! Unable to get response
                n32SXiLogWriteMsg(hSTI, (SXIAPI_OPCODE)un16Opcode,
                    sSxiHdr.tTransactionID, "DROPPED");
                break;
            }

            // Populate received SXi header
            psControlConnection->psSxiInd->sHdr = sSxiHdr;

            bSuccess = bDecodeInd(
                psRx->hPayload, psControlConnection->psSxiInd, hSTI);
            if(bSuccess == TRUE)
            {
                // Re-allocate for confirm
                SXI_vReAllocateResponse(psControlConnection->psSxiInd);

                // Send response to the MODULE
                bResponseHandled = RADIO_bPostModuleEvent(
                    hModule,
                    (SMS_EVENT_TYPE_ENUM)RADIO_EVENT_SXI_IND,
                    SMS_EVENT_OPTION_NONBLOCK,
                    psControlConnection->psSxiInd
                        );
                if(bResponseHandled == TRUE)
                {
                    // Send CFM for all IND messages received properly.
                    // If the CFM cannot be sent, no worries they will
                    // (well maybe will) send it again.
                    vSendCfm(hProtocol, hSTI,
                        psControlConnection->psSxiInd);
                }
                else
                {
                    SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                        "bDecodeControl: Control Ind Payload post failed.");

                    n32SXiLogWriteMsg(hSTI, (SXIAPI_OPCODE)un16Opcode,
                        sSxiHdr.tTransactionID, "UNABLE_TO_POST");

                    // Free allocated response
                    SXI_vFreeResponse(psControlConnection->psSxiInd);
                    psControlConnection->psSxiInd = (SXIAPI_RX_STRUCT*)NULL;
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                    "bDecodeControl: Control Ind Payload decode failed.");

                n32SXiLogWriteMsg(hSTI, (SXIAPI_OPCODE)un16Opcode,
                    sSxiHdr.tTransactionID, "UNABLE_TO_DECODE");
            }

            // Free allocated response
            SXI_vFreeResponse(psControlConnection->psSxiInd);
            psControlConnection->psSxiInd = (SXIAPI_RX_STRUCT*)NULL;
        }
        else if (tMessop == SXIAPI_MESSOP_RESP)
        {
            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                "Control: OpType = %#04.4x, tId=%u (RSP - %#04.4x)\n",
                sSxiHdr.tOpType, sSxiHdr.tTransactionID, tMessop);

            // Use local rx data structure, used to decode the RESP

            // Populate received SXi header
            psControlConnection->sSxiResp.sHdr = sSxiHdr;

            bResponseHandled = bDecodeResp(psRx->hPayload,
                &psControlConnection->sSxiResp);
            if(bResponseHandled == TRUE)
            {
                vSignalResponse(psControlConnection->psReTx,
                    &psControlConnection->sSxiResp.sHdr,
                    psControlConnection->sSxiResp.uData.uResp.eStatusCode);
            }
            else
            {
                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                    "bDecodeControl: Control Ind Payload handle failed.");

                n32SXiLogWriteMsg(hSTI, (SXIAPI_OPCODE)un16Opcode,
                    sSxiHdr.tTransactionID, "UNHANDLED");
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                "Opcode = %#04.4x, tId=%u (UNKNOWN - %#04.4x)\n",
                sSxiHdr.tOpType, sSxiHdr.tTransactionID, tMessop);
        }

    } while(FALSE);

    return bResponseHandled;
}

/*****************************************************************************
*
*   SXI_bDecodeDataPacket
*
*****************************************************************************/
BOOLEAN SXI_bDecodeDataPacket(
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_DATA_PACKET_TYPE_ENUM *peType,
    SXIAPI_DMI *ptDMI
        )
{
    BOOLEAN bSuccess;
    UN8 un8Data;

    /////////////////////////////
    // Decode SXI Data Packet  //
    /////////////////////////////

    do
    {
        // Check input
        if(peType == NULL || ptDMI == NULL)
        {
            break;
        }

        // Read packet type
        bSuccess = SXIAPI_bReadUint8(hPayload, &un8Data);

        if (bSuccess == FALSE)
        {
            break;
        }

        *peType = (SXIAPI_DATA_PACKET_TYPE_ENUM)un8Data;

        // Read DMI
        bSuccess = SXIAPI_bReadUint16(hPayload, ptDMI);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Skip past payload length
        OSAL.tBufferSeekHead(hPayload, sizeof(UN16));

        // Now the payload just has the data packet
        // so we are all good
        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   SXI_n16CompareDataDest
*
*****************************************************************************/
N16 SXI_n16CompareDataDest (
    void *pvObj1,
    void *pvObj2
        )
{
    SXI_DATA_PAYLOAD_DEST_STRUCT *psDest1 =
        (SXI_DATA_PAYLOAD_DEST_STRUCT *)pvObj1;
    SXI_DATA_PAYLOAD_DEST_STRUCT *psDest2 =
        (SXI_DATA_PAYLOAD_DEST_STRUCT *)pvObj2;

    if (((SXI_DATA_PAYLOAD_DEST_STRUCT *)NULL == psDest1) ||
        (SXI_DATA_PAYLOAD_DEST_STRUCT *)NULL == psDest2)
    {
        return N16_MIN;
    }

    return COMPARE(psDest1->tDMI, psDest2->tDMI);
}

/*****************************************************************************
*
*   SXI_vFreeDataDest
*
*****************************************************************************/
void SXI_vFreeDataDest (
    void *pvObj
        )
{
    BOOLEAN bValid;
    SXI_DATA_PAYLOAD_DEST_STRUCT *psDest =
        (SXI_DATA_PAYLOAD_DEST_STRUCT *)pvObj;

    bValid = SMSO_bValid((SMS_OBJECT)psDest);
    if ((TRUE == bValid) &&
        (OSAL_INVALID_OBJECT_HDL != psDest->hDestPools))
    {
        OSAL.eLinkedListRemoveAll(psDest->hDestPools, NULL);
        OSAL.eLinkedListDelete(psDest->hDestPools);
        psDest->hDestPools = OSAL_INVALID_OBJECT_HDL;

        SMSO_vDestroy((SMS_OBJECT)psDest);
    }

    return;
}

/*****************************************************************************
*
*   hCreateResponseQueue
*
*****************************************************************************/
static SXI_RESPONSE_QUEUE_STRUCT *psCreateResponseQueue(
    char const *pacConnectionName,
    UN32 un32Size
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SXI_RESPONSE_QUEUE_STRUCT *psObj;

    // Create memory for this object
    psObj = (SXI_RESPONSE_QUEUE_STRUCT *)
        OSAL.pvMemoryAllocate("SXI:ResponseQueue",
            sizeof(SXI_RESPONSE_QUEUE_STRUCT), TRUE);
    if (psObj != NULL)
    {
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Construct a unique name for this queue object
        snprintf(&acName[0], sizeof(acName),
            "SXI:%s:Response", pacConnectionName);

        // Create response queue w/message
        eReturnCode =
            OSAL.eQueueCreate(
                &psObj->hQueue,
                &acName[0],
                un32Size,
                sizeof(SXIAPI_RX_STRUCT),
                OSAL_QUEUE_OPTION_FIXED_SIZE
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME": Unable to create SXI Response Queue.");

            vDestroyResponseQueue(psObj);
            return NULL;
        }

        // Construct a unique name for this object
        snprintf(&acName[0], sizeof(acName),
            "SXI:%s:Mutex", pacConnectionName);
        // Create response mutex
        eReturnCode =
            OSAL.eSemCreate(
                &psObj->hMutex,
                &acName[0],
                1,
                1,
                OSAL_SEM_OPTION_NONE
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME": Unable to create SXI Response Queue Mutex.");

            vDestroyResponseQueue(psObj);
            return NULL;
        }
    }

    return psObj;
}

/*****************************************************************************
*
*   vDestroyResponseQueue
*
*****************************************************************************/
static void vDestroyResponseQueue(
    SXI_RESPONSE_QUEUE_STRUCT *psObj
        )
{
    if(psObj->hQueue != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eQueueDelete(psObj->hQueue);
        psObj->hQueue = OSAL_INVALID_OBJECT_HDL;
    }

    if(psObj->hMutex != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eSemDelete(psObj->hMutex);
        psObj->hMutex = OSAL_INVALID_OBJECT_HDL;
    }

    OSAL.vMemoryFree(psObj);

    return;
}

/*****************************************************************************
*
*   psCreateReTx
*
*****************************************************************************/
static SXI_RETX_STRUCT *psCreateReTx ( char const *pacConnectionName )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SXI_RETX_STRUCT *psReTx;

    // Create memory for this object
    psReTx = (SXI_RETX_STRUCT *)OSAL.pvMemoryAllocate("SXI:ReTx",
        sizeof(SXI_RETX_STRUCT), TRUE);
    if (psReTx != NULL)
    {
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Invalidate STI handle
        psReTx->hSTI = STI_INVALID_HDL;

        // Construct a unique name for this object
        snprintf( &acName[0], sizeof(acName),
            "SXI:%s:Synch", pacConnectionName);

        // Create synchronization mutex
        eReturnCode =
            OSAL.eSemCreate(
                &psReTx->hMutex,
                &acName[0],
                1,
                1,
                OSAL_SEM_OPTION_NONE
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME": Unable to create %s synch mutex. %s",
                OSAL.pacGetReturnCodeName(eReturnCode), pacConnectionName);

            vDestroyReTx(psReTx);
            return NULL;
        }

        // Construct a unique name for this object
        snprintf( &acName[0], sizeof(acName),
            "SXI:%s:Response", pacConnectionName);

        // Create timeout/response mutex
        eReturnCode =
            OSAL.eSemCreate(
                &psReTx->hResponse,
                &acName[0],
                0,
                1,
                OSAL_SEM_OPTION_NONE
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME": Unable to create %s resp mutex. %s",
                OSAL.pacGetReturnCodeName(eReturnCode), pacConnectionName);

            vDestroyReTx(psReTx);
            return NULL;
        }

        // Construct a unique name for this re-tx timer
        snprintf( &acName[0], sizeof(acName), "SXI:%s:ReTxTimer", pacConnectionName);

        // Create a re-tx timer
        eReturnCode =
            OSAL.eTimerCreate(
                &psReTx->hTimer,
                &acName[0],
                vReTxTimerCallback,
                (void *)&psReTx->hSTI
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME": Unable to create %s re-tx timer. %s",
                OSAL.pacGetReturnCodeName(eReturnCode), pacConnectionName);

            vDestroyReTx(psReTx);
            return NULL;
        }
    }

    return psReTx;
}

/*****************************************************************************
*
*   vDestroyReTx
*
*****************************************************************************/
static void vDestroyReTx (
    SXI_RETX_STRUCT *psReTx
        )
{
    // Destroy re-tx timer if one exists
    if(psReTx->hTimer != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode =
            OSAL.eTimerDelete(psReTx->hTimer);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psReTx->hTimer = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy response mutex if it exists
    if(psReTx->hResponse != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eSemDelete(psReTx->hResponse);
        psReTx->hResponse = OSAL_INVALID_OBJECT_HDL;
    }

    // Destroy synchro mutex if it exists
    if(psReTx->hMutex != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eSemDelete(psReTx->hMutex);
        psReTx->hMutex = OSAL_INVALID_OBJECT_HDL;
    }

    OSAL.vMemoryFree(psReTx);

    return;
}

/*****************************************************************************
*
*   bDecodeInd
*
*****************************************************************************/
static BOOLEAN bDecodeInd(
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_RX_STRUCT *psResp,
    STI_HDL hControlCxn
        )
{
    BOOLEAN bSuccess = TRUE;
    UN8 un8IndCode;
    SXIAPI_INDCODE tIndCode = SXIAPI_INDCODE_NOMINAL;

    // All indications except EventInd contain an indication code
    if (psResp->sHdr.tOpType != SXIAPI_MESSOP_EVENT)
    {
        bSuccess &= SXIAPI_bReadUint8(hPayload, &un8IndCode);
        tIndCode = (SXIAPI_INDCODE)un8IndCode;
    }

    if (bSuccess == TRUE)
    {
        psResp->uData.sInd.eIndCode =
            (SXIAPI_INDCODE_ENUM)tIndCode;

        // Parse payload based on optype (See RX223)
        // to parse out additional data
        switch(psResp->sHdr.tOpType)
        {
            /** SXI Command & Control Indications **/

            // See RX223 - 6.3
            case SXIAPI_MESSOP_MODULECFG:
            {
                SXIAPI_MODULECFGIND_STRUCT *psModCfg =
                    &psResp->uData.sInd.uData.sModCfg;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                    "ModuleCfgInd\n");

                // Initialize the structure
                OSAL.bMemSet(psModCfg, 0, sizeof(*psModCfg));

                SXIAPI_bReadUint8(hPayload, &psModCfg->un8ModuleTypeIDA);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8ModuleTypeIDB);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8ModuleTypeIDC);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8ModuleHWRevA);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8ModuleHWRevB);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8ModuleHWRevC);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8ModSWRevMajor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8ModSWRevMinor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8ModSWRevInc);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8SXIRevMajor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8SXIRevMinor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8SXIRevInc);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8BBRevMajor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8BBRevMinor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8BBRevInc);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8HDecRevMajor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8HDecRevMinor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8HDecRevInc);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8RFRevMajor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8RFRevMinor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8RFRevInc);
                SXIAPI_bReadUint32(hPayload, &psModCfg->un32Capability);
                SXIAPI_bReadUint16(hPayload, &psModCfg->un16DurationOfBuffer);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8SPLRevMajor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8SPLRevMinor);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8SPLRevInc);

                // SXi v3.0 Features
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8MaxSmartFavorites);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8MaxTuneMix);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8MaxSportsFlash);
                SXIAPI_bReadUint8(hPayload, &psModCfg->un8MaxTWNow);
            }
            break;

            // See SX-9845-0097 - 7.1
            case SXIAPI_MESSOP_SUBSTATUS:
            {
                SXIAPI_SUBSTATUSIND_STRUCT *psSubStatus =
                    &psResp->uData.sInd.uData.sSubStatus;
                UN32 un32IVSM;
                UN8 un8Data, un8Day, un8Month, un8Year, un8Reserved, un8ESS;
                struct tm sStruct;

                SXIAPI_bReadString(
                    hPayload,
                    &psSubStatus->acRadioID[0],
                    sizeof(psSubStatus->acRadioID));
                psSubStatus->acRadioID[SXIAPI_MAX_RADIO_ID_LENGTH] = '\0';

                SXIAPI_bReadUint8(hPayload, &un8Data);
                psSubStatus->eStatus = (SXIAPI_SUB_STATUS_ENUM)un8Data;

                SXIAPI_bReadUint8(hPayload, &psSubStatus->un8ReasonCode);

                SXIAPI_bReadUint8(hPayload, &un8Day);
                SXIAPI_bReadUint8(hPayload, &un8Month);
                SXIAPI_bReadUint8(hPayload, &un8Year);

                // default to 00:00:00 Jan 1 1970
                sStruct.tm_sec = 0; // seconds after the minute (from 0)
                sStruct.tm_min = 0; // minutes after the hour (from 0)
                sStruct.tm_hour = 0; // hour of the day (from 0)
                sStruct.tm_mday = 1; // day of the month (from 1)
                sStruct.tm_mon = 0; // month of the year (from 0)
                sStruct.tm_year = 70; // years since 1900 (from 0)
                sStruct.tm_wday = 0; // filled in by OSAL.mktime, days since sunday (from 0)
                sStruct.tm_yday = 0; // filled in by OSAL.mktime, day of the year (from 0)
                sStruct.tm_isdst = 0; // No dst in effect for this time

                if ((un8Day != 255) &&
                    (un8Month != 255) &&
                    (un8Year != 255))
                {
                    // we got a valid suspend date.
                    sStruct.tm_mday = un8Day; // day of the month (from 1)
                    sStruct.tm_mon = un8Month - 1; // month of the year (from 0)
                    sStruct.tm_year = (2000 + un8Year) - 1900; // years since 1900 (from 0)
                }

                // compute the wday and yday
                psSubStatus->un32UTCSuspendTime =
                    (UN32)OSAL.mktime(&sStruct); // calendar time

                SXIAPI_bReadString(
                    hPayload,
                    &psSubStatus->acReasonText[0],
                    sizeof(psSubStatus->acReasonText));
                psSubStatus->acReasonText[SXIAPI_MAX_REASON_TEXT_LENGTH] = '\0';

                SXIAPI_bReadString(
                    hPayload,
                    &psSubStatus->acPhoneNumber[0],
                    sizeof(psSubStatus->acPhoneNumber));
                psSubStatus->acPhoneNumber[SXIAPI_MAX_PHONE_NUMBER_LENGTH] = '\0';

                /* Read reserved field */
                SXIAPI_bReadUint8(hPayload, &un8Reserved);

                /* Read IVSM field since it's not support it's equal to skip */
                SXIAPI_bReadUint32(hPayload, &un32IVSM);

                /* Read Extended Sub-Status field */
                SXIAPI_bReadUint8(hPayload, &un8ESS);

                if ((un8ESS & SXIAPI_SUB_STATUS_EXT_AUDIO_MASK) == SXIAPI_SUB_STATUS_EXT_AUDIO_MASK)
                {
                    psSubStatus->bIsAudioSubscribed = TRUE;
                }
                else
                {
                    psSubStatus->bIsAudioSubscribed = FALSE;
                }
            }
            break;

            case SXIAPI_MESSOP_PWRMODE:
            {
                // Nothing but an IndCode for now.
            }
            break;

            // See RX223 - 7.3
            case SXIAPI_MESSOP_TIME:
            {
                SXIAPI_TIMEIND_STRUCT *psTime =
                    &psResp->uData.sInd.uData.sTime;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                    "TimeInd\n");

                // Extract current uptime (timestamp)
                // The purpose is to time stamp when we received
                // this message and compensate for any offset
                // induced by the time we actually handle this.
                OSAL.vTimeUp(&psTime->un32Seconds, &psTime->un16Msecs);

                // Note: If enabled, the message is initially sent after the
                // Module is synchronized with the SiriusXM network time.
                // Thereafter, the message is sent on minute boundaries
                // (whenever the SiriusXM network time transitions from 59
                // seconds to 00 seconds).  The initial TimeInd may not be
                // aligned to '0 seconds' but subsequent TimeInds will be
                // aligned to '0 seconds'. This may result in two TimeInds
                // within the first minute after enabling the time monitor.

                // Extract satellite info
                bSuccess = SXIAPI_bReadUint8(hPayload, &psTime->un8Minute);
                bSuccess &= SXIAPI_bReadUint8(hPayload, &psTime->un8Hour);
                bSuccess &= SXIAPI_bReadUint8(hPayload, &psTime->un8Day);
                bSuccess &= SXIAPI_bReadUint8(hPayload, &psTime->un8Month);
                bSuccess &= SXIAPI_bReadUint8(hPayload, &psTime->un8Year);

            }
            break;

            // See SX-9845-0097 - 7.11
            case SXIAPI_MESSOP_EVENT:
            {
                SXIAPI_EVENTIND_STRUCT *psEvent =
                    &psResp->uData.sInd.uData.sEvent;
                UN8 un8Data, un8Idx;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                    "EventInd\n");

                // read the event code
                SXIAPI_bReadUint8(hPayload, &un8Data);
                if (un8Data < SXIAPI_EVENT_CODE_INVALID)
                {
                    psEvent->eEventCode = (SXIAPI_EVENT_CODE_ENUM)un8Data;
                }
                else
                {
                    psEvent->eEventCode = SXIAPI_EVENT_CODE_INVALID;
                }

                // read the event data
                for (un8Idx = 0; un8Idx < SXIAPI_EVENT_IND_DATA_MAX_SIZE; un8Idx++)
                {
                    SXIAPI_bReadUint16(hPayload,
                                       &psEvent->aun16EventData[un8Idx]);
                }
            }
            break;

            // See RX223 -
            case SXIAPI_MESSOP_STATUS:
            {
                SXIAPI_STATUSIND_STRUCT *psStatus =
                    &psResp->uData.sInd.uData.sStatus;
                UN8 un8Data;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                    "StatusInd\n");

                SXIAPI_bReadUint8(hPayload, &un8Data);
                psStatus->eStatusItem =
                    (SXIAPI_STATUS_ITEM_ENUM)un8Data;

                // By default don't call anything 'engineering data'
                psStatus->bEngineeringData = FALSE;

                switch (psStatus->eStatusItem)
                {
                    case SXIAPI_STATUS_ITEM_SIGNAL:
                    {
                        SXIAPI_bReadUint8(hPayload, &un8Data);
                        psStatus->uData.sSignal.eSignal =
                            (SXIAPI_SIGNAL_ENUM)un8Data;
                        SXIAPI_bReadUint8(hPayload, &un8Data);
                        psStatus->uData.sSignal.eAntenna =
                            (SXIAPI_ANTENNA_ENUM)un8Data;
                    }
                    break;

                    case SXIAPI_STATUS_ITEM_ANTENNA_AIMING:
                    {
                        SXIAPI_bReadUint8(hPayload,
                            &psStatus->uData.sAntenna.un8SatPercentage);
                        SXIAPI_bReadUint8(hPayload,
                            &psStatus->uData.sAntenna.un8TerPercentage);
                    }
                    break;

                    case SXIAPI_STATUS_ITEM_AUDIO_DECODER_BITRATE:
                    {
                        SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                            "AUDIO DECODER BITRATE\n");
                        SXIAPI_bReadUint8(hPayload,
                            &psStatus->uData.un8AudioDecoderBitrate);
                        // This is engineering data
                        psStatus->bEngineeringData = TRUE;
                    }
                    break;

                    case SXIAPI_STATUS_ITEM_SIGNAL_QUALITY:
                    {
                        SXIAPI_DETAILED_SIGNAL_QUALITY_STRUCT
                                *psDetailedSignalQuality =
                                    &psStatus->uData.sDetailedSignalQuality;

                        SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                            "SIGNAL QUALITY\n");

                        SXIAPI_bReadUint8(hPayload,
                                &psDetailedSignalQuality->un8SignalStrength);
                        SXIAPI_bReadUint8(hPayload,
                                &psDetailedSignalQuality->un8TunerStatus);
                        SXIAPI_bReadUint8(hPayload,
                                &psDetailedSignalQuality->un8ENSALockStatus);
                        SXIAPI_bReadUint8(hPayload,
                                &psDetailedSignalQuality->un8ENSBLockStatus);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedSignalQuality->un16BERS1);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedSignalQuality->un16BERS2);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedSignalQuality->un16BERT);
                        SXIAPI_bReadUint8(hPayload,
                                &psDetailedSignalQuality->un8CNS1A);
                        SXIAPI_bReadUint8(hPayload,
                                &psDetailedSignalQuality->un8CNS1B);
                        SXIAPI_bReadUint8(hPayload,
                                &psDetailedSignalQuality->un8CNS2A);
                        SXIAPI_bReadUint8(hPayload,
                                &psDetailedSignalQuality->un8CNS2B);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedSignalQuality->un16RSErrsWords);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedSignalQuality->un16RSErrsSatSymb);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedSignalQuality->un16RRSErrsTerrSymb);
                        SXIAPI_bReadSint16(hPayload,
                                &psDetailedSignalQuality->n16TunerCarrierFreqOffset);
                        SXIAPI_bReadSint16(hPayload,
                                &psDetailedSignalQuality->n16RSSI);
                        // This is engineering data
                        psStatus->bEngineeringData = TRUE;
                    }
                    break;

                    case SXIAPI_STATUS_ITEM_OVERLAY_SIGNAL_QUALITY:
                    {
                        SXIAPI_DETAILED_OVERLAY_SIGNAL_QUALITY_STRUCT
                                *psDetailedOverlaySignalQuality =
                                    &psStatus->uData.sDetailedOverlaySignalQuality;

                        SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                            "OVERLAY SIGNAL QUALITY\n");

                        SXIAPI_bReadUint8(hPayload,
                                &psDetailedOverlaySignalQuality->un8ReceiverState);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->un16OberS1A);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->un16OberS2A);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->un16OberTA);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->un16OberS1B);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->un16OberS2B);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->un16OberTB);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->
                                    un16TurboWordErrorRate0A);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->
                                    un16TurboWordErrorRate1A);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->
                                    un16TurboWordErrorRate2A);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->
                                    un16TurboWordErrorRate0B);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->
                                    un16TurboWordErrorRate1B);
                        SXIAPI_bReadUint16(hPayload,
                                &psDetailedOverlaySignalQuality->
                                    un16TurboWordErrorRate2B);
                        // This is engineering data
                        psStatus->bEngineeringData = TRUE;
                    }
                    break;

                    case SXIAPI_STATUS_ITEM_LINK_INFORMATION:
                    {
                        SXIAPI_LINK_INFORMATION_STRUCT *psLinkStatus =
                            &psStatus->uData.sLinkStatus;

                        SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                            "LINK STATUS\n");

                        SXIAPI_bReadUint32(hPayload,
                                &psLinkStatus->un32NumberConfirmTimeouts);
                        SXIAPI_bReadUint32(hPayload,
                                &psLinkStatus->un32NumberDataPacketsDropped);
                        SXIAPI_bReadUint32(hPayload,
                                &psLinkStatus->un32NumberDataPacketsTx);
                        SXIAPI_bReadUint32(hPayload,
                                &psLinkStatus->un32NumberAudioPacketsDropped);
                        SXIAPI_bReadUint32(hPayload,
                                &psLinkStatus->un32NumberAudioPacketsTx);
                        // This is engineering data
                        psStatus->bEngineeringData = TRUE;
                    }
                    break;

                    case SXIAPI_STATUS_ITEM_SCAN_ITEMS_AVAILABLE:
                    {
                        SXIAPI_SCAN_ITEMS_AVAILABLE_STRUCT *psScanItems =
                            &psStatus->uData.sScanItems;

                        SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                            "SCAN ITEMS AVAILABLE");

                        // Per updated SXI Messages Spec v2.2.0:
                        // For MonitorID = 0x12, the Module should always set Byte0 = 0 in the StatusInd.
                        // This change will go into a module release following v9.12
                        // CCB determined the parameter indicating the source of the Scan criteria
                        // (Default or Host provided) was not needed and should be changed to Reserved.
                        // This change is also documented in SXI bugzilla_bug7967
                        SXIAPI_bReadUint8(hPayload,
                            &psScanItems->un8ScanCriteriaSource);
                        SXIAPI_bReadUint16(hPayload,
                            &psScanItems->un16ScannableUnplayedTrackCnt);
                        SXIAPI_bReadUint16(hPayload,
                            &psScanItems->un16ScannableUnscannedTrackCnt);
                    }
                    break;

                    case SXIAPI_STATUS_ITEM_AUDIO_PRESENCE:
                    {
                        UN8 un8AudioPresence = 0;

                        SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                            "AUDIO PRESENCE");

                        // Per SXI Messages Spec v3.4.2:
                        // For MonitorID = 0x13, the Module should always send
                        // Audio Status in Byte0
                        SXIAPI_bReadUint8(hPayload,
                            &un8AudioPresence);

                        psStatus->uData.eAudioPresence = 
                            (SXIAPI_AUDIO_PRESENCE_ENUM)un8AudioPresence;
                    }
                    break;

                    case SXIAPI_STATUS_ITEM_INVALID:
                    default:
                    break;
                }
            }
            break;

            // See RX223 - 7.13
            case SXIAPI_MESSOP_DISPLAY:
            {
                SXIAPI_DISPADVISORYIND_STRUCT *psDisplay =
                    &psResp->uData.sInd.uData.sDisplay;

                SXIAPI_bReadUint8(hPayload, &psDisplay->bChanInfoValid);

                // if a value of 1 is read, then the channel info is valid
                // so read it in.
                if (psDisplay->bChanInfoValid == TRUE)
                {
                    SXIAPI_bReadUint16(hPayload, &psDisplay->tChanID);
                    bDecodeBDCastNames(hPayload, &psDisplay->sChanNames);
                }

            }
            break;

            // See RX223 - 8.3
            case SXIAPI_MESSOP_CHANBROWSE:
            case SXIAPI_MESSOP_CHANSELECT:
            {
                SXIAPI_CHANBROWSESELECTIND_STRUCT *psChanBrowseSel =
                    &psResp->uData.sInd.uData.sChanBrowseSelect;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                    "ChanBrowseInd/ChanSelectInd\n");

                // These two fields are always present, regardless of IndCode
                SXIAPI_bReadUint16(hPayload, &psChanBrowseSel->sChannel.tChanID);
                SXIAPI_bReadUint16(hPayload, &psChanBrowseSel->sChannel.tSID);
                SXIAPI_bReadUint8(hPayload, &psChanBrowseSel->sChannel.tCatID);
                SXIAPI_bReadUint8(hPayload, &psChanBrowseSel->sChannel.tRefCatID);
                SXIAPI_bReadUint32(hPayload, &psChanBrowseSel->sTrackInfo.tProgID);
                SXIAPI_bReadUint8(hPayload, &psChanBrowseSel->sChannel.tChanAttrib);
                SXIAPI_bReadUint8(hPayload, &psChanBrowseSel->sChannel.un8RecRes);
                bDecodeBDCastNames(hPayload, &psChanBrowseSel->sChannel.sChanNames);
                bDecodeBDCastNames(hPayload, &psChanBrowseSel->sCatNames);
                bDecodeTrackText(hPayload, &psChanBrowseSel->sTrackInfo);
                bDecodeExtChanMetadata(hPayload, &psChanBrowseSel->sChannel.sChanMetaData);
                bDecodeExtTrackMetadata(hPayload, &psChanBrowseSel->sTrackInfo);

                // SXI 3.x feature

                // Pre-initialize the field with invalid value in case SXI 3.x
                // is not supported in module
                psChanBrowseSel->tSecondaryID = SXIAPI_SECONDARY_INVALID_ID;
                SXIAPI_bReadUint16(hPayload, &psChanBrowseSel->tSecondaryID);
            }
            break;

            // See RX223 - 8.7
            case SXIAPI_MESSOP_CATINFO:
            {
                SXIAPI_CATINFOIND_STRUCT *psCatInfo =
                    &psResp->uData.sInd.uData.sCatInfo;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "CatInfoInd\n");

                SXIAPI_bReadUint8(hPayload, &psCatInfo->tCatID);
                bDecodeBDCastNames(hPayload, &psCatInfo->sCatNames);
            }
            break;

            // See RX223 - 9.1
            case SXIAPI_MESSOP_CHANINFO:
            {
                SXIAPI_CHANINFOIND_STRUCT *psChanInfo =
                    &psResp->uData.sInd.uData.sChanInfo;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "ChanInfoInd\n");

                SXIAPI_bReadUint16(hPayload, &psChanInfo->sChannel.tChanID);
                SXIAPI_bReadUint16(hPayload, &psChanInfo->sChannel.tSID);
                SXIAPI_bReadUint8(hPayload, &psChanInfo->sChannel.tChanAttrib);
                SXIAPI_bReadUint8(hPayload, &psChanInfo->sChannel.un8RecRes);
                SXIAPI_bReadUint8(hPayload, &psChanInfo->sChannel.tCatID);
                bDecodeBDCastNames(hPayload, &psChanInfo->sChannel.sChanNames);
            }
            break;

            // See RX223 - 9.9
            case SXIAPI_MESSOP_METADATA:
            {
                SXIAPI_METADATAIND_STRUCT *psMetaData =
                    &psResp->uData.sInd.uData.sMetaData;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "MetaDataInd\n");

                // We don't get these fields in the metadata indication
                psMetaData->sChannel.sChanNames.acLong[0] = '\0';
                psMetaData->sChannel.sChanNames.acMed[0] = '\0';
                psMetaData->sChannel.sChanNames.acShort[0] = '\0';
                psMetaData->sChannel.tCatID = SXIAPI_CATEGORYID_SERVICENOTASSIGNED;
                psMetaData->sChannel.tChanAttrib = SXIAPI_CHANNELATTR_NONE;

                SXIAPI_bReadUint16(hPayload, &psMetaData->sChannel.tChanID);
                SXIAPI_bReadUint16(hPayload, &psMetaData->sChannel.tSID);
                SXIAPI_bReadUint32(hPayload, &psMetaData->sTrackInfo.tProgID);
                SXIAPI_bReadUint8(hPayload, &psMetaData->sChannel.un8RecRes);

                bDecodeTrackText(hPayload, &psMetaData->sTrackInfo);
                bDecodeExtTrackMetadata(hPayload, &psMetaData->sTrackInfo);
            }
            break;

            case SXIAPI_MESSOP_CHAN_METADATA:
            {
                SXIAPI_CHAN_METADATA_STRUCT *psChanMetaData =
                    &psResp->uData.sInd.uData.sChanMetaData;
                UN16 un16ChanID, un16SID;
                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "ChanMetaDataInd\n");

                SXIAPI_bReadUint16(hPayload, &un16ChanID);
                SXIAPI_bReadUint16(hPayload, &un16SID);
                psChanMetaData->tChanID = un16ChanID;
                psChanMetaData->tSID = un16SID;
                bDecodeExtChanMetadata(hPayload, psChanMetaData);
            }
            break;

            case SXIAPI_MESSOP_GLOBAL_METADATA:
            {
                SXIAPI_EXT_GLOBAL_METADATAIND_STRUCT *psGlobalMetaData =
                    &psResp->uData.sInd.uData.sGlobalMetaData;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "GlobalMetaDataInd\n");

                // Decode incoming GMD into the GMD object
                bSuccess = SXI_GMD_bDecodeExtGlobalMetadata(
                    hControlCxn, hPayload, psGlobalMetaData);

                if (NULL != psGlobalMetaData->pvReleaseArg)
                {
                    // there are external data associated with this message
                    // associate external data release handler
                    psResp->sInstance.tvReleaseFunction =
                        SXI_GMD_vReleaseDecodedExtGlobalMetadata;
                }
            }
            break;

            // See SX-9845-0097 - 9.15
            case SXIAPI_MESSOP_LA_METADATA:
            {
                SXIAPI_METADATAIND_STRUCT *psMetaData =
                    &psResp->uData.sInd.uData.sMetaData;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "LAMetaDataInd\n");

                // We don't get these fields in the la metadata indication
                psMetaData->sChannel.sChanNames.acLong[0] = '\0';
                psMetaData->sChannel.sChanNames.acMed[0] = '\0';
                psMetaData->sChannel.sChanNames.acShort[0] = '\0';
                psMetaData->sChannel.tCatID = SXIAPI_CATEGORYID_SERVICENOTASSIGNED;
                psMetaData->sChannel.tChanAttrib = SXIAPI_CHANNELATTR_NONE;
                psMetaData->sTrackInfo.acArtistBasic[0] = '\0';
                psMetaData->sTrackInfo.acArtistExtd[0] = '\0';
                psMetaData->sTrackInfo.acSongBasic[0] = '\0';
                psMetaData->sTrackInfo.acSongExtd[0] = '\0';

                SXIAPI_bReadUint16(hPayload, &psMetaData->sChannel.tChanID);
                SXIAPI_bReadUint16(hPayload, &psMetaData->sChannel.tSID);
                SXIAPI_bReadUint32(hPayload, &psMetaData->sTrackInfo.tProgID);

                bDecodeExtTrackMetadata(hPayload, &psMetaData->sTrackInfo);
            };
            break;

            // See SX-9845-0097 - 11.3
            case SXIAPI_MESSOP_IRPLAYBACK_CONTROL:
            {
                UN8 un8State;
                SXIAPI_IRPLAYBACKINFO_STRUCT *psPlayback =
                    &psResp->uData.sInd.uData.sIRPlayback;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "IRPlaybackInfoInd\n");

                SXIAPI_bReadUint8(hPayload, &un8State);
                psPlayback->eState = (SXIAPI_PLAYBACK_STATE_ENUM)un8State;
                SXIAPI_bReadUint8(hPayload, &psPlayback->un8Position);
                SXIAPI_bReadUint16(hPayload, &psPlayback->un16PlaybackId);
                SXIAPI_bReadUint16(hPayload, &psPlayback->un16DurationOfTrack);
                SXIAPI_bReadUint16(hPayload,
                    &psPlayback->un16TimeFromStartOfTrack);
                SXIAPI_bReadUint16(hPayload, &psPlayback->un16TracksRemaining);
                SXIAPI_bReadUint16(hPayload, &psPlayback->un16TimeRemaining);
                SXIAPI_bReadUint16(hPayload, &psPlayback->un16TimeBefore);

            };
            break;

            // See SX-9845-0097 - 11.5
            case SXIAPI_MESSOP_IRPLAYBACK_METADATA:
            {
                SXIAPI_IR_PLAYBACK_METADATA_IND_STRUCT *psIRMetadata =
                    &psResp->uData.sInd.uData.sIRPlaybackMetadataInfo;

                UN16 un16SID;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "IRPlaybackMetadata\n");

                SXIAPI_bReadUint8(hPayload, &psIRMetadata->un8Status);
                SXIAPI_bReadUint16(hPayload, &psIRMetadata->un16PlaybackId);
                SXIAPI_bReadUint16(hPayload, &psIRMetadata->tChanID);
                SXIAPI_bReadUint32(hPayload, &psIRMetadata->sTrackInfo.tProgID);
                SXIAPI_bReadUint8(hPayload, &psIRMetadata->sCatInfo.tCatID);

                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sChanNames.acShort[0],
                    sizeof(psIRMetadata->sChanNames.acShort));
                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sChanNames.acMed[0],
                    sizeof(psIRMetadata->sChanNames.acMed));
                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sChanNames.acLong[0],
                    sizeof(psIRMetadata->sChanNames.acLong));

                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sCatInfo.sCatNames.acShort[0],
                    sizeof(psIRMetadata->sCatInfo.sCatNames.acShort));
                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sCatInfo.sCatNames.acMed[0],
                    sizeof(psIRMetadata->sCatInfo.sCatNames.acMed));
                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sCatInfo.sCatNames.acLong[0],
                    sizeof(psIRMetadata->sCatInfo.sCatNames.acLong));

                bDecodeTrackText(hPayload, &psIRMetadata->sTrackInfo);

                bDecodeExtChanMetadata(hPayload, &psIRMetadata->sChanMetaData);
                bDecodeExtTrackMetadata(hPayload, &psIRMetadata->sTrackInfo);
                SXIAPI_bReadUint16(hPayload, &un16SID);

                psIRMetadata->tSID = (SXIAPI_SID)un16SID;
            }
            break;

            // See SX-9845-0097 - 11.7
            case SXIAPI_MESSOP_IRRECORD:
            {
                SXIAPI_IR_RECORD_INFO_IND_STRUCT *psIRInfo =
                    &psResp->uData.sInd.uData.sIRRecordInfo;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "IRRecordInd\n");

                SXIAPI_bReadUint8(hPayload, &psIRInfo->un8RecordState);
                SXIAPI_bReadUint8(hPayload, &psIRInfo->un8BufferUsage);
                SXIAPI_bReadUint16(hPayload, &psIRInfo->un16NewestPlaybackId);
                SXIAPI_bReadUint16(hPayload, &psIRInfo->un16OldestPlaybackId);
                SXIAPI_bReadUint16(hPayload,
                   &psIRInfo->un16DurationOfNewestTrack);
                SXIAPI_bReadUint16(hPayload,
                   &psIRInfo->un16DurationOfOldestTrack);

            };
            break;

            // See SX-9845-0097 - 11.9
            case SXIAPI_MESSOP_IRRECORD_METADATA:
            {
                SXIAPI_IR_RECORD_METADATA_IND_STRUCT *psIRMetadata =
                    &psResp->uData.sInd.uData.sIRRecordMetadataInfo;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "IRRecordMetadata\n");

                SXIAPI_bReadUint8(hPayload,
                    &psIRMetadata->sMetaData.un8Status);
                SXIAPI_bReadUint16(hPayload,
                    &psIRMetadata->sMetaData.un16PlaybackId);
                SXIAPI_bReadUint16(hPayload,
                    &psIRMetadata->un16Duration);
                SXIAPI_bReadUint16(hPayload,
                    &psIRMetadata->sMetaData.tChanID);
                SXIAPI_bReadUint32(hPayload,
                    &psIRMetadata->sMetaData.sTrackInfo.tProgID);
                SXIAPI_bReadUint8(hPayload,
                    &psIRMetadata->sMetaData.sCatInfo.tCatID);

                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sMetaData.sChanNames.acShort[0],
                    sizeof(psIRMetadata->sMetaData.sChanNames.acShort));
                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sMetaData.sChanNames.acMed[0],
                    sizeof(psIRMetadata->sMetaData.sChanNames.acMed));
                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sMetaData.sChanNames.acLong[0],
                    sizeof(psIRMetadata->sMetaData.sChanNames.acLong));

                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sMetaData.sCatInfo.sCatNames.acShort[0],
                    sizeof(psIRMetadata->sMetaData.sCatInfo.sCatNames.acShort));
                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sMetaData.sCatInfo.sCatNames.acMed[0],
                    sizeof(psIRMetadata->sMetaData.sCatInfo.sCatNames.acMed));
                SXIAPI_bReadString(hPayload,
                    &psIRMetadata->sMetaData.sCatInfo.sCatNames.acLong[0],
                    sizeof(psIRMetadata->sMetaData.sCatInfo.sCatNames.acLong));

                bDecodeTrackText(hPayload, &psIRMetadata->sMetaData.sTrackInfo);

                bDecodeExtChanMetadata(hPayload, &psIRMetadata->sMetaData.sChanMetaData);
                bDecodeExtTrackMetadata(hPayload, &psIRMetadata->sMetaData.sTrackInfo);
            }
            break;

            // See RX223 - 14.3
            case SXIAPI_MESSOP_DATASERVICE:
            {
                SXIAPI_DATASERVICESTATUSIND_STRUCT *psDataStatus =
                    &psResp->uData.sInd.uData.sDataStatus;
                UN8 un8Data;
                UN8 un8DMIIdx;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "DataServiceStatusInd\n");

                SXIAPI_bReadUint16(hPayload, &psDataStatus->tDSI);

                SXIAPI_bReadUint8(hPayload, &un8Data);
                psDataStatus->eStatus =
                    (SXIAPI_DATASERVICE_STATUS_ENUM)un8Data;

                SXIAPI_bReadUint8(hPayload, &psDataStatus->un8DMICount);

                for (un8DMIIdx = 0;
                     un8DMIIdx < SXIAPI_DMI_CNT_MAX &&
                     un8DMIIdx < psDataStatus->un8DMICount;
                     un8DMIIdx++)
                {
                    SXIAPI_bReadUint16(hPayload,
                        &psDataStatus->atDMIs[un8DMIIdx]);
                }
            }
            break;

            // See SX-9845-0097 6.15
            case SXIAPI_MESSOP_PACKAGE:
            {
                SXIAPI_PKG_IND_STRUCT *psPkg =
                    &psResp->uData.sInd.uData.sPkg;
                size_t tNumBytesInPayload, tNumBytes;
                UN8 *pun8Index;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4, "PkgInd\n");

                // how many bytes are left in the payload?
                tNumBytesInPayload = OSAL.tBufferGetSize(hPayload);

                // remember that the Trans ID, OpCode and Ind Code were
                // already consumed. But we have to include them in response
                tNumBytes = tNumBytesInPayload +
                    sizeof(SXIAPI_OPCODE) +
                    sizeof(SXIAPI_TRANID) +
                    sizeof(SXIAPI_INDCODE);

                psPkg->pun8PkgInd = OSAL.pvMemoryAllocate("SXI: PkgInd",
                    tNumBytes, TRUE);
                if (psPkg->pun8PkgInd != NULL)
                {
                    SXIAPI_OPCODE tOpCode;
                    size_t tNumBytesWritten;

                    // memory allocated successfully.

                    // point to the start of allocated memory
                    pun8Index = psPkg->pun8PkgInd;

                    // OpCode was already consumed from the payload, so
                    // we'll have to add it back into our copy of the PkgInd.
                    // Copy, update bytes written count, then advance pointer
                    tNumBytes = sizeof(SXIAPI_OPCODE);

                    tOpCode = (SXIAPI_OPCODE)psResp->sHdr.tOpType | SXIAPI_MESSOP_IND;

                    #ifdef OSAL_CPU_LITTLE_ENDIAN
                        // Remember that SXIAPI_bWriteUint16 reversed the
                        // byte order, so we need to do a swap to get
                        // it back to the proper order.
                        tOpCode = (SXIAPI_OPCODE)SWAPUN16(tOpCode);
                    #elif !defined OSAL_CPU_BIG_ENDIAN
                        #error Error! You must define OSAL_CPU_LITTLE_ENDIAN or OSAL_CPU_BIG_ENDIAN
                    #endif /* OSAL_CPU_LITTLE_ENDIAN */

                    OSAL.bMemCpy(pun8Index, (void *)&tOpCode,
                        tNumBytes);
                    tNumBytesWritten = tNumBytes;
                    pun8Index = pun8Index + tNumBytes;

                    // Transaction ID was already consumed from the payload, so
                    // we'll have to add it back into our copy of the PkgInd.
                    // Copy, update bytes written count, then advance pointer
                    tNumBytes = sizeof(SXIAPI_TRANID);
                    OSAL.bMemCpy(
                        pun8Index, (void *)&psResp->sHdr.tTransactionID,
                        tNumBytes);
                    tNumBytesWritten += tNumBytes;
                    pun8Index = pun8Index + tNumBytes;

                    // Ind Code was already consumed from the payload, so
                    // we'll have to add it back into our copy of the PkgInd.
                    // Copy, update bytes written count, then advance pointer
                    tNumBytes = sizeof(SXIAPI_INDCODE);
                    OSAL.bMemCpy(pun8Index, (void *)&tIndCode, tNumBytes);
                    tNumBytesWritten += tNumBytes;
                    pun8Index = pun8Index + tNumBytes;

                    // now we can just copy out from the payload directly
                    tNumBytes = OSAL.tBufferReadHead(hPayload, pun8Index,
                        tNumBytesInPayload);

                    tNumBytesWritten += tNumBytes;

                    // save off the number of bytes written
                    psPkg->tNumBytes = tNumBytesWritten;
                }
            }
            break;

            case SXIAPI_MESSOP_SEEK_MON:
            {
                SXIAPI_SEEKMON_IND_STRUCT *psSeekMon =
                    &psResp->uData.sInd.uData.sSeekMon;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                    "SeekInd\n");

                SXIAPI_bReadUint16(hPayload, &psSeekMon->tChanID);
                SXIAPI_bReadUint16(hPayload, &psSeekMon->tSID);
                SXIAPI_bReadUint8(hPayload, &psSeekMon->tChanAttrib);
                SXIAPI_bReadUint32(hPayload, &psSeekMon->sTrackInfo.tProgID);

                SXIAPI_bReadUint8(hPayload, &psSeekMon->un8SeekMonID);
                SXIAPI_bReadUint16(hPayload, &psSeekMon->tMatchedTMI);

                if (psSeekMon->tMatchedTMI == SXIAPI_TMI_TEAM_BCAST_ID)
                {
                    SXIAPI_bReadUint32(hPayload, &psSeekMon->uMatchedVal.un32Val);
                }

                bDecodeExtTrackMetadata(hPayload, &psSeekMon->sTrackInfo);
            }
            break;

            case SXIAPI_MESSOP_FLASH:
            {
                SXIAPI_FLASH_IND_STRUCT *psFlash =
                    &psResp->uData.sInd.uData.sFlash;

                UN8 un8Data;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                    "FlashInd\n");

                SXIAPI_bReadUint16(hPayload, &psFlash->tChanID);
                SXIAPI_bReadUint16(hPayload, &psFlash->tSID);
                SXIAPI_bReadUint32(hPayload, &psFlash->tProgID);
                SXIAPI_bReadUint8(hPayload, &psFlash->tChanAttrib);

                SXIAPI_bReadUint8(hPayload, &un8Data);
                psFlash->eFlashType = (SXIAPI_FLASH_TYPE_ENUM)un8Data;

                SXIAPI_bReadUint8(hPayload, &un8Data);
                psFlash->eFlashStatus = (SXIAPI_FLASH_STATUS_ENUM)un8Data;

                SXIAPI_bReadUint16(hPayload, &psFlash->tFlashEventID);
                SXIAPI_bReadUint32(hPayload, &psFlash->un32FlashEventData);

                bDecodeBDCastNames(hPayload, &psFlash->sChanNames);
            }
            break;

            case SXIAPI_MESSOP_BULLETIN_MON:
            {
                SXIAPI_BULLETIN_STATUS_IND_STRUCT *psBulletinStatus =
                    &psResp->uData.sInd.uData.sBulletinStatus;
                UN8 un8Data;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                    "BulletinStatusInd\n");

                SXIAPI_bReadUint8(hPayload, &un8Data);
                psBulletinStatus->eType = (SXIAPI_BULLETIN_TYPE_ENUM)un8Data;

                SXIAPI_bReadUint16(hPayload, &psBulletinStatus->tBulletinID);
                SXIAPI_bReadUint16(hPayload, &psBulletinStatus->un16Param1);

                SXIAPI_bReadUint8(hPayload, &un8Data);
                psBulletinStatus->eStatus = (SXIAPI_BULLETIN_STATUS_ENUM)un8Data;
            }
            break;

            // See SX-9845-0097 11.11
            case SXIAPI_MESSOP_CONTENT_BUFFERED:
            {
                SXIAPI_CONTENT_BUFFERED_IND_STRUCT *psCB =
                    &psResp->uData.sInd.uData.sContentBuffered;

                SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                    "ContentBufferedInd\n");

                SXIAPI_bReadUint16(hPayload, &psCB->tChanID);
                SXIAPI_bReadUint16(hPayload, &psCB->tSID);
                SXIAPI_bReadUint8(hPayload, &psCB->un8ProgramIDCnt);

                if (psCB->un8ProgramIDCnt > 0)
                {
                    psCB->ptProgramIDList = (SXIAPI_PROGRAM_ID*)
                        OSAL.pvMemoryAllocate(
                        "ContentBufferedInd:PIDsList",
                        sizeof(*psCB->ptProgramIDList) * psCB->un8ProgramIDCnt,
                        FALSE);
                    if (psCB->ptProgramIDList == NULL)
                    {
                        // We failed - stop here
                        SMSAPI_DEBUG_vPrintErrorFull(
                            gpacThisFile, __LINE__, SXI_LOG_BASE_NAME
                            ": failed to allocate memory for %d PID(s)",
                            (int)psCB->un8ProgramIDCnt);
                    }
                    else
                    {
                        UN8 un8ProgramIDIdx;

                        // Read out list of PIDs
                        for(un8ProgramIDIdx = 0;
                            un8ProgramIDIdx != psCB->un8ProgramIDCnt;
                            ++un8ProgramIDIdx)
                        {
                            SXIAPI_bReadUint32(hPayload,
                                &psCB->ptProgramIDList[un8ProgramIDIdx]);
                        }
                    }
                }
                else
                {
                    psCB->ptProgramIDList = (SXIAPI_PROGRAM_ID *)NULL;
                }
            }
            break;

            default:
            {
                // Do Nothing
            }
            break;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bDecodeResp
*
*****************************************************************************/
static BOOLEAN bDecodeResp(
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_RX_STRUCT *psResp
        )
{
    // Initialize return code
    BOOLEAN bSuccess;
    UN8 un8Data;

    // For now, all responses are just a status code, so decode that
    bSuccess = SXIAPI_bReadUint8(hPayload, &un8Data);
    if (bSuccess == TRUE)
    {
        psResp->uData.uResp.eStatusCode =
            (SXIAPI_STATUSCODE_ENUM)un8Data;

        // Parse payload based on op-type (See RX223)
        // to parse out additional data (nothing for now)
        switch(psResp->sHdr.tOpType)
        {
            /** SXI Command & Control Responses **/

            default:
                // Do Nothing
                break;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bDecodeLink
*
*****************************************************************************/
static BOOLEAN bDecodeLink(
    STI_PROTOCOL_HDL hProtocol,
    STI_HDL hSTI,
    SXI_LINK_CONNECTION_STRUCT *psLinkConnection,
    STI_PROTOCOL_RX_PAYLOAD_STRUCT *psRx
        )
{
    // Initialize return code
    BOOLEAN bSuccess = FALSE;

    BOOLEAN bReadSuccess;
    UN16 un16LinkOpcode;
    SXIAPI_LINK_OPCODE tLinkOpCode;

    if (psRx == NULL)
    {
        return FALSE;
    }

    n32SXiLogWritePayload(hSTI, psRx->hPayload, "OK");

    // Read payload enough to get the link opcode so we can figure
    // out what to do with it.
    bReadSuccess = SXIAPI_bReadUint16(
        psRx->hPayload, &un16LinkOpcode);
    if (bReadSuccess == FALSE)
    {
        // Error! Unable to get link opcode
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SXI_LOG_BASE_NAME": Unable to get link opcode.");
        return FALSE;
    }

    // Link responses have a link-opcode
    tLinkOpCode = (SXIAPI_OPTYPE)un16LinkOpcode;
    psLinkConnection->sRxData.sHdr.tOpType = (SXIAPI_OPTYPE)un16LinkOpcode;

    // Link responses have no transaction id
    psLinkConnection->sRxData.sHdr.tTransactionID = 0;

    // Initialize status code (there are none for link responses)
    psLinkConnection->sRxData.uData.uResp.eStatusCode =
        SXIAPI_STATUSCODE_MSG_RECEIVED;

    SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
        "Link Opcode = %#04.4x (LINK RESP)\n", tLinkOpCode);

    // Parse payload based on link opcode (See RX224)...
    switch(tLinkOpCode)
    {
        /** UART Link Layer Responses **/

        // See RX224 - 3.3
        case SXIAPI_MESSOP_UARTLINKSTARTRESP:
        {
            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                "UART Link Start Resp\n");

            // UART Link Start Status Code
            bSuccess = SXIAPI_bReadUint8(
                psRx->hPayload,
                &psLinkConnection->sRxData.uData.uResp.sUartLinkStartResponse.un8StatusCode
                    );
            if(bSuccess == TRUE)
            {
                vSignalResponse(psLinkConnection->psReTx,
                    &psLinkConnection->sRxData.sHdr,
                        SXIAPI_STATUSCODE_MSG_RECEIVED);
            }
        }
        break;

        case SXIAPI_MESSOP_UARTLINKERR:
        {
            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 1, "UART Link Error\n");

            bSuccess = SXIAPI_bReadUint8(
                psRx->hPayload,
                &psLinkConnection->sRxData.uData.uResp.sUartLinkError.un8ErrorCode
                    );

            // Keep track of the errors we've see
            // But don't pass the message to the application.
            psLinkConnection->un32LinkErrors++;

            if(SXI_LINK_ERROR_UNSUPPORTED_PAYLOAD_TYPE ==
                psLinkConnection->sRxData.uData.uResp.sUartLinkError.un8ErrorCode)
            {
                psLinkConnection->un32UnsupportedPayloadType++;
            }
            else if(SXI_LINK_ERROR_UNKNOWN_OPCODE ==
                psLinkConnection->sRxData.uData.uResp.sUartLinkError.un8ErrorCode)
            {
                psLinkConnection->un32UnknownOpCode++;
            }
            else
            {
                psLinkConnection->un32UnknownError++;
            }

            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                "bDecodeLink: %u "
                "UnsupportedPayloadTypes, %u "
                "UnknownOpCodes, %u UnknownErrors\n",
                psLinkConnection->un32UnsupportedPayloadType,
                psLinkConnection->un32UnknownOpCode,
                psLinkConnection->un32UnknownError);
        }
        break;

        // See RX224 - 3.5
        case SXIAPI_MESSOP_UARTLINKKEEPALIVE:
        {
            // No info with the keep alive
            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                "UART Link Keep Alive\n");

            // Do nothing. Actually we should NEVER get these here
            // because our SXiLL handler will absorb these first.
            psLinkConnection->un32KeepAlivesReceived++;
            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                "bDecodeLink: %u UartLinkKeepAlives\n",
                psLinkConnection->un32KeepAlivesReceived);
        }
        break;

        case SXIAPI_MESSOP_UARTLINKSTATSRESP:
        {
            SXIAPI_UART_LINK_STATS_RESP_DATA_STRUCT *psUartLinkStatsResponse =
                &psLinkConnection->sRxData.uData.uResp.sUartLinkStatsResponse;

            SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
                "UART Link Stats Response\n");

            do
            {
                bSuccess = SXIAPI_bReadUint32(
                    psRx->hPayload, &psUartLinkStatsResponse->un32NumBytesRcvd);
                if(bSuccess == FALSE)
                {
                    break;
                }
                bSuccess = SXIAPI_bReadUint32(
                    psRx->hPayload, &psUartLinkStatsResponse->un32NumBytesXmtd);
                if(bSuccess == FALSE)
                {
                    break;
                }
                bSuccess = SXIAPI_bReadUint32(
                    psRx->hPayload, &psUartLinkStatsResponse->un32NumMsgFramesRcvd);
                if(bSuccess == FALSE)
                {
                    break;
                }
                bSuccess = SXIAPI_bReadUint32(
                    psRx->hPayload, &psUartLinkStatsResponse->un32NumMsgFramesXmtd);
                if(bSuccess == FALSE)
                {
                    break;
                }
                bSuccess = SXIAPI_bReadUint32(
                    psRx->hPayload, &psUartLinkStatsResponse->un32IndicationResends);
                if(bSuccess == FALSE)
                {
                    break;
                }

                vSignalResponse(psLinkConnection->psReTx,
                    &psLinkConnection->sRxData.sHdr,
                        SXIAPI_STATUSCODE_MSG_RECEIVED);

            } while(FALSE);
        }
        break;
        default:
        // DO NOTHING
        break;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bDecodeBDCastNames
*
*****************************************************************************/
static BOOLEAN bDecodeBDCastNames(
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_BDCAST_NAMES_STRUCT *psNames
        )
{
    BOOLEAN bRead = SXIAPI_bReadString(
        hPayload,
        &psNames->acShort[0],
        sizeof(psNames->acShort));
    bRead &= SXIAPI_bReadString(
        hPayload,
        &psNames->acMed[0],
        sizeof(psNames->acMed));
    bRead &= SXIAPI_bReadString(
        hPayload,
        &psNames->acLong[0],
        sizeof(psNames->acLong));

    return bRead;
}

/*****************************************************************************
*
*   bDecodeTrackText
*
*****************************************************************************/
static BOOLEAN bDecodeTrackText (
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_TRACK_INFO_STRUCT *psTrackInfo
        )
{
    BOOLEAN bRead = SXIAPI_bReadString(
        hPayload,
        &psTrackInfo->acArtistBasic[0],
        sizeof(psTrackInfo->acArtistBasic));
    bRead &= SXIAPI_bReadString(
        hPayload,
        &psTrackInfo->acSongBasic[0],
        sizeof(psTrackInfo->acSongBasic));
    bRead &= SXIAPI_bReadString(
        hPayload,
        &psTrackInfo->acArtistExtd[0],
        sizeof(psTrackInfo->acArtistExtd));
    bRead &= SXIAPI_bReadString(
        hPayload,
        &psTrackInfo->acSongExtd[0],
        sizeof(psTrackInfo->acSongExtd));
    bRead &= SXIAPI_bReadString(
        hPayload,
        &psTrackInfo->acContentInfo[0],
        sizeof(psTrackInfo->acContentInfo));

    return bRead;
}

/*****************************************************************************
*
*   bDecodeExtTrackMetadata
*
*****************************************************************************/
static BOOLEAN bDecodeExtTrackMetadata (
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_TRACK_INFO_STRUCT *psTrackInfo
        )
{
    BOOLEAN bWritten;
    UN8 un8Idx;
    UN32 un32TeamIdx, un32NumTeams;
    UN16 un16TMI;

    // initialize our mask to indicate no TMIs were received
    // for each TMI we read we'll set the bit to indicate we got new info
    psTrackInfo->sExtTrackMetaData.un32ValidFieldMask = SXIAPI_VALID_TMI_NONE;

    // Read how many TMIs will follow
    bWritten = SXIAPI_bReadUint8(hPayload,
                   (UN8 *)&psTrackInfo->un8TrackMetaDataCnt);

    for (un8Idx = 0; un8Idx < psTrackInfo->un8TrackMetaDataCnt; un8Idx++)
    {
        // Read the next TMI
        bWritten &= SXIAPI_bReadUint16(hPayload, &un16TMI);

        if(bWritten == FALSE)
        {
            // Payload is empty. Abort!
            break;
        }

        // Read the data according to which TMI we're dealing with
        switch(un16TMI)
        {
            case SXIAPI_TMI_SONG_ID:
            {
                // read song id
                bWritten &= SXIAPI_bReadUint32(hPayload,
                       (UN32 *)&psTrackInfo->sExtTrackMetaData.un32SongId);
                // set the bit to indicate we got the song id
                psTrackInfo->sExtTrackMetaData.un32ValidFieldMask |=
                    SXIAPI_VALID_TMI_SONG_ID;
            }
            break;

            case SXIAPI_TMI_ARTIST_ID:
            {
                // read artist id
                bWritten &= SXIAPI_bReadUint32(hPayload,
                       (UN32 *)&psTrackInfo->sExtTrackMetaData.un32ArtistId);
                // set the bit to indicate we got the artist id
                psTrackInfo->sExtTrackMetaData.un32ValidFieldMask |=
                    SXIAPI_VALID_TMI_ARTIST_ID;
            }
            break;

            case SXIAPI_TMI_ALBUM_NAME:
            {
                // read song id
                bWritten &= SXIAPI_bReadString(hPayload,
                       (char *)&psTrackInfo->sExtTrackMetaData.acAlbumName,
                       sizeof(psTrackInfo->sExtTrackMetaData.acAlbumName));
                // set the bit to indicate we got the song id
                psTrackInfo->sExtTrackMetaData.un32ValidFieldMask |=
                    SXIAPI_VALID_TMI_ALBUM_NAME;
            }
            break;
            case SXIAPI_TMI_LEAGUE_BCAST_ID:
            {
                // read sports league id
                bWritten &= SXIAPI_bReadUint8(hPayload,
                    (UN8 *)&psTrackInfo->sExtTrackMetaData.un8BCastLeagueId);
                // set the bit to indicate we got the league id
                psTrackInfo->sExtTrackMetaData.un32ValidFieldMask |=
                    SXIAPI_VALID_TMI_LEAGUE_BCAST_ID;
            }
            break;

            case SXIAPI_TMI_TEAM_BCAST_ID:
            {
                // Read how many elements (teams) are in the array to follow
                bWritten &= SXIAPI_bReadUint32(hPayload, (UN32 *)&un32NumTeams);

                // Team2 may go uninitialized and cause unnecessary CID allocation /
                // freed without being used. This happens for a single 'team' sport
                // events. To prevent this and keep the current design as is, Team2
                // could be initialized with value of 0. According to the SX-9845-0150,
                // "If the head-to-head game involves a team that is not identified
                // in the Sports Team Table, the Sports Team ID for the unidentified
                // team will be 0".

                psTrackInfo->sExtTrackMetaData.aun32BCastTeamId[0] = 0;
                psTrackInfo->sExtTrackMetaData.aun32BCastTeamId[1] = 0;

                // make sure they aren't trying to send us more than we can handle
                if (un32NumTeams <= SXIAPI_TRACK_METADATA_NUM_TEAMS)
                {
                    for (un32TeamIdx = 0;
                         un32TeamIdx < un32NumTeams;
                         un32TeamIdx++)
                    {
                        // get the team
                        bWritten &= SXIAPI_bReadUint32(hPayload,
                            (UN32 *)&psTrackInfo->sExtTrackMetaData.aun32BCastTeamId[un32TeamIdx]);
                        // set the bit to indicate we got the game team id
                        psTrackInfo->sExtTrackMetaData.un32ValidFieldMask |=
                            SXIAPI_VALID_TMI_TEAM_BCAST_ID;
                    }
                }
            }
            break;

            case SXIAPI_TMI_SPORT_BCAST_TYPE:
            {
                // Read sports broadcast type
                bWritten &= SXIAPI_bReadUint8(hPayload,
                    (UN8 *)&psTrackInfo->sExtTrackMetaData.un8SportBCastType);
                // set the bit to indicate we got the broadcast type
                psTrackInfo->sExtTrackMetaData.un32ValidFieldMask |=
                    SXIAPI_VALID_TMI_SPORT_BCAST_TYPE;
            }
            break;

            case SXIAPI_TMI_TRAFFIC_ID:
            {
                UN8 un8TrafficId;
                // read the traffic id into a temp var.
                // we rx it in a un8, but we store it as a un32.
                // so we need to do this to avoid a possible
                // exception being thrown
                bWritten &= SXIAPI_bReadUint8(hPayload, &un8TrafficId);
                // save the read value into our struct
                psTrackInfo->sExtTrackMetaData.un32TrafficId = (UN32)un8TrafficId;
                // set the bit to indicate we got the traffic id
                psTrackInfo->sExtTrackMetaData.un32ValidFieldMask |=
                    SXIAPI_VALID_TMI_TRAFFIC_ID;
            }
            break;

            case SXIAPI_TMI_ITUNES_SONG_ID:
            {
                // read itunes song id
                bWritten &= SXIAPI_bReadUint32(hPayload,
                    (UN32 *)&psTrackInfo->sExtTrackMetaData.un32ITunesSongId);
                // set the bit to indicate we got the song id
                psTrackInfo->sExtTrackMetaData.un32ValidFieldMask |=
                    SXIAPI_VALID_TMI_ITUNES_SONG_ID;
            }
            break;

            default:
            {
                // don't do anything
            }
            break;
        }
    }

    return bWritten;
}

/*****************************************************************************
*
*   bDecodeExtChanMetadata
*
*****************************************************************************/
static BOOLEAN bDecodeExtChanMetadata (
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_CHAN_METADATA_STRUCT *psChanInfo
        )
{
    BOOLEAN bWritten;
    UN8 un8Idx;
    UN32 un32ChanIdx;
    UN16 un16CMI;

    // initialize our mask to indicate no CMIs were received
    // for each CMI we read we'll set the bit to indicate we got new info
    psChanInfo->sExtChanMetaData.un32ValidFieldMask = SXIAPI_VALID_CMI_NONE;
    psChanInfo->un8ChanMetadataCnt = 0;

    // Read the count of CMIs that will follow
    bWritten = SXIAPI_bReadUint8(hPayload,
                   &psChanInfo->un8ChanMetadataCnt);

    for (un8Idx = 0; un8Idx < psChanInfo->un8ChanMetadataCnt; un8Idx++)
    {
        // Read the next CMI
        bWritten &= SXIAPI_bReadUint16(hPayload, &un16CMI);

        if(bWritten == FALSE)
        {
            // Payload is empty. Aborting!
            break;
        }

        switch(un16CMI)
        {
            // Read the data according to which CMI we're dealing with.
            case SXIAPI_CMI_SHORT_CHAN_DESC:
            {
                // we got the short channel description
                bWritten &= SXIAPI_bReadString(
                    hPayload,
                    &psChanInfo->sExtChanMetaData.acShortDesc[0],
                    sizeof(psChanInfo->sExtChanMetaData.acShortDesc));
                psChanInfo->sExtChanMetaData.acShortDesc[
                    sizeof(psChanInfo->sExtChanMetaData.acShortDesc) - 1] = '\0';

                if( TRUE == bWritten )
                {
                    // set the bit to indicate we got the short channel
                    // description
                    psChanInfo->sExtChanMetaData.un32ValidFieldMask |=
                        SXIAPI_VALID_CMI_SHORT_CHAN_DESC;
                }
            }
            break;

            case SXIAPI_CMI_LONG_CHAN_DESC:
            {
                // we got the long channel description
                bWritten &= SXIAPI_bReadString(
                    hPayload,
                    &psChanInfo->sExtChanMetaData.acLongDesc[0],
                    sizeof(psChanInfo->sExtChanMetaData.acLongDesc));
                psChanInfo->sExtChanMetaData.acLongDesc[
                    sizeof(psChanInfo->sExtChanMetaData.acLongDesc) - 1] = '\0';

                if( TRUE == bWritten )
                {
                    // set the bit to indicate we got the long channel
                    // description
                    psChanInfo->sExtChanMetaData.un32ValidFieldMask |=
                        SXIAPI_VALID_CMI_LONG_CHAN_DESC;
                }
            }
            break;

            case SXIAPI_CMI_RELATED_CHAN_LIST:
            {
                // we got the related channel list
                psChanInfo->sExtChanMetaData.un32NumChans = 0;

                // find out how many channels we need to read
                bWritten &= SXIAPI_bReadUint32(hPayload,
                                &psChanInfo->sExtChanMetaData.un32NumChans);

                if (psChanInfo->sExtChanMetaData.un32NumChans <= sizeof(psChanInfo->sExtChanMetaData.aun16RelatedChan))
                {
                    for (un32ChanIdx = 0;
                         ((un32ChanIdx < psChanInfo->sExtChanMetaData.un32NumChans) && (un32ChanIdx < SXIAPI_MAX_RELATED_CHAN));
                         un32ChanIdx++)
                    {
                        // get the channels
                        bWritten &= SXIAPI_bReadUint16(hPayload,
                              (UN16 *)&psChanInfo->sExtChanMetaData.aun16RelatedChan[un32ChanIdx]);
                    }
                }

                if( TRUE == bWritten )
                {
                    // set the bit to indicate we got the similar channel list
                    psChanInfo->sExtChanMetaData.un32ValidFieldMask |=
                        SXIAPI_VALID_CMI_RELATED_CHAN_LIST;
                }
            }
            break;

            case SXIAPI_CMI_CHAN_LIST_ORDER:
            {
                // we got the List Order
                bWritten &= SXIAPI_bReadUint16(
                    hPayload,
                    &psChanInfo->sExtChanMetaData.un16ACO);

                if( TRUE == bWritten )
                {
                    // set the bit to indicate we got the list order
                    psChanInfo->sExtChanMetaData.un32ValidFieldMask |=
                        SXIAPI_VALID_CMI_CHAN_LIST_ORDER;
                }
            }
            break;

            case SXIAPI_CMI_CHAN_CONTENT_TYPE:
            {
                // we got the Content Type
                bWritten &= SXIAPI_bReadUint8(
                    hPayload,
                    &psChanInfo->sExtChanMetaData.un8ContentType);

                if( TRUE == bWritten )
                {
                    // set the bit to indicate we got the Cotent Type
                    psChanInfo->sExtChanMetaData.un32ValidFieldMask |=
                        SXIAPI_VALID_CMI_CHAN_CONTENT_TYPE;
                }
            }
            break;

            case SXIAPI_CMI_IR_NAVIGATION_CLASS:
            {
                // we got the IR Navigation Class
                bWritten &= SXIAPI_bReadUint8(
                    hPayload,
                    &psChanInfo->sExtChanMetaData.un8IRNavigationClass);

                if( TRUE == bWritten )
                {
                    // set the bit to indicate we got the IR Navigation Class
                    psChanInfo->sExtChanMetaData.un32ValidFieldMask |=
                        SXIAPI_VALID_CMI_IR_NAVIGATION_CLASS;
                }
            }
            break;

            case SXIAPI_CMI_PLAY_ON_SELECT:
            {
                // we got the play on select
                bWritten &= SXIAPI_bReadUint8(
                    hPayload,
                    &psChanInfo->sExtChanMetaData.un8PlayOnSelect);

                if( TRUE == bWritten )
                {
                    // set the bit to indicate we got the play on select
                    psChanInfo->sExtChanMetaData.un32ValidFieldMask |=
                        SXIAPI_VALID_CMI_PLAY_ON_SELECT;
                }
            }
            break;

            default:
            {
                // don't do anything
            }
            break;
        }
    }

    return bWritten;
}

/*****************************************************************************
*
*   vSendCfm
*
*****************************************************************************/
static void vSendCfm (
    STI_PROTOCOL_HDL hProtocol,
    STI_HDL hSTI,
    SXIAPI_RX_STRUCT const *psResp
        )
{
    OSAL_BUFFER_HDL hPayload;

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        BOOLEAN bWritten;
        UN16 un16OpCode;

        // Populate message opcode
        un16OpCode = (UN16)((psResp->sHdr.tOpType &
            SXIAPI_MESSOP_TYPE_MASK) | SXIAPI_MESSOP_CFM);
        bWritten =
            SXIAPI_bWriteUint16(hPayload, un16OpCode);

        // Populate transaction id
        bWritten &=
            SXIAPI_bWriteUint8(hPayload, (UN8)psResp->sHdr.tTransactionID);

        // Populate message based on op-type
        switch (psResp->sHdr.tOpType)
        {
            case SXIAPI_MESSOP_STATUS:
                bWritten &= SXIAPI_bWriteUint8(
                    hPayload, (UN8)psResp->uData.sInd.uData.sStatus.eStatusItem);
                break;
            case SXIAPI_MESSOP_CATINFO:
                bWritten &= SXIAPI_bWriteUint8(
                    hPayload, (UN8)psResp->uData.sInd.uData.sCatInfo.tCatID);
                break;
            case SXIAPI_MESSOP_CHANINFO:
                bWritten &= SXIAPI_bWriteUint16(
                    hPayload, psResp->uData.sInd.uData.sChanInfo.sChannel.tSID);
                break;
            case SXIAPI_MESSOP_METADATA:
                bWritten &= SXIAPI_bWriteUint16(
                    hPayload, (UN16)psResp->uData.sInd.uData.sMetaData.sChannel.tSID);
                break;
            case SXIAPI_MESSOP_LA_METADATA:
                bWritten &= SXIAPI_bWriteUint16(
                    hPayload, (UN16)psResp->uData.sInd.uData.sMetaData.sChannel.tSID);
                break;
            case SXIAPI_MESSOP_CHAN_METADATA:
                bWritten &= SXIAPI_bWriteUint16(
                    hPayload, (UN16)psResp->uData.sInd.uData.sChanMetaData.tSID);
                break;
            default:
                // No additional message payload
                break;
        }

        // All written ok?
        if(bWritten == TRUE)
        {
            BOOLEAN bSentCfm;

            n32SXiLogWritePayload(hSTI, hPayload, "OK");

            // Send this payload directly via SXi-LL
            bSentCfm = SXILL_bTxPayload(
                hProtocol, SXILL_PAYLOAD_TYPE_CONTROL, hPayload);
            if(bSentCfm == FALSE)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SXI_LOG_BASE_NAME": Unable to tx CFM payload.");
            }
        }

        // we are done with this payload
        STI_bFreePayload(hPayload);
    }

    return;
}

/*****************************************************************************
 *
 *   vReTxTimerCallback
 *
 *****************************************************************************/
static void vReTxTimerCallback (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        )
{
    STI_HDL *phSTI = (STI_HDL *)pvArg;

    if(phSTI != NULL)
    {
        STI_bSendTimeout((void *)*phSTI);
    }

    return;
}

/*****************************************************************************
 *
 *   vReleaseReTx
 *
 *****************************************************************************/
static void vReleaseReTx (
    SXI_RETX_STRUCT *psReTx
        )
{
    // Stop timer
    OSAL.eTimerStop(psReTx->hTimer);

#if (DEBUG_OBJECT == 1)
    // Determine when we gave up
    OSAL.vTimeUp(&psReTx->un32EndSeconds, &psReTx->un16EndMsec);
    // Do the math
    psReTx->un32DurationMsec =
        ((psReTx->un32EndSeconds * 1000) + psReTx->un16EndMsec) -
        ((psReTx->un32StartSeconds * 1000) + psReTx->un16StartMsec);
#else
    // Re-init re-tx msg ctrl
    psReTx->sMsg.sHdr.tOpType = 0;
    psReTx->sMsg.sHdr.tTransactionID = 0;
    psReTx->sMsg.n32Timeout = 0;
    psReTx->sMsg.un8Retries = 0;
#endif

    // Does a payload exist?
    if(psReTx->sMsg.hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        // Free this payload
        STI_bFreePayload(psReTx->sMsg.hPayload);
        psReTx->sMsg.hPayload = OSAL_INVALID_BUFFER_HDL;
    }

    // Release response mutex
    OSAL.eSemGive(psReTx->hResponse);

    return;
}

/*****************************************************************************
 *
 *   vSignalResponse
 *
 *****************************************************************************/
static void vSignalResponse (
    SXI_RETX_STRUCT *psReTx,
    SXI_MSG_HDR_STRUCT const *psMsgHdr,
    SXIAPI_STATUSCODE_ENUM eStatusCode
        )
{
    // Check if this message matches what we need to signal
    if((psMsgHdr->tOpType == psReTx->sMsg.sHdr.tOpType) &&
        (psMsgHdr->tTransactionID == psReTx->sMsg.sHdr.tTransactionID))
    {
        // Matches!

        // Write status code back to sender
        psReTx->sMsg.eStatusCode = eStatusCode;

        // Give up(release) re-tx msg ctrl
        vReleaseReTx(psReTx);
    }
    else
    {
        SMSAPI_DEBUG_vPrint(SXI_LOG_BASE_NAME, 4,
            "vSignalResponse: Ignoring OpType=%u, tid=%u [expected: %u,%u]",
            psMsgHdr->tOpType, psMsgHdr->tTransactionID,
            psReTx->sMsg.sHdr.tOpType, psReTx->sMsg.sHdr.tTransactionID);
    }

    return;
}

/*****************************************************************************
 *
 *   vSignalTimeout
 *
 *****************************************************************************/
static void vSignalTimeout (
    STI_PROTOCOL_HDL hProtocol,
    SXI_RETX_STRUCT *psReTx
        )
{
#if (DEBUG_OBJECT == 1)
    // Increment timeout counter
    psReTx->un32TimeoutCounter++;
#endif

    // Another attempt?
    if((psReTx->sMsg.un8Retries > 0) && (psReTx->sMsg.n32Timeout > 0))
    {
        // Try again
        psReTx->sMsg.un8Retries--;

        // Does a payload exist?
        if(psReTx->sMsg.hPayload != OSAL_INVALID_BUFFER_HDL)
        {
            STI_PROTOCOL_TX_PAYLOAD_STRUCT sTx;

            // Populate a pseudo-event for re-tx
            sTx.hSTI = psReTx->hSTI;
            sTx.hPayload = psReTx->sMsg.hPayload;
            sTx.un8Retries = psReTx->sMsg.un8Retries;
            sTx.n32Timeout = psReTx->sMsg.n32Timeout;

            // Re-transmit payload directly
            SXILL_vHandleTx(hProtocol, &sTx);
            return;
        }
    }

    // No more attempts remain or an error happened

    // Re-init re-tx msg ctrl
    psReTx->sMsg.eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Give up(release) re-tx msg ctrl
    vReleaseReTx(psReTx);

    return;
}

/*****************************************************************************
 *
 *   bExtractResponse
 *
 *****************************************************************************/
static BOOLEAN bExtractResponse (
    SXILL_PAYLOAD_TYPE tPayloadType,
    OSAL_BUFFER_HDL hPayload,
    SXI_MSG_HDR_STRUCT *psMsgHdr
        )
{
    BOOLEAN bValidData;
    UN16 un16Opcode = 0;

    // Extract opcode from message to be sent
    bValidData = SXIAPI_bPeekUint16(hPayload, &un16Opcode, 0);
    psMsgHdr->tOpType =
        (SXIAPI_OPTYPE)(un16Opcode & SXIAPI_MESSOP_TYPE_MASK);
    if(bValidData == TRUE)
    {
        // Based on connection type, find retx struct and populate
        // response parameters
        switch(tPayloadType)
        {
            case SXILL_PAYLOAD_TYPE_LINK:
            {
                // Prepare the resp arguments
                switch (psMsgHdr->tOpType)
                {
                    case SXIAPI_MESSOP_UARTLINKSTARTCMD:
                        psMsgHdr->tOpType =
                            SXIAPI_MESSOP_UARTLINKSTARTRESP;
                        bValidData = TRUE;
                    break;

                    case SXIAPI_MESSOP_UARTLINKSTATSCMD:
                        psMsgHdr->tOpType =
                            SXIAPI_MESSOP_UARTLINKSTATSRESP;
                        bValidData = TRUE;
                    break;

                    default:
                        // I have no idea what this is
                    break;
                }

                // Link types never have a transaction id
                psMsgHdr->tTransactionID = 0;
            }
            break;

            case SXILL_PAYLOAD_TYPE_CONTROL:
            {
                UN8 un8TransactionId = 0;

                // Control types also have a transaction id
                bValidData =
                    SXIAPI_bPeekUint8(
                        hPayload, &un8TransactionId, 2); // Opcode=2 bytes
                psMsgHdr->tTransactionID = (SXIAPI_TRANID)un8TransactionId;
            }
            break;

            case SXILL_PAYLOAD_TYPE_DATA:
            {
                // No responses for these
            }
            break;

            case SXILL_PAYLOAD_TYPE_AUDIO:
            {
                // No responses for these
            }
            break;

            default:
            {
                // Unknown type
            }
            break;
        }
    }

    return bValidData;
}

/*****************************************************************************
*
*   bHandleTxPayload
*
*****************************************************************************/
static BOOLEAN bHandleTxPayload(
    STI_HDL hSTI,
    SXILL_PAYLOAD_TYPE tPayloadType,
    STI_PROTOCOL_TX_PAYLOAD_STRUCT *psTx,
    SXI_RETX_STRUCT *psReTx
        )
{
    BOOLEAN bResponseHandled = FALSE;

    // Do we need to wait for a response?
    if((psTx->un8Retries > 0) || (psTx->n32Timeout > 0))
    {
        BOOLEAN bExtracted;

        // We may need to re-tx this message
        bExtracted = bExtractResponse(
            tPayloadType, psTx->hPayload,
            &psReTx->sMsg.sHdr);
        if(bExtracted == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Populate re-tx configuration
            psReTx->hSTI = hSTI;
            psReTx->sMsg.hPayload = psTx->hPayload;
            psReTx->sMsg.n32Timeout = psTx->n32Timeout;
            psReTx->sMsg.un8Retries = psTx->un8Retries;
#if (DEBUG_OBJECT == 1)
            OSAL.vTimeUp(&psReTx->un32StartSeconds, &psReTx->un16StartMsec);
            psReTx->un32EndSeconds = 0;
            psReTx->un16EndMsec = 0;
            psReTx->un32DurationMsec = 0;
#endif
            eReturnCode = OSAL.eTimerSetHandlerArg(
                psReTx->hTimer, &psReTx->hSTI);
            if(eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = OSAL.eTimerStartRelative(
                    psReTx->hTimer,
                    psTx->n32Timeout, 0);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Message has been handled
                    psTx->hPayload = OSAL_INVALID_BUFFER_HDL;
                    bResponseHandled = TRUE;
                }
            }
        }
    }
    else // No waiting, just send and go
    {
        // Message has been handled
        bResponseHandled = TRUE;
    }

    return bResponseHandled;
}

/*****************************************************************************
*
*   pvGetConnectionSpecificInfo
*
*****************************************************************************/
static void *pvGetConnectionSpecificInfo (
    STI_HDL hSTI,
    SXI_CONNECTION_ARG_STRUCT **ppsConnectionArg
        )
{
    void *pvConnectionSpecificPtr = NULL;
    SXI_CONNECTION_ARG_STRUCT *psConnectionArg;

    if(ppsConnectionArg == NULL)
    {
        // Use our own
        ppsConnectionArg = &psConnectionArg;
    }

    *ppsConnectionArg = (SXI_CONNECTION_ARG_STRUCT *)
        STI_pvGetConnectionArgument(hSTI);
    if(*ppsConnectionArg != SXI_CONNECTION_ARG_INVALID_HANDLE)
    {
        // Check the payload type from the connection argument
        switch((*ppsConnectionArg)->tPayloadType)
        {
            case SXILL_PAYLOAD_TYPE_LINK:
            {
                // Assign link stuff
                pvConnectionSpecificPtr =
                    (void *)&((*ppsConnectionArg)->uConnection.sLink);
            }
            break;

            case SXILL_PAYLOAD_TYPE_CONTROL:
            {
                // Assign control stuff...
                pvConnectionSpecificPtr =
                    (void *)&((*ppsConnectionArg)->uConnection.sControl);
            }
            break;

            case SXILL_PAYLOAD_TYPE_DATA:
            {
                // Assign data stuff...
                pvConnectionSpecificPtr =
                    (void *)&((*ppsConnectionArg)->uConnection.sData);
            }
            break;

            case SXILL_PAYLOAD_TYPE_AUDIO:
            {
                // TODO:
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SXI_LOG_BASE_NAME": Invalid Payload Type: %d.",
                    (*ppsConnectionArg)->tPayloadType);
            }
            break;
        }
    }

    return pvConnectionSpecificPtr;
}

/*****************************************************************************
 *
 *   hCreateBlockPoolForDSI
 *
 *****************************************************************************/
static SXI_BLOCKPOOL_ENTRY_STRUCT *psCreateBlockPoolForDSI (
    SXI_CONTROL_CONNECTION_STRUCT *psControl,
    SXIAPI_DSI tDSI,
    size_t tSuggestedOTABufferByteSize
        )
{
    SXI_BLOCKPOOL_ENTRY_STRUCT *psBlockPool =
        (SXI_BLOCKPOOL_ENTRY_STRUCT *)NULL;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        SXI_BLOCKPOOL_ENTRY_STRUCT sBlockPool;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        UN16 un16NumBlocks;

        // Initialize search structure
        sBlockPool.tDSI = tDSI;

        // Look for this DSI in the list
        eReturnCode = OSAL.eLinkedListSearch(
            psControl->hDataBlockPools,
            &hEntry,
            &sBlockPool);
        if (OSAL_SUCCESS == eReturnCode)
        {
            // All done -- just return now
            psBlockPool = (SXI_BLOCKPOOL_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(hEntry);

            return psBlockPool;
        }

        // Did we experience an error?
        if (OSAL_OBJECT_NOT_FOUND != eReturnCode)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME
                ": Unable to add search block pool list. (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));

            break;
        }

        // Create a new block pool list entry
        psBlockPool = (SXI_BLOCKPOOL_ENTRY_STRUCT *)
            SMSO_hCreate("SXI:BlockPoolEntry",
            sizeof(SXI_BLOCKPOOL_ENTRY_STRUCT),
            SMS_INVALID_OBJECT,FALSE);

        if ((SXI_BLOCKPOOL_ENTRY_STRUCT *)NULL == psBlockPool)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME": Unable to create block pool entry.");
            break;
        }

        // Block pool is for this DSI
        psBlockPool->tDSI = tDSI;

        snprintf(&acName[0], sizeof(acName),
            "SXI:BlkPool (DSI: %u)", tDSI);

        // Calculate the number of blocks to create
        // don't worry if the suggested size was zero -
        // we'll always guarantee at least 1 block
        un16NumBlocks = (UN16)SXILL_NUM_BLOCKS_FOR_SIZE(
            tSuggestedOTABufferByteSize);

        // Generate a block pool for this service
        // Create an rx frame block pool
        eReturnCode =
            OSAL.eBlockPoolCreate (
                &psBlockPool->hBlockPool,
                &acName[0],
                (UN16)SXILL_RX_FRAME_BLOCK_SIZE,
                un16NumBlocks,
                OSAL_BLOCK_POOL_OPTION_NONE
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME
                ": Failed to create block pool for DSI %u", tDSI);
            break;
        }

        // Add this block pool to the list
        eReturnCode = OSAL.eLinkedListAdd(
            psControl->hDataBlockPools,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            (void *)psBlockPool);
        if (OSAL_SUCCESS != eReturnCode)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME
                ": Unable to add new block pool to list. (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));

            break;
        }

        // Provide block pool
        return psBlockPool;
    } while (FALSE);

    vReleaseBlockPoolEntry(psBlockPool);

    return NULL;
}

/*****************************************************************************
*
*   bHandleNewDataDestination
*
*****************************************************************************/
static BOOLEAN bHandleNewDataDestination (
    SXI_CONTROL_CONNECTION_STRUCT *psControl,
    SXI_ADD_DATA_PAYLOAD_DEST_STRUCT *psAddDest
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN8 un8Index;
    SXI_DATA_PAYLOAD_DEST_STRUCT *psDest;
    BOOLEAN bSuccess = TRUE;
    SXI_BLOCKPOOL_ENTRY_STRUCT *psBlockPool;

    // Create the block pool for this DSI
    psBlockPool = psCreateBlockPoolForDSI(
        psControl, psAddDest->tDSI, psAddDest->tSuggestedOTABufferByteSize);

    if (NULL == psBlockPool)
    {
        // We failed - stop here
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SXI_LOG_BASE_NAME": Unable to create/locate block pool for DSI %u",
            psAddDest->tDSI);

        // We're done with this now
        SMSO_vDestroy((SMS_OBJECT)psAddDest);

        return FALSE;
    }

    // Add all of these new destinations
    for (un8Index = 0; un8Index < psAddDest->un8NumDMIs; un8Index++)
    {
        // Create a new destination object now
        psDest = (SXI_DATA_PAYLOAD_DEST_STRUCT *)
            SMSO_hCreate("SXI:DestEntry",
            sizeof(SXI_DATA_PAYLOAD_DEST_STRUCT),
            SMS_INVALID_OBJECT, FALSE);

        // Did that work?
        if (NULL != psDest)
        {
            OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            // Populate this new entry
            psDest->hDestPools = OSAL_INVALID_OBJECT_HDL;
            psDest->tDMI = psAddDest->atNewDMIs[un8Index];

            // Add this new destination to the payload list
            eReturnCode = OSAL.eLinkedListAdd(
                psControl->hDataPayloadDestinations,
                &hEntry,
                (void *)psDest);

            if (OSAL_ERROR_LIST_ITEM_NOT_UNIQUE == eReturnCode)
            {
                // existing DMI

                // destroy new object
                SMSO_vDestroy((SMS_OBJECT)psDest);
                // use existing destination object
                psDest = (SXI_DATA_PAYLOAD_DEST_STRUCT *)
                    OSAL.pvLinkedListThis(hEntry);
            }
            else if (OSAL_SUCCESS != eReturnCode)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SXI_LOG_BASE_NAME
                    ": Unable to add new payload destination. (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));

                // Free this memory now
                SMSO_vDestroy((SMS_OBJECT)psDest);

                // Just do our best and try any more that are listed
                // but remember that this failed
                bSuccess = FALSE;
                continue;
            }

            if (OSAL_INVALID_OBJECT_HDL == psDest->hDestPools)
            {
                // Create list of payload destination pools
                eReturnCode = OSAL.eLinkedListCreate(
                    &psDest->hDestPools,
                    "SXI:DestPools",
                    (OSAL_LL_COMPARE_HANDLER)n16CompareBlockPoolEntry,
                    OSAL_LL_OPTION_UNIQUE);

                if (OSAL_SUCCESS != eReturnCode)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        "SXi: Unable to create new payload destination pools list. (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));

                    bSuccess = FALSE;
                    continue;
                }
            }

            // Add destinalion pool reference
            eReturnCode = OSAL.eLinkedListAdd(
                psDest->hDestPools,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                psBlockPool);

            if ((OSAL_SUCCESS != eReturnCode) &&
                (OSAL_ERROR_LIST_ITEM_NOT_UNIQUE != eReturnCode))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "SXi: Unable to add payload destination pool to pools list. (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));

                bSuccess = FALSE;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SXI_LOG_BASE_NAME": Can't create payload dest object.");

            // Just do our best and try any more that are listed
            // but remember that this failed
            bSuccess = FALSE;
        }
    }

    // We're done with this now
    SMSO_vDestroy((SMS_OBJECT)psAddDest);

    return bSuccess;
}

/*****************************************************************************
*
*   n16CompareBlockPoolEntry
*
*****************************************************************************/
static N16 n16CompareBlockPoolEntry (
    SXI_BLOCKPOOL_ENTRY_STRUCT *psPool1,
    SXI_BLOCKPOOL_ENTRY_STRUCT *psPool2
        )
{
    // Validate pointers
    if ((NULL == psPool1) || (NULL == psPool2))
    {
        return N16_MIN;
    }

    return COMPARE(psPool1->tDSI, psPool2->tDSI);
}

/*****************************************************************************
*
*   vReleaseBlockPoolEntry
*
*****************************************************************************/
static void vReleaseBlockPoolEntry (
    SXI_BLOCKPOOL_ENTRY_STRUCT *psPool
        )
{
    BOOLEAN bValid;

    // Validate object
    bValid = SMSO_bValid((SMS_OBJECT)psPool);
    if (TRUE == bValid)
    {
        if (OSAL_INVALID_OBJECT_HDL != psPool->hBlockPool)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Delete the block pool
            eReturnCode = OSAL.eBlockPoolDelete(psPool->hBlockPool);
            if (OSAL_SUCCESS != eReturnCode)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SXI_LOG_BASE_NAME":  cannot delete block pool");
            }

            // Clear the handle
            psPool->hBlockPool = OSAL_INVALID_OBJECT_HDL;
        }

        // Destroy object now
        SMSO_vDestroy((SMS_OBJECT)psPool);
    }

    return;
}

/*****************************************************************************
*
*   bTransferPayloadIterator
*
*****************************************************************************/
static BOOLEAN bTransferPayloadIterator (
    SXI_BLOCKPOOL_ENTRY_STRUCT *psPool,
    SXI_BLOCKPOOL_ITERATOR_STRUCT *psData
        )
{
    BOOLEAN bResult = TRUE;
    BOOLEAN bHandled;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_BUFFER_HDL hNewPayload;
    SMS_EVENT_HDL hEvent;
    SXIAPI_DATA_PAYLOAD_EVENT_STRUCT *psDataPayload =
        (SXIAPI_DATA_PAYLOAD_EVENT_STRUCT *)NULL;

    // Decrease this counter to be able to detect the last destination pool
    psData->un32Items--;

    // Allocate a buffer to send to the manager
    // using the manager's block pool
    hNewPayload = OSAL.hBufferAllocate(
        psPool->hBlockPool,
        FALSE, FALSE, OSAL_BUFFER_ALLOCATE_OPTION_NONE);

    if (hNewPayload == OSAL_INVALID_BUFFER_HDL)
    {
        // Not a fatal error -- can continue with other destinations
        return TRUE;
    }

    if (0 == psData->un32Items)
    {
        // Single or last block pool destination
        eReturnCode = OSAL.eBufferAppend(hNewPayload, psData->hPayload);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Free the new buffer if an error was experienced
            OSAL.eBufferFree(hNewPayload);
            return FALSE;
        }
        // Single destination -- stop iterating
        bResult = FALSE;
    }
    else
    {
        // Multiple block pool destinations
        SXI_BLOCK_COPY_ITERATOR_STRUCT sArg;

        sArg.hPayload = hNewPayload;
        sArg.bOk = FALSE;

        eReturnCode = OSAL.eBufferBlocksIterate(
            psData->hPayload,
            (OSAL_BUFFER_ITERATOR_HANDLER)eBlockCopyIterator,
            &sArg);

        if ((OSAL_SUCCESS != eReturnCode) || (FALSE == sArg.bOk))
        {
            OSAL.eBufferFree(hNewPayload);
            return TRUE;
        }
    }

    // Allocate an event for this payload
    hEvent = DATASERVICE_MGR_hAllocateRadioEvent(
        (SMS_EVENT_TYPE_ENUM)RADIO_EVENT_SXI_DATA_PAYLOAD,
        SMS_EVENT_OPTION_NONBLOCK,
        (void **)&psDataPayload);

    if ((hEvent == SMS_INVALID_EVENT_HDL) ||
        (psDataPayload == NULL))
    {
        OSAL.eBufferFree(hNewPayload);
        return bResult;
    }

    // Populate event
    psDataPayload->eType = psData->ePayloadType;
    psDataPayload->tDSI = psPool->tDSI;
    psDataPayload->hPayload = hNewPayload;
    psDataPayload->hBlockPool = psPool->hBlockPool;

    // Post event
    bHandled = SMSE_bPostEvent(hEvent);
    if (TRUE == bHandled)
    {
        psData->bHandled = TRUE;
    }
    else
    {
        // Error! Unable to post the event
        OSAL.eBufferFree(hNewPayload);
    }

    return bResult;
}

/*****************************************************************************
*
*   eBlockCopyIterator
*
*****************************************************************************/
static OSAL_BUFFER_ITERATOR_RESULT_ENUM eBlockCopyIterator (
    void *pvData,
    size_t tDataSize,
    SXI_BLOCK_COPY_ITERATOR_STRUCT *psArg
        )
{
    size_t tSize;

    tSize = OSAL.tBufferWriteTail(
        psArg->hPayload, pvData, tDataSize,
        OSAL_OBJ_TIMEOUT_NONE);

    psArg->bOk = (tDataSize == tSize) ? TRUE : FALSE;

    return (TRUE == psArg->bOk)
        ? OSAL_BUFFER_ITERATOR_RESULT_KEEP
        : OSAL_BUFFER_ITERATOR_RESULT_STOP;
}

#if SMS_LOGGING == 1

/*****************************************************************************
*
*   pacOpCodeToStr
*
*****************************************************************************/
static const char *pacOpCodeToStr (
    SXIAPI_OPCODE tOpCode
        )
{
    const char *pacCtrlOpCode;

    switch(tOpCode)
    {
        case SXIAPI_MESSOP_CMD:
            pacCtrlOpCode = "CMD";
        break;

        case SXIAPI_MESSOP_RESP:
            pacCtrlOpCode = "RSP";
        break;

        case SXIAPI_MESSOP_IND:
            pacCtrlOpCode = "IND";
        break;

        case SXIAPI_MESSOP_CFM:
            pacCtrlOpCode = "CFM";
        break;

        default:
            pacCtrlOpCode = "???";
        break;
    }

    return pacCtrlOpCode;
}

/*****************************************************************************
*
*   n32SXiLogWrite
*
*****************************************************************************/
static N32 n32SXiLogWrite (
    STI_HDL hSTI,
    const char *pcFormat,
    ...
        )
{
    SXI_CONNECTION_ARG_STRUCT *psConnectionArg;
    N32 n32NumWritten = 0;

    psConnectionArg =  (SXI_CONNECTION_ARG_STRUCT *)
        STI_pvGetConnectionArgument(hSTI);
    if(psConnectionArg != NULL)
    {
        va_list tList; // variable arguments list
        SXI_LOG_STRUCT *psLog;
        UN32 un32Seconds;
        UN16 un16MSecs;
        struct tm sTmBuf;
        TIME_T tTime;

        psLog = &psConnectionArg->sLog;

        OSAL.eTimeGetLocal(&un32Seconds);
        OSAL.vTimeUp((UN32 *)NULL, &un16MSecs);

        tTime = (TIME_T)un32Seconds;
        OSAL.localtime_r(&tTime, &sTmBuf);

        // grab variable arguments (pop)
        va_start(tList, pcFormat);

       n32NumWritten = OSAL.n32LogWrite(
           psLog->hLog, "%02u:%02u:%02u.%03u\t",
            sTmBuf.tm_hour, sTmBuf.tm_min, sTmBuf.tm_sec, un16MSecs);
        n32NumWritten += OSAL.n32VLogWrite(
            psLog->hLog, pcFormat, &tList);

        // restore stack (push)
        va_end(tList);
    }

    return n32NumWritten;
}

/*****************************************************************************
*
*   n32SXiLogWriteMsg
*
*****************************************************************************/
static N32 n32SXiLogWriteMsg (
    STI_HDL hSTI,
    SXIAPI_OPCODE tOpCode,
    SXIAPI_TRANID tId,
    const char *pacInfoText
        )
{
    N32 n32NumWritten;

    n32NumWritten = n32SXiLogWrite(hSTI, SXI_LOG_MSG_FORMAT,
        pacOpCodeToStr(tOpCode & ~SXIAPI_MESSOP_TYPE_MASK),
        tOpCode & ~SXIAPI_MESSOP_TYPE_MASK,
        tOpCode & SXIAPI_MESSOP_TYPE_MASK,
        tId,
        pacInfoText
            );

    return n32NumWritten;
}

/*****************************************************************************
*
*   n32SXiLogWritePayload
*
*****************************************************************************/
static N32 n32SXiLogWritePayload (
    STI_HDL hSTI,
    OSAL_BUFFER_HDL hPayload,
    const char *pacInfoText
        )
{
    N32 n32NumWritten = 0;
    UN16 un16Opcode = 0;
    UN8 un8TransactionId = 0;
    BOOLEAN bSuccess;

    // Extract opcode
    bSuccess = SXIAPI_bPeekUint16(hPayload, &un16Opcode, 0);
    if (bSuccess == FALSE)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SXI_LOG_BASE_NAME": Unable to get opcode.");
        return 0;
    }

    // Extract transaction id
    bSuccess = SXIAPI_bPeekUint8(
        hPayload, &un8TransactionId, 2); // Opcode=2 bytes
    if (bSuccess == FALSE)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SXI_LOG_BASE_NAME": Unable to get transaction id.");
        return 0;
    }

    n32NumWritten = n32SXiLogWriteMsg(
        hSTI, (SXIAPI_OPCODE)un16Opcode,
        (SXIAPI_TRANID)un8TransactionId,
        pacInfoText
            );

    return n32NumWritten;
}

/*****************************************************************************
*
*   SXI_n32SXiLogWritePayload
*
*****************************************************************************/
N32 SXI_n32SXiLogWritePayload (
    STI_HDL hSTI,
    OSAL_BUFFER_HDL hPayload,
    const char *pacInfoText
        )
{
    N32 n32NumWritten;
    n32NumWritten = n32SXiLogWritePayload(hSTI, hPayload, pacInfoText);
    return n32NumWritten;
}

#elif __STDC_VERSION__ < 199901L

static const char *pacOpCodeToStrNothing (
    SXIAPI_OPCODE tOpCode
        )
{
    return "Nothing";
}

static N32 n32SXiLogWriteNothing (
    STI_HDL hSTI,
    const char *pcFormat,
    ...
        )
{
    return 0;
}

static N32 n32SXiLogWriteMsgNothing (
    STI_HDL hSTI,
    SXIAPI_OPCODE tOpCode,
    SXIAPI_TRANID tId,
    const char *pacInfoText
        )
{
    return 0;
}

static N32 n32SXiLogWritePayloadNothing (
    STI_HDL hSTI,
    OSAL_BUFFER_HDL hPayload,
    const char *pacInfoText
        )
{
    return 0;
}

N32 SXI_n32SXiLogWritePayloadNothing (
    STI_HDL hSTI,
    OSAL_BUFFER_HDL hPayload,
    const char *pacInfoText
        )
{
    return 0;
}

#endif
