/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
#include <stdlib.h>
#include <stdio.h>

#include "standard.h"
#include "osal.h"
#include "sti_api.h"
#include "sxi.h"
#include "sxi_rw.h"
#include "sxiapi.h"

/*****************************************************************************
*
*       SXIAPI_eFWUpdateProgCmd
*
*       Inputs:
*           STI_HDL hSTI,
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eFWUpdateProgCmd (
    STI_HDL hSTI
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_FIRMWAREUPDATE_PROG);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_FWUPDATE_PROG_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
   }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eFWUpdateCfgCmd
*
*       Inputs:
*           STI_HDL hSTI,
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eFWUpdateCfgCmd (
    STI_HDL hSTI,
    UN32 un32BytesInPacket,
    UN32 un32NumPackets
        )

{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;
        SXIAPI_TRANID tTransactionId;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_FIRMWAREUPDATE_CFG);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See SX-9845-0097, sec 13.3
        SXIAPI_bWriteUint32(hPayload, un32BytesInPacket);
        SXIAPI_bWriteUint32(hPayload, un32NumPackets);

        // Send payload
        eResultCode =
            SXI_eSendPayload(
                hSTI, hPayload,
                SXI_DEFAULT_RETRIES,
                SXI_FWUPDATE_CFG_TIMEOUT_MSEC,
                &eStatusCode
                    );
        if((STI_RESULT_CODE_OK != eResultCode) && // ok
            (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                )
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}

/*****************************************************************************
*
*       SXIAPI_eFWUpdatePacketCmd
*
*       Inputs:
*           STI_HDL hSTI,
*
*       Outputs:
*           SXIAPI_STATUSCODE_SUCCESS
*               SXIAPI_STATUSCODE_SUCCESS if sucessful
*               otherwise, see SXPAPI.h for error codes
*
*****************************************************************************/
SXIAPI_STATUSCODE_ENUM SXIAPI_eFWUpdatePacketCmd (
    STI_HDL hSTI,
    UN16  un16PacketID,
    UN8   *pun8PacketData,
    UN32  un32BytesInPacket
        )
{
    OSAL_BUFFER_HDL hPayload;
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;

    // Allocate a payload buffer for this command
    hPayload = STI_hAllocatePayload( hSTI );
    if(hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        SXIAPI_TRANID tTransactionId;
        size_t tBytesWritten;

        // Get transaction id
        tTransactionId =
            SXI_tGetNextTransIDForConnection(hSTI);

        // Populate command header
        SXIAPI_bWriteUint16(hPayload, (UN16)SXIAPI_MESSOP_FIRMWAREUPDATE_PACKET);
        SXIAPI_bWriteUint8(hPayload, (UN8)tTransactionId);

        // Populate message
        // See SX-9845-0097, sec 13.5
        SXIAPI_bWriteUint16(hPayload, un16PacketID);

        tBytesWritten = OSAL.tBufferWriteTail(hPayload, pun8PacketData,
                                              un32BytesInPacket);

        if ( tBytesWritten == un32BytesInPacket )
        {
            STI_PROTOCOL_RESULT_CODE_ENUM eResultCode;

            // Send payload
            eResultCode =
                SXI_eSendPayload(
                    hSTI, hPayload,
                    SXI_DEFAULT_RETRIES,
                    SXI_FWUPDATE_PACKET_TIMEOUT_MSEC,
                    &eStatusCode
                        );
            if((STI_RESULT_CODE_OK != eResultCode) && // ok
                (STI_RESULT_CODE_TIMEOUT != eResultCode) // sent, but no response
                    )
            {
                // Error! Unable to send this payload.
                STI_bFreePayload(hPayload);
            }
        }
        else
        {
            // Error! Unable to send this payload.
            STI_bFreePayload(hPayload);
        }
    }

    return eStatusCode;
}
