/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SXI_H_
#define _SXI_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include "sti_api.h"
#include "sxiapi.h"
#include "sxi.h"

/**********************/
 /** LOCAL CONSTANTS  **/
/**********************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// If logging hasn't been configured already
// then only enable when we're in debug
#ifndef SMS_LOGGING
#if SMS_DEBUG == 1 && OSAL_LOG == 1
#define SMS_LOGGING 1
#else
#define SMS_LOGGING 0
#endif
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

#define SXI_LINK_ERROR_UNSUPPORTED_PAYLOAD_TYPE (0x00)
#define SXI_LINK_ERROR_UNKNOWN_OPCODE (0x01)

// For SMS Logging
#define SXI_LOG_BASE_NAME "SXi"
#define SXI_LOG_SIZE ((256 * 1024) - 1)  // 256k
#define SXI_LOG_HEADER "Log:%s\n"
#define SXI_LOG_MSG_FORMAT "%s=0x%04.4x, OpType=0x%04.4x, tId=%u, %s\n"

  /*********************/
 /** LOCAL TYPEDEFS  **/
/*********************/

// SXI Response Queue
typedef struct sxi_response_queue_struct
{
    // Queue of response messages
    OSAL_OBJECT_HDL hQueue;

    // Instance synchronization object
    OSAL_OBJECT_HDL hMutex;

} SXI_RESPONSE_QUEUE_STRUCT;

typedef struct sxi_retx_ctrl_struct
{
    // The SXi payload header we are waiting for
    SXI_MSG_HDR_STRUCT sHdr;

    // Message payload to tx and re-tx
    OSAL_BUFFER_HDL hPayload;

    // Number of retries allowed
    UN8 un8Retries;

    // Response timeout (msec)
    // 0 = No response required, retries must be 0
    // -1 = Infinate wait
    N32 n32Timeout;

    // Response status code from received msg
    SXIAPI_STATUSCODE_ENUM eStatusCode;

} SXI_RETX_CTRL_STRUCT;

typedef struct sxi_retx_struct
{
    // STI Connection using this re-tx control structure
    STI_HDL hSTI;

    // Re-Tx Timer (used for timeouts)
    OSAL_OBJECT_HDL hTimer;

    // Re-Tx Control Synchronization object
    OSAL_OBJECT_HDL hMutex;

    // Re-Tx Response Synchronization object
    OSAL_OBJECT_HDL hResponse;

    // The SXi payload header we're looking for
    // and matching re-tx payload
    SXI_RETX_CTRL_STRUCT sMsg;

#if (DEBUG_OBJECT == 1)
    // Time timer was set
    UN32 un32StartSeconds;
    UN16 un16StartMsec;

    // Time timer was expired
    UN32 un32EndSeconds;
    UN16 un16EndMsec;

    // Timeout
    UN32 un32DurationMsec;

    // Timeout counter
    UN32 un32TimeoutCounter;
#endif

} SXI_RETX_STRUCT;

typedef struct sxi_link_connection_struct
{
    // Local message for parsing status
    SXIAPI_RX_STRUCT sRxData;

    // ReTx struct
    SXI_RETX_STRUCT *psReTx;

    // Link Stats
    UN32 un32LinkErrors;
    UN32 un32KeepAlivesReceived;
    UN32 un32UnsupportedPayloadType;
    UN32 un32UnknownOpCode;
    UN32 un32UnknownError;

} SXI_LINK_CONNECTION_STRUCT;

typedef struct sxi_control_connection_struct
{
    // Received message queue
    SXI_RESPONSE_QUEUE_STRUCT *psRespQueue;

    // Local message for parsing RESPs
    SXIAPI_RX_STRUCT sSxiResp;

    // Message for parsing INDs
    SXIAPI_RX_STRUCT *psSxiInd;

    // ReTx struct
    SXI_RETX_STRUCT *psReTx;

    // Current SXi transaction id
    SXIAPI_TRANID tTransactionID;

    // Global Metadata Object
    SXI_GMD_HDL hGMD;

    // List which determines how to
    // demux data payloads
    OSAL_OBJECT_HDL hDataPayloadDestinations;

    // The list of block pools used by
    // data services
    OSAL_OBJECT_HDL hDataBlockPools;

} SXI_CONTROL_CONNECTION_STRUCT;

typedef struct sxi_data_connection_struct
{
    // List which determines how to
    // demux data payloads
    OSAL_OBJECT_HDL hDataPayloadDestinations;

} SXI_DATA_CONNECTION_STRUCT;

typedef struct sxi_audio_connection_struct
{
    // TODO:
    UN32 un32Dummy;

} SXI_AUDIO_CONNECTION_STRUCT;

typedef union sxi_connection_union
{
    SXI_LINK_CONNECTION_STRUCT sLink;

    SXI_CONTROL_CONNECTION_STRUCT sControl;

    SXI_DATA_CONNECTION_STRUCT sData;

    SXI_AUDIO_CONNECTION_STRUCT sAudio;

} SXI_CONNECTION_UNION;

#if SMS_LOGGING == 1

typedef struct sxi_log_statistics_struct
{
    // Nothing right now, possible use in the future
    UN32 un32Dummy;

} SXI_LOG_STATISTICS_STRUCT;

typedef struct sxi_log_struct
{
    OSAL_OBJECT_HDL hLog;
    SXI_LOG_STATISTICS_STRUCT sStatistics;

} SXI_LOG_STRUCT;

#endif

// SXI Connection Argument
typedef struct sxi_connection_arg_struct
{
    // SXILL Connections require a payload type which defines
    // this unique connection
    SXILL_PAYLOAD_TYPE tPayloadType;

    // SXi Connection Name
    const char *pacName;

    // SXi Connection union
    SXI_CONNECTION_UNION uConnection;

    // Convenience pointer to the connection's retx struct
    SXI_RETX_STRUCT *psReTx;

    // SXi FSM
    SXI_FSM_HDL hFSM;

#if SMS_LOGGING == 1
    SXI_LOG_STRUCT sLog;
#endif

} SXI_CONNECTION_ARG_STRUCT;

typedef struct sxi_data_payload_dest_struct
{
    SXIAPI_DMI tDMI;

    // The list of data block pool destinations
    OSAL_OBJECT_HDL hDestPools;

} SXI_DATA_PAYLOAD_DEST_STRUCT;

typedef struct sxi_blockpool_entry_struct
{
    // The DSI of the service and stream
    SXIAPI_DSI tDSI;

    // The block pool to allocate payloads from
    OSAL_OBJECT_HDL hBlockPool;

} SXI_BLOCKPOOL_ENTRY_STRUCT;

typedef struct sxi_blockpool_iterator_struct
{
    SXIAPI_DATA_PACKET_TYPE_ENUM ePayloadType;
    UN32 un32Items;
    OSAL_BUFFER_HDL hPayload;
    BOOLEAN bHandled;

} SXI_BLOCKPOOL_ITERATOR_STRUCT;

typedef struct sxi_block_copy_iterator_struct
{
    OSAL_BUFFER_HDL hPayload;
    BOOLEAN bOk;

} SXI_BLOCK_COPY_ITERATOR_STRUCT;

  /*****************************/
 /** LOCAL STRUCT TEMPLATES  **/
/*****************************/

  /****************************/
 /** LOCAL UNION TEMPLATES  **/
/****************************/

  /**********************/
 /** LOCAL VARIABLES  **/
/**********************/

  /*******************/
 /** LOCAL MACROS  **/
/*******************/

  /***********************/
 /** LOCAL PROTOTYPES  **/
/***********************/

static SXI_RESPONSE_QUEUE_STRUCT *psCreateResponseQueue(
    char const *pacConnectionName,
    UN32 un32Size
        );

static void vDestroyResponseQueue(
    SXI_RESPONSE_QUEUE_STRUCT *psObj
        );

static SXI_RETX_STRUCT *psCreateReTx ( char const *pacConnectionName );

static void vDestroyReTx (
    SXI_RETX_STRUCT *psReTx
        );

static BOOLEAN bDecodeInd(
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_RX_STRUCT *psResp,
    STI_HDL hControlCxn
        );

static BOOLEAN bDecodeResp(
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_RX_STRUCT *psResp
        );

static BOOLEAN bDecodeLink(
    STI_PROTOCOL_HDL hProtocol,
    STI_HDL hSTI,
    SXI_LINK_CONNECTION_STRUCT *psLinkConnection,
    STI_PROTOCOL_RX_PAYLOAD_STRUCT *psRx
        );

static BOOLEAN bDecodeBDCastNames (
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_BDCAST_NAMES_STRUCT *psNames
        );

static BOOLEAN bDecodeTrackText (
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_TRACK_INFO_STRUCT *psTrackInfo
        );

static BOOLEAN bDecodeExtTrackMetadata (
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_TRACK_INFO_STRUCT *psTrackInfo
        );

static BOOLEAN bDecodeExtChanMetadata (
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_CHAN_METADATA_STRUCT *psChanInfo
        );

static void vSendCfm (
    STI_PROTOCOL_HDL hProtocol,
    STI_HDL hSTI,
    SXIAPI_RX_STRUCT const *psResp
        );

static SXIAPI_RX_STRUCT *psAllocateResponse(
    SXI_RESPONSE_QUEUE_STRUCT *psObj
        );

static void vInitInstance (
    SXI_INSTANCE_STRUCT *psInstance,
    OSAL_OBJECT_HDL hMutex
        );

static void vAddInstance (
    SXI_INSTANCE_STRUCT *psInstance
        );

static BOOLEAN bRemoveInstance (
    SXI_INSTANCE_STRUCT *psInstance
        );

static BOOLEAN bDecodeControl(
    STI_PROTOCOL_HDL hProtocol,
    STI_HDL hSTI,
    SXI_CONTROL_CONNECTION_STRUCT *psControlConnection,
    STI_PROTOCOL_RX_PAYLOAD_STRUCT *psRx,
    void *pvArg
        );

static void vReTxTimerCallback (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        );

static void vReleaseReTx (
    SXI_RETX_STRUCT *psReTx
        );

static void vSignalResponse (
    SXI_RETX_STRUCT *psReTx,
    SXI_MSG_HDR_STRUCT const *psMsgHdr,
    SXIAPI_STATUSCODE_ENUM eStatusCode
        );

static void vSignalTimeout (
    STI_PROTOCOL_HDL hProtocol,
    SXI_RETX_STRUCT *psReTx
        );

static BOOLEAN bExtractResponse (
    SXILL_PAYLOAD_TYPE tPayloadType,
    OSAL_BUFFER_HDL hPayload,
    SXI_MSG_HDR_STRUCT *psMsgHdr
        );

static BOOLEAN bHandleTxPayload(
    STI_HDL hSTI,
    SXILL_PAYLOAD_TYPE tPayloadType,
    STI_PROTOCOL_TX_PAYLOAD_STRUCT *psTx,
    SXI_RETX_STRUCT *psReTx
        );

static void *pvGetConnectionSpecificInfo (
    STI_HDL hSTI,
    SXI_CONNECTION_ARG_STRUCT **ppsConnectionArg
        );

static SXI_BLOCKPOOL_ENTRY_STRUCT *psCreateBlockPoolForDSI (
    SXI_CONTROL_CONNECTION_STRUCT *psControl,
    SXIAPI_DSI tDSI,
    size_t tSuggestedOTABufferByteSize
        );

static BOOLEAN bHandleNewDataDestination (
    SXI_CONTROL_CONNECTION_STRUCT *psControl,
    SXI_ADD_DATA_PAYLOAD_DEST_STRUCT *psAddDest
        );

static N16 n16CompareBlockPoolEntry (
    SXI_BLOCKPOOL_ENTRY_STRUCT *psPool1,
    SXI_BLOCKPOOL_ENTRY_STRUCT *psPool2
        );

static void vReleaseBlockPoolEntry (
    SXI_BLOCKPOOL_ENTRY_STRUCT *psPool
        );

static BOOLEAN bTransferPayloadIterator (
    SXI_BLOCKPOOL_ENTRY_STRUCT *psPool,
    SXI_BLOCKPOOL_ITERATOR_STRUCT *psData
        );

static OSAL_BUFFER_ITERATOR_RESULT_ENUM eBlockCopyIterator (
    void *pvData,
    size_t tDataSize,
    SXI_BLOCK_COPY_ITERATOR_STRUCT *psArg
        );

#if SMS_LOGGING == 1

static const char *pacOpCodeToStr (
    SXIAPI_OPCODE tOpCode
        );

static N32 n32SXiLogWrite (
    STI_HDL hSTI,
    const char *pcFormat,
    ...
        );

static N32 n32SXiLogWritePayload (
    STI_HDL hSTI,
    OSAL_BUFFER_HDL hPayload,
    const char *pacInfoText
        );

static N32 n32SXiLogWriteMsg (
    STI_HDL hSTI,
    SXIAPI_OPCODE tOpCode,
    SXIAPI_TRANID tId,
    const char *pacInfoText
        );

#elif __STDC_VERSION__ < 199901L

static const char *  pacOpCodeToStrNothing (
    SXIAPI_OPCODE tOpCode
        );
#define pacOpCodeToStr if(1);else pacOpCodeToStrNothing

static N32 n32SXiLogWriteNothing (
    STI_HDL hSTI,
    const char *pcFormat,
    ...
        );
#define n32SXiLogWrite if(1);else n32SXiLogWriteNothing

static N32 n32SXiLogWritePayloadNothing (
    STI_HDL hSTI,
    OSAL_BUFFER_HDL hPayload,
    const char *pacInfoText
        );
#define n32SXiLogWritePayload if(1);else n32SXiLogWritePayloadNothing

static N32 n32SXiLogWriteMsgNothing (
    STI_HDL hSTI,
    SXIAPI_OPCODE tOpCode,
    SXIAPI_TRANID tId,
    const char *pacInfoText
        );
#define n32SXiLogWriteMsg if(1);else n32SXiLogWriteMsgNothing

#else

#define pacOpCodeToStr(...)
#define n32SXiLogWrite(...)
#define n32SXiLogWritePayload(...)
#define n32SXiLogWriteMsg(...)

#endif

  /*****************************/
 /** LOCAL INLINE FUNCTIONS  **/
/*****************************/

#endif // _SXI_H_

