/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
* DESCRIPTION
*
*  This module contains the SXi GMD implementation.
*
******************************************************************************/

#include <string.h>

#include "standard.h"
#include "osal.h"

// Include things I need from SMS
#include "sms_version.h"
#include "sms.h"
#include "sms_obj.h"
#include "sms_event.h"
#include "radio_event_types.h"
#include "sti_api.h"
#include "sxiapi.h"
#include "srh.h"
#include "com.h"

#include "sxi.h"
#include "sxi_rw.h"
#include "report_obj.h"
#include "league_obj.h"
#include "team_obj.h"
#include "string_obj.h"
#include "sxi_sports_id.h"
#include "sxi_market_id.h"

#include "decoder_obj.h"

#include "preset_band_obj.h"

// Include SXI GMD module headers
#include "sxi_gmd.h"
#include "_sxi_gmd.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
FRIEND FUNCTIONS
*****************************************************************************/

/*******************************************************************************
*
*   SXI_GMD_bInit
*
*******************************************************************************/
SXI_GMD_HDL SXI_GMD_hInit ( void )
{
    SXI_GMD_HDL hGMD = SXI_GMD_INVALID_HDL;
    SXI_GMD_STRUCT *psGMD;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Create name for this GMD instance
    snprintf(&acName[0], sizeof(acName), "SXi-GMD");

    // Allocate memory for the GMD instance
    psGMD =
        (SXI_GMD_STRUCT *)
        SMSO_hCreate(
        acName,
        sizeof(SXI_GMD_STRUCT),
        SMS_INVALID_OBJECT, // Parent, none
        FALSE); // don't need lock
    if(psGMD == NULL)
    {
        // Error!
        return SXI_GMD_INVALID_HDL;
    }

    // Initialize GMD
    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32TeamId = 0, un32LeagueId = 0;

        // initially no GMIs are enabled
        psGMD->un64ExtMetadataGMIMask = SXIAPI_GMI_MASK_NONE;

        // Initialize GMD array
        psGMD->un8ExtMetadataEmiCnt = 0;
        OSAL.bMemSet(psGMD->atExtMetadataGID, 0,
            sizeof(psGMD->atExtMetadataGID));

        // Initialize GMD types
        OSAL.bMemSet(&psGMD->sMarkets, 0, sizeof(psGMD->sMarkets));
        OSAL.bMemSet(&psGMD->sLeagues, 0, sizeof(psGMD->sLeagues));
        psGMD->sLeagues.n32Version = LEAGUE_INVALID_VERSION;
        OSAL.bMemSet(&psGMD->sTeams, 0, sizeof(psGMD->sTeams));
        psGMD->sTeams.n32Version = TEAM_INVALID_VERSION;
        OSAL.bMemSet(&psGMD->sFF, 0, sizeof(psGMD->sFF));

        // Create working CIDs
        psGMD->hWorkingTeamId = CID_hCreate(
            CID_POOL_INVALID_OBJECT, CID_SXI_SPORTS_ID, 
            &un32TeamId);
        if(psGMD->hWorkingTeamId == CID_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        psGMD->hWorkingLeagueId = CID_hCreate(
            CID_POOL_INVALID_OBJECT, CID_SXI_SPORTS_ID, 
            &un32LeagueId);
        if(psGMD->hWorkingLeagueId == CID_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // Create name for this GMD instance
        snprintf(&acName[0], sizeof(acName), "SXi-GMD-Lists");

        // Allocate a memory block which can be locked to contain our
        // lists.
        psGMD->psLists = (SXI_GMD_LISTS_STRUCT *)SMSO_hCreate(
            acName, sizeof(SXI_GMD_LISTS_STRUCT), SMS_INVALID_OBJECT, TRUE);
        if(psGMD->psLists == NULL)
        {
            // Error!
            break;
        }

        // create a linked list of allocated memory
        // create a linked list of in-use memory blocks for global metadata
        eReturnCode =
            OSAL.eLinkedListCreate(
            &psGMD->psLists->hInUseBlocks,
            "SXI:LL:In-Use",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS |
            OSAL_LL_OPTION_PROTECT
            );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // create a linked list of available memory blocks for global metadata
        eReturnCode =
            OSAL.eLinkedListCreate(
            &psGMD->psLists->hAvailBlocks,
            "SXI:LL:Avail",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS |
            OSAL_LL_OPTION_PROTECT
            );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Create the list of in use linked lists for global metadata
        eReturnCode =
            OSAL.eLinkedListCreate(
            &psGMD->psLists->hInUseLists,
            "SXI:LL:In-UseLLs",
            (OSAL_LL_COMPARE_HANDLER)n16CompareHandles,
            OSAL_LL_OPTION_PROTECT
            );

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Create the list of available linked lists for global metadata
        eReturnCode =
            OSAL.eLinkedListCreate(
            &psGMD->psLists->hAvailLists,
            "SXI:LL:AvailLLs",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_PROTECT
            );

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Create ExtMetadataMonCmd control mutex
        eReturnCode = OSAL.eSemCreate(
            &psGMD->hCmdMutex,
            "SXI:ExtMetadataMonCmd control mutex",
            1, 1, OSAL_SEM_OPTION_MUTEX);

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Assign pointer to GMD handle
        hGMD = (SXI_GMD_HDL)psGMD;

    } while (FALSE);

    // Unlock the lists
    if(psGMD->psLists != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT)psGMD->psLists);
    }

    // Check if not successful
    if(hGMD == SXI_GMD_INVALID_HDL)
    {
        // Clean-up the mess
        SXI_GMD_vUninit((SXI_GMD_HDL)psGMD);
    }

    return hGMD;
}

/*******************************************************************************
*
*   SXI_GMD_vUninit
*
*******************************************************************************/
void SXI_GMD_vUninit (
    SXI_GMD_HDL hGMD
    )
{
    BOOLEAN bLocked;
    SXI_GMD_STRUCT *psGMD = (SXI_GMD_STRUCT *)hGMD;

    // Re-initialize ext metadata monitor items (Global Metadata)
    psGMD->un64ExtMetadataGMIMask = SXIAPI_GMI_MASK_NONE;

    // Lock the lists
    bLocked = SMSO_bLock((SMS_OBJECT)psGMD->psLists, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // destroy ExtMetadataMonCmd control mutex
        if(psGMD->hCmdMutex != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            OSAL_OBJECT_HDL hMutex;

            eReturnCode = OSAL.eSemTake(psGMD->hCmdMutex, OSAL_OBJ_TIMEOUT_INFINITE);
            if(eReturnCode == OSAL_SUCCESS)
            {
                hMutex = psGMD->hCmdMutex;
                psGMD->hCmdMutex = OSAL_INVALID_OBJECT_HDL;
                eReturnCode = OSAL.eSemDelete(hMutex);
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, "Failed to take GMD"
                    " command mutex (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }

        // remove and free all elements of "available" LL
        if(psGMD->psLists->hAvailBlocks != OSAL_INVALID_OBJECT_HDL)
        {
            // Remove all entries from the list and destroy each content entry

            // Since these are pre-allocated LL entries, we can't use eRemoveAll
            // and free the entry in the RemoveAll callback because that would
            // end up freeing the LL portion of the memory and then eRemoveAll
            // gets confused. So do it the old fashioned way.
            OSAL_LINKED_LIST_ENTRY hEntry;
            SXIAPI_EXT_METADATA_STRUCT *psStruct;

            do
            {
                hEntry = OSAL.hLinkedListFirst(psGMD->psLists->hAvailBlocks,
                    (void**)&psStruct);
                if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
                {
                    OSAL.eLinkedListRemove(hEntry);
                    OSAL.vLinkedListMemoryFree(psStruct);
                }
            } while (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY);

            // Destroy the list itself
            OSAL.eLinkedListDelete(psGMD->psLists->hAvailBlocks);
            psGMD->psLists->hAvailBlocks = OSAL_INVALID_OBJECT_HDL;
        }

        // remove and free all elements of "in-use" LL
        if(psGMD->psLists->hInUseBlocks != OSAL_INVALID_OBJECT_HDL)
        {
            // Remove all entries from the list and destroy each content entry
            // since these are pre-allocated LL entries, we can't use eRemoveAll
            // and free the entry in the RemoveAll callback because that would
            // end up freeing the LL portion of the memory and then eRemoveAll
            // gets confused. So do it the old fashioned way.

            OSAL_LINKED_LIST_ENTRY hEntry;
            SXIAPI_EXT_METADATA_STRUCT *psStruct;

            do
            {
                hEntry = OSAL.hLinkedListFirst(
                    psGMD->psLists->hInUseBlocks, (void**)&psStruct);
                if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
                {
                    OSAL.eLinkedListRemove(hEntry);
                    OSAL.vLinkedListMemoryFree(psStruct);
                }
            } while (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY);

            // Destroy the list itself
            OSAL.eLinkedListDelete(psGMD->psLists->hInUseBlocks);
            psGMD->psLists->hInUseBlocks = OSAL_INVALID_OBJECT_HDL;
        }

        // remove and free all elements of "available LLs" LL
        if(psGMD->psLists->hAvailLists != OSAL_INVALID_OBJECT_HDL)
        {
            // Remove all entries from the list and destroy each entry
            OSAL.eLinkedListRemoveAll(
                psGMD->psLists->hAvailLists,
                (OSAL_LL_RELEASE_HANDLER)vFreeLLs
                );

            // Destroy the list itself
            OSAL.eLinkedListDelete(psGMD->psLists->hAvailLists);
            psGMD->psLists->hAvailLists = OSAL_INVALID_OBJECT_HDL;
        }

        // remove and free all elements of "in-use LLs" LL
        if(psGMD->psLists->hInUseLists != OSAL_INVALID_OBJECT_HDL)
        {
            // Remove all entries from the list and destroy each entry
            OSAL.eLinkedListRemoveAll(
                psGMD->psLists->hInUseLists,
                (OSAL_LL_RELEASE_HANDLER)vFreeLLs
                );

            // Destroy the list itself
            OSAL.eLinkedListDelete(psGMD->psLists->hInUseLists);
            psGMD->psLists->hInUseLists = OSAL_INVALID_OBJECT_HDL;
        }

        // Destroy list object
        SMSO_vDestroy((SMS_OBJECT)psGMD->psLists);
        psGMD->psLists = NULL;

        // Destroy working CIDs
        if(psGMD->hWorkingLeagueId != CID_INVALID_OBJECT)
        {
            CID_vDestroy(psGMD->hWorkingLeagueId);
            psGMD->hWorkingLeagueId = CID_INVALID_OBJECT;
        }
        if(psGMD->hWorkingTeamId != CID_INVALID_OBJECT)
        {
            CID_vDestroy(psGMD->hWorkingTeamId);
            psGMD->hWorkingTeamId = CID_INVALID_OBJECT;
        }
    }

    // Destroy object
    SMSO_vDestroy((SMS_OBJECT)hGMD);

    return;
}

/*****************************************************************************
*
*   SXI_GMD_bDecodeExtGlobalMetadata
*
*****************************************************************************/
BOOLEAN SXI_GMD_bDecodeExtGlobalMetadata (
    STI_HDL hControlCxn,
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_EXT_GLOBAL_METADATAIND_STRUCT *psGlobalMetaData
    )
{
    BOOLEAN bWritten = TRUE, bDone = FALSE, bOk;
    UN16 un16GMI;
    SXI_GMD_STRUCT *psGMD;

    // Given the STI-Connection Handle, extract the GMD-handle
    psGMD = (SXI_GMD_STRUCT *)SXI_hGMD(hControlCxn);

    // Check input
    bOk = SMSO_bValid((SMS_OBJECT)psGMD);
    if(bOk == FALSE)
    {
        return FALSE;
    }

    // Initialize global metadata
    OSAL.bMemSet(psGlobalMetaData, 0, sizeof(*psGlobalMetaData));

    // initialize these values explicitly
    psGlobalMetaData->sSongTagServiceStruct.un32ValidFieldMask =
        SXIAPI_VALID_TG_FIELD_NONE;

    psGlobalMetaData->sTrafficStruct.un32ValidFieldMask =
        SXIAPI_VALID_GMI_TW_NONE;

    psGlobalMetaData->sSportsStruct.un32ValidFieldMask =
        SXIAPI_VALID_GMI_SP_NONE;

    psGlobalMetaData->sBankStruct.un32ValueFieldMask =
        SXIAPI_VALID_GMI_FF_BANK_NONE;

    // we got some global metadata
    // now let's figure out what it all means

    // find out how many fields we got
    bWritten &= SXIAPI_bReadUint8(
        hPayload, &psGlobalMetaData->un8Fields);

    if (bWritten == TRUE)
    {
        SXI_GMD_LISTS_STRUCT *psLists =
            psGMD->psLists;

        // go through the fields one by one
        while ((psGlobalMetaData->un8Fields > 0) && (bDone == FALSE))
        {
            // Read the next GMI
            bWritten &= SXIAPI_bPeekUint16(hPayload, &un16GMI, 0);

            // Read the data according to which GMI we're dealing with
            switch(un16GMI)
            {
            case SXIAPI_GMI_TG_ITUNES_URL:
                {
                    // consume the GMI
                    bWritten &= SXIAPI_bReadUint16(hPayload, &un16GMI);

                    // we got the traffic weather table version
                    bWritten &= SXIAPI_bReadString(
                        hPayload,
                        &psGlobalMetaData->sSongTagServiceStruct.acITunesURL[0],
                        sizeof(psGlobalMetaData->sSongTagServiceStruct.acITunesURL));
                    psGlobalMetaData->sSongTagServiceStruct.un32ValidFieldMask |=
                        SXIAPI_VALID_TG_FIELD_ITUNES_URL;

                    psGlobalMetaData->un8Fields--;
                }
                break;

            case SXIAPI_GMI_TW_TBL_VER:
                {
                    // consume the GMI
                    bWritten &= SXIAPI_bReadUint16(hPayload, &un16GMI);

                    // we got the traffic weather table version
                    bWritten &=
                        SXIAPI_bReadUint16(hPayload,
                        &psGlobalMetaData->sTrafficStruct.un16TableVersion);
                    psGlobalMetaData->sTrafficStruct.un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_TW_TABLE_VER;
                    psGlobalMetaData->un8Fields--;
                }
                break;

            case SXIAPI_GMI_TW_CITY_CNT:
                {
                    // consume the GMI
                    bWritten &= SXIAPI_bReadUint16(hPayload, &un16GMI);

                    // we got the traffic weather city count
                    bWritten &=
                        SXIAPI_bReadUint16(hPayload,
                        &psGlobalMetaData->sTrafficStruct.un16Count);
                    psGlobalMetaData->sTrafficStruct.un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_TW_COUNT;
                    psGlobalMetaData->un8Fields--;
                }
                break;

            case SXIAPI_GMI_TW_CITY_ID:
                {
                    // we got a city id. we won't consume it.
                    // the rest of the payload are cities. we won't parse them here.
                    printf(OBJECT_NAME": Handle multiple TW\n");
                    psGlobalMetaData->sTrafficStruct.un8Fields = psGlobalMetaData->un8Fields;
                    bWritten = bHandleMultipleGlobalTWEntries(
                        psLists,
                        hPayload,
                        &psGlobalMetaData->sTrafficStruct
                        );

                    // set aux data for message release function
                    psGlobalMetaData->pvReleaseArg = (void*)psLists;

                    psGlobalMetaData->sTrafficStruct.un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_TW_CITY_ID;
                    // the rest of the payload are cities so after we've handled
                    // markets we're done with the payload
                    bDone = TRUE;
                }
                break;

            case SXIAPI_GMI_SP_TEAM_TABLE_VER:
                {
                    // consume the GMI
                    bWritten &= SXIAPI_bReadUint16(hPayload, &un16GMI);

                    // we got the sports team table version
                    bWritten &=
                        SXIAPI_bReadUint16(hPayload,
                        &psGlobalMetaData->sSportsStruct.un16Version);
                    psGlobalMetaData->sSportsStruct.un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_TEAM_TABLE_VER;
                    psGlobalMetaData->un8Fields--;
                }
                break;

            case SXIAPI_GMI_SP_TEAM_CNT:
                {
                    // consume the GMI
                    bWritten &= SXIAPI_bReadUint16(hPayload, &un16GMI);

                    // we got the sports team count
                    bWritten &=
                        SXIAPI_bReadUint16(hPayload,
                        &psGlobalMetaData->sSportsStruct.un16Count);
                    psGlobalMetaData->sSportsStruct.un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_TEAM_CNT;
                    psGlobalMetaData->un8Fields--;
                }
                break;

            case SXIAPI_GMI_SP_LEAGUE_TABLE_VER:
                {
                    // consume the GMI
                    bWritten &= SXIAPI_bReadUint16(hPayload, &un16GMI);

                    // we got the sports league table version
                    bWritten &=
                        SXIAPI_bReadUint16(hPayload,
                        &psGlobalMetaData->sSportsStruct.un16Version);
                    psGlobalMetaData->sSportsStruct.un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_LEAGUE_TABLE_VER;
                    psGlobalMetaData->un8Fields--;
                }
                break;

            case SXIAPI_GMI_SP_LEAGUE_CNT:
                {
                    // consume the GMI
                    bWritten &= SXIAPI_bReadUint16(hPayload, &un16GMI);

                    // we got the sports league count
                    bWritten &=
                        SXIAPI_bReadUint16(hPayload,
                        &psGlobalMetaData->sSportsStruct.un16Count);
                    psGlobalMetaData->sSportsStruct.un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_LEAGUE_CNT;
                    psGlobalMetaData->un8Fields--;
                }
                break;

            case SXIAPI_GMI_SP_LEAGUE_ID:
                {
                    // we got a league id. we won't consume it.
                    // the rest of the payload is leagues. we won't parse them here.
                    printf(OBJECT_NAME": Handle multiple LEAGUE\n");
                    psGlobalMetaData->sSportsStruct.un8Fields = psGlobalMetaData->un8Fields;
                    bWritten = bHandleMultipleGlobalLEAGUEEntries(
                        psLists,
                        hPayload,
                        &psGlobalMetaData->sSportsStruct
                        );

                    // set aux data for message release function
                    psGlobalMetaData->pvReleaseArg = (void*)psLists;

                    psGlobalMetaData->sSportsStruct.un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_LEAGUE_ID;
                    // the rest of the payload are league's so after we've handled
                    // leagues we're done with the payload
                    bDone = TRUE;
                }
                break;

            case SXIAPI_GMI_SP_TEAM_ID:
                {
                    // we got a team id. we won't consume it.
                    // the rest of the payload is teams. we won't parse them here.
                    printf(OBJECT_NAME": Handle multiple TEAM\n");
                    psGlobalMetaData->sSportsStruct.un8Fields = psGlobalMetaData->un8Fields;
                    bWritten = bHandleMultipleGlobalTEAMEntries(
                        psLists,
                        hPayload,
                        &psGlobalMetaData->sSportsStruct
                        );

                    // set aux data for message release function
                    psGlobalMetaData->pvReleaseArg = (void*)psLists;

                    psGlobalMetaData->sSportsStruct.un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_TEAM_ID;
                    // the rest of the payload are teams so after we've handled
                    // teams we're done with the payload
                    bDone = TRUE;
                }
                break;

            case SXIAPI_GMI_FF_BANK_TBL_VER:
                {
                    // consume the GMI
                    bWritten &= SXIAPI_bReadUint16(hPayload, &un16GMI);

                    // we got the featured favorites table version
                    bWritten &=
                        SXIAPI_bReadUint16(hPayload,
                        &psGlobalMetaData->sBankStruct.un16Version);

                    psGlobalMetaData->sBankStruct.un32ValueFieldMask |=
                        SXIAPI_VALID_GMI_FF_BANK_TBL_VER;

                    psGlobalMetaData->un8Fields--;
                }
                break;

            case SXIAPI_GMI_FF_BANK_CNT:
                {
                    // consume the GMI
                    bWritten &= SXIAPI_bReadUint16(hPayload, &un16GMI);

                    // we got the featured favorites count
                    bWritten &=
                        SXIAPI_bReadUint16(hPayload,
                        &psGlobalMetaData->sBankStruct.un16Count);

                    psGlobalMetaData->sBankStruct.un32ValueFieldMask |=
                        SXIAPI_VALID_GMI_FF_BANK_CNT;

                    psGlobalMetaData->un8Fields--;
                }
                break;

            case SXIAPI_GMI_FF_BANK_ORDER:
                {
                    // we got a bank order. we won't consume it.
                    // the rest of the payload is banks. we won't parse them here.
                    printf(OBJECT_NAME": Handle multiple FEATURED FAVORITES\n");
                    psGlobalMetaData->sBankStruct.un8Fields = psGlobalMetaData->un8Fields;

                    bWritten = bHandleMultipleGlobalFFEntries(
                        psLists,
                        hPayload,
                        &psGlobalMetaData->sBankStruct
                        );

                    // set aux data for message release function
                    psGlobalMetaData->pvReleaseArg = (void*)psLists;

                    psGlobalMetaData->sBankStruct.un32ValueFieldMask |=
                        SXIAPI_VALID_GMI_FF_BANK_ORDER;
                    // the rest of the payload are banks so after we've handled
                    // banks we're done with the payload
                    bDone = TRUE;
                }
                break;

            default:
                // unrecognized GMI. cannot process further, so skipping 
                // the rest of payload
                bDone = TRUE;
                printf( OBJECT_NAME": Unrecognized GMI %u\n", un16GMI );
                break;
            }
        }
    }

    return bWritten;
}

/*****************************************************************************
*
*   SXI_GMD_vReleaseDecodedExtGlobalMetadata
*
*****************************************************************************/
void SXI_GMD_vReleaseDecodedExtGlobalMetadata (
    SXIAPI_RX_UNION *puData
        )
{
    SXIAPI_EXT_GLOBAL_METADATAIND_STRUCT *psGlobalMetaData =
        &puData->sInd.uData.sGlobalMetaData;
    SXI_GMD_LISTS_STRUCT *psLists = (SXI_GMD_LISTS_STRUCT *)
        psGlobalMetaData->pvReleaseArg;

    printf(OBJECT_NAME": RELEASE HANDLER psGlobalMetaData=%p, psLists=%p\n",
           psGlobalMetaData, psLists);

    if (NULL != psLists)
    {
        // in spite the sports and traffic lists are destroyed during processing,
        // have to check all the lists in order to release memory for the case
        // when event is failed to be posted into module/decoder queue
        OSAL_OBJECT_HDL ahLLs[] =
        {
            psGlobalMetaData->sTrafficStruct.hLL,
            psGlobalMetaData->sSportsStruct.hLL,
            psGlobalMetaData->sBankStruct.hLL
        };
        size_t tIdx;

        for (tIdx = 0; tIdx < sizeof(ahLLs)/sizeof(ahLLs[0]); tIdx++)
        {
            if (OSAL_INVALID_OBJECT_HDL != ahLLs[tIdx])
            {
                BOOLEAN bLocked;

                printf(OBJECT_NAME": processing list #%u (%p)\n", tIdx, ahLLs[tIdx]);

                // try locking the lists before accessing them in order
                // to be sure that it is still valid
                bLocked = SMSO_bLock((SMS_OBJECT)psLists, OSAL_OBJ_TIMEOUT_INFINITE);
                if (TRUE == bLocked)
                {
                    OSAL.eLinkedListIterate(
                        ahLLs[tIdx],
                        (OSAL_LL_ITERATOR_HANDLER)bSxiDataDestroyIterator,
                        psLists);
                    OSAL.eLinkedListRemoveAll(ahLLs[tIdx], NULL);
                    vFreeLL(psLists, ahLLs[tIdx]);

                    // the list handles are part of SXI Message which is freed
                    // right after this function, so overwrite with
                    // OSAL_INVALID_OBJECT_HDL is not needed

                    SMSO_vUnlock((SMS_OBJECT)psLists);
                }
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   SXI_GMD_vModuleUpdate
*
*****************************************************************************/
void SXI_GMD_vModuleUpdate (
    STI_HDL hControlCxn,
    SXIAPI_EXT_GLOBAL_METADATAIND_STRUCT *psGlobalMetadata,
    BOOLEAN *pbDispatchToDecoders
    )
{
    BOOLEAN bOk;
    SXI_GMD_STRUCT *psGMD;

    // Given the STI-Connection Handle, extract the GMD-handle
    psGMD = (SXI_GMD_STRUCT *)SXI_hGMD(hControlCxn);

    // Check input
    bOk = SMSO_bValid((SMS_OBJECT)psGMD);
    if(bOk == FALSE)
    {
        return;
    }

    if (psGlobalMetadata->sSongTagServiceStruct.un32ValidFieldMask
        != SXIAPI_VALID_TG_FIELD_NONE)
    {
        // handle song tag url
        vUpdateSongTagURL(&psGlobalMetadata->sSongTagServiceStruct);
    }

    if (psGlobalMetadata->sTrafficStruct.un32ValidFieldMask
        != SXIAPI_VALID_GMI_TW_NONE)
    {
        // handle TRAFFIC/WEATHER global metadata
        vUpdateTrafficWeather(psGMD, hControlCxn,
            &psGlobalMetadata->sTrafficStruct,
            psGlobalMetadata->un8Fields);
    }

    if (psGlobalMetadata->sSportsStruct.un32ValidFieldMask
        != SXIAPI_VALID_GMI_SP_NONE)
    {
        // handle SPORTS global metadata
        vUpdateSports(psGMD, hControlCxn,
            &psGlobalMetadata->sSportsStruct,
            psGlobalMetadata->un8Fields);
    }

    if (psGlobalMetadata->sBankStruct.un32ValueFieldMask
        != SXIAPI_VALID_GMI_FF_BANK_NONE)
    {
        // TODO: This needs to be fixed!

        // We are NOT going to do this here right now.
        // We only do this if we are a DECODER.

        // Dispatch to DECODERS (Totally wrong to be handled by
        // DECODERs, this should be handled by the MODULE). Plus if
        // there are multiple DECODERs involved this will break miserably.
        // The whole GMD processing for FFs needs to be re-worked, but it
        // works for now as is.
        if(pbDispatchToDecoders != NULL)
        {
            *pbDispatchToDecoders = TRUE;
        }
    }

    return;
}

/*****************************************************************************
*
*   SXI_GMD_vDecoderUpdate
*
*****************************************************************************/
void SXI_GMD_vDecoderUpdate (
    DECODER_OBJECT hDecoder,
    STI_HDL hControlCxn,
    SXIAPI_EXT_GLOBAL_METADATAIND_STRUCT *psGlobalMetadata
    )
{
    BOOLEAN bOk;
    SXI_GMD_STRUCT *psGMD;

    // Given the STI-Connection Handle, extract the GMD-handle
    psGMD = (SXI_GMD_STRUCT *)SXI_hGMD(hControlCxn);

    // Check input
    bOk = SMSO_bValid((SMS_OBJECT)psGMD);
    if(bOk == FALSE)
    {
        return;
    }

    if (psGlobalMetadata->sBankStruct.un32ValueFieldMask
        != SXIAPI_VALID_GMI_FF_BANK_NONE)
    {
        // handle FEATURED FAVORITES metadata
        vUpdateFeaturedFavorites(hDecoder, psGMD,
            hControlCxn, &psGlobalMetadata->sBankStruct,
            psGlobalMetadata->un8Fields);
    }

    return;
}

/*******************************************************************************
*
*   SXI_GMD_bConfigure
*
*******************************************************************************/
BOOLEAN SXI_GMD_bConfigure (
    STI_HDL hControlCxn,
    UN64 un64Mask,
    BOOLEAN bEnable
    )
{
    BOOLEAN bOk, bSuccess = FALSE;
    SXI_GMD_STRUCT *psGMD;

    // Given the STI-Connection Handle, extract the GMD-handle
    psGMD = (SXI_GMD_STRUCT *)SXI_hGMD(hControlCxn);

    // Check input
    bOk = SMSO_bValid((SMS_OBJECT)psGMD);
    if(bOk == FALSE)
    {
        return FALSE;
    }

    // Need to reset GMD table, if someone asked to 
    // terminate GMD processing
    if (bEnable == FALSE)
    {
        if ((un64Mask & SXIAPI_GMI_MASK_FF_TABLE_VER) ==
            SXIAPI_GMI_MASK_FF_TABLE_VER)
        {
            // FF table processing can be interrupted
            // by the user, so need to make sure that the
            // table is in the correct state at the next start.
            psGMD->sFF.sManaged.sInfo.bUpdating = FALSE;
            psGMD->sFF.sManaged.sInfo.un16Expected = 0;
            psGMD->sFF.sManaged.sInfo.n32Version = 0;
            psGMD->sFF.sManaged.bVersionValid = FALSE;
        }

        // TW and Sports cannot be interrupted while module
        // is running, so there is no need to handle them here.
    }

    // Go do it
    bSuccess = bConfigureGlobalExtendedMetaDataMonitor (
        psGMD, hControlCxn, un64Mask, bEnable);

    if ((un64Mask & SXI_GMD_FF) != 0)
    {
        psGMD->sFF.sManaged.sInfo.bUpdating = bEnable;
    }

    return bSuccess;
}

/*******************************************************************************
 *
 *   SXI_GMD_bSetFFTableStatus
 *
 *******************************************************************************/
BOOLEAN SXI_GMD_bSetFFTableStatus (
    STI_HDL hControlCxn,
    BOOLEAN bComplete
    )
{
    BOOLEAN bSuccess;
    SXI_GMD_STRUCT *psGMD;

    // Given the STI-Connection Handle, extract the GMD-handle
    psGMD = (SXI_GMD_STRUCT *)SXI_hGMD(hControlCxn);

    bSuccess = bSetFFTableComplete(&psGMD->sFF, bComplete);

    return bSuccess;
}

/*******************************************************************************
 *
 *   SXI_GMD_bGetFFTableStatus
 *
 *******************************************************************************/
BOOLEAN SXI_GMD_bGetFFTableStatus (
    STI_HDL hControlCnx
        )
{
    BOOLEAN bComplete;
    SXI_GMD_STRUCT *psGMD;

    psGMD = (SXI_GMD_STRUCT *)SXI_hGMD(hControlCnx);
    bComplete = bIsFFTableComplete(&psGMD->sFF);

    return bComplete;
}

/*****************************************************************************
PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vUpdateTrafficWeather
*   (global metadata for defining traffic markets)
*
*****************************************************************************/
static void vUpdateTrafficWeather(
    SXI_GMD_STRUCT *psGMD,
    STI_HDL hControlCxn,
    SXIAPI_EXT_METADATA_TRAFFIC_STRUCT *psTrafficStruct,
    UN8 un8Fields
    )
{
    SXI_GMD_MANAGED_VERSION_STRUCT *psMarkets = &psGMD->sMarkets;
    BOOLEAN bNeedUpdate = psMarkets->sInfo.bUpdating;

    if ((psTrafficStruct->un32ValidFieldMask & SXIAPI_VALID_GMI_TW_COUNT)
        != SXIAPI_VALID_GMI_TW_NONE)
    {
        // save the count so we know how many markets we need to see in order
        // to confirm that we've seen them all
        psMarkets->sInfo.un16Expected = psTrafficStruct->un16Count;
    }

    // Did we get the traffic table version number?
    if ((psTrafficStruct->un32ValidFieldMask & SXIAPI_VALID_GMI_TW_TABLE_VER)
        != SXIAPI_VALID_GMI_TW_NONE)
    {
        BOOLEAN bNewTable;

        bNewTable = bIsNewTable(hGetMarketsVersionTag,
            psMarkets, psTrafficStruct->un16TableVersion);

        if (TRUE == bNewTable)
        {
            // Clear out all old table related data
            REPORT_vClear();

            vSaveTableVersion(hGetMarketsVersionTag,
                psMarkets, psTrafficStruct->un16TableVersion);

            bNeedUpdate = TRUE;
        }
        else
        {
            bNeedUpdate = (FALSE == REPORT.bContentComplete()) ? TRUE : FALSE;
        }
    }

    if ((psTrafficStruct->un32ValidFieldMask & SXIAPI_VALID_GMI_TW_CITY_ID)
        == SXIAPI_VALID_GMI_TW_CITY_ID)
    {
        SXI_GMD_LISTS_STRUCT const * psLists = psGMD->psLists;

        while (TRUE == bNeedUpdate)
        {
            BOOLEAN bOk;
            SXIAPI_EXT_METADATA_MARKET_STRUCT sTrafficStruct;

            // get the next market in our list
            bOk = bGetNextMarket(psLists, psTrafficStruct->hLL, &sTrafficStruct);
            if (FALSE == bOk)
            {
                break;
            }

            bOk = bUpdateTWData(&sTrafficStruct);
            if (TRUE == bOk)
            {
                bNeedUpdate = bIsMoreTWDataNeeded(
                    psMarkets->sInfo.un16Expected);
            }
        }

        vFreeLL(psLists, psTrafficStruct->hLL);
        // avoid destroying the list twice
        psTrafficStruct->hLL = OSAL_INVALID_OBJECT_HDL;
    }

    if (bNeedUpdate != psMarkets->sInfo.bUpdating)
    {
        BOOLEAN bSuccess;

        bSuccess = bConfigureGlobalExtendedMetaDataMonitor(
            psGMD, hControlCxn, SXI_GMD_MARKETS, bNeedUpdate);

        if (TRUE == bSuccess)
        {
            psMarkets->sInfo.bUpdating = bNeedUpdate;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Cannot modify markets data monitors");
        }
    }

    return;
}

/*****************************************************************************
*
*   bUpdateTWData
*
*****************************************************************************/
static BOOLEAN bUpdateTWData (
    SXIAPI_EXT_METADATA_MARKET_STRUCT *psTrafficStruct
    )
{
    BOOLEAN bUpdated = FALSE;
    CID_OBJECT hId;
    UN32 un32TrafficId = (UN32)psTrafficStruct->un8CityId;

    // Create a CID from the provided ID
    hId = CID_hCreateConst(
        CID_SXI_MARKET_ID, (void *)&un32TrafficId);

    if (CID_INVALID_OBJECT != hId)
    {
        MARKET_STRUCT sMarket;

        sMarket.pacId = &psTrafficStruct->acAbbrev[0];
        sMarket.pacName = &psTrafficStruct->acName[0];

        bUpdated = REPORT_bAdd(hId, &sMarket, TRUE);
        if (FALSE == bUpdated)
        {
            CID_vDestroy(hId);
        }
    }

    return bUpdated;
}

/*****************************************************************************
*
*   bIsMoreTWDataNeeded
*
*****************************************************************************/
static BOOLEAN bIsMoreTWDataNeeded (
    UN16 un16Expected
    )
{
    BOOLEAN bNeedMoreData = TRUE;
    UN16 un16Rxd;

    un16Rxd = REPORT_un16CountConfirmedMarkets();

    printf(OBJECT_NAME": %u of %u markets have been received\n",
        un16Rxd, un16Expected);

    // check if this is the last market we were looking for
    if (un16Rxd >= un16Expected)
    {
        // yes, all markets have been seen, so confirm...
        REPORT_vAllMarketsFound();

        // we're done
        bNeedMoreData = FALSE;
    }

    return bNeedMoreData;
}

/*****************************************************************************
*
*   vUpdateSports
*
*****************************************************************************/
static void vUpdateSports(
    SXI_GMD_STRUCT *psGMD,
    STI_HDL hControlCxn,
    SXIAPI_EXT_METADATA_SPORTS_STRUCT *psSportsStruct,
    UN8 un8Fields
    )
{
    // Leagues
    if ((psSportsStruct->un32ValidFieldMask & SXIAPI_VALID_GMI_SP_LEAGUE_INFO)
        != SXIAPI_VALID_GMI_SP_NONE)
    {
        vUpdateLeagues(psGMD, hControlCxn, psSportsStruct, un8Fields);
    }

    // Teams
    else if ((psSportsStruct->un32ValidFieldMask & SXIAPI_VALID_GMI_SP_TEAM_INFO)
        != SXIAPI_VALID_GMI_SP_NONE)
    {
        vUpdateTeams(psGMD, hControlCxn, psSportsStruct, un8Fields);
    }

    return;
}

/*****************************************************************************
*
*   vUpdateLeagues
*
*****************************************************************************/
static void vUpdateLeagues(
    SXI_GMD_STRUCT *psGMD,
    STI_HDL hControlCxn,
    SXIAPI_EXT_METADATA_SPORTS_STRUCT *psSportsStruct,
    UN8 un8Fields
    )
{
    SXI_GMD_TYPE_STRUCT *psLeagues = &psGMD->sLeagues;
    BOOLEAN bNeedUpdate = psLeagues->bUpdating;

    // League table version
    if ((psSportsStruct->un32ValidFieldMask & 
        SXIAPI_VALID_GMI_SP_LEAGUE_TABLE_VER)
        != SXIAPI_VALID_GMI_SP_NONE)
    {
        psLeagues->n32Version = (N32)psSportsStruct->un16Version;

        printf(OBJECT_NAME": Current league table version is %d.\n",
            psLeagues->n32Version);
    }

    // League table version length
    if ((psSportsStruct->un32ValidFieldMask & 
        SXIAPI_VALID_GMI_SP_LEAGUE_CNT)
        != SXIAPI_VALID_GMI_SP_NONE)
    {
        UN16 un16Rxd = 0;
        psLeagues->un16Expected = psSportsStruct->un16Count;

        // see if we already have all the leagues for this table version
        bNeedUpdate = bIsMoreLeaguesDataNeeded(
            psLeagues->n32Version, psLeagues->un16Expected, &un16Rxd);

        printf(OBJECT_NAME": Current league table version %d has %u "
            "leagues, and we have %u leagues so we do %sneed an update.\n",
            psLeagues->n32Version, psLeagues->un16Expected, un16Rxd,
            bNeedUpdate ? "" : "not ");
    }

    // Leagues
    if ((psSportsStruct->un32ValidFieldMask & 
        SXIAPI_VALID_GMI_SP_LEAGUE_ID)
        != SXIAPI_VALID_GMI_SP_NONE)
    {
        BOOLEAN bAvailable;
        SXI_GMD_LISTS_STRUCT const * psLists = psGMD->psLists;
        UN16 un16NumProcessed = 0, un16NumAddedReplaced = 0;

        do
        {
            SXIAPI_EXT_METADATA_SPORTS_LEAGUE_STRUCT sLeague;

            bAvailable = bGetNextLeague(
                psLists, psSportsStruct->hLL, &sLeague);
            if (TRUE == bAvailable)
            {
                SPORTS_ENUM eSport;
                LEAGUE_ENUM eLeague;
                LEAGUE_OBJECT hLeague;
                UN32 un32LeagueId = sLeague.un16LeagueId;

                un16NumProcessed++;

                eSport = RADIO_eMapSport(
                    sLeague.acLeagueType);

                eLeague = RADIO_eMapLeague(
                    (UN32)sLeague.un16LeagueId);

                CID_bModify(&psGMD->hWorkingLeagueId,
                    &un32LeagueId);

                hLeague = LEAGUE_hAdd(
                    eSport, 
                    eLeague,
                    psGMD->hWorkingLeagueId, 
                    psLeagues->n32Version,
                    sLeague.acLeagueLongName,
                    sLeague.acLeagueShortName,
                    sLeague.bSportsFlashEnabled,
                    sLeague.un8SportsFlashTiers
                    );
                if(hLeague != LEAGUE_INVALID_OBJECT)
                {
                    un16NumAddedReplaced++;
                }
            }

        } while (TRUE == bAvailable);

        printf(OBJECT_NAME": Processed %u leagues, and add/replaced"
            " %u leagues.\n",
            un16NumProcessed, un16NumAddedReplaced);

        vFreeLL(psLists, psSportsStruct->hLL);
        // avoid destroying the list twice
        psSportsStruct->hLL = OSAL_INVALID_OBJECT_HDL;

        bNeedUpdate = bIsMoreLeaguesDataNeeded(
            psLeagues->n32Version, psLeagues->un16Expected, NULL);
    }

    // After the previous update, do we still expect more?
    if (bNeedUpdate != psLeagues->bUpdating)
    {
        BOOLEAN bSuccess;

        // Turn on/off leagues
        bSuccess = bConfigureGlobalExtendedMetaDataMonitor(
            psGMD, hControlCxn, SXI_GMD_LEAGUES, bNeedUpdate);
        if (TRUE == bSuccess)
        {
            // Leagues turned on/off
            psLeagues->bUpdating = bNeedUpdate;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Cannot modify leagues data monitors");
        }
    }

    return;
}

/*****************************************************************************
*
*   bIsMoreLeaguesDataNeeded
*
*****************************************************************************/
static BOOLEAN bIsMoreLeaguesDataNeeded (
    N32 n32Version,
    UN16 un16Expected,
    UN16 *pun16Rxd
    )
{
    BOOLEAN bNeedMoreData = TRUE;
    UN32 un32Rxd, un32Total;

    un32Rxd = LEAGUE_un32Count(n32Version, &un32Total, un16Expected);

    printf(OBJECT_NAME": %u of %u leagues with version %d have been "
        "received. We have %u total leagues.\n",
        un32Rxd, un16Expected, n32Version, un32Total);

    if (un32Rxd == (UN32)un16Expected)
    {
        BOOLEAN bSaved;

        // Nothing else required, save'em!
        bSaved = LEAGUE_bSave(n32Version);

        // we're done
        bNeedMoreData = !bSaved;
    }

    if(pun16Rxd != NULL)
    {
        *pun16Rxd = (UN16)un32Rxd;
    }

    return bNeedMoreData;
}

/*****************************************************************************
*
*   vUpdateTeams
*
*****************************************************************************/
static void vUpdateTeams(
    SXI_GMD_STRUCT *psGMD,
    STI_HDL hControlCxn,
    SXIAPI_EXT_METADATA_SPORTS_STRUCT *psSportsStruct,
    UN8 un8Fields
    )
{
    SXI_GMD_TYPE_STRUCT *psTeams = &psGMD->sTeams;
    BOOLEAN bNeedUpdate = psTeams->bUpdating;

    // Team Table Version
    if ((psSportsStruct->un32ValidFieldMask & 
        SXIAPI_VALID_GMI_SP_TEAM_TABLE_VER)
        != SXIAPI_VALID_GMI_SP_NONE)
    {
        psTeams->n32Version = (N32)psSportsStruct->un16Version;

        printf(OBJECT_NAME": Current team table version is %d.\n",
            psTeams->n32Version);
    }

    // Team Table Count
    if ((psSportsStruct->un32ValidFieldMask & 
        SXIAPI_VALID_GMI_SP_TEAM_CNT)
        != SXIAPI_VALID_GMI_SP_NONE)
    {
        UN16 un16Rxd = 0;
        psTeams->un16Expected = psSportsStruct->un16Count;

        // see if we already have all the teams for this table version
        bNeedUpdate = bIsMoreTeamsDataNeeded(
            psTeams->n32Version, psTeams->un16Expected, &un16Rxd);

        printf(OBJECT_NAME": Current team table version %d has %u "
            "teams, and we have %u teams so we do %sneed an update.\n",
            psTeams->n32Version, psTeams->un16Expected, un16Rxd,
            bNeedUpdate ? "" : "not ");
    }

    // Team Ids
    if ((psSportsStruct->un32ValidFieldMask & 
        SXIAPI_VALID_GMI_SP_TEAM_ID)
        != SXIAPI_VALID_GMI_SP_NONE)
    {
        SXI_GMD_LISTS_STRUCT const * psLists = psGMD->psLists;
        UN16 un16NumProcessed = 0, un16NumAddedReplaced = 0;
        BOOLEAN bAvailable;

        do
        {
            SXIAPI_EXT_METADATA_SPORTS_TEAM_STRUCT sTeam;

            bAvailable = bGetNextTeam(psLists, psSportsStruct->hLL, &sTeam);
            if (TRUE == bAvailable)
            {
                TEAM_OBJECT hTeam;
                UN32 un32TeamId = sTeam.un16TeamId;

                un16NumProcessed++;

                CID_bModify(&psGMD->hWorkingTeamId, &un32TeamId);

                printf(OBJECT_NAME": Processing team %s %s(%s) id#%u, "
                    "#leagues=%u, #tiers=%u - v%d\n",
                    sTeam.acTeamName, sTeam.acTeamNickname,
                    sTeam.acTeamAbbrev, sTeam.un16TeamId, sTeam.tNumLeagues,
                    sTeam.tNumTiers,
                    psTeams->n32Version);

                // Add/Replace this team
                hTeam = TEAM_hAdd(psGMD->hWorkingTeamId, 
                    psTeams->n32Version,
                    sTeam.acTeamName, sTeam.acTeamNickname,
                    sTeam.acTeamAbbrev, sTeam.tNumLeagues, 
                    sTeam.aun8TeamLeagueList,
                    sTeam.tNumTiers,
                    sTeam.aun8TeamTierList);
                if(hTeam != TEAM_INVALID_OBJECT)
                {
                    un16NumAddedReplaced++;
                }
            }

        } while (TRUE == bAvailable);

        printf(OBJECT_NAME": Processed %u teams, and add/replaced"
            " %u teams.\n",
            un16NumProcessed, un16NumAddedReplaced);

        vFreeLL(psLists, psSportsStruct->hLL);
        // avoid destroying the list twice
        psSportsStruct->hLL = OSAL_INVALID_OBJECT_HDL;

        bNeedUpdate = bIsMoreTeamsDataNeeded(
            psTeams->n32Version, psTeams->un16Expected, NULL);
    }

    // After the previous update, do we still expect more?
    if (bNeedUpdate != psTeams->bUpdating)
    {
        BOOLEAN bSuccess;

        // Turn on/off team info
        bSuccess = bConfigureGlobalExtendedMetaDataMonitor(
            psGMD, hControlCxn, SXI_GMD_TEAMS, bNeedUpdate);
        if (TRUE == bSuccess)
        {
            // Teams turned on/off.
            psTeams->bUpdating = bNeedUpdate;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Cannot modify teams data monitors");
        }
    }

    return;
}

/*****************************************************************************
*
*   bIsMoreTeamsDataNeeded
*
*****************************************************************************/
static BOOLEAN bIsMoreTeamsDataNeeded (
    N32 n32Version,
    UN16 un16Expected,
    UN16 *pun16Rxd
    )
{
    BOOLEAN bNeedMoreData = TRUE;
    UN32 un32Rxd, un32Total;

    un32Rxd = TEAM_un32Count(n32Version, &un32Total, un16Expected);

    printf(OBJECT_NAME": %u of %u teams with version %d have been "
        "received. We have %u total teams.\n",
        un32Rxd, un16Expected, n32Version, un32Total);

    if (un32Rxd == (UN32)un16Expected)
    {
        BOOLEAN bSaved;

        // Nothing else required, save'em!
        bSaved = TEAM_bSave(n32Version);

        // we're done
        bNeedMoreData = !bSaved;
    }

    if(pun16Rxd != NULL)
    {
        *pun16Rxd = (UN16)un32Rxd;
    }

    return bNeedMoreData;
}

/*****************************************************************************
*
*   vUpdateFeaturedFavorites
*
*****************************************************************************/
static void vUpdateFeaturedFavorites(
    DECODER_OBJECT hDecoder,
    SXI_GMD_STRUCT *psGMD,
    STI_HDL hControlCxn,
    SXIAPI_EXT_METADATA_FEATURED_FAVORITES_STRUCT *psFeaturedFavorites,
    UN8 un8Fields
    )
{
    PRESETS_OBJECT hPresets = PRESETS_INVALID_OBJECT;
    SXI_GMD_FF_STRUCT *psFF = &psGMD->sFF;
    BOOLEAN bNeedUpdate = psFF->sManaged.sInfo.bUpdating;

    // Did we get the bank count?
    if ((psFeaturedFavorites->un32ValueFieldMask
        & SXIAPI_VALID_GMI_FF_BANK_CNT)
        != SXIAPI_VALID_GMI_FF_BANK_NONE)
    {
        // save the count so we know how many banks we need to see in order
        // to confirm that we've seen them all
        psFF->sManaged.sInfo.un16Expected = psFeaturedFavorites->un16Count;
    }

    // Did we get the bank table version number
    if ((psFeaturedFavorites->un32ValueFieldMask
        & SXIAPI_VALID_GMI_FF_BANK_TBL_VER)
        != SXIAPI_VALID_GMI_FF_BANK_NONE)
    {
        BOOLEAN bNewTable;
        hPresets = DECODER_hPresets(hDecoder);

        bNewTable = bIsNewTable(hGetFFVersionTag,
            &psFF->sManaged, psFeaturedFavorites->un16Version);

        if (TRUE == bNewTable)
        {
            bNeedUpdate = bInitFFTable(
                hPresets, psFF, psFeaturedFavorites->un16Version);
        }
        else
        {
            bNeedUpdate = bContinueFFTable(
                hPresets, psFF);
        }
    }

    if ((psFeaturedFavorites->un32ValueFieldMask
        & SXIAPI_VALID_GMI_FF_BANK_ORDER) ==
        SXIAPI_VALID_GMI_FF_BANK_ORDER)
    {
        FAVORITES_RECEIVER_INFO_STRUCT sInfo;
        SXI_GMD_BANK_ITERATOR_STRUCT sData;

        if (PRESETS_INVALID_OBJECT == hPresets)
        {
            hPresets = DECODER_hPresets(hDecoder);
        }

        PRESETS_bGetCapabilities(hPresets, &sInfo);

        sData.hPresets = hPresets;
        sData.psFF = psFF;
        sData.psInfo = &sInfo;
        sData.bNeedUpdate = bNeedUpdate;

        // in order to let this shared data to be processed
        // by several recipients, use non-destructive way
        OSAL.eLinkedListIterate(
            psFeaturedFavorites->hLL,
            (OSAL_LL_ITERATOR_HANDLER)bProcessBankIterator,
            &sData);

        bNeedUpdate = sData.bNeedUpdate;
    }

    if (bNeedUpdate != psFF->sManaged.sInfo.bUpdating)
    {
        BOOLEAN bSuccess;

        bSuccess = bConfigureGlobalExtendedMetaDataMonitor(
            psGMD, hControlCxn, SXI_GMD_FF, bNeedUpdate);

        if (TRUE == bSuccess)
        {
            psFF->sManaged.sInfo.bUpdating = bNeedUpdate;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Cannot modify FF data monitors");
        }
    }

    return;
}

/*****************************************************************************
*
*   bInitFFTable
*
*****************************************************************************/
static BOOLEAN bInitFFTable (
    PRESETS_OBJECT hPresets,
    SXI_GMD_FF_STRUCT *psFF,
    UN16 un16Version
    )
{
    BOOLEAN bOk;

    // Clear out all old table related data
    bOk = PRESETS_bSetAllFeaturedBanksRemovedFlag(hPresets, TRUE);
    if (FALSE == bOk)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            OBJECT_NAME": Cannot delete obsolete Featured Bands");

        return FALSE;
    }

    bOk = bSetFFTableComplete(psFF, FALSE);
    if (TRUE == bOk)
    {
        // update table version if only the complete flag has been saved
        // in order to prevent possible data inconsistency
        vSaveTableVersion(hGetFFVersionTag, &psFF->sManaged, un16Version);
    }
    else
    {
        // FF banks can be received still -- not a fatal error
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            OBJECT_NAME": Cannot mark FF table as incomplete");
    }

    return TRUE;
}

/*****************************************************************************
*
*   bContinueFFTable
*
*****************************************************************************/
static BOOLEAN bContinueFFTable(
    PRESETS_OBJECT hPresets,
    SXI_GMD_FF_STRUCT *psFF
    )
{
    BOOLEAN bComplete, bNeedUpdate = FALSE;

    bComplete = bIsFFTableComplete(psFF);
    if (bComplete == FALSE)
    {   
        // Mark old bands as 'removed'
        bNeedUpdate = PRESETS_bSetAllFeaturedBanksRemovedFlag(hPresets, TRUE);
        if (FALSE == bNeedUpdate)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Failed to set FF removed flag");
        }
    }

    return bNeedUpdate;
}

/*****************************************************************************
*
*   bUpdateFFData
*
*****************************************************************************/
static BOOLEAN bUpdateFFData (
    SXIAPI_EXT_METADATA_BANK_STRUCT *psBank,
    PRESETS_OBJECT hPresets,
    FAVORITES_RECEIVER_INFO_STRUCT *psInfo
    )
{
    BOOLEAN bOk = FALSE;
    PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT sIteratorStruct;
    UN8 un8Idx;
    PRESET_BAND_STATE_ENUM eState;

    // Add this bank to the unique ll
    if ((psBank->un32ValueFieldMask & SXIAPI_VALID_GMI_FF_BANK_ID) ==
        SXIAPI_VALID_GMI_FF_BANK_ID)
    {
        OSAL_RETURN_CODE_ENUM eReturn;

        eReturn = PRESETS_bAddToID(hPresets, (FAVORITES_BAND_ID)psBank->un8Id);

        bOk = (OSAL_SUCCESS == eReturn) ? TRUE : FALSE;
    }

    // Check a bank purpose
    if (((psBank->un8Purpose & psInfo->tPurpose) == FAVORITES_BAND_PURPOSE_NONE) &&
        ((psInfo->tPurpose != FAVORITES_BAND_PURPOSE_NONE) ||
        (psBank->un8Purpose != FAVORITES_BAND_PURPOSE_NONE)))
    {
        return bOk;
    }

    // Check a receiver targeting information
    sIteratorStruct.un8ArrIndex = UN8_MAX;
    for (un8Idx = 0; un8Idx < SXIAPI_MAX_BANK_ARR_CNT; un8Idx++)
    {
        if ((psBank->un16Arrangement[un8Idx][0] >= psInfo->tCapacity) &&
            (psInfo->tCapacity >= psBank->un16Arrangement[un8Idx][1]) &&
            (psBank->un8SIDCount[un8Idx] >= SXI_GMD_FF_SID_OFFSET))
        {
            // The first arrangement matches
            sIteratorStruct.un8ArrIndex = un8Idx;
            break;
        }
    }

    if (UN8_MAX == sIteratorStruct.un8ArrIndex)
    {
        // no match
        return bOk;
    }

    printf(OBJECT_NAME":FF:****************************************\n");
    printf(OBJECT_NAME":FF:BANK_ORDER:     %u\n", psBank->un8Order);
    printf(OBJECT_NAME":FF:BANK_ID:        %u\n", psBank->un8Id);
    printf(OBJECT_NAME":FF:BANK_SEQ:       %u\n", psBank->un8Sequence);
    printf(OBJECT_NAME":FF:BANK_NAME:      %s\n", &psBank->acTitleShort[0]);
    printf(OBJECT_NAME":FF:BANK_LONG_NAME: %s\n", &psBank->acTitleLong[0]);
    printf(OBJECT_NAME":FF:BANK_VERBOSE:   %s\n", &psBank->acTitleVerbose[0]);
    printf(OBJECT_NAME":FF:BANK_DESCR:     %s\n", &psBank->acDescription[0]);
    printf(OBJECT_NAME":FF:BANK_PURPOSE:   %u\n", psBank->un8Purpose);
    printf(OBJECT_NAME":FF:****************************************\n");

    // Initialize Iterator Struct
    sIteratorStruct.hBand = PRESET_BAND_INVALID_OBJECT;
    // Fill the Bank Attributes what we are looking for
    sIteratorStruct.tId = (FAVORITES_BAND_ID)psBank->un8Id;
    sIteratorStruct.tSeq = (size_t)psBank->un8Sequence;
    // Initialize Featured Favorites Bank Count
    sIteratorStruct.un32BandsCount = 0;
    sIteratorStruct.un32Removed = 0;

    // Search for Bank and compare
    eState = PRESETS_eGetPresetBandAndState (
        hPresets,
        &sIteratorStruct,
        psInfo->tBandsNumber
        );

    // Process featured bank state
    vProcessFeaturedBankState(
        hPresets,
        eState,
        psInfo->tCapacity,
        psBank,
        &sIteratorStruct
        );

    return bOk;
}

/*****************************************************************************
*
*   bIsMoreFFDataNeeded
*
*****************************************************************************/
static BOOLEAN bIsMoreFFDataNeeded (
    PRESETS_OBJECT hPresets,
    SXI_GMD_FF_STRUCT *psFF
    )
{
    BOOLEAN bNeedMoreData = TRUE;
    UN32 un32Rxd;

    un32Rxd = PRESETS_un32BankIDCount(hPresets);

    printf(OBJECT_NAME":FF: %u of %u banks have been received\n",
        un32Rxd, psFF->sManaged.sInfo.un16Expected);

    if (un32Rxd >= (UN32)psFF->sManaged.sInfo.un16Expected)
    {
        BOOLEAN bOk;

        bOk = bSetFFTableComplete(psFF, TRUE);
        if (FALSE == bOk)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                ": Failed to set FF table completion status");

            // Not so critical to leave something behing.
            // Just update it one more time then on the next caroule cycle.
        }
        else
        {
            bNeedMoreData = FALSE;
        }

        bOk = PRESETS_bResetFeaturedFavoritesId(hPresets);
        if (FALSE == bOk)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Cannot disable Featured "
                "Favorites data monitors");
        }

        // Perform clean up if required
        bOk = PRESETS_bRemoveAllSelectedFFBanks(hPresets);
        if (FALSE == bOk)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Cannot cleanup obsolete "
                "Featured Bands");
        }
    }

    return bNeedMoreData;
}

/*****************************************************************************
*
*   vUpdateSongTagURL
*
*****************************************************************************/
static void vUpdateSongTagURL(
    SXIAPI_EXT_METADATA_SONG_TAG_SERVICE_STRUCT const *psTagServiceStruct
    )
{
    // Did we get the itunes song tag url?
    if ((psTagServiceStruct->un32ValidFieldMask & SXIAPI_VALID_TG_FIELD_ITUNES_URL)
        != SXIAPI_VALID_TG_FIELD_NONE)
    {
        if (psTagServiceStruct->acITunesURL[0] != '\0')
        {
            // yes we did, go update it
            SONGTAG_vSongTagServiceURLUpdate(
                SONG_TAG_SERVICE_ITUNES,
                &psTagServiceStruct->acITunesURL[0]);
        }
    }
}

/*****************************************************************************
*
*   hGetMarketsVersionTag
*
*****************************************************************************/
static TAG_OBJECT hGetMarketsVersionTag(void)
{
    TAG_OBJECT hTag = TAG_INVALID_OBJECT;
    TAG_OBJECT hParentTag;

    hParentTag = REPORT_hGetRadioSpecificTag();
    if (TAG_INVALID_OBJECT != hParentTag)
    {
        // Get the table version number tag
        TAG_eGet(TABLE_VERSION_TAG, hParentTag, &hTag, NULL, TRUE);
    }

    return hTag;
}

/*****************************************************************************
*
*   hGetFFVersionTag
*
*****************************************************************************/
static TAG_OBJECT hGetFFVersionTag(void)
{
    TAG_OBJECT hTag;

    hTag = hFFGetSubTag(PRESETS_FEATURED_TABLE_VERSION);

    return hTag;
}

/*****************************************************************************
*
*   bIsNewTable
*
*****************************************************************************/
static BOOLEAN bIsNewTable(
    GET_TAG_FUNCTION hGetTagFunc,
    SXI_GMD_MANAGED_VERSION_STRUCT *psManaged,
    UN16 un16Version
    )
{
    while (FALSE == psManaged->bVersionValid)
    {
        TAG_OBJECT hTag;

        hTag = hGetTagFunc();
        if (TAG_INVALID_OBJECT != hTag)
        {
            // Retrieve the value
            N32 n32Value;
            N32 *pn32Value = &n32Value;
            size_t tSize = sizeof(N32);
            SMSAPI_RETURN_CODE_ENUM eResult;

            eResult = TAG_eGetTagValue(
                hTag, TAG_TYPE_INTEGER, (void **)&pn32Value, &tSize);

            if (SMSAPI_RETURN_CODE_SUCCESS == eResult)
            {
                psManaged->sInfo.n32Version = n32Value;
                psManaged->bVersionValid = TRUE;
                break;
            }
        }

        // Can't get current table number -- assume new table
        return TRUE;
    }

    return ((N32)un16Version != psManaged->sInfo.n32Version) ? TRUE : FALSE;
}

/*****************************************************************************
*
*   vSaveTableVersion
*
*****************************************************************************/
static void vSaveTableVersion(
    GET_TAG_FUNCTION hGetTagFunc,
    SXI_GMD_MANAGED_VERSION_STRUCT *psManaged,
    UN16 un16Version
    )
{
    TAG_OBJECT hTag;

    hTag = hGetTagFunc();
    if (TAG_INVALID_OBJECT != hTag)
    {
        N32 n32Value = un16Version;

        TAG_eSetTagValue(
            hTag, TAG_TYPE_INTEGER, (void *)&n32Value, sizeof(N32), FALSE);
    }

    psManaged->bVersionValid = TRUE;
    psManaged->sInfo.n32Version = (N32)un16Version;

    return;
}

/*****************************************************************************
*
* bIsFFTableComplete
*
*****************************************************************************/
static BOOLEAN bIsFFTableComplete(
    SXI_GMD_FF_STRUCT *psFF
    )
{
    while (FALSE == psFF->bCompleteFlagValid)
    {
        TAG_OBJECT hTag;

        hTag = hFFGetSubTag(PRESETS_FEATURED_TABLE_COMPLETE);
        if (TAG_INVALID_OBJECT != hTag)
        {
            // Retrieve the value
            N32 n32Value;
            N32 *pn32Value = &n32Value;
            size_t tSize = sizeof(N32);
            SMSAPI_RETURN_CODE_ENUM eResult;

            eResult = TAG_eGetTagValue(
                hTag, TAG_TYPE_INTEGER, (void **)&pn32Value, &tSize);

            if (SMSAPI_RETURN_CODE_SUCCESS == eResult)
            {
                psFF->bComplete = (0 == n32Value) ? FALSE : TRUE;
                psFF->bCompleteFlagValid = TRUE;
                break;
            }
        }

        // can't get complete flag value -- assume incomplete
        return FALSE;
    }

    return psFF->bComplete;
}

/*****************************************************************************
*
* bSetFFTableComplete
*
*****************************************************************************/
static BOOLEAN bSetFFTableComplete(
    SXI_GMD_FF_STRUCT *psFF,
    BOOLEAN bComplete
    )
{
    TAG_OBJECT hTag;

    psFF->bComplete = bComplete;
    psFF->bCompleteFlagValid = TRUE;

    hTag = hFFGetSubTag(PRESETS_FEATURED_TABLE_COMPLETE);
    if (TAG_INVALID_OBJECT != hTag)
    {
        SMSAPI_RETURN_CODE_ENUM eResult;
        N32 n32Value = (N32)bComplete;

        eResult = TAG_eSetTagValue(
            hTag, TAG_TYPE_INTEGER, (void *)&n32Value, sizeof(N32), FALSE);

        if (SMSAPI_RETURN_CODE_SUCCESS == eResult)
        {
            return TRUE;
        }
    }

    return FALSE;
}

/*******************************************************************************
*
*   hFFGetSubTag
*
*******************************************************************************/
static TAG_OBJECT hFFGetSubTag(
    const char *pacName
    )
{
    TAG_OBJECT hSMSTag;
    TAG_OBJECT hFFTag = TAG_INVALID_OBJECT;
    TAG_OBJECT hSubTag = TAG_INVALID_OBJECT;

    // Get the SMS tag. The FF tags are children of the SMS tag
    hSMSTag = SMS_hGetTag();
    TAG_eGet(PRESETS_FF_TAG, hSMSTag, &hFFTag, NULL, TRUE);

    if (TAG_INVALID_OBJECT != hFFTag)
    {
        TAG_eGet(pacName, hFFTag, &hSubTag, NULL, TRUE);
    }

    return hSubTag;
}

/*****************************************************************************
*
*   bGetNextLeague
*
*****************************************************************************/
BOOLEAN bGetNextLeague(
    SXI_GMD_LISTS_STRUCT const *psLists,
    OSAL_OBJECT_HDL hLL,
    SXIAPI_EXT_METADATA_SPORTS_LEAGUE_STRUCT *psLeague
    )
{
    OSAL_LINKED_LIST_ENTRY hEntry;
    SXIAPI_EXT_METADATA_STRUCT *psStruct;

    hEntry = OSAL.hLinkedListFirst(hLL, (void**)&psStruct);
    if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return FALSE;
    }

    // copy the info out
    *psLeague = psStruct->uData.sTeamLeagueStruct.uData.sLeagueStruct;

    // don't need anymore
    OSAL.eLinkedListRemove(hEntry);
    vFreeStruct(psLists, psStruct);

    return TRUE;
}

/*****************************************************************************
*
*   bGetNextTeam
*
*****************************************************************************/
BOOLEAN bGetNextTeam(
    SXI_GMD_LISTS_STRUCT const *psLists,
    OSAL_OBJECT_HDL hLL,
    SXIAPI_EXT_METADATA_SPORTS_TEAM_STRUCT *psTeam
    )
{
    OSAL_LINKED_LIST_ENTRY hEntry;
    SXIAPI_EXT_METADATA_STRUCT *psStruct;

    hEntry = OSAL.hLinkedListFirst(hLL, (void**)&psStruct);
    if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return FALSE;
    }

    // copy the info out
    *psTeam = psStruct->uData.sTeamLeagueStruct.uData.sTeamStruct;

    // don't need anymore
    OSAL.eLinkedListRemove(hEntry);
    vFreeStruct(psLists, psStruct);

    return TRUE;
}

/*****************************************************************************
*
*   bGetNextMarket
*
*****************************************************************************/
BOOLEAN bGetNextMarket(
    SXI_GMD_LISTS_STRUCT const *psLists,
    OSAL_OBJECT_HDL hLL,
    SXIAPI_EXT_METADATA_MARKET_STRUCT *psMarket
    )
{
    OSAL_LINKED_LIST_ENTRY hEntry;
    SXIAPI_EXT_METADATA_STRUCT *psStruct;

    hEntry = OSAL.hLinkedListFirst(hLL, (void**)&psStruct);
    if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return FALSE;
    }

    // copy the info out
    *psMarket = psStruct->uData.sTrafficStruct;

    // don't need anymore
    OSAL.eLinkedListRemove(hEntry);
    vFreeStruct(psLists, psStruct);

    return TRUE;
}

/*****************************************************************************
*
*   vProcessFeaturedBankState
*
*****************************************************************************/
static void vProcessFeaturedBankState(
    PRESETS_OBJECT hPresets,
    PRESET_BAND_STATE_ENUM eState,
    FAVORITES_BAND_CAPACITY tCapacity,
    SXIAPI_EXT_METADATA_BANK_STRUCT const *psBank,
    PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT *psIteratorStruct
    )
{
    UN8 un8ArrIndex;
    PRESET_BAND_MARKER_STRUCT sMarkerStruct;
    PRESET_BAND_PRIVATE_ATTRS_STRUCT sAttrs;

    OSAL.bMemSet(&sAttrs, 0, sizeof(sAttrs));

    if (eState == PRESET_BAND_STATE_NEW)
    {
        printf( "\t+ The Band ID: %u is new. Adding to Service\n",
            psBank->un8Id);

        sAttrs.hDescription = STRING.hCreate(
            psBank->acDescription,
            strlen(psBank->acDescription));

        sAttrs.hLongName = STRING.hCreate(
            psBank->acTitleLong,
            strlen(psBank->acTitleLong));

        sAttrs.hName = STRING.hCreate(
            psBank->acTitleShort,
            strlen(psBank->acTitleShort));

        sAttrs.hVerboseName = STRING.hCreate(
            psBank->acTitleVerbose,
            strlen(psBank->acTitleVerbose));

        sAttrs.eType = PRESET_BAND_TYPE_FEATURED;
        sAttrs.tCapacity = tCapacity;
        sAttrs.sFeatured.bRemoved = FALSE;
        sAttrs.sFeatured.tOrder = (FAVORITES_BAND_ORDER)psBank->un8Order;
        sAttrs.sFeatured.tPurpose = (FAVORITES_BAND_PURPOSE_MASK)psBank->un8Purpose;
        sAttrs.sFeatured.tId = (FAVORITES_BAND_ID)psBank->un8Id;
        sAttrs.sFeatured.tSequence = (FAVORITES_BAND_SEQUENCE)psBank->un8Sequence;

        // Get a selected bank arrangement
        un8ArrIndex = psIteratorStruct->un8ArrIndex;

        // Create a featured favorites bank
        //
        // First two elements in the Arrangement are RMIN and RMAX values
        // so actual channel SIDs are started from the third element
        //
        // Ignoring return value, since Featured Band is automatically
        // inserted into the list upon creation
        PRESET_BAND_hCreateFeaturedBand (
            hPresets,
            tCapacity,
            &sAttrs,
            (SERVICE_ID const *)
            &psBank->un16Arrangement[un8ArrIndex][SXI_GMD_FF_SID_OFFSET],
            (size_t)(psBank->un8SIDCount[un8ArrIndex] - SXI_GMD_FF_SID_OFFSET)
            );

    }
    else if (eState == PRESET_BAND_STATE_MODIFIED)
    {
        BOOLEAN bSuccess;

        printf( "\t+ The Band ID: %u was Updated. Update Channel Presets of this Bank\n",
            psBank->un8Id);

        // Get a selected bank arrangement
        un8ArrIndex = psIteratorStruct->un8ArrIndex;

        // First two elements in the Arrangement are RMIN and RMAX values
        // so actual channel SIDs are started from the third element
        bSuccess = PRESET_BAND_bUpdate(
            psIteratorStruct->hBand,
            (FAVORITES_BAND_ID)psBank->un8Id,
            (FAVORITES_BAND_ORDER)psBank->un8Order,
            (FAVORITES_BAND_SEQUENCE)psBank->un8Sequence,
            (FAVORITES_BAND_PURPOSE_MASK)psBank->un8Purpose,
            &psBank->acTitleShort[0],
            &psBank->acTitleLong[0],
            &psBank->acTitleVerbose[0],
            &psBank->acDescription[0],
            (SERVICE_ID const *)
            &psBank->un16Arrangement[un8ArrIndex][SXI_GMD_FF_SID_OFFSET],
            (size_t)(psBank->un8SIDCount[un8ArrIndex] - SXI_GMD_FF_SID_OFFSET)
            );

        if (bSuccess == FALSE)
        {
            puts(OBJECT_NAME": Failed to update Featured Band");
        }

        // Notifying subscribed client
        bSuccess = PRESETS_bUpdate(
            hPresets,
            psIteratorStruct->hBand,
            PRESETS_OBJECT_EVENT_BAND_MODIFIED
            );

        if (bSuccess == FALSE)
        {
            puts(OBJECT_NAME": Failed to notify application about Band Modification");
        }
    }
    else if (eState == PRESET_BAND_STATE_EXISTING)
    {
        printf( "\t  The Band ID: %u is not updated\n", psBank->un8Id);

        sMarkerStruct.tBankId = (FAVORITES_BAND_ID)psBank->un8Id;
        sMarkerStruct.bRemoved = FALSE;

        PRESETS_bSetSelectedFavoritesBankRemoved(
            hPresets,
            sMarkerStruct
            );
    }
    else if (eState == PRESET_BAND_STATE_REJECTED)
    {
        printf( "\t  The Band ID: %u is rejected\n", psBank->un8Id);
    }

    if( sAttrs.hName != STRING_INVALID_OBJECT )
    {
        STRING_vDestroy( sAttrs.hName );
    }
    if( sAttrs.hLongName != STRING_INVALID_OBJECT )
    {
        STRING_vDestroy( sAttrs.hLongName );
    }
    if( sAttrs.hVerboseName != STRING_INVALID_OBJECT )
    {
        STRING_vDestroy( sAttrs.hVerboseName );
    }
    if( sAttrs.hDescription != STRING_INVALID_OBJECT )
    {
        STRING_vDestroy( sAttrs.hDescription );
    }

    return;
}

/*****************************************************************************
*
*   vFreeStruct
*
* called when we are done using a SXIAPI_GLOBAL_METADATA_PAYLOAD_STRUCT.
* Some clean up to the struct is done, and it is put back on the "available"
* list, making it available for re-use
*
*****************************************************************************/
static void vFreeStruct(
    SXI_GMD_LISTS_STRUCT const *psLists,
    SXIAPI_EXT_METADATA_STRUCT *psStruct
    )
{
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    BOOLEAN bLocked;

    // Lock lists
    bLocked = SMSO_bLock((SMS_OBJECT)psLists, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // we're done using this block.
        printf(OBJECT_NAME": Returning memory back to available list.\n");

        // clear it out
        OSAL.bMemSet(&psStruct->uData, 0, sizeof(psStruct->uData));

        // remove it from the "in use" LL
        eOsalReturnCode = OSAL.eLinkedListRemove(psStruct->hEntry);
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            printf(OBJECT_NAME": Removed from in use list.\n");

            // add it to the "available" LL
            eOsalReturnCode = OSAL.eLinkedListAdd(
                psLists->hAvailBlocks,
                &hEntry,
                (void *)psStruct
                );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                printf(OBJECT_NAME": Error in adding to avail list\n");
                OSAL.vLinkedListMemoryFree(psStruct);
                psStruct = NULL;
            }
            else
            {
                printf(OBJECT_NAME": Added back to available list.\n");
                // remember where we are in the list
                psStruct->hEntry = hEntry;
            }
        }

        // Unlock lists
        SMSO_vUnlock((SMS_OBJECT)psLists);
    }

    return;
}

/*****************************************************************************
*
*   vFreeLL
*
* called when we are done using a traffic or sports LL. This LL will be taken
* out of the "in use" list of LLs and put into the "available" list of LLs,
* making it available for re-use
*
*****************************************************************************/
static void vFreeLL(
    SXI_GMD_LISTS_STRUCT const *psLists,
    OSAL_OBJECT_HDL hLL
    )
{
    BOOLEAN bLocked;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    if (hLL == OSAL_INVALID_OBJECT_HDL)
    {
        return;
    }

    // Lock lists
    bLocked = SMSO_bLock((SMS_OBJECT)psLists, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // is there a struct available for reuse?
        OSAL.eLinkedListSearch (
            psLists->hInUseLists,
            &hEntry, (void *)hLL);

        if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // yes we found the ll
            printf(OBJECT_NAME": Returning list back to the available list.\n");

            // remove from the "in use" LL
            eOsalReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eOsalReturnCode == OSAL_SUCCESS)
            {
                printf(OBJECT_NAME": Removed list from in use list.\n");

                hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                // add this block to the "avail" LL
                eOsalReturnCode = OSAL.eLinkedListAdd(
                    psLists->hAvailLists,
                    (OSAL_LINKED_LIST_ENTRY *)&hEntry,
                    (void *)hLL
                    );
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    printf(OBJECT_NAME": Error. adding LL to avail.\n");
                    OSAL.eLinkedListDelete(hLL);
                }
                else
                {
                    printf(OBJECT_NAME": Added list back to available list.\n");
                }
            }
        }
        else
        {
            printf(OBJECT_NAME": Error. Couldn't find in the in use list.\n");
            OSAL.eLinkedListDelete(hLL);
        }

        // Unlock lists
        SMSO_vUnlock((SMS_OBJECT)psLists);
    }

    return;
}

/*****************************************************************************
*
*   psGetStruct
*
* Function to get a SXIAPI_GLOBAL_METADATA_PAYLOAD_STRUCT. Either we
* reuse a previously allocated block that is available or we create a new one
*
*****************************************************************************/
static SXIAPI_EXT_METADATA_STRUCT *psGetStruct(
    SXI_GMD_LISTS_STRUCT const *psLists
    )
{
    BOOLEAN bLocked;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    SXIAPI_EXT_METADATA_STRUCT *psStruct = NULL;

    // Lock lists
    bLocked = SMSO_bLock((SMS_OBJECT)psLists, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // is there a struct available for reuse?
        hEntry = OSAL.hLinkedListFirst (
            psLists->hAvailBlocks, (void**)&psStruct);
        if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // yes we can reuse a previously allocated block

            printf(OBJECT_NAME": we can reuse a previously allocated block\n");
            // remove from the "available" LL
            eOsalReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eOsalReturnCode == OSAL_SUCCESS)
            {
                hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                // add this block to the "in use" LL
                eOsalReturnCode = OSAL.eLinkedListAdd(
                    psLists->hInUseBlocks,
                    (OSAL_LINKED_LIST_ENTRY *)&hEntry,
                    (void *)psStruct
                    );
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    printf(OBJECT_NAME": Error. adding memory to in use2.\n");
                    OSAL.vLinkedListMemoryFree(psStruct);
                    psStruct = NULL;
                }
                else
                {
                    psStruct->hEntry = hEntry;
                }
            }
        }
        else
        {
            printf(OBJECT_NAME": no available blocks. need to create one\n");

            // no available blocks. need to create one.
            // create a new block
            psStruct = (SXIAPI_EXT_METADATA_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(OBJECT_NAME,
                sizeof(SXIAPI_EXT_METADATA_STRUCT), TRUE);

            if (psStruct != NULL)
            {
                // add this block to the "in use" LL
                eOsalReturnCode = OSAL.eLinkedListAdd(
                    psLists->hInUseBlocks,
                    (OSAL_LINKED_LIST_ENTRY *)&hEntry,
                    (void *)psStruct
                    );
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    printf(OBJECT_NAME": Error. adding memory to in use..\n");
                    OSAL.vLinkedListMemoryFree(psStruct);
                    psStruct = NULL;
                }
                else
                {
                    psStruct->hEntry = hEntry;
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Failed to allcoate SXIAPI_EXT_METADATA_STRUCT");
            }
        }

        // Unlock lists
        SMSO_vUnlock((SMS_OBJECT)psLists);
    }

    return psStruct;
}

/*****************************************************************************
*
*   hGetLL
*
* Function to get a LL. Either we
* reuse a previously allocated list that is available or we create a new one
*
*****************************************************************************/
static OSAL_OBJECT_HDL hGetLL(
    SXI_GMD_LISTS_STRUCT const *psLists
    )
{
    BOOLEAN bLocked;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_OBJECT_HDL hLL = OSAL_INVALID_OBJECT_HDL;

    // Lock lists
    bLocked = SMSO_bLock((SMS_OBJECT)psLists, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // is there a struct available for reuse?
        hEntry = OSAL.hLinkedListFirst (
            psLists->hAvailLists, (void**)&hLL);
        if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // yes we can reuse a previously allocated block

            printf(OBJECT_NAME": we can reuse a previously allocated LL\n");
            // remove from the "available" LL
            eOsalReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eOsalReturnCode == OSAL_SUCCESS)
            {
                hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                // add this block to the "in use" LL
                eOsalReturnCode = OSAL.eLinkedListAdd(
                    psLists->hInUseLists,
                    (OSAL_LINKED_LIST_ENTRY *)&hEntry,
                    (void *)hLL
                    );
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    printf(OBJECT_NAME": Error. adding LL to in use.\n");
                    OSAL.eLinkedListDelete(hLL);
                }
            }
        }
        else
        {
            printf(OBJECT_NAME": no available LLs. need to create one\n");
            // no available blocks. need to create one.
            // create a new block
            eOsalReturnCode =
                OSAL.eLinkedListCreate(&hLL, "SXI:LL", NULL, OSAL_LL_OPTION_NONE);

            if ((hLL != OSAL_INVALID_OBJECT_HDL) ||
                (eOsalReturnCode != OSAL_SUCCESS))
            {
                // add this block to the "in use" LL
                eOsalReturnCode = OSAL.eLinkedListAdd(
                    psLists->hInUseLists,
                    (OSAL_LINKED_LIST_ENTRY *)&hEntry,
                    (void *)hLL
                    );
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    printf(OBJECT_NAME": Error. adding memory to in use..\n");
                    OSAL.eLinkedListDelete(hLL);
                    hLL = OSAL_INVALID_OBJECT_HDL;
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Failed to create LL");
            }
        }

        // Unlock lists
        SMSO_vUnlock((SMS_OBJECT)psLists);
    }

    return hLL;
}

/*****************************************************************************
*
*   vFreeLLs
*
* LL Remove All callback - Frees the memory
*
*****************************************************************************/
static void vFreeLLs(
    OSAL_OBJECT_HDL hLL
    )
{
    OSAL.eLinkedListRemoveAll(
        hLL,
        NULL
        );

    // Destroy the list itself
    OSAL.eLinkedListDelete(hLL);
}

/*****************************************************************************
*
*   bHandleMultipleGlobalTWEntries
*
*****************************************************************************/
static BOOLEAN bHandleMultipleGlobalTWEntries(
    SXI_GMD_LISTS_STRUCT const *psLists,
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_EXT_METADATA_TRAFFIC_STRUCT *psStruct
    )
{
    SXIAPI_EXT_METADATA_STRUCT *psMetaStruct;
    BOOLEAN bReturn = TRUE, bMarketDecoded = FALSE;

    // there are potentially multiple global meta data entries
    // just decode the markets, and add them to the LL
    // first get a LL
    psStruct->hLL = hGetLL(psLists);

    if (psStruct->hLL == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            "Failed to get list");
        return FALSE;
    }

    // markets received
    do
    {
        // get a metadata struct to hold market info
        psMetaStruct = psGetStruct(psLists);

        if (psMetaStruct == NULL)
        {
            bReturn = FALSE;
            break;
        }

        // decode the next market in our payload
        bMarketDecoded =
            bDecodeMarket(hPayload, &psStruct->un8Fields,
            &psMetaStruct->uData.sTrafficStruct
            );

        if (bMarketDecoded == TRUE)
        {
            // market decoded

            OSAL_RETURN_CODE_ENUM eOsalReturnCode;

            // add the struct to the LL

            // add this block to the LL
            eOsalReturnCode = OSAL.eLinkedListAdd(
                psStruct->hLL,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)psMetaStruct
                );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                printf(OBJECT_NAME": Error. adding to ll.\n");
                OSAL.vLinkedListMemoryFree(psMetaStruct);
            }
        }
        else
        {
            // didn't decode a market, return the struct
            vFreeStruct(psLists, psMetaStruct);
        }

    } while (bMarketDecoded == TRUE);


    return bReturn;
}

/*****************************************************************************
*
*   bHandleMultipleGlobalTEAMEntries
*
*****************************************************************************/
static BOOLEAN bHandleMultipleGlobalTEAMEntries(
    SXI_GMD_LISTS_STRUCT const *psLists,
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_EXT_METADATA_SPORTS_STRUCT *psStruct
    )
{
    BOOLEAN bReturn = TRUE, bTeamDecoded = FALSE;
    SXIAPI_EXT_METADATA_STRUCT *psMetaStruct;

    // there are potentially multiple global meta data entries
    // (market, league, teams, favorites)
    // we don't want to allocate an array of structures for the highest number
    // of possible entries 'cause that would be big (like 20k).
    // we'll just copy everything in the payload into a new buffer, pass it up
    // to the radio and then the radio can use SXI friend functions to parse
    // the buffer

    // get a pointer to a SXIAPI_GLOBAL_METADATA_PAYLOAD_STRUCT
    psStruct->hLL = hGetLL(psLists);

    if (psStruct->hLL == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            "Failed to get list");
        return FALSE;
    }

    // teams received
    do
    {
        psMetaStruct = psGetStruct(psLists);

        if (psMetaStruct == NULL)
        {
            bReturn = FALSE;
            break;
        }

        // decode the next team in our buffer
        bTeamDecoded =
            bDecodeTeam(hPayload, &psStruct->un8Fields,
            &psMetaStruct->uData.sTeamLeagueStruct.uData.sTeamStruct);

        if (bTeamDecoded == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;

            // add the struct to the LL
            eOsalReturnCode = OSAL.eLinkedListAdd(
                psStruct->hLL,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)psMetaStruct
                );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                printf(OBJECT_NAME": Error. adding to ll.\n");
                OSAL.vLinkedListMemoryFree(psMetaStruct);
            }
        }
        else
        {
            // return the stuct
            vFreeStruct(psLists, psMetaStruct);
        }

    } while (bTeamDecoded == TRUE);


    return bReturn;
}

/*****************************************************************************
*
*   bHandleMultipleGlobalLEAGUEEntries
*
*****************************************************************************/
static BOOLEAN bHandleMultipleGlobalLEAGUEEntries(
    SXI_GMD_LISTS_STRUCT const *psLists,
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_EXT_METADATA_SPORTS_STRUCT *psStruct
    )
{
    BOOLEAN bReturn = TRUE, bLeagueDecoded = FALSE;
    SXIAPI_EXT_METADATA_STRUCT *psMetaStruct;

    // there are potentially multiple global meta data entries
    // (market, league, teams, favorites)
    // we don't want to allocate an array of structures for the highest number
    // of possible entries 'cause that would be big (like 20k).
    // we'll just copy everything in the payload into a new buffer, pass it up
    // to the radio and then the radio can use SXI friend functions to parse
    // the buffer

    // get a pointer to a SXIAPI_GLOBAL_METADATA_PAYLOAD_STRUCT
    psStruct->hLL = hGetLL(psLists);

    if (psStruct->hLL == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            "Failed to get list");
        return FALSE;
    }

    // leagues received
    do
    {
        psMetaStruct = psGetStruct(psLists);

        if (psMetaStruct == NULL)
        {
            bReturn = FALSE;
            break;
        }

        // decode the next league in our buffer
        bLeagueDecoded =
            bDecodeLeague(hPayload, &psStruct->un8Fields,
            &psMetaStruct->uData.sTeamLeagueStruct.uData.sLeagueStruct);

        if (bLeagueDecoded == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;

            // add the struct to the LL

            // add this block to the "in use" LL
            eOsalReturnCode = OSAL.eLinkedListAdd(
                psStruct->hLL,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)psMetaStruct
                );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                printf(OBJECT_NAME": Error. adding memory to in use.\n");
                OSAL.vLinkedListMemoryFree(psMetaStruct);
            }
        }
        else
        {
            // return the stuct
            vFreeStruct(psLists, psMetaStruct);
        }

    } while (bLeagueDecoded == TRUE);


    return bReturn;
}

/*****************************************************************************
*
*   bHandleMultipleGlobalFFEntries
*
*****************************************************************************/
static BOOLEAN bHandleMultipleGlobalFFEntries(
    SXI_GMD_LISTS_STRUCT const *psLists,
    OSAL_BUFFER_HDL hPayload,
    SXIAPI_EXT_METADATA_FEATURED_FAVORITES_STRUCT *psStruct
    )
{
    BOOLEAN bReturn = TRUE, bBankDecoded = FALSE;
    SXIAPI_EXT_METADATA_STRUCT *psMetaStruct;

    // there are potentially multiple global meta data entries
    // (market, league, teams, favorites)
    // we don't want to allocate an array of structures for the highest number
    // of possible entries 'cause that would be big (like 20k).
    // we'll just copy everything in the payload into a new buffer, pass it up
    // to the radio and then the radio can use SXI friend functions to parse
    // the buffer

    // get a pointer to a SXIAPI_GLOBAL_METADATA_PAYLOAD_STRUCT
    psStruct->hLL = hGetLL(psLists);

    if (psStruct->hLL == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            "Failed to get list");
        return FALSE;
    }

    // banks received
    do
    {
        psMetaStruct = psGetStruct(psLists);

        if (psMetaStruct == NULL)
        {
            bReturn = FALSE;
            break;
        }

        // decode the next bank in our buffer
        bBankDecoded =
            bDecodeBank(hPayload, &psStruct->un8Fields,
            &psMetaStruct->uData.sBankStruct
            );

        if (bBankDecoded == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;

            // add the struct to the LL

            // add this block to the "in use" LL
            eOsalReturnCode = OSAL.eLinkedListAdd(
                psStruct->hLL,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)psMetaStruct
                );

            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                printf(OBJECT_NAME": Error. adding memory to in use.\n");
                OSAL.vLinkedListMemoryFree(psMetaStruct);
            }
        }
        else
        {
            // return the stuct
            vFreeStruct(psLists, psMetaStruct);
        }

    } while (bBankDecoded == TRUE);


    return bReturn;
}

/*****************************************************************************
*
*   bDecodeTeam
*
*****************************************************************************/
static BOOLEAN bDecodeTeam(
    OSAL_BUFFER_HDL hBuffer,
    UN8 *pun8Fields,
    SXIAPI_EXT_METADATA_SPORTS_TEAM_STRUCT *psTeam
    )
{
    BOOLEAN bReturn = FALSE, bDone = FALSE, bReadingTeam = FALSE;
    UN16 un16GMI;

    // go through the fields one by one
    while (*pun8Fields > 0)
    {
        // peek the next GMI
        if( !SXIAPI_bPeekUint16(hBuffer, &un16GMI, 0) )
        {
            break;
        }

        // process the data according to which GMI we're dealing with
        switch(un16GMI)
        {
        case SXIAPI_GMI_SP_TEAM_ID:
            {
                if (bReadingTeam == FALSE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // set these to defualts
                    psTeam->acTeamAbbrev[0] = '\0';
                    psTeam->acTeamName[0] = '\0';
                    psTeam->acTeamNickname[0] = '\0';

                    // we got the sports team id
                    SXIAPI_bReadUint16(hBuffer,&psTeam->un16TeamId);

                    psTeam->un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_TEAM_ID;
                    (*pun8Fields)--;

                    // set the flag to indicate we are processign a team
                    bReadingTeam = TRUE;
                    bReturn = TRUE;
                }
                else
                {
                    // we are starting to read the next team therefore we are
                    // done with the team we're currently reading
                    // we need to stop reading so this team can be handled.
                    bReturn = TRUE;
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_SP_TEAM_ABBREV:
            {
                if (bReadingTeam == TRUE)
                {

                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the sports team abbrev.
                    SXIAPI_bReadString(
                        hBuffer,
                        &psTeam->acTeamAbbrev[0],
                        sizeof(psTeam->acTeamAbbrev));

                    psTeam->un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_TEAM_ABBREV;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a team
                    bReturn = FALSE;
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_SP_TEAM_NAME:
            {
                if (bReadingTeam == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the sports team name
                    SXIAPI_bReadString(
                        hBuffer,
                        &psTeam->acTeamName[0],
                        sizeof(psTeam->acTeamName));

                    psTeam->un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_TEAM_NAME;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a team
                    bReturn = FALSE;
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_SP_TEAM_NICKNAME:
            {
                if (bReadingTeam == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the sports team mascot
                    SXIAPI_bReadString(
                        hBuffer,
                        &psTeam->acTeamNickname[0],
                        sizeof(psTeam->acTeamNickname));
                    psTeam->un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_TEAM_NICKNAME;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a team
                    bReturn = FALSE;
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_SP_TEAM_LEAGUE_IDS:
            {
                if (bReadingTeam == TRUE)
                {
                    UN32 un32NumLeagues = 0;
                    size_t tNumLeagues;

                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the team league ids

                    // find out how many leagues we need to read
                    SXIAPI_bReadUint32(hBuffer, &un32NumLeagues);
                    tNumLeagues = (size_t)un32NumLeagues;
                    (*pun8Fields)--;

                    if (tNumLeagues <= SXIAPI_MAX_SPORTS_TEAM_LEAGUE_LIST_SIZE)
                    {
                        size_t tLeagueIndex;

                        psTeam->tNumLeagues = tNumLeagues;

                        for (tLeagueIndex = 0;
                            tLeagueIndex < SXIAPI_MAX_SPORTS_TEAM_LEAGUE_LIST_SIZE;
                            tLeagueIndex++)
                        {
                            if (tLeagueIndex < tNumLeagues)
                            {
                                UN8 un8LeagueId;
                                BOOLEAN bWrote;

                                bWrote = SXIAPI_bReadUint8(hBuffer,
                                    &un8LeagueId);
                                // we got the league id
                                if (bWrote == TRUE)
                                {
                                    psTeam->aun8TeamLeagueList[tLeagueIndex] =
                                        un8LeagueId;
                                }

                                // set the bit to indicate we got the league list
                                psTeam->un32ValidFieldMask |=
                                    SXIAPI_VALID_GMI_SP_TEAM_LEAGUE_IDS;
                            }
                            else
                            {
                                // fill in the rest of the array with "invalid"
                                // leagueId
                                psTeam->aun8TeamLeagueList[tLeagueIndex] =
                                    LEAGUE_INVALID_ID;
                            }
                        }
                    }
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a team
                    bReturn = FALSE;
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_SP_TEAM_TIERS:
            {
                if (bReadingTeam == TRUE)
                {
                    UN32 un32NumTiers = 0;

                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the team tiers

                    // find out how many tiers we need to read
                    SXIAPI_bReadUint32(hBuffer, (UN32 *)&un32NumTiers);
                    (*pun8Fields)--;

                    if (un32NumTiers <= SXIAPI_MAX_SPORTS_TEAM_TIER_LIST_SIZE)
                    {
                        size_t tTierIdx;

                        psTeam->tNumTiers = (size_t)un32NumTiers;

                        for (tTierIdx = 0;
                            ((tTierIdx < un32NumTiers) &&
                            (tTierIdx < SXIAPI_MAX_SPORTS_TEAM_TIER_LIST_SIZE));
                        tTierIdx++)
                        {
                            // get the tiers
                            SXIAPI_bReadUint8(hBuffer,
                                (UN8 *)&psTeam->aun8TeamTierList[tTierIdx]);
                        }
                    }

                    // set the bit to indicate we got the tiers
                    psTeam->un32ValidFieldMask |= SXIAPI_VALID_GMI_SP_TEAM_TIERS;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a team
                    bReturn = FALSE;
                    bDone = TRUE;
                }
            }
            break;

        default:
            // something strange happened. stop processing
            bDone = TRUE;
            break;
        }

        if (bDone == TRUE)
        {
            // stop parsing
            break;
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   bDecodeLeague
*
*****************************************************************************/
static BOOLEAN bDecodeLeague(
    OSAL_BUFFER_HDL hBuffer,
    UN8 *pun8Fields,
    SXIAPI_EXT_METADATA_SPORTS_LEAGUE_STRUCT *psLeague)
{
    UN16 un16GMI;
    BOOLEAN bDone = FALSE, bReturn = FALSE, bReadingLeague = FALSE;

    puts(OBJECT_NAME": bDecodeLeague Called");

    // go through the fields one by one. when we have parsed a complete
    // league we'll return TRUE
    // if no league can be parsed we'll return FALSE to let the
    // caller know the buffer is used up.
    while (*pun8Fields > 0)
    {
        // Read the next GMI
        if( !SXIAPI_bPeekUint16(hBuffer, &un16GMI, 0) )
        {
            break;
        }

        // Read the data according to which GMI we're dealing with
        switch(un16GMI)
        {
        case SXIAPI_GMI_SP_LEAGUE_ID:
            {
                if (bReadingLeague == FALSE)
                {
                    UN8 un8LeagueId;
                    BOOLEAN bWrote;

                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    psLeague->acLeagueShortName[0] = '\0';
                    psLeague->acLeagueLongName[0] = '\0';
                    psLeague->acLeagueType[0] = '\0';
                    psLeague->bSportsFlashEnabled = FALSE;
                    psLeague->un8SportsFlashTiers = 0;

                    bWrote = SXIAPI_bReadUint8(hBuffer,
                        &un8LeagueId);
                    // we got the league id
                    if (bWrote == TRUE)
                    {
                        psLeague->un16LeagueId = (UN16)un8LeagueId;
                    }

                    printf(OBJECT_NAME": Decoding LEAGUE: %d\n", 
                        psLeague->un16LeagueId);

                    psLeague->un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_LEAGUE_ID;
                    (*pun8Fields)--;

                    // set the flag to indicate we are processing a league
                    bReadingLeague = TRUE;
                    // a league id is the minimimum amount of info to define a
                    // league
                    bReturn = TRUE;
                }
                else
                {
                    // we are starting to read the next league therefore we are
                    // done with the league we're currently reading
                    // we need to stop reading so this league can be handled.
                    bReturn = TRUE;
                    bDone = TRUE;
                }

            }
            break;

        case SXIAPI_GMI_SP_LEAGUE_SHORT_NAME:
            {
                if (bReadingLeague == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the league name
                    SXIAPI_bReadString(
                        hBuffer,
                        &psLeague->acLeagueShortName[0],
                        sizeof(psLeague->acLeagueShortName));

                    psLeague->un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_LEAGUE_SHORT_NAME;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a league
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_SP_LEAGUE_LONG_NAME:
            {
                if (bReadingLeague == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the league name
                    SXIAPI_bReadString(
                        hBuffer,
                        &psLeague->acLeagueLongName[0],
                        sizeof(psLeague->acLeagueLongName));

                    psLeague->un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_LEAGUE_LONG_NAME;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a league
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_SP_LEAGUE_TYPE:
            {
                if (bReadingLeague == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the league name
                    SXIAPI_bReadString(
                        hBuffer,
                        &psLeague->acLeagueType[0],
                        sizeof(psLeague->acLeagueType));

                    psLeague->un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_LEAGUE_TYPE;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a league
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_SP_LEAGUE_SF_ENABLED:
            {
                if (bReadingLeague == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the SportsFlash Enabled value
                    SXIAPI_bReadBoolean(
                        hBuffer,
                        &psLeague->bSportsFlashEnabled);

                    printf(OBJECT_NAME": LEAGUE: %d, SFEnabled= %u\n",
                        psLeague->un16LeagueId, psLeague->bSportsFlashEnabled);

                    psLeague->un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_LEAGUE_SF_ENABLED;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a league
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_SP_LEAGUE_SF_TIERS:
            {
                if (bReadingLeague == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the SportsFlash Tiers value
                    SXIAPI_bReadUint8(
                        hBuffer,
                        &psLeague->un8SportsFlashTiers);

                    printf(OBJECT_NAME": LEAGUE: %d, SFTiers= %u\n",
                        psLeague->un16LeagueId, psLeague->un8SportsFlashTiers);

                    psLeague->un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_SP_LEAGUE_SF_TIERS;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a league
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        default:
            // something strange happened. stop parsing
            bDone = TRUE;
            break;
        }

        if (bDone == TRUE)
        {
            // stop parsing
            break;
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   bDecodeMarket
*
*****************************************************************************/
static BOOLEAN bDecodeMarket(
    OSAL_BUFFER_HDL hBuffer,
    UN8 *pun8Fields,
    SXIAPI_EXT_METADATA_MARKET_STRUCT *psMarket)
{
    UN16 un16GMI;
    BOOLEAN bDone = FALSE, bReturn = FALSE, bReadingMarket = FALSE;

    // go through the fields one by one. when we have parsed a complete
    // market we'll return TRUE
    // if no market can be parsed we'll return FALSE to let the
    // caller know the buffer is used up.
    while (*pun8Fields > 0)
    {
        // Read the next GMI
        if( !SXIAPI_bPeekUint16(hBuffer, &un16GMI, 0) )
        {
            break;
        }

        // Read the data according to which GMI we're dealing with
        switch(un16GMI)
        {
        case SXIAPI_GMI_TW_CITY_ID:
            {
                if (bReadingMarket == FALSE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the traffic weather city id
                    SXIAPI_bReadUint8(hBuffer, &psMarket->un8CityId);
                    psMarket->un32ValidFieldMask |= SXIAPI_VALID_GMI_TW_CITY_ID;
                    (*pun8Fields)--;
                    // set the flag to indicate that we are in the process
                    // of parsing a market
                    bReadingMarket = TRUE;
                    bReturn = TRUE;
                }
                else
                {
                    // we are starting to read the next mkt therefore we are
                    // done with the mkt we're currently reading
                    // we need to stop reading so this mkt can be handled.
                    bReturn = TRUE;
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_TW_CITY_ABBREV:
            {
                if (bReadingMarket == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the traffic weather city abbrev
                    SXIAPI_bReadString(
                        hBuffer,
                        &psMarket->acAbbrev[0],
                        sizeof(psMarket->acAbbrev));
                    psMarket->un32ValidFieldMask |=
                        SXIAPI_VALID_GMI_TW_ABBREV;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a market
                    bReturn = FALSE;
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_TW_CITY_NAME:
            {
                if (bReadingMarket == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the traffic weather city name
                    SXIAPI_bReadString(hBuffer,
                        &psMarket->acName[0], sizeof(psMarket->acName));
                    psMarket->un32ValidFieldMask |= SXIAPI_VALID_GMI_TW_NAME;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a market
                    bReturn = FALSE;
                    bDone = TRUE;
                }
            }
            break;

        default:
            // something strange happened.
            bDone = TRUE;
            break;
        }

        if (bDone == TRUE)
        {
            // we're done processing the buffer
            break;
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   bDecodeBank
*
*****************************************************************************/
static BOOLEAN bDecodeBank(
    OSAL_BUFFER_HDL hBuffer,
    UN8 *pun8Fields,
    SXIAPI_EXT_METADATA_BANK_STRUCT *psBank
    )
{
    UN16 un16GMI;
    BOOLEAN bDone = FALSE, bReturn = FALSE, bOk,
        bReadingFeaturedFavorites = FALSE;

    // go through the fields one by one. when we have parsed a complete
    // featured favorites we'll return TRUE
    // if no featured favorites can be parsed we'll return FALSE to let the
    // caller know the buffer is used up.
    while (*pun8Fields > 0)
    {
        // Read the next GMI
        bOk = SXIAPI_bPeekUint16(hBuffer, &un16GMI, 0);
        if (bOk == FALSE)
        {
            break;
        }

        // Read the data according to which GMI we're dealing with
        switch(un16GMI)
        {
        case SXIAPI_GMI_FF_BANK_ORDER:
            {
                if (bReadingFeaturedFavorites == FALSE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the bank order
                    SXIAPI_bReadUint8(hBuffer, &psBank->un8Order);
                    psBank->un32ValueFieldMask |=
                        SXIAPI_VALID_GMI_FF_BANK_ORDER;
                    (*pun8Fields)--;
                    // set the flag to indicate that we are in the process
                    // of parsing a featured favorites
                    bReadingFeaturedFavorites = TRUE;
                    bReturn = TRUE;
                }
                else
                {
                    // we are starting to read the next featured favorites therefore we are
                    // done with the featured favorites we're currently reading
                    // we need to stop reading so this featured favorites can be handled.
                    bDone = TRUE;
                    bReturn = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_FF_BANK_ID:
            {
                if (bReadingFeaturedFavorites == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the bank ID
                    SXIAPI_bReadUint8(hBuffer, &psBank->un8Id);
                    psBank->un32ValueFieldMask |=
                        SXIAPI_VALID_GMI_FF_BANK_ID;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a featured favorites
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_FF_BANK_SEQ:
            {
                if (bReadingFeaturedFavorites == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the bank sequence
                    SXIAPI_bReadUint8(hBuffer, &psBank->un8Sequence);
                    psBank->un32ValueFieldMask |=
                        SXIAPI_VALID_GMI_FF_BANK_SEQ;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a featured favorites
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_FF_BANK_TTL_SHORT:
            {
                if (bReadingFeaturedFavorites == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the bank title - short
                    SXIAPI_bReadString(
                        hBuffer,
                        &psBank->acTitleShort[0],
                        sizeof(psBank->acTitleShort));

                    psBank->un32ValueFieldMask |=
                        SXIAPI_VALID_GMI_FF_BANK_TTL_SHORT;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a featured favorites
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_FF_BANK_TTL_LONG:
            {
                if (bReadingFeaturedFavorites == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the bank title - long
                    SXIAPI_bReadString(
                        hBuffer,
                        &psBank->acTitleLong[0],
                        sizeof(psBank->acTitleLong));

                    psBank->un32ValueFieldMask |=
                        SXIAPI_VALID_GMI_FF_BANK_TTL_LONG;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a featured favorites
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_FF_BANK_TTL_VERBOSE:
            {
                if (bReadingFeaturedFavorites == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the bank title - verbose
                    SXIAPI_bReadString(
                        hBuffer,
                        &psBank->acTitleVerbose[0],
                        sizeof(psBank->acTitleVerbose));

                    psBank->un32ValueFieldMask |=
                        SXIAPI_VALID_GMI_FF_BANK_TTL_VERBOSE;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a featured favorites
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_FF_BANK_DESCR:
            {
                if (bReadingFeaturedFavorites == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the bank description
                    SXIAPI_bReadString(
                        hBuffer,
                        &psBank->acDescription[0],
                        sizeof(psBank->acDescription));

                    psBank->un32ValueFieldMask |=
                        SXIAPI_VALID_GMI_FF_BANK_DESCR;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a featured favorites
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_FF_BANK_PURPOSE:
            {
                if (bReadingFeaturedFavorites == TRUE)
                {
                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the bank purpose
                    SXIAPI_bReadUint8(hBuffer, &psBank->un8Purpose);
                    psBank->un32ValueFieldMask |=
                        SXIAPI_VALID_GMI_FF_BANK_PURPOSE;
                    (*pun8Fields)--;
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a featured favorites
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        case SXIAPI_GMI_FF_BANK_ARR_1:
        case SXIAPI_GMI_FF_BANK_ARR_2:
        case SXIAPI_GMI_FF_BANK_ARR_3:
        case SXIAPI_GMI_FF_BANK_ARR_4:
            {
                UN8 un8ArrIndex = 0;
                UN64 un64Mask = 0;

                switch( un16GMI )
                {
                case SXIAPI_GMI_FF_BANK_ARR_1:
                    {
                        un8ArrIndex = 0;
                        un64Mask = SXIAPI_VALID_GMI_FF_BANK_ARR1;
                    }
                    break;

                case SXIAPI_GMI_FF_BANK_ARR_2:
                    {
                        un8ArrIndex = 1;
                        un64Mask = SXIAPI_VALID_GMI_FF_BANK_ARR2;
                    }
                    break;

                case SXIAPI_GMI_FF_BANK_ARR_3:
                    {
                        un8ArrIndex = 2;
                        un64Mask = SXIAPI_VALID_GMI_FF_BANK_ARR3;
                    }
                    break;

                case SXIAPI_GMI_FF_BANK_ARR_4:
                    {
                        un8ArrIndex = 3;
                        un64Mask = SXIAPI_VALID_GMI_FF_BANK_ARR4;
                    }
                    break;
                }

                if (bReadingFeaturedFavorites == TRUE)
                {
                    BOOLEAN bProcessed;

                    // consume the GMI
                    SXIAPI_bReadUint16(hBuffer, &un16GMI);

                    // we got the bank arrangement
                    bProcessed = bDecodeArrangement(
                        hBuffer,
                        &psBank->un16Arrangement[un8ArrIndex][0],
                        &psBank->un8SIDCount[un8ArrIndex]);

                    // arrangement processed successfully
                    if (bProcessed == TRUE)
                    {
                        // make the mark, that we have compatible arrangement
                        psBank->un32ValueFieldMask |= un64Mask;
                        (*pun8Fields)--;
                    }
                    else
                    {
                        // something strange happened
                        bReturn = FALSE;
                        // stop processing
                        bDone = TRUE;
                    }
                }
                else
                {
                    // we never should have gotten this GMI unless we are
                    // reading a featured favorites
                    bReturn = FALSE;
                    // stop processing
                    bDone = TRUE;
                }
            }
            break;

        default:
            {
                printf(OBJECT_NAME": Failed to decode Featured Favorites Bank\n");
                // Failed to decode this bank
                bReturn = FALSE;
                // stop processing
                bDone = TRUE;
            }
            break;
        }

        if (bDone == TRUE)
        {
            // we're done processing the buffer
            break;
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   bDecodeArrangement
*
*****************************************************************************/
static BOOLEAN bDecodeArrangement (
    OSAL_BUFFER_HDL hBuffer,
    UN16 *pun16Arrangement,
    UN8 *pun8Count
    )
{
    BOOLEAN bReturn = FALSE;
    UN32 un32ArrLen, un32ArrIdx;

    // read the array length
    // see SX-9845-0205_Featured_Favorites_Protocol_v1.0_2011-04-01, p.19,
    // 5.7.9 Channel_Array_1 - Channel_Array_4
    SXIAPI_bReadUint32((hBuffer), &un32ArrLen);

    // process the arrangement
    for (un32ArrIdx = 0;
        un32ArrIdx < SXIAPI_MAX_BANK_ARR &&
        un32ArrIdx < un32ArrLen;
    un32ArrIdx++)
    {
        UN16 un16SID;
        BOOLEAN bRead;

        // read next SID
        bRead = SXIAPI_bReadUint16((hBuffer), &un16SID);

        // we got the channel SID
        if (bRead == TRUE)
        {
            bReturn = TRUE;

            pun16Arrangement[un32ArrIdx] = un16SID;
            *pun8Count = *pun8Count + 1;

            printf("SID%u: %u\n", un32ArrIdx,
                pun16Arrangement[un32ArrIdx]);
        }
        else
        {
            break;
        }
    }

    printf("**********************************\n"
        "  Featured Favorites Arrangement  \n"
        "**********************************\n"
        "RMAX: %u, RMIN: %u, CNT: %u       \n",
        pun16Arrangement[0],
        pun16Arrangement[1],
        *pun8Count);

    return bReturn;
}

/*****************************************************************************
*
*   bConfigureGlobalExtendedMetaDataMonitor
*
*****************************************************************************/
static BOOLEAN bConfigureGlobalExtendedMetaDataMonitor (
    SXI_GMD_STRUCT *psGMD,
    STI_HDL hControlCxn,
    UN64 un64Mask,
    BOOLEAN bEnable
        )
{
    BOOLEAN bSuccess = TRUE;
    OSAL_OBJECT_HDL hMutex;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Validate inputs
    if((hControlCxn == STI_INVALID_HDL) || (un64Mask == 0))
    {
        return FALSE;
    }

    // The whole command sequence has to be executed in exclusive mode
    // in order to prevent commands interleaving when this function
    // is called from several threads at the same time
    hMutex = psGMD->hCmdMutex;
    eReturnCode = OSAL.eSemTake(hMutex, OSAL_OBJ_TIMEOUT_INFINITE);
    if (OSAL_SUCCESS != eReturnCode)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            OBJECT_NAME":SXIAPI_eExtMetadataMonCmd() "
            "failed to take control mutex");
        return FALSE;
    }

    do
    {
        UN8 un8Index;
        SXIAPI_STATUSCODE_ENUM eStatusCode;
        UN64 un64NewMask, un64CurrentMask;

        // Remember current mask
        un64CurrentMask = psGMD->un64ExtMetadataGMIMask;

        // Based on what we want to do, reflect this in the mask
        if(bEnable == TRUE)
        {
            psGMD->un64ExtMetadataGMIMask |= un64Mask;
        }
        else
        {
            psGMD->un64ExtMetadataGMIMask &= ~un64Mask;
        }

        // Copy out new mask
        un64NewMask = psGMD->un64ExtMetadataGMIMask;

        // Do we have any?
        if(psGMD->un8ExtMetadataEmiCnt > 0)
        {
            // Disable all Global Metadata Monitor Items which
            // are active.
            eStatusCode =
                SXIAPI_eExtMetadataMonCmd(
                    hControlCxn,
                    SXIAPI_EXT_GLOBAL_METADATA_MONITOR,
                    SXIAPI_MONITOR_OP_DISABLE,
                    psGMD->un8ExtMetadataEmiCnt,
                    psGMD->atExtMetadataGID
                        );

            if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
            {
                printf(OBJECT_NAME":SXIAPI_eExtMetadataMonCmd()"
                    " disabled all EMIs (0x%0llx).\n",
                    un64CurrentMask);
                // The mask has now been changed, everything
                // of interest has been disabled.
            }
            else
            {
                // Error
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    OBJECT_NAME":SXIAPI_eExtMetadataMonCmd()"
                    " failed disable of all EMIs (0x%0llx).",
                    un64CurrentMask);
                // The mask had no impact
                bSuccess = FALSE;
            }
        }

        // Reenable all GMI monitors we have remaining active

        // Construct an array of GMIs based on the provided mask
        psGMD->un8ExtMetadataEmiCnt = 0;
        for (un8Index = 0; un8Index < SXIAPI_MAX_EXT_METADATA_GMI_COUNT;
            un8Index++)
        {
            if ( (un64NewMask &
                gasGMIMap[un8Index].un64Mask)
                == gasGMIMap[un8Index].un64Mask )
            {
                psGMD->atExtMetadataGID[psGMD->un8ExtMetadataEmiCnt] =
                    (UN16)gasGMIMap[un8Index].eGMI;
                psGMD->un8ExtMetadataEmiCnt++;
            }
        }

        // Do we have any?
        if(psGMD->un8ExtMetadataEmiCnt > 0)
        {
            // Enable Extended Metadata Monitor Items
            eStatusCode =
                SXIAPI_eExtMetadataMonCmd(
                    hControlCxn,
                    SXIAPI_EXT_GLOBAL_METADATA_MONITOR,
                    SXIAPI_MONITOR_OP_ENABLE,
                    psGMD->un8ExtMetadataEmiCnt,
                    psGMD->atExtMetadataGID
                        );

            if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
            {
                printf(OBJECT_NAME":SXIAPI_eExtMetadataMonCmd()"
                    " enabled %u EMIs (0x%0llx)",
                    psGMD->un8ExtMetadataEmiCnt, un64NewMask);
                printf(" new EMIs are 0x%0llx.\n", un64Mask);
                // The mask has now been changed, everything
                // of interest has been enabled.
            }
            else
            {
                // Error
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    OBJECT_NAME":SXIAPI_eExtMetadataMonCmd()"
                    " failed enable of %u EMIs (0x%0llx) new EMIs (0x%0llx).",
                    psGMD->un8ExtMetadataEmiCnt, un64NewMask, un64Mask);
                // Nothing changed, everything of interest remains disabled
                bSuccess = FALSE;
            }
        }

    } while(FALSE);

    OSAL.eSemGive(hMutex);

    return bSuccess;
}

/*******************************************************************************
*
*   n16CompareHandles
*
*******************************************************************************/
static N16 n16CompareHandles(
    void *pvArg1,
    void *pvArg2
    )
{
    N16 n16Result;

    n16Result = COMPARE(pvArg1, pvArg2);

    return n16Result;
}

/*******************************************************************************
*
*   bProcessBankIterator
*
*******************************************************************************/
static BOOLEAN bProcessBankIterator (
    SXIAPI_EXT_METADATA_STRUCT *psStruct,
    SXI_GMD_BANK_ITERATOR_STRUCT *psData
        )
{
    SXIAPI_EXT_METADATA_BANK_STRUCT *psBank = &psStruct->uData.sBankStruct;
    BOOLEAN bOk;
    BOOLEAN bContinue = TRUE;

    bOk = bUpdateFFData(psBank, psData->hPresets, psData->psInfo);
    if (TRUE == bOk)
    {
        bContinue = psData->bNeedUpdate =
            bIsMoreFFDataNeeded(psData->hPresets, psData->psFF);
    }

    return bContinue;
}

/*******************************************************************************
*
*   bProcessBankIterator
*
*******************************************************************************/
static BOOLEAN bSxiDataDestroyIterator (
    SXIAPI_EXT_METADATA_STRUCT *psStruct,
    SXI_GMD_LISTS_STRUCT *psLists
        )
{
    printf(OBJECT_NAME": releasing entry data=%p\n", psStruct);
    vFreeStruct(psLists, psStruct);
    return TRUE;
}
