/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the SXi FSM implementation.
 *
 ******************************************************************************/

#include <string.h>

#include "standard.h"
#include "osal.h"

// Include things I need from SMS
#include "sms_version.h"
#include "sms.h"
#include "sms_obj.h"
#include "sms_event.h"
#include "radio_event_types.h"
#include "sti_api.h"
#include "sxiapi.h"
#include "srh.h"
#include "com.h"

#include "sxi_gmd.h"
#include "module_obj.h"
#include "report_obj.h"
#include "radio.h"

#include "sms_fcsxm.h"

// Include SXI ID module headers
#include "sxi_fsm.h"
#include "_sxi_fsm.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*******************************************************************************
 *
 *   SXI_FSM_bInit
 *
 *******************************************************************************/
SXI_FSM_HDL SXI_FSM_hInit (
    const char *pacSRMName,
    OSAL_FILE_STRUCT *psDevice,
    MODULE_OBJECT hModule,
    SRH_DEVICE_CAPABILITIES_MASK tCapabilities,
    STI_HDL *phControlCxn
        )
{
    SXI_FSM_LINK_STRUCT *psFSM;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SXI_FSM_EVENT_STRUCT sFsmEvent;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_OBJECT_HDL hDataPayloadDestinations = OSAL_INVALID_OBJECT_HDL;

    SMSAPI_DEBUG_vPrint(OBJECT_NAME, 4,
        "%s(): Init SXI FSM...\n", __FUNCTION__);

    // Create name for this FSM instance
    snprintf(&acName[0], sizeof(acName), "SXi-FSM:%s", pacSRMName);

    // Allocate memory for the FSM instance
    psFSM =
        (SXI_FSM_LINK_STRUCT *)
            SMSO_hCreate(
                acName,
                sizeof(SXI_FSM_LINK_STRUCT),
                (SMS_OBJECT)hModule, // Child-of
                FALSE); // inherit lock
    if(psFSM == NULL)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            OBJECT_NAME": Can't Create SXI FSM object.");
        return SXI_FSM_INVALID_HDL;
    }

    // Copy in caller provided parameters
    psFSM->psDevice = psDevice;
    psFSM->hModule = hModule;
    psFSM->pacSRMName = pacSRMName;
    psFSM->tCapabilities = tCapabilities;

    // Initialize FSM
    psFSM->ePreviousState = SXI_FSM_LINK_STATE_UNKNOWN;
    psFSM->eCurrentState = SXI_FSM_LINK_STATE_INITIAL;
    psFSM->hStartUpTimer = OSAL_INVALID_OBJECT_HDL;
    psFSM->un8ResetRetries = 0;
    psFSM->hModCfgIndWaitTimer = OSAL_INVALID_OBJECT_HDL;
    psFSM->eErrorCode = SXI_FSM_RADIO_OK;

    // Initialize UART control
    psFSM->eUARTControl = SXIAPI_UARTCONTROL_UNKNOWN;
    psFSM->pcUARTSettings = NULL;

    // Initialize STI connections
    psFSM->hSxiLinkCxn = STI_INVALID_HDL;
    psFSM->hLinkConnectionArg = SXI_CONNECTION_ARG_INVALID_HANDLE;

    psFSM->hControlCxn = STI_INVALID_HDL;
    psFSM->hSxiControlCxnArg = SXI_CONNECTION_ARG_INVALID_HANDLE;

    psFSM->hSXiDataCxn = STI_INVALID_HDL;
    psFSM->hSxiDataCxnArg = SXI_CONNECTION_ARG_INVALID_HANDLE;

    // Clear rx flags
    psFSM->bModuleCfgIndRxd = FALSE;
    psFSM->bModuleEsnRxd = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE;
        SXI_MODULE_CFG_STRUCT sSXiModuleCfg;
        BOOLEAN bIsOk;
        SRH_DEVICE_PORT_TUPLE_STRUCT sPort;
        N32 n32Err;

        {
            SMS_EVENT_DATA_UNION *puEventData;

            // PRE-ALLOCATE EVENT (SXI-FSM)
            psFSM->hPowerOffStateEvent =
                MODULE_hAllocateEvent(psFSM->hModule, 
                    (SMS_EVENT_TYPE_ENUM)RADIO_EVENT_SXI_FSM,
                    SMS_EVENT_OPTION_NONE,
                    &puEventData
                        );
            if(psFSM->hPowerOffStateEvent == SMS_INVALID_EVENT_HDL)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    OBJECT_NAME": Can't Create Radio Power State Event.");
                break;
            }
            else
            {
                // Populate event
                SXI_FSM_EVENT_STRUCT *psFsmEvent =
                    (SXI_FSM_EVENT_STRUCT *)
                        puEventData->sRadio.apvData;

                // Populate a power off event (from PwrModeInd)
                psFsmEvent->eEventType = SXI_FSM_EVENT_RADIO_POWER_STATE;
                psFsmEvent->uEventData.ePwrMode = SXI_FSM_POWER_MODE_OFF;
            }
        }

        // Create a supervisory event/timer which will force a RADIO-level
        // power off state just in case things do not go as planned.

        // Create the timer that will be used to power off state event
        eReturnCode = OSAL.eTimerCreate(
            &psFSM->hForcePwrOffStateTimer,
            OBJECT_NAME":hForcePwrOffStateTimer",
                vPullTheChuteTimerCallback,
                (const void *)psFSM->hPowerOffStateEvent);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Can't Create Force power off state Timer.");
            break;
        }

        // Get the desired communication settings
        n32Err = ioctl(psFSM->psDevice, SRH_IOCTL_DESCRIBE_PORT,
            TRUE, psFSM->pacSRMName,
            MODULE_ID_INVALID, DECODER_ID_INVALID, &sPort);

        if(n32Err == DEV_OK)
        {
            UN8 un8Idx;

            // extract setting which matches the slected baudrate
            for (un8Idx = 0; un8Idx < SXI_UART_SETTINGS_MAX; un8Idx++)
            {
                if (gasUARTSettings[un8Idx].un32BaudRate == sPort.un32BaudRate)
                {
                    psFSM->eUARTControl =
                        gasUARTSettings[un8Idx].eControl;
                    psFSM->pcUARTSettings =
                        gasUARTSettings[un8Idx].pcSettings;
                    break;
                }
            }

            // Check if we found a setting
            if(psFSM->pcUARTSettings == NULL)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    OBJECT_NAME": Unable to map the chosen baud rate (%u)."
                    " Please check your srh driver.", sPort.un32BaudRate);
                break;
            }
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Unable to describe the device (%s)."
                " Please check your srh driver.", psFSM->pacSRMName);
            break;
        }

        // Request the module configuration information
        bSuccess = bGetSXIModuleConfiguration(psFSM, &sSXiModuleCfg);
        if (bSuccess == TRUE)
        {
            // Populate the ModuleCfg Settings Structure
            RADIO_SET_VALUE(psFSM->sModuleCfg.eFading,
                SXIAPI_FADING_CFG_ENUM,
                sSXiModuleCfg.un8Fade,
                SXIAPI_FADING_CFG_RESERVED, SXIAPI_FADING_CFG_ENABLED,
                bSuccess);
            RADIO_SET_VALUE(psFSM->sModuleCfg.eMaxCatLabelLen,
                SXIAPI_CAT_CHAN_LABEL_LENGTH_ENUM,
                sSXiModuleCfg.un8CategoryLabelLength,
                SXIAPI_CAT_CHAN_LABEL_LENGTH_8, SXIAPI_CAT_CHAN_LABEL_LENGTH_16,
                bSuccess);
            RADIO_SET_VALUE(psFSM->sModuleCfg.eMaxChanLabelLen,
                SXIAPI_CAT_CHAN_LABEL_LENGTH_ENUM,
                sSXiModuleCfg.un8ChannelLabelLength,
                SXIAPI_CAT_CHAN_LABEL_LENGTH_8, SXIAPI_CAT_CHAN_LABEL_LENGTH_16,
                bSuccess);
            RADIO_SET_VALUE(psFSM->sModuleCfg.eMaxMetadataLabelLen,
                SXIAPI_METADATA_LABEL_LENGTH_ENUM,
                sSXiModuleCfg.un8MetadataLabelLength,
                SXIAPI_METADATA_LABEL_LENGTH_8, SXIAPI_METADATA_LABEL_LENGTH_16,
                bIsOk);
            RADIO_SET_VALUE(psFSM->sModuleCfg.eMaxPendingInds,
                SXIAPI_MAX_PENDING_IND_CFG_ENUM,
                sSXiModuleCfg.un8MaxPendingIndications,
                SXIAPI_MAX_PENDING_IND_CFG_1, SXIAPI_MAX_PENDING_IND_CFG_4,
                bSuccess);
            RADIO_SET_VALUE(psFSM->sModuleCfg.tConfirmationWaitTime,
                SXIAPI_CONFIRMATION_WAIT_TIME,
                sSXiModuleCfg.un8ConfirmationWaitTime,
                SXIAPI_MODULECFG_CMD_CONFIRMATION_WAIT_TIME_MIN,
                SXIAPI_MODULECFG_CMD_CONFIRMATION_WAIT_TIME_MAX,
                bSuccess);
            RADIO_SET_VALUE(psFSM->sModuleCfg.eIRControl,
                SXIAPI_IR_CONTROL_CFG_ENUM,
                sSXiModuleCfg.un8IRControl,
                SXIAPI_IR_CONTROL_CFG_DISABLED, SXIAPI_IR_CONTROL_CFG_MODULE,
                bSuccess);
            RADIO_SET_VALUE(psFSM->sModuleCfg.eIRMarkNewTrack,
                SXIAPI_IR_MARK_NEW_TRACK_CFG_ENUM,
                sSXiModuleCfg.un8IRMarkNewTrack,
                SXIAPI_IR_MARK_NEW_TRACK_CFG_ON_PID,
                SXIAPI_IR_MARK_NEW_TRACK_CFG_ON_LABEL,
                bSuccess);
            RADIO_SET_VALUE(psFSM->sModuleCfg.tExtendedControl,
                SXIAPI_EXTENDED_CONTROL_MASK,
                sSXiModuleCfg.un8ExtendedControl,
                SXIAPI_EXTENDED_CONTROL_VAL_MIN,
                SXIAPI_EXTENDED_CONTROL_VAL_MAX,
                bSuccess);
            RADIO_SET_VALUE(psFSM->sModuleCfg.tPrioritySmartFavCnt,
                SXIAPI_PRIORITY_SMART_FAV_COUNT,
                sSXiModuleCfg.un8PrioritySmartFavCnt,
                SXIAPI_PRIORITY_SMART_FAV_CNT_MIN,
                SXIAPI_PRIORITY_SMART_FAV_CNT_MAX,
                bSuccess);
            RADIO_SET_VALUE(psFSM->sModuleCfg.eIRDeleteOnTune,
                SXIAPI_IR_DELETE_ON_TUNE_CFG_ENUM,
                sSXiModuleCfg.un8IRDeleteOnTune,
                SXIAPI_IR_DELETE_ON_TUNE_CFG_DISABLED,
                SXIAPI_IR_DELETE_ON_TUNE_CFG_MODULE,
                bSuccess);

            // Set default values

            //  - Since the SMS doesn't support recording this field
            //    will be initialized by default values
            psFSM->sModuleCfg.tRecordControl = gsModuleCfgDefault.tRecordControl;
        }
        else
        {
            puts(OBJECT_NAME
                ": Cannot get Module configuration, but ignore it since "
                "the defaults can be used");
        }

        if (bSuccess == FALSE)
        {
            // Populate the ModuleCfg Settings Structure by default values
            puts(OBJECT_NAME": applying default SXi Module Configuration");

            psFSM->sModuleCfg = gsModuleCfgDefault;
        }

        // Update supported features
        MODULE_vUpdateIRSupport(psFSM->hModule,
            (psFSM->sModuleCfg.eIRControl ==
                SXIAPI_IR_CONTROL_CFG_MODULE) ? TRUE : FALSE);

        MODULE_vUpdateAdvancedIRSupport(psFSM->hModule,
            (psFSM->sModuleCfg.eIRDeleteOnTune ==
            SXIAPI_IR_DELETE_ON_TUNE_CFG_MODULE) ? TRUE : FALSE);

        // Create data payload list
        eReturnCode = OSAL.eLinkedListCreate(
            &hDataPayloadDestinations,
            "SXI:DataList",
            (OSAL_LL_COMPARE_HANDLER)SXI_n16CompareDataDest,
            OSAL_LL_OPTION_RBTREE);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Can't create data payload list.");
            break;
        }

        // Open STI Connections

        // Connect to Link...
        psFSM->hLinkConnectionArg =
            SXI_hInitConnectionArg(
                (SXI_FSM_HDL)psFSM,
                SXILL_PAYLOAD_TYPE_LINK,
                0, // No response queue needed
                hDataPayloadDestinations
                    );

        if (psFSM->hLinkConnectionArg !=
            SXI_CONNECTION_ARG_INVALID_HANDLE)
        {
            psFSM->hSxiLinkCxn =
                STI_hConnect(
                    "SXI:Link", // Name of connection
                    SXI_bLinkConnectionCallback, // Callback
                    (void*)NULL, // Callback arg
                    psFSM->hLinkConnectionArg, // Connection argument
                    &GsSXILL, // Protocol Interface
                    (void *)psFSM->psDevice, // I/O Device
                    /*** Connection Specific ***/
                    SXILL_PAYLOAD_TYPE_LINK
                        );

            if(psFSM->hSxiLinkCxn == STI_INVALID_HDL)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    OBJECT_NAME": Can't create SXI connection for link layer.");
                break;
            }
        }
        else
        {
            // Error! Could not create connection arguments

            // Destroy the list now
            OSAL.eLinkedListDelete(hDataPayloadDestinations);

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Can't create SXI connection args for link layer.");
            break;
        }

        // Connect to Control...
        psFSM->hSxiControlCxnArg =
            SXI_hInitConnectionArg(
                (SXI_FSM_HDL)psFSM,
                SXILL_PAYLOAD_TYPE_CONTROL,
                MODULE_CONTROL_IND_QUEUE_SIZE,
                hDataPayloadDestinations
                    );

        if (psFSM->hSxiControlCxnArg !=
            SXI_CONNECTION_ARG_INVALID_HANDLE)
        {
            psFSM->hControlCxn =
                STI_hConnect(
                    "SXI:Control", // Name of connection
                    SXI_bControlConnectionCallback, // Callback
                    (void *)hModule, // Callback arg
                    psFSM->hSxiControlCxnArg, // Connection argument
                    &GsSXILL, // Protocol Interface
                    (void *)psFSM->psDevice, // I/O Device
                    /*** Connection Specific ***/
                    SXILL_PAYLOAD_TYPE_CONTROL
                        );

            if(psFSM->hControlCxn == STI_INVALID_HDL)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    OBJECT_NAME": Can't create SXI connection for control layer.");
                break;
            }
        }
        else
        {
            // Error! Could not create connection arguments for
            // command & control connection

            // Destroy the list now
            OSAL.eLinkedListDelete(hDataPayloadDestinations);

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                ": Can't create SXI connection args for control layer.");
            break;
        }

        // Connect to Data...
        psFSM->hSxiDataCxnArg =
            SXI_hInitConnectionArg(
                (SXI_FSM_HDL)psFSM,
                SXILL_PAYLOAD_TYPE_DATA,
                0, // No response queue needed
                hDataPayloadDestinations
                    );

        if (psFSM->hSxiDataCxnArg !=
            SXI_CONNECTION_ARG_INVALID_HANDLE)
        {
            psFSM->hSXiDataCxn =
                STI_hConnect(
                    "SXI:Data", // Name of connection
                    SXI_bDataConnectionCallback, // Callback
                    (void *)NULL, // Callback arg
                    psFSM->hSxiDataCxnArg, // Connection argument
                    &GsSXILL, // Protocol Interface
                    (void *)psFSM->psDevice, // I/O Device
                    /*** Connection Specific ***/
                    SXILL_PAYLOAD_TYPE_DATA
                        );

            if(psFSM->hSXiDataCxn == STI_INVALID_HDL)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    OBJECT_NAME": Can't create SXI connection for data layer.");
                break;
            }
        }
        else
        {
            // Error! Could not create connection arguments for
            // data connection
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                ": Can't create SXI connection args for data layer.");
            break;
        }

        // Create a Link Start Timer for this Module. It's a one shot timer
        // which will reset the module if it takes too long to try to set the timer
        eReturnCode = OSAL.eTimerCreate(&psFSM->hStartUpTimer,
            OBJECT_NAME":StartupTimer", vFsmTimeoutCallback,
            (void *)psFSM);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Can't Create Startup Timer.");
            break;
        }

        // Create the time that will be used to wait for the ModCfgInd message
        eReturnCode = OSAL.eTimerCreate(&psFSM->hModCfgIndWaitTimer,
            OBJECT_NAME":ModCfgIndWaitTimer",
                vFsmTimeoutCallback,
                (void *)psFSM);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Can't Create Startup Timer.");
            break;
        }

        // Signal event to FSM to be run later
        sFsmEvent.eEventType = SXI_FSM_EVENT_START;
        bSuccess = bPostEvent (
            psFSM,
            SMS_EVENT_OPTION_NONE, &sFsmEvent);
        if(bSuccess == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Can't send FSM event (#%u).",
                    sFsmEvent.eEventType);
            break;
        }

        // pass back SXi Control Cxn
        if(phControlCxn != NULL)
        {
            *phControlCxn = psFSM->hControlCxn;
        }

        // Everything's well.
        SMSAPI_DEBUG_vPrint(OBJECT_NAME, 4,
            "%s(): SXI FSM is successfully initialized.\n", __FUNCTION__);

        return (SXI_FSM_HDL)psFSM;

    } while(FALSE);

    // Error!
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        OBJECT_NAME": SXI FSM initialization failed.");

    SXI_FSM_vUnInit((SXI_FSM_HDL)psFSM);

    return SXI_FSM_INVALID_HDL;
}

/*******************************************************************************
 *
 *   SXI_FSM_vUnInit
 *
 *******************************************************************************/
void SXI_FSM_vUnInit (
    SXI_FSM_HDL hFSM
        )
{
    BOOLEAN bOk;
    SXI_FSM_LINK_STRUCT *psFSM = (SXI_FSM_LINK_STRUCT *)hFSM;

    SMSAPI_DEBUG_vPrint(OBJECT_NAME, 4,
        "%s(): Uninit SXI FSM...\n", __FUNCTION__);

    // Check input
    bOk = SMSO_bOwner((SMS_OBJECT)hFSM);
    if(bOk == FALSE)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            OBJECT_NAME": Not owner.");
        return;
    }

    // Since connection argument is created separately it 
    // should be destroyed independently from an STI connection.
    // Under certain conditions, STI connection initialization
    // might fail leading to leaked connection argument.

    // Disconnect data connection if one exists
    if(psFSM->hSXiDataCxn != STI_INVALID_HDL)
    {
        BOOLEAN bOk = STI_bDisconnect(psFSM->hSXiDataCxn);
        if(bOk == TRUE)
        {
            psFSM->hSXiDataCxn = STI_INVALID_HDL;
        }
    }

    if (psFSM->hSxiDataCxnArg != SXI_CONNECTION_ARG_INVALID_HANDLE)
    {
        SXI_vFreeConnectionArg(psFSM->hSxiDataCxnArg);
        psFSM->hSxiDataCxnArg = SXI_CONNECTION_ARG_INVALID_HANDLE;
    }

    // Disconnect control connection if one exists
    if(psFSM->hControlCxn != STI_INVALID_HDL)
    {
        BOOLEAN bOk = STI_bDisconnect(psFSM->hControlCxn);
        if(bOk == TRUE)
        {
            psFSM->hControlCxn = STI_INVALID_HDL;
        }
    }

    if (psFSM->hSxiControlCxnArg != SXI_CONNECTION_ARG_INVALID_HANDLE)
    {
        SXI_vFreeConnectionArg(psFSM->hSxiControlCxnArg);
        psFSM->hSxiControlCxnArg = SXI_CONNECTION_ARG_INVALID_HANDLE;
    }

    // Disconnection link layer connection if one exists
    if(psFSM->hSxiLinkCxn != STI_INVALID_HDL)
    {
        BOOLEAN bOk = STI_bDisconnect(psFSM->hSxiLinkCxn);
        if(bOk == TRUE)
        {
            psFSM->hSxiLinkCxn = STI_INVALID_HDL;
        }
    }

    if (psFSM->hLinkConnectionArg != SXI_CONNECTION_ARG_INVALID_HANDLE)
    {
        SXI_vFreeConnectionArg(psFSM->hLinkConnectionArg);
        psFSM->hLinkConnectionArg = SXI_CONNECTION_ARG_INVALID_HANDLE;
    }

    // Destroy reset timer if one exists
    if(psFSM->hStartUpTimer != OSAL_INVALID_OBJECT_HDL)
    {
        // Delete timer
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL.eTimerDelete(
            psFSM->hStartUpTimer);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psFSM->hStartUpTimer = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy ModCfgInd timer if one exists
    if(psFSM->hModCfgIndWaitTimer != OSAL_INVALID_OBJECT_HDL)
    {
        // Delete timer
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL.eTimerDelete(
            psFSM->hModCfgIndWaitTimer);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psFSM->hModCfgIndWaitTimer = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy forced power off state timer if one exists
    if(psFSM->hForcePwrOffStateTimer != OSAL_INVALID_OBJECT_HDL)
    {
        // Delete timer
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL.eTimerDelete(
            psFSM->hForcePwrOffStateTimer);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psFSM->hForcePwrOffStateTimer = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy FSM object
    SMSO_vDestroy((SMS_OBJECT)psFSM);

    SMSAPI_DEBUG_vPrint(OBJECT_NAME, 4,
        "%s(): SXI FSM is uninitialized.\n", __FUNCTION__);

    return;
}

/*******************************************************************************
 *
 *   SXI_FSM_bPostPowerControl
 *
 *  Used to tell the module what power mode to go into
 *
 *******************************************************************************/
BOOLEAN SXI_FSM_bPostPowerControl (
    SXI_FSM_HDL hFSM,
    SXI_FSM_POWER_MODE_ENUM ePwrMode 
        )
{
    BOOLEAN bPosted = FALSE, bValid;
    SXI_FSM_LINK_STRUCT *psFSM =
        (SXI_FSM_LINK_STRUCT *)hFSM;

    // Check input
    bValid = SMSO_bValid((SMS_OBJECT)hFSM);
    if(bValid == TRUE)
    {
        SXI_FSM_EVENT_STRUCT sFsmEvent;

        // Prepare event
        sFsmEvent.eEventType = SXI_FSM_EVENT_POWER_CONTROL;
        sFsmEvent.uEventData.ePwrMode = ePwrMode;

        // Send event
        bPosted = bPostEvent(
            psFSM,
            SMS_EVENT_OPTION_NONE,
            &sFsmEvent
                );
    }

    return bPosted;
}

/*******************************************************************************
 *
 *   SXI_FSM_bPostRadioPower
 *
 * Used to report the power mode the module is in
 *
 *******************************************************************************/
BOOLEAN SXI_FSM_bPostRadioPower (
    SXI_FSM_HDL hFSM,
    SXI_FSM_POWER_MODE_ENUM ePwrMode
        )
{
    BOOLEAN bPosted = FALSE, bValid;
    SXI_FSM_LINK_STRUCT *psFSM =
        (SXI_FSM_LINK_STRUCT *)hFSM;

    // Check input
    bValid = SMSO_bValid((SMS_OBJECT)hFSM);
    if(bValid == TRUE)
    {
        SXI_FSM_EVENT_STRUCT sFsmEvent;

        // Prepare event
        sFsmEvent.eEventType = SXI_FSM_EVENT_RADIO_POWER_STATE;
        sFsmEvent.uEventData.ePwrMode = ePwrMode;

        // Send event
        bPosted = bPostEvent(
            psFSM,
            SMS_EVENT_OPTION_NONE,
            &sFsmEvent
                );
    }

    return bPosted;
}

/*******************************************************************************
 *
 *   SXI_FSM_bPostCfgIndRxd
 *
 *******************************************************************************/
BOOLEAN SXI_FSM_bPostCfgIndRxd (
    SXI_FSM_HDL hFSM
        )
{
    BOOLEAN bPosted = FALSE, bValid;
    SXI_FSM_LINK_STRUCT *psFSM =
        (SXI_FSM_LINK_STRUCT *)hFSM;

    // Check input
    bValid = SMSO_bValid((SMS_OBJECT)hFSM);
    if(bValid == TRUE)
    {
        SXI_FSM_EVENT_STRUCT sFsmEvent;

        // Prepare event
        sFsmEvent.eEventType = SXI_FSM_EVENT_MODULE_CFG_IND_RXD;

        // Send event
        bPosted = bPostEvent(
            psFSM,
            SMS_EVENT_OPTION_NONE,
            &sFsmEvent
                );
    }

    return bPosted;
}

/*******************************************************************************
 *
 *   SXI_FSM_bPostEsnRxd
 *
 *******************************************************************************/
BOOLEAN SXI_FSM_bPostEsnRxd (
    SXI_FSM_HDL hFSM
        )
{
    BOOLEAN bPosted = FALSE, bValid;
    SXI_FSM_LINK_STRUCT *psFSM =
        (SXI_FSM_LINK_STRUCT *)hFSM;

    // Check input
    bValid = SMSO_bValid((SMS_OBJECT)hFSM);
    if(bValid == TRUE)
    {
        SXI_FSM_EVENT_STRUCT sFsmEvent;

        // Prepare event
        sFsmEvent.eEventType = SXI_FSM_EVENT_MODULE_ESN_RXD;

        // Send event
        bPosted = bPostEvent(
            psFSM,
            SMS_EVENT_OPTION_NONE,
            &sFsmEvent
                );
    }

    return bPosted;
}

/*******************************************************************************
 *
 *   SXI_FSM_bPostError
 *
 *******************************************************************************/
BOOLEAN SXI_FSM_bPostError (
    SXI_FSM_HDL hFSM,
    MODULE_ERROR_CODE_ENUM eModuleErrorCode
        )
{
    BOOLEAN bPosted = FALSE, bValid;
    SXI_FSM_LINK_STRUCT *psFSM =
        (SXI_FSM_LINK_STRUCT *)hFSM;

    // Check input
    bValid = SMSO_bValid((SMS_OBJECT)hFSM);
    if(bValid == TRUE)
    {
        SXI_FSM_EVENT_STRUCT sFsmEvent;

        // Prepare event
        sFsmEvent.eEventType = SXI_FSM_EVENT_ERROR;
        sFsmEvent.uEventData.eModuleErrorCode = eModuleErrorCode;

        // Send event
        bPosted = bPostEvent(
            psFSM,
            SMS_EVENT_OPTION_NONE,
            &sFsmEvent
                );
    }

    return bPosted;
}

/*******************************************************************************
 *
 *   SXI_FSM_bPostOOB
 *
 *******************************************************************************/
BOOLEAN SXI_FSM_bPostOOB (
    SXI_FSM_HDL hFSM,
    STI_PROTOCOL_OOB_STRUCT const *psOOB
        )
{
    BOOLEAN bPosted = FALSE, bValid;
    SXI_FSM_LINK_STRUCT *psFSM =
        (SXI_FSM_LINK_STRUCT *)hFSM;

    // Check input
    bValid = SMSO_bValid((SMS_OBJECT)hFSM);
    if(bValid == TRUE)
    {
        SXI_FSM_EVENT_STRUCT sFsmEvent;

        // Prepare event
        sFsmEvent.eEventType = SXI_FSM_EVENT_OOB_MSG;
        sFsmEvent.uEventData.sOOB = *psOOB;

        // Send event
        bPosted = bPostEvent(
            psFSM,
            SMS_EVENT_OPTION_NONBLOCK,
            &sFsmEvent
                );
    }

    return bPosted;
}

/*******************************************************************************
 *
 *   SXI_FSM_bPostAudioEvent
 *
 *******************************************************************************/
BOOLEAN SXI_FSM_bPostAudioEvent (
    STI_HDL hControlCxn,
    SXI_FSM_AUDIO_EVENT_STRUCT const *psAudioEvent
        )
{
    BOOLEAN bPosted = FALSE;

    // Check input
    if(hControlCxn != STI_INVALID_HDL)
    {
        SXI_FSM_HDL hFSM;

        // Extract FSM handle
        hFSM = SXI_hFSM(hControlCxn);
        {
            BOOLEAN bValid;

            bValid = SMSO_bValid((SMS_OBJECT)hFSM);
            if(bValid == TRUE)
            {
                SXI_FSM_LINK_STRUCT *psFSM =
                    (SXI_FSM_LINK_STRUCT *)hFSM;
                SXI_FSM_EVENT_STRUCT sFsmEvent;

                // Prepare event
                sFsmEvent.eEventType = SXI_FSM_EVENT_AUDIO;
                sFsmEvent.uEventData.sAudioEvent = *psAudioEvent;

                // Send event
                bPosted = bPostEvent(
                    psFSM,
                    SMS_EVENT_OPTION_NONBLOCK,
                    &sFsmEvent
                        );
            }
        }
    }

    return bPosted;
}

/*******************************************************************************
 *
 *   SXI_FSM_vRun
 *
 *******************************************************************************/
void SXI_FSM_vRun (
    SXI_FSM_HDL hFSM,
    SXI_FSM_EVENT_STRUCT const *psFsmEvent
        )
{
    BOOLEAN bOk;
    SXI_FSM_LINK_STRUCT *psFSM =
        (SXI_FSM_LINK_STRUCT *)hFSM;
    BOOLEAN bSelfTransition = FALSE;
    SXI_FSM_LINK_STATE_ENUM eCurrentState;
    SXI_FSM_LINK_STATE_ENUM ePreviousState;
    SXI_FSM_LINK_STATE_ENUM eNextState;

    // Check input
    bOk = SMSO_bOwner((SMS_OBJECT)hFSM);
    if(bOk == FALSE)
    {
        return;
    }

    // Make local assignments
    eCurrentState = eNextState = psFSM->eCurrentState;
    ePreviousState = psFSM->ePreviousState;

#if DEBUG_OBJECT == 1
    printf(OBJECT_NAME
        ":vFsmSxiLink: Current State:%s, Event Type:%s (%u)\n",
        pacFsmSxiLinkStateText(eCurrentState),
        pacFsmEventText(psFsmEvent->eEventType),
        psFsmEvent->eEventType
            );
#endif

    // Any ERROR event triggers a transition to the ERROR state
    if(psFsmEvent->eEventType == SXI_FSM_EVENT_ERROR)
    {
        ePreviousState = eCurrentState;
        eCurrentState = eNextState = SXI_FSM_LINK_STATE_ERROR;
        psFSM->eErrorCode = (SXI_FSM_ERROR_CODE_ENUM)
            psFsmEvent->uEventData.eModuleErrorCode;
    }

    switch(eCurrentState)
    {
        case SXI_FSM_LINK_STATE_INITIAL:
        {
            // On entry...
            if(eCurrentState != ePreviousState)
            {
                psFSM->eErrorCode = eFsmEnterInitState(psFSM);
                if(psFSM->eErrorCode != SXI_FSM_RADIO_OK)
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }

            // Handle events...
            if(psFsmEvent->eEventType == SXI_FSM_EVENT_START)
            {
                // Transition
                eNextState = SXI_FSM_LINK_STATE_START;
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_SELF_TRANSITION)
            {
                // Do nothing.
            }

            // On exit...
            if(eCurrentState != eNextState)
            {
                // Nothing
            }
        }
        break;

        case SXI_FSM_LINK_STATE_START:
        {
            // On entry...
            if(eCurrentState != ePreviousState)
            {
                psFSM->eErrorCode = eFsmResetLink(psFSM);
                if(psFSM->eErrorCode != SXI_FSM_RADIO_OK)
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }

            // Handle events...
            if(psFsmEvent->eEventType == SXI_FSM_EVENT_TIMEOUT)
            {
                // A timeout causes a re-try to occur
                if(psFSM->un8ResetRetries < LINK_RESET_LIMIT)
                {
                    psFSM->un8ResetRetries++;

                    printf(OBJECT_NAME
                        ": Self Reset requested %d times\n",
                        psFSM->un8ResetRetries
                            );

                    psFSM->eErrorCode = eFsmResetLink(psFSM);
                    if(psFSM->eErrorCode != SXI_FSM_RADIO_OK)
                    {
                        // Error!
                        eNextState = SXI_FSM_LINK_STATE_ERROR;
                        break;
                    }
                }
                else
                {
                    // We've used all our allowable resets and
                    // resetting apparently doesn't correct the problem
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        OBJECT_NAME": Reset limit exceeded.");

                    // Error! Go to ERROR state
                    psFSM->eErrorCode = SXI_FSM_RADIO_DEVICE_ERROR;
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                }
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_SELF_TRANSITION)
            {
                // Actions & Guard Conditions
                psFSM->eErrorCode = eFsmActionStartLink(psFSM);
                if(psFSM->eErrorCode == SXI_FSM_RADIO_COMM_ERROR)
                {
                	// If the module is already OFF due to low-voltage,
                	// there is no need to try again
                	if (fc_sxm_bIsSMSLIB_CvOn())
                	{
                		eNextState = SXI_FSM_LINK_STATE_ERROR;
                		break;
                	}

                	// Try again
                    bSelfTransition = TRUE;
                }
                else if(psFSM->eErrorCode == SXI_FSM_RADIO_OK)
                {
                    // Transition
                    eNextState = SXI_FSM_LINK_STATE_CONFIG;
                }
                else
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }

            // On exit...
            if(eCurrentState != eNextState)
            {
                psFSM->eErrorCode = eFsmExitStartState(psFSM);
                if(psFSM->eErrorCode != SXI_FSM_RADIO_OK)
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }
        }
        break;

        case SXI_FSM_LINK_STATE_CONFIG:
        {
            // On entry...
            if(eCurrentState != ePreviousState)
            {
                psFSM->eErrorCode = eFsmEnterConfigState(psFSM);
                if(psFSM->eErrorCode != SXI_FSM_RADIO_OK)
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }

            // Handle events...
            if(psFsmEvent->eEventType == SXI_FSM_EVENT_TIMEOUT)
            {
                // Transition
                eNextState = SXI_FSM_LINK_STATE_START;
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_MODULE_CFG_IND_RXD)
            {
                OSAL_RETURN_CODE_ENUM eReturnCode;

                // Indicate Module Config Ind has been rxd
                psFSM->bModuleCfgIndRxd = TRUE;

                eReturnCode = OSAL.eTimerStop(psFSM->hModCfgIndWaitTimer);
                if ((eReturnCode != OSAL_SUCCESS) &&
                    (eReturnCode != OSAL_TIMER_NOT_ACTIVE))
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }

                // Check if we can go on
                if(psFSM->bModuleCfgIndRxd && psFSM->bModuleEsnRxd)
                {
                    // Transition
                    eNextState = SXI_FSM_LINK_STATE_READY;
                }
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_MODULE_ESN_RXD)
            {
                // Indicate ESN has been rxd
                psFSM->bModuleEsnRxd = TRUE;

                // Check if we can go on
                if(psFSM->bModuleCfgIndRxd && psFSM->bModuleEsnRxd)
                {
                    // Transition
                    eNextState = SXI_FSM_LINK_STATE_READY;
                }
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_POWER_CONTROL)
            {
                // Transition
                if(psFsmEvent->uEventData.ePwrMode == SXI_FSM_POWER_MODE_OFF)
                {
                    eNextState = SXI_FSM_LINK_STATE_PWR_DOWN;
                }
                else if(psFsmEvent->uEventData.ePwrMode == SXI_FSM_POWER_MODE_LOW)
                {
                    // We would probably handle the low power mode here
                    // but for now there is no transition anywhere
                }
            }
            else if( (psFsmEvent->eEventType == SXI_FSM_EVENT_OOB_MSG) &&
                     (psFSM->bModuleCfgIndRxd) )
            {
                // Action & Transition
                psFSM->eErrorCode = eFsmHandleOOBEvent(psFSM,
                    &psFsmEvent->uEventData.sOOB);
                if(psFSM->eErrorCode != SXI_FSM_RADIO_OK)
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_SELF_TRANSITION)
            {
                // Do nothing.
            }
        }
        break;

        case SXI_FSM_LINK_STATE_READY:
        {
            // On entry...
            if(eCurrentState != ePreviousState)
            {
                psFSM->eErrorCode = eFsmEnterReadyState(psFSM);
                if(psFSM->eErrorCode != SXI_FSM_RADIO_OK)
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }

            // Handle events...
            if(psFsmEvent->eEventType == SXI_FSM_EVENT_OOB_MSG)
            {
                // Action & Transition
                psFSM->eErrorCode = eFsmHandleOOBEvent(psFSM,
                    &psFsmEvent->uEventData.sOOB);
                if(psFSM->eErrorCode != SXI_FSM_RADIO_OK)
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_AUDIO)
            {
                // Action & Transition
                psFSM->eErrorCode = eFsmHandleAudioEvent(psFSM,
                    &psFsmEvent->uEventData.sAudioEvent);
                if(psFSM->eErrorCode != SXI_FSM_RADIO_OK)
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_POWER_CONTROL)
            {
                // Transition
                if(psFsmEvent->uEventData.ePwrMode == SXI_FSM_POWER_MODE_OFF)
                {
                    eNextState = SXI_FSM_LINK_STATE_PWR_DOWN;
                }
                else if(psFsmEvent->uEventData.ePwrMode == SXI_FSM_POWER_MODE_LOW)
                {
                    // TODO: We would handle the low power mode here.
                }
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_RADIO_POWER_STATE)
            {
                // Transition
                if(psFsmEvent->uEventData.ePwrMode == SXI_FSM_POWER_MODE_OFF)
                {
                    // Radio has indicated it has powered off
                    eNextState = SXI_FSM_LINK_STATE_FINAL;
                }
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_SELF_TRANSITION)
            {
                // Do nothing.
            }

            // On exit...
            if(eCurrentState != eNextState)
            {
                psFSM->eErrorCode = eFsmExitReadyState(psFSM);
                if ((psFSM->eErrorCode != SXI_FSM_RADIO_OK) &&
                    (eNextState != SXI_FSM_LINK_STATE_PWR_DOWN))
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }
        }
        break;

        case SXI_FSM_LINK_STATE_PWR_DOWN:
        {
            // On entry...
            if(eCurrentState != ePreviousState)
            {
                psFSM->eErrorCode = eFsmEnterPwrDownState(psFSM);
                if(psFSM->eErrorCode != SXI_FSM_RADIO_OK)
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }

            // Handle events...
            if(psFsmEvent->eEventType == SXI_FSM_EVENT_RADIO_POWER_STATE)
            {
                // Transition
                if(psFsmEvent->uEventData.ePwrMode == SXI_FSM_POWER_MODE_OFF)
                {
                    // Radio has indicated it has powered off
                    eNextState = SXI_FSM_LINK_STATE_FINAL;
                }
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_SELF_TRANSITION)
            {
                // Do nothing.
            }

            // On exit...
            if(eCurrentState != eNextState)
            {
                // Nothing
            }
        }
        break;

        case SXI_FSM_LINK_STATE_ERROR:
        {
            // On entry...
            if(eCurrentState != ePreviousState)
            {
                vFsmEnterErrorState(psFSM);
            }

            // Handle events...
            if(psFsmEvent->eEventType == SXI_FSM_EVENT_POWER_CONTROL)
            {
                // Transition
                if(psFsmEvent->uEventData.ePwrMode == SXI_FSM_POWER_MODE_OFF)
                {
                    eNextState = SXI_FSM_LINK_STATE_FINAL;
                }
            }
            else if(psFsmEvent->eEventType == SXI_FSM_EVENT_SELF_TRANSITION)
            {
                // Do nothing.
            }

            // On exit...
            if(eCurrentState != eNextState)
            {
                // Nothing
            }
        }
        break;

        case SXI_FSM_LINK_STATE_FINAL:
        {
            // On entry...
            if(eCurrentState != ePreviousState)
            {
                psFSM->eErrorCode = eFsmEnterFinalState(psFSM);
                if(psFSM->eErrorCode != SXI_FSM_RADIO_OK)
                {
                    // Error!
                    eNextState = SXI_FSM_LINK_STATE_ERROR;
                    break;
                }
            }

            // Handle events...
            if(psFsmEvent->eEventType == SXI_FSM_EVENT_SELF_TRANSITION)
            {
                // Do nothing.
            }

            // On exit...
            if(eCurrentState != eNextState)
            {
                // Nothing
            }
        }
        break;

        default:
        {
            // Unknown state, fix it.
            eNextState = SXI_FSM_LINK_STATE_INITIAL;
        }
        break;

    }

    // Apply new state
    psFSM->ePreviousState = eCurrentState;
    psFSM->eCurrentState = eNextState;

    // Check if there needs to be a state transition (event),
    // or a self-transition.
    if((eCurrentState != eNextState) || (bSelfTransition == TRUE))
    {
        SXI_FSM_EVENT_STRUCT sFsmEvent;

        // Transition
        sFsmEvent.eEventType =
            SXI_FSM_EVENT_SELF_TRANSITION; // event ignored
        bPostEvent(psFSM,
            SMS_EVENT_OPTION_NONE, &sFsmEvent);
    }

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*******************************************************************************
 *
 *   eFsmEnterInitState
 *
 *******************************************************************************/
static SXI_FSM_ERROR_CODE_ENUM eFsmEnterInitState (
    SXI_FSM_LINK_STRUCT *psFSM
        )
{
    // Nothing to do.

    return SXI_FSM_RADIO_OK;
}

/*******************************************************************************
 *
 *   eFsmResetLink
 *
 *******************************************************************************/
static SXI_FSM_ERROR_CODE_ENUM eFsmResetLink (
    SXI_FSM_LINK_STRUCT const *psFSM
        )
{
    SXI_FSM_ERROR_CODE_ENUM eErrorCode = SXI_FSM_RADIO_OBJECT_ERROR;
    N32 n32Err;
    BOOLEAN bAppResetControl = FALSE;

    puts(OBJECT_NAME": Resetting SRM...\n");

    // Determine if application desired to control the hardware lines
    n32Err = ioctl(
        psFSM->psDevice, SRH_IOCTL_GET_APP_RESET_CONTROL, 
        psFSM->pacSRMName, &bAppResetControl);
    if(n32Err != DEV_OK)
    {
        // Error!
        return eErrorCode;
    }

    if (bAppResetControl == TRUE)
    {
        // Ask the driver to perform an SRM reset.
        n32Err = ioctl(
            psFSM->psDevice, SRH_IOCTL_APP_RESET, psFSM->pacSRMName);
    }
    else
    {
        // Perform an SRM reset.
        n32Err = ioctl(
            psFSM->psDevice, SRH_IOCTL_RESET, psFSM->pacSRMName);
    }

    if(n32Err == DEV_OK)
    {
        printf(OBJECT_NAME": Setting baud rate %s...\n",
            gasUARTSettings[SXI_DEFAULT_COM_SETTINGS_INDEX].pcSettings);

        // Set the port settings to the initial setting
        n32Err =  ioctl(
            psFSM->psDevice,
            COM_IOCTL_CHANGE_SETTINGS,
            gasUARTSettings[SXI_DEFAULT_COM_SETTINGS_INDEX].pcSettings );
        if(n32Err == DEV_OK)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            eReturnCode = OSAL.eTimerStartRelative(
                psFSM->hStartUpTimer,
                MODULE_LINK_STARTUP_TIMER_MS, 0);
            if (eReturnCode == OSAL_SUCCESS)
            {
                // All is well
                eErrorCode = SXI_FSM_RADIO_OK;
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    OBJECT_NAME": Can't Start wait timer.");
            }
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    OBJECT_NAME" Cannot change settings on '%s' to '%s'.\n",
                        psFSM->pacSRMName,
                        gasUARTSettings[SXI_DEFAULT_COM_SETTINGS_INDEX].pcSettings);

            eErrorCode = SXI_FSM_RADIO_DEVICE_ERROR;
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME" Cannot reset on '%s'.\n", psFSM->pacSRMName);

        eErrorCode = SXI_FSM_RADIO_DEVICE_ERROR;
    }

    return eErrorCode;
}

/*******************************************************************************
 *
 *   eFsmActionStartLink
 *
 *******************************************************************************/
static SXI_FSM_ERROR_CODE_ENUM eFsmActionStartLink (
    SXI_FSM_LINK_STRUCT const *psFSM
        )
{
    SXI_FSM_ERROR_CODE_ENUM eErrorCode = SXI_FSM_RADIO_OBJECT_ERROR;
    SXIAPI_STATUSCODE_ENUM eUARTLinkStartCmdStatusCode;

    // Send Uart-Link Start Cmd, with baud rate request
    eUARTLinkStartCmdStatusCode =
        SXIAPI_eUARTLinkStartCmd(
            psFSM->hSxiLinkCxn,
            psFSM->eUARTControl
                );
    if(eUARTLinkStartCmdStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
    {
        N32 n32Err;

        printf(OBJECT_NAME": Setting new baud rate %s...\n",
            psFSM->pcUARTSettings);

        n32Err =  ioctl(
            psFSM->psDevice,
            COM_IOCTL_CHANGE_SETTINGS,
            psFSM->pcUARTSettings );
        if (n32Err == DEV_OK)
        {
            // All is well
            eErrorCode = SXI_FSM_RADIO_OK;

            // Delay for SXi compliance
            OSAL.eTaskDelay(TIME_TO_WAIT_BEFORE_SENDING_MODCFGCMD_MS);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                " Cannot change settings on '%s' to '%s'.\n",
                psFSM->pacSRMName,
                psFSM->pcUARTSettings
                    );

            eErrorCode = SXI_FSM_RADIO_DEVICE_ERROR;
        }
    }
    else if(eUARTLinkStartCmdStatusCode == SXIAPI_STATUSCODE_ERROR)
    {
        // Error! But not fatal
        printf(OBJECT_NAME
            ": SXIAPI_eUARTLinkStartCmd did not respond (%u).\n",
            eUARTLinkStartCmdStatusCode
                );

        // It's ok, we will try again via the FSM
        eErrorCode = SXI_FSM_RADIO_COMM_ERROR;
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            OBJECT_NAME": SXIAPI_eUARTLinkStartCmd responded with unexpected result (%u).\n",
            eUARTLinkStartCmdStatusCode
                );
    }

    return eErrorCode;
}

/*******************************************************************************
 *
 *   eFsmExitStartState
 *
 *******************************************************************************/
static SXI_FSM_ERROR_CODE_ENUM eFsmExitStartState (
    SXI_FSM_LINK_STRUCT *psFSM
        )
{
    SXI_FSM_ERROR_CODE_ENUM eErrorCode = SXI_FSM_RADIO_OBJECT_ERROR;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Stop the timer waiting for the Start
    eReturnCode = OSAL.eTimerStop(psFSM->hStartUpTimer);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // re-initialize link re-tries
        psFSM->un8ResetRetries = 0;

        eErrorCode = SXI_FSM_RADIO_OK;
    }

    return eErrorCode;
}

/*******************************************************************************
 *
 *   eFsmEnterConfigState
 *
 *******************************************************************************/
static SXI_FSM_ERROR_CODE_ENUM eFsmEnterConfigState (
    SXI_FSM_LINK_STRUCT const *psFSM
        )
{
    SXI_FSM_ERROR_CODE_ENUM eErrorCode = SXI_FSM_RADIO_OBJECT_ERROR;
    SXIAPI_STATUSCODE_ENUM eModuleConfigCmdStatusCode;

    // Issue the ModuleCfgCmd
    eModuleConfigCmdStatusCode =
        SXIAPI_eModuleCfgCmd(
        psFSM->hControlCxn,
        &psFSM->sModuleCfg
                );

    if(eModuleConfigCmdStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Everything has gone okay.  Start the mod cfg timer (one-time).
        // It'll be stopped if we rx the Module Ind, otherwise module is reset
        eReturnCode = OSAL.eTimerStartRelative(
            psFSM->hModCfgIndWaitTimer,
            MODCFGIND_WAIT_TIMEOUT_MS, 0);

        if (eReturnCode == OSAL_SUCCESS)
        {
            // All is well.
            eErrorCode = SXI_FSM_RADIO_OK;
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Can't Start wait timer.");
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            OBJECT_NAME": SXIAPI_eModuleCfgCmd responded with unexpected result (%u).\n",
            eModuleConfigCmdStatusCode
                );

        eErrorCode = SXI_FSM_RADIO_COMM_ERROR;
    }

    return eErrorCode;
}

/*******************************************************************************
 *
 *   bFsmEnterReadyState
 *
 *******************************************************************************/
static SXI_FSM_ERROR_CODE_ENUM eFsmEnterReadyState(
    SXI_FSM_LINK_STRUCT const *psFSM
        )
{
    SXI_FSM_ERROR_CODE_ENUM eErrorCode = SXI_FSM_RADIO_OBJECT_ERROR;
    BOOLEAN bSuccess = FALSE;

    do
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        // Enable the module-level monitors.
        // This needs to be done regardless of the capabilities.
        eStatusCode =
            SXIAPI_eFeatureMonCmd(
                psFSM->hControlCxn,
                SXIAPI_MONITOR_OP_ENABLE,
                sizeof(gaeModuleMonitorItems) /
                    sizeof(gaeModuleMonitorItems[0]),
                &gaeModuleMonitorItems[0]
                    );
        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": SXIAPI_eFeatureMonCmd(ENABLE) failed. "
                "Status Code %d", eStatusCode);

            eErrorCode = SXI_FSM_RADIO_COMM_ERROR;
            break;
        }

        // All is well

        // Tell MODULE the radio is ready...
        bSuccess = MODULE_bRadioReady(
            psFSM->hModule, psFSM->hControlCxn);
        if(bSuccess == TRUE)
        {
            eErrorCode = SXI_FSM_RADIO_OK;
        }

    } while (FALSE);

    return eErrorCode;
}

/*******************************************************************************
 *
 *   eFsmExitReadyState
 *
 *******************************************************************************/
static SXI_FSM_ERROR_CODE_ENUM eFsmExitReadyState(
    SXI_FSM_LINK_STRUCT const *psFSM
        )
{
    SXI_FSM_ERROR_CODE_ENUM eErrorCode = SXI_FSM_RADIO_OBJECT_ERROR;

    do
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        // Disable the module-level monitors.
        // This needs to be done regardless of the capabilities.
        eStatusCode =
            SXIAPI_eFeatureMonCmd(
                psFSM->hControlCxn,
                SXIAPI_MONITOR_OP_DISABLE,
                sizeof(gaeModuleMonitorItems) /
                    sizeof(gaeModuleMonitorItems[0]),
                &gaeModuleMonitorItems[0]
                    );
        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": SXIAPI_eFeatureMonCmd(DISABLE) failed. "
                "Status Code %d", eStatusCode);

            eErrorCode = SXI_FSM_RADIO_COMM_ERROR;
            break;
        }

        // All is well
        eErrorCode = SXI_FSM_RADIO_OK;

    } while (FALSE);

    return eErrorCode;
}

/*******************************************************************************
 *
 *   eFsmEnterPwrDownState
 *
 *******************************************************************************/
static SXI_FSM_ERROR_CODE_ENUM eFsmEnterPwrDownState (
    SXI_FSM_LINK_STRUCT *psFSM
        )
{
    SXI_FSM_ERROR_CODE_ENUM eErrorCode = SXI_FSM_RADIO_OBJECT_ERROR;

    do
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;
        SXIAPI_PWRMODE_ENUM ePwrMode;
        SMS_BEHAVIOR_VALUE_UNION uBehaviorValue;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32Timeout;

        // Now, start the force power off state timer (just in case)
        eReturnCode = OSAL.eTimerStartRelative(
            psFSM->hForcePwrOffStateTimer,
            DEFAULT_POWER_DOWN_TIMEOUT_MS, 0);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Can't Start Force power off state Timer.");
            break;
        }

        // Extract reset factory defaults flag
        SMS_bGetBehavior(SMS_BEHAVIOR_RESTORE_FACTORY_DEFAULTS,
            &uBehaviorValue);
        if(FALSE == uBehaviorValue.bValue)
        {
            ePwrMode = SXIAPI_PWRMODE_DOWN;
            un32Timeout = POWER_DOWN_MODE_TIMEOUT_MS;
        }
        else
        {
            ePwrMode = SXIAPI_PWRMODE_FACTORY_DEFAULT;
            un32Timeout = FACTORY_DEFAULT_MODE_TIMEOUT_MS;
        }

        // Tell MODULE to power down

        /*
            NOTE: the Module will send the PwrModeInd after the Module
            completes the power down sequence with requested storage
            of system persistent parameters.  The Host must wait for
            this PwrModeInd before removing power or resetting the
            Module. Host must reset the Module to reestablish the
            SXi link.
        */
        eStatusCode =
            SXIAPI_ePwrModeCmd(
                    psFSM->hControlCxn,
                    ePwrMode
                        );
        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": SXIAPI_ePwrModeCmd() failed. "
                "Status Code %d", eStatusCode);

            // No sense in waiting for something which will never
            // happen, so we just force the power-ctrl response on our own.
            un32Timeout = OSAL_OBJ_TIMEOUT_NONE;
        }

        // Now we wait for the PwrModeInd to come and finish up.
        // We shouldn't be receiving much from the radio any more

        // Now, restart the force power off state timer (just in case)
        // but now use appropriate timeout for type of power-ctrl
        // being performed.
        eReturnCode = OSAL.eTimerStartRelative(
            psFSM->hForcePwrOffStateTimer, un32Timeout, 0);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                ": Can't Start Force power off state Timer.");
            break;
        }

        // All is well
        eErrorCode = SXI_FSM_RADIO_OK;

    } while(FALSE);

    return eErrorCode;
}

/*******************************************************************************
 *
 *   vFsmEnterErrorState
 *
 *******************************************************************************/
static void vFsmEnterErrorState (
    SXI_FSM_LINK_STRUCT const *psFSM
        )
{
    BOOLEAN bSuccess;
    MODULE_ERROR_CODE_ENUM eModuleErrorCode;

    // Translate FSM error code into a MODULE error code
    switch(psFSM->eErrorCode)
    {
        case SXI_FSM_RADIO_OBJECT_ERROR:
            eModuleErrorCode = MODULE_ERROR_CODE_RADIO_OBJECT_ERROR;
        break;

        case SXI_FSM_RADIO_COMM_ERROR:
            eModuleErrorCode = MODULE_ERROR_CODE_RADIO_COMM_FAILURE;
        break;

        case SXI_FSM_RADIO_DEVICE_ERROR:
            eModuleErrorCode = MODULE_ERROR_CODE_RADIO_DEVICE_ERROR;
        break;

        default: // Anything else is just a general error
            eModuleErrorCode = MODULE_ERROR_CODE_GENERAL;
        break;
    }

    // The link state is now in ERROR, so we simply report this error
    // back to the MODULE object. This will cause the MODULE object to go
    // into the ERROR state along with all sub-DECODER objects and
    // data services as well.
    bSuccess = MODULE_bSetError(psFSM->hModule, eModuleErrorCode);
    if(bSuccess == FALSE)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            OBJECT_NAME": Unable to set MODULE error (%u).",
            eModuleErrorCode
                );
    }

    return;
}

/*******************************************************************************
 *
 *   eFsmEnterFinalState
 *
 *******************************************************************************/
static SXI_FSM_ERROR_CODE_ENUM eFsmEnterFinalState (
    SXI_FSM_LINK_STRUCT *psFSM
        )
{
    SXI_FSM_ERROR_CODE_ENUM eErrorCode = SXI_FSM_RADIO_OBJECT_ERROR;
    BOOLEAN bSuccess;

    // Initialize FSM
    psFSM->un8ResetRetries = 0;
    psFSM->eErrorCode = SXI_FSM_RADIO_OK;

    // Post a RADIO event to shutdown
    // DEFERRED event means that all remaining events
    // will be processed before radio layer un-initialization.
    bSuccess = RADIO_bPostModuleEvent(
        psFSM->hModule,
        (SMS_EVENT_TYPE_ENUM)RADIO_EVENT_SHUTDOWN,
        SMS_EVENT_OPTION_DEFERRED, NULL);
    if(bSuccess == FALSE)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            OBJECT_NAME": Can't send SHUTDOWN event.");
    }
    else
    {
        // All is well.
        eErrorCode = SXI_FSM_RADIO_OK;
    }

    return eErrorCode;
}

/*******************************************************************************
 *
 *   eFsmHandleOOBEvent
 *
 *******************************************************************************/
static SXI_FSM_ERROR_CODE_ENUM eFsmHandleOOBEvent (
    SXI_FSM_LINK_STRUCT const *psFSM,
    STI_PROTOCOL_OOB_STRUCT const *psOOB
        )
{
    SXI_FSM_ERROR_CODE_ENUM eErrorCode = SXI_FSM_RADIO_OBJECT_ERROR;
    STI_OOB_TYPE_ENUM eType = psOOB->eType;
    SXILL_RX_ERROR_ENUM eError =
        (SXILL_RX_ERROR_ENUM)(size_t)psOOB->pvData;

    // Determine what type of OOB happened
    if(eType == STI_OOB_TYPE_LINK_ERROR)
    {
        // Handle OOB Msg regarding link-stuff
        switch(eError)
        {
            case SXILL_RX_ERROR_NO_RX:
            {
                SXIAPI_STATUSCODE_ENUM eStatusCode;

                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,
                     __LINE__, OBJECT_NAME
                    ": SXILL_RX_ERROR_NO_RX receievd.");

                // Since the link has been established at one
                // point, but now there has been no rx activity
                // let's just try to ping the module and see if
                // it responds. If not, we're done.
                eStatusCode =
                    SXIAPI_ePingCmd(psFSM->hControlCxn);
                if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
                {
                    // Nevermind, don't panic. Everything is ok
                    eErrorCode = SXI_FSM_RADIO_OK;
                }
                else
                {
                    // Nevermind, signal an error, wait for app
                    // to do something about it.
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,
                            __LINE__,
                        OBJECT_NAME
                        ": SXIAPI_ePingCmd() failed. "
                        "Status Code %d", eStatusCode);

                    // Transition to the error state
                    eErrorCode = SXI_FSM_RADIO_COMM_ERROR;
                }
            }
            break;

            case SXILL_RX_ERROR_DEVICE:
            {
                // In this case, the Rx-Device (perhaps a driver)
                // has experienced an un recoverable error.

                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,
                        __LINE__,
                    OBJECT_NAME
                    ": SXILL_RX_ERROR_DEVICE() receievd.");

                // Transition to the error state
                eErrorCode = SXI_FSM_RADIO_DEVICE_ERROR;
            }
            break;

            default:
                // Don't know this one, ignore it.
            break;
        }
    }

    return eErrorCode;
}

/*******************************************************************************
 *
 *   eFsmHandleAudioEvent
 *
 *******************************************************************************/
static SXI_FSM_ERROR_CODE_ENUM eFsmHandleAudioEvent (
    SXI_FSM_LINK_STRUCT const *psFSM,
    SXI_FSM_AUDIO_EVENT_STRUCT const *psAudioEvent
        )
{
    SXI_FSM_ERROR_CODE_ENUM eErrorCode = SXI_FSM_RADIO_OK;
    SXIAPI_STATUSCODE_ENUM eStatusCode;

    if(psAudioEvent->bInitialize == TRUE)
    {
        // Now initialize radio hw
        eStatusCode = eInitAudioCmds(psFSM, &psAudioEvent->sStatus);
        if(eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Cannot initialize audio-cmds. "
                "Status Code %d", eStatusCode);

            eErrorCode = SXI_FSM_RADIO_COMM_ERROR;
        }
    }
    else
    {
        // Now uninitialize radio hw
        eStatusCode = eUnInitAudioCmds(psFSM);
        if(eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Cannot uninitialize audio-cmds. "
                "Status Code %d", eStatusCode);

            eErrorCode = SXI_FSM_RADIO_COMM_ERROR;
        }
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *   vFsmTimeoutCallback
 *
 *****************************************************************************/
static void vFsmTimeoutCallback (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        )
{
    SXI_FSM_EVENT_STRUCT sFsmEvent;
    SXI_FSM_LINK_STRUCT const *psFSM =
        (SXI_FSM_LINK_STRUCT const *)pvArg;

    // The timer has expired, post the timeout event
    sFsmEvent.eEventType = SXI_FSM_EVENT_TIMEOUT;
    bPostEvent(psFSM, SMS_EVENT_OPTION_NONE, &sFsmEvent);

    return;
}

/*****************************************************************************
*
*   vPullTheChuteTimerCallback
*
*****************************************************************************/
static void vPullTheChuteTimerCallback(
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        )
{
    SMSE_bPostEvent((SMS_EVENT_HDL) pvArg);
    return;
}

/*******************************************************************************
 *
 *   bDecoderInitAudioCmds
 *
 *******************************************************************************/
static SXIAPI_STATUSCODE_ENUM eInitAudioCmds (
    SXI_FSM_LINK_STRUCT const *psFSM,
    SXI_FSM_STATUS_MON_STRUCT const *psMonitorStatus
        )
{
    SXIAPI_STATUSCODE_ENUM eStatusCode;

    do
    {
        BOOLEAN bSuccess;
        TAG_OBJECT hParentTag;
        UN64 un64ExtMetadataGMIMask;
        BOOLEAN bIRSupported;
        UN8 un8FeatureMonitorItems = sizeof(gaeAudioMonitorItems) /
            sizeof(gaeAudioMonitorItems[0]);
        SXIAPI_STATUS_ITEM_ENUM aeAudioStatusItems[] = {
            SXIAPI_STATUS_ITEM_INVALID,
            SXIAPI_STATUS_ITEM_INVALID,
            SXIAPI_STATUS_ITEM_INVALID
        };
        UN8 un8StatusMonitorItems = 0;

        // Unmute audio to prepare for a tune
        eStatusCode =
            SXIAPI_eAudioMuteCmd(
                psFSM->hControlCxn,
                SXIAPI_MUTE_UNMUTE
                    );

        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Failed to set Audio Mute Command.");
            break;
        }

        if (psMonitorStatus->bSignal == TRUE)
        {
            aeAudioStatusItems[un8StatusMonitorItems++] =
                SXIAPI_STATUS_ITEM_SIGNAL;
        }

        if (psMonitorStatus->bAntenna == TRUE)
        {
            aeAudioStatusItems[un8StatusMonitorItems++] =
                SXIAPI_STATUS_ITEM_ANTENNA_AIMING;
        }

        if (psMonitorStatus->bAudioPresence == TRUE)
        {
            aeAudioStatusItems[un8StatusMonitorItems++] =
                SXIAPI_STATUS_ITEM_AUDIO_PRESENCE;
        }

        if (un8StatusMonitorItems > 0)
        {
            // Set Status Monitor Items
            eStatusCode =
                SXIAPI_eStatusMonCmd(
                    psFSM->hControlCxn,
                    SXIAPI_MONITOR_OP_ENABLE,
                    un8StatusMonitorItems,
                    &aeAudioStatusItems[0]
                        );

            if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
            {
                // Error
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    OBJECT_NAME": SXIAPI_eStatusMonCmd() failed. "
                    "Status Code %d", eStatusCode);
                break;
            }
        }

        // Enable feature monitor items

        // Check IR support by the Module
        bIRSupported = MODULE_bIsIRSupported(psFSM->hModule);
        if (bIRSupported == FALSE)
        {
            // Don't include IR
            // Note: This assumes that IR is the last in the list
            // and we are just gonna leave it off.
            if(un8FeatureMonitorItems > 0)
            {
                un8FeatureMonitorItems--;
            }
        }

        eStatusCode =
            SXIAPI_eFeatureMonCmd(
                psFSM->hControlCxn,
                SXIAPI_MONITOR_OP_ENABLE,
                un8FeatureMonitorItems,
                &gaeAudioMonitorItems[0]
                    );
        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": SXIAPI_eFeatureMonCmd() failed. "
                "Status Code %d", eStatusCode);
            break;
        }

        // Disable all Extended Metadata Monitor Items.
        eStatusCode =
            SXIAPI_eExtMetadataMonCmd(
                psFSM->hControlCxn,
                SXIAPI_EXT_GLOBAL_METADATA_MONITOR, // Note:
                // per SXi spec this value is ignored
                SXIAPI_MONITOR_OP_DISABLE_ALL,
                0,
                NULL
                    );

        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                ": Failed to disable ALL Extended Metadata.");
            break;
        }

        // Enable ext metadata monitor items (Track Metadata for Active/Browsed)
        eStatusCode =
            SXIAPI_eExtMetadataMonCmd(
                psFSM->hControlCxn,
                SXIAPI_EXT_TRACK_METADATA_MONITOR_ACTIVE_BROWSED,
                SXIAPI_MONITOR_OP_ENABLE,
                sizeof(gatExtMetadataTID) /
                    sizeof(gatExtMetadataTID[0]),
                &gatExtMetadataTID[0]
                    );
        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                ": Failed to enable Extended Track Metadata Active/Browsed.");
            break;
        }

        // Enable ext metadata monitor items (Track Metadata for all chans)
        eStatusCode =
            SXIAPI_eExtMetadataMonCmd(
                psFSM->hControlCxn,
                SXIAPI_EXT_TRACK_METADATA_MONITOR_ALL_AVAIL,
                SXIAPI_MONITOR_OP_ENABLE,
                sizeof(gatExtMetadataTID) /
                    sizeof(gatExtMetadataTID[0]),
                &gatExtMetadataTID[0]
                    );

        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                ": Failed to enable Extended Track Metadata All.");
            break;
        }

        // Enable ext metadata monitor items (Channel Metadata for all chans)
        eStatusCode =
            SXIAPI_eExtMetadataMonCmd(
                psFSM->hControlCxn,
                SXIAPI_EXT_CHANNEL_METADATA_MONITOR_ALL_AVAIL,
                SXIAPI_MONITOR_OP_ENABLE,
                sizeof(gatExtMetadataCID) /
                    sizeof(gatExtMetadataCID[0]),
                &gatExtMetadataCID[0]
                    );

        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                ": Failed to enable Extended Channel Metadata.");
            break;
        }

        // Disable all GMD
        bSuccess = SXI_GMD_bConfigure(
            psFSM->hControlCxn, SXIAPI_GMI_ALL,
            FALSE);

        if (bSuccess == FALSE)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                ": Failed to disable all Global MetaData.");
            eStatusCode = SXIAPI_STATUSCODE_ERROR;
            break;
        }

        // Check for existance of the radio specific market tag
        // absense of tag indicates no file system
        hParentTag = REPORT_hGetRadioSpecificTag();
        if (hParentTag == TAG_INVALID_OBJECT)
        {
            // if no file system, don't bother with teams
            // because they are only stored in db
            un64ExtMetadataGMIMask =
                (SXIAPI_GMI_MASK_ITUNES_URL |
                    SXIAPI_GMI_MASK_TRAFFIC_TABLE_VER |
                    SXIAPI_GMI_MASK_CITY_COUNT |
                    SXIAPI_GMI_MASK_LEAGUE_TABLE_VER |
                    SXIAPI_GMI_MASK_LEAGUE_CNT );
        }
        else
        {
            un64ExtMetadataGMIMask =
                (SXIAPI_GMI_MASK_ITUNES_URL |
                    SXIAPI_GMI_MASK_TRAFFIC_TABLE_VER |
                    SXIAPI_GMI_MASK_CITY_COUNT |
                    SXIAPI_GMI_MASK_TEAM_TABLE_VER |
                    SXIAPI_GMI_MASK_TEAM_CNT |
                    SXIAPI_GMI_MASK_LEAGUE_TABLE_VER |
                    SXIAPI_GMI_MASK_LEAGUE_CNT );
        }


        bSuccess = SXI_GMD_bConfigure(
            psFSM->hControlCxn, un64ExtMetadataGMIMask,
            TRUE);

        if (bSuccess == FALSE)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                ": Failed to enable Global MetaData.");
            eStatusCode = SXIAPI_STATUSCODE_ERROR;
            break;
        }

    } while (FALSE);

    return eStatusCode;
}

/*******************************************************************************
 *
 *   eUnInitAudioCmds
 *
 *******************************************************************************/
static SXIAPI_STATUSCODE_ENUM eUnInitAudioCmds (
    SXI_FSM_LINK_STRUCT const *psFSM
        )
{
    SXIAPI_STATUSCODE_ENUM eStatusCode;

    do
    {
        // Disable all Extended Metadata Monitor Items.
        eStatusCode =
            SXIAPI_eExtMetadataMonCmd(
                psFSM->hControlCxn,
                SXIAPI_EXT_GLOBAL_METADATA_MONITOR, // Note:
                // per SXi spec this value is ignored
                SXIAPI_MONITOR_OP_DISABLE_ALL,
                0,
                NULL
                    );

        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME
                ": Failed to disable ALL Extended Metadata.");
            break;
        }

        // Disable audio feature monitor items
        eStatusCode =
            SXIAPI_eFeatureMonCmd(
                psFSM->hControlCxn,
                SXIAPI_MONITOR_OP_DISABLE,
                sizeof(gaeAudioMonitorItems) /
                    sizeof(gaeAudioMonitorItems[0]),
                &gaeAudioMonitorItems[0]
                    );

        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Failed to disable FeatureMonCmd.");
            break;
        }

        // Mute audio
        eStatusCode =
            SXIAPI_eAudioMuteCmd(
                psFSM->hControlCxn,
                SXIAPI_MUTE_AUDIO_AND_I2S
                    );

        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Failed to mute AudioMuteCmd.");
            break;
        }

        // Clear Status Monitor Items
        eStatusCode =
            SXIAPI_eStatusMonCmd(
                psFSM->hControlCxn,
                SXIAPI_MONITOR_OP_DISABLE,
                sizeof(gaeAudioStatusItems) /
                    sizeof(gaeAudioStatusItems[0]),
                &gaeAudioStatusItems[0]
                    );

        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Failed to clear StatusMonCmd.");
            break;
        }

    } while (FALSE);

    return eStatusCode;
}

/*****************************************************************************
 *
 *   bGetSXIModuleConfiguration
 *
 *****************************************************************************/
static BOOLEAN bGetSXIModuleConfiguration(
    SXI_FSM_LINK_STRUCT const *psFSM,
    SXI_MODULE_CFG_STRUCT *psSXiCfg
        )
{
    BOOLEAN bResult = FALSE;

    if ((psFSM->psDevice != NULL) && (psSXiCfg != NULL))
    {
        N32 n32Err;

        // Request the module configuration information
        n32Err = ioctl(psFSM->psDevice,
                    SRH_IOCTL_MODULE_CONFIGURATION,
                    psFSM->pacSRMName,
                    psFSM->tId,
                    psSXiCfg
                        );
        if (n32Err == DEV_OK)
        {
            bResult = TRUE;
        }
        else
        {
            printf(OBJECT_NAME
                ": Cannot get Module configuration (Error: %d)",
                n32Err);
        }
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bPostEvent
 *
 *****************************************************************************/
static BOOLEAN bPostEvent (
    SXI_FSM_LINK_STRUCT const *psFSM,
    EVENT_OPTIONS_TYPE tOptions,
    SXI_FSM_EVENT_STRUCT const *psEvent
        )
{
    BOOLEAN bPosted = FALSE;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    // Allocate an event
    hEvent =
        MODULE_hAllocateEvent(
            psFSM->hModule,
            (SMS_EVENT_TYPE_ENUM)RADIO_EVENT_SXI_FSM,
            tOptions,
            &puEventData
                );
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        // Populate message
        SXI_FSM_EVENT_STRUCT *psFsmEvent =
            (SXI_FSM_EVENT_STRUCT *)
                puEventData->sRadio.apvData;

        // Populate event
        *psFsmEvent = *psEvent;

        // Post event
        bPosted = SMSE_bPostEvent(hEvent);
        if(bPosted == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                OBJECT_NAME": Can't post event (#%u).",
                    RADIO_EVENT_SXI_FSM);
        }
    }

    return bPosted;
}

#if DEBUG_OBJECT == 1

/*****************************************************************************
*
*   pacFsmSxiLinkStateText
*
*****************************************************************************/
static const char *pacFsmSxiLinkStateText (
    SXI_FSM_LINK_STATE_ENUM eState
        )
{
    const char *pacReturnString = "UNKNOWN";

    switch (eState)
    {
        case SXI_FSM_LINK_STATE_INITIAL:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_LINK_STATE_INITIAL);
        break;

        case SXI_FSM_LINK_STATE_START:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_LINK_STATE_START);
        break;

        case SXI_FSM_LINK_STATE_CONFIG:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_LINK_STATE_CONFIG);
        break;

        case SXI_FSM_LINK_STATE_READY:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_LINK_STATE_READY);
        break;

        case SXI_FSM_LINK_STATE_PWR_DOWN:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_LINK_STATE_PWR_DOWN);
        break;

        case SXI_FSM_LINK_STATE_ERROR:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_LINK_STATE_ERROR);
        break;

        case SXI_FSM_LINK_STATE_FINAL:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_LINK_STATE_FINAL);
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   pacFsmEventText
*
*****************************************************************************/
static const char *pacFsmEventText (
    SXI_FSM_EVENT_TYPE_ENUM eEventType
        )
{
    const char *pacReturnString = "UNKNOWN";

    switch (eEventType)
    {
        case SXI_FSM_EVENT_SELF_TRANSITION:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_EVENT_SELF_TRANSITION);
        break;

        case SXI_FSM_EVENT_START:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_EVENT_START);
        break;

        case SXI_FSM_EVENT_TIMEOUT:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_EVENT_TIMEOUT);
        break;

        case SXI_FSM_EVENT_POWER_CONTROL:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_EVENT_POWER_CONTROL);
        break;

        case SXI_FSM_EVENT_RADIO_POWER_STATE:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_EVENT_RADIO_POWER_STATE);
        break;

        case SXI_FSM_EVENT_OOB_MSG:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_EVENT_OOB_MSG);
        break;

        case SXI_FSM_EVENT_MODULE_CFG_IND_RXD:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_EVENT_MODULE_CFG_IND_RXD);
        break;

        case SXI_FSM_EVENT_MODULE_ESN_RXD:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_EVENT_MODULE_ESN_RXD);
        break;

        case SXI_FSM_EVENT_AUDIO:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_EVENT_AUDIO);
        break;

        case SXI_FSM_EVENT_ERROR:
            pacReturnString = MACRO_TO_STRING(SXI_FSM_EVENT_ERROR);
        break;
    }

    return pacReturnString;
}

#endif

