/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Radio interface implementation for the
 *  Satellite Module Services (SMS).
 *
 *  The purpose of this interface is to abstract the common radio
 *  functions which either configure or extract data from the radio hardware.
 *  The idea is that in the future we may compile in-out or swap radio.c
 *  modules with other implementations.
 *
 ******************************************************************************/
#include <string.h>
#include <time.h>

#include "standard.h"
#include "osal.h"

#include "srh.h"
#include "com.h"

#include "sms_version.h"
#include "playback_obj.h"
#include "decoder_obj.h"
#include "sms_event.h"
#include "sms_event_types.h"
#include "sms_update.h"
#include "cdo_obj.h"
#include "channel_obj.h"
#include "category_obj.h"
#include "song_obj.h"
#include "scache.h"
#include "ccache.h"
#include "srm_obj.h"
#include "module_obj.h"
#include "string_obj.h"
#include "report_obj.h"
#include "league_obj.h"
#include "preset_band_obj.h"
#include "presets_obj.h"
#include "sti_api.h"
#include "dataservice_base.h"
#include "dataservice_mgr_obj.h"
#include "sms_obj.h"

#include "module_version_obj.h"
#include "detailed_signal_quality_obj.h"
#include "detailed_overlay_signal_quality_obj.h"
#include "link_status_information_obj.h"

#include "sports_flash_obj.h"

// Protocol Specific Includes (SXi)

#include "song_tag_id.h"

#include "sports.h"
#include "sxi_id.h"
#include "sxi_sports_id.h"
#include "sxi_market_id.h"
#include "sxi_artist_id.h"
#include "sxi_song_id.h"
#include "sxi_fsm.h"
#include "sxi_gmd.h"

#include "sxill.h"
#include "sxi.h"

// Protocol Specific Includes (SDTP)
#include "sdtp.h"

// Radio includes
#include "radio_event_types.h"
#include "radio.h"
#include "_radio.h"

#include "sms_fcsxm.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   RADIO_bInitializeSMS
 *
 *   This function is used to perform an SMS initialization of anything
 *   RADIO specific. This function is called once when SMS itself is
 *   initialized. Anything initialized here should be available to all of
 *   SMS while a specific instance of SMS exists.
 *
 *   Inputs:
 *       None.
 *
 *   Returns:
 *       BOOLEAN - TRUE if initialization is successful. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bInitializeSMS(
    void)
{
    BOOLEAN bInitialized = FALSE;

    do
    {
        BOOLEAN bRegistered;

        // We need to register these CIDs
        bRegistered = CID_bRegister(&GsSxiId);
        bRegistered &= CID_bRegister(&GsSxiArtistId);
        bRegistered &= CID_bRegister(&GsSxiSongId);
        bRegistered &= CID_bRegister(&GsSxiMarketId);
        bRegistered &= CID_bRegister(&GsSxiSportsId);
        if (bRegistered == FALSE)
        {
            break;
        }

        bInitialized = TRUE;
    }
    while(FALSE);

    return bInitialized;
}

/*****************************************************************************
 *
 *   RADIO_vUninitializeSMS
 *
 *   This function is used to uninitialize anything previously initialized
 *   by the RADIO_bInitializeSMS() call. This function is called once
 *   when SMS is finally uninitialized.
 *
 *   Inputs:
 *       None.
 *
 *   Returns:
 *       None.
 *
 *****************************************************************************/
void RADIO_vUninitializeSMS(
    void)
{
    return;
}

/*****************************************************************************
 *
 *   RADIO_bInitializeSrm
 *
 *   This function is used to initialize a SRM object and or physical
 *   hardware specific to a RADIO.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE SRM OBJECT.
 *
 *   Inputs:
 *       hSRM - A valid handle to an SRM object mapped to the physical
 *       hardware to initialize.
 *       pacDriverName - The SRH driver name associated with this
 *       physical device.
 *
 *   Returns:
 *       BOOLEAN - TRUE if initialization is successful. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bInitializeSrm(
    SRM_OBJECT hSRM,
    const char *pacName,
    const char *pacDriverName,
    SRH_DEVICE_CAPABILITIES_MASK *ptCapabilities,
    void *pvEventHdlr
        )
{
    RADIO_SRM_OBJECT_STRUCT *psSRM = NULL;
    BOOLEAN bSuccess;

    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
        "%s(): Init RADIO SRM...\n", __FUNCTION__);

    // Verify inputs, caller owns the SRM object
    bSuccess = SMSO_bOwner((SMS_OBJECT)hSRM);
    if (FALSE == bSuccess)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME": Not owner.");
        return FALSE;
    }

    if (NULL == pacDriverName)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME": pacDriverName is not specified.");
        return FALSE;
    }

    do
    {
        SRH_DEVICE_MODULE_TUPLE_STRUCT *psModuleTuple;
        UN32 un32NumModules;
        N32 n32Err;
        char acPortName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Allocate an SRM's radio specific data
        psSRM = (RADIO_SRM_OBJECT_STRUCT *)SMSO_hCreate(
            RADIO_OBJECT_NAME":SRM", sizeof(RADIO_SRM_OBJECT_STRUCT),
            (SMS_OBJECT)hSRM, // SRM OBJECT is parent, inherit lock-feature
            FALSE );
        if (NULL == psSRM)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot create RADIO SRM object.");
            break;
        }

        // Set radio specific SRM data
        bSuccess = SRM_bSetRadioSpecificData(hSRM,
            (RADIO_PRIVATE_DATA_OBJECT)psSRM);
        if (FALSE == bSuccess)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot set RADIO SRM data.");
            break;
        }

        // Associate event handler of parent (for allocating messages)
        psSRM->hEventHdlr = (SMS_EVENT_HANDLER) pvEventHdlr;

        // Associate provided SRM handle with this object
        psSRM->hSRM = hSRM;

        // Extract SRM name
        psSRM->pacName = pacName;

        // Create name and open connection to SRM's Device Port (SXi)
        snprintf(&acPortName[0], sizeof(acPortName), "sxi:%s", psSRM->pacName);

        psSRM->psDevice = fopen(pacDriverName, &acPortName[0]);
        if(psSRM->psDevice == (FILE*)NULL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot open device '%s' with mode '%s'.\n"
                "Something is wrong with the physical device mapped to '%s'.\n"
                "Check your SRH driver initialization in your SRH header file.\n",
                pacDriverName, &acPortName[0], &acPortName[0]);
            break;
        }

        // Startup the SRM
        bSuccess = bPowerOnSRM(psSRM->psDevice, psSRM->pacName);
        if (FALSE == bSuccess)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot Power On SRM.");

            // Close Device now
            fclose(psSRM->psDevice);
            psSRM->psDevice = (FILE*)NULL;

            break;
        }

        // Ask the SRH driver about the modules associated with this SRM
        // This will allocate memory for psTuple which we will have to
        // clean up ourselves when we're done with it. Calling this ioctl
        // will help us learn what capabilities this SRM has.
        n32Err = ioctl(psSRM->psDevice, SRH_IOCTL_DESCRIBE_MODULES, psSRM->pacName,
            &psModuleTuple, &un32NumModules);
        if(n32Err == DEV_OK)
        {
            UN32 un32ModuleIndex;

            // Initialize capabilities
            psSRM->tCapabilities = SRH_DEVICE_CAPABILITY_NONE;

            // Iterate over the modules we discovered and
            // inspect our module descriptor list with
            // the relevant data
            for(un32ModuleIndex = 0; un32ModuleIndex < un32NumModules;
                un32ModuleIndex++)
            {
                // Process the relevant information

                // This SRM's capabilities are equal to the capabilities
                // of all it's constituent modules
                psSRM->tCapabilities
                        |= psModuleTuple[un32ModuleIndex].tModuleCapabilities;
            }

            // Return capabilities to caller
            if(ptCapabilities != NULL)
            {
                *ptCapabilities = psSRM->tCapabilities;
            }

            // Free the tuple's memory allocated by the IOCTL
            OSAL.vMemoryFree(psModuleTuple);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot get associated modules.");
            break;
        }

        // Tell SRM the radio is ready...
        bSuccess = SRM_bRadioReady(hSRM, psSRM->psDevice);
        if (FALSE == bSuccess)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Unable to post RADIO_READY to SRM.");
            break;
        }

        // If we got here, all is well.
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
            "%s(): RADIO SRM is initialized.\n", __FUNCTION__);

        return TRUE;

    } while(FALSE);

    // Error!
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        RADIO_OBJECT_NAME": Cannot initialize RADIO SRM.");

    vUninitializeSrm(psSRM);

    return FALSE;
}

/*****************************************************************************
 *
 *   RADIO_vUninitializeSrm
 *
 *   This function is used to uninitialize an SRM object and or physical
 *   hardware specific to a RADIO.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE SRM OBJECT.
 *
 *   Inputs:
 *       hSRM - A valid handle to an SRM object mapped to the physical
 *       hardware to uninitialize.
 *
 *   Returns:
 *       None.
 *
 *****************************************************************************/
void RADIO_vUninitializeSrm(
    SRM_OBJECT hSRM)
{
    RADIO_SRM_OBJECT_STRUCT *psSRM;

    // Obtain radio data
    psSRM = (RADIO_SRM_OBJECT_STRUCT *)SRM_hGetRadioSpecificData(hSRM);
    if (NULL == psSRM)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME": RADIO SRM is not initialized.");
        return;
    }

    vUninitializeSrm(psSRM);

    return;
}

/*****************************************************************************
 *
 *   RADIO_bPostSrmEvent
 *
 *   This function is used to post an event to the SRM object, generally for
 *   the purposes of having that event handled by the RADIO specific
 *   SRM event handler.
 *
 *   THIS FUNCTION ALWAYS RUNS OUTSIDE THE CONTEXT OF THE SRM OBJECT. IT IS
 *   CALLED BY SMS INTERNALLY WHEN SRM_bPostRadioEvent() IS CALLED.
 *
 *   Inputs:
 *       hEvent - A valid event handle which has been obtained from the object
 *       this event is to be posted to. SMS allocates this event for the caller.
 *       pvArg - A caller defined argument which was provided when
 *       SRM_bPostRadioEvent() was called. The implementation must dereference
 *       this argument appropriately.
 *
 *   Returns:
 *       BOOLEAN - TRUE if post was successful. Otherwise
 *       FALSE is returned on error.
 *
 *****************************************************************************/
BOOLEAN RADIO_bPostSrmEvent(
    SMS_EVENT_HDL hEvent, void *pvArg)
{
    // We don't have any SRM RADIO events
    return FALSE;
}

/*****************************************************************************
 *
 *   RADIO_bSrmEventHandler
 *
 *   This event handler is called by SMS whenever the SRM object needs
 *   to process an event and that event has not been handled by SMS itself
 *   (meaning the event should be handled by specific RADIO hardware).
 *
 *   NOTE: This handler always runs in the context of the SRM object
 *   thus it should be assumed the implementation here has exclusive access
 *   to the object already.
 *
 *   Inputs:
 *       hSRM - A valid handle to a SRM object mapped to the physical
 *           hardware to process this event.
 *       hEvent - A handle to the event being processed.
 *
 *   Returns:
 *       BOOLEAN - TRUE if this handler handled the event. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bSrmEventHandler(
    SRM_OBJECT hSRM,
    SMS_EVENT_TYPE_ENUM eEventType,
    const void *pvEventData
        )
{
    // We don't have any SRM RADIO events
    return FALSE;
}


/*****************************************************************************
 *
 *   RADIO_tGetUniqueModuleId
 *
 *   This function is used to retrieve a unique Module Id from the
 *   SRH driver for a particular Module belonging to the specified decoder.
 * *
 *   Inputs:
 *       hSRM - A valid SRM handle for which to identify the MODULE
 *       which contains the DECODER specified by name.
 *       pacDecoderName - A valid decoder name for which to find a unique
 *       MODULE id that contains it.
 *
 *   Returns:
 *       MODULE_ID - >=0 if a decoder with the provided name was found.
 *       Otherwise a value < 0 is returned if not.
 *
 *****************************************************************************/
MODULE_ID RADIO_tGetUniqueModuleId(
    SRM_OBJECT hSRM,
    const char *pacDecoderName
        )
{
    MODULE_ID tId = MODULE_ID_INVALID;
    BOOLEAN bLocked;

    // Obtain radio data from SRM
    bLocked = SMSO_bLock((SMS_OBJECT)hSRM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        const RADIO_SRM_OBJECT_STRUCT *psSRM;

        psSRM
            = (const RADIO_SRM_OBJECT_STRUCT *)SRM_hGetRadioSpecificData(
                hSRM);
        if(psSRM != NULL)
        {
            BOOLEAN bPrimary;
            N32 n32Err;

            // Extract physical I/O device handle and consult driver for
            // an id of a module which contains the provided
            // decoder by name.
            n32Err = ioctl(psSRM->psDevice, SRH_IOCTL_MODULE_EXISTS,
                pacDecoderName, &tId, &bPrimary);

            if (n32Err != DEV_OK)
            {
                tId = MODULE_ID_INVALID;
            }

        }

        // We're done with SRM
        SMSO_vUnlock((SMS_OBJECT)hSRM);
    }

    return tId;
}

/*****************************************************************************
 *
 *   RADIO_bInitializeModule
 *
 *   This function is used to initialize a MODULE object and or physical
 *   hardware specific to a RADIO.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE MODULE OBJECT.
 *
 *   Inputs:
 *       hModule - A valid handle to a MODULE object mapped to the physical
 *       hardware to initialize.
 *       tId - A unique module id.
 *       hSRM - A valid SRM handle to associate this MODULE with.
 *       ptCapabilities - A pointer to a device capabilities mask supplied
 *       by the call to the RADIO specific function.
 *       pvEventHdlr - The parent object's event handler handle. This is needed
 *       when pre-allocation of events is required.
 *
 *   Returns:
 *       BOOLEAN - TRUE if initialization is successful. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bInitializeModule(
    MODULE_OBJECT hModule,
    MODULE_ID tId,
    SRM_OBJECT hSRM,
    const char *pacSRMName,
    FILE *psDevice,
    void *pvEventHdlr
        )
{
    BOOLEAN bSuccess;
    RADIO_MODULE_OBJECT_STRUCT *psModule;

    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
        "%s(): Init RADIO MODULE...\n", __FUNCTION__);

    // Verify inputs, caller own's the MODULE object
    bSuccess = SMSO_bOwner((SMS_OBJECT)hModule);
    if (FALSE == bSuccess)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME": Not owner.");
        return FALSE;
    }

    do
    {
        N32 n32Err;
        UN32 un32DecoderIndex;

        // Allocate a MODULE's radio specific data
        psModule = (RADIO_MODULE_OBJECT_STRUCT *)SMSO_hCreate(
            RADIO_OBJECT_NAME":Module", sizeof(RADIO_MODULE_OBJECT_STRUCT),
            (SMS_OBJECT)hModule, // MODULE Object is parent,
            FALSE                // inherit lock-feature
                        );
        if(psModule == NULL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Failed to create RADIO MODULE object.");
            break;
        }

        // Set radio specific MODULE data
        bSuccess = MODULE_bSetRadioSpecificData(hModule,
            (RADIO_PRIVATE_DATA_OBJECT)psModule);
        if(bSuccess == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Failed to set RADIO MODULE data.");
            break;
        }

        // Clear reset flag
        psModule->bReset = FALSE;

        // Associate event handler of parent (for allocating messages)
        psModule->hEventHdlr = (SMS_EVENT_HANDLER) pvEventHdlr;

        // Associate device
        psModule->psDevice = psDevice;

        // Associate MODULE handle
        psModule->hModule = hModule;

        // Make some explicit initializations of the RADIO specific data
        psModule->hSRM = hSRM;
        psModule->pacSRMName = pacSRMName;
        psModule->tId = tId;
        psModule->hESN = STRING_INVALID_OBJECT;
        psModule->pacESN = NULL;
        psModule->sSXI.eCurDispAdvisoryIndCode = SXIAPI_INDCODE_ERROR;
        psModule->sFWU.psFWUpdateFile = NULL;
        psModule->sFWU.un32FWUpdateBufferSize = 0;
        psModule->sFWU.un32FWUpdateCurrentPacket = 0;
        psModule->sFWU.un32FWUpdateNumberPackets = 0;
        psModule->sFWU.eFWUpdateState = RADIO_FWUPDATE_STATE_INITIAL;
        psModule->sFWU.hFWUpdateEraseTimer = OSAL_INVALID_OBJECT_HDL;
        psModule->sFWU.hEraseTimeoutEvent = SMS_INVALID_EVENT_HDL;

        // Initialize package pointer
        psModule->sSXI.sPkgInd.tNumBytes = 0;
        psModule->sSXI.sPkgInd.pun8PkgInd = NULL;

        // Initialize connection
        psModule->sSXI.hControlCxn = STI_INVALID_HDL;

        // for now we initialize.
        // At some point we need to retrieve last known values from the
        // config file if they are present.
        psModule->sSubstatus.eStatus = SXIAPI_SUB_STATUS_INVALID;
        psModule->sSubstatus.tReasonCode = MODULE_SUBSTATUS_REASON_CODE_INVALID;
        psModule->sSubstatus.un32UTCTime = 0;
        psModule->sSubstatus.hReasonText = STRING_INVALID_OBJECT;
        psModule->sSubstatus.hPhoneNumber = STRING_INVALID_OBJECT;
        psModule->psDecoderTuple = NULL;
        psModule->un32NumDecoders = 0;

        // Ask the SRH driver about the DECODERs.
        // This will allocate memory for psTuple which we will have to
        // clean up ourselves when we're done with it. Calling this ioctl
        // will help us learn what capabilities this MODULE has.
        n32Err = ioctl(psModule->psDevice,
            SRH_IOCTL_DESCRIBE_DECODERS,
            psModule->tId,
            &psModule->psDecoderTuple, &psModule->un32NumDecoders);
        if(n32Err != DEV_OK)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Failed to get associated decoders.");
            break;
        }

        // Initialize capabilities
        psModule->tCapabilities = SRH_DEVICE_CAPABILITY_NONE;

        // Iterate over the decoders we discovered and
        // inspect our decoder descriptor list with
        // the relevant data
        for(un32DecoderIndex = 0; un32DecoderIndex <
                psModule->un32NumDecoders;
            un32DecoderIndex++
                )
        {
            // Process the relevant information

            // This MODULE's capabilities are equal to the
            // capabilities of all it's constituent decoders
            psModule->tCapabilities |=
                psModule->psDecoderTuple[un32DecoderIndex].
                tDecoderCapabilities;
        }

        // Initialize FSM
        psModule->sSXI.hFSM = SXI_FSM_hInit(
            psModule->pacSRMName, psModule->psDevice,
            psModule->hModule, psModule->tCapabilities,
            &psModule->sSXI.hControlCxn);
        if(psModule->sSXI.hFSM == SXI_FSM_INVALID_HDL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Can't init SXi-FSM.");
            break;
        }

        // If we got here, all is well.
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
            "%s(): RADIO MODULE is initialized.\n", __FUNCTION__);

        return TRUE;

    } while(FALSE);

    // Error!
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        RADIO_OBJECT_NAME": RADIO MODULE initialization failed.");

    vUninitializeModule(psModule);

    return FALSE;
}

/*****************************************************************************
 *
 *   RADIO_vUninitializeModule
 *
 *   This function is used to uninitialize a MODULE object and or physical
 *   hardware specific to a RADIO.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE MODULE OBJECT.
 *
 *   Inputs:
 *       hModule - A valid handle to a MODULE object mapped to the physical
 *       hardware to uninitialize.
 *
 *   Returns:
 *       None.
 *
 *****************************************************************************/
void RADIO_vUninitializeModule(
    MODULE_OBJECT hModule,
    BOOLEAN bReset
        )
{
    RADIO_MODULE_OBJECT_STRUCT *psModule;
    BOOLEAN bPosted;

    // Obtain radio data
    psModule = (RADIO_MODULE_OBJECT_STRUCT *)
        MODULE_hGetRadioSpecificData(hModule);
    if (NULL == psModule)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME": RADIO MODULE is not initialized.");
        return;
    }

    // Post command to power down the module
    bPosted = SXI_FSM_bPostPowerControl(
        psModule->sSXI.hFSM, SXI_FSM_POWER_MODE_OFF);
    if (TRUE == bPosted)
    {
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
            "%s(): POWER_CONTROL event is posted.\n", __FUNCTION__);

        // Set reset flag
        psModule->bReset = bReset;

        // All is done here
        return;
    }

    // Error!
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        RADIO_OBJECT_NAME": Can't Post POWER_CONTROL event.");

    // Signal the MODULE an error has occurred
    bPosted = MODULE_bSetError(hModule,
        MODULE_ERROR_CODE_RADIO_OBJECT_ERROR);
    if (FALSE == bPosted)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME": Unable to post MODULE event (ERROR).");

        // As one last ditch, try this....
        bPosted = MODULE_bRadioReleased(hModule);
        if (FALSE == bPosted)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Unable to post MODULE event (RADIO_RELEASED).");

            // Sorry! That is all I can do.
        }
    }

    return;
}

/*****************************************************************************
 *
 *   RADIO_bModuleEventHandler
 *
 *   This event handler is called by SMS whenever the MODULE object needs
 *   to process an event and that event has not been handled by SMS itself
 *   (meaning the event should be handled by specific RADIO hardware).
 *
 *   NOTE: This handler always runs in the context of the MODULE object
 *   thus it should be assumed the implementation here has exclusive access
 *   to the object already.
 *
 *   Inputs:
 *       hModule - A valid handle to a MODULE object mapped to the physical
 *           hardware to process this event.
 *       hEvent - A handle to the event being processed.
 *
 *   Returns:
 *       BOOLEAN - TRUE if this handler handled the event. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bModuleEventHandler(
    MODULE_OBJECT hModule,
    SMS_EVENT_TYPE_ENUM eEventType,
    const void *pvEventData
        )
{
    SMS_EVENT_DATA_UNION const *puEventData =
        (SMS_EVENT_DATA_UNION const *)pvEventData;
    BOOLEAN bHandledEvent = TRUE;
    RADIO_MODULE_OBJECT_STRUCT *psModule;

    // Obtain radio data
    psModule = (RADIO_MODULE_OBJECT_STRUCT *)MODULE_hGetRadioSpecificData(
        hModule);
    if(psModule == NULL)
    {
        // Error! Cannot obtain RADIO data
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME": Unexpected RADIO MODULE event: %d",
            eEventType);
        return FALSE;
    }

    // Process event
    switch((size_t)eEventType)
    {
        case RADIO_EVENT_SXI_IND:
        {
            RADIO_EVENT_UNION *puRadio = (RADIO_EVENT_UNION *)
                &puEventData->sRadio.apvData[0];
            SXIAPI_RX_STRUCT *psRxData = puRadio->psRxData;

            // Handle SXi Ind Rx
            bHandledEvent = bHandleModuleInd(psModule, psRxData);
        }
        break;

        case RADIO_EVENT_SXI_FSM:
        {
            RADIO_EVENT_UNION *puRadio =
                (RADIO_EVENT_UNION *)
                    &puEventData->sRadio.apvData[0];

            // Run module FSM
            SXI_FSM_vRun(psModule->sSXI.hFSM, &puRadio->sFsmEvent);
        }
        break;

        case RADIO_EVENT_SHUTDOWN:
        {
            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
                "RADIO_EVENT_SHUTDOWN\n");

            // We now tell the MODULE to either STOP or INITIALIZE
            if(psModule->bReset == TRUE)
            {
                MODULE_ERROR_CODE_ENUM eErrorCode;

                eErrorCode = MODULE_eInitialize(hModule);
                if(eErrorCode != MODULE_ERROR_CODE_NONE)
                {
                    // Error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        RADIO_OBJECT_NAME": Unable to initialize MODULE.");

                    bSetModuleError(psModule, eErrorCode);
                }
            }
            else
            {
                BOOLEAN bPosted;

                bPosted = MODULE_bRadioReleased(hModule);
                if (FALSE == bPosted)
                {
                    // Error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        RADIO_OBJECT_NAME": Unable to post RADIO_RELEASED event.");

                    // Sorry! That is all I can do.
                    bSetModuleError(psModule, MODULE_ERROR_CODE_EVENT_POST_ERROR);
                }
            }

            // Finish up uninitialization
            vUninitializeModule(psModule);

            // Note, at this point the pointer psModule
            // is no longer valid!
        }
        break;

        case RADIO_EVENT_FWUPDATE:
        {
            RADIO_EVENT_UNION *puRadio =
                (RADIO_EVENT_UNION *)&puEventData->sRadio.apvData[0];
            RADIO_FWUPDATE_STRUCT sFWUpdateInfo =
                puRadio->sFirmwareUpdate;

            // Run firmware update process state machine
            vProcessFWUpdateEvent(psModule, &sFWUpdateInfo);
        }
        break;

        default:
        {
            // Unhandled event
            bHandledEvent = FALSE;
        }
        break;
    }

    return bHandledEvent;
}

/*****************************************************************************
 *
 *   RADIO_tDecoderCapabilities
 *
 *   This function is used to ask the parent MODULE what a specific
 *   DECODER's capabilities are.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE MODULE OBJECT.
 *
 *   Returns:
 *       SRH_DEVICE_CAPABILITIES_MASK
 *
 *****************************************************************************/
SRH_DEVICE_CAPABILITIES_MASK RADIO_tDecoderCapabilities (
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder
        )
{
    SRH_DEVICE_CAPABILITIES_MASK tCapabilities = SRH_DEVICE_CAPABILITY_NONE;
    N32 n32Err;
    const char *pacDecoderName;
    DECODER_ID tDecoderId = DECODER_ID_INVALID;

    RADIO_MODULE_OBJECT_STRUCT *psModule;

    // Obtain radio data
    psModule = (RADIO_MODULE_OBJECT_STRUCT *)MODULE_hGetRadioSpecificData(
        hModule);
    if(psModule == NULL)
    {
        // Error! Cannot obtain RADIO data
        return SRH_DEVICE_CAPABILITY_NONE;
    }

    // Extract DECODER name from DECODER object
    pacDecoderName = DECODER_pacName(hDecoder);

    // Check if this decoder name exists
    n32Err = ioctl(psModule->psDevice, SRH_IOCTL_DECODER_EXISTS,
        pacDecoderName, &tDecoderId);
    if (n32Err == DEV_OK)
    {
        UN32 un32DecoderIndex;

        // Check if we already have the tuple
        if(psModule->psDecoderTuple == NULL)
        {
            // Ask the SRH driver about the decoders.
            // This will allocate memory for psTuple which we will have to
            // clean up ourselves when we're done with it. Calling this ioctl
            // will help us learn what capabilities this DECODER has.
            n32Err = ioctl(psModule->psDevice,
                SRH_IOCTL_DESCRIBE_DECODERS,
                psModule->tId,
                &psModule->psDecoderTuple, &psModule->un32NumDecoders);
        }

        if (n32Err == DEV_OK)
        {
            // Iterate over the decoders we discovered and
            // inspect our decoder descriptor list with
            // the relevant data
            for (un32DecoderIndex = 0;
                 un32DecoderIndex < psModule->un32NumDecoders;
                 un32DecoderIndex++)
            {
                // Process the relevant information for our specific decoder
                if (tDecoderId ==
                    psModule->psDecoderTuple[un32DecoderIndex].tDecoderPathId)
                {
                    // Extract this DECODER's capabilities
                    tCapabilities = psModule->psDecoderTuple[un32DecoderIndex].
                        tDecoderCapabilities;
                    break;
                }
            }
        }
    }

    return tCapabilities;
}

/*****************************************************************************
 *
 *   RADIO_bInitializeDecoder
 *
 *   This function is used to initialize a DECODER object and or physical
 *   hardware specific to a RADIO based on it's capabilities.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   Returns:
 *       BOOLEAN - TRUE if initialization is successful. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bInitializeDecoder(
    DECODER_OBJECT hDecoder,
    SRH_DEVICE_CAPABILITIES_MASK tCapabilities,
    STI_HDL hSTI,
    MODULE_SUBSTATUS_ENUM eSubStatus,
    MODULE_SUBSTATUS_REASON_CODE tSubReasonCode,
    UN32 un32SubSuspendDate,
    void *pvEventHdlr
        )
{
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;
    BOOLEAN bSuccess;

    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
        "%s(): Init RADIO DECODER...\n", __FUNCTION__);

    // Verify inputs, caller own's the DECODER object
    bSuccess = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (FALSE == bSuccess)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME": Not owner.");
        return FALSE;
    }

    do
    {
        // Allocate a DECODER's radio specific data
        psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)SMSO_hCreate(
            RADIO_OBJECT_NAME":Decoder", sizeof(RADIO_DECODER_OBJECT_STRUCT),
            (SMS_OBJECT)hDecoder, // DECODER Object is parent,
            FALSE                 // inherit lock-feature
                        );
        if (NULL == psDecoder)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot create RADIO DECODER object.");
            break;
        }

        // Set radio specific DECODER data
        bSuccess = DECODER_bSetRadioSpecificData(hDecoder,
            (RADIO_PRIVATE_DATA_OBJECT)psDecoder);
        if (FALSE == bSuccess)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot set RADIO DECODER data.");
            break;
        }

        // recondition flag
        bSuccess = FALSE;

        // Associate event handler of parent (for allocating messages)
        psDecoder->hEventHdlr = (SMS_EVENT_HANDLER) pvEventHdlr;

        // Associate DECODER handle
        psDecoder->hDecoder = hDecoder;

        // Initialize STI connection handle
        psDecoder->hControlCxn = hSTI;

        // Extract CCache Handle
        psDecoder->hCCache = DECODER_hCCache(hDecoder);

        // Initialize the state variables
        psDecoder->eState = DECODER_STATE_INITIAL;

        // Just initialize this as TRUE to abandon gating the
        // DECODER object becoming ready based on ChanId-0
        // reception. Otherwise, leave it as FALSE.
        psDecoder->bChannel0IndRxd = FALSE;

        // Initialize subscription status info
        psDecoder->sSubStatus.eStatus = eSubStatus;
        psDecoder->sSubStatus.tReasonCode = tSubReasonCode;
        psDecoder->sSubStatus.un32SubSuspendDate = un32SubSuspendDate;

        // Initialize pending service ID
        psDecoder->sRadioSelection.eSelType = SXIAPI_SELECT_TYPE_SID;
        psDecoder->sRadioSelection.uId.tService = SERVICE_INVALID_ID;
        psDecoder->sRadioSelection.tCatId = SXIAPI_CATEGORYID_SERVICENOTASSIGNED;
        psDecoder->sRadioSelection.tOverrides = SXIAPI_CHANNELATTR_NONE;
        psDecoder->sRadioSelection.tNominalId = SERVICE_INVALID_ID;

        // Update the decoder state based on the current
        // DispAdvisoryIndCode received by the module

        // We cannot block the DECODER from getting this, but we
        // can at least prevent a race-condition, between update
        // and reading this value.
        // TODO: Note this doesn't work if we have multiple MODULE
        // objects. So we must revisit if that can happen.
        OSAL.eEnterTaskSafeSection();
        psDecoder->eCurDispAdvisoryIndCode = geDispAdvisoryIndCode;
        OSAL.eExitTaskSafeSection();

        vUpdateDisplayAdvisories(
            psDecoder, psDecoder->eCurDispAdvisoryIndCode, TRUE);

        // Initialize the current signal strength
        psDecoder->sSignalQuality.eComposite = SIGNAL_QUALITY_INVALID;
        psDecoder->sSignalQuality.eSatellite = SIGNAL_QUALITY_INVALID;
        psDecoder->sSignalQuality.eTerrestrial = SIGNAL_QUALITY_INVALID;

        // Initialize Category Vector
        OSAL.bMemSet(&psDecoder->sCategory.aun8CatIDVector[0], 0,
            sizeof(psDecoder->sCategory.aun8CatIDVector));

        // Initialize decoder id
        psDecoder->tDecoderId = DECODER_ID_INVALID;

        // Remember DECODER capabilities
        psDecoder->tCapabilities = tCapabilities;

        // Initialize based on this DECODER's capabilities
        if((psDecoder->tCapabilities & SRH_DEVICE_CAPABILITY_AUDIO)
                        == SRH_DEVICE_CAPABILITY_AUDIO)
        {
            bSuccess = bInitializeAudioDecoder(psDecoder);
        }
        else if((psDecoder->tCapabilities & SRH_DEVICE_CAPABILITY_VIDEO)
                        == SRH_DEVICE_CAPABILITY_VIDEO)
        {
            // Nothing special to do here.  No one wants video anyways
            bSuccess = TRUE;
        }
        else if((psDecoder->tCapabilities & SRH_DEVICE_CAPABILITY_DATA)
                        == SRH_DEVICE_CAPABILITY_DATA)
        {
            // We don't use data decoders for sxi
            bSuccess = FALSE;
        }

        if (FALSE == bSuccess)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot initialize specific Decoder.");
            break;
        }

        // If we got here, all is well.
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
            "%s(): RADIO DECODER is initialized.\n", __FUNCTION__);

        return TRUE;

    } while(FALSE);

    // Error!
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        RADIO_OBJECT_NAME": RADIO DECODER initialization failed.");

    vUninitializeDecoder(psDecoder);

    return FALSE;
}

/*****************************************************************************
 *
 *   RADIO_vUninitializeDecoder
 *
 *   This function is used to uninitialize a DECODER object and or physical
 *   hardware specific to a RADIO based on it's capabilities.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *       hardware to uninitialize.
 *       tDeviceCapabilities - A mask indicating which one or more capabilities
 *       this DECODER has to guide the uninitialization.
 *
 *   Returns:
 *       None.
 *
 *****************************************************************************/
void RADIO_vUninitializeDecoder(
    DECODER_OBJECT hDecoder)
{
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);
    if (NULL == psDecoder)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME": RADIO DECODER is not initialized.");
        return;
    }

    vUninitializeDecoder(psDecoder);

    return;
}

/*****************************************************************************
 *
 *   RADIO_bDecoderEventHandler
 *
 *   This event handler is called by SMS whenever the DECODER object needs
 *   to process an event and that event has not been handled by SMS itself
 *   (meaning the event should be handled by specific RADIO hardware).
 *
 *   NOTE: This handler always runs in the context of the DECODER object
 *   thus it should be assumed the implementation here has exclusive access
 *   to the object already.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to process this event.
 *       hEvent - A handle to the event being processed.
 *
 *   Returns:
 *       BOOLEAN - TRUE if this handler handled the event. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bDecoderEventHandler(
    DECODER_OBJECT hDecoder,
    SMS_EVENT_TYPE_ENUM eEventType,
    const void *pvEventData
        )
{
    SMS_EVENT_DATA_UNION const *puEventData =
        (SMS_EVENT_DATA_UNION const *)pvEventData;
    BOOLEAN bHandledEvent = TRUE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);
    if (psDecoder == NULL)
    {
        // Decoder is not ready to accept events.
        if (((RADIO_EVENT_TYPE_ENUM)eEventType) == RADIO_EVENT_SXI_RX_DATA)
        {
            RADIO_EVENT_UNION *puRadio = (RADIO_EVENT_UNION *)
                &puEventData->sRadio.apvData[0];
            SXIAPI_RX_STRUCT *psRxData = puRadio->psRxData;

            // we don't have an active decoder, so free the indication
            SXI_vFreeResponse(psRxData);
        }
        else // other events are not handled
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Unhandled RADIO event while"
                " DECODER is not ready to accept events.");
            bHandledEvent = FALSE;
        }

        // All cases are 'handled' other than final else case.
        return bHandledEvent;
    }

    // Process event
    switch((size_t)eEventType)
    {
        case RADIO_EVENT_SXI_RX_DATA:
        {
            RADIO_EVENT_UNION *puRadio = (RADIO_EVENT_UNION *)
                &puEventData->sRadio.apvData[0];
            SXIAPI_RX_STRUCT *psRxData = puRadio->psRxData;

            // Process SXI Control Ind
            vProcessDecoderInd(psDecoder, psRxData);

            // Free data
            SXI_vFreeResponse(psRxData);
        }
        break;

        case SMS_EVENT_UPDATE_SIGNAL:
        {
            DECODER_vUpdateSignal(psDecoder->hDecoder,
                &psDecoder->sSignalQuality);
        }
        break;

        case SMS_EVENT_UPDATE_ANTENNA:
        {
            // We only have support for one antenna, so magic number
            DECODER_vUpdateAntennaState(psDecoder->hDecoder, 0,
                psDecoder->eAntennaState);
        }
        break;

        default:
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Unhandled RADIO event.");
            bHandledEvent = FALSE;
        break;
    };

    return bHandledEvent;
}

/*****************************************************************************
 *
 *   RADIO_tCategoryIdFromOffset
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   This function is used when the caller would like to return
 *   the category id of the category which is n16Offset from the provided
 *   category id. This is typically used for category navigation.
 *
 *****************************************************************************/
CATEGORY_ID RADIO_tCategoryIdFromOffset(
    DECODER_OBJECT hDecoder, CATEGORY_ID tCategoryId, N16 n16Offset)
{
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Verify inputs
    if(tCategoryId == CATEGORY_INVALID_ID)
    {
        // Error!
        return CATEGORY_INVALID_ID;
    }

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)DECODER_hGetRadioSpecificData(
        hDecoder);
    if(psDecoder != NULL)
    {
        N16 n16StartIndex, n16StartBit, n16CurrentIndex, n16CurrentBit;

        // Verify information
        if(psDecoder->sCategory.n16NumCategories == 0)
        {
            // Error!
            return CATEGORY_INVALID_ID;
        }

        // Offset cannot be larger than the number of categories our
        // vector represents. So we just truncate it now
        n16Offset = (abs(n16Offset) % (SXIAPI_CAT_ID_VECTOR_SIZE * 8));

        // Start at channel tCategoryId...

        // Determine the vector index and bit within the CSV
        // for the reference category. This is our starting point.
        // The index can be from 0 thru MAX-1 and bits can be 0-7
        n16StartIndex = (SXIAPI_CAT_ID_VECTOR_SIZE - (tCategoryId / 8) - 1);
        n16StartBit = (tCategoryId % 8);

        // Initialize current index and bit within it
        n16CurrentIndex = n16StartIndex;
        n16CurrentBit = n16StartBit;

        while(n16Offset != 0)
        {
            // Move one bit over
            n16CurrentBit++;

            // Adjust for wrap around in any direction
            if(n16CurrentBit >= 8)
            {
                n16CurrentBit = 0;
                n16CurrentIndex--;
            }
            if(n16CurrentBit < 0)
            {
                n16CurrentBit = 7;
                n16CurrentIndex--;
            }
            if(n16CurrentIndex >= SXIAPI_CAT_ID_VECTOR_SIZE)
            {
                n16CurrentIndex = 0;
                n16CurrentBit = 7;
            }
            if(n16CurrentIndex < 0)
            {
                n16CurrentIndex = SXIAPI_CAT_ID_VECTOR_SIZE - 1;
                n16CurrentBit = 0;
            }

            // If there is a valid category here, then we can move our offset
            if(psDecoder->sCategory.aun8CatIDVector[n16CurrentIndex]
                            & ((0x01 << n16CurrentBit) & 0xFF))
            {
                // Move offset to the next bit in the vector
                n16Offset--;
            }
        }

        // Calculate this category
        tCategoryId = ((SXIAPI_CAT_ID_VECTOR_SIZE - n16CurrentIndex - 1) * 8)
                        + n16CurrentBit;
    }

    return tCategoryId;
}

/*****************************************************************************
 *
 *   RADIO_n16CategoryOffsetFromId
 *
 *   Determine the offset within the list of valid broadcast categories for
 *   a particular category Id.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       tCategoryId - A valid category id.
 *
 *   Returns:
 *       >= 0 If the category id provided is valid and exists within the
 *       radio's list of known categories.
 *       < 0 If the category id provided is invalid or unknown by the radio.
 *
 *****************************************************************************/
N16 RADIO_n16CategoryOffsetFromId(
    DECODER_OBJECT hDecoder, CATEGORY_ID tCategoryId)
{
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;
    N16 n16Offset = -1;

    // Verify inputs
    if(tCategoryId == CATEGORY_INVALID_ID)
    {
        // Error!
        return -1;
    }

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)DECODER_hGetRadioSpecificData(
        hDecoder);
    if(psDecoder != NULL)
    {
        BOOLEAN bFound = FALSE;

        do
        {
            CATEGORY_ID tCurrentCategoryId = 0;
            N16 n16CurrentIndex, n16CurrentBit;

            // Verify information
            if(psDecoder->sCategory.n16NumCategories == 0)
            {
                // Error!
                break;
            }

            // Initialize offset
            n16Offset = 0;

            for(n16CurrentIndex = SXIAPI_CAT_ID_VECTOR_SIZE - 1; (n16CurrentIndex
                            >= 0) && (bFound == FALSE); n16CurrentIndex--)
            {
                for(n16CurrentBit = 0; n16CurrentBit < 8; n16CurrentBit++)
                {
                    // Check if there is a valid channel here.
                    if(psDecoder->sCategory.aun8CatIDVector[n16CurrentIndex]
                                    & ((0x01 << n16CurrentBit) & 0xFF))
                    {
                        // Check if it's our category
                        if(tCurrentCategoryId == tCategoryId)
                        {
                            // It's our category
                            bFound = TRUE;
                            break;
                        }

                        // Valid, category
                        n16Offset++;
                    }

                    tCurrentCategoryId++;
                }
            }

        }
        while(FALSE);

        // Check if we didn't find anything
        if(bFound == FALSE)
        {
            // Error. Cannot find it or it is invalid
            n16Offset = -1;
        }
    }

    return n16Offset;
}

/*****************************************************************************
 *
 *   RADIO_bTuneServiceId
 *
 *****************************************************************************/
BOOLEAN RADIO_bTuneServiceId(
    DECODER_OBJECT hDecoder,
    SERVICE_ID tServiceId,
    BOOLEAN bMature,
    BOOLEAN bLocked,
    BOOLEAN bSkipped,
    BOOLEAN bPlayUnrestricted
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    SXIAPI_SID tSID = (SXIAPI_SID)tServiceId;

    // Note that we didn't include SXIAPI_CHANNELATTR_FREE_TO_AIR here
    // According to Sec 4.2.5SX-9845-0097, that override works differently
    // from the other overrides.  If you set SXIAPI_CHANNELATTR_FREE_TO_AIR,
    // you are limited to only channels that are free-to-air.
    SXIAPI_CH_ATTRIB tOverrides = SXIAPI_CHANNELATTR_UNSUBSCRIBED;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(
                        hDecoder);
    if(psDecoder == NULL)
    {
        // Error!
        return FALSE;
    }

    // Configure overrides
    if(TRUE == bMature)
    {
        tOverrides |= SXIAPI_CHANNELATTR_MATURE;
    }
    if(TRUE == bLocked)
    {
        tOverrides |= SXIAPI_CHANNELATTR_LOCKED;
    }
    if(TRUE == bSkipped)
    {
        tOverrides |= SXIAPI_CHANNELATTR_SKIPPED;
    }
    if(TRUE == bPlayUnrestricted)
    {
        tOverrides |= SXIAPI_CHANNELATTR_PLAY_UNRESTRICTED;
    }


    // Tune...
    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
        "Requesting tune for service id:%u"
        " with overrides: 0x%x\n", tSID, tOverrides);

    // Request channel select (tune)
    eStatusCode = SXIAPI_eChanSelectCmd(
        psDecoder->hControlCxn,
        SXIAPI_SELECT_TYPE_SID,
        tSID,
        SXIAPI_CATEGORYID_SERVICENOTASSIGNED,
        tOverrides
            );
    if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
    {
        // Successful channel selected
        psDecoder->sRadioSelection.uId.tService = tSID;
        bSuccess = TRUE;
    }
    else
    {
        // This command failed. Channel not selected
        // due to bad sid, or overrides, timeout, etc. eitherway
        // some type of major failure, unknown.
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "RADIO Error! Requesting tune for service id:%u"
            " failed with error %u.\n", tSID, eStatusCode);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bScanContent
 *
 *****************************************************************************/
BOOLEAN RADIO_bScanContent (
    DECODER_OBJECT hDecoder,
    BOOLEAN bMusicOnly
        )
{
    BOOLEAN bSuccess;
    RADIO_CHANNEL_SELECTION_STRUCT sRadioSelection;

    sRadioSelection.tCatId = SXIAPI_CATEGORYID_SERVICENOTASSIGNED;

    if (bMusicOnly == TRUE)
    {
        sRadioSelection.eSelType = SXIAPI_SELECT_TYPE_SCAN_CONTENT_MUSIC;
    }
    else
    {
        sRadioSelection.eSelType = SXIAPI_SELECT_TYPE_SCAN_CONTENT_ALL;
    }

    bSuccess = bScan(hDecoder, &sRadioSelection);

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bScanBack
 *
 *****************************************************************************/
BOOLEAN RADIO_bScanBack (
    DECODER_OBJECT hDecoder,
    BOOLEAN bReverse
        )
{
    BOOLEAN bSuccess;
    RADIO_CHANNEL_SELECTION_STRUCT sRadioSelection;

    sRadioSelection.tCatId = SXIAPI_CATEGORYID_SERVICENOTASSIGNED;

    if (bReverse == TRUE)
    {
        sRadioSelection.eSelType = SXIAPI_SELECT_TYPE_SCAN_REVERSE;
    }
    else
    {
        sRadioSelection.eSelType = SXIAPI_SELECT_TYPE_SCAN_SKIP_BACK;
    }

    bSuccess = bScan(hDecoder, &sRadioSelection);

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bScanForward
 *
 *****************************************************************************/
BOOLEAN RADIO_bScanForward (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bSuccess;
    RADIO_CHANNEL_SELECTION_STRUCT sRadioSelection;

    sRadioSelection.tCatId = SXIAPI_CATEGORYID_SERVICENOTASSIGNED;
    sRadioSelection.eSelType = SXIAPI_SELECT_TYPE_SCAN_SKIP_FORWARD;

    bSuccess = bScan(hDecoder, &sRadioSelection);

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bScanTerminate
 *
 *****************************************************************************/
BOOLEAN RADIO_bScanTerminate (
    DECODER_OBJECT hDecoder,
    BOOLEAN bAbort
        )
{
    BOOLEAN bSuccess;
    RADIO_CHANNEL_SELECTION_STRUCT sRadioSelection;

    sRadioSelection.tCatId = SXIAPI_CATEGORYID_SERVICENOTASSIGNED;

    if (bAbort == TRUE)
    {
        sRadioSelection.eSelType = SXIAPI_SELECT_TYPE_SCAN_ABORT;
    }
    else
    {
        sRadioSelection.eSelType = SXIAPI_SELECT_TYPE_SCAN_STOP;
    }

    bSuccess = bScan(hDecoder, &sRadioSelection);

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bScanSelectCfg
 *
 *****************************************************************************/
BOOLEAN RADIO_bScanSelectCfg (
    DECODER_OBJECT hDecoder,
    BOOLEAN bMusicOnly,
    BOOLEAN bScanMature,
    BOOLEAN bScanLocked,
    BOOLEAN bScanSkipped
        )
{
    BOOLEAN bSuccess;
    RADIO_CHANNEL_SELECTION_STRUCT sRadioSelection;

    sRadioSelection.tCatId = SXIAPI_CATEGORYID_SERVICENOTASSIGNED;

    if (bMusicOnly == FALSE)
    {
    sRadioSelection.eSelType = SXIAPI_SELECT_TYPE_SCAN_CFG_ALL;
    }
    else
    {
        sRadioSelection.eSelType = SXIAPI_SELECT_TYPE_SCAN_CFG_MUSIC;
    }

    sRadioSelection.tOverrides = SXIAPI_CHANNELATTR_NONE;

    if (bScanMature == TRUE)
    {
        sRadioSelection.tOverrides |= SXIAPI_CHANNELATTR_MATURE;
    }

    if (bScanLocked == TRUE)
    {
        sRadioSelection.tOverrides |= SXIAPI_CHANNELATTR_LOCKED;
    }

    if (bScanSkipped == TRUE)
    {
        sRadioSelection.tOverrides |= SXIAPI_CHANNELATTR_SKIPPED;
    }

    bSuccess = bScan(hDecoder, &sRadioSelection);

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bScanItemsMon
 *
 *****************************************************************************/
BOOLEAN RADIO_bScanItemsMon (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        )
{
    BOOLEAN bSuccess = FALSE;
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;
    SXIAPI_MONITOR_OP_ENUM eOperation = SXIAPI_MONITOR_OP_DISABLE;

    const SXIAPI_STATUS_ITEM_ENUM aeStatusItems[] = {
        SXIAPI_STATUS_ITEM_SCAN_ITEMS_AVAILABLE
            };

    const UN8 aeStatusItemsCount =
        sizeof(aeStatusItems) / sizeof(SXIAPI_STATUS_ITEM_ENUM);

    // Obtain radio data
    psDecoder =
        (RADIO_DECODER_OBJECT_STRUCT *)
            DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        if (bEnable == TRUE)
        {
            eOperation = SXIAPI_MONITOR_OP_ENABLE;
        }

        // Set Status Monitor Items
        eStatusCode =
            SXIAPI_eStatusMonCmd(
                psDecoder->hControlCxn,
                eOperation,
                aeStatusItemsCount,
                &aeStatusItems[0]
                    );

        if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bSignalMonitorSwitch
 *
 *****************************************************************************/
BOOLEAN RADIO_bSignalMonitorSwitch (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        )
{
    BOOLEAN bSuccess;
    size_t tNumItems = 
        sizeof(gaeSignalStatusItems) / sizeof(gaeSignalStatusItems[0]);

    bSuccess = bStatusMonitorSwitch( hDecoder, 
        gaeSignalStatusItems, 
        tNumItems,
        bEnable );

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bAntennaAimingMonitorSwitch
 *
 *****************************************************************************/
BOOLEAN RADIO_bAntennaAimingMonitorSwitch (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        )
{
    BOOLEAN bSuccess;
    size_t tNumItems = 
        sizeof(gaeAntennaStatusItems) / sizeof(gaeAntennaStatusItems[0]);

    bSuccess = bStatusMonitorSwitch( hDecoder, 
        gaeAntennaStatusItems, 
        tNumItems,
        bEnable );

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bAudioPresenceMonitorSwitch
 *
 *****************************************************************************/
BOOLEAN RADIO_bAudioPresenceMonitorSwitch (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        )
{
    BOOLEAN bSuccess;
    size_t tNumItems = 
        sizeof(gaeAudioPresenceItems) / sizeof(gaeAudioPresenceItems[0]);

    bSuccess = bStatusMonitorSwitch( hDecoder, 
        gaeAudioPresenceItems, 
        tNumItems,
        bEnable );

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bRequestChannel
 *
 *   This API requests the most current channel information from the
 *   physical RADIO hardware.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the channel info from.
 *       tServiceId - A service id describing the channel which is to be
 *           used as a request to update.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bRequestChannel(
    DECODER_OBJECT hDecoder,
    SERVICE_ID tServiceId
        )
{
    BOOLEAN bRequested = FALSE;

    if(tServiceId != SERVICE_INVALID_ID)
    {
        RADIO_DECODER_OBJECT_STRUCT *psDecoder;

        // Obtain radio data
        psDecoder =
            (RADIO_DECODER_OBJECT_STRUCT *)
                DECODER_hGetRadioSpecificData(hDecoder);
        if (psDecoder != NULL)
        {
            SXIAPI_STATUSCODE_ENUM eStatusCode;
            SXIAPI_SID tSID = (SXIAPI_SID)tServiceId;

            eStatusCode = SXIAPI_eChanBrowseCmd(psDecoder->hControlCxn,
                SXIAPI_BROWSE_TYPE_SID, tSID,
                SXIAPI_CATEGORYID_SERVICENOTASSIGNED,
                // Note that we didn't include SXIAPI_CHANNELATTR_FREE_TO_AIR
                // here According to Sec 4.2.5SX-9845-0097, that override works
                // differently from the other overrides.  If you set
                // SXIAPI_CHANNELATTR_FREE_TO_AIR, you are limited to only
                // channels that are free-to-air.
                SXIAPI_CHANNELATTR_UNSUBSCRIBED |
                SXIAPI_CHANNELATTR_MATURE |
                SXIAPI_CHANNELATTR_LOCKED |
                SXIAPI_CHANNELATTR_SKIPPED
                    );
            if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
            {
                // Success
                bRequested = TRUE;
            }
        }
    }

    return bRequested;
}

/*****************************************************************************
 *
 *   RADIO_ePlay
 *
 *   This function is used to resume playback
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 * Inputs:
 *   hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to handle this API.
 *
 * Returns:
 *       PLAYBACK_ERROR_CODE_ENUM.
 *
 *****************************************************************************/
PLAYBACK_ERROR_CODE_ENUM RADIO_ePlay(
    DECODER_OBJECT hDecoder)
{
    PLAYBACK_ERROR_CODE_ENUM eReturnCode = PLAYBACK_ERROR_CODE_INVALID_OBJECT;
    const RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (const RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(
                        hDecoder);
    if(psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        // Request play
        eStatusCode = SXIAPI_eIRPlaybackControlCmd(
            psDecoder->hControlCxn,
             SXIAPI_IR_CONTROL_PLAY,
             0,
             0
                );

        if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            eReturnCode = PLAYBACK_ERROR_CODE_NONE;
        }
        else if(eStatusCode == SXIAPI_STATUSCODE_MSG_FAIL)
        {
            eReturnCode = PLAYBACK_ERROR_CODE_PLAYBACK_OP_FAILED;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   RADIO_ePause
 *
 *   This function is used to pause playback
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to handle this API.
 *
 *   Returns:
 *       PLAYBACK_ERROR_CODE_ENUM.
 *
 *****************************************************************************/
PLAYBACK_ERROR_CODE_ENUM RADIO_ePause(
    DECODER_OBJECT hDecoder)
{
    PLAYBACK_ERROR_CODE_ENUM eReturnCode = PLAYBACK_ERROR_CODE_INVALID_OBJECT;
    const RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (const RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(
                        hDecoder);
    if(psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        // Request pause
        eStatusCode = SXIAPI_eIRPlaybackControlCmd(
            psDecoder->hControlCxn,
            SXIAPI_IR_CONTROL_PAUSE,
            0,
            0
               );

        if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            eReturnCode = PLAYBACK_ERROR_CODE_NONE;
        }
        else if(eStatusCode == SXIAPI_STATUSCODE_MSG_FAIL)
        {
            eReturnCode = PLAYBACK_ERROR_CODE_PLAYBACK_OP_FAILED;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   RADIO_eSeek
 *
 *   This API requests a program seek into the physical radio's playback
 *   buffer. There are two kinds of seek (index and time) as well as an
 *   offset provided. There is also a choice the caller can make which
 *   is to either pause after the seek is performed or simply continue
 *   playback from that point onward.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the seek.
 *       eType - A radio seek type for which to interpret the provided offset
 *           value. Typically index or time.
 *       n16Offset - Either an index to seek or time in seconds from the current
 *           playback position.
 *       bPauseAfterSeek - TRUE if the radio is to pause after seeking,
 *           otherwise simply continue playback from that point onward.
 *
 *   Returns:
 *       PLAYBACK_ERROR_CODE_ENUM.
 *
 *****************************************************************************/
PLAYBACK_ERROR_CODE_ENUM RADIO_eSeek(
    DECODER_OBJECT hDecoder, RADIO_SEEK_TYPE eType, N32 n32Offset,
    BOOLEAN bPauseAfterSeek)
{
    PLAYBACK_ERROR_CODE_ENUM eReturnCode = PLAYBACK_ERROR_CODE_INVALID_OBJECT;
    const RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (const RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(
                        hDecoder);
    if(psDecoder != NULL)
    {
        SXIAPI_IR_CONTROL_ENUM eControl = SXIAPI_IR_CONTROL_UNKNOWN;
        SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;
        N16 n16Offset;

        // clip offset
        if (n32Offset > N16_MAX)
        {
            n16Offset = N16_MAX;
        }
        else if (n32Offset < N16_MIN)
        {
            n16Offset = N16_MIN;
        }
        else
        {
            n16Offset = (N16)n32Offset;
        }

        switch(eType)
        {
            case RADIO_SEEK_TYPE_TIME:
            {
                if (bPauseAfterSeek == TRUE)
                {
                    eControl = SXIAPI_IR_CONTROL_JUMP_TO_TIME_OFFSET_PAUSE;
                }
                else
                {
                    eControl = SXIAPI_IR_CONTROL_JUMP_TO_TIME_OFFSET_RESUME;
                }

                // Request seek by time
                eStatusCode = SXIAPI_eIRPlaybackControlCmd(
                    psDecoder->hControlCxn,
                    eControl,
                    (SXIAPI_PLAYBACK_TIME_OFFSET)n16Offset,
                    0
                        );

            }
            break;

            case RADIO_SEEK_TYPE_SONG:
            {
                SXIAPI_PLAYBACK_ID tPlaybackId = 0, tRange;
                N16 n16TimeOffset = 0;

                if (bPauseAfterSeek == TRUE)
                {
                    eControl = SXIAPI_IR_CONTROL_JUMP_TO_PLAYBACK_ID_PAUSE;
                }
                else
                {
                    eControl = SXIAPI_IR_CONTROL_JUMP_TO_PLAYBACK_ID_RESUME;
                }


                if (n16Offset < 0)
                {
                    // compute the range of playback id's
                    if (psDecoder->sPlayback.tOldestPlaybackId <
                        psDecoder->sPlayback.tCurrentPlaybackId)
                    {
                        tRange = psDecoder->sPlayback.tCurrentPlaybackId -
                                 psDecoder->sPlayback.tOldestPlaybackId;
                    }
                    else
                    {
                        tRange = psDecoder->sPlayback.tOldestPlaybackId -
                                 psDecoder->sPlayback.tCurrentPlaybackId;
                    }

                    // moving backward in time
                    if ( (n16Offset + tRange) > 0 )
                    {
                        // offset magnitude is less than or equal to the range
                        tPlaybackId =
                            psDecoder->sPlayback.tCurrentPlaybackId + n16Offset;
                    }
                    else
                    {
                        // offset magnitude is greater than range.
                        // we can only jump to the start of the buffer
                        if (bPauseAfterSeek == TRUE)
                        {
                            eControl = SXIAPI_IR_CONTROL_JUMP_TO_START_PAUSE;
                        }
                        else
                        {
                            eControl = SXIAPI_IR_CONTROL_JUMP_TO_START_RESUME;
                        }
                    }
                }
                else if (n16Offset > 0)
                {
                    // compute the range of playback id's
                    if (psDecoder->sPlayback.tNewestPlaybackId <
                        psDecoder->sPlayback.tCurrentPlaybackId)
                    {
                        tRange = psDecoder->sPlayback.tCurrentPlaybackId -
                            psDecoder->sPlayback.tNewestPlaybackId;
                    }
                    else
                    {
                        tRange = psDecoder->sPlayback.tNewestPlaybackId -
                                 psDecoder->sPlayback.tCurrentPlaybackId;
                    }

                    // moving forward in time
                    if ( (n16Offset - tRange) <= 0 )
                    {
                        // offset magnitude is less than or equal to the range
                        tPlaybackId =
                            psDecoder->sPlayback.tCurrentPlaybackId + n16Offset;
                    }
                    else
                    {
                        // offset magnitude is greater than range.  we can
                        // only jump to the end of the buffer (i.e. live)

                        if (bPauseAfterSeek == TRUE)
                        {
                            // there is no "jump to live and pause" control
                            // but by using a gigantic time offset we can can use
                            // SXIAPI_IR_CONTROL_JUMP_TO_TIME_OFFSET_PAUSE and
                            // accomplish the same end result
                            eControl = SXIAPI_IR_CONTROL_JUMP_TO_TIME_OFFSET_PAUSE;
                            n16TimeOffset = N16_MAX;
                            tPlaybackId = 0;
                        }
                        else
                        {
                           eControl = SXIAPI_IR_CONTROL_LIVE;
                        }
                    }
                }
                else
                {
                    tPlaybackId = psDecoder->sPlayback.tCurrentPlaybackId;
                }

                // Request seek operation
                eStatusCode = SXIAPI_eIRPlaybackControlCmd(
                    psDecoder->hControlCxn,
                    eControl,
                    n16TimeOffset,
                    (SXIAPI_PLAYBACK_ID)tPlaybackId
                       );
            }
            break;

            case RADIO_SEEK_TYPE_PREVIOUS:
            {
                if (bPauseAfterSeek == TRUE)
                {
                    eControl = SXIAPI_IR_CONTROL_PREV;
                }
                else
                {
                    eControl = SXIAPI_IR_CONTROL_PREVIOUS_RESUME;
                }

                // Request seek by time
                eStatusCode = SXIAPI_eIRPlaybackControlCmd(
                    psDecoder->hControlCxn,
                    eControl,
                    0,
                    0
                        );

            }
            break;

            case RADIO_SEEK_TYPE_NEXT:
            {
                if (bPauseAfterSeek == TRUE)
                {
                    eControl = SXIAPI_IR_CONTROL_NEXT;
                }
                else
                {
                    eControl = SXIAPI_IR_CONTROL_NEXT_RESUME;
                }

                // Request seek by time
                eStatusCode = SXIAPI_eIRPlaybackControlCmd(
                    psDecoder->hControlCxn,
                     eControl,
                     0,
                     0
                        );

            }
            break;

            default:
            break;
        }

        if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            eReturnCode = PLAYBACK_ERROR_CODE_NONE;
        }
        else if(eStatusCode == SXIAPI_STATUSCODE_MSG_FAIL)
        {
            eReturnCode = PLAYBACK_ERROR_CODE_PLAYBACK_OP_FAILED;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   RADIO_eSetPlaybackParams
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the params from.
 *       un32WarningOffset - A warning offset.
 *
 *   Returns:
 *       Specific PLAYBACK_ERROR_CODE_ENUM value
 *
 *****************************************************************************/
PLAYBACK_ERROR_CODE_ENUM RADIO_eSetPlaybackParams(
    DECODER_OBJECT hDecoder, UN32 un32WarningOffset)
{
    PLAYBACK_ERROR_CODE_ENUM eReturnCode = PLAYBACK_ERROR_CODE_INVALID_OBJECT;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(
                        hDecoder);
    if(psDecoder != NULL)
    {
        PLAYBACK_OBJECT hPlayback;

        if (un32WarningOffset >= psDecoder->sPlayback.un16DurationOfBuffer)
        {
            eReturnCode = PLAYBACK_ERROR_CODE_OUT_OF_RANGE;
        }
        else
        {
            // Extract the playback handle
            hPlayback = DECODER.hPlayback(psDecoder->hDecoder);
            psDecoder->sPlayback.un16WarningLimit = (UN16)un32WarningOffset;
            PLAYBACK_vUpdatePlaybackParams(hPlayback, un32WarningOffset);
            eReturnCode = PLAYBACK_ERROR_CODE_NONE;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   RADIO_hCreateCategory
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *****************************************************************************/
CATEGORY_OBJECT RADIO_hCreateCategory(
    DECODER_OBJECT hDecoder, CATEGORY_ID tCategoryId)
{
    CATEGORY_OBJECT hCategory = CATEGORY_INVALID_OBJECT;
    const RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (const RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);

    if(psDecoder != NULL)
    {
        SXIAPI_CATINFOIND_STRUCT *psCatInfo;
        SXIAPI_CATINFOIND_STRUCT sCatInfoToFind;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "RADIO_hCreateCategory called for %u...\n", tCategoryId);

        // Look up category in linked list
        sCatInfoToFind.tCatID = (SXIAPI_CATEGORY_ID)tCategoryId;
        eReturnCode = OSAL.eLinkedListSearch(
                psDecoder->sCategory.hCatList,
                &hEntry, &sCatInfoToFind);

        // Is it there, then create the category
        if (eReturnCode == OSAL_SUCCESS)
        {
            const char *pcCatNameShort = NULL;
            const char *pcCatNameLong = NULL;
            const char *pcCatNameMed = NULL;

            psCatInfo =
                (SXIAPI_CATINFOIND_STRUCT*) OSAL.pvLinkedListThis(hEntry);

            // Create a new category object and
            // populate it with our cached info

            if (psCatInfo->sCatNames.acLong[0] != '\0')
            {
                pcCatNameLong = &psCatInfo->sCatNames.acLong[0];
            }

            if (psCatInfo->sCatNames.acMed[0] != '\0')
            {
                pcCatNameMed = &psCatInfo->sCatNames.acMed[0];
            }

            if (psCatInfo->sCatNames.acShort[0] != '\0')
            {
                pcCatNameShort = &psCatInfo->sCatNames.acShort[0];
            }

            hCategory = CATEGORY_hCreateCategory(hDecoder,
                (SMS_OBJECT)psDecoder->hCCache,
                CATEGORY_TYPE_BROADCAST, psCatInfo->tCatID,
                pcCatNameLong, pcCatNameMed, pcCatNameShort, 0, FALSE, FALSE);

            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "FOUND\n");
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Category not found.");
        }
    }

    return hCategory;
}

/******************************************************************************
 *
 *   RADIO_bGetESN
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE MODULE OBJECT.
 *
 *****************************************************************************/
BOOLEAN RADIO_bGetESN(
    MODULE_OBJECT hModule, STRING_OBJECT *phESN)
{
    BOOLEAN bSuccess = FALSE;
    RADIO_MODULE_OBJECT_STRUCT *psModule;

    // Verify input
    if(phESN == NULL)
    {
        // Input error!
        return FALSE;
    }

    // Obtain radio data
    psModule = (RADIO_MODULE_OBJECT_STRUCT *)
        MODULE_hGetRadioSpecificData(
            hModule);
    if(psModule != NULL)
    {
        // Return what we have.
        // We may not have anything though since we
        // have to wait for the SubStatusInd message to be sent
        *phESN = psModule->hESN;

        // All is well
        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_eSubStatus
 *
 *****************************************************************************/
MODULE_SUBSTATUS_ENUM RADIO_eSubStatus(
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder
        )
{
    MODULE_SUBSTATUS_ENUM eStatus = MODULE_SUBSTATUS_INVALID;

    if (hModule != MODULE_INVALID_OBJECT)
    {
        RADIO_MODULE_OBJECT_STRUCT *psModule;

        // Obtain radio data
        psModule = (RADIO_MODULE_OBJECT_STRUCT *)
            MODULE_hGetRadioSpecificData(
                hModule);
        if(psModule != NULL)
        {
            // Return what we have.
            // We may not have anything though since we
            // have to wait for the SubStatusInd message to be sent
            eStatus = eSXISubStatusToSMSSubStatus(psModule->sSubstatus.eStatus);
        }
    }
    else if (hDecoder != DECODER_INVALID_OBJECT)
    {
        RADIO_DECODER_OBJECT_STRUCT *psDecoder;

        // Obtain radio data
        psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
            DECODER_hGetRadioSpecificData(
                hDecoder);
        if(psDecoder != NULL)
        {
            // Return what we have.
            eStatus = psDecoder->sSubStatus.eStatus;
        }
    }

    return eStatus;
}

/*****************************************************************************
 *
 *   RADIO_tReasonCode
 *
 *****************************************************************************/
MODULE_SUBSTATUS_REASON_CODE RADIO_tReasonCode(
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder
        )
{
    MODULE_SUBSTATUS_REASON_CODE tReasonCode =
        MODULE_SUBSTATUS_REASON_CODE_INVALID;

    if (hModule != MODULE_INVALID_OBJECT)
    {
        RADIO_MODULE_OBJECT_STRUCT *psModule;

        // Obtain radio data
        psModule = (RADIO_MODULE_OBJECT_STRUCT *)
            MODULE_hGetRadioSpecificData(
                hModule);
        if(psModule != NULL)
        {
            // Return what we have.
            // We may not have anything though since we
            // have to wait for the SubStatusInd message to be sent
            tReasonCode = psModule->sSubstatus.tReasonCode;
        }
    }
    else if (hDecoder != DECODER_INVALID_OBJECT)
    {
        RADIO_DECODER_OBJECT_STRUCT *psDecoder;

        // Obtain radio data
        psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
            DECODER_hGetRadioSpecificData(
                hDecoder);
        if(psDecoder != NULL)
        {
            // Return what we have.
            tReasonCode = psDecoder->sSubStatus.tReasonCode;
        }
    }

    return tReasonCode;
}

/*****************************************************************************
 *
 *   RADIO_un32SuspendDate
 *
 *****************************************************************************/
UN32 RADIO_un32SuspendDate(
    MODULE_OBJECT hModule,
    DECODER_OBJECT hDecoder
        )
{
    UN32 un32SuspendDate = MODULE_SUSPED_DATE_INVALID;

    if (hModule != MODULE_INVALID_OBJECT)
    {
        RADIO_MODULE_OBJECT_STRUCT *psModule;

        // Obtain radio data
        psModule = (RADIO_MODULE_OBJECT_STRUCT *)
            MODULE_hGetRadioSpecificData(
                hModule);
        if(psModule != NULL)
        {
            // Return what we have.
            // We may not have anything though since we
            // have to wait for the SubStatusInd message to be sent
            un32SuspendDate = psModule->sSubstatus.un32UTCTime;
        }
    }
    else if (hDecoder != DECODER_INVALID_OBJECT)
    {
        RADIO_DECODER_OBJECT_STRUCT *psDecoder;

        // Obtain radio data
        psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
            DECODER_hGetRadioSpecificData(
                hDecoder);
        if(psDecoder != NULL)
        {
            struct tm sUTCTime;
            TIME_T tSecs;

            // get what we have
            tSecs = (TIME_T)psDecoder->sSubStatus.un32SubSuspendDate;

            // now we need to convert to time structure
            OSAL.gmtime_r(&tSecs, &sUTCTime);

            if (sUTCTime.tm_year > (RADIO_EPOCH+RADIO_SXI_SUB_NOTIFICATION_OFFSET))
            {
                // then subtract the offset.
                sUTCTime.tm_year = sUTCTime.tm_year - RADIO_SXI_SUB_NOTIFICATION_OFFSET;

                // then convert back to seconds
                un32SuspendDate = (UN32)OSAL.mktime(&sUTCTime);
            }
            else
            {
                // Either we haven't rxd a suspend date or rxd a weird one.
                // Anyway, no sub_notification for you!
                un32SuspendDate = 0;
            }
        }
    }

    return un32SuspendDate;
}



/*****************************************************************************
 *
 *   RADIO_hReasonText
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE MODULE OBJECT.
 *
 *****************************************************************************/
STRING_OBJECT RADIO_hReasonText(
    MODULE_OBJECT hModule)
{
    RADIO_MODULE_OBJECT_STRUCT *psModule;
    STRING_OBJECT hReasonText = STRING_INVALID_OBJECT;

    // Obtain radio data
    psModule = (RADIO_MODULE_OBJECT_STRUCT *)
        MODULE_hGetRadioSpecificData(
            hModule);
    if(psModule != NULL)
    {
        // Return what we have.
        // We may not have anything though since we
        // have to wait for the SubStatusInd message to be sent
        hReasonText = psModule->sSubstatus.hReasonText;
    }

    return hReasonText;
}

/*****************************************************************************
 *
 *   RADIO_hPhoneNumber
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE MODULE OBJECT.
 *
 *****************************************************************************/
STRING_OBJECT RADIO_hPhoneNumber(
    MODULE_OBJECT hModule)
{
    RADIO_MODULE_OBJECT_STRUCT *psModule;
    STRING_OBJECT hPhoneNumber = STRING_INVALID_OBJECT;

    // Obtain radio data
    psModule = (RADIO_MODULE_OBJECT_STRUCT *)
        MODULE_hGetRadioSpecificData(
            hModule);
    if(psModule != NULL)
    {
        // Return what we have.
        // We may not have anything though since we
        // have to wait for the SubStatusInd message to be sent
        hPhoneNumber = psModule->sSubstatus.hPhoneNumber;
    }

    return hPhoneNumber;
}

/*****************************************************************************
*
*   RADIO_bIsAudioSubscribed
*
*   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE MODULE OBJECT.
*
*****************************************************************************/
BOOLEAN RADIO_bIsAudioSubscribed(
    MODULE_OBJECT hModule)
{
    RADIO_MODULE_OBJECT_STRUCT *psModule;
    BOOLEAN bIsAudioSubscribed = FALSE;

    // Obtain radio data
    psModule = (RADIO_MODULE_OBJECT_STRUCT *)
        MODULE_hGetRadioSpecificData(
            hModule);
    if (psModule != NULL)
    {
        // Return what we have.
        // We may not have anything though since we
        // have to wait for the SubStatusInd message to be sent
        bIsAudioSubscribed = psModule->sSubstatus.bIsAudioSubscribed;
    }

    return bIsAudioSubscribed;
}

/*****************************************************************************
 *
 *   RADIO_bDaylightSavingsTimeSupported
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE MODULE OBJECT.
 *
 *****************************************************************************/
BOOLEAN RADIO_bDaylightSavingsTimeSupported(
    MODULE_OBJECT hModule)
{
    BOOLEAN bReturn;
    UN32 un32SwVer = 0;

    //Get the version information of the running module to
    //determine if we can support DST
    bReturn = MODULE_VERSION_bGetSwVersion(hModule, &un32SwVer);
    if ((bReturn == TRUE) && (un32SwVer >= SXI_DST_CAPABLE_VERSION))
    {
        bReturn = TRUE;
    }
    else
    {
        bReturn = FALSE;
    }
    return bReturn;
}

/*****************************************************************************
 *
 *   RADIO_bSetTimeParameters
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE MODULE OBJECT.
 *
 *****************************************************************************/
BOOLEAN RADIO_bSetTimeParameters(
    MODULE_OBJECT hModule, N16 n16GmtOffset,
    BOOLEAN bDaylightSavingsTimeObserved)
{
    BOOLEAN bSuccess = FALSE;
    const RADIO_MODULE_OBJECT_STRUCT *psModule;

    // Obtain radio data
    psModule = (const RADIO_MODULE_OBJECT_STRUCT *)
        MODULE_hGetRadioSpecificData(
                        hModule);
    if(psModule != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;
        SXIAPI_TIMEZONE_ENUM eTimeZone;

        // Make sure we know this offset
        switch (n16GmtOffset)
        {
            case SXIAPI_TIMEZONE_GMT_OFFSET_MIN_NONE:
                eTimeZone = SXIAPI_TIMEZONE_UTC;
                break;
            case SXIAPI_TIMEZONE_GMT_OFFSET_MIN_NT:
                eTimeZone = SXIAPI_TIMEZONE_NT;
                break;
            case SXIAPI_TIMEZONE_GMT_OFFSET_MIN_AT:
                eTimeZone = SXIAPI_TIMEZONE_AT;
                break;
            case SXIAPI_TIMEZONE_GMT_OFFSET_MIN_ET:
                eTimeZone = SXIAPI_TIMEZONE_ET;
                break;
            case SXIAPI_TIMEZONE_GMT_OFFSET_MIN_CT:
                eTimeZone = SXIAPI_TIMEZONE_CT;
                break;
            case SXIAPI_TIMEZONE_GMT_OFFSET_MIN_MT:
                eTimeZone = SXIAPI_TIMEZONE_MT;
                break;
            case SXIAPI_TIMEZONE_GMT_OFFSET_MIN_PT:
                eTimeZone = SXIAPI_TIMEZONE_PT;
                break;
            case SXIAPI_TIMEZONE_GMT_OFFSET_MIN_AKT:
                eTimeZone = SXIAPI_TIMEZONE_AKT;
                break;
            case SXIAPI_TIMEZONE_GMT_OFFSET_MIN_HAT:
                eTimeZone = SXIAPI_TIMEZONE_HAT;
                break;
            default:
                eTimeZone = SXIAPI_TIMEZONE_INVALID;
                break;
        }

        if (eTimeZone != SXIAPI_TIMEZONE_INVALID)
        {
            if (bDaylightSavingsTimeObserved == TRUE)
            {
                BOOLEAN bReturn;

                bReturn = RADIO_bDaylightSavingsTimeSupported(
                                psModule->hModule);
                if (bReturn == FALSE)
                {
                    // We can't enable daylight savings time if it isn't supported
                    // by sms and the module

                    // Error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            RADIO_OBJECT_NAME": Unable to enable daylight"
                            " savings time observation on module.");

                    return FALSE;
                }
            }
            eStatusCode =
                SXIAPI_eTimeCfgCmd(
                    psModule->sSXI.hControlCxn,
                    eTimeZone,
                            // If the firmware version is less than SXI_DST_CAPABLE_VERSION
                            // we always disable DST. Otherwise we can do what is requested
                            // of us.
                    bDaylightSavingsTimeObserved
                        );

            if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
            {
                bSuccess = TRUE;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_eFirmwareUpdate
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eFirmwareUpdate(
    MODULE_OBJECT hModule,
    const char *pcFirmwareFileName
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    RADIO_FWUPDATE_STRUCT sFirmwareUpdate =
                        {
                            RADIO_FWUPDATE_EVENT_ERROR,
                            RADIO_FWUPDATE_ERROR_GENERAL,
                            NULL,
                            NULL,
                            0
                        };

    eReturnCode = eLoadFWUpdate( hModule,
                                 pcFirmwareFileName,
                                 &sFirmwareUpdate );

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        BOOLEAN bPosted;

        sFirmwareUpdate.eFWUpdateEvent =
                   RADIO_FWUPDATE_EVENT_READ_COMPLETE;
        sFirmwareUpdate.eFWUpdateError =
                   RADIO_FWUPDATE_ERROR_GENERAL;

        bPosted = bPostModuleEvent(
            hModule, (SMS_EVENT_TYPE_ENUM)RADIO_EVENT_FWUPDATE,
            SMS_EVENT_OPTION_NONE,
            &sFirmwareUpdate
                );
        if (bPosted != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
    }
    return eReturnCode;
}

/*****************************************************************************
 *
 *   RADIO_vAbortFirmwareUpdate
 *
 *****************************************************************************/
void RADIO_vAbortFirmwareUpdate(
    MODULE_OBJECT hModule
        )
{
    RADIO_FWUPDATE_STRUCT sFirmwareUpdate;
    BOOLEAN bPosted;

    sFirmwareUpdate.eFWUpdateEvent = RADIO_FWUPDATE_EVENT_ABORT;
    sFirmwareUpdate.eFWUpdateError = RADIO_FWUPDATE_ERROR_GENERAL;

    bPosted = bPostModuleEvent(
        hModule, (SMS_EVENT_TYPE_ENUM)RADIO_EVENT_FWUPDATE,
        SMS_EVENT_OPTION_NONE,
        &sFirmwareUpdate
            );
    if (bPosted != TRUE)
    {
        //not much we can do here
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME
            ": Unable to post abort firmware update event");
    }

    return;
}

/*****************************************************************************
 *
 *   RADIO_n8GetNumAntennas
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *****************************************************************************/
N8 RADIO_n8GetNumAntennas(
    DECODER_OBJECT hDecoder)
{
    // NOTE: We can look at the SRH, but since none of the SXI commands
    // support multiple antennas, why bother?
    return 1;
}

/*****************************************************************************
 *
 *   RADIO_eGetAntennaState
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *****************************************************************************/
ANTENNA_STATE_ENUM RADIO_eGetAntennaState(
    DECODER_OBJECT hDecoder, N8 n8Antenna)
{
    ANTENNA_STATE_ENUM eAntennaState = ANTENNA_STATE_UNKNOWN;
    const RADIO_DECODER_OBJECT_STRUCT *psDecoder;
    N8 n8NumAntennas;

    n8NumAntennas = RADIO_n8GetNumAntennas(hDecoder);

    // make sure they aren't requesting out of bounds
    if (n8Antenna < n8NumAntennas)
    {
        // Obtain radio data
        psDecoder = (const RADIO_DECODER_OBJECT_STRUCT *)
            DECODER_hGetRadioSpecificData(
                            hDecoder);
        if(psDecoder != NULL)
        {
            eAntennaState = psDecoder->eAntennaState;
        }
    }

    return eAntennaState;
}

/*****************************************************************************
*
*   RADIO_bFeaturedFavoritesEnable
*
*   This function is used to enable or disable featured favorites processing
*
*****************************************************************************/
BOOLEAN RADIO_bFeaturedFavoritesEnable (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        )
{
    BOOLEAN bReturn = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
            DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        UN64 un64Mask;

        if(bEnable == TRUE)
        {
            // We just need the table version and count to start with
            un64Mask =
                (SXIAPI_GMI_MASK_FF_TABLE_VER |
                 SXIAPI_GMI_MASK_FF_CNT
                    );
        }
        else
        {
            // Disable everything
            un64Mask =
                (SXIAPI_GMI_MASK_FF_TABLE_VER |
                 SXIAPI_GMI_MASK_FF_CNT |
                 SXIAPI_GMI_MASK_FF_BANK_ORDER |
                 SXIAPI_GMI_MASK_FF_BANK_ID |
                 SXIAPI_GMI_MASK_FF_BANK_SEQ |
                 SXIAPI_GMI_MASK_FF_BANK_TTL_SHORT |
                 SXIAPI_GMI_MASK_FF_BANK_TTL_LONG |
                 SXIAPI_GMI_MASK_FF_BANK_TTL_VERBOSE |
                 SXIAPI_GMI_MASK_FF_BANK_DESCR |
                 SXIAPI_GMI_MASK_FF_BANK_PURPOSE |
                 SXIAPI_GMI_MASK_FF_BANK_ARR_1 |
                 SXIAPI_GMI_MASK_FF_BANK_ARR_2 |
                 SXIAPI_GMI_MASK_FF_BANK_ARR_3 |
                 SXIAPI_GMI_MASK_FF_BANK_ARR_4
                    );
        }

        bReturn = SXI_GMD_bConfigure(
            psDecoder->hControlCxn, un64Mask, bEnable);
        if(bReturn == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot set Featured Favorites "
                "GMD monitor");
        }
    }

    return bReturn;
}

#if SMS_LOGGING == 1

/*****************************************************************************
*
*   RADIO_pacEventTypeText
*
* This is a radio-specific function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
const char *RADIO_pacEventTypeText(
    SMS_EVENT_TYPE_ENUM eType
        )
{
    const char *pacReturnString = "UNKNOWN";

    switch ((RADIO_EVENT_TYPE_ENUM)eType)
    {
        case RADIO_EVENT_SXI_DATA_PAYLOAD:
            pacReturnString = MACRO_TO_STRING(RADIO_EVENT_SXI_DATA_PAYLOAD);
        break;

        case RADIO_EVENT_SXI_RX_DATA:
            pacReturnString = MACRO_TO_STRING(RADIO_EVENT_SXI_RX_DATA);
        break;

        case RADIO_EVENT_SXI_IND:
            pacReturnString = MACRO_TO_STRING(RADIO_EVENT_SXI_IND);
        break;

        case RADIO_EVENT_SXI_FSM:
            pacReturnString = MACRO_TO_STRING(RADIO_EVENT_SXI_FSM);
        break;

        case RADIO_EVENT_SHUTDOWN:
            pacReturnString = MACRO_TO_STRING(RADIO_EVENT_SHUTDOWN);
        break;

        case RADIO_EVENT_FWUPDATE:
            pacReturnString = MACRO_TO_STRING(RADIO_EVENT_FWUPDATE);
        break;
    }

    return pacReturnString;
}

#endif /* SMS_LOGGING == 1 */

/*****************************************************************************
*
*   RADIO_eMapLeague
*
*****************************************************************************/
LEAGUE_ENUM RADIO_eMapLeague(UN32 un32LeagueId)
{
    LEAGUE_ENUM eLeague;

    // map the league id to the league enum
    switch(un32LeagueId)
    {
        case 0:
            eLeague = LEAGUE_NFL;
        break;
        case 1:
            eLeague = LEAGUE_MLB;
        break;
        case 2:
            eLeague = LEAGUE_NBA;
        break;
        case 3:
            eLeague = LEAGUE_NHL;
        break;
        case 4:
            eLeague = LEAGUE_COLLEGE_FOOTBALL;
        break;
        case 5:
            eLeague = LEAGUE_COLLEGE_BASKETBALL;
        break;
        case 6:
            eLeague = LEAGUE_WOMENS_COLLEGE_BASKETBALL;
        break;
        default:
            eLeague = LEAGUE_UNKNOWN;
        break;
    }

    return eLeague;
}

/*****************************************************************************
*
*   RADIO_eMapSport
*
*****************************************************************************/
SPORTS_ENUM RADIO_eMapSport(char const *pacSportType)
{
    SPORTS_ENUM eSport = SPORTS_UNKNOWN;
    UN8 un8Index;

    // map text to sport type
    if (pacSportType != NULL)
    {
        for (un8Index = 0; un8Index < NUM_SPORTS; un8Index++)
        {
            if (strcmp(gasSport[un8Index].pacName, pacSportType) == 0)
            {
                eSport = gasSport[un8Index].eSubType;
                break;
            }
        }
    }

    return eSport;
}

/*****************************************************************************
*
*   RADIO_bConfirmDiscoveredMarkets
*
*****************************************************************************/
BOOLEAN RADIO_bConfirmDiscoveredMarkets(void)
{
    // SXi broadcasts global metadata. Don't confirm based on markets broadcast
    return FALSE;
}

/*****************************************************************************
 *
 *   RADIO_ePackage
 *   (when called, hModule is locked, inputs have been checked)
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_ePackage (
    MODULE_OBJECT hModule,
    MODULE_PACKGE_CMD_ENUM eCmd,
    UN8 un8PackageIndex
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    SXIAPI_PKG_CMD_OPTION_ENUM eMPFACmd = SXIAPI_PKG_CMD_OPTION_UNKNOWN;

    // Translate SMS package cmd into SXi pkg cmd
    switch (eCmd)
    {
        case MODULE_PACKGE_CMD_SELECT:
            eMPFACmd = SXIAPI_PKG_CMD_OPTION_SELECT;
        break;
        case MODULE_PACKGE_CMD_QUERY:
            eMPFACmd = SXIAPI_PKG_CMD_OPTION_QUERY;
        break;
        case MODULE_PACKGE_CMD_VALIDATE:
            eMPFACmd = SXIAPI_PKG_CMD_OPTION_VALIDATE;
        break;
        case MODULE_PACKGE_CMD_REPORT:
            eMPFACmd = SXIAPI_PKG_CMD_OPTION_REPORT;
        break;
        case MODULE_PACKGE_CMD_INVALID:
        default:
            eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
        break;
    }

    // Check if this is a valid cmd
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        const RADIO_MODULE_OBJECT_STRUCT *psModule;

        // Obtain radio data
        psModule = (const RADIO_MODULE_OBJECT_STRUCT *)
            MODULE_hGetRadioSpecificData(hModule);
        if(psModule != NULL)
        {
            SXIAPI_STATUSCODE_ENUM eStatusCode;

            // Issue PkgCmd
            eStatusCode = SXIAPI_ePkgCmd(psModule->sSXI.hControlCxn,
                              eMPFACmd, un8PackageIndex);
            if(eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
            {
                // couldn't send the message
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": SXIAPI_ePkgCmd(%u) returned %u.",
                    eMPFACmd, eStatusCode);

                eReturn = SMSAPI_RETURN_CODE_ERROR;
            }
        }
        else
        {
           // couldn't get radio data
           eReturn = SMSAPI_RETURN_CODE_ERROR;
        }
    }

    return eReturn;
}

/*****************************************************************************
 *
 *   RADIO_bDefaultDst
 *
 *****************************************************************************/
BOOLEAN RADIO_bDefaultDst(void)
{
    return SXIAPI_RADIO_DEFAULT_DST;
}

/*****************************************************************************
 *
 *   RADIO_eToneGenerationStart
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eToneGenerationStart(
    DECODER_OBJECT hDecoder,
    UN32 un32ToneFreqHz,
    N8 n8Volume,
    DECODER_TONE_GENERATION_BALANCE_ENUM eBalance,
    BOOLEAN bAlert
        )
{
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    UN8 un8ToneFreqhHz;
    UN8 un8AudioToneOptions = SXIAPI_TONE_GEN_OPTIONS_INVALID;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)DECODER_hGetRadioSpecificData(
        hDecoder);

    if (psDecoder != NULL)
    {
        if (un32ToneFreqHz > SXIAPI_TONE_GEN_MAX_FREQ_HZ)
        {
            // clip to max allowable freq
            // convert to hecto Hertz (hHz)
            un8ToneFreqhHz = SXIAPI_TONE_GEN_MAX_FREQ_HZ / 100;
        }
        else
        {
            // convert to hecto Hertz (hHz), add 50 for rounding
            // to nearest hHz.
            un8ToneFreqhHz = (un32ToneFreqHz + 50 ) / 100;
        }

        if (n8Volume > SXIAPI_TONE_GEN_MAX_VOL_ADJ)
        {
            // clip to max allowable volume adjust and scale (dB)
            n8Volume = (SXIAPI_TONE_GEN_MAX_VOL_ADJ *
                        SXIAPI_TONE_GEN_SCALE_NUMERATOR ) /
                        SXIAPI_TONE_GEN_SCALE_DENOMINATORATOR;
        }
        else if (n8Volume > SXIAPI_TONE_GEN_MIN_VOL_ADJ)
        {
            // scale vol adj (dB)
            n8Volume = (n8Volume * SXIAPI_TONE_GEN_SCALE_NUMERATOR ) /
                SXIAPI_TONE_GEN_SCALE_DENOMINATORATOR;
        }
        else
        {
            // clip min allowable volume adjust and scale (dB)
            n8Volume = (SXIAPI_TONE_GEN_MIN_VOL_ADJ *
                        SXIAPI_TONE_GEN_SCALE_NUMERATOR ) /
                        SXIAPI_TONE_GEN_SCALE_DENOMINATORATOR;
        }

        switch(eBalance)
        {
            case DECODER_TONE_GENERATION_BALANCE_LEFT:
                un8AudioToneOptions = SXIAPI_TONE_GEN_CHANNEL_LEFT;
            break;

            case DECODER_TONE_GENERATION_BALANCE_RIGHT:
                un8AudioToneOptions = SXIAPI_TONE_GEN_CHANNEL_RIGHT;
            break;

            case DECODER_TONE_GENERATION_BALANCE_BOTH:
                un8AudioToneOptions = SXIAPI_TONE_GEN_CHANNEL_BOTH;
            break;

            default:
                un8AudioToneOptions = SXIAPI_TONE_GEN_OPTIONS_INVALID;
                eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        if (un8AudioToneOptions < SXIAPI_TONE_GEN_OPTIONS_INVALID)
        {
            if (bAlert == TRUE)
            {
                un8AudioToneOptions |= SXIAPI_TONE_GEN_ALERT;
            }

            // Generate Tone
            eStatusCode =
                SXIAPI_eToneGenerateCmd(
                    psDecoder->hControlCxn,
                    un8ToneFreqhHz,
                    un8AudioToneOptions,
                    n8Volume
                        );

            if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
            {
                // Error
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Failed to set Tone Gen (Start) Command.");
            }
            else
            {
                eReturn = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }

    return eReturn;

}

/*****************************************************************************
 *
 *   RADIO_eToneGenerationStop
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eToneGenerationStop(
    DECODER_OBJECT hDecoder
        )
{
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)DECODER_hGetRadioSpecificData(
        hDecoder);

    if (psDecoder != NULL)
    {
        // Generate Tone
        eStatusCode =
            SXIAPI_eToneGenerateCmd(
                psDecoder->hControlCxn,
                0,
                SXIAPI_TONE_GEN_CHANNEL_NONE,
                0
                    );

        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Failed to set Tone Gen (Stop) Command.");
        }
        else
        {
            eReturn = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    return eReturn;
}

/*****************************************************************************
 *
 *   RADIO_eAdjustVolume
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eAdjustVolume(
    DECODER_OBJECT hDecoder,
    N16 n16Level
        )
{
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)DECODER_hGetRadioSpecificData(
        hDecoder);

    if (psDecoder != NULL)
    {
        SXIAPI_MUTE_ENUM eAction = SXIAPI_MUTE_UNMUTE;
        N8 n8Level = 0;

        if (n16Level == SMSAPI_AUDIO_LEVEL_MUTE)
        {
            // mute audio level
            eAction = SXIAPI_MUTE_AUDIO;
        }
        else
        {
            // clip the requested level to valid values...
            if (n16Level < SXIAPI_VOL_MIN)
            {
                n8Level = SXIAPI_VOL_MIN;

            }
            else if (n16Level > SXIAPI_VOL_MAX)
            {
                n8Level = SXIAPI_VOL_MAX;
            }
            else
            {
                n8Level = (N8)n16Level;
            }

            // adjust the volume
            eStatusCode =
                SXIAPI_eAudioVolCmd (
                    psDecoder->hControlCxn,
                    n8Level
                        );
            if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
            {
                // Error
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Failed to Adjust Audio Level.");
                eReturn = SMSAPI_RETURN_CODE_ERROR;
            }
        }

        // Adjust audio mute (if we adjusted volume, we will be unmuting)
        eStatusCode =
            SXIAPI_eAudioMuteCmd(
                psDecoder->hControlCxn,
                eAction
                    );

        if (eStatusCode != SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Failed to Adjust Mute.");
            eReturn = SMSAPI_RETURN_CODE_ERROR;
        }
    }

    return eReturn;
}

/*****************************************************************************
 *
 *   RADIO_eModifyEngineeringData
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE MODULE OBJECT.
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eModifyEngineeringData(
    MODULE_OBJECT hModule,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eEnable
        )

{
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SXIAPI_MONITOR_OP_ENUM eOperation;

    if ( eEnable == SMSAPI_MODIFY_EVENT_MASK_ENABLE )
    {
        eOperation = SXIAPI_MONITOR_OP_ENABLE;
    }
    else if ( eEnable == SMSAPI_MODIFY_EVENT_MASK_DISABLE )
    {
        eOperation = SXIAPI_MONITOR_OP_DISABLE;
    }
    else
    {
        return SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
    }

    do
    {
        RADIO_MODULE_OBJECT_STRUCT *psModule;

        const SXIAPI_STATUS_ITEM_ENUM aeStatusItems[] =
        {
            SXIAPI_STATUS_ITEM_AUDIO_DECODER_BITRATE,
            SXIAPI_STATUS_ITEM_SIGNAL_QUALITY,
            SXIAPI_STATUS_ITEM_OVERLAY_SIGNAL_QUALITY,
            SXIAPI_STATUS_ITEM_LINK_INFORMATION
        };
        const UN8 aeStatusItemsCount =
             sizeof(aeStatusItems) / sizeof(SXIAPI_STATUS_ITEM_ENUM);

        // Obtain radio data
        psModule = (RADIO_MODULE_OBJECT_STRUCT *)
                     MODULE_hGetRadioSpecificData(hModule);

        if (psModule == NULL)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        // Set Status Monitor Items
        eStatusCode =   SXIAPI_eStatusMonCmd(
                                         psModule->sSXI.hControlCxn,
                                         eOperation,
                                         aeStatusItemsCount,
                                         &aeStatusItems[0]
                                            );
        if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    } while(FALSE);
    return eReturnCode;
}


/*****************************************************************************
 *
 *   RADIO_eGetFirmwareFileVersion
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eGetFirmwareFileVersion(
    FILE *psFile,
    UN32 *pun32FileVersion,
    UN32 *pun32FileEarliestVersion,
    UN32 *pun32FileLatestVersion,
    UN32 *pun32FileTypeID
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    UN8 *pun8Memory;

    do
    {
        UN32 un32Result, un32ReadTotal=0;
        int iResult;
        pun8Memory = (UN8 *)OSAL.pvMemoryAllocate(
                                              "FWUpdateFileVersionBuffer",
                                              SXI_FWUPDATE_PACKET_SIZE,
                                              TRUE);
        if( pun8Memory == NULL )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                RADIO_OBJECT_NAME": Unable to allocate buffer");
            return SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
        }

        // copy the first chunk into the buffer:
        do
        {
            un32Result = (UN32)fread(pun8Memory,
                             1, SXI_FWUPDATE_HEADER_SIZE, psFile);
            un32ReadTotal += un32Result;
        }
        while (  (un32ReadTotal < SXI_FWUPDATE_HEADER_SIZE)
              && (un32Result > 0)
              );

        if (un32ReadTotal != SXI_FWUPDATE_HEADER_SIZE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                RADIO_OBJECT_NAME": Unable to read file");
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        //reset the file pointer back to begining
        iResult = fseek(psFile, 0 , SEEK_SET);
        if (iResult != 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                RADIO_OBJECT_NAME": Unable to reset file pointer");
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        //Extract the version information from the firmware update file
        *pun32FileVersion = (pun8Memory[SXI_FWUPDATE_SW_REV_MAJOR_POS] << 16) |
                            (pun8Memory[SXI_FWUPDATE_SW_REV_MINOR_POS] << 8)  |
                            (pun8Memory[SXI_FWUPDATE_SW_REV_INC_POS]);

        *pun32FileTypeID = (pun8Memory[SXI_FWUPDATE_MOD_TYPE_A_POS] << 16) |
                           (pun8Memory[SXI_FWUPDATE_MOD_TYPE_B_POS] << 8)  |
                           (pun8Memory[SXI_FWUPDATE_MOD_TYPE_C_POS]);

        *pun32FileEarliestVersion = (pun8Memory[SXI_FWUPDATE_EARLIEST_SW_REV_MAJOR_POS] << 16) |
                                    (pun8Memory[SXI_FWUPDATE_EARLIEST_SW_REV_MINOR_POS] << 8) |
                                    (pun8Memory[SXI_FWUPDATE_EARLIEST_SW_REV_INC_POS]);

        *pun32FileLatestVersion = (pun8Memory[SXI_FWUPDATE_LATEST_SW_REV_MAJOR_POS] << 16) |
                                  (pun8Memory[SXI_FWUPDATE_LATEST_SW_REV_MINOR_POS] << 8) |
                                  (pun8Memory[SXI_FWUPDATE_LATEST_SW_REV_INC_POS]);


    } while(FALSE);

    OSAL.vMemoryFree(pun8Memory);
    return eReturnCode;
}

/*****************************************************************************
 *
 *   RADIO_eGetIRDurationOfBuffer
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM RADIO_eGetIRDurationOfBuffer(
    DECODER_OBJECT hDecoder,
    UN32 *pun32IRDurationOfBuffer
        )
{
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)DECODER_hGetRadioSpecificData(
                                                hDecoder);
    if (   (psDecoder != NULL)
        && (pun32IRDurationOfBuffer != NULL)
       )
    {
        *pun32IRDurationOfBuffer = psDecoder->sPlayback.un16DurationOfBuffer;
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   RADIO_bSmartSelect
 *
 *   This API causes the RADIO to tune a set of Smart Favorite (BIR) channels
 *   specified by Service Ids.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the tune from.
 *       tServiceId - an array of channels defined by service ids to
 *           tune as the current smart favorites bank.
 *       un16NumServiceIds - the number of service ids in the tServiceId
 *           array.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bSmartSelect(
    DECODER_OBJECT hDecoder,
    SERVICE_ID atServiceId[],
    UN16 un16NumServiceIds
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(
                        hDecoder);

    if(psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "Requesting tune of Smart (Favorites) Bank\n");

        // Request tune
        eStatusCode = SXIAPI_eChanAttrListCmd(
                psDecoder->hControlCxn,
                SXIAPI_CHANNELATTRLIST_SMART_FAVORITES,
                (SXIAPI_SID *)atServiceId,
                un16NumServiceIds
                    );

        if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bSetSmartFavsCfg
 *
 *   This API causes the RADIO to config how the Module should handle
 *   the playback audio stream and how many IR buffers shall be reserved
 *   for Smart Favorites.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the tune from.
 *       ePlayPoint - Controls where a track begins playing for set
 *           channel and channel scan.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bSetSmartFavsCfg(
    DECODER_OBJECT hDecoder,
    SMART_FAVORITES_PLAY_POINT_CTRL_ENUM ePlayPoint
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        // Set Play Point Ctrl variable
        // All other parameters remain as last configured
        switch ( ePlayPoint )
        {
            case SMART_FAVORITES_PLAY_POINT_CTRL_AUTOMATIC:
            {
                psDecoder->sChanSelConfig.ePlayPoint =
                    SXIAPI_ADV_IR_PLAYPOINT_AUTOMATIC;
            }
            break;

            case SMART_FAVORITES_PLAY_POINT_CTRL_LIVE:
            {
                psDecoder->sChanSelConfig.ePlayPoint =
                    SXIAPI_ADV_IR_PLAYPOINT_AT_LIVE;
            }
            break;

            case SMART_FAVORITES_PLAY_POINT_CTRL_START:
            {
                psDecoder->sChanSelConfig.ePlayPoint =
                    SXIAPI_ADV_IR_PLAYPOINT_AT_START;
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                    "Incorrect Play On Select Method requested\n");
                return FALSE;
            }
        }

        eStatusCode = SXIAPI_eChanSelectCfgCmd(
                psDecoder->hControlCxn,
                &psDecoder->sChanSelConfig
                    );

        if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bSetPlaySeconds
 *
 *   This API causes the RADIO to config number of seconds to play a scan
 *   item before going to the next
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the tune from.
 *       un8PlaySeconds - A number of seconds to play a scan item before
 *           going to the next.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bSetPlaySeconds (
    DECODER_OBJECT hDecoder,
    UN8 un8PlaySeconds
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        // Verify Play Seconds range
        if ((un8PlaySeconds > SXIAPI_ADV_IR_SCAN_PLAY_SECONDS_MAX) ||
            (un8PlaySeconds < SXIAPI_ADV_IR_SCAN_PLAY_SECONDS_MIN))
        {
            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Incorrect Play Seconds value requested\n");
            return FALSE;
        }

        psDecoder->sChanSelConfig.tPlaySeconds =
            (SXIAPI_ADV_IR_SCAN_PLAY_SECONDS)un8PlaySeconds;

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "Requesting Channel Select Play Seconds Configuration\n");

        eStatusCode = SXIAPI_eChanSelectCfgCmd(
                psDecoder->hControlCxn,
                &psDecoder->sChanSelConfig
                    );

        if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bLockChannel
 *
 *   This API causes the RADIO to set 'Locked' channel attribute flag
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the tune from.
 *       tChangeAttrib - Indexed list of change types to make to the
 *           specified list of channel
 *       tServiceId - an array of channels defined by service ids
 *       un16NumServiceIds - the number of service ids in the tServiceId array
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bLockChannel (
    DECODER_OBJECT hDecoder,
    SERVICE_ID atServiceId[],
    UN16 un16NumServiceIds
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder =
        (RADIO_DECODER_OBJECT_STRUCT *)
            DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "Requesting channels lock\n");

        // Request channel(s) lock
        eStatusCode = SXIAPI_eChanAttrCfgCmd(
                psDecoder->hControlCxn,
                SXIAPI_CHANNELATTRSIDV_LOCK,
               (SXIAPI_SID *)atServiceId,
                un16NumServiceIds
                    );

        if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bSkipChannel
 *
 *   This API causes the RADIO to set 'Skipped' channel attribute flag
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the tune from.
 *       tChangeAttrib - Indexed list of change types to make to the
 *           specified list of channel
 *       tServiceId - an array of channels defined by service ids
 *       un16NumServiceIds - the number of service ids in the tServiceId array
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bSkipChannel (
    DECODER_OBJECT hDecoder,
    SERVICE_ID atServiceId[],
    UN16 un16NumServiceIds
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder =
        (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "Requesting channels skip\n");

        // Request channel(s) skip
        eStatusCode = SXIAPI_eChanAttrCfgCmd(
                psDecoder->hControlCxn,
                SXIAPI_CHANNELATTRSIDV_SKIP,
               (SXIAPI_SID *)atServiceId,
                un16NumServiceIds
                    );

        if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bAudioVolume
 *
 *   This API uses to configure the Module audio port volume (gain) level
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the tune from.
 *       n8Volime - A signed byte value which configures the Module audio
 *           port volume gain adjustment
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bAudioVolume(
    DECODER_OBJECT hDecoder,
    N8 n8Volume
        )
{
    BOOLEAN bReturn = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder = NULL;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(
            hDecoder);

    if (psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "Requesting to change audio volume\n");

        eStatusCode = SXIAPI_eAudioVolCmd(
            psDecoder->hControlCxn,
            n8Volume
                );

        if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bReturn = TRUE;
        }
    }

    return bReturn;
}

/*****************************************************************************
 *
 *   RADIO_bPostModuleEvent
 *
 *****************************************************************************/
BOOLEAN RADIO_bPostModuleEvent (
    MODULE_OBJECT hModule,
    SMS_EVENT_TYPE_ENUM eEvent,
    EVENT_OPTIONS_TYPE tOptions,
    const void *pvData
        )
{
    return bPostModuleEvent(
        hModule, eEvent, tOptions, pvData);
}
/*****************************************************************************
 *
 *   RADIO_bTuneMixConfigure
 *
 *   This API causes the RADIO to assign a list of TuneMix components
 *   specified by Service IDs.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONTEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the channel attributes set.
 *       ptServiceId - an array of channels defined for service
 *       un16NumServiceIds - the number of Service IDs in the tServiceId
 *           array.
         un8TuneMixChannelIndex - TuneMix service Id.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bTuneMixConfigure (
    DECODER_OBJECT hDecoder,
    SERVICE_ID *ptServiceId,
    UN16 un16NumServiceIds,
    UN8 un8TuneMixChannelIndex
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData( hDecoder );

    if (psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "Requesting configure of Tune Mix\n");

        // Request tune
        eStatusCode = SXIAPI_eChanAttrListCmd(
            psDecoder->hControlCxn,
            (SXIAPI_CH_ATTRIB)(un8TuneMixChannelIndex +
                TUNEMIX_CHANNELATTRIB_OFFSET),
            (SXIAPI_SID *)ptServiceId,
            un16NumServiceIds
            );

        if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bUpdateFeaturedFavoritesState
 *
 *****************************************************************************/
BOOLEAN RADIO_bUpdateFeaturedFavoritesState (
    DECODER_OBJECT hDecoder,
    BOOLEAN bUpdateComplete
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        // Wrap to the protocol specific method
        bSuccess = SXI_GMD_bSetFFTableStatus(
            psDecoder->hControlCxn, bUpdateComplete);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bGetFFTableStatus
 *
 *****************************************************************************/
BOOLEAN RADIO_bGetFFTableStatus (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bComplete = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        // Wrap to the protocol specific method
        bComplete = SXI_GMD_bGetFFTableStatus(
            psDecoder->hControlCxn);
    }

    return bComplete;
}

/*****************************************************************************
 *
 *   RADIO_bSeekMonitorEnable
 *   This API causes the RADIO to set up a Seek Monitor
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONTEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the seek monitor setup from
 *       eType - the Seek Monitor to set up
 *       un8ValueCnt - number of items in the pvValues array
 *       un8ValueLen - size of each item
 *       pvValues - array of meta data items to monitor
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bSeekMonitorEnable (
    DECODER_OBJECT hDecoder,
    SEEK_MONITOR_TYPE_ENUM eType,
    UN8 un8ValueCnt,
    UN8 un8ValueLen,
    const void *pvValues
        )
{
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    if (eType >= SEEK_MONITOR_TYPE_INVALID)
    {
        return FALSE;
    }

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        // Wrap to the protocol specific method
        eStatusCode = SXIAPI_eSeekMonCmd(
            psDecoder->hControlCxn,
            (UN8)eType,
            SXIAPI_MONITOR_OP_ENABLE,
            gaeSeekMonTMI[eType],
            un8ValueCnt,
            un8ValueLen,
            pvValues,
            sizeof(gatSeekMonSportsFlashReportTMI) /
                sizeof(gatSeekMonSportsFlashReportTMI[0]),
            &gatSeekMonSportsFlashReportTMI[0],
            gaun8SeekMonControlMask[eType]
                );

        if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bSeekMonitorDisable
 *   This API causes the RADIO to cancel a Seek Monitor
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONTEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the seek monitor setup from
 *       eType - the Seek Monitor to cancel.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bSeekMonitorDisable (
    DECODER_OBJECT hDecoder,
    SEEK_MONITOR_TYPE_ENUM eType
        )
{
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        // Wrap to the protocol specific method
        eStatusCode = SXIAPI_eSeekMonCmd(
            psDecoder->hControlCxn,
            (UN8)eType,
            SXIAPI_MONITOR_OP_DISABLE,
            gaeSeekMonTMI[eType],
            0,
            0,
            NULL,
            0,
            NULL,
            0
                );

        if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bSeekMonitorDisableAll
 *   This API causes the RADIO to cancel all Seek Monitors
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONTEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the seek monitor setup from
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bSeekMonitorDisableAll (
    DECODER_OBJECT hDecoder
        )
{
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        // Wrap to the protocol specific method
        eStatusCode = SXIAPI_eSeekMonCmd(
            psDecoder->hControlCxn,
            0,
            SXIAPI_MONITOR_OP_DISABLE_ALL,
            SXIAPI_TMI_SONG_ID,
            0,
            0,
            NULL,
            0,
            NULL,
            0
                );

        if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bSetGamesMonitor
 *
 *   This API causes the RADIO to set a Sports Flash games monitor for the
 *   channels specified by Service Ids.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONTEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the games monitor setup from.
 *       atServiceId[] - an array of channels defined by service ids to
 *           set the games monitor to.
 *       un16NumServiceIds - the number of service ids in the tServiceId
 *           array.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bSetGamesMonitor(
    DECODER_OBJECT hDecoder,
    SERVICE_ID atServiceId[],
    UN16 un16NumServiceIds
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(
                        hDecoder);

    if(psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "Requesting Sports Flash Games Monitor\n");

        // Request tune
        eStatusCode = SXIAPI_eChanAttrListCmd(
                psDecoder->hControlCxn,
                SXIAPI_CHANNELATTRLIST_SPORTS_FLASH,
                (SXIAPI_SID *)atServiceId,
                un16NumServiceIds
                    );

        if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bPlayFlashEvent
 *
 *   This API causes the RADIO to tune a Flash Event channel.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONTEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the tune from.
 *       tEventId - the flash event id.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bPlayFlashEvent(
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tEventId
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    SXIAPI_SID tSID = (SXIAPI_SID)tEventId;

    SXIAPI_CH_ATTRIB tOverrides = SXIAPI_CHANNELATTR_MATURE |
                                  SXIAPI_CHANNELATTR_LOCKED |
                                  SXIAPI_CHANNELATTR_SKIPPED;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);
    if(psDecoder == NULL)
    {
        // Error!
        return FALSE;
    }

    // Tune...
    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
        "Requesting tune for Flash event id:%u"
        " with overrides: 0x%x\n", tSID, tOverrides);

    // Request channel select (tune)
    eStatusCode = SXIAPI_eChanSelectCmd(
        psDecoder->hControlCxn,
        SXIAPI_SELECT_TYPE_PLAY_FLASH_EVENT,
        tSID,
        SXIAPI_CATEGORYID_SERVICENOTASSIGNED,
        tOverrides
            );
    if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
    {
    	// Command sent successfully
        bSuccess = TRUE;
    }
    else
    {
        // This command failed due to bad sid, or overrides, timeout, etc.
        // eitherway some type of major failure, unknown.
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "RADIO Error! Requesting tune for flash event id:%u"
            " failed with error %u.\n", tSID, eStatusCode);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bAbortFlashEvent
 *   This API causes the RADIO to abort a Flash Event channel playback.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONTEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the abort from.
 *       tEventId - the flash event id.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bAbortFlashEvent(
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tEventId
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    SXIAPI_SID tSID = (SXIAPI_SID)tEventId;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(
                        hDecoder);
    if(psDecoder == NULL)
    {
        // Error!
        return FALSE;
    }

    // Tune...
    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
        "Requesting abort for Flash event id:%u\n", tSID);

    // Request channel select (tune)
    eStatusCode = SXIAPI_eChanSelectCmd(
        psDecoder->hControlCxn,
        SXIAPI_SELECT_TYPE_ABORT_FLASH_EVENT,
        tSID,
        SXIAPI_CATEGORYID_SERVICENOTASSIGNED,
        SXIAPI_CHANNELATTR_NONE
            );
    if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
    {
        // Command sent successfully
        bSuccess = TRUE;
    }
    else
    {
        // This command failed due to bad sid, or overrides, timeout, etc.
        // eitherway some type of major failure, unknown.
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "RADIO Error! Requesting abort for flash event id:%u"
            " failed with error %u.\n", tSID, eStatusCode);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bRemainFlashEventChannel
 *   This API causes the RADIO to remain on a Flash Event channel.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONTEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the remain from.
 *       tEventId - the flash event id.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bRemainFlashEventChannel(
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tEventId
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    SXIAPI_SID tSID = (SXIAPI_SID)tEventId;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(
                        hDecoder);
    if(psDecoder == NULL)
    {
        // Error!
        return FALSE;
    }

    // Tune...
    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
        "Requesting remain for Flash event id:%u\n", tSID);

    // Request channel select (tune)
    eStatusCode = SXIAPI_eChanSelectCmd(
        psDecoder->hControlCxn,
        SXIAPI_SELECT_TYPE_REMAIN_FLASH_EVENT_CHANNEL,
        tSID,
        SXIAPI_CATEGORYID_SERVICENOTASSIGNED,
        SXIAPI_CHANNELATTR_NONE
            );
    if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
    {
        // Command sent successfully
        bSuccess = TRUE;
    }
    else
    {
        // This command failed due to bad sid, or overrides, timeout, etc.
        // eitherway some type of major failure, unknown.
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "RADIO Error! Requesting remain for flash event id:%u"
            " failed with error %u.\n", tSID, eStatusCode);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_vSelfTuneComplete
 *
 *****************************************************************************/
void RADIO_vSelfTuneComplete(
    DECODER_OBJECT hDecoder
        )
{
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        psDecoder->bSelfTuneComplete = TRUE;
    }

    return;
}

/*****************************************************************************
 *
 *   RADIO_bBulletinMonitor
 *   This API causes the RADIO to set up a Bulletin Monitor
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONTEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the seek monitor setup from
 *       eType - the Bulletin Monitor to set up
 *       un8ParamCnt - number of items in the pun16ParamList array
 *       pun16ParamList - array of Bulletin monitor parameters
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bBulletinMonitor (
    DECODER_OBJECT hDecoder,
    BULLETIN_MONITOR_TYPE_ENUM eType,
    UN8 un8ParamCnt,
    const UN16 *pun16ParamList
        )
{
    SXIAPI_STATUSCODE_ENUM eStatusCode = SXIAPI_STATUSCODE_ERROR;
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    if ((eType >= BULLETIN_MONITOR_TYPE_INVALID) ||
        ((pun16ParamList == NULL) && (un8ParamCnt > 0)))
    {
        // Invalid parameters
        return bSuccess;
    }

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        // Wrap to the protocol specific method
        eStatusCode = SXIAPI_eBulletinMonCmd(
            psDecoder->hControlCxn,
            (UN8)eType,
            un8ParamCnt,
            pun16ParamList
                );

        if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bPlayBulletin
 *
 *   This API causes the RADIO to tune a TW Now Bulletin channel.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONTEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the tune from.
 *       tBulletinId - the bulletin id.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bPlayBulletin(
    DECODER_OBJECT hDecoder,
    TW_NOW_BULLETIN_ID tBulletinId
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    SXIAPI_SID tSID = (SXIAPI_SID)tBulletinId;

    SXIAPI_CH_ATTRIB tOverrides = SXIAPI_CHANNELATTR_MATURE |
                                  SXIAPI_CHANNELATTR_LOCKED |
                                  SXIAPI_CHANNELATTR_SKIPPED;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(hDecoder);
    if(psDecoder == NULL)
    {
        // Error!
        return bSuccess;
    }

    // Tune...
    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
        "Requesting tune for TW Now Bulletin id:%u"
        " with overrides: 0x%x\n", tSID, tOverrides);

    // Request channel select (tune)
    eStatusCode = SXIAPI_eChanSelectCmd(
        psDecoder->hControlCxn,
        SXIAPI_SELECT_TYPE_PLAY_BULLETIN,
        tSID,
        SXIAPI_CATEGORYID_SERVICENOTASSIGNED,
        tOverrides
            );
    if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
    {
        // Command sent successfully
        bSuccess = TRUE;
    }
    else
    {
        // This command failed due to bad sid, or overrides, timeout, etc.
        // eitherway some type of major failure, unknown.
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "RADIO Error! Requesting tune for bulletin id:%u"
            " failed with error %u.\n", tSID, eStatusCode);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RADIO_bAbortBulletin
 *   This API causes the RADIO to abort a TW Now Bulletin channel playback.
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONTEXT OF THE DECODER OBJECT.
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the abort from.
 *       tBulletinId - the bulletin id.
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned.
 *
 *****************************************************************************/
BOOLEAN RADIO_bAbortBulletin(
    DECODER_OBJECT hDecoder,
    TW_NOW_BULLETIN_ID tBulletinId
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;
    SXIAPI_STATUSCODE_ENUM eStatusCode;
    SXIAPI_SID tSID = (SXIAPI_SID)tBulletinId;

    // Obtain radio data
    psDecoder = (RADIO_DECODER_OBJECT_STRUCT *)
        DECODER_hGetRadioSpecificData(
                        hDecoder);
    if(psDecoder == NULL)
    {
        // Error!
        return FALSE;
    }

    // Tune...
    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
        "Requesting abort for TW Now Bulletin id:%u\n", tSID);

    // Request channel select (tune)
    eStatusCode = SXIAPI_eChanSelectCmd(
        psDecoder->hControlCxn,
        SXIAPI_SELECT_TYPE_ABORT_BULLETIN,
        tSID,
        SXIAPI_CATEGORYID_SERVICENOTASSIGNED,
        SXIAPI_CHANNELATTR_NONE
            );
    if(eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
    {
        // Command sent successfully
        bSuccess = TRUE;
    }
    else
    {
        // This command failed due to bad sid, or overrides, timeout, etc.
        // eitherway some type of major failure, unknown.
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "RADIO Error! Requesting abort for TW Now Bulletin id:%u"
            " failed with error %u.\n", tSID, eStatusCode);
    }

    return bSuccess;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   vUninitializeSrm
 *
 *****************************************************************************/
static void vUninitializeSrm(
    RADIO_SRM_OBJECT_STRUCT *psSRM
        )
{
    BOOLEAN bSuccess;

    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
        "%s(): Uninit RADIO SRM...\n", __FUNCTION__);

    if (NULL == psSRM)
    {
        // Nothing to uninit
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
            "%s(): Nothing to uninit.\n", __FUNCTION__);
        return;
    }

    if (SRM_INVALID_OBJECT != psSRM->hSRM)
    {
        // Set radio specific SRM data to NULL
        bSuccess = SRM_bSetRadioSpecificData(psSRM->hSRM,
            RADIO_PRIVATE_DATA_INVALID_OBJECT);
        if (FALSE == bSuccess)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Can't set RADIO SRM data.");
            return;
        }
    }

    // Close connections to SRM interfaces (SXi)
    if (NULL != psSRM->psDevice)
    {
        // Turn off the SRM
        bSuccess = bPowerOffSRM(psSRM->psDevice, psSRM->pacName);
        if (FALSE == bSuccess)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot power down the SRM");

            // We should perform un-initialization regardless of 
            // the status of hardware, since the object is going 
            // to be destroyed anyway.
        }

        // Close Device Port, using the SRM driver
        fclose(psSRM->psDevice);
    }

    // Destroy RADIO specific SRM data object
    SMSO_vDestroy((SMS_OBJECT)psSRM);

    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
        "%s(): RADIO SRM is uninitialized.\n", __FUNCTION__);

    return;
}

/*****************************************************************************
 *
 *   vUninitializeModule
 *
 *****************************************************************************/
static void vUninitializeModule(
    RADIO_MODULE_OBJECT_STRUCT *psModule)
{
    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
        "%s(): Uninit RADIO MODULE...\n", __FUNCTION__);

    if (NULL == psModule)
    {
        // Nothing to uninit
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
            "%s(): Nothing to uninit.\n", __FUNCTION__);
        return;
    }

    if (MODULE_INVALID_OBJECT != psModule->hModule)
    {
        BOOLEAN bSuccess;

        // Set radio specific MODULE data to NULL
        bSuccess = MODULE_bSetRadioSpecificData(
            psModule->hModule, RADIO_PRIVATE_DATA_INVALID_OBJECT);
        if (FALSE == bSuccess)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Can't set RADIO MODULE data.");
            return;
        }
    }

    // Release ESN memory resources
    vReleaseESN(psModule);

    // Release package memory (if it exists)
    if(psModule->sSXI.sPkgInd.pun8PkgInd != NULL)
    {
        OSAL.vMemoryFree(psModule->sSXI.sPkgInd.pun8PkgInd);
        psModule->sSXI.sPkgInd.tNumBytes = 0;
        psModule->sSXI.sPkgInd.pun8PkgInd = NULL;
    }

    // Clear remaining SXi info
    psModule->sSXI.eCurDispAdvisoryIndCode = SXIAPI_INDCODE_ERROR;
    psModule->sSXI.hControlCxn = STI_INVALID_HDL;

    if(psModule->sSXI.hFSM != SXI_FSM_INVALID_HDL)
    {
        SXI_FSM_vUnInit(psModule->sSXI.hFSM);
        psModule->sSXI.hFSM = SXI_FSM_INVALID_HDL;
    }

    // Invalidate subscription stuff
    psModule->sSubstatus.eStatus = SXIAPI_SUB_STATUS_INVALID;
    psModule->sSubstatus.tReasonCode = MODULE_SUBSTATUS_REASON_CODE_INVALID;
    psModule->sSubstatus.un32UTCTime = 0;
    psModule->sSubstatus.bIsAudioSubscribed = FALSE;

    // Release STRING objects
    STRING_vDestroy(psModule->sSubstatus.hReasonText);
    psModule->sSubstatus.hReasonText = STRING_INVALID_OBJECT;
    STRING_vDestroy(psModule->sSubstatus.hPhoneNumber);
    psModule->sSubstatus.hPhoneNumber = STRING_INVALID_OBJECT;

    // Invalidate remaining object entries
    psModule->hSRM = SRM_INVALID_OBJECT;
    psModule->hModule = MODULE_INVALID_OBJECT;
    psModule->psDevice = NULL;

    // Free the tuple's memory allocated by the IOCTL
    if(psModule->psDecoderTuple != NULL)
    {
        OSAL.vMemoryFree(psModule->psDecoderTuple);
        psModule->psDecoderTuple = NULL;
        psModule->un32NumDecoders = 0;
    }

    // Destroy RADIO specific MODULE data object
    SMSO_vDestroy((SMS_OBJECT)psModule);

    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
        "%s(): RADIO MODULE uninitialized.\n", __FUNCTION__);

    return;
}

/*****************************************************************************
 *
 *   vUninitializeDecoder
 *
 *****************************************************************************/
static void vUninitializeDecoder (
    RADIO_DECODER_OBJECT_STRUCT *psDecoder
        )
{
    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
        "%s(): Uninit RADIO DECODER...\n", __FUNCTION__);

    if (NULL == psDecoder)
    {
        // Nothing to uninit
        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
            "%s(): Nothing to uninit.\n", __FUNCTION__);
        return;
    }

    if (DECODER_INVALID_OBJECT != psDecoder->hDecoder)
    {
        BOOLEAN bSuccess;

        // Set radio specific DECODER data to NULL
        bSuccess = DECODER_bSetRadioSpecificData(
            psDecoder->hDecoder,
            RADIO_PRIVATE_DATA_INVALID_OBJECT);
        if (FALSE == bSuccess)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Can't set RADIO DECODER data.");
            return;
        }
    }

    if((psDecoder->tCapabilities & SRH_DEVICE_CAPABILITY_AUDIO)
                    == SRH_DEVICE_CAPABILITY_AUDIO)
    {
        // Remove capability
        psDecoder->tCapabilities &= ~SRH_DEVICE_CAPABILITY_AUDIO;

        // Uninit audio services
        vUnInitializeAudioDecoder(psDecoder);
    }
    else if((psDecoder->tCapabilities & SRH_DEVICE_CAPABILITY_VIDEO)
                    == SRH_DEVICE_CAPABILITY_VIDEO)
    {
        // Remove capability
        psDecoder->tCapabilities &= ~SRH_DEVICE_CAPABILITY_VIDEO;
    }
    else if((psDecoder->tCapabilities & SRH_DEVICE_CAPABILITY_DATA)
                    == SRH_DEVICE_CAPABILITY_DATA)
    {
        // Remove capability
        psDecoder->tCapabilities &= ~SRH_DEVICE_CAPABILITY_DATA;
    }

    // Destroy RADIO specific DECODER data object
    SMSO_vDestroy((SMS_OBJECT)psDecoder);

    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 4,
        "%s(): RADIO DECODER uninitialized.\n", __FUNCTION__);

    return;
}

/*****************************************************************************
 *
 *   vProcessModuleInd
 *
 *****************************************************************************/
static void vProcessModuleInd(
    RADIO_MODULE_OBJECT_STRUCT *psModule,
    SXIAPI_RX_STRUCT *psRxData,
    BOOLEAN *pbDispatchToDecoders,
    BOOLEAN *pbEngineeringData
        )
{
    BOOLEAN bDispatchToDecoders, bEngineeringData;

    // If caller did not provide a dispatch flag, then use our own
    if(pbDispatchToDecoders == NULL)
    {
        // Use our own
        pbDispatchToDecoders = &bDispatchToDecoders;
    }

    // Initially do not dispatch to decoders unless specifically
    // determined to do so.
    *pbDispatchToDecoders = FALSE;

    // If caller did not provide an engineering data flag, then use our own
    if(pbEngineeringData == NULL)
    {
        // Use our own
        pbEngineeringData = &bEngineeringData;
    }

    // Initially everything is not engineering data.
    *pbEngineeringData = FALSE;

    ///////////////////////////
    // Decode SXI Indication //
    ///////////////////////////

    // Parse payload based on op-type (See RX223)...
    switch(psRxData->sHdr.tOpType)
    {
        case SXIAPI_MESSOP_MODULECFG:
        {
            // The MODULE has indicated it is READY, but the
            // IndCode may indicate there is an error
            if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL)
            {
                BOOLEAN bPosted;
                MODULE_VERSION_INFO_STRUCT sModuleVerInfo;
                UN16 un16NumOfVersions, un16Counter;
                UN32 un32CompareValue;
                const SXIAPI_MODULECFGIND_STRUCT *psModuleCfgIndSource =
                    &psRxData->uData.sInd.uData.sModCfg;

                // We are ok.  Set the flag to indicate we are ready
                sModuleVerInfo.un8ModuleTypeIDA =
                    psModuleCfgIndSource->un8ModuleTypeIDA;
                sModuleVerInfo.un8ModuleTypeIDB =
                    psModuleCfgIndSource->un8ModuleTypeIDB;
                sModuleVerInfo.un8ModuleTypeIDC =
                    psModuleCfgIndSource->un8ModuleTypeIDC;

                sModuleVerInfo.un8ModuleHWRevA =
                    psModuleCfgIndSource->un8ModuleHWRevA;
                sModuleVerInfo.un8ModuleHWRevB =
                    psModuleCfgIndSource->un8ModuleHWRevB;
                sModuleVerInfo.un8ModuleHWRevC =
                    psModuleCfgIndSource->un8ModuleHWRevC;

                sModuleVerInfo.un8ModSWRevMajor =
                    psModuleCfgIndSource->un8ModSWRevMajor;
                sModuleVerInfo.un8ModSWRevMinor =
                    psModuleCfgIndSource->un8ModSWRevMinor;
                sModuleVerInfo.un8ModSWRevInc =
                    psModuleCfgIndSource->un8ModSWRevInc;

                sModuleVerInfo.un8HDecRevMajor =
                    psModuleCfgIndSource->un8HDecRevMajor;
                sModuleVerInfo.un8HDecRevMinor =
                    psModuleCfgIndSource->un8HDecRevMinor;
                sModuleVerInfo.un8HDecRevInc =
                    psModuleCfgIndSource->un8HDecRevInc;

                sModuleVerInfo.un8BBRevMajor =
                    psModuleCfgIndSource->un8BBRevMajor;
                sModuleVerInfo.un8BBRevMinor =
                    psModuleCfgIndSource->un8BBRevMinor;
                sModuleVerInfo.un8BBRevInc =
                    psModuleCfgIndSource->un8BBRevInc;

                sModuleVerInfo.un8RFRevMajor =
                    psModuleCfgIndSource->un8RFRevMajor;
                sModuleVerInfo.un8RFRevMinor =
                    psModuleCfgIndSource->un8RFRevMinor;
                sModuleVerInfo.un8RFRevInc =
                    psModuleCfgIndSource->un8RFRevInc;

                sModuleVerInfo.un8ProtocolRevMajor =
                    psModuleCfgIndSource->un8SXIRevMajor;
                sModuleVerInfo.un8ProtocolRevMinor =
                    psModuleCfgIndSource->un8SXIRevMinor;
                sModuleVerInfo.un8ProtocolRevInc =
                    psModuleCfgIndSource->un8SXIRevInc;

                sModuleVerInfo.un8SPLRevMajor =
                    psModuleCfgIndSource->un8SPLRevMajor;
                sModuleVerInfo.un8SPLRevMinor =
                    psModuleCfgIndSource->un8SPLRevMinor;
                sModuleVerInfo.un8SPLRevInc =
                    psModuleCfgIndSource->un8SPLRevInc;

                // SXi v3.0 Features
                // Expanded Smart Favorites are set below,
                // based on the module capabilities
                sModuleVerInfo.un8MaxSportsFlash = 
                    psModuleCfgIndSource->un8MaxSportsFlash;
                sModuleVerInfo.un8MaxTuneMix = 
                    psModuleCfgIndSource->un8MaxTuneMix;
                sModuleVerInfo.un8MaxTWNow = 
                    psModuleCfgIndSource->un8MaxTWNow;

                sModuleVerInfo.un32Capability =
                    psModuleCfgIndSource->un32Capability;

                un16NumOfVersions = sizeof(sModVersionStruct)/
                                    sizeof(RADIO_SXI_MOD_VERSION_NAME_STRUCT);
                un32CompareValue = sModuleVerInfo.un8ModuleTypeIDA << 16 |
                                   sModuleVerInfo.un8ModuleTypeIDB << 8  |
                                   sModuleVerInfo.un8ModuleTypeIDC;

                for (un16Counter=0; un16Counter < un16NumOfVersions;
                    un16Counter++)
                {
                    if (sModVersionStruct[un16Counter].un32Type ==
                            un32CompareValue)
                    {
                        sModuleVerInfo.pcModuleHwName =
                            sModVersionStruct[un16Counter].pacNameString;
                        break;
                    }

                }

                if (un16Counter == un16NumOfVersions)
                {
                    // The string shall be 10 bytes (9 + NULL) to be properly
                    // copied into appropriate MODULE_VERSION_OBJECT_STRUCT 
                    // field
                     sModuleVerInfo.pcModuleHwName =("UNKNOWN  ");
                }

                // Get the Duration of IR Buffer
                // We cannot block the MODULE from getting this, but we
                // can at least prevent a race-condition, between update
                // and reading this value.
                // TODO: Note this doesn't work if we have multiple MODULE
                // objects. So we must revisit if that can happen.
                OSAL.eEnterTaskSafeSection();
                psModule->un16DurationOfBuffer =
                gun16DurationOfBuffer =
                    psRxData->uData.sInd.uData.sModCfg.un16DurationOfBuffer;
                OSAL.eExitTaskSafeSection();

                sModuleVerInfo.tFeatures =
                    SXIAPI_MODULECFG_IND_CAPABILITY_NONE;

                // Copy over the capabilities into our feature mask
                if ( (psRxData->uData.sInd.uData.sModCfg.un32Capability &
                        SXIAPI_MODULECFG_IND_CAPABILITY_IR) ==
                                SXIAPI_MODULECFG_IND_CAPABILITY_IR)
                {
                    sModuleVerInfo.tFeatures |= RADIO_MODULE_FEATURE_IR;
                }

                if ( (psRxData->uData.sInd.uData.sModCfg.un32Capability &
                        SXIAPI_MODULECFG_IND_CAPABILITY_AUDIO_REC) ==
                            SXIAPI_MODULECFG_IND_CAPABILITY_AUDIO_REC)
                {
                    sModuleVerInfo.tFeatures |=
                        RADIO_MODULE_FEATURE_AUDIO_RECORDING;
                }

                if ( (psRxData->uData.sInd.uData.sModCfg.un32Capability &
                        SXIAPI_MODULECFG_IND_CAPABILITY_EXTD_CHAN) ==
                            SXIAPI_MODULECFG_IND_CAPABILITY_EXTD_CHAN)
                {
                    sModuleVerInfo.tFeatures |=
                        RADIO_MODULE_FEATURE_OVERLAY;
                }

                if ( (psRxData->uData.sInd.uData.sModCfg.un32Capability &
                        SXIAPI_MODULECFG_IND_CAPABILITY_I2S_SLAVE) ==
                            SXIAPI_MODULECFG_IND_CAPABILITY_I2S_SLAVE)
                {
                    sModuleVerInfo.tFeatures |=
                        RADIO_MODULE_FEATURE_I2S_SLAVE;
                }

                if ( (psRxData->uData.sInd.uData.sModCfg.un32Capability &
                        SXIAPI_MODULECFG_IND_CAPABILITY_ADVANCED_IR) ==
                            SXIAPI_MODULECFG_IND_CAPABILITY_ADVANCED_IR)
                {
                    sModuleVerInfo.tFeatures |=
                        RADIO_MODULE_FEATURE_ADVANCED_IR;
                }

                if ( (psRxData->uData.sInd.uData.sModCfg.un32Capability &
                        SXIAPI_MODULECFG_IND_CAPABILITY_EXPANDED_SMART_FAVS) ==
                            SXIAPI_MODULECFG_IND_CAPABILITY_EXPANDED_SMART_FAVS)
                {
                    sModuleVerInfo.tFeatures |=
                        RADIO_MODULE_FEATURE_EXPANDED_SMART_FAVS;

                    // Set Expanded Smart Favorites provided by the Module
                    sModuleVerInfo.un8MaxSmartFavorites = 
                        psModuleCfgIndSource->un8MaxSmartFavorites;
                }
                else
                {
                    // Expanded Smart Favorites are not supported.
                    // Per SX-9845-0097, "0 = Expanded Smart Favorites not supported. 
                    // Supports 6 Smart Favorites if Advanced IR Available = 1."
                    sModuleVerInfo.un8MaxSmartFavorites = 
                        RADIO_SMART_FAVORITES_DEFAULT;
                }

                // Tell Link-FSM link we have received the ModuleCfgInd.
                bPosted = SXI_FSM_bPostCfgIndRxd(psModule->sSXI.hFSM);
                if(bPosted == FALSE)
                {
                    // Error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        RADIO_OBJECT_NAME": Can't Post SXi FSM event.");
                }

                MODULE_VERSION_vUpdate(psModule->hModule, &sModuleVerInfo);
            }
            else
            {
                BOOLEAN bPosted;

                // some sort of error or something that we don't
                // know what to do with.
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Error IndCode(%u) received from ModCfgInd",
                    psRxData->uData.sInd.eIndCode
                        );

                bPosted = SXI_FSM_bPostError(
                    psModule->sSXI.hFSM, MODULE_ERROR_CODE_RADIO_DEVICE_ERROR);
                if(bPosted == FALSE)
                {
                    // Error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        RADIO_OBJECT_NAME": Can't Post SXI FSM event.");
                }
            }
        }
        break;

        case SXIAPI_MESSOP_PWRMODE:
        {
            if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL)
            {
                /*
                    NOTE: the Module will send the PwrModeInd after the Module
                    completes the power down sequence with requested storage
                    of system persistent parameters.  The Host must wait for
                    this PwrModeInd before removing power or resetting the
                    Module. Host must reset the Module to reestablish the
                    SXi link.
                */

                BOOLEAN bPosted;

                // Tell Link-FSM link is now powered off and to stop
                // monitoring it.
                bPosted = SXI_FSM_bPostRadioPower(
                    psModule->sSXI.hFSM, SXI_FSM_POWER_MODE_OFF);
                if(bPosted == FALSE)
                {
                    // Error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        RADIO_OBJECT_NAME": Can't Post SXi FSM event.");
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Error IndCode received from PwrModeInd");
            }
        }
        break;

        case SXIAPI_MESSOP_TIME:
        {
            SXIAPI_TIMEIND_STRUCT *psTime =
                &psRxData->uData.sInd.uData.sTime;

            // Time as provided in broken down format
            struct tm sTime;

            // Time converted into TIME_T
            TIME_T tTod;

            // Extract time from satellite info...

            // SXM values are 1..12
            sTime.tm_mon = psTime->un8Month - 1;
            sTime.tm_mday = psTime->un8Day;
            // According to X-9845-0097, the year is calculated
            // by adding 2000 to the value read
            sTime.tm_year = ((int)psTime->un8Year + 2000) - 1900;
            sTime.tm_hour = psTime->un8Hour;
            sTime.tm_min = psTime->un8Minute;
            // We don't know the seconds, so use 0
            sTime.tm_sec = 0;
            // filled in by OSAL.mktime, days since sunday (from 0)
            sTime.tm_wday = 0;
            // filled in by OSAL.mktime, day of the year (from 0)
            sTime.tm_yday = 0;
            sTime.tm_isdst = 0; // No DST adjustment

            // 2009 according to SXi spec is the minimum year we should get.
            if (psTime->un8Year >= 9)
            {
                BOOLEAN bTimeSet;

                // Call API to determine new time in seconds since
                // epoch  (in calendar time)
                tTod = OSAL.mktime(&sTime);

                // Tell MODULE to set time using this information and include
                // further information which best describes when it was this time.
                bTimeSet = MODULE_bSetTime(psModule->hModule,
                    tTod, psTime->un32Seconds, psTime->un16Msecs);
                if(FALSE == bTimeSet)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        RADIO_OBJECT_NAME": Unable to set time.");
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Invalid Time (year = %u) Received.", psTime->un8Year);
            }
        }
        break;

        case SXIAPI_MESSOP_STATUS:
        {
            if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL)
            {
                if(psRxData->uData.sInd.uData.sStatus.bEngineeringData == TRUE)
                {
                    // Idicate this is engineering data
                    *pbEngineeringData =
                        psRxData->uData.sInd.uData.sStatus.bEngineeringData;
                }

                // Pass along to all DECODERs
                *pbDispatchToDecoders = TRUE;
            }
        }
        break;

        case SXIAPI_MESSOP_EVENT:
        {
            switch (psRxData->uData.sInd.uData.sEvent.eEventCode)
            {
                case SXIAPI_EVENT_CODE_INFO:
                {
                    // System - Information
                    // chipset providing informative event indication.
                    // No Host action is required.

                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        RADIO_OBJECT_NAME": EventInd ( EV_SYS_INFO ): ");

                    // Please note, that dump will reorder the bytes order
                    OSAL.vDump(stderr, NULL,
                        (void *)&psRxData->uData.sInd.uData.sEvent.aun16EventData[0],
                            sizeof(psRxData->uData.sInd.uData.sEvent.aun16EventData) /
                            sizeof(psRxData->uData.sInd.uData.sEvent.aun16EventData[0]));
                }
                break;

                case SXIAPI_EVENT_CODE_RESET:
                {
                    MODULE_EVENT_DATA_STRUCT sEventData;

                    OSAL.bMemCpy(&sEventData.aun16Data[0],
                        &psRxData->uData.sInd.uData.sEvent.aun16EventData[0],
                        sizeof(sEventData.aun16Data));

                    // Go to error with RADIO_REQUESTED_RESET code.
                    MODULE_bSetEventData(psModule->hModule, &sEventData);
                    bSetModuleError(psModule,
                        MODULE_ERROR_CODE_RADIO_REQUESTED_RESET);
                }
                break;

                case SXIAPI_EVENT_CODE_ACTION:
                    // Host Action
                    // There are currently no defined Event Host Action Codes.
                break;

                case SXIAPI_EVENT_CODE_INVALID:
                default:
                    // invalid. ignore it
                break;
            }
        }
        break;

        case SXIAPI_MESSOP_DISPLAY:
        {
            // This indication is saved by the module
            // for decoders to reference if need be (like during init)
            // The DispAdvisoryInd is a "module-level" message, but
            // most of the data is only interesting to the decoder.
            // We will run this through the module
            // and then let it be dispatched to the decoder

            psModule->sSXI.eCurDispAdvisoryIndCode = psRxData->uData.sInd.eIndCode;

            // We cannot block the DECODER from getting this, but we
            // can at least prevent a race-condition, between update
            // and reading this value.
            // TODO: Note this doesn't work if we have multiple MODULE
            // objects. So we must revisit if that can happen.
            OSAL.eEnterTaskSafeSection();
            geDispAdvisoryIndCode = psModule->sSXI.eCurDispAdvisoryIndCode;
            OSAL.eExitTaskSafeSection();

            // DECODERs may want this
            *pbDispatchToDecoders = TRUE;
        }
        break;

        case SXIAPI_MESSOP_SUBSTATUS:
        {
            // This indication is processed / handled by the module
            // but the information is also tracked by the decoders,
            // so we'll have both process this message

            if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL)
            {
                // get the sub info...

                psModule->sSubstatus.eStatus =
                    psRxData->uData.sInd.uData.sSubStatus.eStatus;
                psModule->sSubstatus.tReasonCode =
                    (MODULE_SUBSTATUS_REASON_CODE)
                        psRxData->uData.sInd.uData.sSubStatus.un8ReasonCode;
                psModule->sSubstatus.bIsAudioSubscribed =
                    psRxData->uData.sInd.uData.sSubStatus.bIsAudioSubscribed;

                psModule->sSubstatus.un32UTCTime =
                    psRxData->uData.sInd.uData.sSubStatus.un32UTCSuspendTime;

                if (psModule->sSubstatus.hPhoneNumber == STRING_INVALID_OBJECT)
                {
                    psModule->sSubstatus.hPhoneNumber =
                        STRING.hCreate(&psRxData->uData.sInd.uData.sSubStatus.acPhoneNumber[0],
                            sizeof(psRxData->uData.sInd.uData.sSubStatus.acPhoneNumber));
                }
                else
                {
                    STRING.bModifyCStr(psModule->sSubstatus.hPhoneNumber,
                        &psRxData->uData.sInd.uData.sSubStatus.acPhoneNumber[0]);
                }

                if (psModule->sSubstatus.hReasonText == STRING_INVALID_OBJECT)
                {
                    psModule->sSubstatus.hReasonText =
                        STRING.hCreate(&psRxData->uData.sInd.uData.sSubStatus.acReasonText[0],
                            sizeof(psRxData->uData.sInd.uData.sSubStatus.acReasonText));
                }
                else
                {
                    STRING.bModifyCStr(psModule->sSubstatus.hReasonText,
                        &psRxData->uData.sInd.uData.sSubStatus.acReasonText[0]);
                }

                // if we don't have an ESN yet...
                if(psModule->hESN == STRING_INVALID_OBJECT)
                {
                    // Release ESN memory resources (if they exist)
                    vReleaseESN(psModule);

                    // Allocate the space based upon what we received
                    psModule->pacESN = (const char *)SMSO_hCreate("ESN",
                        sizeof(psRxData->uData.sInd.uData.sSubStatus.acRadioID),
                        (SMS_OBJECT)psModule, FALSE);

                    // Make sure we allocated
                    if(psModule->pacESN != NULL)
                    {
                        BOOLEAN bPosted;
                        char *pacESN = (char *)psModule->pacESN;

                        // Since we have a valid pointer at this juncture, let's
                        // perform the copy to the "persistent location".
                        strncpy(pacESN,
                            &psRxData->uData.sInd.uData.sSubStatus.acRadioID[0],
                            sizeof(psRxData->uData.sInd.uData.sSubStatus.acRadioID));

                        // Create the constant based upon the ESN we received in
                        // the response
                        psModule->hESN = STRING_hCreateConst(psModule->pacESN,
                            strlen(psModule->pacESN));

                        // If the string constant creation attempt failed
                        // we can't do much but try to get it next time

                        // Tell Link-FSM link we have received the ESN.
                        bPosted = SXI_FSM_bPostEsnRxd(psModule->sSXI.hFSM);
                        if(bPosted == FALSE)
                        {
                            // Error!
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                RADIO_OBJECT_NAME": Can't Post SXi FSM event.");
                        }
                    }
                }

                MODULE_vUpdateSubStatus(psModule->hModule);

                // DECODERs may want this
                *pbDispatchToDecoders = TRUE;
            }
            else if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_FA_ERROR_CAP_IO_ERROR)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Factory Activation - Error - "
                        "SDEC Master I2C communication error with EEPROM");

                bSetModuleError(psModule, MODULE_ERROR_CODE_RADIO_DEVICE_ERROR);
            }
            else if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_FA_ERROR_NVM_INVALID)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Factory Activation - Error - crypto image error in NVM");

                bSetModuleError(psModule, MODULE_ERROR_CODE_RADIO_DEVICE_ERROR);
            }
            else if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_FA_ERROR_UNBOUND)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Factory Activation - Error - unbound");

                bSetModuleError(psModule, MODULE_ERROR_CODE_RADIO_DEVICE_ERROR);
            }
            else if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_FA_ERROR_OTP_INVALID)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Factory Activation - Error - "
                        "critical chip error, one time programmable register value is invalid");

                bSetModuleError(psModule, MODULE_ERROR_CODE_RADIO_DEVICE_ERROR);
            }
        }
        break;

        case SXIAPI_MESSOP_GLOBAL_METADATA:
        {
             SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                 "GlobalMetaDataInd\n");

            // MODULE IND - Handle GMD...
            SXI_GMD_vModuleUpdate(
                psModule->sSXI.hControlCxn,
                &psRxData->uData.sInd.uData.sGlobalMetaData,
                pbDispatchToDecoders
                    );
        }
        break;

        case SXIAPI_MESSOP_DATASERVICE:
        {
            BOOLEAN bSuccess;

            // Replicate response structure for dispatch to the DSM.
            SXI_vReAllocateResponse(psRxData);

            bSuccess = bPostDataEvent(
                (SMS_EVENT_TYPE_ENUM)RADIO_EVENT_SXI_RX_DATA,
                psRxData);
            if (bSuccess == FALSE)
            {
                // Free replicated response
                SXI_vFreeResponse(psRxData);
            }
        }
        break;

        case SXIAPI_MESSOP_CONTENT_BUFFERED:
        {
            SXIAPI_CONTENT_BUFFERED_IND_STRUCT *psCB =
                &psRxData->uData.sInd.uData.sContentBuffered;
            BOOLEAN bPosted = FALSE;

            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "ContentBufferedInd\n");

            if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL)
            {
                SMS_EVENT_DATASERVICE_SXI_MESSAGE_EVENT_UNION uSxiMessage;
                SXM_EVENT_DATASERVICE_SXI_CONTENT_BUFFERED_STRUCT *psDSCB;

                psDSCB = &uSxiMessage.sContentBuffered;

                psDSCB->tChannleId = (CHANNEL_ID)psCB->tChanID;
                psDSCB->tServiceId = (SERVICE_ID)psCB->tSID;
                psDSCB->un8PIDCount = psCB->un8ProgramIDCnt;
                psDSCB->atPIDList = psCB->ptProgramIDList;

                bPosted =
                    DATASERVICE_MGR_bPostEvent(DATASERVICE_MGR_INVALID_OBJECT,
                        DATASERVICE_FW_EVENT_SXI_MESSAGE,
                        &uSxiMessage);
            }

            if (bPosted == FALSE)
            {
                OSAL.vMemoryFree(psCB->ptProgramIDList);
                psCB->ptProgramIDList = NULL;
            }
        }
        break;

        case SXIAPI_MESSOP_FIRMWAREUPDATE_ERASE:
        {
            BOOLEAN bReturn;
            RADIO_FWUPDATE_STRUCT sFWUpdateEventStruct;

            OSAL.bMemSet(&sFWUpdateEventStruct,
                    sizeof(sFWUpdateEventStruct), 0);

            if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_FAIL)
            {
                sFWUpdateEventStruct.eFWUpdateEvent =
                    RADIO_FWUPDATE_EVENT_ERROR;
                sFWUpdateEventStruct.eFWUpdateError =
                    RADIO_FWUPDATE_ERROR_ERASE_TIMEOUT;
            }
            else
            {
                sFWUpdateEventStruct.eFWUpdateEvent =
                    RADIO_FWUPDATE_EVENT_ERASE_COMPLETE;
            }

            bReturn = bPostModuleEvent(
                psModule->hModule,
                (SMS_EVENT_TYPE_ENUM) RADIO_EVENT_FWUPDATE,
                SMS_EVENT_OPTION_NONE,
                (void *)&sFWUpdateEventStruct);
            if (bReturn == FALSE)
            {
                //not much we can do here
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,
                    __LINE__,
                    RADIO_OBJECT_NAME
                    ": Unable to post firmware erased event");
            }
        }
        break;

        case SXIAPI_MESSOP_PACKAGE:
        {
            // We transfer in the package info here as long was
            // we have something to transfer.
            if(psRxData->uData.sInd.uData.sPkg.pun8PkgInd != NULL) // new?
            {
                MODULE_bSetPkgData(psModule->hModule,
                    psRxData->uData.sInd.uData.sPkg.tNumBytes,
                    psRxData->uData.sInd.uData.sPkg.pun8PkgInd
                        );

                // Data transferred
            }
        }
        break;

        default:
            // Do nothing in the MODULE, however maybe DECODERs need it?
            *pbDispatchToDecoders = TRUE;
        break;
    }

    return;
}

/*******************************************************************************
 *
 *   vUpdateSCacheState
 *
 *******************************************************************************/
static void vUpdateSCacheState(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SCACHE_OBJECT hSCache
        )
{
    if (psDecoder->sPlayback.bReady == TRUE)
    {
        if (psDecoder->sPlayback.bReadySent == FALSE)
        {
            // If SCache is ready, and notification isn't sent yet,
            // it's time, since the current song offset is 'in-range'.
            SCACHE_vReady(hSCache);
            psDecoder->sPlayback.bReadySent = TRUE;

            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "SCache Ready has been sent\n");
        }
    }
    else
    {
        // SCache is not ready yet, but the current song offset
        // is already 'in-range'. Wait until all tracks will be received.
        psDecoder->sPlayback.bOffsetReady = TRUE;

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "Current Song offset is ready\n");
    }
}

/*******************************************************************************
 *
 *   vProcessDecoderInd
 *
 *******************************************************************************/
static void vProcessDecoderInd(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_RX_STRUCT *psRxData)
{
    ///////////////////////////
    // Decode SXI Indication //
    ///////////////////////////

    if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_ERROR)
    {
        // This is an SMS general error.
        // Indicates some SMS system issue
        return;
    }

    if((psDecoder->tCapabilities & SRH_DEVICE_CAPABILITY_AUDIO)
                    == SRH_DEVICE_CAPABILITY_AUDIO)
    {

        // Parse payload based on op-type (See RX223)...
        switch(psRxData->sHdr.tOpType)
        {
            case SXIAPI_MESSOP_STATUS:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "StatusInd\n");
                if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL)
                {
                    vUpdateStatus(psDecoder, &psRxData->uData.sInd.uData.sStatus);
                }
            }
            break;

            case SXIAPI_MESSOP_DISPLAY:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "DisplayAdvisoryInd\n");

                // We should always update our current notion of display advisories
                vUpdateDisplayAdvisories(psDecoder,
                    psRxData->uData.sInd.eIndCode, FALSE);
            }
            break;

            case SXIAPI_MESSOP_SUBSTATUS:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "SubStatusInd\n");

                if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL)
                {
                    psDecoder->sSubStatus.eStatus =
                        eSXISubStatusToSMSSubStatus(psRxData->uData.sInd.uData.sSubStatus.eStatus);

                    psDecoder->sSubStatus.tReasonCode =
                        (MODULE_SUBSTATUS_REASON_CODE)psRxData->uData.sInd.uData.sSubStatus.un8ReasonCode;

                    psDecoder->sSubStatus.un32SubSuspendDate =
                        psRxData->uData.sInd.uData.sSubStatus.un32UTCSuspendTime;
                }
            }
            break;

            case SXIAPI_MESSOP_CHANINFO:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "ChanInfoInd\n");

                if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_ADD_UPDATE)
                {
                    // Make sure we have an entry for this category in our list
                    vUpdateCategory(psDecoder,
                        psRxData->uData.sInd.uData.sChanInfo.sChannel.tCatID,
                        NULL, NULL, NULL, FALSE);
                    // Find this channel in the channel cache, if it exists
                    // update the contents. If not, create it and update it
                    vUpdateChannel(psDecoder,
                        psRxData->uData.sInd.uData.sChanInfo.sChannel.tChanID,
                        psRxData->uData.sInd.uData.sChanInfo.sChannel.tSID,
                        &psRxData->uData.sInd.uData.sChanInfo.sChannel,
                        (SXIAPI_TRACK_INFO_STRUCT *)NULL);

                    /*
                        SXi Spec SX-9845-0097, Section 9.1 ChanInfoInd says...
                        Version: 2.0.2  INTERNAL  ONLY December, 15, 2011

                        "The Module will indicate that it has sent at least one ChanInfoInd message for
                        every Available channel by sending the ChanInfoInd with ChanID and SID set to
                        0.  In other words, the Module sends the ChanInfoInd for ChanID=0 to indicate
                        it completed sending the latest information for all Available channels.  (This
                        allows the Host to build or verify a master list of all Available channels, if
                        desired.)  Thereafter, the Module will send ChanInfoInd messages when it
                        detects a change in an Available channel."
                    */
                    if (psRxData->uData.sInd.uData.sChanInfo.sChannel.tChanID == 0)
                    {
                        psDecoder->bChannel0IndRxd = TRUE;
                    }

                    if ( (psDecoder->bChannel0IndRxd == TRUE) &&
                         (psDecoder->eState == DECODER_STATE_INITIAL)
                       )
                    {
                        // Mark ready
                        BOOLEAN bSuccess =
                            DECODER_bRadioReady(psDecoder->hDecoder);
                        if (bSuccess == TRUE)
                        {
                            psDecoder->eState = DECODER_STATE_READY;
                        }
                    }
                }
                else if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_DELETE)
                {
                    CHANNEL_OBJECT hChannel;

                    hChannel =
                        CCACHE_hChannelFromIds(
                            psDecoder->hCCache,
                            (SERVICE_ID)psRxData->uData.sInd.uData.sChanInfo.sChannel.tSID,
                            (CHANNEL_ID)psRxData->uData.sInd.uData.sChanInfo.sChannel.tChanID,
                            FALSE
                                );

                    CCACHE_bRemoveChannel(psDecoder->hCCache, hChannel);
                }
            }
            break;

            case SXIAPI_MESSOP_CATINFO:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "CatInfoInd\n");

                if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_ADD_UPDATE)
                {
                    // Update this category in our local category list
                    vUpdateCategory(psDecoder,
                        psRxData->uData.sInd.uData.sCatInfo.tCatID,
                        &psRxData->uData.sInd.uData.sCatInfo.sCatNames.acShort[0],
                        &psRxData->uData.sInd.uData.sCatInfo.sCatNames.acMed[0],
                        &psRxData->uData.sInd.uData.sCatInfo.sCatNames.acLong[0],
                        FALSE);
                }
            }
            break;

            case SXIAPI_MESSOP_METADATA:
            case SXIAPI_MESSOP_LA_METADATA:
            {
                CHANNEL_ID tTunedChannelId;

                // Find this channel in the channel cache, if it exists
                // update the contents. If not, create it and update it
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "%sMetaDataInd\n",
                    (psRxData->sHdr.tOpType ==
                    SXIAPI_MESSOP_LA_METADATA) ? " LA" : " ");

                tTunedChannelId = DECODER.tCurrentChannelId(psDecoder->hDecoder);

                // only update the channel if not the tuned chan or we're live
                // if it is the tuned channel and we are time shifted we
                // don't want to update the channel with live metadata
                // since that isn't what's playing
                if ( (  tTunedChannelId != psRxData->uData.sInd.uData.sMetaData.sChannel.tChanID ) ||
                     (  psDecoder->sPlayback.eState == SXIAPI_PLAYBACK_STATE_INACTIVE ) )
                {
                    vUpdateChannel(psDecoder,
                    psRxData->uData.sInd.uData.sMetaData.sChannel.tChanID,
                        psRxData->uData.sInd.uData.sMetaData.sChannel.tSID,
                    NULL, &psRxData->uData.sInd.uData.sMetaData.sTrackInfo);
                }
            }
            break;

            case SXIAPI_MESSOP_GLOBAL_METADATA:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "GlobalMetaDataInd\n");

                // DECODER IND - Handle GMD...
                SXI_GMD_vDecoderUpdate(
                    psDecoder->hDecoder,
                    psDecoder->hControlCxn,
                    &psRxData->uData.sInd.uData.sGlobalMetaData);
            }
            break;

            case SXIAPI_MESSOP_CHANBROWSE:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "ChanBrowseInd\n");

                if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL)
                {
                    // Make sure we have an entry for this category in our list
                    vUpdateCategory(psDecoder,
                        psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tCatID,
                        psRxData->uData.sInd.uData.sChanBrowseSelect.sCatNames.acShort,
                        psRxData->uData.sInd.uData.sChanBrowseSelect.sCatNames.acMed,
                        psRxData->uData.sInd.uData.sChanBrowseSelect.sCatNames.acLong, FALSE);
                    // Find this channel in the channel cache, if it exists
                    // update the contents. If not, create it and update it
                    vUpdateChannel(psDecoder,
                        psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tChanID,
                        psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tSID,
                        &psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel,
                        &psRxData->uData.sInd.uData.sChanBrowseSelect.sTrackInfo);
                }
                else if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NO_CHAN)
                {
                    printf(RADIO_OBJECT_NAME": Service ID: %d is not available\n",
                        psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tSID);

                    if (psDecoder->bSelfTuneComplete == FALSE)
                    {
                        DECODER_vSelfTuneFallback(
                            psDecoder->hDecoder, 
                            psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tSID);
                    }
                }
            }
            break;

            case SXIAPI_MESSOP_CHANSELECT:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "ChanSelectInd\n");

                if ((psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL) ||
                    (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_SCAN_NOMINAL) ||
                    (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_TUNEMIX_NOMINAL) ||
                    (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_FLASH_EVENT_NOMINAL) ||
                    (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_BULLETIN_NOMINAL) ||
                    (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_SCAN_ABORTED))
                {
                    BOOLEAN bPaused = FALSE;
                    PLAYBACK_OBJECT hPlayback = DECODER.hPlayback(psDecoder->hDecoder);

                    if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_BULLETIN_NOMINAL)
                    {
                        DECODER_vUpdateTWNowStatus(
                            psDecoder->hDecoder,
                            (TW_NOW_BULLETIN_ID)
                                psRxData->uData.sInd.uData.sChanBrowseSelect.tSecondaryID,
                                FALSE);
                    }
                    else
                    {
                        // Here we've to handle transition from TWN bulletin to
                        // regular tune.
                        DECODER_vUpdateTWNowStatus(
                            psDecoder->hDecoder,
                            TW_NOW_INVALID_BULLETIN_ID,
                            TRUE);
                    }

                    if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_FLASH_EVENT_NOMINAL)
                    {
                        // Notify Sports Flash service about flash event playback start
                        SPORTS_FLASH_vUpdateFlashPlaybackStatus(
                            psDecoder->hDecoder,
                            (SPORTS_FLASH_EVENT_ID)
                                psRxData->uData.sInd.uData.sChanBrowseSelect.tSecondaryID);
                    }
                    else
                    {
                        // Notify Sports Flash service about flash event playback end
                        SPORTS_FLASH_vUpdateFlashPlaybackStatus(
                            psDecoder->hDecoder,
                            SPORTS_FLASH_INVALID_EVENT_ID);
                    }

                    if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_TUNEMIX_NOMINAL)
                    {
                        TUNEMIX_vUpdateTuneMixStatus(
                            psDecoder->hDecoder,
                            (CHANNEL_ID)psRxData->uData.sInd.uData.sChanBrowseSelect.tSecondaryID,
                            TRUE, 
                            TUNEMIX_STATUS_AVAILABLE);

                        // Cleanup song cache
                        vFlushSCache(psDecoder);
                    }
                    else
                    {
                        TUNEMIX_OBJECT hActive;
                        BOOLEAN bNeedToSave = TRUE;

                        // If we had active Tune Mix, we need to clear
                        // it now. This shall happen only once when first
                        // not TUNEMIX NOMINAL indication comes after
                        // TUNEMIX_NOMINAL
                        hActive = DECODER_hGetTuneMixActive(psDecoder->hDecoder);
                        if (hActive != TUNEMIX_INVALID_OBJECT)
                        {
                            // TuneMix is non-active now
                            TUNEMIX_vUpdateTuneMixStatus(psDecoder->hDecoder, 
                                CHANNEL_INVALID_ID,
                                TRUE, 
                                TUNEMIX_STATUS_AVAILABLE);
                        }

                        // Make sure we have an entry for this category in our list
                        vUpdateCategory(psDecoder,
                            psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tCatID,
                            psRxData->uData.sInd.uData.sChanBrowseSelect.sCatNames.acShort,
                            psRxData->uData.sInd.uData.sChanBrowseSelect.sCatNames.acMed,
                            psRxData->uData.sInd.uData.sChanBrowseSelect.sCatNames.acLong, FALSE);
                        // Find this channel in the channel cache, if it exists
                        // update the contents. If not, create it and update it
                        vUpdateChannel(psDecoder,
                            psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tChanID,
                            psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tSID,
                            &psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel,
                            &psRxData->uData.sInd.uData.sChanBrowseSelect.sTrackInfo);

                        if ((psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL) ||
                            (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_SCAN_ABORTED))
                        {
                            // Update nominal channel selection
                            vUpdateChannelSelection(psDecoder,
                                psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tSID, TRUE);
                        }

                        // Update tuned service id
                        if ((psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_FLASH_EVENT_NOMINAL) ||
                            (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_BULLETIN_NOMINAL))
                        {
                            // Do not save Flash or TWNow event tune to config file
                            bNeedToSave = FALSE;
                        }

                        DECODER_vUpdateTunedServiceId(psDecoder->hDecoder,
                            psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tSID,
                            bNeedToSave);
                    }

                    if (psDecoder->sPlayback.eState == SXIAPI_PLAYBACK_STATE_INACTIVE)
                    {
                        SCACHE_OBJECT hSCache;

                        // The newly tuned channel could be a
                        // BIR (Smart Favorites) channel and module might set playpoint
                        // in playback (beginning of newest song/track),
                        // or it might set playpoint to live (inactive playback state).

                        // Must set bIRPlaybackInfoValid to true since playback state
                        // is currently inactive and it is not known if any following
                        // IRPlaybackInfoInd messages will be sent (not sent iSf new
                        // channel playpoint is live, will be sent if new channel
                        // playpoint is in buffer).
                        psDecoder->sPlayback.bIRPlaybackInfoValid = TRUE;

                        // Updating current songID in SCache
                        hSCache = PLAYBACK_hSCache(hPlayback);
                        SCACHE_bChangeCurrentSongID(hSCache, SONG_LIVE_ID);
                    }
                    else
                    {
                        // set false since current playback state is active so
                        // we will wait to receive an IRPlaybackInfoInd message.
                        // We will receive at least one IRPlaybackInfoInd
                        // (one if new channel playpoint is live, multiple
                        // if new channel playpoint is in buffer).
                        // By waiting for new IR Record and Playback Info messages,
                        // we can avoid sending "transient" playback updates to
                        // the application where the info is based on
                        // parameters received from both these messages.
                        psDecoder->sPlayback.bIRPlaybackInfoValid = FALSE;
                    }

                    PLAYBACK.eIsPaused(hPlayback, &bPaused);
                    if (bPaused == TRUE)
                    {
                        // change status to play
                        PLAYBACK_vUpdatePause(hPlayback, FALSE);
                    }
                }
                else if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_BULLETIN_UNAVAIL)
                {
                    DECODER_vUpdateTWNowStatus(
                        psDecoder->hDecoder,
                        TW_NOW_INVALID_BULLETIN_ID,
                        FALSE);
                }
                else if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_FLASH_EVENT_UNAVAIL)
                {
                    // Notify Sports Flash service about flash event unavailability.
                    SPORTS_FLASH_vUpdateFlashPlaybackStatus(
                        psDecoder->hDecoder,
                        SPORTS_FLASH_INVALID_EVENT_ID);
                }
                else if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NO_CHAN)
                {
                    if ((psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tSID == 0) &&
                        (psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tChanID == 0))
                    {
                        // Seems like this indication comes from the failed Scan request
                        // (All fields are zero / null). We assume, that the Service / Channel ID 0
                        // (RADIO ID) is always present and cannot be removed from broadcast.
                        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                            "Scan Terminated: No Channels\n");

                        // Update non-nominal channel selection
                        vUpdateChannelSelection(psDecoder, SERVICE_INVALID_ID, FALSE);
                    }
                    else
                    {
                        CHANNEL_OBJECT hChannel;

                        hChannel = CCACHE_hChannelFromIds(
                                psDecoder->hCCache,
                               (SERVICE_ID)psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tSID,
                               (CHANNEL_ID)psRxData->uData.sInd.uData.sChanBrowseSelect.sChannel.tChanID,
                                FALSE);

                        CCACHE_bRemoveChannel(psDecoder->hCCache, hChannel);
                    }
                }
                else if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NO_TRACKS)
                {
                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                        "Scan Terminated: No Tracks\n");

                    // Update non-nominal channel selection
                    vUpdateChannelSelection(psDecoder, SERVICE_INVALID_ID, FALSE);
                }
            }
            break;

            case SXIAPI_MESSOP_CHAN_METADATA:
            {
                CHANNEL_OBJECT hChannel;
                BOOLEAN bBlocked;

                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "ChanMetaDataInd\n");

                hChannel = CCACHE_hChannelFromIds(
                    psDecoder->hCCache,
                    (SERVICE_ID)psRxData->uData.sInd.uData.sChanMetaData.tSID,
                    (CHANNEL_ID)psRxData->uData.sInd.uData.sChanMetaData.tChanID,
                    FALSE
                        );

                bBlocked = CHANNEL_bBlockNotifications(hChannel);
                if(bBlocked == TRUE)
                {
                    if ( (psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un32ValidFieldMask &
                          SXIAPI_VALID_CMI_SHORT_CHAN_DESC) == SXIAPI_VALID_CMI_SHORT_CHAN_DESC)
                    {
                        CHANNEL_bUpdateShortChanDesc(
                            hChannel,
                            &psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.acShortDesc[0]);
                    }

                    if ( (psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un32ValidFieldMask &
                          SXIAPI_VALID_CMI_LONG_CHAN_DESC) == SXIAPI_VALID_CMI_LONG_CHAN_DESC)
                    {
                        CHANNEL_bUpdateLongChanDesc(
                            hChannel,
                            &psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.acLongDesc[0]);
                    }

                    if ( (psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un32ValidFieldMask &
                          SXIAPI_VALID_CMI_RELATED_CHAN_LIST) == SXIAPI_VALID_CMI_RELATED_CHAN_LIST)
                    {
                        CHANNEL_bUpdateSimilarChannels(
                            hChannel,
                            psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un32NumChans,
                            (SERVICE_ID *)&psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.aun16RelatedChan[0]);
                    }

                    if ( (psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un32ValidFieldMask &
                          SXIAPI_VALID_CMI_PLAY_ON_SELECT) == SXIAPI_VALID_CMI_PLAY_ON_SELECT)
                    {
                        CHANNEL_PLAY_ON_SELECT_METHOD_ENUM ePlayOnSelect =
                            eMapPlayOnSelectMethod(
                                psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un8PlayOnSelect);

                        CHANNEL_bUpdatePlayOnSelectMethod( hChannel, ePlayOnSelect);
                    }

                    if ( (psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un32ValidFieldMask &
                          SXIAPI_VALID_CMI_CHAN_CONTENT_TYPE) == SXIAPI_VALID_CMI_CHAN_CONTENT_TYPE)
                    {
                        CONTENT_TYPE_ENUM eContentType =
                            eMapContentType(
                                psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un8ContentType);

                        CHANNEL_bUpdateContentType( hChannel, eContentType);
                    }

                    if ( (psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un32ValidFieldMask &
                          SXIAPI_VALID_CMI_IR_NAVIGATION_CLASS) == SXIAPI_VALID_CMI_IR_NAVIGATION_CLASS)
                    {
                        IR_NAVIGATION_CLASS_ENUM eIRNavClass =
                            eMapIRNavigationClass(
                                psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un8IRNavigationClass);

                        CHANNEL_bUpdateIRNavigationClass( hChannel, eIRNavClass);
                    }

                    if ( (psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un32ValidFieldMask &
                          SXIAPI_VALID_CMI_CHAN_LIST_ORDER) == SXIAPI_VALID_CMI_CHAN_LIST_ORDER)
                    {
                        CHANNEL_ACO tACO = 
                            tMapACO(
                                psRxData->uData.sInd.uData.sChanMetaData.sExtChanMetaData.un16ACO);

                        // Calling DECODER function because we don't only need to update
                        // CHANNEL object, but also start timer for ACO file save
                        DECODER_vUpdateACO(
                            psDecoder->hDecoder, hChannel, tACO);
                    }
                }
                // Let'em rip!
                CHANNEL_vReleaseNotifications(hChannel);
            }
            break;

            case SXIAPI_MESSOP_IRPLAYBACK_CONTROL:
            {
                PLAYBACK_STATS_STRUCT sStats;
                PLAYBACK_OBJECT hPlayback;
                SCACHE_OBJECT hSCache;
                UN16 un16TimeToStartOfBuffer;
                BOOLEAN bPaused = FALSE, bScanActive;

                // Extract the playback handle
                hPlayback = DECODER.hPlayback(psDecoder->hDecoder);
                hSCache = PLAYBACK_hSCache(hPlayback);

                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "IRPlaybackInfoInd\n");

                // Indicate info from this playback info message is available
                // now that it is being processed.
                psDecoder->sPlayback.bIRPlaybackInfoValid = TRUE;

                // Query scan status
                bScanActive = DECODER_bTuneScanActive(psDecoder->hDecoder);

                // 1. Let's handle the state information

                // state changed?
                if (psDecoder->sPlayback.eState !=
                    psRxData->uData.sInd.uData.sIRPlayback.eState)
                {
                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "Playback State Changed: %s -> %s\n",
                        pacPlaybackStateToString(psDecoder->sPlayback.eState),
                        pacPlaybackStateToString(psRxData->uData.sInd.uData.sIRPlayback.eState) );

                    PLAYBACK.eIsPaused(hPlayback, &bPaused);

                    // check if state changed to live
                    if (psRxData->uData.sInd.uData.sIRPlayback.eState ==
                        SXIAPI_PLAYBACK_STATE_INACTIVE)
                    {
                        // transitioning to LIVE from TIME-SHIFTED, we need to
                        // adjust our SCACHE offset
                        SCACHE_bChangeCurrentSongID(hSCache, SONG_LIVE_ID);

                        // There is no need in querying of the channel info from
                        // the Module, since in transition to 'live' module will
                        // send ChanSelectInd

                        if (bPaused == TRUE)
                        {
                            // change status to play
                            PLAYBACK_vUpdatePause(hPlayback, FALSE);
                        }
                    }
                    else
                    {
                        if ((psRxData->uData.sInd.uData.sIRPlayback.eState ==
                            SXIAPI_PLAYBACK_STATE_PLAYING) && (bPaused == TRUE))
                        {
                            PLAYBACK_vUpdatePause(hPlayback, FALSE);
                        }
                        else if ((psRxData->uData.sInd.uData.sIRPlayback.eState ==
                            SXIAPI_PLAYBACK_STATE_PAUSED) && (bPaused == FALSE))
                        {
                            PLAYBACK_vUpdatePause(hPlayback, TRUE);

                            // If we transition from <<LIVE>> to paused, need to fix
                            // current song ID
                            SCACHE_bChangeCurrentSongID(hSCache, 
                                psRxData->uData.sInd.uData.sIRPlayback.un16PlaybackId);
                        }
                    }
                    // update the state
                    psDecoder->sPlayback.eState =
                        psRxData->uData.sInd.uData.sIRPlayback.eState;
                }

                // get play point based on what state we're in
                if (psDecoder->sPlayback.eState != SXIAPI_PLAYBACK_STATE_INACTIVE)
                {
                    // WE'RE TIME SHIFTED, so update play pt with rx'd info
                    psDecoder->sPlayback.un8PlayPt =
                        psRxData->uData.sInd.uData.sIRPlayback.un8Position;
                }
                else
                {
                    // WE'RE LIVE so we didn't get a valid play pt.
                    // but by definition it must equal the buffer usage when live
                    psDecoder->sPlayback.un8PlayPt =
                        psDecoder->sPlayback.un8BufferUsage;
                }

                // change scache offset if not live and scan is idle
                if ((psDecoder->sPlayback.eState != SXIAPI_PLAYBACK_STATE_INACTIVE) &&
                    (bScanActive == FALSE))
                {
                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                        "Current Playback ID: %d, Received Playback ID: %d\n",
                        psDecoder->sPlayback.tCurrentPlaybackId,
                        psRxData->uData.sInd.uData.sIRPlayback.un16PlaybackId );

                    if (psDecoder->sPlayback.tCurrentPlaybackId !=
                        psRxData->uData.sInd.uData.sIRPlayback.un16PlaybackId)
                    {
                        BOOLEAN bInRange;

                        // Creates the song in cache if it was not there
                        (void)SCACHE_hGetSong(hSCache, 
                            (SONG_ID)psRxData->uData.sInd.uData.sIRPlayback.un16PlaybackId);

                        // When transition from 'Live' to 'Time-Shifted', the current
                        // song's offset shall be changed. At least, it shall be taken
                        // off from the 'Live' song.
                        // If we are jumping at the beginning of the current song (which
                        // is at 'Live' now), both Playback IDs will be the same,
                        // however we are not at the 'Live' anymore ...
                        // The same is for Pause and Tune Scan stop on the another channel.
                        bInRange = SCACHE_bChangeCurrentSongID(hSCache, 
                            (SONG_ID)psRxData->uData.sInd.uData.sIRPlayback.un16PlaybackId);

                        // Should not change anything if the offset is out of range.
                        // This usually happens when the scan is stopped on another channel,
                        // but IRPlaybackInfoInd may come somewhere in the middle of the
                        // SCache filling. So, we can get an offset which is currently 'out',
                        // but will be 'in' pretty soon, when all tracks will be populated
                        // thru IRRecordMetadataInd.
                        // Anyway, we will receive a bunch of the IRPlaybackInfoInd later.
                        if (bInRange == TRUE)
                        {
                            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                                "Offset is in range. Updating Current Playback ID: %d -> %d\n",
                                psDecoder->sPlayback.tCurrentPlaybackId,
                                psRxData->uData.sInd.uData.sIRPlayback.un16PlaybackId );

                            psDecoder->sPlayback.tCurrentPlaybackId =
                                psRxData->uData.sInd.uData.sIRPlayback.un16PlaybackId;

                            vUpdateSCacheState(psDecoder, hSCache);
                        }
                    }
                }

                // 2. Let's see if a playback warning or playback limit occurred

                // did our status change
                PLAYBACK.eIsPaused(hPlayback, &bPaused);

                if(psDecoder->sPlayback.eState != SXIAPI_PLAYBACK_STATE_INACTIVE)
                {
                    // Per SXI spec, when state is SXIAPI_PLAYBACK_STATE_INACTIVE,
                    // parameters of this IRPlaybackInd message are not valid,
                    // including un16TimeRemaining.
                    // Time offset changed
                    if ( (psDecoder->sPlayback.un16TimeOffset !=
                          psRxData->uData.sInd.uData.sIRPlayback.un16TimeRemaining) ||
                         (bPaused != psDecoder->sPlayback.bPaused) )
                    {
                        psDecoder->sPlayback.un16TimeOffset =
                            psRxData->uData.sInd.uData.sIRPlayback.un16TimeRemaining;

                        // the reported time remaining should never exceed the total duration of the buffer
                        // but at times it is reported as such.
                        if (psDecoder->sPlayback.un16TimeOffset >
                                        psDecoder->sPlayback.un16DurationOfBuffer)
                        {
                            psDecoder->sPlayback.un16TimeOffset =
                                        psDecoder->sPlayback.un16DurationOfBuffer;
                        }

                        un16TimeToStartOfBuffer =
                            psDecoder->sPlayback.un16DurationOfBuffer -
                            psDecoder->sPlayback.un16TimeOffset;

                        if (bPaused == TRUE)
                        {
                            // need to check limit / warning

                            if (un16TimeToStartOfBuffer == 0)
                            {
                                if (psDecoder->sPlayback.bLimitSent == FALSE)
                                {
                                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                                        "Playback Limit Occurred!\n");
                                    PLAYBACK_vLimitOccurred(hPlayback);
                                    psDecoder->sPlayback.bLimitSent = TRUE;
                                }
                            }
                            else if (un16TimeToStartOfBuffer <
                                     psDecoder->sPlayback.un16WarningLimit)
                            {
                                psDecoder->sPlayback.bLimitSent = FALSE;

                                if (psDecoder->sPlayback.bWarningSent == FALSE)
                                {
                                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                                        "Playback Warning Occurred!\n");
                                    PLAYBACK_vWarningOccurred(hPlayback);
                                    psDecoder->sPlayback.bWarningSent = TRUE;
                                }
                            }
                            else
                            {
                                // we're outside the window.
                                // reset the flags
                                psDecoder->sPlayback.bWarningSent = FALSE;
                                psDecoder->sPlayback.bLimitSent = FALSE;
                            }
                        }
                        else
                        {
                            // we're not paused.
                            // reset the flags
                            psDecoder->sPlayback.bWarningSent = FALSE;
                            psDecoder->sPlayback.bLimitSent = FALSE;
                        }
                    }
                }
                else
                {
                    // wer're live.

                    // timeoffset must be 0.
                    psDecoder->sPlayback.un16TimeOffset = 0;

                    // must be outside the window.
                    // reset the flags
                    psDecoder->sPlayback.bWarningSent = FALSE;
                    psDecoder->sPlayback.bLimitSent = FALSE;
                }

                // remember the latest state
                psDecoder->sPlayback.bPaused = bPaused;

                // The un16TimeBefore parameter is new in SXM2.0 SXI.
                // The un16TimeBefore + un16TimeRemaining can be used to
                // calculate an accurate buffer duration, available when playback
                // is active. This calculation is done in IRRecordMetadataInd
                // processing, where PLAYBACK_vUpdateTotalDuration() is called.
                // This is used (when playback active / not-live) instead of the
                // elaborate buffer duration tracking done in the IRRecordInfoInd.
                // This accumulate calculation in IRRecordInfoInd needs to be updated
                // to support operation with SXM2.0 Adv. IR when tuning to a BIR channel
                // that already has content (burst update of rec. metadata).
                psDecoder->sPlayback.un16TimeBefore =
                    psRxData->uData.sInd.uData.sIRPlayback.un16TimeBefore;

                // updating (confirming) buffer duration
                PLAYBACK_vUpdateTotalDuration(hPlayback,
                    psRxData->uData.sInd.uData.sIRPlayback.un16TimeBefore +
                    psRxData->uData.sInd.uData.sIRPlayback.un16TimeRemaining);

                // the reported time before playpoint should never exceed the total
                // duration of the buffer
                if (psDecoder->sPlayback.un16TimeBefore >
                                psDecoder->sPlayback.un16DurationOfBuffer)
                {
                    psDecoder->sPlayback.un16TimeBefore =
                                psDecoder->sPlayback.un16DurationOfBuffer;
                }

                // 3. Update the stats

                // package up the playback stats
                sStats.un8FillPercentage = psDecoder->sPlayback.un8BufferUsage;
                sStats.un8PlayPercentage = psDecoder->sPlayback.un8PlayPt;
                sStats.n32TimeOffset = -1 * (N32)psDecoder->sPlayback.un16TimeOffset;
                
                if (psDecoder->sPlayback.tCurrentPlaybackId >= psDecoder->sPlayback.tOldestPlaybackId)
                {
                    sStats.un16TracksBefore =
                        psDecoder->sPlayback.tCurrentPlaybackId - psDecoder->sPlayback.tOldestPlaybackId;
                }
                else
                {
                    sStats.un16TracksBefore =
                        UN16_MAX - psDecoder->sPlayback.tOldestPlaybackId + psDecoder->sPlayback.tCurrentPlaybackId;
                }
                psDecoder->sPlayback.un32TimeFromTrackStart =
                    psRxData->uData.sInd.uData.sIRPlayback.un16TimeFromStartOfTrack;
                sStats.un32TimeFromTrackStart =
                    psDecoder->sPlayback.un32TimeFromTrackStart;
                psDecoder->sPlayback.un32DurationOfTrack =
                    psRxData->uData.sInd.uData.sIRPlayback.un16DurationOfTrack;
                sStats.un32DurationOfTrack =
                    psDecoder->sPlayback.un32DurationOfTrack;

                psDecoder->sPlayback.un16TracksRemaining =
                    psRxData->uData.sInd.uData.sIRPlayback.un16TracksRemaining;
                sStats.un16TracksRemaining =
                    psDecoder->sPlayback.un16TracksRemaining;

                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                    "Tracks Before = %d, Tracks After = %d\n",
                    sStats.un16TracksBefore,
                    sStats.un16TracksRemaining);

                if(psDecoder->sPlayback.bIRRecordInfoValid &&
                   psDecoder->sPlayback.bIRPlaybackInfoValid)
                {
                    // playback info derived from valid info.
                    // update the playback stats
                    PLAYBACK_vUpdatePlaybackInfo(hPlayback, &sStats);
                }
            }
            break;

            case SXIAPI_MESSOP_IRPLAYBACK_METADATA:
            {
                BOOLEAN bScanActive;
                PLAYBACK_OBJECT hPlayback;
                SCACHE_OBJECT hSCache;
                SONG_OBJECT hSong;

                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "IRPlaybackMetadataInd\n" );

                // "IRPlaybackMetadataInd" on each track transition
                // as the TuneMix channel plays
                if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_TUNEMIX_NOMINAL)
                {

                    // Make sure we have an entry for this category in our list
                    vUpdateCategory(psDecoder,
                        psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sCatInfo.tCatID,
                        psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sCatInfo.sCatNames.acShort,
                        psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sCatInfo.sCatNames.acMed,
                        psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sCatInfo.sCatNames.acLong,
                        FALSE);
                    // Find this channel in the channel cache, if it exists
                    // update the contents. If not, create it and update it
                    vUpdateChannel(psDecoder,
                        psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.tChanID,
                        psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.tSID,
                        NULL,
                        &psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sTrackInfo);

                    // Update Last Tuned channel selection
                    vUpdateLastTunedChannelSelection(psDecoder,
                        psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.tSID);

                    if (psDecoder->eCurDispAdvisoryIndCode == SXIAPI_INDCODE_NOMINAL)
                    {
                        DECODER_vUpdateTunedServiceId(psDecoder->hDecoder,
                            psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.tSID,
                            FALSE);
                    }
                }

                // Extract objects handles
                hPlayback = DECODER.hPlayback(psDecoder->hDecoder);
                hSCache = PLAYBACK_hSCache(hPlayback);

                // Update current song
                hSong = SCACHE_hGetSong(hSCache, 
                    (SONG_ID)psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.un16PlaybackId);
                if (hSong != SONG_INVALID_OBJECT)
                {
                    UN8 un8Status;
                    SMSAPI_SONG_STATUS_ENUM eStatus;

                    // process the metadata ....
                    if (psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sTrackInfo.acArtistExtd[0] != '\0')
                    {
                        // we prefer extended
                        SONG_bUpdateArtist(hSong,
                            psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sTrackInfo.acArtistExtd);
                    }
                    else if (psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sTrackInfo.acArtistBasic[0] != '\0')
                    {
                        // but if basic is all we got...
                        SONG_bUpdateArtist(hSong,
                            psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sTrackInfo.acArtistBasic);
                    }

                    if (psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sTrackInfo.acSongExtd[0] != '\0')
                    {
                        // we prefer extended
                        SONG_bUpdateTitle(hSong,
                            psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sTrackInfo.acSongExtd);
                    }
                    else if (psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sTrackInfo.acSongBasic[0] != '\0')
                    {
                        // but if basic is all we got...
                        SONG_bUpdateTitle(hSong,
                            psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sTrackInfo.acSongBasic);
                    }

                    SONG_bUpdateChannelId(hSong,
                        (CHANNEL_ID)psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.tChanID);

                    // get the song status
                    un8Status =
                        psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.un8Status;

                    // interpret song status
                    if ((un8Status & (SXIAPI_IR_PLAYBACK_BEGINNING_STORED | SXIAPI_IR_PLAYBACK_END_STORED)) ==
                        (SXIAPI_IR_PLAYBACK_BEGINNING_STORED | SXIAPI_IR_PLAYBACK_END_STORED))
                    {
                        eStatus = SMSAPI_SONG_STATUS_COMPLETE;
                    }
                    else
                    {
                        eStatus = SMSAPI_SONG_STATUS_INCOMPLETE;
                    }

                    // update song status
                    SONG_bSetStatus(hSong, eStatus);
                }

                // Query Tune Scan status
                bScanActive = DECODER_bTuneScanActive(psDecoder->hDecoder);
                if (bScanActive == FALSE)
                {
                    // process the metadata ....
                    UN8 un8Status =
                        psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.un8Status;
                    SMSAPI_SONG_STATUS_ENUM eStatus = SMSAPI_SONG_STATUS_UNKNOWN;
                    SONG_OBJECT hCurrentSong;
                    CHANNEL_OBJECT hChannel;
                    BOOLEAN bBlocked;

                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                        "Current Playback ID: %d, Received Playback ID: %d\n",
                        psDecoder->sPlayback.tCurrentPlaybackId,
                        psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.un16PlaybackId );

                    if (psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.un16PlaybackId !=
                        psDecoder->sPlayback.tCurrentPlaybackId)
                    {
                        BOOLEAN bInRange;

                        bInRange = SCACHE_bChangeCurrentSongID(hSCache,
                            (SONG_ID)psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.un16PlaybackId);

                        if (bInRange == TRUE)
                        {
                            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                                "Offset is in range. Updating Current Playback ID: %d -> %d\n",
                                psDecoder->sPlayback.tCurrentPlaybackId,
                                psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.un16PlaybackId );

                            psDecoder->sPlayback.tCurrentPlaybackId =
                                psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.un16PlaybackId;

                            // Note, that this case is very rare. However, we must take
                            // it into account. Otherwise, it can slay scache.
                            vUpdateSCacheState(psDecoder, hSCache);
                        }
                    }

                    // get the song status
                    if ((un8Status & (SXIAPI_IR_PLAYBACK_BEGINNING_STORED | SXIAPI_IR_PLAYBACK_END_STORED)) ==
                        (SXIAPI_IR_PLAYBACK_BEGINNING_STORED | SXIAPI_IR_PLAYBACK_END_STORED))
                    {
                        eStatus = SMSAPI_SONG_STATUS_COMPLETE;
                    }
                    else
                    {
                        eStatus = SMSAPI_SONG_STATUS_INCOMPLETE;
                    }

                    hCurrentSong = SCACHE_hCurrentSong(hSCache);
                    SONG_bSetStatus(hCurrentSong, eStatus);

                    // If this indication came for a nominal channel while scan is in progress,
                    // there is no need to update track metadata once again, since for each
                    // scanned track module will send ChanSelectInd

                    // Who came up to put a channel id here?!
                    hChannel = CCACHE_hChannel(psDecoder->hCCache,
                        (CHANNEL_ID *)&psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.tChanID, NULL);

                    // Block future CHANNEL object notifications
                    bBlocked = CHANNEL_bBlockNotifications(hChannel);
                    if(bBlocked == TRUE)
                    {
                        vUpdateCDO(hChannel,
                            &psRxData->uData.sInd.uData.sIRPlaybackMetadataInfo.sTrackInfo);

                        // Let'em rip!
                        CHANNEL_vReleaseNotifications(hChannel);
                    }
                }
            }
            break;

            case SXIAPI_MESSOP_IRRECORD:
            {
                PLAYBACK_STATS_STRUCT sStats;
                PLAYBACK_OBJECT hPlayback;
                SCACHE_OBJECT hSCache;
                N16 n16DeltaDurationNewest = 0, n16DeltaDurationOldest = 0, n16DeltaDuration;
                BOOLEAN bUpdate = FALSE;
                UN32 un32OldestRelativePlaybackId, un32NewestRelativePlaybackId;
                UN32 un32CurrentRelativePlaybackId;
                N32 n32DeltaOldest;

                // Extract the playback handle
                hPlayback = DECODER.hPlayback(psDecoder->hDecoder);
                hSCache = PLAYBACK_hSCache(hPlayback);

                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "IRRecordInfoInd\n");

                // If oldest id = 0, newest id = 0 and both newest and oldest durations are 0
                // then we need some cleanup to be done.
                if ((psRxData->uData.sInd.uData.sIRRecordInfo.un16OldestPlaybackId == 0) &&
                    (psRxData->uData.sInd.uData.sIRRecordInfo.un16NewestPlaybackId == 0) &&
                    (psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfNewestTrack == 0) &&
                    (psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfOldestTrack == 0))
                {
                    // Resetting IR Record validity flag.
                    psDecoder->sPlayback.bIRRecordInfoValid = FALSE;

                    // Set true since playback state is being set to inactive
                    // and other playback params are set appropriately.
                    psDecoder->sPlayback.bIRPlaybackInfoValid = TRUE;

                    // Should set IR to 'inactive' when the module says it is 'inactive'.
                    // Do not change the state under nominal channel selection, since in
                    // some cases it could remain 'time-shifted'.
                    psDecoder->sPlayback.eState = SXIAPI_PLAYBACK_STATE_INACTIVE;

                    // Some logic in the SCache is based on a relative offsets. To prevent
                    // possible collisions / race conditions, all stats will be set to 0
                    // under the nominal channel selection.

                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "Module indicates IR buffer flush\n" );

                    // In case of subscription 'issues', if this indication came,
                    // FIR buffer will be flushed
                    if ((psDecoder->sSubStatus.eStatus == MODULE_SUBSTATUS_NOT_SUBSCRIBED) ||
                        (psDecoder->sSubStatus.eStatus == MODULE_SUBSTATUS_SUSPENDED))
                    {
                        vFlushSCache( psDecoder );
                    }
                    else
                    {
                        // change status to play
                        PLAYBACK_vUpdatePause(hPlayback, FALSE);
                    }

                    break;
                }

                // Indicate info from this message is available
                // now that it is being processed.
                psDecoder->sPlayback.bIRRecordInfoValid = TRUE;

                // Handle Playback ID wrap around cases.

                un32OldestRelativePlaybackId =
                    psRxData->uData.sInd.uData.sIRRecordInfo.un16OldestPlaybackId;

                // Delta between Newest and Oldest

                n32DeltaOldest =
                    (N32) psRxData->uData.sInd.uData.sIRRecordInfo.un16NewestPlaybackId -
                    (N32) psRxData->uData.sInd.uData.sIRRecordInfo.un16OldestPlaybackId;

                if(n32DeltaOldest < 0)
                {
                    if(n32DeltaOldest == -1)
                    {
                        // treat this as jitter, not wrap around.
                        n32DeltaOldest = 0;
                    }
                    else
                    {
                        // Wrap around case
                        n32DeltaOldest += UN16_MAX;
                    }
                }
                un32NewestRelativePlaybackId =
                    un32OldestRelativePlaybackId + n32DeltaOldest;

                // Handle SCache Ready state
                // If the flushed state is not handled yet,
                // that means that the Module doesn't sent
                // any songs and therefore, there is nothing
                // to do at this point.
                if ((psDecoder->sPlayback.bReady == FALSE) &&
                    (psDecoder->sPlayback.bFlushed == FALSE))
                {
                    N16 n16Songs;

                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "Newest / Oldest Delta: %d\n",
                        n32DeltaOldest);

                    n16Songs = SCACHE_n16NumberOfSongs(hSCache);

                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "SCache Songs Number: %d\n", n16Songs);

                    // Compare number of songs in SCache and Playback IDs delta.
                    // Please note, that there is a fake 'live' song and delta is 0
                    // while only 1 track is in IR buffer.
                    if (n16Songs == (N16)(n32DeltaOldest + 2))
                    {
                        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "SCache is Ready!\n");
                        psDecoder->sPlayback.bReady = TRUE;

                        // All tracks of the current IR buffer were received.
                        // However, notification sending depends on the current
                        // playback state.

                        // If playback is not active, the current song offset is always at 'live'.
                        // In this case we can just send notification right now.
                        // Note: per SX-9845-0008, "when a product supports Smart Favorites the
                        // Host must be able to handle the potential that IR playback may begin
                        // immediately after any channel selection."
                        if ((psDecoder->sPlayback.eState == SXIAPI_PLAYBACK_STATE_INACTIVE) ||

                            // If playback is active (play point is 'time-shifted'),
                            // we have to wait until the current song's offset will
                            // be 'in-range' (the current song received). Otherwise,
                            // the current song's offset will be still need to be updated.
                            // Note, that Tune Scan may jump on any track across all of IR
                            // buffers. In this case, to set the current song offset will
                            // be enough to receive only few first IRRecordMetadataInds.
                            // To prevent excessive events, even if the current song
                            // offset is up-to-date, we are going to wait until all tracks
                            // will be received.

                            (psDecoder->sPlayback.bOffsetReady == TRUE))
                        {
                            SCACHE_vReady(hSCache);
                            psDecoder->sPlayback.bReadySent = TRUE;

                            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                                "SCache Ready has been sent\n");
                        }
                    }
                }

                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                    "Oldest Playback ID = %d, Current Playback ID = %d, Newest Playback ID = %d\n",
                    psRxData->uData.sInd.uData.sIRRecordInfo.un16OldestPlaybackId,
                    psDecoder->sPlayback.tCurrentPlaybackId,
                    psRxData->uData.sInd.uData.sIRRecordInfo.un16NewestPlaybackId);

                // Delta between Current and Oldest
                n32DeltaOldest =
                    (N32) psDecoder->sPlayback.tCurrentPlaybackId -
                    (N32) psRxData->uData.sInd.uData.sIRRecordInfo.un16OldestPlaybackId;

                if(n32DeltaOldest < 0)
                {
                    if(n32DeltaOldest == -1)
                    {
                        // treat this as jitter, not wrap around.
                        n32DeltaOldest = 0;
                    }
                    else
                    {
                        // Wrap around case
                        n32DeltaOldest += UN16_MAX;
                    }
                }
                un32CurrentRelativePlaybackId =
                    un32OldestRelativePlaybackId + n32DeltaOldest;

                // Update Tracks Before and Tracks Remaining
                if(psDecoder->sPlayback.eState == SXIAPI_PLAYBACK_STATE_INACTIVE)
                {
                    // Live state
                    psDecoder->sPlayback.un16TracksBefore =
                        un32NewestRelativePlaybackId -
                        un32OldestRelativePlaybackId;

                    psDecoder->sPlayback.un16TracksRemaining = 0;
                }
                else if(un32CurrentRelativePlaybackId <=
                        un32NewestRelativePlaybackId)
                {
                    // Not Live State and Current Playback is in
                    // the expected range.

                    psDecoder->sPlayback.un16TracksBefore =
                        un32CurrentRelativePlaybackId -
                        un32OldestRelativePlaybackId;

                    psDecoder->sPlayback.un16TracksRemaining =
                        un32NewestRelativePlaybackId -
                        un32CurrentRelativePlaybackId;
                }
                // else out of expected range, update with existing values

                sStats.un16TracksBefore =
                     psDecoder->sPlayback.un16TracksBefore;

                sStats.un16TracksRemaining =
                     psDecoder->sPlayback.un16TracksRemaining;

                // did the oldest playback id change?
                if (psDecoder->sPlayback.tOldestPlaybackId !=
                    psRxData->uData.sInd.uData.sIRRecordInfo.un16OldestPlaybackId)
                {
                    // the oldest song in the buffer was removed, so find the corresponding song
                    // and remove it from the scache

                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                        "Remove oldest song: %u\n", psDecoder->sPlayback.tOldestPlaybackId);
                    SCACHE_vRemoveSongFromSCache(hSCache, psDecoder->sPlayback.tOldestPlaybackId);

                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                        "Updating Oldest Playback Id: %d -> %d\n",
                        psDecoder->sPlayback.tOldestPlaybackId,
                        psRxData->uData.sInd.uData.sIRRecordInfo.un16OldestPlaybackId );

                    psDecoder->sPlayback.tOldestPlaybackId =
                        psRxData->uData.sInd.uData.sIRRecordInfo.un16OldestPlaybackId;
                }

                if (psDecoder->sPlayback.eState == SXIAPI_PLAYBACK_STATE_INACTIVE)
                {
                    if (psDecoder->sPlayback.tCurrentPlaybackId !=
                        psRxData->uData.sInd.uData.sIRRecordInfo.un16NewestPlaybackId)
                    {
                        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                            "Updating Current Playback Id: %d -> %d\n",
                            psDecoder->sPlayback.tCurrentPlaybackId,
                            psRxData->uData.sInd.uData.sIRRecordInfo.un16NewestPlaybackId );

                        // we're live
                        psDecoder->sPlayback.tCurrentPlaybackId =
                            psRxData->uData.sInd.uData.sIRRecordInfo.un16NewestPlaybackId;
                    }
                }

                // update buffer usage
                psDecoder->sPlayback.un8BufferUsage =
                    psRxData->uData.sInd.uData.sIRRecordInfo.un8BufferUsage;
                sStats.un8FillPercentage = psDecoder->sPlayback.un8BufferUsage;
                if (psDecoder->sPlayback.eState != SXIAPI_PLAYBACK_STATE_INACTIVE)
                {
                    // if we aren't live, we're getting play pt updates from
                    // SXIAPI_MESSOP_IRPLAYBACK_CONTROL
                    sStats.un8PlayPercentage = psDecoder->sPlayback.un8PlayPt;

                }
                else
                {
                    // we're live, so play pt is equal to the buffer fill pct.
                    sStats.un8PlayPercentage = psDecoder->sPlayback.un8BufferUsage;
                    // since we're live, the time from the start is the same as the duration
                    psDecoder->sPlayback.un32TimeFromTrackStart =
                        psDecoder->sPlayback.un16DurationOfNewestTrack;

                    // since we're live, the duration of current track is the same as
                    // duration of the newest track.
                    psDecoder->sPlayback.un32DurationOfTrack =
                        psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfNewestTrack;

                    // TimeOffset must also be 0 since were live.
                    psDecoder->sPlayback.un16TimeOffset = 0;
                }

                // fill in the current timeoffset into the stats structure
                sStats.n32TimeOffset = -1 * (N32)psDecoder->sPlayback.un16TimeOffset;
                // fill in the current time from start of track into the stats structure
                sStats.un32TimeFromTrackStart =
                    psDecoder->sPlayback.un32TimeFromTrackStart;
                // fill in the current duration of track into the stats structure
                sStats.un32DurationOfTrack =
                    psDecoder->sPlayback.un32DurationOfTrack;

                if (psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfNewestTrack !=
                    psDecoder->sPlayback.un16DurationOfNewestTrack)

                {
                    OSAL_LINKED_LIST_ENTRY hEntry;
                    SONG_OBJECT hSong;

                    // duration of newest track changed.

                    if (psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfNewestTrack <
                        psDecoder->sPlayback.un16DurationOfNewestTrack)
                    {
                        // somehow we got an indication that the newset track decreased in size
                        // that should not happen, so we'll consider no change in that case
                        n16DeltaDurationNewest = 0;
                    }
                    else
                    {
                        // compute the change in the newest track's duration
                        n16DeltaDurationNewest =
                            psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfNewestTrack -
                            psDecoder->sPlayback.un16DurationOfNewestTrack;
                        // set flag to indicate we need to update the duration
                        bUpdate = TRUE;
                    }

                    // update our notion of the newset track's duration
                    psDecoder->sPlayback.un16DurationOfNewestTrack =
                        psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfNewestTrack;

                    // get the handle and update the corresponding song in the scache
                    hEntry = SCACHE_hSongLLEntry(hSCache,
                        psRxData->uData.sInd.uData.sIRRecordInfo.un16NewestPlaybackId);
                    hSong = SCACHE_hSongFromLLEntry(hEntry);
                    SONG_bSetDurationValue(hSong,
                        (UN32)psDecoder->sPlayback.un16DurationOfNewestTrack);
                }

                if (psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfOldestTrack !=
                    psDecoder->sPlayback.un16DurationOfOldestTrack)
                {
                    SONG_OBJECT hSong;

                    // duration of oldest track changed.
                    if (psDecoder->sPlayback.un16DurationOfOldestTrack > 0)
                    {
                        // compute the change in the newest track's duration
                        n16DeltaDurationOldest =
                            psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfOldestTrack -
                            psDecoder->sPlayback.un16DurationOfOldestTrack;
                    }
                    else
                    {
                        // If the oldest track's duration was 0
                        // then that means it is removed from the buffer and the next track
                        // becomes the oldest track it is in it sentirety, so it will look
                        // like a jump increase, but really the duration just decreased by 1
                        n16DeltaDurationOldest = -1;
                    }

                    // update our notion of the newset track's duration
                    psDecoder->sPlayback.un16DurationOfOldestTrack =
                        psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfOldestTrack;

                    // set flag to indicate we need to update the duration
                    bUpdate = TRUE;

                    // update the corresponding song in the scache
                    hSong = SCACHE_hSong(hSCache, 0);
                    SONG_bSetDurationValue(hSong,
                        (UN32)psDecoder->sPlayback.un16DurationOfOldestTrack);
                }

                // do we need to update the duration?
                if (bUpdate == TRUE)
                {
                    // yes

                    // the oldest track's duration can be:
                    // 1. increasing (we're still recording it
                    // 2. not changing (partially filled buffer, oldest track is no longer
                    //    being played live i.e we aren't still recording it)
                    // 3. decreasing (full buffer)
                    if (n16DeltaDurationOldest <= 0)
                    {
                        // the change in total duration is the sum of the change in duration of
                        // oldest and newest
                        n16DeltaDuration = n16DeltaDurationNewest + n16DeltaDurationOldest;
                        if (n16DeltaDuration <= 0)
                        {
                            // total duration can't decrease (unless it goes to zero when flushed)
                            // must have gotten the newset track length to be the same
                            n16DeltaDuration = 0;
                        }
                    }
                    else
                    {
                        // when buffer is basically empty, the newest and oldest are
                        // the same song and the oldest duration could be increasing
                        // we don't want to double count the increase.
                        n16DeltaDuration = n16DeltaDurationNewest;
                    }

                    // compute the new duration
                    psDecoder->sPlayback.un16Duration += n16DeltaDuration;
                    // the computed duration should never exceed the buffer duration
                    // due to timing of the messages and rounding inside the chipset it can
                    // happen that we'll be slightly off. so let's clip the duration to be safe
                    if (psDecoder->sPlayback.un16Duration >
                          psDecoder->sPlayback.un16DurationOfBuffer)
                    {
                        psDecoder->sPlayback.un16Duration =
                        psDecoder->sPlayback.un16DurationOfBuffer;
                    }
                }

                // alternate calculation, not based on accumulated duration method.
                {
                    UN32 un32AlternateDuration;

                    if (psDecoder->sPlayback.eState == SXIAPI_PLAYBACK_STATE_INACTIVE)
                    {
                        // Live state, so IRPlaybackInfoInd is not active and
                        // accurate duration params from it are not available.
                        // If Oldest and Newest tracks are the same track or adjacent
                        // tracks, use Oldest and Newest tracks to derive duration.
                        // Otherwise, use BufferUsage percentage to generate a course duration value

                        UN16 un16DeltaPlaybackIds;

                        un16DeltaPlaybackIds = psRxData->uData.sInd.uData.sIRRecordInfo.un16NewestPlaybackId -
                            psRxData->uData.sInd.uData.sIRRecordInfo.un16OldestPlaybackId;

                        if(un16DeltaPlaybackIds == 0)
                        {
                            // Oldest and Newest are the same track
                            un32AlternateDuration =
                                psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfNewestTrack;
                        }
                        else if(un16DeltaPlaybackIds == 1)
                        {
                            // Oldest and Newest are adjacent tracks
                            // Note: un16DeltaPlaybackIds = 1 result is also true for module wrap
                            // case of newest 0 - oldest 0xffff = 1 with unsigned arithmetic (UN16 vars).

                            un32AlternateDuration =
                                psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfNewestTrack +
                                psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfOldestTrack;
                        }
                        else
                        {
                            // There's at least one other track between oldest and newest,
                            // so can't use oldest and newest to derive the live total duration.
                            // Use the coarse duration derived from buffer full percentage instead.
                            un32AlternateDuration = ((UN32) psDecoder->sPlayback.un8BufferUsage *
                                                        psDecoder->sPlayback.un16DurationOfBuffer) / 100;

                            if(un32AlternateDuration <
                                (UN32) (psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfNewestTrack +
                                        psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfOldestTrack) )
                            {
                                // However, the course duration rounded value is less than Oldest + Newest,
                                // so use Oldest + Newest as duration instead.
                                un32AlternateDuration =
                                    psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfNewestTrack +
                                    psRxData->uData.sInd.uData.sIRRecordInfo.un16DurationOfOldestTrack;
                            }
                        }
                    }
                    else
                    {
                        // A non-live state, use accurate duration params from the
                        // active IRPlaybackInfoInd.
                        un32AlternateDuration = psDecoder->sPlayback.un16TimeBefore +
                                                psDecoder->sPlayback.un16TimeOffset;
                    }

                    if (un32AlternateDuration >
                          psDecoder->sPlayback.un16DurationOfBuffer)
                    {
                        un32AlternateDuration = psDecoder->sPlayback.un16DurationOfBuffer;
                    }

                    if(psDecoder->sPlayback.bIRRecordInfoValid &&
                       psDecoder->sPlayback.bIRPlaybackInfoValid)
                    {
                        // duration info derived from valid info.
                        // update the duration information
                        PLAYBACK_vUpdateTotalDuration(hPlayback, un32AlternateDuration);
                    }
                }

                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                    "Tracks Before = %d, Tracks After = %d\n",
                    sStats.un16TracksBefore,
                    sStats.un16TracksRemaining);

                if(psDecoder->sPlayback.bIRRecordInfoValid &&
                   psDecoder->sPlayback.bIRPlaybackInfoValid)
                {
                    // playback info derived from valid info.
                    // update the playback information
                    PLAYBACK_vUpdatePlaybackInfo(hPlayback, &sStats);
                }
            }
            break;

            case SXIAPI_MESSOP_IRRECORD_METADATA:
            {
                SONG_OBJECT hSong = SONG_INVALID_OBJECT;
                PLAYBACK_OBJECT hPlayback;
                SCACHE_OBJECT hSCache;
                UN8 un8Status;
                SMSAPI_SONG_STATUS_ENUM eStatus = SMSAPI_SONG_STATUS_UNKNOWN;

                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "IRRecordMetadataInd\n" );

                // Extract the playback handle
                hPlayback = DECODER.hPlayback(psDecoder->hDecoder);
                hSCache = PLAYBACK_hSCache(hPlayback);

                if (psDecoder->sPlayback.bFlushed == TRUE)
                {
                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "IR buffer just flushed\n" );

                    // Create the first song
                    hSong = SCACHE_hGetSong(hSCache,
                        (SONG_ID)psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.un16PlaybackId);

                    // Since we just flushed, the newest, the oldest and the current Playback IDs
                    // will be equal to the received Playback ID

                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "Updating Current Playback ID: %d -> %d\n",
                        psDecoder->sPlayback.tCurrentPlaybackId,
                        psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.un16PlaybackId );

                    // Update current Playback ID
                    psDecoder->sPlayback.tCurrentPlaybackId =
                        psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.un16PlaybackId;

                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "Updating Newest Playback ID: %d -> %d\n",
                        psDecoder->sPlayback.tNewestPlaybackId,
                        psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.un16PlaybackId );

                    // Update Newest Playback ID
                    psDecoder->sPlayback.tNewestPlaybackId =
                        psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.un16PlaybackId;

                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "Updating Oldest Playback ID: %d -> %d\n",
                        psDecoder->sPlayback.tOldestPlaybackId,
                        psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.un16PlaybackId );

                    // Update Oldest Playback ID
                    psDecoder->sPlayback.tOldestPlaybackId =
                        psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.un16PlaybackId;

                    // SCache Flush handled
                    psDecoder->sPlayback.bFlushed = FALSE;
                }
                else
                {
                    N16 n16Compare;
                    UN16 un16PlaybackId = psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.un16PlaybackId;

                    n16Compare = SCACHE_n16CompareSongOrder(
                        un16PlaybackId,
                        psDecoder->sPlayback.tOldestPlaybackId);

                    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                        "Comparing Received: %d / Oldest: %d Playback IDs: %d\n",
                        un16PlaybackId, psDecoder->sPlayback.tOldestPlaybackId, n16Compare );

                    if (n16Compare < 0)
                    {
                        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                            "Updating Oldest Playback Id: %d -> %d\n",
                            psDecoder->sPlayback.tOldestPlaybackId, un16PlaybackId );

                        psDecoder->sPlayback.tOldestPlaybackId = un16PlaybackId;
                    }
                    else
                    {
                        n16Compare = SCACHE_n16CompareSongOrder(
                            un16PlaybackId,
                            psDecoder->sPlayback.tNewestPlaybackId);

                        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                            "Comparing Received: %d / Newest: %d Playback IDs: %d\n",
                            un16PlaybackId, psDecoder->sPlayback.tNewestPlaybackId, n16Compare );

                        if (n16Compare > 0)
                        {
                            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                                "Updating Newest Playback Id: %d -> %d\n",
                                psDecoder->sPlayback.tNewestPlaybackId, un16PlaybackId );

                            psDecoder->sPlayback.tNewestPlaybackId = un16PlaybackId;

                            if (psDecoder->sPlayback.eState == SXIAPI_PLAYBACK_STATE_INACTIVE)
                            {
                                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "Updating Current Playback ID: %d -> %d\n",
                                    psDecoder->sPlayback.tCurrentPlaybackId,
                                    psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.un16PlaybackId );

                                // We must keep the Current Playback ID equal to the Newest Playback ID 
                                // when playback is inactive. It is necessary to keep both Current 
                                // Song Offset and Current Playback ID in sync, so when calculating 
                                // the delta offset or additional offset for Tune Start conditions 
                                // the Current Song Offset can be set properly.

                                // Update current Playback ID
                                psDecoder->sPlayback.tCurrentPlaybackId =
                                    psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.un16PlaybackId;
                            }
                        }
                    }

                    hSong = SCACHE_hGetSong(hSCache, (SONG_ID)un16PlaybackId);
                }

                if (hSong == SONG_INVALID_OBJECT)
                {
                    break;
                }

                // process the metadata ....
                if (psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.sTrackInfo.acArtistExtd[0] != '\0')
                {
                   // we prefer extended
                   SONG_bUpdateArtist(hSong,
                        psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.sTrackInfo.acArtistExtd);
                }
                else if (psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.sTrackInfo.acArtistBasic[0] != '\0')
                {
                    // but if basic is all we got...
                    SONG_bUpdateArtist(hSong,
                        psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.sTrackInfo.acArtistBasic);
                }

                if (psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.sTrackInfo.acSongExtd[0] != '\0')
                {
                    // we prefer extended
                    SONG_bUpdateTitle(hSong,
                        psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.sTrackInfo.acSongExtd);
                }
                else if (psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.sTrackInfo.acSongBasic[0] != '\0')
                {
                    // but if basic is all we got...
                    SONG_bUpdateTitle(hSong,
                        psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.sTrackInfo.acSongBasic);
                }

                SONG_bUpdateChannelId(hSong,
                    (CHANNEL_ID)psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.tChanID);

                // get the song status
                un8Status =
                    psRxData->uData.sInd.uData.sIRRecordMetadataInfo.sMetaData.un8Status;

                SONG_bSetDurationValue(hSong,
                    (UN32)psRxData->uData.sInd.uData.sIRRecordMetadataInfo.un16Duration);

                // interpret song status
                if ((un8Status & (SXIAPI_IR_PLAYBACK_BEGINNING_STORED | SXIAPI_IR_PLAYBACK_END_STORED)) ==
                    (SXIAPI_IR_PLAYBACK_BEGINNING_STORED | SXIAPI_IR_PLAYBACK_END_STORED))
                {
                    eStatus = SMSAPI_SONG_STATUS_COMPLETE;
                }
                else
                {
                    eStatus = SMSAPI_SONG_STATUS_INCOMPLETE;
                }

                // update song status
                SONG_bSetStatus(hSong, eStatus);
            }
            break;

            case SXIAPI_MESSOP_SEEK_MON:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "SeekInd\n" );

                if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL)
                {
                    vProcessSeekInd(psDecoder, FALSE, &psRxData->uData.sInd.uData.sSeekMon);
                }
                else if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_SEEK_END)
                {
                    vProcessSeekInd(psDecoder, TRUE, &psRxData->uData.sInd.uData.sSeekMon);
                }
            }
            break;

            case SXIAPI_MESSOP_FLASH:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "FlashInd\n" );

                if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL)
                {
                    vProcessFlashInd(psDecoder, &psRxData->uData.sInd.uData.sFlash);
                }
            }
            break;

            case SXIAPI_MESSOP_BULLETIN_MON:
            {
                SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "BulletinStatusInd\n");

                if (psRxData->uData.sInd.eIndCode == SXIAPI_INDCODE_NOMINAL)
                {
                    vProcessBulletinStatusInd(psDecoder,
                        &psRxData->uData.sInd.uData.sBulletinStatus);
                }
            }
            break;

            default:
            {
                // Do nothing
            }
            break;
        }
    }
    else if ((psDecoder->tCapabilities & SRH_DEVICE_CAPABILITY_DATA)
                    == SRH_DEVICE_CAPABILITY_DATA)
    {
        // Do Nothing
    }

    return;
}

/*****************************************************************************
 *
 *   vProcessSeekInd
 *
 *****************************************************************************/
static void vProcessSeekInd(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    BOOLEAN bEnd,
    SXIAPI_SEEKMON_IND_STRUCT *psInd
        )
{
    SEEK_MONITOR_TYPE_ENUM eSeekType;

    eSeekType = (SEEK_MONITOR_TYPE_ENUM)psInd->un8SeekMonID;

    switch (eSeekType)
    {
        case SEEK_MONITOR_TYPE_SPORTS_FLASH:
        {
            SPORTS_FLASH_EVENT_GAME_EVENT_STRUCT sEvent;

            if ((bEnd == FALSE) &&
                ((psInd->tChanAttrib & SXIAPI_CHANNELATTR_UNSUBSCRIBED) ==
                  SXIAPI_CHANNELATTR_UNSUBSCRIBED))
            {
                // Do not process unsubscribed channels
                break;
            }

            OSAL.bMemSet(&sEvent, 0, sizeof(sEvent));

            sEvent.tChannelID = psInd->tChanID;
            sEvent.un16Bias = 0;
            sEvent.bEnded = bEnd;

            if((psInd->sTrackInfo.sExtTrackMetaData.un32ValidFieldMask &
                    SXIAPI_VALID_TMI_SPORT_BCAST_TYPE) ==
                    SXIAPI_VALID_TMI_SPORT_BCAST_TYPE)
            {
                sEvent.un16Bias = psInd->sTrackInfo.sExtTrackMetaData.un8SportBCastType;
            }

            if((bEnd == TRUE) || 
               (psInd->sTrackInfo.sExtTrackMetaData.un32ValidFieldMask &
                    SXIAPI_VALID_TMI_TEAM_BCAST_ID) ==
                    SXIAPI_VALID_TMI_TEAM_BCAST_ID)
            {
                BOOLEAN bHandled;

                // Game End indication does not contain League / Team IDs
                if(bEnd == FALSE)
                {
                    sEvent.un8AwayLeagueID =
                        RADIO_EXTRACT_LEAGUE_ID(psInd->sTrackInfo.sExtTrackMetaData.aun32BCastTeamId[0]);
                    sEvent.un16AwayTeamID =
                        RADIO_EXTRACT_TEAM_ID(psInd->sTrackInfo.sExtTrackMetaData.aun32BCastTeamId[0]);
                    sEvent.un8HomeLeagueID =
                        RADIO_EXTRACT_LEAGUE_ID(psInd->sTrackInfo.sExtTrackMetaData.aun32BCastTeamId[1]);
                    sEvent.un16HomeTeamID =
                        RADIO_EXTRACT_TEAM_ID(psInd->sTrackInfo.sExtTrackMetaData.aun32BCastTeamId[1]);
                }

                // Post event
                bHandled = SPORTS_FLASH_bHandleGameEvent(psDecoder->hDecoder, &sEvent);
                if (bHandled == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        RADIO_OBJECT_NAME": Unable to handle Sports Flash Game Event.");
                }
            }
        }
        break;

        default:
            // Unknown monitor, do nothing
        break;
    }

    return;
}

/*****************************************************************************
 *
 *   vProcessFlashInd
 *
 *****************************************************************************/
static void vProcessFlashInd(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_FLASH_IND_STRUCT *psInd
        )
{
    if (psInd == NULL)
    {
        return;
    }

    switch (psInd->eFlashType)
    {
        case SXIAPI_FLASH_TYPE_SPORTS_FLASH:
        {
            CHANNEL_OBJECT hChannel;
            SPORTS_FLASH_EVENT_FLASH_EVENT_STRUCT sEvent;
            BOOLEAN bHandled;
            UN32 un32UMask;

            if ((psInd->tChanAttrib & SXIAPI_CHANNELATTR_UNSUBSCRIBED) ==
                SXIAPI_CHANNELATTR_UNSUBSCRIBED)
            {
                // Do not process unsubscribed channels
                break;
            }

            // Check Urgency mask field
            un32UMask = psInd->un32FlashEventData & SXIAPI_FLASH_EVENT_URGENCY_MASK;

            if (
                (un32UMask != SXIAPI_SPORTS_FLASH_SUPPORTED_URGENCY) &&
                ((un32UMask != SXIAPI_SPORTS_FLASH_EXPIRY_URGENCY) || 
                 (psInd->eFlashStatus != SXIAPI_FLASH_STATUS_EXPIRED))
               )
            {
                // Do not process events with unsupported urgency
                break;
            }

            // Find the channel object in the channel cache. Create if necessary.
            hChannel = CCACHE_hChannelFromIds(
                psDecoder->hCCache,
                (SERVICE_ID)psInd->tSID,
                (CHANNEL_ID)psInd->tChanID,
                TRUE);
            if (hChannel == CHANNEL_INVALID_OBJECT)
            {
                // Error!
                break;
            }

            sEvent.tChannelID = CHANNEL.tChannelId(hChannel);
            sEvent.eEventStatus =
                (SPORTS_FLASH_EVENT_STATUS_ENUM)psInd->eFlashStatus;
            sEvent.tFlashEventID =
                (SPORTS_FLASH_EVENT_ID)psInd->tFlashEventID;
            sEvent.tProgramID =
                (PROGRAM_ID)psInd->tProgID;

            // Post event only if we have valid channel id
            bHandled = SPORTS_FLASH_bHandleFlashEvent(psDecoder->hDecoder, &sEvent);
            if (bHandled == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Unable to handle Sports Flash Flash Event.");
            }
        }
        break;

        default:
        {
            // Do nothing
        }
        break;
    }

    return;
}

/*****************************************************************************
 *
 *   vProcessBulletinStatusInd
 *
 *****************************************************************************/
static void vProcessBulletinStatusInd(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_BULLETIN_STATUS_IND_STRUCT *psInd
        )
{
    switch (psInd->eType)
    {
        case SXIAPI_BULLETIN_TYPE_TW_NOW:
        {
            BOOLEAN bHandled;

            // Call TW Now event handler
            bHandled = TW_NOW_bHandleBulletinEvent(
                psDecoder->hDecoder,
                (TW_NOW_BULLETIN_ID)psInd->tBulletinID,
                (TW_NOW_BULLETIN_STATUS_ENUM)psInd->eStatus,
                (TRAFFIC_MARKET)psInd->un16Param1
                    );
            if (bHandled == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Unable to handle TW Now Bulletin Event.");
            }
        }
        break;

        default:
        {
            // Do nothing
        }
        break;
    }

    return;
}

/*****************************************************************************
 *
 *   vUpdateCategory
 *
 *****************************************************************************/
static void vUpdateCategory (
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_CATEGORY_ID tCatID,
    const char *psCatNameShort,
    const char *psCatNameMedium,
    const char *psCatNameLong,
    BOOLEAN bRemove)
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SXIAPI_CATINFOIND_STRUCT sCatInfoToFind;
    SXIAPI_CATINFOIND_STRUCT *psCurCatInfo;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    BOOLEAN bAdd = FALSE, bRename = FALSE;

    if (tCatID == CATEGORY_ID_BROADCAST_NOT_ASSIGNED ||
        tCatID == SXIAPI_CATEGORYID_ALLCATEGORIES)
    {
        return;
    }

    // Update this categories status in our vector
    vUpdateCategoryVector(psDecoder, tCatID, (BOOLEAN)!bRemove);

    do
    {
        // Look up category in linked list
        sCatInfoToFind.tCatID = tCatID;
        eReturnCode = OSAL.eLinkedListSearch(
                psDecoder->sCategory.hCatList,
                &hEntry, (void *)&sCatInfoToFind);

        if (eReturnCode == OSAL_SUCCESS)
        {
            // We already know about this cat id,
            // so we can just update our info
            psCurCatInfo = (SXIAPI_CATINFOIND_STRUCT *)OSAL.pvLinkedListThis(hEntry);

            if (psCatNameLong != NULL)
            {
                // If it changed, update the CCACHE
                if (strcmp(psCatNameLong, psCurCatInfo->sCatNames.acLong) != 0)
                {
                    bRename = TRUE;
                }
            }
            if (psCatNameMedium != NULL)
            {
                // If it changed, update the CCACHE
                if (strcmp(psCatNameMedium, psCurCatInfo->sCatNames.acMed) != 0)
                {
                    bRename = TRUE;
                }
            }
            if (psCatNameShort != NULL)
            {
                // If it changed, update the CCACHE
                if (strcmp(psCatNameShort, psCurCatInfo->sCatNames.acShort) != 0)
                {
                    bRename = TRUE;
                }
            }
        }
        else if (eReturnCode == OSAL_OBJECT_NOT_FOUND && bRemove == FALSE)
        {
            char acName[OSAL_MAX_OBJECT_NAME_LENGTH];

            // Add a new entry for this category
            sprintf(&acName[0],
                RADIO_OBJECT_NAME":Decoder:Cat:%u", tCatID);
            psCurCatInfo = (SXIAPI_CATINFOIND_STRUCT *)SMSO_hCreate(
                &acName[0], sizeof(SXIAPI_CATINFOIND_STRUCT),
                (SMS_OBJECT)psDecoder, // DECODER is parent,
                FALSE                  // inherit lock-feature
                );

            if (psCurCatInfo == NULL)
            {
                break;
            }

            psCurCatInfo->tCatID = tCatID;
            psCurCatInfo->sCatNames.acLong[0] = '\0';
            psCurCatInfo->sCatNames.acMed[0] = '\0';
            psCurCatInfo->sCatNames.acShort[0] = '\0';
            bAdd = TRUE;
        }
        else
        {
            // ERROR
            break;
        }

        if (bRemove == TRUE)
        {
            // Remove the category from our list
            eReturnCode = OSAL.eLinkedListRemove(hEntry);

            if (eReturnCode == OSAL_SUCCESS)
            {
                // Remove the category from the CCACHE
                CCACHE_vRemoveCategory(psDecoder->hCCache, tCatID);
            }

        }
        else
        {
            if (psCatNameLong != NULL)
            {
                strncpy(&psCurCatInfo->sCatNames.acLong[0],
                    psCatNameLong, sizeof(psCurCatInfo->sCatNames.acLong) - 1);
                psCurCatInfo->sCatNames.acLong[sizeof(psCurCatInfo->sCatNames.acLong)-1] = '\0';
            }

            if (psCatNameMedium != NULL)
            {
                strncpy(&psCurCatInfo->sCatNames.acMed[0],
                    psCatNameMedium, sizeof(psCurCatInfo->sCatNames.acMed) - 1);
                psCurCatInfo->sCatNames.acMed[sizeof(psCurCatInfo->sCatNames.acMed)-1] = '\0';
            }

            if (psCatNameShort != NULL)
            {
                strncpy(&psCurCatInfo->sCatNames.acShort[0],
                    psCatNameShort, sizeof(psCurCatInfo->sCatNames.acShort) - 1);

                psCurCatInfo->sCatNames.acShort[sizeof(psCurCatInfo->sCatNames.acShort)-1] = '\0';
            }

            if (bAdd == TRUE)
            {
                // Add category to our linked list
                eReturnCode = OSAL.eLinkedListAdd(
                        psDecoder->sCategory.hCatList,
                        &hEntry, psCurCatInfo);

                if (eReturnCode != OSAL_SUCCESS)
                {
                    // ERROR
                    SMSO_vDestroy((SMS_OBJECT)psCurCatInfo);
                    break;
                }
            }
            else if (bRename == TRUE)
            {
                CATEGORY_eRename(psDecoder->hDecoder, tCatID, 
                    psCatNameLong, psCatNameMedium, psCatNameShort);

            }
        }

    } while (FALSE);

    return;
}

/*****************************************************************************
 *
 *   vUpdateChannel
 *
 *****************************************************************************/
static void vUpdateChannel(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    CHANNEL_ID tChannelId,
    SERVICE_ID tServiceId,
    const SXIAPI_CHANNEL_STRUCT *psChannel,
    const SXIAPI_TRACK_INFO_STRUCT *psTrack)
{
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;

    // Check inputs
    if((tChannelId == CHANNEL_INVALID_ID) ||
        (tServiceId == SERVICE_INVALID_ID))
    {
        // Sorry, we cannot update any channel without a notion
        // of a channel id and service id.
        return;
    }

    // Find this channel in the channel cache. We should use a 
    // service-id to find a channel since some channels might be
    // restored from sms configuration based on their service ids
    // (e.g. locked / skipped channels).

    // We are not going to create a channel immediately if that is 
    // not in the cache, as it leads to uncontrolled notification.
    hChannel = CCACHE_hChannelFromIds(
        psDecoder->hCCache, tServiceId, CHANNEL_INVALID_ID, FALSE);

    // If channel does not exist, create it.
    if (hChannel == CHANNEL_INVALID_OBJECT)
    {
        hChannel = CCACHE_hCreateChannel(
            psDecoder->hCCache, tServiceId, CHANNEL_INVALID_ID, FALSE);
    }
    
    if (CHANNEL_INVALID_OBJECT != hChannel)
    {
        BOOLEAN bBlocked;

        // Block future CHANNEL object notifications
        bBlocked = CHANNEL_bBlockNotifications(hChannel);
        if(bBlocked == TRUE)
        {
            // Update this channel info with the provided channel & track info
            if (psChannel != NULL)
            {
                vUpdateChannelInfo(hChannel, psChannel);
                vUpdateCategoryVector(psDecoder, psChannel->tCatID, TRUE);
            }

            if (psTrack != NULL)
            {
                vUpdateCDO(hChannel, psTrack);
            }

            // Let'em rip!
            CHANNEL_vReleaseNotifications(hChannel);
        }
    }

    return;
}

/*****************************************************************************
 *
 *   vUpdateCategoryVector
 *
 *****************************************************************************/
static void vUpdateCategoryVector(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_CATEGORY_ID tCatID,
    BOOLEAN bPresent
        )
{
    N16 n16CurrentIndex =
        (SXIAPI_CAT_ID_VECTOR_SIZE - (tCatID / 8) - 1);
    N16 n16CurrentBit = (tCatID % 8);
    BOOLEAN bCategorySet = FALSE;

    // Check input parameters
    if ((tCatID == CATEGORY_ID_BROADCAST_NOT_ASSIGNED) ||
        (tCatID > CATEGORY_ID_BROADCAST_MAX))
    {
        return;
    }

    if(psDecoder->sCategory.aun8CatIDVector[n16CurrentIndex]
       & ((0x01 << n16CurrentBit) & 0xFF))
    {
        // Category is currently set
        bCategorySet = TRUE;
    }

    if (bPresent == TRUE)
    {
        // Available
        if (bCategorySet == FALSE)
        {
            psDecoder->sCategory.n16NumCategories++;
            psDecoder->sCategory.aun8CatIDVector[n16CurrentIndex] |=
                (0x01 << n16CurrentBit);
        }
    }
    else
    {
        // Not there
        if (bCategorySet == TRUE)
        {
            psDecoder->sCategory.n16NumCategories--;
            psDecoder->sCategory.aun8CatIDVector[n16CurrentIndex] &=
                ~(0x01 << n16CurrentBit);
        }
    }

    return;
}


/*****************************************************************************
 *
 *   vUpdateChannelInfo
 *
 *****************************************************************************/
static void vUpdateChannelInfo(
    CHANNEL_OBJECT hChannel, const SXIAPI_CHANNEL_STRUCT *psChannel)
{
    CATEGORY_ID tCategoryId;
    CHANNEL_ID tChannelId;

    do
    {
        // Verify inputs
        if((hChannel == CHANNEL_INVALID_OBJECT) || (psChannel == NULL))
        {
            // Error!
            break;
        }

        // Populate with new channel information...

        // Extract channel id
        tChannelId = ((psChannel->tChanID ==
            SXIAPI_CHANNELID_SERVICENOTASSIGNED)
            ? CHANNEL_INVALID_ID : (CHANNEL_ID)psChannel->tChanID);

        // Update channel id for this service id
        CHANNEL_bUpdateChannelId(hChannel, tChannelId);

        // Update channel type
        CHANNEL_bUpdateType(hChannel, CHANNEL_TYPE_AUDIO);

        // Update channel names
        CHANNEL_bUpdateShortName(hChannel, &psChannel->sChanNames.acShort[0]);
        CHANNEL_bUpdateMediumName(hChannel, &psChannel->sChanNames.acMed[0]);
        CHANNEL_bUpdateLongName(hChannel, &psChannel->sChanNames.acLong[0]);


        // Update subsription state
        if ((psChannel->tChanAttrib & SXIAPI_CHANNELATTR_UNSUBSCRIBED)
                        != SXIAPI_CHANNELATTR_UNSUBSCRIBED)
        {
            CHANNEL_eSetAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_SUBSCRIBED);
        }
        else
        {
            CHANNEL_eClearAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_SUBSCRIBED);
        }

        // Update mature state
        if ((psChannel->tChanAttrib & SXIAPI_CHANNELATTR_MATURE)
                        != SXIAPI_CHANNELATTR_MATURE)
        {
            CHANNEL_eClearAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_MATURE);
        }
        else
        {
            CHANNEL_eSetAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_MATURE);
        }

        // Update locked state
        if ((psChannel->tChanAttrib & SXIAPI_CHANNELATTR_LOCKED)
                        != SXIAPI_CHANNELATTR_LOCKED)
        {
            CHANNEL_eClearAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_LOCKED);
        }
        else
        {
            CHANNEL_eSetAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_LOCKED);
        }

        // Update skipped state
        if ((psChannel->tChanAttrib & SXIAPI_CHANNELATTR_SKIPPED)
                        != SXIAPI_CHANNELATTR_SKIPPED)
        {
            CHANNEL_eClearAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_SKIPPED);
        }
        else
        {
            CHANNEL_eSetAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_SKIPPED);
        }

        // Update free-to-air state
        if ((psChannel->tChanAttrib & SXIAPI_CHANNELATTR_FREE_TO_AIR)
                        != SXIAPI_CHANNELATTR_FREE_TO_AIR)
        {
            CHANNEL_eClearAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_FREE_TO_AIR);
        }
        else
        {
            CHANNEL_eSetAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_FREE_TO_AIR);
        }

        // Update category information...
        tCategoryId  =
            ((psChannel->tCatID  == SXIAPI_CATEGORYID_SERVICENOTASSIGNED)
              ? CATEGORY_INVALID_ID : (CATEGORY_ID)psChannel->tCatID);
        CHANNEL_bUpdateCategoryId(hChannel, tCategoryId);

    }
    while(FALSE);

    return;
}

/*****************************************************************************
 *
 *   eProcessSportsBroadcastType
 *
 *****************************************************************************/
static SPORTS_BROADCAST_ENUM eProcessSportsBroadcastType(
    UN32 un32ValidFieldMask, UN8 un8SportBCastType, UN8 *pun8Index)
{
    SPORTS_BROADCAST_ENUM eBCastType = SPORTS_BROADCAST_UNKNOWN;

    if (pun8Index != NULL)
    {
        if ( (un32ValidFieldMask & SXIAPI_VALID_TMI_SPORT_BCAST_TYPE) !=
            SXIAPI_VALID_TMI_NONE)
        {
            switch(un8SportBCastType)
            {
                case SXIAPI_SPORT_BCAST_NATIONAL:
                {
                    eBCastType = SPORTS_BROADCAST_NATIONAL;
                }
                break;

                case SXIAPI_SPORT_BCAST_TEAM_2:
                {
                    eBCastType = SPORTS_BROADCAST_TEAM;
                    // SMS APIs are base 0, so SXI's team 2 in SMS team 1
                    *pun8Index = 1;
                }
                break;

                case SXIAPI_SPORT_BCAST_TEAM_1:
                {
                    eBCastType = SPORTS_BROADCAST_TEAM;
                    // SMS APIs are base 0, so SXI's team 1 in SMS team 0
                    *pun8Index = 0;
                }
                break;

                case SXIAPI_SPORT_BCAST_OTHER:
                {
                    eBCastType = SPORTS_BROADCAST_OTHER;
                }
                break;

                case SXIAPI_SPORT_BCAST_INVALID:
                default:
                {
                    eBCastType = SPORTS_BROADCAST_UNKNOWN;
                }
                break;
            }
        }
    }

    return eBCastType;

}

/*****************************************************************************
 *
 *   vUpdateCDO
 *
 * Create/Update a CDO which belongs to the channel object provided
 * using SXI information. A CDO only processes "common" information such
 * as Artist, Title, Composer, etc in a Text form. Other information
 * such as the Song Id and Artist Id are processed by the CDO and so that
 * information is passed to the CDO after textual information has been
 * processed here.
 *
 * hChannel - A channel object for which to associate this CDO object with.
 * This is only required if this is a new object to be created. Otherwise
 * providing CHANNEL_INVALID_OBJECT maybe provided.
 *
 * psTrack - A pointer to some Track information to either create or populate this
 * CDO with. This is the Track information which will be parsed.
 *
 *****************************************************************************/
static void vUpdateCDO(
    CHANNEL_OBJECT hChannel,
    const SXIAPI_TRACK_INFO_STRUCT *psTrack)
{
    CD_OBJECT hCDO;
    void *pvData = NULL;
    BOOLEAN bUpdated = FALSE;
    CDO_TYPE_ENUM eType;

    // Check input. At the very least valid track info must be provided
    if(psTrack == NULL)
    {
        // Error!
        return;
    }

    // Retrieve CDO handle from object handle using a helper function.
    hCDO = CHANNEL_hCDO(hChannel, TRUE);
    if(hCDO == CD_INVALID_OBJECT)
    {
        // Error!
        return;
    }

    // First, save off the program ID of the content (used during album
    // art lookups)
    CDO_vAssignPID( hCDO, (PROGRAM_ID)psTrack->tProgID );

    eType = CDO.eType(hCDO);

    // look at the extended track metatdata rx'd to determine what kind of CDO
    // we should use

    if ( (psTrack->sExtTrackMetaData.un32ValidFieldMask &
          SXIAPI_VALID_TMI_SPORTS) != SXIAPI_VALID_TMI_NONE)
    {
        SPORTS_PARSE_STRUCT sSportsParse;
        CDO_TYPE_ENUM eOrigType;

        eOrigType = eType;

        if (eOrigType != CDO_SPORTS)
        {
            // this is SPORTS content
            CDO_bAssignType(hCDO, CDO_SPORTS);
            bUpdated = TRUE;
        }

        sSportsParse.un8LeagueId =
          psTrack->sExtTrackMetaData.un8BCastLeagueId;
        sSportsParse.un16Team1 = (UN16)(psTrack->sExtTrackMetaData.aun32BCastTeamId[0] & 0xFFFF);
        sSportsParse.un16Team2 = (UN16)(psTrack->sExtTrackMetaData.aun32BCastTeamId[1] & 0xFFFF);
        sSportsParse.eBCastType = eProcessSportsBroadcastType(
            psTrack->sExtTrackMetaData.un32ValidFieldMask,
            psTrack->sExtTrackMetaData.un8SportBCastType,
            &sSportsParse.un8BCastTeamIndex);
        sSportsParse.bFound = FALSE;

        pvData = (void *)&sSportsParse;

        bUpdated |= GsSxiSportsId.bParse(hCDO, pvData);

        if (sSportsParse.bFound == FALSE)
        {
            // this is UNKNOWN content
            CDO_bAssignType(hCDO, CDO_UNKNOWN);
            if (eOrigType != CDO_UNKNOWN)
            {
                bUpdated = TRUE;
            }
        }
    }
    else if ( (psTrack->sExtTrackMetaData.un32ValidFieldMask &
               SXIAPI_VALID_TMI_TRAFFIC_ID) != SXIAPI_VALID_TMI_NONE)
    {
        MARKET_PARSE_STRUCT sMarketParseStruct;
        CDO_TYPE_ENUM eOrigType;

        eOrigType = eType;

        if (eOrigType != CDO_REPORT)
        {
            // this is REPORT content
            CDO_bAssignType(hCDO, CDO_REPORT);
            bUpdated = TRUE;
        }

        sMarketParseStruct.un32MarketId = psTrack->sExtTrackMetaData.un32TrafficId;
        sMarketParseStruct.bMktFound = FALSE;

        // since we have a traffic id, update the report id
        pvData = (void *)&sMarketParseStruct;

        bUpdated |= GsSxiMarketId.bParse(hCDO, pvData);

        if (sMarketParseStruct.bMktFound == FALSE)
        {
            // this is UNKNOWN content
            CDO_bAssignType(hCDO, CDO_UNKNOWN);
            if (eOrigType != CDO_UNKNOWN)
            {
                bUpdated = TRUE;
            }
        }
    }
    else if ( (psTrack->sExtTrackMetaData.un32ValidFieldMask &
               SXIAPI_VALID_TMI_MUSIC) != SXIAPI_VALID_TMI_NONE)
    {
        SONG_TAG_ID_PARSE_STRUCT sSongTagParseStruct;

        if (eType != CDO_MUSIC)
        {
            // this is MUSIC content
            CDO_bAssignType(hCDO, CDO_MUSIC);
            bUpdated = TRUE;
        }

        // is there a song id?
        if ( (psTrack->sExtTrackMetaData.un32ValidFieldMask &
              SXIAPI_VALID_TMI_SONG_ID) != SXIAPI_VALID_TMI_NONE)
        {
            // get the song id
            pvData = (void *)&psTrack->sExtTrackMetaData.un32SongId;
        }
        // parse the song id into CID
        bUpdated |= GsSxiSongId.bParse(hCDO, pvData);

        pvData = NULL;
        // is there an artist id?
        if ( (psTrack->sExtTrackMetaData.un32ValidFieldMask &
              SXIAPI_VALID_TMI_ARTIST_ID) != SXIAPI_VALID_TMI_NONE)
        {
            // get the artist id
            pvData = (void *)&psTrack->sExtTrackMetaData.un32ArtistId;
        }
        // parser the artist id into an CID
         bUpdated |= GsSxiArtistId.bParse(hCDO, pvData);

        // iTunes service
        sSongTagParseStruct.eService = SONG_TAG_SERVICE_ITUNES;
        sSongTagParseStruct.pvData = NULL;

        // is there a itunes song tag?
        if ( (psTrack->sExtTrackMetaData.un32ValidFieldMask &
              SXIAPI_VALID_TMI_ITUNES_SONG_ID) != SXIAPI_VALID_TMI_NONE)
        {
            // yes.
            sSongTagParseStruct.pvData =
                (void *)&psTrack->sExtTrackMetaData.un32ITunesSongId;
        }

        bUpdated |= GsSongTagIntf.bParse(hCDO, &sSongTagParseStruct);
    }
    else
    {
        // we can't determine what type of content this is
        if (eType != CDO_UNKNOWN)
        {
            // this is UNKNOWN content
            CDO_bAssignType(hCDO, CDO_UNKNOWN);
                bUpdated = TRUE;
        }
    }

    // We favor the extended fields over the basic, but if basic is all we have..
    if (psTrack->acArtistExtd[0] == '\0')
    {
        if (psTrack->acArtistBasic[0] != '\0')
        {
            CDO_vUpdate(hChannel, &psTrack->acArtistBasic[0], CDO_FIELD_ARTIST);
        }
        else
        {
            // Set blank string since CDO_vUpdate doesn't like null text
            CDO_vUpdate(hChannel, "", CDO_FIELD_ARTIST);
        }
    }
    else
    {
        CDO_vUpdate(hChannel, &psTrack->acArtistExtd[0], CDO_FIELD_ARTIST);
    }

    if (psTrack->acSongExtd[0] == '\0')
    {
        if (psTrack->acSongBasic[0] != '\0')
        {
            CDO_vUpdate(hChannel, &psTrack->acSongBasic[0], CDO_FIELD_TITLE);
        }
        else
        {
            // Set blank string since CDO_vUpdate doesn't like null text
            CDO_vUpdate(hChannel, "", CDO_FIELD_TITLE);
        }
    }
    else
    {
        CDO_vUpdate(hChannel, &psTrack->acSongExtd[0], CDO_FIELD_TITLE);
    }

    if ((psTrack->sExtTrackMetaData.un32ValidFieldMask & 
        SXIAPI_VALID_TMI_ALBUM_NAME) == SXIAPI_VALID_TMI_ALBUM_NAME)
    {
        CDO_vUpdate(hChannel, &psTrack->sExtTrackMetaData.acAlbumName[0], CDO_FIELD_ALBUM);
    }
    else
    {
        CDO_vUpdate(hChannel, "", CDO_FIELD_ALBUM);
    }
    
    CDO_vUpdate(hChannel, &psTrack->acContentInfo[0], CDO_FIELD_CONTENTINFO);

    // Now that we've finished updating, let the art manager update
    // the channel art for this CDO.
    CDO_vUpdateArt( hCDO, hChannel );

    if (bUpdated == TRUE)
    {
        CHANNEL_vSetEvents(hChannel, CHANNEL_OBJECT_EVENT_TYPE_INFO);
    }

    return;
}

/*******************************************************************************
 *
 *   vUpdateStatus
 *
 *******************************************************************************/
static void vUpdateStatus(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    const SXIAPI_STATUSIND_STRUCT *psStatus)
{

    switch (psStatus->eStatusItem)
    {
        case SXIAPI_STATUS_ITEM_SIGNAL:
        {
            SIGNAL_QUALITY_ENUM eQuality;
            ANTENNA_STATE_ENUM eAntennaState;

            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "Signal Quality\n");

            // Populate with new signal information...
            switch (psStatus->uData.sSignal.eSignal)
            {
                case SXIAPI_SIGNAL_0BARS:
                    eQuality = SIGNAL_QUALITY_NONE;
                break;

                case SXIAPI_SIGNAL_1BAR:
                    eQuality = SIGNAL_QUALITY_WEAK;
                break;

                case SXIAPI_SIGNAL_2BARS:
                    eQuality = SIGNAL_QUALITY_GOOD;
                break;

                case SXIAPI_SIGNAL_3BARS:
                    eQuality = SIGNAL_QUALITY_EXCELLENT;
                break;

                default:
                    eQuality = SIGNAL_QUALITY_INVALID;
                break;
            }

            // Inform decoder of new info
            psDecoder->sSignalQuality.eComposite = eQuality;
            DECODER_vUpdateSignal(psDecoder->hDecoder, &psDecoder->sSignalQuality);

            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "Antenna State\n");

            // Populate with new signal information...
            switch (psStatus->uData.sSignal.eAntenna)
            {
                case SXIAPI_ANTENNA_OK:
                    eAntennaState = ANTENNA_STATE_DETECTED;
                break;

                case SXIAPI_ANTENNA_DISCONNECTED:
                    eAntennaState = ANTENNA_STATE_NOT_DETECTED;
                break;

                case SXIAPI_ANTENNA_SHORTED:
                    eAntennaState = ANTENNA_STATE_SHORTED;
                break;

                case SXIAPI_ANTENNA_UNKNOWN:
                default:
                    eAntennaState = ANTENNA_STATE_UNKNOWN;
                break;
            }

            // Inform decoder of new info
            psDecoder->eAntennaState = eAntennaState;
            // Single antenna
            DECODER_vUpdateAntennaState(
                psDecoder->hDecoder, 0, psDecoder->eAntennaState);

        }
        break;
        case SXIAPI_STATUS_ITEM_ANTENNA_AIMING:
        {
            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Signal Quality/Antenna Aiming\n");

            // The aiming value is a value between 0% and 100%
            // Normalize to the SMS API signal quality enum
            psDecoder->sSignalQuality.eSatellite = eConvertRSLToQuality(
                psStatus->uData.sAntenna.un8SatPercentage);
            psDecoder->sSignalQuality.eTerrestrial = eConvertRSLToQuality(
                psStatus->uData.sAntenna.un8TerPercentage);
            psDecoder->sSignalQuality.sAntennaAimingData.un8SatPercentSigLevel =
                psStatus->uData.sAntenna.un8SatPercentage;
            psDecoder->sSignalQuality.sAntennaAimingData.un8TerPercentSigLevel =
                psStatus->uData.sAntenna.un8TerPercentage;

            // Inform decoder of new info
            DECODER_vUpdateSignal(psDecoder->hDecoder, &psDecoder->sSignalQuality);
        }
        break;
        case SXIAPI_STATUS_ITEM_AUDIO_DECODER_BITRATE:
        {
            DECODER_vUpdateAudioDecoderBitRate(psDecoder->hDecoder,
                                psStatus->uData.un8AudioDecoderBitrate);
        }
            break;

        case SXIAPI_STATUS_ITEM_SIGNAL_QUALITY:
        {
            DETAILED_SIGNAL_QUALITY_STRUCT sDetailedSignalQuality;
            const SXIAPI_DETAILED_SIGNAL_QUALITY_STRUCT *psSourceSigQuality
                                = &psStatus->uData.sDetailedSignalQuality;

            sDetailedSignalQuality.eSignalStrength =
                                (DETAILED_SIGNAL_STRENGTH_ENUM)
                                 psSourceSigQuality->un8SignalStrength;
            sDetailedSignalQuality.un8TunerStatus =
                                psSourceSigQuality->un8TunerStatus;
            sDetailedSignalQuality.un8ENSALockStatus=
                                psSourceSigQuality->un8ENSALockStatus;
            sDetailedSignalQuality.un8ENSBLockStatus=
                                psSourceSigQuality->un8ENSBLockStatus;
            sDetailedSignalQuality.un16BERS1=psSourceSigQuality->un16BERS1;
            sDetailedSignalQuality.un16BERS2=psSourceSigQuality->un16BERS2;
            sDetailedSignalQuality.un16BERT=psSourceSigQuality->un16BERT;
            sDetailedSignalQuality.un8CNS1A=psSourceSigQuality->un8CNS1A;
            sDetailedSignalQuality.un8CNS1B=psSourceSigQuality->un8CNS1B;
            sDetailedSignalQuality.un8CNS2A=psSourceSigQuality->un8CNS2A;
            sDetailedSignalQuality.un8CNS2B=psSourceSigQuality->un8CNS2B;
            sDetailedSignalQuality.un16RSErrsWords=
                                psSourceSigQuality->un16RSErrsWords;
            sDetailedSignalQuality.un16RSErrsSatSymb=
                                psSourceSigQuality->un16RSErrsSatSymb;
            sDetailedSignalQuality.un16RRSErrsTerrSymb=
                                psSourceSigQuality->un16RRSErrsTerrSymb;
            sDetailedSignalQuality.n16TunerCarrierFreqOffset=
                                psSourceSigQuality->n16TunerCarrierFreqOffset;
            sDetailedSignalQuality.n16RSSI=psSourceSigQuality->n16RSSI;

            DECODER_vUpdateDetailedSignalQuality(psDecoder->hDecoder,
                                &sDetailedSignalQuality);
        }
        break;

        case SXIAPI_STATUS_ITEM_OVERLAY_SIGNAL_QUALITY:
        {
            DETAILED_OVERLAY_SIGNAL_QUALITY_STRUCT
                        sDetailedOverlaySignalQuality;
            const SXIAPI_DETAILED_OVERLAY_SIGNAL_QUALITY_STRUCT
                        *psSourceOverlaySigQuality =
                        &psStatus->uData.sDetailedOverlaySignalQuality;

            sDetailedOverlaySignalQuality.un8ReceiverState =
                        psSourceOverlaySigQuality->un8ReceiverState;
            sDetailedOverlaySignalQuality.un16OberS1A =
                        psSourceOverlaySigQuality->un16OberS1A;
            sDetailedOverlaySignalQuality.un16OberS2A =
                        psSourceOverlaySigQuality->un16OberS2A;
            sDetailedOverlaySignalQuality.un16OberTA =
                        psSourceOverlaySigQuality->un16OberTA;
            sDetailedOverlaySignalQuality.un16OberS1B =
                        psSourceOverlaySigQuality->un16OberS1B;
            sDetailedOverlaySignalQuality.un16OberS2B =
                        psSourceOverlaySigQuality->un16OberS2B;
            sDetailedOverlaySignalQuality.un16OberTB =
                        psSourceOverlaySigQuality->un16OberTB;
            sDetailedOverlaySignalQuality.un16TurboWordErrorRate0A =
                        psSourceOverlaySigQuality->un16TurboWordErrorRate0A;
            sDetailedOverlaySignalQuality.un16TurboWordErrorRate1A =
                        psSourceOverlaySigQuality->un16TurboWordErrorRate1A;
            sDetailedOverlaySignalQuality.un16TurboWordErrorRate2A =
                        psSourceOverlaySigQuality->un16TurboWordErrorRate2A;
            sDetailedOverlaySignalQuality.un16TurboWordErrorRate0B =
                        psSourceOverlaySigQuality->un16TurboWordErrorRate0B;
            sDetailedOverlaySignalQuality.un16TurboWordErrorRate1B =
                        psSourceOverlaySigQuality->un16TurboWordErrorRate1B;
            sDetailedOverlaySignalQuality.un16TurboWordErrorRate2B =
                        psSourceOverlaySigQuality->un16TurboWordErrorRate2B;

            DECODER_vUpdateDetailedOverlaySignalQuality(
                        psDecoder->hDecoder,
                        &sDetailedOverlaySignalQuality);
        }
        break;

        case SXIAPI_STATUS_ITEM_LINK_INFORMATION:
        {
            LINK_INFORMATION_STRUCT sLinkInformation;
            const SXIAPI_LINK_INFORMATION_STRUCT *psSourceLinkInfo=
                            &psStatus->uData.sLinkStatus;

            sLinkInformation.un32NumberConfirmTimeouts =
                            psSourceLinkInfo->un32NumberConfirmTimeouts;
            sLinkInformation.un32NumberDataPacketsDropped=
                            psSourceLinkInfo->un32NumberDataPacketsDropped;
            sLinkInformation.un32NumberDataPacketsTx =
                            psSourceLinkInfo->un32NumberDataPacketsTx;
            sLinkInformation.un32NumberAudioPacketsDropped =
                            psSourceLinkInfo->un32NumberAudioPacketsDropped;
            sLinkInformation.un32NumberAudioPacketsTx =
                            psSourceLinkInfo->un32NumberAudioPacketsTx;

            DECODER_vUpdateLinkStatusInformation(psDecoder->hDecoder,
                            &sLinkInformation);
        }
        break;

        case SXIAPI_STATUS_ITEM_SCAN_ITEMS_AVAILABLE:
        {
            const SXIAPI_SCAN_ITEMS_AVAILABLE_STRUCT *psScanItems =
                &psStatus->uData.sScanItems;

            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Scan Items Available: %d, %d, %d\n",
                psScanItems->un8ScanCriteriaSource,
                psScanItems->un16ScannableUnplayedTrackCnt,
                psScanItems->un16ScannableUnscannedTrackCnt);

            if (psScanItems->un16ScannableUnplayedTrackCnt < 2) // SX-9845-0097
            {
                // Update Tune Scan State
                DECODER_vUpdateTuneScanContentState(psDecoder->hDecoder, FALSE);
            }
            else
            {
                // Update Tune Scan State
                DECODER_vUpdateTuneScanContentState(psDecoder->hDecoder, TRUE);
            }
        }
        break;

        case SXIAPI_STATUS_ITEM_AUDIO_PRESENCE:
        {
            DECODER_AUDIO_PRESENCE_ENUM eAudioPresence;

            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "Audio Presence\n");

            // Populate with new Audio Presence information...
            switch (psStatus->uData.eAudioPresence)
            {
                case SXIAPI_SIGNAL_AUDIO_PRESENT:
                {
                    eAudioPresence = DECODER_AUDIO_PRESENCE_AUDIO_PRESENT;
                }
                break;

                case SXIAPI_SIGNAL_AUDIO_NOT_PRESENT:
                {
                    eAudioPresence = DECODER_AUDIO_PRESENCE_AUDIO_NOT_PRESENT;
                }
                break;

                default:
                {
                    eAudioPresence = DECODER_AUDIO_PRESENCE_UNKNOWN;
                }
                break;
            }

            // Inform decoder of new info
            DECODER_vUpdateAudioPresence(psDecoder->hDecoder, 
                eAudioPresence);
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "Not Handled\n");
        }
        break;
    }
}

/*******************************************************************************
 *
 *   vUpdateDisplayAdvisories
 *
 *******************************************************************************/
static void vUpdateDisplayAdvisories (
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_INDCODE_ENUM eIndCode,
    BOOLEAN bInitialUpdate)
{
    // Record latest display advisory as nominal for all except for
    // SXIAPI_INDCODE_CHECK_ANTENNA and SXIAPI_INDCODE_NO_SIGNAL
    SXIAPI_INDCODE_ENUM eLatestIndCode = SXIAPI_INDCODE_NOMINAL;

    // Any indication other than NO_SIGNAL or
    // CHECK_ANTENNA will mean that those conditions have cleared
    // if they were present. In other words signal is ACQUIRED
    if ( (eIndCode != SXIAPI_INDCODE_CHECK_ANTENNA) &&
         (eIndCode != SXIAPI_INDCODE_NO_SIGNAL) )
    {
        // Set the signal state to a good state
        // if we weren't already in a good state.
        if (psDecoder->sSignalQuality.eState != SIGNAL_STATE_ACQUIRED)
        {
            vSetDecoderAudioSignalState(psDecoder, SIGNAL_STATE_ACQUIRED);
        }
    }

    switch (eIndCode)
    {
        case SXIAPI_INDCODE_NOMINAL:
        {
            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Received SXIAPI_INDCODE_NOMINAL.\n");
        }
        break;
        case SXIAPI_INDCODE_CHECK_ANTENNA:
        {
            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Received SXIAPI_INDCODE_CHECK_ANTENNA.\n");
            // Indicate that the antenna is in error
            vSetDecoderAudioSignalState(psDecoder, SIGNAL_STATE_ANTENNA_ERROR);
            eLatestIndCode = eIndCode;
        }
        break;
        case SXIAPI_INDCODE_NO_SIGNAL:
        {
            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Received SXIAPI_INDCODE_NO_SIGNAL.\n");
            // Indicate that there is no signal
            vSetDecoderAudioSignalState(psDecoder, SIGNAL_STATE_ACQUIRING);
            eLatestIndCode = eIndCode;
        }
        break;
        case SXIAPI_INDCODE_SUB_UPDATE:
        {
            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Received SXIAPI_INDCODE_SUB_UPDATE.\n");
            // We only use this sub update to cause audio changes
            // data service subscription changes can be handled
            // through the DataServiceStatusInd
            if((psDecoder->tCapabilities & SRH_DEVICE_CAPABILITY_AUDIO)
                                    == SRH_DEVICE_CAPABILITY_AUDIO)
            {
                DECODER_vUpdateSubscription(psDecoder->hDecoder);
            }
        }
        break;
        case SXIAPI_INDCODE_CHAN_UNAVAIL:
        {
            TUNEMIX_OBJECT hTuneMixActive =
                DECODER_hGetTuneMixActive(psDecoder->hDecoder);
            TUNE_STATE_ENUM eTuneState =
                DECODER.eTuneState(psDecoder->hDecoder);

            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Received SXIAPI_INDCODE_CHAN_UNAVAIL.\n");
            // Indicate that the channel is unavailable.
            // If it is the channel we tuned to, then Channel Info
            // Indication message will come. Channel removal processing
            // will be done there.

            // This indication code makes sense only
            // if DECODER is up and running OR TuneMix is tunning
            if (((bInitialUpdate == FALSE) &&
                (eTuneState == TUNE_STATE_TUNING_IN_PROGRESS)) ||
                (hTuneMixActive != TUNEMIX_INVALID_OBJECT))
            {
                DECODER_vTunedChanNoLongerAvail(psDecoder->hDecoder);
            }

            if (hTuneMixActive != TUNEMIX_INVALID_OBJECT)
            {
                // TuneMix is unavailable now
                TUNEMIX_vUpdateTuneMixStatus(psDecoder->hDecoder, 
                    CHANNEL_INVALID_ID,
                    TRUE, 
                    TUNEMIX_STATUS_UNAVAILABLE);
            }
        }
        break;

        case SXIAPI_INDCODE_CHAN_UNSUB:
        {
            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Received SXIAPI_INDCODE_CHAN_UNSUB.\n");
            // Indicate that the channel is unsubscribed.
            // We should never get this since we override this check
            // when selecting a channel. Ignore, but put out a msg.
        }
        break;

        case SXIAPI_INDCODE_CHAN_LOCKED:
        {
            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Received SXIAPI_INDCODE_CHAN_LOCKED.\n");
            // Indicate that the channel is locked.
            // We should never get this since we override this check
            // when selecting a channel. Ignore, but put out a msg.
        }
        break;

        case SXIAPI_INDCODE_CHAN_MATURE:
        {
            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Received SXIAPI_INDCODE_CHAN_MATURE.\n");
            // Indicate that the channel is mature.
            // We should never get this since we override this check
            // when selecting a channel. Ignore, but put out a msg.
        }
        break;

        default:
        {
            // Do nothing
        }
        break;
    }

    // Record latest display advisory
    psDecoder->eCurDispAdvisoryIndCode = eLatestIndCode;

    return;
}

/*****************************************************************************
 *
 *   vReleaseESN
 *
 *****************************************************************************/
static void vReleaseESN(
    RADIO_MODULE_OBJECT_STRUCT *psModule)
{
    // Release ESN memory resources
    if(psModule->hESN != STRING_INVALID_OBJECT)
    {
        // Release STRING object
        STRING_vDestroy(psModule->hESN);
        psModule->hESN = STRING_INVALID_OBJECT;
    }

    if(psModule->pacESN != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psModule->pacESN);
        psModule->pacESN = NULL;
    }

    return;
}

/*******************************************************************************
 *
 *   bDispatchToDecoder
 *
 *   This function is called externally from the context of the MODULE.
 *
 *******************************************************************************/
static BOOLEAN bDispatchToDecoder(
    DECODER_OBJECT hDecoder,
    SMS_EVENT_TYPE_ENUM eType,
    const void *pvArg
        )
{
    RADIO_EVENT_DISPATCH_STRUCT const *psDispatch =
        (RADIO_EVENT_DISPATCH_STRUCT const *)pvArg;

    // Post event to this DECODER...
    switch((size_t)eType)
    {
        case RADIO_EVENT_SXI_RX_DATA:
        {
            BOOLEAN bPosted;
            SXIAPI_RX_STRUCT const *psRxData =
                (SXIAPI_RX_STRUCT const *)psDispatch->uRadio.psRxData;

            // Check if we have a valid DECODER handle. If we don't then
            // this is just a dummy call to free the response structure.
            if(hDecoder == DECODER_INVALID_OBJECT)
            {
                // Free original instance of the response
                SXI_vFreeResponse((SXIAPI_RX_STRUCT *)psRxData);
                break;
            }

            // Replicate response structure for each decoder this is
            // posted to.
            SXI_vReAllocateResponse((SXIAPI_RX_STRUCT *)psRxData);

            // Post this RADIO event to the DECODER
            bPosted = bPostDecoderEvent(
                hDecoder,
                (RADIO_EVENT_TYPE_ENUM)eType,
                SMS_EVENT_OPTION_NONE,
                psRxData
                    );

            if(bPosted == FALSE)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME": Unable to post RADIO event.");

                // Free response
                SXI_vFreeResponse((SXIAPI_RX_STRUCT *)psRxData);
                break;
            }

        }
        break;
        default:
            // Unhandled event (ok)
        break;
    }

    // Continue
    return TRUE;
}

/*****************************************************************************
 *
 *   bPowerOnSRM
 *
 *****************************************************************************/
static BOOLEAN bPowerOnSRM (
    FILE *psDevice,
    const char *pacSRMName
        )
{
    N32 n32Err;
    BOOLEAN bAppResetControl = FALSE;

    // Determine if application desired to control the hardware lines
    n32Err = ioctl(
        psDevice, SRH_IOCTL_GET_APP_RESET_CONTROL, 
        pacSRMName, &bAppResetControl);
    if(n32Err != DEV_OK)
    {
        // Error!
        return FALSE;
    }

    if (bAppResetControl == TRUE)
    {
		// Bosch ID #2 bosch api for power off on and reseting x65
    	fc_sxm_vSwitchOnX65();
        return TRUE;
    }

    do
    {
        // Turn the module's power on (SHDN)
        n32Err = ioctl(
            psDevice, SRH_IOCTL_POWER_ON, pacSRMName);
        if(n32Err != DEV_OK)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot enable power on '%s'.", pacSRMName);
            break;
        }

        // Turn the module's antenna on (V_ANT)
        n32Err = ioctl(
            psDevice, SRH_IOCTL_ANTENNA_POWER_ENABLE, pacSRMName);
        if(n32Err != DEV_OK)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Cannot enable antenna on '%s'.", pacSRMName);
            break;
        }

        // According to SX-9840-0017 Section 7.6 we need to
        // wait a minimum of 250 msec before doing anything else.
        OSAL.eTaskDelay(250);

        // We got here so all is good
        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
 *
 *   bPowerOffSRM
 *
 *****************************************************************************/
static BOOLEAN bPowerOffSRM (
    FILE *psDevice,
    const char *pacSRMName
        )
{
    N32 n32Err;
    BOOLEAN bSuccess = FALSE;
    BOOLEAN bAppResetControl = FALSE;

    // Determine if application desired to control the hardware lines
    n32Err = ioctl(
        psDevice, SRH_IOCTL_GET_APP_RESET_CONTROL, 
        pacSRMName, &bAppResetControl);
    if(n32Err != DEV_OK)
    {
        // Error!
        return FALSE;
    }

    if (bAppResetControl == TRUE)
    {
		// Bosch ID #2 bosch api for power off on and reseting x65
    	fc_sxm_vSwitchOffX65();
        return TRUE;
    }

    do
    {
        // De-assert the RESET_M signal (put it in reset)
        n32Err = ioctl(
            psDevice, SRH_IOCTL_RESET_ENABLE, pacSRMName);
        if(n32Err != DEV_OK)
        {
            // Error!
            break;
        }

        // According to SX-9840-0017 Section 7.6 we need to
        // wait a minimum of 250 msec before doing anything else.
        OSAL.eTaskDelay(250);

        // Turn the SRM's power off (SHDN)
        n32Err = ioctl(
            psDevice, SRH_IOCTL_POWER_OFF, pacSRMName);
        if(n32Err != DEV_OK)
        {
            // Error!
            break;
        }

        // Turn the SRM's antenna off (V_ANT)
        n32Err = ioctl(
            psDevice, SRH_IOCTL_ANTENNA_POWER_DISABLE, pacSRMName);
        if(n32Err != DEV_OK)
        {
            // Error!
            break;
        }

        // According to SX-9840-0017 Section 7.6 we need to
        // wait a minimum of 50 msec before doing anything else.
        OSAL.eTaskDelay(50);

        bSuccess = TRUE;

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
 *
 *   bInitializeAudioDecoder
 *
 *****************************************************************************/
static BOOLEAN bInitializeAudioDecoder(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Initialize Self Tune completion flag
    psDecoder->bSelfTuneComplete = FALSE;

    // Set initial/default values for Advanced IR Configuration.
    psDecoder->sChanSelConfig.eFeatureControl =
        SXIAPI_ADV_IR_FEATURE_ENABLE;
    psDecoder->sChanSelConfig.ePlayPoint =
        SXIAPI_ADV_IR_PLAYPOINT_AUTOMATIC;
    psDecoder->sChanSelConfig.tPlaySeconds =
        SXIAPI_ADV_IR_SCAN_PLAY_SECONDS_DEFAULT;
    psDecoder->sChanSelConfig.tChanScanIncludeMask =
        SXIAPI_ADV_IR_CHAN_SCAN_INCLUDE_DEFAULT;
    psDecoder->sChanSelConfig.tChanScanExcludeMask =
        SXIAPI_ADV_IR_CHAN_SCAN_EXCLUDE_DEFAULT;

    psDecoder->sPlayback.eState = SXIAPI_PLAYBACK_STATE_INACTIVE;
    psDecoder->sPlayback.un8BufferUsage = 0;
    psDecoder->sPlayback.un8PlayPt = 0;
    psDecoder->sPlayback.un16TimeOffset = 0;
    psDecoder->sPlayback.un16WarningLimit = 0;
    psDecoder->sPlayback.tOldestPlaybackId = 0;
    psDecoder->sPlayback.tCurrentPlaybackId = 0;
    psDecoder->sPlayback.tNewestPlaybackId = 0;
    psDecoder->sPlayback.un16DurationOfOldestTrack = 0;
    psDecoder->sPlayback.un16DurationOfNewestTrack = 0;
    psDecoder->sPlayback.un32TimeFromTrackStart =
        PLAYBACK_INVALID_TIME_FROM_START;
    psDecoder->sPlayback.un32DurationOfTrack =
        PLAYBACK_INVALID_DURATION_OF_TRACK;
    psDecoder->sPlayback.un16TracksBefore = 0;
    psDecoder->sPlayback.un16TracksRemaining = 0;
    psDecoder->sPlayback.bIRRecordInfoValid = FALSE;
    // set true since playback state is being set to inactive
    // and other playback params are set appropriately.
    psDecoder->sPlayback.bIRPlaybackInfoValid = TRUE;
    psDecoder->sPlayback.bFlushed = TRUE;
    psDecoder->sPlayback.bFlush = FALSE;
    psDecoder->sPlayback.bReady = FALSE;
    psDecoder->sPlayback.bReadySent = FALSE;
    psDecoder->sPlayback.bOffsetReady = FALSE;
    psDecoder->sPlayback.un16Duration = 0;
    psDecoder->sPlayback.bWarningSent = FALSE;
    psDecoder->sPlayback.bLimitSent = FALSE;
    psDecoder->sPlayback.bPaused = FALSE;

    // Obtain radio data
    // We cannot block the MODULE from getting this, but we
    // can at least prevent a race-condition, between update
    // and reading this value.
    // TODO: Note this doesn't work if we have multiple MODULE
    // objects. So we must revisit if that can happen.
    OSAL.eEnterTaskSafeSection();
    // get the module's ir buffer duration
    psDecoder->sPlayback.un16DurationOfBuffer = gun16DurationOfBuffer;
    OSAL.eExitTaskSafeSection();

    do
    {
        CHANNEL_OBJECT hChannel;
        BOOLEAN bPosted;
        SXI_FSM_AUDIO_EVENT_STRUCT sAudioEvent;
        DECODER_EVENT_MASK tStatusMask;

        // Always create safe service id based channel, this is a virtual
        // channel and always exists.
        // We are told the service-id == channel-id
        hChannel = CCACHE_hCreateChannel(
            psDecoder->hCCache, 
            SAFE_SERVICE_ID, 
            SAFE_SERVICE_ID, 
            FALSE); // Postpone notification until the channel update.

        if (hChannel == CHANNEL_INVALID_OBJECT)
        {
            // ERROR
            break;
        }

        // Construct a unique name for this object.
        snprintf( &acName[0], sizeof(acName),
            RADIO_OBJECT_NAME":Pool");

        // Init category list
        eReturnCode = OSAL.eLinkedListCreate(
            &psDecoder->sCategory.hCatList,
            RADIO_OBJECT_NAME":Decoder:CatList",
            n16CompareCatID,
            (OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE));

        if (eReturnCode != OSAL_SUCCESS)
        {
            // ERROR
            break;
        }

        // Populate and post audio event
        sAudioEvent.bInitialize = TRUE;
        sAudioEvent.sStatus.bSignal = FALSE;
        sAudioEvent.sStatus.bAntenna = FALSE;
        sAudioEvent.sStatus.bAudioPresence = FALSE;

        // Check if any status monitors are requested
        tStatusMask = DECODER_tRequestedEventMask(psDecoder->hDecoder);

        // Signal level
        if (tStatusMask & DECODER_OBJECT_EVENT_ANTENNA_SIGNAL)
        {
            sAudioEvent.sStatus.bSignal = TRUE;
        }

        // Antenna status
        if (tStatusMask & DECODER_OBJECT_EVENT_ANTENNA_AIMING)
        {
            sAudioEvent.sStatus.bAntenna = TRUE;
        }

        // Audio Presence status
        if (tStatusMask & DECODER_OBJECT_EVENT_AUDIO_PRESENCE)
        {
            sAudioEvent.sStatus.bAudioPresence = TRUE;
        }

        bPosted = SXI_FSM_bPostAudioEvent(
            psDecoder->hControlCxn, &sAudioEvent);
        if (bPosted == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Unable to post Audio Event.");
            break;
        }

        return TRUE;

    } while(FALSE);

    return FALSE;
}

/*****************************************************************************
 *
 *   vUnInitializeAudioDecoder
 *
 *****************************************************************************/
static void vUnInitializeAudioDecoder(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SXI_FSM_AUDIO_EVENT_STRUCT sAudioEvent;

    // Populate and post audio event
    sAudioEvent.bInitialize = FALSE;
    SXI_FSM_bPostAudioEvent(psDecoder->hControlCxn, &sAudioEvent);

    // Remove the saved category list
    eReturnCode = OSAL.eLinkedListRemoveAll(
        psDecoder->sCategory.hCatList,
        (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy);
    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psDecoder->sCategory.hCatList);
        psDecoder->sCategory.hCatList = OSAL_INVALID_OBJECT_HDL;
    }

    return;
}

/*****************************************************************************
 *
 *   vSetDecoderAudioSignalState
 *
 *****************************************************************************/
static void vSetDecoderAudioSignalState (
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SIGNAL_STATE_ENUM eSignalState
        )
{
    // Populate with new signal information...
    psDecoder->sSignalQuality.eState = eSignalState;

    // Inform decoder of new info
    DECODER_vUpdateSignal(psDecoder->hDecoder, &psDecoder->sSignalQuality);

    return;
}

/*******************************************************************************
*
*   n16CompareCatID
*
*******************************************************************************/
static N16 n16CompareCatID (
    void *pvObj1,
    void *pvObj2
        )
{
    SXIAPI_CATINFOIND_STRUCT *psCatInfo1 =
        (SXIAPI_CATINFOIND_STRUCT *)pvObj1;
    SXIAPI_CATINFOIND_STRUCT *psCatInfo2 =
        (SXIAPI_CATINFOIND_STRUCT *)pvObj2;
    N16 n16Result = N16_MIN;

    // Ensure we have proper pointers...
    if ((psCatInfo1 != NULL) && (psCatInfo2 != NULL))
    {
        n16Result = COMPARE(psCatInfo1->tCatID, psCatInfo2->tCatID);
    }

    return n16Result;
}

/*****************************************************************************
*
*   eConvertRSLToQuality
*
*****************************************************************************/
static SIGNAL_QUALITY_ENUM eConvertRSLToQuality (
    UN8 un8SignalRSLValue
        )
{
    if (un8SignalRSLValue == 0)
    {
        return SIGNAL_QUALITY_NONE;
    }
    else if (un8SignalRSLValue <= 33)
    {
        return SIGNAL_QUALITY_WEAK;
    }
    else if (un8SignalRSLValue <= 66)
    {
        return SIGNAL_QUALITY_GOOD;
    }
    else if (un8SignalRSLValue <= 100)
    {
        return SIGNAL_QUALITY_EXCELLENT;
    }

    // Who knows....
    return SIGNAL_QUALITY_INVALID;
}

/*****************************************************************************
*
*   vProcessFWUpdateEvent
*
*   Function implements the sub state machine for the module updating state
*
*****************************************************************************/
static void vProcessFWUpdateEvent(
    RADIO_MODULE_OBJECT_STRUCT *psModule,
    RADIO_FWUPDATE_STRUCT const *psFWUpdateInfo
        )
{
    MODULE_ERROR_CODE_ENUM eErrorCode = MODULE_ERROR_CODE_NONE;

    switch (psModule->sFWU.eFWUpdateState)
    {
        case RADIO_FWUPDATE_STATE_INITIAL:
        {
            switch (psFWUpdateInfo->eFWUpdateEvent)
            {
                case RADIO_FWUPDATE_EVENT_READ_COMPLETE:
                {
                    psModule->sFWU.psFWUpdateFile = psFWUpdateInfo->psFWUpdateFile;
                    psModule->sFWU.un32FWUpdateNumberPackets = psFWUpdateInfo->un32FWUpdateNumberPackets;
                    psModule->sFWU.pun8FWUpdateBuffer = psFWUpdateInfo->pun8FWUpdateBuffer;

                    MODULE_vUpdateProgress(psModule->hModule,
                                           0,
                                           SMSAPI_RETURN_CODE_SUCCESS);

                    eErrorCode = eFWUpdateTransitionToErasing(psModule);
                }
                break;

                case RADIO_FWUPDATE_EVENT_ABORT:
                {
                    vFWUpdateHandleAbort(psModule);
                }
                break;

                // Just to handle unexpected erase complete event
                // after firmware update abort
                case RADIO_FWUPDATE_EVENT_ERASE_COMPLETE:
                {
                    // do nothing here
                }
                break;

                default:
                {
                    eErrorCode = MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL;
                }
                break;

            }
        }
        break;

        case RADIO_FWUPDATE_STATE_ERASING:
        {
            switch (psFWUpdateInfo->eFWUpdateEvent)
            {
                case RADIO_FWUPDATE_EVENT_ERASE_COMPLETE:
                {
                    OSAL.eTimerStop(psModule->sFWU.hFWUpdateEraseTimer);
                    OSAL.eTimerDelete(psModule->sFWU.hFWUpdateEraseTimer);
                    psModule->sFWU.hFWUpdateEraseTimer =
                        OSAL_INVALID_OBJECT_HDL;
                    psModule->sFWU.un32FWUpdateCurrentPacket = 0;
                    eErrorCode = eFWUpdateTransitionToSending(psModule);

                }
                break;

                case RADIO_FWUPDATE_EVENT_ABORT:
                {
                    vFWUpdateHandleAbort(psModule);
                }
                break;

                case RADIO_FWUPDATE_EVENT_ERROR:
                {
                    if (psFWUpdateInfo->eFWUpdateError ==
                            RADIO_FWUPDATE_ERROR_ERASE_TIMEOUT
                       )
                    {
                        eErrorCode =
                            MODULE_ERROR_CODE_FIRMWARE_UPDATE_ERASE_TIMEOUT;
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                            RADIO_OBJECT_NAME": FWU: Timeout waiting"
                            " for erased indication");
                    }
                    else if (psFWUpdateInfo->eFWUpdateError ==
                            RADIO_FWUPDATE_ERROR_ERASE_FAILED
                            )
                    {
                        eErrorCode =
                            MODULE_ERROR_CODE_FIRMWARE_UPDATE_ERASE_FAILED;
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                            RADIO_OBJECT_NAME": FWU: Erase Failed");
                    }
                    else
                    {
                        eErrorCode = MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL;
                    }
                }
                break;

                default:
                {
                    eErrorCode =
                        MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL;
                }
                break;
            }
        }
        break;

        case RADIO_FWUPDATE_STATE_SENDING:
        {
            switch (psFWUpdateInfo->eFWUpdateEvent)
            {
                case RADIO_FWUPDATE_EVENT_SENT_PACKET:
                {
                    eErrorCode = eFWUpdateTransitionToSending(psModule);
                }
                break;

                case RADIO_FWUPDATE_EVENT_LOAD_COMPLETE:
                {
                    eErrorCode = eFWUpdateHandleSent(psModule);
                }
                break;

                case RADIO_FWUPDATE_EVENT_ABORT:
                {
                    vFWUpdateHandleAbort(psModule);
                }
                break;

                default:
                {
                    eErrorCode =
                        MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL;
                }
                break;
            }
        }
        break;

        case RADIO_FWUPDATE_STATE_ABORTING:
        {
            // do nothing
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                RADIO_OBJECT_NAME":FWU:Unknown FWU state");
            eErrorCode = MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL;
        }
        break;
    }

    if ( eErrorCode != MODULE_ERROR_CODE_NONE )
    {
        vFWUpdateHandleError(psModule, eErrorCode);
    }
    return;

}

/*****************************************************************************
 *
 *   vFWUpdateHandleError
 *
 *****************************************************************************/
static void vFWUpdateHandleError(
    RADIO_MODULE_OBJECT_STRUCT *psModule,
    MODULE_ERROR_CODE_ENUM eErrorCode
        )
{
    BOOLEAN bReturn;

    // clean up
    vFWUpdateCleanup(psModule);

    //set the module error state
    bReturn = bSetModuleError(psModule, eErrorCode);
    if (bReturn == FALSE)
    {
        //not much we can do.
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME": Unable to set module error state");
    }

    //set our firmware update state to initial
    psModule->sFWU.eFWUpdateState = RADIO_FWUPDATE_STATE_INITIAL;

    return;
}

/*****************************************************************************
 *
 *   vFWUpdateCleanup
 *
 *****************************************************************************/
static void vFWUpdateCleanup(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        )
{
    //clean up timer if necessary
    if (psModule->sFWU.hFWUpdateEraseTimer != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eTimerStop(psModule->sFWU.hFWUpdateEraseTimer);
        OSAL.eTimerDelete(psModule->sFWU.hFWUpdateEraseTimer);
        psModule->sFWU.hFWUpdateEraseTimer = OSAL_INVALID_OBJECT_HDL;
    }

    //clean up file handle
    if (psModule->sFWU.psFWUpdateFile != NULL)
    {
        fclose(psModule->sFWU.psFWUpdateFile);
        psModule->sFWU.psFWUpdateFile = NULL;
    }

    //cleanup memory
    if (psModule->sFWU.pun8FWUpdateBuffer != NULL)
    {
        OSAL.vMemoryFree((void *)psModule->sFWU.pun8FWUpdateBuffer);
        psModule->sFWU.pun8FWUpdateBuffer=NULL;
    }

    return;
}

/*****************************************************************************
 *
 *   eLoadFWUpdate
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eLoadFWUpdate(
    MODULE_OBJECT hModule,
    const char *pcFirmwareFileName,
    RADIO_FWUPDATE_STRUCT *psFirmwareUpdate
       )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    do
    {
        BOOLEAN bResult;
        size_t tFileSize = 0;

        //Load File into memory
        psFirmwareUpdate->psFWUpdateFile = fopen(pcFirmwareFileName,"rb");
        if ( psFirmwareUpdate->psFWUpdateFile == NULL )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                RADIO_OBJECT_NAME": Unable to open firmware update file");
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bResult = OSAL.bFileSystemGetFileSize(psFirmwareUpdate->psFWUpdateFile,
                                              &tFileSize);

        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                RADIO_OBJECT_NAME": Unable to determine firmware update file size");
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        //make sure we got a positive file size that is a
        //multiple of our packet size
        if ( ( tFileSize<=0 )
             ||
             ( (tFileSize % SXI_FWUPDATE_PACKET_SIZE) != 0 )
           )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                RADIO_OBJECT_NAME": Invalid firmware update file size");
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        psFirmwareUpdate->un32FWUpdateNumberPackets = tFileSize/
                                                    SXI_FWUPDATE_PACKET_SIZE;

        psFirmwareUpdate->pun8FWUpdateBuffer = (UN8 *)OSAL.pvMemoryAllocate(
                                                     "modFirmUpdateBuf",
                                                     SXI_FWUPDATE_PACKET_SIZE,
                                                     TRUE);

        if( psFirmwareUpdate->pun8FWUpdateBuffer == NULL )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                RADIO_OBJECT_NAME": Unable to allocate firmware update buffer");
            eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
            break;
        }

        bResult = bVerifyFWUpdateFileValid(hModule, psFirmwareUpdate);

        if ( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                RADIO_OBJECT_NAME": Invalid Firmware update file");
            eReturnCode =
                SMSAPI_RETURN_CODE_MODULE_FWUPDATE_FILE_VERSION_INVALID;
            break;
        }

    } while(FALSE);

    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        fclose(psFirmwareUpdate->psFWUpdateFile);
        psFirmwareUpdate->psFWUpdateFile = NULL;
        if( psFirmwareUpdate->pun8FWUpdateBuffer != NULL )
        {
            OSAL.vMemoryFree((void *)psFirmwareUpdate->pun8FWUpdateBuffer);
            psFirmwareUpdate->pun8FWUpdateBuffer=NULL;
        }
    }
    return(eReturnCode);
}

/*****************************************************************************
*
*   bVerifyFWValid
*
*****************************************************************************/
static BOOLEAN bVerifyFWUpdateFileValid(
    MODULE_OBJECT hModule,
    RADIO_FWUPDATE_STRUCT *psFirmwareUpdate
        )
{
    BOOLEAN bReturn;
    UN32 un32ModuleType = 0,
         un32SwVer=0,
         un32FileVer=0,
         un32FileModuleType=0,
         un32FileSwVerEarliest=0,
         un32FileSwVerLatest=0;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    //Get the version information of the running module to
    //validate that the firmware update is applicable
    bReturn = MODULE_VERSION_bGetSwVersion( hModule,
                                            &un32SwVer );
    if ( bReturn == FALSE )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
            RADIO_OBJECT_NAME": Current Module Sw Version Not available");
        return FALSE;
    }

    //Get the module type of the running module to
    //validate that the firmware update is applicable
    bReturn = MODULE_VERSION_bGetModuleType( hModule,
                                             &un32ModuleType );
    if ( bReturn == FALSE )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
            RADIO_OBJECT_NAME": Current Module Type Not available");
        return FALSE;
    }

    eReturnCode = RADIO_eGetFirmwareFileVersion(psFirmwareUpdate->psFWUpdateFile,
                                             &un32FileVer,
                                             &un32FileSwVerEarliest,
                                             &un32FileSwVerLatest,
                                             &un32FileModuleType
                                             );
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        return FALSE;
    }


    //If the Software Major version falls outside the range
    //go ahead and return false
    if ( ( un32ModuleType != un32FileModuleType )
        ||
         ( un32SwVer < un32FileSwVerEarliest )
        ||
         ( un32SwVer > un32FileSwVerLatest )
       )
    {
        //not much we can do here
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                        "\n********************************************************\n"
                        "The module types for the firwmare update file\n"
                        "and the module must match exactly and the sw\n"
                        "version of the module must fall between the\n"
                        "earliest and latestest as allowed by the update\n"
                        "file\n"
                        "********************************************************\n");
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                        "\n"
                        "Firmware Update File\n"
                        "SwVer=%d.%d.%d ModuleType=%d.%d.%d\n"
                        "Earliest SwVer=%d.%d.%d Latest SwVer=%d.%d.%d\n"
                        "Running ModuleType=%d.%d.%d SwVer=%d.%d.%d\n"
                        "\n",
                        //SwVer
                        BYTE2(un32FileVer),
                        BYTE1(un32FileVer),
                        BYTE0(un32FileVer),
                        //ModuleType
                        BYTE2(un32FileModuleType),
                        BYTE1(un32FileModuleType),
                        BYTE0(un32FileModuleType),
                        //Earliest SwVer
                        BYTE2(un32FileSwVerEarliest),
                        BYTE1(un32FileSwVerEarliest),
                        BYTE0(un32FileSwVerEarliest),
                        //Latest SwVer
                        BYTE2(un32FileSwVerLatest),
                        BYTE1(un32FileSwVerLatest),
                        BYTE0(un32FileSwVerLatest),
                        //ModuleType
                        BYTE2(un32ModuleType),
                        BYTE1(un32ModuleType),
                        BYTE0(un32ModuleType),
                        //SwVer
                        BYTE2(un32SwVer),
                        BYTE1(un32SwVer),
                        BYTE0(un32SwVer));

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   vFWUpdateEraseTimerCallback
*
*****************************************************************************/
static void vFWUpdateEraseTimerCallback(
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        )
{
    SMSE_bPostEvent((SMS_EVENT_HDL) pvArg);
    return;
}

/*****************************************************************************
*
*   eFWUpdateTransitionToErasing
*
*****************************************************************************/
static MODULE_ERROR_CODE_ENUM eFWUpdateTransitionToErasing(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        )
{
    MODULE_ERROR_CODE_ENUM eReturnCode = MODULE_ERROR_CODE_NONE;

    //****************************************
    // Handle initial state transition request
    // by sending firmware update
    // cfg command to the module and starting a timer to
    // wait for the erase indication
    //****************************************
    do {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        // Create a Timer for firmware updates of this module. It's a one
        // shot timer which will error out the firmware update process if
        // it is not cleared before firing
        if (psModule->sFWU.hFWUpdateEraseTimer != OSAL_INVALID_OBJECT_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                       RADIO_OBJECT_NAME
                       ": FWU: unexpected firmware erase timer "
                       "present, terminating firmware update");
            eReturnCode = MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL;
            break;
        }

        // PRE-ALLOCATE EVENT (RADIO_EVENT_FWUPDATE)
        if(SMS_INVALID_EVENT_HDL == psModule->sFWU.hEraseTimeoutEvent)
        {
            SMS_EVENT_DATA_UNION *puEventData;

            psModule->sFWU.hEraseTimeoutEvent =
                SMSE_hAllocateEvent( psModule->hEventHdlr,
                    (SMS_EVENT_TYPE_ENUM)RADIO_EVENT_FWUPDATE, &puEventData,
                    SMS_EVENT_OPTION_STATIC
                        );
            if(psModule->sFWU.hEraseTimeoutEvent == SMS_INVALID_EVENT_HDL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                           RADIO_OBJECT_NAME
                           ": FWU: cannot allocate FWU event ");
                eReturnCode = MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL;
                break;
            }
            else
            {
                RADIO_EVENT_UNION *puRadio =
                    (RADIO_EVENT_UNION *)
                        &puEventData->sRadio.apvData[0];
                RADIO_FWUPDATE_STRUCT *psFirmwareEventStruct =
                        &puRadio->sFirmwareUpdate;

                // Populate event
                psFirmwareEventStruct->eFWUpdateEvent =
                    RADIO_FWUPDATE_EVENT_ERROR;
                psFirmwareEventStruct->eFWUpdateError =
                    RADIO_FWUPDATE_ERROR_ERASE_TIMEOUT;
            }
        }

        eOsalReturnCode = OSAL.eTimerCreate(
                    &psModule->sFWU.hFWUpdateEraseTimer,
                    RADIO_OBJECT_NAME":FWU:FirmwareEraseTimer",
                    vFWUpdateEraseTimerCallback,
                    (const void *)psModule->sFWU.hEraseTimeoutEvent);

        if ( eOsalReturnCode != OSAL_SUCCESS )
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RADIO_OBJECT_NAME
                    ": FWU: Can't Create firmware erase Timer.");
            eReturnCode = MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL;
            break;
        }

        eStatusCode = SXIAPI_eFWUpdateCfgCmd (
                        psModule->sSXI.hControlCxn,
                        SXI_FWUPDATE_PACKET_SIZE,
                        psModule->sFWU.un32FWUpdateNumberPackets);

        if ( eStatusCode != SXIAPI_STATUSCODE_MSG_COMPLETE )
        {
            eReturnCode =
                    MODULE_ERROR_CODE_FIRMWARE_UPDATE_CONFIGURE_FAILED;
            break;
        }

        eOsalReturnCode = OSAL.eTimerStartRelative(
                                     psModule->sFWU.hFWUpdateEraseTimer,
                                     SXI_FWUPDATE_ERASE_TIMEOUT_MSEC,
                                     0);
        if( eOsalReturnCode != OSAL_SUCCESS )
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,
                __LINE__,
                RADIO_OBJECT_NAME
                ": FWU: Can't start firmware erase Timer.");
            eReturnCode = MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL;
            break;
        }
    } while(FALSE);

    psModule->sFWU.eFWUpdateState =
                            RADIO_FWUPDATE_STATE_ERASING;

    return eReturnCode;
}

/*****************************************************************************
*
*   eFWUpdateTransitionToSending
*
*****************************************************************************/
static MODULE_ERROR_CODE_ENUM eFWUpdateTransitionToSending(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        )
{
    MODULE_ERROR_CODE_ENUM eReturnCode = MODULE_ERROR_CODE_NONE;

    //****************************************
    //if we are transitioning to the sending state
    //we send the next packet commands with the right data
    do
    {
        RADIO_FWUPDATE_STRUCT sFWUpdateEventStruct =
                        {
                            RADIO_FWUPDATE_EVENT_ERROR,
                            RADIO_FWUPDATE_ERROR_GENERAL,
                            NULL,
                            NULL,
                            0
                        };
        SXIAPI_STATUSCODE_ENUM eStatusCode;
        BOOLEAN bPosted;
        UN8 un8NewPercentageCompletion, un8PreviousPercentageCompletion;
        UN32 un32Result,un32ReadTotal=0;

        // copy the next  chunk to send into the buffer:
        do
        {
            un32Result = (UN32)fread(psModule->sFWU.pun8FWUpdateBuffer,
                             1, SXI_FWUPDATE_PACKET_SIZE,
                             psModule->sFWU.psFWUpdateFile);
            un32ReadTotal += un32Result;
        }
        while (  (un32ReadTotal < SXI_FWUPDATE_PACKET_SIZE)
              && (un32Result > 0)
              );

        if (un32ReadTotal != SXI_FWUPDATE_PACKET_SIZE)
        {
            eReturnCode = MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL;
            break;
        }

        eStatusCode =
                SXIAPI_eFWUpdatePacketCmd (
                    psModule->sSXI.hControlCxn,
                    (UN16)psModule->sFWU.un32FWUpdateCurrentPacket,
                    psModule->sFWU.pun8FWUpdateBuffer,
                    SXI_FWUPDATE_PACKET_SIZE);

        if ( eStatusCode == SXIAPI_STATUSCODE_MSG_FAIL )
        {
            eReturnCode = MODULE_ERROR_CODE_FIRMWARE_UPDATE_PACKET_FAILED;
            break;
        }
        if ( eStatusCode != SXIAPI_STATUSCODE_MSG_COMPLETE )
        {
            eReturnCode = MODULE_ERROR_CODE_FIRMWARE_UPDATE_PACKET_TIMEOUT;
            break;
        }

        //Calculate the integer percentage of packets that we have sent
        //and the integer percentage at the packet before this one
        //so we can see if a new integer value has been reached.
        // (Total packets sent *100)/(total number of packets)
        un8PreviousPercentageCompletion =
            ((psModule->sFWU.un32FWUpdateCurrentPacket )* 100)
                /
            (psModule->sFWU.un32FWUpdateNumberPackets);

        un8NewPercentageCompletion =
            ((psModule->sFWU.un32FWUpdateCurrentPacket + 1)* 100)
                /
            (psModule->sFWU.un32FWUpdateNumberPackets);

        //Notify app if we have sent the next integer increment
        //percentage of the packets to the module
        if ( un8PreviousPercentageCompletion != un8NewPercentageCompletion )
        {
            MODULE_vUpdateProgress(psModule->hModule,
                                   un8NewPercentageCompletion,
                                   SMSAPI_RETURN_CODE_SUCCESS);
        }

        //check to see if we have sent the last packet
        if ( un8NewPercentageCompletion == 100 )
        {
            //sent all the data packets
            //send an event to ourselves with the notification
            sFWUpdateEventStruct.eFWUpdateEvent =
                    RADIO_FWUPDATE_EVENT_LOAD_COMPLETE;
        }
        else
        {
            //send an event indicating we should send the
            //next data packet with the next data packet to send
            sFWUpdateEventStruct.eFWUpdateEvent =
                    RADIO_FWUPDATE_EVENT_SENT_PACKET;
            psModule->sFWU.un32FWUpdateCurrentPacket++;
        }

        // Post event
        //post message to inform our state machine there has been an error
        bPosted = bPostModuleEvent(psModule->hModule,
             (SMS_EVENT_TYPE_ENUM)RADIO_EVENT_FWUPDATE,
             SMS_EVENT_OPTION_NONE,
            (void *) &sFWUpdateEventStruct);
        if (bPosted != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                RADIO_OBJECT_NAME": FWU: Unable to post event.");
            eReturnCode = MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL;
        }

    } while (FALSE);

    if (eReturnCode == MODULE_ERROR_CODE_NONE)
    {
        psModule->sFWU.eFWUpdateState =
                            RADIO_FWUPDATE_STATE_SENDING;
    }
    return eReturnCode;
}

/*****************************************************************************
*
*   eFWUpdateHandleSent
*
*****************************************************************************/
static MODULE_ERROR_CODE_ENUM eFWUpdateHandleSent(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        )
{
    MODULE_ERROR_CODE_ENUM eReturnCode = MODULE_ERROR_CODE_NONE;

    //****************************************
    // in this state we have sent all the packets
    // to the module so wemust send programm cmd
    // and reset the module
    do
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;
        BOOLEAN bReset;

        eStatusCode = SXIAPI_eFWUpdateProgCmd (
                        psModule->sSXI.hControlCxn);
        if ( eStatusCode == SXIAPI_STATUSCODE_MSG_FAIL )
        {
            eReturnCode =
                    MODULE_ERROR_CODE_FIRMWARE_UPDATE_PROGRAMMING_FAILED;
            break;
        }
        if ( eStatusCode != SXIAPI_STATUSCODE_MSG_COMPLETE )
        {
            eReturnCode =
                    MODULE_ERROR_CODE_FIRMWARE_UPDATE_PROGRAMMING_TIMEOUT;
            break;
        }

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "FWU: programmed\n");

        //clean up file handle if necessary
        if (psModule->sFWU.psFWUpdateFile != NULL)
        {
            fclose(psModule->sFWU.psFWUpdateFile);
            psModule->sFWU.psFWUpdateFile=NULL;
        }

        //cleanup memory
        if (psModule->sFWU.pun8FWUpdateBuffer != NULL)
        {
            OSAL.vMemoryFree((void *)psModule->sFWU.pun8FWUpdateBuffer);
            psModule->sFWU.pun8FWUpdateBuffer=NULL;
        }

        // The only way to make an SXi based module to reset, is to
        // reset it's parent SRM.
        bReset = SRM_bReset(psModule->hSRM);
        if (bReset == FALSE)
        {
            eReturnCode = MODULE_ERROR_CODE_RADIO_OBJECT_ERROR;
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                RADIO_OBJECT_NAME": FWU: Unable to reset the SRM.");
            break;
        }

    } while (FALSE);

    if (eReturnCode == MODULE_ERROR_CODE_NONE)
    {
        psModule->sFWU.eFWUpdateState =
                            RADIO_FWUPDATE_STATE_INITIAL;
    }
    return eReturnCode;
}

/*****************************************************************************
*
*   vFWUpdateHandleAbort
*
*****************************************************************************/
static void vFWUpdateHandleAbort(
    RADIO_MODULE_OBJECT_STRUCT *psModule
        )
{
    // clean up
    vFWUpdateCleanup(psModule);

    psModule->sFWU.eFWUpdateState = RADIO_FWUPDATE_STATE_ABORTING;

    return;
}

/*****************************************************************************
 *
 *   bScan
 *
 *   This API causes the RADIO to scan a specified content or channel group
 *
 *   THIS FUNCTION ALWAYS RUNS WITHIN THE CONEXT OF THE DECODER OBJECT
 *
 *   Inputs:
 *       hDecoder - A valid handle to a DECODER object mapped to the physical
 *           hardware to request the scan from
 *       psRadioSelection - pointer to Radio Selection structure describing
 *           the type of selection
 *
 *   Returns:
 *       BOOLEAN - TRUE if carried out successfully. Otherwise
 *       FALSE is returned
 *
 *****************************************************************************/
static BOOLEAN bScan (
    DECODER_OBJECT hDecoder,
    RADIO_CHANNEL_SELECTION_STRUCT const *psRadioSelection
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Check selection type
    if ((psRadioSelection->eSelType < SXIAPI_SELECT_TYPE_SCAN_CAT) ||
        (psRadioSelection->eSelType > SXIAPI_SELECT_TYPE_SCAN_CFG_MUSIC))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RADIO_OBJECT_NAME": Unsupported selection type.");
        return FALSE;
    }

    // Obtain radio data
    psDecoder =
        (RADIO_DECODER_OBJECT_STRUCT *)
            DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "Requesting scan %u\n", psRadioSelection->eSelType);

        // Update overrides
        if ((psRadioSelection->eSelType == SXIAPI_SELECT_TYPE_SCAN_CFG_ALL) ||
            (psRadioSelection->eSelType == SXIAPI_SELECT_TYPE_SCAN_CFG_MUSIC))
        {
            psDecoder->sRadioSelection.tOverrides = psRadioSelection->tOverrides;
        }

        // Request channel scan
        eStatusCode = SXIAPI_eChanSelectCmd(
            psDecoder->hControlCxn,
            psRadioSelection->eSelType,
           (SXIAPI_SID)0, // Never used for scan
           (SXIAPI_CATEGORY_ID)psRadioSelection->tCatId,
            psDecoder->sRadioSelection.tOverrides
                );

        if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   eSXISubStatusToSMSSubStatus
*
*****************************************************************************/
static MODULE_SUBSTATUS_ENUM eSXISubStatusToSMSSubStatus(
    SXIAPI_SUB_STATUS_ENUM eSXISubStatus
        )
{
    MODULE_SUBSTATUS_ENUM eSMSSubStatus;

    switch(eSXISubStatus)
    {
        case SXIAPI_SUB_STATUS_NOT_SUBSCRIBED:
            eSMSSubStatus = MODULE_SUBSTATUS_NOT_SUBSCRIBED;
        break;

        case SXIAPI_SUB_STATUS_SUBSCRIBED:
            eSMSSubStatus = MODULE_SUBSTATUS_SUBSCRIBED;
        break;

        case SXIAPI_SUB_STATUS_SUSPEND_ALERT:
            eSMSSubStatus = MODULE_SUBSTATUS_SUSPEND_ALERT;
        break;

        case SXIAPI_SUB_STATUS_SUSPENDED:
            eSMSSubStatus = MODULE_SUBSTATUS_SUSPENDED;
        break;

        case SXIAPI_SUB_STATUS_INVALID:
        default:
            eSMSSubStatus = MODULE_SUBSTATUS_INVALID;
        break;
    }

    return eSMSSubStatus;
}

/*****************************************************************************
 *
 *   bPostModuleEvent
 *
 *   This function is used to post an event to an object, generally for
 *   the purposes of having that event handled by the RADIO specific
 *   event handler.
 *
 *   Inputs:
 *       hModule - An SMS active object (MODULE).
 *       eEvent - Event type
 *       pvData - Event data
 *
 *   Returns:
 *       BOOLEAN - TRUE if post was successful. Otherwise
 *       FALSE is returned on error.
 *
 *****************************************************************************/
static BOOLEAN bPostModuleEvent (
    MODULE_OBJECT hModule,
    SMS_EVENT_TYPE_ENUM eEvent,
    EVENT_OPTIONS_TYPE tOptions,
    const void *pvData
        )
{
    BOOLEAN bPosted = FALSE;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    // Allocate an event
    hEvent =
        MODULE_hAllocateEvent(
            hModule,
            eEvent,
            tOptions,
            &puEventData
                );
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        // Based on known events, populate message
        switch((size_t)eEvent)
        {
            // Process based on type
            case RADIO_EVENT_SXI_IND:
            {
                SXIAPI_RX_STRUCT ** ppsRxData =
                    (SXIAPI_RX_STRUCT **)
                        puEventData->sRadio.apvData;
                SXIAPI_RX_STRUCT *psRx =
                    (SXIAPI_RX_STRUCT *)pvData;

                // Populate event
                *ppsRxData = psRx;
            }
            break;

            case RADIO_EVENT_FWUPDATE:
            {
                RADIO_EVENT_UNION *puRadio =
                    (RADIO_EVENT_UNION *)
                        puEventData->sRadio.apvData;

                if(pvData != NULL)
                {
                    RADIO_FWUPDATE_STRUCT *psObj =
                        ( RADIO_FWUPDATE_STRUCT *)pvData;

                    puRadio->sFirmwareUpdate.eFWUpdateEvent =
                                    psObj->eFWUpdateEvent;
                    puRadio->sFirmwareUpdate.eFWUpdateError =
                                    psObj->eFWUpdateError;
                    puRadio->sFirmwareUpdate.psFWUpdateFile =
                                    psObj->psFWUpdateFile;
                    puRadio->sFirmwareUpdate.pun8FWUpdateBuffer =
                                    psObj->pun8FWUpdateBuffer;
                    puRadio->sFirmwareUpdate.un32FWUpdateNumberPackets =
                                    psObj->un32FWUpdateNumberPackets;

                }
                else
                {
                    puRadio->sFirmwareUpdate.eFWUpdateEvent =
                                    RADIO_FWUPDATE_EVENT_ERROR;
                    puRadio->sFirmwareUpdate.eFWUpdateError =
                                    RADIO_FWUPDATE_ERROR_GENERAL;
                }
            }
            break;

            default:
            break;
        }

        // Post event
        bPosted = SMSE_bPostEvent(hEvent);
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   bPostDecoderEvent
 *
 *   This function is used to post an event to an object, generally for
 *   the purposes of having that event handled by the RADIO specific
 *   event handler.
 *
 *   Inputs:
 *       hDecoder - An SMS active object (DECODER).
 *       eEvent - Event type
 *       pvData - Event data
 *
 *   Returns:
 *       BOOLEAN - TRUE if post was successful. Otherwise
 *       FALSE is returned on error.
 *
 *****************************************************************************/
static BOOLEAN bPostDecoderEvent (
    DECODER_OBJECT hDecoder,
    RADIO_EVENT_TYPE_ENUM eEvent,
    EVENT_OPTIONS_TYPE tOptions,
    const void *pvData
        )
{
    BOOLEAN bPosted = FALSE;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    switch(eEvent)
    {
        case RADIO_EVENT_SXI_RX_DATA:
        {
            // Allocate an event using caller provided options
            hEvent =
                DECODER_hAllocateEvent(
                    hDecoder,
                    (SMS_EVENT_TYPE_ENUM)eEvent,
                    tOptions,
                    &puEventData
                );
            if(hEvent != SMS_INVALID_EVENT_HDL)
            {
                RADIO_EVENT_UNION *puRadio =
                    (RADIO_EVENT_UNION *)
                    &puEventData->sRadio.apvData[0];
                SXIAPI_RX_STRUCT *psRxData =
                    (SXIAPI_RX_STRUCT *)pvData;

                // Populate event
                puRadio->psRxData = psRxData;

                // Post event
                bPosted = SMSE_bPostEvent(hEvent);
            }
        }
        break;

        default:
            // Unhandled type
        break;
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   bPostDataEvent
 *
 *   This function is used to post an event to the DSM, for
 *   the purposes of having that event handled by the RADIO specific
 *   event handler.
 *
 *   Inputs:
 *       eEvent - Event type
 *       pvData - Event data
 *
 *   Returns:
 *       BOOLEAN - TRUE if post was successful. Otherwise
 *       FALSE is returned on error.
 *
 *****************************************************************************/
static BOOLEAN bPostDataEvent (
    SMS_EVENT_TYPE_ENUM eEvent,
    const void *pvData
        )
{
    BOOLEAN bPosted = FALSE;
    SMS_EVENT_HDL hEvent;
    RADIO_EVENT_UNION *puRadio;

    // Allocate an event
    hEvent =
        DATASERVICE_MGR_hAllocateRadioEvent(
            eEvent,
            SMS_EVENT_OPTION_NONE,
            (void **)&puRadio
                );
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        // Populate event
        puRadio->psRxData =
            (SXIAPI_RX_STRUCT *)pvData;

        // Post event
        bPosted = SMSE_bPostEvent(hEvent);
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   eMapPlayOnSelectMethod
 *
 *   This function maps SXI values of Play On Select Extended metadata to
 *   CHANNEL object's known enumeration.
 *
 *****************************************************************************/
static CHANNEL_PLAY_ON_SELECT_METHOD_ENUM eMapPlayOnSelectMethod (
    UN8 un8PlayOnSelectMethod
    )
{
    CHANNEL_PLAY_ON_SELECT_METHOD_ENUM eChannelMethod =
        CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_B;

    switch (un8PlayOnSelectMethod)
    {
        case 0: //SXIAPI_PLAY_ON_SELECT_METHOD_NEWEST_A:
        {
            eChannelMethod = CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_A;
        }
        break;

        case 1: //SXIAPI_PLAY_ON_SELECT_METHOD_NEWEST_B:
        {
            eChannelMethod = CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_B;
        }
        break;

        case 2: //SXIAPI_PLAY_ON_SELECT_METHOD_NEWEST_C:
        {
            eChannelMethod = CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_C;
        }
        break;

        case 3: //SXIAPI_PLAY_ON_SELECT_METHOD_NEWEST_D:
        {
            eChannelMethod = CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_D;
        }
        break;

        case 4: //SXIAPI_PLAY_ON_SELECT_METHOD_CONSTRAINED_A:
        {
            eChannelMethod = CHANNEL_PLAY_ON_SELECT_METHOD_CONSTRAINED_TYPE_A;
        }
        break;

        case 5: //SXIAPI_PLAY_ON_SELECT_METHOD_CONSTRAINED_B:
        {
            eChannelMethod = CHANNEL_PLAY_ON_SELECT_METHOD_CONSTRAINED_TYPE_B;
        }
        break;

        case 6: //SXIAPI_PLAY_ON_SELECT_METHOD_REALTIME_A:
        {
            eChannelMethod = CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_A;
        }
        break;

        case 7: //SXIAPI_PLAY_ON_SELECT_METHOD_REALTIME_B:
        {
            eChannelMethod = CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_B;
        }
        break;
    }

    return eChannelMethod;
}

/*****************************************************************************
 *
 *   eMapContentType
 *
 *   This function maps SXI values of Channel Content Type Extended metadata
 *   to CHANNEL object's enumeration.
 *
 *****************************************************************************/
static CONTENT_TYPE_ENUM eMapContentType (
    UN8 un8ContentType
    )
{
    CONTENT_TYPE_ENUM eChannelContentType =
        CONTENT_TYPE_UNKNOWN;

    switch (un8ContentType)
    {
        case 0: //SXIAPI_CHAN_CONTENT_TYPE_MUSIC_PID:
        {
            eChannelContentType = CONTENT_TYPE_MUSIC_PID;
        }
        break;

        case 1: //SXIAPI_CHAN_CONTENT_TYPE_MUSIC_AT:
        {
            eChannelContentType = CONTENT_TYPE_MUSIC_AT;
        }
        break;

        case 2: //SXIAPI_CHAN_CONTENT_TYPE_TALK_PID:
        {
            eChannelContentType = CONTENT_TYPE_TALK_PID;
        }
        break;

        case 3: //SXIAPI_CHAN_CONTENT_TYPE_TALK_AT:
        {
            eChannelContentType = CONTENT_TYPE_TALK_AT;
        }
        break;

        case 4: //SXIAPI_CHAN_CONTENT_TYPE_LIVE_PID:
        {
            eChannelContentType = CONTENT_TYPE_LIVE_PID;
        }
        break;

        case 5: //SXIAPI_CHAN_CONTENT_TYPE_LIVE_AT:
        {
            eChannelContentType = CONTENT_TYPE_LIVE_AT;
        }
        break;
    }

    return eChannelContentType;
}

/*****************************************************************************
 *
 *   eMapIRNavigationClass
 *
 *   This function maps SXI values of IR Navigation Class Extended metadata
 *   to CHANNEL object's enumeration.
 *
 *****************************************************************************/
static IR_NAVIGATION_CLASS_ENUM eMapIRNavigationClass (
    UN8 un8IRNavigationClass
    )
{
    IR_NAVIGATION_CLASS_ENUM eChannelClass =
        IR_NAVIGATION_CLASS_DISALLOWED_TYPE_B;

    switch (un8IRNavigationClass)
    {
        case 0://SXIAPI_IR_NAVIGATION_CLASS_UNRESTRICTED_A:
        {
            eChannelClass = IR_NAVIGATION_CLASS_UNRESTRICTED_TYPE_A;
        }
        break;

        case  1://SXIAPI_IR_NAVIGATION_CLASS_UNRESTRICTED_B:
        {
            eChannelClass = IR_NAVIGATION_CLASS_UNRESTRICTED_TYPE_B;
        }
        break;

        case  2://SXIAPI_IR_NAVIGATION_CLASS_RESTRICTED_A:
        {
            eChannelClass = IR_NAVIGATION_CLASS_RESTRICTED_TYPE_A;
        }
        break;

        case  3://SXIAPI_IR_NAVIGATION_CLASS_RESTRICTED_B:
        {
            eChannelClass = IR_NAVIGATION_CLASS_RESTRICTED_TYPE_B;
        }
        break;

        case  4://SXIAPI_IR_NAVIGATION_CLASS_DISALLOWED_A:
        {
            eChannelClass = IR_NAVIGATION_CLASS_DISALLOWED_TYPE_A;
        }
        break;

        case  5://SXIAPI_IR_NAVIGATION_CLASS_DISALLOWED_B:
        {
            eChannelClass = IR_NAVIGATION_CLASS_DISALLOWED_TYPE_B;
        }
        break;

    }

    return eChannelClass;
}

/*****************************************************************************
 *
 *   tMapACO
 *
 *   This function maps SXI values of Alternative Channel Order CMI
 *   to CHANNEL object's value.
 *
 *****************************************************************************/
static CHANNEL_ACO tMapACO (
    UN16 un16ACO
        )
{
    // This API should know everything about protocol 
    // specific values and limitations and shall be used 
    // to preprocess list order

    CHANNEL_ACO tACO;

    // Compare with default value for SXi
    if (un16ACO == SXIAPI_DEFAULT_CHAN_LIST_ORDER)
    {
        tACO = CHANNEL_ACO_DEFAULT;
    }
    else
    {
        tACO = (CHANNEL_ACO)un16ACO;
    }

    return tACO;
}

/*****************************************************************************
 *
 *   bHandleModuleInd
 *
 *****************************************************************************/
static BOOLEAN bHandleModuleInd (
    RADIO_MODULE_OBJECT_STRUCT *psModule,
    SXIAPI_RX_STRUCT *psRxData
        )
{
    BOOLEAN bHandledEvent = TRUE;
    BOOLEAN bSendToDecoder = FALSE;
    BOOLEAN bEngineeringData = FALSE;

    // DEBUG Output for SXI Rx
    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
        "MODULE: OpType = %#04.4x\n", psRxData->sHdr.tOpType);

    // Use this SXi IND and allow the MODULE object to process it.
    vProcessModuleInd(psModule, psRxData, &bSendToDecoder, &bEngineeringData);

    // Now we have allowed the MODULE to process this IND, however
    // we now must decide if we also need to pass this along (dispatch)
    // to our DECODERs.
    if(bSendToDecoder == TRUE)
    {
        BOOLEAN bDispatchedToDecoder;
        RADIO_EVENT_DISPATCH_STRUCT sDispatch;

        // Populate dispatcher
        sDispatch.uRadio.psRxData = psRxData;

        // Response to be handled by multiple decoders
        // Dispatch to all MODULE DECODERs...
        bDispatchedToDecoder =
            MODULE_bDispatchDecoderEvent(psModule->hModule,
                (SMS_EVENT_TYPE_ENUM)RADIO_EVENT_SXI_RX_DATA,
                (void *)&sDispatch,
                bDispatchToDecoder,
                bEngineeringData
                    );
        if(bDispatchedToDecoder == FALSE)
        {
            // Error! We were unable to post this response.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RADIO_OBJECT_NAME": Unable to dispatch DECODER event.");

            // Indicate we wanted to handle this, but could not
            bHandledEvent = FALSE;
        }
    }

    // Always free our instance of the response.
    SXI_vFreeResponse(psRxData);

    return bHandledEvent;
}

/*****************************************************************************
*
*   vUpdateLastTunedChannelSelection
*
*****************************************************************************/
static void vUpdateLastTunedChannelSelection(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_SID tSID
        )
{
    // If the received nominal Service ID is not equal to the last nominal
    // Service ID, then update has occurred.
    if (psDecoder->sRadioSelection.tNominalId != tSID)
    {
        // Update the last nominal channel. Also, keep requested id updated
        // with nominal.
        psDecoder->sRadioSelection.tNominalId =
            psDecoder->sRadioSelection.uId.tService = tSID;

        SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
            "Lats Tuned Channel Selection was updated to Service ID: %d\n",
            psDecoder->sRadioSelection.tNominalId );
    }
}

/*****************************************************************************
*
*   vUpdateChannelSelection
*
*****************************************************************************/
static void vUpdateChannelSelection(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder,
    SXIAPI_SID tSID,
    BOOLEAN bNominal
        )
{
    if (bNominal == TRUE)
    {
        // If the received nominal Service ID is not equal to the last nominal
        // Service ID, then update has occurred.
        if (psDecoder->sRadioSelection.tNominalId != tSID)
        {
            // we just tuned. flush the ccache
            // NOTE: We used to rely on a flushed buffer triggering an scache flush.
            // We have to put this in because with Tune Start, there
            // is buffered content. so we won't see the buffer stats go to 0
            // because there is BIR content there. We have to actually
            // watch for the tune to flush the ccache
            psDecoder->sPlayback.bFlush = TRUE;

            // Update the last nominal channel. Also, keep requested id updated
            // with nominal.
            psDecoder->sRadioSelection.tNominalId =
                psDecoder->sRadioSelection.uId.tService = tSID;

            // On new channel tune, always set bIRRecordInfoValid
            // to false, since IR buffer was flushed.
            psDecoder->sPlayback.bIRRecordInfoValid = FALSE;

            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Nominal Channel Selection has occurred on Service ID: %d\n",
                psDecoder->sRadioSelection.tNominalId );
        }
    }
    else
    {
        SERVICE_ID tCurrentId;
        tCurrentId = DECODER.tCurrentServiceId(psDecoder->hDecoder);

        // Non-nominal update has occurred. Compare current channel
        // and the last nominal to determine channel change.
        if (psDecoder->sRadioSelection.tNominalId != tCurrentId)
        {
            // we just tuned. flush the ccache
            // NOTE: We used to rely on a flushed buffer triggering an scache flush.
            // We have to put this in because with Tune Start, there
            // is buffered content. so we won't see the buffer stats go to 0
            // because there is BIR content there. We have to actually
            // watch for the tune to flush the ccache
            psDecoder->sPlayback.bFlush = TRUE;

            // Update the last nominal channel. Also, keep requested id updated
            // with nominal.
            psDecoder->sRadioSelection.tNominalId =
                psDecoder->sRadioSelection.uId.tService = (SXIAPI_SID)tCurrentId;

            SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5,
                "Non-Nominal Channel Selection has occurred on Service ID: %d\n",
                psDecoder->sRadioSelection.tNominalId );
        }
    }

    if (psDecoder->sPlayback.bFlush == TRUE)
    {
        // FIR buffer's flushing has been detected
        vFlushSCache( psDecoder );
    }

    // Handle Channel Selection for Tune Scan. If the
    // Tune Scan is in progress, and Tune Direct was requested,
    // Tune Scan state should be set to inactive. This scenario
    // considered as scan 'abort'.
    DECODER_vUpdateTuneScanTerminateState(psDecoder->hDecoder, TRUE);

    return;
}

/*****************************************************************************
*
*   pacPlaybackStateToString
*
*****************************************************************************/
static const char *pacPlaybackStateToString(
    SXIAPI_PLAYBACK_STATE_ENUM eState
        )
{
    const char *pacState = "Unknown";

    switch ( eState )
    {
        case SXIAPI_PLAYBACK_STATE_INACTIVE:
        {
            pacState = "Inactive";
        }
        break;

        case SXIAPI_PLAYBACK_STATE_PAUSED:
        {
            pacState = "Paused";
        }
        break;

        case SXIAPI_PLAYBACK_STATE_PLAYING:
        {
            pacState = "Playing";
        }
        break;

        case SXIAPI_PLAYBACK_STATE_FF:
        {
            pacState = "FF";
        }
        break;

        case SXIAPI_PLAYBACK_STATE_REW:
        {
            pacState = "RW";
        }
        break;

        case SXIAPI_PLAYBACK_STATE_UNKNOWN:
        {
            // Already unknown. Just to eliminate warning
        }
        break;
    };

    return pacState;
}

/*****************************************************************************
*
*   vFlushSCache
*
*****************************************************************************/
static void vFlushSCache(
    RADIO_DECODER_OBJECT_STRUCT *psDecoder
        )
{
    PLAYBACK_OBJECT hPlayback;
    SCACHE_OBJECT hSCache;
    PLAYBACK_STATS_STRUCT sStats;

    SMSAPI_DEBUG_vPrint(RADIO_OBJECT_NAME, 5, "SCache Flush\n");

    // FIR buffer flush has occurred. Need to flush the SCache. Also, need to
    // clear the IR stats here. When you tune from a Smart Favorite to Channel 0,
    // since you are not really tuning to a new channel and you are on a Smart Favorite,
    // so the current buffer is not cleared, there is no IRRecordInfoInd sent
    // to indicate the new cleared state.

    // When you tune between 2 channels that are non-Smart Favorites, the buffer is
    // cleared on each channel change. This results in an IRRecordInfoInd indicating
    // this new starting point with empty buffers and durations cleared.

    // Channel 0 is a virtual channel, and you are not actually tuning to a new service.
    // However, the Module code is still behaving as if it tunes to a new channel and
    // thus sends this initial IRRecordInfoInd to show the new cleared state. Since
    // Channel 0 is not a real channel, nothing is stored to IR and thus the values do
    // not change and no additional IRRecordInfoInds are sent.

    hPlayback = DECODER.hPlayback( psDecoder->hDecoder );
    hSCache = PLAYBACK_hSCache( hPlayback );

    // Flush the SCache
    SCACHE_vFlush( hSCache );

    psDecoder->sPlayback.bFlushed = TRUE;
    psDecoder->sPlayback.bFlush = FALSE;
    psDecoder->sPlayback.bReady = FALSE;
    psDecoder->sPlayback.bReadySent = FALSE;
    psDecoder->sPlayback.bOffsetReady = FALSE;

    psDecoder->sPlayback.un8BufferUsage = 0;
    psDecoder->sPlayback.un8PlayPt = 0;
    psDecoder->sPlayback.un16TimeOffset = 0;

    psDecoder->sPlayback.tOldestPlaybackId = 0;
    psDecoder->sPlayback.tCurrentPlaybackId = 0;
    psDecoder->sPlayback.tNewestPlaybackId = 0;
    psDecoder->sPlayback.un16DurationOfOldestTrack = 0;
    psDecoder->sPlayback.un16DurationOfNewestTrack = 0;
    psDecoder->sPlayback.un32TimeFromTrackStart = 0;
    psDecoder->sPlayback.un32DurationOfTrack = 0;
    psDecoder->sPlayback.un16TracksBefore = 0;
    psDecoder->sPlayback.un16TracksRemaining = 0;

    PLAYBACK_vFlushTimeInfo( hPlayback );

    // Reset Playback stats
    OSAL.bMemSet((void *)&sStats, 0, sizeof( PLAYBACK_STATS_STRUCT ));

    // update the playback information
    PLAYBACK_vUpdatePlaybackInfo( hPlayback, &sStats );

    // change status to play
    PLAYBACK_vUpdatePause( hPlayback, FALSE );

    return;
}

/*****************************************************************************
*
*   bSetModuleError
*
*****************************************************************************/
static BOOLEAN bSetModuleError (
    RADIO_MODULE_OBJECT_STRUCT *psModule,
    MODULE_ERROR_CODE_ENUM eErrorCode
        )
{
    BOOLEAN bPosted;

    // Inform MODULE about the error in the radio
    bPosted = MODULE_bSetError(psModule->hModule, eErrorCode);

    return bPosted;
}

/*****************************************************************************
 *
 *   bStatusMonitorSwitch
 *
 *****************************************************************************/
static BOOLEAN bStatusMonitorSwitch (
    DECODER_OBJECT hDecoder,
    SXIAPI_STATUS_ITEM_ENUM const *peStatusItems,
    size_t tStatusItemsNum,
    BOOLEAN bEnable
        )
{
    BOOLEAN bSuccess = FALSE;
    RADIO_DECODER_OBJECT_STRUCT *psDecoder;

    // Obtain radio data
    psDecoder =
        (RADIO_DECODER_OBJECT_STRUCT *)
            DECODER_hGetRadioSpecificData(hDecoder);

    if (psDecoder != NULL)
    {
        SXIAPI_STATUSCODE_ENUM eStatusCode;
        SXIAPI_MONITOR_OP_ENUM eOperation = SXIAPI_MONITOR_OP_DISABLE;

        if (bEnable == TRUE)
        {
            eOperation = SXIAPI_MONITOR_OP_ENABLE;
        }

        // Set Status Monitor Items
        eStatusCode =
            SXIAPI_eStatusMonCmd(
                psDecoder->hControlCxn,
                eOperation,
                (UN8)tStatusItemsNum,
                peStatusItems
                    );

        if (eStatusCode == SXIAPI_STATUSCODE_MSG_RECEIVED)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}
