/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SXI_FSM_H_
#define _SXI_FSM_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include "sms_event_types.h"
#include "sxiapi.h"
#include "sxi.h"
#include "sti_api.h"
#include "sxi_fsm.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define OBJECT_NAME "SXI_FSM"

// Constants used for UART settings map table
#define SXI_DEFAULT_COM_SETTINGS_INDEX   0
#define SXI_UART_SETTINGS_MAX            7

// Constants involved with starting the modules
#define TIME_TO_WAIT_BEFORE_SENDING_MODCFGCMD_MS (20)
#define MODULE_LINK_STARTUP_TIMER_MS (10000)
#define MODCFGIND_WAIT_TIMEOUT_MS (1000)  // According to the v1.3.2 implementation guide
#define LINK_RESET_LIMIT 3

// Constants involved with shutting down the modules
#define POWER_DOWN_MODE_TIMEOUT_MS (2000)
#define FACTORY_DEFAULT_MODE_TIMEOUT_MS (10000)
#define DEFAULT_POWER_DOWN_TIMEOUT_MS (10000)

// Number of control responses which can be queued. This includes
// only IND messages. RESP messages are NOT allocated from this queue.
// Our rational for this value is that we can queue enough responses
// that the MODULE can handle which has the same size.
// Bosch ID#22: Module Control queue size increases from 64 to 256
#define MODULE_CONTROL_IND_QUEUE_SIZE  (256)

  /**************/
 /** TYPEDEFS **/
/**************/

typedef enum sxi_fsm_error_code_enum
{
    SXI_FSM_RADIO_OK,
    SXI_FSM_RADIO_OBJECT_ERROR = MODULE_ERROR_CODE_RADIO_OBJECT_ERROR,
    SXI_FSM_RADIO_COMM_ERROR = MODULE_ERROR_CODE_RADIO_COMM_FAILURE,
    SXI_FSM_RADIO_DEVICE_ERROR = MODULE_ERROR_CODE_RADIO_DEVICE_ERROR

} SXI_FSM_ERROR_CODE_ENUM;

typedef enum sxi_fsm_link_state_enum
{
    SXI_FSM_LINK_STATE_UNKNOWN,
    SXI_FSM_LINK_STATE_INITIAL,
    SXI_FSM_LINK_STATE_START,
    SXI_FSM_LINK_STATE_CONFIG,
    SXI_FSM_LINK_STATE_READY,
    SXI_FSM_LINK_STATE_PWR_DOWN,
    SXI_FSM_LINK_STATE_ERROR,
    SXI_FSM_LINK_STATE_FINAL

} SXI_FSM_LINK_STATE_ENUM;

typedef struct sxi_fsm_link_struct
{
    // Startup Timer
    OSAL_OBJECT_HDL hStartUpTimer;

    // Startup Timer
    OSAL_OBJECT_HDL hModCfgIndWaitTimer;

    // Force power off state timer
    OSAL_OBJECT_HDL hForcePwrOffStateTimer;

    // Module FSM - Current state
    SXI_FSM_LINK_STATE_ENUM eCurrentState;

    // Module FSM - Previous state
    SXI_FSM_LINK_STATE_ENUM ePreviousState;

    // keep track of the number of self-generated reset tries
    UN8 un8ResetRetries;

    // RADIO error
    SXI_FSM_ERROR_CODE_ENUM eErrorCode;

    // SRM-Name
    const char *pacSRMName;

    // Device
    OSAL_FILE_STRUCT *psDevice;

    // Module Info
    MODULE_OBJECT hModule;
    MODULE_ID tId;
    SRH_DEVICE_CAPABILITIES_MASK tCapabilities;

    // SXI-Connections
    STI_HDL hSxiLinkCxn;
    SXI_CONNECTION_ARG_HANDLE hLinkConnectionArg;

    STI_HDL hControlCxn;
    SXI_CONNECTION_ARG_HANDLE hSxiControlCxnArg;

    STI_HDL hSXiDataCxn;
    SXI_CONNECTION_ARG_HANDLE hSxiDataCxnArg;

    // Uart data
    SXIAPI_UARTCONTROL_ENUM eUARTControl;
    const char *pcUARTSettings;

    // SXi Module Config
    SXIAPI_MODULECFGCMD_STRUCT sModuleCfg;

    // Event flags
    BOOLEAN bModuleCfgIndRxd;
    BOOLEAN bModuleEsnRxd;

    // Pre-Allocated power-off event
    SMS_EVENT_HDL hPowerOffStateEvent;

} SXI_FSM_LINK_STRUCT;

// Entry in the table for the possible UART settings supported by SXI
typedef struct sxi_fsm_uart_settings_map_struct
{
    UN32 un32BaudRate;

    SXIAPI_UARTCONTROL_ENUM eControl;

    const char *pcSettings;

} SXI_FSM_UART_SETTINGS_MAP_STRUCT;

  /************/
 /** MACROS **/
/************/

// Choose proper value for the field.
#define RADIO_SET_VALUE(_Var, _VarType, _Val, _Min, _Max, _bOk)              \
{                                                                            \
    SET_VALUE_BOUNDED(_Var, (_VarType)(_Val), _Min, _Max, _bOk)               \
    if ((_bOk) == FALSE)                                                     \
    {                                                                        \
        printf(OBJECT_NAME": failed to set %d to %s at line %d\n",     \
            (int)_Val, #_Var, __LINE__);                                     \
    }                                                                        \
}

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Private Prototypes */

static SXI_FSM_ERROR_CODE_ENUM eFsmEnterInitState (
    SXI_FSM_LINK_STRUCT *psFSM
        );

static SXI_FSM_ERROR_CODE_ENUM eFsmResetLink (
    SXI_FSM_LINK_STRUCT const *psFSM
        );

static SXI_FSM_ERROR_CODE_ENUM eFsmActionStartLink (
    SXI_FSM_LINK_STRUCT const *psFSM
        );

static SXI_FSM_ERROR_CODE_ENUM eFsmExitStartState (
    SXI_FSM_LINK_STRUCT *psFSM
        );

static SXI_FSM_ERROR_CODE_ENUM eFsmEnterConfigState (
    SXI_FSM_LINK_STRUCT const *psFSM
        );

static SXI_FSM_ERROR_CODE_ENUM eFsmEnterReadyState(
    SXI_FSM_LINK_STRUCT const *psFSM
        );

static SXI_FSM_ERROR_CODE_ENUM eFsmExitReadyState(
    SXI_FSM_LINK_STRUCT const *psFSM
        );

static SXI_FSM_ERROR_CODE_ENUM eFsmEnterPwrDownState(
    SXI_FSM_LINK_STRUCT *psFSM
        );

static void vFsmEnterErrorState(
    SXI_FSM_LINK_STRUCT const *psFSM
        );

static SXI_FSM_ERROR_CODE_ENUM eFsmEnterFinalState (
    SXI_FSM_LINK_STRUCT *psFSM
        );

static SXI_FSM_ERROR_CODE_ENUM eFsmHandleOOBEvent (
    SXI_FSM_LINK_STRUCT const *psFSM,
    STI_PROTOCOL_OOB_STRUCT const *psOOB
        );

static SXI_FSM_ERROR_CODE_ENUM eFsmHandleAudioEvent (
    SXI_FSM_LINK_STRUCT const *psFSM,
    SXI_FSM_AUDIO_EVENT_STRUCT const *psAudioEvent
        );

static void vFsmTimeoutCallback(
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        );

static void vPullTheChuteTimerCallback(
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        );

static SXIAPI_STATUSCODE_ENUM eInitAudioCmds (
    SXI_FSM_LINK_STRUCT const *psFSM,
    SXI_FSM_STATUS_MON_STRUCT const *psMonitorStatus
        );

static SXIAPI_STATUSCODE_ENUM eUnInitAudioCmds (
    SXI_FSM_LINK_STRUCT const *psFSM
        );

static BOOLEAN bGetSXIModuleConfiguration(
    SXI_FSM_LINK_STRUCT const *psFSM,
    SXI_MODULE_CFG_STRUCT *psSXiCfg
        );

static BOOLEAN bPostEvent (
    SXI_FSM_LINK_STRUCT const *psFSM,
    EVENT_OPTIONS_TYPE tOptions,
    SXI_FSM_EVENT_STRUCT const *psEvent
        );

#if DEBUG_OBJECT == 1

static const char *pacFsmSxiLinkStateText (
    SXI_FSM_LINK_STATE_ENUM eState
        );

static const char *pacFsmEventText (
    SXI_FSM_EVENT_TYPE_ENUM eEventType
        );

#endif

  /***************/
 /** VARIABLES **/
/***************/

// Table used for all the available UART settings in SXI.
static const SXI_FSM_UART_SETTINGS_MAP_STRUCT gasUARTSettings[SXI_UART_SETTINGS_MAX] =
{
    {
        57600,
        SXIAPI_UARTCONTROL_57600,
        "57600,N,8,1"
    },

    {
        115200,
        SXIAPI_UARTCONTROL_115200,
        "115200,N,8,1"
    },

    {
        230400,
        SXIAPI_UARTCONTROL_230400,
        "230400,N,8,1"
    },

    {
        460800,
        SXIAPI_UARTCONTROL_460800,
        "460800,N,8,1"
    },

    {
        921600,
        SXIAPI_UARTCONTROL_921600,
        "921600,N,8,1"
    },

    {
        1230769,
        SXIAPI_UARTCONTROL_1230769,
        "1230769,N,8,1"
    },

    {
        1846154,
        SXIAPI_UARTCONTROL_1846154,
        "1846154,N,8,1"
    }

};

// Add an assert so that our default index always at least points to
// a valid address.
COMPILE_TIME_ASSERT(SXI_DEFAULT_COM_SETTINGS_INDEX < sizeof(gasUARTSettings),
    SXI_DEFAULT_COM_SETTINGS_INDEX_must_be_valid);

// General Feature Monitor Items

static const SXIAPI_FEATURE_MONITOR_ENUM gaeModuleMonitorItems[] =
{
    SXIAPI_FEATURE_MONITOR_TIME
};

// Audio Status Monitor Items

static const SXIAPI_STATUS_ITEM_ENUM gaeAudioStatusItems[] =
{
    SXIAPI_STATUS_ITEM_SIGNAL,
    SXIAPI_STATUS_ITEM_ANTENNA_AIMING
};

// Audio Feature Monitor Items

static const SXIAPI_FEATURE_MONITOR_ENUM gaeAudioMonitorItems[] =
{
    SXIAPI_FEATURE_MONITOR_CHAN_INFO,
    SXIAPI_FEATURE_MONITOR_CAT_INFO,
    SXIAPI_FEATURE_MONITOR_METADATA,
    // IR must be the last one in the list (see eInitAudioCmds)
    // for the reason why.
    SXIAPI_FEATURE_MONITOR_IR_RECORD_METADATA
};

// Audio Extended Metadata Items

static const SXIAPI_METADATA_IDENTIFIER gatExtMetadataTID[] = {
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_TMI_SONG_ID,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_TMI_ARTIST_ID,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_TMI_ALBUM_NAME,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_TMI_TRAFFIC_ID,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_TMI_ITUNES_SONG_ID,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_TMI_LEAGUE_BCAST_ID,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_TMI_TEAM_BCAST_ID,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_TMI_SPORT_BCAST_TYPE
        };

static const SXIAPI_METADATA_IDENTIFIER gatExtMetadataCID[] = {
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_CMI_SHORT_CHAN_DESC,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_CMI_LONG_CHAN_DESC,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_CMI_RELATED_CHAN_LIST,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_CMI_PLAY_ON_SELECT,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_CMI_CHAN_CONTENT_TYPE,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_CMI_IR_NAVIGATION_CLASS,
    (SXIAPI_METADATA_IDENTIFIER)SXIAPI_CMI_CHAN_LIST_ORDER
};

// Defaults for Module Configuration
static const SXIAPI_MODULECFGCMD_STRUCT gsModuleCfgDefault =
{
    /* eFading = */ SXIAPI_FADING_CFG_ENABLED,
    /* eMaxCatLabelLen = */ SXIAPI_CAT_CHAN_LABEL_LENGTH_16,
    /* eMaxChanLabelLen = */ SXIAPI_CAT_CHAN_LABEL_LENGTH_16,
    /* eMaxMetadataLabelLen = */ SXIAPI_METADATA_LABEL_LENGTH_16,
    /* eMaxPendingInds */ SXIAPI_MAX_PENDING_IND_CFG_4,
    /* tConfirmationWaitTime = */ SXIAPI_MODULECFG_CMD_CONFIRMATION_WAIT_TIME_DEFAULT,
    /* eIRControl = */ SXIAPI_IR_CONTROL_CFG_MODULE,
    /* eIRDeleteOnTune = */ SXIAPI_IR_DELETE_ON_TUNE_CFG_MODULE,
    /* eIRMarkNewTrack = */ SXIAPI_IR_MARK_NEW_TRACK_CFG_ON_LABEL,
    /* tRecordControl = */ SXIAPI_RECORD_CONTROL_MASK_NONE,
    /* tExtendedControl = */ SXIAPI_I2S_CONTROL_VAL_MODE_MASTER |
                             SXIAPI_I2S_CONTROL_VAL_MODE_CONTENT_BUFFERED,
    /* tPrioritySmartFavCnt = */ SXIAPI_PRIORITY_SMART_FAV_CNT_DEFAULT
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _SXI_FSM_H_
