/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _RADIO_DATA_SERVICE_H_
#define _RADIO_DATA_SERVICE_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include "sxiapi.h"
#include "sdtp.h"
#include "radio_data_service.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define RADIO_DATA_SERVICE_NAME "RDS"

/**** Data Service Logging ****/

#define DS_LOGGING_RATE_SECONDS (60)

#define DS_LOGGING_BANNER \
    "\n*************** Data Service Statistics ***************"

#define DS_LOGGING_END \
    "*******************************************************\n"

#define DS_BUFFER_STATS_MSG \
    "\n\tOTA Buffer Statistics:\n" \
    "\t    Buffer Byte Size: %u\n" \
    "\t    Num Bytes Used: %u\n" \
    "\t    Max Bytes Used: %u\n" \
    "\t    Num Failed Block Allocations: %u\n"

// CSV values for post-processing in a spreadsheet
// Format: DSI, Buffer Size, Num Bytes Used, Max Bytes Used, Num Failed
#define DS_BUFFER_STATS_CSV \
    "\n\t  CSV Buffer, %u, %u, %u, %u, %u\n"

#define DS_BUFFER_STATS_OUTPUT_COMBINED \
    DS_BUFFER_STATS_MSG""DS_BUFFER_STATS_CSV

// A set of DMI's attributes
#define DMI_ATTR_NONE       (0x0)
#define DMI_ATTR_AVAILABLE  (0x1)
#define DMI_ATTR_ENABLED    (0x2)

  /**************/
 /** TYPEDEFS **/
/**************/

struct radio_dsi_struct;
struct radio_dmi_struct;

typedef struct radio_data_dmi_list_update_iterator_arg
{
    // input: DSI which DMIs belong to
    struct radio_dsi_struct *psDSI;

    // input: the set of incoming DMIs
    // output: the detected new DMIs
    SXIAPI_DMI atNewDMIs[SXIAPI_DMI_CNT_MAX];

    // input/output: the number of items in the above array
    UN8 un8NewDMIsCount;

    // input: has to be initialized by FALSE
    // output: indicates any chages of attributes in any existing DMIs.
    // Mostly when DMI_ATTR_AVAILABLE attribute is changed.
    BOOLEAN bExistingDMIsChanged;

} RADIO_DATA_DMI_LIST_UPDATE_ITERATOR_ARG;

// Data Control Structure
typedef struct radio_data_ctrl_struct
{
    SMS_OBJECT hSMSDataCtrl;

    // This field is a placeholder for quite a big structure
    // which is too dangerous to be allocated on stack
    RADIO_DATA_DMI_LIST_UPDATE_ITERATOR_ARG sDMIUpdateArg;

    // STI Control Cnx Handle
    STI_HDL hControlCxn;

#if SMS_LOGGING == 1

    // The last time this decoder logged data service stats
    UN32 un32LastLogSeconds;

#endif

} RADIO_DATA_CTRL_STRUCT;

typedef struct radio_data_service_struct
{
    // The list of associated DSIs
    OSAL_OBJECT_HDL hDSIList;

    // Data Service Manager Handle
    DATASERVICE_MGR_OBJECT hManager;

    // Data Service Demux (DMIs) List
    OSAL_OBJECT_HDL hDemuxList;

} RADIO_DATA_SERVICE_STRUCT;

typedef struct radio_dsi_struct
{
    // DSI This structure refers to
    DSI tDSI;

    // The current data service status as reported by the radio
    SXIAPI_DATASERVICE_STATUS_ENUM eStatus;

    // Flag indicating default value for enabling DMIs
    BOOLEAN bEnableAllDMIs;

    // The size the block pool should be for this DSI
    size_t tSuggestedOTABufferByteSize;

    // SDTP connection handle
    // for SDTP-based services.
    // This will be invalid (SDTP_INVALID_HDL)
    // if the service uses another protocol
    SDTP_HDL hSDTP;
    BOOLEAN bSDTPActive;

    // Entry handle (for direct remove)
    OSAL_LINKED_LIST_ENTRY hEntry;

    // Reference to the master service
    RADIO_DATA_SERVICE_STRUCT *psService;

} RADIO_DSI_STRUCT;

typedef struct radio_dmi_struct
{
    // Demux list entry handle (for direct remove)
    OSAL_LINKED_LIST_ENTRY hDemuxEntry;

    // The DMI
    SXM_DMI tDMI;

    // Bit-fields variable to hold DMI_ATTR_... attributes
    // Indicates if DMI is available and/or enabled
    UN8 un8Attributes;

    // Reference to the master DSI
    RADIO_DSI_STRUCT *psDSI;

} RADIO_DMI_STRUCT;

typedef struct radio_data_dmi_info_iterator_arg
{
    // input: DSI which DMIs belong to
    DSI tDSI;

    // input: number of elements in ptDMIs and appsDMIs
    // Can be 0 if not required
    UN8 un8DMIsCountMax;

    // input: array has to be provided if required
    // or can be NULL if not required
    // output: DMIs for specified DSI
    SXM_DMI *ptDMIs;

    // input: array has to be provided if required
    // or can be NULL if not required
    // output: DMIs objects for specified DSI
    RADIO_DMI_STRUCT **apsDMIs;

    // input: has to be initialized (usually by 0)
    // output: number of found DMIs
    UN8 un8DMIsCount;

} RADIO_DATA_DMI_INFO_ITERATOR_ARG;

typedef struct radio_data_find_dsi_iterator_arg
{
    DSI tDSI;
    RADIO_DSI_STRUCT *psDSI;

} RADIO_DATA_FIND_DSI_ITERATOR_ARG;

typedef struct radio_data_dsi_match_struct
{
    DSI tDSI;
    RADIO_DATA_SERVICE_STRUCT *psDstService;

} RADIO_DATA_DSI_MATCH_STRUCT;

typedef struct radio_data_update_dmi_filter_iterator_arg
{
    // The DSI to configure (if known)
    DSI tDSI;

    BOOLEAN bTurnFilterOn;
    BOOLEAN bSuccess;
    RADIO_DATA_CTRL_STRUCT *psDataCtrl;

} RADIO_DATA_UPDATE_DMI_FILTER_ITERATOR_ARG;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Private Prototypes */

static BOOLEAN bRadioDataFindDSIIterator (
    RADIO_PRIVATE_DATA_OBJECT hRadioData,
    RADIO_DATA_FIND_DSI_ITERATOR_ARG *psArg
        );

static BOOLEAN bRadioDataUpdateDMIFilterIterator (
    RADIO_DSI_STRUCT *psDSI,
    RADIO_DATA_UPDATE_DMI_FILTER_ITERATOR_ARG *psArg
        );

static BOOLEAN bRadioDataUpdateDMIFilter (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    RADIO_DSI_STRUCT *psDSI,
    BOOLEAN bTurnFilterOn
        );

static BOOLEAN bRadioDataStopDSIIterator (
    RADIO_DSI_STRUCT *psDSI,
    RADIO_DATA_CTRL_STRUCT *psDataCtrl
        );

static void vRadioDataStopDSI (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    RADIO_DSI_STRUCT *psDSI
        );

static BOOLEAN bRadioDataCreateDSI (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    RADIO_DATA_SERVICE_STRUCT *psService,
    DSI tDSI,
    size_t tSuggestedOTABufferByteSize,
    BOOLEAN bEnableAllDMIs
        );

static void vRadioDataDestroyDSI (
    RADIO_DSI_STRUCT *psDSI
        );

static BOOLEAN bRadioDataDestroyAllDMIs (
    RADIO_DSI_STRUCT *psDSI
        );

static RADIO_DSI_STRUCT *psRadioDataFindDSI (
    RADIO_DATA_SERVICE_STRUCT *psService,
    DSI tDSI
        );

static RADIO_DSI_STRUCT *psRadioDataFindDSIAnywhere (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    DSI tDSI
        );

static N16 n16RadioDataDSICompare (
    RADIO_DSI_STRUCT *psDSI1,
    RADIO_DSI_STRUCT *psDSI2
        );

static void vRadioDataSetDSIStatus (
    RADIO_DSI_STRUCT *psDSI,
    SXIAPI_DATASERVICE_STATUS_ENUM eStatus
        );

static void vRadioDataDSIReleaseHandler (
    RADIO_DSI_STRUCT *psDSI
        );

static RADIO_DMI_STRUCT *psRadioDataCreateNewDMI (
    SXM_DMI tDMI
        );

static BOOLEAN bRadioDataRegisterNewDMI (
    SXM_DMI tDMI,
    BOOLEAN bEnabled,
    RADIO_DSI_STRUCT *psDSI
        );

static BOOLEAN bRadioDataRegisterDMIinDemuxList (
    RADIO_DATA_SERVICE_STRUCT *psService,
    RADIO_DMI_STRUCT *psDMI
        );

static void vRadioDataSetDMIAttribute (
    RADIO_DMI_STRUCT *psDMI,
    UN8 un8DMIAttribute,
    BOOLEAN bAttrValue
        );

static BOOLEAN bRadioDataGetDMIAttribute (
    RADIO_DMI_STRUCT *psDMI,
    UN8 un8DMIAttribute
        );

static void vRadioDataDestroyDMI (
    RADIO_DMI_STRUCT *psDMI
        );

static RADIO_DMI_STRUCT *psRadioDataFindDMI (
    OSAL_OBJECT_HDL hDemuxList,
    SXM_DMI tDMI
        );

static N16 n16RadioDataDMICompare (
    RADIO_DMI_STRUCT *psDMI1,
    RADIO_DMI_STRUCT *psDMI2
        );

static BOOLEAN bRadioDataDMIListUpdateIterator (
    RADIO_DMI_STRUCT *psDMI,
    RADIO_DATA_DMI_LIST_UPDATE_ITERATOR_ARG *psArg
        );

static BOOLEAN bRadioDataFilterComposeIterator (
    RADIO_DMI_STRUCT *psDMI,
    RADIO_DATA_DMI_INFO_ITERATOR_ARG *psArg
        );

static BOOLEAN bRadioDataDMIInfoIterator (
    RADIO_DMI_STRUCT *psDMI,
    RADIO_DATA_DMI_INFO_ITERATOR_ARG *psArg
        );

static BOOLEAN bRadioDataUpdateDSIStatus (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    SXIAPI_DATASERVICESTATUSIND_STRUCT *psDataStatus
        );

static RADIO_DATA_SERVICE_STRUCT *psCreateDataServiceRadioData (
    DATASERVICE_MGR_OBJECT hManager,
    DATASERVICE_ID tDataId
        );

static void vDestroyDataServiceRadioData (
    RADIO_DATA_SERVICE_STRUCT *psService
        );

static void vProcessDataInd (
    RADIO_DATA_CTRL_STRUCT *psDataCtrl,
    SXIAPI_OPTYPE tOpType,
    SXIAPI_INDCODE_ENUM eIndCode,
    SXIAPI_DATASERVICESTATUSIND_STRUCT *psDataStatus
        );

static OSAL_BUFFER_HDL hHandleSDTPPacket (
    RADIO_DSI_STRUCT *psDSI,
    OSAL_BUFFER_HDL *phBuffer
        );

#if SMS_LOGGING == 1

static BOOLEAN bDSILogIterator (
    RADIO_DSI_STRUCT *psDSI,
    OSAL_OBJECT_HDL hBlockPool
        );

static BOOLEAN bIterateServicesForLog (
    RADIO_PRIVATE_DATA_OBJECT hDataServiceRadioData,
    OSAL_OBJECT_HDL hBlockPool
        );

#endif

  /***************/
 /** VARIABLES **/
/***************/

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _RADIO_DATA_SERVICE_H_
