/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the SMS API DEBUG implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <time.h>

#include "standard.h"
#include "osal.h"
#include "sms_api.h"

#include "radio.h"

#include "sms_event.h"
#include "sms_obj.h"
#include "sms_event.h"
#include "ccache.h"
#include "scache.h"
#include "cid_obj.h"
#include "song_obj.h"
#include "decoder_obj.h"
#include "device_obj.h"
#include "playback_obj.h"
#include "cm.h"
#include "league_obj.h"
#include "eprogram_obj.h"

#include "_sms_api_debug.h"

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
*
*   SMSAPI_DEBUG_pacReturnCodeText
*
* This is a local function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
const char *SMSAPI_DEBUG_pacReturnCodeText(
    SMSAPI_RETURN_CODE_ENUM eReturnCode
        )
{
    const char *pacReturnString = "unknown";

    switch (eReturnCode)
    {
        case SMSAPI_RETURN_CODE_SUCCESS:
            pacReturnString = MACRO_TO_STRING(SMSAPI_RETURN_CODE_SUCCESS);
        break;

        case SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED);
        break;

        case SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER);
        break;

        case SMSAPI_RETURN_CODE_NO_OBJECTS:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_NO_OBJECTS);
        break;

        case SMSAPI_RETURN_CODE_ALREADY_INITIALIZED:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_ALREADY_INITIALIZED);
        break;

        case SMSAPI_RETURN_CODE_NOT_INITIALIZED:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_NOT_INITIALIZED);
        break;

        case SMSAPI_RETURN_CODE_NOT_OWNER:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_NOT_OWNER);
        break;

        case SMSAPI_RETURN_CODE_NOT_FOUND:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_NOT_FOUND);
        break;

        case SMSAPI_RETURN_CODE_STILL_ACTIVE:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_STILL_ACTIVE);
        break;

        case SMSAPI_RETURN_CODE_OUT_OF_MEMORY:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_OUT_OF_MEMORY);
        break;

        case SMSAPI_RETURN_CODE_INVALID_INPUT:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_INVALID_INPUT);
        break;

        case SMSAPI_RETURN_CODE_INVALID_DECODER:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_INVALID_DECODER);
        break;

        case SMSAPI_RETURN_CODE_UNSUPPORTED_API:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_UNSUPPORTED_API);
        break;

        case SMSAPI_RETURN_CODE_API_NOT_ALLOWED:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_API_NOT_ALLOWED);
        break;

        case SMSAPI_RETURN_CODE_INVALID_OPTIONS:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_INVALID_OPTIONS);
        break;

        case SMSAPI_RETURN_CODE_INSUFFICIENT_SIZE:
            pacReturnString =
                    MACRO_TO_STRING(SMSAPI_RETURN_CODE_INSUFFICIENT_SIZE);
        break;

        case SMSAPI_RETURN_CODE_PARTIAL_TRANSACTION:
            pacReturnString =
                    MACRO_TO_STRING(SMSAPI_RETURN_CODE_PARTIAL_TRANSACTION);
        break;

        case SMSAPI_RETURN_CODE_CID_NOT_ALLOWED:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_CID_NOT_ALLOWED);
        break;

        case SMSAPI_RETURN_CODE_INVALID_CID_TYPE:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_INVALID_CID_TYPE);
        break;

        case SMSAPI_RETURN_CODE_DUPLICATE_CONTENT:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_DUPLICATE_CONTENT);
        break;

        case SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST);
        break;

        case SMSAPI_RETURN_CODE_UNABLE_TO_CREATE_CONTENT:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_UNABLE_TO_CREATE_CONTENT);
        break;

        case SMSAPI_RETURN_CODE_UNABLE_TO_ADD_CONTENT:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_UNABLE_TO_ADD_CONTENT);
        break;

        case SMSAPI_RETURN_CODE_WRONG_SEEK_SERVICE:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_WRONG_SEEK_SERVICE);
        break;

        case SMSAPI_RETURN_CODE_INVALID_CHANNEL:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_INVALID_CHANNEL);
        break;

        case SMSAPI_RETURN_CODE_LOCKED_CHANNEL:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_LOCKED_CHANNEL);
        break;

        case SMSAPI_RETURN_CODE_MATURE_CHANNEL:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_MATURE_CHANNEL);
        break;

        case SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED);
        break;

        case SMSAPI_RETURN_CODE_SERVICE_NOT_STARTED:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_SERVICE_NOT_STARTED);
        break;

        case SMSAPI_RETURN_CODE_NEW_SERVICE:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_NEW_SERVICE);
        break;

        case SMSAPI_RETURN_CODE_LOAD_SERVICE:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_LOAD_SERVICE);
            break;

        case SMSAPI_RETURN_CODE_EMPTY_PRESET:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_EMPTY_PRESET);
        break;

        case SMSAPI_RETURN_CODE_CFG_VALUE_TOO_LONG:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_CFG_VALUE_TOO_LONG);
        break;

        case SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET);
        break;

        case SMSAPI_RETURN_CODE_CFG_TAG_REMOVED:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_CFG_TAG_REMOVED);
        break;

        case SMSAPI_RETURN_CODE_CFG_MALFORMED_FILE:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_CFG_MALFORMED_FILE);
        break;

        case SMSAPI_RETURN_CODE_CFG_FILE_BAD_CHECKSUM:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_CFG_FILE_BAD_CHECKSUM);
        break;

        case SMSAPI_RETURN_CODE_CFG_BASE64_ERROR:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_CFG_BASE64_ERROR);
        break;

        case SMSAPI_RETURN_CODE_CFG_NO_PARENT:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_CFG_NO_PARENT);
        break;

        case SMSAPI_RETURN_CODE_MODULE_FWUPDATE_FILE_VERSION_INVALID:
            pacReturnString =
                MACRO_TO_STRING(
                    SMSAPI_RETURN_CODE_MODULE_FWUPDATE_FILE_VERSION_INVALID);
        break;

        case SMSAPI_RETURN_CODE_MODULE_FWUPDATE_ALREADY_IN_PROGRESS:
            pacReturnString =
                MACRO_TO_STRING(
                    SMSAPI_RETURN_CODE_MODULE_FWUPDATE_ALREADY_IN_PROGRESS);
        break;

        case SMSAPI_RETURN_CODE_ERROR:
            pacReturnString =
                MACRO_TO_STRING(SMSAPI_RETURN_CODE_ERROR);
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   LEAGUE_DEBUG_bPrint
*
*  This is a DEBUG function used as an object iterator to simply format print
*  the League name and CID information.
*
*****************************************************************************/
BOOLEAN LEAGUE_DEBUG_bPrint (
    LEAGUE_OBJECT hLeague,
    void *pvContentIteratorCallbackArg
        )
{
    FILE *psFile = (FILE *)pvContentIteratorCallbackArg;
    LEAGUE_ENUM eLeague;
    SPORTS_ENUM eSport;
    STRING_OBJECT hString;
    N32 n32Version;
    UN8 un8LeagueId;
    BOOL bSFEnabled;

    // Check inputs
    if(psFile == NULL)
    {
        // Stop iterating
        return FALSE;
    }

    // Print league info

    eLeague = LEAGUE.eLeague(hLeague);
    eSport = LEAGUE.eSport(hLeague);
    un8LeagueId = LEAGUE.un8Id(hLeague);

    fprintf(psFile, "\t#%u ", un8LeagueId);
    hString = LEAGUE.hName(hLeague);
    STRING.n32FWrite(hString, psFile);
    fprintf(psFile, " (");
    hString = LEAGUE.hAbbreviation(hLeague);
    STRING.n32FWrite(hString, psFile);
    n32Version = LEAGUE.n32Version(hLeague);
    bSFEnabled = LEAGUE.bIsSportsFlashEnabled(hLeague);

    fprintf(psFile, ") - %s(%u) <%s> [v%d] SF:%s\n", pacLeagueEnumText(eLeague),
        eLeague, SPORTS_pacEnumText(eSport), n32Version,
        (bSFEnabled == TRUE? "ok" : "n/a" ));

    // Keep iterating
    return TRUE;
}

/*******************************************************************************
*
*   SONG_vPrint
*
*****************************************************************************/
void SONG_vPrint (
    SONG_OBJECT hSong,
    BOOLEAN bVerbose
        )
{
    STRING_OBJECT
        hTitleName, hArtistName;
    N16 n16Offset;
    CHANNEL_ID tId;
    UN32 un32Duration, un32TimeStamp;
    SMSAPI_SONG_STATUS_ENUM eStatus;

    char acArtistName[SMS_ARTIST_NAME_MAX_LENGTH + 1],
         acTitleName[SMS_TITLE_NAME_MAX_LENGTH + 1];
    struct tm sUTCTime;
    TIME_T tTime;
    BOOLEAN bValid;
    SONG_ID tSongId;

    // Verify inputs
    bValid = SMSO_bIsValid((SMS_OBJECT)hSong);
    if(bValid == FALSE)
    {
        // Just print header
        if(bVerbose == FALSE)
        {
            printf("| %-*.*s | %-*.*s | %-*.*s | %-*.*s | %s | %s | %s |\n",

                SMS_ARTIST_NAME_MAX_LENGTH/2,
                SMS_ARTIST_NAME_MAX_LENGTH/2,
                "Artist",
                SMS_TITLE_NAME_MAX_LENGTH/2,
                SMS_TITLE_NAME_MAX_LENGTH/2,
                "Title",
                18,
                18,
                "Offset",
                18,
                18,
                "Channel",
                "Timestamp",
                "Duration",
                "Id"
                    );
        }
        else
        {
            printf("----------------------------------------\n");
        }
        return;
    }

    // Extract object handles from this song...

    // Artist Name
    hArtistName = SONG.hArtist(hSong);
    STRING.tCopyToCStr(hArtistName,
                 &acArtistName[0], sizeof(acArtistName));

    // Title Name
    hTitleName = SONG.hTitle(hSong);
    STRING.tCopyToCStr(hTitleName,
                 &acTitleName[0], sizeof(acTitleName));

    n16Offset = SONG_n16Offset(hSong);
    SONG.eChannelId(hSong, &tId);
    SONG.eDuration(hSong, &un32Duration);
    SONG.eTimeStamp(hSong, &un32TimeStamp);
    SONG.eStatus(hSong, &eStatus);
    tSongId = SONG_tId(hSong);

    if(bVerbose == TRUE)
    {
        printf("Artist: %s\n",
            &acArtistName[0]);
        printf("Title: %s\n",
            &acTitleName[0]);
        printf("Offset: %i ",
            n16Offset);
        printf("\t\tChannel: %u\n",
            (UN32)tId);
                if (un32TimeStamp == 0)
        {
            printf("Timestamp: Unknown ");
        }
        else
        {
            tTime = un32TimeStamp;
            OSAL.gmtime_r( &tTime, &sUTCTime );
            printf("Timestamp: %02u:%02u:%02u ",
               sUTCTime.tm_hour,
               sUTCTime.tm_min,
               sUTCTime.tm_sec);
        }
        if (un32Duration == 0)
        {
            printf("\tDuration: Unknown ");
        }
        else
        {
            tTime = un32Duration;
            OSAL.gmtime_r( &tTime, &sUTCTime );
            printf("\tDuration: %02u:%02u:%02u ",
                sUTCTime.tm_hour,
                sUTCTime.tm_min,
                sUTCTime.tm_sec);
        }

        printf("\tSongID: %u ", tSongId);

        printf("\tStatus: %s\n", SONG_pacStatusText(eStatus));

        printf("----------------------------------------\n");
    }
    else
    {
        printf("| %-*.*s | %-*.*s | %-*.*i | %-*.*u | ",
            SMS_ARTIST_NAME_MAX_LENGTH/2,
            SMS_ARTIST_NAME_MAX_LENGTH/2,
            &acArtistName[0],
            SMS_TITLE_NAME_MAX_LENGTH/2,
            SMS_TITLE_NAME_MAX_LENGTH/2,
            &acTitleName[0],
            18,
            18,
            n16Offset,
            18,
            18,
            (UN32)tId
                );

        tTime = un32TimeStamp;
        OSAL.gmtime_r( &tTime, &sUTCTime );
        printf("%02u:%02u:%02u |\n",
            sUTCTime.tm_hour,
            sUTCTime.tm_min,
            sUTCTime.tm_sec
                );
        tTime = un32Duration;
        OSAL.gmtime_r( &tTime, &sUTCTime );
        printf("%02u:%02u:%02u |\n",
            sUTCTime.tm_hour,
            sUTCTime.tm_min,
            sUTCTime.tm_sec
                );
    }

    return;
}

/*****************************************************************************
*
*   TEAM_bPrint
*
*  This is a DEBUG function used as an object iterator to simply format print
*  the Team name and CID information.
*
*****************************************************************************/
BOOLEAN TEAM_bPrint (
    TEAM_OBJECT hTeam,
    LEAGUE_OBJECT hLeague,
    void *pvContentIteratorCallbackArg
        )
{
    FILE *psFile = (FILE *)pvContentIteratorCallbackArg;
    N32 n32Version;
    UN16 un16TeamId;
    N8 n8Index;
    LEAGUE_MEMBERSHIP_STRUCT sLeagueMembership;
    STRING_OBJECT hString;
    BOOL bSFEligible;

    // Check inputs
    if((hTeam == TEAM_INVALID_OBJECT) || (psFile == NULL))
    {
        // Stop iterating
        return FALSE;
    }

    OSAL.bMemSet(&sLeagueMembership, 0, sizeof(sLeagueMembership));
    un16TeamId = TEAM.un16Id(hTeam);

    // Extract team information
    fprintf(psFile, "\t(#%u)", un16TeamId);

    hString = TEAM.hName(hTeam);
    STRING.n32FWrite(hString, psFile);

    fprintf(psFile, " ");

    hString = TEAM.hNickname(hTeam);
    STRING.n32FWrite(hString, psFile);

    fprintf(psFile, " (");

    hString = TEAM.hAbbreviation(hTeam);
    STRING.n32FWrite(hString, psFile);

    fprintf(psFile, ")");

    fprintf(psFile, " {");

    TEAM.eLeagueMembership(hTeam, &sLeagueMembership);
	for (n8Index = NUMBER_OF_LEAGUE_ELEMENTS-1; n8Index >= 0; n8Index--)
	{
	    fprintf(psFile, " %02x", sLeagueMembership.un8Mask[n8Index]);
	}

	fprintf(psFile, " }");

    n32Version = TEAM.n32Version(hTeam);
    bSFEligible =
        (hLeague != LEAGUE_INVALID_OBJECT) ?
        TEAM.bIsSportsFlashEligible(hLeague, hTeam) :
        FALSE;
    fprintf(psFile, " [v%d] SF:%s\n",
        n32Version, (bSFEligible == TRUE ? "ok" : "n/a"));

    // Keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   CCACHE_DEBUG_bPrintChannelIterator
*
*****************************************************************************/
BOOLEAN CCACHE_DEBUG_bPrintChannelIterator (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSubscribed = FALSE;
    SMSAPI_DEBUG_CCACHE_CHANNEL_ITERATOR_STRUCT *psIterator =
        (SMSAPI_DEBUG_CCACHE_CHANNEL_ITERATOR_STRUCT *)pvArg;
    CHANNEL_ID tChannelId;

    // Print channel
    CHANNEL_DEBUG_vPrint(hChannel, psIterator->eMode);

    // Accumulate statistics
    psIterator->n16NumChannels++;

    tChannelId = CHANNEL.tChannelId(hChannel);
    if(CHANNEL_INVALID_ID != tChannelId)
    {
        eReturnCode = CHANNEL.eIsSubscribed(hChannel, &bSubscribed);
        if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            if(bSubscribed == TRUE)
            {
                psIterator->n16NumSubscribedChannels++;
            }
            else
            {
                psIterator->n16NumUnSubscribedChannels++;
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   CCACHE_DEBUG_bPrintCategoryIterator
*
*****************************************************************************/
BOOLEAN CCACHE_DEBUG_bPrintCategoryIterator (
    CATEGORY_OBJECT hCategory,
    void *pvArg
        )
{
    SMSAPI_DEBUG_CCACHE_CATEGORY_ITERATOR_STRUCT *psIterator =
        (SMSAPI_DEBUG_CCACHE_CATEGORY_ITERATOR_STRUCT *)pvArg;

    // Print channel
    CATEGORY_DEBUG_vPrint(hCategory, psIterator->bVerbose);

    // Accumulate statistics
    psIterator->n16NumCategories++;

    return TRUE;
}

/*******************************************************************************
*
*   CATEGORY_DEBUG_vPrint
*
*****************************************************************************/
void CATEGORY_DEBUG_vPrint (
    CATEGORY_OBJECT hCategory,
    BOOLEAN bVerbose
        )
{
    UN32 un32NumChannels = 0;
    CATEGORY_ID tCategoryId;
    STRING_OBJECT hCategoryLongName, 
                  hCategoryMediumName,
                  hCategoryShortName;
    CHANNEL_ART_OBJECT hArt;
    CATEGORY_ART_INFO_STRUCT sArtInfo;
    DECODER_OBJECT hDecoder;
    CCACHE_OBJECT hCCache;
    char acShortCategoryName[SMS_CATEGORY_SHORT_NAME_MAX_LEN + 1] = "?",
         acMediumCategoryName[SMS_CATEGORY_MEDIUM_NAME_MAX_LEN + 1] = "?", 
         acLongCategoryName[SMS_CATEGORY_LONG_NAME_MAX_LEN + 1] = "?",
         acCategoryId[6];

    if(hCategory == CATEGORY_INVALID_OBJECT)
    {
        // Error
        return;
    }

    // Extract object handles from this channel...
    hCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)hCategory);
    if(hCCache == CCACHE_INVALID_OBJECT)
    {
        // Error
        return;
    }

    hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)hCCache);
    if(hDecoder == DECODER_INVALID_OBJECT)
    {
        // Error
        return;
    }

    // Category Name
    hCategoryShortName = CATEGORY.hShortName(hCategory);
    if (hCategoryShortName != STRING_INVALID_OBJECT)
    {
        STRING.tCopyToCStr(hCategoryShortName,
                     &acShortCategoryName[0], sizeof(acShortCategoryName));
    }
    hCategoryMediumName = CATEGORY.hMediumName(hCategory);
    if (hCategoryMediumName != STRING_INVALID_OBJECT)
    {
        STRING.tCopyToCStr(hCategoryMediumName,
            &acMediumCategoryName[0], sizeof(acMediumCategoryName));
    }
    hCategoryLongName = CATEGORY.hLongName(hCategory);
    if (hCategoryLongName != STRING_INVALID_OBJECT)
    {
        STRING.tCopyToCStr(hCategoryLongName,
                     &acLongCategoryName[0], sizeof(acLongCategoryName));
    }
    // Category Id
    tCategoryId = CATEGORY.tGetCategoryId(hCategory);
    if(tCategoryId == CATEGORY_INVALID_ID)
    {
        strcpy(&acCategoryId[0], "?");
    }
    else
    {
        snprintf(&acCategoryId[0],
            sizeof(acCategoryId), "%03u", tCategoryId);
    }

    // Number of channels
    un32NumChannels = CATEGORY.tSize(hDecoder, tCategoryId);

    printf("%s %s (%s) [%s]\n",
        &acCategoryId[0],
        &acLongCategoryName[0],
        &acMediumCategoryName[0],
        &acShortCategoryName[0]
            );

    // Print list of channels which are members of this category
    if ( un32NumChannels == 0 )
    {
        printf("Channels (%u): \n", un32NumChannels);
    }
    else
    {
        printf("Channels (%u): ", un32NumChannels);
        CATEGORY.eIterate(hDecoder, tCategoryId,
            n16PrintChannelIterator, &un32NumChannels);
    }

    // Print channel art if available
    hArt = CATEGORY.hArt(hCategory);
    sArtInfo.tAvailableImage =
        CHANNEL_ART_AVAILABLE_IMAGE_NONE;

    if (hArt != CHANNEL_ART_INVALID_OBJECT)
    {
        CHANNEL_ART.eUseArt(
            hArt,
            vPrintCategoryArtCallback,
            &sArtInfo );
    }

    // Only print the category art if it is present
    if (sArtInfo.tAvailableImage != CHANNEL_ART_AVAILABLE_IMAGE_NONE)
    {
        printf("Category Art:\n");
        printf("\t%s File name: %s\n",
            &sArtInfo.acImageType[0], &sArtInfo.acImageFilename[0]);
        printf("\t%s File Format: %s\n",
            &sArtInfo.acImageType[0], &sArtInfo.acImageFormat[0]);
    }

    printf("----------------------------------------"
           "----------------------------------------"
           "\n");

    return;
}

/*******************************************************************************
*
*   CHANNEL_DEBUG_vPrint
*
* Ah yes...there must be 50 ways to print a channel
*
*****************************************************************************/
void CHANNEL_DEBUG_vPrint (
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_PRINT_MODE_ENUM eMode
        )
{
    CHANNEL_INFO_STRUCT sInfo;
    CHANNEL_ART_OBJECT hArt;

    // Fill entire structure with 0's
    OSAL.bMemSet(&sInfo, 0, sizeof(sInfo));

    // Remember channel object
    sInfo.hChannel = hChannel;

    if(hChannel != CHANNEL_INVALID_OBJECT)
    {
        // Extract all channel information

        // Channel id
        sInfo.tChannelId = CHANNEL.tChannelId(hChannel);
        if(sInfo.tChannelId == CHANNEL_INVALID_ID)
        {
            strcpy(&sInfo.acChannelId[0], "?");
        }
        else
        {
            snprintf(&sInfo.acChannelId[0],
                sizeof(sInfo.acChannelId), "%03u", sInfo.tChannelId
                    );
        }

        // Service id
        sInfo.tServiceId = CHANNEL.tServiceId(sInfo.hChannel);
        if(sInfo.tServiceId == SERVICE_INVALID_ID)
        {
            strcpy(&sInfo.acServiceId[0], "?");
        }
        else
        {
            snprintf(&sInfo.acServiceId[0], sizeof(sInfo.acServiceId),
                "%03u", sInfo.tServiceId);
        }

        // Channel Name
        sInfo.hChannelShortName = CHANNEL.hShortName(sInfo.hChannel);
        if(sInfo.hChannelShortName != STRING_INVALID_OBJECT)
        {
            STRING.tCopyToCStr(sInfo.hChannelShortName,
                         &sInfo.acChannelShortName[0],
                         sizeof(sInfo.acChannelShortName));
        }
        else
        {
            snprintf(
                &sInfo.acChannelShortName[0],
                sizeof(sInfo.acChannelShortName), "?"
                    );
        }

        // Medium name
        sInfo.hChannelMediumName = CHANNEL.hMediumName(sInfo.hChannel);
        if(sInfo.hChannelMediumName != STRING_INVALID_OBJECT)
        {
            STRING.tCopyToCStr(sInfo.hChannelMediumName,
                         &sInfo.acChannelMediumName[0],
                         sizeof(sInfo.acChannelMediumName));
        }
        else
        {
            snprintf(
                &sInfo.acChannelMediumName[0],
                sizeof(sInfo.acChannelMediumName), "?"
                    );
        }

        // Long name
        sInfo.hChannelLongName = CHANNEL.hLongName(sInfo.hChannel);
        if(sInfo.hChannelLongName != STRING_INVALID_OBJECT)
        {
            STRING.tCopyToCStr(sInfo.hChannelLongName,
                         &sInfo.acChannelLongName[0],
                         sizeof(sInfo.acChannelLongName));
        }
        else
        {
            snprintf(
                &sInfo.acChannelLongName[0],
                sizeof(sInfo.acChannelLongName), "?"
                    );
        }

        // Long Description
        sInfo.hChannelLongDesc = CHANNEL.hLongDescription(sInfo.hChannel);
        STRING.tCopyToCStr(sInfo.hChannelLongDesc,
                     &sInfo.acChannelLongDesc[0], sizeof(sInfo.acChannelLongDesc));

        // Short Description
        sInfo.hChannelShortDesc = CHANNEL.hShortDescription(sInfo.hChannel);
        STRING.tCopyToCStr(sInfo.hChannelShortDesc,
                     &sInfo.acChannelShortDesc[0], sizeof(sInfo.acChannelShortDesc));

        // CDO
        sInfo.hCDO = CHANNEL.hCDO(sInfo.hChannel);

        // Artist Name
        sInfo.hArtistName = CDO.hArtist(sInfo.hCDO);
        STRING.tCopyToCStr(sInfo.hArtistName,
                     &sInfo.acArtistName[0], sizeof(sInfo.acArtistName));

        // Title Name
        sInfo.hTitleName = CDO.hTitle(sInfo.hCDO);
        STRING.tCopyToCStr(sInfo.hTitleName,
                     &sInfo.acTitleName[0], sizeof(sInfo.acTitleName));

        // Composer Name
        sInfo.hComposerName = CDO.hComposer(sInfo.hCDO);
        STRING.tCopyToCStr(sInfo.hComposerName,
                     &sInfo.acComposerName[0], sizeof(sInfo.acComposerName));

        // Album Name
        sInfo.hAlbumName = CDO.hAlbum(sInfo.hCDO);
        STRING.tCopyToCStr(sInfo.hAlbumName,
                     &sInfo.acAlbumName[0], sizeof(sInfo.acAlbumName));

        // ContentInfo Name
        sInfo.hContentInfoName = CDO.hContentInfo(sInfo.hCDO);
        STRING.tCopyToCStr(sInfo.hContentInfoName,
                     &sInfo.acContentInfoName[0], sizeof(sInfo.acContentInfoName));

        // Content Art
        hArt = CDO.hArt( sInfo.hCDO );
        if ( CHANNEL_ART_INVALID_OBJECT != hArt )
        {
            SMSAPI_RETURN_CODE_ENUM eReturnCode;

            // Use this image
            eReturnCode = CHANNEL_ART.eUseArt( hArt, vPrintContentArtCallback,
                &sInfo );

            if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
            {
                // If that didn't work, then
                // make sure we don't display this image
                // by indicating this CDO has no
                // art of this type associated with it
                sInfo.sContentArtInfo.tAvailableImages =
                       CHANNEL_ART_AVAILABLE_IMAGE_NONE;
            }
        }

        // Category Name
        sInfo.hCategory = CHANNEL.hCategory(sInfo.hChannel, 0);
        sInfo.hCategoryShortName = CATEGORY.hShortName(sInfo.hCategory);
        if (sInfo.hCategoryShortName != STRING_INVALID_OBJECT)
        {
            STRING.tCopyToCStr(sInfo.hCategoryShortName,
                         &sInfo.acShortCategoryName[0],
                         sizeof(sInfo.acShortCategoryName));
        }
        else
        {
            strncpy(&sInfo.acShortCategoryName[0], "?",
                    sizeof(sInfo.acShortCategoryName));
        }
        sInfo.hCategoryLongName = CATEGORY.hLongName(sInfo.hCategory);
        if (sInfo.hCategoryLongName != STRING_INVALID_OBJECT)
        {
            STRING.tCopyToCStr(sInfo.hCategoryLongName,
                         &sInfo.acLongCategoryName[0],
                         sizeof(sInfo.acLongCategoryName));
        }
        else
        {
            strncpy(&sInfo.acLongCategoryName[0], "?",
                    sizeof(sInfo.acLongCategoryName));
        }
        // Category Id
        sInfo.tCategoryId = CATEGORY.tGetCategoryId(sInfo.hCategory);
        if(sInfo.tCategoryId == CATEGORY_INVALID_ID)
        {
            strcpy(&sInfo.acCategoryId[0], "?");
        }
        else
        {
            snprintf(&sInfo.acCategoryId[0],
                sizeof(sInfo.acCategoryId), "%03u", sInfo.tCategoryId);
        }

        // Category Art
        hArt = CATEGORY.hArt( sInfo.hCategory );
        sInfo.sCategoryArtInfo.tAvailableImage =
            CHANNEL_ART_AVAILABLE_IMAGE_NONE;

        if (hArt != CHANNEL_ART_INVALID_OBJECT)
        {
            SMSAPI_RETURN_CODE_ENUM eReturnCode;

            // Use this art object to get
            // access to the image we're looking for
            eReturnCode = CHANNEL_ART.eUseArt(
                hArt,
                vPrintCategoryArtCallback,
                &sInfo.sCategoryArtInfo );

            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // If that didn't work, then
                // make sure we don't display anything
                // by indicating this category has no
                // art associated with it
                sInfo.sCategoryArtInfo.tAvailableImage =
                    CHANNEL_ART_AVAILABLE_IMAGE_NONE;
            }
        }

        sInfo.bSubscribed = sInfo.bLocked = sInfo.bSkipped = FALSE;
        CHANNEL.eIsSubscribed(sInfo.hChannel, &sInfo.bSubscribed);
        CHANNEL.eIsLocked(sInfo.hChannel, &sInfo.bLocked);
        CHANNEL.eIsSkipped(sInfo.hChannel, &sInfo.bSkipped);
        CHANNEL.eIsMature(sInfo.hChannel, &sInfo.bMature);
        CHANNEL.eIsFreeToAir(sInfo.hChannel, &sInfo.bFreeToAir);
        CHANNEL.ePlayOnSelect(sInfo.hChannel, &sInfo.ePlayOnSelect);

        // Channel art
        hArt = CHANNEL.hArt( sInfo.hChannel );

        if ( CHANNEL_ART_INVALID_OBJECT != hArt )
        {
            SMSAPI_RETURN_CODE_ENUM eReturnCode;

            // Use this image
            eReturnCode = CHANNEL_ART.eUseArt(
                hArt,
                vPrintChannelArtCallback,
                &sInfo );

            if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
            {
                // If that didn't work, then
                // make sure we don't display this image
                // by indicating this channel has no
                // art of this type associated with it
                sInfo.sChannelArtInfo.tAvailableImages =
                       CHANNEL_ART_AVAILABLE_IMAGE_NONE;
            }
        }

        // Extract Preset data
        strncpy(&sInfo.acPresetBandName[0], "Not a preset",
            sizeof(sInfo.acPresetBandName));
        strncpy(&sInfo.acPresetName[0], "Not a preset",
            sizeof(sInfo.acPresetName));
        strncpy(&sInfo.acPresetIndex[0], "Not a preset",
            sizeof(sInfo.acPresetIndex));
        sInfo.hPresetBand = PRESET_BAND_INVALID_OBJECT;
        sInfo.hPresetName = STRING_INVALID_OBJECT;

        CHANNEL.ePreset(
            sInfo.hChannel, &sInfo.hPresetBand,
            &sInfo.hPresetName, &sInfo.tPresetIndex
                );

        if (sInfo.hPresetBand != PRESET_BAND_INVALID_OBJECT)
        {
            STRING_OBJECT hPresetBandName;

            hPresetBandName = BAND.hName( sInfo.hPresetBand );
            if (hPresetBandName != STRING_INVALID_OBJECT)
            {
                STRING.tCopyToCStr(
                    hPresetBandName,
                    &sInfo.acPresetBandName[0],
                    sizeof(sInfo.acPresetBandName)
                        );
            }

            // Only attempt to extract the preset name if
            // a band was found.  The presence or absence of
            // a band tells us if this channel is part of a preset
            if (sInfo.hPresetName != STRING_INVALID_OBJECT)
            {
                STRING.tCopyToCStr(
                    sInfo.hPresetName,
                    &sInfo.acPresetName[0],
                    sizeof(sInfo.acPresetName)
                        );
            }
            else
            {
                strncpy(&sInfo.acPresetName[0], "Not set",
                    sizeof(sInfo.acPresetName));
            }

            snprintf(&sInfo.acPresetIndex[0],
                     sizeof(sInfo.acPresetIndex),
                     "%02u", sInfo.tPresetIndex);
        }

        sInfo.tACO = CHANNEL.tACO(hChannel);

        // Call printer
        CHANNEL_DEBUG_vPrintInfo(&sInfo, eMode);
    }
    else
    {
        // Print header only
        CHANNEL_DEBUG_vPrintInfo(NULL, eMode);
    }

    return;
}

/*******************************************************************************
*
*   CHANNEL_DEBUG_pacPlayOnSelectMethod
*
*****************************************************************************/
const char *CHANNEL_DEBUG_pacPlayOnSelectMethod(
    CHANNEL_PLAY_ON_SELECT_METHOD_ENUM ePlayOnSelectMethod
        )
{
    char *pacPlayOnSelectMethod = "Default";

    switch (ePlayOnSelectMethod)
    {
        case CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_A:
        {
            pacPlayOnSelectMethod = "Newest (0)";
        }
        break;

        case CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_B:
        {
            pacPlayOnSelectMethod = "Newest (1)";
        }
        break;

        case CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_C:
        {
            pacPlayOnSelectMethod = "Newest (2)";
        }
        break;

        case CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_D:
        {
            pacPlayOnSelectMethod = "Newest (3)";
        }
        break;

        case CHANNEL_PLAY_ON_SELECT_METHOD_CONSTRAINED_TYPE_A:
        {
            pacPlayOnSelectMethod = "Constrained (4)";
        }
        break;

        case CHANNEL_PLAY_ON_SELECT_METHOD_CONSTRAINED_TYPE_B:
        {
            pacPlayOnSelectMethod = "Constrained (5)";
        }
        break;

        case CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_A:
        {
            pacPlayOnSelectMethod = "Realtime (6)";
        }
        break;

        case CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_B:
        {
            pacPlayOnSelectMethod = "Realtime (7)";
        }
        break;
    }

    return pacPlayOnSelectMethod;
}

/*******************************************************************************
*
*   CHANNEL_DEBUG_vPrintInfo
*
*****************************************************************************/
void CHANNEL_DEBUG_vPrintInfo (
    CHANNEL_INFO_STRUCT *psInfo,
    CHANNEL_OBJECT_PRINT_MODE_ENUM eMode
        )
{
    if(psInfo == NULL)
    {
        // Just print headers

        switch(eMode)
        {
            case CHANNEL_OBJECT_PRINT_TERSE:

                fprintf(stdout,
                    "| %4.4s | %4.4s | %5.5s | %-*.*s | %-*.*s | %-*.*s | %-*.*s | %s |\n",
                    "Ch#",
                    "Sid",
                    "ACO",
                    SMS_SHORT_CHANNEL_NAME_MAX_LENGTH,
                    SMS_SHORT_CHANNEL_NAME_MAX_LENGTH,
                    "Ch. Name",
                    SMS_ARTIST_NAME_MAX_LENGTH/2,
                    SMS_ARTIST_NAME_MAX_LENGTH/2,
                    "Artist",
                    SMS_TITLE_NAME_MAX_LENGTH/2,
                    SMS_TITLE_NAME_MAX_LENGTH/2,
                    "Title",
                    SMS_COMPOSER_NAME_MAX_LENGTH/2,
                    SMS_COMPOSER_NAME_MAX_LENGTH/2,
                    "Addt'l Info",
                    "S"
                        );
            break;

            case CHANNEL_OBJECT_PRINT_VERBOSE:
            break;

            case CHANNEL_OBJECT_PRINT_GROSS:
            break;

            default:
            break;
        }
    }
    else
    {
        switch(eMode)
        {
            case CHANNEL_OBJECT_PRINT_TERSE:
            {
                char *pcAddtionalInfo;

                if (psInfo->acComposerName[0] != '\0')
                {
                    pcAddtionalInfo = &psInfo->acComposerName[0];
                }
                else
                {
                    pcAddtionalInfo = &psInfo->acContentInfoName[0];
                }

                fprintf(stdout,
                    "| %4s | %4s |",
                    &psInfo->acChannelId[0],
                    &psInfo->acServiceId[0]
                        );
                
                if (psInfo->tACO == CHANNEL_ACO_DEFAULT)
                {
                    fprintf(stdout, "   N/A |" );
                }
                else
                {
                    fprintf(stdout, " %5d |", psInfo->tACO);
                }
                
                fprintf(stdout,
                    " %-*.*s | %-*.*s | %-*.*s | %-*.*s | %s |\n",
                    SMS_SHORT_CHANNEL_NAME_MAX_LENGTH,
                    SMS_SHORT_CHANNEL_NAME_MAX_LENGTH,
                    &psInfo->acChannelShortName[0],
                    SMS_ARTIST_NAME_MAX_LENGTH/2,
                    SMS_ARTIST_NAME_MAX_LENGTH/2,
                    &psInfo->acArtistName[0],
                    SMS_TITLE_NAME_MAX_LENGTH/2,
                    SMS_TITLE_NAME_MAX_LENGTH/2,
                    &psInfo->acTitleName[0],
                    SMS_COMPOSER_NAME_MAX_LENGTH/2,
                    SMS_COMPOSER_NAME_MAX_LENGTH/2,
                    pcAddtionalInfo,
                    ((psInfo->bSubscribed == TRUE) ? "S" : "U")
                    );
            }
            break;

            case CHANNEL_OBJECT_PRINT_VERBOSE:
            {
                fprintf(stdout,
                    "    Channel Id: %u\n",
                    CHANNEL.tChannelId(psInfo->hChannel));
                fprintf(stdout,
                    "    Service Id: %u\n",
                    CHANNEL.tServiceId(psInfo->hChannel));
                fprintf(stdout,
                    "    Broadcast Category (%u): %s (%s)\n",
                    psInfo->tCategoryId,
                    &psInfo->acLongCategoryName[0],
                    &psInfo->acShortCategoryName[0]
                    );
                fprintf(stdout,
                    "    Channel Name: %s (%s) [%s]\n",
                    &psInfo->acChannelLongName[0],
                    &psInfo->acChannelMediumName[0],
                    &psInfo->acChannelShortName[0]
                    );
                fprintf(stdout,
                    "    Long Channel Description: %s\n",
                    &psInfo->acChannelLongDesc[0]);
                fprintf(stdout,
                    "    Short Channel Description: %s\n",
                    &psInfo->acChannelShortDesc[0]);
                fprintf(stdout,
                    "    Similar Channels:");
                CHANNEL.eIterateSimilarChannels(psInfo->hChannel,
                    bPrintSimilarChannels, NULL);
                fprintf(stdout, "\n");
                fprintf(stdout,
                    "    Artist: %s\n",
                    &psInfo->acArtistName[0]);
                fprintf(stdout,
                    "    Title: %s\n",
                    &psInfo->acTitleName[0]);
                fprintf(stdout,
                    "    Composer: %s\n",
                    &psInfo->acComposerName[0]);
                fprintf(stdout,
                    "    Album: %s\n",
                    &psInfo->acAlbumName[0]);
                fprintf(stdout,
                    "    ContentInfo: %s\n",
                    &psInfo->acContentInfoName[0]);
                fprintf(stdout,
                    "    Subscribed: %s\n",
                    ATTRIBUTE_STATE_TEXT(psInfo->bSubscribed));
                fprintf(stdout,
                    "    Skipped: %s\n",
                    ATTRIBUTE_STATE_TEXT(psInfo->bSkipped));
                fprintf(stdout,
                    "    Locked: %s\n",
                    ATTRIBUTE_STATE_TEXT(psInfo->bLocked));
                fprintf(stdout,
                    "    Mature: %s\n",
                    ATTRIBUTE_STATE_TEXT(psInfo->bMature));
                fprintf(stdout,
                    "    Free-to-air: %s\n",
                    ATTRIBUTE_STATE_TEXT(psInfo->bFreeToAir));
                fprintf(stdout,
                    "    Play On Select Method: %s\n",
                    CHANNEL_DEBUG_pacPlayOnSelectMethod(psInfo->ePlayOnSelect));
                fprintf(stdout,
                    "    Content Type: %s\n",
                    CHANNEL_pacContentType(psInfo->hChannel));

                if (psInfo->tACO == CHANNEL_ACO_DEFAULT)
                {
                    fprintf(stdout, "    ACO: Not assigned\n");
                }
                else
                {
                    fprintf(stdout, "    ACO: %d\n", psInfo->tACO);
                }

                fprintf(stdout,
                        "    On air:");
                CHANNEL.eIteratePrograms(psInfo->hChannel,
                        bPrintEpgEvents, (void*)eMode);
                fprintf(stdout, "\n");

                if (psInfo->sChannelArtInfo.tAvailableImages !=
                    CHANNEL_ART_AVAILABLE_IMAGE_NONE)
                {
                    // Print the Logo for this channel
                    fprintf(stdout,
                        "    Channel Art:\n");
                    fprintf(stdout,
                        "    \t%s file name: %s\n",
                        &psInfo->sChannelArtInfo.sLogoInfo.acImageType[0],
                        &psInfo->sChannelArtInfo.sLogoInfo.acFilename[0]
                            );
                }

                // Print the category art if it exists
                if (psInfo->sCategoryArtInfo.tAvailableImage !=
                    CHANNEL_ART_AVAILABLE_IMAGE_NONE)
                {
                    fprintf(stdout,
                        "    Category Art:\n");
                    fprintf(stdout,
                        "    \t%s file name: %s\n",
                        &psInfo->sCategoryArtInfo.acImageType[0],
                        &psInfo->sCategoryArtInfo.acImageFilename[0]
                            );
                }

                // Print the album art if it exists
                if ( psInfo->sContentArtInfo.tAvailableImages !=
                     CHANNEL_ART_AVAILABLE_IMAGE_NONE )
                {
                    fprintf(stdout,
                        "    Album Art:\n");
                    fprintf(stdout,
                        "    \t%s file name: %s\n",
                        &psInfo->sContentArtInfo.sAlbumInfo.acImageType[0],
                        &psInfo->sContentArtInfo.sAlbumInfo.acFilename[0]
                            );
                }
            }
            break;

            case CHANNEL_OBJECT_PRINT_GROSS:
            {
                SMSAPI_RETURN_CODE_ENUM eReturn;
                N16 n16NumCats;
                N32 n32Return;

                // Call ourselves again, but with the verbose mode
                CHANNEL_DEBUG_vPrintInfo(psInfo, CHANNEL_OBJECT_PRINT_VERBOSE);

                // print out the user categories this channel belongs to
                eReturn = CHANNEL.eNumCategories(psInfo->hChannel, &n16NumCats);
                if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
                {
                    N16 n16Index;
                    CATEGORY_OBJECT hCategory;
                    char acShortCategoryName[SMS_SHORT_CATEGORY_NAME_MAX_LENGTH + 1] = "?",
                         acLongCategoryName[SMS_LONG_CATEGORY_NAME_MAX_LENGTH + 1] = "?";
                    CATEGORY_ID tCatId;
                    CATEGORY_TYPE_ENUM eCatType;
                    STRING_OBJECT hCategoryShortName, hCategoryLongName;
                    const char *pcFormat = NULL;

                    fputs("\n\n    SMS Categories:\n", stdout);

                    // print out cat info for all categories
                    // found after index 0
                    for (n16Index = 1; n16Index < n16NumCats; n16Index++)
                    {
                        N16 n16Offset = N16_MIN;

                        hCategory =
                            CHANNEL.hCategory(psInfo->hChannel, n16Index);

                        if (hCategory == CATEGORY_INVALID_OBJECT)
                        {
                            continue;
                        }

                        // Grab the type
                        eCatType = CATEGORY.eType(hCategory);

                        // Verify category type is valid
                        if (   ( eCatType != CATEGORY_TYPE_BROADCAST)
                            && ( eCatType != CATEGORY_TYPE_USER)
                            && ( eCatType != CATEGORY_TYPE_VIRTUAL)
                           )
                        {
                            continue;
                        }

                        // Grab the id
                        tCatId = CATEGORY.tGetCategoryId(hCategory);

                        // Category Names
                        hCategoryShortName =
                            CATEGORY.hShortName(hCategory);
                        if (hCategoryShortName != STRING_INVALID_OBJECT)
                        {
                            STRING.tCopyToCStr(hCategoryShortName,
                                         &acShortCategoryName[0],
                                         sizeof(acShortCategoryName));
                        }
                        hCategoryLongName =
                            CATEGORY.hLongName(hCategory);
                        if (hCategoryShortName != STRING_INVALID_OBJECT)
                        {
                            STRING.tCopyToCStr(hCategoryLongName,
                                         &acLongCategoryName[0],
                                         sizeof(acLongCategoryName));
                        }
                        eReturn = CHANNEL.eCategoryOffset(psInfo->hChannel, tCatId, &n16Offset);
                        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
                        {
                            printf("SMSD: Failed to get offset of category %d: %s (#%d)\n",
                                tCatId, SMSAPI_DEBUG_pacReturnCodeText(eReturn), eReturn);
                        }

                        // Get the appropriate format string
                        if (eCatType == CATEGORY_TYPE_USER)
                        {
                            pcFormat = "    User Category (%u), Offset (%d): %s (%s)\n";
                        }
                        else if (eCatType == CATEGORY_TYPE_VIRTUAL)
                        {
                            pcFormat = "    Virtual Category (%u), Offset (%d): %s (%s)\n";
                        }
                        else if (eCatType == CATEGORY_TYPE_BROADCAST)
                        {
                            pcFormat = "    Broadcast Category (%u), Offset (%d): %s (%s)\n";
                        }

                        // Output the category info
                        fprintf(stdout,
                            pcFormat,
                            tCatId,
                            n16Offset,
                            &acLongCategoryName[0],
                            &acShortCategoryName[0]);
                    }
                }

                n32Return = CDO.n32FPrintf(psInfo->hCDO, stdout);
                if (n32Return < 0)
                {
                    printf("CDO.n32FPrintf returned error %d",
                                          n32Return);
                }

                // Print the content / album art info
                if ( CHANNEL_ART_AVAILABLE_IMAGE_NONE != 
                     ( psInfo->sContentArtInfo.tAvailableImages & CHANNEL_ART_AVAILABLE_IMAGE_ALBUM ) )
                {
                    fprintf(stdout,
                        "    Album Art:\n");
                    fprintf(stdout,
                        "    \t%s file name: %s\n",
                        &psInfo->sContentArtInfo.sAlbumInfo.acImageType[0],
                        &psInfo->sContentArtInfo.sAlbumInfo.acFilename[0]
                            );
                }

                // Print the full Program Guide Schedule
                // for this channel
                fprintf(stdout, "    Program Schedule:\n" );
                CHANNEL.eIteratePrograms(psInfo->hChannel,
                        bPrintEpgEvents, (void*)(size_t)CHANNEL_OBJECT_PRINT_GROSS);
                fprintf(stdout, "\n");

                // Print the channel art header if we have
                // some channel art available to us
                if (psInfo->sChannelArtInfo.tAvailableImages
                        != CHANNEL_ART_AVAILABLE_IMAGE_NONE)
                {
                    fprintf(stdout, "    Channel Art:\n");
                }

                // Print the logo art if it exists
                if ((psInfo->sChannelArtInfo.tAvailableImages
                     & CHANNEL_ART_AVAILABLE_IMAGE_LOGO)
                         == CHANNEL_ART_AVAILABLE_IMAGE_LOGO)
                {
                    fprintf(stdout,
                        "    \t%s file name: %s\n",
                        &psInfo->sChannelArtInfo.sLogoInfo.acImageType[0],
                        &psInfo->sChannelArtInfo.sLogoInfo.acFilename[0]
                            );
                    fprintf(stdout,
                        "    \t%s File Format: %s\n",
                        &psInfo->sChannelArtInfo.sLogoInfo.acImageType[0],
                        &psInfo->sChannelArtInfo.sLogoInfo.acFormat[0]
                            );
                    fprintf(stdout,
                        "    \t%s Uses Default Image: %s\n",
                        &psInfo->sChannelArtInfo.sLogoInfo.acImageType[0],
                        (psInfo->sChannelArtInfo.sLogoInfo.bUsingDefault == TRUE)?"Yes":"No"
                           );
                    fprintf(stdout,
                        "    \t%s Uses Background Color: ",
                        &psInfo->sChannelArtInfo.sLogoInfo.acImageType[0]
                           );

                    if (psInfo->sChannelArtInfo.sLogoInfo.
                            bUseBkgrndColor == TRUE)
                    {
                        fprintf(stdout, "Red: %u, Green: %u, Blue: %u\n",
                            psInfo->sChannelArtInfo.sLogoInfo.
                                sBkgrndColor.un8Red,
                            psInfo->sChannelArtInfo.sLogoInfo.
                                sBkgrndColor.un8Green,
                            psInfo->sChannelArtInfo.sLogoInfo.
                                sBkgrndColor.un8Blue
                                );
                    }
                    else
                    {
                        fprintf(stdout, "False\n");
                    }
                    fprintf(stdout,
                        "    \t%s Uses Line Bitmap: ",
                        &psInfo->sChannelArtInfo.sLogoInfo.acImageType[0]
                           );
                    if (psInfo->sChannelArtInfo.sLogoInfo.
                            bUseLineBitmap == TRUE)
                    {
                        fprintf(stdout, "Line Bitmap Entries: %u\n",
                            psInfo->sChannelArtInfo.sLogoInfo.
                                tLineBitmapSize);
                    }
                    else
                    {
                        fprintf(stdout, "False\n");
                    }
                }

                // Print the 2nd logo art if it exists
                if ((psInfo->sChannelArtInfo.tAvailableImages
                         & CHANNEL_ART_AVAILABLE_IMAGE_SECONDARY_LOGO)
                             == CHANNEL_ART_AVAILABLE_IMAGE_SECONDARY_LOGO)
                {
                    fprintf(stdout,
                        "    \t%s file name: %s\n",
                        &psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                            acImageType[0],
                        &psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                            acFilename[0]
                            );
                    fprintf(stdout,
                        "    \t%s File Format: %s\n",
                        &psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                            acImageType[0],
                        &psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                            acFormat[0]
                            );
                    fprintf(stdout,
                        "    \t%s Uses Default Image: %s\n",
                        &psInfo->sChannelArtInfo.sSecondaryLogoInfo.acImageType[0],
                        (psInfo->sChannelArtInfo.sSecondaryLogoInfo.bUsingDefault == TRUE)?"Yes":"No"
                           );
                    fprintf(stdout,
                        "    \t%s Uses Background Color: ",
                        &psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                            acImageType[0]
                           );
                    if (psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                            bUseBkgrndColor == TRUE)
                    {
                        fprintf(stdout, "Red: %u, Green: %u, Blue: %u\n",
                            psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                                sBkgrndColor.un8Red,
                            psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                                sBkgrndColor.un8Green,
                            psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                                sBkgrndColor.un8Blue
                                );
                    }
                    else
                    {
                        fprintf(stdout, "False\n");
                    }
                    fprintf(stdout,
                        "    \t%s Uses Line Bitmap: ",
                        &psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                            acImageType[0]
                           );
                    if (psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                            bUseLineBitmap == TRUE)
                    {
                        fprintf(stdout, "Line Bitmap Entries: %u\n",
                            psInfo->sChannelArtInfo.sSecondaryLogoInfo.
                                tLineBitmapSize);
                    }
                    else
                    {
                        fprintf(stdout, "False\n");
                    }
                }

                // Print the background image art if it exists
                if ((psInfo->sChannelArtInfo.tAvailableImages
                         & CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND)
                             == CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND)
                {
                    fprintf(stdout,
                        "    \t%s file name: %s\n",
                        &psInfo->sChannelArtInfo.sBackgroundInfo.
                            acImageType[0],
                        &psInfo->sChannelArtInfo.sBackgroundInfo.
                            acFilename[0]
                            );
                    fprintf(stdout,
                        "    \t%s File Format: %s\n",
                        &psInfo->sChannelArtInfo.sBackgroundInfo.
                            acImageType[0],
                        &psInfo->sChannelArtInfo.sBackgroundInfo.
                            acFormat[0]
                            );
                    fprintf(stdout,
                        "    \t%s Uses Default Image: %s\n",
                        &psInfo->sChannelArtInfo.sBackgroundInfo.acImageType[0],
                        (psInfo->sChannelArtInfo.sBackgroundInfo.bUsingDefault == TRUE)?"Yes":"No"
                           );
                    fprintf(stdout,
                        "    \t%s Uses Background Color: ",
                        &psInfo->sChannelArtInfo.sBackgroundInfo.
                            acImageType[0]
                           );
                    if (psInfo->sChannelArtInfo.sBackgroundInfo.
                            bUseBkgrndColor == TRUE)
                    {
                        fprintf(stdout, "Red: %u, Green: %u, Blue: %u\n",
                            psInfo->sChannelArtInfo.sBackgroundInfo.
                                sBkgrndColor.un8Red,
                            psInfo->sChannelArtInfo.sBackgroundInfo.
                                sBkgrndColor.un8Green,
                            psInfo->sChannelArtInfo.sBackgroundInfo.
                                sBkgrndColor.un8Blue
                                );
                    }
                    else
                    {
                        fprintf(stdout, "False\n");
                    }
                    fprintf(stdout,
                        "    \t%s Uses Line Bitmap: ",
                        &psInfo->sChannelArtInfo.sBackgroundInfo.
                            acImageType[0]
                           );
                    if (psInfo->sChannelArtInfo.sBackgroundInfo.
                            bUseLineBitmap == TRUE)
                    {
                        fprintf(stdout, "Line Bitmap Entries: %u\n",
                            psInfo->sChannelArtInfo.sBackgroundInfo.
                                tLineBitmapSize);
                    }
                    else
                    {
                        fprintf(stdout, "False\n");
                    }
                }

                // Print the category art if it exists
                if (psInfo->sCategoryArtInfo.tAvailableImage !=
                    CHANNEL_ART_AVAILABLE_IMAGE_NONE)
                {
                    fprintf(stdout,
                        "    Category Art:\n");
                    fprintf(stdout,
                        "    \t%s file name: %s\n",
                        &psInfo->sCategoryArtInfo.acImageType[0],
                        &psInfo->sCategoryArtInfo.acImageFilename[0]
                            );
                    fprintf(stdout,
                        "    \t%s File Format: %s\n",
                        &psInfo->sCategoryArtInfo.acImageType[0],
                        &psInfo->sCategoryArtInfo.acImageFormat[0]
                            );
                    fprintf(stdout,
                        "    \t%s Uses Default Image: %s\n",
                        &psInfo->sCategoryArtInfo.acImageType[0],
                        (psInfo->sCategoryArtInfo.bUsingDefault == TRUE)?"Yes":"No"
                           );
                }

                // Print preset info
                fprintf(stdout,
                    "    Preset Band: %s\n",
                    &psInfo->acPresetBandName[0]
                        );
                fprintf(stdout,
                    "    Preset Index: %s\n",
                    &psInfo->acPresetIndex[0]
                        );
                fprintf(stdout,
                    "    Preset Name: %s\n",
                    &psInfo->acPresetName[0]
                        );
            }
            break;

            default:
            break;
        }
    }

    return;
}

/*****************************************************************************
*
*   PLAYBACK_vPrintSongs
*
*       This friend function is used to tell PLAYBACK to print out all its songs
*
*       Inputs:
*           hPlayback - the playback obj whose is being modified
*           bVerbose -
*       Outputs:
*           none
*
*****************************************************************************/
void PLAYBACK_vPrintSongs(PLAYBACK_OBJECT hPlayback, BOOLEAN bVerbose)
{
    SCACHE_OBJECT hSCache;

    hSCache = PLAYBACK_hSCache(hPlayback);
    if(hSCache != SCACHE_INVALID_OBJECT)
    {
        BOOLEAN bDebugEnabled;

        // Remember output enable state
        bDebugEnabled = OSAL.bOutputEnabledThisTask();

        // Enable debug output
        OSAL.vControlOutputThisTask(TRUE);

        vPrintSongs(hSCache, bVerbose);

        // Disable output only if it was disabled when we started
        if(bDebugEnabled == FALSE)
        {
            // Disable debug output
            OSAL.vControlOutputThisTask(FALSE);
        }
    }

    return;
}

/*****************************************************************************
*
*   SPORTS_pacEnumText
*
* This local function translates a provided SPORTS_ENUM into a text
* string representation.
*
*****************************************************************************/
const char *SPORTS_pacEnumText(
    SPORTS_ENUM eType
        )
{
    const char *pacReturnString;

    switch (eType)
    {
        case SPORTS_FOOTBALL:
            pacReturnString = MACRO_TO_STRING(SPORTS_FOOTBALL);
        break;

        case SPORTS_HOCKEY:
            pacReturnString = MACRO_TO_STRING(SPORTS_HOCKEY);
        break;

        case SPORTS_BASKETBALL:
            pacReturnString = MACRO_TO_STRING(SPORTS_BASKETBALL);
        break;

        case SPORTS_BASEBALL:
            pacReturnString = MACRO_TO_STRING(SPORTS_BASEBALL);
        break;

        case SPORTS_SOCCER:
            pacReturnString = MACRO_TO_STRING(SPORTS_SOCCER);
        break;

        case SPORTS_AUTO_RACING:
            pacReturnString = MACRO_TO_STRING(SPORTS_AUTO_RACING);
        break;

        case SPORTS_OTHER:
            pacReturnString = MACRO_TO_STRING(SPORTS_OTHER);
        break;

        case SPORTS_UNKNOWN:
        default:
            pacReturnString = MACRO_TO_STRING(SPORTS_UNKNOWN);
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   SPORTS_pacBCastEnumText
*
* This local function translates a provided SPORTS_BROADCAST_ENUM into a text
* string representation.
*
*****************************************************************************/
const char *SPORTS_pacBCastEnumText(
    SPORTS_BROADCAST_ENUM eType
        )
{
    const char *pacReturnString;

    switch (eType)
    {
        case SPORTS_BROADCAST_NATIONAL:
            pacReturnString = MACRO_TO_STRING(SPORTS_BROADCAST_NATIONAL);
        break;

        case SPORTS_BROADCAST_TEAM:
            pacReturnString = MACRO_TO_STRING(SPORTS_BROADCAST_TEAM);
        break;

        case SPORTS_BROADCAST_OTHER:
            pacReturnString = MACRO_TO_STRING(SPORTS_BROADCAST_OTHER);
        break;

        case SPORTS_BROADCAST_UNKNOWN:
        default:
            pacReturnString = MACRO_TO_STRING(SPORTS_BROADCAST_UNKNOWN);
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   SONG_DEBUG_pacPrintStatus
*
* This local function translates a provided SMSAPI_SONG_STATUS_ENUM into a
* text string representation.
*
*****************************************************************************/
const char *SONG_pacStatusText(
    SMSAPI_SONG_STATUS_ENUM eStatus
        )
{
    const char *pacReturnString;

    switch (eStatus)
    {
        case SMSAPI_SONG_STATUS_INCOMPLETE:
            pacReturnString = "Incomplete";
        break;

        case SMSAPI_SONG_STATUS_COMPLETE:
            pacReturnString = "Complete";
        break;

        case SMSAPI_SONG_STATUS_UNKNOWN:
        default:
            pacReturnString = "Unknown";
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   SCAN_pacCmdText
*
*****************************************************************************/
const char *SCAN_pacCmdText (
    DECODER_TUNE_SCAN_CMD_ENUM eTuneScanCmd
        )
{
    const char *pacReturnSring = "Unknown";

    switch (eTuneScanCmd)
    {
    case DECODER_TUNE_SCAN_CMD_START:
        pacReturnSring = "Start";
        break;
    case DECODER_TUNE_SCAN_CMD_STOP:
        pacReturnSring = "Stop";
        break;
    case DECODER_TUNE_SCAN_CMD_ABORT:
        pacReturnSring = "Abort";
        break;
    case DECODER_TUNE_SCAN_CMD_SKIP_FORWARD:
        pacReturnSring = "Skip Forward";
        break;
    case DECODER_TUNE_SCAN_CMD_SKIP_BACKWARD:
        pacReturnSring = "Skip Backward";
        break;
    }

    return pacReturnSring;
}

/******************************************************************************
*
*   CM_DEBUG_ePrint
*
*   This function calls an SMS friend function to print out the config Tags.
*   It allows the cli to print the tags without having
*   access to the internal cm friend functions
*
******************************************************************************/
SMSAPI_RETURN_CODE_ENUM CM_DEBUG_ePrint(
    void
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;

    eReturn = CM_ePrintTags(stdout);

    return eReturn;
}

/******************************************************************************
*
*   DEVICE_DEBUG_ePrint
*
*   This function calls an SMS friend function to print out the config Tags.
*   It allows the cli to print Device object's information without having
*   access to the internal Device functions
*
******************************************************************************/
SMSAPI_RETURN_CODE_ENUM DEVICE_DEBUG_ePrint(
    void
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    N32 n32BytesWritten = 0;

    n32BytesWritten = DEVICE_n32FPrintf(stdout, SMS_OUTPUT_OPTION_TERSE);
    if( n32BytesWritten <= 0)
    {
        printf("DEVICE_n32FPrintf returned error %d",n32BytesWritten);
        eReturn = SMSAPI_RETURN_CODE_NO_OBJECTS;
    }

    return eReturn;
}

/*****************************************************************************
*
*   SMSAPI_DEBUG_vPrintChannels
*
*****************************************************************************/
void SMSAPI_DEBUG_vPrintChannels (
    DECODER_OBJECT hDecoder,
    BOOLEAN bVerbose
        )
{
    SMSAPI_DEBUG_CCACHE_CHANNEL_ITERATOR_STRUCT sIterator =
        {0, 0, 0, CHANNEL_OBJECT_PRINT_TERSE};
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Translate input
    if(bVerbose == TRUE)
    {
        sIterator.eMode = CHANNEL_OBJECT_PRINT_VERBOSE;
    }

    printf("\n");
    CHANNEL_DEBUG_vPrint(CHANNEL_INVALID_OBJECT, sIterator.eMode);
    LINE(101);
    eReturnCode = CHANNEL.eIterateAll(
        hDecoder, CCACHE_DEBUG_bPrintChannelIterator,
        NULL, &sIterator
            );
    LINE(101);
    if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        printf("%d channels in cache.\n\n\n",
            sIterator.n16NumChannels);
        printf("%d subscribed channels.\n",
            sIterator.n16NumSubscribedChannels);
        printf("%d unsubscribed channels.\n",
            sIterator.n16NumUnSubscribedChannels);
    }
    else
    {
        // Error!
        printf("Error! Unable to iterate channel cache.\n\n\n");
    }

    return;
}

/*****************************************************************************
*
*   SMSAPI_DEBUG_vPrintCategories
*
*****************************************************************************/
void SMSAPI_DEBUG_vPrintCategories (
    DECODER_OBJECT hDecoder,
    BOOLEAN bVerbose
        )
{
    SMSAPI_DEBUG_CCACHE_CATEGORY_ITERATOR_STRUCT sIterator;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    sIterator.n16NumCategories = 0;
    sIterator.bVerbose = bVerbose;

    LINE(90);
    eReturnCode = CATEGORY.eIterateAll(
        hDecoder, CCACHE_DEBUG_bPrintCategoryIterator,
        NULL, &sIterator
            );
    LINE(90);
    if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        printf("%d categories in cache.\n\n\n", sIterator.n16NumCategories);
    }
    else
    {
        // Error!
        printf("Error! Unable to iterate category cache.\n\n\n");
    }

    return;
}

#if SMS_DEBUG == 0
/*******************************************************************************
*
*   SMSAPI_DEBUG_vPrintErrorDummy
*
*   This function will ...
*
*******************************************************************************/
void SMSAPI_DEBUG_vPrintErrorDummy (
    const char *pcThisFile,
    UN32 un32LineNum,
    const char *pcMessage,
    ...
        )
{
    return;
}

#else /* #if SMS_DEBUG == 0 */
/*******************************************************************************
*
*   SMSAPI_DEBUG_vPrintErrorFull
*
*   This function will ...
*
*******************************************************************************/
void SMSAPI_DEBUG_vPrintErrorFull (
    const char *pcThisFile,
    UN32 un32LineNum,
    const char *pcMessage,
    ...
        )
{
    char *pcError = NULL;
    va_list tArgList;
    UN32 un32CharCount;

    // allocate a buffer dynamically and zero it out
    // so we don't have to worry about terminating the string later.
    pcError = (char*)OSAL.pvMemoryAllocate( "PrintErrorFull_Buffer",
        SMS_PRINT_ERROR_MSG_MAX_LEN, TRUE );
    if (NULL == pcError)
    {
        // if we failed to allocate our buffer,
        // we can't print the error message.
        fprintf(stderr, "Error: Error message not available (OUT_OF_MEMORY");
        return;
    }

    // Grab the variable argument list
    va_start(tArgList, pcMessage);

    // Append the formatted error message
    vsnprintf(pcError,
        SMS_PRINT_ERROR_MSG_MAX_LEN,
        pcMessage,
        tArgList);
    un32CharCount = strlen(pcError);

    // Done with varargs now
    va_end(tArgList);

    if( pcThisFile != NULL )
    {
        // print the file name and line number
        strncat (pcError, "\n",
            SMS_PRINT_ERROR_MSG_MAX_LEN - un32CharCount);
        un32CharCount += 1; // plus one character added
        strncat (pcError, pcThisFile,
            SMS_PRINT_ERROR_MSG_MAX_LEN - un32CharCount);
        un32CharCount += strlen (pcThisFile);

        if (un32LineNum != 0)
        {
            char acNumber[11] = "";    // UN32 cannot be longer than 4G
                                       // which is 10 decimal characters

            strncat (pcError, " at line ",
                SMS_PRINT_ERROR_MSG_MAX_LEN - un32CharCount);
            un32CharCount += 9; // Length of " at line " text

            // Converting line number into string
            snprintf(acNumber, 10, "%d", un32LineNum);
            strncat (pcError, acNumber,
                SMS_PRINT_ERROR_MSG_MAX_LEN - un32CharCount);
            un32CharCount += strlen(acNumber);
        }

        strncat (pcError, "\n",
            SMS_PRINT_ERROR_MSG_MAX_LEN - un32CharCount);
    }
    else
    {
        strncat (pcError, "\n",
            SMS_PRINT_ERROR_MSG_MAX_LEN - un32CharCount);
    }

    // Now, print it with debug level 0 to make sure it is mandatory printed
    SMSAPI_DEBUG_vPrint("ERROR", DEBUG_PRINT_LEVEL_ALWAYS, pcError);

    // free the buffer
    OSAL.vMemoryFree( pcError );

    return;
}

/*******************************************************************************
 *
 *  SMSAPI_DEBUG_vPrint
 *
 *  Outputs debug message for un8Level <= un8DebugPrintingDepth,
 *  if bDebugPrintSpecifiedDepthOnly == FALSE.
 *
 *  Outputs debug message for un8Level == un8DebugPrintingDepth,
 *  if bDebugPrintSpecifiedDepthOnly == TRUE.
 *
 *  Outputs task (thread) ID, if bDebugPrintSpecifiedDepthOnly == TRUE.
 *
 *******************************************************************************/
void SMSAPI_DEBUG_vPrint (
    const char *pcPrefix,
    UN8 un8Level,
    char *pcText,
    ...
        )
{
    // Checking if requested debug level is within allowed depth
    if (((un8Level <= un8DebugPrintLevel) &&
         (bDebugPrintSpecifiedLevelOnly == FALSE)) ||
        ((un8Level == un8DebugPrintLevel) &&
         (bDebugPrintSpecifiedLevelOnly == TRUE)) ||
        (un8Level == DEBUG_PRINT_LEVEL_ALWAYS))
    {
        OSAL_TASK_ID tThisTaskId = 0;
        BOOLEAN bPrintTaskId = FALSE;
        int iNumPrinted = 0;
        char acBuffer[BUFSIZ];
        va_list tArgList;

        // Check if Task ID output is requested
        if (TRUE == bDebugPrintTaskID)
        {
            bPrintTaskId = OSAL.bTaskGetId(&tThisTaskId);
        }

        // First print task id (if required) and prefix to the beginning
        if (TRUE == bPrintTaskId)
        {
            iNumPrinted += snprintf(&acBuffer[iNumPrinted],
                    BUFSIZ - iNumPrinted, "0x%04X | %s | ",
                    tThisTaskId, pcPrefix);
        }
        else
        {
            iNumPrinted += snprintf(&acBuffer[iNumPrinted],
                BUFSIZ - iNumPrinted, "%s | ", pcPrefix);
        }

        if (iNumPrinted > 0)
        {
            va_start(tArgList, pcText);

            // Now print additional arguments to the end
            vsnprintf(&acBuffer[iNumPrinted], BUFSIZ - iNumPrinted,
                pcText, tArgList);

            va_end(tArgList);
        }
        else
        {
            sprintf(acBuffer, "LOG OUTPUT ERROR!");
        }

        // Now output the resulting buffer to the standard output (if required)
        if (0 != (DEBUG_PRINT_OUTPUT_OPTIONS & DEBUG_PRINT_OPTION_STD_OUTPUT))
        {
            if (DEBUG_PRINT_LEVEL_ALWAYS == un8DebugPrintLevel)
            {
                fputs(acBuffer, stderr);
            }
            else
            {
                puts(acBuffer);
            }
        }

        // Output to the file located in the SMS temp path (if required)
        if (0 != (DEBUG_PRINT_OUTPUT_OPTIONS & DEBUG_PRINT_OPTION_FILE_OUTPUT))
        {
            char acFilePath[SMS_PATH_MAX_LENGTH];
            FILE *pFile = NULL;

            // Create file path
            OSAL.tFileSystemGetTempPath(SMS_PATH_MAX_LENGTH, acFilePath);
            strncat(acFilePath, DEBUG_PRINT_FILE_NAME, SMS_PATH_MAX_LENGTH);

            pFile = fopen(acFilePath, "a");
            if (NULL != pFile)
            {
                fputs(acBuffer, pFile);
                fputs("\n", pFile);
                fclose(pFile);
            }
        }
    }

    return;
}

#endif /* #if SMS_DEBUG == 0 ... #else ...*/

#if __STDC_VERSION__ < 199901L
/*******************************************************************************
 *
 * SMSAPI_DEBUG_vPrintDummy
 *
 * Dummy placeholder
 *
 *******************************************************************************/
void SMSAPI_DEBUG_vPrintDummy (
    const char *pcPrefix,
    UN8 un8Level,
    char *pcText,
    ...
        )
{
    return;
}

#endif

/*****************************************************************************
*
*   SMSAPI_DEBUG_vSetPrintLevel
*
*****************************************************************************/
void SMSAPI_DEBUG_vSetPrintLevel(
    UN8 un8PrintLevel
        )
{
    un8DebugPrintLevel = un8PrintLevel;

    return;
}

/*****************************************************************************
*
*   SMSAPI_DEBUG_vSetPrintSpecifiedLevelOnly
*
*****************************************************************************/
void SMSAPI_DEBUG_vSetPrintSpecifiedLevelOnly(
    BOOLEAN bOneLevelOnly
        )
{
    bDebugPrintSpecifiedLevelOnly = bOneLevelOnly;

    return;
}

/*****************************************************************************
*
*   SMSAPI_DEBUG_vSetPrintingDepth
*
*****************************************************************************/
void SMSAPI_DEBUG_vSetPrintTaskID(
    BOOLEAN bPrintTaskID
        )
{
    bDebugPrintTaskID = bPrintTaskID;

    return;
}

/*****************************************************************************
*
*   SMSAPI_DEBUG_vInitPrintingParameters
*
*****************************************************************************/
void SMSAPI_DEBUG_vInitPrintingParameters( void )
{
    un8DebugPrintLevel = DEBUG_PRINT_LEVEL;
    bDebugPrintSpecifiedLevelOnly = DEBUG_PRINT_DEFINED_LEVEL_ONLY;
    bDebugPrintTaskID = DEBUG_PRINT_TASK_ID;

    return;
}

#if (SMS_DEBUG == 1) || (__STDC_VERSION__ < 199901L)

/*****************************************************************************
*
*   SMSAPI_DEBUG_pacSRMErrorCodeText
*
*   This function prints text corresponding to the provided error code.
*
*   Inputs:
*      eError - the error code whose textural representaion is requested
*
*   Outputs:
*       pacReturnString
*
*****************************************************************************/
char const *SMSAPI_DEBUG_pacSRMErrorCodeText (
    SRM_ERROR_CODE_ENUM eError
        )
{
    char const *pacReturnString =
        MACRO_TO_STRING(SRM_ERROR_CODE_UNKNOWN);;

    switch (eError)
    {
        case SRM_ERROR_CODE_NONE:
            pacReturnString = MACRO_TO_STRING(SRM_ERROR_CODE_NONE);
        break;
        case SRM_ERROR_CODE_INVALID_OBJECT:
            pacReturnString = MACRO_TO_STRING(SRM_ERROR_CODE_INVALID_OBJECT);
        break;
        case SRM_ERROR_CODE_EVENT_POST_ERROR:
            pacReturnString = MACRO_TO_STRING(SRM_ERROR_CODE_EVENT_POST_ERROR);
        break;
        case SRM_ERROR_CODE_EVENT_ALLOCATION_ERROR:
            pacReturnString = MACRO_TO_STRING(SRM_ERROR_CODE_EVENT_ALLOCATION_ERROR);
        break;
        case SRM_ERROR_CODE_NO_CAPABLE_MODULE:
            pacReturnString = MACRO_TO_STRING(SRM_ERROR_CODE_NO_CAPABLE_MODULE);
        break;
        case SRM_ERROR_CODE_INITIALIZATION_ERROR:
            pacReturnString = MACRO_TO_STRING(SRM_ERROR_CODE_INITIALIZATION_ERROR);
        break;
        case SRM_ERROR_CODE_UNABLE_TO_REMOVE_SRM:
            pacReturnString = MACRO_TO_STRING(SRM_ERROR_CODE_UNABLE_TO_REMOVE_SRM);
        break;
        case SRM_ERROR_CODE_UNABLE_TO_LOCK_SMS:
            pacReturnString = MACRO_TO_STRING(SRM_ERROR_CODE_UNABLE_TO_LOCK_SMS);
        break;
        case SRM_ERROR_CODE_MODULE_LIST_ERROR:
            pacReturnString = MACRO_TO_STRING(SRM_ERROR_CODE_MODULE_LIST_ERROR);
        break;
        case SRM_ERROR_CODE_GENERAL:
            pacReturnString = MACRO_TO_STRING(SRM_ERROR_CODE_GENERAL);
        break;
        case SRM_ERROR_CODE_UNKNOWN:
            pacReturnString = MACRO_TO_STRING(SRM_ERROR_CODE_UNKNOWN);
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   SMSAPI_DEBUG_pacModuleErrorCodeText
*
*   This function prints text corresponding to the provided error code.
*
*   Inputs:
*      eError - the error code whose textural representaion is requested
*
*   Outputs:
*       pacReturnString
*
*****************************************************************************/
char const *SMSAPI_DEBUG_pacModuleErrorCodeText (
    MODULE_ERROR_CODE_ENUM eError
        )
{
    char const *pacReturnString =
        MACRO_TO_STRING(MODULE_ERROR_CODE_UNKNOWN);;

    switch (eError)
    {
        case MODULE_ERROR_CODE_NONE:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_NONE);
        break;
        case MODULE_ERROR_CODE_INVALID_OBJECT:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_INVALID_OBJECT);
        break;
        case MODULE_ERROR_CODE_INVALID_INPUT:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_INVALID_INPUT);
        break;
        case MODULE_ERROR_CODE_EVENT_POST_ERROR:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_EVENT_POST_ERROR);
        break;
        case MODULE_ERROR_CODE_EVENT_ALLOCATION_ERROR:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_EVENT_ALLOCATION_ERROR);
        break;
        case MODULE_ERROR_CODE_UNABLE_TO_RESET_SRM:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_UNABLE_TO_RESET_SRM);
        break;
        case MODULE_ERROR_CODE_NO_CAPABLE_DECODER:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_NO_CAPABLE_DECODER);
        break;
        case MODULE_ERROR_CODE_INITIALIZATION_ERROR:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_INITIALIZATION_ERROR);
        break;
        case MODULE_ERROR_CODE_UNABLE_TO_REMOVE_MODULE:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_UNABLE_TO_REMOVE_MODULE);
        break;
        case MODULE_ERROR_CODE_DECODER_LIST_ERROR:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_DECODER_LIST_ERROR);
        break;
        case MODULE_ERROR_CODE_UNABLE_TO_STOP_DECODER:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_UNABLE_TO_STOP_DECODER);
        break;
        case MODULE_ERROR_CODE_RADIO_OBJECT_ERROR:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_RADIO_OBJECT_ERROR);
        break;
        case MODULE_ERROR_CODE_RADIO_REQUESTED_RESET:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_RADIO_REQUESTED_RESET);
        break;
        case MODULE_ERROR_CODE_RADIO_COMM_FAILURE:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_RADIO_COMM_FAILURE);
        break;
        case MODULE_ERROR_CODE_RADIO_DEVICE_ERROR:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_RADIO_DEVICE_ERROR);
        break;
        case MODULE_ERROR_CODE_SRM_ERROR:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_SRM_ERROR);
        break;
        case MODULE_ERROR_CODE_FIRMWARE_UPDATE_PROGRAMMING_FAILED:
            pacReturnString = MACRO_TO_STRING(
                MODULE_ERROR_CODE_FIRMWARE_UPDATE_PROGRAMMING_FAILED);
        break;
        case MODULE_ERROR_CODE_FIRMWARE_UPDATE_PROGRAMMING_TIMEOUT:
            pacReturnString = MACRO_TO_STRING(
                MODULE_ERROR_CODE_FIRMWARE_UPDATE_PROGRAMMING_TIMEOUT);
        break;
        case MODULE_ERROR_CODE_FIRMWARE_UPDATE_CONFIGURE_FAILED:
            pacReturnString = MACRO_TO_STRING(
                MODULE_ERROR_CODE_FIRMWARE_UPDATE_CONFIGURE_FAILED);
        break;
        case MODULE_ERROR_CODE_FIRMWARE_UPDATE_ERASE_FAILED:
            pacReturnString = MACRO_TO_STRING(
                MODULE_ERROR_CODE_FIRMWARE_UPDATE_ERASE_FAILED);
        break;
        case MODULE_ERROR_CODE_FIRMWARE_UPDATE_ERASE_TIMEOUT:
            pacReturnString = MACRO_TO_STRING(
                MODULE_ERROR_CODE_FIRMWARE_UPDATE_ERASE_TIMEOUT);
        break;
        case MODULE_ERROR_CODE_FIRMWARE_UPDATE_PACKET_FAILED:
            pacReturnString = MACRO_TO_STRING(
                MODULE_ERROR_CODE_FIRMWARE_UPDATE_PACKET_FAILED);
        break;
        case MODULE_ERROR_CODE_FIRMWARE_UPDATE_PACKET_TIMEOUT:
            pacReturnString = MACRO_TO_STRING(
                MODULE_ERROR_CODE_FIRMWARE_UPDATE_PACKET_TIMEOUT);
        break;
        case MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL);
        break;
        case MODULE_ERROR_CODE_GENERAL:
            pacReturnString = MACRO_TO_STRING(
                MODULE_ERROR_CODE_GENERAL);
        break;
        case MODULE_ERROR_CODE_UNKNOWN:
            pacReturnString = MACRO_TO_STRING(MODULE_ERROR_CODE_UNKNOWN);
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   SMSAPI_DEBUG_pacDecoderErrorCodeText
*
*   This function prints text corresponding to the provided error code.
*
*   Inputs:
*      eError - the error code whose textural representaion is requested
*
*   Outputs:
*       pacReturnString
*
*****************************************************************************/
char const *SMSAPI_DEBUG_pacDecoderErrorCodeText (
    DECODER_ERROR_CODE_ENUM eError
        )
{
    char const *pacReturnString =
        MACRO_TO_STRING(DECODER_ERROR_CODE_UNKNOWN);

    switch (eError)
    {
        case DECODER_ERROR_CODE_NONE:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_NONE);
        break;
        case DECODER_ERROR_CODE_INVALID_OBJECT:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_INVALID_OBJECT);
        break;
        case DECODER_ERROR_CODE_EVENT_POST_ERROR:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_EVENT_POST_ERROR);
        break;
        case DECODER_ERROR_CODE_EVENT_ALLOCATION_ERROR:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_EVENT_ALLOCATION_ERROR);
        break;
        case DECODER_ERROR_CODE_UNABLE_TO_RESET_MODULE:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_UNABLE_TO_RESET_MODULE);
        break;
        case DECODER_ERROR_CODE_INITIALIZATION_ERROR:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_INITIALIZATION_ERROR);
        break;
        case DECODER_ERROR_CODE_RADIO_ERROR:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_RADIO_ERROR);
        break;
        case DECODER_ERROR_CODE_UNABLE_TO_REMOVE_DECODER:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_UNABLE_TO_REMOVE_DECODER);
        break;
        case DECODER_ERROR_CODE_CCACHE_ERROR:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_CCACHE_ERROR);
        break;
        case DECODER_ERROR_CODE_MODULE_ERROR:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_MODULE_ERROR);
        break;
        case DECODER_ERROR_CODE_TIMER_ERROR:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_TIMER_ERROR);
        break;
        case DECODER_ERROR_CODE_GENERAL:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_GENERAL);
        break;
        case DECODER_ERROR_CODE_UNKNOWN:
            pacReturnString = MACRO_TO_STRING(DECODER_ERROR_CODE_UNKNOWN);
        break;
    }

    return pacReturnString;
}

#endif

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
*
*   vPrintCategoryArtCallback
*
*   This is a callback function provided to
*   CHANNEL_ART_SERVICE.eUseChannelArt in order to get access to the necessary
*   art and print it.
*
*****************************************************************************/
static void vPrintCategoryArtCallback(
    CHANNEL_ART_OBJECT hArt,
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tAvailableImages,
    void *pvCategoryArtAccessCallbackArg
        )
{
    IMAGE_OBJECT hImage;
    BOOLEAN bUsingDefault = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    STRING_OBJECT hFilename;
    IMAGE_FORMAT_ENUM eImageFormat;
    CHANNEL_ART_IMAGETYPE_ENUM eImageType =
         CHANNEL_ART_IMAGETYPE_MAX;
    CATEGORY_ART_INFO_STRUCT *psInfo =
        (CATEGORY_ART_INFO_STRUCT *)pvCategoryArtAccessCallbackArg;

    // Validate that our info argument is not null (otherwise
    // there's no point in populating it.)
    if ( psInfo == (CATEGORY_ART_INFO_STRUCT *)NULL )
    {
        return;
    }

    // Categories may have a logo or a
    // background image; get which image is available
    psInfo->tAvailableImage = tAvailableImages;

    // Try the logo first
    if ( CHANNEL_ART_AVAILABLE_IMAGE_NONE !=
            ( tAvailableImages & CHANNEL_ART_AVAILABLE_IMAGE_LOGO ) )
    {
        eImageType = CHANNEL_ART_IMAGETYPE_LOGO;

        strncpy(&psInfo->acImageType[0],
                "Logo", sizeof(psInfo->acImageType));
    }
    // Then the background image
    else if ( CHANNEL_ART_AVAILABLE_IMAGE_NONE !=
              ( tAvailableImages & CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND) )
    {
        eImageType = CHANNEL_ART_IMAGETYPE_BKGRND;

        strncpy(&psInfo->acImageType[0],
                "Background", sizeof(psInfo->acImageType));
    }
    else
    {
        printf("Unexpected / unknown art found for category: %u\n",
                tAvailableImages);
    }

    // Get the Image
    eReturnCode = CHANNEL_ART.eImage( hArt, eImageType,
                        &hImage, &bUsingDefault );

    // Make sure it's not NULL / invalid
    if ( ( IMAGE_INVALID_OBJECT == hImage ) ||
          ( SMSAPI_RETURN_CODE_SUCCESS != eReturnCode ) )
    {
        // Odd, but there's not much we can do about it. We have to
        // bail out now, as the rest of these operations depend on
        // having a valid image.
        return;
    }

    // Store the default usage
    psInfo->bUsingDefault = bUsingDefault;

    // Get the image's file name
    hFilename = IMAGE.hFileName( hImage );

    if (hFilename != STRING_INVALID_OBJECT)
    {
        // Copy it for display
        STRING.tCopyToCStr(
            hFilename,
            &psInfo->acImageFilename[0],
            sizeof(psInfo->acImageFilename));
    }

    // Get the image file's format
    eImageFormat = IMAGE.eFormat( hImage );

    switch( eImageFormat )
    {
        case IMAGE_FORMAT_PNG:
        {
            strncpy(&psInfo->acImageFormat[0], PNG_NAME,
                sizeof(psInfo->acImageFormat));
        }
        break;

        case IMAGE_FORMAT_JPEG:
        {
            strncpy(&psInfo->acImageFormat[0], JPG_NAME,
                sizeof(psInfo->acImageFormat));
        }
        break;

        case IMAGE_INVALID_FORMAT:
        default:
        break;
    }

    return;
}

/*****************************************************************************
*
*   n16PrintChannelIterator
*
*****************************************************************************/
static N16 n16PrintChannelIterator(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        )
{
    CHANNEL_ID tChannelId;
    char acChannelId[6];
    UN32 *pun32NumberOfChannelsLeft = (UN32 *)pvIterateArg;

    // Channel Id
    tChannelId = CHANNEL.tChannelId(hChannel);
    if(tChannelId == CHANNEL_INVALID_ID)
    {
        strcpy(&acChannelId[0], "?");
    }
    else
    {
        snprintf(&acChannelId[0],
            sizeof(acChannelId), "%u", tChannelId);
    }

    // Print it.
    if((*pun32NumberOfChannelsLeft)-- > 1)
    {
        printf("%s,", &acChannelId[0]);
        return 1;
    }
    else
    {
        printf("%s\n", &acChannelId[0]);
        return 0;
    }
}

/*****************************************************************************
*
*   vPrintChannelArtCallback
*
*   A callback function provided to CHANNEL_ART_SERVICE.eUseChannelArt in
*   order to create formatted strings which will be used to print the
*   channel.
*
*****************************************************************************/
static void vPrintChannelArtCallback(
    CHANNEL_ART_OBJECT hArt,
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tAvailableImages,
    void *pvChannelArtAccessCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CHANNEL_ART_IMAGE_INFO_STRUCT *psImageInfo = NULL;
    IMAGE_OBJECT hImage;
    BOOLEAN bUsingDefault;
    const char *pcString = "";

    CHANNEL_ART_IMAGETYPE_ENUM eImageType = CHANNEL_ART_IMAGETYPE_MAX;
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tCurrentImage =
            CHANNEL_ART_AVAILABLE_IMAGE_LOGO;
    CHANNEL_INFO_STRUCT *psChanArtInfo =
        (CHANNEL_INFO_STRUCT *)pvChannelArtAccessCallbackArg;

    // Make sure our art info struct pointer is not a null
    if ( (CHANNEL_INFO_STRUCT*)NULL == psChanArtInfo )
    {
        return;
    }

    // Find out which image(s) are available
    // for this channel
    psChanArtInfo->sChannelArtInfo.tAvailableImages = tAvailableImages;

    // Iterate through the images, left-shifting
    // by one to get the next image mask
    for ( ; tCurrentImage <= CHANNEL_ART_AVAILABLE_IMAGE_ALBUM ;
           tCurrentImage <<= 1 )
    {
        // If the available image mask doesn't contain the current
        // image, skip to the next one.
        if ( CHANNEL_ART_AVAILABLE_IMAGE_NONE ==
                ( tAvailableImages & tCurrentImage ) )
        {
            continue;
        }

        // Is this the logo image?
        if ( tCurrentImage == CHANNEL_ART_AVAILABLE_IMAGE_LOGO )
        {
            eImageType = CHANNEL_ART_IMAGETYPE_LOGO;
            psImageInfo = &psChanArtInfo->sChannelArtInfo.sLogoInfo;
            pcString = LOGO_NAME;
        }
        // Secondary logo image?
        else if ( tCurrentImage
                    == CHANNEL_ART_AVAILABLE_IMAGE_SECONDARY_LOGO )
        {
            eImageType = CHANNEL_ART_IMAGETYPE_SECONDARY_LOGO;
            psImageInfo = &psChanArtInfo->sChannelArtInfo.sSecondaryLogoInfo;
            pcString = SECONDARY_LOGO_NAME;
        }
        // Background image?
        else if ( tCurrentImage
                    == CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND )
        {
            eImageType = CHANNEL_ART_IMAGETYPE_BKGRND;
            psImageInfo = &psChanArtInfo->sChannelArtInfo.sBackgroundInfo;
            pcString = BACKGR0UND_NAME;
        }
        // Album art image?
        else if ( tCurrentImage
                    == CHANNEL_ART_AVAILABLE_IMAGE_ALBUM )
        {
            eImageType = CHANNEL_ART_IMAGETYPE_ALBUM;
            psImageInfo = &psChanArtInfo->sChannelArtInfo.sAlbumInfo;

            strncpy(&psImageInfo->acImageType[0],
                    ALBUMART_NAME, sizeof(psImageInfo->acImageType));
        }
        // Print an error if we have unexpected art
        else
        {
            printf("Unknown art found for channel: %u\n",
                    tCurrentImage);
            continue;
        }

        if (psImageInfo == NULL)
        {
            printf("Image info is NULL for channel: %u\n",
                    tCurrentImage);
            continue;
        }

        strncpy(&psImageInfo->acImageType[0], pcString,
            sizeof(psImageInfo->acImageType));

        // Get the Image
        eReturnCode = CHANNEL_ART.eImage(hArt, eImageType,
                         &hImage, &bUsingDefault);

        // Make sure it's not NULL / invalid
        if ( ( IMAGE_INVALID_OBJECT == hImage ) ||
              ( SMSAPI_RETURN_CODE_SUCCESS != eReturnCode ))
        {
            // Odd, but there's not much we can do about it. We have to
            // bail out now, as the rest of these operations depend on
            // having a valid image.
            continue;
        }

        // Store the image's default status
        psImageInfo->bUsingDefault = bUsingDefault;

        // Get the image's file name
        psImageInfo->hFilename = IMAGE.hFileName( hImage );

        if ( psImageInfo->hFilename != STRING_INVALID_OBJECT )
        {
            // Copy it for display
            STRING.tCopyToCStr(
                    psImageInfo->hFilename,
                    &psImageInfo->acFilename[0],
                    sizeof(psImageInfo->acFilename));
        }

        // Get the image's background display rules
        eReturnCode = IMAGE.eImageBackgroundDisplayRules(
            hImage,
            &psImageInfo->bUseBkgrndColor,
            &psImageInfo->bUseLineBitmap,
            &psImageInfo->tLineBitmapSize,
            &psImageInfo->sBkgrndColor);

        if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
        {
            // Ensure we don't print this info
            psImageInfo->bUseBkgrndColor = FALSE;
            psImageInfo->bUseLineBitmap = FALSE;
        }

        // Get the image file's code type
        psImageInfo->eFormat = IMAGE.eFormat( hImage );

        switch( psImageInfo->eFormat )
        {
            case IMAGE_FORMAT_PNG:
            {
                strncpy(&psImageInfo->acFormat[0], PNG_NAME,
                    sizeof(psImageInfo->acFormat));
            }
            break;

            case IMAGE_FORMAT_JPEG:
            {
                strncpy(&psImageInfo->acFormat[0], JPG_NAME,
                    sizeof(psImageInfo->acFormat));
            }
            break;

            case IMAGE_INVALID_FORMAT:
            default:
            {
                printf("Invalid format found: %u\n",
                        psImageInfo->eFormat);

            }
            break;
        }
    }

    return;
}

/*****************************************************************************
*
*   vPrintContentArtCallback
*
*   A callback function provided to CHANNEL_ART_SERVICE.eUseArt in
*   order to create formatted strings which will be used to print the
*   content / album art info.
*
*****************************************************************************/
static void vPrintContentArtCallback(
    CHANNEL_ART_OBJECT hArt,
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tAvailableImages,
    void *pvContentArtAccessCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CHANNEL_ART_IMAGE_INFO_STRUCT *psImageInfo = NULL;
    IMAGE_OBJECT hImage;
    BOOLEAN bUsingDefault;

    CHANNEL_ART_IMAGETYPE_ENUM eImageType = CHANNEL_ART_IMAGETYPE_MAX;
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tCurrentImage =
            CHANNEL_ART_AVAILABLE_IMAGE_LOGO;
    CHANNEL_INFO_STRUCT *psArtInfo =
        (CHANNEL_INFO_STRUCT *)pvContentArtAccessCallbackArg;

    // Make sure our art info struct pointer is not a null
    if ( (CHANNEL_INFO_STRUCT*)NULL == psArtInfo )
    {
        return;
    }

    // Find out which image(s) are available
    // for this channel
    psArtInfo->sContentArtInfo.tAvailableImages = tAvailableImages;

    // Iterate through the images, left-shifting 
    // by one to get the next image mask
    for ( ; tCurrentImage <= CHANNEL_ART_AVAILABLE_IMAGE_ALBUM ;
           tCurrentImage <<= 1 )
    {
        // If the available image mask doesn't contain the current
        // image, skip to the next one.
        if ( CHANNEL_ART_AVAILABLE_IMAGE_NONE ==
                ( tAvailableImages & tCurrentImage ) )
        {
            continue;
        }

        // Is this the album art image?
        if ( tCurrentImage == CHANNEL_ART_AVAILABLE_IMAGE_ALBUM )
        {
            eImageType = CHANNEL_ART_IMAGETYPE_ALBUM;
            psImageInfo = &psArtInfo->sContentArtInfo.sAlbumInfo;

            strncpy(&psImageInfo->acImageType[0],
                    ALBUMART_NAME, sizeof(psImageInfo->acImageType));
        }
        // Print an error if we have unexpected art
        else
        {
            printf("Unknown album art found for channel: %u\n",
                    tCurrentImage);
        }

        // Get the Image
        eReturnCode = CHANNEL_ART.eImage(hArt, eImageType,
                         &hImage, &bUsingDefault);

        // Make sure it's not NULL / invalid
        if ( ( IMAGE_INVALID_OBJECT == hImage ) ||
              ( SMSAPI_RETURN_CODE_SUCCESS != eReturnCode ))
        {
            // Odd, but there's not much we can do about it. We have to
            // bail out now, as the rest of these operations depend on
            // having a valid image.
            continue;
        }

        if (NULL == psImageInfo)
        {
            // Error. Something really bad happened
            break;
        }

        // Store the image's default status
        psImageInfo->bUsingDefault = bUsingDefault;

        // Get the image's file name
        psImageInfo->hFilename = IMAGE.hFileName( hImage );

        if ( psImageInfo->hFilename != STRING_INVALID_OBJECT )
        {
            // Copy it for display
            STRING.tCopyToCStr(
                    psImageInfo->hFilename,
                    &psImageInfo->acFilename[0],
                    sizeof(psImageInfo->acFilename));
        }

        // Get the image file's code type
        psImageInfo->eFormat = IMAGE.eFormat( hImage );

        switch( psImageInfo->eFormat )
        {
            case IMAGE_FORMAT_PNG:
            {
                strncpy(&psImageInfo->acFormat[0], PNG_NAME,
                    sizeof(psImageInfo->acFormat));
            }
            break;

            case IMAGE_FORMAT_JPEG:
            {
                strncpy(&psImageInfo->acFormat[0], JPG_NAME,
                    sizeof(psImageInfo->acFormat));
            }
            break;

            case IMAGE_INVALID_FORMAT:
            default:
            {
                printf("Invalid format found: %u\n",
                        psImageInfo->eFormat);

            }
            break;
        }
    }

    return;
}

/*****************************************************************************
*
*   pacLeagueEnumText
*
* This is a local function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacLeagueEnumText(
    LEAGUE_ENUM eType
        )
{
    const char *pacReturnString;

    switch (eType)
    {
        case LEAGUE_NFL:
            pacReturnString = MACRO_TO_STRING(LEAGUE_NFL);
        break;

        case LEAGUE_MLB:
            pacReturnString = MACRO_TO_STRING(LEAGUE_MLB);
        break;

        case LEAGUE_NBA:
            pacReturnString = MACRO_TO_STRING(LEAGUE_NBA);
        break;

        case LEAGUE_NHL:
            pacReturnString = MACRO_TO_STRING(LEAGUE_NHL);
        break;

        case LEAGUE_COLLEGE_FOOTBALL:
            pacReturnString = MACRO_TO_STRING(LEAGUE_COLLEGE_FOOTBALL);
        break;

        case LEAGUE_COLLEGE_BASKETBALL:
            pacReturnString = MACRO_TO_STRING(LEAGUE_COLLEGE_BASKETBALL);
        break;

        case LEAGUE_WOMENS_COLLEGE_BASKETBALL:
            pacReturnString =
                MACRO_TO_STRING(LEAGUE_WOMENS_COLLEGE_BASKETBALL);
        break;

        case LEAGUE_COLLEGE_OTHER:
            pacReturnString = MACRO_TO_STRING(LEAGUE_COLLEGE_OTHER);
        break;

        case LEAGUE_AUTORACING:
            pacReturnString = MACRO_TO_STRING(LEAGUE_AUTORACING);
        break;

        case LEAGUE_SOCCER:
            pacReturnString = MACRO_TO_STRING(LEAGUE_SOCCER);
        break;

        case LEAGUE_UNKNOWN:
            pacReturnString = MACRO_TO_STRING(LEAGUE_UNKNOWN);
        break;

        default:
            pacReturnString = "unknown";
        break;
    }

    return pacReturnString;
}

#if SMS_DEBUG == 1
/*****************************************************************************
*
*   bPrintSongIterator
*
*****************************************************************************/
static BOOLEAN bPrintSongIterator(
    SONG_OBJECT hSong,
    void *pvArg
        )
{
    BOOLEAN *pbVerbose = (BOOLEAN *)pvArg;
    SONG_vPrint(hSong, *pbVerbose);
    return TRUE;
}
#endif

/*****************************************************************************
*
*   vPrintSongs
*
*****************************************************************************/
static void vPrintSongs (
    SCACHE_OBJECT hSCache,
    BOOLEAN bVerbose
        )
{
    if(hSCache != SCACHE_INVALID_OBJECT)
    {
        UN32 un32NumSongs;
        N16 n16Offset;

        un32NumSongs = SCACHE_n16NumberOfSongs(hSCache);
        n16Offset = SCACHE_n16CurrentSongOffset(hSCache);

        printf("\nSONG CACHE: %u songs\n", un32NumSongs);
        SONG_vPrint(SONG_INVALID_OBJECT, bVerbose);
        printf("========================================"
               "========================================\n");
        SCACHE_bIterateSongs(hSCache, bPrintSongIterator, &bVerbose);
        printf("========================================"
               "========================================\n");
        printf("%u is the current song.\n\n\n", n16Offset);
    }

    return;
}

/*******************************************************************************
*
* bPrintSimilarChannels
*
*******************************************************************************/
static BOOLEAN bPrintSimilarChannels(
    CHANNEL_OBJECT hChannel, void *pvArg)
{
    CHANNEL_ID tChannelId;

    tChannelId = CHANNEL.tChannelId(hChannel);
    fprintf(stdout, " %u", tChannelId);

    return TRUE;
}

/*******************************************************************************
*
* bPrintEpgEvents
*
*******************************************************************************/
static BOOLEAN bPrintEpgEvents(
    PROGRAM_OBJECT hEpgEvent,
    void *pvArg
        )
{
    CHANNEL_OBJECT_PRINT_MODE_ENUM eMode =
                (CHANNEL_OBJECT_PRINT_MODE_ENUM) pvArg;
    TIME_T tStartTime = 0, tEndTime = 0, tCurTime = 0;
    UN32 un32CurTime;

    STRING_OBJECT hProgram = STRING_INVALID_OBJECT,
                  hEpisode = STRING_INVALID_OBJECT;
    BOOLEAN bResult = FALSE;

    switch(eMode)
    {
        case CHANNEL_OBJECT_PRINT_TERSE:
            break;

        case CHANNEL_OBJECT_PRINT_VERBOSE:
            OSAL.eTimeGetLocal(&un32CurTime);
            tCurTime = (TIME_T)un32CurTime;
            tStartTime = PROGRAM.tStartTime(hEpgEvent);
            tEndTime = PROGRAM.tEndTime(hEpgEvent);
            if( tStartTime <= tCurTime && tCurTime < tEndTime )
            {
                char acBuffer[100];
                char acEnd[30];
                OSAL.ctime_r(&tStartTime, acBuffer);
                OSAL.ctime_r(&tEndTime, acEnd);
                hProgram = STRING.hCreate("",0);
                bResult = PROGRAM.bGetShortName(hEpgEvent, hProgram);
                if (bResult == TRUE)
                {
                    hEpisode = STRING.hCreate("",0);
                    bResult = PROGRAM.bGetProgramDescription(hEpgEvent, hEpisode);
                    if( bResult == TRUE)
                    {
                        fprintf(stdout, "\n\t[%s - %s] - %s (%s)",
                            acBuffer,
                            acEnd,
                            STRING.pacCStr(hProgram),
                            STRING.pacCStr(hEpisode));
                    }
                    else
                    {
                        fprintf(stdout, "\n\t[%s - %s] - %s",
                            acBuffer,
                            acEnd,
                            STRING.pacCStr(hProgram));
                    }
                }
                else
                {
                    printf( "Program name is absent\n");
                }

                STRING.vDestroy(hProgram);
                STRING.vDestroy(hEpisode);
            }
            break;

        case CHANNEL_OBJECT_PRINT_GROSS:
            {
                N32 n32Return;
                n32Return = PROGRAM.n32FPrintf(hEpgEvent, stdout);
                if (n32Return < 0)
                {
                    printf("PROGRAM.n32FPrintf returned error %d",n32Return);
                }
            }
            break;
    }
    return TRUE;
}

#if 0
/*******************************************************
*
* SMSAPI_DEBUG_fFixed2Float
*
* The purpose of this function is to convert OSAL FIXED
* number to float type.
*
* Inputs:
*   hFixed - The handle of OSAL FIXED object
*
* Outputs:
*   float value
*
********************************************************/
float SMSAPI_DEBUG_fFixed2Float(OSAL_FIXED_OBJECT hFixed)
{
    float fReturnValue = 0.0;

    if(hFixed != OSAL_INVALID_OBJECT_HDL)
    {
        fReturnValue = (float)OSAL_FIXED.n32Value(hFixed) / (float)(1UL << OSAL_FIXED.un8NumFractionalBits(hFixed));
    }

    return fReturnValue;
}
#endif
