/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Presets debug implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include <stdio.h>
#include "osal.h"

#include "_presets_debug.h"
#include "presets_debug.h"

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *       PRESETS_DEBUG_n32PrintPresetsService
 *
 *       This API is used to print the presets service to a file.  This
 *       function is meant to aid in debugging and is not meant to assist
 *       the presets service to persist across power cycles.
 *
 *       Inputs:
 *           *psPresetsAttrs - A pointer to a PRESETS_DEBUG_ATTRS_STRUCT
 *              contains the attributes of the presets service to print.
 *           *psFile - A pointer to a device which is to receive formatted
 *               text output.
 *
 *       Outputs:
 *           The number of characters written.
 *
 *****************************************************************************/
N32 PRESETS_DEBUG_n32PrintPresetsService (
    PRESETS_DEBUG_ATTRS_STRUCT *psPresetsAttrs,
    FILE *psFile
        )
{
    N32 n32Return, n32Temp;

    // Verify inputs
    if ((psFile == NULL) || (psPresetsAttrs == NULL))
    {
        return 0;
    }

    // First, print out top-level presets information
    n32Return = fprintf(
        psFile, "Presets: hPresets = 0x%p\n",
        psPresetsAttrs->hPresets);


    // Provide the long name
    if (psPresetsAttrs->hLongName != STRING_INVALID_OBJECT)
    {
        // Print some info text
        n32Return += fprintf(
            psFile,
            "\t Long name: ");

        // Print the name
        n32Temp = STRING.n32FWrite(
            psPresetsAttrs->hLongName,
            psFile);

        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }

        // Finish the line
        n32Return += fprintf(
            psFile,
            "\n");
    }

    // Provide the short name
    if (psPresetsAttrs->hShortName != STRING_INVALID_OBJECT)
    {
        // Print some info text
        n32Return += fprintf(
            psFile,
            "\t Short name: ");

        // Print the name
        n32Temp = STRING.n32FWrite(
            psPresetsAttrs->hShortName,
            psFile);

        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }

        // Finish the line
        n32Return += fprintf(
            psFile,
            "\n");
    }

    // Print the category Id
    n32Return += fprintf(
        psFile,
        "\t Category Id: %u\n",
        psPresetsAttrs->tCategoryId);

    // Print the number of bands
    n32Return += fprintf(psFile,"\t Number of Bands: %u\n",
        psPresetsAttrs->tNumberOfBands);

    n32Return += fprintf(psFile, "\t Modes:%s%s\n",
        (psPresetsAttrs->bFeaturedFavoritesEnabled == TRUE) ? " Featured" : "",
        (psPresetsAttrs->tCategorySyncId != CATEGORY_INVALID_ID) ? " Smart" : ""
            );

    return n32Return;
}

/*****************************************************************************
 *
 *       PRESETS_DEBUG_n32PrintBand
 *
 *       This API is used to print a preset band to a  file.  This
 *       function is meant to aid in debugging and is not meant to assist
 *       the presets service to persist across power cycles.
 *
 *       Inputs:
 *           *psBandAttr - A pointer to a PRESETS_DEBUG_BAND_ATTR_STRUCT
 *              contains the attributes of the preset band to print.
 *           *psFile - A pointer to a device which is to receive formatted
 *               text output.
 *
 *       Outputs:
 *           The number of characters written.
 *
 *****************************************************************************/
N32 PRESETS_DEBUG_n32PrintBand (
    PRESETS_DEBUG_BAND_ATTR_STRUCT *psBandAttr,
    FILE *psFile,
    const char *pacFormat
        )
{
    N32 n32CharsWritten = 0, n32Temp;

    // Verify inputs
    if (   (psBandAttr == NULL)
        || (psFile == NULL)
        || (pacFormat == NULL)
        )
    {
        return 0;
    }

    // Print the band handle
    n32CharsWritten = fprintf(
        psFile,
        "\n%s Band: hBand = 0x%p\n",
        pacFormat,
        psBandAttr->hBand);

    // Print the band name
    n32CharsWritten += fprintf(
        psFile, "%s\t",
        pacFormat);

    // Print header
    n32CharsWritten += fprintf( psFile, " Name: ");

    // Print the name
    n32Temp = STRING.n32FWrite(
        psBandAttr->hName,
        psFile );
    if( n32Temp > 0 )
    {
        n32CharsWritten += n32Temp;
    }

    n32CharsWritten += fprintf(psFile, "\n");

    // Print the long name
    n32CharsWritten += fprintf( psFile, "\t Long Name: ");

    n32Temp = STRING.n32FWrite(
        psBandAttr->hLongName,
        psFile );
    if( n32Temp > 0 )
    {
        n32CharsWritten += n32Temp;
    }

    n32CharsWritten += fprintf(psFile, "\n");

    // Print the Verbose name
    n32CharsWritten += fprintf( psFile, "\t Verbose Name: ");

    n32Temp = STRING.n32FWrite(
        psBandAttr->hVerbose,
        psFile );
    if( n32Temp > 0 )
    {
        n32CharsWritten += n32Temp;
    }

    n32CharsWritten += fprintf(psFile, "\n");

    // Print the description
    n32CharsWritten += fprintf( psFile, "\t Description: ");

    n32Temp = STRING.n32FWrite(
        psBandAttr->hDescription,
        psFile );
    if( n32Temp > 0 )
    {
        n32CharsWritten += n32Temp;
    }

    n32CharsWritten += fprintf(psFile, "\n");

    // Print the band capacity
    n32CharsWritten += fprintf(
        psFile,
        "%s\t Capacity: %u\n",
        pacFormat,
        psBandAttr->tCapacity);

    // Print the Order
    n32CharsWritten += fprintf(
        psFile,
        "%s Order: %u\n",
        pacFormat,
        psBandAttr->tOrder);

    // Print the Loaded Flag
    n32CharsWritten += fprintf(
        psFile,
        "%s Type: %s\n",
        pacFormat,
        ((psBandAttr->eType ==
            PRESET_BAND_TYPE_USER) ?
                "USER" : "FEATURED"));

    // Print the State
    n32CharsWritten += fprintf(
        psFile,
        "%s Removed: %s\n",
        pacFormat,
       (psBandAttr->bRemoved == TRUE) ? "Yes" : "No");

    // Print the Current Band sign
    n32CharsWritten += fprintf(
        psFile,
        "%s Current: %s\n",
        pacFormat,
       (psBandAttr->bCurrent == TRUE) ? "Yes" : "No");

    // Print the ID
    n32CharsWritten += fprintf(
        psFile,
        "%s ID: %u\n",
        pacFormat,
        psBandAttr->tId);

    // Print the ID
    n32CharsWritten += fprintf(
        psFile,
        "%s BandID: %u\n",
        pacFormat,
        psBandAttr->tBandId);

    // Print the sequence
    n32CharsWritten += fprintf(
        psFile,
        "%s Sequence: %u\n",
        pacFormat,
        psBandAttr->tSequence);

    // Print the Purpose
    n32CharsWritten += fprintf(
        psFile,
        "%s Purpose: %u\n",
        pacFormat,
        psBandAttr->tPurpose);

    return n32CharsWritten;
}

/*****************************************************************************
 *
 *       PRESETS_DEBUG_n32PrintPreset
 *
 *       This API is used to print a preset to a  file.  This function
 *       is meant to aid in debugging and is not meant to assist the
 *       presets service to persist across power cycles.
 *
 *       Inputs:
 *           *psPresetAttr - A pointer to a PRESETS_DEBUG_PRESET_ATTR_STRUCT
 *              contains the attributes of the preset to print.
 *           *psFile - A pointer to a device which is to receive formatted
 *               text output.
 *
 *       Outputs:
 *           The number of characters written.
 *
 *****************************************************************************/
N32 PRESETS_DEBUG_n32PrintPreset (
    PRESETS_DEBUG_PRESET_ATTR_STRUCT *psPresetAttr,
    FILE *psFile,
    const char *pacFormat
        )
{
    N32 n32CharsWritten, n32Temp;

    // Verify inputs
    if (   (psPresetAttr == NULL)
        || (psFile == NULL)
        || (pacFormat == NULL)
        )
    {
        return 0;
    }

    // Print preset header
    n32CharsWritten = fprintf(
        psFile,
        "%s\tPreset %u:\n",
        pacFormat,
        psPresetAttr->tPresetIndex);

	// Start printing line for channel number
    n32CharsWritten += fprintf(
        psFile,
        "%s\t\tServiceId Id: ",
        pacFormat);

    // Print current value of this preset's channel id
    if (psPresetAttr->tServiceId == SERVICE_INVALID_ID)
    {
        n32CharsWritten += fprintf(
            psFile,
            "Not Set \n");
    }
    else
    {
        n32CharsWritten += fprintf(
            psFile,
            "%u \n",
            psPresetAttr->tServiceId);
    }

    // Start printing line for channel number
    n32CharsWritten += fprintf(
        psFile,
        "%s\t\tChannel Id: ",
        pacFormat);

    // Print current value of this preset's channel id
    if (psPresetAttr->tChannelId == CHANNEL_INVALID_ID)
    {
        n32CharsWritten += fprintf(
            psFile,
            "Not Set ");
    }
    else
    {
        n32CharsWritten += fprintf(
            psFile,
            "%u ",
            psPresetAttr->tChannelId);
    }

    if (psPresetAttr->hChannelName != STRING_INVALID_OBJECT)
    {
        n32CharsWritten += fprintf(
            psFile,
            "(%s)",
            STRING.pacCStr(psPresetAttr->hChannelName)
                );
    }

    n32CharsWritten += fprintf(psFile, "\n");

    // Start printing line for preset name
    n32CharsWritten += fprintf(
        psFile,
        "%s\t\tName: ",
        pacFormat);

    // Print current value of this preset's name
    if (psPresetAttr->hName == STRING_INVALID_OBJECT)
    {
        n32CharsWritten += fprintf(
            psFile,
            "Not Set \n");
    }
    else
    {
        // Print the preset name
        n32Temp = STRING.n32FWrite(
            psPresetAttr->hName,
            psFile);
        if (n32Temp > 0)
        {
            n32CharsWritten += n32Temp;
        }

        // Finish the line
        n32CharsWritten += fprintf(
            psFile,
            "\n");
    }

    return n32CharsWritten;
}

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/
