/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SMS_APP_H_
#define _SMS_APP_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Local constants; these could go in a separate "configuration" header file, but
// we leave them here to keep this example simple. A good rule of thumb is to
// start the task size bigger, and shrink later as needed.

#define SXM_RADIO_NAME              "radio:"
#define SRH_DRIVER_NAME             "srh:"
#define DECODER_NAME                "Audio"
#define SXM_INITIALIZER_FILE        "default_sms.cfg"
#define LOG_HEADER_COLUMN_LEN_MAX   (30)
#define LOG_ACTION_COLUMN_LEN_MAX   (45)
#define LOG_STATE_COLUMN_LEN_MAX    (16)
#define SMSD_TASK_OPTIONS           (OSAL_TASK_OPTION_DEBUG_OUTPUT)
#define SMSD_TASK_STACK_SIZE        (32768)
#define INDENT_WIDTH                (3)
#define INNER_CYCLE_SEPARATOR       ("*************************************************************************************************")
#define OUTER_CYCLE_SEPARATOR       ("=================================================================================================")
#define MID_CYCLE_SEPARATOR         ("-------------------------------------------------------------------------------------------------")
#define OUTER_CYCLER_TEST_STOP      ("**********************  C Y C L E R    T E S T    I S    C O M P L E T E D  *********************")
#define OUTER_CYCLER_TASK_STOP      ("************************************* CYCLER TASK HAS EXITED ************************************")

// Data Services related constants
#define CHANNEL_ART_AVAILABLE_IMAGE_TYPE_ALL (    \
    (CHANNEL_ART_AVAILABLE_IMAGE_MASK)            \
     CHANNEL_ART_AVAILABLE_IMAGE_LOGO           | \
     CHANNEL_ART_AVAILABLE_IMAGE_SECONDARY_LOGO | \
     CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND     | \
     CHANNEL_ART_AVAILABLE_IMAGE_ALBUM            \
    )
#define TRAFFIC_DSI             (480)
#define EPG_NUM_SEGMENTS_MAX    (8)
#define MILLISECONDS_IN_SECOND  (1000)

// Test default runtime related definitions

// Default values for cycler when no special parameters are specified
#define DEFAULT_ASSET_START_INTERVAL    (0)     // milliseconds
#define DEFAULT_ASSET_STOP_INTERVAL     (0)     // milliseconds
#define DEFAULT_OPERATIONAL_TIME        (0)     // milliseconds
#define DEFAULT_START_WAIT_CONDITION    (TRUE)
#define DEFAULT_STOP_WAIT_CONDITION     (TRUE)
#define DEFAULT_TUNE_CHANNEL_ID         (1)

// Default values for random testing
#define ASSET_START_INTERVAL_MAX        (1000)  // milliseconds
#define ASSET_STOP_INTERVAL_MAX         (1000)  // milliseconds
#define RANDOM_OPERATIONAL_TIME_MAX     (30000) // milliseconds

  /**************/
 /** TYPEDEFS **/
/**************/

// An enumeration of the event-generating assets
// used by our application.
// IMPORTANT NOTE:
// When a new asset is added, its definition should be added
// to the gasAssetsInfo as well. Position in the gasAssetsInfo array
// must correspond to the appropriate index in this enum.
typedef enum sms_app_asset_enum
{
    FIRST_ASSET = 0,

    // Mandatory assets
    SMS_ASSET = FIRST_ASSET,

    SRM_ASSET,

    MODULE_ASSET,

    DECODER_ASSET,
    DECODER_TUNE_ASSET,

    ART_SERVICE_ASSET,
    CHANNEL_ART_PRODUCT_ASSET,
    ALBUM_ART_PRODUCT_ASSET,

    // Optional assets
    EPG_ASSET,

    TRAFFIC_ASSET,

    MOVIE_LISTINGS_ASSET,

    FUEL_PRICE_SERVICE_ASSET,
    FUEL_DB_UPDATE_PRODUCT_ASSET,
    FUEL_PRICE_PRODUCT_ASSET,

    EV_SERVICE_ASSET,
    EV_DB_UPDATE_PRODUCT_ASSET,
    EV_STATION_LOGO_PRODUCT_ASSET,
    EV_AVAILABILITY_PRODUCT_ASSET,

    SPORTS_SERVICE_ASSET,

    GRAPHICAL_WEATHER_ASSET,

    TABULAR_WEATHER_ASSET,

    STOCK_TICKER_ASSET,

    SAFETY_CAMERAS_ASSET,

    TRAFFIC_CAMERAS_ASSET,

    WS_ALERTS_ASSET,

    MAPS_ASSET,

    PHONETICS_ASSET,

    // Number of assets
    NUM_ASSETS

} SMS_APP_ASSET_ENUM;

// An enumeration of the states these assets can be in
// that we'd like to act on. (Note: there are others, we
// just care about starting, stopping, and errors.)
typedef enum sms_app_asset_state_enum
{
    ASSET_STATE_STOPPED = 0,
    ASSET_STATE_STARTING,
    ASSET_STATE_START_FAILED,
    ASSET_STATE_STARTED,
    ASSET_STATE_STOPPING,
    ASSET_STATE_ERROR,

    ASSET_NUM_STATES

} SMS_APP_ASSET_STATE_ENUM;

// Typedefs for SMS object state changes, sent from event handlers
// to the application task, as well as a message for starting SMS.
// SMS_APP_MSG_MAX is used as a sentinel value.
typedef enum sms_app_msg_enum
{
    SMS_APP_MSG_START_CYCLE = 0,
    SMS_APP_MSG_ASSET_STATE,
    SMS_APP_MSG_START_CUNIT,
    SMS_APP_MSG_TIME_INTERVAL,

    SMS_APP_MSG_MAX

} SMS_APP_MSG_ENUM;

// We need to define a struct that will represent a state change
// in an asset, to be packed in an SMS message.
typedef struct sms_app_state_update_struct
{
    SMS_APP_ASSET_ENUM          eAsset;
    SMS_APP_ASSET_STATE_ENUM    eState;

} SMS_APP_STATE_UPDATE_STRUCT;

// Here we create a union of all the possible types of payload for
// our app messages. Right now we only have the SMS_APP_STATE_UPDATE
// message.
typedef union sms_app_msg_union
{
    SMS_APP_STATE_UPDATE_STRUCT sStateUpdate;

} SMS_APP_MSG_UNION;

// Our "master" message struct consists of the message
// type followed by the payload.
typedef struct sms_app_msg_struct
{
    SMS_APP_MSG_ENUM    eMsgType;
    SMS_APP_MSG_UNION   uMsg;

} SMS_APP_MSG_STRUCT;

// A function pointer to handle messages emitted
// by the application and event handlers.
typedef BOOLEAN (*SMS_APP_MSG_HANDLER)(SMS_APP_MSG_STRUCT *psMsg);

// Each sms object (Decoder/Module/etc) has a similar notion of
// being started, stopped, or in an error state. The enums representing
// these states are specific to their objects. Here we create a struct
// that stores these equivalent states for use by our generic event handler.
// This is made possible by storing the  various enum values as UN32s.
typedef struct sms_app_object_states_struct
{
    // The state enum for when this object is ready / up
    UN32    un32ReadyState;

    // The state enum for when this object is stopped / released
    UN32    un32StoppedState;

    // The state enum for when this object is faulted / errored
    UN32    un32ErrorState;

} SMS_APP_OBJECT_STATES_STRUCT;

// Asset's properties (bit flags)
typedef UN8 SMS_APP_ASSET_PROPS;
// Flag indicating that asset is required to be started.
// It means that we cannot go on if this asset is not started.
// In case of random test, start of such asset could be skipped.
#define SMS_APP_ASSET_PROP_MANDATORY_START          (0x01)
// Flag indicating that asset is required to be stopped.
// Shutdown cannot be completed if this asset is not stopped.
// In case of random test, stop of such asset could be skipped.
#define SMS_APP_ASSET_PROP_MANDATORY_STOP           (0x02)
// If set, STARTED (READY) state could be handled
#define SMS_APP_ASSET_PROP_READY_EVENT_SUPPORTED    (0x04)
// If set, STARTED (READY) state MUST be handled
#define SMS_APP_ASSET_PROP_READY_EVENT_REQUIRED     (0x08)
// If set, STOPPED (RELEASED) state could be handled
#define SMS_APP_ASSET_PROP_STOPPED_EVENT_SUPPORTED  (0x10)
// If set, STOPPED (RELEASED) state MUST be handled
#define SMS_APP_ASSET_PROP_STOPPED_EVENT_REQUIRED   (0x20)

#define SMS_APP_ASSET_PROPS_MANDATORY               (SMS_APP_ASSET_PROPS) \
                                                    (SMS_APP_ASSET_PROP_MANDATORY_START | \
                                                     SMS_APP_ASSET_PROP_MANDATORY_STOP)

#define SMS_APP_ASSET_PROPS_STATE_EVENT_SUPPORTED   (SMS_APP_ASSET_PROPS) \
                                                    (SMS_APP_ASSET_PROP_READY_EVENT_SUPPORTED | \
                                                     SMS_APP_ASSET_PROP_STOPPED_EVENT_SUPPORTED)

#define SMS_APP_ASSET_PROPS_DATA_SERVICE            (SMS_APP_ASSET_PROPS) \
                                                    (SMS_APP_ASSET_PROPS_STATE_EVENT_SUPPORTED | \
                                                     SMS_APP_ASSET_PROP_MANDATORY_STOP)

#define SMS_APP_ASSET_PROPS_GENERAL                 (SMS_APP_ASSET_PROPS) \
                                                    (SMS_APP_ASSET_PROPS_MANDATORY | \
                                                     SMS_APP_ASSET_PROPS_STATE_EVENT_SUPPORTED)

// Asset control function (generic for asset start\stop)
typedef BOOLEAN (*SMS_APP_ASSET_START_FUNC)(void *pvAssetObjectData);
typedef BOOLEAN (*SMS_APP_ASSET_STOP_FUNC)(void *pvAssetObjectData);

// Asset definition struct
typedef struct sms_app_asset_info_struct
{
    // User-friendly asset's name
    const char *pacName;

    // Asset's states values mapping
    SMS_APP_OBJECT_STATES_STRUCT sStatesMap;

    // Asset control functions
    SMS_APP_ASSET_START_FUNC tFnStart; // Must be specified
    SMS_APP_ASSET_STOP_FUNC tFnStop;   // Can be NULL

    // Output indent
    UN8 un8Indent;

    // A set of bit flags specified as SMS_APP_ASSET_PROP_...
    SMS_APP_ASSET_PROPS tProps;

    // Range of time before next asset start (in milliseconds)
    UN32 un32StartInterval;

    // Range of time before next asset stop (in milliseconds)
    UN32 un32StopInterval;

    // Current state
    SMS_APP_ASSET_STATE_ENUM eState;

    // Precedent asset which MUST be surely started before this asset start
    // (e.g. DECODER must be started before TUNE)
    SMS_APP_ASSET_ENUM ePrecedentAsset;

    // Asset's specific data (object handle, specific parameters, etc.)
    void *pvObjectData;

} SMS_APP_ASSET_INFO_STRUCT;

// Current cycle state
typedef enum sms_app_cycle_state_enum
{
    SMS_APP_CYCLE_STATE_STOPPED = 0,
    SMS_APP_CYCLE_STATE_STARTING,
    SMS_APP_CYCLE_STATE_STOPPING

} SMS_APP_CYCLE_STATE_ENUM;

// A struct to keep track of the state on a per-cycle basis
typedef struct sms_app_cycle_info_struct
{
    // Current cycle state
    SMS_APP_CYCLE_STATE_ENUM eState;

    // Cycle start time (in seconds, as measured by vTimeUp)
    UN32 un32StartTime;

    // Current pending asset (waiting for state update)
    SMS_APP_ASSET_ENUM ePendingAsset;

    // Maximum number of assets to start in a cycle
    UN8 un8NumAssetsToStart;

    // Number of started assets (wich asset's start is called for)
    UN8 un8NumStartedAssets;

    // Operational time between start completion and stop beginning
    // (in milliseconds)
    UN32 un32OperationalTime;

    // Number of errors per 1 cycle (reset each iteration)
    UN16 un16NumErrors;

} SMS_APP_CYCLE_INFO_STRUCT;

// Cycler's common properties (bit flags)
typedef UN16 SMS_APP_CYCLER_PROPS;
// Empty properties value
#define SMS_APP_CYCLER_PROP_NONE            (0x000)
// Means to cycle indefinitely
#define SMS_APP_CYCLER_PROP_CYCLE_FOREVER   (0x0001)
// Means only to start all services (no cycling)
#define SMS_APP_CYCLER_PROP_START_ALL       (0x0002)
// Means to run a com test in the background
#define SMS_APP_CYCLER_PROP_COM_TEST        (0x0004)
// CUnit running trigger
#define SMS_APP_CYCLER_PROP_RUN_CUNIT       (0x0008)
// Random cycler test trigger
#define SMS_APP_CYCLER_PROP_RANDOM_TEST     (0x0010)
// Means that next asset start shall be initiated
// without waiting for STARTED state of previously started asset
#define SMS_APP_CYCLER_PROP_NO_START_WAIT   (0x0020)
// Means that next asset stop shall be initiated
// without waiting for STOPPED state of previously stopped asset
#define SMS_APP_CYCLER_PROP_NO_STOP_WAIT    (0x0040)
// Means that Cycler shall be stopped immediately if any error occurs
#define SMS_APP_CYCLER_PROP_STOP_ON_ERROR   (0x0080)

// The "master" structure that collects control related data
// used by the app during execution.

typedef struct sms_app_data_struct
{
    // OSAL "App" Task Priority; allows the user to specify the
    // priority of the SMS app task from the command line; useful
    // for debugging.
    OSAL_TASK_PRIORITY_ENUM eAppTaskPriority;

    // Global queue handle
    OSAL_OBJECT_HDL hMsgQueueObj;

    // Interval Timer
    OSAL_OBJECT_HDL hTimer;

    // Common Cycler's parameters
    SMS_APP_CYCLER_PROPS tProps;

    // Current cycle parameters
    SMS_APP_CYCLE_INFO_STRUCT sCycleInfo;

    // Max Iterations, specified by the user
    UN32 un32MaxCycles;

    // Number of successful Cycles
    UN32 un32SuccessfulCycles;

    // Number of failed Cycles
    UN32 un32FailedCycles;

    // Memory usage
    UN32 un32MaxUserBytes;

    // Number of detected memory leaks
    UN32 un32NumMemoryLeaks;

#ifdef SUPPORT_CUNIT
    // CUnit suite to run
    SMS_CUNIT_TEST_SUITE_MASK tCUnitSuite;
#endif

} SMS_APP_DATA_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

// Prototypes for functions to (un-)register drivers and to
// platform-specific setup / functions.

BOOLEAN bRegisterDrivers (
    void *pvArg
        );

void vUnRegisterDrivers (
    void *pvArg
        );

BOOLEAN bPlatformSpecificInit ( void );

static void vMonitorViolationHandler (
    OSAL_RETURN_CODE_ENUM eErrorCode,
    const OSAL_TASK_INFO_STRUCT *psTaskInfo,
    void *pvArg
        );

static BOOLEAN SMS_APP_bStartHandler (
    const void *pvArg
        );

// Common helper functions

void vToggleSmsConsoleColor ( void );

void vResetSmsConsoleColor ( void );

static void SMS_APP_vGenericStateCallback (
    SMS_APP_ASSET_ENUM eAsset,
    UN32 un32StateValue
        );

static BOOLEAN SMS_APP_bUpdateMemoryStatistics (
    BOOLEAN bCheckMemoryUsage
        );

static void SMS_APP_vPrintStatDump ( void );

static BOOLEAN SMS_APP_bSendMessage (
    SMS_APP_MSG_STRUCT *psMsg
        );

static BOOLEAN SMS_APP_bStartIntervalTimer (
    UN32 un32Timeout
        );

static N32 SMS_APP_n32Task (
    void *pvArg
        );

static void SMS_APP_vHeartbeat (
    void *pvArg
        );

static void vIndentedPrint (
    UN8 un8Indent,
    char cPaddingCharacter,
    UN8 un8DstLen,
    const char *format,
    ...
        );

static void vCharPaddingPrint (
    char cPaddingCharacter,
    UN8 un8DstLen,
    const char *format,
    ...
        );

static void SMS_APP_vTimeoutCallback (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        );

static const char *SMS_APP_pacAssetStateToString (
    SMS_APP_ASSET_STATE_ENUM eState
        );

// Functions to get test runtime related values
// depending on specified parameters

static UN32 SMS_APP_un32GetTestValue (
    UN32 un32MaxValue,
    UN32 un32DefaultValue
        );

static UN32 SMS_APP_un32GetAssetStartWaitTime (
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo
        );

static UN32 SMS_APP_un32GetAssetStopWaitTime (
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo
        );

static UN32 SMS_APP_un32GetOperationalTime ( void );

// Message handlers, called in response to specific
// messages sent to the test task.

static BOOLEAN SMS_APP_bHandleStartCycleMsg (
    SMS_APP_MSG_STRUCT *psMsg
        );

static BOOLEAN SMS_APP_bHandleStateChangeMsg (
    SMS_APP_MSG_STRUCT *psMsg
        );

static BOOLEAN SMS_APP_bHandleStartCUnitMsg (
    SMS_APP_MSG_STRUCT *psMsg
        );

static BOOLEAN SMS_APP_bHandleTimerMsg (
    SMS_APP_MSG_STRUCT *psMsg
        );

// Events "action" functions for starting and stopping a cycle.
// (These also trigger printing the cycle header / footer).

static BOOLEAN SMS_APP_bInitCycler ( void );

static BOOLEAN SMS_APP_bBeginCycleStop ( void );

static BOOLEAN SMS_APP_bCompleteCycle ( void );

static BOOLEAN SMS_APP_bStartNext ( void );

static BOOLEAN SMS_APP_bStopNext ( void );

static BOOLEAN SMS_APP_bStopCycler ( void );

static BOOLEAN SMS_APP_bOnStartingEventHandler (
    SMS_APP_ASSET_ENUM eAsset,
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo,
    SMS_APP_ASSET_STATE_ENUM eState
        );

static BOOLEAN SMS_APP_bOnStoppingEventHandler (
    SMS_APP_ASSET_ENUM eAsset,
    SMS_APP_ASSET_INFO_STRUCT *psAssetInfo,
    SMS_APP_ASSET_STATE_ENUM eState
        );

// SMS asset implementation

static BOOLEAN SMS_APP_bStartSMS (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopSMS (
    void *pvAssetObjectData
        );

// SRM asset implementation

static BOOLEAN SMS_APP_bStartSRM (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopSRM (
    void *pvAssetObjectData
        );

static void SRM_vEventCallback (
    SRM_OBJECT hSRM,
    SRM_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static SRM_OBJECT SMS_APP_hSrm ( void );

// MODULE asset implementation

static BOOLEAN SMS_APP_bStartModule (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopModule (
    void *pvAssetObjectData
        );

static void MODULE_vEventCallback (
    MODULE_OBJECT hModule,
    MODULE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static MODULE_OBJECT SMS_APP_hModule ( void );

// DECODER and DECODER_TUNE assets implementation

static BOOLEAN SMS_APP_bStartDecoder (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopDecoder (
    void *pvAssetObjectData
        );

static void DECODER_vEventCallback (
    DECODER_OBJECT hDecoder,
    DECODER_EVENT_MASK tEventMask,
    CHANNEL_OBJECT hChannel,
    void *pvEventCallbackArg
        );

static BOOLEAN SMS_APP_bStartDecoderTune (
    void *pvAssetObjectData
        );

static DECODER_OBJECT SMS_APP_hDecoder ( void );

// Art Service and Art Products assets implementation

static BOOLEAN SMS_APP_bStartArtService (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopArtService (
    void *pvAssetObjectData
        );

static void CHANNEL_ART_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static DATASERVICE_MGR_OBJECT SMS_APP_hArtService ( void );

static BOOLEAN SMS_APP_bStartChannelArtProduct (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopChannelArtProduct (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStartAlbumArtProduct (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopAlbumArtProduct (
    void *pvAssetObjectData
        );

// EPG Service asset implementation

static BOOLEAN SMS_APP_bStartEPG (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopEPG (
    void *pvAssetObjectData
        );

static void EPG_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

#ifdef USE_GENERIC_TRAFFIC
// Generic Traffic Service asset implementation

static BOOLEAN SMS_APP_bStartGenericTraffic (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopGenericTraffic (
    void *pvAssetObjectData
        );

static void GENERIC_TRAFFIC_vEventCallback(
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventArg,
    void *pvEventCallbackArg
        );
#else // #ifdef USE_GENERIC_TRAFFIC

// Traffic Service asset implementation

static BOOLEAN SMS_APP_bStartTraffic (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopTraffic (
    void *pvAssetObjectData
        );

static void TRAFFIC_vEventCallback(
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );
#endif

// Movies Service asset implementation

static BOOLEAN SMS_APP_bStartMovies (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopMovies (
    void *pvAssetObjectData
        );

static void MOVIES_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// Fuel Price Service and Fuel Products assets implementation

static BOOLEAN SMS_APP_bStartFuelPriceService (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopFuelPriceService (
    void *pvAssetObjectData
        );

static void FUEL_PRICE_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static DATASERVICE_MGR_OBJECT SMS_APP_hFuelService ( void );

static BOOLEAN SMS_APP_bStartFuelDbUpdateProduct (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopFuelDbUpdateProduct (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStartFuelPriceProduct (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopFuelPriceProduct (
    void *pvAssetObjectData
        );

// EV Stations Service and EV Products assets implementation

static BOOLEAN SMS_APP_bStartEvService (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopEvService (
    void *pvAssetObjectData
        );

static void EV_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static DATASERVICE_MGR_OBJECT SMS_APP_hEvService ( void );

static BOOLEAN SMS_APP_bStartEvDbUpdateProduct (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopEvDbUpdateProduct (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStartEvStationLogoProduct (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopEvStationLogoProduct (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStartEvAvailabilityProduct (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopEvAvailabilityProduct (
    void *pvAssetObjectData
        );

// Sports Service asset implementation

static BOOLEAN SMS_APP_bStartSportsService (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopSportsService (
    void *pvAssetObjectData
        );

static void SPORTS_SERVICE_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// Graphical Weather Service asset implementation

static BOOLEAN SMS_APP_bStartGraphicalWeather (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopGraphicalWeather (
    void *pvAssetObjectData
        );

static void GRAPHICAL_WEATHER_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// Tabular Weather Service asset implementation

static BOOLEAN SMS_APP_bStartTabularWeather (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopTabularWeather (
    void *pvAssetObjectData
        );

static void TABULAR_WEATHER_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// Stocks Service asset implementation

static BOOLEAN SMS_APP_bStartStockTicker (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopStockTicker (
    void *pvAssetObjectData
        );

static void STOCK_TICKER_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// Saftety Cameras Service asset implementation

static BOOLEAN SMS_APP_bStartSafetyCameras (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopSafetyCameras (
    void *pvAssetObjectData
        );

static void SAFETY_CAMERAS_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// Traffic Cameras Service asset implementation

static BOOLEAN SMS_APP_bStartTrafficCameras (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopTrafficCameras (
    void *pvAssetObjectData
        );

static void TRAFFIC_CAMERAS_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// WS Alerts Service asset implementation

static BOOLEAN SMS_APP_bStartWsAlerts (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopWsAlerts (
    void *pvAssetObjectData
        );

static void WS_ALERTS_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// Maps Service asset implementation

static BOOLEAN SMS_APP_bStartMaps (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopMaps (
    void *pvAssetObjectData
        );

static void MAPS_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// Phonetics Service asset implementation

static BOOLEAN SMS_APP_bStartPhonetics (
    void *pvAssetObjectData
        );

static BOOLEAN SMS_APP_bStopPhonetics (
    void *pvAssetObjectData
        );

static void PHONETICS_vEventCallback (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

  /***************/
 /** VARIABLES **/
/***************/

// An externally defined variable for the temp directory path (platform-specifc)

extern char *gpacTempDirPath;

// Serial Device
// We leave this outside our asset struct for easy access by the
// platform-specific device driver file.

char *gpacSerialDevice = NULL;
char *gpacRadioName = SXM_RADIO_NAME;

// The following two strings are defined in the platform-specific
// device driver file.

extern char *gpacDefaultSmsCfgPath;
extern char *gpacDefaultCliResourcesDir;

// Assets definitions
// IMPORTANT NOTE:
// When a new asset is added, it should be added
// to the SMS_APP_ASSET_ENUM as well. Position in the gasAssetsInfo array
// must correspond to the appropriate index in the SMS_APP_ASSET_ENUM.

static SMS_APP_ASSET_INFO_STRUCT gasAssetInfo[NUM_ASSETS] =
{
    // SMS
    {
        /*.pacName=*/               "SMS",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    ASSET_STATE_STARTED,
            /*.un32StoppedState=*/  ASSET_STATE_STOPPED,
            /*.un32ErrorState=*/    ASSET_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartSMS,
        /*.tFnStop=*/               SMS_APP_bStopSMS,
        /*.un8Indent=*/             0,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_MANDATORY,
        /*.un32StartInterval=*/     0,
        /*.un32StopInterval=*/      0,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // SRM
    {
        /*.pacName=*/               "SRM",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    SRM_STATE_READY,
            /*.un32StoppedState=*/  SRM_STATE_STOPPED,
            /*.un32ErrorState=*/    SRM_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartSRM,
        /*.tFnStop=*/               SMS_APP_bStopSRM,
        /*.un8Indent=*/             1,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_GENERAL,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // MODULE
    {
        /*.pacName=*/               "MODULE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    MODULE_STATE_READY,
            /*.un32StoppedState=*/  MODULE_STATE_STOPPED,
            /*.un32ErrorState=*/    MODULE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartModule,
        /*.tFnStop=*/               SMS_APP_bStopModule,
        /*.un8Indent=*/             2,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_GENERAL,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // DECODER
    {
        /*.pacName=*/               "DECODER",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DECODER_STATE_READY,
            /*.un32StoppedState=*/  DECODER_STATE_RELEASED,
            /*.un32ErrorState=*/    DECODER_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartDecoder,
        /*.tFnStop=*/               SMS_APP_bStopDecoder,
        /*.un8Indent=*/             3,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_GENERAL,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // DECODER_TUNE
    {
        /*.pacName=*/               "DECODER TUNE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    ASSET_STATE_STARTED,
            /*.un32StoppedState=*/  ASSET_STATE_STOPPED,
            /*.un32ErrorState=*/    ASSET_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartDecoderTune,
        /*.tFnStop=*/               NULL,
        /*.un8Indent=*/             3,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_STATE_EVENT_SUPPORTED,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       DECODER_ASSET,
        /*.pvObjectData=*/          NULL
    },
    // Art Service
    {
        /*.pacName=*/               "ART SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartArtService,
        /*.tFnStop=*/               SMS_APP_bStopArtService,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // Channel Art Product
    {
        /*.pacName=*/               "CHANNEL ART PRODUCT",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATA_PRODUCT_STATE_READY,
            /*.un32StoppedState=*/  DATA_PRODUCT_STATE_DISABLED,
            /*.un32ErrorState=*/    DATA_PRODUCT_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartChannelArtProduct,
        /*.tFnStop=*/               SMS_APP_bStopChannelArtProduct,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_STATE_EVENT_SUPPORTED,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       ART_SERVICE_ASSET,
        /*.pvObjectData=*/          NULL
    },
    // Album Art Product
    {
        /*.pacName=*/               "ALBUM ART PRODUCT",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATA_PRODUCT_STATE_READY,
            /*.un32StoppedState=*/  DATA_PRODUCT_STATE_DISABLED,
            /*.un32ErrorState=*/    DATA_PRODUCT_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartAlbumArtProduct,
        /*.tFnStop=*/               SMS_APP_bStopAlbumArtProduct,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_STATE_EVENT_SUPPORTED,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       ART_SERVICE_ASSET,
        /*.pvObjectData=*/          NULL
    },
    // EPG Service
    {
        /*.pacName=*/               "EPG SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartEPG,
        /*.tFnStop=*/               SMS_APP_bStopEPG,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
#ifdef USE_GENERIC_TRAFFIC
    // Generic Traffic Service
    {
        /*.pacName=*/               "GENERIC TRAFFIC SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartGenericTraffic,
        /*.tFnStop=*/               SMS_APP_bStopGenericTraffic,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
#else // #ifdef USE_GENERIC_TRAFFIC
    // Traffic Service
    {
        /*.pacName=*/               "TRAFFIC SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartTraffic,
        /*.tFnStop=*/               SMS_APP_bStopTraffic,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
#endif
    // Movies Service
    {
        /*.pacName=*/               "MOVIES SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartMovies,
        /*.tFnStop=*/               SMS_APP_bStopMovies,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // Fuel Price Service
    {
        /*.pacName=*/               "FUEL PRICE SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartFuelPriceService,
        /*.tFnStop=*/               SMS_APP_bStopFuelPriceService,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // Fuel DB Update Product
    {
        /*.pacName=*/               "FUEL DB UPDATE PRODUCT",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATA_PRODUCT_STATE_READY,
            /*.un32StoppedState=*/  DATA_PRODUCT_STATE_DISABLED,
            /*.un32ErrorState=*/    DATA_PRODUCT_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartFuelDbUpdateProduct,
        /*.tFnStop=*/               SMS_APP_bStopFuelDbUpdateProduct,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_STATE_EVENT_SUPPORTED,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       FUEL_PRICE_SERVICE_ASSET,
        /*.pvObjectData=*/          NULL
    },
    // Fuel Price Product
    {
        /*.pacName=*/               "FUEL PRICE PRODUCT",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATA_PRODUCT_STATE_READY,
            /*.un32StoppedState=*/  DATA_PRODUCT_STATE_DISABLED,
            /*.un32ErrorState=*/    DATA_PRODUCT_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartFuelPriceProduct,
        /*.tFnStop=*/               SMS_APP_bStopFuelPriceProduct,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_STATE_EVENT_SUPPORTED,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       FUEL_PRICE_SERVICE_ASSET,
        /*.pvObjectData=*/          NULL
    },
    // EV Stations Service
    {
        /*.pacName=*/               "EV SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartEvService,
        /*.tFnStop=*/               SMS_APP_bStopEvService,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // EV DB Update Product
    {
        /*.pacName=*/               "EV DB UPDATE PRODUCT",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATA_PRODUCT_STATE_READY,
            /*.un32StoppedState=*/  DATA_PRODUCT_STATE_DISABLED,
            /*.un32ErrorState=*/    DATA_PRODUCT_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartEvDbUpdateProduct,
        /*.tFnStop=*/               SMS_APP_bStopEvDbUpdateProduct,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_STATE_EVENT_SUPPORTED,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       EV_SERVICE_ASSET,
        /*.pvObjectData=*/          NULL
    },
    // EV Station Logos Product
    {
        /*.pacName=*/               "EV STATION LOGO PRODUCT",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATA_PRODUCT_STATE_READY,
            /*.un32StoppedState=*/  DATA_PRODUCT_STATE_DISABLED,
            /*.un32ErrorState=*/    DATA_PRODUCT_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartEvStationLogoProduct,
        /*.tFnStop=*/               SMS_APP_bStopEvStationLogoProduct,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_STATE_EVENT_SUPPORTED,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       EV_SERVICE_ASSET,
        /*.pvObjectData=*/          NULL
    },
    // EV Availability Product
    {
        /*.pacName=*/               "EV AVAILABILITY PRODUCT",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATA_PRODUCT_STATE_READY,
            /*.un32StoppedState=*/  DATA_PRODUCT_STATE_DISABLED,
            /*.un32ErrorState=*/    DATA_PRODUCT_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartEvAvailabilityProduct,
        /*.tFnStop=*/               SMS_APP_bStopEvAvailabilityProduct,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_STATE_EVENT_SUPPORTED,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       EV_SERVICE_ASSET,
        /*.pvObjectData=*/          NULL
    },
    // Sports Service
    {
        /*.pacName=*/               "SPORTS SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartSportsService,
        /*.tFnStop=*/               SMS_APP_bStopSportsService,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // Graphical Weather Service
    {
        /*.pacName=*/               "GRAPHICAL WEATHER SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartGraphicalWeather,
        /*.tFnStop=*/               SMS_APP_bStopGraphicalWeather,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // Tabular Weather Service
    {
        /*.pacName=*/               "TABULAR WEATHER SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartTabularWeather,
        /*.tFnStop=*/               SMS_APP_bStopTabularWeather,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // Stock Ticker Service
    {
        /*.pacName=*/               "STOCK TICKER SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartStockTicker,
        /*.tFnStop=*/               SMS_APP_bStopStockTicker,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // Safety Cameras Service
    {
        /*.pacName=*/               "SAFETY CAMERAS SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartSafetyCameras,
        /*.tFnStop=*/               SMS_APP_bStopSafetyCameras,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // Traffic Cameras Service
    {
        /*.pacName=*/               "TRAFFIC CAMERAS SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartTrafficCameras,
        /*.tFnStop=*/               SMS_APP_bStopTrafficCameras,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // WS Alerts Service
    {
        /*.pacName=*/               "W/S ALERTS SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartWsAlerts,
        /*.tFnStop=*/               SMS_APP_bStopWsAlerts,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // Maps Service
    {
        /*.pacName=*/               "NON-NAV MAPS SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartMaps,
        /*.tFnStop=*/               SMS_APP_bStopMaps,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
    // Phonetics Service
    {
        /*.pacName=*/               "PHONETICS SERVICE",
        /*.sStatesMap=*/
        {
            /*.un32ReadyState=*/    DATASERVICE_STATE_READY,
            /*.un32StoppedState=*/  DATASERVICE_STATE_STOPPED,
            /*.un32ErrorState=*/    DATASERVICE_STATE_ERROR,
        },
        /*.tFnStart=*/              SMS_APP_bStartPhonetics,
        /*.tFnStop=*/               SMS_APP_bStopPhonetics,
        /*.un8Indent=*/             4,
        /*.tProps=*/                SMS_APP_ASSET_PROPS_DATA_SERVICE,
        /*.un32StartInterval=*/     ASSET_START_INTERVAL_MAX,
        /*.un32StopInterval=*/      ASSET_STOP_INTERVAL_MAX,
        /*.eState=*/                ASSET_STATE_STOPPED,
        /*.ePrecedentAsset=*/       NUM_ASSETS,
        /*.pvObjectData=*/          NULL
    },
};

// Common Cycler data initialization

static SMS_APP_DATA_STRUCT gsAppData =
{
    /* .eAppTaskPriority */     OSAL_TASK_PRIORITY_LOW,
    /* .hMsgQueueObj = */       OSAL_INVALID_OBJECT_HDL,
    /* .hTimer = */             OSAL_INVALID_OBJECT_HDL,
    /* .tProps = */             SMS_APP_CYCLER_PROP_NONE,
    /* .sCycleInfo = */
    {
        /* .eState = */                 SMS_APP_CYCLE_STATE_STOPPED,
        /* .startTime = */              0,
        /* .ePendingAsset = */          NUM_ASSETS, // No pending asset
        /* .un8NumAssetsToStart = */    NUM_ASSETS,
        /* .un8NumStartedAssets = */    0,
        /* .un32OperationalTime = */    UN32_MAX, // Means not specified
        /* .un16NumErrors = */          0,
    },
    /* .un32MaxCycles = */              0,
    /* .un32SuccessfulCycles = */       0,
    /* .un32FailedCycles = */           0,
    /* .un32MaxUserBytes = */           0,
    /* .un32NumMemoryLeaks = */         0
#ifdef SUPPORT_CUNIT
    /* .tCUnitSuite = */                ,SMS_CUNIT_TEST_SUITE_NONE,
#endif
};

// Message handlers. These MUST match the order in the SMS_APP_MSG enum.

static const SMS_APP_MSG_HANDLER gsCyclerMsgHandlers[SMS_APP_MSG_MAX] =
{
    SMS_APP_bHandleStartCycleMsg,
    SMS_APP_bHandleStateChangeMsg,
    SMS_APP_bHandleStartCUnitMsg,
    SMS_APP_bHandleTimerMsg
};

// Our OSAL Start Handler, used to register handlers for driver
// registration, startup, and task monitoring.

static const OSAL_START_HANDLER_STRUCT gsStartHandlers =
{
    // OSAL Device Driver Registration
    /* .bRegisterDrivers = */            bRegisterDrivers,
    /* .vUnregisterDrivers = */          vUnRegisterDrivers,
    /* .pvRegisterDriversHandlerArg = */ NULL,

    // OSAL Bootstrap(Start)
    /* .bStartHandler = */               SMS_APP_bStartHandler,
    /* .pvStartHandlerArg = */           NULL,

    // OSAL Monitor Handlers
    {
        /* .vMonitorErrorHandler = */     vMonitorViolationHandler,
        /* .pvMonitorErrorHandlerArg = */ NULL,

        /* .vHeartBeatHandler = */        SMS_APP_vHeartbeat,
        /* .pvHeartBeatHandlerArg = */    NULL,

        /* .vWatchdogStartHandler = */    NULL,
        /* .vWatchdogStopHandler = */     NULL,
        /* .vWatchdogKickHandler = */     NULL,
        /* .pvWatchdogHandlerArg = */     NULL
    }
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif // _SMS_APP_H_
