////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_receiver\rfdr_storage.c
///
/// @brief	rfdr storage class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////


#include "rfd_receiver.h"

////////////////////////////////////////////////////////////////////////////////
BOOL IsValidRfdFileSetDirName( RFD_FSET_MDATA_STRUCT * fileSetInfo, const TCHAR * DirName )
{
	int i;
	TCHAR fileSetDirName[FILE_SET_DIR_NAME_SIZE+1];
	BOOL isValidName = FALSE; // init. to false.

	for(i=0;i<fileSetInfo->numFiles;i++)
	{

		/* Note. A more efficient implementation would be to convert the DirName to an integer
		 *       for comparison to each integer id, instead of converting each id to a string.
		 *       In this alternate implementation, caution should be taken to check for
		 *       number of leading zeros in DirName for precise comparison.
		 */

		// create dir name string for i-th file set
		RFD_SPRINTF_S(fileSetDirName, FILE_SET_DIR_NAME_SIZE+1, FILE_SET_DIR_NAME_SPRINT_FORMAT,
			fileSetInfo->fileInfo[i].id);

		if(0 == RFD_STRCMP(DirName, fileSetDirName))
		{
			isValidName = TRUE;
			break;
		}
	}

	return isValidName;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ClientCreateStorage(RFDR_CLIENT_INFO_STRUCT * clientInfo)
{
	int i;
	TCHAR * pathBuf;
	RFD_STATUS status;

	if(NULL == (pathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
	// Create the Backup Storage directory for
	// this corresponding Client Directory.

	// Create file path string for the client directory.
	RFDR_ConstructClientDirPathName(pathBuf, RFD_MAX_PATH_LEN, clientInfo);

	status = RFD_CreateBackupStorage(pathBuf);
	if(status != RFD_STATUS_OK) {
		RFD_DPrint( TEXT("RFD_CreateBackupStorage failed, status (%d)\n"), status);
		RFD_FREE(pathBuf);
		return status;
	}
#endif

	//////////
	// For each fragment (file instance), create the top-level fragment directory
	// and the underlying fragment directories (process and collect).
	//////////

	for(i=0;i<clientInfo->fileSetInfo.numFiles;i++)
	{
		//////////
		// Create the fragment (file instance) level directory.
		//////////

		// Construct base file path string used by this file instance
		// (with trailing delimiter for call to Create Directory)
		RFDR_ConstructFragmentDirPathName(pathBuf, RFD_MAX_PATH_LEN, clientInfo,
										  clientInfo->fileSetInfo.fileInfo[i].id);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);

		RFD_DPrint( TEXT("Creating directory: %s \n"),pathBuf);

		if(RFD_CREATE_DIR(pathBuf)) {
			status = RFD_STATUS_ERROR_DIRECTORY_CREATE;
			RFD_DPrint( TEXT("CreateDirectory failed, status (%d)\n"), status);

			RFD_FREE(pathBuf);
			return status;
		}

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
		// Note: there is no dedicated backup directory for the fragment-level directory
		// (only the underlying directories).
#endif

		//////////
		// Create the fragment's underlying Collect directory.
		//////////

		// Construct base file path string for the underlying Collect directory for this fragment (file instance)
		// (with trailing delimiter for call to Create Directory)
		RFDR_ConstructFragmentDirPathName(pathBuf, RFD_MAX_PATH_LEN, clientInfo,
										  clientInfo->fileSetInfo.fileInfo[i].id);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);

		RFD_DPrint( TEXT("Creating directory: %s \n"),pathBuf);

		if(RFD_CREATE_DIR(pathBuf)) {
			status = RFD_STATUS_ERROR_DIRECTORY_CREATE;
			RFD_DPrint( TEXT("CreateDirectory failed, status (%d)\n"), status);

			RFD_FREE(pathBuf);
			return status;
		}

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
		// Create file path string,
		// without the trailing delimiter.
		RFDR_ConstructFragmentDirPathName(pathBuf, RFD_MAX_PATH_LEN, clientInfo,
										  clientInfo->fileSetInfo.fileInfo[i].id);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);

		// Create the backup storage directory for the corresponding directory.
		status = RFD_CreateBackupStorage(pathBuf);
		if(status != RFD_STATUS_OK) {
			RFD_DPrint( TEXT("RFD_CreateBackupStorage failed, status (%d)\n"), status);
			RFD_FREE(pathBuf);
			return status;
		}
#endif

		/////////
		// Do Not Mark the end of storage transaction.
		// The directory has been created, but the contents files have
		// not yet been created.
		// Could mark the start of transaction
		// but absence of a transaction file also effectively indicates
		// transaction in progress.
		/////////

		//////////
		// Create the fragment's underlying Process directory.
		//////////

		// Construct base file path string for the underlying Process directory for this fragment (file instance)
		// (with trailing delimiter for call to Create Directory)
		RFDR_ConstructFragmentDirPathName(pathBuf, RFD_MAX_PATH_LEN, clientInfo,
										  clientInfo->fileSetInfo.fileInfo[i].id);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);

		RFD_DPrint( TEXT("Creating directory: %s \n"),pathBuf);

		if(RFD_CREATE_DIR(pathBuf)) {
			status = RFD_STATUS_ERROR_DIRECTORY_CREATE;
			RFD_DPrint( TEXT("CreateDirectory failed, status (%d)\n"), status);

			RFD_FREE(pathBuf);
			return status;
		}

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
		// Create file path string,
		// without the trailing delimiter.
		RFDR_ConstructFragmentDirPathName(pathBuf, RFD_MAX_PATH_LEN, clientInfo,
										  clientInfo->fileSetInfo.fileInfo[i].id);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

		// Create the backup storage directory for the corresponding directory.
		status = RFD_CreateBackupStorage(pathBuf);
		if(status != RFD_STATUS_OK) {
			RFD_DPrint( TEXT("RFD_CreateBackupStorage failed, status (%d)\n"), status);
			RFD_FREE(pathBuf);
			return status;
		}
#endif

		/////////
		// Do Not Mark the end of storage transaction.
		// The directory has been created, but the contents files have
		// not yet been created.
		// Could mark the start of transaction
		// but absence of a transaction file also effectively indicates
		// transaction in progress.
		/////////
	}

	RFD_FREE(pathBuf);
	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Cleanup client storage.
///
/// Remove all files and directories under a client directory. The client directory itself is not
/// removed.
///
/// @param  clientInfo  Information describing the client.
///
/// @return true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_ClientCleanupStorage(RFDR_CLIENT_INFO_STRUCT * clientInfo)
{
	TCHAR * pathBuf;
	BOOL errorNone;

	pathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );

	if(pathBuf == NULL) {
		return FALSE;
	}

	// build client directory path name
	RFDR_ConstructClientDirPathName(pathBuf, RFD_MAX_PATH_LEN, clientInfo);

	RFD_DPrint( TEXT("RFD_ClientCleanupStorage(): performing cleanup of directory %s\n"), pathBuf);

	// Delete contents of client directory.
	errorNone = RFD_DeleteDirectoryRecursive(pathBuf,  TRUE /* delete dir contents only, not the client directory itself */);

	if(!errorNone) {
		RFD_DPrint( TEXT("RFD_ClientCleanupStorage(): error in cleanup of directory %s\n"), pathBuf);
	}

	RFD_FREE(pathBuf);
	return errorNone;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Mark the start of a storage transaction.
///
/// This function is called to indicate that a storage transaction is started.
///
/// A normal usage sequence is:
///
/// 1. Call \e RFD_MarkStorageTransaction()
///
/// 2. Update/Create/Delete files in storage corresponding to this transaction marker (typically
///    in same directory).
///    For an RFD configuration with file and directory syncing enabled (i.e. IS_RFDR_DIR_AND_FILE_SYNC_ENABLED = 1)
///    it is important that the all files of concern as part of this transaction are
///    File Synced after any file modification and before call to the next function (RFD_MarkStorageTransactionComplete)
///    that completes the transaction.
///
/// 3. Call \e RFD_MarkStorageTransactionComplete()
/// Alternately, call RFD_CancelStorageTransaction().
///
/// If file storage operations are interrupted due to power-down/reset/ect. during step (2), then
/// on the ensuing power-up, an incomplete storage transaction will be indicated because
/// RFD_MarkStorageTransactionComplete() or RFD_CancelStorageTransactionStart()
/// was not executed.
///
/// The \e RFD_IsStorageTransactionComplete() function is called to determine if a transaction is
/// complete or incomplete.
///
/// @param  path    Full pathname of the corresponding directory in which the transaction marker
///                 file is stored.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_MarkStorageTransactionStart(const TCHAR path[])
{
	RFD_STATUS status;
	TCHAR * fileName;
	INT32 actualReadDataSize;
	RFD_STORAGE_TRANSACTION_STRUCT transInfo;

	transInfo.transactionCount = 0;
	transInfo.syncCount = 0;

	fileName = (TCHAR *) RFD_MALLOC(sizeof(TCHAR) * RFD_MAX_PATH_LEN);

	if(fileName == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	RFD_STRCPY_S(fileName, RFD_MAX_PATH_LEN, path);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFDR_TRANSACTION_STATE_FNAME);

	// Read the current transaction count.
	status = RFD_ReadFile(fileName, &transInfo, sizeof(transInfo),
						  &actualReadDataSize, TRUE);
	if(status != RFD_STATUS_OK) {
		// If read fails, could be due to first time marking a transaction
		// and the transaction file has not been created yet.
		// set the counts to initial value = 0.
		transInfo.transactionCount = 0;
		transInfo.syncCount = 0;
	}

	// Increment the transaction count (each start transaction increments the transaction count).
	// The transactionCount counts up (on trans. start) and down (on trans. complete).
	transInfo.transactionCount++;

	// Increment the Sync Count (each start transaction increments the sync count)
	// The syncCount normally counts up only and counts the total number of transactions.
	// It can be used to track that the same number of transactions have been performed
	// over multiple transaction instances.
	transInfo.syncCount++;

	// Write the incremented transaction count back to file
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
	// with File Sync enabled, to sync the transaction file to disk.
#endif
	status = RFD_WriteFile(fileName, &transInfo, sizeof(transInfo), IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
    // However, there's no need to sync the parent directory since if the transaction file is not listed
    // in the parent directory file, this is also flagged  as a incomplete transaction in
    // the checks following powerup using RFD_IsStorageTransactionComplete().
#endif

	RFD_FREE(fileName);

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Mark storage transaction complete.
///
/// This function is called to indicate that the storage transaction is complete for the
/// corresponding directory specified by \a path.
/// For a configuration with file and directory syncing enabled (i.e. IS_RFDR_DIR_AND_FILE_SYNC_ENABLED = 1)
/// it is important that the caller ensure that all files of concern as part of this transaction are
/// File Synced before call to this function.
///
/// See description of function: \e RFD_MarkStorageTransactionStart() for transaction usage
/// details.
///
/// @param  path                            Full pathname of the corresponding directory in which
///                                         the transaction marker file is stored.
/// @param  doPreSyncOfDirectory            Do Directory Sync before writing the transaction file to complete the transaction.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_MarkStorageTransactionComplete(
    const TCHAR path[],
    BOOL doPreSyncOfDirectory)
{
	RFD_STATUS status;
	TCHAR * fileName;
	INT32 actualReadDataSize;
	RFD_STORAGE_TRANSACTION_STRUCT transInfo;

	transInfo.transactionCount = 0;
	transInfo.syncCount = 0;

	fileName = (TCHAR *) RFD_MALLOC(sizeof(TCHAR) * RFD_MAX_PATH_LEN);

	if(fileName == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	RFD_STRCPY_S(fileName, RFD_MAX_PATH_LEN, path);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFDR_TRANSACTION_STATE_FNAME);

	// Read the current transaction count.
	status = RFD_ReadFile(fileName, &transInfo, sizeof(transInfo),
						  &actualReadDataSize, TRUE);
	if(status != RFD_STATUS_OK) {
		// If read fails, could be due to first time marking completion of a transaction
		// and the transaction file has not been created yet.
		// set the count to initial value = 1, it will be decremented below.
		// This is not a normal use case. However, RFD_MarkStorageTransactionComplete()
		// may be used to initially create and initialize the transaction file
		// (note: alternative is to use a specific fcn: RFD_InitiailizeStorageTransaction())
		transInfo.transactionCount = 1;
		transInfo.syncCount = 0;
	}

	// Decrement the transaction count (each complete transaction decrements the transaction count)
	// The syncCount remains the same.
	transInfo.transactionCount--;

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
    if(doPreSyncOfDirectory) {
        // Sync the directory, e.g. if a new file was created in this directory
        // do a directory sync to ensure it is persisted in the directory file.
        // Do this before marking the transaction complete.
        RFD_DirectorySync( path );
    }
#endif

	// Write the decremented transaction count back to file
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
	// with File Sync enabled, to sync the transaction file to disk.
#endif
	status = RFD_WriteFile(fileName, &transInfo, sizeof(transInfo), IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
    // Sync the directory one last time. This is in case the RFD_WriteFile() operation, which opens the
    // transaction file for writing (not appending), might result in a transitory case where
    // the file is removed from the directory file.
    // An alternative to this would be to open the transaction file for appending to
    // ensure the existing file remains. However, keeping the same RFD_WriteFile()
    // preserves legacy behavior of previous RFD library releases (more conservative).
    RFD_DirectorySync( path );
#endif

	RFD_FREE(fileName);

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Cancel a storage transaction.
///
/// This function is called to indicate that the storage transaction has been cancelled. The
/// function is similar to the \e RFD_MarkStorageTransactionComplete() function except that the
/// syncCount (total count of the number of transactions) is decremented
/// For a configuration with file and directory syncing enabled (i.e. IS_RFDR_DIR_AND_FILE_SYNC_ENABLED = 1)
/// it is important that the caller ensure that all files of concern as part of this transaction are
/// File Synced before call to this function.
///
/// See description of function: \e RFD_MarkStorageTransactionStart() for transaction function
/// usage details.
///
/// @param  path    Full pathname of the corresponding directory in which the transaction marker
///                 file is stored.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CancelStorageTransaction(const TCHAR path[])
{
	RFD_STATUS status;
	TCHAR * fileName;
	INT32 actualReadDataSize;
	RFD_STORAGE_TRANSACTION_STRUCT transInfo;

	transInfo.transactionCount = 0;
	transInfo.syncCount = 0;

	fileName = (TCHAR *) RFD_MALLOC(sizeof(TCHAR) * RFD_MAX_PATH_LEN);

	if(fileName == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	RFD_STRCPY_S(fileName, RFD_MAX_PATH_LEN, path);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFDR_TRANSACTION_STATE_FNAME);

	// Read the current transaction count.
	status = RFD_ReadFile(fileName, &transInfo, sizeof(transInfo),
						  &actualReadDataSize, TRUE);
	if(status != RFD_STATUS_OK) {
		// If read fails, could be due to first time marking completion of a transaction
		// and the transaction file has not been created yet.
		// set the count to initial value = 1, it will be decremented below.
		// This is not a normal use case. However, RFD_MarkStorageTransactionComplete()
		// may be used to initially create and initialize the transaction file
		// (note: alternative is to use a specific fcn: RFD_InitiailizeStorageTransaction())
		transInfo.transactionCount = 1;
		transInfo.syncCount = 0;
	}

	// Decrement the transaction count as part of the Cancelling Transaction Start procedure.
	// Decrement the Sync Count as part of the Cancelling Transaction Start procedure.
	transInfo.transactionCount--;
	transInfo.syncCount--;

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
    // Sync the directory, e.g. if a new file was created in this directory
    // do a directory sync to ensure it is persisted in the directory file.
    // Do this before marking the transaction complete.
    RFD_DirectorySync( path );
#endif

	// Write the decremented transaction count back to file.
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
	// with File Sync enabled, to sync the transaction file to disk.
#endif
	status = RFD_WriteFile(fileName, &transInfo, sizeof(transInfo), IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
    // Sync the directory one last time. This is in case the RFD_WriteFile() operation, which opens the
    // transaction file for writing (not appending), might result in a transitory case where
    // the file is removed from the directory file.
    // An alternative to this would be to open the transaction file for appending to
    // ensure the existing file remains. However, keeping the same RFD_WriteFile()
    // preserves legacy behavior of previous RFD library releases (more conservative).
    RFD_DirectorySync( path );
#endif

	RFD_FREE(fileName);

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Initialize a storage transaction.
///
/// This function is called to 'reset' a transaction to the initial state. i.e. The transaction
/// count set to 0, and the sync count (count of total transactions) set to
///  \a initialSyncCount.
/// For a configuration with file and directory syncing enabled (i.e. IS_RFDR_DIR_AND_FILE_SYNC_ENABLED = 1)
/// it is important that the caller ensure that all files of concern as part of this transaction are
/// File Synced before call to this function.
///
/// See description of function: \e RFD_MarkStorageTransactionStart() for transaction function
/// usage details.
///
/// @param  path                Full pathname of the corresponding directory in which the
///                             transaction marker file is stored.
/// @param  initialSyncCount    The initial Sync Count value (count of the number of total
///                             transactions).
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_InitializeStorageTransaction(const TCHAR path[],
											RFD_STORAGE_SYNC_COUNT_TYPE initialSyncCount)
{
	RFD_STATUS status;
	TCHAR * fileName;
	RFD_STORAGE_TRANSACTION_STRUCT transInfo;

	fileName = (TCHAR *) RFD_MALLOC(sizeof(TCHAR) * RFD_MAX_PATH_LEN);

	if(fileName == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	RFD_STRCPY_S(fileName, RFD_MAX_PATH_LEN, path);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFDR_TRANSACTION_STATE_FNAME);

	// Initialize the counts.
	transInfo.transactionCount = 0;
	transInfo.syncCount = initialSyncCount;

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
    // Sync the directory, e.g. if a new file was created in this directory
    // do a directory sync to ensure it is persisted in the directory file.
    // Do this before marking the transaction complete.
    RFD_DirectorySync( path );
#endif

	// Write the transaction count to file.
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
	// with File Sync enabled, to sync the transaction file to disk.
#endif
	status = RFD_WriteFile(fileName, &transInfo, sizeof(transInfo), IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
	// Sync the directory one last time. This is in case the RFD_WriteFile() operation, which opens the
    // transaction file for writing (not appending), might result in a transitory case where
    // the file is removed from the directory file.
    RFD_DirectorySync( path );
#endif

	RFD_FREE(fileName);

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Is a storage transaction complete.
///
/// This function is called to determine if a storage transaction is complete or incomplete.
///
/// See description of function: \e RFD_MarkStorageTransactionStart() for transaction function
/// usage details.
///
/// @param  path    Full pathname of the corresponding directory in which the transaction marker
///                 file is stored.
///
/// @return BOOL true is transaction is complete, false if incomplete or failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_IsStorageTransactionComplete(const TCHAR path[])
{
	RFD_STATUS status;
	TCHAR * fileName;
	INT32 actualReadDataSize;
	RFD_STORAGE_TRANSACTION_STRUCT transInfo;
	BOOL isComplete = FALSE;

	transInfo.transactionCount = 0;
	transInfo.syncCount = 0;

	fileName = (TCHAR *) RFD_MALLOC(sizeof(TCHAR) * RFD_MAX_PATH_LEN);

	if(fileName == NULL) {
		return isComplete;
	}

	RFD_STRCPY_S(fileName, RFD_MAX_PATH_LEN, path);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFDR_TRANSACTION_STATE_FNAME);

	status = RFD_ReadFile(fileName, &transInfo, sizeof(transInfo),
						  &actualReadDataSize, TRUE);
	RFD_FREE(fileName);

	if(status == RFD_STATUS_OK) {
		if(transInfo.transactionCount == 0) {
			isComplete = TRUE;
		}
	}

	return isComplete;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Get the storage transaction sync count.
///
/// This function is called to get the total number of transactions that have been performed for
/// the corresponding transaction. It can be used to synchronize multiple transaction instances.
/// i.e. to determine if the same number of transactions have occurred across multiple
/// transaction instances.
///
/// See description of function: \e RFD_MarkStorageTransactionStart() for transaction function
/// usage details.
///
/// @param  path        Full pathname of the corresponding directory in which the transaction
///                     marker file is stored.
/// @param  syncCount   (Output) Total number of transactions counted.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_GetStorageTransSyncCount(const TCHAR path[], RFD_STORAGE_SYNC_COUNT_TYPE * syncCount)
{
	RFD_STATUS status;
	TCHAR * fileName;
	INT32 actualReadDataSize;
	RFD_STORAGE_TRANSACTION_STRUCT transInfo;

	transInfo.transactionCount = 0;
	transInfo.syncCount = 0;

	fileName = (TCHAR *) RFD_MALLOC(sizeof(TCHAR) * RFD_MAX_PATH_LEN);

	if(fileName == NULL) {
		*syncCount = 0;
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	RFD_STRCPY_S(fileName, RFD_MAX_PATH_LEN, path);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileName, RFD_MAX_PATH_LEN, RFDR_TRANSACTION_STATE_FNAME);

	status = RFD_ReadFile(fileName, &transInfo, sizeof(transInfo),
						  &actualReadDataSize, TRUE);
	RFD_FREE(fileName);

	if(status != RFD_STATUS_OK) {
		*syncCount = 0;
		return status;
	}

	*syncCount = transInfo.syncCount;

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Are transactions in-sync.
///
/// Are the transactions represented by two transaction marker files "in-sync". Technically
/// indicates whether the same number of transactions have been performed on both instances. Can
/// be used to indicate if storage data corresponding to both transaction instances are in-sync
/// (the same).
///
/// See description of function: \e RFD_MarkStorageTransactionStart() for transaction function
/// usage details.
///
/// @param  path1   The path of directory containing the first transaction marker file.
/// @param  path2   The path of directory containing the second transaction marker file.
///
/// @return true if transactions are In-Sync, false if not in-sync or error.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_AreStorageTransactionsInSync(const TCHAR path1[], const TCHAR path2[])
{
	RFD_STATUS status;
	RFD_STORAGE_SYNC_COUNT_TYPE syncCount1;
	RFD_STORAGE_SYNC_COUNT_TYPE syncCount2;

	status = RFD_GetStorageTransSyncCount(path1, &syncCount1) ;

	if(status != RFD_STATUS_OK) {
		return FALSE;
	}

	status = RFD_GetStorageTransSyncCount(path2, &syncCount2) ;

	if(status != RFD_STATUS_OK) {
		return FALSE;
	}

	if(syncCount1 == syncCount2) {
		return TRUE;
	}
	else {
		return FALSE;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Copy directory contents transaction protected.
///
/// This function supports backup and restore functionality. The entire file contents of a
/// directory (not including subdirectories) are copied from the specified source directory to
/// the specified destination directory. The shared data Mutex should be aquired before calling
/// this function.
///
/// @param  dstPathBuf  destination path name string.
/// @param  srcPathBuf  source path name string.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CopyDirectoryContentsTransactionProtected(const TCHAR dstPathBuf[], const TCHAR srcPathBuf[])
{
	RFD_STORAGE_SYNC_COUNT_TYPE syncCount = 0;
	RFD_STATUS status;

	// Check if transaction complete for the src directory.
	if(!RFD_IsStorageTransactionComplete(srcPathBuf)) {
		// The src directory indicates an incomplete transaction
		// so the contents cannot be relied on to be valid.
		// return with error indication.
		return(RFD_STATUS_ERROR_TRANSACTION_IN_PROGRESS);
	}

	// Mark start of transaction before deleting files
	// in case of power-down during the following file deletions.
	// In some cases, the transaction may already be marked as incomplete,
	// but this may not be all cases (e.g. this fcn. called due to
	// a detected currupted file instead of transaction incomplete).
	// (the order in which files are deleted is not guaranteed so)

	RFD_MarkStorageTransactionStart(dstPathBuf);

	// Delete all files of the destination directory.

	if(!RFD_DeleteDirectoryContentsNonRecursive(dstPathBuf, FALSE /* !includeSubDirRemoval (files only) */)) {
		return RFD_STATUS_ERROR_INVALID_STORAGE;
	}

	// Copy all files except the transaction marker file
	// from the backup directory to the destination directory.
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
	// Specify file sync of the copied files, to ensure all files of the directory 
	// are on disk before the call to the following RFD_InitializeStorageTransaction, 
	// which will "commit" the transaction.
#endif
	if(!RFD_CopyDirectoryFilesSkipSingleFile(dstPathBuf, srcPathBuf, RFDR_TRANSACTION_STATE_FNAME, 
			IS_RFDR_DIR_AND_FILE_SYNC_ENABLED)) {
		return RFD_STATUS_ERROR_INVALID_STORAGE;
	}

	// Initialize the Storage Transaction file in the destination directory.
	// This is done last, after all other files are sucessfully copied from
	// source location.
	// Use RFD_InitializeStorageTransaction() instead of RFD_MarkStorageTransactionComplete()
	// because the preceeding RFD_DeleteDirectoryContentsNonRecursive()
	// will have deleted the transaction file.
	// Use RFD_GetStorageTransSyncCount() to get the syncCount of the source directory,
	// and then use this value in the initialization with RFD_InitializeStorageTransaction().
	// The source and destination tansaction info must have the same sync counts (not the same
	// as transaction counts) to indicate that the contents of the two directories are the
	// same (in-sync).

	status = RFD_GetStorageTransSyncCount(srcPathBuf, &syncCount);
	if(status != RFD_STATUS_OK) {
		return status;
	}

	status = RFD_InitializeStorageTransaction(dstPathBuf, syncCount);
	if(status != RFD_STATUS_OK) {
		return status;
	}

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Is backup storage in-sync with the corresponding directory.
///
/// Being In-Sync is an indication that the contents of the backup storage and the corresponding
/// base directory are the same (in-sync). It is possible, depending on exactly when a power-
/// down/reset occurs, that the two directories may have transactions complete, but the data
/// contents of the two directories are not the same (not in-sync). This occurs if power-
/// down/reset occurs rigth after an update to the base directory, but before the following
/// update to the backup directory.
///
/// @param  pathBuf Path name for the base directory (the backup storage directory is a
///                 subdirectory of this base directory).
///
/// @return true backup storage directory is in-sync with base directory.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_IsBackupStorageInSync(const TCHAR pathBuf[])
{
	TCHAR * backupPathBuf = NULL;
	BOOL  isInSync;

	backupPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(backupPathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	RFD_STRCPY_S(backupPathBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(backupPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(backupPathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);

	isInSync = RFD_AreStorageTransactionsInSync(pathBuf, backupPathBuf);

	RFD_FREE(backupPathBuf);

	return isInSync;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Restore directory contents from backup storage.
///
/// The entire file contents of a directory (not including subdirectories) are restored from the
/// corresponding backup storage directory for this directory. The shared data Mutex should be
/// aquired before calling this function.
///
/// @param  pathBuf Path name string of directory to be restored.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_RestoreDirectoryContents(const TCHAR pathBuf[])
{
	const TCHAR * dstPathBuf = NULL;
	TCHAR * srcPathBuf = NULL;
	RFD_STATUS  status;

	dstPathBuf = pathBuf;

	srcPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(srcPathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	RFD_STRCPY_S(srcPathBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(srcPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(srcPathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);

	status = RFD_CopyDirectoryContentsTransactionProtected(dstPathBuf, srcPathBuf);

	RFD_FREE(srcPathBuf);

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Backup directory contents to backup storage.
///
/// The entire file contents of a directory (not including subdirectories) are backed-up to the
/// corresponding backup storage directory for this directory. If the backup directory does not
/// exist, it is created. The shared data Mutex should be aquired before calling this function.
///
/// @param  pathBuf Path name string of directory to be backed-up.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_BackupDirectoryContents(const TCHAR pathBuf[])
{
	TCHAR * dstPathBuf = NULL;
	const TCHAR * srcPathBuf = NULL;
	RFD_STATUS  status;

	srcPathBuf = pathBuf;

	dstPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(dstPathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	RFD_STRCPY_S(dstPathBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(dstPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(dstPathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);

	// Check if the backup directory exists.
	// If not, create the backup directory.
	if(!RFD_IsDirectory(dstPathBuf)) {

		RFD_DPrint( TEXT("directory Not found, Creating: %s \n"),dstPathBuf);

		if(RFD_CREATE_DIR(dstPathBuf)) {
			RFD_DPrint( TEXT("Create Directory failed\n"));
			RFD_FREE(dstPathBuf);
			return RFD_STATUS_ERROR_DIRECTORY_CREATE;
		}
	}

	// Do the copy to backup directory, with transaction protection.
	status =RFD_CopyDirectoryContentsTransactionProtected(dstPathBuf, srcPathBuf);

	RFD_FREE(dstPathBuf);

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Is backup storage valid.
///
/// The transaction completion status is checked for the backup storage directory. Backup storage
/// is valid if transaction complete is true.
///
/// @param  pathBuf Path name string for the directory for which the backup storage is being
///                 checked.
///
/// @return true if backup storage is valid, false otherwise.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_IsBackupStorageValid(const TCHAR pathBuf[])
{
	BOOL isBackupStorageValid;
	TCHAR * backupPathBuf;

	backupPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(backupPathBuf == NULL) {
		return FALSE;
	}

	// Construct backup directory path Buffer.
	RFD_STRCPY_S(backupPathBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(backupPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(backupPathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);

	// Check if transaction complete for the src directory.
	if(RFD_IsStorageTransactionComplete(backupPathBuf)) {
		// transaction complete, so contents should be valid.
		isBackupStorageValid = TRUE;
	}
	else {
		// The backup directory indicates an incomplete transaction
		// so the contents cannot be relied on to be valid.
		isBackupStorageValid = FALSE;
	}

	RFD_FREE(backupPathBuf);

	return(isBackupStorageValid);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Cleanup a backup storage directory for a specified directory.
///
/// The actual backup storage directory is not deleted, just the contents of the directory. The
/// cleanup operation is transaction protected in that the transaction-in- progress marker is set
/// first before directory cleanup begins.
///
/// @param  pathBuf path name string of directory for which the backup storage is being cleanedup.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CleanupBackupStorageContents(const TCHAR pathBuf[])
{
	TCHAR * backupPathBuf = NULL;

	backupPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(backupPathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Construct backup path name.
	RFD_STRCPY_S(backupPathBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(backupPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(backupPathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);

	// Mark start of transaction before deleting files
	// in case of power-down during the following file deletions.

	RFD_MarkStorageTransactionStart(backupPathBuf);

	// Delete all files of the backup directory.
	// The transaction file will also be deleted.

	if(!RFD_DeleteDirectoryContentsNonRecursive(backupPathBuf, TRUE /* includeSubDirRemoval, even though there shouldn't be any in backup dir */)) {
		RFD_FREE(backupPathBuf);
		return RFD_STATUS_ERROR_INVALID_STORAGE;
	}

	RFD_FREE(backupPathBuf);

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Create backup storage direcory for a corresponding directory.
///
/// @param	pathBuf	Directory path name string.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CreateBackupStorage(const TCHAR pathBuf[])
{
	TCHAR * backupPathBuf = NULL;
	RFD_STATUS  status;

	backupPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(backupPathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Construct path name string for the backup storage directory.
	RFD_STRCPY_S(backupPathBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(backupPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(backupPathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);

	// Check if the backup directory already exists.

	if(!RFD_IsDirectory(backupPathBuf)) {
		// Backup storage Directory not found
		// Create the backup directory.

		RFD_DPrint( TEXT("directory Not found, Creating: %s \n"),backupPathBuf);

		if(RFD_CREATE_DIR(backupPathBuf)) {
			RFD_DPrint( TEXT("Create Directory failed\n"));
			RFD_FREE(backupPathBuf);
			return RFD_STATUS_ERROR_DIRECTORY_CREATE;
		}
		else {
			status = RFD_STATUS_OK;
		}
	}
	else {
		// An existing Backup storage Directory was found.
		// Cleanup contents of the existing backup storage directory.
		status = RFD_CleanupBackupStorageContents(pathBuf);
	}

	// Cleanup and return.

	RFD_FREE(backupPathBuf);

	return status;
}

