////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_receiver\rfdr_processor_xcc_code.c
///
/// @brief	rfdr processor for eXtended-ecc-code class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd_receiver.h"
#include "rfd_codec_xcc.h"

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_DecodeInfoXccCodeCreateWithDefInit( RFDR_FILE_PROCESS_INFO * fileProcessInfo )
{
	RFD_XCC_DECODE_STATE_INFO * decoderStateInfo;
	RFD_XCC_CODEC_CONFIG_INFO * codecConfigInfo;
	RFD_XCC_DECODE_EVENT_BLOCK_THRESH_INFO * decodeBlockThresholdInfo;
	RFD_STATUS status;

	fileProcessInfo->decInfo.xccDecodeInfo = (RFDR_XCC_DECODE_INFO *) RFD_MALLOC( sizeof(RFDR_XCC_DECODE_INFO) );

	if(fileProcessInfo->decInfo.xccDecodeInfo == NULL)	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Clear entire structure.
	RFD_MEMSET(fileProcessInfo->decInfo.xccDecodeInfo, 0, sizeof(RFDR_XCC_DECODE_INFO));

	// Initialize codecConfigInfo

	codecConfigInfo = &fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo;

	codecConfigInfo->blkLenBytes = fileProcessInfo->fileInfo.blkLenBytes;
	codecConfigInfo->fileLenBytes = fileProcessInfo->fileInfo.size;
	codecConfigInfo->fileLenBlocks = (fileProcessInfo->fileInfo.size / fileProcessInfo->fileInfo.blkLenBytes)
		+ ((fileProcessInfo->fileInfo.size % fileProcessInfo->fileInfo.blkLenBytes) ? 1:0);

	// Initialize decodeEventBlockThreshInfo, which is specific to codeType.

	decodeBlockThresholdInfo = &fileProcessInfo->decInfo.xccDecodeInfo->decodeEventBlockThreshInfo;
	// Call function to initial codeType specific information of decodeEventBlockThreshInfo.
	status = RFD_InitializeDecodeBlockThresholdInfo(fileProcessInfo->fileInfo.codeType, decodeBlockThresholdInfo);
	if(status != RFD_STATUS_OK) {
		return status;
	}

	// Initialize maxExtraBlocksForSolution, which is specific to codeType.

	// Call function to initialize codeType specific maxExtraBlocksForSolution.
	status = RFD_InitializeMaxExtraBlocksForSolution(fileProcessInfo->fileInfo.codeType,
										&fileProcessInfo->decInfo.xccDecodeInfo->maxExtraBlocksForSolution);
	if(status != RFD_STATUS_OK) {
		return status;
	}

	// Initialize decoderStateInfo

	decoderStateInfo = &fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo;

	decoderStateInfo->decodeChunkCount = 0;
	decoderStateInfo->blksCollectedEventThresh = ((UINT32) decodeBlockThresholdInfo->blockThresholdNumeratorArray[0] *
												   codecConfigInfo->fileLenBlocks) /
												   decodeBlockThresholdInfo->blockThreshDenominator;
	decoderStateInfo->nChkBlksInBuf = 0;
	decoderStateInfo->nSrcBlksInBuf = 0;
	decoderStateInfo->nextPivotNo = 0;
	decoderStateInfo->nTransformRows = 0;
	decoderStateInfo->nChkBlksCollectedTotal = 0;
	decoderStateInfo->nSrcBlksCollectedTotal = 0;
	decoderStateInfo->nChkBlksEliminated = 0;

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_DecodeInfoXccCodeReadFromStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo )
{
	RFD_STATUS status = RFD_STATUS_ERROR_GENERAL;
	INT32 ActualReadDataSize;
	RFD_XCC_DECODE_STATE_INFO * decStateInfo;
	RFD_XCC_CODEC_CONFIG_INFO * codecInfo;
	RFDR_XCC_SRC_BLKS_COLLECTED_STATUS * srcBlksCollectedStatusArray;
	int i;
	UINT16 checkedNumSrcBlksCollected;
	TCHAR * fileNameBuf;
	UINT32 fileSize;
	UINT16 maxExtraBlocksForSolution;

	codecInfo = &fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo;
	decStateInfo = &fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo;
	srcBlksCollectedStatusArray = fileProcessInfo->decInfo.xccDecodeInfo->srcBlksCollectedStatus;
	maxExtraBlocksForSolution = fileProcessInfo->decInfo.xccDecodeInfo->maxExtraBlocksForSolution;

	if(NULL == (fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	//////////////////////
	// Load the Decoder State file
	//////////////////////

	// create file path string for the State filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // the main dec state file is in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_STATE_FNAME);

	// read the file
	if(RFD_STATUS_OK != (status = RFD_ReadFile( fileNameBuf, &fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo,
		sizeof(RFD_XCC_DECODE_STATE_INFO), &ActualReadDataSize, TRUE) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	//////////////////////
	// Load the 'Private decoder' version of the Decoder State file
	//////////////////////

	// create file path string for the State filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // the private dec state file is in the process dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_STATE_PRIVATE_FNAME);

	// read the file
	if(RFD_STATUS_OK != (status = RFD_ReadFile( fileNameBuf, &fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfoPrivate,
		sizeof(RFD_XCC_DECODE_STATE_INFO), &ActualReadDataSize, TRUE) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	//////////////////////
	// Load the Decoder Config file
	// Note: Alternative is to derive/calculate config info from fset i.e. no need to save/store this to file.
	//////////////////////

	// create file path string for the State filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_CODEC_CONFIG_FNAME);

	// read the file
	if(RFD_STATUS_OK != (status = RFD_ReadFile( fileNameBuf, &fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo,
		sizeof(RFD_XCC_CODEC_CONFIG_INFO), &ActualReadDataSize, TRUE) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}


	/////////////////////////////
	// Error check Decode state and config parameters
	/////////////////////////////

	status = RFD_STATUS_OK;

	// codecConfigInfo params
	if(    codecInfo->blkLenBytes > RFD_MAX_BLK_LEN_BYTES
		|| codecInfo->blkLenBytes < RFD_MIN_BLK_LEN_BYTES)
	{
		status = RFD_STATUS_ERROR_DECODER_BLK_LEN;
	}
	else if(    codecInfo->fileLenBlocks > RFD_MAX_FILE_LEN_BLOCKS
		|| codecInfo->fileLenBlocks < RFD_MIN_FILE_LEN_BLOCKS)
	{
		status = RFD_STATUS_ERROR_DECODER_EQU_LEN;
	}
	else if(    codecInfo->fileLenBytes > RFD_MAX_FILE_LEN_BYTES
		|| codecInfo->fileLenBytes < RFD_MIN_FILE_LEN_BYTES)
	{
		status = RFD_STATUS_ERROR_FILE_LEN;
	}
	// decoderStateInfo params
	else if( (  decStateInfo->nChkBlksInBuf
		+ decStateInfo->nSrcBlksInBuf
		+ decStateInfo->nTransformRows )
		> (codecInfo->fileLenBlocks + maxExtraBlocksForSolution) )
	{
		status = RFD_STATUS_ERROR_NUM_COLLECTED_BLKS;
	}
	else if( (decStateInfo->nSrcBlksCollectedTotal
		+ decStateInfo->nChkBlksCollectedTotal)
		> (codecInfo->fileLenBlocks + maxExtraBlocksForSolution) )
	{
		status = RFD_STATUS_ERROR_NUM_COLLECTED_BLKS;
	}
	else if( decStateInfo->nextPivotNo > codecInfo->fileLenBlocks -1 ) // caution: decStateInfo->nextPivotNo is zero relative
	{
		status = RFD_STATUS_ERROR_PIVOT_NUM;
	}

	// don't check decStateInfo->blksCollectedEventThresh

	// if error, cleanup and return
	if(status != RFD_STATUS_OK)
	{
		RFD_FREE(fileNameBuf);
		return status;
	}


	//////////////////////
	// Load srcBlksCollectedStatus array
	//////////////////////

	// create file path string for the filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_SRC_BLKS_COLLECTED_FNAME);

	// read the file
	if(RFD_STATUS_OK != (status = RFD_ReadFile( fileNameBuf, srcBlksCollectedStatusArray,
		sizeof(fileProcessInfo->decInfo.xccDecodeInfo->srcBlksCollectedStatus), &ActualReadDataSize, TRUE) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	// check contents of loaded srcBlksCollectedStatus and calculate number of Source Blocks indicated as collected.
	checkedNumSrcBlksCollected = 0;
	for(i=0;i<codecInfo->fileLenBlocks;i++)
	{
		if(srcBlksCollectedStatusArray[i] == TRUE)
		{
			checkedNumSrcBlksCollected++;
		}
		else if(srcBlksCollectedStatusArray[i] != FALSE)
		{
			// error, invalid entry (value not TRUE nor FALSE). cleanup and return with error code.
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_INVALID_VALUE;
		}
	}

	// validate consistency of the number of source blocks collected.
	if(checkedNumSrcBlksCollected != decStateInfo->nSrcBlksCollectedTotal)
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_NUM_COLLECTED_BLKS;
	}

	//////////////////////
	// Check that the source block buffer file is the expected length
	//////////////////////

	// create file path string for the filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_SRC_BLKS_INBUF_FNAME);

	if(RFD_STATUS_OK != RFD_GetFileSize( fileNameBuf, &fileSize))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	// the src block buffer file contains the block index along with the src block payload
	if(fileSize !=
		(UINT32) decStateInfo->nSrcBlksInBuf *
		((UINT32) codecInfo->blkLenBytes * sizeof(UCHAR) + sizeof(RFD_XCC_SRC_BLK_INDEX)))
	{
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
	}

	//////////////////////
	// Load encBlksCollectedList,
	// the list of all the enc type blocks that have been collected.
	//////////////////////

	// create file path string for the filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_CHK_BLKS_COLLECTED_FNAME);

	// read the file.
	// calling RFD_ReadFile with CheckFileSize flag true checks that the file length is the expected
	// size based on nChkBlksCollectedTotal.
	if(RFD_STATUS_OK != (status = RFD_ReadFile( fileNameBuf, fileProcessInfo->decInfo.xccDecodeInfo->encBlksCollectedList,
		decStateInfo->nChkBlksCollectedTotal * sizeof(RFD_XCC_ENC_BLK_INDEX), &ActualReadDataSize, TRUE) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	//////////////////////
	// Check that the encode block buffer file is the expected length
	//////////////////////

	// create file path string for the filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_CHK_BLKS_INBUF_FNAME);

	if(RFD_STATUS_OK != RFD_GetFileSize( fileNameBuf, &fileSize))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	// the enc block buffer file contains the block index along with the enc block payload
	if(fileSize !=
		(UINT32) decStateInfo->nChkBlksInBuf *
		((UINT32) codecInfo->blkLenBytes * sizeof(UCHAR) + sizeof(RFD_XCC_ENC_BLK_INDEX)))
	{
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
	}


	//////////////////////////
	// cleanup and return;
	//////////////////////////

	RFD_FREE(fileNameBuf);

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_DecodeInfoXccCodeWriteToStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo )
{
	RFD_STATUS status = RFD_STATUS_ERROR_GENERAL;
	RFD_XCC_DECODE_STATE_INFO * decStateInfo;
	RFD_XCC_CODEC_CONFIG_INFO * codecInfo;
	RFDR_XCC_SRC_BLKS_COLLECTED_STATUS * srcBlksCollectedStatusArray;
	TCHAR * fileNameBuf;

	codecInfo = &fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo;
	decStateInfo = &fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo;
	srcBlksCollectedStatusArray = fileProcessInfo->decInfo.xccDecodeInfo->srcBlksCollectedStatus;

	if(NULL == (fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	//////////////////////
	// Write the Decoder State file
	//////////////////////

	// create file path string for the State filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // the main dec state file is in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_STATE_FNAME);

	// write the file, with compile-time configured file sync setting
	if(RFD_STATUS_OK != (status = RFD_WriteFile( fileNameBuf, 
										&fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo,
										sizeof(RFD_XCC_DECODE_STATE_INFO), 
										IS_RFDR_DIR_AND_FILE_SYNC_ENABLED ) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	//////////////////////
	// Write the 'Private decoder' version of the Decoder State file
	//////////////////////

	// create file path string for the State filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // the private dec state file is in the process dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_STATE_PRIVATE_FNAME);

	// write the file, with compile-time configured file sync setting
	if(RFD_STATUS_OK != (status = RFD_WriteFile( fileNameBuf, 
										&fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfoPrivate,
										sizeof(RFD_XCC_DECODE_STATE_INFO), 
										IS_RFDR_DIR_AND_FILE_SYNC_ENABLED) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	//////////////////////
	// Write the Decoder Config file
	// Note: Alternative is to derive/calculate config info from fset i.e. no need to save/store this to file.
	//////////////////////

	// create file path string for the State filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_CODEC_CONFIG_FNAME);

	// write the file, with compile-time configured file sync setting
	if(RFD_STATUS_OK != (status = RFD_WriteFile( fileNameBuf, 
										&fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo,
										sizeof(RFD_XCC_CODEC_CONFIG_INFO), 
										IS_RFDR_DIR_AND_FILE_SYNC_ENABLED) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	//////////////////////
	// Write srcBlksCollectedStatus array
	//////////////////////

	// create file path string for the filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_SRC_BLKS_COLLECTED_FNAME);

	// write the file, with compile-time configured file sync setting
	if(RFD_STATUS_OK != (status = RFD_WriteFile( fileNameBuf, srcBlksCollectedStatusArray,
										sizeof(fileProcessInfo->decInfo.xccDecodeInfo->srcBlksCollectedStatus), 
										IS_RFDR_DIR_AND_FILE_SYNC_ENABLED) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	//////////////////////
	// If nSrcBlksInBuf is 0, then assume the decode info storage is being initialized for the first time
	// (being created). So create RFDR_SRC_BLKS_COLLECTED_FNAME with zero size as part of initialization.
	// Note: could possibly pass a 'create flag' in function to request creation of these files.
	//////////////////////

	if(decStateInfo->nSrcBlksInBuf == 0)
	{
		// create file path string for the filename
		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_SRC_BLKS_INBUF_FNAME);

		// create the file, with compile-time configured file sync setting
		if(RFD_STATUS_OK != RFD_CreateFile( fileNameBuf,IS_RFDR_DIR_AND_FILE_SYNC_ENABLED ))
		{
			// error, cleanup and return with error code.
			RFD_FREE(fileNameBuf);
			return status;
		}
	}


	//////////////////////
	// Write encBlksCollectedList,
	// the list of all the enc type blocks that have been collected.
	//////////////////////

	// create file path string for the filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_CHK_BLKS_COLLECTED_FNAME);

	// write the file, with compile-time configured file sync setting
	if(RFD_STATUS_OK != (status = RFD_WriteFile( fileNameBuf, 
										fileProcessInfo->decInfo.xccDecodeInfo->encBlksCollectedList,
										decStateInfo->nChkBlksCollectedTotal * sizeof(RFD_XCC_ENC_BLK_INDEX), 
										IS_RFDR_DIR_AND_FILE_SYNC_ENABLED) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}


	//////////////////////
	// If nChkBlksInBuf is 0, then assume the decode info storage is being initialized for the first time
	// (being created). So create RFDR_CHK_BLKS_COLLECTED_FNAME with zero size as part of initialization.
	// Note: could possibly pass a 'create flag' in funtion to request creation of these files.
	//////////////////////

	if(decStateInfo->nChkBlksInBuf == 0)
	{
		// create file path string for the filename
		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_CHK_BLKS_INBUF_FNAME);

		// create the file, with compile-time configured file sync setting
		if(RFD_STATUS_OK != RFD_CreateFile( fileNameBuf, IS_RFDR_DIR_AND_FILE_SYNC_ENABLED ))
		{
			// error, cleanup and return with error code.
			RFD_FREE(fileNameBuf);
			return status;
		}
	}


	//////////////////////////
	// cleanup and return;
	//////////////////////////

	RFD_FREE(fileNameBuf);

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_DecodeInfoXccCodeCleanupStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo )
{
	TCHAR * fileNameBuf;

	if(NULL == (fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	//////////////////////
	// Delete the Decoder State file
	//////////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // the main dec state file is in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_STATE_FNAME);
	RFD_DELETE_FILE(fileNameBuf);

	//////////////////////
	// Delete the 'Private decoder' version of the Decoder State file
	//////////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // the private dec state file is in the process dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_STATE_PRIVATE_FNAME);
	RFD_DELETE_FILE(fileNameBuf);

	//////////////////////
	// Delete the Decoder Config file
	//////////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_CODEC_CONFIG_FNAME);
	RFD_DELETE_FILE(fileNameBuf);


	//////////////////////
	// Delete srcBlksCollectedStatus array
	//////////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_SRC_BLKS_COLLECTED_FNAME);
	RFD_DELETE_FILE(fileNameBuf);


	//////////////////////
	// Delete the Source Blocks Input Buffer
	//////////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_SRC_BLKS_INBUF_FNAME);
	RFD_DELETE_FILE(fileNameBuf);


	//////////////////////
	// Delete the encBlksCollectedList,
	// the list of all the enc type blocks that have been collected.
	//////////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_CHK_BLKS_COLLECTED_FNAME);
	RFD_DELETE_FILE(fileNameBuf);


	//////////////////////
	// Delete the Encoded Blocks Input Buffer
	//////////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_CHK_BLKS_INBUF_FNAME);
	RFD_DELETE_FILE(fileNameBuf);

	//////////////////////
	// In case a cleanup occurs before Decode Complete state
	// (i.e. Decode In Progress),
	// the following files also will still exist and need to be cleaned-up:
	//   RFDR_XCC_DEC_BLOCK_TRANS_MATRIX_FNAME
	//   RFDR_XCC_DEC_COEFF_TRANS_MATRIX_FNAME
	//   RFDR_XCC_DEC_EQU_INFO_FNAME
	//////////////////////

	//////////////////////
	// Delete the Block Transfom matrix file
	//////////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // the private dec state file is in the process dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_XCC_DEC_BLOCK_TRANS_MATRIX_FNAME);
	RFD_DELETE_FILE(fileNameBuf);

	//////////////////////
	// Delete the Coefficient Transfom matrix file
	//////////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // the private dec state file is in the process dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_XCC_DEC_COEFF_TRANS_MATRIX_FNAME);
	RFD_DELETE_FILE(fileNameBuf);


	//////////////////////
	// Delete the Decoder Equation Info file
	//////////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // the private dec state file is in the process dir.
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_XCC_DEC_EQU_INFO_FNAME);
	RFD_DELETE_FILE(fileNameBuf);

	//////////////////////////
	// cleanup and return;
	//////////////////////////

	RFD_FREE(fileNameBuf);

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_PrepareForDecodeXccCode(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo, BOOL doStorageBackupOps)
{
	RFD_XCC_CODEC_CONFIG_INFO * codecConfigInfo = &fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo;
	RFD_XCC_DECODE_STATE_INFO * decoderStateInfo = &fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo;
	RFD_XCC_DECODE_STATE_INFO * decoderStateInfoPrivate = &fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfoPrivate;
	UINT16 maxExtraBlocksForSolution = fileProcessInfo->decInfo.xccDecodeInfo->maxExtraBlocksForSolution;
	RFD_STATUS status;
	BOOL prepareSrcBlksInBuf, prepareChkBlksInBuf;

	// Do State Info updates onle if doStorageBackupOps is False.
	// doStorageBackupOps enables repeating the 'prepare for decode' file update steps
	// in a backup storage location. i.e.
	// 1. a first pass call with doStorageBackupOps False will update State Info and then
	// update files in preparation for decode.
	//
	// 2. a second pass call with doStorageBackupOps True will then skip the State Info updates
	// and just do the same file updates as the first pass, but this time on the files
	// in a backup storage location.

	if(!doStorageBackupOps) {

		// Copy the shared decoderStateInfo structure to the decoderStateInfoPrivate stucture that is used by the decoder.
		RFD_MEMCPY(decoderStateInfoPrivate, decoderStateInfo, sizeof(RFD_XCC_DECODE_STATE_INFO));

		// update dec state file to indicate the current src and enc blocks in the input buffer have been
		// transferred to the solution buffer.
		// Then clear the counts to zero for the current src and enc blocks in the input buffer
		decoderStateInfo->nTransformRows +=
						(decoderStateInfo->nChkBlksInBuf +
						 decoderStateInfo->nSrcBlksInBuf);

		decoderStateInfo->nChkBlksInBuf = 0;
		decoderStateInfo->nSrcBlksInBuf = 0;

		// while decoding, set collection threshold to maximum, effectively disabling the update event
		// (the collector continues to collect blocks, but most likely will never reach the threshold
		// to signal collection event).
		// After decoding complete, the processor task will recalculate and set the collection threshold.
		decoderStateInfo->blksCollectedEventThresh = codecConfigInfo->fileLenBlocks + maxExtraBlocksForSolution;
	}
	// else only do the file storage updates, with existing state info.

	/////////////////////////////////////////////
	// Update storage in preparation for decode.
	/////////////////////////////////////////////

	// Note: set these flags after decoderStateInfo has been
	// copied to decoderStateInfoPrivate
	if(decoderStateInfoPrivate->nSrcBlksInBuf > 0) {
		prepareSrcBlksInBuf = TRUE;
	}
	else {
		prepareSrcBlksInBuf = FALSE;
	}

	if(decoderStateInfoPrivate->nChkBlksInBuf > 0) {
		prepareChkBlksInBuf = TRUE;
	}
	else {
		prepareChkBlksInBuf = FALSE;
	}

	// Note: call this function after the decoder state struct (and 'private' version) has been updated
	// (decode state struct (and 'private' version) is written to storage).
	status = RFDR_PrepareStorageForDecodeXcc(pathBuf, fileProcessInfo->decInfo.xccDecodeInfo,
											 prepareSrcBlksInBuf, prepareChkBlksInBuf, doStorageBackupOps);
	if(RFD_STATUS_OK != status) {
		return status;
	}

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_PostDecodeOpsXccCode(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo, BOOL * pIsDecodeComplete)
{
	RFD_XCC_CODEC_CONFIG_INFO * codecConfigInfo = &fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo;
	RFD_XCC_DECODE_STATE_INFO * decoderStateInfo = &fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo;
	RFD_XCC_DECODE_STATE_INFO * decoderStateInfoPrivate = &fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfoPrivate;
	RFD_XCC_DECODE_EVENT_BLOCK_THRESH_INFO * decodeEventBlockThreshInfo = &fileProcessInfo->decInfo.xccDecodeInfo->decodeEventBlockThreshInfo;
	TCHAR * fileNameBuf = NULL;
	RFD_STATUS status = RFD_STATUS_ERROR_GENERAL;

	*pIsDecodeComplete = FALSE;

	if(NULL == (fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	//////////
	// a decode chunk is completed prior to this fcn. call.
	// This fcn updates the decoder state info that is shared with the collector thread.
	// The collector/processor mutex must be aquired prior to this fcn. call.
	//////////

	if(decoderStateInfoPrivate->nextPivotNo == codecConfigInfo->fileLenBlocks)
	{
		// The entire file has been decoded, signalled by nextPivotNo = fileLenBlocks.
		// just set blksCollectedEventThresh to some default value (won't be used anymore since decoding is complete.
		decoderStateInfo->blksCollectedEventThresh = codecConfigInfo->fileLenBlocks + decodeEventBlockThreshInfo->extraBlocksIncrement;
		*pIsDecodeComplete = TRUE;
	}
	else if(decoderStateInfoPrivate->nextPivotNo < codecConfigInfo->fileLenBlocks)
	{
		// decode still in progess.

		//////////
		// update collection threshold
		//////////
		if(decoderStateInfo->decodeChunkCount >= (decodeEventBlockThreshInfo->numeratorArrayLen-1))
		{
			decoderStateInfo->blksCollectedEventThresh = codecConfigInfo->fileLenBlocks +
														 decodeEventBlockThreshInfo->extraBlocksIncrement;
			// Note: checks to this event threshold should also be accompanied by
			// checks to Max Number of Blocks Collected condition.
			// i.e. decode should be called for either check in case the event threshold
			// is effectively higher than the Max block collection limit.
		}
		else
		{
			while(decoderStateInfo->decodeChunkCount < (decodeEventBlockThreshInfo->numeratorArrayLen-1))
			{
				decoderStateInfo->decodeChunkCount++;
				decoderStateInfo->blksCollectedEventThresh =
					((UINT32) decodeEventBlockThreshInfo->blockThresholdNumeratorArray[decoderStateInfo->decodeChunkCount] *
					 codecConfigInfo->fileLenBlocks) / decodeEventBlockThreshInfo->blockThreshDenominator;

				if(decoderStateInfo->blksCollectedEventThresh > /* note: Private */ decoderStateInfoPrivate->nextPivotNo)
				{
					break;
				}
			}
		}
	}
	else // decoderStateInfoPrivate->nextPivotNo > codecConfigInfo->fileLenBlocks
	{
		// this is an error condition, shouldn't happen.
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_INVALID_VALUE;
	}

	//////////
	// "synchronize" decoderStateInfoPrivate and decoderStateInfo
	//////////

	decoderStateInfo->nTransformRows = decoderStateInfoPrivate->nTransformRows;
	decoderStateInfo->nChkBlksEliminated = decoderStateInfoPrivate->nChkBlksEliminated;
	decoderStateInfo->nSrcBlocksInSolution = decoderStateInfoPrivate->nSrcBlocksInSolution;
	decoderStateInfo->nextPivotNo = decoderStateInfoPrivate->nextPivotNo;

	//////////
	// Write Decode State to storage.
	//////////

	// create file path string for the Decoder State filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in the collect dir
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_STATE_FNAME);

	// write the Decoder State structure to the Decoder State file,
	// with compile-time configured file sync setting
	if(RFD_STATUS_OK != (status = RFD_WriteFile( fileNameBuf, decoderStateInfo,
		sizeof(RFD_XCC_DECODE_STATE_INFO), IS_RFDR_DIR_AND_FILE_SYNC_ENABLED) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	//////////
	// Write 'Private decoder' version of Decode State to storage.
	//
	// This update is technically not needed since 'private' decode state
	// won't be used anymore until the next decode operation where
	// the main decoder state is copied to the private decode state.
	// An alternative is to not update, or even delete the file.
	// Writing the file here may be advantagous only for debug/testing.
	//////////

	// create file path string for the Decoder State filename
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_STATE_PRIVATE_FNAME);

	// write the Private Decoder State structure to the Decoder State file,
	// with compile-time configured file sync setting
	if(RFD_STATUS_OK != (status = RFD_WriteFile( fileNameBuf, decoderStateInfoPrivate,
		sizeof(RFD_XCC_DECODE_STATE_INFO), IS_RFDR_DIR_AND_FILE_SYNC_ENABLED) ))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	// Note: could do some bounds error checking here.

	RFD_FREE(fileNameBuf);
	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_PrepareStorageForDecodeXcc(const TCHAR pathBuf[], RFDR_XCC_DECODE_INFO * xccDecodeInfo,
										   BOOL prepareSrcBlksInBuf, BOOL prepareChkBlksInBuf, BOOL doStorageBackupOps)
{
	TCHAR * fileNameBuf1, * fileNameBuf2;
	RFD_STATUS status;
	RFD_XCC_DECODE_STATE_INFO * decoderStateInfo = &xccDecodeInfo->decoderStateInfo;
	RFD_XCC_DECODE_STATE_INFO * decoderStateInfoPrivate = &xccDecodeInfo->decoderStateInfoPrivate;

	if(NULL == (fileNameBuf1 = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	if(NULL == (fileNameBuf2 = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		RFD_FREE(fileNameBuf1);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	///////
	// write the decode state to storage
	///////

	// create file path string for the Decoder State filename
	RFD_STRCPY_S(fileNameBuf1, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in collect directory
	RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	if(doStorageBackupOps) {
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	}
	RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_DECODE_STATE_FNAME);

	// write the Decoder State structure to the Decoder State file,
	// with compile-time configured file sync setting
	if(RFD_STATUS_OK != (status = RFD_WriteFile( fileNameBuf1, decoderStateInfo,
		sizeof(RFD_XCC_DECODE_STATE_INFO), IS_RFDR_DIR_AND_FILE_SYNC_ENABLED)))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf1);
		RFD_FREE(fileNameBuf2);
		return status;
	}

	///////
	// write the 'Private decoder' version of the decode state to storage
	///////

	// create file path string for the Decoder State filename
	RFD_STRCPY_S(fileNameBuf1, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in process directory
	RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	if(doStorageBackupOps) {
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	}
	RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_DECODE_STATE_PRIVATE_FNAME);

	// write the Decoder State structure to the Decoder State file,
	// with compile-time configured file sync setting
	if(RFD_STATUS_OK != (status = RFD_WriteFile( fileNameBuf1, decoderStateInfoPrivate,
		sizeof(RFD_XCC_DECODE_STATE_INFO), IS_RFDR_DIR_AND_FILE_SYNC_ENABLED)))
	{
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf1);
		RFD_FREE(fileNameBuf2);
		return status;
	}

	if(prepareSrcBlksInBuf) {

		///////////
		// rename src blk buf file to tmp filename used by the decoder
		// and then recreate the src blk buf file (zero len).
		///////////

		// create file path strings for the source block input buffer
		// and the temp filename it will be renamed to.
		RFD_STRCPY_S(fileNameBuf1, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in collect directory
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		if(doStorageBackupOps) {
			RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);
			RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		}
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_SRC_BLKS_INBUF_FNAME);

		RFD_STRCPY_S(fileNameBuf2, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in process directory
		RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		if(doStorageBackupOps) {
			RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);
			RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		}
		RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFDR_SRC_BLKS_INBUF_TMP_DECODE_FNAME);

		// rename the src block buffer file to the decoder tmp src block file name.
		if(RFD_RENAME( fileNameBuf1 /*old*/, fileNameBuf2 /*new*/))
		{
			// error, cleanup and return with error code.
			RFD_FREE(fileNameBuf1);
			RFD_FREE(fileNameBuf2);
			return RFD_STATUS_ERROR_FILE_RENAME;
		}

		// Create a zero length file for the src block buffer
		// with compile-time configured file sync setting
		if(RFD_STATUS_OK != (status = RFD_CreateFile(fileNameBuf1, IS_RFDR_DIR_AND_FILE_SYNC_ENABLED)))
		{
			// error, cleanup and return with error code.
			RFD_FREE(fileNameBuf1);
			RFD_FREE(fileNameBuf2);
			return status;
		}
	}

	if(prepareChkBlksInBuf) {

		///////////
		// rename enc blk buf file to tmp filename used by the decoder
		// and then recreate the enc blk buf file (zero len).
		///////////

		// create file path strings for the encoded block input buffer
		// and the temp filename it will be renamed to.
		RFD_STRCPY_S(fileNameBuf1, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME); // in collect directory
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		if(doStorageBackupOps) {
			RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);
			RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		}
		RFD_STRCAT_S(fileNameBuf1, RFD_MAX_PATH_LEN, RFDR_CHK_BLKS_INBUF_FNAME);

		RFD_STRCPY_S(fileNameBuf2, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in process directory
		RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		if(doStorageBackupOps) {
			RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);
			RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		}
		RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFDR_CHK_BLKS_INBUF_TMP_DECODE_FNAME);

		// rename the enc block buffer file to the decoder tmp enc block file name.
		if(RFD_RENAME( fileNameBuf1 /*old*/, fileNameBuf2 /*new*/))
		{
			// error, cleanup and return with error code.
			RFD_FREE(fileNameBuf1);
			RFD_FREE(fileNameBuf2);
			return RFD_STATUS_ERROR_FILE_RENAME;
		}

		// Create a zero length file for the enc block buffer
		// with compile-time configured file sync setting
		if(RFD_STATUS_OK != (status = RFD_CreateFile(fileNameBuf1, IS_RFDR_DIR_AND_FILE_SYNC_ENABLED)))
		{
			// error, cleanup and return with error code.
			RFD_FREE(fileNameBuf1);
			RFD_FREE(fileNameBuf2);
			return status;
		}
	}

	// cleanup and return
	RFD_FREE(fileNameBuf1);
	RFD_FREE(fileNameBuf2);
	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_CollectBlockXccCode(const TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo, RFD_BLOCK_MSG_STRUCT * hBlockMsg, BOOL * pSignalRequest)
{
	int i;
	UINT16 nChkBlksCollectedTotal;
	UINT32 blkIndex;
	RFD_XCC_ENC_BLK_INDEX * pEncBlkCollected;
	RFD_STATUS status;
	RFDR_XCC_DECODE_INFO * xccDecodeInfo = fileProcessInfo->decInfo.xccDecodeInfo;

	blkIndex = hBlockMsg->blkIndex;

	// initialize signal request to false.
	*pSignalRequest = FALSE;

	//////
	// Check if maximum number of allowable collected blocks is exceeded.
	//////
	if(RFDR_IsMaxAllowableNumBlocksCollectedXccCode( &fileProcessInfo->decInfo )) {
		// already collected the maximum allowable number of blocks.

		// It is possible that the block collection 'call decode' threshold was set
		// higher than the max allowable collected blocks threshold. Therefore, set the
		// 'call decode' signal in this max collected block condition. Otherwise, decode will never
		// be signalled for the final attempt.
		// The decode operation should attempt decode and if decode still not complete
		// it should reinitialize and restart this file process.
		*pSignalRequest = TRUE;

		return RFD_STATUS_OK;
	}

	//////
	// do collection processing based on block type
	//////
	switch(hBlockMsg->blkType)
	{

	case BLK_TYPE_ECC1_CHECK:
	case BLK_TYPE_ECC2_CHECK:
	case BLK_TYPE_SIMPLE_PARITY:

		// check if this block index has already been collected.

		pEncBlkCollected = xccDecodeInfo->encBlksCollectedList;
		nChkBlksCollectedTotal = xccDecodeInfo->decoderStateInfo.nChkBlksCollectedTotal;

		for(i=0;i<nChkBlksCollectedTotal;i++)
		{
			if(blkIndex == *pEncBlkCollected++)
			{
				// this block has already been collected.
				return RFD_STATUS_OK;
			}
		}

		// This is a new block.

		// Update structures for new block.
		xccDecodeInfo->encBlksCollectedList[nChkBlksCollectedTotal] = blkIndex;
		xccDecodeInfo->decoderStateInfo.nChkBlksCollectedTotal++;
		xccDecodeInfo->decoderStateInfo.nChkBlksInBuf++;

		// write block and updated structures to storage
		if(RFD_STATUS_OK != (status = RFDR_StoreCollectedBlockXccCode_Ex(pathBuf, fileProcessInfo, hBlockMsg)))
		{
			// error in store operation, try to undo changes to structures
			xccDecodeInfo->decoderStateInfo.nChkBlksCollectedTotal--;
			xccDecodeInfo->decoderStateInfo.nChkBlksInBuf--;
			return status;
		}
		break;

	case BLK_TYPE_SOURCE:

		// check if this block index has already been collected.

		if(xccDecodeInfo->srcBlksCollectedStatus[blkIndex] == TRUE)
		{
			// this block has already been collected.
			return RFD_STATUS_OK;
		}

		// This is a new block.

		// Update structures for new block.
		xccDecodeInfo->srcBlksCollectedStatus[blkIndex] = TRUE;
		xccDecodeInfo->decoderStateInfo.nSrcBlksCollectedTotal++;
		xccDecodeInfo->decoderStateInfo.nSrcBlksInBuf++;

		// write block and updated structures to storage
		if(RFD_STATUS_OK != (status = RFDR_StoreCollectedBlockXccCode_Ex(pathBuf, fileProcessInfo, hBlockMsg)))
		{
			// error is store operation, try to undo changes to structures
			xccDecodeInfo->srcBlksCollectedStatus[blkIndex] = FALSE;
			xccDecodeInfo->decoderStateInfo.nSrcBlksCollectedTotal--;
			xccDecodeInfo->decoderStateInfo.nSrcBlksInBuf--;

			return status;
		}
		break;

	default:
		return RFD_STATUS_ERROR_INVALID_BLOCK_TYPE;
	}

	// the block has been collected

	RFD_DPrint( TEXT("RFDR Collector: Block Stored, fileId %d, blkType %d, nSrcBlksInBuf %d, nSrcBlksCollectedTotal %d, nChkBlksInBuf %d, nChkBlksCollectedTotal %d,\n"),
				fileProcessInfo->fileInfo.id,
				hBlockMsg->blkType,
				xccDecodeInfo->decoderStateInfo.nSrcBlksInBuf,
				xccDecodeInfo->decoderStateInfo.nSrcBlksCollectedTotal,
				xccDecodeInfo->decoderStateInfo.nChkBlksInBuf,
				xccDecodeInfo->decoderStateInfo.nChkBlksCollectedTotal);

	////////
	// check if collection event should be signalled to the RFDR processor
	///////
	if(RFDR_IsBlockCollectionStateReadyForDecodeXccCode(&fileProcessInfo->decInfo))
	{
		// set signal request.
		*pSignalRequest = TRUE;
	}

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_StoreCollectedBlockXccCode(const TCHAR * pathBuf,
										   RFDR_FILE_PROCESS_INFO * fileProcessInfo,
										   RFD_BLOCK_TYPE blkType,
										   UCHAR * blockStorageBuf,
										   UINT16 blockStorageBufSize)
{
	TCHAR * fileNameBuf = NULL;
	RFD_STATUS status;
	UINT16 nChkBlksCollectedTotal;

	fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(fileNameBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	/////////
	// Mark start of storage transaction.
	// Note: Mutex must be aquired before calling transaction fcn.
	/////////
	RFD_MarkStorageTransactionStart(pathBuf);
	{
		/////////
		// Update storage elements dependent on block type.
		/////////

		switch(blkType)
		{

		case BLK_TYPE_ECC1_CHECK:
		case BLK_TYPE_ECC2_CHECK:
		case BLK_TYPE_SIMPLE_PARITY:

			/////////
			// Write the Encoded (Check) Block to Storage
			/////////

			// create file path string for the Source Block Buffer filename
			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_CHK_BLKS_INBUF_FNAME);

			// Append the block to the Block Storage Buffer File
			// with compile-time configured file sync setting
			status = RFD_AppendDataToFile( fileNameBuf, blockStorageBuf, blockStorageBufSize, IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);
			if(status != RFD_STATUS_OK) {
				// error, cleanup and return with error code.
				RFD_FREE(fileNameBuf);
				return status;
			}

			/////////
			// Write the enc block collected list to Storage.
			/////////

			// create file path string for the Decoder State filename
			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_CHK_BLKS_COLLECTED_FNAME);

			nChkBlksCollectedTotal = fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo.nChkBlksCollectedTotal;

			// write the enc block collected list to the enc block collected list file
#if 1
			// write the entire list, even though only a single entry has been appended to end of list.
			// this may be more efficient than appending to the file since this list is relatively short
			// i.e. probably shorter that a flash page size for flash type storage media

			// write the enc block collected list to the enc block collected list file,
			// with compile-time configured file sync setting
			status = RFD_WriteFile( fileNameBuf, &fileProcessInfo->decInfo.xccDecodeInfo->encBlksCollectedList,
				nChkBlksCollectedTotal * sizeof(RFD_XCC_ENC_BLK_INDEX), IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);
			if(status != RFD_STATUS_OK) {
				// error, cleanup and return with error code.
				RFD_FREE(fileNameBuf);
				return status;
			}
#else
			// alternative is to just update the file with the latest entry (append to the end of file),
			// with compile-time configured file sync setting
			status = RFD_AppendDataToFile( fileNameBuf,
				&fileProcessInfo->decInfo.xccDecodeInfo->encBlksCollectedList[nChkBlksCollectedTotal-1],
				sizeof(RFD_XCC_ENC_BLK_INDEX), IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);
			if(status != RFD_STATUS_OK) {
				// error, cleanup and return with error code.
				RFD_FREE(fileNameBuf);
				return status;
			}
#endif
			break;

		case BLK_TYPE_SOURCE:

			/////////
			// Write the Source Block to Storage
			/////////

			// create file path string for the Source Block Buffer filename
			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_SRC_BLKS_INBUF_FNAME);

			// Append the block to the Block Storage Buffer File,
			// with compile-time configured file sync setting
			status = RFD_AppendDataToFile( fileNameBuf, blockStorageBuf, blockStorageBufSize, 
										   IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);
			if(status != RFD_STATUS_OK) {
				// error, cleanup and return with error code.
				RFD_FREE(fileNameBuf);
				return status;
			}

			//////////////////////
			// Write srcBlksCollectedStatus array
			//////////////////////

			// create file path string for the source block collected status array
			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_SRC_BLKS_COLLECTED_FNAME);

			// write the source block collected status array to the corresponding file
			// with compile-time configured file sync setting
			status = RFD_WriteFile( fileNameBuf, &fileProcessInfo->decInfo.xccDecodeInfo->srcBlksCollectedStatus,
									sizeof(fileProcessInfo->decInfo.xccDecodeInfo->srcBlksCollectedStatus), 
									IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);
			if(status != RFD_STATUS_OK) {
				// error, cleanup and return with error code.
				RFD_FREE(fileNameBuf);
				return status;
			}
			break;

		default:

			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_INVALID_BLOCK_TYPE;
		}

		/////////
		// Write the Decoder State File to Storage
		/////////

		// create file path string for the Decoder State filename
		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_STATE_FNAME);

		// write the Decoder State structure to the Decoder State file
		// with compile-time configured file sync setting
		status = RFD_WriteFile( fileNameBuf, &fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo,
								sizeof(RFD_XCC_DECODE_STATE_INFO),
								IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);
		if(status != RFD_STATUS_OK) {
			// error, cleanup and return with error code.
			RFD_FREE(fileNameBuf);
			return status;
		}
		// Note: the 'private decoder' version of the Decoder State File is NOT written by the collector.

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
		/////////
		// Sync the (Parent) Directory
		/////////
        // Sync the directory, e.g. if a new file was created in this directory
        // (e.g. due to open for write/create instead of appending)
        // do a directory sync to ensure it is persisted in the directory file.
        // Do this before marking the transaction complete.
        RFD_DirectorySync( pathBuf );
#endif

	/////////
	// Mark completion of storage transaction.
	// Note: Mutex must be aquired before calling transaction fcn.
	/////////
	}
    // Set argument doPreSyncOfDirectory = FALSE since we have already done this operation
    // here in this function.
	RFD_MarkStorageTransactionComplete(pathBuf, FALSE);

	// Cleanup and return

	RFD_FREE(fileNameBuf);
	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_StoreCollectedBlockXccCode_Ex(const TCHAR * pathBuf,
											  RFDR_FILE_PROCESS_INFO * fileProcessInfo,
											  RFD_BLOCK_MSG_STRUCT * hBlockMsg)
{
	TCHAR * collectPathBuf = NULL;
	UCHAR * blockStorageBuf = NULL;
	RFD_STATUS status;
	UINT16 blockStorageBufSize;

	// allocate the collector directory path buffer.
	collectPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(collectPathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	/////////
	// Construct a single buffer containing block index and the block payload
	// in preparation for storing the single buffer.
	/////////

	switch(hBlockMsg->blkType)
	{

	case BLK_TYPE_ECC1_CHECK:
	case BLK_TYPE_ECC2_CHECK:
	case BLK_TYPE_SIMPLE_PARITY:

		/////////
		// Construct block buffer for case of a Check/Parity type block.
		/////////

		// calc size of the block Storage buffer which contains the block index and the block payload
		blockStorageBufSize =	sizeof(RFD_XCC_ENC_BLK_INDEX) +
						fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo.blkLenBytes * sizeof(UCHAR);

		// allocate the block storage buffer.
		// the block (index and payload) is written to file storage in one step/one buffer
		// as this may be generally faster/more efficient (unless the file system has a decent write cache).
		blockStorageBuf = (UCHAR *) RFD_MALLOC( blockStorageBufSize );
		if(blockStorageBuf == NULL) {
			RFD_FREE(collectPathBuf);
			return RFD_STATUS_ERROR_MEM_ALLOC;
		}

		// copy the block index into the buffer
		*(RFD_XCC_ENC_BLK_INDEX *)blockStorageBuf =  (RFD_XCC_ENC_BLK_INDEX) hBlockMsg->blkIndex;

		// copy the block payload into the buffer
		RFD_MEMCPY(blockStorageBuf + sizeof(RFD_XCC_ENC_BLK_INDEX), hBlockMsg->block,
			fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo.blkLenBytes * sizeof(UCHAR));

		break;

	case BLK_TYPE_SOURCE:

		/////////
		// Construct block buffer for the case of a Source type block.
		/////////

		// calc size of the block Storage buffer which contains the block index and the block payload
		blockStorageBufSize =	sizeof(RFD_XCC_SRC_BLK_INDEX) +
						fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo.blkLenBytes * sizeof(UCHAR);

		// allocate the block storage buffer.
		// the block (index and payload) is written to file storage in one step/one buffer
		// as this may be generally faster/more efficient (unless the file system has a decent write cache).
		blockStorageBuf = (UCHAR *) RFD_MALLOC( blockStorageBufSize );
		if(blockStorageBuf == NULL) {
			RFD_FREE(collectPathBuf);
			return RFD_STATUS_ERROR_MEM_ALLOC;
		}

		// copy the block index into the buffer
		*(RFD_XCC_SRC_BLK_INDEX *)blockStorageBuf =  (RFD_XCC_SRC_BLK_INDEX) hBlockMsg->blkIndex;

		// copy the block payload into the buffer
		RFD_MEMCPY(blockStorageBuf + sizeof(RFD_XCC_SRC_BLK_INDEX), hBlockMsg->block,
			fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo.blkLenBytes * sizeof(UCHAR));

		break;

	default:

		RFD_FREE(collectPathBuf);
		return RFD_STATUS_ERROR_INVALID_BLOCK_TYPE;
	}

	// Construct collect directory path
	RFD_STRCPY_S(collectPathBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(collectPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(collectPathBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);

	/////////////////////////
	// Call final function to write the block buffer and other related structures to storage.
	/////////////////////////
	status = RFDR_StoreCollectedBlockXccCode(collectPathBuf, fileProcessInfo,
											 hBlockMsg->blkType, blockStorageBuf, blockStorageBufSize);
	if(status != RFD_STATUS_OK) {
		RFD_FREE(collectPathBuf);
		RFD_FREE(blockStorageBuf);
		return status;
	}

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
	// Store the block buffer and related structures to the backup location.

	// Construct collect backup directory path
	RFD_STRCPY_S(collectPathBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(collectPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(collectPathBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);
	RFD_STRCAT_S(collectPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(collectPathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);

	/////////////////////////
	// Call final function to write the block buffer and other related structures
	// to the redundant/backup storage location.
	/////////////////////////
	status = RFDR_StoreCollectedBlockXccCode(collectPathBuf, fileProcessInfo,
											 hBlockMsg->blkType, blockStorageBuf, blockStorageBufSize);
	if(status != RFD_STATUS_OK) {
		RFD_FREE(collectPathBuf);
		RFD_FREE(blockStorageBuf);
		return status;
	}
#endif

	// Cleanup and return

	RFD_FREE(collectPathBuf);
	RFD_FREE(blockStorageBuf);

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
BOOL RFDR_IsMaxAllowableNumBlocksCollectedXccCode( RFDR_DECODE_INFO * decodeInfo )
{
	RFDR_XCC_DECODE_INFO * xccDecodeInfo = decodeInfo->xccDecodeInfo;
#if 0
	// This is an alternative limit check. To implement this 'Revert Decode' operation
	// must be changed to ensure that the 'copy back' appended total does not exceed limits
	// See notes below.

	// Check if maximum number of allowable collected blocks is exceeded.
	if( (  xccDecodeInfo->decoderStateInfo.nChkBlksInBuf
		+ xccDecodeInfo->decoderStateInfo.nSrcBlksInBuf
		+ xccDecodeInfo->decoderStateInfo.nTransformRows )
		>= (xccDecodeInfo->codecConfigInfo.fileLenBlocks + xccDecodeInfo->maxExtraBlocksForSolution) )
	{
		// already collected the maximum allowable number of blocks.
		return TRUE;
	}
	else if( (  xccDecodeInfo->decoderStateInfo.nChkBlksCollectedTotal )
		     >= (xccDecodeInfo->codecConfigInfo.fileLenBlocks + xccDecodeInfo->maxExtraBlocksForSolution) )
	{
		// already collected the maximum allowable number of blocks.
		return TRUE;
	}
	else {
		return FALSE;
	}
#else
	// Check if maximum number of allowable collected blocks is exceeded.
	// Note: some number of these total blocks may be 'eliminated' by the decoder.
	// The comparison is done against the 'Total' counts instead of the 'InBuf' plus nTransformRows.
	// This is simpler than comparison against (nChkBlksInBuf + nSrcBlksInBuf + nTransformRows)
	// because of the Decode Revert implementation whereby an interrupted decode will 'copy back append'
	// transferred blocks back to the 'InBuf'. It's possible that this 'copy back append' could exceed
	// (fileLenBlocks + maxExtraBlocksForSolution) if block collection is fast and decode is slow or suspended
	// for some reason.
	// The disadvantage is that this puts a hard limit on the block limit before failure.
	//
	// However, we have a hard limit on the check block nChkBlksCollectedTotal anyway due to
	// the finite encBlksCollectedList array size. So a 'total' limit comparison is required anyway
	// i.e.  nChkBlksCollectedTotal >= (fileLenBlocks + maxExtraBlocksForSolution)
	// instead of (nChkBlksCollectedTotal + nSecBlksCollectedTotal) >= (fileLenBlocks + maxExtraBlocksForSolution)
	//
	// With this 'total' limit comparison implementation, maxExtraBlocksForSolution should be set to ensure decoding success
	// with very high probability. In the very rare case that decoding fails due to this limit,
	// file processing should be reset.
	if( (  xccDecodeInfo->decoderStateInfo.nChkBlksCollectedTotal
		 + xccDecodeInfo->decoderStateInfo.nSrcBlksCollectedTotal )
		>= (xccDecodeInfo->codecConfigInfo.fileLenBlocks + xccDecodeInfo->maxExtraBlocksForSolution) )
	{
		// already collected the maximum allowable number of blocks.
		return TRUE;
	}
	else {
		return FALSE;
	}
#endif
}

///////////////////////////////////////////////////////////////////////////////
BOOL RFDR_IsBlockCollectionStateReadyForDecodeXccCode(RFDR_DECODE_INFO * decodeInfo)
{
	int i;
	UINT16 srcBlockCheckLimit;
	int numSrcBlksBelowCollectThreshold;
	BOOL isReadyForDecode = FALSE;
	RFDR_XCC_DECODE_INFO * xccDecodeInfo = decodeInfo->xccDecodeInfo;

	////////
	// check if collection event should be signalled to the RFDR processor
	///////

	if( xccDecodeInfo->decoderStateInfo.nChkBlksInBuf > 0 ||
		xccDecodeInfo->decoderStateInfo.nSrcBlksInBuf > 0) {
		// There is at least one source block or check block in collection buffer,
		// therefore, new input is available to decode.

		if(!RFDR_IsMaxAllowableNumBlocksCollectedXccCode(decodeInfo)) {
			// Maximum allowable number of collected blocks is Not reached.
			// Do standard blksCollectedEventThresh checking.

			if(xccDecodeInfo->decoderStateInfo.blksCollectedEventThresh > xccDecodeInfo->codecConfigInfo.fileLenBlocks) {
				srcBlockCheckLimit = xccDecodeInfo->codecConfigInfo.fileLenBlocks;
			}
			else {
				srcBlockCheckLimit = xccDecodeInfo->decoderStateInfo.blksCollectedEventThresh;
			}

			// count the number of collected blocks below the current event threshold.
			numSrcBlksBelowCollectThreshold = 0;
			for(i=0;i<srcBlockCheckLimit;i++) {
				if(xccDecodeInfo->srcBlksCollectedStatus[i] == TRUE) {
					numSrcBlksBelowCollectThreshold++;
				}
			}

			// Note: Here is an alternate calculation of nChkBlksEliminated, used in below calculation.
			// numEliminatedEncBlocks =
			//	(xccDecodeInfo->decoderStateInfo.nChkBlksCollectedTotal
			//	 - xccDecodeInfo->decoderStateInfo.nChkBlksInBuf)  // =   number of check blocks in transform + number of check blocks eleminated
			//	+ (xccDecodeInfo->decoderStateInfo.nSrcBlksCollectedTotal
			//	 - xccDecodeInfo->decoderStateInfo.nSrcBlksInBuf)  // =  + number of source blocks in transform
			//	- xccDecodeInfo->decoderStateInfo.nTransformRows; // = -(number of check blocks in transform + source blocks in transform)

			// signal processing event if
			// number of source blocks collected below current event threshold
			// plus the number of NON-eliminated collected enc blocks
			// is greater than or equal to the current event threshold.
			if( (numSrcBlksBelowCollectThreshold
				+ xccDecodeInfo->decoderStateInfo.nChkBlksCollectedTotal
				- xccDecodeInfo->decoderStateInfo.nChkBlksEliminated)
				>= xccDecodeInfo->decoderStateInfo.blksCollectedEventThresh ) {
				// set signal status.
				isReadyForDecode = TRUE;
			}
		}
		else {
			// Maximum allowable number of colleted blocks is reached.
			// Decoding should be attempted once more, regardless of blksCollectedEventThresh check
			// i.e. the threshold may never be met since no new blocks will be collected.
			// Set signal status.
			isReadyForDecode = TRUE;
		}
	}
	// else no new data is available for decode.

	return isReadyForDecode;
}

///////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Get decoder solution stats for the extended ecc code.
///
/// This funcion can be called after decoding is complete to get the decoding statistics. It must
/// be called while still in the decoding state, before fileProcessInfo->decInfo.xccDecodeInfo is
/// de-allocated.
///
/// @param  fileProcessInfo Information describing the file process.
/// @param  decodeStats     (output) The decoder statistics.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFDR_GetDecoderSolutionStatsXccCode(RFDR_FILE_PROCESS_INFO * fileProcessInfo, RFDR_DECODE_STATS * decodeStats)
{
	if(fileProcessInfo == NULL) {
		return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
	}

	if(fileProcessInfo->decInfo.xccDecodeInfo == NULL) {
		return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
	}

	decodeStats->nChkBlksEliminated		= fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo.nChkBlksEliminated;
	decodeStats->nSrcBlocksInSolution		= fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo.nSrcBlocksInSolution;
	decodeStats->nTotalBlocksInSolution    = fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo.fileLenBlocks;
	decodeStats->nChkBlocksInSolution		= decodeStats->nTotalBlocksInSolution - decodeStats->nSrcBlocksInSolution;
	decodeStats->nDecodeChunks				= fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfo.decodeChunkCount;

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_InitializeDecodeBlockThresholdInfoEccCode(RFD_XCC_DECODE_EVENT_BLOCK_THRESH_INFO * decodeBlockThresholdInfo)
{
	UINT32 totalTime, currThreshTime, nextThreshTime, threshDeltaTime;
	UINT32 currStep, nDecodeChunks, i;

	nDecodeChunks = RFD_ECC_DECODE_NUM_DECODE_CHUNKS;

	if(nDecodeChunks > RFDR_MAX_NUM_DECODE_CHUNKS) {
		return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
	}

	// clear the entire threshold array (max size) for consistent contents.
	RFD_MEMSET(decodeBlockThresholdInfo->blockThresholdNumeratorArray, 0, RFDR_MAX_NUM_DECODE_CHUNKS * sizeof(INT16));

	decodeBlockThresholdInfo->blockThreshDenominator = RFD_ECC_DECODE_CHUNK_BLK_THRESH_DENOM;
	decodeBlockThresholdInfo->extraBlocksIncrement = RFD_ECC_DECODE_EXTRA_BLKS_CHUNK_INC;
	decodeBlockThresholdInfo->numeratorArrayLen = nDecodeChunks;

	/////////////////////////////////////////////////////////////////
	// Estimate the total processing time as proportional to the square of the number of blocks availiable.
	// This "square" relationship should be accurate when the block size is much larger than
	// the coefficient matrix dimension (number of blocks that makeup a file).
	// If the coefficient matrix dimension (actually, the number of bytes per row of the matrix)
	// is on the same order as the block length (in bytes), then the processing of the coefficient
	// matrix part of the overall solution has a more significant contribution.
	// However, a block length on the order of the size of the matrix dimension implies a small file size
	// in which case the overall processing time is lower, and the accuracy in "chunking" the decoding
	// into equal processing-periods is not as critical.
	/////////////////////////////////////////////////////////////////

	totalTime = RFD_ECC_DECODE_CHUNK_BLK_THRESH_DENOM * RFD_ECC_DECODE_CHUNK_BLK_THRESH_DENOM;
	threshDeltaTime = totalTime / nDecodeChunks;

	if(threshDeltaTime == 0) {
		return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
	}

	currStep = 0;
	currThreshTime = 0;
	//nextThreshTime = threshDeltaTime;

	for(i=0;i<nDecodeChunks;i++) {

		nextThreshTime = threshDeltaTime * (i+1);

		while(currThreshTime < nextThreshTime) {
			currStep++;
			currThreshTime = (currStep * currStep);

			if(currStep >= RFD_ECC_DECODE_CHUNK_BLK_THRESH_DENOM) {
				currStep = RFD_ECC_DECODE_CHUNK_BLK_THRESH_DENOM;
				break;
			}
		}

		if(i > 0) {
			// ensure that the determined step for new chunk is greater than the
			// step for the previous chunk.
			if(currStep <= decodeBlockThresholdInfo->blockThresholdNumeratorArray[i-1]) {
				currStep = decodeBlockThresholdInfo->blockThresholdNumeratorArray[i-1] + 1;
			}
		}

		if(i == nDecodeChunks -1 )  {
			// last chunk threshold should always be exacly equal to RFD_ECC_DECODE_CHUNK_BLK_THRESH_DENOM.
			// Set this here in case of roundoff error resulting in lower value currStep.
			decodeBlockThresholdInfo->blockThresholdNumeratorArray[i] = RFD_ECC_DECODE_CHUNK_BLK_THRESH_DENOM;
		}
		else if(currStep >= RFD_ECC_DECODE_CHUNK_BLK_THRESH_DENOM) {
			// else not the last chunk, but the last step is prematurely reached.
			decodeBlockThresholdInfo->blockThresholdNumeratorArray[i] = RFD_ECC_DECODE_CHUNK_BLK_THRESH_DENOM;
			// in case last step reached earlier than expected due to roundoff errors.
			decodeBlockThresholdInfo->numeratorArrayLen = i;
			break;
		}
		else {
			decodeBlockThresholdInfo->blockThresholdNumeratorArray[i] = currStep;
		}
	}

	return RFD_STATUS_OK;
}
