////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_receiver\rfdr_processor.c
///
/// @brief	rfdr processor class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd_receiver.h"
#include "rfd_codec_xcc.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Construct RFDR File Instance (fragment) directory path name.
///
/// @param	fragmentPathBuf			String Buffer for output fragment directory path.
/// @param	fragmentPathBufLength	Length of the string buffer.
/// @param	clientInfo				Information describing the client.
/// @param	fileId					Identifier of the file instance (fragment).
////////////////////////////////////////////////////////////////////////////////////////////////////

void RFDR_ConstructFragmentDirPathName(TCHAR fragmentPathBuf[],
									   int fragmentPathBufLength,
									   RFDR_CLIENT_INFO_STRUCT * clientInfo, UINT16 fragmentId)
{
	RFD_SPRINTF_S(fragmentPathBuf, fragmentPathBufLength, TEXT("%s%s%s%s%05d"),
				  RFDR_BASE_PATH,
				  RFD_PATH_SEPARATOR,
				  clientInfo->clientDirName,
				  RFD_PATH_SEPARATOR,
				  fragmentId);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Construct RFDR Client directory path name.
///
/// @param	clientPathBuf		String Buffer for output fragment directory path.
/// @param	clientPathBufLength	Length of the client path buffer.
/// @param	clientInfo			Information describing the client.
////////////////////////////////////////////////////////////////////////////////////////////////////

void RFDR_ConstructClientDirPathName(TCHAR clientPathBuf[],
									 int clientPathBufLength,
									 RFDR_CLIENT_INFO_STRUCT * clientInfo)
{
	RFD_SPRINTF_S(clientPathBuf, clientPathBufLength, TEXT("%s%s%s"),
		          RFDR_BASE_PATH,
				  RFD_PATH_SEPARATOR,
				  clientInfo->clientDirName);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Do storage backup of a fragment (file instance).
///
/// The two underlying directories (process and collect) are backed up to their corresponding
/// storage locations. The fragment level directory does not itself have a dedicated backup.
///
/// An alternate use case is to cleanup (remove all files) from the process backup directory
/// (controlled by the doCleanupForProcessBackupDir parameter). This alternate use case is in
/// support of a storage space optimization so that all decoding files are not simultaneously
/// backed-up in storage for each fragment instance
/// i.e. during collection there is no need to have all files for all the  process directories
/// backed-up.
///
/// The storage transaction markers must be in the Complete state for the fragement level
/// directory and the two underlying directories.
///
/// @param  fragmentPathBuf                 Path name to the fragment (file instance) level
///                                         directory.
/// @param  doCleanupForProcessBackupDir    true to do cleanup for process backup directory
///                                         instead of normal backup from process directory.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_DoFragmentBackup(const TCHAR fragmentPathBuf[], BOOL doCleanupForProcessBackupDir)
{
	TCHAR * pathBuf, * backupPathBuf;
	RFD_STATUS status;
	RFD_STORAGE_SYNC_COUNT_TYPE syncCount;

	pathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(pathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	backupPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(backupPathBuf == NULL) {
		RFD_FREE(pathBuf);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	/////////////
	// Backup the Collector directory
	/////////////

	// Construct file path for the Collector directory
	RFD_STRCPY_S(pathBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);

	status = RFD_BackupDirectoryContents(pathBuf);

	if(status != RFD_STATUS_OK) {
		// Backup failed.
		RFD_DPrint( TEXT("RFDR Processor: Collector Directory backup failed, status = %d, path = %s\n"), status, pathBuf);
		RFD_FREE(pathBuf);
		RFD_FREE(backupPathBuf);
		return status;
	}

	/////////////
	//    Backup the Process directory
	// OR Cleanup the Process directory if doCleanupForProcessDir is requested.
	/////////////

	// Construct file path for the Process directory
	RFD_STRCPY_S(pathBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

	if(doCleanupForProcessBackupDir) {

		// Do cleanup of Process backup directory instead of a regular backup operation.

		status = RFD_CleanupBackupStorageContents(pathBuf);

		if(status != RFD_STATUS_OK) {
			// Cleanup failed.
			RFD_DPrint( TEXT("RFDR Processor: Process Directory cleanup failed, status = %d, path = %s\n"), status, pathBuf);
			RFD_FREE(pathBuf);
			RFD_FREE(backupPathBuf);
			return status;
		}

		// The Cleanup operation removed the backup transaction marker file.
		// Create/Initialize the backup tranaction marker file, synchronizing with
		// the sync. count of the 'parent' directory.

		RFD_STRCPY_S(backupPathBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(backupPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(backupPathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);

		status = RFD_GetStorageTransSyncCount(pathBuf, &syncCount);
		if(status != RFD_STATUS_OK) {
			RFD_FREE(pathBuf);
			RFD_FREE(backupPathBuf);
			return status;
		}

		status = RFD_InitializeStorageTransaction(backupPathBuf, syncCount);
		if(status != RFD_STATUS_OK) {
			RFD_FREE(pathBuf);
			RFD_FREE(backupPathBuf);
			return status;
		}

	}
	else {
		// do a regular backup of the process directory.
		status = RFD_BackupDirectoryContents(pathBuf);

		if(status != RFD_STATUS_OK) {
			// Backup failed.
			RFD_DPrint( TEXT("RFDR Processor: Process Directory backup failed, status = %d, path = %s\n"), status, pathBuf);
			RFD_FREE(pathBuf);
			RFD_FREE(backupPathBuf);
			return status;
		}
	}

	RFD_FREE(pathBuf);
	RFD_FREE(backupPathBuf);
	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Restore a fragment (file instance) from backup storage.
///
/// The two underlying directories (process and collect) are restored from their corresponding
/// storage locations. The fragment level directory does not itself have a dedicated backup (no
/// restore of files in fragment dir).
///
/// @param  fragmentPathBuf Path name for the fragment (file instance) level directory.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_DoFragmentRestore(const TCHAR fragmentPathBuf[])
{
	TCHAR * pathBuf;
	RFD_STATUS status;

	pathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(pathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	/////////////
	// Restore the Collector directory
	/////////////

	// Construct file path for the Collector directory
	RFD_STRCPY_S(pathBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);

	status = RFD_RestoreDirectoryContents(pathBuf);

	if(status != RFD_STATUS_OK) {
		// Backup failed.
		RFD_DPrint( TEXT("RFDR Processor: Collector Directory restore failed, status = %d, path = %s\n"), status, pathBuf);
		RFD_FREE(pathBuf);
		return status;
	}

	/////////////
	// Restore the Process directory
	/////////////

	// Construct file path for the Process directory
	RFD_STRCPY_S(pathBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

	status = RFD_RestoreDirectoryContents(pathBuf);

	if(status != RFD_STATUS_OK) {
		// Backup failed.
		RFD_DPrint( TEXT("RFDR Processor: Process Directory restore failed, status = %d, path = %s\n"), status, pathBuf);
		RFD_FREE(pathBuf);
		return status;
	}

	RFD_FREE(pathBuf);
	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Mark storage transaction start involving multiple directories under a fragment (file
/// instance).
///
/// For the primary use case (markForBackup = FALSE):
///
/// To accomodate atomically updating the contents of the multiple directories under a fragement
/// dir, all transaction files corresponding to each directory are marked as
/// *Started*. The three directories are the top level fragment directory and the
/// underlying process and collect directories.
///
/// For the alternate use case (markForBackup = TRUE):
///
/// Only the *backup* locations corresponding to the underlying process and collect directories
/// are marked (the top level fragement dir does not have a corresponding backup). This use case
/// is intended for 'optimized' backup whereby specific files (only files changed) are 'manually'
/// writted to backup, rather than all files of a directory written in which case the standard
/// backup functions manage the transaction marking for the backup directory.
///
/// Using the isProcessTransacInProgress paramete:
///
/// The \e isProcessTransacInProgress parameter may be used to 'tranaction couple' a process
/// transaction that is functionally being completed and a fragment transaction that is being
/// started. It is set True when it is desireable that a the process transaction NOT be marked
/// for complete for the period between the end of the RFD_MarkStorageTransactionComplete() and
/// the RFD_DoFragmentMarkStorageTransactionStart() (e.g. if a powerdown/abort at this point
/// would result in an error contition because this point does not correspond to a 'well defined'
/// continuation state on the following powerup). To use this feature, the user skips the call to
/// RFD_MarkStorageTransactionComplete() for the process directory and then calls
/// RFD_DoFragmentMarkStorageTransactionStart() with parameter isProcessTransacInProgress = True.
/// Note: One side effect of this usage is that the process transaction sync count will not be
/// incremented by the RFD_DoFragmentMarkStorageTransactionStart() (because the
/// RFD_MarkStorageTransactionStart()
/// for the process directory is skipped internally). The user should be aware of this in case a
/// 'manual' update of the corresponding backup directories is to follow. This is not an issue if
/// a standard 'backup' function is used (i.e. RFD_DoFragmentBackup())
/// which ensures the backup transaction sync count is equal to the base directory's sync count
/// on completion.
///
/// @param  fragmentPathBuf             Path name for the fragment (file instance) level
///                                     directory.
/// @param  markForBackup               true to mark transactions in underlying backup
///                                     directories in preparation for 'manual/customized' backup
///                                     type operation.
/// @param  isProcessTransacInProgress  true if process directory transaction is already in
///                                     progress. This function then avoids calling
///                                     RFD_DoFragmentMarkStorageTransactionStart() for the
///                                     process directory.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_DoFragmentMarkStorageTransactionStart(const TCHAR fragmentPathBuf[], BOOL markForBackup, BOOL isProcessTransacInProgress)
{
	TCHAR * pathBuf;
	RFD_STATUS status;

	pathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(pathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	/////////////
	// For the base use case (markForBackup = FALSE):
	// Mark Transaction Start for the top-level fragment directory first,
	// followed by the two underlying directories.
	// The order is not critical in the case of *Starting* the multi-directory transaction
	// (unlike the case of *Completing* the multi-directory transaction).
	//
	// For the alternate use case (markForBackup = TRUE):
	// The top-level fragement transaction marker is Not marked (there is no backup location
	// corresponding this directory). Only the backup locations for the underlying process and collect
	// directories are marked.
	/////////////

	// Do not mark the top-level fragment transaction for start in the case of backup preparation usage.
	// (there is no corresponding backup dir for the top-level fragment directory).
	if(!markForBackup) {

		/////////////
		// Mark Transaction Start for the top-level fragment directory
		/////////////

		status = RFD_MarkStorageTransactionStart(fragmentPathBuf);

		if(status != RFD_STATUS_OK) {
			RFD_DPrint( TEXT("Transaction Start failed, status = %d, path = %s\n"), status, pathBuf);
			RFD_FREE(pathBuf);
			return status;
		}
	}

	/////////////
	// Mark Transaction Start for the Collect directory
	/////////////

	// Construct file path for the Collector directory
	RFD_STRCPY_S(pathBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);
	if(markForBackup) {
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);
	}

	status = RFD_MarkStorageTransactionStart(pathBuf);

	if(status != RFD_STATUS_OK) {
		RFD_DPrint( TEXT("Transaction Start failed, status = %d, path = %s\n"), status, pathBuf);
		RFD_FREE(pathBuf);
		return status;
	}

	/////////////
	// Mark Transaction Start for the Process directory.
	//
	// Mark only if the process directory is indicated as
	// already marked as started (in progress) via the
	// isProcessTransacInProgress input parameter set to false.
	/////////////
	if(!isProcessTransacInProgress) {

		// Construct file path for the Process directory
		RFD_STRCPY_S(pathBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);
		if(markForBackup) {
			RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);
		}

		status = RFD_MarkStorageTransactionStart(pathBuf);

		if(status != RFD_STATUS_OK) {
			RFD_DPrint( TEXT("Transaction Start failed, status = %d, path = %s\n"), status, pathBuf);
			RFD_FREE(pathBuf);
			return status;
		}
	}

	RFD_FREE(pathBuf);
	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Mark storage transaction complete involving multiple directories under a fragment (file
/// instance).
///
/// For the primary use case (markForBackup = FALSE):
///
/// To accomodate atomically updating the contents of the multiple directories under a fragement
/// dir, all transaction files corresponding to each directory are marked as
/// *Completed*. The three directories are the top level fragment directory and the
/// underlying process and collect directories.
///
/// For the alternate use case (markForBackup = TRUE):
///
/// Only the *backup* locations corresponding to the underlying process and collect directories
/// are marked (the top level fragement dir does not have a corresponding backup). This use case
/// is intended for 'optimized' backup whereby specific files (only files changed) are 'manually'
/// writted to backup, rather than all files of a directory written in which case the standard
/// backup functions manage the transaction marking for the backup directory.
///
/// @param  fragmentPathBuf Path name for the fragment (file instance) level directory.
/// @param  markForBackup   true to mark transactions in underlying backup directories in
///                         preparation for 'manual/customized' backup type operation.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_DoFragmentMarkStorageTransactionComplete(const TCHAR fragmentPathBuf[], BOOL markForBackup)
{
	TCHAR * pathBuf;
	RFD_STATUS status;

	pathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(pathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	/////////////
	// For the base use case (markForBackup = FALSE):
	// Mark Transaction Complete for the underlying directories first,
	// followed by top-level fragment directory last.
	// The order is critical in the case of *Completing* the multi-directory transaction.
	//
	// For the alternate use case (markForBackup = TRUE):
	// The top-level fragement transaction marker is Not marked (there is no backup location
	// corresponding this directory). Only the backup locations for the underlying process and collect
	// directories are marked.
	/////////////


	/////////////
	// Mark Transaction Complete for the Collect directory
	/////////////

	// Construct file path for the Collector directory
	RFD_STRCPY_S(pathBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);
	if(markForBackup) {
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);
	}

	status = RFD_MarkStorageTransactionComplete(pathBuf, TRUE);

	if(status != RFD_STATUS_OK) {
		RFD_DPrint( TEXT("Transaction Complete failed, status = %d, path = %s\n"), status, pathBuf);
		RFD_FREE(pathBuf);
		return status;
	}

	/////////////
	// Mark Transaction Complete for the Process directory
	/////////////

	// Construct file path for the Process directory
	RFD_STRCPY_S(pathBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);
	if(markForBackup) {
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);
	}

	status = RFD_MarkStorageTransactionComplete(pathBuf, TRUE);

	if(status != RFD_STATUS_OK) {
		RFD_DPrint( TEXT("Transaction Complete failed, status = %d, path = %s\n"), status, pathBuf);
		RFD_FREE(pathBuf);
		return status;
	}

	// Do not mark the top-level fragment transaction in the case of fragment backup operation usage.
	if(!markForBackup) {

		/////////////
		// Mark Transaction Complete for the top-level fragment directory.
		// This top-level directory must be the last directory to mark as complete.
		/////////////

		status = RFD_MarkStorageTransactionComplete(fragmentPathBuf, TRUE);

		if(status != RFD_STATUS_OK) {
			RFD_DPrint( TEXT("Transaction Complete failed, status = %d, path = %s\n"), status, pathBuf);
			RFD_FREE(pathBuf);
			return status;
		}
	}

	RFD_FREE(pathBuf);
	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
void RFDR_SignalErrorExit( RFD_EVENT_HANDLE errorExitEvent )
{
	RFD_DPrint(TEXT("RFDR_SignalErrorExit() setting error event\n"));
	RFD_SetEvent(errorExitEvent);
}

///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFDR_IsMaxAllowableNumBlocksCollected(RFDR_FILE_PROCESS_INFO * fileProcessInfo)
{
	BOOL isMaxAllowableNumBlocksCollected = FALSE;

	switch(fileProcessInfo->fileInfo.codeType)
	{
	case RFD_CODE_TYPE_ECC1:
	case RFD_CODE_TYPE_ECC2:

		isMaxAllowableNumBlocksCollected = RFDR_IsMaxAllowableNumBlocksCollectedXccCode(&fileProcessInfo->decInfo);
		break;

	case RFD_CODE_TYPE_SIMPLE_BLK:

		isMaxAllowableNumBlocksCollected = RFDR_IsMaxAllowableNumBlocksCollectedSimpleCode(&fileProcessInfo->decInfo);
		break;

	default:
		isMaxAllowableNumBlocksCollected = FALSE;
		break;
	}

	return isMaxAllowableNumBlocksCollected;
}

///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFDR_IsBlockCollectionStateReadyForDecode(RFDR_FILE_PROCESS_INFO * fileProcessInfo)
{
	BOOL isSuffientForDecode = FALSE;

	switch(fileProcessInfo->fileInfo.codeType)
	{
	case RFD_CODE_TYPE_ECC1:
	case RFD_CODE_TYPE_ECC2:

		isSuffientForDecode = RFDR_IsBlockCollectionStateReadyForDecodeXccCode(&fileProcessInfo->decInfo);
		break;

	case RFD_CODE_TYPE_SIMPLE_BLK:

		isSuffientForDecode = RFDR_IsBlockCollectionStateReadyForDecodeSimpleCode(&fileProcessInfo->decInfo);
		break;

	default:
		isSuffientForDecode = FALSE;
		break;
	}

	return isSuffientForDecode;
}

///////////////////////////////////////////////////////////////////////////////
void RFD_FileSetDefaultInit(RFD_FSET_MDATA_STRUCT * hFileSet )
{
	memset(hFileSet, 0, sizeof(RFD_FSET_MDATA_STRUCT));
	hFileSet->numFiles = 0;
	return;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_InitializeMaxExtraBlocksForSolution(RFD_CODE_TYPE_ENUM codeType,
												  UINT16 * maxExtraBlocksForSolution)
{
	RFD_STATUS status;

	switch(codeType) {

		case RFD_CODE_TYPE_ECC1:
		case RFD_CODE_TYPE_ECC2:

			*maxExtraBlocksForSolution = RFDR_ECC_MAX_EXTRA_BLKS_FOR_SOLUTION;
			status = RFD_STATUS_OK;
			break;

		case RFD_CODE_TYPE_SIMPLE_BLK:

			*maxExtraBlocksForSolution = RFDR_SIMPLE_CODE_MAX_EXTRA_BLKS_FOR_SOLUTION;
			status = RFD_STATUS_OK;
			break;

		default:

			*maxExtraBlocksForSolution = 0;
			status = RFD_STATUS_ERROR_INVALID_CODE_TYPE;
			break;
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_InitializeDecodeBlockThresholdInfo(RFD_CODE_TYPE_ENUM codeType,
												  RFD_XCC_DECODE_EVENT_BLOCK_THRESH_INFO * decodeBlockThresholdInfo)
{
	RFD_STATUS status;

	switch(codeType) {
		case RFD_CODE_TYPE_ECC1:
		case RFD_CODE_TYPE_ECC2:

			status = RFD_InitializeDecodeBlockThresholdInfoEccCode(decodeBlockThresholdInfo);
			break;

		case RFD_CODE_TYPE_SIMPLE_BLK:

			status = RFD_InitializeDecodeBlockThresholdInfoSimpleCode(decodeBlockThresholdInfo);
			break;

		default:

			status = RFD_STATUS_ERROR_INVALID_CODE_TYPE;
			break;
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_DecodeInfoCreateWithDefInit( RFDR_FILE_PROCESS_INFO * fileProcessInfo )
{
	RFD_STATUS status = RFD_STATUS_ERROR_GENERAL;

	switch( fileProcessInfo->fileInfo.codeType )
	{
	case RFD_CODE_TYPE_ECC1:
	case RFD_CODE_TYPE_ECC2:

		status = RFD_DecodeInfoXccCodeCreateWithDefInit( fileProcessInfo );
		break;

	case RFD_CODE_TYPE_SIMPLE_BLK:

		status = RFD_DecodeInfoSimpleCodeCreateWithDefInit( fileProcessInfo );
		break;

	default:

		status = RFD_STATUS_ERROR_INVALID_CODE_TYPE;
		break;
	}

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Write the file process state structure to storage file.
///
/// @param	pathBuf				Name of directory in which the file process state file resides.
/// @param	fileProcessState	Pointer to the file process state structure (data to be written).
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS  RFD_FileProcessStateWriteToStorage( const TCHAR * pathBuf, RFDR_FILE_RECONSTRUCT_STATE * fileProcessState )
{
	RFD_STATUS status = RFD_STATUS_OK;
	TCHAR * fileNameBuf;

	fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );

	if(fileNameBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	//////////////////////
	// Write the Process State file
	//////////////////////

	// Construct the state file name string.
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_STATE_FNAME);

	// Write the file, with compile-time configured file sync setting.
	status = RFD_WriteFile( fileNameBuf, fileProcessState,
							sizeof(RFDR_FILE_RECONSTRUCT_STATE), 
							IS_RFDR_DIR_AND_FILE_SYNC_ENABLED );

	if(status != RFD_STATUS_OK) {
		// error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	RFD_FREE(fileNameBuf);
	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Read the file process state structure from storage file.
///
/// @param	pathBuf				Name of directory in which the file process state file resides.
/// @param	fileProcessState	Pointer to the file process state structure which is updated by
/// 							this function.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS  RFD_FileProcessStateReadFromStorage( const TCHAR * pathBuf, RFDR_FILE_RECONSTRUCT_STATE * fileProcessState )
{
	RFD_STATUS status = RFD_STATUS_OK;
	INT32 ActualReadDataSize;
	TCHAR * fileNameBuf;

	fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );

	if(fileNameBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	//////////////////////
	// Load the Process State file
	//////////////////////

	// Construct the state file name string.
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_STATE_FNAME);

	// Read the file.
	status = RFD_ReadFile( fileNameBuf, fileProcessState,
							sizeof(RFDR_FILE_RECONSTRUCT_STATE), &ActualReadDataSize, TRUE);

	if(status != RFD_STATUS_OK) {
		// Error, cleanup and return with error code.
		RFD_FREE(fileNameBuf);
		return status;
	}

	RFD_FREE(fileNameBuf);
	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_FileProcessInfoReadFromStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo )
{
	RFD_STATUS status = RFD_STATUS_OK;

	//////////////////////
	// Load the Process State file
	//////////////////////

	status = RFD_FileProcessStateReadFromStorage(pathBuf, &fileProcessInfo->state);
	if(status != RFD_STATUS_OK) {
		// error, cleanup and return with error code.
		return status;
	}

	//////////////////////
	// check for valid process state
	//////////////////////
	if(  fileProcessInfo->state > RFDR_STATE_ALL_COMPLETE ) {
		return RFD_STATUS_ERROR_DECODER_STATE;
	}

	//////////////////////
	// check for Decoding state
	//////////////////////
	if(    fileProcessInfo->state < RFDR_STATE_DECODE_COMPLETE
		&& fileProcessInfo->state > RFDR_STATE_UNINIT) {

		///////////////////////////
		// Decoding State, call function to load Decode Info based on codeType.
		///////////////////////////

		switch( fileProcessInfo->fileInfo.codeType )
		{
		case RFD_CODE_TYPE_ECC1:
		case RFD_CODE_TYPE_ECC2:

			status = RFD_DecodeInfoXccCodeReadFromStorage( pathBuf, fileProcessInfo );
			break;

		case RFD_CODE_TYPE_SIMPLE_BLK:

			status = RFD_DecodeInfoSimpleCodeReadFromStorage( pathBuf, fileProcessInfo );
			break;

		default:

			status = RFD_STATUS_ERROR_INVALID_CODE_TYPE;
			break;
		}
	}
	else if( fileProcessInfo->state < RFDR_STATE_DECOMPRESS_COMPLETE ) {
		///////////////////////////
		// Decompressing State, call function to load Decompress Info .
		///////////////////////////
		// no load/initialize of decompress info is necessary.
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_FileProcessInfoWriteToStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo)
{
	TCHAR * processPathBuf;
	RFD_STATUS status = RFD_STATUS_OK;

	processPathBuf = (TCHAR *) RFD_MALLOC(sizeof(TCHAR) * RFD_MAX_PATH_LEN);

	if(processPathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	//////////////////////
	// Write the Process State file
	//////////////////////

	// Construct directory path to where the Process State File is stored.
	RFD_STRCPY_S(processPathBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(processPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(processPathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

	status = RFD_FileProcessStateWriteToStorage(processPathBuf, &fileProcessInfo->state);
	if(status != RFD_STATUS_OK) {
		// error, cleanup and return with error code.
		RFD_FREE(processPathBuf);
		return status;
	}

	//////////////////////
	// check for valid process state
	//////////////////////
	if(  fileProcessInfo->state > RFDR_STATE_ALL_COMPLETE ) {
		RFD_FREE(processPathBuf);
		return RFD_STATUS_ERROR_DECODER_STATE;
	}

	//////////////////////
	// check for Decoding state
	//////////////////////
	if(    fileProcessInfo->state < RFDR_STATE_DECODE_COMPLETE
		&& fileProcessInfo->state > RFDR_STATE_UNINIT) {

		///////////////////////////
		// Decoding State, call function to Write Decode Info based on codeType.
		///////////////////////////

		switch( fileProcessInfo->fileInfo.codeType ) {

		case RFD_CODE_TYPE_ECC1:
		case RFD_CODE_TYPE_ECC2:

			status = RFD_DecodeInfoXccCodeWriteToStorage( pathBuf, fileProcessInfo );
			break;

		case RFD_CODE_TYPE_SIMPLE_BLK:

			status = RFD_DecodeInfoSimpleCodeWriteToStorage( pathBuf, fileProcessInfo );
			break;

		default:

			status = RFD_STATUS_ERROR_INVALID_CODE_TYPE;
			break;
		}
	}
	else if( fileProcessInfo->state < RFDR_STATE_DECOMPRESS_COMPLETE ) {
		///////////////////////////
		// Decompressing State, call function to write Decompress Info .
		///////////////////////////
		// no decompress data or state info currently written for decompress state .
	}

	RFD_FREE(processPathBuf);
	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_FileProcessInfoArrayReadFromStorage( RFDR_CLIENT_INFO_STRUCT * clientInfo)
{
	int i;
	RFD_FSET_MDATA_STRUCT * pFileSetInfo;
	RFD_STATUS status = RFD_STATUS_OK; // init to success
	TCHAR * pathBuf;

	if(NULL == (pathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// The FileSetInfo must already be initialized.

	pFileSetInfo = &clientInfo->fileSetInfo;

	// range check numFiles
	if(    pFileSetInfo->numFiles < 0
		|| pFileSetInfo->numFiles > RFD_FSET_MAX_NUM_FILES)
	{
		RFD_FREE(pathBuf);
		return RFD_STATUS_ERROR_INVALID_FSET_NUM_FILES;
	}

	if( pFileSetInfo->numFiles == 0 )
	{
		// Empty FileSet, a valid value but no need to proceed.
		RFD_FREE(pathBuf);
		return RFD_STATUS_OK;
	}


	// Allocate and init fileProcessInfo structure and all fileProcessInfo child structures for each file
	for(i=0;i<pFileSetInfo->numFiles;i++)
	{
		// create base file path string used by this file instance
		RFDR_ConstructFragmentDirPathName(pathBuf, RFD_MAX_PATH_LEN, clientInfo, pFileSetInfo->fileInfo[i].id);

		if(RFD_STATUS_OK != (status = RFD_FileProcessInfoReadFromStorage( pathBuf, clientInfo->fileProcessInfo[i] )))
		{
			break;
		}
	}

	RFD_FREE(pathBuf);
	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_FileProcessInfoArrayWriteToStorage( RFDR_CLIENT_INFO_STRUCT * clientInfo)
{
	int i;
	RFD_FSET_MDATA_STRUCT * pFileSetInfo;
	RFD_STATUS status = RFD_STATUS_OK; // init to success
	TCHAR * pathBuf;

	pathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(pathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// The FileSetInfo must already be initialized.

	pFileSetInfo = &clientInfo->fileSetInfo;

	// range check numFiles
	if(    pFileSetInfo->numFiles < 0
		|| pFileSetInfo->numFiles > RFD_FSET_MAX_NUM_FILES) {
		RFD_FREE(pathBuf);
		return RFD_STATUS_ERROR_INVALID_FSET_NUM_FILES;
	}

	if( pFileSetInfo->numFiles == 0 ) {
		// Empty FileSet, a valid value but no need to proceed.
		RFD_FREE(pathBuf);
		return RFD_STATUS_OK;
	}


	// Write fileProcessInfo structures for each file
	for(i=0;i<pFileSetInfo->numFiles;i++) {

		// construct base file path string used by this file instance
		RFDR_ConstructFragmentDirPathName(pathBuf, RFD_MAX_PATH_LEN, clientInfo, pFileSetInfo->fileInfo[i].id);

		/////////
		// Mark start of storage transaction for the overall fragment i.e. the
		// top-level fragment dir and the underlying collect and process directories.
		// RFD_FileProcessInfoWriteToStorage() writes to files in both underlying directories.
		// Note: Mutex must be aquired before calling this function.
		/////////
		RFD_DoFragmentMarkStorageTransactionStart(pathBuf, FALSE, FALSE);
		{
			/////////
			// Write objects for this file of the file set to storage (this fragment).
			/////////
			status = RFD_FileProcessInfoWriteToStorage( pathBuf, clientInfo->fileProcessInfo[i]);

			if(status != RFD_STATUS_OK) {
				break;
			}

		/////////
		// Mark completion of storage transaction for the overall fragment.
		// Note: Mutex must be aquired before calling this function.
		/////////
		}
		RFD_DoFragmentMarkStorageTransactionComplete(pathBuf, FALSE);

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
		///////////
		// Backup the files just written for this file process instance
		// to the corresponding Backup Storage directory.
		// Do this after the marking transaction complete for the
		// corresponding directory file process instance.
		//
		// RFD_DoFragmentBackup() is called to backup for both the fragment's underlying
		// directories (process and collect), since RFD_FileProcessInfoWriteToStorage()
		// updates both of these directories.
		///////////

		status = RFD_DoFragmentBackup(pathBuf, FALSE /* regular backup for process dir */);
		if(status != RFD_STATUS_OK) {
			break;
		}
#endif
	}

	RFD_FREE(pathBuf);
	return status;
}
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_Decode(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo, UCHAR * exitRequestFlagPtr)
{
	RFD_STATUS status;

	switch(fileProcessInfo->fileInfo.codeType)
	{
	case RFD_CODE_TYPE_ECC1:

		status = RFD_DecodeXcc1Code(pathBuf, &gDecodeXccFileNameList,
					&fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo,
					&fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfoPrivate, //note: pass the copy, decoderStateInfoPrivate
					&RFD_ConstructSrcTypeCoeffRowEcc1,
					&RFD_ConstructChkTypeCoeffRowEcc1,
					ECC_MAX_CHK_BLOCK_INDEX,
					RFDR_ECC_MAX_EXTRA_BLKS_FOR_SOLUTION,
					exitRequestFlagPtr);
		break;

	case RFD_CODE_TYPE_ECC2:

		status = RFD_DecodeXcc2Code(pathBuf, &gDecodeXccFileNameList,
					&fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo,
					&fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfoPrivate, //note: pass the copy, decoderStateInfoPrivate
					&RFD_ConstructSrcTypeCoeffRowEcc2,
					&RFD_ConstructChkTypeCoeffRowEcc2,
					ECC_MAX_CHK_BLOCK_INDEX,
					RFDR_ECC_MAX_EXTRA_BLKS_FOR_SOLUTION,
					exitRequestFlagPtr);
		break;

	case RFD_CODE_TYPE_SIMPLE_BLK:

		// Use the generic 'Xcc1' Decoder to decode the Simple Code,
		// passing in the specific parameters needed for the Simple Code.
		status = RFD_DecodeXcc1Code(pathBuf, &gDecodeXccFileNameList,
					&fileProcessInfo->decInfo.xccDecodeInfo->codecConfigInfo,
					&fileProcessInfo->decInfo.xccDecodeInfo->decoderStateInfoPrivate, //note: pass the copy, decoderStateInfoPrivate
					&RFD_ConstructSrcTypeCoeffRowSimpleCode,
					&RFD_ConstructChkTypeCoeffRowSimpleCode,
					SIMPLE_CODE_MAX_CHK_BLOCK_INDEX,
					RFDR_SIMPLE_CODE_MAX_EXTRA_BLKS_FOR_SOLUTION,
					exitRequestFlagPtr);
		break;

	default:
		status = RFD_STATUS_ERROR_INVALID_CODE_TYPE;
		break;
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_PrepareForDecode(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo, BOOL doStorageBackupOps)
{
	RFD_STATUS status;

	switch(fileProcessInfo->fileInfo.codeType)
	{
	case RFD_CODE_TYPE_ECC1:
	case RFD_CODE_TYPE_ECC2:

		status = RFDR_PrepareForDecodeXccCode(pathBuf, fileProcessInfo, doStorageBackupOps);
		break;

	case RFD_CODE_TYPE_SIMPLE_BLK:

		status = RFDR_PrepareForDecodeSimpleCode(pathBuf, fileProcessInfo, doStorageBackupOps);
		break;

	default:
		status = RFD_STATUS_ERROR_INVALID_CODE_TYPE;
		break;
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_PostDecodeOps(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo, BOOL * pIsDecodeComplete)
{
	RFD_STATUS status = RFD_STATUS_ERROR_GENERAL;

	*pIsDecodeComplete = FALSE;

	switch(fileProcessInfo->fileInfo.codeType)
	{
	case RFD_CODE_TYPE_ECC1:
	case RFD_CODE_TYPE_ECC2:

		status = RFDR_PostDecodeOpsXccCode(pathBuf, fileProcessInfo, pIsDecodeComplete);
		// let caller handle any errors (RFDR_PostDecodeOpsXccCode() could possibly return several "OK" codes)
		break;

	case RFD_CODE_TYPE_SIMPLE_BLK:

		status = RFDR_PostDecodeOpsSimpleCode(pathBuf, fileProcessInfo, pIsDecodeComplete);
		break;

	default:
		status = RFD_STATUS_ERROR_INVALID_CODE_TYPE;
		break;
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_DecodeInfoCleanupDyn( RFDR_FILE_PROCESS_INFO * fileProcessInfo )
{
	switch( fileProcessInfo->fileInfo.codeType )
	{
	case RFD_CODE_TYPE_ECC1:
	case RFD_CODE_TYPE_ECC2:

		if(fileProcessInfo->decInfo.xccDecodeInfo != NULL) {
			RFD_FREE( fileProcessInfo->decInfo.xccDecodeInfo );
			fileProcessInfo->decInfo.xccDecodeInfo = NULL;
		}
		break;

	case RFD_CODE_TYPE_SIMPLE_BLK:

#if 0
		if(fileProcessInfo->decInfo.simpleDecodeInfo != NULL) {
			RFD_FREE( fileProcessInfo->decInfo.simpleDecodeInfo );
			fileProcessInfo->decInfo.simpleDecodeInfo = NULL;
		}
#else
		// Use the generic 'Xcc' code for the Simple Code.
		if(fileProcessInfo->decInfo.xccDecodeInfo != NULL) {
			RFD_FREE( fileProcessInfo->decInfo.xccDecodeInfo );
			fileProcessInfo->decInfo.xccDecodeInfo = NULL;
		}
#endif
		break;

	default:

		return RFD_STATUS_ERROR_GENERAL;
		break;
	}

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_GetDecoderSolutionStats(RFDR_FILE_PROCESS_INFO * fileProcessInfo, RFDR_DECODE_STATS * decodeStats)
{
	RFD_STATUS status = RFD_STATUS_ERROR_GENERAL;

	switch(fileProcessInfo->fileInfo.codeType)
	{
	case RFD_CODE_TYPE_ECC1:
	case RFD_CODE_TYPE_ECC2:

		status = RFDR_GetDecoderSolutionStatsXccCode(fileProcessInfo, decodeStats);
		break;

	case RFD_CODE_TYPE_SIMPLE_BLK:

		status = RFDR_GetDecoderSolutionStatsSimpleCode(fileProcessInfo, decodeStats);
		break;

	default:
		status = RFD_STATUS_ERROR_INVALID_CODE_TYPE;
		break;
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
void RFDR_PrintDecoderSolutionStats(RFDR_DECODE_STATS * decodeStats)
{
	RFD_DPrint( TEXT("\n"));
	RFD_DPrint( TEXT("                Decoder Solution Statistics:\n"));
	RFD_DPrint( TEXT("                ============================\n"));
	RFD_DPrint( TEXT("                Total Number of Blocks                    = %d\n"),decodeStats->nTotalBlocksInSolution);
	RFD_DPrint( TEXT("                Number of Source Blocks used in solution  = %d\n"),decodeStats->nSrcBlocksInSolution);
	RFD_DPrint( TEXT("                Number of Check Blocks used in solution   = %d\n"),decodeStats->nChkBlocksInSolution);
	RFD_DPrint( TEXT("                Number of Extra Check Blocks              = %d\n"),decodeStats->nChkBlksEliminated);
	RFD_DPrint( TEXT("                Number of Decode Chunks                  = %d\n"),decodeStats->nDecodeChunks);
}

///////////////////////////////////////////////////////////////////////////////////////////////
void RFDR_PrintAccumulatedDecoderSolutionStats(RFDR_ACCUMULATED_DECODE_STATS * accumDecodeStats)
{
	int i;

	if(accumDecodeStats->nFilesProcessed > 0) {

		RFD_DPrint( TEXT("\n"));
		RFD_DPrint( TEXT("                Accumulated Decoder Solution Statistics:\n"));
		RFD_DPrint( TEXT("                ========================================\n"));
		RFD_DPrint( TEXT("                Total Number of Files (Fragments) Processed  = %d\n"),accumDecodeStats->nFilesProcessed);
		RFD_DPrint( TEXT("                Minimum Extra Blocks                         = %d\n"),accumDecodeStats->minExtraChkBlocks);
		RFD_DPrint( TEXT("                Maximum Extra Blocks                         = %d\n"),accumDecodeStats->maxExtraChkBlocks);
		RFD_DPrint( TEXT("\n"));

		RFD_DPrint( TEXT("                Histogram of number of extra block required for decoder solution\n"));
		RFD_DPrint( TEXT("                ================================================================\n"));
		for(i=0;i<RFDR_MAX_EXTRA_BLKS_FOR_SOLUTION;i++) {
			RFD_DPrint( TEXT("                Extra Block Bin: [%2d], Count: %8d, Rate: %8.5f percent\n"),
				(INT32) i,
				(INT32) accumDecodeStats->extraChkBlocksHistogram[i],
				(float) ((float)accumDecodeStats->extraChkBlocksHistogram[i]/accumDecodeStats->nFilesProcessed*100) );
		}
		RFD_DPrint( TEXT("\n"));

		RFD_DPrint( TEXT("                Histogram of decoder failures for each extra block count\n"));
		RFD_DPrint( TEXT("                ========================================================\n"));
		RFD_DPrint( TEXT("                (i.e. for given extra block count, how many times were more extra blocks than that count required for decoder solution)\n"));
		for(i=0;i<RFDR_MAX_EXTRA_BLKS_FOR_SOLUTION;i++) {
			RFD_DPrint( TEXT("                Extra Block Bin: [%2d], Threshold Failure Count: %8d, Rate: %8.5f percent\n"),
				(INT32) i,
				(INT32) accumDecodeStats->failureThreshHistogram[i],
				(float) ((float)accumDecodeStats->failureThreshHistogram[i]/accumDecodeStats->nFilesProcessed*100) );
		}
		RFD_DPrint( TEXT("\n"));
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////
void RFDR_UpdateAccumulatedDecoderSolutionStats(RFDR_ACCUMULATED_DECODE_STATS * accumDecodeStats, RFDR_DECODE_STATS * decodeStats)
{
	int i;

	accumDecodeStats->nFilesProcessed++;
	if(decodeStats->nChkBlksEliminated > accumDecodeStats->maxExtraChkBlocks) {
		accumDecodeStats->maxExtraChkBlocks = decodeStats->nChkBlksEliminated;
	}
	if(decodeStats->nChkBlksEliminated < accumDecodeStats->minExtraChkBlocks) {
		accumDecodeStats->minExtraChkBlocks = decodeStats->nChkBlksEliminated;
	}
	if(decodeStats->nChkBlksEliminated < RFDR_MAX_EXTRA_BLKS_FOR_SOLUTION) {

		accumDecodeStats->extraChkBlocksHistogram[decodeStats->nChkBlksEliminated]++;

		for(i=0;i<decodeStats->nChkBlksEliminated;i++) {
			accumDecodeStats->failureThreshHistogram[i]++;
		}
	}
	else {
		RFD_DPrint( TEXT("Warning, unexpectedly high number of Extra Check Blocks were required for decode\n"));
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////
void RFDR_InitializeAccumulatedDecoderSolutionStats(RFDR_ACCUMULATED_DECODE_STATS * accumDecodeStats)
{
	int i;

	accumDecodeStats->nFilesProcessed = 0;
	accumDecodeStats->maxExtraChkBlocks = 0;
	accumDecodeStats->minExtraChkBlocks = 0x0ffff;
	for(i=0;i<RFDR_MAX_EXTRA_BLKS_FOR_SOLUTION;i++) {
		accumDecodeStats->extraChkBlocksHistogram[i] = 0;
		accumDecodeStats->failureThreshHistogram[i] = 0;
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_FileProcessInfoCleanupDyn(RFDR_FILE_PROCESS_INFO * fileProcessInfo)
{
	////////////
	// Cleanup all dynamic structures under fileProcessInfo.
	// Then free fileProcessInfo
	////////////

	RFD_STATUS status;

	// Cleanup Decode info under fileProcessInfo
	status = RFDR_DecodeInfoCleanupDyn( fileProcessInfo );

	// free fileProcessInfo
	RFD_FREE( fileProcessInfo );

	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_FileProcessInfoArrayCleanupDyn(RFDR_CLIENT_INFO_STRUCT * clientInfo)
{
	int i;
	RFD_FSET_MDATA_STRUCT * pFileSetInfo;

	// The FileSetInfo must already be initialized.

	pFileSetInfo = &clientInfo->fileSetInfo;

	for(i=0;i<RFD_FSET_MAX_NUM_FILES;i++)
	{
		if(clientInfo->fileProcessInfo[i] != NULL)
		{
			RFDR_FileProcessInfoCleanupDyn(clientInfo->fileProcessInfo[i]);
			clientInfo->fileProcessInfo[i] = NULL;
		}
	}

	return RFD_STATUS_OK;
}


///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_FileProcessInfoCreateWithDefInit( RFD_FSET_MDATA_STRUCT * hFileSetInfo, int FileIdx, RFDR_FILE_PROCESS_INFO ** hhFileProcessInfo)
{
	RFD_STATUS status = RFD_STATUS_ERROR_GENERAL;
	RFDR_FILE_PROCESS_INFO * fileProcessInfo;

	// Allocate the fileProcessInfo structure
	if(NULL == (fileProcessInfo = (RFDR_FILE_PROCESS_INFO *) RFD_MALLOC(sizeof(RFDR_FILE_PROCESS_INFO))))
	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}
	*hhFileProcessInfo = fileProcessInfo;

	// might as well clear the whole structure.
	RFD_MEMSET(fileProcessInfo, 0, sizeof(RFDR_FILE_PROCESS_INFO));

	// Initialize the base file info. Copy from hFileSetInfo structure.
	// Note: maintaining this info in each separate process structure is redundant, but is helps to decouple modules a little.
	fileProcessInfo->fileInfo.codeType = hFileSetInfo->codeType;
	fileProcessInfo->fileInfo.compType = hFileSetInfo->compType;
	fileProcessInfo->fileInfo.crc = hFileSetInfo->fileInfo[FileIdx].crc;
	fileProcessInfo->fileInfo.id = hFileSetInfo->fileInfo[FileIdx].id;
	fileProcessInfo->fileInfo.size = hFileSetInfo->fileInfo[FileIdx].size;
	fileProcessInfo->fileInfo.blkLenBytes = hFileSetInfo->fileInfo[FileIdx].blkLenBytes;

	// Initialize to first state.
	// Note: in an alternate implementation, this state could
	//       be made available prior to this function call.
	//       and allocations could be done based on the state.
	fileProcessInfo->state = RFDR_STATE_COLLECTING;

	// allocate and init all dynamic structures under fileProcessInfo
	// Note: for simplicity, all dynamic structures are initialized. An alternative is to only
	//       allocate structures required for the current fileProcessInfo->state
	//       e.g. if decoding is complete then there's no need to allocate decoding or collecting related structers.
	//       If this alternative is implemented, then need to be more careful with cleanup and allocations
	//       upon state changes.
	status = RFD_DecodeInfoCreateWithDefInit( fileProcessInfo );

	return status;
}


///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_FileProcessInfoArrayCreateDynWithDefInit( RFDR_CLIENT_INFO_STRUCT * clientInfo)
{
	int i;
	RFD_FSET_MDATA_STRUCT * pFileSetInfo;
	RFD_STATUS status = RFD_STATUS_OK; // init to success

	// The FileSetInfo must already be initialized.

	pFileSetInfo = &clientInfo->fileSetInfo;

	// file Process handles must be freed prior, should not have to set to NULL, but just to be safe.
	for(i=0;i<RFD_FSET_MAX_NUM_FILES;i++)
	{
		clientInfo->fileProcessInfo[i] = NULL;
	}

	// range check numFiles
	if(    pFileSetInfo->numFiles < 0
		|| pFileSetInfo->numFiles > RFD_FSET_MAX_NUM_FILES)
	{
		return RFD_STATUS_ERROR_INVALID_FSET_NUM_FILES;
	}

	if( pFileSetInfo->numFiles == 0 )
	{
		// Empty FileSet, a valid value but no need to proceed.
		return RFD_STATUS_OK;
	}


	// Allocate and init fileProcessInfo structure and all fileProcessInfo child structures for each file
	for(i=0;i<pFileSetInfo->numFiles;i++)
	{
		if(RFD_STATUS_OK != (status = RFD_FileProcessInfoCreateWithDefInit( pFileSetInfo, i, &clientInfo->fileProcessInfo[i])))
		{
			break;
		}
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_FileProcessInfoCleanupStateSpecificStorage( TCHAR * pathBuf,
														    RFDR_FILE_PROCESS_INFO * fileProcessInfo,
															RFDR_FILE_RECONSTRUCT_STATE fileProcessState)
{
	RFD_STATUS status = RFD_STATUS_OK;
	TCHAR * fileNameBuf;

	if(NULL == (fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// The Process State file is Not deleted, only state specific files.

	//////////////////////
	// check for valid process state
	//////////////////////
	if(  fileProcessState > RFDR_STATE_ALL_COMPLETE )
	{
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_DECODER_STATE;
	}

	//////////////////////
	// check for Decoding state
	//////////////////////
	if( fileProcessState <= RFDR_STATE_DECODE_COMPLETE &&
		fileProcessState > RFDR_STATE_UNINIT )
	{
		///////////////////////////
		// Decoding State, call function to Cleanup Decode Storage based on codeType.
		///////////////////////////

		switch( fileProcessInfo->fileInfo.codeType )
		{
		case RFD_CODE_TYPE_ECC1:
		case RFD_CODE_TYPE_ECC2:

			status = RFD_DecodeInfoXccCodeCleanupStorage( pathBuf, fileProcessInfo );
			break;

		case RFD_CODE_TYPE_SIMPLE_BLK:

			status = RFD_DecodeInfoSimpleCodeCleanupStorage( pathBuf, fileProcessInfo );
			break;

		default:

			status = RFD_STATUS_ERROR_INVALID_CODE_TYPE;
			break;
		}
	}
	else if( fileProcessState == RFDR_STATE_DEFRAG_COMPLETE )
	{
		///////////////////////////
		// Defrag Complete State, call function to cleanup Defrag Info for a file instance.
		///////////////////////////
		// currently no cleanup needed for defrag complete state.
	}
	else if( fileProcessState == RFDR_STATE_DECOMPRESS_COMPLETE )
	{
		///////////////////////////
		// Decompressing State, call function to cleanup Decompress Info .
		///////////////////////////
		// currently no cleanup needed for decompress state.
	}

	RFD_FREE(fileNameBuf);
	return status;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_DecompressGzip(const TCHAR decompressOutputFileName[],
							  const TCHAR decodeOutputFileName[],
							  UCHAR * exitRequestFlagPtr,
							  INT32 decodeOutputFileSize /* not needed */)
{
	RFD_STATUS status;
	UINT32 fileSize;

	// Do uncompress,
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
	// with File Sync enabled, to ensure the output uncompressed file
	// is synced to the disk before completion of transaction.
#endif
	status = rfd_gz_file_uncompress(decompressOutputFileName, decodeOutputFileName, exitRequestFlagPtr, 
									IS_RFDR_DIR_AND_FILE_SYNC_ENABLED );

	// If uncompress function exited early due to early exit request
	// delete the unfinished output file.
	if(status == RFD_STATUS_EARLY_EXIT_ON_REQUEST) {
		RFD_DELETE_FILE(decompressOutputFileName);
		return status;
	}
	else if(status != RFD_STATUS_OK) {
		// else for any error, return with error status.
		return status;
	}
	// else decompress reports successful.

	status = RFD_GetFileSize(decompressOutputFileName, &fileSize);
	if(status != RFD_STATUS_OK) {
		return status;
	}
	// Note: decompressed file size could be checked for some max decompress file size.
	// could do this check incrementally during decompress and abort if decomp goes over limit.

	// Finished decompressing, delete the compressed file.
	RFD_DELETE_FILE(decodeOutputFileName);

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_DecompressNone(const TCHAR decompressOutputFileName[],
							  const TCHAR decodeOutputFileName[],
							  INT32 decodeOutputFileSize)
{
	RFD_STATUS status;
	UINT32 fileSize;

	///////////
	// rename decoded ouput file name to decompress output file name.
	///////////

	// rename the file.
	if(RFD_RENAME( decodeOutputFileName /*old*/, decompressOutputFileName /*new*/)) {
		// error, return with error code.
		return RFD_STATUS_ERROR_FILE_RENAME;
	}

	// validate the file size.
	status = RFD_GetFileSize( decompressOutputFileName, &fileSize);
	if(status != RFD_STATUS_OK) {
		return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
	}
	if(fileSize != (UINT32)decodeOutputFileSize) {
		return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
	}

	// return.
	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_DecompressFileSet(RFDR_CLIENT_INFO_STRUCT * clientInfo, UCHAR * exitRequestFlagPtr)
{
	TCHAR * dstFileNameBuf, * srcFileNameBuf;
	RFD_STATUS status;

	// The directory in which the source and destination file resides is the first file instance (frag) directory
	// (at index = 0).
	BOOL srcDstFileInstanceIndex = 0;

	//////////
	// Allocate memory for the file names
	//////////

	dstFileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(dstFileNameBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	srcFileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(srcFileNameBuf == NULL) {
		RFD_FREE(dstFileNameBuf);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Construct file name for the source file.
	RFDR_ConstructFragmentDirPathName(srcFileNameBuf, RFD_MAX_PATH_LEN,
									  clientInfo,
									  clientInfo->fileSetInfo.fileInfo[srcDstFileInstanceIndex].id);
	RFD_STRCAT_S(srcFileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(srcFileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir
	RFD_STRCAT_S(srcFileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(srcFileNameBuf, RFD_MAX_PATH_LEN, RFDR_DEFRAGMENTED_DATA_FNAME);

	// Construct file name for the destination/decompressed file.
	RFDR_ConstructFragmentDirPathName(dstFileNameBuf, RFD_MAX_PATH_LEN,
									  clientInfo,
									  clientInfo->fileSetInfo.fileInfo[srcDstFileInstanceIndex].id);
	RFD_STRCAT_S(dstFileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(dstFileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir
	RFD_STRCAT_S(dstFileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(dstFileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECOMPRESS_OUTPUT_FNAME);

	//////////
	// Call compression-type specific functions to complete decompression.
	//////////
	switch(clientInfo->fileSetInfo.compType)
	{
	case RFD_COMP_TYPE_GZIP:

		status = RFD_DecompressGzip(dstFileNameBuf,
									srcFileNameBuf,
									exitRequestFlagPtr,
									clientInfo->fileSetInfo.totalSize);
		break;

	case RFD_COMP_TYPE_NONE:

		status = RFD_DecompressNone(dstFileNameBuf,
									srcFileNameBuf,
									clientInfo->fileSetInfo.totalSize);
		break;

	default:
		status = RFD_STATUS_ERROR_INVALID_COMP_TYPE;
		break;
	}

	//////////
	// Cleanup and return.
	//////////
	RFD_FREE(dstFileNameBuf);
	RFD_FREE(srcFileNameBuf);

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Defragment RFD file set with full power-loss resilience (when enabled).
///
/// Client Mutex must be acquired before calling this funtion. This function maintains power-loss
/// resilience while minimizing storage space by minimizing the amound of data simultaneously
/// backed-up in the main and backup directories of the file set and the individual file
/// instances (fragments).
///
/// The appoach to ensuring power-loss resilience on file opearations across the File Set
/// directory and the file instance (fragment) directories is to:
///
/// a. Mark Transaction Start on the special case *destination* file instance directory first,
/// THEN
///
/// b. Mark Transaction Start on each of the other *source* file instance directories, then
///
/// c. Do file copies/moves/appends/deletes across the files as part of defragment operation.
///
/// d. Update the state variable for the *destination* file instance directory to Defrag-
/// Complete. the states for the other source file instance directories are NOT updated. They
/// remain at Decode-Complete.
///
/// e. Mark Transaction Complete on the *destination* file instance directory, THEN
///
/// f. Mark Transaction Complete on each of the other *source* file instance directories.
///
/// With this sequence of Transaction Complete on the destination directory *FIRST* and then
/// Transaction Completions on the other source File Instance (fragment) directories
/// *LAST*.
///
/// 1. On powerup, If an incomplete transaction is detected on the destination file instance
/// directory, the original states of all directories (File Set and File Instance/Fragments) are
/// recoverable (restored) i.e. all the file instance directories must also be detected with
/// incomplete transactions. Defragmentation is then attempted again on the power up.
///
/// 2. On powerup, If the transaction was completed for the destionation file instance directory,
/// then the defragment operation was completed and the defrag result is valid in the destination
/// directory. However, it is possible that some or all of the transactions for the other source
/// file instances were not were not completed. This would result in a Restore from backup for
/// these instances, which would restore the decoded output files for those incomplete
/// transactions. Therefore, on powerup, if the overall file set state is Defrag-Complete, then
/// these source file instance directories should be *cleaned-up* by deleting the decoded output
/// files from the file instance directories.
///
/// @param  clientInfo  Information describing the client.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_DefragmentFileSet(RFDR_CLIENT_INFO_STRUCT * clientInfo)
{
	TCHAR * srcPathBuf, * srcFileNameBuf;
	TCHAR * dstPathBuf, * dstFileNameBuf;
	RFD_FILE * dstFileHandle = NULL;
	RFD_FILE * srcFileHandle = NULL;
	UCHAR * dataBuf = NULL;
	INT32 actualReadDataSize;
	INT32 dataBufSize;
	INT32 expectedFileSize, actualFileSize;
	int i;
	BOOL isFileCopyComplete;
	RFD_STATUS status;
	// File instance index 0 serves as the destination for the defragment output.
	int dstFileInstanceIndex = RFDR_FILE_INSTANCE_INDEX_0;

	srcPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(srcPathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	srcFileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(srcFileNameBuf == NULL) {
		RFD_FREE(srcPathBuf);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	dstPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(dstPathBuf == NULL) {
		RFD_FREE(srcPathBuf);
		RFD_FREE(srcFileNameBuf);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	dstFileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(dstFileNameBuf == NULL) {
		RFD_FREE(srcPathBuf);
		RFD_FREE(srcFileNameBuf);
		RFD_FREE(dstPathBuf);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Construct path name for the destination defragment output dir.
	// The process sub-dir of the file instance (fragment) dir index 0 serves as the output/destination directory.
	RFDR_ConstructFragmentDirPathName(dstPathBuf, RFD_MAX_PATH_LEN,
									  clientInfo, clientInfo->fileSetInfo.fileInfo[dstFileInstanceIndex].id);
	RFD_STRCAT_S(dstPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(dstPathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir

	// Construct filename for defrag output file.
	RFD_STRCPY_S(dstFileNameBuf, RFD_MAX_PATH_LEN, dstPathBuf);
	RFD_STRCAT_S(dstFileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(dstFileNameBuf, RFD_MAX_PATH_LEN, RFDR_DEFRAGMENTED_DATA_FNAME);

	/////////
	// Mark start of storage transaction for the destination file.
	/////////
	RFD_MarkStorageTransactionStart(dstPathBuf);
	{
		/////////
		// Mark start of storage transaction for each source File instance.
		// The destination file instance has already been marked, so skip it
		// i.e. start at index: dstFileInstanceIndex+1.
		/////////
		for(i=dstFileInstanceIndex+1;i<clientInfo->fileSetInfo.numFiles;i++) {

			// Construct the source path string for this file instance (in process dir).
			RFDR_ConstructFragmentDirPathName(srcPathBuf, RFD_MAX_PATH_LEN,
											  clientInfo,
											  clientInfo->fileSetInfo.fileInfo[i].id);
			RFD_STRCAT_S(srcPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(srcPathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir

			RFD_MarkStorageTransactionStart(srcPathBuf);
		}


		/////////
		// Defragment by copying each File instance of the File Set into the destination file.
		// Then Delete the source File after copying.
		// Then Update state of the file to Defrag Complete.
		//
		// Instead of copying, just rename the first file instance file name to the
		// destination/defrag file name.
		// Any remaining file instances will then be appended to this renamed first one.
		/////////

		// Construct the source path string for the first file instance.
		RFDR_ConstructFragmentDirPathName(srcPathBuf, RFD_MAX_PATH_LEN,
										  clientInfo,
										  clientInfo->fileSetInfo.fileInfo[dstFileInstanceIndex].id);
		RFD_STRCAT_S(srcPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(srcPathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir

		// Construct the source file name string
		RFD_STRCPY_S(srcFileNameBuf, RFD_MAX_PATH_LEN, srcPathBuf);
		RFD_STRCAT_S(srcFileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(srcFileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_OUTPUT_FNAME);

		// Delete the destination File Set file before attempting a rename (just in case).
		RFD_DELETE_FILE(dstFileNameBuf);

		// Rename (move) the file to the destination
		if(RFD_RENAME(srcFileNameBuf, dstFileNameBuf)) {
			// Error.
			// Don't Complete each started storage transaction, let further error handing clean this up.
			RFD_FREE(srcPathBuf);
			RFD_FREE(srcFileNameBuf);
			RFD_FREE(dstPathBuf);
			RFD_FREE(dstFileNameBuf);
			return RFD_STATUS_ERROR_FILE_RENAME;
		}

		/////////
		// If there is more than one file in the file set,
		// append each of the remaining files to the first file.
		/////////

		if(clientInfo->fileSetInfo.numFiles > 1) {

			dataBufSize = RFDR_MAX_DATA_COPY_BUF_LEN;

			dataBuf = (UCHAR *) RFD_MALLOC( sizeof(UCHAR) * dataBufSize );
			if(dataBuf == NULL) {
				RFD_FREE(srcPathBuf);
				RFD_FREE(srcFileNameBuf);
				RFD_FREE(dstPathBuf);
				RFD_FREE(dstFileNameBuf);
				return RFD_STATUS_ERROR_MEM_ALLOC;
			}

			// Open the destination file for append.
			if(RFD_FOPEN_S(&dstFileHandle, dstFileNameBuf, TEXT("ab"))) {
				// Error.
				// Don't Complete each started storage transaction, let further error handing clean this up.
				RFD_FREE(srcPathBuf);
				RFD_FREE(srcFileNameBuf);
				RFD_FREE(dstPathBuf);
				RFD_FREE(dstFileNameBuf);
				RFD_FREE(dataBuf);
				return RFD_STATUS_ERROR_FILE_OPEN;
			}

			// Start at the second file instance (zero relative index = 1).
			for(i=dstFileInstanceIndex+1;i<clientInfo->fileSetInfo.numFiles;i++) {

				// Construct the source path string for this file instance (in process dir).
				RFDR_ConstructFragmentDirPathName(srcPathBuf, RFD_MAX_PATH_LEN,
												  clientInfo,
												  clientInfo->fileSetInfo.fileInfo[i].id);
				RFD_STRCAT_S(srcPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
				RFD_STRCAT_S(srcPathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir

				// Construct the source file name string
				RFD_STRCPY_S(srcFileNameBuf, RFD_MAX_PATH_LEN, srcPathBuf);
				RFD_STRCAT_S(srcFileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
				RFD_STRCAT_S(srcFileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_OUTPUT_FNAME);

				// Open the source file for read.
				if(RFD_FOPEN_S(&srcFileHandle,srcFileNameBuf, TEXT("rb"))) {
					// Error.
					// Don't Complete each started storage transaction, let further error handing clean this up.
					RFD_FREE(srcPathBuf);
					RFD_FREE(srcFileNameBuf);
					RFD_FREE(dstPathBuf);
					RFD_FREE(dstFileNameBuf);
					RFD_FREE(dataBuf);
					RFD_FCLOSE(dstFileHandle);
					return RFD_STATUS_ERROR_FILE_OPEN;
				}

				actualFileSize = 0;
				expectedFileSize = clientInfo->fileSetInfo.fileInfo[i].size;
				isFileCopyComplete = FALSE;

				while(!isFileCopyComplete) {

					// read next section of the source file into buffer
					actualReadDataSize = (INT32) RFD_FREAD(dataBuf, 1, dataBufSize, srcFileHandle);

					if(actualReadDataSize > 0) {
						RFD_FWRITE(dataBuf, 1, actualReadDataSize, dstFileHandle);
						actualFileSize += actualReadDataSize;
					}

					if(actualReadDataSize < dataBufSize) {
						isFileCopyComplete = TRUE;
					}
				}

				// Close the source file.
				RFD_FCLOSE(srcFileHandle);

				// Delete the source file since it is now copied.
				RFD_DELETE_FILE(srcFileNameBuf);

				if(actualFileSize != expectedFileSize) {
					// File size error.
					// Don't Complete each started storage transaction, let further error handing clean this up.
					RFD_FREE(srcPathBuf);
					RFD_FREE(srcFileNameBuf);
					RFD_FREE(dstPathBuf);
					RFD_FREE(dstFileNameBuf);
					RFD_FREE(dataBuf);
					RFD_FCLOSE(dstFileHandle);
					return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
				}

				//////////////////////
				// The state of the other 'non-destination' file instances are not updated.
				// They remain at Decode-Complete. The state variable for the single destination file instance
				// now tracks the overall state of the file set.
				//////////////////////

			} // for(i=1;i<clientInfo->fileSetInfo.numFiles;i++)

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
			// File-Sync the dest output file to disk as part of
			// RFD storage transaction management to handle power-loss or other 
			// program abort conditions.
		
			// Must fflush first to flush upper layer buffers, before file sync
			RFD_FFLUSH(dstFileHandle);

			// Do the File Sync.
			RFD_FileSyncByHandle(dstFileHandle);
#endif
			// Close the destination file handle.
			RFD_FCLOSE(dstFileHandle);

			// Free the data buffer.
			RFD_FREE(dataBuf);
			dataBuf = NULL;

		} // if(clientInfo->fileSetInfo.numFiles > 1)

		//////////////////////
		// Update the overall file set state to Defragment Complete.
		// The state variable for the single destination file instance
		// now tracks the overall state of the file set.
		//////////////////////

		clientInfo->fileProcessInfo[dstFileInstanceIndex]->state = RFDR_STATE_DEFRAG_COMPLETE;

		// Write the Process State file
		status = RFD_FileProcessStateWriteToStorage(dstPathBuf, &clientInfo->fileProcessInfo[dstFileInstanceIndex]->state);

		if(status != RFD_STATUS_OK) {
			// Error.
			// Don't Complete each started storage transaction, let further error handing clean this up.
			RFD_FREE(srcPathBuf);
			RFD_FREE(srcFileNameBuf);
			RFD_FREE(dstPathBuf);
			RFD_FREE(dstFileNameBuf);
			return RFD_STATUS_ERROR_FILE_WRITE;
		}

	//////////////////////
	// Mark storage transaction Complete for the destination file.
	// For full power-loss resilience, it is critical that storage transaction
	// is marked Complete for this destination file (directory update),
	// BEFORE the source file (directory updates) are marked Complete.
	//
	// To minimize storage size for all the simultaneous Files in Backup,
	// postpone the Backup step until all File instances have been Backed-up.
	// Note that the Backup step in this case is actually deleting the corresponding File instances
	// that have been deleted above.
	//////////////////////
	}
	RFD_MarkStorageTransactionComplete(dstPathBuf, TRUE);

	/////////
	// Mark Completion of storage transaction for each source File instance.
	// Then Backup the corresponding directory - backup storage receives the idendical file contents
	// as the originating File instance directory
	// (effectively deletes the data File instance and copies the state file).
	// Skip marking transaction complete for the destination file instance index, as it has already been
	// marked.
	/////////
	for(i=dstFileInstanceIndex+1;i<clientInfo->fileSetInfo.numFiles;i++) {

		// Construct the source path string for this file instance (in process dir).
		RFDR_ConstructFragmentDirPathName(srcPathBuf, RFD_MAX_PATH_LEN,
										  clientInfo,
										  clientInfo->fileSetInfo.fileInfo[i].id);
		RFD_STRCAT_S(srcPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(srcPathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir

		/////////
		// Mark Completion of storage transaction for this RFD File instance.
		/////////
		RFD_MarkStorageTransactionComplete(srcPathBuf, TRUE);

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
		/////////
		// Backup directory for this RFD File instance.
		/////////
		status = RFD_BackupDirectoryContents(srcPathBuf);

		if(status != RFD_STATUS_OK) {
			// Backup failed.
			RFD_FREE(srcPathBuf);
			RFD_FREE(srcFileNameBuf);
			RFD_FREE(dstPathBuf);
			RFD_FREE(dstFileNameBuf);
			return status;
		}
#endif

	}

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
	/////////
	// Backup the destination directory for the File Set (process dir).
	// This is done after the RFD_BackupDirectoryContents() for the individual source File instances
	// so as to avoid unnecessary simultaneous storage of the defragmented file in backup directory
	// and the File fragments in the backup directory of each File (fragement) instance.
	/////////
	status = RFD_BackupDirectoryContents(dstPathBuf);

	if(status != RFD_STATUS_OK) {
		// Backup failed.
		RFD_FREE(srcPathBuf);
		RFD_FREE(srcFileNameBuf);
		RFD_FREE(dstPathBuf);
		RFD_FREE(dstFileNameBuf);
		return status;
	}
#endif

	/////////
	// Cleanup then return.
	/////////
	RFD_FREE(srcPathBuf);
	RFD_FREE(srcFileNameBuf);
	RFD_FREE(dstPathBuf);
	RFD_FREE(dstFileNameBuf);

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_ValidateFileCrc(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo)
{
	UINT32 calculatedCrc;
	RFD_STATUS status;
	TCHAR * fileNameBuf;

	if(NULL == (fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	///////////////////////
	// create file path string for the Decoder output filename
	///////////////////////
	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_OUTPUT_FNAME);

	///////////////////////
	// Calculate CRC over the decoded file
	///////////////////////
	status = RFD_CalculateFileCrc(fileNameBuf, &calculatedCrc);

	RFD_FREE(fileNameBuf);

	if(status != RFD_STATUS_OK) {
		return status;
	}

	///////////////////////
	// Compare calculated crc with the expected crc value.
	///////////////////////
	if(calculatedCrc != fileProcessInfo->fileInfo.crc) {
		return RFD_STATUS_ERROR_CRC;
	}

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_IsNewFileSetMetadata( RFD_FSET_MDATA_STRUCT * currFileSetInfo,
							   RFD_FSET_MDATA_STRUCT * newFileSetInfo)
{
	// Note: do not use RFD_MEMCMP() to determine equality of the two file set structures;
	// because for the case of !RFD_FSET_MESSAGE_INCLUDE_FILE_SIZES_ENABLE,
	// size field may not be calculated yet.
    //
    // Do Not include currFileSetInfo->lifeTime in change check for indication of new file.
    // This check was removed as of RFD SDK version 1.4.00.
    // The uplink might dynamically update the Lifetime parameter during transmission
    // of the same File Set (Update File) version.

	int i;

	if( currFileSetInfo->codeType  != newFileSetInfo->codeType ||
		currFileSetInfo->compType  != newFileSetInfo->compType ||
		currFileSetInfo->numFiles  != newFileSetInfo->numFiles ||
		currFileSetInfo->totalSize != newFileSetInfo->totalSize ) {

		return TRUE;
	}
	if(RFD_STRCMP(currFileSetInfo->name, newFileSetInfo->name)) {

		return TRUE;
	}

	for(i=0;i<currFileSetInfo->numFiles;i++) {
		if(currFileSetInfo->fileInfo[i].blkLenBytes != newFileSetInfo->fileInfo[i].blkLenBytes ||
		   currFileSetInfo->fileInfo[i].crc         != newFileSetInfo->fileInfo[i].crc ||
		   currFileSetInfo->fileInfo[i].id          != newFileSetInfo->fileInfo[i].id ) {
			// Note: do not compare fileInfo[i].size.
		    // because for the case of !RFD_FSET_MESSAGE_INCLUDE_FILE_SIZES_ENABLE,
		    // size field may not be calculated yet.

			return TRUE;
		}
	}

	// All compared fields are the same, return False to indicate Not New.
	return FALSE;
}

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Cleanup after powerup in the file set Defragment-Complete state.
///
/// This function is called to cleanup (delete) decode-output files (the inputs to defragment
/// process)
/// when a powerup occurs in the file set defragment-complete state. An abrupt power-down could
/// have occured after the defragment ouput file was completed and the state variable was updated
/// to indicate defragement-complete, but occured before some or all of the decode output files
/// (input to defragment process) were cleaned-up/deleted. If any of these decode output files
/// still exists in the file instance (fragment)
/// directories, they are deleted by this funtion. The file deletions are storage transaction
/// protected. After each deletion, a Backup of each file instance directory is performed.
///
/// The Mutex must be aquired before calling this funtion.
///
/// @param  clientInfo  Information describing the client.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CleanupOnPowerUpInDefragmentState(RFDR_CLIENT_INFO_STRUCT * clientInfo)
{
	TCHAR * pathBuf, * fileNameBuf;
	int fileToProcessIndex;
	RFDR_FILE_PROCESS_INFO * fileProcessInfo;
	int i;
	RFD_STATUS status = RFD_STATUS_OK; // in case numFiles = 0.

	// Allocate pathBuf string
	pathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(pathBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Allocate fileNameBuf string
	fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(fileNameBuf == NULL) {
		RFD_FREE(pathBuf);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	for(i=0;i<clientInfo->fileSetInfo.numFiles;i++) {

		fileToProcessIndex = i;
		fileProcessInfo = clientInfo->fileProcessInfo[i];

		// Initialize pathbuf string to the frag/process directory (where the decoder output file resides).
		RFDR_ConstructFragmentDirPathName(pathBuf, RFD_MAX_PATH_LEN, clientInfo, fileProcessInfo->fileInfo.id);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(pathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

		// Construct the full filename path.
		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECODE_OUTPUT_FNAME);

		// Delete the file if it exists
		if(RFD_IsFile(fileNameBuf)) {

			RFD_MarkStorageTransactionStart(pathBuf);
			{
				//////////////
				// Delete the Decode Output File if it exists.
				//////////////

				RFD_DPrint( TEXT("RFDR Processor: deleting file: %s, Client Index %d, fileIndex %d\n"),fileNameBuf, clientInfo->clientIndex, fileToProcessIndex);

				RFD_DELETE_FILE(fileNameBuf);
			}
			RFD_MarkStorageTransactionComplete(pathBuf, TRUE);

			//////
			// Backup Storage after above RFD_MarkStorageTransactionComplete().
			// An 'update' action is complete.
			//////
			status = RFD_BackupDirectoryContents(pathBuf);

			if(status != RFD_STATUS_OK) {
				// Backup failed.
				break;
			}
		}
	}

	// Cleanup and return.

	RFD_FREE(pathBuf);
	RFD_FREE(fileNameBuf);

	return status;

}
#endif

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Is the block collection activity complete for a file set.
///
/// @param	clientInfo	Information describing the client.
///
/// @return	true if block collection activity complete, false if not.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFDR_IsFileSetCollectionComplete(RFDR_CLIENT_INFO_STRUCT *clientInfo)
{
	int i;
	BOOL isFileSetCollectionComplete;

	// If any single file of the file set is in the decoding/collection states,
	// then set isFileSetCollectionComplete to False.
	isFileSetCollectionComplete = TRUE;
	if(clientInfo->fileSetInfo.numFiles > 0) {
		for(i=0;i<clientInfo->fileSetInfo.numFiles;i++) {
			if( clientInfo->fileProcessInfo[i]->state < RFDR_STATE_DECODE_COMPLETE &&
				clientInfo->fileProcessInfo[i]->state > RFDR_STATE_UNINIT ) {

					isFileSetCollectionComplete = FALSE;
					break;
			}
		}
	}

	return isFileSetCollectionComplete;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Is the overall file set in Decode complete state.
///
/// @param	clientInfo	Information describing the client.
///
/// @return	true if file set is in decode complete state, false if not.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFDR_IsFileSetInDecodeCompleteState(RFDR_CLIENT_INFO_STRUCT *clientInfo)
{
	BOOL isFileSetComplete;
	int i;

	if(clientInfo->fileSetInfo.numFiles == 0) {
		return FALSE;
	}

	isFileSetComplete = TRUE;
	for(i=0;i<clientInfo->fileSetInfo.numFiles;i++) {
		if(clientInfo->fileProcessInfo[i]->state != RFDR_STATE_DECODE_COMPLETE) {
			isFileSetComplete = FALSE;
			break; // break from numFiles loop
		}
	}

	return isFileSetComplete;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Is the overall file set in defrag complete state.
///
/// @param	clientInfo	Information describing the client.
///
/// @return	true if file set is in defrag complete state, false if not.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFDR_IsFileSetInDefragCompleteState(RFDR_CLIENT_INFO_STRUCT *clientInfo)
{
	// The state for file instance index 0 contains the overall file set state
	// after all file instances have achieved the Decode-Complete state.
	int fileInstanceIndex = RFDR_FILE_INSTANCE_INDEX_0;

	if(clientInfo->fileSetInfo.numFiles == 0) {
		return FALSE;
	}

	if(clientInfo->fileProcessInfo[fileInstanceIndex]->state == RFDR_STATE_DEFRAG_COMPLETE) {
		return TRUE;
	}
	else {
		return FALSE;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Is the overall file set in decompress complete state.
///
/// @param	clientInfo	Information describing the client.
///
/// @return	true if file set is in decompress complete state, false if not.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFDR_IsFileSetInDecompressCompleteState(RFDR_CLIENT_INFO_STRUCT *clientInfo)
{
	// The state for file instance index 0 contains the overall file set state
	// after all file instances have achieved the Decode-Complete state.
	int fileInstanceIndex = RFDR_FILE_INSTANCE_INDEX_0;

	if(clientInfo->fileSetInfo.numFiles == 0) {
		return FALSE;
	}

	if(clientInfo->fileProcessInfo[fileInstanceIndex]->state == RFDR_STATE_DECOMPRESS_COMPLETE) {
		return TRUE;
	}
	else {
		return FALSE;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Is the overall file set in the All-Complete state.
///
/// @param	clientInfo	Information describing the client.
///
/// @return	true if file set is in defrag complete state, false if not.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFDR_IsFileSetInAllCompleteState(RFDR_CLIENT_INFO_STRUCT *clientInfo)
{
	// The state for file instance index 0 contains the overall file set state
	// after all file instances have achieved the Decode-Complete state.
	int fileInstanceIndex = RFDR_FILE_INSTANCE_INDEX_0;

	if(clientInfo->fileSetInfo.numFiles == 0) {
		return FALSE;
	}

	if(clientInfo->fileProcessInfo[fileInstanceIndex]->state == RFDR_STATE_ALL_COMPLETE) {
		return TRUE;
	}
	else {
		return FALSE;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Get the overall (client) file set state.
///
/// @param	clientInfo	Information describing the client.
///
/// @return	the overall (client) file set state, one of the states enumerated by RFDR_CLIENT_STATE.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFDR_CLIENT_STATE RFDR_GetClientFileSetState(RFDR_CLIENT_INFO_STRUCT *clientInfo)
{
    // The state for file instance index 0 contains the overall file set state
    // after all file instances have achieved the Decode-Complete state.
    int fileInstanceIndex = RFDR_FILE_INSTANCE_INDEX_0;

    RFDR_CLIENT_STATE clientFileSetState;

    if(clientInfo->fileSetInfo.numFiles == 0) {
        clientFileSetState = RFDR_CLIENT_STATE_RESET;
    }
    else if(clientInfo->fileProcessInfo[fileInstanceIndex]->state == RFDR_STATE_ALL_COMPLETE) {
        clientFileSetState = RFDR_CLIENT_STATE_COMPLETE;
    }
    else {
        clientFileSetState = RFDR_CLIENT_STATE_IN_PROGRESS;
    }

    return clientFileSetState;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_ProcessorCloseLL(
    RFD_PROCESSOR_THREAD_DATA_HANDLE hRfdr,
    RFD_STATUS * statusPtr
        )
{
    RFDR_PROCESSOR_LL_CONTROL_STRUCT * ctl = NULL;
	RFDR_CLIENT_INFO_STRUCT * clientInfo = NULL;
	int nClients = hRfdr->rfdrClientSet->nClients;
    int i;

	// Cleanup Processor related mem in each client.
	for(i=0;i<nClients;i++)
	{
	   clientInfo = &hRfdr->rfdrClientSet->clientInfo[i];

	   // Acquire Mutex to disable other threads from accessing client info as
	   // it is cleaned up.
	   RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);
	   {
	       // Cleanup FileProcessInfo array and all FileProcessInfo children
	       RFDR_FileProcessInfoArrayCleanupDyn( clientInfo );
	       // Reset FileSetInfo to effectively disable the client (clientInfo->fileSetInfo.numFiles = 0)
	       // If not, the Collector thread may try to access a cleaned-up clientInfo->fileProcessInfo
	       // after the mutex is released, but before the Collector thread itself exits upon signal from manager.
	       RFD_FileSetDefaultInit( &clientInfo->fileSetInfo);
	   }
	   RFD_ReleaseMutex(clientInfo->clientMutex);
	}

    // cleanup allocations managed under hRfdr->llControl (Low Level Control object)
    // and the finally cleanup hRfdr->llControl.
    if(hRfdr->llControl != NULL) {

        ctl = hRfdr->llControl;

        if(ctl->isClientPollForActionComplete != NULL) {
	        RFD_FREE(ctl->isClientPollForActionComplete);
            ctl->isClientPollForActionComplete = NULL;
        }

        if(ctl->clientPathBuf != NULL) {
	        RFD_FREE(ctl->clientPathBuf);
            ctl->clientPathBuf = NULL;
        }
        if(ctl->fragmentPathBuf != NULL) {
	        RFD_FREE(ctl->fragmentPathBuf);
            ctl->fragmentPathBuf = NULL;
        }
        if(ctl->fileNameBuf != NULL) {
	        RFD_FREE(ctl->fileNameBuf);
            ctl->fileNameBuf = NULL;
        }
        if(ctl->multiCollectWaitEventObject != NULL) {
	        RFD_DeleteMultiWaitEventObject(ctl->multiCollectWaitEventObject);
            ctl->multiCollectWaitEventObject = NULL;
        }

	    RFD_FREE(hRfdr->llControl);
        hRfdr->llControl = NULL;
        ctl = NULL;
    }

    RFD_DPrint( TEXT("RFDR Processor: Ending\n"));

    *statusPtr = RFD_STATUS_OK;

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_ProcessorOpenLL(
	RFD_PROCESSOR_THREAD_DATA_HANDLE hRfdr,
    RFD_STATUS * statusPtr,
    BOOL *isThreadExitRequestedPtr
        )
{
	int nClients;
	int i, j;
	BOOL isValid;
	INT32 actualReadDataSize;
	RFD_FILE_INFO_STRUCT * fileInfo;
	RFD_STATUS status;
	BOOL isThreadExitRequested;
	RFD_EVENT_HANDLE initCompleteEvent;
	RFD_EVENT_HANDLE errorExitEvent;
    //////////////////////////////////////////////
	TCHAR * clientPathBuf;
    TCHAR * fragmentPathBuf;
    TCHAR * fileNameBuf;
	int clientIndex;
	RFDR_CLIENT_INFO_STRUCT * clientInfo;
	RFDR_COLLECTOR_INFO_STRUCT * collectorInfo;
	RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo;
	RFD_FSET_MDATA_STRUCT * fileSetInfo;
	RFDR_FILE_PROCESS_INFO * fileProcessInfo;
	int fileToProcessIndex;
    //////////////////////////////////////////////
    RFDR_PROCESSOR_LL_CONTROL_STRUCT * ctl = NULL;
    status = RFD_STATUS_OK;
    isValid = TRUE;
    //////////////////////////////////////////////

	errorExitEvent = hRfdr->rfdrProcessorInfo->errorExitEvent;
	initCompleteEvent  = hRfdr->rfdrProcessorInfo->initCompleteEvent;
	nClients = hRfdr->rfdrClientSet->nClients;
    isThreadExitRequested = FALSE;

    do { // do once

	    // Create Low Level Control object that maintains data needed in outer scope of Run loop.
        hRfdr->llControl = (RFDR_PROCESSOR_LL_CONTROL_STRUCT *) RFD_CALLOC( sizeof(RFDR_PROCESSOR_LL_CONTROL_STRUCT));
	    if(hRfdr->llControl == NULL) {
		    status = RFD_STATUS_ERROR_MEM_ALLOC;
            isThreadExitRequested = TRUE;
            break;
	    }
        ctl = hRfdr->llControl;

	    ctl->clientPathBuf = NULL;
        ctl->fragmentPathBuf = NULL;
        ctl->fileNameBuf = NULL;
	    ctl->isClientPollForActionComplete = NULL;
	    ctl->multiCollectWaitEventObject = NULL;
	    ctl->clientIndex = 0;
	    ctl->clientInfo = NULL;
	    ctl->collectorInfo = NULL;
	    ctl->consumerInfo = NULL;
	    ctl->fileSetInfo = NULL;
	    ctl->fileProcessInfo = NULL;
	    ctl->fileToProcessIndex = 0;
	    ctl->forceNextActionCompletion = FALSE;
	    ctl->actionRequest = RFD_PROC_ACTION__POLL_FOR_ACTION;
#ifdef ENABLE_ACCUMULATED_DECODER_TEST_STATS
	    RFDR_InitializeAccumulatedDecoderSolutionStats(&ctl->accumDecodeStats);
#endif

	    // Create MultWaitEventObject
	    ctl->multiCollectWaitEventObject = RFD_CreateMultiWaitEventObject( nClients ) ;
	    if(ctl->multiCollectWaitEventObject == NULL) {
		    status = RFD_STATUS_ERROR_MEM_ALLOC;
            isThreadExitRequested = TRUE;
            break;
	    }

	    // Add each ClientEvent to the multiEventObject
	    for(i=0;i<nClients;i++) {
		    status = RFD_AddEventToMultiWaitEventObject(
							    ctl->multiCollectWaitEventObject,
							    hRfdr->rfdrClientSet->clientInfo[i].clientEvent,
							    i);
		    if(status != RFD_STATUS_OK) {
                isThreadExitRequested = TRUE;
                break;
		    }
	    }
        if(isThreadExitRequested) {
            break;
        }

	    // Allocate clientPathBuf string
	    ctl->clientPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	    if(ctl->clientPathBuf == NULL) {
		    status = RFD_STATUS_ERROR_MEM_ALLOC;
            isThreadExitRequested = TRUE;
            break;
	    }

	    // Allocate fragmentPathBuf string
	    ctl->fragmentPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	    if(ctl->fragmentPathBuf == NULL) {
		    status = RFD_STATUS_ERROR_MEM_ALLOC;
            isThreadExitRequested = TRUE;
            break;
	    }

	    // Allocate fileNameBuf string
	    ctl->fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	    if(ctl->fileNameBuf == NULL) {
		    status = RFD_STATUS_ERROR_MEM_ALLOC;
            isThreadExitRequested = TRUE;
            break;
	    }

	    // Allocate isClientPollForActionComplete array
	    ctl->isClientPollForActionComplete = (BOOL *) RFD_MALLOC( sizeof(BOOL) * nClients );
	    if(ctl->isClientPollForActionComplete == NULL) {
		    status = RFD_STATUS_ERROR_MEM_ALLOC;
            isThreadExitRequested = TRUE;
            break;
	    }

	    ////////////////////////////////////
	    // RFDR Initialization
	    ////////////////////////////////////

	    clientPathBuf                  = ctl->clientPathBuf;
        fragmentPathBuf                = ctl->fragmentPathBuf;
        fileNameBuf                    = ctl->fileNameBuf;
	    //isClientPollForActionComplete  = ctl->isClientPollForActionComplete;
	    //multiCollectWaitEventObject    = ctl->multiCollectWaitEventObject;
	    clientIndex                    = ctl->clientIndex;
	    clientInfo                     = ctl->clientInfo;
	    collectorInfo                  = ctl->collectorInfo;
	    consumerInfo                   = ctl->consumerInfo;
	    fileSetInfo                    = ctl->fileSetInfo;
	    fileProcessInfo                = ctl->fileProcessInfo;
	    fileToProcessIndex             = ctl->fileToProcessIndex;
	    //forceNextActionCompletion      = ctl->forceNextActionCompletion;
	    //actionRequest                  = ctl->actionRequest;

	    ////////////////////////////////////
	    // Create RFDR Base directory if it doesn't exist
	    ////////////////////////////////////

	    if(!RFD_IsDirectory(RFDR_BASE_PATH)) {

		    RFD_DPrint( TEXT("Base directory Not found, Creating: %s \n"),RFDR_BASE_PATH);

		    if(RFD_CREATE_DIR(RFDR_BASE_PATH)) {
			    status = RFD_STATUS_ERROR_DIRECTORY_CREATE;
			    RFD_DPrint( TEXT("CreateDirectory failed, status (%d)\n"), status);
	            isThreadExitRequested = TRUE;
                break;
		    }
	    }

	    ////////////////////////////////////
	    // For each Client:
	    // - Validate storage contents
	    // - If storage contents not valid,
	    //   then default initialize the storage contents
	    // - Allocate client dynamic structures,
	    //   and load structures from storage.
	    ////////////////////////////////////
	    for(i=0;i<nClients;i++) {

		    isValid = TRUE;
		    clientInfo = &hRfdr->rfdrClientSet->clientInfo[i];
		    fileSetInfo = &clientInfo->fileSetInfo;
		    clientIndex = i;

		    RFDR_ConstructClientDirPathName(clientPathBuf, RFD_MAX_PATH_LEN, clientInfo);

		    ////////
		    // Acquire Mutex
		    ////////
		    status = RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);
		    if(status != RFD_STATUS_OK) {
			    // Error, signal error exit and break out of client init loop
			    isThreadExitRequested = TRUE;
			    break;
		    }

		    if(!RFD_IsDirectory(clientPathBuf)) {
			    // Client directory not found. This may be the first time running.

			    // Create the Client directory.

			    RFD_DPrint( TEXT("directory Not found, Creating: %s \n"),clientPathBuf);

			    if(RFD_CREATE_DIR(clientPathBuf)) {
				    status = RFD_STATUS_ERROR_DIRECTORY_CREATE;
				    RFD_DPrint( TEXT("CreateDirectory failed, status (%d)\n"), status);
				    // Error, signal error exit and break out of client init loop
				    RFD_ReleaseMutex(clientInfo->clientMutex);
				    isThreadExitRequested = TRUE;
				    break;
			    }

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
			    // Create the Backup Storage directory for
			    // this corresponding Client Directory.

			    status = RFD_CreateBackupStorage(clientPathBuf);
			    if(status != RFD_STATUS_OK) {
				    RFD_DPrint( TEXT("RFD_CreateBackupStorage failed, status (%d)\n"), status);
				    // Error, signal error exit and break out of client init loop
				    RFD_ReleaseMutex(clientInfo->clientMutex);
				    isThreadExitRequested = TRUE;
				    break;
			    }
 #endif
			    // set isValid false to avoid further validity checks and force
			    // initialization.
			    isValid = FALSE;
		    }
#if !defined(RFDR_DISABLE_CLEANUP_FS_CORRUPTED_DIR_ENTRIES)
			else {

				UINT32 nCorruptedDirectoryEntriesFound = 0;

				/////////////
				// Try to cleanup anomalous corrupted files that might occur on some platforms.
				// These corrupted files may show as directory entries, but opening the file or
				// trying to get attributes of the file may fail. Trying to delete these files
				// through regular file system calls may also fail. Typical file validation checking
				// and any necessary cleanup in presence of these types of files may therefore
				// not successfully restore state or cleanup to initial state as expected.
				// 
				// So this specialized function is used to try to cleanup corrupted files of
				// this nature. It is called first, before any other validation, recovery or cleanup
				// steps that are part of the normal flow of application initialization.
				/////////////

				status = RFD_DeleteFileSysCorruptedDirectoryEntries(
							clientPathBuf, 
							&nCorruptedDirectoryEntriesFound, 
							RFDR_TRANSACTION_STATE_FNAME, /* specify the transaction state file as the ancillary file to delete also */
							TRUE, /* do sync after transaction state file is removed */
							TRUE, /* do final dir sync if any invalid entries found and removed. */
							TRUE  /* recursively check all subdirs */ );
				if(status != RFD_STATUS_OK) {

					// The function reported failure for some reason, 
					// but don't declare the RFD client storage as invalid (avoid false negatives).
					// Keep going and perform the regular storage validation.

					RFD_DPrint( TEXT("RFDR Processor: RFD_DeleteFileSysCorruptedDirectoryEntries() failed, status = %d, path = %s, Client Index %d\n"), 
						status, clientPathBuf, clientIndex);
				}

				if(nCorruptedDirectoryEntriesFound != 0) {
					// Found invalid/corrupted directory entry or entries and attempted their deletion. 
					// Any corresponding transaction state file(s) would have also been deleted by the function.
					// Therefore, continue with normal application validation steps, which will detect the incomplete
					// transaction state(s) and perform restore operations.
					RFD_DPrint( TEXT("RFDR Processor: RFD_DeleteFileSysCorruptedDirectoryEntries() found corrupted file(s)"
									 " and attempted cleanup, count = %d, path = %s, Client Index %d\n"), 
						nCorruptedDirectoryEntriesFound, clientPathBuf, clientIndex);
				}
			}
#endif

		    /////////////
		    // Check the client directory Storage Transaction file
		    /////////////
		    if(isValid) {
			    if(!RFD_IsStorageTransactionComplete(clientPathBuf)) {
				    // A transaction was incomplete, content of
				    // this directory is possibly corrupted.

				    RFD_DPrint( TEXT("RFDR Processor: Incomplete Storage Transaction Detected, path = %s, Client Index %d\n"),
							    clientPathBuf, clientIndex);

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
				    // Attempt to restore client directory from
				    // the backup location.

				    RFD_DPrint( TEXT("RFDR Processor: Attempting to restore directory files from backup storage\n"));

				    status = RFD_RestoreDirectoryContents(clientPathBuf);
				    if(status != RFD_STATUS_OK) {
					    // Restore failed.
					    RFD_DPrint( TEXT("RFDR Processor: Directory restore failed, status = %d.\n"), status);
					    isValid = FALSE;
				    }
				    else {
					    RFD_DPrint( TEXT("RFDR Processor: Directory sucessfully restored.\n"));
				    }
#else
				    RFD_DPrint( TEXT("RFDR Processor: Marking directory as Invalid.\n"));
				    isValid = FALSE;
#endif
			    }
		    }

		    /////////////
		    // Check and read the FileSetInfo File
		    /////////////
		    if(isValid) {

			    // create file path string for the File Set filename
			    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, clientPathBuf);
			    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_FSET_FNAME);

			    // Read the file.
			    status = RFD_ReadFile( fileNameBuf, fileSetInfo, sizeof(RFD_FSET_MDATA_STRUCT),
								       &actualReadDataSize, TRUE);
			    if(status != RFD_STATUS_OK) {
				    // error reading file, file may not exist or size error, etc.
				    isValid = FALSE;
			    }
			    #if 0 // note, possible error check could be done here: RFD_RangeCheckFileSetInfo()
			    else if(!RFD_RangeCheckFileSetInfo(fileSetInfo)){
				    // range check of fileSetInfo structure failed
				    isValid = FALSE;
			    }
			    #endif
		    }

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
		    // Backup this directory to backup storage if:
		    //
		    // 1. The Backup Storage is invalid E.g. a power-loss could have
		    // occurred during file update to the backup location in which case
		    // transaction-in-progress status should be indicated.
		    //
		    // 2. The Backup Storage is indicated as out-of-sync with
		    // this directory. E.g. a power-loss could have
		    // right after a completed transaction to this directory, but before
		    // the start of the corresponding transaction to the Backup Storage.
		    // transaction-in-progress status should be indicated.
		    // If not valid, copy the current directory to the backup location.
		    if(isValid) {

			    if(!RFD_IsBackupStorageValid(clientPathBuf) ||
			       !RFD_IsBackupStorageInSync(clientPathBuf) ) {

				    RFD_DPrint( TEXT("RFDR Processor: Backup Storage detected as Invalid or Out-of-Sync, for path = %s, Client Index %d\n"),
							    clientPathBuf, clientIndex);
				    RFD_DPrint( TEXT("RFDR Processor: RFD_IsBackupStorageValid() = %d, RFD_IsBackupStorageInSync() = %d\n"),
							    RFD_IsBackupStorageValid(clientPathBuf), RFD_IsBackupStorageInSync(clientPathBuf));
				    RFD_DPrint( TEXT("RFDR Processor: Copying directory contents to backup storage\n"));

				    status = RFD_BackupDirectoryContents(clientPathBuf);
				    if(status != RFD_STATUS_OK) {
					    // Backup failed. Set valid status false
					    // to force overall initialization.
					    isValid = FALSE;
				    }
			    }
		    }
#endif
		    /////////////
		    // Check individual file directories of the file set.
		    /////////////
		    if(isValid) {

			    for(j=0;j<fileSetInfo->numFiles;j++) {

				    fileInfo = &fileSetInfo->fileInfo[j];
				    fileToProcessIndex = j;

				    /////////////
				    // Check that each file directory exists
				    /////////////

				    // Construct file path for jth file (id) directory
				    RFDR_ConstructFragmentDirPathName(fragmentPathBuf, RFD_MAX_PATH_LEN, clientInfo, fileInfo->id);

				    if(!RFD_IsDirectory(fragmentPathBuf)) {
					    isValid = FALSE;
					    break;
				    }
				    else {

					    // Check that the Collection directory exists

					    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);

					    if(!RFD_IsDirectory(fileNameBuf)) {
						    isValid = FALSE;
						    break;
					    }

					    // Check that the Process directory exists

					    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

					    if(!RFD_IsDirectory(fileNameBuf)) {
						    isValid = FALSE;
						    break;
					    }
				    }


				    /////////////
				    // Check the Storage Transaction file in each file instance (fragment) directory.
				    // This transaction marker is used to synchronize (make atomic) file updates
				    // involving both of the underlying directories i.e. the collection dir and the process dir.
				    /////////////

				    if(!RFD_IsStorageTransactionComplete(fragmentPathBuf)) {

					    // A fragment-level transaction was incomplete, content of
					    // underlying collection and process directories are possibly corrupted.

					    RFD_DPrint( TEXT("RFDR Processor: Incomplete Storage Transaction Detected, path = %s, Client Index %d, File Index %d\n"),
								    fragmentPathBuf, clientIndex, fileToProcessIndex);

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
					    /////////////
					    // Attempt to restore the underlying collection and process directories from
					    // the backup locations.
					    // Call RFD_DoFragmentRestore() to restore the two underlying directories (process and collect).
					    /////////////

					    status = RFD_DoFragmentRestore(fragmentPathBuf);

					    if(status != RFD_STATUS_OK) {
						    // Restore failed.
						    RFD_DPrint( TEXT("RFDR Processor: Fragment (file instance) restore failed, status = %d.\n"), status);
						    isValid = FALSE;
						    break;
					    }
					    else {
						    RFD_DPrint( TEXT("RFDR Processor: Fragment (file instance) sucessfully restored.\n"));
					    }

					    /////////
					    // Initialize the fragment-level storage transaction marker to indicate complete.
					    /////////
					    RFD_InitializeStorageTransaction(fragmentPathBuf, 0);
#else
					    RFD_DPrint( TEXT("RFDR Processor: Marking directory as Invalid.\n"));
					    isValid = FALSE;
					    break;
#endif

				    }
				    else {
					    /////////////
					    // The fragment-level transaction is indicated as complete.
					    // Now check transaction complete status for the two individual underlying directories.
					    /////////////

					    /////////////
					    // Collector directory
					    /////////////

					    // Construct file path for the Collector directory
					    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);

					    if(!RFD_IsStorageTransactionComplete(fileNameBuf)) {

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
						    RFD_DPrint( TEXT("RFDR Processor: Incomplete Storage Transaction Detected, path = %s, Client Index %d, File Index %d\n"),
									    fileNameBuf, clientIndex, fileToProcessIndex);

						    status = RFD_RestoreDirectoryContents(fileNameBuf);
						    if(status != RFD_STATUS_OK) {
							    // Restore failed.
							    RFD_DPrint( TEXT("RFDR Processor: Collect Directory restore failed, status = %d.\n"), status);
							    isValid = FALSE;
							    break;
						    }
						    else {
							    RFD_DPrint( TEXT("RFDR Processor: Collect Directory sucessfully restored.\n"));
						    }
#else
						    RFD_DPrint( TEXT("RFDR Processor: Marking directory as Invalid.\n"));
						    isValid = FALSE;
						    break;
#endif
					    }

					    /////////////
					    // Process directory
					    /////////////

					    // Construct file path for the Collector directory
					    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

					    if(!RFD_IsStorageTransactionComplete(fileNameBuf)) {

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
						    RFD_DPrint( TEXT("RFDR Processor: Incomplete Storage Transaction Detected, path = %s, Client Index %d, File Index %d\n"),
									    fileNameBuf, clientIndex, fileToProcessIndex);

						    status = RFD_RestoreDirectoryContents(fileNameBuf);
						    if(status != RFD_STATUS_OK) {
							    // Restore failed.
							    RFD_DPrint( TEXT("RFDR Processor: Process Directory restore failed, status = %d.\n"), status);
							    isValid = FALSE;
							    break;
						    }
						    else {
							    RFD_DPrint( TEXT("RFDR Processor: Process Directory sucessfully restored.\n"));
						    }
#else
						    RFD_DPrint( TEXT("RFDR Processor: Marking directory as Invalid.\n"));
						    isValid = FALSE;
						    break;
#endif
					    }
				    }

				    /////////////
				    // Note: Additional tests can be done here on individual files
				    // of each file directory. The files present are dependent
				    // on the state of the file and the code and compression types for the file set.
				    // The dynamic structures under clientInfo may need to be allocated first
				    // if they will be loaded as part of these checks.
				    // -- Some of this type of checking is done below in RFD_FileProcessInfoArrayReadFromStorage()
				    /////////////

#if !defined(RFDR_ENABLE_STORAGE_OPTIM_1) && defined(RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE)

				    // Backup directories to backup storage if:
				    //
				    // 1. The Backup Storage is invalid E.g. a power-loss could have
				    // occurred during file update to the backup location in which case
				    // transaction-in-progress status should be indicated.
				    //
				    // 2. The Backup Storage is indicated as out-of-sync with
				    // this directory. E.g. a power-loss could have
				    // right after a completed transaction to this directory, but before
				    // the start of the corresponding transaction to the Backup Storage.
				    // transaction-in-progress status should be indicated.
				    // If not valid, copy the current directory to the backup location.

				    /////////////
				    // Collector directory
				    /////////////

				    // Construct file path for the Collector directory
				    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
				    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
				    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);

				    if(!RFD_IsBackupStorageValid(fileNameBuf) ||
				       !RFD_IsBackupStorageInSync(fileNameBuf) ) {

					    RFD_DPrint( TEXT("RFDR Processor: Backup Storage detected as Invalid or Out-of-Sync, for path = %s, Client Index %d, File Index %d\n"),
								    fileNameBuf, clientIndex, fileToProcessIndex);
					    RFD_DPrint( TEXT("RFDR Processor: RFD_IsBackupStorageValid() = %d, RFD_IsBackupStorageInSync() = %d\n"),
								    RFD_IsBackupStorageValid(fileNameBuf), RFD_IsBackupStorageInSync(fileNameBuf));
					    RFD_DPrint( TEXT("RFDR Processor: Copying directory contents to backup storage\n"));

					    status = RFD_BackupDirectoryContents(fileNameBuf);
					    if(status != RFD_STATUS_OK) {
						    // Backup failed. Set valid status false
						    // to force overall initialization.
						    isValid = FALSE;
						    break;
					    }
				    }

				    /////////////
				    // Process directory
				    /////////////

				    // Construct file path for the Collector directory
				    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
				    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
				    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

				    if(!RFD_IsBackupStorageValid(fileNameBuf) ||
				       !RFD_IsBackupStorageInSync(fileNameBuf) ) {

					    RFD_DPrint( TEXT("RFDR Processor: Backup Storage detected as Invalid or Out-of-Sync, for path = %s, Client Index %d, File Index %d\n"),
								    fileNameBuf, clientIndex, fileToProcessIndex);
					    RFD_DPrint( TEXT("RFDR Processor: RFD_IsBackupStorageValid() = %d, RFD_IsBackupStorageInSync() = %d\n"),
								    RFD_IsBackupStorageValid(fileNameBuf), RFD_IsBackupStorageInSync(fileNameBuf));
					    RFD_DPrint( TEXT("RFDR Processor: Copying directory contents to backup storage\n"));

					    status = RFD_BackupDirectoryContents(fileNameBuf);
					    if(status != RFD_STATUS_OK) {
						    // Backup failed. Set valid status false
						    // to force overall initialization.
						    isValid = FALSE;
						    break;
					    }
				    }
#else
    // Postpone this operation until after the file processing state has been loaded below.
#endif
			    } // for(j=0;j<fileSetInfo->numFiles;j++)
		    }


		    if(isValid) {
			    //////////////////
			    // Initialize File Process Info structures and files.
			    //////////////////
			    status = RFD_FileProcessInfoArrayCreateDynWithDefInit( clientInfo );

			    if(status != RFD_STATUS_OK) {
				    RFD_DPrint( TEXT("Error in RFD_FileProcessInfoArrayCreateDynWithDefInit: status = %d \n"),status);
				    // Cleanup FileProcessInfo array and all FileProcessInfo children
				    RFDR_FileProcessInfoArrayCleanupDyn( clientInfo );

				    isValid = FALSE;
			    }
		    }

		    if(isValid) {
			    //////////////////
			    // Load File Process Info structures and files from File System.
			    //////////////////
			    status = RFD_FileProcessInfoArrayReadFromStorage( clientInfo );

			    if(status != RFD_STATUS_OK) {
				    RFD_DPrint( TEXT("Error in RFD_FileProcessInfoArrayReadFromStorage: status = %d \n"),status);
				    // Cleanup FileProcessInfo array and all FileProcessInfo children
				    RFDR_FileProcessInfoArrayCleanupDyn( clientInfo );

				    isValid = FALSE;
			    }
		    }


#if defined(RFDR_ENABLE_STORAGE_OPTIM_1) && defined(RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE)
		    //////////////////
		    // Moved the backup incomplete check until after initialization of the processing state variebles
		    // by RFDR_FileProcessInfoArrayCleanupDyn() and RFD_FileProcessInfoArrayReadFromStorage() above.
		    // The backup is done conditionally for the process directory, depending on the processing state.
		    //////////////////

		    if(isValid) {

			    for(j=0;j<fileSetInfo->numFiles;j++) {

				    fileInfo = &fileSetInfo->fileInfo[j];
				    fileToProcessIndex = j;

				    /////////////
				    // Check that each file directory exists
				    /////////////

				    // Construct file path for jth file (id) directory
				    RFDR_ConstructFragmentDirPathName(fragmentPathBuf, RFD_MAX_PATH_LEN, clientInfo, fileInfo->id);

				    // Backup directories to backup storage if:
				    //
				    // 1. The Backup Storage is invalid E.g. a power-loss could have
				    // occurred during file update to the backup location in which case
				    // transaction-in-progress status should be indicated.
				    //
				    // 2. The Backup Storage is indicated as out-of-sync with
				    // this directory. E.g. a power-loss could have
				    // right after a completed transaction to this directory, but before
				    // the start of the corresponding transaction to the Backup Storage.
				    // transaction-in-progress status should be indicated.
				    // If not valid, copy the current directory to the backup location.

				    /////////////
				    // Collector directory
				    /////////////

				    // Construct file path for the Collector directory
				    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
				    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
				    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_COLLECT_DIR_NAME);

				    if(!RFD_IsBackupStorageValid(fileNameBuf) ||
				       !RFD_IsBackupStorageInSync(fileNameBuf) ) {

					    RFD_DPrint( TEXT("RFDR Processor: Backup Storage detected as Invalid or Out-of-Sync, for path = %s, Client Index %d, File Index %d\n"),
								    fileNameBuf, clientIndex, fileToProcessIndex);
					    RFD_DPrint( TEXT("RFDR Processor: RFD_IsBackupStorageValid() = %d, RFD_IsBackupStorageInSync() = %d\n"),
								    RFD_IsBackupStorageValid(fileNameBuf), RFD_IsBackupStorageInSync(fileNameBuf));
					    RFD_DPrint( TEXT("RFDR Processor: Copying directory contents to backup storage\n"));

					    status = RFD_BackupDirectoryContents(fileNameBuf);
					    if(status != RFD_STATUS_OK) {
						    // Backup failed. Set valid status false
						    // to force overall initialization.
						    isValid = FALSE;
						    // Cleanup FileProcessInfo array and all FileProcessInfo children
						    // (because RFD_FileProcessInfoArrayCreateDynWithDefInit() has already been called).
						    RFDR_FileProcessInfoArrayCleanupDyn( clientInfo );
						    break;
					    }
				    }

				    /////////////
				    // Process directory
				    //
				    // Special case condition for storage optimization for full power-loss resilience build :
				    // Backup is not done for process directory if file (fragment) processing state
				    // is RFDR_STATE_COLLECTING.
				    // In this case, the backup has been postponed as part of the normal operation
				    // to minimize storage space utilization; the backup of the process directory will occur
				    // at the begining of the next 'prepare-for-decode' stage.
				    ////////

				    if(clientInfo->fileProcessInfo[j]->state != RFDR_STATE_COLLECTING) {

					    // Construct file path for the Collector directory
					    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

					    if(!RFD_IsBackupStorageValid(fileNameBuf) ||
					       !RFD_IsBackupStorageInSync(fileNameBuf) ) {

						    RFD_DPrint( TEXT("RFDR Processor: Backup Storage detected as Invalid or Out-of-Sync, for path = %s, Client Index %d, File Index %d\n"),
									    fileNameBuf, clientIndex, fileToProcessIndex);
						    RFD_DPrint( TEXT("RFDR Processor: RFD_IsBackupStorageValid() = %d, RFD_IsBackupStorageInSync() = %d\n"),
									    RFD_IsBackupStorageValid(fileNameBuf), RFD_IsBackupStorageInSync(fileNameBuf));
						    RFD_DPrint( TEXT("RFDR Processor: Copying directory contents to backup storage\n"));

						    status = RFD_BackupDirectoryContents(fileNameBuf);
						    if(status != RFD_STATUS_OK) {
							    // Backup failed. Set valid status false
							    // to force overall initialization.
							    isValid = FALSE;
							    // Cleanup FileProcessInfo array and all FileProcessInfo children
							    // (because RFD_FileProcessInfoArrayCreateDynWithDefInit() has already been called).
							    RFDR_FileProcessInfoArrayCleanupDyn( clientInfo );
							    break;
						    }
					    }
				    }
			    } // for(j=0;j<fileSetInfo->numFiles;j++)
		    }
#endif

		    /////////////
		    // Default Initialize the Client storage if checks determined storage is not valid.
		    /////////////
		    if(!isValid) {

			    RFD_DPrint( TEXT("Client validity check failed. Perform Default Initialize of client storage\n"));

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
			    /////////
			    // Mark start of storage transaction for the client *Backup* direcory.
			    //
			    // This is done before marking the transaction start for the base client directory.
			    // This avoids the unwanted restore of the base directory in case of powerdown during
			    // this initialization process that involves deletion and creation of
			    // file instance directories and files. The file instance directories have
			    // their own transaction marker files and are not 'synchronized' with the client directory.
			    // Furthermore, the cleanup process does not do transaction protected deletion
			    // of the file instance directories.
			    // If a power-down occurs during this process, it's best to ensure that
			    // the overall file set is detected as invalid, thus forcing another full cleanup and client re-init,
			    // in the power-up initialization.
			    // This is ensured by marking the client backup directory transaction as in progress (started) first,
			    // followed by the corresponding base client directory. In this way, on power-up, the base client directory
			    // will not be restored from the backup, and the base directory will be determined as invalid, causing
			    // a 'reset' of the client and any underlying file instances (fragments).
			    /////////

			    // create path string for backup directory.
			    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, clientPathBuf);
			    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);

			    RFD_MarkStorageTransactionStart(fileNameBuf);
#endif
			    //////////
			    // Mark start of storage transaction for the base client directory.
			    //////////
			    RFD_MarkStorageTransactionStart(clientPathBuf);
			    {

				    RFD_ClientCleanupStorage(clientInfo);

				    //////////
				    // Initialize and mark start of storage transaction (again).
				    // (RFD_ClientCleanupStorage() will have deleted the transaction file).
				    //
				    // Note: if a power-down occurs here, after RFD_InitializeStorageTransaction() but
				    // before RFD_MarkStorageTransactionStart(), then the transaction will NOT be
				    // detected as incomplete on the following power-up. However, the FileSet file
				    // (RFDR_FSET_FNAME) has not been created yet. Therefore, the client will be detected
				    // as invalid due to error in reading this file.
				    //////////
				    RFD_InitializeStorageTransaction(clientPathBuf, 0);
				    RFD_MarkStorageTransactionStart(clientPathBuf);

				    //////////////////
				    // Initialize default content of FileSet structure.
				    //////////////////

				    RFD_FileSetDefaultInit( fileSetInfo);

				    // create file path string for the File Set filename
				    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, clientPathBuf);
				    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
				    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_FSET_FNAME);

				    // write the file
				    status = RFD_WriteFile(fileNameBuf, fileSetInfo, sizeof(RFD_FSET_MDATA_STRUCT), 
								IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);
				    if(status != RFD_STATUS_OK) {
					    RFD_ReleaseMutex(clientInfo->clientMutex);
					    isThreadExitRequested = TRUE;
					    break;
				    }
			    }

			    // Note: Since the default FileSet specifies zero files, there are
			    // no file process directories to default initialize.

			    //////////
			    // Mark completion of storage transaction
			    //////////
			    RFD_MarkStorageTransactionComplete(clientPathBuf, TRUE);

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
			    //////////
			    // Now that the client storage has been initialized,
			    // backup the client directory to backup location.
			    // Do this after marking storage transaction complete on the client directory.
			    //
			    // Note: the backup directory was marked for Transaction Start above.
			    // The following RFD_BackupDirectoryContents() call serves as the matching
			    // RFD_MarkStorageTransactionComplete(). Therefore another RFD_MarkStorageTransactionComplete()
			    // should Not be called.
			    //////////

			    status = RFD_BackupDirectoryContents(clientPathBuf);
			    if(status != RFD_STATUS_OK) {
				    // Backup failed.
				    RFD_DPrint( TEXT("Error in RFD_BackupDirectoryContents(): status = %d \n"),status);
				    RFD_ReleaseMutex(clientInfo->clientMutex);
				    isThreadExitRequested = TRUE;
				    break;
			    }
#endif
			    //////////////////
			    // Initialize File Process Info structures and files.
			    //////////////////
			    status = RFD_FileProcessInfoArrayCreateDynWithDefInit( clientInfo );

			    if(status != RFD_STATUS_OK) {
				    RFD_DPrint( TEXT("Error in RFD_FileProcessInfoArrayCreateDynWithDefInit: status = %d \n"),status);
				    // Error, signal error exit and break out of client init loop
				    RFD_ReleaseMutex(clientInfo->clientMutex);
				    isThreadExitRequested = TRUE;
				    break;
			    }

			    //////////////////
			    // Load File Process Info structures and files from File System.
			    //////////////////
			    status = RFD_FileProcessInfoArrayReadFromStorage( clientInfo );

			    if(status != RFD_STATUS_OK) {
				    RFD_DPrint( TEXT("Error in RFD_FileProcessInfoArrayReadFromStorage: status = %d \n"),status);
				    // Error, signal error exit and break out of client init loop
				    RFD_ReleaseMutex(clientInfo->clientMutex);
				    isThreadExitRequested = TRUE;
				    break;
			    }
		    }

		    //////////////
		    // Release Mutex
		    //////////////
		    status = RFD_ReleaseMutex(clientInfo->clientMutex);
		    if (status != RFD_STATUS_OK) {
			    // Error, signal error exit and break out of client init loop
			    isThreadExitRequested = TRUE;
			    break;
		    }

	    } // for(i=0;i<nClients;i++)

        if(isThreadExitRequested) {
            break;
        }


	    ////////////////////////////////////
	    // Set the collector isFileSetCollectionComplete status
	    // based on the decoding/collecting state of the files of the file set.
	    // The collector can avoid parsing block messages if isFileSetCollectionComplete
	    // is True.
	    ////////////////////////////////////
	    for(i=0;i<nClients;i++) {

		    clientInfo = &hRfdr->rfdrClientSet->clientInfo[i];
		    collectorInfo = &hRfdr->rfdrClientSet->collectorInfo[i];

		    collectorInfo->isFileSetCollectionComplete = (UCHAR) RFDR_IsFileSetCollectionComplete(clientInfo);
	    }

#ifdef DYNAMIC_BLOCK_MSG_CARID_ENABLE
	    ////////////////////////////////////
	    // Update the Block Message Carousel Id to the
	    // collector message configuration.
	    ////////////////////////////////////
	    for(i=0;i<nClients;i++) {

		    clientInfo = &hRfdr->rfdrClientSet->clientInfo[i];
		    collectorInfo = &hRfdr->rfdrClientSet->collectorInfo[i];

		    if(clientInfo->fileSetInfo.numFiles > 0) {
			    collectorInfo->messageInfo->configInfo.blockMsgCarouselId =
				    clientInfo->fileSetInfo.blockMsgCarouselId;
		    }
	    }
#endif


#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE

	    ///////////////////////
	    // Check for Power-Up in Defragment-Complete state.
	    // The Cleanup as part of the normal defragmentation process may not have completed due to
	    // possible abrupt power-down during prior power-cycle defragmentation operation.
	    ///////////////////////

		for(i=0;i<nClients;i++) {

		    clientInfo = &hRfdr->rfdrClientSet->clientInfo[i];
		    clientIndex = i;

		    RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);

		    // Check if file set is in the Defrag-Complete state.
		    if(RFDR_IsFileSetInDefragCompleteState(clientInfo)) {

			    RFD_DPrint( TEXT("RFDR Processor: Found File Set in Defragment Complete-State, Client Index %d\n"), clientIndex);
			    RFD_DPrint( TEXT("                Cleanup after Powerup in Defragment-Complete state\n"));

			    ////////////////////////////////////
			    // Handle possible condition of  power-loss during defragment operation
			    // whereby deletion of all or some of the defragment input files (decode-output files)
			    // was not completed.
			    ////////////////////////////////////
			    status = RFD_CleanupOnPowerUpInDefragmentState(clientInfo);

			    if(status != RFD_STATUS_OK) {
				    // Unrecoverable error, signal error to manager and exit processing loop to exit thread.
				    RFD_DPrint( TEXT("RFDR Processor: RFD_CleanupOnPowerUpInDefragmentState() error , status %d, Client Index %d\n"),status, clientIndex);
		            RFD_ReleaseMutex(clientInfo->clientMutex);
			        isThreadExitRequested = TRUE;
				    break;
			    }
		    }

		    RFD_ReleaseMutex(clientInfo->clientMutex);
		}

        if(isThreadExitRequested) {
            break;
        }
#endif

	    //////////////////////////////////////////
	    // Initialize vars for first entry the the main processing (Run) loop of the thread.
	    /////////////////////////////////////////

	    ctl->forceNextActionCompletion = FALSE;
	    ctl->actionRequest = RFD_PROC_ACTION__POLL_FOR_ACTION;
	    for(i=0;i<nClients;i++) {
		    // Set Poll For Action Complete to False to indicate polling is required
		    // for this client instance.
		    ctl->isClientPollForActionComplete[i] = FALSE;
	    }

	    ctl->clientIndex                    = clientIndex;
	    ctl->clientInfo                     = clientInfo;
	    ctl->collectorInfo                  = collectorInfo;
	    ctl->consumerInfo                   = consumerInfo;
	    ctl->fileSetInfo                    = fileSetInfo;
	    ctl->fileProcessInfo                = fileProcessInfo;
	    ctl->fileToProcessIndex             = fileToProcessIndex;

        // If we get here, everything is successful - no thread exit requested.
        isThreadExitRequested = FALSE;

    } while(0); // do once

	////////////////////////////////////
	// Set the Init Complete Event
	////////////////////////////////////

    RFD_DPrint( TEXT("RFDR Processor: Initialization Complete, setting Init Complete event\n"));

	status = RFD_SetEvent(initCompleteEvent);
	if (status != RFD_STATUS_OK)  {
	    RFD_DPrint( TEXT("SetEvent failed (%d)\n"), RFD_GetLastError());
	    isThreadExitRequested = TRUE;
    }

	////////////////////////////////////
	// Check for Thread Exit Request
	////////////////////////////////////
    if(isThreadExitRequested) {

        RFD_STATUS closeStatus;

        RFD_DPrint( TEXT("RFDR Processor: Thread Exit condition determined during initialization, signalling error exit\n"));

        // Cleanup if Open fails with Exit Request
        RFDR_ProcessorCloseLL(
            hRfdr,
            &closeStatus );

        // Signal Error Exit
        RFDR_SignalErrorExit(errorExitEvent);
    }

    /////////////////////////////////
    // Set output parameters
    /////////////////////////////////

    *isThreadExitRequestedPtr = isThreadExitRequested;
    *statusPtr = status;

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_ProcessorRunLL(
    RFD_PROCESSOR_THREAD_DATA_HANDLE hRfdr,
    RFD_STATUS * statusPtr,
    BOOL *isThreadExitRequestedPtr,
    BOOL doRunUntilIdle
        )
{
	int nClients;
	int i, j;
	RFD_STATUS status;
	BOOL foundFileToProcess;
	BOOL isDecodeComplete;
	BOOL isFileSetComplete;
	BOOL isThreadExitRequested;
	RFD_EVENT_HANDLE errorExitEvent;
	int ocurredEventIndex;
	BOOL isDecodeWithMaxAllowableNumBlocksCollected;
    BOOL isIdle = FALSE;
    ////////////////////////////////////////////////////
	TCHAR * clientPathBuf;
    TCHAR * fragmentPathBuf;
    TCHAR * fileNameBuf;
	int clientIndex;
	RFDR_CLIENT_INFO_STRUCT * clientInfo;
	RFDR_COLLECTOR_INFO_STRUCT * collectorInfo;
	RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo;
	RFD_FSET_MDATA_STRUCT * fileSetInfo;
	RFDR_FILE_PROCESS_INFO * fileProcessInfo;
	int fileToProcessIndex;
	BOOL forceNextActionCompletion;
	RFD_PROCESSOR_ACTION_ENUM actionRequest;
	RFD_MULTI_WAIT_EVENT_STRUCT * multiCollectWaitEventObject;
	BOOL * isClientPollForActionComplete;
	RFDR_DECODE_STATS decodeStats;
#ifdef ENABLE_ACCUMULATED_DECODER_TEST_STATS
	RFDR_ACCUMULATED_DECODE_STATS accumDecodeStats;
#endif
    RFDR_PROCESSOR_LL_CONTROL_STRUCT * ctl = NULL;
    status = RFD_STATUS_OK;

	errorExitEvent = hRfdr->rfdrProcessorInfo->errorExitEvent;
	nClients = hRfdr->rfdrClientSet->nClients;
    isThreadExitRequested = FALSE;

    ctl = hRfdr->llControl;

    clientPathBuf                  = ctl->clientPathBuf;
    fragmentPathBuf                = ctl->fragmentPathBuf;
    fileNameBuf                    = ctl->fileNameBuf;
    isClientPollForActionComplete  = ctl->isClientPollForActionComplete;
    multiCollectWaitEventObject    = ctl->multiCollectWaitEventObject;
    ////////////////////////////////////////////////////
    clientIndex                    = ctl->clientIndex;
    clientInfo                     = ctl->clientInfo;
    collectorInfo                  = ctl->collectorInfo;
    consumerInfo                   = ctl->consumerInfo;
    fileSetInfo                    = ctl->fileSetInfo;
    fileProcessInfo                = ctl->fileProcessInfo;
    fileToProcessIndex             = ctl->fileToProcessIndex;
    forceNextActionCompletion      = ctl->forceNextActionCompletion;
    actionRequest                  = ctl->actionRequest;
#ifdef ENABLE_ACCUMULATED_DECODER_TEST_STATS
    accumDecodeStats               = ctl->accumDecodeStats;
#endif

    ////////////////////////////////////////////
    // The Run Loop
    ////////////////////////////////////////////

	do {

		if(!forceNextActionCompletion) {

			///////////////////////////////////////////////////
			// Check for Shutdown Request Event
			///////////////////////////////////////////////////

			status = RFD_WaitForEvent(hRfdr->rfdrProcessorInfo->exitRequestEvent, 0);
			if(status == RFD_STATUS_OK) {
				RFD_DPrint( TEXT("RFDR Processor: Shutdown Request Event Occurred\n"));
				// exit the processing loop
				isThreadExitRequested = TRUE;
				break;
			}
		}
		else {
			forceNextActionCompletion = FALSE;
		}

		switch(actionRequest) {

		///////////////////////////////////////////////////////////////////////
		// Poll for Pending Actions
		///////////////////////////////////////////////////////////////////////
		case RFD_PROC_ACTION__POLL_FOR_ACTION:

			RFD_DPrint( TEXT("RFDR Processor: Poll-For-Action Begin\n"));

			////////////////////////////////////
			// For each Client:
			////////////////////////////////////

			for(i=0;i<nClients;i++) {

				// This 'wait for event' action request may be changed in polling this or other client instances,
				// otherwise it will be the next action to take.
				actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;

				clientInfo = &hRfdr->rfdrClientSet->clientInfo[i];

				//////////////////////////
				// Initialize: clientInfo, collectorInfo, and consumerInfo pointers (also fileSetInfo)
				//////////////////////////
				clientIndex = i;
				clientInfo = &hRfdr->rfdrClientSet->clientInfo[clientIndex];
				collectorInfo = &hRfdr->rfdrClientSet->collectorInfo[clientIndex];
				consumerInfo = &hRfdr->rfdrClientSet->consumerInfo[clientIndex];

				fileSetInfo = &clientInfo->fileSetInfo;

				////////
				// Acquire Mutex for current client
				////////
				RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);

				if(clientInfo->fileSetInfo.numFiles == 0) {
					// This client is in a reset state (null fileSet / no files).
					// Set the poll-complete flag then continue to check the next client.
				    isClientPollForActionComplete[clientIndex] = TRUE;
					RFD_ReleaseMutex(clientInfo->clientMutex);
					continue;
				}

				//////////////////////////
				// Check all files of this file set for 'prepared for decode' state.
				// This check must be done before the following check for 'ready for decode'
				// (or the RFDR_STATE_PREPARED_FOR_DECODE should be explicitly excluded by
				// the 'ready for decode' check).
				//////////////////////////
				foundFileToProcess = FALSE;

				for(j=0;j<clientInfo->fileSetInfo.numFiles;j++) {

					if( clientInfo->fileProcessInfo[j]->state == RFDR_STATE_PREPARED_FOR_DECODE ) {

						RFD_DPrint( TEXT("RFDR Processor: Detected 'Prepared for Decode' State on polling, Client Index %d, File Index %d\n"),clientIndex, j);
						foundFileToProcess = TRUE;
						break;
					}
				}

				if(foundFileToProcess) {

					// Initialize fileToProcessIndex and fileProcessInfo that are also expected initialized
					// for the decode action.
					fileToProcessIndex = j;
					fileProcessInfo = clientInfo->fileProcessInfo[j];

					// The following variables are intialized at this point:
					//  clientIndex
					//  clientInfo
					//  collectorInfo
					//  consumerInfo
					//  fileToProcessIndex
					//  fileProcessInfo

					// set action request to Decode (not 'Prepare for decode' which is the normal
					// sequence for uniterrupted decoding).
					RFD_ReleaseMutex(clientInfo->clientMutex);
					actionRequest = RFD_PROC_ACTION__DECODE;

                    // Do NOT clear the poll-for-action-complete flag for this client here because
                    // there may be other file (fragments) of this client's file set that need to
                    // be polled for action.

					// Break from client loop to perform the Decode action.
					break;
				}
                else {

				    //////////////////////////
				    // Check all files of this file set for 'ready for decode' condition
				    // i.e. if one of the decoding/collecting states and enough blocks
				    // and buffered to do a decode operation.
				    //////////////////////////
				    foundFileToProcess = FALSE;

				    for(j=0;j<clientInfo->fileSetInfo.numFiles;j++) {

					    // Verify that file is in the expected state
					    // (collecting/decoding) for a block update event.
					    // Explicitly exclude the RFDR_STATE_PREPARED_FOR_DECODE state that is
					    // a special case that is handled separately.
					    if( (clientInfo->fileProcessInfo[j]->state < RFDR_STATE_DECODE_COMPLETE) &&
					        (clientInfo->fileProcessInfo[j]->state > RFDR_STATE_UNINIT) &&
					        (clientInfo->fileProcessInfo[j]->state != RFDR_STATE_PREPARED_FOR_DECODE) ) {

						    if(RFDR_IsBlockCollectionStateReadyForDecode(clientInfo->fileProcessInfo[j])) {

							    RFD_DPrint( TEXT("RFDR Processor: Detected Block Collection State Ready for Decode on polling, Client Index %d, File Index %d\n"),clientIndex, j);
							    collectorInfo->fileProcessCollectorUpdate[j] = TRUE;
							    foundFileToProcess = TRUE;
							    // continue polling all file instances of this client.
						    }
					    }
				    }

				    if(foundFileToProcess) {
					    // Set event signal. The specific file instance flag/s is/are set above.
					    // They will be processed the next time the 'wait event' action
					    // is taken.
					    RFD_SetEvent(clientInfo->clientEvent);
					    // set action request to wait for event. It will be overwritten
					    // if an action is found while polling the other client instances
					    // in this loop. If this occurs, this file instance will be processed
					    // whenever the next action goes to 'wait for event'.
					    RFD_ReleaseMutex(clientInfo->clientMutex);
					    actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;

                        // Need to set the poll-for-action-complete flag, otherwise we are locked up
                        // in infinite loop cycling between RFD_PROC_ACTION__WAIT_FOR_EVENT action and
                        // RFD_PROC_ACTION__POLL_FOR_ACTION actions (RFD_PROC_ACTION__WAIT_FOR_EVENT
                        // action first checks to see if polling is completed and jumps back here if
                        // it is not).
                        // At this point, threfore (since the flag is being cleared), the only current
                        // pending actions for this client's fileSet and all
                        // it's file instances (fragments) must be this processing of this 'ready-to-decode'
                        // condition (driven by setting the clientEvent).
				        isClientPollForActionComplete[clientIndex] = TRUE;

					    // Do Not break from client loop; perform 'continue' instead
					    // to poll other clients for actions.
					    continue;
				    }
                }


				///////////////////////////////////
				// Check if all file instances (fragments) are in the decode complete state.
				// If yes, then set flag to indicate ready for defrag.
				///////////////////////////////////
				isFileSetComplete = RFDR_IsFileSetInDecodeCompleteState(clientInfo);

				if(isFileSetComplete) {
					RFD_DPrint( TEXT("RFDR Processor: Detected 'All fragments Decode Complete' State on polling, Client Index %d\n"),clientIndex);
					RFD_ReleaseMutex(clientInfo->clientMutex);
					actionRequest = RFD_PROC_ACTION__DEFRAGMENT;
					break; // break from client loop
				}

				///////////////////////////////////
				// Check if file set is in the defrag complete state.
				// If yes, then set flag to indicate ready for decompress.
				///////////////////////////////////
				isFileSetComplete = RFDR_IsFileSetInDefragCompleteState(clientInfo);

				if(isFileSetComplete) {
					RFD_DPrint( TEXT("RFDR Processor: Detected 'Defragment Complete' State on polling, Client Index %d\n"),clientIndex);
					RFD_ReleaseMutex(clientInfo->clientMutex);
					actionRequest = RFD_PROC_ACTION__DECOMPRESS;
					break; // break from client loop
				}

				///////////////////////////////////
				// Check if file set in the decompress complete state.
				// If yes, then set event to signal consumer thread to transfer the data file.
				///////////////////////////////////
				isFileSetComplete = RFDR_IsFileSetInDecompressCompleteState(clientInfo);

				if(isFileSetComplete) {
					// The fileSet data is ready for transfer to the consumer.
					// Signal consumer thread to do the transfer.
					RFD_DPrint( TEXT("RFDR Processor: Detected 'Decompress Complete' State on polling, Client Index %d\n"),clientIndex);
					// No explicit next action has been is here for the processor thread
                    // (just the default RFD_PROC_ACTION__WAIT_FOR_EVENT).
					// Only an Event is set to signal action by the consumer thread.
                    RFD_SetEvent(consumerInfo->fileSetReadyEvent);
					// set action request to wait for event. It will be overwritten
					// if an action is found while polling the other client instances
					// in this loop.
					RFD_ReleaseMutex(clientInfo->clientMutex);
					actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;

                    // Need to set the poll-for-action-complete flag. Otherwise, until this client
                    // processes the fileSetReadyEvent event to get the RFD file to progress to
                    // the finalized state, we are locked up
                    // cycling between RFD_PROC_ACTION__WAIT_FOR_EVENT action and
                    // RFD_PROC_ACTION__POLL_FOR_ACTION actions (RFD_PROC_ACTION__WAIT_FOR_EVENT
                    // action first checks to see if polling is completed and jumps back here if
                    // it is not).
				    isClientPollForActionComplete[clientIndex] = TRUE;

					// Do Not break from client loop; perform 'continue' instead
					// to poll other clients for actions.
					continue;
				}

				//////////////////////////
				// If this point reached, then no actions detected for this client instance.
				// Set Poll For Action Complete to True to indicate no more polling required
				// for this client instance.
				//////////////////////////
				isClientPollForActionComplete[clientIndex] = TRUE;

				// Release Mutex for this client instance.
				RFD_ReleaseMutex(clientInfo->clientMutex);

			} // for(i=0;i<nClients;i++)

			break;


		///////////////////////////////////////////////////////////////////////
		// Wait For Event Action
		///////////////////////////////////////////////////////////////////////
		case RFD_PROC_ACTION__WAIT_FOR_EVENT:

			// Check if polling all clients is complete. If not,
			// then change the action request to 'poll for action'
			// and then break out of this current action.
			for(i=0;i<nClients;i++) {
				if(isClientPollForActionComplete[i] == FALSE) {
					actionRequest = RFD_PROC_ACTION__POLL_FOR_ACTION;
                    // break from this inner for loop
					break;
				}
			}
            if(actionRequest == RFD_PROC_ACTION__POLL_FOR_ACTION) {
                // break from this action to do the poll action.
                break;
            }

			///////////////////////////////////////////////////
			// Wait for Collect Event
			///////////////////////////////////////////////////

			status = RFD_WaitForOneOfMultiEvents(multiCollectWaitEventObject,
												 RFDR_PROCESSOR_WAIT_COLLECT_EVENT_TIMEOUT_MSEC,
												 &ocurredEventIndex);
			if(status == RFD_STATUS_OBJECT_TIMEOUT) {
				// Timeout, actionRequest remains the same to loopback and
				// do same action
				actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;
                // there's nothing to do but wait for an event,
                // set the idle flag.
                isIdle = TRUE;
				break;
			}
			else if(status != RFD_STATUS_OK) {
				// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
				RFDR_SignalErrorExit(errorExitEvent);
				isThreadExitRequested = TRUE;
				break;
			}
			else {
				//////////////////////////
				// A client collect event ocurred.
				// Initialize: clientInfo, collectorInfo, and consumerInfo pointers
				//////////////////////////
				clientIndex = ocurredEventIndex;
				clientInfo = &hRfdr->rfdrClientSet->clientInfo[clientIndex];
				collectorInfo = &hRfdr->rfdrClientSet->collectorInfo[clientIndex];
				consumerInfo = &hRfdr->rfdrClientSet->consumerInfo[clientIndex];
				RFD_DPrint( TEXT("RFDR Processor: Collect Event Occurred, Client Index %d\n"),clientIndex);
			}

			/////////////////////////////////////////////////
			// Event occurred, Process the Event
			/////////////////////////////////////////////////

			////////
			// Acquire Mutex
			////////
			RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);

			////////////
			// Access shared data
			////////////

			if(collectorInfo->fileSetCollectorUpdate) {

				/////////////////////////
				// File Set Metadata Update occurred.
				//   Check if different from current File Set Metadata.
                //
				//   If different, excluding lifeTime field, cleanup previous file set file sys and
				//   dynamic allocations, and reinitialize for the File Set Metadata.
                //
                //   If only the lifeTime value is different, just update the lifeTime value to
                //   Client File Set Info and to the stored File Set Info, so the new value is
                //   available to the application (by subsequent calls to RFD_CollectorGetClientStateInfo().
				////////////////////////

				collectorInfo->fileSetCollectorUpdate = FALSE;

				// Compare current and new file set info structures.
                // Note: This check ignores changes in the Life (lifeTime) field, as
                //       a change in Life value does not have to be treated as a completely new File Set.
                //       A change in Life value is handled later below.
				if(RFD_IsNewFileSetMetadata(&clientInfo->fileSetInfo,
											&collectorInfo->fileSetInfoBuf)) {

					RFD_DPrint( TEXT("RFDR Processor: File Set Metadata update detected, Client Index %d\n"),clientIndex);

					// contents are new, copy new to current.
					RFD_MEMCPY(&clientInfo->fileSetInfo, &collectorInfo->fileSetInfoBuf, sizeof(RFD_FSET_MDATA_STRUCT));

#ifndef RFD_FSET_MESSAGE_INCLUDE_FILE_SIZES_ENABLE
					// calulate and set the fileSize fields for each file of the file set.
					RFD_CalculateFileSetMetatdataFileSizes(&clientInfo->fileSetInfo);
#endif
					// No need to reset the fileSet since this file set update effectively does this
					// (restarts file set processing)
					consumerInfo->fileSetResetRequest = FALSE;

					// With fileSetInfo set, next action is to inialize the process using
					// the fileSetInfo.
					actionRequest = RFD_PROC_ACTION__INITIALIZE_PROCESS;
                    // Ignore any other flags associated with this event that are also set
                    // (e.g. block update flag); release mutex and break to
                    // process this next action.
			        RFD_ReleaseMutex(clientInfo->clientMutex);
                    break;
				}
                else if(clientInfo->fileSetInfo.lifeTime != collectorInfo->fileSetInfoBuf.lifeTime) {

                    // The Metadata Life (liefTime) value has changed, perform the Life value update
                    // to the Client File Set Info structure and to the stored File Set Info. After this update, the
                    // new Life value is then available to the RFD Receiver application (e.g. MFM Multifile Manager)
                    // through any subsequent calls to RFD_CollectorGetClientStateInfo().

					RFD_DPrint( TEXT("RFDR Processor: File Set Metadata Life value update detected, from (%d) to (%d), Client Index %d\n"),
                        clientInfo->fileSetInfo.lifeTime,
                        collectorInfo->fileSetInfoBuf.lifeTime,
                        clientIndex);

                    // Update the value to Client File Set Info structure.
                    clientInfo->fileSetInfo.lifeTime = collectorInfo->fileSetInfoBuf.lifeTime;

                    // Print the new FileSet (Metadata) Info.
                    RFD_PrintFileSetInfo( &clientInfo->fileSetInfo );

			        // Construct client path name string.
			        RFDR_ConstructClientDirPathName(clientPathBuf, RFD_MAX_PATH_LEN, clientInfo);

                    RFD_MarkStorageTransactionStart(clientPathBuf);
			        {
				        RFD_DPrint( TEXT("RFDR Processor: Writing the File Set Metadata file, Client Index\n"), clientInfo->clientIndex);

					    //////////////////
					    // write the updated File Set Metadata file
					    //////////////////

					    // Create file path string for the File Set filename
					    RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, clientPathBuf);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					    RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_FSET_FNAME);

					    // Write the fileset file
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
						// with File Sync enabled, to sync file to disk before Completeing the transaction.
#endif
					    status = RFD_WriteFile(fileNameBuf, &clientInfo->fileSetInfo,
													      sizeof(RFD_FSET_MDATA_STRUCT),
														  IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);
					    if(status != RFD_STATUS_OK) {
						    // Unrecoverable error, signal error to manager and exit processing loop to exit thread.
						    RFD_ReleaseMutex(clientInfo->clientMutex);
						    RFDR_SignalErrorExit(errorExitEvent);
						    isThreadExitRequested = TRUE;
						    break;
					    }
			        }
			        RFD_MarkStorageTransactionComplete(clientPathBuf, TRUE);

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
			        //////
			        // Backup Storage after above RFD_MarkStorageTransactionComplete().
			        // An 'update' action is complete.
			        //////
			        status = RFD_BackupDirectoryContents(clientPathBuf);

					if(status != RFD_STATUS_OK) {
                        RFD_ReleaseMutex(clientInfo->clientMutex);
                        RFDR_SignalErrorExit(errorExitEvent);
                        isThreadExitRequested = TRUE;
					    break;
                    }
#endif
					// the change in FileSet Metadata Life has been handled here,
                    // so there's no additional File Set related action.
					// Set next action to wait for event, but don't break-out so as to
                    // check other flags associated with this event.
					actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;
                }
				else {
					// new FileSet Metadata is the same as previous, so no File Set related action.
					// set next action to wait for event, but don't break-out so as to
                    // check other flags associated with this event.
					actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;
				}
			}

            if(consumerInfo->fileSetResetRequest) {

				// consumerInfo->fileSetResetRequest would typically be set
				// due to error detection by the consumer, so force next action completion.
                //
                // Postpone clearing the consumerInfo->fileSetResetRequest. This same flag will be checked
                // at end of the reset and following process initialialization to then perform all
                // final actions to clear the reset request.

				forceNextActionCompletion = TRUE;
				actionRequest = RFD_PROC_ACTION__RESET_FSET_MDATA;

                // Ignore any other flags associated with this event that are also set
                // (e.g. block update flag); release mutex and break to
                // process this next action.
			    RFD_ReleaseMutex(clientInfo->clientMutex);
                break;
			}

			////////////////////////
			// check each block update event flag for each fid.
			////////////////////////

			foundFileToProcess = FALSE;

			for(i=0;i<clientInfo->fileSetInfo.numFiles;i++) {
				if(collectorInfo->fileProcessCollectorUpdate[i]) {

					// Clear update flag.
					collectorInfo->fileProcessCollectorUpdate[i] = FALSE;
					// Verify that file is in the expected state
					// (collecting/decoding) for a block update event and
					//      the block collection is sufficient for decode OR
					//      the Maxumum Allowable Number of Blocks has been collected.
					if( clientInfo->fileProcessInfo[i]->state < RFDR_STATE_DECODE_COMPLETE &&
						clientInfo->fileProcessInfo[i]->state > RFDR_STATE_UNINIT ) {

						if( RFDR_IsBlockCollectionStateReadyForDecode(clientInfo->fileProcessInfo[i]) ||
							RFDR_IsMaxAllowableNumBlocksCollected(clientInfo->fileProcessInfo[i]) ) {

							RFD_DPrint( TEXT("RFDR Processor: Block Collection Update dectected, Client Index %d, File Index %d\n"),clientIndex, i);
							foundFileToProcess = TRUE;
							fileToProcessIndex = i;
							fileProcessInfo = clientInfo->fileProcessInfo[i];
							break;
						}
					}
				}
			}

			if(foundFileToProcess) {
				// Found a file ready to process (decode).

				// Additonal block updates could be flagged for this detected event.
				// If additional updates, then re-set the event so that processor task processes the event on the next wait-for-event.
				for(i=0;i<clientInfo->fileSetInfo.numFiles;i++) {
					if(collectorInfo->fileProcessCollectorUpdate[i]) {
						RFD_DPrint( TEXT("RFDR Processor: Additional Block Collection Update dectected, re-setting event, Client Index %d, File Index %d\n"),clientIndex, i);

						RFD_SetEvent(clientInfo->clientEvent);
						break;
					}
				}

				// The following variables are intialized at this point:
				//  clientIndex
				//  clientInfo
				//  collectorInfo
				//  consumerInfo
				//  fileToProcessIndex
				//  fileProcessInfo

				// Next state is prepare-for-decode.
				actionRequest = RFD_PROC_ACTION__PREPARE_FOR_DECODE;
			}
			else {
				actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;
			}


			RFD_ReleaseMutex(clientInfo->clientMutex);
			break;

		///////////////////////////////////////////////////////////////////////
		// Prepare for Decode Action
		///////////////////////////////////////////////////////////////////////
		case RFD_PROC_ACTION__PREPARE_FOR_DECODE:

			RFD_DPrint( TEXT("RFDR Processor: Prepare-For-Decode-Action Begin, Client Index %d, fileIndex %d\n"), clientIndex, fileToProcessIndex);

			// Initialize pathbuf string for the fragment (file instance) directory.
			RFDR_ConstructFragmentDirPathName(fragmentPathBuf, RFD_MAX_PATH_LEN,
											  clientInfo, fileProcessInfo->fileInfo.id);

			//////////////
			// Acquire Mutex
			//////////////
			RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);

#if defined(RFDR_ENABLE_STORAGE_OPTIM_1) && defined(RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE)
			//////////////
			// Backup the Process directory before updating the process directory for prepare-for-decode operation.
			// This is in support of a storage space optimization. It is a special case usage scenario. In this
			// case, the backup operation has been postponed from the end of the last decode
			// operation for this fragment until now. Storage space is saved because the process dir is
			// instead cleared/cleanup up at the end to the last decode so that only one fragment at a time
			// has decoding files backed up for the process dir.
			// This alternate use case also requires a special check during initialization to avoid backup
			// for the process dirs if the processing state is 'collecting'.
			//////////////

			// Construct file path for the Process directory
			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

			status = RFD_BackupDirectoryContents(fileNameBuf);

			if(status != RFD_STATUS_OK) {
				// Reset the FileSet to attempt to recover from error.
				// This resets all files of the file set, not just the single file
				// detected as failed here.
				// Do Not mark transaction complete (i.e. RFD_MarkStorageTransactionComplete())
				// for error conditions (in case powerdown right after marking transaction complete).

				RFD_DPrint( TEXT("RFDR Processor: RFD_BackupDirectoryContents error during prepare for decode, status %d, Client Index %d, fileIndex %d\n"),status, clientIndex, fileToProcessIndex);

				RFD_ReleaseMutex(clientInfo->clientMutex);

				// force completion of next action.
				forceNextActionCompletion = TRUE;
				// Next action is to initialize the RFD Process for the current client instance
				// using the default fileSet.
				actionRequest = RFD_PROC_ACTION__RESET_FSET_MDATA;
				break;
			}
#endif

			/////////
			// Mark (multi-directory) Fragment Transaction start.
			// Note: Mutex must be aquired before calling this function.
			/////////

			RFD_DoFragmentMarkStorageTransactionStart(fragmentPathBuf, FALSE, FALSE);
			{
				// rename src and enc input buffer files to dec...
				//
				// create new (zero len) src and enc input buffer files
				//
				// Copy dec state structures to dec...
				//
				// update dec state src and enc count variables i.e. move inbuf counts to in-solution count
				// and update collect threshold to next level, so that collector will continue to collect
				// and signal events. Note: decoder may reduce in-solution count due to linear dependence elimination
				// so that after decoder is complete, this processor task should reduce the
				// dec state in-solution count and check for signal read status and set signal event
				// (normally done by collector task)
				//

				//////////////
				// Prepare for Decode (with mutex aquired).
				//////////////

				// Call with doStorageUpdatesOnly parameter = FALSE so that
				// Data/variable updates are done, along with Storage Updates
				// as part of the prepare for decode.
				status = RFDR_PrepareForDecode(fragmentPathBuf, fileProcessInfo, FALSE);

				if(status != RFD_STATUS_OK) {
					// Reset the FileSet to attempt to recover from error.
					// This resets all files of the file set, not just the single file
					// detected as failed here.
					// Do Not mark transaction complete (i.e. RFD_MarkStorageTransactionComplete())
					// for error conditions (in case powerdown right after marking transaction complete).

					RFD_DPrint( TEXT("RFDR Processor: RFDR_PrepareForDecode error, status %d, Client Index %d, fileIndex %d\n"),status, clientIndex, fileToProcessIndex);

					RFD_ReleaseMutex(clientInfo->clientMutex);

					// force completion of next action.
					forceNextActionCompletion = TRUE;
					// Next action is to initialize the RFD Process for the current client instance
					// using the default fileSet.
					actionRequest = RFD_PROC_ACTION__RESET_FSET_MDATA;
					break;
				}

				//////////////////////
				// update state to preparing-for-decode (with mutex acquired)
				// so that if power-down reset occurs during decode,
				// initialization knows to call RFDR_RevertDecodeOps()
				// to undo the RFDR_PrepareForDecode() changes.
				//////////////////////
				fileProcessInfo->state = RFDR_STATE_PREPARED_FOR_DECODE;

				//////////////////////
				// Write the Process State file
				//////////////////////

				// Construct directory path to Process State file.
				RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
				RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
				RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process directory

				status = RFD_FileProcessStateWriteToStorage(fileNameBuf, &fileProcessInfo->state);

				if(status != RFD_STATUS_OK) {
					// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
					// Do Not mark transaction complete (i.e. RFD_MarkStorageTransactionComplete())
					// for error conditions (in case powerdown right after marking transaction complete).
					RFD_ReleaseMutex(clientInfo->clientMutex);
					RFDR_SignalErrorExit(errorExitEvent);
					isThreadExitRequested = TRUE;
					break;
				}

			//////////////
			// Mark (multi-directory) Fragment Transaction complete.
			//////////////
			}
			RFD_DoFragmentMarkStorageTransactionComplete(fragmentPathBuf, FALSE);

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE

			///////////////
			// Repeat the same Decode-Prepare file operations on the Backup Directory.
			///////////////

			//////////////
			// Prepare for backup following the file changes due to the
			// preceeding RFDR_PrepareForDecode() operations.
			//////////////

			RFD_DoFragmentMarkStorageTransactionStart(fragmentPathBuf, TRUE /* mark backup locations */, FALSE);
			{
				//////////////
				// Call RFDR_PrepareForDecode() again to update the backup storage.
				// Call with path name pointing to the corresponding backup directory.
				// Call with doStorageUpdatesOnly parameter = TRUE so that
				// only storage updates are done, not Data/variable updates.
				//
				// Calling RFDR_PrepareForDecode() to update the backup storage
				// in this manner is quicker that blindly copying the whole directory contents
				// into the backup location. This is because RFDR_PrepareForDecode()
				// does file renaming, not copying on a subset of files (block input buffers)
				// and minimal other file updates (decoder state file).
				//////////////

				status = RFDR_PrepareForDecode(fragmentPathBuf, fileProcessInfo, TRUE);

				if(status != RFD_STATUS_OK) {
					// Reset the FileSet to attempt to recover from error.
					// This resets all files of the file set, not just the single file
					// detected as failed here.
					// Do Not mark transaction complete (i.e. RFD_MarkStorageTransactionComplete())
					// for error conditions (in case powerdown right after marking transaction complete).

					RFD_DPrint( TEXT("RFDR Processor: RFDR_PrepareForDecode error on backup storage pass, status %d, Client Index %d, fileIndex %d\n"),status, clientIndex, fileToProcessIndex);
					RFD_ReleaseMutex(clientInfo->clientMutex);

					// force completion of next action.
					forceNextActionCompletion = TRUE;
					// Next action is to initialize the RFD Process for the current client instance
					// using the default fileSet.
					actionRequest = RFD_PROC_ACTION__RESET_FSET_MDATA;
					break;
				}

				//////////////////////
				// Write the Process State to the Process State file
				// as was done above to maintain synchronized backup.
				//////////////////////

				// Construct backup directory path
				RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
				RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
				RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process directory
				RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
				RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);

				// write the state file to the Backup Directory (fileNameBuf).
				status = RFD_FileProcessStateWriteToStorage(fileNameBuf, &fileProcessInfo->state);

				if(status != RFD_STATUS_OK) {
					// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
					// Do Not mark transaction complete (i.e. RFD_MarkStorageTransactionComplete())
					// for error conditions (in case powerdown right after marking transaction complete).

					RFD_ReleaseMutex(clientInfo->clientMutex);
					RFDR_SignalErrorExit(errorExitEvent);
					isThreadExitRequested = TRUE;
					break;
				}

			//////////////
			// Backup operations finished, so complete the fragment-level transaction for the backup locations.
			//////////////
			}
			RFD_DoFragmentMarkStorageTransactionComplete(fragmentPathBuf, TRUE /* mark backup locations */);
#endif
			//////////////
			// Release Mutex
			//////////////
			RFD_ReleaseMutex(clientInfo->clientMutex);

            RFD_DPrint( TEXT("RFDR Processor: Prepare-For-Decode-Action Completed, Client Index %d, fileIndex %d\n"), clientIndex, fileToProcessIndex);

			// Prepare-for-Decode step is complete.
			// Next step is to do the Decode operation.
			actionRequest = RFD_PROC_ACTION__DECODE;
			break;


		///////////////////////////////////////////////////////////////////////
		// Decode Action
		///////////////////////////////////////////////////////////////////////
		case RFD_PROC_ACTION__DECODE:

			RFD_DPrint( TEXT("RFDR Processor: Decode-Action Begin, Client Index %d, fileIndex %d\n"), clientIndex, fileToProcessIndex);

			// Construct string for the fragment (file instance) directory.
			RFDR_ConstructFragmentDirPathName(fragmentPathBuf, RFD_MAX_PATH_LEN,
											  clientInfo, fileProcessInfo->fileInfo.id);

			// Construct the process directory path name.
			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

			//////////////
			// Acquire Mutex
			//////////////
			RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);

			//////////////
			// Mark storage transaction start in only the process directory
			// for next decoding step. The decoder only updates files in the process direcory,
			// not the collect directory.
			// By marking only the process direcotory for transaction started,
			// the mutex may be released around the long running decode process. This allows
			// the collector thread to run. The collector thread only updates files in the collect directory.
			// Therefore there is no possibility in both the base and backup directory transaction in progress
			// due to multithreaded opeartion.
			//////////////
			RFD_MarkStorageTransactionStart(fileNameBuf);
			{

				//////////////
				// Decode step
				//////////////

				isDecodeComplete = FALSE;

				// Check if this decode (chunk) will be done with the maximum number of allowable
				// blocks collected. If the decode (chunk) does not complete the overall decode
				// with max blocks collected, then the file process will be reset to reattempt the overall decode.
				// This check is done before calling decode (and with mutex aquired).
				// If the check is done after decode, and additional blocks were collected during the decode,
				// then the check is not accurate i.e. there are more blocks to be decoded even though the
				// IsMaxAllowableNumBlocksCollected funtion indicates true.
				isDecodeWithMaxAllowableNumBlocksCollected =
					RFDR_IsMaxAllowableNumBlocksCollectedXccCode( &fileProcessInfo->decInfo );

				////////////
				// Release the Mutex before the potentially long running Decode Operation
				////////////
				RFD_DPrint( TEXT("RFDR Processor: Releasing Mutex before decode, Client Index %d\n"),clientIndex);

				RFD_ReleaseMutex(clientInfo->clientMutex);

				//////////////
				// Decode
				//////////////
				status = RFDR_Decode(fileNameBuf, fileProcessInfo, hRfdr->rfdrProcessorInfo->exitRequestFlagPtr);

				//////////////
				// Re-Acquire Mutex
				//////////////
				RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);

				if(status == RFD_STATUS_EARLY_EXIT_ON_REQUEST) {
					RFD_DPrint( TEXT("RFDR Processor: Exit Requested during Decode, Client Index %d, fileIndex %d\n"), clientIndex, fileToProcessIndex);

					// The Decoder has exited early due to polling the exitRequestFlag.
					// The corresponding event, exitRequestEvent, should still be set; however,
					// re-asserting the SetEvent() for clarity is done here.
					RFD_SetEvent(hRfdr->rfdrProcessorInfo->exitRequestEvent);

					// Cancel the storage transaction.
					// In the case of early decoder exit on request,
					// the decoder has Not modified any files in storage. The state is still 'prepare for decode',
					// and the decoder should resume from this state on the next powerup.
#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
					// Comment for Power-Loss Resilient Option:
					// Cancel the storage transaction instead of the Completing the storage transaction
					// so that the 'sync count' (total number of transactions counted) is decremented.
					// There is not a corresponding Backup transaction operation here, so the Cancel operation keeps
					// the two transaction instances in-sync. Therefore, the base directory content
					// will Not be unecessarily copied to backup on powerup due to out-of-sync detection.
#endif
					RFD_CancelStorageTransaction(fileNameBuf);

					RFD_ReleaseMutex(clientInfo->clientMutex);
					// The exitRequestEvent is tested on each action request loop.
					// actionRequest is set to some default anyway.
					actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;
					break;
				}
				else if(status != RFD_STATUS_OK) {
					// Reset the FileSet to attempt to recover from error.
					// This resets all files of the file set, not just the single file
					// detected as failed here.
					//
					// Do Not mark the transaction complete (i.e. RFD_MarkStorageTransactionComplete()).
					// If the transaction was to be marked complete, there is the danger that
					// a power-down could occur right after this but before the FileSet reset.
					// On the next powerup, it would appear that the storage is in a normal configuration
					// in the 'prepared for decode state; however, the storage might not correpond to this
					// state depending on the error.
					// By Not marking the transaction complete, in the case of an ill timed power down,
					// a reset of the fileSet will be forced on the next powerup (for full power-loss resilience
					// build, the incomplete transaction will be detected and the backup will be restored to
					// achieve the valid previous storage).

					RFD_DPrint( TEXT("RFDR Processor: RFDR_Decode error, status %d, Client Index %d, fileIndex %d\n"),status, clientIndex, fileToProcessIndex);
					RFD_ReleaseMutex(clientInfo->clientMutex);
					// force completion of next action.
					forceNextActionCompletion = TRUE;
					// Next action is to initialize the RFD Process for the current client instance
					// using the default fileSet.
					actionRequest = RFD_PROC_ACTION__RESET_FSET_MDATA;
					break;
				}
				// Else Decode executed normally

				// Decode executed normally (either a partial decode or completed decode).

                RFD_DPrint( TEXT("RFDR Processor: Decode-Action Completed, Client Index %d, fileIndex %d\n"), clientIndex, fileToProcessIndex);

			}
			//////////////
			// Skip the Marking of storage transaction complete (RFD_DoFragmentMarkStorageTransactionComplete())
			// for the *process* directory. Instead, the following *fragment*
			// RFD_DoFragmentMarkStorageTransactionStart() is called with the isProcessTransacInProgress
			// set True. In this way, the process directory is *contiguously* marked as transaction-in-progress.
			// The RFD_DoFragmentMarkStorageTransactionStart() will internally skip marking
			// the process dir transaction started so that after the final RFD_DoFragmentMarkStorageTransactionComplete(),
			// the same number of transaction starts and transaction completes will be performed.
			// This implementation ensures that the process directory is always marked with transaction-in-progress
			// during this sequence and the following error condition is avoided:
			//
			// If the decode sub-step does a RFD_MarkStorageTransactionComplete() on the process directory
			// followed by the post-decode sub-step doing a RFD_DoFragmentMarkStorageTransactionStart() on the overall fragment,
			// a problem would occur if a power-down/abort occured between these two transaction markers.
			// i.e. upon power-up, the decode state would be detected and attempted again, however, the decode
			// had already been performed (and process dir storage updated) so an error condition would occur.
			//
			// The solution here is to ensure that the process directory is always marked tranaction-in-progress (started)
			// for the entire sequence. In this way, a powerdown at any point in this sequence will result
			// in a restore from backup on the following powerup, resulting in a return to the
			// 'prepared-for-decode' state.
			//
			// An alternate solution to this issue would have been to split this decode state into two states,
			// decode and post-decode states (and corresponding actions) between the RFD_MarkStorageTransactionComplete()
			// and RFD_DoFragmentMarkStorageTransactionStart(). However, this requires storing the variable
			// isDecodeWithMaxAllowableNumBlocksCollected, the private decode info, and process state because
			// this info would need to be restored in case of powerdown between the two new states
			// (between the two trans markers).
			//////////////


			//////////////
			// Mark storage transaction start for the overall fragment.
			// The RFDR_PostDecodeOps() and possibly other following functions
			// will modify files in both the collect and process directories underlying this fragment.
			// The mutex must NOT be released while the overall fragment transaction is started.
			// This ensures that the collector thread will Not run and perform any file updates (in collector dir).
			// If that were to happen, it is possible that both a backup dir and it's base collect directory
			// are open with transactions in progress when a power loss occurs. For the case of the full power loss
			//  resilience build, this means that the backup will not be restored, and file processing is reset.
			//
			// Call with the isProcessTransacInProgress = True as described above.
			//////////////
			RFD_DoFragmentMarkStorageTransactionStart(fragmentPathBuf,
													  FALSE,
													  TRUE /* indicate process dir already marked in-progress/started */ );
			{
				//////////////
				// Do Post Decode operations, with the collector mutex acquired.
				//////////////
				status = RFDR_PostDecodeOps(fragmentPathBuf, fileProcessInfo, &isDecodeComplete);

				if(status != RFD_STATUS_OK) {
					// Reset the FileSet to attempt to recover from error.
					// This resets all files of the file set, not just the single file
					// detected as failed here.
					//
					// Do Not mark the transaction complete (i.e. RFD_MarkStorageTransactionComplete()).
					// If the transaction was to be marked complete, there is the danger that
					// a power-down could occur right after this but before the FileSet reset.
					// On the next powerup, it would appear that the storage is in a normal configuration
					// in the 'prepared for decode state; however, the storage might not correpond to this
					// state depending on the error.
					// By Not marking the transaction complete, in the case of an ill timed power down,
					// a reset of the fileSet will be forced on the next powerup (for full power-loss resilience
					// build, the incomplete transaction will be detected and the backup will be restored to
					// achieve the valid previous storage).

					RFD_DPrint( TEXT("RFDR Processor: RFDR_PostDecodeOps error, status %d, Client Index %d, fileIndex %d\n"),status, clientIndex, fileToProcessIndex);
					RFD_ReleaseMutex(clientInfo->clientMutex);
					// force completion of next action.
					forceNextActionCompletion = TRUE;
					// Next action is to initialize the RFD Process for the current client instance
					// using the default fileSet.
					actionRequest = RFD_PROC_ACTION__RESET_FSET_MDATA;
					break;
				}

				if(isDecodeComplete) {
					RFD_DPrint( TEXT("RFDR Processor: Decoding Complete, Client Index %d, fileIndex %d\n"),clientIndex, fileToProcessIndex);

					//////////////////////////////////
					// Print Decode Solution Satistics
					//////////////////////////////////
					if(RFDR_GetDecoderSolutionStats(fileProcessInfo, &decodeStats) == RFD_STATUS_OK) {

						RFDR_PrintDecoderSolutionStats(&decodeStats);

#ifdef ENABLE_ACCUMULATED_DECODER_TEST_STATS
						RFDR_UpdateAccumulatedDecoderSolutionStats(&accumDecodeStats, &decodeStats);
						RFDR_PrintAccumulatedDecoderSolutionStats(&accumDecodeStats);
#endif
					}

					//////////////////////
					// CRC Check the decoded file.
					//////////////////////

					//////////////////////
					// RFDR_ValidateFileCrc() is a potentially long running operation.
					// It might be beneficial to release mutex to enable the colletor thread
					// to run and collect blocks. However, the mutex should NOT be released
					// while the overall fragment transaction is in progress, to avoid possible loss
					// of data due to powerdown (for full power-loss resilience build).
					// An alternative to enable release of mutex around RFDR_ValidateFileCrc()
					// is to mark the fragement transaction complete also around RFDR_ValidateFileCrc().
					// This results in more file i/o operations and may erase some of the gains.
					// For example:
					//		RFD_DoFragmentMarkStorageTransactionComplete(fragmentPathBuf, FALSE, FALSE);
					//		RFD_ReleaseMutex(clientInfo->clientMutex);
					//		status = RFDR_ValidateFileCrc(fragmentPathBuf, fileProcessInfo);
					//		RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);
					//		RFD_DoFragmentMarkStorageTransactionStart(fragmentPathBuf, FALSE, FALSE);
					//////////////////////

					// Construct the process directory path name.
					RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
					RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME);

					RFD_DPrint( TEXT("RFDR Processor: Validate-CRC Begin, Client Index %d, fileIndex %d\n"),clientIndex, fileToProcessIndex);

					status = RFDR_ValidateFileCrc(fileNameBuf, fileProcessInfo);

					if(status != RFD_STATUS_OK) {
						// Reset the FileSet to attempt to recover from error.
						// This resets all files of the file set, not just the single file
						// detected as failed here.
						//
						// Do Not mark the transaction complete (i.e. RFD_MarkStorageTransactionComplete()).
						// If the transaction was to be marked complete, there is the danger that
						// a power-down could occur right after this but before the FileSet reset.
						// On the next powerup, it would appear that the storage is in a normal configuration
						// in the 'prepared for decode' state; however, the storage might not correpond to this
						// state depending on where the error occured in the decode operation.
						// By Not marking the transaction complete, in the case of an ill timed power down,
						// a reset of the fileSet will be forced on the next powerup (for full power-loss resilience
						// build, the incomplete transaction will be detected and the backup will be restored to
						// achieve the valid previous storage).

						RFD_DPrint( TEXT("RFDR Processor: RFDR_ValidateFileCrc error, status %d, Client Index %d, fileIndex %d\n"),status, clientIndex, fileToProcessIndex);
						RFD_ReleaseMutex(clientInfo->clientMutex);
						// force completion of next action.
						forceNextActionCompletion = TRUE;
						// Next action is to initialize the RFD Process for the current client instance
						// using the default fileSet.
						actionRequest = RFD_PROC_ACTION__RESET_FSET_MDATA;
						break;
					}

					RFD_DPrint( TEXT("RFDR Processor: Validate-CRC Completed, Client Index %d, fileIndex %d\n"),clientIndex, fileToProcessIndex);

					//////////////////////
					// update state to decode complete (with mutex acquired)
					//////////////////////
					fileProcessInfo->state = RFDR_STATE_DECODE_COMPLETE;

					//////////////////////
					// Write the Process State file
					//////////////////////

					// Construct directory path to Process State file.
					RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
					RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process directory

					status = RFD_FileProcessStateWriteToStorage(fileNameBuf, &fileProcessInfo->state);

					if(status != RFD_STATUS_OK) {
						// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
						// Do Not mark transaction complete (i.e. RFD_MarkStorageTransactionComplete()
						// for error conditions (in case powerdown right after marking transaction complete).
						RFD_ReleaseMutex(clientInfo->clientMutex);
						RFDR_SignalErrorExit(errorExitEvent);
						isThreadExitRequested = TRUE;
						break;
					}

					//////////////////////
					// Now that decoding is complete, Clear collector update flag in case
					// another collection event was signalled during the last decode,
					// before fileProcessInfo->state set to RFDR_STATE_DECODE_COMPLETE.
					// e.g. enough blocks were collected for another decode during the last decode.
					//////////////////////
					collectorInfo->fileProcessCollectorUpdate[fileToProcessIndex] = FALSE;

					//////////////////////
					// Now that decoding is complete, cleanup storage related to decoding,
					// for this file instance.
					// Note: this operation could be postponed until after all files of the file set
					// are complete if this operation takes too long i.e. collection of blocks of
					// other files of the file set are blocked here because mutex is aquired here.
					//////////////////////
					status = RFD_FileProcessInfoCleanupStateSpecificStorage( fragmentPathBuf, fileProcessInfo,
																			 RFDR_STATE_DECODE_COMPLETE);
					if(status != RFD_STATUS_OK) {
						// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
						// Do Not mark transaction complete (i.e. RFD_MarkStorageTransactionComplete()
						// for error conditions (in case powerdown right after marking transaction complete).
						RFD_ReleaseMutex(clientInfo->clientMutex);
						RFDR_SignalErrorExit(errorExitEvent);
						isThreadExitRequested = TRUE;
						break;
					}

					//////////////////////
					// Now that decoding is complete, cleanup dynamic allocations,
					// for this file instance.
					//////////////////////
					status = RFDR_DecodeInfoCleanupDyn( fileProcessInfo );
					if(status != RFD_STATUS_OK) {
						// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
						// Do Not mark transaction complete (i.e. RFD_MarkStorageTransactionComplete()
						// for error conditions (in case powerdown right after marking transaction complete).
						RFD_ReleaseMutex(clientInfo->clientMutex);
						RFDR_SignalErrorExit(errorExitEvent);
						isThreadExitRequested = TRUE;
						break;
					}

					// Check if all files are in the decode complete state.
					// If yes, then set flag to indicate ready for defrag.
					isFileSetComplete = RFDR_IsFileSetInDecodeCompleteState(clientInfo);

					if(isFileSetComplete) {
						//////////////
						// Decode complete for all file instances
						//////////////

						//////////////
						// Set collector isFileSetCollectionComplete status to True.
						// Collector can avoid any further parsing of block messages
						// for this file set.
						//////////////
						collectorInfo->isFileSetCollectionComplete = TRUE;

						//////////////
						// next step is to defragment.
						//////////////
						actionRequest = RFD_PROC_ACTION__DEFRAGMENT;
					}
					else {
						// Decode Complete for this file process instance,
						// but not for all file process instances.
						actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;
					}
				}
				else {
					// Decode Not complete for this file instance (!isDecodeComplete))

					//////////////////////
					// update state back to collecting (with mutex acquired)
					//////////////////////
					fileProcessInfo->state = RFDR_STATE_COLLECTING;

					//////////////////////
					// Write the Process State file
					//////////////////////

					// Construct directory path to Process State file.
					RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, fragmentPathBuf);
					RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process directory

					status = RFD_FileProcessStateWriteToStorage(fileNameBuf, &fileProcessInfo->state);

					if(status != RFD_STATUS_OK) {
						// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
						// Do Not mark transaction complete (i.e. RFD_MarkStorageTransactionComplete())
						// for error conditions (in case powerdown right after marking transaction complete).
						RFD_ReleaseMutex(clientInfo->clientMutex);
						RFDR_SignalErrorExit(errorExitEvent);
						isThreadExitRequested = TRUE;
						break;
					}

					//////////////////////
					// Check if decode was performed without maximum allowable number of
					// collected blocks.
					// Ensure that this 'maximum allowable number of collecgted blocks' status
					// is sampled before the decode incase addional blocks were collected
					// during the decode.
					//////////////////////
					if(!isDecodeWithMaxAllowableNumBlocksCollected) {

						// Maximum collected blocks Not exceeded,
						// continue decoding process.

						////////////////////
						// Check if Block Collection is sufficient for decode.
						// This is normally signalled by the Collector thread. However
						// there is a possibility that Collector will not signal if collector
						// does not collect any further blocks and
						// because RFDR_PostDecodeOps() call in this thread updates
						// block collection threshold i.e. the last block collected did
						// not exceed collection threshold before RFDR_PostDecodeOps() update,
						// but would have exceeded threshold after RFDR_PostDecodeOps() update.
						////////////////////
						if(RFDR_IsBlockCollectionStateReadyForDecode(fileProcessInfo)) {
							collectorInfo->fileProcessCollectorUpdate[fileToProcessIndex] = TRUE;
							RFD_SetEvent(clientInfo->clientEvent);
						}
						actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;
					}
					else {
						//////////////////////
						// Maximum allowable number of blocks has been collected.
						// This should be an extremely low probability occurence.
						// Cleanup and re-inialize this file instance to re-attempt decoding process from
						// initial state.
						//////////////////////

						RFD_DPrint( TEXT("RFDR Processor: Maxumum number of blocks collected, and decode is not complete for File Index %d, Client Index %d\n"), fileToProcessIndex, clientIndex);
						RFD_DPrint( TEXT("                Cleanup and re-inialize this file instance to re-start decoding process\n"));

						//////////////////////
						// Cleanup storage related to decoding,
						// for this file instance.
						//////////////////////
						status = RFD_FileProcessInfoCleanupStateSpecificStorage( fragmentPathBuf, fileProcessInfo,
																				 RFDR_STATE_PREPARED_FOR_DECODE /* one of the decoding states */);
						//////////////////////
						// Cleanup dynamic allocations,
						// for this file instance.
						//////////////////////
						if(status == RFD_STATUS_OK) {
							status = RFDR_FileProcessInfoCleanupDyn( fileProcessInfo );
						}

						//////////////////////
						// Re-Create and re-initialize data for this file instance.
						//////////////////////
						if(status == RFD_STATUS_OK) {
							status = RFD_FileProcessInfoCreateWithDefInit( &clientInfo->fileSetInfo,
																		   fileToProcessIndex,
																		   &fileProcessInfo);
							// Must copy newly allocated fileProcessInfo pointer back to main structure
							// where the pointer is maintained.
							clientInfo->fileProcessInfo[fileToProcessIndex] = fileProcessInfo;
						}

						//////////////////////
						// Initialize Storage for this file process instance.
						//////////////////////
						if(status == RFD_STATUS_OK) {
							status = RFD_FileProcessInfoWriteToStorage( fragmentPathBuf, fileProcessInfo);
						}

						//////////////////////
						// Handle any Error in the above re-initialization sequence..
						//////////////////////
						if(status != RFD_STATUS_OK) {
							// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
							// Do Not mark transaction complete (i.e. RFD_MarkStorageTransactionComplete()
							// for error conditions (in case powerdown right after marking transaction complete).
							RFD_ReleaseMutex(clientInfo->clientMutex);
							RFDR_SignalErrorExit(errorExitEvent);
							isThreadExitRequested = TRUE;
							break;
						}
						actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;
					}
				}

			/////////
			// Mark completion of storage transaction for the overall fragment.
			// Note: Mutex must be aquired before calling this function.
			// fragmentPathBuf string must represent the file instance (fragment) directory of the file set.
			/////////
			}
			RFD_DoFragmentMarkStorageTransactionComplete(fragmentPathBuf, FALSE);

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
			//////
			// Backup Storage after above RFD_DoFragmentMarkStorageTransactionComplete().
			// A decoding action is complete (chunk decoded, overall decode completed or decoding failed).
			// RFD_DoFragmentBackup() is called to backup for both the fragment's underlying
			// directories (process and collect).
			//////

#ifdef RFDR_ENABLE_STORAGE_OPTIM_1
			if(fileProcessInfo->state == RFDR_STATE_COLLECTING) {
				// Updating the process dir is safely completed and no further updates
				// will be done until the next decode (prepare-for-decode) operation.
				// Postpone the backup of the process directory until prepare-for-decode operation.
				// Instead, clear the process backup directory as storage space optimization
				// (only one fragment at a time will utilize its process dir backup storage for decode processing).
				status = RFD_DoFragmentBackup(fragmentPathBuf, TRUE /* do cleanup for process dir */);
			}
			else {
				status = RFD_DoFragmentBackup(fragmentPathBuf, FALSE /* do regular backup for process dir */);
			}
#else
			status = RFD_DoFragmentBackup(fragmentPathBuf, FALSE);
#endif
			if(status != RFD_STATUS_OK) {
				// Backup failed.
				RFD_ReleaseMutex(clientInfo->clientMutex);
				RFDR_SignalErrorExit(errorExitEvent);
				isThreadExitRequested = TRUE;
				break;
			}
#endif
			RFD_ReleaseMutex(clientInfo->clientMutex);
			// actionRequest already set.
			break;

		///////////////////////////////////////////////////////////////////////
		// Defragment Action
		///////////////////////////////////////////////////////////////////////
		case RFD_PROC_ACTION__DEFRAGMENT:

			RFD_DPrint( TEXT("RFDR Processor: Defragment Process Begin, Client Index %d\n"), clientIndex);

			//////////////
			// Acquire Mutex
			//////////////
			RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);

			status = RFD_DefragmentFileSet(clientInfo);

			if(status != RFD_STATUS_OK) {
				// Reset the FileSet to attempt to recover from error.
				// This resets all files of the file set, not just the single file
				// detected as failed here.
				RFD_DPrint( TEXT("RFDR Processor: RFD_DefragmentFileSet error, status %d, Client Index %d\n"),status, clientIndex);
				RFD_ReleaseMutex(clientInfo->clientMutex);
				// force completion of next action.
				forceNextActionCompletion = TRUE;
				// Next action is to initialize the RFD Process for the current client instance
				// using the default fileSet.
				actionRequest = RFD_PROC_ACTION__RESET_FSET_MDATA;
				break;
			}

			RFD_DPrint( TEXT("RFDR Processor: Defragment Process Completed, Client Index %d\n"), clientIndex);

			//////////////
			// Release Mutex
			//////////////
			RFD_ReleaseMutex(clientInfo->clientMutex);
			actionRequest = RFD_PROC_ACTION__DECOMPRESS;

			break;

		///////////////////////////////////////////////////////////////////////
		// Decompress Action
		///////////////////////////////////////////////////////////////////////
		case RFD_PROC_ACTION__DECOMPRESS:

			RFD_DPrint( TEXT("RFDR Processor: Decompress-Action Begin, Client Index %d\n"), clientIndex);

			//////////////
			// This 'Decompress-action' sequence is compatible with or without
			// RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE defined.
			//////////////

			// Construct the path string for the diretorory containing the decompress input file, output
			// file and the state variable file to be updated.
			// The proces sub-dir of the first file instance (fragment) directory (index 0) serves as this directory.

			RFDR_ConstructFragmentDirPathName(fragmentPathBuf, RFD_MAX_PATH_LEN,
											  clientInfo,
											  clientInfo->fileSetInfo.fileInfo[RFDR_FILE_INSTANCE_INDEX_0].id);
			RFD_STRCAT_S(fragmentPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fragmentPathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir

			// Note: fragmentPathBuf contains the path to the process sub-dir, not the
			// top level fragment directory as the variable name implies.

			//////////////
			// Acquire Mutex
			//////////////
			RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);

			//////////////
			// Mark storage transaction start. Mutex must be aquired for this.
			//////////////
			RFD_MarkStorageTransactionStart(fragmentPathBuf);
			{

				//////////////
				// Release Mutex before long running decompress task.
				//////////////
				RFD_ReleaseMutex(clientInfo->clientMutex);

				//////////////
				// Decompress
				//////////////
				status = RFDR_DecompressFileSet(clientInfo, hRfdr->rfdrProcessorInfo->exitRequestFlagPtr);

				//////////////
				// Re-Acquire Mutex
				//////////////
				RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);

				if(status == RFD_STATUS_EARLY_EXIT_ON_REQUEST) {
					RFD_DPrint( TEXT("RFDR Processor: Exit Requested during Decompress, Client Index %d\n"), clientIndex);
					// The Decompressor has exited early due to polling the exitRequestFlag.
					// The corresponding event, exitRequestEvent, should still be set; however,
					// re-asserting the SetEvent() for clarity is done here.
					RFD_SetEvent(hRfdr->rfdrProcessorInfo->exitRequestEvent);
					// Cancel the Storage Transaction for the FileSet.
					// In this case of early exit on request,
					// the decompressor has Not modified any files in storage. The state is still 'defragment complete',
					// and the decompressor should resume from this state on the next powerup.
	#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
					// Comment for Power-Loss Resilient Option:
					// Cancel the storage transaction instead of the Completing the storage transaction
					// so that the 'sync count' (total number of transactions counted) is decremented.
					// There is not a corresponding Backup transaction operation here, so the Cancel operation keeps
					// the two transaction instances in-sync. Therefore, the base directory content
					// will Not be unecessarily copied to backup on powerup due to out-of-sync detection.
	#endif
					RFD_CancelStorageTransaction(fragmentPathBuf);

					RFD_ReleaseMutex(clientInfo->clientMutex);
					// The exitRequestEvent is tested on each action request loop.
					// actionRequest is set to some default anyway.
					actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;
					break;
				}
				else if(status != RFD_STATUS_OK) {
					RFD_DPrint( TEXT("RFDR Processor: RFDR Decompress error, status %d, Client Index %d\n"),status, clientIndex);
					// Reset the FileSet to attempt to recover from error.
					// This resets all files of the file set, not just the single file
					// detected as failed here.
					//
					// Do Not Mark Transaction Complete, incase of powerdown right after (error would not be handled on powerup).
					// Instead, Transaction incomplete will be detected on powerup, forcing a reset of the FileSet (for non-full-power-down-resilience option).

					RFD_ReleaseMutex(clientInfo->clientMutex);
					// force completion of next action.
					forceNextActionCompletion = TRUE;
					// Next action is to initialize the RFD Process for the current client instance
					// using the default fileSet.
					actionRequest = RFD_PROC_ACTION__RESET_FSET_MDATA;
					break;
				}

				//////////////////////
				// Update state of the file set to decompress complete.
				// The state variable (and corresponding state file) for the first file instance represents
				// the state of the overall file set.
				//////////////////////

				clientInfo->fileProcessInfo[RFDR_FILE_INSTANCE_INDEX_0]->state = RFDR_STATE_DECOMPRESS_COMPLETE;

				//////////////////////
				// Write the Process State file
				//////////////////////

				status = RFD_FileProcessStateWriteToStorage(fragmentPathBuf, &clientInfo->fileProcessInfo[RFDR_FILE_INSTANCE_INDEX_0]->state);

				if(status != RFD_STATUS_OK) {
					// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
					//
					// Do Not Mark Transaction Complete, incase of powerdown right after (error would not be handled on powerup).
					// Instead, Transaction incomplete will be detected on powerup, forcing a reset of the FileSet.

					RFD_ReleaseMutex(clientInfo->clientMutex);
					RFDR_SignalErrorExit(errorExitEvent);
					isThreadExitRequested = TRUE;
					break; // break from for loop
				}

			/////////////////
			// Mark Storage Transaction Complete for the FileSet, (which is represented by
			// the transaction for first file instance).
			/////////////////
			}
			RFD_MarkStorageTransactionComplete(fragmentPathBuf, TRUE);

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
			/////////////////
			// Backup Store the FileSet directory.
			/////////////////

			status = RFD_BackupDirectoryContents(fragmentPathBuf);

			if(status != RFD_STATUS_OK) {
				RFD_ReleaseMutex(clientInfo->clientMutex);
				RFDR_SignalErrorExit(errorExitEvent);
				isThreadExitRequested = TRUE;
				break;
			}
#endif
			////////////////////////
			// Set event to the consumer thread to indicate that this file set output data file
			// is complete. The consumer should detect the event, move this file, then
			// update state from decompress complete to all complete.
			////////////////////////
			RFD_DPrint( TEXT("RFDR Processor: Setting FileSet Ready Event to consumer, Client Index %d\n"), clientIndex);

			RFD_SetEvent(consumerInfo->fileSetReadyEvent);

			/*
			 * Note: in case the consumer thread does not copy output before shutdown occurs,
			 *       the processor thread should check for file set in the RFDR_STATE_DECOMPRESS_COMPLETE state
			 *       upon powerup, and signal client output again.
			 */

			//////////////
			// Release Mutex
			//////////////
			RFD_ReleaseMutex(clientInfo->clientMutex);

			RFD_DPrint( TEXT("RFDR Processor: Decompress-Action Completed, Client Index %d\n"), clientIndex);

			// Decompress complete for this file set.
			actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;
			break;

		///////////////////////////////////////////////////////////////////////
		// Reset FileSet Metadata Action
		///////////////////////////////////////////////////////////////////////
		case RFD_PROC_ACTION__RESET_FSET_MDATA:

			/////////////////////////
			// File Set Reset Request
			// (from Consumer or localy in this fcn. due to error detection).
			////////////////////////
			RFD_DPrint( TEXT("RFDR Processor: Begin FileSet-Reset-Action, Client Index %d\n"),clientIndex);

			RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);
			{
				// Initialize default content of file set structure.
				RFD_FileSetDefaultInit(&clientInfo->fileSetInfo);
			}
			RFD_ReleaseMutex(clientInfo->clientMutex);

			// force completion of next action.
			forceNextActionCompletion = TRUE;

			// Next action is to initialize the RFD Process for the current client instance
			// using the default fileSet.
			actionRequest = RFD_PROC_ACTION__INITIALIZE_PROCESS;
			break;

		///////////////////////////////////////////////////////////////////////
		// Initialize Processing Action
		///////////////////////////////////////////////////////////////////////
		case RFD_PROC_ACTION__INITIALIZE_PROCESS:

			RFD_DPrint( TEXT("RFDR Processor: FileSet-Initialization-Action Begin, Client Index %d\n"),clientIndex);

            // Print FileSet (Metadata) Info.
            RFD_PrintFileSetInfo( &clientInfo->fileSetInfo );

			// Construct client path name string.
			RFDR_ConstructClientDirPathName(clientPathBuf, RFD_MAX_PATH_LEN, clientInfo);

			RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);
			{
				// Cleanup all dynamic allocations.
				RFDR_FileProcessInfoArrayCleanupDyn( clientInfo );;

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
				/////////
				// Mark start of storage transaction for the client *Backup* direcory.
				//
				// This is done before marking the transaction start for the base client directory.
				// This avoids the unwanted restore of the base directory in case of powerdown during
				// this initialization process that involves deletion and creation of
				// file instance directories and files. The file instance directories have
				// their own transaction marker files and are not 'synchronized' with the client directory.
				// Furthermore, the cleanup process does not do transaction protected deletion
				// of the file instance directories.
				// If a power-down occurs during this process, it's best to ensure that
				// the overall file set is detected as invalid, thus forcing another full cleanup and client re-init,
				// in the power-up initialization.
				// This is ensured by marking the client backup directory transaction as in progress (started) first,
				// followed by the corresponding base client directory. In this way, on power-up, the base client directory
				// will not be restored from the backup, and the base directory will be determined as invalid, causing
				// a 'reset' of the client and any underlying file instances (fragments).
				/////////

				// construct path string for client backup directory
				RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, clientPathBuf);
				RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
				RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_BACKUP_DIR_NAME);

				RFD_MarkStorageTransactionStart(fileNameBuf);
#endif
				/////////
				// Mark start of storage transaction for the client direcory.
				/////////

				RFD_MarkStorageTransactionStart(clientPathBuf);
				{
					//////////////////
					// Cleanup client directory
					//////////////////
					RFD_ClientCleanupStorage(clientInfo);

					// RFD_ClientCleanupStorage() will remove the Storage Transaction File,
					// So call RFD_InitializeStorageTransaction() then
					// RFD_MarkStorageTransactionStart() again.
					//
					// Note: if a power-down occurs here, after RFD_InitializeStorageTransaction() but
					// before RFD_MarkStorageTransactionStart(), then the transaction will NOT be
					// detected as incomplete on the following power-up. However, the FileSet file
					// (RFDR_FSET_FNAME) has not been created yet. Therefore, the client will be detected
					// as invalid due to error in reading this file.

					RFD_InitializeStorageTransaction(clientPathBuf, 0);
					RFD_MarkStorageTransactionStart(clientPathBuf);

					//////////////////
					// Create the file processing directories as indicated by
					// the FileSet structure, for this client.
					//////////////////
					RFD_ClientCreateStorage(clientInfo);

					//////////////////
					// write the new File Set Metadata file
					//////////////////

					// Create file path string for the File Set filename
					RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, clientPathBuf);
					RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFDR_FSET_FNAME);

					// Write the fileset file
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
					// with File Sync enabled, to sync file to disk before Completeing the transaction.
#endif
					status = RFD_WriteFile(fileNameBuf, &clientInfo->fileSetInfo,
													  sizeof(RFD_FSET_MDATA_STRUCT),
													  IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);
					if(status != RFD_STATUS_OK) {
						// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
						RFD_ReleaseMutex(clientInfo->clientMutex);
						RFDR_SignalErrorExit(errorExitEvent);
						isThreadExitRequested = TRUE;
						break;
					}

					//////////////////
					// Reallocate and initialize all dynamic allocations.
					//////////////////
					status = RFD_FileProcessInfoArrayCreateDynWithDefInit( clientInfo );

					if(status != RFD_STATUS_OK) {
						RFD_DPrint( TEXT("Error in RFD_FileProcessInfoArrayCreateDynWithDefInit: status = %d \n"),status);

						// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
						RFD_ReleaseMutex(clientInfo->clientMutex);
						RFDR_SignalErrorExit(errorExitEvent);
						isThreadExitRequested = TRUE;
						break;
					}

					//////////////////
					// Write the initialized Process Info Array to Storage.
					//////////////////
					status = RFD_FileProcessInfoArrayWriteToStorage( clientInfo );

					if(status != RFD_STATUS_OK) {
						RFD_DPrint( TEXT("Error in RFD_FileProcessInfoArrayWriteToStorage: status = %d \n"),status);

						// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
						RFD_ReleaseMutex(clientInfo->clientMutex);
						RFDR_SignalErrorExit(errorExitEvent);
						isThreadExitRequested = TRUE;
						break;
					}

				/////////
				// Mark end of storage transaction.
				// Do this after RFD_FileProcessInfoArrayWriteToStorage() as the process
				// of updating the fset file and initialializing each individual file directory
				// of the file set is one coupled transaction.
				/////////
				}
				RFD_MarkStorageTransactionComplete(clientPathBuf, TRUE);

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
				///////////
				// Backup the client directory level files just written for this
				// file client instance to the corresponding Backup Storage directory.
				// Do this after the RFD_MarkStorageTransactionComplete() for the
				// corresponding directory.
				//
				// Note: the backup directory was marked for Transaction Start above.
				// The following RFD_BackupDirectoryContents() call serves as the matching
				// RFD_MarkStorageTransactionComplete(). Therefore another RFD_MarkStorageTransactionComplete()
				// should Not be called.
				///////////

				// Do the backup.
				status = RFD_BackupDirectoryContents(clientPathBuf);
				if(status != RFD_STATUS_OK) {
					RFD_ReleaseMutex(clientInfo->clientMutex);
					RFDR_SignalErrorExit(errorExitEvent);
					isThreadExitRequested = TRUE;
					break;
				}
#endif
				//////////////////
				// Clear any fileProcess Updates that may also have been flaged
				// Since we've reinitialized for the new file set metadata
				// i.e. any block updates have been cleaned out.
				//////////////////
				//for(i=0;i<clientInfo->fileSetInfo.numFiles;i++) // note: could go up to RFD_FSET_MAX_NUM_FILES.
				for(i=0;i<RFD_FSET_MAX_NUM_FILES;i++) {
					collectorInfo->fileProcessCollectorUpdate[i] = FALSE;
				}

				//////////////
				// Set collector isFileSetCollectionComplete status to False, unless number of files is file set is 0.
				// Collector must parse block messages when isFileSetCollectionComplete is False.
				//////////////
				if(clientInfo->fileSetInfo.numFiles > 0) {
					collectorInfo->isFileSetCollectionComplete = FALSE;
				}
				else {
					collectorInfo->isFileSetCollectionComplete = TRUE;
				}

				//////////////
				// Check for FileSet Reset Request
				//////////////
                if( clientInfo->fileSetInfo.numFiles == 0 &&
                    consumerInfo->fileSetResetRequest == TRUE) {

                    //////////////
                    // The file set has been reset (.numFiles == 0) and the reset request is flaged
                    // so we are here due to a reset request.
                    // Clear the File Set Reset Request
                    // - Clear the reset request flag
                    // - Delete the file in consumer dir that is used to mark the pending reset request
                    //   in a power-loss-resilient manner.
                    //
                    //   The file delete is done with clientInfo->clientMutex aquired
                    //   to avoid simultaneous operations on the same file, e.g. another file set
                    //   reset could be requested ansynchronously from another thread.
                    //
                    //   The file delete is done here after the marking end of storage transaction
                    //   for this client for this reset/process initialization action,
                    //   so we are sure the reset has fully completed before clearing the reset request.
                    //   This fset reset req file is in the consumer directory (not under the same
                    //   storage transaction mechanisms as the client directory).
                    //////////////

                    consumerInfo->fileSetResetRequest = FALSE;

                    // Construct the reset-request-pending file name
                    RFD_ConsumerConstructFSetResetReqFilePathName(consumerInfo, fileNameBuf, RFD_MAX_PATH_LEN);

                    // Delete the reset-request-pending file to fully clear the reset request.
                    // No sync of the containing directory is needed afterwards.
                    RFD_DELETE_FILE(fileNameBuf);

			        RFD_DPrint( TEXT("RFDR Processor: Cleared the FileSet Reset Request, deleted file %s, Client Index %d\n"),
                        fileNameBuf,
                        clientIndex);
                }

				actionRequest = RFD_PROC_ACTION__WAIT_FOR_EVENT;
			}
			RFD_ReleaseMutex(clientInfo->clientMutex);
			break;

		///////////////////////////////////////////////////////////////////////
		// Default
		///////////////////////////////////////////////////////////////////////
		default:
			// Unexpected Error if this condition is met.
            status = RFD_STATUS_ERROR_GENERAL;
			RFDR_SignalErrorExit(errorExitEvent);
			isThreadExitRequested = TRUE;
			break;
		}


		///////////////////////////////////////////////////////////////////////
		// Check run loop exit conditions
		///////////////////////////////////////////////////////////////////////
        if( isThreadExitRequested ||
            (doRunUntilIdle && isIdle) ) {
            // break out of run loop.
            break;
        }

    } while(TRUE); // run until forced exit.

    ////////////////////////
    // Save vars for next run
    ////////////////////////
	ctl->clientIndex                    = clientIndex;
	ctl->clientInfo                     = clientInfo;
	ctl->collectorInfo                  = collectorInfo;
	ctl->consumerInfo                   = consumerInfo;
	ctl->fileSetInfo                    = fileSetInfo;
	ctl->fileProcessInfo                = fileProcessInfo;
	ctl->fileToProcessIndex             = fileToProcessIndex;
	ctl->forceNextActionCompletion      = forceNextActionCompletion;
	ctl->actionRequest                  = actionRequest;
#ifdef ENABLE_ACCUMULATED_DECODER_TEST_STATS
	ctl->accumDecodeStats               = accumDecodeStats;
#endif

    /////////////////////////////////
    // Set output parameters
    /////////////////////////////////

    *isThreadExitRequestedPtr = isThreadExitRequested;
    *statusPtr = status;

    return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////////////////////
DWORD RFD_PLATFORM_API RFDR_Processor( LPVOID lpParam )
{
	RFD_PROCESSOR_THREAD_DATA_HANDLE hProcessor;
    RFD_STATUS status;
    BOOL isThreadExitRequested;

	hProcessor = (RFD_PROCESSOR_THREAD_DATA_HANDLE) lpParam;

    ////////////////////////////////
    // Open (create and initialize)
    ////////////////////////////////
    RFDR_ProcessorOpenLL(
        hProcessor,
        &status,
        &isThreadExitRequested );

    ////////////////////////////////
    // Run loop
    ////////////////////////////////
    while(!isThreadExitRequested) {

        RFDR_ProcessorRunLL(
            hProcessor,
            &status,
            &isThreadExitRequested,
            TRUE );
    }
    // Run loop exit

    ////////////////////////////////
    // Close (cleanup)
    ////////////////////////////////
    RFDR_ProcessorCloseLL(
        hProcessor,
        &status );

    // Thread exit
    return 0;
}





