////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_receiver\rfdr_msg_collector.c
///
/// @brief	rfdr message collector class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2012 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd_receiver.h"
#include "rfd_msg_collector.h"

///////////////////////////////////////////////////////////////////////////////////////////////////
/// @addtogroup RfdrInputApi RFDR Input API
/// @ingroup RFDR_API
///////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrInputApi
///
/// @brief
/// Determine if a Message is an RFD type Message that is collectable by an RFD Message Collector
/// (RFDR Collector).
///
/// The Message Producer calls this function to determine if a newly produced message should be
/// collected by the RFDR Collector identified by the \e collectorHandle handle. If this function
/// returns True, then the \e RFD_CollectorPutMessage() function should then be called to
/// transfer this message to the RFDR Collector. This function may indicate that this message is
/// e not collectable if the message's Protocol Version Number or Carousel ID values do not match
/// any of the valid values of this RFDR Collector. This function may also indicate this message
/// is not collectable if the message is targeted to this RFDR Collector, but the RFDR is not in
/// a \e collecting state for this type of message. For example, the RFD File processing is in a
/// completed state so no further Block Messages need to be collected for this File.
///
/// @param  collectorHandle Handle of the Collector Thread Data object.
/// @param  messageBuf      Buffer containing RFD message data.
/// @param  messageSize     Size of the RFD message.
///
/// @return
/// RFD_STATUS_OK if the message is an RFD message to be collected by this RFDR Collector Thread,
/// RFD_STATUS_FSET_COMPLETE if the message is part of file set that is already complete, Some
/// other error if there was a problem with this message.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CollectorIsMessageCollectable(RFD_COLLECTOR_THREAD_DATA_HANDLE collectorHandle, UCHAR * messageBuf, UINT32 messageSize)
{

    RFD_STATUS status;

    status = RFD_CollectorIsMessageCollectableExt(
                    collectorHandle,
                    messageBuf,
                    messageSize,
                    FALSE, // dont use Blk Msg File Id for filtering (for legacy behavior)
                    FALSE, // dont use following DMI parameter for filtering (for legacy behavior)
                    RFD_DMI_UNUSED_VALUE );

    if(status == RFD_STATUS_CAROUSEL_ID_NOT_MATCHED) {
        // for backwards compatibility with original RFD_CollectorIsMessageCollectable() behavior.
        status = RFD_STATUS_ERROR_GENERAL;
    }

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrInputApi
///
/// @brief
/// Determine if a Message is an RFD type Message that is collectable by an RFD Message Collector
/// (RFDR Collector). This function extends the capability of the
/// RFD_CollectorIsMessageCollectable() function by optionally including Block File IDs and/or
/// the Block DMI in the "is collectable" determination.
///
/// The Message Producer calls this function to determine if a newly produced message should be
/// collected by the RFDR Collector identified by the \e collectorHandle handle. If this function
/// returns RFD_STATUS_OK, then the RFD_CollectorPutMessage() function should then be called to
/// transfer this message to the RFDR Collector. This function may indicate that this message is
/// e not collectable if the message's Protocol Version Number or Carousel ID values do not match
/// any of the valid values of this RFDR Collector. This function may also indicate this message
/// is not collectable if the message is targeted to this RFDR Collector, but the RFDR is not in
/// a \e collecting state for this type of message. For example, the RFD File processing is in a
/// completed state so no further Block Messages need to be collected for this File.
///
/// The caller may also specify the use of Block Message File IDs for the "Is Collectable"
/// determination. This is typically done for Data Services supporting multiple Update Files per
/// Carousel. In this case, the File IDs are used to determine which of possibly serveral RFD
/// Clients (supporting that Data Service) to route messages to (i.e. route the message to the
/// RFD Client that reports "Is Collectable").
///
/// The caller may also specify the use of Block Message DMI filtering for the "Is Collectable"
/// decision. This is in support of Data Services that use the Block DMI field of Update File
/// Metadata to specify the DMI source from which to obtain Block Messages for a corresponding
/// Update File. When this type of filtering is enabled, the caller must also set the blockMsgDmi
/// parameter to DMI from which this (Block) message was obtained.
///
/// @param  collectorHandle             Handle of the Collector Thread Data object.
/// @param  messageBuf                  Buffer containing RFD message data.
/// @param  messageSize                 Size of the RFD message.
/// @param  doBlockMsgFileIdFiltering   If TRUE, a Block type message will be determined "Is
///                                     Collectable" only if the File ID parameter of the Block
///                                     message matches one of the File IDs defined by the Update
///                                     File Metadata currently being processed by this
///                                     Collector. If FALSE, the Block message File Id parameter
///                                     is ignored in the "Is Collectable" decision.
/// @param  doBlockMsgDmiFiltering      If TRUE, a Block type message will be determined "Is
///                                     Collectable" only if the DMI source of the Block Message,
///                                     passed in parameter blockMsgDmi, matches the Block DMI
///                                     specified by the Update File Metadata currently being
///                                     processed by this Collector. If FALSE, the passed
///                                     parameter, blockMsgDmi, is ignored in the "Is
///                                     Collectable" decision.
/// @param  blockMsgDmi                 The DMI source of a passed in Block type Message. When
///                                     doBlockMsgDmiFiltering = TRUE, and the passed in message
///                                     is a Block type message, then, for the Block Message to
///                                     be "Is Collectable", this blockMsgDmi must match the
///                                     configured Block Message DMI specified by the FileSetInfo
///                                     (Update File Metadata) currently being processed by this
///                                     Collector. If the currently processed FileSetInfo (Update
///                                     File Metadata)
///                                     does not specify a Block Message DMI (i.e. if this
///                                     parameter is not used by corresponding SXM Data Service),
///                                     then the blockMsgDmi parameter is ignored. When
///                                     doBlockMsgDmiFiltering = FALSE, the blockMsgDmi parameter
///                                     is ignored and the caller should set blockMsgDmi value to
///                                     RFD_FSET_MDATA_BLK_MSG_DMI_DEFAULT_VALUE.
///
/// @return
/// RFD_STATUS_OK if the message is an RFD message to be collected by this RFDR Collector Thread.
/// If the message is determined "not collectable", then one of the following codes is returned:
/// - RFD_STATUS_FSET_COMPLETE if the message is part of file set (Update File) that is already
/// complete.
/// - RFD_STATUS_BLK_MSG_FILE_ID_NOT_MATCHED if the message is a Block type message with a
///     Block Message File ID that does NOT match a File ID of the FileSetInfo (Update File
///     Metadata)
///     being processed by this Collector.
/// - RFD_STATUS_BLK_MSG_DMI_SOURCE_NOT_MATCHED if the message is a Block type message and
///     the passed blockMsgDmi does NOT match the configured Block Message DMI value of the
///     FileSetInfo (Update File Metadata) being processed by this Collector.
/// - RFD_STATUS_CAROUSEL_ID_NOT_MATCHED if the Carousel ID value of the message does not
///     match the expected Carousel ID value of an RFD message (Block message or Update File
///     Metadata message).
/// - RFD_STATUS_MSG_PROTOCOL_VERSION_NUM_OUT_OF_RANGE if the Protocol Version Number of the
/// message
///     does not match the protocol version configured for the RFD Receiver.
/// - Some other RFD_STATUS error status enumeration if there was any detected problem with this
/// message or
///     for any other reason the message is determined as "not collectable".
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CollectorIsMessageCollectableExt(
    RFD_COLLECTOR_THREAD_DATA_HANDLE collectorHandle,
    UCHAR * messageBuf,
    UINT32 messageSize,
    BOOL doBlockMsgFileIdFiltering,
    BOOL doBlockMsgDmiFiltering,
    UINT16 blockMsgDmi)
{
    RFD_STATUS status;
    RFD_MESSAGE_CONFIG_INFO * configInfo;
    UCHAR protocolVersionNo;
    UCHAR carouselId;

    // Note:
    // Some of the parameters that are read by this function may be concurrently written by the
    // RFD Collector Thread (configInfo->blockMsgCarouselId
    //                       collectorHandle->collectorInfo->isFileSetCollectionComplete).
    // However, the reads by this function are intentionally Not protected
    // by Mutex for performance reasons i.e. this function might be called for all messages
    // received by the message producer including messages that may targeted to to other modules.
    // This non-mutex-protected behavior is harmless. This function performs a 'pre-filter'
    // type functionality only. Complete mutex-protected parsing (PVN and CarouselId checking etc.)
    // is done by the RFDR Collector Thread once the message is put to the message queue by
    // a following RFD_CollectorPutMessage().
    // Therefore, in rare instances of invalid data read due to concurrent updates of thes variables,
    // this function may return True or False indication that is technically incorrect.
    // For the 'invalid true' case, as described, this behavior is harmless because the
    // RFDR Collector Thread(), after the transfer of the message, will perform correct parsing
    // and handling of the message.
    // The 'invalid false' case might occur only on 'the boundary' of a collection state that was
    // just being updated. A single message might then be 'thrown-out, which is once again, a
    // harmless action.

    // Initialize to OK (Is Collectable = True).
    status = RFD_STATUS_OK;

    do {

        configInfo = &collectorHandle->collectorInfo->messageInfo->configInfo;

        status = RFD_PeekMessageHeader(messageBuf,
            messageSize,
            configInfo->nBitsCarouselIdField,
            &protocolVersionNo,
            &carouselId);

        if(status != RFD_STATUS_OK) {
            break;
        }

        if(protocolVersionNo != configInfo->protocolVersionNo) {
            status = RFD_STATUS_MSG_PROTOCOL_VERSION_NUM_OUT_OF_RANGE;
            break;
        }

        if(carouselId == configInfo->fileSetMdataMsgCarouselId) {
            // Message is a FileSet Metadata Message
            status = RFD_STATUS_OK;
            break;
        }

        if(collectorHandle->collectorInfo->isFileSetCollectionComplete)
        {
            status = RFD_STATUS_FILE_SET_COMPLETE;
            break;
        }

        if(configInfo->blockMsgCarouselId != -1 &&
            carouselId == configInfo->blockMsgCarouselId) {

                // The FileSet processing is not complete (Block Data collection may still be in progress),
                // and the block message carousel Id has a valid value (this is a dynamically updated field),
                // and the carousel Id of this message matches the Block Data Message Carousel Id.

                if(doBlockMsgFileIdFiltering) {

                    // Block Message File ID Filtering is requested

                    UINT16 blkMsgFileId;
                    UCHAR numFiles;
                    UCHAR iFile;
                    BOOL foundFileId;
                    RFD_FILE_INFO_STRUCT * fileInfo;

                    status = RFD_PeekBlockMessageFileId(messageBuf,
                        messageSize,
                        configInfo,
                        &blkMsgFileId);

                    if(status != RFD_STATUS_OK) {
                        break;
                    }

                    fileInfo = collectorHandle->collectorInfo->fileSetInfoBuf.fileInfo;
                    foundFileId = FALSE;

                    // IMPORTANT! copy the collectorInfo->fileSetInfoBuf.numFiles to a separate variable and
                    // use this separate variable for both the limit check and the end case in processing "for loop".
                    // This is because access to the fileSetInfoBuf.numFiles here is not mutex protected
                    // and it is possible (but rare) that the collector thread may also modify this data
                    // concurrently. We don't want the variable to change while it's used in the for loop
                    // as it could possible contain some intermediate out of range value with a simultaneous
                    // read in context of this thread while the collector thread is writing it.
                    numFiles = collectorHandle->collectorInfo->fileSetInfoBuf.numFiles;

                    if(numFiles > RFD_FSET_MAX_NUM_FILES) {
                        status = RFD_STATUS_ERROR_GENERAL;
                    }

                    for(iFile=0;iFile<numFiles;iFile++) {

                        if(blkMsgFileId == fileInfo[iFile].id) {
                            foundFileId = TRUE;
                            // break from for loop.
                            break;
                        }
                    }

                    if(!foundFileId) {
                        // Did not find the Blk Msg File Id in the active File Set of this collector.
                        status = RFD_STATUS_BLK_MSG_FILE_ID_NOT_MATCHED;
                        break;
                    }
                }

                if(doBlockMsgDmiFiltering) {

                    // Block Message DMI Filtering is requested

                    UINT16 fileSetInfoBlockMsgDmi;
                    fileSetInfoBlockMsgDmi = collectorHandle->collectorInfo->fileSetInfoBuf.blockMsgDmi;

                    // Perform the DMI comparison only if the FileSetInfo DMI value is configured,
                    // i.e., is not the default/invalid value RFD_FSET_MDATA_BLK_MSG_DMI_DEFAULT_VALUE.
                    // The FileSetInfo DMI will NOT be configured for FileSetInfo (Update File Metadata)
                    // corresponding to Data Services that do NOT use the DMI field (i.e. for these
                    // Data Services, the Block DMI field is not present, and Block Messages are delivered
                    // on the same DMI as the Update File Metadata Messages or some other predetermined DMI
                    // that is defined by the Data Service.

                    if(fileSetInfoBlockMsgDmi != RFD_FSET_MDATA_BLK_MSG_DMI_DEFAULT_VALUE &&
                        fileSetInfoBlockMsgDmi != blockMsgDmi) {
                            status = RFD_STATUS_BLK_MSG_DMI_SOURCE_NOT_MATCHED;
                            break;
                    }
                }
        }
        else {
            // unknown carouseId or carouselId not initialized
            status = RFD_STATUS_CAROUSEL_ID_NOT_MATCHED;
            break;
        }
    }
    while(FALSE); // do once

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrInputApi
///
/// @brief  Put an RFD Message to an RFD Message Collector.
///
/// @param  collectorHandle Handle of an RFD Message Collector.
/// @param  messageBuf      Buffer containing RFD message data.
/// @param  messageSize     Size of the RFD message.
///
/// @return
/// - RFD_STATUS_OK if successful.
/// - RFD_STATUS_OK_BUF_FULL indicates message not written due to buffer full.
/// - RFD_STATUS_OK_MAX_SEMAPHORE_COUNT indicates message not written due to semaphore count at
/// maximum.
/// - RFD_STATUS_OK_MSG_OVERSIZED indicates message not written because message size is larger
/// than the internal buffer.
/// - Other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CollectorPutMessage(RFD_COLLECTOR_THREAD_DATA_HANDLE collectorHandle, UCHAR * messageBuf, UINT32 messageSize)
{
	RFD_STATUS status;

	status = RFD_MSG_QUEUE_PutMessage(collectorHandle->collectorInfo->messageQueue, messageBuf, messageSize);

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrInputApi
///
/// @brief
/// Get pointer to the specified RFD Message Collector's Default Message Configuration structure.
///
/// The Default Message Configuration structure contains configuration parameters necessary to
/// parse the RFD Update File Metadata message (e.g. field sizes and protocol/carousel id values,
/// etc.). Note: The Default Message Config structure cannot be used for parsing operations on
/// the RFD Block message because some config fields related to Block message parsing are
/// dynamically updated during RFD Message Collector run time (i.e., the Block Msg Carousel Id
/// parameter).
///
/// @param  collectorHandle             Handle of an RFD Message Collector.
/// @param  defaultMessageConfigInfo    (output) Pointer to the Default Message Config structure
///                                     on successful return.
///
/// @return RFD_STATUS_OK on success, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CollectorGetDefaultMessageConfig(
                RFD_COLLECTOR_THREAD_DATA_HANDLE collectorHandle,
                const RFD_MESSAGE_CONFIG_INFO ** defaultMessageConfigInfo)
{
    if(collectorHandle == NULL || defaultMessageConfigInfo == NULL) {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    // Copy the message collector's message config structure to the user's destination message config structure.
	*defaultMessageConfigInfo = &collectorHandle->collectorInfo->defaultMsgConfigInfo;

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrInputApi
///
/// @brief	Determine if an RFD message is an RFD Update File Metadata type message.
///
/// @param	messageConfigInfo           Pointer to an RFD Message Config structure, which is obtained
///                                     using RFD_CollectorGetDefaultMessageConfig()
/// @param	messageBuf		            Buffer containing RFD message data.
/// @param	messageSize		            Size of the RFD message.
/// @param	isUpdateFileMetadataMessage	(output) Pointer to BOOLEAN variable set to true by this
///                                     function if the message is an Update File Metadata message,
///                                     or set to false if not. Value is valid when return status
///                                     is RFD_STATUS_OK.
///
/// @return
/// RFD_STATUS_OK if the message was successfully parsed to determine if the message is or isn't an
/// RFD Update File Metadata Message.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CollectorIsUpdateFileMetadataMessage(
                const RFD_MESSAGE_CONFIG_INFO * messageConfigInfo,
                UCHAR * messageBuf,
                UINT32 messageSize,
                BOOL * isUpdateFileMetadataMessage)
{
	RFD_STATUS status = RFD_STATUS_ERROR_GENERAL;
	UCHAR protocolVersionNo;
	UCHAR carouselId;

    if(isUpdateFileMetadataMessage == NULL) {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }
    *isUpdateFileMetadataMessage = FALSE;

	status = RFD_PeekMessageHeader(messageBuf,
								   messageSize,
								   messageConfigInfo->nBitsCarouselIdField,
								   &protocolVersionNo,
								   &carouselId);

	if(status == RFD_STATUS_OK) {

		if(protocolVersionNo == messageConfigInfo->protocolVersionNo &&
           carouselId == messageConfigInfo->fileSetMdataMsgCarouselId ) {

				// Message is a FileSet Metadata Message
                *isUpdateFileMetadataMessage = TRUE;
				status = RFD_STATUS_OK;
        }
	}

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrInputApi
///
/// @brief  Parse an Update File Metadata message into the provided Update File Metadata structure.
///
/// @param  messageConfigInfo   Pointer to an RFD Message Config structure, which is obtained
///                             using RFD_CollectorGetDefaultMessageConfig()
/// @param  messageBuf          Buffer containing RFD message data.
/// @param  messageSize         Size of the RFD message.
/// @param  fileSetMetadata     (output) A pointer to the Update File (FileSet) Metadata
///                             structure which
///                             is written by this function upon successful parsing as indicated
///                             by
///                             return with RFD_STATUS_OK status. This structure is allocated by
///                             the caller.
///
/// @return
/// RFD_STATUS_OK if the message was successfully parsed to output the Update File (FileSet)
/// Metadata structure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CollectorParseUpdateFileMetadataMsg(
                const RFD_MESSAGE_CONFIG_INFO * messageConfigInfo,
                UCHAR * messageBuf,
                UINT32 messageSize,
                RFD_FSET_MDATA_STRUCT * fileSetMetadata)
{
	RFD_STATUS status = RFD_STATUS_ERROR_GENERAL;
    RFD_MSG_TYPE messageType;

    status = RFD_ParseMessageExt(
	                &messageType,
	                fileSetMetadata,
	                NULL,
	                messageBuf,
	                messageSize,
	                messageConfigInfo);

    if(status == RFD_STATUS_OK) {

        if(messageType != MSG_TYPE_FSET_MDATA) {

            // No error was reported, but this is not the expected message type.
            // report as error.
            status = RFD_STATUS_ERROR_INVALID_MESSAGE_TYPE;
        }
        // else status == RFD_STATUS_OK , return with this status value.
    }
    // else status indicates some error, return with this error status value.

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrInputApi
///
/// @brief	Get the RFDR Client State information describing the processing state of the
///         RFDR Client. When the processing state is not the reset state, then the Metadata
///         for the corresponding Update File being processed or that was last completed is also
///         output by the function.
///
/// @param	collectorHandle	    Handle of an RFD Message Collector for a corresponding RFDR Client.
/// @param	clientState         (output) A pointer to the RFDR Client processing state describing
///                             the processing state of this client. One of the RFDR_CLIENT_STATE
///                             enumeration values. The output value is only valid when function
///                             return status is RFD_STATUS_OK.
/// @param	updateFileMetadata  (output) A pointer to the Update File (FileSet) Metadata structure
///                             for the Update File currently being processed by this RFD Client.
///                             The structure is only valid when the correponding
///                             clientState is NOT RFDR_CLIENT_STATE_RESET and
///                             the return status value is RFD_STATUS_OK.
///                             This structure is allocated by the caller.
///
/// @return	RFD_STATUS_OK on success.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CollectorGetClientStateInfo(
                RFD_COLLECTOR_THREAD_DATA_HANDLE collectorHandle,
                RFDR_CLIENT_STATE * clientState,
                RFD_FSET_MDATA_STRUCT * updateFileMetadata)
{
	RFD_STATUS status = RFD_STATUS_ERROR_GENERAL;
	RFDR_CLIENT_INFO_STRUCT * clientInfo = collectorHandle->clientInfo;

	//////////////////////////////////////////
	// Acquire the Mutex to access shared data
	/////////////////////////////////////////

	status = RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);
	if(status != RFD_STATUS_OK) {
        return status;
	}

    // Get the FileSet State
    *clientState = RFDR_GetClientFileSetState(clientInfo);

    // Copy the FileSet Metadata.
    if(*clientState != RFDR_CLIENT_STATE_RESET) {
        // Content of Update File Metadata structure is valid if state is not RFDR_CLIENT_STATE_RESET.

        // Note: copy from the "main" client version, not the collector buffered version
        // (collectionInfo->fileSetInfoBuf) as these may be temporarily out-of-sync.
        // Note: this is a copy of the structure, not a pointer assignment.
        *updateFileMetadata = clientInfo->fileSetInfo;
    }
    else {
        // The caller should NOT use the Update File Metadata structure
        // in the case of Update File processing state = RFDR_CLIENT_STATE_RESET,
        // but just in case, initialize with some fixed default data for consistency.
        RFD_MEMSET(updateFileMetadata, 0, sizeof(RFD_FSET_MDATA_STRUCT));
        RFD_STRCPY_S(updateFileMetadata->name, RFD_FSET_NAME_MAX_LEN, TEXT(""));
    }

    /////////////////////////////////////
    // Release Mutex
    /////////////////////////////////////
    status = RFD_ReleaseMutex(clientInfo->clientMutex);
    if (status != RFD_STATUS_OK) {
        return status;
    }

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrInputApi
///
/// @brief	Get pointer to the Name string field from the provided Update File Metadata structure.
///
/// @param	fileSetMetadata	Pointer to the Update File (FileSet) Metadata structure which contains
///                         valid data. This structure is generated using
///                         RFD_CollectorParseUpdateFileMetadataMsg().
/// @param	name	        (output) Pointer to the Name field from the Update File Metadata structure.
///
/// @return	RFD_STATUS_OK on success, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CollectorGetUpdateFileMetadataName(
                RFD_FSET_MDATA_STRUCT * fileSetMetadata,
                TCHAR ** name)
{
	*name = fileSetMetadata->name;
    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrInputApi
///
/// @brief	Get the Life field from the provided Update File Metadata structure.
///
/// @param	fileSetMetadata	Pointer to the Update File (FileSet) Metadata message structure which
///                         contains valid data. This structure is generated using
///                         RFD_CollectorParseUpdateFileMetadataMsg().
/// @param	life	        The Life(Time) field value from the Update File Metadata structure.
///                         The Life field specifies the final day on which this update will be
///                         transmitted. The value is coded as the number of days since
///                         1st January 2009.
///
/// @return	RFD_STATUS_OK on success, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CollectorGetUpdateFileMetadataLife(
                RFD_FSET_MDATA_STRUCT * fileSetMetadata,
                INT32 * life)
{
	*life = fileSetMetadata->lifeTime;
    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrInputApi
///
/// @brief	Get the Update Files DMI field from the provided Update File Metadata structure.
///
/// @param	fileSetMetadata	Pointer to the Update File (FileSet) Metadata message structure which
///                         contains valid data. This structure is generated using
///                         RFD_CollectorParseUpdateFileMetadataMsg().
/// @param	updateFilesDmi	The Update Files DMI field value from the Update File Metadata structure.
///                         A value of 0 indicates this field was not available from the corresponding
///                         RFD Update File Metadata message which was originally parsed
///                         (e.g. optional Extension Field was present)
///
/// @return	RFD_STATUS_OK on success, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CollectorGetUpdateFileMetadataUpdateFilesDmi(
                RFD_FSET_MDATA_STRUCT * fileSetMetadata,
                UINT16 * updateFilesDmi)
{
	*updateFilesDmi = fileSetMetadata->blockMsgDmi;
    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Wait for input message available in the specified messageQueue.
///
/// If message is available it is copied from the messageQueue and into the messageInfo message
/// buffer.
///
/// @param  messageQueue        Message queue handle.
/// @param  messageInfo         Message info structure.
/// @param  timeOutMilliseconds The wait time out in milliseconds.
///
/// @return
/// RFDR_COLLECTOR_IN_MSG_RECEIVED if message received, RFDR_COLLECTOR_IN_MSG_TIMEOUT if time out
/// with no error, RFDR_COLLECTOR_IN_MSG_ERROR on error.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFDR_COLLECTOR_MSG_WAIT_STATUS  RFDR_CollectorWaitForMessage(RFD_MSG_QUEUE_HANDLE messageQueue,  RFD_MESSAGE_INFO * messageInfo, UINT16 timeOutMilliseconds)
{
	RFD_STATUS status;
	RFDR_COLLECTOR_MSG_WAIT_STATUS waitInputMessageStatus;

	status = RFD_MSG_QUEUE_WaitForMessage(messageQueue, timeOutMilliseconds);

	if(status == RFD_STATUS_OK) {
		status = RFD_MSG_QUEUE_GetMessage(messageQueue,
									messageInfo->bitstreamBuf, messageInfo->bitstreamBufLenBytes,
									&messageInfo->messageBitstreamLenBytes);
		if(status == RFD_STATUS_OK) {
			waitInputMessageStatus = RFDR_COLLECTOR_IN_MSG_RECEIVED;
		}
		else {
			waitInputMessageStatus = RFDR_COLLECTOR_IN_MSG_ERROR;
		}
	}
	else if(status == RFD_STATUS_OBJECT_TIMEOUT) {
		waitInputMessageStatus = RFDR_COLLECTOR_IN_MSG_TIMEOUT;
	}
	else { // error
		waitInputMessageStatus = RFDR_COLLECTOR_IN_MSG_ERROR;
	}

	return waitInputMessageStatus;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Parse available input message.
///
/// @param	messageInfo					information describing the message.
/// @param	isFileSetCollectionComplete	file set collection complete status.
/// @param	messageType					type of the message, block or file set metadata message.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS  RFDR_CollectorParseMessage(RFD_MESSAGE_INFO * messageInfo,
									   BOOL isFileSetCollectionComplete,
									   RFD_MSG_TYPE * messageType)
{
	RFD_STATUS status;
	BOOL isBlockMsgParsingEnabled;

	// Disable Block Messaging Parsing if the current File Set Collection is complete.
	if(isFileSetCollectionComplete) {
		isBlockMsgParsingEnabled = FALSE;
	}
	else {
		isBlockMsgParsingEnabled = TRUE;
	}

	// Parse the message.
	status = RFD_ParseMessage(messageInfo, isBlockMsgParsingEnabled);

	// Output the message type that was parsed.
	*messageType = messageInfo->messageType;
	return(status);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Collect a parsed block type message. If the corresponding file is in the
/// 		collection/decode state and if the block has not been previously collected, the block
/// 		is stored.
///
/// @param	collectorInfo	The message collector information handle.
/// @param	clientInfo		The client information handle.
/// @param	pSignalRequest	Pointer to the signal request, true if enough blocks have been
/// 						collected to signal rfd processor to decode the collected blocks.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS RFDR_CollectBlock(RFDR_COLLECTOR_INFO_STRUCT * collectorInfo, RFDR_CLIENT_INFO_STRUCT * clientInfo, BOOL * pSignalRequest)
{
	// a block is parsed before call.
	// if file is not in collecting/decoding state, return with signalRequest false.
	// check if this is a new block,
	// if new block, store the new block
	// if >= decode threshold, then set signal flag and signalRequest output parameter.

	int i;
	int fileIdx = 0 /* avoid compiler warning */;
	BOOL foundId;
	RFDR_FILE_PROCESS_INFO * fileProcessInfo;
	RFD_STATUS status;
	TCHAR * pathBuf;

	// initialize signal request to false.
	*pSignalRequest = FALSE;

	// allocate pathBuf
	if(NULL == (pathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN )))
	{
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// find the file for this block based on fileId
	foundId = FALSE;
	for(i=0;i<clientInfo->fileSetInfo.numFiles;i++)
	{
		if(clientInfo->fileSetInfo.fileInfo[i].id == collectorInfo->messageInfo->message.blockMsg.fileId)
		{
			foundId = TRUE;
			fileIdx = i;
			break;
		}
	}

	if(!foundId)
	{
		// file not found
		RFD_FREE(pathBuf);
		return RFD_STATUS_OK;
	}

	fileProcessInfo = clientInfo->fileProcessInfo[fileIdx];

	// only collect the block if processor is in the collecting/decoding state.
	if( !(fileProcessInfo->state < RFDR_STATE_DECODE_COMPLETE &&
		  fileProcessInfo->state > RFDR_STATE_UNINIT ))
	{
		// not in a collecting mode.
		RFD_FREE(pathBuf);
		if(fileProcessInfo->state >= RFDR_STATE_DECODE_COMPLETE) {
			RFD_DPrint( TEXT("RFDR Collector: Message not stored, decode is already complete for file (fragment) id %d\n"), fileProcessInfo->fileInfo.id);
		}
		else {
			RFD_DPrint( TEXT("RFDR Collector: Message not stored, not in decoding state for file (fragment) id %d (state = %d)\n"),
				fileProcessInfo->state, fileProcessInfo->fileInfo.id);
		}
		return RFD_STATUS_OK;
	}

	// create base file path string used by this file instance (fragment)
	RFDR_ConstructFragmentDirPathName(pathBuf, RFD_MAX_PATH_LEN,
									  clientInfo, fileProcessInfo->fileInfo.id);

	switch(fileProcessInfo->fileInfo.codeType)
	{
		case RFD_CODE_TYPE_ECC1:
		case RFD_CODE_TYPE_ECC2:

			status = RFDR_CollectBlockXccCode(pathBuf, fileProcessInfo, &collectorInfo->messageInfo->message.blockMsg, pSignalRequest);
			break;

		case RFD_CODE_TYPE_SIMPLE_BLK:

			status = RFDR_CollectBlockSimpleCode(pathBuf, fileProcessInfo, &collectorInfo->messageInfo->message.blockMsg, pSignalRequest);
			break;

		default:

			status = RFD_STATUS_ERROR_INVALID_CODE_TYPE;
			break;
	}

	// If the signal request is set, set the update flag
	// corresponding to the file index.
	if(*pSignalRequest)
	{
		collectorInfo->fileProcessCollectorUpdate[fileIdx] = TRUE;
	}

	RFD_FREE(pathBuf);
	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_MsgCollectorOpenLL(
    RFD_COLLECTOR_THREAD_DATA_HANDLE hCollectorThread,
    RFD_STATUS * statusPtr,
    BOOL *isThreadExitRequestedPtr
        )
{
	DWORD clientIdNo;
	RFDR_CLIENT_INFO_STRUCT * clientInfo;

	clientInfo = hCollectorThread->clientInfo;
	clientIdNo = clientInfo->clientIdNo;

    RFD_DPrint( TEXT("RFDR Message Collector Started, clientIdNo %d\n"), clientIdNo);

    *isThreadExitRequestedPtr = FALSE;
    *statusPtr = RFD_STATUS_OK;

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_MsgCollectorCloseLL(
    RFD_COLLECTOR_THREAD_DATA_HANDLE hCollectorThread,
    RFD_STATUS * statusPtr
        )
{
	DWORD clientIdNo;
	RFDR_CLIENT_INFO_STRUCT * clientInfo;

	clientInfo = hCollectorThread->clientInfo;
	clientIdNo = clientInfo->clientIdNo;

	/////////////////////////
	// Shutdown - exit thread. Cleanup and return.
	/////////////////////////

    RFD_DPrint( TEXT("RFDR Message Collector Ending, clientIdNo %d\n"), clientIdNo);

    *statusPtr = RFD_STATUS_OK;

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_MsgCollectorRunLL(
    RFD_COLLECTOR_THREAD_DATA_HANDLE hCollectorThread,
    RFD_STATUS * statusPtr,
    BOOL * isThreadExitRequestedPtr,
    BOOL * isIdlePtr
        )
{
	DWORD clientIdNo;
	RFDR_COLLECTOR_MSG_WAIT_STATUS messageWaitResult;
	BOOL isMessageAvailable;
	BOOL signalRequest;
	BOOL isThreadExitRequested;
	RFD_MSG_TYPE messageType;
	RFD_STATUS status;
	RFDR_COLLECTOR_INFO_STRUCT * collectorInfo;
	RFDR_CLIENT_INFO_STRUCT * clientInfo;
	RFD_EVENT_HANDLE errorExitEvent;
    BOOL isIdle;

	clientInfo = hCollectorThread->clientInfo;
	collectorInfo = hCollectorThread->collectorInfo;
	clientIdNo = clientInfo->clientIdNo;
	errorExitEvent = collectorInfo->errorExitEvent;

    isIdle = FALSE;
	isThreadExitRequested = FALSE;

	do {

		//////////////////////////////////////////
		// Wait for Input Message and parse if available
		/////////////////////////////////////////

		messageWaitResult = RFDR_CollectorWaitForMessage( collectorInfo->messageQueue, collectorInfo->messageInfo, RFDR_IN_MSG_TIMEOUT_MSEC);

		isMessageAvailable = FALSE;
		messageType = MSG_TYPE_INVALID;

		switch (messageWaitResult)
		{
			case RFDR_COLLECTOR_IN_MSG_RECEIVED:

				if(RFD_STATUS_OK == RFDR_CollectorParseMessage(  collectorInfo->messageInfo,
																 collectorInfo->isFileSetCollectionComplete,
																 &messageType))
				{
					isMessageAvailable = TRUE;
				}

				break;

			case RFDR_COLLECTOR_IN_MSG_TIMEOUT:
                // set idle flag after timeout waiting for message.
				isIdle = TRUE;
				break;

			case RFDR_COLLECTOR_IN_MSG_ERROR:
				RFD_DPrint( TEXT("RFDR Collector: RFDR_CollectorWaitForMessage error messageWaitResult %d, clientIdNo %d\n"), messageWaitResult, clientIdNo);
				// To keep legacy implementation behavior, don't signal error or signal exit thread,
                // but do treat this as idle condition.
                isIdle = TRUE;
				break;

			default:
				RFD_DPrint( TEXT("RFDR Collector: RFDR_CollectorWaitForMessage error messageWaitResult %d, clientIdNo %d\n"), messageWaitResult, clientIdNo);
				// To keep legacy implementation behavior, don't signal error or signal exit thread,
                // but do treat this as idle condition.
                isIdle = TRUE;
				break;
		}

		//////////////////////////////////////////
		// Check for Shutdown request event
		/////////////////////////////////////////

		status = RFD_WaitForEvent(collectorInfo->exitRequestEvent, 0);
		if(status == RFD_STATUS_OK) {
			RFD_DPrint( TEXT("RFDR Collector: Shutdown Request Event Occurred, clientIdNo %d\n"), clientIdNo);
			// exit processing loop to shutdown
			isThreadExitRequested = TRUE;
			continue;
		}

		if(isMessageAvailable) {
			//////////////////////////////////////////
			// A new message is available, update the Processor with the new  message.
			/////////////////////////////////////////

			isMessageAvailable = FALSE;

			//////////////////////////////////////////
			// Acquire the Mutex to access shared data
			/////////////////////////////////////////

			status = RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);
			if(status != RFD_STATUS_OK) {
				RFD_DPrint( TEXT("RFDR Collector: Error acquiring collect mutex, clientIdNo %d\n"), clientIdNo);
				// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
				RFDR_SignalErrorExit(errorExitEvent);
				isThreadExitRequested = TRUE;
				continue;
			}

			// Access shared resource

			if(messageType == MSG_TYPE_FSET_MDATA) {
				RFD_DPrint( TEXT("RFDR Collector: FileSet Metadata Type Message Received, clientIdNo %d\n"), clientIdNo);

#ifdef DYNAMIC_BLOCK_MSG_CARID_ENABLE
				// Update the CarouselId value used for Block Message delivery
				collectorInfo->messageInfo->configInfo.blockMsgCarouselId =
							collectorInfo->messageInfo->message.fileSetInfoMsg.blockMsgCarouselId;
#endif
				// Copy newly recevied fileSet Metadata message to the shared (mutex protected) fileSetInfoBuf.
				RFD_MEMCPY( &collectorInfo->fileSetInfoBuf, &collectorInfo->messageInfo->message.fileSetInfoMsg, sizeof(RFD_FSET_MDATA_STRUCT));

				// set fileSet update flag.
				collectorInfo->fileSetCollectorUpdate = TRUE;

				// Signal the Colletion Event
				RFD_DPrint( TEXT("RFDR Collector: Setting Event for FSet Metadata Msg, clientIdNo %d\n"), clientIdNo);
				status = RFD_SetEvent(clientInfo->clientEvent);
				if (status != RFD_STATUS_OK) {
					RFD_DPrint( TEXT("SetEvent failed (%d), clientIdNo %d\n"), RFD_GetLastError(), clientIdNo);
					// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
					RFDR_SignalErrorExit(errorExitEvent);
					isThreadExitRequested = TRUE;
					RFD_ReleaseMutex(clientInfo->clientMutex);
					continue;
				}
			}
			else if(messageType == MSG_TYPE_BLOCK) {
				RFD_DPrint( TEXT("RFDR Collector: Block Type Message Received, clientIdNo %d\n"), clientIdNo);

				signalRequest = FALSE;
				status = RFDR_CollectBlock(collectorInfo, clientInfo, &signalRequest);
				if(status == RFD_STATUS_OK) {
					if(signalRequest) {

						// Signal the Colletion Event
						RFD_DPrint( TEXT("RFDR Collector: Setting Event for Block Collection, clientIdNo %d\n"), clientIdNo);
						status = RFD_SetEvent(clientInfo->clientEvent);
						if (status != RFD_STATUS_OK) {
							RFD_DPrint( TEXT("SetEvent failed (%d), clientIdNo %d\n"), RFD_GetLastError(), clientIdNo);
							// Unr5ecoverable error, signal error to manager and exit processing loop to exit thread.
							RFDR_SignalErrorExit(errorExitEvent);
							isThreadExitRequested = TRUE;
							RFD_ReleaseMutex(clientInfo->clientMutex);
							continue;
						}
					}
				}
				else {
					RFD_DPrint( TEXT("RFDR Collector: RFDR_CollectBlock error, status %d, clientIdNo %d\n"), clientIdNo);
					// Note: this could be different categories of error e.g. one time error due to parsing error, or
					// fatal error due to file system write error.
					// Since the category of error is not know, treat it as benign and contine collection processing
					// (i.e. don't exit thread).
				}
			}

			/////////////////////////////////////
			// Release Mutex
			/////////////////////////////////////
			status = RFD_ReleaseMutex(clientInfo->clientMutex);
			if (status != RFD_STATUS_OK) {
				// Unrecoverable error, signal error to manager and exit processing loop to exit thread.
				RFDR_SignalErrorExit(errorExitEvent);
				isThreadExitRequested = TRUE;
				continue;
			}
		}

	} while(FALSE); // do once

    *isThreadExitRequestedPtr = isThreadExitRequested;
    *isIdlePtr = isIdle;
    *statusPtr = status;

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @internal
/// @brief	The message collector thread function.
///
/// 		This thread receives and parses messages for a particular rfd client index. Block
/// 		type messages are stored and the rfd processor thread is signalled when enough blocks
/// 		are collected for decode. File set metadata messages are buffered and passed to the
/// 		rfd processor.
///
/// @param	lpParam	The RFD_COLLECTOR_THREAD_DATA_HANDLE handle passed into this thread.
///
/// @return	0 on normal thread exit.
////////////////////////////////////////////////////////////////////////////////////////////////////
DWORD RFD_PLATFORM_API RFDR_MsgCollector( LPVOID lpParam )
{
	RFD_COLLECTOR_THREAD_DATA_HANDLE hCollector;
    RFD_STATUS status;
    BOOL isThreadExitRequested;
    BOOL isIdle;

    hCollector = (RFD_COLLECTOR_THREAD_DATA_HANDLE) lpParam;

    ////////////////////////////////
    // Open (create and initialize)
    ////////////////////////////////
    RFDR_MsgCollectorOpenLL(
        hCollector,
        &status,
        &isThreadExitRequested
            );

    ////////////////////////////////
    // Run loop
    ////////////////////////////////
    while(!isThreadExitRequested) {

        RFDR_MsgCollectorRunLL(
            hCollector,
            &status,
            &isThreadExitRequested,
            &isIdle
                );
    }
    // Run loop exit

    ////////////////////////////////
    // Close (cleanup)
    ////////////////////////////////
    RFDR_MsgCollectorCloseLL(
        hCollector,
        &status
            );

    // Thread exit
    return 0;
}
