////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_receiver\rfdr_manager.c
///
/// @brief	rfdr manager class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd_receiver.h"
#include "rfd_file_consumer.h"
#include "rfd_msg_collector.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
// PRIVATE FUNCTIONS
////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Start the RFDR  (RFD Receiver Module) in the alternate  single-threaded mode.
///
/// All serial run objects of the RFDR are started.
///
/// @param  rfdReceiverHandle   Handle of the RFDR.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _RFD_StartReceiverAltSingleThreadMode(RFDR_HANDLE rfdReceiverHandle)
{
	RFDR_HANDLE	hRfdr;
	RFDR_THREAD_MANAGEMENT_STRUCT * manageInfo;
	int i;
	RFD_STATUS status;
    BOOL isThreadExitRequested;

	hRfdr = rfdReceiverHandle;
    if( hRfdr == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

	manageInfo = &hRfdr->rfdrThreadManagementInfo;

	/////////////////////////////////////////////////////
	// Create RFDR Processor run object first
	/////////////////////////////////////////////////////

    RFDR_ProcessorOpenLL(
        manageInfo->rfdProcessorThreadInfo,
        &status,
        &isThreadExitRequested
            );
	RFD_DPrint( TEXT("RFDR Processor Open\n"));

	if (isThreadExitRequested)
	{
	   RFD_DPrint( TEXT("Error on RFDR Processor Open\n"));
	   RFD_StopReceiver(hRfdr);
	   return RFD_STATUS_ERROR_CREATE_THREAD;
	}

	/////////////////////////////////////////////////////
	// Wait for Processor Initialization Complete before starting the Client Threads
	/////////////////////////////////////////////////////

	status = RFD_WaitForEvent(hRfdr->rfdrProcessorInfo->initCompleteEvent,
							  RFDR_PROC_THREAD_INIT_TIMEOUT_MSEC);

	if(status != RFD_STATUS_OK) {
		// timeout error or other error.
		if(status == RFD_STATUS_OBJECT_TIMEOUT) {
			RFD_DPrint( TEXT("RFD-Receiver: Timeout waiting for Processor Initialization Complete Event\n"));
			RFD_StopReceiver(hRfdr);
			return RFD_STATUS_OBJECT_TIMEOUT;
		}
		else {
			RFD_DPrint( TEXT("RFD-Receiver: Error in WaitEvent waitingfor Processor Initialization Complete Event\n"));
			RFD_StopReceiver(hRfdr);
			return RFD_STATUS_ERROR_GENERAL;
		}
	}
	else {
		RFD_DPrint( TEXT("RFD-Receiver: Processor Initialization Complete Event\n"));
	}

	/////////////////////////////////////////////////////
	// Create RFDR Client Collector Threads now that the Processor thread is ready
	/////////////////////////////////////////////////////
	for(i=0;i<RFDR_NUM_CLIENTS;i++)
	{
        RFDR_MsgCollectorOpenLL(
            manageInfo->rfdCollectorThreadInfo[i],
            &status,
            &isThreadExitRequested
                );
		RFD_DPrint( TEXT("RFDR Msg Collector Open, Collector Index %d\n"), i);

	    if (isThreadExitRequested)
	    {
		   RFD_DPrint( TEXT("Error on RFDR Msg Collector Open, Collector Index %d\n"), i);
	       RFD_StopReceiver(hRfdr);
	       return RFD_STATUS_ERROR_CREATE_THREAD;
	    }
	}

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Start the RFDR  (RFD Receiver Module) in the standard multi-threaded mode.
///
/// All threads of the RFDR are started.
///
/// @param  rfdReceiverHandle   Handle of the RFDR.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _RFD_StartReceiverStandardMultiThreadMode(RFDR_HANDLE rfdReceiverHandle)
{
	RFDR_HANDLE	hRfdr;
	RFDR_THREAD_MANAGEMENT_STRUCT * manageInfo;
	int i;
	RFD_STATUS status;

	hRfdr = rfdReceiverHandle;
    if( hRfdr == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

	manageInfo = &hRfdr->rfdrThreadManagementInfo;

	/////////////////////////////////////////////////////
	// Create RFDR Processor thread first
	/////////////////////////////////////////////////////

	manageInfo->processorThread = RFD_CreateThread(
											RFDR_PROCESSOR_THREAD_STACK_SIZE,
											"RFDR Processor",
											RFDR_Processor,                    // thread function name
											manageInfo->rfdProcessorThreadInfo,// argument to thread function
											&manageInfo->processorThreadId);   // returns the thread identifier

	// Check the return value for success.
	// If RFD_CreateThread fails, terminate execution.
	// This will automatically clean up threads and memory.

	if (manageInfo->processorThread == NULL)
	{
	   RFD_StopReceiver(hRfdr);
	   return RFD_STATUS_ERROR_CREATE_THREAD;
	}

	/////////////////////////////////////////////////////
	// Wait for Processor Initialization Complete before starting the Client Threads
	/////////////////////////////////////////////////////

	status = RFD_WaitForEvent(hRfdr->rfdrProcessorInfo->initCompleteEvent,
							  RFDR_PROC_THREAD_INIT_TIMEOUT_MSEC);

	if(status != RFD_STATUS_OK) {
		// timeout error or other error.
		if(status == RFD_STATUS_OBJECT_TIMEOUT) {
			RFD_DPrint( TEXT("RFD-Receiver: Timeout waiting for Processor Initialization Complete Event\n"));
			RFD_StopReceiver(hRfdr);
			return RFD_STATUS_OBJECT_TIMEOUT;
		}
		else {
			RFD_DPrint( TEXT("RFD-Receiver: Error in WaitEvent waitingfor Processor Initialization Complete Event\n"));
			RFD_StopReceiver(hRfdr);
			return RFD_STATUS_ERROR_GENERAL;
		}
	}
	else {
		RFD_DPrint( TEXT("RFD-Receiver: Processor Initialization Complete Event\n"));
	}


	/////////////////////////////////////////////////////
	// Set Processor Thread Priority
	/////////////////////////////////////////////////////

	if(!RFD_SetThreadPriority(manageInfo->processorThread, RFDR_PROCESSOR_THREAD_PRIORITY))
	{
		RFD_StopReceiver(hRfdr);
		return RFD_STATUS_ERROR_SET_THREAD_PRIORITY;
	}

	/////////////////////////////////////////////////////
	// Create RFDR Client Collector Threads now that the Processor thread is ready
	/////////////////////////////////////////////////////
	for(i=0;i<RFDR_NUM_CLIENTS;i++)
	{
		manageInfo->collectorThread[i] = RFD_CreateThread(
												RFDR_COLLECTOR_THREAD_STACK_SIZE,
												"RFDR Collector",
												RFDR_MsgCollector,      // thread function name
												manageInfo->rfdCollectorThreadInfo[i],   // argument to thread function
												&manageInfo->collectorThreadId[i]);     // returns the thread identifier

		// Check the return value for success.
		// If RFD_CreateThread fails, terminate execution.
		// This will automatically clean up threads and memory.

		if (manageInfo->collectorThread[i] == NULL)
		{
			RFD_StopReceiver(hRfdr);
			return RFD_STATUS_ERROR_CREATE_THREAD;
		}
	}

	/////////////////////////////////////////////////////
	// Set Collector Thread Priorities
	/////////////////////////////////////////////////////


	for(i=0;i<RFDR_NUM_CLIENTS;i++)
	{
		if(!RFD_SetThreadPriority(manageInfo->collectorThread[i], RFDR_COLLECTOR_THREAD_PRIORITY))
		{
			RFD_StopReceiver(hRfdr);
			return RFD_STATUS_ERROR_SET_THREAD_PRIORITY;
		}
	}

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Stop the RFDR (RFD Receiver Module) that is running in the standard multi-threaded mode
///
/// @param	rfdReceiverHandle	Handle of the RFDR.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _RFD_StopReceiverStandardMultiThreadMode(RFDR_HANDLE rfdReceiverHandle)
{
	RFDR_COLLECTOR_INFO_STRUCT * collectorInfo;
	RFDR_THREAD_MANAGEMENT_STRUCT * manageInfo;
	int i;
	RFDR_HANDLE	hRfdr;

	hRfdr = rfdReceiverHandle;
    if( hRfdr == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

	manageInfo = &hRfdr->rfdrThreadManagementInfo;

	/////////////////////////////////////////////////////
	// Request Shutdown of Threads
	/////////////////////////////////////////////////////

	// Request shutdown of RFDR Processor Thread
	RFD_SetEvent(hRfdr->rfdrProcessorInfo->exitRequestEvent);
	*hRfdr->rfdrProcessorInfo->exitRequestFlagPtr = TRUE;

	RFD_DPrint( TEXT("Set RFDR Processor Shutdown Request Event\n"));


	// Request shutdown of Client Collector Threads
	for(i=0;i<RFDR_NUM_CLIENTS;i++)
	{
		collectorInfo = &hRfdr->rfdrClientSet->collectorInfo[i];

		RFD_SetEvent(collectorInfo->exitRequestEvent);
		RFD_DPrint( TEXT("Set RFDR Collector Shutdown Request Event, Collector Index %d\n"), i);
	}

	////////////////////////////////////////////////////
	// Wait until all threads have terminated.
	////////////////////////////////////////////////////

	// Wait for processor thread
	if(manageInfo->processorThread != NULL) {
		RFD_WaitForThreadExit(manageInfo->processorThread, RFD_INFINITE);
	}
	RFD_DPrint( TEXT("RFDR Processor Shutdown Requests Complete\n"));

	// Wait for collector threads
	for(i=0;i<RFDR_NUM_CLIENTS;i++)
	{
		if(manageInfo->collectorThread[i] != NULL) {
			RFD_WaitForThreadExit(manageInfo->collectorThread[i], RFD_INFINITE);
		}
	}
	RFD_DPrint( TEXT("Set RFDR Collector Shutdown Requests Complete\n"));

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Stop the RFDR (RFD Receiver Module) that is running in the alternate single-threaded mode.
///
/// @param  rfdReceiverHandle   Handle of the RFDR.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _RFD_StopReceiverAltSingleThreadMode(RFDR_HANDLE rfdReceiverHandle)
{
	RFDR_HANDLE	hRfdr;
	RFDR_THREAD_MANAGEMENT_STRUCT * manageInfo;
	int i;
	RFD_STATUS status;

	hRfdr = rfdReceiverHandle;
    if( hRfdr == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

	manageInfo = &hRfdr->rfdrThreadManagementInfo;

	/////////////////////////////////////////////////////
	// Stop (close) the RFDR Processor run object first
	/////////////////////////////////////////////////////

    RFDR_ProcessorCloseLL(
        manageInfo->rfdProcessorThreadInfo,
        &status
            );
	RFD_DPrint( TEXT("RFDR Processor Close\n"));


	/////////////////////////////////////////////////////
	// Stop (close) the RFDR Client Collector Threads
	/////////////////////////////////////////////////////
	for(i=0;i<RFDR_NUM_CLIENTS;i++)
	{
        RFDR_MsgCollectorCloseLL(
            manageInfo->rfdCollectorThreadInfo[i],
            &status
                );
		RFD_DPrint( TEXT("RFDR Msg Collector Close, Collector Index %d\n"), i);
	}

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// PUBLIC FUNCTIONS
////////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////
/// @addtogroup RFDR_API RFD Receiver API
///////////////////////////////////////////////////////////////////////////////////////////////////

// doxygen group start symbol, don't remove.
///////////////////////////////////////////////////////////////////////////////////////////////////
/// @addtogroup RfdrManagerApi RFDR Manager API
/// @ingroup RFDR_API
/// @{
///////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Get the RFD SDK version information.
///
/// The function can be called at anytime to retrieve the RFD SDK version information.
///
/// @param  rfdSdkVersionInfo   Pointer to the \e RFD_SDK_VERSION_INFO data structure that this
///                             function fills with the version information. The caller allocates
///                             this data structure.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_GetSdkVersionInfo(RFD_SDK_VERSION_INFO * rfdSdkVersionInfo)
{
	if(rfdSdkVersionInfo == NULL) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
	}

	// Fill out the version structure with the version info.

	rfdSdkVersionInfo->rfdMajorVersionNo   = RFD_SDK_MAJOR_VERSION_NO;

	rfdSdkVersionInfo->rfdMinorVersionNo   = RFD_SDK_MINOR_VERSION_NO;

	rfdSdkVersionInfo->rfdVersionReleaseNo = RFD_SDK_VERSION_RELEASE_NO;

	RFD_SPRINTF_S(rfdSdkVersionInfo->rfdVersionString, RFD_SDK_VERSION_STRING_MAX_LEN,
				  RFD_SDK_VERSION_STRING_PRINTF_FORMAT, // Format
				  rfdSdkVersionInfo->rfdMajorVersionNo,
				  rfdSdkVersionInfo->rfdMinorVersionNo,
				  rfdSdkVersionInfo->rfdVersionReleaseNo );

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Open the RFDR (RFD Receiver Module) in the specified mode of operation.
///
/// The RFDR data object is created. The RFDR threads are \e not started.
///
/// @param  rfdReceiverHandle       Handle of the RFDR.
/// @param  isAltSingleThreadMode   TRUE to select the alternate single-thread mode of RFDR
///                                 operation, FALSE to select the standard multi-threaded mode
///                                 of RFDR operation.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_OpenReceiverExt(
    RFDR_HANDLE * rfdReceiverHandle,
    BOOL isAltSingleThreadMode
    )
{
	RFDR_THREAD_MANAGEMENT_STRUCT * manageInfo;
	RFDR_CLIENT_INFO_STRUCT * clientInfo;
	RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo;
	RFDR_COLLECTOR_INFO_STRUCT * collectorInfo;
	RFDR_HANDLE	hRfdr;
	int i;
	RFD_STATUS status = RFD_STATUS_OK;

	// Initialize to null.
	*rfdReceiverHandle = NULL;

    // Allocate memory for RFDR processor data, and clear structure (calloc).
	hRfdr = (RFDR_HANDLE) RFD_CALLOC(sizeof(RFDR_STRUCT));
    if( hRfdr == NULL ) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
    }

    // Set the requested RFDR operating mode
    if(isAltSingleThreadMode) {
        // alternate single-threaded mode
        hRfdr->isAltSingleThreadMode = TRUE;
    }
    else {
        // standard multi-threaded mode
        hRfdr->isAltSingleThreadMode = FALSE;
    }

    // Initialize run state
    hRfdr->isRunning = FALSE;

	manageInfo = &hRfdr->rfdrThreadManagementInfo;

    // Allocate memory for RFDR Processor data handle.
	manageInfo->rfdProcessorThreadInfo = (RFD_PROCESSOR_THREAD_DATA_HANDLE) RFD_CALLOC(sizeof(RFD_PROCESSOR_THREAD_DATA_STRUCT));
	if( manageInfo->rfdProcessorThreadInfo == NULL ) {
		RFD_CloseReceiver(hRfdr);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

    // Initialize to NULL the low level control stucture pointer that's
    // managed and used by the "run/thread" layer
    // With calloc() (which zero initialzes) of rfdProcessorThreadInfo don't really need to do this,
    // but just to be explicit.
    manageInfo->rfdProcessorThreadInfo->llControl = NULL;

    // Allocate memory for RFDR File Consumer data handle.
	for(i=0;i<RFDR_NUM_CLIENTS;i++) {
		manageInfo->rfdConsumerThreadInfo[i] = (RFD_CONSUMER_THREAD_DATA_HANDLE) RFD_CALLOC(sizeof(RFD_FILE_CONSUMER_THREAD_STRUCT));
		if( manageInfo->rfdConsumerThreadInfo[i] == NULL ) {
			RFD_CloseReceiver(hRfdr);
			return RFD_STATUS_ERROR_MEM_ALLOC;
		}
	}

    // Allocate memory for RFDR File Collector data handle.
	for(i=0;i<RFDR_NUM_CLIENTS;i++) {
		manageInfo->rfdCollectorThreadInfo[i] = (RFD_COLLECTOR_THREAD_DATA_HANDLE) RFD_CALLOC(sizeof(RFD_FILE_COLLECTOR_THREAD_STRUCT));
		if( manageInfo->rfdCollectorThreadInfo[i] == NULL ) {
			RFD_CloseReceiver(hRfdr);
			return RFD_STATUS_ERROR_MEM_ALLOC;
		}
	}

	/////////////////////////////////////////////////////
	// Create single Error Exit Event object.
	// The event is set by any thread to indicate
	// thread is exiting due to unexpected error condition.
	// The event is monitored here by the single manager thread.
	/////////////////////////////////////////////////////

	// hRfdr->rfdrThreadManagementInfo.errorExitEvent is primary location
	// from which to maintain this errorExitEvent event.
	// It is also stored to other structures to pass to the rfdr threads.
	hRfdr->rfdrThreadManagementInfo.errorExitEvent = RFD_CreateEvent();

	if (hRfdr->rfdrThreadManagementInfo.errorExitEvent == NULL) {
		RFD_CloseReceiver(hRfdr);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	/////////////////////////////////////////////////////
	// Create and initializize Processor related objects
	/////////////////////////////////////////////////////

    // Allocate memory for RFDR processor info structure, and clear structure (calloc).
	hRfdr->rfdrProcessorInfo = (RFDR_PROCESSOR_INFO_STRUCT *) RFD_CALLOC(sizeof(RFDR_PROCESSOR_INFO_STRUCT));
    if( hRfdr->rfdrProcessorInfo == NULL ) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
    }
	manageInfo->rfdProcessorThreadInfo->rfdrProcessorInfo = hRfdr->rfdrProcessorInfo;

	// Create Processor Shutdown Request Event object

	hRfdr->rfdrProcessorInfo->exitRequestEvent = RFD_CreateEvent();

	if (hRfdr->rfdrProcessorInfo->exitRequestEvent == NULL) {
		RFD_CloseReceiver(hRfdr);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Initialize the exitRequestFlag.
    // exitRequestFlagPtr initialized to point to the internal internalExitRequestFlag
    // as the flag location. Before starting the RFD Reciever, the application can change
    // this address to an externally provided flag address using RFD_SetExitRequestFlagAddress().
	hRfdr->rfdrProcessorInfo->defaultExitRequestFlag = FALSE;
	hRfdr->rfdrProcessorInfo->exitRequestFlagPtr = &hRfdr->rfdrProcessorInfo->defaultExitRequestFlag;

	// Create Processor Initialization Complete Event object

	hRfdr->rfdrProcessorInfo->initCompleteEvent = RFD_CreateEvent();

	if (hRfdr->rfdrProcessorInfo->initCompleteEvent == NULL) {
		RFD_CloseReceiver(hRfdr);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Initialize the error exit event.
	hRfdr->rfdrProcessorInfo->errorExitEvent = hRfdr->rfdrThreadManagementInfo.errorExitEvent;

	/////////////////////////////////////////////////////
	// Create and initialize the 'common' Client related objects
	/////////////////////////////////////////////////////

    // Allocate memory for RFDR client set structure, and clear structure (calloc).
	hRfdr->rfdrClientSet = (RFDR_CLIENT_SET_STRUCT *) RFD_CALLOC(sizeof(RFDR_CLIENT_SET_STRUCT));
    if( hRfdr->rfdrClientSet == NULL ) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
    }
	hRfdr->rfdrClientSet->nClients = RFDR_NUM_CLIENTS;
	manageInfo->rfdProcessorThreadInfo->rfdrClientSet = hRfdr->rfdrClientSet;

	for(i=0;i<RFDR_NUM_CLIENTS;i++)
	{
		clientInfo = &hRfdr->rfdrClientSet->clientInfo[i];

		RFD_MEMSET(clientInfo, 0, sizeof( RFDR_CLIENT_INFO_STRUCT ));

		// Set Client Index.
		clientInfo->clientIndex = i;

		// Set Client ID.
		clientInfo->clientIdNo = gClientId[i];

		// Set Client Directory Name
		clientInfo->clientDirName = gClientDirName[i];

		// Create Collect Mutex object

		clientInfo->clientMutex = RFD_CreateMutex();

		if (clientInfo->clientMutex == NULL) {
			RFD_CloseReceiver(hRfdr);
			return RFD_STATUS_ERROR_MEM_ALLOC;
		}

		// Create Client Event object

		clientInfo->clientEvent = RFD_CreateEvent();

		if (clientInfo->clientEvent == NULL)
		{
			RFD_CloseReceiver(hRfdr);
			return RFD_STATUS_ERROR_MEM_ALLOC;
		}
	}

	/////////////////////////////////////////////////////
	// Create and initialize the Collector related objects
	/////////////////////////////////////////////////////

	for(i=0;i<RFDR_NUM_CLIENTS;i++)
	{
		collectorInfo = &hRfdr->rfdrClientSet->collectorInfo[i];

		RFD_MEMSET(collectorInfo, 0, sizeof( RFDR_COLLECTOR_INFO_STRUCT ));

		// Allocate memory for messageInfo
		collectorInfo->messageInfo = (RFD_MESSAGE_INFO *) RFD_CALLOC(sizeof(RFD_MESSAGE_INFO));
		if( collectorInfo->messageInfo == NULL ) {
			RFD_CloseReceiver(hRfdr);
			return RFD_STATUS_ERROR_MEM_ALLOC;
		}
		// Set messageInfo bitstream buf size.
		collectorInfo->messageInfo->bitstreamBufLenBytes = RFD_MAX_BITSTREAM_MESSAGE_LEN_BYTES;

		// Set the default configInfo
		collectorInfo->defaultMsgConfigInfo = gMessageConfigInfo[i];
#ifdef DYNAMIC_BLOCK_MSG_CARID_ENABLE
		// Set initial value to -1 to indicate this parameter is not initialized.
		collectorInfo->defaultMsgConfigInfo.blockMsgCarouselId = -1;
#endif
		// Set the (dynamic) messageInfo.configInfo that will be dynamically updated at run time
        // by updates to blockMsgCarouselId field derived from File Set Metadata messages
		collectorInfo->messageInfo->configInfo = gMessageConfigInfo[i];
#ifdef DYNAMIC_BLOCK_MSG_CARID_ENABLE
		// Set initial value to -1 to indicate this parameter is not yet initialized.
		collectorInfo->messageInfo->configInfo.blockMsgCarouselId = -1;
#endif

		// Create the Message Queue for message passing from the
		// application message producer to this message collector instance.
		 status = RFD_MSG_QUEUE_Create( &collectorInfo->messageQueue, RFDR_MESSAGE_QUEUE_BUF_SIZE);
		 if(status != RFD_STATUS_OK || collectorInfo->messageQueue == NULL) {
			RFD_CloseReceiver(hRfdr);
			return RFD_STATUS_ERROR_MESSAGE_QUEUE_CREATE;
		}

		// Create Collector Shutdown Request Event object
		collectorInfo->exitRequestEvent = RFD_CreateEvent();

		if (collectorInfo->exitRequestEvent == NULL) {
			RFD_CloseReceiver(hRfdr);
			return RFD_STATUS_ERROR_MEM_ALLOC;
		}

		// Initialize the error exit event.
		collectorInfo->errorExitEvent = hRfdr->rfdrThreadManagementInfo.errorExitEvent;

		manageInfo->rfdCollectorThreadInfo[i]->clientInfo = &hRfdr->rfdrClientSet->clientInfo[i];
		manageInfo->rfdCollectorThreadInfo[i]->collectorInfo = &hRfdr->rfdrClientSet->collectorInfo[i];
	}

	/////////////////////////////////////////////////////
	// Create and initialize Client File Consumer specific objects
	/////////////////////////////////////////////////////

	for(i=0;i<RFDR_NUM_CLIENTS;i++) {

		consumerInfo = &hRfdr->rfdrClientSet->consumerInfo[i];

		RFD_MEMSET(consumerInfo, 0, sizeof( RFD_FILE_CONSUMER_INFO_STRUCT ));

		// Set Client Index.
		consumerInfo->clientIndex = i;

		// Set Client Directory Name
		consumerInfo->consumerDirName = gFileConsumerDirName[i];

		// Create FileSet Ready Event object

		consumerInfo->fileSetReadyEvent = RFD_CreateEvent();
		if (consumerInfo->fileSetReadyEvent == NULL) {
			RFD_CloseReceiver(hRfdr);
			return RFD_STATUS_ERROR_MEM_ALLOC;
		}

		manageInfo->rfdConsumerThreadInfo[i]->clientInfo = &hRfdr->rfdrClientSet->clientInfo[i];
		manageInfo->rfdConsumerThreadInfo[i]->consumerInfo = &hRfdr->rfdrClientSet->consumerInfo[i];
	}

	*rfdReceiverHandle = hRfdr;
	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Open the RFDR (RFD Receiver Module).
///
/// The RFDR data object is created. The RFDR threads are \e not started.
///
/// @param  rfdReceiverHandle   Handle of the RFDR.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_OpenReceiver(RFDR_HANDLE * rfdReceiverHandle)
{
    RFD_STATUS status;

    status = RFD_OpenReceiverExt(rfdReceiverHandle, FALSE);

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Close the RFDR  (RFD Receiver Module).
///
/// The RFDR data object resources are cleaned-up.
///
/// @param  rfdReceiverHandle   Handle of the RFDR.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_CloseReceiver(RFDR_HANDLE rfdReceiverHandle)
{
	RFDR_CLIENT_INFO_STRUCT * clientInfo;
	RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo;
	RFDR_COLLECTOR_INFO_STRUCT * collectorInfo;
	RFDR_THREAD_MANAGEMENT_STRUCT * manageInfo;

	RFDR_HANDLE	hRfdr;
	int i;

	hRfdr = rfdReceiverHandle;

    if( hRfdr != NULL ) {

		/////////////////////////////////////////////////////
		// Close all thread handles, event handles and free memory allocations.
		/////////////////////////////////////////////////////

		//////////////
		// Free objects under hRfdr->rfdrThreadManagementInfo
		//////////////
		manageInfo = &hRfdr->rfdrThreadManagementInfo;

		// objects common to all threads

		if(manageInfo->errorExitEvent != NULL) {
			RFD_CloseEvent(manageInfo->errorExitEvent);
			manageInfo->errorExitEvent = NULL;
		}

		// Processor Thread objs
        if(hRfdr->isAltSingleThreadMode == FALSE) {
            // Close thread only in standard mode, not the alt-single-thread mode.
		    if(manageInfo->processorThread != NULL) {
			    RFD_CloseThread(manageInfo->processorThread);
			    manageInfo->processorThread = NULL;
		    }
        }

		if(manageInfo->rfdProcessorThreadInfo != NULL) {
			RFD_FREE(manageInfo->rfdProcessorThreadInfo);
			manageInfo->rfdProcessorThreadInfo = NULL;
		}

		for(i=0;i<RFDR_NUM_CLIENTS;i++) {

			if(manageInfo->rfdConsumerThreadInfo[i] != NULL) {
				RFD_FREE(manageInfo->rfdConsumerThreadInfo[i]);
				manageInfo->rfdConsumerThreadInfo[i] = NULL;
			}

			// Collector Thread objs
            if(hRfdr->isAltSingleThreadMode == FALSE) {
                // Close thread only in standard mode, not the alt-single-thread mode.
			    if(manageInfo->collectorThread[i] != NULL) {
				    RFD_CloseThread(manageInfo->collectorThread[i]);
				    manageInfo->collectorThread[i] = NULL;
			    }
            }

			if(manageInfo->rfdCollectorThreadInfo[i] != NULL) {
				RFD_FREE(manageInfo->rfdCollectorThreadInfo[i]);
				manageInfo->rfdCollectorThreadInfo[i] = NULL;
			}
		}


		//////////////
		// Free objects for RFDR File Consumer.
		//////////////
		if(hRfdr->rfdrClientSet != NULL) {

			for(i=0;i<RFDR_NUM_CLIENTS;i++) {

				consumerInfo = &hRfdr->rfdrClientSet->consumerInfo[i];

				if(consumerInfo->fileSetReadyEvent != NULL) {
					RFD_CloseEvent(consumerInfo->fileSetReadyEvent);
					consumerInfo->fileSetReadyEvent = NULL;
				}
			}

			//////////////
			// Free objects for RFDR Collector
			//////////////
			for(i=0;i<RFDR_NUM_CLIENTS;i++) {

				collectorInfo = &hRfdr->rfdrClientSet->collectorInfo[i];

				if(collectorInfo->messageInfo != NULL) {
					RFD_FREE(collectorInfo->messageInfo);
					collectorInfo->messageInfo = NULL;
				}

				if(collectorInfo->exitRequestEvent != NULL) {
					RFD_CloseEvent(collectorInfo->exitRequestEvent);
					collectorInfo->exitRequestEvent = NULL;
				}

				// Cleanup the message queue.
				if(collectorInfo->messageQueue != NULL) {
					RFD_MSG_QUEUE_Delete( collectorInfo->messageQueue);
				}
			}

			//////////////
			// Free objects common for each client
			//////////////
			for(i=0;i<RFDR_NUM_CLIENTS;i++) {

				clientInfo = &hRfdr->rfdrClientSet->clientInfo[i];

				if(clientInfo->clientMutex != NULL) {
					RFD_CloseMutex(clientInfo->clientMutex);
					clientInfo->clientMutex = NULL;
				}
				if(clientInfo->clientEvent != NULL) {
					RFD_CloseEvent(clientInfo->clientEvent);
					clientInfo->clientEvent = NULL;
				}
			}

			// free hRfdr->rfdrClientSet after all 'child' objects are free'd.\/closed
			RFD_FREE(hRfdr->rfdrClientSet);
			hRfdr->rfdrClientSet = NULL;

		} // if(hRfdr->rfdrClientSet != NULL)


		//////////////
		// Free objects specific to the RFDR Processor
		//////////////
		if(hRfdr->rfdrProcessorInfo != NULL) {

			if(hRfdr->rfdrProcessorInfo->exitRequestEvent != NULL) {
				RFD_CloseEvent(hRfdr->rfdrProcessorInfo->exitRequestEvent);
				hRfdr->rfdrProcessorInfo->exitRequestEvent = NULL;
			}
			if(hRfdr->rfdrProcessorInfo->initCompleteEvent != NULL) {
				RFD_CloseEvent(hRfdr->rfdrProcessorInfo->initCompleteEvent);
				hRfdr->rfdrProcessorInfo->initCompleteEvent = NULL;
			}


			// free hRfdr->rfdrProcessorInfo after all 'child' objects are free'd.\/closed
			RFD_FREE(hRfdr->rfdrProcessorInfo);
			hRfdr->rfdrProcessorInfo = NULL;

		} // if(hRfdr->rfdrProcessorInfo != NULL)

		//////////////
		// Free the main RFDR Handle last
		//////////////
		RFD_FREE(hRfdr);
		hRfdr = NULL;
	}

	RFD_DPrint( TEXT("RFD Receiver Cleanup Complete\n"));

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Start the RFDR  (RFD Receiver Module).
///
/// All threads of the RFDR are started.
///
/// @param  rfdReceiverHandle   Handle of the RFDR.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_StartReceiver(RFDR_HANDLE rfdReceiverHandle)
{
    RFD_STATUS status;

    if( rfdReceiverHandle == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    if(rfdReceiverHandle->isRunning == TRUE) {
		RFD_DPrint( TEXT("RFD-Receiver: Error, Receiver already started\n"));
        return RFD_STATUS_ERROR_INVALID_STATE;
    }

    if(rfdReceiverHandle->isAltSingleThreadMode) {
        status = _RFD_StartReceiverAltSingleThreadMode(rfdReceiverHandle);
    }
    else {
        status = _RFD_StartReceiverStandardMultiThreadMode(rfdReceiverHandle);
    }

    if(status == RFD_STATUS_OK) {
        // update run state
        rfdReceiverHandle->isRunning = TRUE;
    }

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Stop the RFDR (RFD Receiver Module).
///
/// @param	rfdReceiverHandle	Handle of the RFDR.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_StopReceiver(RFDR_HANDLE rfdReceiverHandle)
{
    RFD_STATUS status;

    if( rfdReceiverHandle == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    if(rfdReceiverHandle->isRunning == FALSE) {
		RFD_DPrint( TEXT("RFD-Receiver: Stop not performed because Receiver was not started\n"));
        return RFD_STATUS_ERROR_INVALID_STATE;
    }

    if(rfdReceiverHandle->isAltSingleThreadMode) {
        status = _RFD_StopReceiverAltSingleThreadMode(rfdReceiverHandle);
    }
    else {
        status = _RFD_StopReceiverStandardMultiThreadMode(rfdReceiverHandle);
    }

    // update run state
    rfdReceiverHandle->isRunning = FALSE;

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Check for RFDR (RFD Receiver Module) error.
///
/// The function will block, waiting for an error event indication, for \a timeOutMilliSeconds
/// milliseconds.
///
/// @param  rfdReceiverHandle   Handle of the RFDR.
/// @param  timeOutMilliSeconds The time-out in milliseconds. Value RFD_INFINITE to wait
///                             indefinitely. Value 0 to return immediately.
///
/// @return
/// RFD_STATUS_OK if there is no error, RFD_STATUS_ERROR_THREAD_EXIT if there is an RFDR error,
/// other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_ReceiverErrorCheck(RFDR_HANDLE rfdReceiverHandle, INT32 timeOutMilliSeconds)
{
	RFD_STATUS status;
	RFDR_HANDLE	hRfdr;

	hRfdr = rfdReceiverHandle;
    if( hRfdr == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

	status = RFD_WaitForEvent( hRfdr->rfdrThreadManagementInfo.errorExitEvent,    // handle to the error exit event
							   timeOutMilliSeconds);		// timeout

	if(status == RFD_STATUS_OK) {
		RFD_DPrint( TEXT("RFDR Manager: A thread signalled an error exit\n"));
		status = RFD_STATUS_ERROR_THREAD_EXIT;
	}
	else {
		status = RFD_STATUS_OK;
	}

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Get the Collector Thread Data Handle for the \a clientIndex client index.
///
/// The Collector Thread Data Handle is then the handle used in subsequent calls to the RFDR
/// Output Input functions.
///
/// @param  rfdReceiverHandle   Handle of the RFDR.
/// @param  clientIndex         Zero-based index of the RFDR Client.
///
/// @return the Collector Thread Data Handle for the specified client index.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_COLLECTOR_THREAD_DATA_HANDLE  RFD_GetMessageCollectorHandle(RFDR_HANDLE rfdReceiverHandle, int clientIndex)
{
	return rfdReceiverHandle->rfdrThreadManagementInfo.rfdCollectorThreadInfo[clientIndex];
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Get the Consumer Thread Data Handle for the \a clientIndex client index.
///
/// The Consumer Thread Data Handle is then the handle used in subsequent calls to the RFDR
/// Output API functions.
///
/// @param  rfdReceiverHandle   Handle of the RFDR.
/// @param  clientIndex         Zero-based index of the RFDR Client.
///
/// @return the Consumer Thread Data Handle for the specified client index.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_CONSUMER_THREAD_DATA_HANDLE  RFD_GetFileConsumerHandle(RFDR_HANDLE rfdReceiverHandle, int clientIndex)
{
	return rfdReceiverHandle->rfdrThreadManagementInfo.rfdConsumerThreadInfo[clientIndex];
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Set the Address of the Exit Request Flag.
///
/// In addition to setting the exit request (thread synchronization) event, the application's
/// call to RFD_StopReceiver() also sets the exit request flag (pointed to by this provided
/// address). During long running decoding/processing operations, the RFD Receiver polls this
/// address to check for exit request to then terminate decoding/processing and avoid any long
/// response latency to the exit request. Use of this function to specify this external exit
/// request address may be useful for RFDR running in the "Alt Single Threaded Mode" since the
/// application management of RFDR (e.g. module calling RFD_StopReciever() is typically running
/// in the same thread context as RFDR (thus the manager may be blocked for a long time from
/// responding to exit requests). But this external external exit request flag may be set from
/// some other thread or process (using shared memory) so that RFDR can more quickly respond to
/// the exit request. Calling this RFD_SetExitRequestFlagAddress() function is optional. If not
/// called, the RFD Receiver uses an internal default address for this exit request flag.
///
/// @param  rfdReceiverHandle   Handle of the RFDR.
/// @param  exitRequestFlagPtr  Address of the exit request flag provided by the caller.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_SetExitRequestFlagAddress(
    RFDR_HANDLE rfdReceiverHandle,
    UCHAR * exitRequestFlagPtr
    )
{
    if(exitRequestFlagPtr == NULL) {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    if(rfdReceiverHandle->isRunning == TRUE) {
        return RFD_STATUS_ERROR_INVALID_STATE;
    }

    rfdReceiverHandle->rfdrProcessorInfo->exitRequestFlagPtr = exitRequestFlagPtr;
    // Set initial value to false.
    *rfdReceiverHandle->rfdrProcessorInfo->exitRequestFlagPtr = FALSE;

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Do a "Run Loop" of the RFDR (RFD Receiver Module) to complete all pending work.
///
/// This function is relevant only if RFDR is opened in the "Alternate Single Threaded" operating mode.
/// Calling this funtion will run through all the RFDR "run objects" until all objects are idle
/// (until no work is pending).
///
/// @param	rfdReceiverHandle	        Handle of the RFDR.
/// @param	isThreadExitRequestedPtr	(output) Thread (Run Object) exit is requested.
///                                     Could be due to error condition a Run Object or
///                                     Run Object detected exit request.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_RunReceiver(
    RFDR_HANDLE rfdReceiverHandle,
    BOOL *isThreadExitRequestedPtr
    )
{
    RFD_STATUS status;
    RFD_STATUS statusLL;
	RFDR_THREAD_MANAGEMENT_STRUCT * manageInfo;
    int i;
    int nClients = RFDR_NUM_CLIENTS;
    int idleClientCount = 0;
    BOOL isIdle = FALSE;
    BOOL isThreadExitRequested = FALSE;

    if( rfdReceiverHandle == NULL ) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    if(rfdReceiverHandle->isRunning == FALSE) {
		RFD_DPrint( TEXT("RFD-Receiver: Error, RFDR is not in run state\n"));
        return RFD_STATUS_ERROR_INVALID_STATE;
    }

    if(!rfdReceiverHandle->isAltSingleThreadMode) {
		RFD_DPrint( TEXT("RFD-Receiver: Error, RFD_RunReceiver() cannot be called in standard Multi-Threaded mode\n"));
        return RFD_STATUS_ERROR_INVALID_MODE;
    }

	manageInfo = &rfdReceiverHandle->rfdrThreadManagementInfo;

    /////////////////////////////////////////////////////////////
    // 1. Run each Collector client once
    // 2. Run the Processor to completion of it's current work.
    // 3. If all Collector clients were idle in step 1,
    //    then exit loop.
    //    Else (at least one client may have work to do)
    //    repeat from step 1.
    /////////////////////////////////////////////////////////////
    do {

        ////////////////////////////////////
        // Run each Collector
        ////////////////////////////////////
        idleClientCount = 0;
        for(i=0;i<nClients;i++) {

            RFDR_MsgCollectorRunLL(
                manageInfo->rfdCollectorThreadInfo[i],
                &status,
                &isThreadExitRequested,
                &isIdle );

             if(isThreadExitRequested) {
                break;
             }

             if(isIdle) {
                idleClientCount++;
             }
        }
        if(isThreadExitRequested) {
            break;
        }

        ////////////////////////////////////
        // Run the Processor to completion of work
        // after each run of the Collectors.
        //
        // Even if all Collectors are idle at this point,
        // still run the Processor to handle the case of the first
        // run after powering-up; this is because there may be
        // work pending for the Processor from the previous
        // power cycle.
        ////////////////////////////////////
        status = RFDR_ProcessorRunLL(
                    manageInfo->rfdProcessorThreadInfo,
                    &statusLL,
                    &isThreadExitRequested,
                    TRUE // run until idle
                    );

        if(isThreadExitRequested) {
            break;
        }

        if(idleClientCount == nClients) {
            // All clients Collectors are idle, and the
            // Processor been run so exit the run loop.
            break;
        }

    } while(TRUE);

    *isThreadExitRequestedPtr = isThreadExitRequested;

    return status;
}

///
/// @} // doxygen group end symbol, don't remove.
///
