////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_receiver\rfdr_file_consumer.c
///
/// @brief	rfdr file consumer class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2012 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd_receiver.h"
#include "rfd_file_consumer.h"

///////////////////////////////////////////////////////////////////////////////////////////////////
/// @addtogroup RfdrOutputApi RFDR Output API
/// @ingroup RFDR_API
///////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief Read the rfd consumer info file.
///
/// Reads contents of the rfd consumer info file specified by the fileNamefileId file id into a
/// RFD_CONSUMER_FILE_INFO_STRUCT pointed to by fileInfoPtr.
///
/// @param  consumerPathBuf             Path name to where the info file is located.
/// @param  fileNamefileId              File Id identifying the info file.
/// @param  doDataFileCrossValidation   true to do cross validation with corresponding data file.
/// @param  fileInfoPtr                 (output) The file information structure pointer.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS RFD_ReadConsumerFileInfo(
    TCHAR * consumerPathBuf,
    UINT16 fileNamefileId,
    BOOL doDataFileCrossValidation,
    RFD_CONSUMER_FILE_INFO_STRUCT * fileInfoPtr )
{
    RFD_STATUS status;
    TCHAR * consumerFileNameBuf;
    INT32 actualReadDataSize = 0;
    UINT32 actualNewFileSize = 0;
    size_t stringLen = 0;

    status = RFD_STATUS_ERROR_GENERAL;
    consumerFileNameBuf = NULL;
    fileInfoPtr->fileSize = 0;
    fileInfoPtr->lifeTime = 0;
    fileInfoPtr->name[0] = '\0';

    do { // do Once. Simple way to break out of
        // normal execution path upon error to cleanup and return

        //////////
        // Allocate memory for the file names
        //////////

        consumerFileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );

        if( consumerFileNameBuf == NULL ) {

            // Unrecoverable error
            status = RFD_STATUS_ERROR_MEM_ALLOC;
            break;
        }

        //////////
        // Read file info structure from the info file.
        //////////

        // compose the consumer info file name string
        RFD_SPRINTF_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
            consumerPathBuf,
            RFD_PATH_SEPARATOR,
            fileNamefileId,
            RFDR_CONSUMER_FID_TYPE_INFO_FILE_NAME_EXT);

        status = RFD_ReadFile( consumerFileNameBuf,
            fileInfoPtr, sizeof(RFD_CONSUMER_FILE_INFO_STRUCT),
            &actualReadDataSize,
            TRUE // do actual file size == requested size check
            );
        if(status != RFD_STATUS_OK) {

            // error reading file or unexpected file length.
            break;
        }

        //////////
        // Do basic validation of fields.
        //////////

        // Do basic check of file size field.
        if(fileInfoPtr->fileSize == 0 ) {
            // error, data size is 0 (could do more rigorous min/max file size
            // range check)
            status = RFD_STATUS_ERROR_INVALID_FILE_SIZE;
            break;
        }

        // Do basic length check of name string field.
        stringLen = RFD_STRNLEN(fileInfoPtr->name, RFD_FSET_NAME_MAX_LEN);
        if(stringLen > RFD_FSET_NAME_MAX_LEN) {
            // error, string length is out of range
            status = RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
            break;
        }

        //////////
        // If requested, do basic validation of by cross checking
        // actual data file size vs. data file size field from the info file:
        // (i.e. actual size of the data file = consumerFileInfo.fileSize)
        //////////
        if(doDataFileCrossValidation == TRUE) {

            // compose the consumer data file name string
            RFD_SPRINTF_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
                consumerPathBuf,
                RFD_PATH_SEPARATOR,
                fileNamefileId,
                RFDR_CONSUMER_FID_TYPE_DATA_FILE_NAME_EXT);

            status = RFD_GetFileSize(consumerFileNameBuf, &actualNewFileSize);
            if(status != RFD_STATUS_OK) {

                // error in getting file size
                break;
            }

            if(actualNewFileSize != fileInfoPtr->fileSize ) {
                // error, actual file size does not match expected file size,
                // or actual file size is 0 (could do more rigorous file size
                // range check.
                status = RFD_STATUS_ERROR_INVALID_FILE_SIZE;
                break;
            }
        }

        status = RFD_STATUS_OK;

    } while(0); // do just once

    //////////
    // Free memory for the file names
    //////////

    if(consumerFileNameBuf != NULL) {
        RFD_FREE(consumerFileNameBuf);
    }

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Determine the type of consumer item for the given file name.
///
/// @param	fileNameBuf	File name string.
/// @param	fileId		[output] file id of the evaluated file name. This output is relevant when
///                     the file name is determined to be one of the File-ID based consumer item files.
///
/// @return	the type of rfd consumer file, identified by one of the RFD_CONSUMER_ITEM_FILE_TYPE
/// 		enumeration values.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_CONSUMER_ITEM_FILE_TYPE RFD_InterpretConsumerItemFileName(TCHAR fileNameBuf[], UINT32 * fileId)
{
	int fileNameLen;
	TCHAR fileNameBase[RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_BASE_LEN+1];
	TCHAR fileNameExt[RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_EXT_LEN+1];
	RFD_CONSUMER_ITEM_FILE_TYPE itemType;

    *fileId = 0;

    /////////////////
    // First check if this file is the File Set reset request file.
    /////////////////
	if(!RFD_STRCMP(fileNameBuf, RFD_CONSUMER_FSET_RESET_REQ_FNAME)) {
        // Found the FileSet (File Collect) Reset Request file.
		itemType = RFD_CONSUMER_ITEM_FSET_RESET_REQUEST_FILE;
    }
    else {
        /////////////////
        // Next determine if filename is that of one of the File-ID based item files.
        /////////////////

	    // validate the length of the file name.
	    fileNameLen = (int) RFD_STRNLEN(fileNameBuf, RFD_MAX_PATH_LEN);

	    if(fileNameLen != RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_LEN) {
		    // invalid file name length
		    return RFD_CONSUMER_ITEM_UNDEFINED_FILE;
	    }

	    // Convert the base file name string to integer and output to *fileId output parameter.
	    RFD_MEMCPY(fileNameBase, fileNameBuf,
			       RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_BASE_LEN * sizeof(TCHAR));
	    fileNameBase[RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_BASE_LEN] = '\0';

	    *fileId = RFD_ATOI(fileNameBase);

	    // Determine the File-ID based Consumer Item file type based on the file extension.
	    RFD_MEMCPY(fileNameExt, &fileNameBuf[RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_BASE_LEN],
			       RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_EXT_LEN * sizeof(TCHAR));
	    fileNameExt[RFD_CONSUMER_FID_TYPE_ITEM_FILE_NAME_EXT_LEN] = '\0';

	    if(!RFD_STRCMP(fileNameExt, RFDR_CONSUMER_FID_TYPE_DATA_FILE_NAME_EXT)) {
		    itemType = RFD_CONSUMER_ITEM_FID_TYPE_DATA_FILE;
	    }
	    else if(!RFD_STRCMP(fileNameExt, RFDR_CONSUMER_FID_TYPE_INFO_FILE_NAME_EXT)) {
		    itemType = RFD_CONSUMER_ITEM_FID_TYPE_INFO_FILE;
	    }
	    else if(!RFD_STRCMP(fileNameExt, RFDR_CONSUMER_FID_TYPE_TRANS_IN_PROGRESS_FILE_NAME_EXT)) {
		    itemType = RFD_CONSUMER_ITEM_FID_TYPE_TIP_FILE;
	    }
	    else if(!RFD_STRCMP(fileNameExt, RFDR_CONSUMER_FID_TYPE_HANDOVER_IN_PROGRESS_FILE_NAME_EXT)) {
		    itemType = RFD_CONSUMER_ITEM_FID_TYPE_HIP_FILE;
	    }
	    else {
		    // file name extension not match any of the expected file name extensions.
		    itemType = RFD_CONSUMER_ITEM_UNDEFINED_FILE;
	    }
    }

	return itemType;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Cleanup Rfd consumer storage, removing invalid files and checking for files with unfinished
/// handover status.
///
/// Files identified as valid are unchanged. Files identified as invalid are removed. Also detect
/// if there is a file with an unfinished handover (ownership transfer to app not completed).
///
/// @param  pathBuf                         Directory path to the directory where consumer
///                                         maintains rfd output files.
/// @param  foundUnfinishedFileHandoverPtr  (output) (Pointer to) true if a file was found for
///                                         which transfer to the app was unfinished.
/// @param  unfinishedFileHandoverFileIdPtr (output) (Pointer to) file id identifying a file with
///                                         unfinished file handover, when
///                                         *foundUnfinishedFileHandoverPtr is true.
/// @param  foundFileSetResetRequestFilePtr (output) (Pointer to) true if the File Set Reset Request
///                                          was found.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS RFD_ConsumerCleanupStorage(
            TCHAR pathBuf[],
            BOOL * foundUnfinishedFileHandoverPtr,
            UINT16 * unfinishedFileHandoverFileIdPtr,
            BOOL * foundFileSetResetRequestFilePtr )
{
    RFD_STATUS status;
	RFD_FIND_FILE_HANDLE hSearch;
	RFD_FIND_FILE_DATA_INFO fileData;
	TCHAR * pathBuf2;
	TCHAR * fileNamePtr;
	BOOL startFileSearch;
	BOOL isError;
	BOOL isValidConsumerItem;
	int maxCleanupAttempts = RFD_CONSUMER_DIR_MAX_CLEANUP_ATTEMPTS;
	int cleanupAttemptCount = 0;
	UINT32 fileId;
	RFD_CONSUMER_ITEM_FILE_TYPE consumerFileType;
	RFD_CONSUMER_FILE_INFO_STRUCT consumerFileInfo;

    // Initialize output parameters - indicate no unfinished ownership handover has been found yet.
    *foundUnfinishedFileHandoverPtr = FALSE;
    *unfinishedFileHandoverFileIdPtr = 0;
    // Initialize output parameters - indicate no FileSet Reset Request file has been found yet.
    *foundFileSetResetRequestFilePtr = FALSE;

	pathBuf2 = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(pathBuf2 == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	startFileSearch = TRUE;
	while( startFileSearch &&
		  (cleanupAttemptCount++ < maxCleanupAttempts) ) {

		// find the first file/directory
		hSearch = RFD_FindFirstFile(pathBuf, &fileData, &isError);
		if(hSearch == NULL) {
			RFD_FREE(pathBuf2);
			if(isError) {
				return RFD_STATUS_ERROR_FILE_FIND;
			}
			else {
				// No error with null search handle indicates an empty directory,
				// which is a valid condition.
				return RFD_STATUS_OK;
			}
		}

		while(1)
		{
			fileNamePtr = RFD_FindDataGetFileNamePtr(&fileData);

			if(fileNamePtr != NULL && !RFD_IsReservedFileSystemFileName(fileNamePtr))
			{
				if(RFD_FindDataIsItemADirectory(&fileData))
				{
					////
					// Directories are not expected to be in the Consumer directory,
					// Remove the directory.
					// Deleting a file or dir may possibly effect the FindFile process,
					// so close it and then restart the FindFile process after the delete.
					////

					// First Close the current search.
					RFD_FindClose(hSearch);

					// build the new path for the directory to delete
					RFD_STRCPY_S(pathBuf2, RFD_MAX_PATH_LEN, pathBuf);
					RFD_STRCAT_S(pathBuf2, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
					RFD_STRCAT_S(pathBuf2, RFD_MAX_PATH_LEN, fileNamePtr);

					//////
					// delete the directory and contents
					//////
					RFD_DeleteDirectoryNonRecursive( pathBuf2 );

					RFD_DPrint( TEXT("Deleted Directory: %s\n"),pathBuf2);

					// break out of inner while loop to restart the FindFile sequence.
					break;
				}
				else // is not a directory (is a file)
				{
					/////////////////////
					// Check validity of this file and any corresponding files.
					/////////////////////

					isValidConsumerItem = TRUE;

					// Determine the type of consumer file based on parsing the file name.
					consumerFileType = RFD_InterpretConsumerItemFileName(fileNamePtr, &fileId);

					if(consumerFileType == RFD_CONSUMER_ITEM_UNDEFINED_FILE) {
						// Undefined file. Is not a consumer item file.
						// Delete this file.

						// First Close the current search.
						RFD_FindClose(hSearch);

						// build the new path for the file to delete
						RFD_STRCPY_S(pathBuf2, RFD_MAX_PATH_LEN, pathBuf);
						RFD_STRCAT_S(pathBuf2, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
						RFD_STRCAT_S(pathBuf2, RFD_MAX_PATH_LEN, fileNamePtr);

						RFD_DELETE_FILE(pathBuf2);
						// break out of inner while loop to restart the FindFile sequence.
						break;
					}
                    else if(consumerFileType == RFD_CONSUMER_ITEM_FSET_RESET_REQUEST_FILE) {
                        // Found the FileSet (File Collect) Reset Request file.
                        // This means that a file set reset request
                        // procedure from the previous power cycle was initiated,
                        // but possibly not completed, due to power-loss, etc.
                        // Report existence of the File Set reset request flag to caller.
                        // Do not delete the file.
                        *foundFileSetResetRequestFilePtr = TRUE;
                    }
					else if(consumerFileType == RFD_CONSUMER_ITEM_FID_TYPE_TIP_FILE) {
						// Error, Found a 'tip' file.
						// Tip file (transaction in progress file marker)
						// is only present if a file transaction did not
						// complete successfully.
						// Delete all consumer files corresponding to this file,
						// then delete the tip file.
						isValidConsumerItem = FALSE;
					}
					else if(consumerFileType == RFD_CONSUMER_ITEM_FID_TYPE_DATA_FILE) {
						// this item is a Data file.
						// Ensure that there is a corresponding Info file.

						// Compose the full file name path for the Info file.
						RFD_SPRINTF_S(pathBuf2, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
										pathBuf, RFD_PATH_SEPARATOR, fileId, RFDR_CONSUMER_FID_TYPE_INFO_FILE_NAME_EXT);

						if(!RFD_IsFile(pathBuf2)) {
							// Error, Info file Not found.
							// Set flag to delete this consumer file item.
							isValidConsumerItem = FALSE;
						}

						// If a handover-in-progress file is found for this corresponding data file,
                        // further validity checks will be performed.
					}
					else if(consumerFileType == RFD_CONSUMER_ITEM_FID_TYPE_INFO_FILE) {
						// this item is an Info file.
						// Ensure that there is a corresponding Data file.

						// Compose the full file name path for the Data file.
						RFD_SPRINTF_S(pathBuf2, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
										pathBuf, RFD_PATH_SEPARATOR, fileId, RFDR_CONSUMER_FID_TYPE_DATA_FILE_NAME_EXT);

						if(!RFD_IsFile(pathBuf2)) {
							// Error, Data file Not found.
							// Set flag to delete this consumer file item.
							isValidConsumerItem = FALSE;
						}

						// If a handover-in-progress file is found for this corresponding info file,
                        // further validity checks will be performed.
					}
					else if(consumerFileType == RFD_CONSUMER_ITEM_FID_TYPE_HIP_FILE) {
						// This item is a Handover-in-progress file.
                        //
                        // First check for a TIP file also existing. If TIP file exists, treat this like a TIP case instead of HIP case.
                        // If TIP doesn't exist,
						// Ensure that there is a corresponding Data file and Info file
						// that appear to be valid (correct sizes, etc.).

						// Compose the full file name path for the TIP file.
						RFD_SPRINTF_S(pathBuf2, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
										pathBuf, RFD_PATH_SEPARATOR, fileId, RFDR_CONSUMER_FID_TYPE_TRANS_IN_PROGRESS_FILE_NAME_EXT);

						if(RFD_IsFile(pathBuf2) == TRUE) {
							// A TIP file was found.
                            // This is the case whereby both a TIP and a HIP exist. Treat this case
                            // like the "TIP" case (delete all corresponding files) since the rfd file will be retransferred
                            // from the RFD Processor - see comments in RFD_ConsumerTransferNewFileExt() at point
                            // where both TIP and HIP exists.

                            // Set flag to delete this consumer file item.
							isValidConsumerItem = FALSE;
						}
                        else {
                            // Just the HIP file exists (no TIP), continue validating the data and info files.

						    // Compose the full file name path for the Data file.
						    RFD_SPRINTF_S(pathBuf2, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
										    pathBuf, RFD_PATH_SEPARATOR, fileId, RFDR_CONSUMER_FID_TYPE_DATA_FILE_NAME_EXT);

						    if(!RFD_IsFile(pathBuf2)) {
							    // Error, Data file Not found.
							    // Set flag to delete this consumer file item.
							    isValidConsumerItem = FALSE;
						    }

						    // Compose the full file name path for the Info file.
						    RFD_SPRINTF_S(pathBuf2, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
										    pathBuf, RFD_PATH_SEPARATOR, fileId, RFDR_CONSUMER_FID_TYPE_INFO_FILE_NAME_EXT);

						    if(!RFD_IsFile(pathBuf2)) {
							    // Error, Info file Not found.
							    // Set flag to delete this consumer file item.
							    isValidConsumerItem = FALSE;
						    }

                            // Read the File Info structure from the info file
                            // for the purpose of performing validity checking,
                            // including cross checking data file size info with
                            // actual data file size.
                            status = RFD_ReadConsumerFileInfo(
                                        pathBuf,
                                        fileId,
                                        TRUE, // do Data File Cross Validation
                                        &consumerFileInfo
                                        );
                            if(status != RFD_STATUS_OK) {
							    isValidConsumerItem = FALSE;
						    }

                            if(isValidConsumerItem) {
						        RFD_DPrint( TEXT("Found unfinished File Transfer (Ownership Handover), ID: %d.\n"),fileId);
                                // set output parameters to indicate finding of an unfinished file handover (ownership transfer)
                                // and the corresponding file id for file that was not handed-over to the app.
                                *foundUnfinishedFileHandoverPtr = TRUE;
                                *unfinishedFileHandoverFileIdPtr = fileId;
                            }
                        }
					}

					if(!isValidConsumerItem) {

						RFD_DPrint( TEXT("Found Invalid Consumer Item file, ID: %d, cleaning up corresponding files\n"),fileId);

						/////////////////////
						// Invalid Item.
						// Delete all consumer files corresponding to this Id,
						// then finally delete the TIP file.
						/////////////////////

						// First Close the current search.
						RFD_FindClose(hSearch);

						//////
						// Delete the Info file if it exists.
						//////

						RFD_SPRINTF_S(pathBuf2, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
										pathBuf, RFD_PATH_SEPARATOR, fileId, RFDR_CONSUMER_FID_TYPE_INFO_FILE_NAME_EXT);

						if(RFD_IsFile(pathBuf2)) {
							RFD_DELETE_FILE(pathBuf2);
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
                            // Sync directory after every delete for consistent ordering of file deletions
                            // as recorded on persistent media, in the case of power loss.
                            RFD_DirectorySync(pathBuf);
#endif
						}

						//////
						// Delete the Data file if it exists.
						//////

						RFD_SPRINTF_S(pathBuf2, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
										pathBuf, RFD_PATH_SEPARATOR, fileId, RFDR_CONSUMER_FID_TYPE_DATA_FILE_NAME_EXT);

						if(RFD_IsFile(pathBuf2)) {
							RFD_DELETE_FILE(pathBuf2);
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
                            // Sync directory after every delete for consistent ordering of file deletions
                            // as recorded on persistent media, in the case of power loss.
                            RFD_DirectorySync(pathBuf);
#endif
						}


						//////
						// Delete the HIP file if it exists.
                        // Delete the HIP before the TIP file to guard against power loss right after deleting the TIP fle.
						//////

						RFD_SPRINTF_S(pathBuf2, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
										pathBuf, RFD_PATH_SEPARATOR, fileId, RFDR_CONSUMER_FID_TYPE_HANDOVER_IN_PROGRESS_FILE_NAME_EXT);

						if(RFD_IsFile(pathBuf2)) {
							RFD_DELETE_FILE(pathBuf2);
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
                            // Sync directory after every delete for consistent ordering of file deletions
                            // as recorded on persistent media, in the case of power loss.
                            RFD_DirectorySync(pathBuf);
#endif
						}

						//////
						// Delete the TIP file if it exists.
						//////

						RFD_SPRINTF_S(pathBuf2, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
										pathBuf, RFD_PATH_SEPARATOR, fileId, RFDR_CONSUMER_FID_TYPE_TRANS_IN_PROGRESS_FILE_NAME_EXT);

						if(RFD_IsFile(pathBuf2)) {
							RFD_DELETE_FILE(pathBuf2);
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
                            // Sync the parent directory to ensure deletion of TIP file safely persisted on disk
                            RFD_DirectorySync(pathBuf);
#endif
						}

						// break out of inner while loop to restart the FindFile sequence.
						break;
					}
				}
			}

			//////
			// find the next file/dir
			//////
			if(!RFD_FindNextFile(hSearch,&fileData, &isError)) {
				// file/dir not found.
				if(!isError) {
					RFD_DPrint( TEXT("Cleanup processing complete for: %s\n"),pathBuf);
				}
				else {
					// some error occurred;
					RFD_DPrint( TEXT("FindNextFile: unexpeted error\n"));
					RFD_DPrint( TEXT("Directory removal processing complete for: %s\n"),pathBuf);
				}

				// search complete, exit loops to cleanup and return
				RFD_FindClose(hSearch);
				startFileSearch = FALSE; // to exit from outer loop.
				break; // to exit from inner loop.
			}
		}
	} // while(startFileSearch)

	RFD_FREE(pathBuf2);

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Begin the Transfer a newly available file from the RFDR (RFD Receiver) to the application
/// consumer directory.
///
/// This function performs the first step of transferring a \e ready file. A \e ready file is a
/// file that has been fully processed by RFDR and signaled as ready via the \b
/// RFD_ConsumerWaitForNewFileEvent() function. The ready file is moved from RFD Processor client
/// directory to the consumer directory. If requested, a handover-in-progress is marked (.hip
/// file created) to indicated that the overall transfer is still in progress (ownership not yet
/// fully transferred to the app). This is done as part of guaranteeing complete ownership
/// transfer of the file to the application, in the event of power loss or program abort.
///
/// The RFD Receiver Application calls this function to transfer a \e ready file from RFDR
/// ownership to application ownership. A \e ready file is a file that has been fully processed
/// by RFDR and signaled as ready via the \b RFD_ConsumerWaitForNewFileEvent() function.
///
/// Upon completion of the function, \a *isNewFileAvailablePtr = TRUE and return value of
/// RFD_STATUS_OK confirms that the file was sucessfully transferred. The file is transferred to
/// the \a newFileName file (this file name is an output of the function). Application relevant
/// metadata for the file is provided in \a newFileInfo.
///
/// @param  handle                      Handle of the Consumer Thread Data.
/// @param  isNewFileAvailablePtr       (output) Set to True gives confirmation that new file is
///                                     available, otherwise false.
/// @param  newFileInfoPtr              (output) information structure describing the new file.
/// @param  newFileNamefileIdPtr        (output) file id identifying the new file.
/// @param  markFileHandoverInProgress  Set to true if a handover-in-progress marker file is to
///                                     be created (part of ownership handover procedure).
///
/// @return
/// RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure. RFD_STATUS_OK
/// does \e not indicate that a file was transferred. It is possible due to asynchronous
/// operation that a \e ready file may no longer be available at the time this function is
/// called. Check \a *isNewFileAvailablePtr = TRUE as indication that a file was transferred.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS RFD_ConsumerTransferNewFileBegin(
    RFD_CONSUMER_THREAD_DATA_HANDLE handle,
    BOOL * isNewFileAvailablePtr,
    RFD_CONSUMER_FILE_INFO_STRUCT * newFileInfoPtr,
    UINT16 * newFileNamefileIdPtr,
    BOOL markFileHandoverInProgress )
{
	DWORD clientIdNo;
	DWORD clientIndex;
	RFD_STATUS status;
	BOOL isFileSetComplete;
    BOOL isNewFileAvailable;
	TCHAR * consumerFileNameBuf, * consumerPathBuf;
	TCHAR * clientFileNameBuf, * clientPathBuf;
	RFD_CONSUMER_FILE_INFO_STRUCT consumerFileInfo;
	RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo = handle->consumerInfo;
	RFDR_CLIENT_INFO_STRUCT * clientInfo = handle->clientInfo;
    UINT16 baseFileId;

	clientIdNo = clientInfo->clientIdNo;
	clientIndex = clientInfo->clientIndex;

	clientPathBuf = NULL;
	clientFileNameBuf = NULL;
	consumerPathBuf = NULL;
	consumerFileNameBuf = NULL;
    baseFileId = 0;

	// Initialize status to ok.
	status = RFD_STATUS_OK;
	// Initialize for file Not available.
	isNewFileAvailable = FALSE;
    // Initialize output parameters
    *isNewFileAvailablePtr = FALSE;
    *newFileNamefileIdPtr = 0;
	// Copy the consumerFileInfo as output to the caller.
    // Do NULL check on newFileInfoPtr for backwards compatibility with "legacy RFD_ConsumerTransferNewFile"
	if(newFileInfoPtr != NULL) {
		newFileInfoPtr->fileSize = 0;
        newFileInfoPtr->lifeTime = 0;
        newFileInfoPtr->name[0] = '\0';
	}

	RFD_DPrint( TEXT("RFDR Consumer: RFD_ConsumerTransferNewFileBegin() attempting file transfer to Consumer, clientIdNo %d\n"), clientIdNo);

	//////////////
	// Acquire Mutex
	//////////////
	RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);

	do { // do Once. Simple way to break out of
		 // normal execution path upon error to cleanup and return

		//////////
		// Allocate memory for the file names
		//////////

		clientPathBuf		= (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
		clientFileNameBuf	= (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
		consumerPathBuf		= (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
		consumerFileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );

		if( clientPathBuf		== NULL ||
			clientFileNameBuf	== NULL ||
			consumerPathBuf		== NULL ||
			consumerFileNameBuf == NULL ) {

			// Unrecoverable error
			status = RFD_STATUS_ERROR_MEM_ALLOC;
			break;
		}

		// Validate that file set is in the decompress complete state.
		isFileSetComplete = RFDR_IsFileSetInDecompressCompleteState(clientInfo);

		if(isFileSetComplete) {
			// A FileSet is ready for transfer.

			//////////////
			// The specific sequence of updating the FileSet directory
			// and the individual File Instance directories is done in support
			// of full power-loss resilience (RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE) i.e.
			// order in which Storage Transactions are marked Started and marked Completed
			// for the directories.
			// This sequence is compatible with or without RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
			// defined.
			//
			// In case of power-loss during this sequence, the following actions should be
			// taken on powerup to properly handle the power-loss:
			//
			// For RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE defined,
			//  On Powerup: (after restoring any directories from backup if transaction incomplete),
			//  a check must be made to the File Instance states. If at least one state is
			//  'All Complete', then any other states still in 'Decompress Complete' must be
			//  updated to 'All Complete' and written to storage.
			//
			// For RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE disabled,
			//  On Powerup: If any transactions are detected incomplete (for FileSet or any File Instances)
			//  then Processing must be reset for the entire FileSet.
			//////////////

            // baseFileId is the file id of the file (fragment) at the first index (0) of the file set.
            // The file that is output to the consumer will be named by this file id value.
            // This baseFileId at 0-index also names the process sub-dir where the file set result
            // (the rfd reconstructed file) is currently stored.
            // Note: baseFileId is read here while inside the rfd client mutex lock. It will
            // be used further below after releasing the muxtex lock (i.e. the
            // clientInfo->fileSetInfo.fileInfo[RFDR_FILE_INSTANCE_INDEX_0].id variable must
            // NOT be read outside of the muxtex lock).
            baseFileId = clientInfo->fileSetInfo.fileInfo[RFDR_FILE_INSTANCE_INDEX_0].id;

			// Consturct the path string for the consumer directory.
			RFD_STRCPY_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_FILE_CONSUMER_BASE_PATH);
			RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, consumerInfo->consumerDirName);

			// compose the client 'source' path string for this file. The file set result (rfd reconstructed file)
			// is stored in the process sub-dir of the first file instance (fragment) directory (index 0).
			RFDR_ConstructFragmentDirPathName(clientPathBuf, RFD_MAX_PATH_LEN, clientInfo, baseFileId);
			RFD_STRCAT_S(clientPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(clientPathBuf, RFD_MAX_PATH_LEN, RFDR_PROCESS_DIR_NAME); // in the process dir

			/////////
			// Mark start of storage transaction for the FileSet.
			/////////
			RFD_MarkStorageTransactionStart(clientPathBuf);

			///////////////////////////////////////////////////////////
			// Summary of transfer-to-consumer procedure
			// 1. Create the transaction-in-progress file to mark that this update is in progress.
			// 2. Move (Rename) the rfd data file to the consumer data file
			// 3. Create the corresponding Consumer Info file corresponding to this data file.
			// 4. Delete the transaction-in-progress file to mark that this update is complete.
			// 5. Update the state to All Complete for the FileSet.
			///////////////////////////////////////////////////////////


			/////////////////////////////
			// Create the transaction-in-progress file to mark that this update is
			// in progress.
			/////////////////////////////

			RFD_SPRINTF_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
							consumerPathBuf,
							RFD_PATH_SEPARATOR,
							baseFileId,
							RFDR_CONSUMER_FID_TYPE_TRANS_IN_PROGRESS_FILE_NAME_EXT);
			
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
			// Be conservative and sync the TIP file here to persitent media (file sync may not be
			// needed here since we just check for existence of this marker type file, not the data content).
#endif
			// Create the TIP file, with compile-time configured file sync setting.
			RFD_CreateFile(consumerFileNameBuf, IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
            // Sync parent directory here to guarantee, as recorded to persistent media (in case of power loss),
            // the ordering of file create/delete/rename operations.
            RFD_DirectorySync(consumerPathBuf);
#endif
			/////////////////////////////
			// Move (Rename) the rfd data file to the consumer data file
			/////////////////////////////

			// compose the consumer 'destination' data file name string
			RFD_SPRINTF_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
							consumerPathBuf,
							RFD_PATH_SEPARATOR,
							baseFileId,
							RFDR_CONSUMER_FID_TYPE_DATA_FILE_NAME_EXT);

			// compose the client 'source' data file name string
			RFD_STRCPY_S(clientFileNameBuf, RFD_MAX_PATH_LEN, clientPathBuf);
			RFD_STRCAT_S(clientFileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(clientFileNameBuf, RFD_MAX_PATH_LEN, RFDR_DECOMPRESS_OUTPUT_FNAME);


			// Delete consumer file before the rename in case one already exists with the same name.
			RFD_DELETE_FILE(consumerFileNameBuf);

			// rename (move) the file to the consumer
			if(RFD_RENAME(clientFileNameBuf, consumerFileNameBuf)) {
				// Unrecoverable error, signal error.
				// Do Not mark transaction Complete on error incase power-down occurs right after this.
				status = RFD_STATUS_ERROR_FILE_RENAME;
				break;
			}

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
            // Be conservative and sync the directory here to guarantee ordering of
            // Data File delete(may be older previous version) and rename occurs before the creation
            // of the corresponding Info File to avoid mismatch between an old Data File and newly
            // created Info File.
            RFD_DirectorySync(consumerPathBuf);
#endif
			/////////////////////////////
			// Create the corresponding Consumer Info file corresponding to this data file.
			/////////////////////////////

			// Update the consumerFileInfo structure.
			RFD_GetFileSize(consumerFileNameBuf, &consumerFileInfo.fileSize);
			consumerFileInfo.lifeTime = clientInfo->fileSetInfo.lifeTime;
			RFD_MEMCPY( consumerFileInfo.name, clientInfo->fileSetInfo.name,
						RFD_FSET_NAME_MAX_LEN * sizeof(TCHAR));

			// compose the consumer 'destination' info file name string
			RFD_SPRINTF_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
							consumerPathBuf,
							RFD_PATH_SEPARATOR,
							baseFileId,
							RFDR_CONSUMER_FID_TYPE_INFO_FILE_NAME_EXT);

			// Write the consumerFileInfo structure to the consumer info file,
			// with compile-time configured file sync setting.
			status = RFD_WriteFile( consumerFileNameBuf, 
									&consumerFileInfo, sizeof(RFD_CONSUMER_FILE_INFO_STRUCT),
									IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);

			if(status != RFD_STATUS_OK) {
				// Unrecoverable error, signal error.
				// status variable contains error code for return.
				// Do Not mark transaction Complete on error incase power-down occurs right after this.
				break;
			}

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
            // Sync parent directory here to guarantee, as recorded to persistent media (in case of power loss),
            // the ordering of file create/delete/rename operations.
            RFD_DirectorySync(consumerPathBuf);
#endif

            if(markFileHandoverInProgress == TRUE) {
			    /////////////////////////////
			    // Create the file-ownership-handover-in-progress file to mark that the ownership handover is in progress.
                //
                // Do this before marking the storage transaction complete, otherwise there will be a "hole" where
                // we don't transfer this file to the consumer/application if power loss occurs
                // (hole between marking transaction complete and marking ownership handover in progress).
                // However, the consequence of doing this is that if power loss occurs between these two "marks",
                // this same file will be transferred to twice to the consumer/application on the following power cycle.
                //
                // Also do this before deleting the transaction-in-progress file. By "overlapping" the two
                // files this way, we avoid a "hole" where neither marker file exists. For power loss during this hole,
                // the rfd data file and info file will exist in the consumer directory without any markers;
                // we will therefore not delete them as part of cleanup on the next power cycle. This would normally be ok
                // but there is a corner case whereby this results in a "storage usage leak" due to these file
                // instances being created but never signalled to the app. This corner case more likely to happen for the
                // disabled full power loss resilience build (RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE not defined)
                // where RFDR cleans up after this described power loss in the "hole" and starts recollecting the
                // same rfd file to finally reconstruct and retransfer the file. However, right before or anytime
                // during file collection and reconstuction, the uplink stops broadcast of this file (identified by specific file id(s))
                // and starts broadcasting a new file. Therefore the previous file is never completed to then retranfer
                // to overwrite the "stale" rfd file of same file id in the consumer directory and never signalled
                // to the app. So unless the application does any additional scanning of the consumer directory to cleanup
                // the previous rfd data file and info file stay in storage and cause the "storage usage leak".
                // This condition can also possibly occur for the RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE enabled build
                // even though the rfd data file is restored and re-signalled to the file consumer thread (application)
                // immediately on the next power cycle. This is because there is a very small time window here in which
                // if an rfd metadata file identifying a different/new file is recieved before the file consumer thread calls
                // RFD_ConsumerTransferNewFileExt() upon being signalled then the previous rfd data file is deleted in the
                // RFD Processing directory (but of course not the corresponding rfd data file in the consumer directory).
			    /////////////////////////////

			    RFD_SPRINTF_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
							    consumerPathBuf,
							    RFD_PATH_SEPARATOR,
							    baseFileId,
							    RFDR_CONSUMER_FID_TYPE_HANDOVER_IN_PROGRESS_FILE_NAME_EXT);

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
				// Be conservative and sync the HIP file here to persitent media (file sync may not be
				// needed here since we just check for existence of this marker type file, not the data content).
#endif
				// Create the HIP file, with compile-time configured file sync setting.
				RFD_CreateFile(consumerFileNameBuf, IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
                // Sync parent directory here to guarantee, as recorded to persistent media (in case of power loss),
                // the ordering of file create/delete/rename operations.
                RFD_DirectorySync(consumerPathBuf);
#endif
            }

			/////////////////////////////
            // Note: at this point, both the transaction-in-progress and file-ownership-handover-in-progress file
            // exist. Upon a powerloss, the RFD_ConsumerInitializeStorage() can treat this as the transaction-in-progress cleanup
            // case instead of the file-ownership-handover-in-progress cleanup case, i.e. delete the corresponding rfd files instead
            // of setting up for the second stage handover. This is because we know that the file will be re-transferred
            // again from the RFDR Processor (immediately for the RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE enabled case) so
            // we can avoid transferring the same file twice to the application/file-consumer.
			/////////////////////////////

			/////////////////////////////
			// Mark that the Consumer Data file and Consumer Info file update
			// is complete (part of unexpected power-down handling).
			/////////////////////////////

			// compose the consumer 'destination' transaction-in-progress file name string
			RFD_SPRINTF_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
							consumerPathBuf,
							RFD_PATH_SEPARATOR,
							baseFileId,
							RFDR_CONSUMER_FID_TYPE_TRANS_IN_PROGRESS_FILE_NAME_EXT);

			// Delete the transaction-in-progress file to mark that this update is
			// complete.
			RFD_DELETE_FILE(consumerFileNameBuf);

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
            // Sync parent directory here to guarantee, as recorded to persistent media (in case of power loss),
            // the ordering of file create/delete/rename operations.
            RFD_DirectorySync(consumerPathBuf);
#endif
			//////////////////////
			// Update state to All Complete for the FileSet.
			// The state variable for first file instance represents the state of the overall
			// file set.
			//////////////////////

			clientInfo->fileProcessInfo[RFDR_FILE_INSTANCE_INDEX_0]->state = RFDR_STATE_ALL_COMPLETE;

			// Write the state to file.
			status = RFD_FileProcessStateWriteToStorage(clientPathBuf, &clientInfo->fileProcessInfo[RFDR_FILE_INSTANCE_INDEX_0]->state);

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
            // Use the following RFD_MarkStorageTransactionComplete( , TRUE, TRUE) to manage the syncing for the above
            // file and it's parent directory.
#endif
			if(status != RFD_STATUS_OK) {
				// Unrecoverable error, signal error.
				// status variable contains error code for return.
				// Do Not mark transaction Complete on error incase power-down occurs right after this.
				break;
			}

			/////////////////
			// Mark Storage Transaction Complete for the FileSet.
			/////////////////
			RFD_MarkStorageTransactionComplete(clientPathBuf, TRUE);

#ifdef RFDR_ENABLE_FULL_PWR_LOSS_RESILIENCE
			/////////////////
			// Backup Store the FileSet directory.
			/////////////////
			status = RFD_BackupDirectoryContents(clientPathBuf);

			if(status != RFD_STATUS_OK) {
				// break out of normal processing sequence to cleanup.
				break;
			}
#endif
			/////////////////
			// indicate that file is available.
			/////////////////
			isNewFileAvailable = TRUE;
			status = RFD_STATUS_OK;
		}
		else { // isFileSetComplete is false
			// Event occurred, but state indicated decompress not complete.
			// It's possible that the processor signaled event but then a
			// new FileSet message was received that changed the state.
			// This is a valid scenario so it should not be treated as an error.

			// indicate that file is Not available to the caller.
			isNewFileAvailable = FALSE;
			status = RFD_STATUS_OK;
		}

	} while(0); // do just once

	//////////////
	// Release Mutex
	//////////////
	RFD_ReleaseMutex(clientInfo->clientMutex);

	//////////////
	// Update final values for output parameters.
	//////////////

    *isNewFileAvailablePtr = isNewFileAvailable;

    if(isNewFileAvailable) {
        // use file id value that is read while inside the rfd client mutex lock.
        *newFileNamefileIdPtr = baseFileId;
	    // Copy the consumerFileInfo as output to the caller.
        // Do NULL check on newFileInfoPtr for backwards compatibility with
        // "legacy RFD_ConsumerTransferNewFile"
	    if(newFileInfoPtr != NULL) {
		    *newFileInfoPtr = consumerFileInfo;
	    }
    }

	//////////
	// Free memory for the file names
	//////////

	if(clientPathBuf != NULL) {
		RFD_FREE(clientPathBuf);
	}
	if(clientFileNameBuf != NULL) {
		RFD_FREE(clientFileNameBuf);
	}
	if(consumerPathBuf != NULL) {
		RFD_FREE(consumerPathBuf);
	}
	if(consumerFileNameBuf != NULL) {
		RFD_FREE(consumerFileNameBuf);
	}

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Construct the File Set Reset Request file path and name string into the provided string
/// buffer.
///
/// @param  consumerInfo        Pointer to rfd file consumer info struct.
/// @param  filePathNameBuf     Pointer to string buffer into which the full path name string is
///                             written. This buffer is allocated by the caller.
/// @param  filePathNameBufLen  Length of the filePathNameBuf buffer.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_ConsumerConstructFSetResetReqFilePathName(
    RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo,
    TCHAR filePathNameBuf[],
    int filePathNameBufLen )
{
	RFD_SPRINTF_S(filePathNameBuf, filePathNameBufLen, TEXT("%s%s%s%s%s"),
			        RFD_FILE_CONSUMER_BASE_PATH,
			        RFD_PATH_SEPARATOR,
				    consumerInfo->consumerDirName,
			        RFD_PATH_SEPARATOR,
                    RFD_CONSUMER_FSET_RESET_REQ_FNAME );

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrOutputApi
///
/// @brief
/// Initialize Consumer storage directory for RFD files.
///
/// This function checks that the expected RFDR Consumer storage directory exists and cleans-up
/// any files it identifies as invalid. The RFD Receiver Application calls this function on
/// startup.
///
/// @param  handle  Handle of the Consumer Thread Data.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_ConsumerInitializeStorage(RFD_CONSUMER_THREAD_DATA_HANDLE handle)
{
	RFD_STATUS status;
	TCHAR * consumerPathBuf;
	RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo = handle->consumerInfo;
    BOOL foundUnfinishedFileTransfer;
    UINT16 unfinishedFileTransferFileId;
    BOOL foundFileSetResetRequestFile = FALSE;
	UINT32 nCorruptedDirectoryEntriesFound = 0;

    // Initialize 'found unfinished file transfer' flags to negative/false case as default
    consumerInfo->foundUnfinishedFileTransfer = FALSE;
    consumerInfo->unfinishedFileTransferFileId = 0;

	consumerPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(consumerPathBuf == NULL) {
		// Unrecoverable error
		return RFD_STATUS_ERROR_MEM_ALLOC ;
	}

	//////////////////////////////////////////
	// Check if consumer directory exists.
	// If not, create the directory e.g. for first time running.
	//////////////////////////////////////////

	// Compose the path string for the consumer directory without trailing path separator.
	RFD_STRCPY_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_FILE_CONSUMER_BASE_PATH);
	RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, consumerInfo->consumerDirName);

	if(!RFD_IsDirectory(consumerPathBuf)) {

		RFD_DPrint( TEXT("directory Not found, Creating: %s \n"),consumerPathBuf);

		// Compose the path string for the consumer directory WITH trailing path separator.
		RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);

		// Create the directory, use trailing path separator
		if(RFD_CREATE_DIR(consumerPathBuf)) {
			// Error
			status = RFD_STATUS_ERROR_DIRECTORY_CREATE;
			RFD_DPrint( TEXT("CreateDirectory failed, status (%d)\n"), status);
			RFD_FREE(consumerPathBuf);
			return status;
		}
	}

	// reCompose the path string for the consumer directory without trailing path separator.
	RFD_STRCPY_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_FILE_CONSUMER_BASE_PATH);
	RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, consumerInfo->consumerDirName);

#if !defined(RFDR_DISABLE_CLEANUP_FS_CORRUPTED_DIR_ENTRIES)
	/////////////
	// Try to cleanup anomalous corrupted files that might occur on some platforms.
	// These corrupted files may show as directory entries, but opening the file or
	// trying to get attributes of the file may fail. Trying to delete these files
	// through regular file system calls may also fail. Typical file validation checking
	// and any necessary cleanup in presence of these types of files may therefore
	// not successfully restore state or cleanup to initial state as expected.
	// 
	// So this specialized function is used to try to cleanup corrupted files of
	// this nature. It is called first, before any other validation, recovery or cleanup
	// steps that are part of the normal flow of application initialization.
	/////////////

	status = RFD_DeleteFileSysCorruptedDirectoryEntries(
				consumerPathBuf, 
				&nCorruptedDirectoryEntriesFound, 
				NULL,  /* there is no ancillary file to delete for consumer dir */ 
				FALSE, 
				TRUE,  /* go ahead and sync dir if invalid items found. */
				FALSE  /* no subdirs in consumer dir */ );

	if(status != RFD_STATUS_OK) {

		// The function reported failure for some reason, 
		// but keep going and perform the regular storage validation/cleanup.

		RFD_DPrint( TEXT("RFDR Processor: RFD_DeleteFileSysCorruptedDirectoryEntries() failed, status = %d, path = %s\n"), 
			status, consumerPathBuf);
	}

	if(nCorruptedDirectoryEntriesFound != 0) {
		// Found invalid/corrupted directory entry or entries and attempted their deletion. 
		// Keep going and perform the regular storage validation/cleanup.
		RFD_DPrint( TEXT("RFDR Processor: RFD_DeleteFileSysCorruptedDirectoryEntries() found corrupted file(s)"
			" and attempted cleanup, count = %d, path = %s\n"), 
			nCorruptedDirectoryEntriesFound, consumerPathBuf);
	}
#endif

	// Scan directory and cleanup any incomplete files
	// (cleanup does not mean delete all files in this case).
	status = RFD_ConsumerCleanupStorage(
                        consumerPathBuf,
                        &foundUnfinishedFileTransfer,
                        &unfinishedFileTransferFileId,
                        &foundFileSetResetRequestFile);

    if(status == RFD_STATUS_OK) {

        if(foundUnfinishedFileTransfer == TRUE) {
            // setup to complete the unfinished file transfer/handover to the app
            consumerInfo->foundUnfinishedFileTransfer = TRUE;
            consumerInfo->unfinishedFileTransferFileId = unfinishedFileTransferFileId;
        }

        if(foundFileSetResetRequestFile == TRUE) {
            // Re-Issue the File Set (File Collection) Reset Request
            RFD_ConsumerResetFileCollectionRequest(handle);
        }
    }
    else {
        // error reported
        RFD_DPrint( TEXT("RFD_ConsumerCleanupStorage() reports error, status (%d)\n"), status);
    }

	// free the file path buffer
	RFD_FREE(consumerPathBuf);

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrOutputApi
///
/// @brief
/// Wait for RFDR Consumer \e Output-File-Ready event.
///
/// The RFD Receiver Application calls this function to determine when a new RFD File has been
/// fully received and reconstructed and is ready for transfer to the application.
///
/// The function will block, waiting for a File Ready event indication, for \a
/// timeoutMilliseconds milliseconds.
///
/// After indication that a new file is ready, the application calls RFD_ConsumerTransferNewFile()
/// to transfer the file to application ownership.
///
/// @param  handle              Handle of the Consumer Thread Data.
/// @param  timeoutMilliseconds The time-out in milliseconds. Value RFD_INFINITE to wait
///                             indefinitely. Value 0 to return immediately.
///
/// @return
/// RFD_STATUS_OK if an Output File is Ready, RFD_STATUS_OBJECT_TIMEOUT is Output file is *not*
/// ready, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_ConsumerWaitForNewFileEvent(RFD_CONSUMER_THREAD_DATA_HANDLE handle,
												INT32 timeoutMilliseconds)
{
	RFD_STATUS status;

    if(handle->consumerInfo->foundUnfinishedFileTransfer == TRUE) {
        RFD_DPrint(
            TEXT("RFDR Consumer: RFD_ConsumerWaitForNewFileEvent() An unfinished File Transfer has been flagged, ID: %d. Signal that file is ready, clientIdNo %d\n"),
            handle->consumerInfo->unfinishedFileTransferFileId, handle->clientInfo->clientIdNo);
        status = RFD_STATUS_OK;
    }
    else {
	    status = RFD_WaitForEvent( handle->consumerInfo->fileSetReadyEvent,    // handle to the file set ready event
							   timeoutMilliseconds);  // timeout
    }

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrOutputApi
///
/// @brief
/// Transfer a newly available file from RFDR (RFD Receiver) to application ownership.
///
/// Transfer a newly available file from RFDR (RFD Receiver) to application ownership. The RFD
/// Receiver Application calls this function to transfer a ready file from RFDR ownership to
/// application ownership in a power loss resilient manner. A ready file is a file that has been
/// fully processed by RFDR and signaled as ready via the RFD_ConsumerWaitForNewFileEvent()
/// function. To complete the file ownership transfer procedure, this function calls the
/// application provided callback function passed in parameter consumerFileTransferCallback. This
/// callback function is invoked synchronously in the context of the same thread invoking the
/// RFD_ConsumerTransferNewFileExt function. Upon return from the callback function, the ready
/// file is considered fully owned by the application.
///
/// If a power loss or other type of program abort occurs before or during execution of the
/// application callback function however, then on the next power cycle, the file will still be
/// flagged as being under ownership by RFDR; therefore RFDR will again signal to the application
/// that the same file is ready (in application call to RFD_ConsumerWaitForNewFileEvent()). In
/// this case an ownership transfer will be performed again on the subsequent call to this
/// RFD_ConsumerTransferNewFileExt () function.
///
/// It is also possible for a power loss occurs after the application has finished processing the
/// file in the consumerFileTransferCallback function, but before the application has deleted or
/// moved the file or RFDR has marked the ownership transfer of the file as complete. In this
/// case, RFDR will again signal the same file as ready on the subsequent power cycle and a file
/// transfer will be attempted again. It is then the applications responsibility to either detect
/// that the same file (content) has already been processed and therefore avoid reprocessing, or
/// ensure that processing (or partially processing) the same file content two or more times has
/// no negative or corrupting results.
///
/// In some cases, a ready file may not be available at the time this function in invoked (e.g.
/// RFDR just detected a newly broadcast RFD file version and internally cleaned-up the previous
/// RFD file which had been signaled as ready). In this case, the function will still return with
/// RFD_STATUS_OK status, but the application callback function, passed in parameter
/// consumerFileTransferCallback, will not be called.
///
/// @param  handle                          Consumer Thread Data handle.
/// @param  consumerFileTransferCallback    An application provided callback function  that is
///                                         called from within this function as the final step of
///                                         transferring ownership of the ready file. The
///                                         application can read from the new file as needed for
///                                         processing it's content in this callback function; or
///                                         the application may mark, in some power loss
///                                         resilient manner, that this new file is ready for
///                                         processing elsewhere (e.g. in another thread). The
///                                         application implements this function with function
///                                         prototype compliant to
///                                         RFD_CONSUMER_FILE_TRANSFER_CALLBACK.
/// @param  consumerFileTransferCallbackArg An application provided argument that is passed in
///                                         the call to the consumerFileTransferCallback function.
///
/// @return
/// RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure. RFD_STATUS_OK
/// does *not* indicate that a file was transferred. It is possible due to asynchronous operation
/// that a *ready* file may no longer be available even after indication by
/// RFD_ConsumerWaitForNewFileEvent(). In the case that a ready file is not available, the
/// application callback function, consumerFileTransferCallback, will not be called.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_ConsumerTransferNewFileExt(
    RFD_CONSUMER_THREAD_DATA_HANDLE handle,
    RFD_CONSUMER_FILE_TRANSFER_CALLBACK consumerFileTransferCallback,
    void *consumerFileTransferCallbackArg )
{
    RFD_STATUS status;
    BOOL isNewFileAvailable;
    UINT16 newFileNamefileId;
	TCHAR * consumerFileNameBuf, * consumerPathBuf;
	RFD_CONSUMER_FILE_INFO_STRUCT consumerFileInfo;
	RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo = handle->consumerInfo;
    //INT32 actualReadDataSize = 0;
    //UINT32 actualNewFileSize = 0;

    isNewFileAvailable = FALSE;
    status = RFD_STATUS_ERROR_GENERAL;
	consumerPathBuf = NULL;
	consumerFileNameBuf = NULL;

	do { // do Once. Simple way to break out of
		 // normal execution path upon error to cleanup and return

		//////////
		// Allocate memory for the file names
		//////////

		consumerPathBuf	= (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
		consumerFileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );

		if( consumerPathBuf	== NULL ||
			consumerFileNameBuf == NULL ) {

			// Unrecoverable error
			status = RFD_STATUS_ERROR_MEM_ALLOC;
			break;
		}

		//////////
	    // Consturct the path string for the consumer directory.
		//////////
	    RFD_STRCPY_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_FILE_CONSUMER_BASE_PATH);
	    RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	    RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, consumerInfo->consumerDirName);

		//////////
	    // Check for unfinished File Transfer.
		//////////
        if(consumerInfo->foundUnfinishedFileTransfer == FALSE) {

		    //////////
		    // An unfinished File Transfer is Not flaged. Try to transfer a new file from the RFD processor
		    // to the consumer directory. This function performs the first "begin" stage of transferring
		    // ownership of the file.
		    //////////

            status = RFD_ConsumerTransferNewFileBegin(
                    handle,
                    &isNewFileAvailable,
                    &consumerFileInfo,
                    &newFileNamefileId,
                    TRUE // create the file handover-in-progress file.
                    );

		    //////////
            // On successful function call,
            // return status = RFD_STATUS_OK
            // and output parameter isNewFileAvailable = TRUE
            // (and output parameters consumerFileInfo and newFileNamefileId
            // describe the new file).
            //
	        // At this point, the new file is transferred to the consumer directory
            // and the "handoff-in-progress-file" is present, indicating the file
            // has not yet been handed over to the application (app callback not yet called).
            // On a power loss or program abort at this point, RFD Receiver will attempt
            // to re-transfer this file to the application on the following power cycle).
		    //////////
        }
        else {
            RFD_DPrint(
                TEXT("RFDR Consumer: RFD_ConsumerTransferNewFileExt(): An unfinished File Transfer has been flagged, ID: %d. Setup to re-transfer file, clientIdNo %d\n"),
                handle->consumerInfo->unfinishedFileTransferFileId, handle->clientInfo->clientIdNo);
		    //////////
	        // An unfinished File Transfer has been detected and flagged.
            // As indicated by presence of a handover-in-progress file, on a
            // previous power cycle a new file had been transferred to the consumer
            // directory, but ownership not handed over to the application.
            //
            // So skip the initial step of transferring file from RFD Receiver Processor
            // (file has been cleaned up in RFD Receiver Processor system already).
            // Instead, setup for the final consumer callback step, by doing basic validation
            // of the existing file and corresponding info file and initializing
            // file info to pass to the consumer callback.
		    //////////

            newFileNamefileId = consumerInfo->unfinishedFileTransferFileId;

            // Clear the pending/unfinished file handover so it's not attempted again.
            consumerInfo->foundUnfinishedFileTransfer = FALSE;
            consumerInfo->unfinishedFileTransferFileId = 0;

            // Read the File Info structure from file, performing validity checks.
            status = RFD_ReadConsumerFileInfo(
                        consumerPathBuf,
                        newFileNamefileId,
                        TRUE, // do Data File Cross Validation
                        &consumerFileInfo
                        );
            if(status != RFD_STATUS_OK) {

                // error reading file or unexpected file length.
                break;
            }

            isNewFileAvailable = TRUE;
            status = RFD_STATUS_OK;
        }

		//////////
	    // At this point, the data file should now have been transferred from
        // the RFD Receiver Processor to the consumer directory (and a corresponding
        // info file also created).
        // The remaining steps are to call the consumer callback function
        // and then delete the handover-in-progress file to complete the ownership
        // handover of the file to the consumer application.
		//////////

        if(status == RFD_STATUS_OK && isNewFileAvailable == TRUE) {

		    /////////////////////////////
		    // Complete the ownership handover of the file to the application.
		    /////////////////////////////

		    //////////////////////////////
		    // Call consumer file transfer callback
		    // - Only called if a new file is available and
		    //   with success status for previous operations.
		    // ///////////////////////////

		    // compose the consumer data file name string
		    RFD_SPRINTF_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
					        consumerPathBuf,
					        RFD_PATH_SEPARATOR,
						    newFileNamefileId,
					        RFDR_CONSUMER_FID_TYPE_DATA_FILE_NAME_EXT);

		    // call the consumer file transfer callback
            // This call is done outside of any rfd client mutex acquire/lock so as not to
            // block the rfd processor in case the application callback takes a long time.
            consumerFileTransferCallback(
                            consumerFileNameBuf,
                            &consumerFileInfo,
                            consumerFileTransferCallbackArg);

		    /////////////////////////////
		    // Mark that the transfer (file ownership handover) of the Consumer Data file
            // to the consumer application is complete (part of unexpected power-down handling).
		    /////////////////////////////

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
            // First Sync the parent directory here to guarantee, as recorded to persistent media (in case of power loss),
            // that any deletion of the Data File done in consumerFileTransferCallback() is
            // done before the following HIP file deletion.
            RFD_DirectorySync(consumerPathBuf);
#endif
		    // compose the consumer 'destination' file-ownership-handover-in-progress file name string
		    RFD_SPRINTF_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, TEXT("%s%s%05d%s"),
						    consumerPathBuf,
						    RFD_PATH_SEPARATOR,
						    newFileNamefileId,
						    RFDR_CONSUMER_FID_TYPE_HANDOVER_IN_PROGRESS_FILE_NAME_EXT);

		    // Delete the file-ownership-handover-in-progress file to mark that
            // this transfer is complete (full transfer and handover).
		    RFD_DELETE_FILE(consumerFileNameBuf);

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
            // Be conservative and sync directory after HIP file deletion (not critical here).
            RFD_DirectorySync(consumerPathBuf);
#endif
        }

    } while(0); // do just once

	//////////
	// Free memory for the file names
	//////////

	if(consumerPathBuf != NULL) {
		RFD_FREE(consumerPathBuf);
	}
	if(consumerFileNameBuf != NULL) {
		RFD_FREE(consumerFileNameBuf);
	}

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// (deprecated, use RFD_ConsumerTransferNewFileExt() instead) Transfer a newly available file
/// from RFDR (RFD Receiver) to application ownership.
///
///  Note: this function has been deprecated (was the original RFD_ConsumerTransferNewFile
/// function, renamed to RFD_ConsumerTransferNewFile_Deprecated). Applications should use the
/// RFD_ConsumerTransferNewFileExt() function, which performs a fully power loss resilient
/// ownership transfer of newly available files.
///
/// The RFD Receiver Application calls this function to transfer a \e ready file from RFDR
/// ownership to application ownership. A \e ready file is a file that has been fully processed
/// by RFDR and signaled as ready via the \b RFD_ConsumerWaitForNewFileEvent() function.
///
/// Upon completion of the function, \a *isNewFileAvailablePtr = TRUE and return value of
/// RFD_STATUS_OK confirms that the file was sucessfully transferred. The file is transferred to
/// the \a newFileName file (this file name is an output of the function). Application relevant
/// metadata for the file is provided in \a newFileInfo.
///
/// @param  handle                  Handle of the Consumer Thread Data.
/// @param  isNewFileAvailablePtr   (output) Set to True gives confirmation that new file is
///                                 available, otherwise false.
/// @param  newFileName             (output) Filename of the new file. The caller allocates this
///                                 buffer with  \a newFileNameBufMaxLen length.
/// @param  newFileNameBufMaxLen    maximum Length of the new file name buffer.
/// @param  newFileInfo             (output) information structure describing the new file.
///
/// @return
/// RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure. RFD_STATUS_OK
/// does \e not indicate that a file was transferred. It is possible due to asynchronous
/// operation that a \e ready file may no longer be available even after indication by \b
/// RFD_ConsumerWaitForNewFileEvent(). Check \a *isNewFileAvailablePtr = TRUE as indication that
/// a file was transferred.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_ConsumerTransferNewFile_Deprecated(
    RFD_CONSUMER_THREAD_DATA_HANDLE handle,
    BOOL * isNewFileAvailablePtr,
    TCHAR newFileName[], int newFileNameBufMaxLen,
    RFD_CONSUMER_FILE_INFO_STRUCT * newFileInfo )
{
    RFD_STATUS status;
    TCHAR * consumerPathBuf = NULL;
    UINT16 newFileNamefileId = 0;
    RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo = handle->consumerInfo;

	// Initialize status to ok.
	status = RFD_STATUS_OK;

	// Initialize for file Not available.
	*isNewFileAvailablePtr = FALSE;

	// Initialize new file name output to null string.
	if(newFileName != NULL) {
		newFileName[0] = '\0';
	}

	do { // do Once. Simple way to break out of
		 // normal execution path upon error to cleanup and return

        //////////////
        // Call lower level function to do the file transfer
        // from RFD Reciever Processor to the consumer directory.
        // Call with markFileHandoverInProgress = FALSE to preserve
        // deprecated/legacy behavior (the handover-in-progress file
        // is not created).
        //////////////

        status = RFD_ConsumerTransferNewFileBegin(
                handle,
                isNewFileAvailablePtr, // output - also passed to our caller
                newFileInfo,           // output - also passed to our caller
                &newFileNamefileId,    // output - used in this fcn to construct output data file name.
                FALSE // don't create the file handover-in-progress file (legacy behavior).
                );

        if( status == RFD_STATUS_OK &&
            *isNewFileAvailablePtr == TRUE ) {

            //////////////
            // Construct the data file name string (output to newFileName)
            //////////////

            // Preserve legacy behavior
            // - don't treate newFileName = NULL as error condition,
            //   just don't update string to it.

			if(newFileName != NULL) {

		        consumerPathBuf	= (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );

                if(consumerPathBuf == NULL) {
			        // Unrecoverable error
                    *isNewFileAvailablePtr = FALSE;
			        status = RFD_STATUS_ERROR_MEM_ALLOC;
			        break;
		        }

			    // Consturct the path string for the consumer directory.
			    RFD_STRCPY_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_FILE_CONSUMER_BASE_PATH);
			    RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			    RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, consumerInfo->consumerDirName);

			    // compose the consumer data file name string
			    RFD_SPRINTF_S(newFileName, newFileNameBufMaxLen, TEXT("%s%s%05d%s"),
						        consumerPathBuf,
						        RFD_PATH_SEPARATOR,
						        newFileNamefileId,
						        RFDR_CONSUMER_FID_TYPE_DATA_FILE_NAME_EXT);
            }
        }
    } while(0); // just once

    //////////////
    // free memory
    //////////////
	if(consumerPathBuf != NULL) {
		RFD_FREE(consumerPathBuf);
	}

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrOutputApi
///
/// @brief
/// Requests a reset of the RFD file collection process for the specied consumer (handle).
/// The request is also referred to as a FileSet Reset Request.
///
/// After collecting, reconstructing and transferring an RFD file to consumer, the RFD Receiver
/// normally will not immediately re-collect and re-process the same RFD file (still being broadcasted)
/// again; the RFD Receiver ignores RFD Block messages if corresponding RFD Metadata messages it receives
/// that matches the RFD Metadata of the last successfully reconstructed RFD File.
/// RFD_ConsumerResetFileCollectionRequest() resets the RFD file collection process such that the
/// RFR Reciever will re-collect and re-process the same RFD File that was just completed.
///
/// The request is asynchronous. The actual reset is performed when the RFDR Processor thread
/// recognizes the occurrence of the request event. Currently, there is no callback defined to indicate
/// completion of the reset to the consumer application.
///
/// The request is power-loss-resilient. If a power loss or other type of system abort occurs at the
/// point of this function return or later, before the asynchronous FileSet reset is completed,
/// then the reset of the file collection process will be completed automatically upon the following
/// power-cycle.
///
/// The consumer may use this API function anytime after RFD_ConsumerInitializeStorage() is called.
///
/// This API may be used by the consumer following occurrence of any
/// abnormal error condition as a method of attempting error recovery.
/// The consumer might call this function if an error is returned by any RFDR Consumer API,
/// e.g. by RFD_ConsumerTransferNewFileExt().
/// The consumer may also call this function to reset file collection so as to eventually
/// re-transfer the same RFD file if a newly transferred RFD file is determined to be corrupted
/// during the consumer application processing or use of the RFD file.
/// The developer should be cautious with this approach and set a limit on the number file collection
/// reset attempts of the same RFD file (e.g. file identified by Name field) if there are
/// constraints on Flash wear. This guards against the possibility of the consumer application
/// continuously requesting file collection reset after every RFD file transfer if there
/// is some anomalous incompatibility between the application and the content of a specific
/// RFD file (and the application interprets the incompatibility as a file content error).
///
/// For the purpose of RFD file corruption recovery described above,
/// RFD_ConsumerResetFileCollectionRequest() may be called from within the consumer implemented
/// File Transfer Callback function (RFD_CONSUMER_FILE_TRANSFER_CALLBACK) to ensure full
/// power loss resilience of the entire file transfer process in combination with file collection
/// reset request handling.
///
/// Note that provision of RFD_ConsumerResetFileCollectionRequest() functionality is not intented
/// for the purpose of guarding against RFD File content errors occurring due to internal
/// RFD Receiver processing, as this processing is intended to be highly fault tolerant through use of
/// CRC checks and other error checks; also, the RFD Receiver will internally invoke File Collection
/// resetting (FileSet Resetting) upon any internally detected RFD file processing errors.
///
/// A pseudocode example of Application-implemented File Collection reset requesting with maximum
/// number of reset attempts follows:
///
/// \verbatim
/// IF APPLICATION ERROR IN PROCESSING RFD FILE \n
///
///     ; read number of attempts so far for corresponding newFileName RFD file name field \n
///     ; from application file appMonitorAttemptsFileName (not in the consumer dir) \n
///    currAttempts = APP_READ_NUM_ATTEMPTS_FROM_FILE(newFileName, appMonitorAttemptsFileName)
///
///    IF currAttempts < APP_MAX_ATTEMPTS
///
///       ; Perform Reset Request
///       RFD_ConsumerResetFileCollectionRequest(callbackInfo->hConsumer->rfdFileConsumerHandle);
///
///       ; store number of reset attempts along with corresponding RFD newFileName field string *
///       APP_WRITE_NUM_ATTEMPTS_TO_FILE(currAttempts+1, newFileName, appMonitorAttemptsFileName)
///
///    ELSE
///
///       Print("Max Reset Attempts Reached (%d) %s", currAttempts, newFileName)
///
///    END IF
///
/// END IF
/// \endverbatim
///
/// @param  handle  Consumer Thread handle.
///
/// @return
/// RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_ConsumerResetFileCollectionRequest(RFD_CONSUMER_THREAD_DATA_HANDLE handle)
{
    RFD_STATUS status = RFD_STATUS_OK;
	RFDR_CLIENT_INFO_STRUCT * clientInfo = handle->clientInfo;
	RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo = handle->consumerInfo;
	DWORD clientIdNo;
	TCHAR * consumerFileNameBuf = NULL;
	TCHAR * consumerPathBuf = NULL;

	consumerPathBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if( consumerPathBuf == NULL ) {

		// Unrecoverable error
		status = RFD_STATUS_ERROR_MEM_ALLOC;
        return status;
	}

	consumerFileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if( consumerFileNameBuf == NULL ) {

		// Unrecoverable error
        RFD_FREE(consumerPathBuf);
		status = RFD_STATUS_ERROR_MEM_ALLOC;
        return status;
	}

    clientIdNo = clientInfo->clientIdNo;

	//////////////////////////////////////////
	// Acquire the Mutex to access shared data
	/////////////////////////////////////////
	RFD_AcquireMutex(clientInfo->clientMutex, RFD_INFINITE);
    {

	    /////////////////////////////////////////
        // Create File Collection Reset Request File to mark the pending reset request
        // in power-loss-resilient manner.
	    /////////////////////////////////////////

	    RFD_STRCPY_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_FILE_CONSUMER_BASE_PATH);
	    RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	    RFD_STRCAT_S(consumerPathBuf, RFD_MAX_PATH_LEN, consumerInfo->consumerDirName);

	    RFD_STRCPY_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, consumerPathBuf);
	    RFD_STRCAT_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	    RFD_STRCAT_S(consumerFileNameBuf, RFD_MAX_PATH_LEN, RFD_CONSUMER_FSET_RESET_REQ_FNAME);

	    /////////////////////////////////////////
        // Only create the reset request marker file and do the file/dir sync
        // if this Consumer Directory exists. This is in case this API is called
        // from threads other than the application Consumer thread which is the thread
        // responsible for creating and initializing this consumer directory. In this multi-thread
        // use case, it may be possible that the Consumer thread has not yet created this directory
        // when this API function is called (e.g. directory created on the very first run of
        // the Consumer for this configured RFD Client).
	    /////////////////////////////////////////
	    if(RFD_IsDirectory(consumerPathBuf)) {
			// Create the file
#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
            // with File Sync enabled, to be conservative and sync the FileSet reset request file here to persitent media
            // (file sync may not be needed here since we just check for existence of this marker type file,
            //  not the data content).
#endif
            RFD_CreateFile(consumerFileNameBuf, IS_RFDR_DIR_AND_FILE_SYNC_ENABLED);

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
            // Sync the parent directory after creating the file
            RFD_DirectorySync(consumerPathBuf);
#endif
        }
        else {
            // Directory doesn't exist yet.
            RFD_DPrint(
                TEXT("RFDR Consumer API: RFD_ConsumerResetFileCollectionRequest() ")
                TEXT("bypassing writing of reset request flag file because consumer dir does not yet exist, clientIdNo %d\n"),
                clientIdNo);
        }

	    /////////////////////////////////////////
	    // Set Flag and then Signal the Reset Request Event
	    /////////////////////////////////////////

	    RFD_DPrint(
            TEXT("RFDR Consumer API: Setting Event to request File Collection Reset (FSet Reset), clientIdNo %d\n"),
            clientIdNo);

        // Set Flag
        consumerInfo->fileSetResetRequest = TRUE;

        // Signal Event
	    RFD_SetEvent(clientInfo->clientEvent);

    }
	/////////////////////////////////////
	// Release Mutex
	/////////////////////////////////////
	RFD_ReleaseMutex(clientInfo->clientMutex);

    // Cleanup
    RFD_FREE(consumerPathBuf);
    RFD_FREE(consumerFileNameBuf);

    return status;
}
