////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_receiver\rfd_receiver.h
///
/// @brief	Declares the rfd receiver class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2012 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef RFD_RECEIVER_H
#define RFD_RECEIVER_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include "rfd_config.h"
#include "rfd_receiver_config.h"
#include "rfd_codec_xcc.h"
#include "rfd_codec_simple.h"
#include "rfd_message_queue.h"

/////////////////////////////////////////////////////
// IS_RFDR_DIR_AND_FILE_SYNC_ENABLED :
// This configuration is derived from the RFDR_DISABLE_DIRECTORY_AND_FILE_SYNC macro
// which is defined in "rfd_receiver_config.h". 
// Therefore, ensure that this below configuration is preceeded by #include "rfd_receiver_config.h"
/////////////////////////////////////////////////////
#ifdef RFDR_DISABLE_DIRECTORY_AND_FILE_SYNC
	#define IS_RFDR_DIR_AND_FILE_SYNC_ENABLED   0
#else
	#define IS_RFDR_DIR_AND_FILE_SYNC_ENABLED   1
#endif

/////////////////////////////////////////////////////
// RFDR_ENABLE_STORAGE_OPTIM_1 :
// Enables optimization of storage space usage so that only one fragment at a time
// has it's process directory backed-up during the decode stage.
// After a decode (chunk) operation, and decoding is incomplete (returning to the 'collect' state),
// and after the process directory is safely updated (transaction complete),
// the backup directory is cleared/cleaned-up. The backup is essentailly postoned until the
// begining of the next decode (after enough blocks collected for the
// next decode).
///////////////////////////
#define RFDR_ENABLE_STORAGE_OPTIM_1

/////////////////////////////////////////////////////
// The Block Message Carousel Id is dynamically specified to recevier
// by the File Set Metatdata message.
// -1 value indicates that the ID has not yet been set.
#define RFDR_CLIENT_INITIAL_BLOCK_MSG_CAROUSEL_ID	-1

/////////////////////////////////////////////////////
extern RFD_MESSAGE_CONFIG_INFO gMessageConfigInfo[RFDR_NUM_CLIENTS];
extern DWORD gFileConsumerId[RFDR_NUM_CLIENTS];
extern TCHAR * gFileConsumerDirName[RFDR_NUM_CLIENTS];

///////////////////////////
// Task/thread Stack Sizes
///////////////////////////
#define RFDR_PROCESSOR_THREAD_STACK_SIZE	8192
#define RFDR_COLLECTOR_THREAD_STACK_SIZE	8192
#define RFD_FILE_CONSUMER_THREAD_STACK_SIZE	8192

///////////////////////////
// Task/thread priorities
///////////////////////////
#define RFDR_PROCESSOR_THREAD_PRIORITY		RFD_THREAD_PRIORITY_LOWEST
#define RFDR_COLLECTOR_THREAD_PRIORITY		RFD_THREAD_PRIORITY_NORMAL

//////////////
// RFDR_MAX_NUM_DECODE_CHUNKS :
// Maximum number of allowable decode chunks (code type independent).
//////////////
#define RFDR_MAX_NUM_DECODE_CHUNKS 16

//////////////
// ECC code defines for Decode Event Block Collection Info.
//////////////

// Number of configured Decode Chunks for the ECC code.
// Range must be 1 to RFDR_MAX_NUM_DECODE_CHUNKS.
#define RFD_ECC_DECODE_NUM_DECODE_CHUNKS 4
// The denominator for the gEccDecodeChunkBlkThreshNumerator
#define RFD_ECC_DECODE_CHUNK_BLK_THRESH_DENOM 64
// The decode chunk step size (increment) for decoding after the minimum block
#define RFD_ECC_DECODE_EXTRA_BLKS_CHUNK_INC 2

#if (RFD_ECC_DECODE_NUM_DECODE_CHUNKS > RFDR_MAX_NUM_DECODE_CHUNKS)
#error "Number of configured ECC Decode Chunks exceeds maximum allowed"
#endif

//////////////
// Simple code defines for Decode Event Block Collection Info.
//////////////

// Number of configured Decode Chunks for the Simple code.
// Should always be set to 1. Decoding in more than one
// chunk step adds no gain in overall processing time.
#define RFD_SIMPLE_DECODE_NUM_DECODE_CHUNKS 1
// The denominator for the gSimpleCodeDecodeChunkBlkThreshNumerator
#define RFD_SIMPLE_DECODE_CHUNK_BLK_THRESH_DENOM 1
// The decode chunk step size (increment) for decoding after the minimum block
#define RFD_SIMPLE_DECODE_EXTRA_BLKS_CHUNK_INC 1

#if (RFD_SIMPLE_DECODE_NUM_DECODE_CHUNKS > RFDR_MAX_NUM_DECODE_CHUNKS)
#error "Number of configured Simple-Code Decode Chunks exceeds maximum allowed"
#endif


// The max length buffer that the receiver may allocate for data transfer.
#define RFDR_MAX_DATA_COPY_BUF_LEN 8192

// Size of each Message Queue Buffer, the input to each Collector Instance.
// Should be big enough to handle a 'burst' of block messaging.
// This could be made configurable per 'client' based on
// max file sizes and max 'instantaneous' bitrate for each client.
#define RFDR_MESSAGE_QUEUE_BUF_SIZE 20000

///////////////////////////////////////////////////////////////////////////////////////////////
// As a safety precaution, put a limit on the number of attempts at restarting the
// FindFile process as part of the consumer directory cleanup.
// e.g. each time an invalid file or directory is found and deleted the FindFile
// process is restarted. If the file/directory is not successfully deleted
// (e.g. special file attributes), the
// FindFile process will never end without a limit check.
//
#define RFD_CONSUMER_DIR_MAX_CLEANUP_ATTEMPTS 500

typedef enum
{
	RFDR_COLLECTOR_IN_MSG_RECEIVED = 0,
	RFDR_COLLECTOR_IN_MSG_TIMEOUT,
	RFDR_COLLECTOR_IN_MSG_ERROR

} RFDR_COLLECTOR_MSG_WAIT_STATUS;

/* Block Collecting Directory Name for each file instance (fragment)*/
#define RFDR_COLLECT_DIR_NAME TEXT("collect")

/* Processing Directory Name for each file instance (fragment) */
#define RFDR_PROCESS_DIR_NAME TEXT("process")

/* File Set Metadata filename */
#define RFDR_FSET_FNAME TEXT("fset.dat")

/* Used to indicate that file transaction is in progress (a file or set of files is being updated) */
#define RFDR_TRANSACTION_STATE_FNAME			TEXT("trans_st.dat")

/* Maintain state information about the overall processing i.e. the collection, decoding and decompression */
#define RFDR_PROCESS_STATE_FNAME				TEXT("proc_st.dat")

/* Maintain state information about the decoding */
#define RFDR_DECODE_STATE_FNAME					TEXT("dec_st.dat")

/* Maintain a 'private' version of state information about the decoding.
 * It's 'private' in that only the decoder operates on this version,
 * while the collector updates the 'main' version of the decoder state structure/file.
 * The main version of the structure is copied to the private version,
 * The decoder operates on the private version while the collector
 * operates on the main version (i.e. additional blocks can be collected during decode).
 * After a decoding operation, the main version is updated (consolidated) with the private version.
 * If a decoding operation is cancelled (e.g. power-down request),
 * a 'reversion' is done where the main version is also updated (consolidated)
 * with the private version.
 * This 'private decoder' version of the decoder state struction is stored (not just maintains in volotile memory)
 * in support of 'full power-loss resilience'.
 */
#define RFDR_DECODE_STATE_PRIVATE_FNAME			TEXT("dec_st.tmp")

/* Maintain configuartion information for the decoding */
#define RFDR_CODEC_CONFIG_FNAME					TEXT("cdec_cfg.dat")

/* The collector client adds source type blocks (up to a threshold) to this file buffer
 * if the source block has not already been collected, as indicated by the
 * RFDR_SRC_BLKS_COLLECTED_FNAME file/structure */
#define RFDR_SRC_BLKS_INBUF_FNAME				TEXT("src_blks.dat")
#define RFDR_SRC_BLKS_INBUF_TMP_DECODE_FNAME	TEXT("src_blks.tmp")

/* The collector client adds check-type (encoded) blocks (up to a threshold) to this file buffer
 * if the check block has not already been collected, as indicated by the
 * RFDR_CHK_BLKS_COLLECTED_FNAME file/structure */
#define RFDR_CHK_BLKS_INBUF_FNAME				TEXT("chk_blks.dat")
#define RFDR_CHK_BLKS_INBUF_TMP_DECODE_FNAME	TEXT("chk_blks.tmp")

/* Contains status of each row of the system of equations being transformed including
 * pivot index (if row is a pivot), is_reduced (to current pivot), is_pivot, is_linearly_dependent.
 */
#define RFDR_XCC_DEC_EQU_INFO_FNAME				TEXT("equ_info.dat")

/* The equation coefficient portion of the overall equation
 * (rows of the left side of the system of equations being transformed)
 */
#define RFDR_XCC_DEC_COEFF_TRANS_MATRIX_FNAME	TEXT("coef_mat.dat")

/* The data portion of the overall equation
 * (rows of the right side of the equation).
 * This file becomes the reconstructed data after decoding is complete.
 */
#define RFDR_XCC_DEC_BLOCK_TRANS_MATRIX_FNAME	TEXT("blk_mat.dat")

/* A list that indicates the source-type blocks that have already been collected.
 * The collector client adds a new source block index to the RFDR_SRC_BLKS_INBUF_FNAME
 * only if it hasn't already been collected. The collector client then adds
 * the new source block identifier to this file. This file is indexed by the
 * source block index.
 */
#define RFDR_SRC_BLKS_COLLECTED_FNAME			TEXT("src_list.dat")

/* A list that indicates the check-type (encoded) blocks that have already been collected.
 * The collector client adds a new check block identifier to the RFDR_CHK_BLKS_INBUF_FNAME
 * only if it hasn't already been collected. The collector client then adds
 * the new check block identifier to this file. This file is not indexed, it contains a list
 * of check block identifiers in the order in which they are collected.
 */
#define RFDR_CHK_BLKS_COLLECTED_FNAME			TEXT("chk_list.dat")

/* Name of the decoded output file.
 */
#define RFDR_DECODE_OUTPUT_FNAME				TEXT("dec_out.dat")

/* The name of the defragmented data file (composition of all files of the file set) */
#define RFDR_DEFRAGMENTED_DATA_FNAME			TEXT("defrag.dat")

/* Name of the decompressed output file.
 */
#define RFDR_DECOMPRESS_OUTPUT_FNAME			TEXT("decomp.dat")

/* The directory name for backup storage directories */
#define RFDR_BACKUP_DIR_NAME					TEXT("bak")

/* FILE_SET_DIR_NAME_SIZE, FILE_SET_DIR_NAME_SPRINT_FORMAT
 * Defines for the format of the File (Set) Directory Name
 * The directory name is the File ID zero-extended to fill 5 digits.
 * (max value = 99999 > 2^16).
 * Caution: some functions bypass these definitions and use hard coded values.
 */
#define FILE_SET_DIR_NAME_SIZE 5
#define FILE_SET_DIR_NAME_SPRINT_FORMAT			TEXT("%05d")

// RFDR_FILE_INSTANCE_INDEX_0 :
// The first File Instance (fragment), at Index = 0, serves as a special case instance.
// After Decode-Complete for all individual file instances, state for the first file instance
// serves as the state for the overall file set for all the remaining states - defrag, decompress, all complete.
// The corresponding first file instance directory also maintains all input/output (resulting) files for
// the final operations (defrag and decompress).
#define RFDR_FILE_INSTANCE_INDEX_0 0

#define RFDR_PROC_THREAD_INIT_TIMEOUT_MSEC 180000 // time for manager to wait for processor thread initialization complete.
#define RFDR_WAIT_ERROR_EXIT_TIMEOUT_MSEC 10	  // time for manager to wait for error exit event
#define RFDR_IN_MSG_TIMEOUT_MSEC 100			  // time for collector to wait for input message
#define RFDR_FILE_SET_READY_TIMEOUT_MSEC 100	  // time for consumer waiting for file set ready event
#define RFDR_PROCESSOR_WAIT_COLLECT_EVENT_TIMEOUT_MSEC 1000	// time for processor to wait for a collection event

#define RFDR_ECC_MAX_EXTRA_BLKS_FOR_SOLUTION 30
#define RFDR_SIMPLE_CODE_MAX_EXTRA_BLKS_FOR_SOLUTION 0

#define RFDR_ECC_MAX_NUM_COLLECTED_BLKS			(RFD_MAX_FILE_LEN_BLOCKS + RFDR_ECC_MAX_EXTRA_BLKS_FOR_SOLUTION)
#define RFDR_SIMPLE_CODE_MAX_NUM_COLLECTED_BLKS	(RFD_MAX_FILE_LEN_BLOCKS + RFDR_SIMPLE_CODE_MAX_EXTRA_BLKS_FOR_SOLUTION)


//  RFDR_MAX_NUM_COLLECTED_BLKS is the maximum number of collected blocks over all supported code types.
#if (RFDR_ECC_MAX_NUM_COLLECTED_BLKS > RFDR_SIMPLE_CODE_MAX_NUM_COLLECTED_BLKS)
	#define RFDR_MAX_NUM_COLLECTED_BLKS     RFDR_ECC_MAX_NUM_COLLECTED_BLKS
#else
	#define RFDR_MAX_NUM_COLLECTED_BLKS     RFDR_SIMPLE_CODE_MAX_NUM_COLLECTED_BLKS
#endif
#define RFD_ENC_BLKS_COLLECTED_LIST_MAX_ELEM     RFDR_MAX_NUM_COLLECTED_BLKS
#define RFD_SRC_BLKS_COLLECTED_STATUS_MAX_ELEM   RFD_MAX_NUM_SRC_BLKS

//  RFDR_ECC_MAX_EXTRA_BLKS_FOR_SOLUTION is the maximum number of extra blocks for solution over all supported code types.
#if (RFDR_ECC_MAX_EXTRA_BLKS_FOR_SOLUTION > RFDR_SIMPLE_CODE_MAX_EXTRA_BLKS_FOR_SOLUTION)
	#define RFDR_MAX_EXTRA_BLKS_FOR_SOLUTION     RFDR_ECC_MAX_EXTRA_BLKS_FOR_SOLUTION
#else
	#define RFDR_MAX_EXTRA_BLKS_FOR_SOLUTION     RFDR_SIMPLE_CODE_MAX_EXTRA_BLKS_FOR_SOLUTION
#endif

#define RFD_MAX_NUM_SRC_BLKS		RFD_MAX_FILE_LEN_BLOCKS
#if 0 // PK
#define RFD_MIN_NUM_SRC_BLKS		RFD_MIN_FILE_LEN_BLOCKS
#endif

///////////////////////////////
// RFD_MAX_FILE_LEN_BYTES :
// The maximum allowable file (fragment) size in bytes specified by the RFD protocol.
// Each Client may specify a Maximum File Size value less than
// or equal to RFD_MAX_FILE_LEN_BYTES.
#define RFD_MAX_FILE_LEN_BYTES  (16*1024*1024) // 4 Mbyte

///////////////////////////////
// RFD_MIN_FILE_LEN_BYTES :
// Minimum allowable file size in bytes.
#define RFD_MIN_FILE_LEN_BYTES  (1*1024) // 1 Kbyte


// Storage Transaction Count Variable
typedef INT32 RFD_STORAGE_TRANSACTION_COUNT_TYPE;
typedef INT32 RFD_STORAGE_SYNC_COUNT_TYPE;

typedef struct {
	// transactionCount is incremented upon the transaction start and
	// decremented upon the transaction complete.
	// If the transaction count is non-zero, then a transaction is in-progress.
	RFD_STORAGE_TRANSACTION_COUNT_TYPE transactionCount;
	// syncCount is incremented upon each transaction started. This is used to synchronize
	// the contents of a directory with another directory e.g. a corresponing
	// backup directory. If the syncCount of two directories are the same
	// (and the transactionCounts are 0) then the contents of the
	// directories are in-sync. (it is possible for two directories to have complete transactions
	// but be out-of-sync).
	RFD_STORAGE_SYNC_COUNT_TYPE syncCount;
} RFD_STORAGE_TRANSACTION_STRUCT;

typedef enum {
	RFD_PROC_ACTION__POLL_FOR_ACTION,
	RFD_PROC_ACTION__WAIT_FOR_EVENT,
	RFD_PROC_ACTION__PREPARE_FOR_DECODE,
	RFD_PROC_ACTION__DECODE,
	RFD_PROC_ACTION__DEFRAGMENT,
	RFD_PROC_ACTION__DECOMPRESS,
	RFD_PROC_ACTION__RESET_FSET_MDATA,
	RFD_PROC_ACTION__INITIALIZE_PROCESS
} RFD_PROCESSOR_ACTION_ENUM;

typedef enum {
	RFDR_STATE_UNINIT = 0,
	RFDR_STATE_COLLECTING = 1,			// processor waiting for enough collected blocks to decode/progressive decode
	RFDR_STATE_PREPARED_FOR_DECODE = 2, // processor set up for decode. Includes copying decoder state to 'private' version and copying block buffers to 'private' buffers for decode operation.
	RFDR_STATE_DECODE_COMPLETE = 3,		// decode complete, stop collecting, ready to decompress.
	RFDR_STATE_DEFRAG_COMPLETE = 4,		// decode complete, stop collecting, ready to decompress.
	RFDR_STATE_DECOMPRESS_COMPLETE = 5, // decompress complete, reconstructed file ready.
	RFDR_STATE_ALL_COMPLETE = 6         // file received by client app, all complete (must be last enum value)
} RFDR_FILE_RECONSTRUCT_STATE;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrInputApi
///
/// @brief
/// Enumerator describing the RFDR Client States.
///
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef enum {
    /// Processing is in a reset (unitialized) state. No Update File (FileSet) Metadata is currently
    /// assigned for this Client. An RFDR Client can be in this reset state if it has never collected
    /// an RFD Update File Metadata message (by way of RFD_CollectorPutMessage() API function call),
    /// or following an RFD_ConsumerResetFileCollectionRequest() API function call by the application.
    /// Reception of a valid RFD Update File Metadata message (by RFD_CollectorPutMessage() API) will
    /// result in a transition from this reset state (or any other state) to the processing state.
	RFDR_CLIENT_STATE_RESET = 0,

    /// Processing is in progress. The RFDR Client enters the processing state after collecting a
    /// new RFD Update File Metadata message (by way of RFD_CollectorPutMessage()). The RFDR Client
    /// remains in the processing state until it collects and decodes enough blocks to fully
    /// reconstuct the corresponding Update File and transfer it to the application Consumer.
	RFDR_CLIENT_STATE_IN_PROGRESS = 1,

    /// Processing is complete. An Update File has been reconstructed and transferred to the
    /// application Consumer. The RFDR Client will remain in this completed state until either:
    /// - it collects an RFD Update File Metadata message that's different than the Update File Metada
    /// for the Update File that was last completed, at which time the state will transition back
    /// to the in-progress state for the new Update File.
    /// - or the application issues an RFD_ConsumerResetFileCollectionRequest() API, following which
    /// the RFDR Client will transition back to the reset state.
	RFDR_CLIENT_STATE_COMPLETE = 2

} RFDR_CLIENT_STATE;

typedef struct {
	UINT16 blockThresholdNumeratorArray[RFDR_MAX_NUM_DECODE_CHUNKS]; // table of thresholds for each chunk incremetn..
	UINT16 numeratorArrayLen; // length of array (number of decode chunks).
	UINT16 blockThreshDenominator; // denominator value.
	UINT16 extraBlocksIncrement; // after reaching the final threshold of the array, increment the threshold by this many blocks.
} RFD_XCC_DECODE_EVENT_BLOCK_THRESH_INFO;

typedef struct {
	/* Variables to maintain the decoding state
	 * This structure is read by the Collector Thread and read and written by the Processor Thread.
	 * This is stored, along with other structures, in the file named: RFDR_STATE_FNAME
	 */
	RFD_XCC_DECODE_STATE_INFO decoderStateInfo;

	/* decoderStateInfoPrivate is a snapshot of decoderStateInfo that is used
	 * by the RFD processor decoder.
	 * The decoder can operate on the decoderStateInfoPrivate snapshot,
	 * while the collector thread reads decoderStateInfo that is
	 * possibly in a different state (different contents of structure)
	 */
	RFD_XCC_DECODE_STATE_INFO decoderStateInfoPrivate;

	/* Decode Codec Info.
	 * This is stored, along with other structures, in the file named: RFDR_STATE_FNAME
	 */
	RFD_XCC_CODEC_CONFIG_INFO codecConfigInfo;

	/* indexed by source block number, each entry is TRUE/FALSE to indicate whether
	 * the source block number has been collected.
	 * This array corresponds to the file named: RFDR_SRC_BLKS_COLLECTED_FNAME.
	 * The array also marks blocks that have been most recently buffered to the RFDR_SRC_BLKS_INBUF_FNAME file buffer.
	 */
	RFDR_XCC_SRC_BLKS_COLLECTED_STATUS srcBlksCollectedStatus[RFD_SRC_BLKS_COLLECTED_STATUS_MAX_ELEM];

	/* new encoded-type blocks that have not previously been collected
	 * are added to this list. Each entry is an encoded block id number identifying the block.
	 * This array corresponds to the file named: RFDR_CHK_BLKS_COLLECTED_FNAME.
	 * The list includes blocks that have been most recently buffered to the RFDR_CHK_BLKS_INBUF_FNAME file buffer.
	 */
	RFD_XCC_ENC_BLK_INDEX encBlksCollectedList[RFD_ENC_BLKS_COLLECTED_LIST_MAX_ELEM];

	/* variables related to determining when enough blocks have been collected
	 * to call decoder to do full or progressive decode.
	 */
	RFD_XCC_DECODE_EVENT_BLOCK_THRESH_INFO decodeEventBlockThreshInfo;

	/* The maximum number of extra blocks (over the minimum number) that are needed for
	 * successful decode solution. Sets a limit on the number of blocks that can be accumulated
	 * and input to the decoder.
	 */
	UINT16  maxExtraBlocksForSolution;

} RFDR_XCC_DECODE_INFO;

typedef union {
	RFDR_XCC_DECODE_INFO * xccDecodeInfo;
#if 0
	// 'xcc' is a generalized implementation of RFD 'ecc' (erasure correction code 1 or 2)
	// Xcc is also generalized to support the simple-code.
	// An more specific and optimized implementation may be developed
	// for the simple-code with a specific structure supported here (simpleDecodeInfo)
	// in this union.
	//RFDR_ECC_DECODE_INFO * eccDecodeInfo;
	//RFDR_SIMPLE_DECODE_INFO * simpleDecodeInfo;
#endif

} RFDR_DECODE_INFO;

typedef struct {
	UINT16 nSrcBlocksInSolution;
	UINT16 nChkBlocksInSolution;
	UINT16 nTotalBlocksInSolution;
	UINT16 nChkBlksEliminated;
	UINT16 nDecodeChunks;
} RFDR_DECODE_STATS;

#ifdef ENABLE_ACCUMULATED_DECODER_TEST_STATS
typedef struct {
	INT32 nFilesProcessed;
	INT32 maxExtraChkBlocks;
	INT32 minExtraChkBlocks;
	INT32 extraChkBlocksHistogram[RFDR_MAX_EXTRA_BLKS_FOR_SOLUTION];
	INT32 failureThreshHistogram[RFDR_MAX_EXTRA_BLKS_FOR_SOLUTION];
} RFDR_ACCUMULATED_DECODE_STATS;
#endif

typedef struct {
	UINT16	id;
	UINT32	crc;
	INT32	size;
	UINT16	blkLenBytes;
	RFD_CODE_TYPE_ENUM  codeType;
	RFD_COMP_TYPE_ENUM	compType;

} RFD_FILE_BASE_INFO;

typedef struct  {
	RFD_FILE_BASE_INFO fileInfo; // copy from file set info.
	RFDR_FILE_RECONSTRUCT_STATE state; // this state info is stored, along with other structures, in the file named: RFDR_STATE_FNAME
	RFDR_DECODE_INFO decInfo;

} RFDR_FILE_PROCESS_INFO;

typedef struct {
	RFD_EVENT_HANDLE exitRequestEvent;        // signal power down exit request FROM manager
	RFD_EVENT_HANDLE errorExitEvent;          // signal error exit event TO the manager
	UCHAR   fileProcessCollectorUpdate[RFD_FSET_MAX_NUM_FILES]; /* collector sets flag for particular file object updated */
	UCHAR   fileSetCollectorUpdate; /* collector sets this flag when the fileSetInfoBuf is updated. */
	RFD_FSET_MDATA_STRUCT fileSetInfoBuf; /* This is the File Set structure that is
										   * updated by the Collector Client
										   *(only the RFD Recevier Processor modifies the main 'client' fileSetInfo) */
	RFD_MESSAGE_INFO * messageInfo;
    RFD_MESSAGE_CONFIG_INFO defaultMsgConfigInfo; /* Default message configuration, no dynamic updates during runtime
                                                   * (i.e., no updates to blockMsgCarouselId field derived from File Set Metadata messages */
	UCHAR isFileSetCollectionComplete;	  /* Processor thread sets this boolean variable,
										   * and the Collector thread reads it (variable is not mutex protected
										   * for performance efficiency) */
	RFD_MSG_QUEUE_HANDLE messageQueue;

} RFDR_COLLECTOR_INFO_STRUCT;


typedef struct {
	DWORD  clientIndex;
	TCHAR * consumerDirName;
	BOOL fileSetResetRequest;				// flag from rfdr consumer to rfdr processor to request a reset on the processing of the entire file set.
	RFD_EVENT_HANDLE fileSetReadyEvent;	    // signal from rfdr processor to file consumer that a new file set is ready for the associated client.
    BOOL foundUnfinishedFileTransfer;       // flag to retry a file ownership transfer/handover that was unfinished on previous power cycle.
    UINT16 unfinishedFileTransferFileId;    // file Id identifying the file for which to restart ownership transfer to the app.
} RFD_FILE_CONSUMER_INFO_STRUCT;

typedef struct {
	DWORD  clientIdNo;
	DWORD  clientIndex;
	TCHAR * clientDirName;
	RFD_EVENT_HANDLE clientEvent;
	RFD_MUTEX_HANDLE clientMutex;
	RFDR_FILE_PROCESS_INFO * fileProcessInfo[RFD_FSET_MAX_NUM_FILES]; /* array of file objects. */
	RFD_FSET_MDATA_STRUCT fileSetInfo; /* the main version of the File Set structure */

} RFDR_CLIENT_INFO_STRUCT;


typedef struct {
	DWORD nClients;
	RFDR_CLIENT_INFO_STRUCT	clientInfo[RFDR_NUM_CLIENTS];
	RFDR_COLLECTOR_INFO_STRUCT collectorInfo[RFDR_NUM_CLIENTS];
	RFD_FILE_CONSUMER_INFO_STRUCT consumerInfo[RFDR_NUM_CLIENTS];
} RFDR_CLIENT_SET_STRUCT;

// RFDR_PROCESSOR_LL_RUN_STRUCT :
// Contains variables that have scope over the lifetime of the
// RFDR Processor Thread Run function.
typedef struct  {

	TCHAR * clientPathBuf;
    TCHAR * fragmentPathBuf;
    TCHAR * fileNameBuf;
	BOOL * isClientPollForActionComplete;
	RFD_MULTI_WAIT_EVENT_STRUCT * multiCollectWaitEventObject;

	int clientIndex;
	RFDR_CLIENT_INFO_STRUCT * clientInfo;
	RFDR_COLLECTOR_INFO_STRUCT * collectorInfo;
	RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo;
	RFD_FSET_MDATA_STRUCT * fileSetInfo;
	RFDR_FILE_PROCESS_INFO * fileProcessInfo;
	int fileToProcessIndex;
	BOOL forceNextActionCompletion;
	RFD_PROCESSOR_ACTION_ENUM actionRequest;

#ifdef ENABLE_ACCUMULATED_DECODER_TEST_STATS
	RFDR_ACCUMULATED_DECODE_STATS accumDecodeStats;
#endif

} RFDR_PROCESSOR_LL_CONTROL_STRUCT;

typedef struct
{
	RFD_EVENT_HANDLE errorExitEvent;		// event for signal error exit event TO the manager
	RFD_EVENT_HANDLE initCompleteEvent;		// event for signal inialization complete event TO the manager
	RFD_EVENT_HANDLE exitRequestEvent;		// event for signal power down exit request FROM manager
    UCHAR * exitRequestFlagPtr;             // Pointer to Flag for signal power down exit request FROM manager
											// processor thread can poll this flag instead of waiting for exitRequestEvent.
	UCHAR defaultExitRequestFlag;		    // The initial Exit Request Flag after Open(). The user can
                                            // provide an "external" address to replace this, after Open()
                                            // but before Start().
} RFDR_PROCESSOR_INFO_STRUCT;

typedef struct {
	RFDR_CLIENT_INFO_STRUCT * clientInfo;
	RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo;
} RFD_FILE_CONSUMER_THREAD_STRUCT, *RFD_CONSUMER_THREAD_DATA_HANDLE;

typedef struct {
	RFDR_CLIENT_INFO_STRUCT * clientInfo;
	RFDR_COLLECTOR_INFO_STRUCT * collectorInfo;
} RFD_FILE_COLLECTOR_THREAD_STRUCT, *RFD_COLLECTOR_THREAD_DATA_HANDLE;

typedef struct
{
	RFDR_CLIENT_SET_STRUCT * rfdrClientSet;
	RFDR_PROCESSOR_INFO_STRUCT * rfdrProcessorInfo;
    RFDR_PROCESSOR_LL_CONTROL_STRUCT * llControl;
} RFD_PROCESSOR_THREAD_DATA_STRUCT, *RFD_PROCESSOR_THREAD_DATA_HANDLE;

typedef struct
{
	RFD_EVENT_HANDLE errorExitEvent;

    DWORD   processorThreadId;
	DWORD   collectorThreadId[RFDR_NUM_CLIENTS];
    RFD_THREAD_HANDLE  processorThread;
	RFD_THREAD_HANDLE  collectorThread[RFDR_NUM_CLIENTS];

	RFD_PROCESSOR_THREAD_DATA_HANDLE rfdProcessorThreadInfo;
	RFD_COLLECTOR_THREAD_DATA_HANDLE rfdCollectorThreadInfo[RFDR_NUM_CLIENTS];
	RFD_CONSUMER_THREAD_DATA_HANDLE  rfdConsumerThreadInfo[RFDR_NUM_CLIENTS];

} RFDR_THREAD_MANAGEMENT_STRUCT;


typedef struct
{
	RFDR_CLIENT_SET_STRUCT * rfdrClientSet;
	RFDR_PROCESSOR_INFO_STRUCT * rfdrProcessorInfo;
	RFDR_THREAD_MANAGEMENT_STRUCT rfdrThreadManagementInfo;
    BOOL isAltSingleThreadMode;
    BOOL isRunning;    // TRUE if running (afer successful call to RFD_StartReceiver(), before RFD_StopReceiver().

} RFDR_STRUCT, *RFDR_HANDLE;

typedef struct
{
	TCHAR userReq;
	RFD_MSG_TYPE userReqMsgType;
	RFD_FSET_MDATA_STRUCT testFileSetMsg;
	RFD_BLOCK_MSG_STRUCT testBlockMsg;

} RFD_TEST_STRUCT;

///////////////////////////////////////////////////////////////////////////////

extern DWORD gClientId[RFDR_NUM_CLIENTS];

extern TCHAR * gClientDirName[RFDR_NUM_CLIENTS];

extern UINT16 gEccDecodeChunkBlkThreshNumerator[RFD_ECC_DECODE_NUM_DECODE_CHUNKS];
extern UINT16 gSimpleDecodeChunkBlkThreshNumerator[RFD_SIMPLE_DECODE_NUM_DECODE_CHUNKS];

extern const RFD_DECODE_XCC_FILE_NAME_LIST gDecodeXccFileNameList;

///////////////////////////////////////////////////////////////////////////////
void RFD_TestInit( void );
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL IsValidRfdFileSetDirName( RFD_FSET_MDATA_STRUCT * fileSetInfo, const TCHAR * DirName );
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ClientCreateStorage(RFDR_CLIENT_INFO_STRUCT * clientInfo);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_ClientCleanupStorage(RFDR_CLIENT_INFO_STRUCT * clientInfo);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CopyDirectoryContentsTransactionProtected(const TCHAR dstPathBuf[], const TCHAR srcPathBuf[]);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_BackupDirectoryContents(const TCHAR pathBuf[]);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_IsBackupStorageValid(const TCHAR pathBuf[]);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_IsBackupStorageInSync(const TCHAR pathBuf[]);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CleanupBackupStorageContents(const TCHAR pathBuf[]);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CreateBackupStorage(const TCHAR pathBuf[]);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_RestoreDirectoryContents(const TCHAR pathBuf[]);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_MarkStorageTransactionStart(const TCHAR path[]);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_MarkStorageTransactionComplete(
    const TCHAR path[],
    BOOL doPreSyncOfDirectory);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CancelStorageTransaction(const TCHAR path[]);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_IsStorageTransactionComplete(const TCHAR path[]);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_InitializeStorageTransaction(const TCHAR path[],
											RFD_STORAGE_SYNC_COUNT_TYPE initialSyncCount);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_GetStorageTransSyncCount(const TCHAR path[], RFD_STORAGE_SYNC_COUNT_TYPE * syncCount);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_AreStorageTransactionsInSync(const TCHAR path1[], const TCHAR path2[]);
///////////////////////////////////////////////////////////////////////////////////////////////
void RFD_FileSetDefaultInit(RFD_FSET_MDATA_STRUCT * hFileSet );
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_DecodeInfoCleanupDyn( RFDR_FILE_PROCESS_INFO * fileProcessInfo );
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_FileProcessInfoArrayCleanupDyn(RFDR_CLIENT_INFO_STRUCT * clientInfo);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_DecodeInfoCreateWithDefInit( RFDR_FILE_PROCESS_INFO * fileProcessInfo );
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_FileProcessInfoCreateWithDefInit( RFD_FSET_MDATA_STRUCT * hFileSetInfo, int FileIdx, RFDR_FILE_PROCESS_INFO ** hhFileProcessInfo);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_FileProcessInfoArrayCreateDynWithDefInit( RFDR_CLIENT_INFO_STRUCT * clientInfo);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_FileProcessInfoReadFromStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo );
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_FileProcessInfoWriteToStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_FileProcessInfoArrayReadFromStorage( RFDR_CLIENT_INFO_STRUCT * clientInfo);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_FileProcessInfoArrayWriteToStorage( RFDR_CLIENT_INFO_STRUCT * clientInfo);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_FileProcessStateWriteToStorage( const TCHAR * pathBuf, RFDR_FILE_RECONSTRUCT_STATE * fileProcessState );
///////////////////////////////////////////////////////////////////////////////////////////////
void RFDR_SignalErrorExit( RFD_EVENT_HANDLE errorExitEvent );
////////////////////////////////////////////////////
BOOL RFDR_IsFileSetInDecompressCompleteState(RFDR_CLIENT_INFO_STRUCT *clientInfo);
////////////////////////////////////////////////////
RFDR_CLIENT_STATE RFDR_GetClientFileSetState(RFDR_CLIENT_INFO_STRUCT *clientInfo);
////////////////////////////////////////////////////
void RFDR_ConstructFragmentDirPathName(TCHAR fragmentPathBuf[],
									   int fragmentPathBufLength,
									   RFDR_CLIENT_INFO_STRUCT * clientInfo, UINT16 fragmentId);
////////////////////////////////////////////////////
void RFDR_ConstructClientDirPathName(TCHAR clientPathBuf[],
									 int clientPathBufLength,
									 RFDR_CLIENT_INFO_STRUCT * clientInfo);
////////////////////////////////////////////////////
DWORD RFD_PLATFORM_API RFDR_Processor( LPVOID lpParam );

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_InitializeMaxExtraBlocksForSolution(RFD_CODE_TYPE_ENUM codeType,
												  UINT16 * maxExtraBlocksForSolution);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_InitializeDecodeBlockThresholdInfo(RFD_CODE_TYPE_ENUM codeType,
												  RFD_XCC_DECODE_EVENT_BLOCK_THRESH_INFO * decodeBlockThresholdInfo);

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_DecodeInfoSimpleCodeCreateWithDefInit( RFDR_FILE_PROCESS_INFO * fileProcessInfo );
RFD_STATUS RFD_DecodeInfoSimpleCodeReadFromStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo );
RFD_STATUS RFD_DecodeInfoSimpleCodeWriteToStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo );
RFD_STATUS RFD_DecodeInfoSimpleCodeCleanupStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo );
RFD_STATUS RFDR_PrepareForDecodeSimpleCode(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo, BOOL doStorageBackupOps);
RFD_STATUS RFDR_PostDecodeOpsSimpleCode(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo, BOOL * pIsDecodeComplete);
RFD_STATUS RFDR_RevertDecodeOpsSimpleCode(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo);
RFD_STATUS RFDR_CollectBlockSimpleCode(const TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo, RFD_BLOCK_MSG_STRUCT * hBlockMsg, BOOL * pSignalRequest);
RFD_STATUS RFDR_GetDecoderSolutionStatsSimpleCode(RFDR_FILE_PROCESS_INFO * fileProcessInfo, RFDR_DECODE_STATS * decodeStats);
RFD_STATUS RFD_InitializeDecodeBlockThresholdInfoSimpleCode(RFD_XCC_DECODE_EVENT_BLOCK_THRESH_INFO * decodeBlockThresholdInfo);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_DecodeInfoXccCodeCreateWithDefInit( RFDR_FILE_PROCESS_INFO * fileProcessInfo );
RFD_STATUS RFD_DecodeInfoXccCodeReadFromStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo );
RFD_STATUS RFD_DecodeInfoXccCodeWriteToStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo );
RFD_STATUS RFD_DecodeInfoXccCodeCleanupStorage( TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo );
RFD_STATUS RFDR_PrepareForDecodeXccCode(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo, BOOL doStorageBackupOps);
RFD_STATUS RFDR_PostDecodeOpsXccCode(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo, BOOL * pIsDecodeComplete);
RFD_STATUS RFDR_RevertDecodeOpsXccCode(const TCHAR pathBuf[], RFDR_FILE_PROCESS_INFO * fileProcessInfo);
RFD_STATUS RFDR_PrepareStorageForDecodeXcc(const TCHAR pathBuf[], RFDR_XCC_DECODE_INFO * xccDecodeInfo,
										   BOOL prepareSrcBlksInBuf, BOOL prepareChkBlksInBuf, BOOL doStorageBackupOps);
RFD_STATUS RFDR_GetDecoderSolutionStatsXccCode(RFDR_FILE_PROCESS_INFO * fileProcessInfo, RFDR_DECODE_STATS * decodeStats);
RFD_STATUS RFD_InitializeDecodeBlockThresholdInfoEccCode(RFD_XCC_DECODE_EVENT_BLOCK_THRESH_INFO * decodeBlockThresholdInfo);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_CollectBlockXccCode(const TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo, RFD_BLOCK_MSG_STRUCT * hBlockMsg, BOOL * pSignalRequest);
RFD_STATUS RFDR_StoreCollectedBlockXccCode_Ex(const TCHAR * pathBuf, RFDR_FILE_PROCESS_INFO * fileProcessInfo, RFD_BLOCK_MSG_STRUCT * hBlockMsg);
RFD_STATUS RFDR_StoreCollectedBlockXccCode(const TCHAR * pathBuf,
										   RFDR_FILE_PROCESS_INFO * fileProcessInfo,
										   RFD_BLOCK_TYPE blkType,
										   UCHAR * blockStorageBuf,
										   UINT16 blockStorageBufSize);
BOOL RFDR_IsMaxAllowableNumBlocksCollectedXccCode(RFDR_DECODE_INFO * decodeInfo);
BOOL RFDR_IsMaxAllowableNumBlocksCollectedSimpleCode(RFDR_DECODE_INFO * decodeInfo);
BOOL RFDR_IsBlockCollectionStateReadyForDecodeXccCode(RFDR_DECODE_INFO * decodeInfo);
BOOL RFDR_IsBlockCollectionStateReadyForDecodeSimpleCode(RFDR_DECODE_INFO * decodeInfo);
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_GetSdkVersionInfo(RFD_SDK_VERSION_INFO * rfdSkdVersionInfo) ;
RFD_STATUS RFD_ReceiverErrorCheck(RFDR_HANDLE rfdReceiverHandle, INT32 timeOutMilliSeconds);
RFD_STATUS RFD_OpenReceiver(RFDR_HANDLE * rfdReceiverHandle);
RFD_STATUS RFD_OpenReceiverExt(
    RFDR_HANDLE * rfdReceiverHandle,
    BOOL isAltSingleThreadMode
    );
RFD_STATUS RFD_CloseReceiver(RFDR_HANDLE rfdReceiverHandle);
RFD_STATUS RFD_StartReceiver(RFDR_HANDLE rfdReceiverHandle);
RFD_STATUS RFD_StopReceiver(RFDR_HANDLE rfdReceiverHandle);
RFD_CONSUMER_THREAD_DATA_HANDLE  RFD_GetFileConsumerHandle(RFDR_HANDLE rfdReceiverHandle, int clientIndex);
RFD_COLLECTOR_THREAD_DATA_HANDLE  RFD_GetMessageCollectorHandle(RFDR_HANDLE rfdReceiverHandle, int clientIndex);
RFD_STATUS RFD_SetExitRequestFlagAddress(
    RFDR_HANDLE rfdReceiverHandle,
    UCHAR * exitRequestFlagPtr
    );
RFD_STATUS RFD_RunReceiver(
    RFDR_HANDLE rfdReceiverHandle,
    BOOL *isThreadExitRequestedPtr
    );
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_ProcessorOpenLL(
	RFD_PROCESSOR_THREAD_DATA_HANDLE hRfdr,
    RFD_STATUS * statusPtr,
    BOOL *isThreadExitRequestedPtr
        );
RFD_STATUS RFDR_ProcessorCloseLL(
	RFD_PROCESSOR_THREAD_DATA_HANDLE hRfdr,
    RFD_STATUS * statusPtr
        );
RFD_STATUS RFDR_ProcessorRunLL(
    RFD_PROCESSOR_THREAD_DATA_HANDLE hRfdr,
    RFD_STATUS * statusPtr,
    BOOL *isThreadExitRequestedPtr,
    BOOL doRunUntilIdle
        );
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFDR_MsgCollectorOpenLL(
    RFD_COLLECTOR_THREAD_DATA_HANDLE hCollectorThread,
    RFD_STATUS * statusPtr,
    BOOL *isThreadExitRequestedPtr
        );
RFD_STATUS RFDR_MsgCollectorCloseLL(
    RFD_COLLECTOR_THREAD_DATA_HANDLE hCollectorThread,
    RFD_STATUS * statusPtr
        );
RFD_STATUS RFDR_MsgCollectorRunLL(
    RFD_COLLECTOR_THREAD_DATA_HANDLE hCollectorThread,
    RFD_STATUS * statusPtr,
    BOOL *isThreadExitRequestedPtr,
    BOOL * isIdlePtr
        );
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ConsumerConstructFSetResetReqFilePathName(
    RFD_FILE_CONSUMER_INFO_STRUCT * consumerInfo,
    TCHAR filePathNameBuf[],
    int filePathNameBufLen );

#ifdef __cplusplus
}
#endif

#endif // RFD_RECEIVER_H