////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_receiver\rfd_dec_xcc1.c
///
/// @brief	rfd eXtended ECC decoder.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd_codec_xcc.h"
#include "rfd_receiver.h"

///////////////////////////////////////////////////////////////////////////////////////////////
#ifndef XCC_BITS_PER_ELEMENT
	#define XCC_BITS_PER_ELEMENT 1
	#define XCC_ELEMENT_BIT_MASK 0x01
	#define XCC_IDENTITY_ELEM 1
	#define RFD_XCC1_CODE
	#define RFD_DECODE_XCC		RFD_DecodeXcc1Code
#endif

// XCC_IDENTITY_ELEM_DWORD_ALIGNED
//   e.g. for xcc1 = 0x80000000
//   e.g. for xcc2 = 0x40000000
#define XCC_IDENTITY_ELEM_DWORD_ALIGNED (((DWORD) 1) << (RFD_BITS_PER_DWORD - XCC_BITS_PER_ELEMENT))

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Cleanup decoder temp storage for an incomplete decode condition.
///
/// Cleanup (delete) temporary storage used by decoder in the case that decode is aborted early
/// e.g. exit request. On a regular decode completion this temp storage is cleaned up as part of
/// the regular decoding sequence.
///
/// @param  pathBuf         Directory Path for decoder files.
/// @param  fileNameList    List of file names.
////////////////////////////////////////////////////////////////////////////////////////////////////

static void _CleanupTempStorageOnIncompleteDecode(const TCHAR pathBuf[], const RFD_DECODE_XCC_FILE_NAME_LIST * fileNameList)
{
	TCHAR * fileNameBuf = NULL;

	//////////////////
	// Delete the temporary file used to build the transformation block matrix
	// for multi-block per super block decoding.
	// The decodeOutputFileName is temporarily used for this temp file.
	//////////////////

	fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(fileNameBuf == NULL) {
		return;
	}

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->decodeOutputFileName);

	RFD_DELETE_FILE(fileNameBuf);

	RFD_FREE(fileNameBuf);

	return;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Cleanup decoder bufs.
///
/// @param	bufs	the decoder buffers to cleanup/free.
////////////////////////////////////////////////////////////////////////////////////////////////////

static void _CleanupBufs(XCC_DEC_DYNAMIC_BUFS * bufs)
{

	if(bufs->coeffBuf != NULL) {
		RFD_FREE(bufs->coeffBuf);
		bufs->coeffBuf = NULL;
	}
	if(bufs->blockBuf != NULL) {
		RFD_FREE(bufs->blockBuf);
		bufs->blockBuf = NULL;
	}
	if(bufs->rowInfo != NULL) {
		RFD_FREE(bufs->rowInfo);
		bufs->rowInfo = NULL;
	}
	if(bufs->pivotInfo != NULL) {
		RFD_FREE(bufs->pivotInfo);
		bufs->pivotInfo = NULL;
	}

#if XCC_BITS_PER_ELEMENT > 1
	if(bufs->rowScaleInfo != NULL) {
		_CleanupRowScaleBufs(bufs->rowScaleInfo);
		bufs->rowScaleInfo = NULL;
	}
#else
	bufs->rowScaleInfo = NULL;
#endif
	return;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Calculates the decode base memory usage.
///
/// The base memory excludes memory for block data.
///
/// @param  dimInfo Decode dimension info.
///
/// @return The calculated decode base memory usage.
////////////////////////////////////////////////////////////////////////////////////////////////////

static UINT32 _CalculateDecodeBaseMemoryUsage(XCC_DEC_DIMENSION_INFO * dimInfo)
{
	UINT32 decodeBaseMemoryUsage = 0;

	// rowInfo
	decodeBaseMemoryUsage += (dimInfo->nRowsTotal * sizeof(INT16));

	// pivotInfo
	decodeBaseMemoryUsage += (dimInfo->nEquVars * sizeof(INT16));

	// coeffBuf
	decodeBaseMemoryUsage += (dimInfo->nRowsTotal * dimInfo->coeffRowLenDWords * sizeof(DWORD));

#if XCC_BITS_PER_ELEMENT > 1
	// for coeff scale rows
	decodeBaseMemoryUsage += (XCC_SCALE_ROW_NUM_BUFS * dimInfo->coeffRowLenDWords * sizeof(DWORD));
#endif

	return decodeBaseMemoryUsage;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Allocate bufs.
///
/// @param [out]	bufs	The allocated decoder bufs structure
/// @param [in]		dimInfo	dimension info structure.
///
/// @return	RFD_STATUS_OK is successful.
////////////////////////////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _AllocateBufs(XCC_DEC_DYNAMIC_BUFS * bufs,
								XCC_DEC_DIMENSION_INFO * dimInfo)
{
	bufs->coeffBuf = NULL;
	bufs->blockBuf = NULL;
	bufs->rowInfo = NULL;
	bufs->pivotInfo = NULL;
	bufs->rowScaleInfo = NULL;

	bufs->coeffBuf = (DWORD *) RFD_MALLOC(dimInfo->nRowsTotal * dimInfo->coeffRowLenDWords * sizeof(DWORD));
	if(bufs->coeffBuf == NULL) {
		_CleanupBufs(bufs);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	bufs->blockBuf = (DWORD *) RFD_MALLOC(dimInfo->nRowsTotal * dimInfo->blkLenDWords * sizeof(DWORD));
	if(bufs->blockBuf == NULL) {
		_CleanupBufs(bufs);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	bufs->rowInfo = (INT16 *) RFD_MALLOC(dimInfo->nRowsTotal * sizeof(INT16));
	if(bufs->rowInfo == NULL) {
		_CleanupBufs(bufs);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	bufs->pivotInfo = (INT16 *) RFD_MALLOC(dimInfo->nEquVars * sizeof(INT16));
	if(bufs->pivotInfo == NULL) {
		_CleanupBufs(bufs);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

#if XCC_BITS_PER_ELEMENT > 1
	bufs->rowScaleInfo = _AllocateRowScaleBufs(dimInfo);
	if(bufs->rowScaleInfo == NULL) {
		_CleanupBufs(bufs);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}
#endif

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Adds a source row to a check row.
///
/// @param	chkRowIndex	Zero-based index of the check row.
/// @param	srcRowIndex	Zero-based index of the source row.
/// @param	srcPivotNo	Source pivot no.
/// @param	bufs		The bufs.
/// @param	dimInfo		Dimension info.
////////////////////////////////////////////////////////////////////////////////////////////////////

static void _AddSourceRowToCheckRow(UINT16 chkRowIndex, UINT16 srcRowIndex, UINT16 srcPivotNo,
							 XCC_DEC_DYNAMIC_BUFS * bufs, XCC_DEC_DIMENSION_INFO * dimInfo)
{
	UINT16 i;
	DWORD* pSrc, * pChk;
	DWORD srcWord;

	// Add the block components of the Rows.

	pSrc = bufs->blockBuf + ((UINT32)srcRowIndex * dimInfo->blkLenDWords);
	pChk = bufs->blockBuf + ((UINT32)chkRowIndex * dimInfo->blkLenDWords);

	for(i=0;i<dimInfo->blkLenDWords;i++) {
		// bitwise XOR to add each element of the word.
		*pChk ^= *pSrc;
		pChk++; pSrc++;
	}

	// Add the coefficient components of the Rows.
	// No need to add all elements of the coefficient rows
	// since the source row only has one element set (at the Pivot Position).
	// Just add the words of the rows that contain the Pivot Position.
	// In fact the relevent word containing the pivot postion can be "synthesized"
	// instead of actually reading this word from the src Coeff buffer
	// typically faster operation.

	pChk = bufs->coeffBuf + ((UINT32)chkRowIndex * dimInfo->coeffRowLenDWords) +
		   ((srcPivotNo * XCC_BITS_PER_ELEMENT) / RFD_BITS_PER_DWORD);

	// "synthesize" (construct) the src Coefficient word instead of reading the equivalent
	// actual word.
	srcWord = XCC_IDENTITY_ELEM << (RFD_BITS_PER_DWORD - XCC_BITS_PER_ELEMENT);
	srcWord >>= ((srcPivotNo * XCC_BITS_PER_ELEMENT) % RFD_BITS_PER_DWORD);

	*pChk ^= srcWord;

	return;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Loads source type blocks from storage into buffers.
///
/// This function 'adds' new pivots to PivotInfo., and also modifies PivotInfo if new source
/// blocks correspond to pivot postions that have been previously implemented.
///
/// @param  pathBuf                             Directory path string.
/// @param  fileNameList                        List of file names.
/// @param  bufs                                The bufs.
/// @param  dimInfo                             Information describing the dimension.
/// @param  stateInfo                           Information describing the state.
/// @param  ConstructSrcTypeCoeffRowFunction    The construct source type coeff row function
///                                             pointer.
///
/// @return RFD_STATUS_OK is successful.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _LoadSrcBlksInputBuf(
					const TCHAR pathBuf[],
					const RFD_DECODE_XCC_FILE_NAME_LIST * fileNameList,
					XCC_DEC_DYNAMIC_BUFS * bufs,
					XCC_DEC_DIMENSION_INFO * dimInfo,
					RFD_XCC_DECODE_STATE_INFO * stateInfo,
					FcnPtrConstructSrcTypeCoeffRow ConstructSrcTypeCoeffRowFunction )
{
	RFD_FILE* fileHandle = NULL;
	TCHAR * fileNameBuf = NULL;
	UCHAR * superBlockStructBuf = NULL;
	UINT16 nTransformRows;
	UINT32 iRow; // use UINT32 to ensure no loss if used to calculate address offsets.
	UINT32 actualReadDataSize;
	UINT16 superBlockStorageStructSize;
	RFD_XCC_SRC_BLK_INDEX blkIndex;
	INT16 pivotNo;
	int iSwapRow;

	nTransformRows = stateInfo->nTransformRows;

	////////////////
	// Allocate the file name buffer.
	////////////////

	fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(fileNameBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	////////////////
	// Allocate the block buffer.
	////////////////

	// Clear and Allocate the temporary block structure buffer.
	// The block structure (index and payload) is read/written to file storage in one step/one buffer
	// as this may be generally faster/more efficient (unless the file system has a decent read/write cache).
	// Extend the size of the buffer to DWORD boundary (the block on the storage media is byte based
	// and may be slightly smaller). The blocks in the tranfomation matrix are also DWORD aligned/extended
	// so this superBlockStructBuf may be copied using the extended DWORD size, the last few bytes being zero filled
	// if the byteLen is not a multiple of DWORDS.
	superBlockStructBuf = (UCHAR *) RFD_CALLOC( sizeof(RFD_XCC_SRC_BLK_INDEX) +
												dimInfo->superBlkInfo.superBlkLenDWords * sizeof(DWORD) );

	if(superBlockStructBuf == NULL)
	{
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Calculate size of the block Storage Structure,
	// which contains the block index and the block payload.
	superBlockStorageStructSize = sizeof(RFD_XCC_SRC_BLK_INDEX) +
								  dimInfo->superBlkInfo.superBlkLenBytes * sizeof(UCHAR);

	/////////////////
	// Open the srcBlksInbuf file.
	/////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->srcBlksInbufFileName);

	if(RFD_FOPEN_S(&fileHandle, fileNameBuf, TEXT("rb"))) {
		RFD_FREE(fileNameBuf);
		RFD_FREE(superBlockStructBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	/////////////////
	// Finished using the fileNameBuf, free it.
	/////////////////
	RFD_FREE(fileNameBuf);

	// Initialize the row count to the end of the current transformed rows.
	// The source blocks will be inserted into the transformation matrices
	// starting at this row.

	for(iRow = nTransformRows;
		 iRow < (UINT32) (stateInfo->nSrcBlksInBuf + nTransformRows);
		  iRow++) {

		// read the index and block data from file
		actualReadDataSize = (UINT32) RFD_FREAD(superBlockStructBuf, 1, superBlockStorageStructSize, fileHandle);
		if(actualReadDataSize != superBlockStorageStructSize) {
			RFD_FCLOSE(fileHandle);
			RFD_FREE(superBlockStructBuf);
			return RFD_STATUS_ERROR_FILE_READ;
		}

		blkIndex = *(RFD_XCC_SRC_BLK_INDEX *)superBlockStructBuf;

		// The block's block index is the Pivot Number.
		pivotNo = blkIndex;

		if(pivotNo < dimInfo->nEquVars) { // make sure block index is in valid range

			// Update the row info to indicate this row implements a Pivot.
			// Set the column number to nEquVars to effectively indicate that
			// this is a source-type row with all postions clear (except
			// the pivot position).
			XCC_DEC_SET_ROW_INFO(bufs->rowInfo[iRow], dimInfo->nEquVars, TRUE);

			// copy the block payload
			RFD_MEMCPY( &bufs->blockBuf[iRow * dimInfo->blkLenDWords],
						superBlockStructBuf + sizeof(RFD_XCC_SRC_BLK_INDEX) + dimInfo->superBlkInfo.blkOffsetBytes,
						dimInfo->blkLenDWords * sizeof(DWORD) /* note: could just be dimInfo->blkLenBytes */);

			ConstructSrcTypeCoeffRowFunction(&bufs->coeffBuf[iRow * dimInfo->coeffRowLenDWords],
											 pivotNo, dimInfo->coeffRowLenDWords);

			///////////////////////////////
			// Do 'Pivot Swap' if this pivot postion is already implemented
			// by a transformed row (It will be swapped with this new Source row).
			///////////////////////////////

			// Get the current contents of this pivot position in case
			// this pivot is already implemented by a transformed row
			// (it will be swapped with this new Source row).
			iSwapRow = XCC_DEC_PIVOT_INFO_GET_ROW(bufs->pivotInfo[pivotNo]);

			// Write the Pivot Info structure to indicate that
			// this Pivot Position is implemented by this Source Block.
			XCC_DEC_SET_PIVOT_INFO_PIVOT_TRUE(bufs->pivotInfo[pivotNo], iRow);

			// Update nSrcBlocksInSolution, a decoder statistic type variable.
			stateInfo->nSrcBlocksInSolution++;

			if(pivotNo < stateInfo->nextPivotNo) {

				///////////////
				// The source index corresponds to a pivot position that has
				// already been reduced in the current transformation matrix.
				// Change the status row info corresponding to the previous
				// row that implemented the Pivot Postion
				// so that it is no longer flagged as a pivot row.
				///////////////

				if(iSwapRow < 0 ||
				   iSwapRow >= nTransformRows) {
					// Error, the pivotInfo should give a row that is one of the current transformed rows.
					// iSwapRow less than 0 indicates there is No row yet for that pivot, but there
					// must be one for that pivotNo if pivotNo is less than stateInfo->nextPivotNo.
		   			RFD_FCLOSE(fileHandle);
					RFD_FREE(superBlockStructBuf);
					return RFD_STATUS_ERROR_INVALID_BLOCK_INDEX;
				}

				/////////
				// Convert the 'swapped' previous Row from a pivot type row to a
				// non-pivot row.
				/////////

				// Add the Source-Type Row to the Check Row to zero the
				// Pivot Position in the Check Coefficient Row.
				_AddSourceRowToCheckRow((UINT16)iSwapRow, (UINT16)iRow, pivotNo, bufs, dimInfo);

				// Convert the Transformation Row Info from a pivot position row to
				// a non-pivot row, the next position to be cleared
				// for that row remains the same.
				XCC_DEC_SET_ROW_INFO_PIVOT_FALSE(bufs->rowInfo[iSwapRow]);
			}
		}
		else { //(blkIndex >= dimInfo->nEquVars)
			// Error, block index is out of range.
			RFD_FCLOSE(fileHandle);
			RFD_FREE(superBlockStructBuf);
			return RFD_STATUS_ERROR_INVALID_BLOCK_INDEX;
		}
	}

	///////////////
	// Check if the file size matches the expected size.
	///////////////

	if(RFD_FSEEK(fileHandle, 0, SEEK_END)) {
		RFD_FCLOSE(fileHandle);
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_SEEK;
	}

	if((UINT32)RFD_FTELL(fileHandle) != ((UINT32)superBlockStorageStructSize * stateInfo->nSrcBlksInBuf)) {
		RFD_FCLOSE(fileHandle);
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
	}

	///////////////
	// Finished, cleanup and exit.
	///////////////

	RFD_FCLOSE(fileHandle);
	RFD_FREE(superBlockStructBuf);

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Loads check-type blocks from storage into buffers.
///
/// @param	pathBuf								directory path string.
/// @param	fileNameList						list of file names.
/// @param	bufs								the bufs.
/// @param	dimInfo								information describing the dimension.
/// @param	stateInfo							information describing the state.
/// @param	ConstructChkTypeCoeffRowFunction	The construct check type coeff row function.
/// @param	maxCheckBlockIndex					maximum value of the check block index of the .
///
/// @return	RFD_STATUS_OK is successful.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _LoadChkBlksInputBuf(
					const TCHAR pathBuf[],
					const RFD_DECODE_XCC_FILE_NAME_LIST * fileNameList,
					XCC_DEC_DYNAMIC_BUFS * bufs,
					XCC_DEC_DIMENSION_INFO * dimInfo,
					RFD_XCC_DECODE_STATE_INFO * stateInfo,
					FcnPtrConstructChkTypeCoeffRow ConstructChkTypeCoeffRowFunction,
					UINT32 maxCheckBlockIndex )
{
	RFD_FILE* fileHandle = NULL;
	TCHAR * fileNameBuf = NULL;
	UCHAR * superBlockStructBuf = NULL;
	UINT16 chkRowStartNo;
	UINT32 iRow; // use UINT32 to ensure no loss if used to calculate address offsets.
	UINT32 actualReadDataSize;
	UINT16 superBlockStorageStructSize;
	RFD_XCC_ENC_BLK_INDEX blkIndex;

	chkRowStartNo = stateInfo->nTransformRows + stateInfo->nSrcBlksInBuf;

	////////////////
	// Allocate the file name buffer.
	////////////////

	fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(fileNameBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	////////////////
	// Allocate the block buffer.
	////////////////

	// Clear and Allocate the temporary block structure buffer.
	// The block structure (index and payload) is read/written to file storage in one step/one buffer
	// as this may be generally faster/more efficient (unless the file system has a decent read/write cache).
	// Extend the size of the buffer to DWORD boundary (the block on the storage media is byte based
	// and may be slightly smaller). The blocks in the tranfomation matrix are also DWORD aligned/extended
	// so this superBlockStructBuf may be copied using the extended DWORD size, the last few bytes being zero filled
	// if the byteLen is not a multiple of DWORDS.
	superBlockStructBuf = (UCHAR *) RFD_CALLOC( sizeof(RFD_XCC_ENC_BLK_INDEX) +
										   dimInfo->superBlkInfo.superBlkLenDWords * sizeof(DWORD) );

	if(superBlockStructBuf == NULL) {
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Calculate size of the size of the block Storage Sturcture,
	// which contains the block index and the block payload.
	superBlockStorageStructSize = sizeof(RFD_XCC_ENC_BLK_INDEX) +
							 dimInfo->superBlkInfo.superBlkLenBytes * sizeof(UCHAR);

	/////////////////
	// Open the chkBlksInbuf file.
	/////////////////

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->chkBlksInbufFileName);

	if(RFD_FOPEN_S(&fileHandle, fileNameBuf, TEXT("rb"))) {
		RFD_FREE(fileNameBuf);
		RFD_FREE(superBlockStructBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	/////////////////
	// Finished using the fileNameBuf, free it.
	/////////////////
	RFD_FREE(fileNameBuf);

	// Initialize the row count to the end of the current transformed rows.
	// The source blocks will be inserted into the transformation matrices
	// starting at this row.

	for(iRow = chkRowStartNo;
		 iRow < (UINT32) (chkRowStartNo + stateInfo->nChkBlksInBuf);
		  iRow++) {

		// read the index and block data from file
		actualReadDataSize = (UINT32) RFD_FREAD(superBlockStructBuf, 1, superBlockStorageStructSize, fileHandle);
		if(actualReadDataSize != superBlockStorageStructSize) {
			RFD_FCLOSE(fileHandle);
			RFD_FREE(superBlockStructBuf);
			return RFD_STATUS_ERROR_FILE_READ;
		}

		blkIndex = *(RFD_XCC_ENC_BLK_INDEX *)superBlockStructBuf;

		if(blkIndex <= maxCheckBlockIndex) { // make sure block index is in valid range

			// Update the row info to indicate that
			// the next pivot position to process for this row is column 0
			// and this row does not yet implement a Pivot Position.
			XCC_DEC_SET_ROW_INFO(bufs->rowInfo[iRow], 0, FALSE);

			// copy the block payload
			RFD_MEMCPY( &bufs->blockBuf[iRow * dimInfo->blkLenDWords],
						superBlockStructBuf + sizeof(RFD_XCC_ENC_BLK_INDEX) + dimInfo->superBlkInfo.blkOffsetBytes,
						dimInfo->blkLenDWords * sizeof(DWORD) /* note: could just be dimInfo->blkLenBytes */);

			ConstructChkTypeCoeffRowFunction(&bufs->coeffBuf[iRow * dimInfo->coeffRowLenDWords],
											 blkIndex, dimInfo->nEquVars);
		}
		else { //(blkIndex > maxCheckBlockIndex)
			// Error, block index is out of range.
			RFD_FCLOSE(fileHandle);
			RFD_FREE(superBlockStructBuf);
			return RFD_STATUS_ERROR_INVALID_BLOCK_INDEX;
		}
	} // end of row processing loop

	///////////////
	// Check if the file size matches the expected size.
	///////////////

	if(RFD_FSEEK(fileHandle, 0, SEEK_END)) {
		RFD_FCLOSE(fileHandle);
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_SEEK;
	}

	if((UINT32)RFD_FTELL(fileHandle) != ((UINT32)superBlockStorageStructSize * stateInfo->nChkBlksInBuf)) {
		RFD_FCLOSE(fileHandle);
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
	}

	///////////////
	// Finished, cleanup and exit.
	///////////////

	RFD_FCLOSE(fileHandle);
	RFD_FREE(superBlockStructBuf);

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Resets pivot information.
///
/// @param	pivotInfo	Information describing the pivot.
/// @param	dimInfo		information describing the dimension.
///
/// @return	RFD_STATUS_OK if successful.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _ResetPivotInfo(
					INT16 pivotInfo[],
					XCC_DEC_DIMENSION_INFO * dimInfo)
{
	int i;

	// Set each pivot position to indicate no pivot row implements.
	for(i=0;i<dimInfo->nEquVars;i++) {
		XCC_DEC_SET_PIVOT_INFO_PIVOT_FALSE(pivotInfo[i]);
	}

	return RFD_STATUS_OK;

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Is decode complete.
///
/// @param	dimInfo		information describing the dimension.
/// @param	stateInfo	information describing the state.
///
/// @return	true if decode complete, false if not.
////////////////////////////////////////////////////////////////////////////////////////////////////

static BOOL _IsDecodeComplete(XCC_DEC_DIMENSION_INFO * dimInfo,
					   RFD_XCC_DECODE_STATE_INFO * stateInfo)
{
	return ((stateInfo->nextPivotNo == dimInfo->nEquVars) ? TRUE : FALSE);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Stores the system of equations for the case of a single block per super block.
///
/// @param	pathBuf			directory path string.
/// @param	fileNameList	list of file names.
/// @param	bufs			the bufs.
/// @param	dimInfo			information describing the dimension.
/// @param	stateInfo		information describing the state.
///
/// @return	RFD_STATUS_OK if successful.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _StoreSysOfEquationsSingleBlockPerSuperBlock(
					const TCHAR pathBuf[],
					const RFD_DECODE_XCC_FILE_NAME_LIST * fileNameList,
					XCC_DEC_DYNAMIC_BUFS * bufs,
					XCC_DEC_DIMENSION_INFO * dimInfo,
					RFD_XCC_DECODE_STATE_INFO * stateInfo)
{
	RFD_FILE* fileHandle = NULL;
	RFD_FILE* blockFileHandle = NULL;
	RFD_FILE* coeffFileHandle = NULL;
	TCHAR * fileNameBuf = NULL;
	UINT16 nTransformRows;
	int iPivot;
	INT32 actualWriteDataSize, blockOutputStorageSize, dataSize;
	INT32 blockTransformStorageSize, coeffTransformStorageSize;
	int rowNo, iRow, iNewRow;
	int progressNo;
	DWORD * pBlock, * pCoeff;
	INT16 * newRowInfo;

	nTransformRows = stateInfo->nTransformRows;

	////////////////
	// Allocate the file name buffer.
	////////////////

	fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(fileNameBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	if(stateInfo->nSrcBlksInBuf > 0) {
		////////////////
		// Delete the source blocks input file.
		////////////////

		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->srcBlksInbufFileName);

		if(RFD_DELETE_FILE(fileNameBuf)) {
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_DELETE;
		}
	}

	if(stateInfo->nChkBlksInBuf > 0) {
		////////////////
		// Delete the check blocks input file.
		////////////////

		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->chkBlksInbufFileName);

		if(RFD_DELETE_FILE(fileNameBuf)) {
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_DELETE;
		}
	}

	if(_IsDecodeComplete(dimInfo, stateInfo)) {

		////////////////
		// Delete the transformation block matrix file.
		////////////////

		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->blockTransformFileName);

		// If decoding completed on the first decode call, then the file will not exist
		// and the Delete File function may return error. Don't treat this as an error.
		RFD_DELETE_FILE(fileNameBuf);


		////////////////
		// Delete the transformation coefficient matrix file.
		////////////////

		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->coeffTransformFileName);

		// If decoding completed on the first decode call, then the file will not exist
		// and the Delete File function may return error. Don't treat this as an error.
		RFD_DELETE_FILE(fileNameBuf);


		////////////////
		// Delete the transformation information file.
		////////////////

		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->transformationInfoFileName);

		// If decoding completed on the first decode call, then the file will not exist
		// and the Delete File function may return error. Don't treat this as an error.
		RFD_DELETE_FILE(fileNameBuf);


		////////////////
		// Write the final reconstructed file.
		// Use the specified decoder output file name.
		////////////////

		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->decodeOutputFileName);

		if(RFD_FOPEN_S(&blockFileHandle, fileNameBuf, TEXT("wb"))) {
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_OPEN;
		}

		// Write the pivot rows in order of the pivot index
		for(iPivot=0;iPivot<dimInfo->nEquVars;iPivot++) {

			if(!XCC_DEC_PIVOT_INFO_IS_PIVOT(bufs->pivotInfo[iPivot])) {
				// Error, the pivot position does not have an implementing row.
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
			}

			// The row corresponding to this pivot position.
			rowNo = XCC_DEC_PIVOT_INFO_GET_ROW(bufs->pivotInfo[iPivot]);

			if(rowNo >= dimInfo->nRowsTotal) {
				// Error, the row number is out of range.
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
			}

			// Pointer to start of block row in the block transformation matrix
			pBlock = bufs->blockBuf + dimInfo->blkLenDWords * rowNo;

			if(iPivot != (dimInfo->nEquVars-1)) {
				// Not the last block.
				// When writing the block to the final ouput, the block
				// is written using the block length in bytes,
				// not the rounded-up byte length (dword aligned
				// as in block length when writing transformation matrix rows).
				blockOutputStorageSize = dimInfo->blkLenBytes * sizeof(UCHAR);
			}
			else {
				// This is the last block. The last block len may be smaller
				// if file len is not an exact multiple of the block len.
				blockOutputStorageSize = dimInfo->superBlkInfo.lastSuperBlkLenBytes * sizeof(UCHAR);
			}

			actualWriteDataSize = (INT32) RFD_FWRITE(pBlock, 1, blockOutputStorageSize, blockFileHandle);
			if(actualWriteDataSize != blockOutputStorageSize) {
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_WRITE;
			}
		}

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
		// File-Sync the block file to disk as part of
		// RFD storage transaction management to handle power-loss or other 
		// program abort conditions.
		
		// Must fflush first to flush upper layer buffers, before file sync
		RFD_FFLUSH(blockFileHandle);

		// Do the File Sync.
		RFD_FileSyncByHandle(blockFileHandle);
#endif
		RFD_FCLOSE(blockFileHandle);
		blockFileHandle = NULL;
	}
	else {
		////////////////
		// Decoding is not complete.
		// Save all necessary data to continue decoding on the
		// next iteration.
		////////////////

		// Allocate a temporary rowInfo array to hold the updated row sequence
		// i.e. the order in which rows are written to the transformation matrices
		// files may be different from their current order.
		newRowInfo = (INT16 *) RFD_MALLOC(dimInfo->nRowsTotal * sizeof(INT16));
		if(newRowInfo == NULL) {
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_MEM_ALLOC;
		}

		////////////////
		// Write transformation block matrix and coeff matrix back to the
		// corresponding files.
		////////////////

		// Open the Block transformation matrix file.
		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->blockTransformFileName);

		if(RFD_FOPEN_S(&blockFileHandle, fileNameBuf, TEXT("wb"))) {
			RFD_FREE(newRowInfo);
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_OPEN;
		}

		// Open the Coefficient transformation matrix file.
		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->coeffTransformFileName);

		if(RFD_FOPEN_S(&coeffFileHandle, fileNameBuf, TEXT("wb"))) {
			RFD_FCLOSE(blockFileHandle);
			RFD_FREE(newRowInfo);
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_OPEN;
		}

		// Write the rows implementing pivots in order of the pivot index

		// The rows are written using the rounded-up DWord aligned size.
		blockTransformStorageSize = dimInfo->blkLenDWords * sizeof(DWORD);
		coeffTransformStorageSize = dimInfo->coeffRowLenDWords * sizeof(DWORD);

		iNewRow = 0;

		for(iPivot=0;iPivot<dimInfo->nEquVars;iPivot++) {

			if(!XCC_DEC_PIVOT_INFO_IS_PIVOT(bufs->pivotInfo[iPivot])) {
				// No row implements this pivot postion yet,
				// continue to process next pivot position.
				continue;
			}

			// The current pivot has an implemented row.

			// The row corresponding to this pivot position.
			rowNo = XCC_DEC_PIVOT_INFO_GET_ROW(bufs->pivotInfo[iPivot]);

			if(rowNo >= dimInfo->nRowsTotal) {
				// Error, the row number is out of range.
				RFD_FCLOSE(coeffFileHandle);
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(newRowInfo);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
			}

			// The progress number corresponding to this row.
			progressNo = XCC_DEC_ROW_INFO_GET_NEXT_POSITION(bufs->rowInfo[rowNo]);

			// Set the progress number and isPivot=TRUE indication
			// in the new RowInfo structure.
			XCC_DEC_SET_ROW_INFO(newRowInfo[iNewRow], progressNo, TRUE);

			// Update the pivotInfo current pivot position with the new row number.
			XCC_DEC_SET_PIVOT_INFO_PIVOT_TRUE(bufs->pivotInfo[iPivot], iNewRow);
			// Increment the written row counter.
			iNewRow++;

			// Pointer to start of block row in the block transformation matrix
			pBlock = bufs->blockBuf + dimInfo->blkLenDWords * rowNo;
			// Pointer to start of coefficient row in the coeff transformation matrix
			pCoeff = bufs->coeffBuf + dimInfo->coeffRowLenDWords * rowNo;

			// Write the block row
			actualWriteDataSize = (INT32) RFD_FWRITE(pBlock, 1, blockTransformStorageSize, blockFileHandle);
			if(actualWriteDataSize != blockTransformStorageSize) {
				RFD_FCLOSE(coeffFileHandle);
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(newRowInfo);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_WRITE;
			}
			// Write the coefficient row
			actualWriteDataSize = (INT32) RFD_FWRITE(pCoeff, 1, coeffTransformStorageSize, coeffFileHandle);
			if(actualWriteDataSize != coeffTransformStorageSize) {
				RFD_FCLOSE(coeffFileHandle);
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(newRowInfo);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_WRITE;
			}
		}

		// Write the rows that do not implement pivot postions
		// and that are not linearly dependent.
		for(iRow=0;iRow<dimInfo->nRowsTotal;iRow++) {

			if(XCC_DEC_ROW_INFO_IS_PIVOT_ROW(bufs->rowInfo[iRow])) {
				// This row implements a pivot postion and would have
				// been written in the previous loop. Continue
				// the the next current row.
				continue;
			}

			// The current row is not a pivot row.

			// The progress number corresponding to this current row.
			progressNo = XCC_DEC_ROW_INFO_GET_NEXT_POSITION(bufs->rowInfo[iRow]);

			if(progressNo == dimInfo->nEquVars) {
				// This row is linearly dependent. Eliminate it by not
				// writing it back to files.
				continue;
			}

			// Set the progress number and isPivot=FALSE indication
			// in the new RowInfo structure.
			XCC_DEC_SET_ROW_INFO(newRowInfo[iNewRow], progressNo, FALSE);

			// Increment the written row counter.
			iNewRow++;

			// Pointer to start of block row in the block transformation matrix
			pBlock = bufs->blockBuf + dimInfo->blkLenDWords * iRow;
			// Pointer to start of coefficient row in the coeff transformation matrix
			pCoeff = bufs->coeffBuf + dimInfo->coeffRowLenDWords * iRow;

			// Write the block row
			actualWriteDataSize = (INT32) RFD_FWRITE(pBlock, 1, blockTransformStorageSize, blockFileHandle);
			if(actualWriteDataSize != blockTransformStorageSize) {
				RFD_FCLOSE(coeffFileHandle);
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(newRowInfo);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_WRITE;
			}
			// Write the coefficient row
			actualWriteDataSize = (INT32) RFD_FWRITE(pCoeff, 1, coeffTransformStorageSize, coeffFileHandle);
			if(actualWriteDataSize != coeffTransformStorageSize) {
				RFD_FCLOSE(coeffFileHandle);
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(newRowInfo);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_WRITE;
			}
		}

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
		// File-Sync the coefficient and block files to disk as part of
		// RFD storage transaction management to handle power-loss or other 
		// program abort conditions.
		
		// Must perform fflush before file sync.
		RFD_FFLUSH(coeffFileHandle);
		RFD_FFLUSH(blockFileHandle);

		// Do the File Sync.
		RFD_FileSyncByHandle(coeffFileHandle);
		RFD_FileSyncByHandle(blockFileHandle);
#endif
		// close the coeff and block transformation matrix files
		RFD_FCLOSE(coeffFileHandle);
		RFD_FCLOSE(blockFileHandle);
		coeffFileHandle = NULL;
		blockFileHandle = NULL;

		////////////////
		// Update state info with the number of rows now in the transformation matrices.
		////////////////
		stateInfo->nTransformRows = iNewRow;

		////////////////
		// Write the new rowInfo and updated pivotInfo to the transformation info file.
		////////////////

		// Open the file.

		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->transformationInfoFileName);

		if(RFD_FOPEN_S(&fileHandle, fileNameBuf, TEXT("wb"))) {
			RFD_FREE(newRowInfo);
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_OPEN;
		}

		// Write the new RowInfo Array

		dataSize = iNewRow * sizeof(INT16);
		actualWriteDataSize = (INT32) RFD_FWRITE(newRowInfo, 1, dataSize, fileHandle);
		if(actualWriteDataSize != dataSize) {
			RFD_FREE(newRowInfo);
			RFD_FCLOSE(fileHandle);
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_READ;
		}

		// Write the PivotInfo Array

		dataSize = dimInfo->nEquVars * sizeof(INT16);
		actualWriteDataSize = (INT32) RFD_FWRITE(bufs->pivotInfo, 1, dataSize, fileHandle);
		if(actualWriteDataSize != dataSize) {
			RFD_FREE(newRowInfo);
			RFD_FCLOSE(fileHandle);
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_READ;
		}

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
		// File-Sync the transformation info file to disk as part of
		// RFD storage transaction management to handle power-loss or other 
		// program abort conditions.
		
		// Must perform fflush to flush upper layer stdio buffers, before file sync.
		RFD_FFLUSH(fileHandle);

		// Do the File Sync.
		RFD_FileSyncByHandle(fileHandle);
#endif
		// Close the transformation info file.
		RFD_FCLOSE(fileHandle);
		fileHandle = NULL;

		// Free the temporary new rowInfo structure.
		RFD_FREE(newRowInfo);
		newRowInfo = NULL;
	}

	// Free the file name buffer.
	RFD_FREE(fileNameBuf);
	fileNameBuf = NULL;

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Stores the system of equations for the case of a multiple blocks per super block.
///
/// @param	pathBuf			directory path string.
/// @param	fileNameList	list of file names.
/// @param	bufs			the bufs.
/// @param	dimInfo			information describing the dimension.
/// @param	stateInfo		information describing the state.
///
/// @return	RFD_STATUS_OK if successful.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _StoreSysOfEquationsMultipleBlocksPerSuperBlock(
					const TCHAR pathBuf[],
					const RFD_DECODE_XCC_FILE_NAME_LIST * fileNameList,
					XCC_DEC_DYNAMIC_BUFS * bufs,
					XCC_DEC_DIMENSION_INFO * dimInfo,
					RFD_XCC_DECODE_STATE_INFO * stateInfo)
{
	RFD_FILE* fileHandle = NULL;
	RFD_FILE* blockFileHandle = NULL;
	RFD_FILE* coeffFileHandle = NULL;
	TCHAR * fileNameBuf = NULL;
	TCHAR * fileNameBuf2 = NULL;
	UINT16 nTransformRows;
	int iPivot;
	INT32 actualWriteDataSize, blockOutputStorageSize, dataSize;
	UINT32 blockTransformStorageSize, superBlockTransformStorageSize, coeffTransformStorageSize;
	int rowNo, iRow, iNewRow;
	int progressNo;
	DWORD * pBlock, * pCoeff;
	INT16 * newRowInfo;

	nTransformRows = stateInfo->nTransformRows;

	////////////////
	// Allocate the file name buffer.
	////////////////

	fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(fileNameBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	if(dimInfo->superBlkInfo.isLastBlock) {

		// This is the last block of the super block.
		// Therefore delete the source and check block input files.
		// Othewise, not the last block, and these files should not be
		// deleted in case decoding may be aborted due to exit request, in which case
		// these input files will be used again on a following decode operation.

		if(stateInfo->nSrcBlksInBuf > 0) {
			////////////////
			// Delete the source blocks input file.
			////////////////

			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->srcBlksInbufFileName);

			if(RFD_DELETE_FILE(fileNameBuf)) {
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_DELETE;
			}
		}

		if(stateInfo->nChkBlksInBuf > 0) {
			////////////////
			// Delete the check blocks input file.
			////////////////

			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->chkBlksInbufFileName);

			if(RFD_DELETE_FILE(fileNameBuf)) {
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_DELETE;
			}
		}
	}

	if(_IsDecodeComplete(dimInfo, stateInfo)) {

		////////////////
		// Delete the transformation block matrix file.
		////////////////

		if(dimInfo->superBlkInfo.isLastBlock) {

			// This is the last block of the super block.
			// Therefore delete the transformation coefficient, block data and info files.
			// Othewise, not the last block, and these files should not be
			// deleted in case decoding may be aborted due to exit request, in which case
			// these input files will be reverted to and used again on a following decode operation.

			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->blockTransformFileName);

			// If decoding completed on the first decode call, then the file will not exist
			// and the Delete File function may return error. Don't treat this as an error.
			RFD_DELETE_FILE(fileNameBuf);


			////////////////
			// Delete the transformation coefficient matrix file.
			////////////////

			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->coeffTransformFileName);

			// If decoding completed on the first decode call, then the file will not exist
			// and the Delete File function may return error. Don't treat this as an error.
			RFD_DELETE_FILE(fileNameBuf);


			////////////////
			// Delete the transformation information file.
			////////////////

			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->transformationInfoFileName);

			// If decoding completed on the first decode call, then the file will not exist
			// and the Delete File function may return error. Don't treat this as an error.
			RFD_DELETE_FILE(fileNameBuf);
		}


		////////////////
		// Write the final reconstructed file.
		// Use the specified decoder output file name.
		////////////////

		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->decodeOutputFileName);

		if(dimInfo->superBlkInfo.isFirstBlock) {
			// This is the first block of the super block.
			// The file is being created for the first time.
			if(RFD_FOPEN_S(&blockFileHandle, fileNameBuf, TEXT("wb"))) {
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_OPEN;
			}
		}
		else {
			// Not the first block of the super block.
			// Open existing file for "update". Previous contents of the file,
			// i.e. for other (sub) blocks, are not deleted.
			// The file is updated with the contents of the current block.
			if(RFD_FOPEN_S(&blockFileHandle, fileNameBuf, TEXT("rb+"))) {
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_OPEN;
			}
		}

		// Initialize file position for the first block write.
		if(RFD_FSEEK(blockFileHandle, dimInfo->superBlkInfo.blkOffsetBytes, SEEK_CUR)) {
			RFD_FCLOSE(blockFileHandle);
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_SEEK;
		}

		// Write the pivot rows in order of the pivot index
		for(iPivot=0;iPivot<dimInfo->nEquVars;iPivot++) {

			if(!XCC_DEC_PIVOT_INFO_IS_PIVOT(bufs->pivotInfo[iPivot])) {
				// Error, the pivot position does not have an implementing row.
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
			}

			// The row corresponding to this pivot position.
			rowNo = XCC_DEC_PIVOT_INFO_GET_ROW(bufs->pivotInfo[iPivot]);

			if(rowNo >= dimInfo->nRowsTotal) {
				// Error, the row number is out of range.
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
			}

			// Pointer to start of block row in the block transformation matrix
			pBlock = bufs->blockBuf + dimInfo->blkLenDWords * rowNo;

			if(iPivot != (dimInfo->nEquVars-1)) {
				// Not the last super block row.
				// When writing the block to the final ouput, the block
				// is written using the block length in bytes,
				// not the rounded-up byte length (dword aligned
				// as in block length when writing transformation matrix rows).
				blockOutputStorageSize = dimInfo->blkLenBytes;
			}
			else {
				// This is the last super block row. The current block may be cropped smaller
				// or even not existing in the last row if the file len is not an exact
				// multiple of the super block len.
				if( (dimInfo->superBlkInfo.blkOffsetBytes + dimInfo->blkLenBytes) > dimInfo->superBlkInfo.lastSuperBlkLenBytes) {
					// This value can be negative or zero. In this case, this block does not exist in the last super block row.
					// Bypass writing this block in this <= 0 case.
					blockOutputStorageSize = (INT32) dimInfo->superBlkInfo.lastSuperBlkLenBytes - dimInfo->superBlkInfo.blkOffsetBytes;
				}
				else {
					blockOutputStorageSize = dimInfo->blkLenBytes;
				}
			}

			// Write the block only if it exists in the last super block row.
			if(blockOutputStorageSize > 0) {
				actualWriteDataSize = (INT32) RFD_FWRITE(pBlock, 1, blockOutputStorageSize, blockFileHandle);
				if(actualWriteDataSize != blockOutputStorageSize) {
					RFD_FCLOSE(blockFileHandle);
					RFD_FREE(fileNameBuf);
					return RFD_STATUS_ERROR_FILE_WRITE;
				}
			}


			if(    (iPivot < (dimInfo->nEquVars-2))   // Not either of the last two rows.
				|| // Or Is the second to last row AND block offset is less that the length of the last row
				   (iPivot == (dimInfo->nEquVars-2) &&
				    dimInfo->superBlkInfo.blkOffsetBytes < dimInfo->superBlkInfo.lastSuperBlkLenBytes) ) {

				// THEN perform the next Seek.

				// Update file position for the next write.
				// For the case of multiple blocks per super block, each row of the block transform matrix
				// is not rounded to dwords. The length of each row is therefore superBlkLenBytes.
				if(RFD_FSEEK(blockFileHandle, dimInfo->superBlkInfo.superBlkLenBytes - blockOutputStorageSize, SEEK_CUR)) {
					RFD_FCLOSE(blockFileHandle);
					RFD_FREE(fileNameBuf);
					return RFD_STATUS_ERROR_FILE_SEEK;
				}
			}
			// Else don't do file seek because it may result in extending the file length erroneously.
		}

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
		// File-Sync the block file to disk as part of
		// RFD storage transaction management to handle power-loss or other 
		// program abort conditions.
		
		// Must perform fflush to flush the upper layer buffers before file sync.
		RFD_FFLUSH(blockFileHandle);

		// Do the File Sync.
		RFD_FileSyncByHandle(blockFileHandle);
#endif
		RFD_FCLOSE(blockFileHandle);
		blockFileHandle = NULL;
	}
	else {
		////////////////
		// Decoding is not complete.
		////////////////

		////////////////
		// Save the Transformation Block Matrix to storage.
		// This is done for each block of the super block.
		//
		// Other data is stored only upon the last block of the super block.
		////////////////

		// Open the Block transformation matrix file.
		// For multiple-blocks per super block implementation,
		// a temporary file is used to contain the incremental updates as each
		// block of the super-blok is processed. Upon the last block of
		// the super block, the temporary file is copied to the regular block transform file.
		// In this way, if a decode is aborted (exit request) during decoding of any block,
		// the original block transform file is preserved.
		// Use the decoder output file as the temporary block transform file.
		RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
		RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->decodeOutputFileName);

		if(dimInfo->superBlkInfo.isFirstBlock) {
			// This is the first block of the super block.
			// The file is being created for the first time.
			if(RFD_FOPEN_S(&blockFileHandle, fileNameBuf, TEXT("wb"))) {
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_OPEN;
			}
		}
		else {
			// Not the first block of the super block.
			// Open existing file for "update". Previous contents of the file,
			// i.e. for other (sub) blocks, are not deleted.
			// The file is updated with the contents of the current block.
			if(RFD_FOPEN_S(&blockFileHandle, fileNameBuf, TEXT("rb+"))) {
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_OPEN;
			}
		}

		// Round-up the super Block storage size to a multiple of DWords.
		// This is done for compatibility with the way blocks/super blocks are stored for the simple
		// case of one block per super block.
		superBlockTransformStorageSize = dimInfo->superBlkInfo.superBlkLenDWords * sizeof(DWORD);

		if(dimInfo->superBlkInfo.isLastBlock) {
			// This is the last block of the super block. Round-up this block storage size
			// so that it 'fills' up to the DWord multiple super block row size.
			// This is done for compatibility with the way blocks/super blocks are stored for the simple
			// case of one block per super block.
			// e.g. the first series of progressive decode operations may implement one block per super block
			// (due to fewer number of available rows) while the last decode operations will require
			// multiple blocks per super block (due to more available rows). In both cases,
			// the storage alignment must be compatible.

			if((dimInfo->superBlkInfo.blkOffsetBytes + dimInfo->blkLenBytes) < superBlockTransformStorageSize) {
				blockTransformStorageSize = dimInfo->blkLenBytes +
					superBlockTransformStorageSize - (dimInfo->superBlkInfo.blkOffsetBytes + dimInfo->blkLenBytes);
			}
			else {
				blockTransformStorageSize = dimInfo->blkLenBytes;
			}
		}
		else {
			// Not the last block.
			blockTransformStorageSize = dimInfo->blkLenBytes;
		}

		// Initialize file position for the first block write.
		if(RFD_FSEEK(blockFileHandle, dimInfo->superBlkInfo.blkOffsetBytes, SEEK_CUR)) {
			RFD_FCLOSE(blockFileHandle);
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_SEEK;
		}

		////////////////////
		// Note: The sequence in which block rows are stored must match the
		// same sequence in which coefficient rows are stored later below,
		// and this sequence must also be synchronized with the the pivot info and
		// row info arrays which reference these rows (all must be syncronized).
		// e.g. rows implementing pivots are stored first, followed by rows
		// that are not yet selected for pivots.
		////////////////////

		for(iPivot=0;iPivot<dimInfo->nEquVars;iPivot++) {

			if(!XCC_DEC_PIVOT_INFO_IS_PIVOT(bufs->pivotInfo[iPivot])) {
				// No row implements this pivot postion yet,
				// continue to process next pivot position.
				continue;
			}

			// The current pivot has an implemented row.

			// The row corresponding to this pivot position.
			rowNo = XCC_DEC_PIVOT_INFO_GET_ROW(bufs->pivotInfo[iPivot]);

			if(rowNo >= dimInfo->nRowsTotal) {
				// Error, the row number is out of range.
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
			}

			// Pointer to start of block row in the block transformation matrix
			pBlock = bufs->blockBuf + dimInfo->blkLenDWords * rowNo;

			// Write the block row
			actualWriteDataSize = (INT32) RFD_FWRITE(pBlock, 1, blockTransformStorageSize, blockFileHandle);
			if((UINT32)actualWriteDataSize != blockTransformStorageSize) {
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_WRITE;
			}

			// seek to start of block in next super block row in preparation for the next block write.
			if(RFD_FSEEK(blockFileHandle, superBlockTransformStorageSize - blockTransformStorageSize, SEEK_CUR)) {
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_SEEK;
			}
		}

		// Write the rows that do not implement pivot postions
		// and that are not linearly dependent.
		for(iRow=0;iRow<dimInfo->nRowsTotal;iRow++) {

			if(XCC_DEC_ROW_INFO_IS_PIVOT_ROW(bufs->rowInfo[iRow])) {
				// This row implements a pivot postion and would have
				// been written in the previous loop. Continue
				// the the next current row.
				continue;
			}

			// The current row is not a pivot row.

			// The progress number corresponding to this current row.
			progressNo = XCC_DEC_ROW_INFO_GET_NEXT_POSITION(bufs->rowInfo[iRow]);

			if(progressNo == dimInfo->nEquVars) {
				// This row is linearly dependent. Eliminate it by not
				// writing it back to files.
				continue;
			}

			// Pointer to start of block row in the block transformation matrix
			pBlock = bufs->blockBuf + dimInfo->blkLenDWords * iRow;

			// Write the block row
			actualWriteDataSize = (INT32) RFD_FWRITE(pBlock, 1, blockTransformStorageSize, blockFileHandle);
			if((UINT32)actualWriteDataSize != blockTransformStorageSize) {
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_WRITE;
			}

			// seek to start of block in next super block row in preparation for the next block write.
			if(RFD_FSEEK(blockFileHandle, superBlockTransformStorageSize - blockTransformStorageSize, SEEK_CUR)) {
				RFD_FCLOSE(blockFileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_SEEK;
			}
		}

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
		// File-Sync the block file to disk as part of
		// RFD storage transaction management to handle power-loss or other 
		// program abort conditions.
		
		// Must perform fflush to flush the upper layer buffers before file sync.
		RFD_FFLUSH(blockFileHandle);

		// Do the File Sync.
		RFD_FileSyncByHandle(blockFileHandle);
#endif
		// close the block transformation matrix files
		RFD_FCLOSE(blockFileHandle);
		blockFileHandle = NULL;

		if(dimInfo->superBlkInfo.isLastBlock) {

			////////////////
			// This is the last block of the super block.
			// There are no further block decodes for this decode operation.
			////////////////

			////////////////
			// There are no further block decodes in which an abort (exit request) might occur
			// and in which the original block transform file will be reverted to.
			// Delete the original (old) block transform file.
			// Then Rename the temporary block transform file back to the permanent block transform file.
			////////////////

			fileNameBuf2 = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
			if(fileNameBuf2 == NULL) {
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_MEM_ALLOC;
			}
			RFD_STRCPY_S(fileNameBuf2, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf2, RFD_MAX_PATH_LEN, fileNameList->decodeOutputFileName);

			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->blockTransformFileName);

			RFD_DELETE_FILE(fileNameBuf);

			RFD_RENAME(fileNameBuf2, fileNameBuf);

			RFD_FREE(fileNameBuf2);
			fileNameBuf2 = NULL;


			////////////////
			// This is the last block of the super block.
			// There are no further block decodes,
			// so store all data and info required for the next decode operation.
			////////////////


			// Allocate a temporary rowInfo array to hold the updated row sequence
			// i.e. the order in which rows are written to the transformation matrices
			// files may be different from their current order.
			newRowInfo = (INT16 *) RFD_MALLOC(dimInfo->nRowsTotal * sizeof(INT16));
			if(newRowInfo == NULL) {
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_MEM_ALLOC;
			}

			////////////////
			// Write transformation coeff matrix back to the
			// corresponding file.
			////////////////

			// Open the Coefficient transformation matrix file.
			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->coeffTransformFileName);

			if(RFD_FOPEN_S(&coeffFileHandle, fileNameBuf, TEXT("wb"))) {
				RFD_FCLOSE(coeffFileHandle);
				RFD_FREE(newRowInfo);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_OPEN;
			}

			// Write the rows implementing pivots in order of the pivot index

			// The coefficient rows are written using the rounded-up DWord aligned size.
			coeffTransformStorageSize = dimInfo->coeffRowLenDWords * sizeof(DWORD);

			iNewRow = 0;

			for(iPivot=0;iPivot<dimInfo->nEquVars;iPivot++) {

				if(!XCC_DEC_PIVOT_INFO_IS_PIVOT(bufs->pivotInfo[iPivot])) {
					// No row implements this pivot postion yet,
					// continue to process next pivot position.
					continue;
				}

				// The current pivot has an implemented row.

				// The row corresponding to this pivot position.
				rowNo = XCC_DEC_PIVOT_INFO_GET_ROW(bufs->pivotInfo[iPivot]);

				if(rowNo >= dimInfo->nRowsTotal) {
					// Error, the row number is out of range.
					RFD_FCLOSE(coeffFileHandle);
					RFD_FREE(newRowInfo);
					RFD_FREE(fileNameBuf);
					return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
				}

				// The progress number corresponding to this row.
				progressNo = XCC_DEC_ROW_INFO_GET_NEXT_POSITION(bufs->rowInfo[rowNo]);

				// Set the progress number and isPivot=TRUE indication
				// in the new RowInfo structure.
				XCC_DEC_SET_ROW_INFO(newRowInfo[iNewRow], progressNo, TRUE);

				// Update the pivotInfo current pivot position with the new row number.
				XCC_DEC_SET_PIVOT_INFO_PIVOT_TRUE(bufs->pivotInfo[iPivot], iNewRow);
				// Increment the written row counter.
				iNewRow++;

				// Pointer to start of coefficient row in the coeff transformation matrix
				pCoeff = bufs->coeffBuf + dimInfo->coeffRowLenDWords * rowNo;

				// Write the coefficient row
				actualWriteDataSize = (INT32) RFD_FWRITE(pCoeff, 1, coeffTransformStorageSize, coeffFileHandle);
				if((UINT32)actualWriteDataSize != coeffTransformStorageSize) {
					RFD_FCLOSE(coeffFileHandle);
					RFD_FREE(newRowInfo);
					RFD_FREE(fileNameBuf);
					return RFD_STATUS_ERROR_FILE_WRITE;
				}
			}

			// Write the rows that do not implement pivot postions
			// and that are not linearly dependent.
			for(iRow=0;iRow<dimInfo->nRowsTotal;iRow++) {

				if(XCC_DEC_ROW_INFO_IS_PIVOT_ROW(bufs->rowInfo[iRow])) {
					// This row implements a pivot postion and would have
					// been written in the previous loop. Continue
					// to the next current row.
					continue;
				}

				// The current row is not a pivot row.

				// The progress number corresponding to this current row.
				progressNo = XCC_DEC_ROW_INFO_GET_NEXT_POSITION(bufs->rowInfo[iRow]);

				if(progressNo == dimInfo->nEquVars) {
					// This row is linearly dependent. Eliminate it by not
					// writing it back to files.
					continue;
				}

				// Set the progress number and isPivot=FALSE indication
				// in the new RowInfo structure.
				XCC_DEC_SET_ROW_INFO(newRowInfo[iNewRow], progressNo, FALSE);

				// Increment the written row counter.
				iNewRow++;

				// Pointer to start of coefficient row in the coeff transformation matrix
				pCoeff = bufs->coeffBuf + dimInfo->coeffRowLenDWords * iRow;

				// Write the coefficient row
				actualWriteDataSize = (INT32) RFD_FWRITE(pCoeff, 1, coeffTransformStorageSize, coeffFileHandle);
				if((UINT32)actualWriteDataSize != coeffTransformStorageSize) {
					RFD_FCLOSE(coeffFileHandle);
					RFD_FREE(newRowInfo);
					RFD_FREE(fileNameBuf);
					return RFD_STATUS_ERROR_FILE_WRITE;
				}
			}

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
			// File-Sync the coeff file to disk as part of
			// RFD storage transaction management to handle power-loss or other 
			// program abort conditions.
		
			// Must perform fflush to flush the upper layer buffers before file sync.
			RFD_FFLUSH(coeffFileHandle);

			// Do the File Sync.
			RFD_FileSyncByHandle(coeffFileHandle);
#endif
			// close the coeff transformation matrix files
			RFD_FCLOSE(coeffFileHandle);
			coeffFileHandle = NULL;


			////////////////
			// Update state info with the number of rows now in the transformation matrices.
			////////////////
			stateInfo->nTransformRows = iNewRow;

			////////////////
			// Write the new rowInfo and updated pivotInfo to the transformation info file.
			////////////////

			// Open the file.

			RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
			RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->transformationInfoFileName);

			if(RFD_FOPEN_S(&fileHandle, fileNameBuf, TEXT("wb"))) {
				RFD_FREE(newRowInfo);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_OPEN;
			}

			// Write the new RowInfo Array

			dataSize = iNewRow * sizeof(INT16);
			actualWriteDataSize = (INT32) RFD_FWRITE(newRowInfo, 1, dataSize, fileHandle);
			if(actualWriteDataSize != dataSize) {
				RFD_FREE(newRowInfo);
				RFD_FCLOSE(fileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_READ;
			}

			// Write the PivotInfo Array

			dataSize = dimInfo->nEquVars * sizeof(INT16);
			actualWriteDataSize = (INT32) RFD_FWRITE(bufs->pivotInfo, 1, dataSize, fileHandle);
			if(actualWriteDataSize != dataSize) {
				RFD_FREE(newRowInfo);
				RFD_FCLOSE(fileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_READ;
			}

#if IS_RFDR_DIR_AND_FILE_SYNC_ENABLED
			// File-Sync the tranformation info file to disk as part of
			// RFD storage transaction management to handle power-loss or other 
			// program abort conditions.
		
			// Must perform fflush to flush the upper layer buffers before file sync.
			RFD_FFLUSH(fileHandle);

			// Do the File Sync.
			RFD_FileSyncByHandle(fileHandle);
#endif
			// Close the transformation info file.
			RFD_FCLOSE(fileHandle);
			fileHandle = NULL;

			// Free the temporary new rowInfo structure.
			RFD_FREE(newRowInfo);
			newRowInfo = NULL;
		}
		// else not the last block of the super block, decoding info and data
		// should Not be stored.
		// The original input info and data needs to be maintaind
		// for each block decode of the super block.
	}

	// Free the file name buffer.
	RFD_FREE(fileNameBuf);
	fileNameBuf = NULL;

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Stores the system of equations.
///
/// @param	pathBuf			directory path string.
/// @param	fileNameList	list of file names.
/// @param	bufs			the bufs.
/// @param	dimInfo			information describing the dimension.
/// @param	stateInfo		information describing the state.
///
/// @return	RFD_STATUS_OK if successful.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _StoreSysOfEquations(
					const TCHAR pathBuf[],
					const RFD_DECODE_XCC_FILE_NAME_LIST * fileNameList,
					XCC_DEC_DYNAMIC_BUFS * bufs,
					XCC_DEC_DIMENSION_INFO * dimInfo,
					RFD_XCC_DECODE_STATE_INFO * stateInfo)
{
	RFD_STATUS status;

	if(dimInfo->superBlkInfo.nBlocks == 1) {
		status = _StoreSysOfEquationsSingleBlockPerSuperBlock(
												pathBuf,
												fileNameList,
												bufs,
												dimInfo,
												stateInfo);
	}
	else {
		status = _StoreSysOfEquationsMultipleBlocksPerSuperBlock(
												pathBuf,
												fileNameList,
												bufs,
												dimInfo,
												stateInfo);
	}

	return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Loads the current system of equations.
///
/// @param	pathBuf			Directory path string.
/// @param	fileNameList	list of file names.
/// @param	bufs			the bufs.
/// @param	dimInfo			information describing the dimension.
/// @param	stateInfo		information describing the state.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _LoadCurrSysOfEquations(
					const TCHAR pathBuf[],
					const RFD_DECODE_XCC_FILE_NAME_LIST * fileNameList,
					XCC_DEC_DYNAMIC_BUFS * bufs,
					XCC_DEC_DIMENSION_INFO * dimInfo,
					RFD_XCC_DECODE_STATE_INFO * stateInfo)
{
	DWORD * blockBufPtr;
	int iRow;
	RFD_FILE* fileHandle = NULL;
	TCHAR * fileNameBuf = NULL;
	UINT16 nTransformRows;
	UINT32 actualReadDataSize, dataSize;
	UINT32 blockTransformStorageSize, superBlockTransformStorageSize;

	nTransformRows = stateInfo->nTransformRows;

	////////////////
	// Allocate the file name buffer.
	////////////////

	fileNameBuf = (TCHAR *) RFD_MALLOC( sizeof(TCHAR) * RFD_MAX_PATH_LEN );
	if(fileNameBuf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}


	////////////////
	// Read the equation RowInfo array and the PivotInfo array
	// from the single transformationInfo file
	////////////////

	// Open the file.

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->transformationInfoFileName);

	if(RFD_FOPEN_S(&fileHandle, fileNameBuf, TEXT("rb"))) {
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	// Read the RowInfo Array

	dataSize = nTransformRows * sizeof(INT16);
	actualReadDataSize = (UINT32) RFD_FREAD(bufs->rowInfo, 1, dataSize, fileHandle);
	if(actualReadDataSize != dataSize) {
		RFD_FCLOSE(fileHandle);
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_READ;
	}

	// Read the PivotInfo Array

	dataSize = dimInfo->nEquVars * sizeof(INT16);
	actualReadDataSize = (UINT32) RFD_FREAD(bufs->pivotInfo, 1, dataSize, fileHandle);
	if(actualReadDataSize != dataSize) {
		RFD_FCLOSE(fileHandle);
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_READ;
	}

	// Check if file file size matches the requested data size.
	dataSize = (nTransformRows * sizeof(INT16)) +
		       (dimInfo->nEquVars * sizeof(INT16));

	if(RFD_FSEEK(fileHandle, 0, SEEK_END)) {
		RFD_FCLOSE(fileHandle);
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_SEEK;
	}

	if((UINT32)RFD_FTELL(fileHandle) != dataSize) {
		RFD_FCLOSE(fileHandle);
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
	}

	RFD_FCLOSE(fileHandle);

	////////////////
	// Read the Coefficient part of the current solution.
	////////////////

	// Open the file.

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->coeffTransformFileName);

	if(RFD_FOPEN_S(&fileHandle, fileNameBuf, TEXT("rb"))) {
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	// Read the file.

	dataSize = nTransformRows * dimInfo->coeffRowLenDWords * sizeof(DWORD);
	actualReadDataSize = (UINT32) RFD_FREAD(bufs->coeffBuf, 1, dataSize, fileHandle);
	if(actualReadDataSize != dataSize) {
		RFD_FCLOSE(fileHandle);
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_READ;
	}

	// Check if file file size matches the requested data size.

	if(RFD_FSEEK(fileHandle, 0, SEEK_END)) {
		RFD_FCLOSE(fileHandle);
		return RFD_STATUS_ERROR_FILE_SEEK;
	}

	if((UINT32)RFD_FTELL(fileHandle) != dataSize) {
		RFD_FCLOSE(fileHandle);
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
	}

	RFD_FCLOSE(fileHandle);


	////////////////
	// Read the Block data part of the current solution.
	////////////////

	// Open the file

	RFD_STRCPY_S(fileNameBuf, RFD_MAX_PATH_LEN, pathBuf);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, RFD_PATH_SEPARATOR);
	RFD_STRCAT_S(fileNameBuf, RFD_MAX_PATH_LEN, fileNameList->blockTransformFileName);

	if(RFD_FOPEN_S(&fileHandle, fileNameBuf, TEXT("rb"))) {
		RFD_FREE(fileNameBuf);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	// Read the file.

	if(dimInfo->superBlkInfo.nBlocks ==	1) {

		// The storage block (super block) is processed as a single (sub) block.

		dataSize = nTransformRows * dimInfo->blkLenDWords * sizeof(DWORD);
		actualReadDataSize = (UINT32) RFD_FREAD(bufs->blockBuf, 1, dataSize, fileHandle);
		if(actualReadDataSize != dataSize) {
			RFD_FCLOSE(fileHandle);
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_READ;
		}

		// Check if file size matches the requested data size.

		if(RFD_FSEEK(fileHandle, 0, SEEK_END)) {
			RFD_FCLOSE(fileHandle);
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_SEEK;
		}

		if((UINT32)RFD_FTELL(fileHandle) != dataSize) {
			RFD_FCLOSE(fileHandle);
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_SIZE_CHECK;
		}
	}
	else {
		// The storage block (super block) is processes as multiple (sub) blocks.
		// Read the block data for the current block of the super block.

		// Initialize block buffer pointer for the first block read.
		blockBufPtr = bufs->blockBuf;

		// Round-up the super Block storage size to a multiple of DWords.
		// This is done for compatibility with the way blocks/super blocks are stored for the simple
		// case of one block per super block.
		superBlockTransformStorageSize = dimInfo->superBlkInfo.superBlkLenDWords * sizeof(DWORD);

		if(dimInfo->superBlkInfo.isLastBlock) {
			// This is the last block of the super block. Round-up this block storage size
			// so that it 'fills' up to the DWord multiple super block row size.
			// This is done for compatibility with the way blocks/super blocks are stored for the simple
			// case of one block per super block.
			// e.g. the first series of progressive decode operations may implement one block per super block
			// (due to fewer number of available rows) while the last decode operations will require
			// multiple blocks per super block (due to more available rows). In both cases,
			// the storage alignment must be compatible.

			if((dimInfo->superBlkInfo.blkOffsetBytes + dimInfo->blkLenBytes) < superBlockTransformStorageSize) {
				blockTransformStorageSize = dimInfo->blkLenBytes +
					superBlockTransformStorageSize - (dimInfo->superBlkInfo.blkOffsetBytes + dimInfo->blkLenBytes);
			}
			else {
				blockTransformStorageSize = dimInfo->blkLenBytes;
			}
		}
		else {
			// Not the last block.
			blockTransformStorageSize = dimInfo->blkLenBytes;
		}

		// Initialize file position for the first block read.
		if(RFD_FSEEK(fileHandle, dimInfo->superBlkInfo.blkOffsetBytes, SEEK_CUR)) {
			RFD_FCLOSE(fileHandle);
			RFD_FREE(fileNameBuf);
			return RFD_STATUS_ERROR_FILE_SEEK;
		}

		for(iRow=0;iRow<nTransformRows;iRow++) {

			// Read block data for the current row into the block buffer.
			actualReadDataSize = (UINT32) RFD_FREAD(blockBufPtr, 1, blockTransformStorageSize, fileHandle);
			if(actualReadDataSize != blockTransformStorageSize) {
				RFD_FCLOSE(fileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_READ;
			}

			// Update block buffer pointer for the next read.
			// Blocks in the block Buffer are DWord aligned.
			blockBufPtr += dimInfo->blkLenDWords;

			// Update file position for the next read.
			// For the case of multiple blocks per super block, each row of the block transform matrix
			// is not rounded to dwords. The length of each row is therefore superBlkLenBytes.
			if(RFD_FSEEK(fileHandle, superBlockTransformStorageSize - blockTransformStorageSize, SEEK_CUR)) {
				RFD_FCLOSE(fileHandle);
				RFD_FREE(fileNameBuf);
				return RFD_STATUS_ERROR_FILE_SEEK;
			}
		}
	}

	RFD_FCLOSE(fileHandle);

	////////////////
	// Cleanup and return.
	////////////////

	RFD_FREE(fileNameBuf);

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @fn	static RFD_STATUS _Decode( XCC_DEC_DYNAMIC_BUFS * bufs, XCC_DEC_DIMENSION_INFO * dimInfo,
///  RFD_XCC_DECODE_STATE_INFO * stateInfo, UCHAR * exitRequestFlagPtr)
///
/// @brief	Incremental Decode system of equations
///
/// @param [in,out]	bufs				the bufs.
/// @param [in    ]	dimInfo				information describing the dimension.
/// @param [in,out]	stateInfo			information describing the state.
/// @param [in    ]	exitRequestFlagPtr	the exit request flag pointer.
///
/// RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS _Decode(  XCC_DEC_DYNAMIC_BUFS * bufs,
					 XCC_DEC_DIMENSION_INFO * dimInfo,
					 RFD_XCC_DECODE_STATE_INFO * stateInfo,
					 UCHAR * exitRequestFlagPtr)
{
	register DWORD * pDst, * pSrc;
	register int i;
	register int nWordsToAdd;
	int iRow, iPivot, nRowsTotal, pivotRowNo = 0 /* avoid compiler warning */;
	int adjustedCoeffRowLenDWords, blkLenDWords, dstNextPivotNo;
	int dstCoeffElement, srcCoeffElement, coeffElemBitShift;
	int coeffStartDWordOffset;
	INT16 rowInfo;
	DWORD * pSrcBlkRowStart, * pSrcCoeffStart;
	DWORD * pDstCoeffStart;
	XCC_ROW_INFO_BLOCK_TYPE_ENUM pivotRowType = XCC_ROW_INFO_SOURCE_TYPE_BLOCK /* avoid compiler warning */;
	BOOL foundPivot;
#if XCC_BITS_PER_ELEMENT > 1
	XCC_ROW_SCALE_INFO * scalePivotInfo = (XCC_ROW_SCALE_INFO *) bufs->rowScaleInfo;
#endif
	BOOL isLastPivot;

	nRowsTotal = stateInfo->nTransformRows +
				 stateInfo->nSrcBlksInBuf +
				 stateInfo->nChkBlksInBuf;
	blkLenDWords = dimInfo->blkLenDWords;

#if XCC_BITS_PER_ELEMENT > 1
	scalePivotInfo->coeffRowLenDWords = dimInfo->coeffRowLenDWords;
	scalePivotInfo->blkLenDWords = dimInfo->blkLenDWords;
#endif

	//for(iPivot=0;iPivot<stateInfo->nextPivotNo;iPivot++) {
	//for(iPivot=stateInfo->nextPivotNo;iPivot<dimInfo->nEquVars;iPivot++) {
	for(iPivot=0;iPivot<dimInfo->nEquVars;iPivot++) {

		if(iPivot != (dimInfo->nEquVars-1)) {
			isLastPivot = FALSE;
		}
		else {
			isLastPivot = TRUE;
		}

		coeffStartDWordOffset = (iPivot * XCC_BITS_PER_ELEMENT) / RFD_BITS_PER_DWORD;
		// The number of bits to shift a coefficient element to the
		// least significant part of a DWord.
		coeffElemBitShift = RFD_BITS_PER_DWORD -
							((iPivot * XCC_BITS_PER_ELEMENT) % RFD_BITS_PER_DWORD) -
							XCC_BITS_PER_ELEMENT;
		adjustedCoeffRowLenDWords = dimInfo->coeffRowLenDWords - coeffStartDWordOffset;

		if(!XCC_DEC_PIVOT_INFO_IS_PIVOT(bufs->pivotInfo[iPivot])) {
			// No row implemented yet for this pivot position.
			pSrcCoeffStart = bufs->coeffBuf + coeffStartDWordOffset;
			foundPivot = FALSE;

			// note: possible optimization is to change row search range to bypass
			// the new source-type input buf rows
			// (these candidate pivots would have been detected in bufs->pivotInfo[iPivot])
			for(iRow=0;iRow<nRowsTotal;iRow++) { // Loop to Find a Pivot Row

				rowInfo = bufs->rowInfo[iRow];
				if(!XCC_DEC_ROW_INFO_IS_PIVOT_ROW(rowInfo)) {
					// This row does not implement a pivot postion
					if(XCC_DEC_ROW_INFO_GET_NEXT_POSITION(rowInfo) == iPivot) {
						// the Next Postion (last potential non-zero position)
						// for this non-pivot row equals the current pivot position,
						// so this is a possible pivot row to select, pending
						// the final check below.
						srcCoeffElement = (*pSrcCoeffStart >> coeffElemBitShift) & XCC_ELEMENT_BIT_MASK;
						if(srcCoeffElement) {
							// Found a row with a non-zero element at this pivot position.
							// Indicate this row as a pivot row,
							// increment the 'next pivot postion to clear',
							// then break out of pivot row search loop
							// to do substitution of this row into the other rows.

							pivotRowNo = iRow;
							// any row that is not a pivot must be a check type block.
							pivotRowType = XCC_ROW_INFO_CHECK_TYPE_BLOCK;
							XCC_DEC_SET_PIVOT_INFO_PIVOT_TRUE(bufs->pivotInfo[iPivot], pivotRowNo);
							XCC_DEC_SET_ROW_INFO(bufs->rowInfo[iRow], (iPivot+1), TRUE /*is pivot*/);
							foundPivot = TRUE;
							break;
						}
						else {
							// Update the Next Pivot Number info for this row, since this pivot position detected as zero.
							// Note: this row must be a Check-type row, otherwise, as a Source-Type row,
							// it would have been indicated as a pivot postion and not processed here.
							XCC_DEC_SET_ROW_INFO(bufs->rowInfo[iRow], (iPivot+1), FALSE /*not a pivot*/);

							// This row is linearly dependent with the other rows of the current system of equations.
							// Update nChkBlksEliminated.
							if(isLastPivot) {
								stateInfo->nChkBlksEliminated++;
							}
						}
					}
				}

				pSrcCoeffStart += dimInfo->coeffRowLenDWords;
			} // Find a Pivot Row Search Loop

			if(!foundPivot) {
				break; // break out of the main pivot advancement loop.
			}
		}
		else {
			// found a row that is already indicated as implementing a pivot at this postion.
			pivotRowNo = XCC_DEC_PIVOT_INFO_GET_ROW(bufs->pivotInfo[iPivot]);

			if(XCC_DEC_ROW_INFO_GET_NEXT_POSITION(bufs->rowInfo[pivotRowNo]) == dimInfo->nEquVars) {
				pivotRowType = XCC_ROW_INFO_SOURCE_TYPE_BLOCK;
			}
			else {
				pivotRowType = XCC_ROW_INFO_CHECK_TYPE_BLOCK;
			}
			pSrcCoeffStart = bufs->coeffBuf + (pivotRowNo * dimInfo->coeffRowLenDWords) +
							 coeffStartDWordOffset;
			srcCoeffElement = (*pSrcCoeffStart >> coeffElemBitShift) & XCC_ELEMENT_BIT_MASK;
		}

		pSrcBlkRowStart = bufs->blockBuf + (pivotRowNo * dimInfo->blkLenDWords);

#if XCC_BITS_PER_ELEMENT > 1
		scalePivotInfo->srcCoeffElement = (*pSrcCoeffStart >> coeffElemBitShift) & XCC_ELEMENT_BIT_MASK;
		scalePivotInfo->iNextAvailableBuf = 0;
		scalePivotInfo->rowType = pivotRowType;
		scalePivotInfo->srcCoeffRow = bufs->coeffBuf + (pivotRowNo * dimInfo->coeffRowLenDWords); // not adjusted pSrcBlkRowStart
		scalePivotInfo->adjustedCoeffRowLenDWords = adjustedCoeffRowLenDWords;
		scalePivotInfo->coeffStartDWordOffset = coeffStartDWordOffset;
		scalePivotInfo->srcBlockRow = pSrcBlkRowStart;

		for(i=0;i<XCC_NUM_ELEMENTS;i++) {
			scalePivotInfo->blockStart[i] = NULL;
			scalePivotInfo->coeffStart[i] = NULL;
		}

		// Can initialize for the trivial case here, for when the dst (pivot) element
		// is equal to the src (pivot) element (i.e. no scale needed).
		scalePivotInfo->blockStart[scalePivotInfo->srcCoeffElement] = pSrcBlkRowStart;
		scalePivotInfo->coeffStart[scalePivotInfo->srcCoeffElement] = pSrcCoeffStart;
#endif

		// Loop for 'substituting' the current pivot row into all other rows
		for(iRow=0;iRow<nRowsTotal;iRow++) {
			rowInfo = bufs->rowInfo[iRow];

			dstNextPivotNo = XCC_DEC_ROW_INFO_GET_NEXT_POSITION(rowInfo);
			if(dstNextPivotNo == iPivot) {
				// This row's next reduction level is equal to the
				// current pivot level, so continue checks to reduce this row.
				pDstCoeffStart = bufs->coeffBuf + (iRow * dimInfo->coeffRowLenDWords) +
					coeffStartDWordOffset;
				dstCoeffElement = (*pDstCoeffStart >> coeffElemBitShift) & XCC_ELEMENT_BIT_MASK;
				if(dstCoeffElement) {
					// The coefficient is non-zero, so a row addition is necessary to
					// reduce the row at this pivot position.

					pDst = pDstCoeffStart;
#if XCC_BITS_PER_ELEMENT == 1
					pSrc = pSrcCoeffStart;
#else
					//if(dstCoeffElement != srcCoeffElement) {
					if(scalePivotInfo->coeffStart[dstCoeffElement] == NULL) {
						_ScaleRow(scalePivotInfo, dstCoeffElement);
					}
					pSrc = scalePivotInfo->coeffStart[dstCoeffElement];
#endif

					// NOTE: Optimization should be done here:
					// In the case of 'catchup mode' (iPivot < stateInfo->nextPivotNo)
					// a row addition should be done at the pivot position coeff word
					// and then all coeff words after the pivot row's 'Next Postion'
					// (in some cases the pivot postion word is the same word
					// as the Next Postion word).
					// The same optimization applies to the _ScaleRow()
					// and the _CopyBackScaledRow() functions (all should be implemented
					// in a consistent way)
					//
					// Example, addition only needed at the following postions (^)
					// of the example pivot row.
					//
					// 00000100000000xxxxxxx......x
					//      ^        ^^^^^^^^^^^^^^

					if(pivotRowType == XCC_ROW_INFO_CHECK_TYPE_BLOCK) {
						nWordsToAdd = adjustedCoeffRowLenDWords;
					}
					else {
						// else the pivot row is a Source-Type row, which have all
						// zero coefficients except the pivot postion coefficient.
						// So no need to add any other coefficient postions besides
						// the pivot position.
						nWordsToAdd = 1;
					}

					/////////////////////////////
					// add pivot coefficient matrix row to the
					// destination coefficient row to clear the pivot postion
					// in the destination row.
					/////////////////////////////
					for(i=0;i<nWordsToAdd;i++) {
						*pDst ^= *pSrc;
						pDst++; pSrc++;
					}

					// pDstBlkRowStart
					pDst = bufs->blockBuf + (iRow * dimInfo->blkLenDWords);
#if XCC_BITS_PER_ELEMENT == 1
					pSrc = pSrcBlkRowStart;
#else
					pSrc = scalePivotInfo->blockStart[dstCoeffElement];
#endif
					nWordsToAdd = blkLenDWords;

					/////////////////////////////
					// add pivot block matrix row to the
					// destination block row
					/////////////////////////////
					for(i=0;i<nWordsToAdd;i++) {
						*pDst ^= *pSrc;
						pDst++; pSrc++;
					}
				}
				// Update the Next Pivot Number info for this row, since this pivot position is now zero.
				// Preserve the existing isPivot state.
				XCC_DEC_SET_ROW_INFO_COLUMN_NUM_ONLY(bufs->rowInfo[iRow], (dstNextPivotNo+1));
			}
		}

#if XCC_BITS_PER_ELEMENT > 1
		// If the (new) pivot row pivot element is not the identity elemement (1)
		// copy the scaled/generated identity element row back to the coeff row
		// in the coefficient matrix.

		if(scalePivotInfo->coeffStart[XCC_IDENTITY_ELEM] != pSrcCoeffStart)
		{
			// Generate the identity element row first if it
			// has not been generated yet.
			if(scalePivotInfo->coeffStart[XCC_IDENTITY_ELEM] == NULL) {
				_ScaleRow(scalePivotInfo, XCC_IDENTITY_ELEM);
			}
#if 1
			_CopyBackScaledRow(scalePivotInfo, XCC_IDENTITY_ELEM);
#else
			// Copy Coefficient part of Row

			pDst = pSrcCoeffStart; // writing back to the orig src.
			pSrc = scalePivotInfo->coeffStart[XCC_IDENTITY_ELEM];

			if(pivotRowType == XCC_ROW_INFO_CHECK_TYPE_BLOCK) {
				nWordsToAdd = adjustedCoeffRowLenDWords;
			}
			else {
				nWordsToAdd = 1;
			}

			for(i=0;i<nWordsToAdd;i++) {
				*pDst = *pSrc;
				pDst++; pSrc++;
			}

			// Copy Block part of Row
			pDst = pSrcBlkRowStart;
			pSrc = scalePivotInfo->blockStart[XCC_IDENTITY_ELEM];

			nWordsToAdd = blkLenDWords;
			for(i=0;i<nWordsToAdd;i++) {
				*pDst = *pSrc;
				pDst++; pSrc++;
			}
#endif
		}
#endif
		// Check for exit request on each pivotNo loop.
		if(*exitRequestFlagPtr) {
			return RFD_STATUS_EARLY_EXIT_ON_REQUEST;
		}
	}

	// Update nextPivotNo.
	stateInfo->nextPivotNo = iPivot;

	// Note: Decoding complete if
	// stateInfo->nextPivotNo == dimInfo->nEquVars

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Rfd eXtended ecc decode operation.
///
/// The operation includes loading data from storage into buffers, performing the decode, and
/// then storing decoded data back to storage. Decoding can be done incrementally in that
/// RFD_DECODE_XCC() may be called multiple times with new input blocks until the solution is
/// complete.
///
/// @param  pathBuf                             Directory path string.
/// @param  fileNameList                        List of file names.
/// @param  codecConfigInfo                     Information describing the codec configuration.
/// @param  stateInfo                           information describing the state.
/// @param  ConstructSrcTypeCoeffRowFunction    The construct source type coeff row function.
/// @param  ConstructChkTypeCoeffRowFunction    The construct check type coeff row function.
/// @param  maxCheckBlockIndex                  maximum check block index value.
/// @param  maxExtraBlocksForSolution           The maximum number of extra blocks for solution.
/// @param  exitRequestFlagPtr                  The exit request flag pointer.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_DECODE_XCC(const TCHAR pathBuf[],
						  const RFD_DECODE_XCC_FILE_NAME_LIST * fileNameList,
						  RFD_XCC_CODEC_CONFIG_INFO * codecConfigInfo,
						  RFD_XCC_DECODE_STATE_INFO * stateInfo,
						  FcnPtrConstructSrcTypeCoeffRow ConstructSrcTypeCoeffRowFunction,
						  FcnPtrConstructChkTypeCoeffRow ConstructChkTypeCoeffRowFunction,
						  UINT32 maxCheckBlockIndex,
						  UINT16 maxExtraBlocksForSolution,
						  UCHAR * exitRequestFlagPtr )
{
	XCC_DEC_DIMENSION_INFO dimInfo;
	XCC_DEC_DYNAMIC_BUFS bufs;
	RFD_STATUS status;
	RFD_XCC_DECODE_STATE_INFO initialStateInfo;
	int iBlock;
	INT32 availMemForBlockData, decodeBaseMemoryUsage;

	RFD_DPrint( TEXT("RFDR XCC Decode: Decode Start, stateInfo: \n\
		blksCollectedEventThresh %d\n\
		decodeChunkCount %d\n\
		nSrcBlksCollectedTotal %d\n\
		nChkBlksCollectedTotal %d\n\
		nSrcBlksInBuf %d\n\
		nChkBlksInBuf %d\n\
		nSrcBlocksInSolution %d\n\
		nTransformRows %d\n\
		nextPivotNo %d\n\
		nChkBlksEliminated %d\n\
		"),\
		stateInfo->blksCollectedEventThresh,
		stateInfo->decodeChunkCount,
		stateInfo->nSrcBlksCollectedTotal,
		stateInfo->nChkBlksCollectedTotal,
		stateInfo->nSrcBlksInBuf,
		stateInfo->nChkBlksInBuf,
		stateInfo->nSrcBlocksInSolution,
		stateInfo->nTransformRows,
		stateInfo->nextPivotNo,
		stateInfo->nChkBlksEliminated
		);

	if(stateInfo->nSrcBlksInBuf == 0 && stateInfo->nChkBlksInBuf == 0) {
		// Decoder shouldn't be called if there is nothing to decode
		return RFD_STATUS_ERROR_DECODER_NUM_INPUT_BLKS;
	}

	///////////////////////////////////////////////////////////////////////////
	// Calculate dimension/size parameters that will be used in decoding.
	// and decoding setup.
	///////////////////////////////////////////////////////////////////////////

	dimInfo.nRowsTotal = stateInfo->nTransformRows +
						 stateInfo->nChkBlksInBuf +
						 stateInfo->nSrcBlksInBuf;

	dimInfo.nEquVars = codecConfigInfo->fileLenBlocks;

	if(dimInfo.nRowsTotal > (dimInfo.nEquVars + maxExtraBlocksForSolution)) {
		return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
	}

	dimInfo.coeffRowLenDWords = (dimInfo.nEquVars * XCC_BITS_PER_ELEMENT) / RFD_BITS_PER_DWORD +
		(((dimInfo.nEquVars * XCC_BITS_PER_ELEMENT) % RFD_BITS_PER_DWORD) ? 1:0);

	dimInfo.superBlkInfo.superBlkLenBytes = codecConfigInfo->blkLenBytes;

	dimInfo.superBlkInfo.superBlkLenDWords = codecConfigInfo->blkLenBytes / sizeof(DWORD) /* RFD_SIZE_OF_DWORD */ +
		((codecConfigInfo->blkLenBytes % sizeof(DWORD)) ? 1:0);

	dimInfo.superBlkInfo.lastSuperBlkLenBytes = ((codecConfigInfo->fileLenBytes % codecConfigInfo->blkLenBytes) ?
		(codecConfigInfo->fileLenBytes % codecConfigInfo->blkLenBytes) : codecConfigInfo->blkLenBytes);

	///////////////////////////////////////////////////////////////////////////
	// Calculate maximum block length, and based on this
	// calculate the number of blocks the super block is split into
	// for decoding.
	///////////////////////////////////////////////////////////////////////////

	decodeBaseMemoryUsage = _CalculateDecodeBaseMemoryUsage(&dimInfo);

	availMemForBlockData = RFDR_MAX_DECODE_MEMORY_BYTES - decodeBaseMemoryUsage;

	if(availMemForBlockData <= 0) {
		return RFD_STATUS_ERROR_DECODER_INSUFFICIENT_MEMORY;
	}

#if (XCC_BITS_PER_ELEMENT > 1)
	dimInfo.superBlkInfo.maxBlkLenBytes = availMemForBlockData / (dimInfo.nRowsTotal + XCC_SCALE_ROW_NUM_BUFS);
#else
	dimInfo.superBlkInfo.maxBlkLenBytes = availMemForBlockData / dimInfo.nRowsTotal;
#endif
	// round-down maxBlkLenBytes to multiple of dword since block buffer allocation rounds-up blocks to dword multiple.
	dimInfo.superBlkInfo.maxBlkLenBytes -= (dimInfo.superBlkInfo.maxBlkLenBytes % RFD_BYTES_PER_DWORD);

	if(dimInfo.superBlkInfo.maxBlkLenBytes == 0) {
		return RFD_STATUS_ERROR_DECODER_INSUFFICIENT_MEMORY;
	}

	dimInfo.superBlkInfo.nBlocks = (dimInfo.superBlkInfo.superBlkLenBytes / dimInfo.superBlkInfo.maxBlkLenBytes) +
		((dimInfo.superBlkInfo.superBlkLenBytes % dimInfo.superBlkInfo.maxBlkLenBytes) ? 1:0);

	///////////////////////////////////////////////////////////////////////////
	// Initialize blkOffsetBytes for first block decoding loop.
	///////////////////////////////////////////////////////////////////////////
	dimInfo.superBlkInfo.blkOffsetBytes = 0;

	///////////////////////////////////////////////////////////////////////////
	// Initialize initialStateInfo with input/output stateInfo.
	///////////////////////////////////////////////////////////////////////////
	initialStateInfo = *stateInfo;

	///////////////////////////////////////////////////////////////////////////
	// Process each (sub)block of the super Block.
	///////////////////////////////////////////////////////////////////////////
	for(iBlock=0;iBlock<dimInfo.superBlkInfo.nBlocks;iBlock++) {

		///////////////////////////////////////////////////////////////////////////
		// Update stateInfo with the initial state info before each (sub) block
		// decoding sequence.
		// At the end of the last block of the super block decoding sequence,
		// stateInfo will contain the updated info which is output to caller
		// (i.e. stateInfo will not be reinitialized to initialStateInfo).
		///////////////////////////////////////////////////////////////////////////
		*stateInfo = initialStateInfo;

		// Initialize isFirstBlock, isLastBlock status for current block of the super block.
		if(iBlock==0) {
			dimInfo.superBlkInfo.isFirstBlock = TRUE;
		}
		else {
			dimInfo.superBlkInfo.isFirstBlock = FALSE;
		}

		if( iBlock == (dimInfo.superBlkInfo.nBlocks-1) ) {
			dimInfo.superBlkInfo.isLastBlock = TRUE;
		}
		else {
			dimInfo.superBlkInfo.isLastBlock = FALSE;
		}

		// Initialize block length for current block of the super block.
		dimInfo.blkLenBytes = dimInfo.superBlkInfo.maxBlkLenBytes;
		if( (dimInfo.blkLenBytes + dimInfo.superBlkInfo.blkOffsetBytes) > dimInfo.superBlkInfo.superBlkLenBytes ) {
			dimInfo.blkLenBytes = dimInfo.superBlkInfo.superBlkLenBytes - dimInfo.superBlkInfo.blkOffsetBytes;
		}

		dimInfo.blkLenDWords = dimInfo.blkLenBytes / sizeof(DWORD) /* RFD_SIZE_OF_DWORD */ +
			((dimInfo.blkLenBytes % sizeof(DWORD)) ? 1:0);

		///////////////////////////////////////////////////////////////////////////
		// allocate memory for the buffers required by the decoder:
		//  - equation coefficient rows
		//  - blocks data rows
		//  - row info array
		//  - pivot info array
		///////////////////////////////////////////////////////////////////////////

		status = _AllocateBufs(&bufs, &dimInfo);

		if(status != RFD_STATUS_OK) {
			_CleanupTempStorageOnIncompleteDecode(pathBuf, fileNameList);
			return status;
		}

		///////////////////////////////////////////////////////////////////////////
		// Load input data from storage in preparation for decoding.
		// Data includes:
		// - the partially decoded solution "System of Equations"
		// - the newly received Source-Type Blocks
		// - the newly received Check(Encoded)-Type Blocks
		///////////////////////////////////////////////////////////////////////////

		// Load the current System of Equations (current transformation matrix).
		if(stateInfo->nTransformRows > 0) {
			status = _LoadCurrSysOfEquations( pathBuf, fileNameList,
											  &bufs, &dimInfo, stateInfo );
			if(status != RFD_STATUS_OK) {
				 _CleanupBufs(&bufs);
				 _CleanupTempStorageOnIncompleteDecode(pathBuf, fileNameList);
				 return status;
			}
		}
		else {
			// Initialize pivotInfo for first time decoding.
			status = _ResetPivotInfo(bufs.pivotInfo, &dimInfo);
			if(status != RFD_STATUS_OK) {
				 _CleanupBufs(&bufs);
				 _CleanupTempStorageOnIncompleteDecode(pathBuf, fileNameList);
				 return status;
			}
		}

		// Load the newly received Source-type blocks.
		if(stateInfo->nSrcBlksInBuf > 0) {
			status = _LoadSrcBlksInputBuf( pathBuf, fileNameList,
										   &bufs, &dimInfo, stateInfo,
										   ConstructSrcTypeCoeffRowFunction);
			if(status != RFD_STATUS_OK) {
				 _CleanupBufs(&bufs);
				 _CleanupTempStorageOnIncompleteDecode(pathBuf, fileNameList);
				 return status;
			}
		}

		// Load the newly received Check-type blocks
		if(stateInfo->nChkBlksInBuf > 0) {
			status = _LoadChkBlksInputBuf( pathBuf, fileNameList,
										   &bufs, &dimInfo, stateInfo,
										   ConstructChkTypeCoeffRowFunction,
										   maxCheckBlockIndex);
			if(status != RFD_STATUS_OK) {
				 _CleanupBufs(&bufs);
				 _CleanupTempStorageOnIncompleteDecode(pathBuf, fileNameList);
				 return status;
			}
		}

		///////////////////////////////////////////////////////////////////////////
		// Decode
		///////////////////////////////////////////////////////////////////////////

		status = _Decode(&bufs, &dimInfo, stateInfo, exitRequestFlagPtr);

		if(status != RFD_STATUS_OK) {
			// Error or early exit due to exit request
			 _CleanupBufs(&bufs);
			 _CleanupTempStorageOnIncompleteDecode(pathBuf, fileNameList);
			 return status;
		}

		///////////////////////////////////////////////////////////////////////////
		// Store result.
		///////////////////////////////////////////////////////////////////////////

		status = _StoreSysOfEquations( pathBuf, fileNameList,
									   &bufs, &dimInfo, stateInfo );
		if(status != RFD_STATUS_OK) {
			// Error
			 _CleanupBufs(&bufs);
			 _CleanupTempStorageOnIncompleteDecode(pathBuf, fileNameList);
			 return status;
		}

		///////////////////////////////////////////////////////////////////////////
		// Cleanup
		///////////////////////////////////////////////////////////////////////////

		_CleanupBufs(&bufs);

		///////////////////////////////////////////////////////////////////////////
		// Update blkOffsetBytes
		///////////////////////////////////////////////////////////////////////////
		dimInfo.superBlkInfo.blkOffsetBytes = dimInfo.superBlkInfo.blkOffsetBytes + dimInfo.blkLenBytes;
	}

	return RFD_STATUS_OK;
}


