////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_time.c
///
/// @brief	rfd time class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2013 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"
#include "rfd_time.h"
#include <time.h>

///////////////////////////////////////////////////////////////////////////////
static RFD_STATUS _RfdEpoc2StdBrokenOutTime( struct tm * stdBrokenOutTime )
{
    RFD_STATUS status = RFD_STATUS_OK;

    do { // once

        RFD_MEMSET( stdBrokenOutTime, 0, sizeof(struct tm));

        ///////////////////////////////////////////////////////////
        // Convert from SXI time to c standard lib Broken-out time.
        ///////////////////////////////////////////////////////////

        // Initialize to std epoc
        stdBrokenOutTime->tm_sec = 0;
        stdBrokenOutTime->tm_min = 0;
        stdBrokenOutTime->tm_hour = 0;
        stdBrokenOutTime->tm_mday = 1; // tm_mday is 1 relative
        stdBrokenOutTime->tm_mon = 0;
        stdBrokenOutTime->tm_year = 0;
        stdBrokenOutTime->tm_isdst = 0;
        stdBrokenOutTime->tm_wday = 0;
        stdBrokenOutTime->tm_yday = 0;

        // add RFD Epoc (relative to standard broken-out Epoc).
        stdBrokenOutTime->tm_year += (RFD_MDATA_LIFE_YEAR0 - RFD_STD_LIB_BROKEN_DOWN_TIME_YEAR0);

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
// input dispYear  "Absolute" Year e.g. 2015
// input dispMonth [1 to 12]
// input dispDay   [1 to 31]
static RFD_STATUS _DisplayYearMonthDay2StdBrokenOutTime(
                UINT32 dispYear,
                UINT32 dispMonth,
                UINT32 dispDay,
                struct tm * stdBrokenOutTime )
{
    RFD_STATUS status = RFD_STATUS_OK;

    do { // once

        ///////////////////////////////////////////////////////////
        // Validate the input time.
        ///////////////////////////////////////////////////////////

        if( dispMonth < 1 || dispMonth > 12 ||
            dispDay < 1 || dispDay > 31 ||
            dispYear < RFD_STD_LIB_BROKEN_DOWN_TIME_YEAR0 ) {

            // error, parameter out of range.
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        RFD_MEMSET( stdBrokenOutTime, 0, sizeof(struct tm));

        stdBrokenOutTime->tm_sec = 0;
        stdBrokenOutTime->tm_min = 0;
        stdBrokenOutTime->tm_hour = 0;
        // Both dispDay and tm_mday are 1 relative, so no offset needed.
        stdBrokenOutTime->tm_mday = dispDay;
        // Subtract one from dispMonth to account for dispMonth 1 relative vs. tm_mon 0 relative.
        stdBrokenOutTime->tm_mon = dispMonth-1;
        stdBrokenOutTime->tm_year = dispYear - RFD_STD_LIB_BROKEN_DOWN_TIME_YEAR0;
        stdBrokenOutTime->tm_isdst = 0;
        stdBrokenOutTime->tm_wday = 0;
        stdBrokenOutTime->tm_yday = 0;

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
time_t RFD_TIME_MakeSimpleCalendarTime( struct tm *brokentime )
{
    return RFD_MKTIME( brokentime );
}

///////////////////////////////////////////////////////////////////////////////
double RFD_TIME_DiffSimpleCalendarTime (time_t time1, time_t time0)
{
    return RFD_DIFFTIME( time1, time0 );
}

///////////////////////////////////////////////////////////////////////////////
BOOL RFD_TIME_IsSxiTimeFormatValid( const RFD_SXI_TIME_INFO_STRUCT * sxiTime )
{
    BOOL isFormatValid;

    if( sxiTime->minute >= 0 && sxiTime->minute <= 59 &&  // minute range 0 to 59
        sxiTime->hour   >= 0 && sxiTime->hour   <= 23 &&  // hour range 0 to 23
        sxiTime->day    >= 1 && sxiTime->day    <= 31 &&  // day range 1 to 31 (note: some months 30, feb sometimes 28 or 29)
        sxiTime->month  >= 1 && sxiTime->month  <= 12 ) { // month range 1 to 12

        isFormatValid = TRUE;
    }
    else {
        isFormatValid = FALSE;
    }

    return isFormatValid;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_TIME_SxiTime2StdCalendarTime(
                time_t * stdCalendarTimePtr,
                const RFD_SXI_TIME_INFO_STRUCT * sxiTime )
{
    RFD_STATUS status = RFD_STATUS_OK;
    struct tm stdBrokenOutTime;
    BOOL isValidSxiTimeFormat;

    do { // once

        ///////////////////////////////////////////////////////////
        // Validate the input sxi time.
        ///////////////////////////////////////////////////////////

        isValidSxiTimeFormat = RFD_TIME_IsSxiTimeFormatValid( sxiTime);

        if( isValidSxiTimeFormat == FALSE ) {

            // error, parameter out of range.
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        RFD_MEMSET( &stdBrokenOutTime, 0, sizeof(struct tm));

        ///////////////////////////////////////////////////////////
        // Convert from SXI time to c standard lib Broken-out time.
        ///////////////////////////////////////////////////////////

        stdBrokenOutTime.tm_sec = 0;
        stdBrokenOutTime.tm_min = sxiTime->minute;
        stdBrokenOutTime.tm_hour = sxiTime->hour;
        stdBrokenOutTime.tm_mday = sxiTime->day; // both SXI mday and standard broken-out mday are 1 relative.
        stdBrokenOutTime.tm_mon = sxiTime->month - 1; // only SXI month is relative to 1, not 0.
        stdBrokenOutTime.tm_year = sxiTime->yearSince2k +
                                    (RFD_SXI_YEAR0 - RFD_STD_LIB_BROKEN_DOWN_TIME_YEAR0);
        stdBrokenOutTime.tm_isdst = 0;
        stdBrokenOutTime.tm_wday = 0;
        stdBrokenOutTime.tm_yday = 0;

        ///////////////////////////////////////////////////////////
        // Convert c standard lib "broken-out" time to c standard lib "calendar" time which is
        // a format that facilitates time comparison operations for expiration checking.
        ///////////////////////////////////////////////////////////

        *stdCalendarTimePtr = RFD_TIME_MakeSimpleCalendarTime( &stdBrokenOutTime );

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_TIME_RfdLifeDays2StdCalendarTime(
                time_t * stdCalendarTimePtr,
                const INT32 rfdLifeTime )
{
    RFD_STATUS status = RFD_STATUS_OK;
    struct tm stdBrokenOutTime;

    do { // once

        if( rfdLifeTime < 0 ) {
            // error, parameter out of range.
            status = RFD_STATUS_ERROR_INVALID_VALUE;
            break;
        }

        RFD_MEMSET( &stdBrokenOutTime, 0, sizeof(struct tm));

        ///////////////////////////////////////////////////////////
        // Convert from SXI time to c standard lib Broken-out time.
        ///////////////////////////////////////////////////////////

        // Initialize to std epoc
        stdBrokenOutTime.tm_sec = 0;
        stdBrokenOutTime.tm_min = 0;
        stdBrokenOutTime.tm_hour = 0;
        stdBrokenOutTime.tm_mday = 1; // tm_mday is 1 relative
        stdBrokenOutTime.tm_mon = 0;
        stdBrokenOutTime.tm_year = 0;
        stdBrokenOutTime.tm_isdst = 0;
        stdBrokenOutTime.tm_wday = 0;
        stdBrokenOutTime.tm_yday = 0;

        // add rfdLifeTime (days)
        stdBrokenOutTime.tm_mday += rfdLifeTime;

        // add RFD Epoc (relative to standard broken-out Epoc).
        stdBrokenOutTime.tm_year += (RFD_MDATA_LIFE_YEAR0 - RFD_STD_LIB_BROKEN_DOWN_TIME_YEAR0);

        ///////////////////////////////////////////////////////////
        // Convert c standard lib "broken-out" time to c standard lib "calendar" time which is
        // a format that facilitates time comparison operations for expiration checking.
        ///////////////////////////////////////////////////////////

        *stdCalendarTimePtr = RFD_TIME_MakeSimpleCalendarTime( &stdBrokenOutTime );

    } while(FALSE);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
BOOL RFD_TIME_IsEqualSxiTime(
            RFD_SXI_TIME_INFO_STRUCT * sxiTime1,
            RFD_SXI_TIME_INFO_STRUCT * sxiTime0,
            RFD_SXI_TIME_RESOLUTION_ENUM timeResolution )
{
    BOOL isEqual = TRUE;

    if( RFD_SXI_TIME_MINUTE_RES >= timeResolution ) {
        isEqual &= (sxiTime1->minute == sxiTime0->minute? TRUE:FALSE);
    }
    if( RFD_SXI_TIME_HOUR_RES >= timeResolution ) {
        isEqual &= (sxiTime1->hour == sxiTime0->hour? TRUE:FALSE);
    }
    if( RFD_SXI_TIME_DAY_RES >= timeResolution ) {
        isEqual &= (sxiTime1->day == sxiTime0->day? TRUE:FALSE);
    }
    if( RFD_SXI_TIME_MONTH_RES >= timeResolution ) {
        isEqual &= (sxiTime1->month == sxiTime0->month? TRUE:FALSE);
    }
    if( RFD_SXI_TIME_YEAR_RES >= timeResolution ) {
        isEqual &= (sxiTime1->yearSince2k == sxiTime0->yearSince2k? TRUE:FALSE);
    }

    return isEqual;
}

///////////////////////////////////////////////////////////////////////////////
void RFD_TIME_SetSxiEpocTime( RFD_SXI_TIME_INFO_STRUCT * sxiTimePtr )
{
    RFD_MEMSET( sxiTimePtr, 0, sizeof(RFD_SXI_TIME_INFO_STRUCT));

    sxiTimePtr->minute = 0;
    sxiTimePtr->hour = 0;
    sxiTimePtr->day = 1; // SXI day (of month) is 1 relative
    sxiTimePtr->month = 1; // SXI month is 1 relative
    sxiTimePtr->yearSince2k = 0;

    return;
}

///////////////////////////////////////////////////////////////////////////////
// input dispYear  "Absolute" Year e.g. 2015
// input dispMonth [1 to 12]
// input dispDay   [1 to 31]
RFD_STATUS RFD_TIME_DisplayYearMonthDay2RfdLifeDays(
                    INT32 year,
                    INT32 month,
                    INT32 day,
                    INT32 * rfdLifeDaysPtr )
{
    RFD_STATUS status;
    struct tm endOfLifeStdBrokenOutTime;
    struct tm rfdEpocStdBrokenOutTime;
    time_t rfdEpocSimpleCalendarTime;
    time_t endOfLifeSimpleCalendarTime;
    double rfdLifeSeconds;
    double rfdLifeDays = 0;

    // initialize output
    *rfdLifeDaysPtr = 0;

    do { // once

        // Convert the input Year/Month/Day to Standard Broken Out Time

        status = _DisplayYearMonthDay2StdBrokenOutTime( year, month, day, &endOfLifeStdBrokenOutTime );

        if(status != RFD_STATUS_OK) {
            // error
            RFD_DPrint(TEXT("RFD_TIME_YearMonthDay2RfdLifeDays(), error in _YearMonthDay2StdBrokenOutTime(), status: %d\n"), status );
            break;
        }

        // Convert the end of life Standard Broken Out Time to Simple Calendar Time

        endOfLifeSimpleCalendarTime = RFD_TIME_MakeSimpleCalendarTime( &endOfLifeStdBrokenOutTime);

        if(endOfLifeSimpleCalendarTime == -1) {
            // error
            status = RFD_STATUS_ERROR_GENERAL;
            RFD_DPrint(TEXT("RFD_TIME_YearMonthDay2RfdLifeDays(), error in RFD_TIME_MakeSimpleCalendarTime()\n") );
            break;
        }

        // Get the RFD Epoc (year0) in Standard Broken Out Time

        status = _RfdEpoc2StdBrokenOutTime( &rfdEpocStdBrokenOutTime );

        if(status != RFD_STATUS_OK) {
            // error
            RFD_DPrint(TEXT("RFD_TIME_YearMonthDay2RfdLifeDays(), error in _RfdEpoc2StdBrokenOutTime(), status: %d\n"), status );
            break;
        }

        // Convert the RFD Epoc Standard Broken Out Time to Simple Calendar Time

        rfdEpocSimpleCalendarTime = RFD_TIME_MakeSimpleCalendarTime( &rfdEpocStdBrokenOutTime );

        if(rfdEpocSimpleCalendarTime == -1) {
            // error
            status = RFD_STATUS_ERROR_GENERAL;
            RFD_DPrint(TEXT("RFD_TIME_YearMonthDay2RfdLifeDays(), error in RFD_TIME_MakeSimpleCalendarTime(), status: %d\n"), status );
            break;
        }

        // Get the difference in seconds, between End of Life Time and RFD Epoc Time.
        // This is RFD Life in seconds.

        rfdLifeSeconds = RFD_TIME_DiffSimpleCalendarTime( endOfLifeSimpleCalendarTime, rfdEpocSimpleCalendarTime );

        // Convert RFD Life from Seconds to Days.

        rfdLifeDays = rfdLifeSeconds/60/60/24;

        // Do bound check on the result.

        if( rfdLifeDays < RFD_MDATA_LIFE_MIN_DAYS ||
            rfdLifeDays > RFD_MDATA_LIFE_MAX_DAYS ) {

            status = RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
		    RFD_DPrint(TEXT("RFD_TIME_YearMonthDay2RfdLifeDays() error, Life value is out of range. Max %d, Min %d (days)\n"),
                RFD_MDATA_LIFE_MIN_DAYS,
                RFD_MDATA_LIFE_MAX_DAYS);
            break;
	    }

    } while(FALSE);

    if(status == RFD_STATUS_OK) {
        *rfdLifeDaysPtr = (INT32) rfdLifeDays;
    }

    return status;
}
