/////////////////////
/// @file	rfd_common\rfd_threads_emul.c
///
/// @brief	emulate thread related functionality for the Alt Single Threaded Mode operation.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2012 Sirius XM Satellite Radio. All rights reserved.
/////////////////////

#include "rfd.h"

#ifdef RFD_ALT_SINGLE_THREAD_MODE_ENABLE

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Set the priority value for the specified thread.
///
/// The Priority value can be one of the following values:
///
/// RFD_THREAD_PRIORITY_HIGHEST         - higher than _ABOVE_NORMAL
///
/// RFD_THREAD_PRIORITY_ABOVE_NORMAL    - above _NORMAL
///
/// RFD_THREAD_PRIORITY_NORMAL          - normal priority
///
/// RFD_THREAD_PRIORITY_BELOW_NORMAL    - below _NORMAL
///
/// RFD_THREAD_PRIORITY_LOWEST          - lower than _BELOW_NORMAL.
///
/// @param  threadHandle    Handle of the thread whose priority value is to be set.
/// @param  nPriority       The priority.
///
/// @return true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_SetThreadPriority(RFD_THREAD_HANDLE threadHandle, int nPriority)
{
    RFD_DPrint( TEXT("RFD ALT Single Thread Mode: Unsupported function \n"));
    return FALSE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create thread.
///
/// The thread is created with a thread priority of RFD_THREAD_PRIORITY_NORMAL. Use the
/// RFD_SetThreadPriority() function to set the priority value of a thread. Indication that the
/// thread has terminated may be determined by a call to the function RFD_WaitForThreadExit().
/// The thread object remains in the system until the thread has terminated and all handles to it
/// have been closed through a call to RFD_CloseThread(). The typical thread completion sequence
/// is to wait for the thread to terminate by calling RFD_WaitForThreadExit(); then calling
/// RFD_CloseThread() to release resources corresponding to the thread.
///
/// @param  stackSize       Size of the stack in bytes to reserve for use by the thread.
/// @param  startAddress    Pointer to the application-defined function to be executed by the
///                         thread and represents the starting address of the thread.
/// @param  parameterPtr    Pointer to a variable to be passed to the thread.
/// @param  threadIdPtr     (output) Pointer to a variable that receives the thread identifier.
///                         If this parameter is NULL, the thread identifier is not returned.
///
/// @return handle to the created thread. If NULL, thread creation failed.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_THREAD_HANDLE RFD_CreateThread( SIZE_T stackSize,
									RFD_THREAD_START_ROUTINE_PTR startAddress,
									VOID * parameterPtr,
									DWORD * threadIdPtr )
{
    RFD_DPrint( TEXT("RFD ALT Single Thread Mode: Unsupported function \n"));
	return NULL;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Close the specified thread. See RFD_CreateThread() for additional info.
///
/// @param	threadHandle	Handle of the thread.
///
/// @return	true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_CloseThread(RFD_THREAD_HANDLE threadHandle)
{
    RFD_DPrint( TEXT("RFD ALT Single Thread Mode: Unsupported function \n"));
	return FALSE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Wait for the specified thread to exit (terminate).
///
/// If the thread has not terminated within the number of milliseconds specified by the
/// timeoutMilliseconds parameter, then RFD_WaitForThreadExit() will return with
/// RFD_STATUS_OBJECT_TIMEOUT status. RFD_WaitForThreadExit() will wait indefinitely if
/// timeoutMilliseconds = RFD_INFINITE. See RFD_CreateThread() for additional info.
///
/// @param  threadHandle        Handle of the thread.
/// @param  timeoutMilliseconds The timeout in milliseconds. No timeout if timeoutMilliseconds =
///                             RFD_INFINITE.
///
/// @return
/// RFD_STATUS_OK if thread has exited normally, RFD_STATUS_OBJECT_TIMEOUT if thread has not
/// termated withing the specified timeout period, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_WaitForThreadExit(RFD_THREAD_HANDLE threadHandle, INT32 timeoutMilliseconds)
{
    RFD_DPrint( TEXT("RFD ALT Single Thread Mode: Unsupported function \n"));
    return RFD_STATUS_ERROR_INVALID_MODE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create a mutex object.
///
/// Mutex objects are used to protect a shared resource from simultaneous access by multiple
/// threads. Each thread must wait for ownership of the mutex before it can execute the code that
/// accesses the shared resource. For example, if several threads share access to a database, the
/// threads can use a mutex object to permit only one thread at a time to write to the database.
///
/// A mutex is first created using RFD_CreateMutex(). To obtain ownership of a mutex, a thread
/// calls RFD_AcquireMutex(). If another thread still has ownership, then the RFD_AcquireMutex()
/// blocks (suspends the calling thread) until the other thread has released ownership or until
/// RFD_AcquireMutex() times-out. Once a thread has ownership of the mutex, the shared resource
/// may be safely evaluated and/or modified by the owning thread. The thread may then call
/// RFD_ReleaseMutex() to release ownership of the mutex and thus allow other threads access to
/// the shared resource. If the mutex is no longer in use by any threads (e.g. closing the
/// application or shutting down) the mutex reasources are cleaned up by calling RFD_CloseMutex().
///
/// @return the handle of the created mutex. NULL if creation failed.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_MUTEX_HANDLE RFD_CreateMutex(void)
{
    //RFD_DPrint( TEXT("RFD ALT Single Thread Mode: Emulated function \n"));
	return (RFD_MUTEX_HANDLE) 1;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Close the specified mutex object. See RFD_CreateMutex() for additional info.
///
/// @param	mutexHandle	Handle of the mutex.
///
/// @return	true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_CloseMutex(RFD_MUTEX_HANDLE mutexHandle)
{
    //RFD_DPrint( TEXT("RFD ALT Single Thread Mode: Emulated function \n"));
    return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Acquire the specified mutex object.
///
/// If the mutex has not been acquired within the number of milliseconds specified by the
/// timeoutMilliseconds parameter, then RFD_AcquireMutex() will return with
/// RFD_STATUS_OBJECT_TIMEOUT status. RFD_AcquireMutex() will wait indefinitely if
/// timeoutMilliseconds = RFD_INFINITE. See RFD_CreateMutex() for additional info.
///
/// @param  mutexHandle         The mutex handle.
/// @param  timeoutMilliseconds The timeout in milliseconds. No timeout if timeoutMilliseconds =
///                             RFD_INFINITE.
///
/// @return
/// RFD_STATUS_OK if the mutex was sucessfully acquired, RFD_STATUS_OBJECT_TIMEOUT if the mutex
/// was not acquired and a timeout has occurred instead, other RFD_STATUS enumeration value if
/// failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_AcquireMutex(RFD_MUTEX_HANDLE mutexHandle, INT32 timeoutMilliseconds)
{
    //RFD_DPrint( TEXT("RFD ALT Single Thread Mode: Emulated function \n"));
	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Release the specified mutex object.
///
/// See RFD_CreateMutex() for additional info.
///
/// @param  mutexHandle The mutex handle.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_ReleaseMutex(RFD_MUTEX_HANDLE mutexHandle)
{
    //RFD_DPrint( TEXT("RFD ALT Single Thread Mode: Emulated function \n"));
	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create an event object.
///
/// An event object is a type of synchronization object. The purpose of an event object is to
/// enable one or multiple threads to wait for a particular event to occur. The waiting threads
/// can be kept suspended until the event is set by another thread. The event is set (to signaled
/// state) by calling RFD_SetEvent(). Threads wait on the event to be signaled by calling
/// RFD_WaitForEvent(). An event in the signaled state is changed to the non-signaled state when
/// a single waiting thread is released. If no threads are waiting, the event object's state
/// remains signaled. If more than one thread is waiting, a waiting thread is selected. Do not
/// assume a particular wait order.
///
/// An event object is created by RFD_CreateEvent(). When created, the event object is initially
/// in the non-signaled state. If the event object is no longer in use by any threads (e.g.
/// closing the application or shutting down) the event object resources are cleaned up by
/// calling RFD_CloseEvent().
///
/// @return the handle of the created event. NULL if creation failed.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_EVENT_HANDLE RFD_CreateEvent(void)
{
    int * emulObjPtr;
    emulObjPtr = (int *) RFD_MALLOC(sizeof(int));

    if(emulObjPtr == NULL) {
        return NULL;
    }

    *emulObjPtr = 0;

    return (RFD_EVENT_HANDLE) emulObjPtr;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Close the specified event object.
///
/// See RFD_CreateEvent() for additional info.
///
/// @param	eventHandle	Handle of the event.
///
/// @return	true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_CloseEvent(RFD_EVENT_HANDLE eventHandle)
{

    if(eventHandle != NULL) {
        RFD_FREE(eventHandle);
    }

    return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Set the specified event object to the signaled state.
///
/// See RFD_CreateEvent() for additional info.
///
/// @param  eventHandle Handle of the event.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_SetEvent(RFD_EVENT_HANDLE eventHandle)
{
    int * emulObjPtr;

    if(eventHandle == NULL) {
		return RFD_STATUS_ERROR_GENERAL;
    }

    emulObjPtr = (int *) eventHandle;

    *emulObjPtr = 1;

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Waits for an event and suspends the calling thread for a specified time as long as the event
/// is not signaled.
///
/// If the specified event object is already set to the signaled state, the calling task resets
/// the event to the non-signaled state and continues program execution. If the specified event
/// object is not set to the signaled state, the calling task is suspended until the event object
/// becomes signaled or the timeout time has expired. RFD_WaitForEvent() will wait indefinitely
/// if timeoutMilliseconds = RFD_INFINITE. See RFD_WaitForEvent() for additional info.
///
/// @param  eventHandle         Handle of the event.
/// @param  timeoutMilliseconds The timeout in milliseconds. No timeout if timeoutMilliseconds =
///                             RFD_INFINITE.
///
/// @return
/// RFD_STATUS_OK if event object was detected as signaled and was reset to non-signaled,
/// RFD_STATUS_OBJECT_TIMEOUT if the event object was not signaled and a timeout has occurred
/// instead, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_WaitForEvent(RFD_EVENT_HANDLE eventHandle, INT32 timeoutMilliseconds)
{
    int * emulObjPtr;
    RFD_STATUS status;

    if(eventHandle == NULL) {
		return RFD_STATUS_ERROR_GENERAL;
    }

    emulObjPtr = (int *) eventHandle;

    if(*emulObjPtr == 0) {
        // event not set
        if(timeoutMilliseconds == RFD_INFINITE) {
            RFD_DPrint( TEXT("RFD ALT Single Thread Mode: error, RFD_WaitForEvent() emulation called with infinite timeout and event is not set \n"));
            status = RFD_STATUS_ERROR_GENERAL;
        }
        else {
            //Sleep(timeoutMilliseconds); // could emulate timeout.
            status = RFD_STATUS_OBJECT_TIMEOUT;
        }
    }
    else {
        // event was set, now clear event
        *emulObjPtr = 0;
        status = RFD_STATUS_OK;
    }

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create a multi wait event object.
///
/// A multi wait event object holds a collection of event objects. See RFD_WaitForEvent()
/// for a description of event objects. The multi wait event object enables a thread to
/// simultaneosly wait for any of the collection of event objects to be signaled (set to signaled
/// state). The RFD_WaitForOneOfMultiEvents() function is used for this purpose.
///
/// A multi wait event object is created using RFD_CreateMultiWaitEventObject(). The number of
/// events in the collection is specified in the call to this function. Individual event objects
/// are then added to the multi wait event object using RFD_AddEventToMultiWaitEventObject(). The
/// individual event objects are previously created using the event object function
/// RFD_CreateEvent(). An individual event of the multi wait event object is signaled (set to
/// signaled state) using the event object function RFD_SetEvent(). The
/// RFD_WaitForOneOfMultiEvents() function allows a thread to wait for any of the events of the
/// multi wait event object to be signaled. The first event to be detected as signaled is then
/// reset to the non-signaled state and program execution continues for the calling thread. If
/// the multi wait even object is no longer in use by any threads (e.g. closing the application
/// or shutting down) the object resources are cleaned up by calling
/// RFD_DeleteMultiWaitEventObject(). This does not clean up the individual event objects that
/// were added to the multi wait event object. The individual event objects are cleaned up by the
/// application using RFD_CloseEvent() for each individual event object.
///
/// @param  nWaitEventEntries   The number of event objects that the multi wait event object is
///                             to contain.
///
/// @return multi-wait-event structure. NULL if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_MULTI_WAIT_EVENT_STRUCT * RFD_CreateMultiWaitEventObject( int nWaitEventEntries)
{
	RFD_MULTI_WAIT_EVENT_STRUCT * multiWaitEventHandle;
	int i;

	// Allocate the overall structure.
	multiWaitEventHandle = (RFD_MULTI_WAIT_EVENT_STRUCT *) RFD_MALLOC(sizeof(RFD_MULTI_WAIT_EVENT_STRUCT));
	if(multiWaitEventHandle == NULL) {
		return NULL;
	}

	// Allocate the array of event objects within the main structure.
	multiWaitEventHandle->waitEventArray = (RFD_EVENT_HANDLE *) RFD_MALLOC(nWaitEventEntries * sizeof(RFD_EVENT_HANDLE));
	if(multiWaitEventHandle->waitEventArray == NULL) {
		RFD_FREE(multiWaitEventHandle);
		return NULL;
	}

	for(i=0;i<nWaitEventEntries;i++) {
		multiWaitEventHandle->waitEventArray[i] = NULL;
	}

	multiWaitEventHandle->nWaitEventEntries = nWaitEventEntries;

	return multiWaitEventHandle;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Delete a multi wait event object to clean up resources of the multi wait event object.
///
/// The individual event objects that were added to the multi wait event object are not cleaned
/// up. See RFD_CreateMultiWaitEventObject() for addition info.
///
/// @param  multiWaitEventHandle    The multi wait event object.
////////////////////////////////////////////////////////////////////////////////////////////////////

void RFD_DeleteMultiWaitEventObject( RFD_MULTI_WAIT_EVENT_STRUCT * multiWaitEventHandle)
{
	if(multiWaitEventHandle != NULL) {
		RFD_FREE(multiWaitEventHandle->waitEventArray);
	}
	RFD_FREE(multiWaitEventHandle);
	return;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Add an individual event object to a multi wait event object.
///
/// Each event object that is added must have a unique event index, specified by the eventIndex
/// parameter, that is used to identify which event is signaled in the call to
/// RFD_WaitForOneOfMultiEvents(). The number of event objects added to the multi wait event
/// object must not exceed the maximum number of event objects configured in the call to
/// RFD_CreateMultiWaitEventObject. As such the eventIndex parameter must be less than this
/// maximum number. See RFD_CreateMultiWaitEventObject() for addition info.
///
/// @param  multiWaitEventHandle    The multi wait event handle.
/// @param  eventHandle             The event object handle being added.
/// @param  eventIndex              Zero-based index of the event object being added.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_AddEventToMultiWaitEventObject(RFD_MULTI_WAIT_EVENT_STRUCT * multiWaitEventHandle,
											  RFD_EVENT_HANDLE eventHandle,
											  int eventIndex)
{
	if(eventIndex < multiWaitEventHandle->nWaitEventEntries) {
		multiWaitEventHandle->waitEventArray[eventIndex] = eventHandle;
		return RFD_STATUS_OK;
	}
	else {
		return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Wait for one of the event objects of the multi wait event object to be signaled.
///
/// The RFD_WaitForOneOfMultiEvents() function allows a thread to simultaneously wait for any of
/// the events of the multi wait event object to be set to the signaled state. The first event to
/// be detected as signaled is reset to the non-signaled state and program execution continues
/// for the calling thread. If no event object is signaled within the timeout period specified by
/// timeoutMilliseconds, the program execution continues but returns a value of
/// RFD_STATUS_OBJECT_TIMEOUT. RFD_WaitForOneOfMultiEvents() will wait indefinitely if
/// timeoutMilliseconds = RFD_INFINITE. See RFD_CreateMultiWaitEventObject() for addition info.
///
/// @param  multiWaitEventHandle    The multi wait event handle.
/// @param  timeoutMilliseconds     The timeout in milliseconds. No timeout if
///                                 timeoutMilliseconds = RFD_INFINITE.
/// @param  ocurredEventIndexPtr    (output) The zero-relative index of the event object that was
///                                 signaled.
///
/// @return
/// RFD_STATUS_OK if an event object was detected as signaled and was reset to non- signaled,
/// RFD_STATUS_OBJECT_TIMEOUT if no event objects were signaled and a timeout has occurred
/// instead, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_WaitForOneOfMultiEvents(RFD_MULTI_WAIT_EVENT_STRUCT * multiWaitEventHandle,
									   INT32 timeoutMilliseconds,
								       int * ocurredEventIndexPtr)
{
	RFD_STATUS status;
    int i;
    BOOL found = FALSE;

	*ocurredEventIndexPtr = 0;

    if(multiWaitEventHandle == NULL) {
        return RFD_STATUS_ERROR_GENERAL;
    }

    for(i=0;i<multiWaitEventHandle->nWaitEventEntries;i++) {
        status = RFD_WaitForEvent(multiWaitEventHandle->waitEventArray[i], 0);
        if(status == RFD_STATUS_OK) {
            found = TRUE;
            break;
        }
    }

    if(found) {
        *ocurredEventIndexPtr = i;
        status = RFD_STATUS_OK;
    }
    else {
        // event not set
        if(timeoutMilliseconds == RFD_INFINITE) {
            RFD_DPrint( TEXT("RFD ALT Single Thread Mode: error, RFD_WaitForEvent() emulation called with infinite timeout and event is not set \n"));
            status = RFD_STATUS_ERROR_GENERAL;
        }
        else {
            //Sleep(timeoutMilliseconds); // could emulate timeout.
            status = RFD_STATUS_OBJECT_TIMEOUT;
        }
    }

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Create a counting semaphore.
///
/// Semaphores are used for synchronizing threads and to manage resources. Counting semaphores
/// are counters that are managed by the Kernel/OS. They are used in situations where a thread
/// needs to wait for something that can be signaled (e.g. by another thread) one or more times.
///
/// A semaphore is first created using RFD_CreateSemaphore(). Each call to RFD_SignalSemaphore()
/// will increment the semaphore's counter. If the counter is not 0, a call to
/// RFD_WaitForSemaphore() decrements the counter by one and execution continues for the thread
/// making the call. If the count is 0, RFD_WaitForSemaphore()
/// waits until the counter is incremented by another thread that calls RFD_SignalSemaphore().
/// Multiple threads can wait on a semaphore. The thread with the highest priority will continue
/// program execution, while other threads will continue waiting if the semaphore counter is 0.
/// If the semaphore is no longer in use by any threads (e.g. closing the application or shutting
/// down) the semaphore resources are cleaned up by calling RFD_CloseSemaphore().
///
/// @param  initialCount    Initial count of the semaphore. This value must be greater than or
///                         equal to zero and less than or equal to maximumCount.
/// @param  maximumCount    Maximum count of the semaphore. This value must be greater than zero.
///
/// @return handle of the created semaphore. NULL if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_SEMAPHORE_HANDLE RFD_CreateSemaphore(int initialCount, int maximumCount)
{
	RFD_EMUL_SEMAPHORE_STRUCT * sem;

    sem = (RFD_EMUL_SEMAPHORE_STRUCT *) RFD_MALLOC(sizeof(RFD_EMUL_SEMAPHORE_STRUCT));
    if(sem == NULL) {
        return NULL;
    }

    sem->count = initialCount;
    sem->initialCount = initialCount;
    sem->maximumCount = maximumCount;

    return (RFD_SEMAPHORE_HANDLE) sem;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Close a specified semaphore.
///
/// See RFD_CreateSemaphore() for additional info.
///
/// @param  semaphoreHandle Handle of the semaphore.
///
/// @return true if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL RFD_CloseSemaphore(RFD_SEMAPHORE_HANDLE semaphoreHandle)
{
    if(semaphoreHandle == NULL) {
        return FALSE;
    }

    RFD_FREE(semaphoreHandle);
    return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Signal a semaphore by incrementing the semaphore's counter.
///
/// The counter is not incremented if the semaphore's current counter is at the maximum value as
/// configured by RFD_CreateSemaphore(). See RFD_CreateSemaphore() for additional info.
///
/// @param  semaphoreHandle Handle of the semaphore.
///
/// @return RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_SignalSemaphore(RFD_SEMAPHORE_HANDLE semaphoreHandle)
{
	RFD_EMUL_SEMAPHORE_STRUCT * sem;

    if(semaphoreHandle == NULL) {
        return RFD_STATUS_ERROR_GENERAL;
    }

    sem = (RFD_EMUL_SEMAPHORE_STRUCT *) semaphoreHandle;

    if(sem->count < sem->maximumCount) {
        sem->count++;
    }

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Waits for a specified semaphore to be signaled.
///
/// If the counter of the semaphore is not 0, the counter is decremented and program execution
/// continues for the calling thread. If the counter is 0, RFD_WaitForSemaphore() waits until the
/// counter is incremented by another thread. The counter is then decremented and program
/// execution continues. If the semaphore was not signaled within the timeout period specified by
/// timeoutMilliseconds, the program execution continues but returns a value of
/// RFD_STATUS_OBJECT_TIMEOUT. RFD_WaitForSemaphore() will wait indefinitely if
/// timeoutMilliseconds = RFD_INFINITE.
///
/// @param  semaphoreHandle     Handle of the semaphore.
/// @param  timeoutMilliseconds The timeout in milliseconds. No timeout if timeoutMilliseconds =
///                             RFD_INFINITE.
///
/// @return
/// RFD_STATUS_OK if semaphore is signaled, RFD_STATUS_OBJECT_TIMEOUT if the semaphore is not
/// signaled and a timeout has occurred instead, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_WaitForSemaphore(RFD_SEMAPHORE_HANDLE semaphoreHandle, INT32 timeoutMilliseconds)
{
	RFD_STATUS status;

	RFD_EMUL_SEMAPHORE_STRUCT * sem;

    if(semaphoreHandle == NULL) {
        return RFD_STATUS_ERROR_GENERAL;
    }

    sem = (RFD_EMUL_SEMAPHORE_STRUCT *) semaphoreHandle;

    if(sem->count > 0) {
        // semaphore available.
        // decrement and set stutus as signaled.
        sem->count--;
        status = RFD_STATUS_OK;
    }
    else {
        // no semaphore available
        if(timeoutMilliseconds == RFD_INFINITE) {
            RFD_DPrint( TEXT("RFD ALT Single Thread Mode: error, RFD_WaitForSemaphore() emulation called with infinite timeout and semaphore is not available\n"));
            status = RFD_STATUS_ERROR_GENERAL;
        }
        else {
            //Sleep(timeoutMilliseconds); // could emulate timeout.
            status = RFD_STATUS_OBJECT_TIMEOUT;
        }
    }

    return status;
}


#endif // RFD_ALT_SINGLE_THREAD_MODE_ENABLE

