////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_platform_port.h
///
/// @brief	Declares the rfd platform port class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef RFD_PLATFORM_PORT_H
#define RFD_PLATFORM_PORT_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

// First Definition : The Platform RFD is built for.
#define RFD_OSAL 1

#if defined(RFD_WIN32)

#define WIN32_LEAN_AND_MEAN		// Exclude rarely-used stuff from Windows headers
#include <stdio.h>
#include <stdlib.h>
#include <tchar.h>
#include <strsafe.h>

#define RFD_FILE  FILE
#define RFD_INFINITE  INFINITE

typedef HANDLE RFD_MUTEX_HANDLE;
typedef HANDLE RFD_SEMAPHORE_HANDLE;
typedef HANDLE RFD_EVENT_HANDLE;
typedef HANDLE RFD_THREAD_HANDLE;
typedef LPTHREAD_START_ROUTINE  RFD_THREAD_START_ROUTINE_PTR;
#define RFD_PLATFORM_API  WINAPI

// typedefs for FindFirst()/FindNext() fcns.
typedef HANDLE   RFD_FIND_FILE_HANDLE;
typedef WIN32_FIND_DATA  RFD_FIND_FILE_DATA_INFO;

typedef unsigned int		UINT32; // 32-bit unsigned integer
typedef int					INT32;  // 32-bit signed integer
typedef unsigned short		UINT16; // 16-bit unsigned integer
typedef short				INT16;  // 16-bit signed integer
typedef unsigned char		UCHAR;  // unsigned character


#if 0
////////
// The following types are defined by WIN32.
// Other platforms may have to assign types here. For example as follows:
///////

// SIZE_T : The maximum number of bytes to which a pointer can point.
// Use for a count that must span the full range of a pointer.
typedef unsigned long		SIZE_T;

typedef unsigned long		DWORD;	// 32-bit unsigned integer
typedef int					BOOL;	// Boolean variable (should be TRUE or FALSE).
typedef void *				LPVOID;	// Pointer to any type.
#define VOID				void	// any type
#endif

#define RFD_SIZE_OF_DWORD	4
#define RFD_BYTES_PER_DWORD 4
#define RFD_BITS_PER_DWORD	32
#define RFD_BITS_PER_BYTE	8


// DWORD_ALIGN
// Force DWORD address alignment of some variables/arrays.
//   Some platfroms (e.g. ARM) require data alignment to the same size as the instruction access size.
//   Visual Studio/WIN32:  use __declspec(align(RFD_SIZE_OF_DWORD))
//   ARM compilers: use __align(RFD_SIZE_OF_DWORD)
#define DWORD_ALIGN __declspec(align(RFD_SIZE_OF_DWORD))

// RFD_PLATFORM_LITTLE_ENDIAN
// Define RFD_PLATFORM_LITTLE_ENDIAN to specify the platform implements
// Litle Endian type memory access (e.g. ARM, Intel x86).
// Undefine RFD_PLATFORM_LITTLE_ENDIAN to specify Big Endian platform.
#if 1
#define RFD_PLATFORM_LITTLE_ENDIAN
#endif

// RFD_MAX_PATH_LEN
// Specify the maximum length of a complete File System directory/file path string.
// Note: To save some relatively small amout of heap space usage, it is possible to
// assign a value lower that the actual maximum allowed for
// the file system. This lower value must be greater than the worse case maximum
// filename + path string used by RFD.
#define RFD_MAX_PATH_LEN    MAX_PATH // MAX_PATH = 260 for WIN32

// RFD_PATH_SEPARATOR
// The character used to separate elements of a file system path or pathname.
// (for example, Unix uses '/' while MS-DOS uses back slash '\')
#define RFD_PATH_SEPARATOR TEXT("\\")

// File operations
#define RFD_FOPEN_S(File, Filename, Mode)			_tfopen_s(File, Filename, Mode)
#define RFD_FREAD(buf, elem_size, count, file_ptr)  fread( buf, elem_size, count, file_ptr)
#define RFD_FWRITE(buf, elem_size, count, file_ptr) fwrite(buf, elem_size, count, file_ptr)
#define RFD_FSEEK(file_ptr, offset, origin)         fseek(file_ptr, offset, origin)
#define RFD_FTELL(file_ptr)							ftell(file_ptr)
#define RFD_FFLUSH(file_ptr)                        fflush(file_ptr)
#define RFD_FCLOSE(file_ptr)                        fclose(file_ptr)
#define RFD_FERROR(file_ptr)						ferror(file_ptr)
#define RFD_FEOF(file_ptr)							feof(file_ptr)
#define RFD_RENAME(oldName, newName)				_trename(oldName, newName)	// rename() - returns 0 if successful.
#define RFD_DELETE_FILE(fileName)					_tremove(fileName)			// remove() - returns 0 if successful.
#define RFD_CREATE_DIR(dirName)						_tmkdir(dirName)			// mkdir() - returns 0 if successful.
#define RFD_REMOVE_DIR(dirName)						_trmdir(dirName)			// rmdir() - returns 0 if successful.

// Dynamic memory allocation
// Use malloc/calloc/free instead of HeapAlloc/HeapFree for compatibility
// with memory leak debugging using _CrtDumpMemoryLeaks().
#define RFD_MALLOC(size)							malloc(size)
#define RFD_CALLOC(size)							calloc(size, 1)
#define RFD_FREE(lpMem)								free(lpMem)

// Memory buffer manipulation
#define RFD_MEMSET(lpMem, val, size)				memset(lpMem, val, size)
#define RFD_MEMCPY(dst, src, size)					memcpy(dst, src, size)
#define RFD_MEMCMP(buf1, buf2, size)				memcmp(buf1, buf2, size)

// String manipulation
#define RFD_STRCPY_S(dst, dst_size, src)			_tcscpy_s(dst, dst_size, src)
#define RFD_STRCAT_S(dst, dst_size, src)			_tcscat_s(dst, dst_size, src)
#define RFD_STRCMP(string1, string2)				_tcscmp(string1, string2)
#define RFD_STRNCMP									_tcsnicmp		// strncmp()
#define RFD_STRNLEN(str, bufLen)					_tcsnlen(str, bufLen)		// strnlen()
#define RFD_ATOI(str)								_tstoi(str)					// atio()
#define RFD_SPRINTF_S								_stprintf_s	// sprinf_s() // StringCchPrintf()

// RFD_CHAR_STRCPY_S :
// Does 'char' string copy regardless of character format configuration (e.g. unicode configuration).
// Note: This macro is implemented in support of the RFD port of the gzip related function, rfd_gzerror().
// If this macro implementation is problematic for a particular platform, an alternative is to
// comment out the rfd_gzerror() function (not currently called).
#define RFD_CHAR_STRCPY_S(dst, dst_size, src)		strcpy_s(dst, dst_size, src)

// RFD_PRINTF is only used by the receiver_app, rfd_transmitter, rfd_message_server.
// It is not needed by the libraries that make up the "RFDR Module".
#define RFD_PRINTF									_tprintf		// printf()

// RFD_FPRINTF is only used by the rfd_transmitter.
// It is not needed by the libraries that make up the "RFDR Module".
#define RFD_FPRINTF									_ftprintf     // fprintf()

// Time support
#define RFD_MKTIME(brokentime)                      mktime(brokentime)
#define RFD_DIFFTIME(time1, time0)                  difftime(time1, time0)

#ifdef RFD_BAUDOT_TEST_ENABLE
#define RFD_STDLIB_SRAND(seed)						srand(seed) // limited srand function from c standard lib
#define RFD_STDLIB_RAND								rand // limited rand function from c standard lib
#endif

// Thread Priorities
#define RFD_THREAD_PRIORITY_HIGHEST			THREAD_PRIORITY_HIGHEST
#define RFD_THREAD_PRIORITY_ABOVE_NORMAL	THREAD_PRIORITY_ABOVE_NORMAL
#define RFD_THREAD_PRIORITY_NORMAL			THREAD_PRIORITY_NORMAL
#define RFD_THREAD_PRIORITY_BELOW_NORMAL	THREAD_PRIORITY_BELOW_NORMAL
#define RFD_THREAD_PRIORITY_LOWEST			THREAD_PRIORITY_LOWEST

#elif defined(RFD_OSAL)
#include <stdio.h>
#include <stdarg.h>
#ifdef OS_WINCE
#include <stdio.h>
#include <tchar.h>
extern int errno;
#else
#include <errno.h>
#endif
#include <string.h>

#include "standard.h"
#include "osal.h"

#define RFD_FILE  FILE
#define RFD_INFINITE  OSAL_OBJ_TIMEOUT_INFINITE

typedef OSAL_OBJECT_HDL RFD_MUTEX_HANDLE;
typedef OSAL_OBJECT_HDL RFD_SEMAPHORE_HANDLE;
typedef void * RFD_EVENT_HANDLE;
typedef void * RFD_THREAD_HANDLE;
#define RFD_PLATFORM_API    // Don't need this for OSAL (or non-WINAPI)

////////
// The following types are defined by WIN32.
// Other platforms may have to assign types here. For example as follows:
///////

// SIZE_T : The maximum number of bytes to which a pointer can point.
// Use for a count that must span the full range of a pointer.
typedef unsigned long       SIZE_T;

typedef unsigned long       DWORD;  // 32-bit unsigned integer
#ifndef OS_WINCE
typedef BOOLEAN             BOOL;   // Boolean variable (should be TRUE or FALSE).
#endif
typedef void *              LPVOID; // Pointer to any type.
#define VOID                void    // any type

// typedefs for FindFirst()/FindNext() fcns.
typedef void * RFD_FIND_FILE_HANDLE;
typedef void * RFD_FIND_FILE_DATA_INFO;

typedef UN32              UINT32;
typedef N32               INT32;
typedef UN16              UINT16;
typedef N16               INT16;
//typedef UN8               UCHAR;  OSAL PORT NOTE: UCHAR defined in standard.h

// We are only supporting single-byte character sets
#ifdef OS_WINCE
typedef WCHAR             TCHAR;
#define TCHAR char
#undef TEXT
#define TEXT(value)       value
#define strerror(arg)     "?"
#else
typedef char              TCHAR;
#define TEXT(value)       value
#endif

#define RFD_SIZE_OF_DWORD   4
#define RFD_BYTES_PER_DWORD 4
#define RFD_BITS_PER_DWORD  32
#define RFD_BITS_PER_BYTE   8

// DWORD_ALIGN
// Force DWORD address alignment of some variables/arrays.
//   Some platfroms (e.g. ARM) require data alignment to the same size as the instruction access size.
//   Visual Studio/WIN32:  use __declspec(align(RFD_SIZE_OF_DWORD))
//   ARM compilers: use __align(RFD_SIZE_OF_DWORD)
//   CSH NOTE: __align doesn't seem to work with GCC
#if __ICCARM__ == 1
    #define DWORD_ALIGN   _Pragma("data_alignment=RFD_SIZE_OF_DWORD")
#elif defined __QNX__
    #define DWORD_ALIGN   __attribute__ ((aligned (RFD_SIZE_OF_DWORD)))
#elif defined OS_WINCE
	#define DWORD_ALIGN
#elif defined _WIN32
    #define DWORD_ALIGN __declspec(align(RFD_SIZE_OF_DWORD))
#else
    #define DWORD_ALIGN   __attribute__ ((aligned (RFD_SIZE_OF_DWORD)))
#endif

#ifndef OSAL_CPU_BIG_ENDIAN
// RFD_PLATFORM_LITTLE_ENDIAN
// Define RFD_PLATFORM_LITTLE_ENDIAN to specify the platform implements
// Litle Endian type memory access (e.g. ARM, Intel x86).
// Undefine RFD_PLATFORM_LITTLE_ENDIAN to specify Big Endian platform.
#define RFD_PLATFORM_LITTLE_ENDIAN
#endif

// RFD_MAX_PATH_LEN
// Specify the maximum length of a complete File System directory/file path string.
// Note: To save some relatively small amout of heap space usage, it is possible to
// assign a value lower that the actual maximum allowed for
// the file system. This lower value must be greater than the worse case maximum
// filename + path string used by RFD.
#define RFD_MAX_PATH_LEN    260 // MAX_PATH = 260 for WIN32

// RFD_PATH_SEPARATOR
// The character used to separate elements of a file system path or pathname.
// (for example, Unix uses '/' while MS-DOS uses back slash '\')
#define RFD_PATH_SEPARATOR TEXT("/")



typedef DWORD (*RFD_THREAD_START_ROUTINE_PTR)(
    LPVOID lpThreadParameter
    );

// File operations
#define RFD_FOPEN_S(File, Filename, Mode)           PORT_fopen(File, Filename, Mode)
#define RFD_FREAD(buf, elem_size, count, file_ptr)  fread( buf, elem_size, count, file_ptr)
#define RFD_FWRITE(buf, elem_size, count, file_ptr) fwrite(buf, elem_size, count, file_ptr)
#define RFD_FSEEK(file_ptr, offset, origin)         fseek(file_ptr, offset, origin)
#define RFD_FTELL(file_ptr)                         ftell(file_ptr)
#define RFD_FFLUSH(file_ptr)                        fflush(file_ptr)
#define RFD_FCLOSE(file_ptr)                        fclose(file_ptr)
#define RFD_FERROR(file_ptr)                        ferror(file_ptr)
#define RFD_FEOF(file_ptr)                          feof(file_ptr)
#define RFD_RENAME(oldName, newName)                rename(oldName, newName)
#define RFD_DELETE_FILE(fileName)                   remove(fileName)
#define RFD_CREATE_DIR(dirName)                     PORT_mkdir(dirName)
#define RFD_REMOVE_DIR(dirName)                     PORT_rmdir(dirName)

// Dynamic memory allocation
#define RFD_MALLOC(size)                            OSAL.pvMemoryAllocate(RFDLIB_NAME":MallocRes", size, FALSE)
#define RFD_CALLOC(size)                            OSAL.pvMemoryAllocate(RFDLIB_NAME":CallocRes", size, TRUE)
#define RFD_FREE(lpMem)                             OSAL.vMemoryFree(lpMem)

// Memory buffer manipulation
#define RFD_MEMSET(lpMem, val, size)                OSAL.bMemSet(lpMem, val, size)
#define RFD_MEMCPY(dst, src, size)                  OSAL.bMemCpy(dst, src, size)
#define RFD_MEMCMP(buf1, buf2, size)                memcmp(buf1, buf2, size)

// String manipulation
#define RFD_STRCPY_S(dst, dst_size, src)            strcpy(dst, src)
#define RFD_STRCAT_S(dst, dst_size, src)            strcat(dst, src)
#define RFD_STRCMP(string1, string2)                strcmp(string1, string2)
#define RFD_STRNCMP                                 strncmp
#define RFD_STRNLEN(str, bufLen)                    strlen(str)
#define RFD_ATOI(str)                               atoi(str)
#define RFD_SPRINTF_S                               PORT_sprintf_s

// RFD_CHAR_STRCPY_S :
// Does 'char' string copy regardless of character format configuration (e.g. unicode configuration).
// Note: This macro is implemented in support of the RFD port of the gzip related function, rfd_gzerror().
// If this macro implementation is problematic for a particular platform, an alternative is to
// comment out the rfd_gzerror() function (not currently called).
#define RFD_CHAR_STRCPY_S(dst, dst_size, src)       RFD_STRCPY_S(dst, dst_size, src)


// RFD_PRINTF is only used by the receiver_app, rfd_transmitter, rfd_message_server.
// It is not needed by the libraries that make up the "RFDR Module".
#define RFD_PRINTF                                  printf

// RFD_FPRINTF is only used by the rfd_transmitter.
// It is not needed by the libraries that make up the "RFDR Module".
#define RFD_FPRINTF                                 fprintf

// Time support
#define RFD_MKTIME(brokentime)                      OSAL.mktime(brokentime)
#define RFD_DIFFTIME(time1, time0)                  OSAL.difftime(time1, time0);

#ifdef RFD_BAUDOT_TEST_ENABLE
#define RFD_STDLIB_SRAND(seed)                      srand(seed) // limited srand function from c standard lib
#define RFD_STDLIB_RAND                             rand // limited rand function from c standard lib
#endif

// Thread Priorities
#define RFD_THREAD_PRIORITY_HIGHEST         OSAL_TASK_PRIORITY_HIGH //OSAL_TASK_PRIORITY_HIGHEST
#define RFD_THREAD_PRIORITY_ABOVE_NORMAL    OSAL_TASK_PRIORITY_HIGH
#define RFD_THREAD_PRIORITY_NORMAL          OSAL_TASK_PRIORITY_MEDIUM
#define RFD_THREAD_PRIORITY_BELOW_NORMAL    OSAL_TASK_PRIORITY_LOW
#define RFD_THREAD_PRIORITY_LOWEST          OSAL_TASK_PRIORITY_LOW //OSAL_TASK_PRIORITY_LOWEST

// Platform-specific constants
#define RFDLIB_NAME                         "RFDLib"

// Platform-specific prototypes

// Wrapper for fopen to make it match prototype of
// the windows function _tfopen_s
INT32 PORT_fopen (
    RFD_FILE **ptFile,
    const char* pcFileName,
    const char* pcMode
        );

INT32 PORT_mkdir (
    const char* pcDirName
        );

INT32 PORT_rmdir (
    const char* pcDirName
        );

INT32 PORT_sprintf_s (
    TCHAR *buffer,
    size_t sizeOfBuffer,
    const TCHAR *format,
    ...
        );

#elif defined(RFD_OEM_PLATFORM)
// Alternate declarations for another platform can go here.
#endif

#ifdef __cplusplus
}
#endif

#endif // RFD_CONFIG_H
