////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_misc.c
///
/// @brief	rfd miscellaneous class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CalculateFileSetMetatdataNumFiles(RFD_FSET_MDATA_STRUCT * fileSetInfo, UINT32 maxFileSizeBytes)
{
	// Inputs are:
	//  fileSetInfo->totalSize
	//  maxFileSizeBytes
	//
	// Output is:
	//  fileSetInfo->numFiles
	//

	int numFiles;
	RFD_STATUS status;

	numFiles =  (fileSetInfo->totalSize / maxFileSizeBytes) +
				((fileSetInfo->totalSize % maxFileSizeBytes != 0) ? 1:0);

	if(numFiles > RFD_FSET_MAX_NUM_FILES) {
		fileSetInfo->numFiles = 0;
		status = RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
	}
	else {
		fileSetInfo->numFiles = numFiles;
		status = RFD_STATUS_OK;
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CalculateFileSetMetatdataFileSizes(RFD_FSET_MDATA_STRUCT * fileSetInfo)
{
	// Inputs are:
	//  fileSetInfo->totalSize
	//  fileSetInfo->numFiles
	//
	// Outputs are, for i = 0 to numFiles-1:
	//  fileSetInfo->fileInfo[i].size
	//

	// Note:
	// This general method of setting file sizes works as long as the
	// totalSize is >= (numFiles-1)*(numFiles)
	// e.g. for numFiles = 16, totalSize must be >= 240 (bytes).
	// Without this constraint, the lastFragmentSize could be 0 or negative.

	UINT32 fragmentSize, lastFragmentSize;
	UINT32 numFiles = fileSetInfo->numFiles;
	UINT32 totalSize = fileSetInfo->totalSize;
	unsigned int i;

	// check valid range of totalSize and numFiles.
	if(totalSize < ((numFiles-1)*numFiles)) {

		for(i = 0; i < numFiles; i++) {
			fileSetInfo->fileInfo[i].size = 0;
		}
		return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
	}

	// Calculate 'base' fragment size and special case last fragment size

	fragmentSize = totalSize / numFiles;

	if(totalSize % numFiles == 0) {
		// Total size is an exact multiple of the fragmentSize,
		// last fragement size is the same as all other fragment sizes.
		lastFragmentSize = fragmentSize;
	}
	else {
		// Round up fragmentSize and reduce the lastFragment size appropriately.
		fragmentSize += 1;
		lastFragmentSize = totalSize - ((numFiles-1)*fragmentSize);
	}

	// Set all fragments except last
	for(i = 0; i < numFiles-1; i++) {
		fileSetInfo->fileInfo[i].size = fragmentSize;
	}

	// Set Last fragment
	fileSetInfo->fileInfo[numFiles-1].size = lastFragmentSize;

	return RFD_STATUS_OK;
}


