////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_message_queue.c
///
/// @brief	rfd message queue class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"
#include "rfd_message_queue.h"

///////////////////////////////////////////////////////////////////////////////
static UINT32 CopyToModuloBuf(UCHAR * moduloBuf, UINT32 moduloBufSize, UINT32 moduloBufIndex,
							UCHAR * srcData, UINT32 srcDataSize)
{
	UINT32 newModuloBufIndex;
	UINT32 firstSize, secondSize;

	if(srcDataSize > moduloBufSize) {
		srcDataSize = moduloBufSize;
	}

	if( (moduloBufIndex + srcDataSize) == moduloBufSize) {
		RFD_MEMCPY(&moduloBuf[moduloBufIndex], srcData, srcDataSize);
		newModuloBufIndex = 0;
	}
	else if( (moduloBufIndex + srcDataSize) < moduloBufSize) {
		RFD_MEMCPY(&moduloBuf[moduloBufIndex], srcData, srcDataSize);
		newModuloBufIndex = moduloBufIndex + srcDataSize;
	}
	else { // (moduloBufIndex + srcDataSize) > moduloBufSize
		firstSize = moduloBufSize - moduloBufIndex;
		secondSize = srcDataSize - firstSize;
		RFD_MEMCPY(&moduloBuf[moduloBufIndex], srcData, firstSize);
		RFD_MEMCPY(&moduloBuf[0], srcData + firstSize, secondSize );
		newModuloBufIndex = secondSize;
	}

	return newModuloBufIndex;
}

///////////////////////////////////////////////////////////////////////////////
static UINT32 CopyFromModuloBuf(UCHAR * dstData, UINT32 dstDataSize,
							  UCHAR * moduloBuf, UINT32 moduloBufSize, UINT32 moduloBufIndex )
{
	UINT32 newModuloBufIndex;
	UINT32 firstSize, secondSize;

	if(dstDataSize > moduloBufSize) {
		dstDataSize = moduloBufSize;
	}

	if( (moduloBufIndex + dstDataSize) == moduloBufSize) {
		RFD_MEMCPY(dstData, &moduloBuf[moduloBufIndex], dstDataSize);
		newModuloBufIndex = 0;
	}
	else if( (moduloBufIndex + dstDataSize) < moduloBufSize) {
		RFD_MEMCPY(dstData, &moduloBuf[moduloBufIndex], dstDataSize);
		newModuloBufIndex = moduloBufIndex + dstDataSize;
	}
	else { // (moduloBufIndex + dstDataSize) > moduloBufSize
		firstSize = moduloBufSize - moduloBufIndex;
		secondSize = dstDataSize - firstSize;
		RFD_MEMCPY(dstData, &moduloBuf[moduloBufIndex], firstSize);
		RFD_MEMCPY(dstData + firstSize, &moduloBuf[0], secondSize );
		newModuloBufIndex = secondSize;
	}

	return newModuloBufIndex;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_MSG_QUEUE_Create(RFD_MSG_QUEUE_HANDLE * messageQueueHandle, INT32 dataBufSize)
{
	RFD_MSG_QUEUE_HANDLE hMessageQ;

	// Initialize to null.
	*messageQueueHandle = NULL;

	// Allocate the main structure, and clear to 0 (calloc)
	hMessageQ = (RFD_MSG_QUEUE_HANDLE) RFD_CALLOC(sizeof(RFD_MSG_QUEUE_STRUCT));
	if(hMessageQ == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
    }

	// create the semaphore
	hMessageQ->semaphoreHandle = RFD_CreateSemaphore(0, // initial count
												   RFD_MSG_QUEUE_SEMAPHORE_MAX_COUNT // max count
												   );
	if(hMessageQ->semaphoreHandle == NULL) {
		RFD_MSG_QUEUE_Delete(hMessageQ);
		return RFD_STATUS_ERROR_MEM_ALLOC;
    }

	// create the mutex
	hMessageQ->mutexHandle = RFD_CreateMutex();

	if (hMessageQ->mutexHandle == NULL) {
		RFD_MSG_QUEUE_Delete(hMessageQ);
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

	// Allocate the message data buffer and clear to 0 (calloc)
	hMessageQ->buf = (UCHAR *) RFD_CALLOC(sizeof(UCHAR)*dataBufSize);
	if(hMessageQ->buf == NULL) {
		RFD_MSG_QUEUE_Delete(hMessageQ);
		return RFD_STATUS_ERROR_MEM_ALLOC;
    }

	// Initialize head and tail indices, buf size and the semaphore count.
	hMessageQ->headIndex = 0;
	hMessageQ->tailIndex = 0;
	hMessageQ->semaphoreCount = 0;
	hMessageQ->bufSize = dataBufSize;

	*messageQueueHandle = hMessageQ;
	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_MSG_QUEUE_Delete(RFD_MSG_QUEUE_HANDLE messageQueueHandle)
{
	RFD_MSG_QUEUE_HANDLE hMessageQ;

	hMessageQ = messageQueueHandle;

	if(hMessageQ == NULL) {
		return RFD_STATUS_OK;
	}

	// Close the Semaphore
	if(hMessageQ->semaphoreHandle != NULL) {
		RFD_CloseSemaphore(hMessageQ->semaphoreHandle);
		hMessageQ->semaphoreHandle = NULL;
	}

	// Close the Mutex
	if (hMessageQ->mutexHandle != NULL) {
		RFD_CloseMutex(hMessageQ->mutexHandle);
		hMessageQ->mutexHandle = NULL;
	}

	// Close the message data buffer.
	if(hMessageQ->buf != NULL) {
		RFD_FREE(hMessageQ->buf);
		hMessageQ->buf = NULL;
	}

	// Free the message queue
	RFD_FREE(hMessageQ);

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_MSG_QUEUE_WaitForMessage(RFD_MSG_QUEUE_HANDLE messageQueueHandle, INT32 timeoutMilliseconds)
{
	RFD_STATUS status;

	status = RFD_WaitForSemaphore(messageQueueHandle->semaphoreHandle, timeoutMilliseconds);

	if(status == RFD_STATUS_OK) {
		// semaphore decremented.
		RFD_AcquireMutex(messageQueueHandle->mutexHandle, RFD_INFINITE);
		{
			// decrement the 'shaddow' semaphore count.
			messageQueueHandle->semaphoreCount--;
		}
		RFD_ReleaseMutex(messageQueueHandle->mutexHandle);
	}
	// else timeout or error.

	return status;
}


////////////////////////////////////////////////////////////////////////////////////////////////////
/// @internal
/// @brief	Get message from the message queue.
///
/// @param	messageQueueHandle	Handle of the message queue.
/// @param	outputMsgBuf		Buffer for output message data.
/// @param	ouputMsgBufSize		Size of the ouput message buffer.
/// @param	messageSizePtr		The message size pointer.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_MSG_QUEUE_GetMessage(RFD_MSG_QUEUE_HANDLE messageQueueHandle,
									UCHAR * outputMsgBuf, UINT32 ouputMsgBufSize,
									UINT32 * messageSizePtr)
{
	// Note: Call RFD_MSG_QUEUE_GetMessage() after each successful call to RFD_MSG_QUEUE_WaitForMessage()

	RFD_MSG_QUEUE_HANDLE hMessageQ;
	UINT32 messageSize;
	RFD_MSG_QUEUE_MSG_HEADER messageHeader;

	hMessageQ = messageQueueHandle;

	// Initialize output parameter.
	*messageSizePtr = 0;

	RFD_AcquireMutex(hMessageQ->mutexHandle, RFD_INFINITE);
	{
		if(hMessageQ->tailIndex == hMessageQ->headIndex) {
			// no messages to get, queue is empty.
			RFD_ReleaseMutex(hMessageQ->mutexHandle);
			RFD_DPrint(TEXT("RFD_MSG_QUEUE_GetMessage(). Function called, but no Messages available in Message Queue.\n"));
			return RFD_STATUS_ERROR_MESSAGE_QUEUE_EMPTY;
		}

		// Get the size of the message which is in the Message Header field.
		// Copy Message Header field, from the data buffer.
		// Also update the tailIndex to reflect the copy.
		hMessageQ->tailIndex = CopyFromModuloBuf((UCHAR *) &messageHeader, sizeof(RFD_MSG_QUEUE_MSG_HEADER),
												 hMessageQ->buf, hMessageQ->bufSize, hMessageQ->tailIndex );
		messageSize = messageHeader;

		if(messageSize > ouputMsgBufSize) {
			// User buffer is too small, this is an error condition.
			// Increment the tail index to flush this message, but don't copy the message.
			// This will recover from error only if the RFD_MSG_QUEUE_PutMessage() does the same adjustemt.
			// Otherwise, the Message Queue is corrupted and the GetMessage() is permanently
			// out-of-sync with the position of messages.
			hMessageQ->tailIndex += messageSize;
			if(hMessageQ->tailIndex >= hMessageQ->bufSize) {
				// handle modulo wrap.
				hMessageQ->tailIndex = hMessageQ->tailIndex %  hMessageQ->bufSize;
			}
			RFD_ReleaseMutex(hMessageQ->mutexHandle);
			RFD_DPrint(TEXT("RFD_MSG_QUEUE_GetMessage(). Error. Insufficient User buffer space, message not copied to user buffer\n"));
			return RFD_STATUS_ERROR_BUFFER_SIZE;
		}

		// Copy the Message from the queue buffer to the user buffer.
		// Also update the tailIndex to reflect the copy.
		hMessageQ->tailIndex = CopyFromModuloBuf(outputMsgBuf, messageSize,
												 hMessageQ->buf, hMessageQ->bufSize, hMessageQ->tailIndex );
	}
	RFD_ReleaseMutex(hMessageQ->mutexHandle);

	*messageSizePtr = messageSize;
	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Put a message to the message queue.
///
/// @param	messageQueueHandle	Handle of the message queue.
/// @param	inputData			Input data buffer.
/// @param	inputDataSize		Size of the input data.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

RFD_STATUS RFD_MSG_QUEUE_PutMessage(RFD_MSG_QUEUE_HANDLE messageQueueHandle, UCHAR * inputData, UINT32 inputDataSize)
{
	RFD_MSG_QUEUE_HANDLE hMessageQ;
	UINT32 availableSize;
	RFD_MSG_QUEUE_MSG_HEADER messageHeader = inputDataSize;

	hMessageQ = messageQueueHandle;

	RFD_AcquireMutex(hMessageQ->mutexHandle, RFD_INFINITE);
	{

		if(hMessageQ->semaphoreCount >= RFD_MSG_QUEUE_SEMAPHORE_MAX_COUNT) {
			// Max Semaphore value reached. Do not put data to buffer.
			// Treat like a buffer full condition.
			RFD_ReleaseMutex(hMessageQ->mutexHandle);
			RFD_DPrint(TEXT("RFD_MSG_QUEUE_PutMessage(). Warning, Max Semaphore Count Reached, data not written to buffer\n"));
			return RFD_STATUS_OK_MAX_SEMAPHORE_COUNT;
		}

		if(inputDataSize >= hMessageQ->bufSize) {
			// Input message size is bigger that the whole buffer.
			RFD_ReleaseMutex(hMessageQ->mutexHandle);
			RFD_DPrint(TEXT("RFD_MSG_QUEUE_PutMessage().Input message size too big for buffer, not written to buffer\n"));
			return RFD_STATUS_OK_MSG_OVERSIZED;
		}

		// Calculate the message queue buffer available space, head to tail.
		if(hMessageQ->headIndex < hMessageQ->tailIndex) {
			availableSize = hMessageQ->tailIndex - hMessageQ->headIndex;
		}
		else { // headIndex >= tailIndex
			availableSize = (hMessageQ->bufSize - hMessageQ->headIndex) +
							hMessageQ->tailIndex;
		}

		if( (inputDataSize + sizeof(RFD_MSG_QUEUE_MSG_HEADER)) >= (availableSize-1)) {
			// Insufficient Message Queue buffer space
			RFD_ReleaseMutex(hMessageQ->mutexHandle);
			RFD_DPrint(TEXT("RFD_MSG_QUEUE_PutMessage(). Insufficient Message Queue buffer space, message not written to buffer\n"));
			return RFD_STATUS_OK_BUF_FULL;
		}

		// Message can be written to queue.

		// Copy input data size, i.e. the Message Header field, to the data buffer.
		// Also update the headerIndex to reflect this copy.
		hMessageQ->headIndex = CopyToModuloBuf(hMessageQ->buf, hMessageQ->bufSize, hMessageQ->headIndex,
											   (UCHAR *) &messageHeader, sizeof(RFD_MSG_QUEUE_MSG_HEADER) );

		// Copy input data to the data buffer.
		// Also update the headerIndex to reflect this copy.
		hMessageQ->headIndex = CopyToModuloBuf(hMessageQ->buf, hMessageQ->bufSize, hMessageQ->headIndex,
											   inputData, inputDataSize );

		// Signal the Semaphore (increment the counting semaphore).
		// and increment the 'shaddow' semaphore count variable.
		RFD_SignalSemaphore(hMessageQ->semaphoreHandle);

		hMessageQ->semaphoreCount++;
	}
	RFD_ReleaseMutex(hMessageQ->mutexHandle);

	return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Get the overhead size in bytes of each message in the message queue.
///
/// This is the number of additional bytes the Message Queue requeires to manage each message
/// in the Message Queue buffer. The user can use this value in calculating
/// its required Message Queue buffer size.
///
/// @param	messageQueueHandle	Handle of the message queue.
/// @param	inputData			Input data buffer.
/// @param	inputDataSize		Size of the input data.
///
/// @return	RFD_STATUS_OK if successful, other RFD_STATUS enumeration value if failure.
////////////////////////////////////////////////////////////////////////////////////////////////////

UINT32 RFD_MSG_QUEUE_GetMessageOverheadSize(void)
{
    return sizeof(RFD_MSG_QUEUE_MSG_HEADER);
}
