////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_message_parse.c
///
/// @brief	rfd message parse class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"
#include "rfd_bitstream_io.h"

///////////////////////////////////////////////////////////////////////////////
void RFD_SetDefaultsFileSetMetadataExtensionFields(RFD_FSET_MDATA_STRUCT * fileSetMetadataMsg)
{
    fileSetMetadataMsg->blockMsgDmi = RFD_FSET_MDATA_BLK_MSG_DMI_DEFAULT_VALUE;
    return;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ParseFileSetMetadataExtensionRecord(
                RFD_FSET_MDATA_STRUCT * fileSetMetadataMsg,
                UCHAR extensionRecord[],
                UCHAR extensionRecordLenBytes)
{
	UINT32 dWord;
	RFD_BITSTREAM_INFO streamInfo;

    if(extensionRecordLenBytes == 0) {
        return RFD_STATUS_ERROR_BUFFER_SIZE;
    }

	////////////////////////////////////
    // Set default values for fields of extension records.
	////////////////////////////////////
    RFD_SetDefaultsFileSetMetadataExtensionFields(fileSetMetadataMsg);

	// Initialize Bitstream processing structure.
	if(RFD_BitstreamInit(&streamInfo,
						 extensionRecord,
						 extensionRecordLenBytes * RFD_BITS_PER_BYTE)) {
		return RFD_STATUS_ERROR_BITSTREAM_INIT_ERROR;
	}

	////////////////////////////////////
	// Read Block Message (Update Files) DMI
	////////////////////////////////////
	if(RFD_BitstreamGetU32(&streamInfo, RFD_FSET_MSG_UPDATE_FILES_DMI_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}
	fileSetMetadataMsg->blockMsgDmi = dWord;

	////////////////////////////////////
	// Skip the RFU field
	////////////////////////////////////
	if(RFD_BitstreamSeek(&streamInfo, RFD_FSET_MSG_UPDATE_FILES_DMIRFU_FIELD_NUM_BITS,
						 RFD_BSTRM_SEEK_OS_CURRENT)) {
		return RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR;
	}

    // No errors, so extension record was at least large enough to contain the
    // expected fields (for this version of the protocol).

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ParseFileSetMetadataMessage(RFD_BITSTREAM_INFO * streamInfo,
										   RFD_FSET_MDATA_STRUCT * fileSetMetadataMsg,
										   const RFD_MESSAGE_CONFIG_INFO * configInfo)
{
	// On entry, the streamInfo state structure will have the current bit position
	// pointing to the field following the Carousel ID.
	UINT32 dWord;
	int i;
	int remainingMessageBits;
	RFD_STATUS status;

	//////////////////////////
	// Convert the Baudot name string in the FSET NAME field to a
	// regularly-formatted string that is stored in fileSetMetadataMsg->name.
	//////////////////////////
	status = RFD_ConvertBaudotBitStreamToString( fileSetMetadataMsg->name, RFD_FSET_NAME_MAX_LEN,
									      streamInfo, RFD_FSET_NAME_MAX_NUM_BAUDOT_SYMBOLS );
	if(status != RFD_STATUS_OK) {
		return status;
	}

	////////////////////////////////////
	// Read FSET SIZE field
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, RFD_FSET_MSG_FSET_SIZE_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}
	fileSetMetadataMsg->totalSize = dWord;

	if(fileSetMetadataMsg->totalSize > configInfo->maxFileLenBytes) {
		return RFD_STATUS_ERROR_INVALID_FILE_SIZE;
	}

#ifdef DYNAMIC_BLOCK_MSG_CARID_ENABLE
	////////////////////////////////////
	// Read CAR field (Carousel on which Block Messages are received)
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, configInfo->nBitsFileSetUpdateCarField, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}
	fileSetMetadataMsg->blockMsgCarouselId = dWord;
#endif

	////////////////////////////////////
	// Read ALG, the Coding Algorithm.
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, RFD_FSET_MSG_CODING_ALG_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}
	fileSetMetadataMsg->codeType = dWord;

	switch(fileSetMetadataMsg->codeType) {

		case RFD_CODE_TYPE_SIMPLE_BLK:
		case RFD_CODE_TYPE_ECC1:
		case RFD_CODE_TYPE_ECC2:
		// These code types are supported so continue parsing.
			break;

		default:
		// Any other code type is not supported, stop parsing.
			return RFD_STATUS_UNSUPPORTED_CODE_TYPE;
	}


	////////////////////////////////////
	// Read COMP, the Compression Algorithm.
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, RFD_FSET_MSG_COMPRESSION_ALG_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}
	fileSetMetadataMsg->compType = dWord;

	switch(fileSetMetadataMsg->compType) {

		case RFD_COMP_TYPE_NONE:
		case RFD_COMP_TYPE_GZIP:
		// These compression types are supported, continue parsing.
			break;

		default:
		// Any other compression type is not supported, stop parsing.
			return RFD_STATUS_UNSUPPORTED_COMPRESSION_TYPE;
	}

	////////////////////////////////////
	// Read the LIFE field
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, RFD_FSET_MSG_LIFE_TIME_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}
	fileSetMetadataMsg->lifeTime = dWord;

	////////////////////////////////////
	// Read the NUM FILES field
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, RFD_FSET_MSG_NUM_FILES_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}
	fileSetMetadataMsg->numFiles = dWord+1;

	if(fileSetMetadataMsg->numFiles > RFD_FSET_MAX_NUM_FILES) {
		return RFD_STATUS_ERROR_INVALID_FSET_NUM_FILES;
	}

	for(i=0;i<fileSetMetadataMsg->numFiles;i++) {

		////////////////////////////////////
		// Read the File Id field for this file instance
		////////////////////////////////////
		if(RFD_BitstreamGetU32(streamInfo, RFD_FSET_MSG_FILE_ID_FIELD_NUM_BITS, &dWord)) {
			return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
		}
		fileSetMetadataMsg->fileInfo[i].id = dWord;

		////////////////////////////////////
		// Read the Block Length field for this file instance
		////////////////////////////////////
		if(RFD_BitstreamGetU32(streamInfo, RFD_FSET_MSG_BLK_LEN_FIELD_NUM_BITS, &dWord)) {
			return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
		}
		fileSetMetadataMsg->fileInfo[i].blkLenBytes = dWord;

		if(fileSetMetadataMsg->fileInfo[i].blkLenBytes > configInfo->maxBlockLenBytes) {
			return RFD_STATUS_ERROR_INVALID_BLOCK_LENGTH;
		}

#ifdef RFD_FSET_MESSAGE_INCLUDE_FILE_SIZES_ENABLE
		////////////////////////////////////
		// Read the File Size field for this file instance
		////////////////////////////////////
		if(RFD_BitstreamGetU32(streamInfo, RFD_FSET_MSG_FILE_SIZE_FIELD_NUM_BITS, &dWord)) {
			return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
		}
		fileSetMetadataMsg->fileInfo[i].size = dWord;

		if(fileSetMetadataMsg->fileInfo[i].size > configInfo->maxFileLenBytes) {
			return RFD_STATUS_ERROR_INVALID_FILE_SIZE;
		}
#else
		// There RFD Receiver will calculate the size for each file of the file set
		// based on shared transmitter/receiver calculation rule.
		fileSetMetadataMsg->fileInfo[i].size = 0;
#endif

		////////////////////////////////////
		// Read the File CRC field for this file instance
		////////////////////////////////////
		if(RFD_BitstreamGetU32(streamInfo, RFD_FSET_MSG_FILE_CRC_FIELD_NUM_BITS, &dWord)) {
			return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
		}
		fileSetMetadataMsg->fileInfo[i].crc = dWord;
	}

	////////////////////////////////////
	// Read the F+EXTL field
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, RFD_MSG_PRESENCE_FLAG_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}

	if(dWord == RFD_MGS_FIELD_IS_PRESENT_TRUE) {
		////////////////////////////////////
		// Ext Field is Present, read the EXT Length field.
		////////////////////////////////////
		if(RFD_BitstreamGetU32(streamInfo, RFD_FSET_MSG_EXT_LENGTH_FIELD_NUM_BITS, &dWord)) {
			return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
		}
		fileSetMetadataMsg->extensionLenBytes = dWord+1;
	}
	else {
		fileSetMetadataMsg->extensionLenBytes = 0;
	}

	if(fileSetMetadataMsg->extensionLenBytes != 0) {

        ////////////////////////////////////
        // Extension Record is present,
        // so read the extension record byte array into byte buffer, then parse the extension record.
        //
        // Note: an optimization would be to avoid copying into the intermediate byte buffer
        // and instead directly parse from the existing buffer. This could be done with the same bitstream instance.
        // Alternatively, a second bitstream instance could be instantiated, setup to start at
        // extension record start and end at extension record end. However, to use this second bitstream instance,
        // some minor changes to bitstream parsing functions would be needed to allow a bitstream
        // to begin at a non-byte aligned bit boundary.
	    ////////////////////////////////////

		if(RFD_BitstreamGetByteArray(streamInfo, fileSetMetadataMsg->extensionLenBytes, fileSetMetadataMsg->extension)) {
			return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
		}

        status = RFD_ParseFileSetMetadataExtensionRecord(fileSetMetadataMsg,
                    fileSetMetadataMsg->extension,
                    fileSetMetadataMsg->extensionLenBytes);

        if(status != RFD_STATUS_OK) {
            return status;
        }
    }
    else {
        ////////////////////////////////////
        // Extension Record is not present,
        // so set default values for fields gotten from the extension record.
	    ////////////////////////////////////
        RFD_SetDefaultsFileSetMetadataExtensionFields(fileSetMetadataMsg);
    }

	////////////////////////////////////
	// Skip PADDING field.
	// Adjust bitstream pointer to the next byte aligned bit postion
	// (no change if already byte aligned).
	////////////////////////////////////
	if(RFD_BitstreamSeek(streamInfo, 0, RFD_BSTRM_SEEK_OS_NEXT_BYTE_ALIGN)) {
		return RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR;
	}

	////////////////////////////////////
	// Validate the overall message length.
	////////////////////////////////////
	remainingMessageBits =  RFD_BitstreamGetStreamLengthBits(streamInfo) -
							RFD_BitstreamGetCurrBitPosition(streamInfo) -
							RFD_MSG_CRC_FIELD_NUM_BITS;

	if(remainingMessageBits != 0) {
		return RFD_STATUS_ERROR_INVALID_MESSAGE_LENGTH;
	}

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ParseBlockMessage(RFD_BITSTREAM_INFO * streamInfo,
								 RFD_BLOCK_MSG_STRUCT * blockMsg,
								 const RFD_MESSAGE_CONFIG_INFO * configInfo)
{
	// On entry, the streamInfo state structure will have the current bit position
	// pointing to the field following the Carousel ID.
	UINT32 dWord;
	int nBitsBlockIndexField = 0;
	int extensionFieldSizeBytes;
	INT32 blockLenBytes;

	////////////////////////////////////
	// Read File ID
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, RFD_MSG_FILE_ID_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}
	blockMsg->fileId = dWord;

	////////////////////////////////////
	// Read Block Type
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, RFD_MSG_BLK_TYPE_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}
	blockMsg->blkType = dWord;

	// Read Block Index, size and presence dependent on Block Type.
	switch(blockMsg->blkType) {

		case BLK_TYPE_SOURCE:
			// Block Index filed size for Source type block.
			nBitsBlockIndexField = RFD_MSG_SRC_TYPE_BLK_INDEX_FIELD_NUM_BITS;
			break;

		case BLK_TYPE_ECC1_CHECK:
			// Block Index filed size for ECC1 Check type block.
			nBitsBlockIndexField = RFD_MSG_ECC1_CHECK_TYPE_BLK_INDEX_FIELD_NUM_BITS;
			break;

		case BLK_TYPE_ECC2_CHECK:
			// Block Index filed size for ECC2 Check type block.
			nBitsBlockIndexField = RFD_MSG_ECC2_CHECK_TYPE_BLK_INDEX_FIELD_NUM_BITS;
			break;

		case BLK_TYPE_SIMPLE_PARITY:
			// Simple Parity Block does not have a Block Index.
			// Just set to 0.
			nBitsBlockIndexField = 0;
			break;

		default:
			// Unsupported Block Type.
			return RFD_STATUS_UNSUPPORTED_BLOCK_TYPE;
	}

	if(nBitsBlockIndexField != 0) {
		////////////////////////////////////
		// Read the Block Index Field
		////////////////////////////////////
		if(RFD_BitstreamGetU32(streamInfo, nBitsBlockIndexField, &dWord)) {
			return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
		}
		blockMsg->blkIndex = dWord;
	}
	else {
		// Block Index not implemented for Simple Parity type block.
		// Set value to zero by design. Generic handling function may be
		// used to handle Simple Parity, ECC1 and ECC2 check blocks
		// in which Simple Parity is treated as a generic check block
		// with a block index (always = 0 in the case of Simple Parity).
		blockMsg->blkIndex = 0;
	}

	////////////////////////////////////
	// Skip PADDING field.
	// Adjust bitstream pointer to the next byte aligned bit postion
	// (no change if already byte aligned).
	////////////////////////////////////
	if(RFD_BitstreamSeek(streamInfo, 0, RFD_BSTRM_SEEK_OS_NEXT_BYTE_ALIGN)) {
		return RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR;
	}

	////////////////////////////////////
	// Read Extension Filed Size.
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, RFD_MSG_BLK_EXT_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}
	extensionFieldSizeBytes = dWord;

	if(extensionFieldSizeBytes != 0) {
		#if (RFDR_ENABLE_BLK_MSG_EXT_FIELD_EXTRACTION == 1)
		blockMsg->extensionLenBytes = extensionFieldSizeBytes;
		// Read the Extension Field.
		if(RFD_BitstreamGetByteArray(streamInfo, extensionFieldSizeBytes, blockMsg->extension)) {
			return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
		}
		#else
		// Skip the Extension Field.
		if(RFD_BitstreamSeek(streamInfo, extensionFieldSizeBytes * RFD_BITS_PER_BYTE,
							 RFD_BSTRM_SEEK_OS_CURRENT)) {
			return RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR;
		}
		#endif
	}

	// Calculate the length of the Block data payload.
	blockLenBytes = (RFD_BitstreamGetStreamLengthBits(streamInfo) -
					 RFD_BitstreamGetCurrBitPosition(streamInfo) -
					 RFD_MSG_CRC_FIELD_NUM_BITS) / RFD_BITS_PER_BYTE;

	// Check block length before copying block to buffer.
	if(blockLenBytes > configInfo->maxBlockLenBytes) {
		return RFD_STATUS_ERROR_INVALID_BLOCK_LENGTH;
	}

	blockMsg->blkLenBytes = blockLenBytes;

	////////////////////////////////////
	// Read the Block Data into the block buffer.
	////////////////////////////////////
	if(RFD_BitstreamGetByteArray(streamInfo, blockLenBytes, blockMsg->block)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ParseMessageExt(
	RFD_MSG_TYPE * messageType,
	RFD_FSET_MDATA_STRUCT * fileSetInfoMsg,
	RFD_BLOCK_MSG_STRUCT * blockMsg,
	UCHAR  bitstreamBuf[],
	UINT32 messageBitstreamLenBytes,
	const RFD_MESSAGE_CONFIG_INFO * configInfo)
{
	UINT32 dWord;
	RFD_BITSTREAM_INFO streamInfo;
	RFD_STATUS status;
	UINT32 calculatedCrc;

	// Initialize message type to invalid.
	*messageType = MSG_TYPE_INVALID;

	// Initialize Bitstream processing structure.
	if(RFD_BitstreamInit(&streamInfo,
						 bitstreamBuf,
						 messageBitstreamLenBytes * RFD_BITS_PER_BYTE)) {
		return RFD_STATUS_ERROR_BITSTREAM_INIT_ERROR;
	}

	////////////////////////////////////
	// Validate the Message CRC
	////////////////////////////////////

	// Calculate the crc.
	RFD_ResetCrc(&calculatedCrc);
	RFD_CalculateCrc(&calculatedCrc,
					 bitstreamBuf,
					 messageBitstreamLenBytes - RFD_MSG_CRC_FIELD_NUM_BYTES);

	// Seek to the Message CRC field, the last field of the bitstream = bitstream_end - crc_len
	if(RFD_BitstreamSeek(&streamInfo, -RFD_MSG_CRC_FIELD_NUM_BITS , RFD_BSTRM_SEEK_OS_END)) {
		return RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR;
	}

	// Read the actual message crc.
	if(RFD_BitstreamGetU32(&streamInfo, RFD_MSG_CRC_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}

	// Return with crc error indication if calculated crc does not match the message crc.
	if(calculatedCrc != dWord) {
		return RFD_STATUS_ERROR_CRC;
	}

	// Seek back to beginning of bitstream.
	if(RFD_BitstreamSeek(&streamInfo, 0 , RFD_BSTRM_SEEK_OS_BEGINING)) {
		return RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR;
	}

	////////////////////////////////////
	// Read PVN
	////////////////////////////////////
	if(RFD_BitstreamGetU32(&streamInfo, RFD_MSG_PVN_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}

	// Return if this PVN is not supported
	if(dWord != configInfo->protocolVersionNo) {
		return RFD_STATUS_MSG_PROTOCOL_VERSION_NUM_OUT_OF_RANGE;
	}

	////////////////////////////////////
	// Read CARID
	////////////////////////////////////
	if(RFD_BitstreamGetU32(&streamInfo, configInfo->nBitsCarouselIdField, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}

	// Determine the message type.
	// and call the specfic parsing function to parse the message.
	if( (blockMsg != NULL) &&
		(configInfo->blockMsgCarouselId != -1) &&
        (dWord == (UINT32)configInfo->blockMsgCarouselId) ) {

		// For case of DYNAMIC_BLOCK_MSG_CARID_ENABLE, blockMsgCarouselId = -1 signals that
		// blockMsgCarouselId is not yet available
		// (dWord = -1 is not a possible dWord value, but include the -1 comparison above for clarity)

		// Safest to clear the block message structure for consistentcy.
		// If performance optimization is required, this could be removed.
		RFD_MEMSET(blockMsg, 0, sizeof(RFD_BLOCK_MSG_STRUCT));

		*messageType = MSG_TYPE_BLOCK;
		status = RFD_ParseBlockMessage(&streamInfo,
									   blockMsg,
									   configInfo);
	}
	else if( (fileSetInfoMsg != NULL) &&
             (dWord == configInfo->fileSetMdataMsgCarouselId) ) {

		// Clear the File Set Info structure into which a File Set Info (metadata) message
		// will be parsed. Change detection of this message may be accomplished by mem compare,
		// in which case clearing the entire structure will provide consistent results.
		// E.g. The Name field may be parsed only up to the END character, in which case the
		// remaining characters of the max length string array may not be written during parsing.
		// (also note that the fileset structure shares the same memory area as the block structure
		//  via union).
		RFD_MEMSET(fileSetInfoMsg, 0, sizeof(RFD_FSET_MDATA_STRUCT));

		*messageType = MSG_TYPE_FSET_MDATA;
		status = RFD_ParseFileSetMetadataMessage(&streamInfo,
												 fileSetInfoMsg,
												 configInfo);
	}
	else {
		*messageType = MSG_TYPE_UNKNOWN;
		status = RFD_STATUS_OK;
	}

	return status;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ParseMessage(RFD_MESSAGE_INFO * messageInfo, BOOL isBlockMsgParsingEnabled)
{
    RFD_STATUS status;
	RFD_FSET_MDATA_STRUCT * fileSetInfoMsg;
	RFD_BLOCK_MSG_STRUCT * blockMsg;

    fileSetInfoMsg = &messageInfo->message.fileSetInfoMsg;

    if(isBlockMsgParsingEnabled) {
        blockMsg = &messageInfo->message.blockMsg;
    }
    else {
        // Null pointer for blockMsg indicates Blk Message Parsing is disabled.
        blockMsg = NULL;
    }

    status = RFD_ParseMessageExt(
	            &messageInfo->messageType,
	            fileSetInfoMsg,
	            blockMsg,
	            messageInfo->bitstreamBuf,
	            messageInfo->messageBitstreamLenBytes,
	            &messageInfo->configInfo);

    return status;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief	Peak at the RFD Message Header to output parameters of the Header.
///
/// This is the LL (low level) version of the header peek function, which expects
/// a pre-initialized Stream Info structure as input.
/// On successful return, the Stream Info struct will point the bitstream
/// to the first bit of Message Body (first byte following the header).
///
/// @param	streamInfo	          (in/out) Handle of the Bitstream Info structure, which must be
///                               pre-initialized by the caller. On successful return, the
///                               streamInfo will point bitstream to the first bit following
///                               the Message Header field.
/// @param	nBitsCarouselIdField  (output) the Protocol Version Number param. of the Msg Header.
/// @param	carouselId		      (output) the Carousel ID param. of the Msg Header.
///
/// @return	RFD_STATUS_OK if successful.
////////////////////////////////////////////////////////////////////////////////////////////////////

static RFD_STATUS RFD_PeekMessageHeaderLL(RFD_BITSTREAM_INFO * streamInfo,
								 UCHAR nBitsCarouselIdField,
								 UCHAR * protocolVersionNo /* output */,
								 UCHAR * carouselId /* output */)
{
	UINT32 dWord;

	// Initialize output variables to some default values
	// incase of early return due to error.
	*protocolVersionNo = 0;
	*carouselId = 0;

	////////////////////////////////////
	// Skip the Message CRC validation.
	// This function is intended for providing the Message header fields
	// so as to determine if this message should be processed.
	// Any further processing would then perform the complete parsing
	// including CRC validation.
	////////////////////////////////////


	////////////////////////////////////
	// Read PVN
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, RFD_MSG_PVN_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}

	// Output the protocol version number.
	*protocolVersionNo = (UCHAR) dWord;


	////////////////////////////////////
	// Read CARID
	////////////////////////////////////
	if(RFD_BitstreamGetU32(streamInfo, nBitsCarouselIdField, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}

	// Output the carousel id.
	*carouselId = (UCHAR) dWord;

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_PeekMessageHeader(UCHAR bitstreamBuf[],
								 UINT32 messageBitstreamLenBytes,
								 UCHAR nBitsCarouselIdField,
								 UCHAR * protocolVersionNo /* output */,
								 UCHAR * carouselId /* output */)
{
	RFD_BITSTREAM_INFO streamInfo;
    RFD_STATUS status;

	// Initialize output variables to some default values
	// incase of early return due to error.
	*protocolVersionNo = 0;
	*carouselId = 0;

	// Initialize Bitstream processing structure.
	if(RFD_BitstreamInit(&streamInfo,
						 bitstreamBuf,
						 messageBitstreamLenBytes * RFD_BITS_PER_BYTE)) {
		return RFD_STATUS_ERROR_BITSTREAM_INIT_ERROR;
	}

    // Call lower level Peak Msg Header function, passsing the initialized Stream Info input.
    status = RFD_PeekMessageHeaderLL(&streamInfo,
								     nBitsCarouselIdField,
								     protocolVersionNo,
								     carouselId);

    return status;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_PeekBlockMessageFileId(UCHAR bitstreamBuf[],
								     UINT32 messageBitstreamLenBytes,
	                                 RFD_MESSAGE_CONFIG_INFO * configInfo,
    								 UINT16 * fileId /* output */)
{
	RFD_BITSTREAM_INFO streamInfo;
    RFD_STATUS status;
	UCHAR protocolVersionNo;
	UCHAR carouselId;
	UINT32 dWord;

	// Initialize output variables to some default values
	// incase of early return due to error.
	protocolVersionNo = 0;
	carouselId = 0;
    *fileId = 0;

	// Initialize Bitstream processing structure.
	if(RFD_BitstreamInit(&streamInfo,
						 bitstreamBuf,
						 messageBitstreamLenBytes * RFD_BITS_PER_BYTE)) {
		return RFD_STATUS_ERROR_BITSTREAM_INIT_ERROR;
	}

    // Call lower level Peak Msg Header function, passsing the initialized Stream Info input.
    status = RFD_PeekMessageHeaderLL(&streamInfo,
								     configInfo->nBitsCarouselIdField,
								     &protocolVersionNo,
								     &carouselId);

    if(status != RFD_STATUS_OK) {
        return status;
    }

    if(protocolVersionNo != configInfo->protocolVersionNo) {
        return RFD_STATUS_MSG_PROTOCOL_VERSION_NUM_OUT_OF_RANGE;
    }

    if(carouselId != configInfo->blockMsgCarouselId) {
        return RFD_STATUS_ERROR_INVALID_MESSAGE_TYPE;
    }

	// On RFD_PeekMessageHeaderLL success return, the streamInfo state structure will have the
    // current bit position pointing to the field following the Carousel ID.

	////////////////////////////////////
	// Read File ID
	////////////////////////////////////
	if(RFD_BitstreamGetU32(&streamInfo, RFD_MSG_FILE_ID_FIELD_NUM_BITS, &dWord)) {
		return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
	}

    // output fileId
	*fileId = (UINT16) dWord;

    return RFD_STATUS_OK;
}
