////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_linked_list.h
///
/// @brief	Declares the rfd linked list class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2013 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef RFD_LINKED_LIST_H
#define RFD_LINKED_LIST_H

#include "rfd_config.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef void * RFD_LLIST_USER_DATA_HANDLE;

typedef struct rfd_llist_node_struct {
    RFD_LLIST_USER_DATA_HANDLE data;
    struct rfd_llist_node_struct * prev;
    struct rfd_llist_node_struct * next;
} RFD_LLIST_NODE_STRUCT, *RFD_LLIST_NODE_HANDLE;

typedef struct {
    RFD_LLIST_NODE_STRUCT * headNode;
    RFD_LLIST_NODE_STRUCT * tailNode;

    // currentBrowseNode: the last node browse result after calls to _First() or _Next().
    // Can be updated from non-null to null if the node that is the current browse
    // node is _Remove()'ed. In this Null case for current browse node, the nullCurrentBrowsePrevNode is updated
    // to contain the resulting browse node that would occur following a subsequent _Prev() (if implmented);
    // and likewise, nullCurrentBrowseNextNode is updated to contain the resulting browse node
    // that would occur following a subsequent _Next().
    RFD_LLIST_NODE_STRUCT * currentBrowseNode;
    RFD_LLIST_NODE_STRUCT * nullCurrentBrowsePrevNode;
    RFD_LLIST_NODE_STRUCT * nullCurrentBrowseNextNode;

} RFD_LLIST_STRUCT, *RFD_LLIST_HANDLE;

typedef RFD_STATUS (*RFD_LLIST_ITERATOR_CALLBACK)(
    RFD_LLIST_NODE_STRUCT * currentNode,
    void * data,
    void * callbackArg
    );

typedef void (*RFD_LLIST_DELETE_USER_DATA_CALLBACK_FCN) (
    void * listNodeUserDataPtr
    );

///////////////////////////////////////////////////////////////////////////////
// RFD_LLIST_SORT_COMPARE_CALLBACK_FCN:
//
// Callback uses the below rules to sort list from high to low user_metric(),
// where RFD_LLIST_First()/Next() will then advance from high to low user_metric().
// return  <negative value> when: user_metric(userDataA) < user_metric(userDataB)
// return                 0 when: user_metric(userDataA) = user_metric(userDataB)
// return  <positive value> when: user_metric(userDataA) > user_metric(userDataB)
//
// Alternatively, to sort from low to hight, the user swaps the '<' and the '>' signs above
// and follows that rule for it's user_metric and callback return value.
///////////////////////////////////////////////////////////////////////////////
typedef int (*RFD_LLIST_SORT_COMPARE_CALLBACK_FCN) (
    void * userDataA,
    void * userDataB,
    void * sortCompareCallbackArg
    );

///////////////////////////////////////////////////////////////////////////////
RFD_LLIST_HANDLE RFD_LLIST_CreateList(void);

void RFD_LLIST_DeleteList(
            RFD_LLIST_HANDLE listHandle,
            RFD_LLIST_DELETE_USER_DATA_CALLBACK_FCN deleteUserDataCallbackFcn );

RFD_STATUS RFD_LLIST_InsertAtTail( RFD_LLIST_HANDLE listHandle, RFD_LLIST_USER_DATA_HANDLE data );
RFD_STATUS RFD_LLIST_InsertAtHead( RFD_LLIST_HANDLE listHandle, RFD_LLIST_USER_DATA_HANDLE data );
RFD_STATUS RFD_LLIST_Insert( RFD_LLIST_HANDLE listHandle, RFD_LLIST_USER_DATA_HANDLE data );

RFD_STATUS RFD_LLIST_TransferNode(
                RFD_LLIST_HANDLE dstListHandle,
                RFD_LLIST_HANDLE srcListHandle,
                RFD_LLIST_NODE_HANDLE node );

RFD_STATUS RFD_LLIST_GetNumNodes(
    RFD_LLIST_HANDLE listHandle,
    size_t * numNodesPtr );

RFD_STATUS RFD_LLIST_First(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE * foundNodeHandlePtr );

RFD_STATUS RFD_LLIST_Next(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE * foundNodeHandlePtr );

RFD_STATUS RFD_LLIST_CurrentBrowseNode(
                RFD_LLIST_HANDLE listHandle,
                RFD_LLIST_NODE_HANDLE * foundNodeHandlePtr );

RFD_LLIST_USER_DATA_HANDLE RFD_LLIST_GetUserData(
    RFD_LLIST_NODE_HANDLE referenceNode,
    RFD_STATUS * statusPtr  );

RFD_STATUS RFD_LLIST_SetUserData(
    RFD_LLIST_NODE_HANDLE referenceNode,
    RFD_LLIST_USER_DATA_HANDLE userDataHandle );

RFD_STATUS RFD_LLIST_InsertAfterNode(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE referenceNode,
    RFD_LLIST_USER_DATA_HANDLE data );

RFD_STATUS RFD_LLIST_InsertBeforeNode(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE referenceNode,
    RFD_LLIST_USER_DATA_HANDLE data );

RFD_STATUS RFD_LLIST_RemoveNode(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE referenceNode );

RFD_STATUS RFD_LLIST_Sort(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_SORT_COMPARE_CALLBACK_FCN sortCompareCallbackFcn,
    void * sortCompareCallbackArg );

RFD_STATUS RFD_LLIST_2DFirst(
                RFD_LLIST_HANDLE hOuterList,
                RFD_LLIST_NODE_HANDLE * hFoundOuterListNodePtr,
                RFD_LLIST_NODE_HANDLE * hFoundInnerListNodePtr );

RFD_STATUS RFD_LLIST_2DNext(
                RFD_LLIST_HANDLE hOuterList,
                RFD_LLIST_NODE_HANDLE * hFoundOuterListNodePtr,
                RFD_LLIST_NODE_HANDLE * hFoundInnerListNodePtr );

RFD_STATUS RFD_LLIST_2DGetNumNodes(
                RFD_LLIST_HANDLE hOuterList,
                size_t * numNodesPtr );

#ifdef __cplusplus
}
#endif

#endif // RFD_LINKED_LIST_H
