////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_linked_list.c
///
/// @brief	rfd linked list class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2013 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"
#include "rfd_linked_list.h"


//////////////////////////////////////////////////////////
static RFD_LLIST_NODE_STRUCT * _CreateNode(void)
{
    RFD_LLIST_NODE_STRUCT * node;

    node = (RFD_LLIST_NODE_STRUCT *) RFD_MALLOC(sizeof(RFD_LLIST_NODE_STRUCT));

    if(node == NULL)
    {
        return NULL;
    }

    // Initialize node
    node->next = NULL;
    node->prev = NULL;
    node->data = NULL;

    return node;
}

//////////////////////////////////////////////////////////
static void _DeleteNode(RFD_LLIST_NODE_STRUCT * node)
{
    if(node == NULL)
    {
        return;
    }

    RFD_FREE(node);

    return;
}

//////////////////////////////////////////////////////////
void  _updateBrowseInfoOnRemoveNode(
            RFD_LLIST_HANDLE listHandle,
            RFD_LLIST_NODE_HANDLE removedNode )
{
    if(listHandle->currentBrowseNode == removedNode) {

        // currentBrowseNode is not null and is equal to removedNode
        listHandle->nullCurrentBrowsePrevNode = listHandle->currentBrowseNode->prev;
        listHandle->nullCurrentBrowseNextNode = listHandle->currentBrowseNode->next;
        listHandle->currentBrowseNode = NULL;

        // currentBrowseNode is now in a "null condition", with special "predictors"
        // nullCurrentBrowsePrevNode and nullCurrentBrowseNextNode tracking where to
        // go next on any subsequent _Next() or _Prev() seeking.
    }
    else if(listHandle->currentBrowseNode == NULL) {

        // currentBrowseNode is already in a "null" condition due to a prior _RemoveNode()
        // or browse past beginning or ending of list.
        // See if the removedNode corresponds to the "predictor" Prev/Next pointers and if so,
        // update the predictor appropriately.

        if( removedNode == listHandle->nullCurrentBrowsePrevNode) {

            listHandle->nullCurrentBrowsePrevNode = removedNode->prev;
        }

        if( removedNode == listHandle->nullCurrentBrowseNextNode) {

            listHandle->nullCurrentBrowseNextNode = removedNode->next;
        }

        // note that it's possible that both if conditions above can be true
        // i.e. that nullCurrentBrowsePrevNode equals nullCurrentBrowseNextNode.
        // This can be due to a Remove of the currently browsed node followed by an Insert
        // at that list postion, followed by a Remove again.
    }

    return;
}

//////////////////////////////////////////////////////////
void  _updateBrowseInfoOnInsertNode(
            RFD_LLIST_HANDLE listHandle,
            RFD_LLIST_NODE_HANDLE insertedNode )
{
    if(listHandle->currentBrowseNode == NULL) {

        if(listHandle->nullCurrentBrowsePrevNode == insertedNode->prev) {
            listHandle->nullCurrentBrowsePrevNode = insertedNode;
        }

        if(listHandle->nullCurrentBrowseNextNode == insertedNode->next) {
            listHandle->nullCurrentBrowseNextNode = insertedNode;
        }

        // Note that it's expected that both the above if conditions are
        // true together or false together.

    }
    // else currentBrowseNode is not null. In this case there's no extra work to update Browse info.

    return;
}

//////////////////////////////////////////////////////////
RFD_STATUS _RemoveNodeNoDelete(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE referenceNode
    )
{
    if( listHandle == NULL || referenceNode == NULL )
    {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    // update forward link around the removed node.
    if( referenceNode->prev != NULL )
    {
        referenceNode->prev->next = referenceNode->next;
    }
    else
    {
        // referenceNode must currently be the head Node,
        // update head Node
        listHandle->headNode = referenceNode->next;

        if( listHandle->headNode != NULL )
        {
            // must set prev of Head to NULL (used to check if a node is head throught llist module)
            listHandle->headNode->prev = NULL;
        }
    }

    // update backward link around the removed node.
    if( referenceNode->next != NULL )
    {
        referenceNode->next->prev = referenceNode->prev;
    }
    else
    {
        // referenceNode must currently be the tail Node,
        // update tail Node
        listHandle->tailNode = referenceNode->prev;

        if( listHandle->tailNode != NULL )
        {
            // must set next of Tail to NULL (used to check if a node is the tail throught llist module)
            listHandle->tailNode->next = NULL;
        }
    }

    // Update Browse info upon _RemoveNode(). Ensure that the
    // removed Node prev and next handle pointers have not been
    // modified before call.
    _updateBrowseInfoOnRemoveNode(listHandle, referenceNode);

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
RFD_STATUS _InsertAllocatedNodeAtHead( RFD_LLIST_HANDLE listHandle, RFD_LLIST_NODE_HANDLE newNode )
{
    RFD_LLIST_NODE_STRUCT * referenceNode;

    if(listHandle->headNode == NULL)
    {
        // list is empty (both head and tail nodes must be null)

        // set new node as the tail node also for initially empty list.
        listHandle->tailNode = newNode;
        // and next pointer is also null for the tail node.
        listHandle->tailNode->next = NULL;
    }
    else
    {
        // list was not empty

        referenceNode = listHandle->headNode;

        // update backward and forward links to nodes involved.
        newNode->next = referenceNode;
        referenceNode->prev = newNode;
    }

    listHandle->headNode = newNode;
    // must set prev of Head to NULL (used to check if a node is head throught llist module)
    listHandle->headNode->prev = NULL;

    // update Browse info after inserting a node.
    // Ensure that the inserted node prev/next pointers have been
    // updated before call.
    _updateBrowseInfoOnInsertNode( listHandle, newNode );

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
RFD_LLIST_HANDLE RFD_LLIST_CreateList(void)
{
    RFD_LLIST_HANDLE listHandle = (RFD_LLIST_HANDLE) RFD_MALLOC(sizeof(RFD_LLIST_STRUCT));

    if(listHandle != NULL)
    {
        // list is initially empty
        listHandle->headNode = NULL;
        listHandle->tailNode = NULL;

        // initialize browse info.
        listHandle->currentBrowseNode = NULL;
        listHandle->nullCurrentBrowsePrevNode = NULL;
        listHandle->nullCurrentBrowseNextNode = NULL;
    }

    return listHandle;
}

//////////////////////////////////////////////////////////
void RFD_LLIST_DeleteList(
            RFD_LLIST_HANDLE listHandle,
            RFD_LLIST_DELETE_USER_DATA_CALLBACK_FCN deleteUserDataCallbackFcn
            )
{
    RFD_LLIST_NODE_STRUCT * current, * next;

    if(listHandle == NULL)
    {
        return;
    }

    // first node in list (can be NULL - empty list)
    current = listHandle->headNode;

    // Free each node
    while(current != NULL)
    {
        next = current->next;

        // delete user data if caller provided delete function.
        if( deleteUserDataCallbackFcn != NULL) {
            deleteUserDataCallbackFcn( current->data );
            current->data = NULL;
        }

        // delete the node.
        _DeleteNode(current);
        current = next;
    }

    // Free the list (container)
    RFD_FREE(listHandle);

    return;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_TransferNode(
                RFD_LLIST_HANDLE dstListHandle,
                RFD_LLIST_HANDLE srcListHandle,
                RFD_LLIST_NODE_HANDLE node )
{
    RFD_STATUS status;

    // remove node from source list
    status = _RemoveNodeNoDelete( srcListHandle, node );

    if(status != RFD_STATUS_OK) {
        return status;
    }

    // insert node to destination list
    status = _InsertAllocatedNodeAtHead( dstListHandle, node );

    return status;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_InsertAtHead( RFD_LLIST_HANDLE listHandle, RFD_LLIST_USER_DATA_HANDLE data )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_STRUCT * newNode;

    // create a new node
    newNode = _CreateNode();

    if(newNode == NULL)
    {
        // error
        return RFD_STATUS_ERROR_MEM_ALLOC;
    }

    // Assign data to the new node.
    newNode->data = data;

    // Complete the insert
    status = _InsertAllocatedNodeAtHead( listHandle, newNode );

    return status;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_Insert( RFD_LLIST_HANDLE listHandle, RFD_LLIST_USER_DATA_HANDLE data )
{
    return RFD_LLIST_InsertAtHead( listHandle, data );
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_GetNumNodes(
    RFD_LLIST_HANDLE listHandle,
    size_t * numNodesPtr )
{
    RFD_LLIST_NODE_STRUCT * current;
    size_t numNodes = 0;

    if( listHandle == NULL )
    {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    // first node in list can be NULL - empty list
    current = listHandle->headNode;

    while(current != NULL)
    {
        numNodes++;
        current = current->next; // last node next ptr will be null.
    }

    // output to caller.
    *numNodesPtr = numNodes;

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_First(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE * foundNodeHandlePtr )
{
    if( listHandle == NULL )
    {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    listHandle->currentBrowseNode = listHandle->headNode;
    listHandle->nullCurrentBrowsePrevNode = NULL;
    listHandle->nullCurrentBrowseNextNode = NULL;

    // output the head node as the first found node.
    // It will be NULL if the list is empty.
    *foundNodeHandlePtr = listHandle->currentBrowseNode;

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_Next(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE * foundNodeHandlePtr )
{

    if( listHandle == NULL)
    {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    if( listHandle->currentBrowseNode !=  NULL )
    {
        RFD_LLIST_NODE_HANDLE nextNode;

        nextNode = listHandle->currentBrowseNode->next;

        if(nextNode == NULL) {
            // this update to browse postion will result in NULL currentBrowseNode
            // (i.e. past end of list). Update nullCurrentBrowsePrevNode appropriately before
            // writing currentBrowseNode.
            listHandle->nullCurrentBrowsePrevNode = listHandle->currentBrowseNode;
            listHandle->nullCurrentBrowseNextNode = NULL;
        }
        else {
            listHandle->nullCurrentBrowsePrevNode = NULL;
            listHandle->nullCurrentBrowseNextNode = NULL;
        }

        // finally, update currentBrowseNode
        listHandle->currentBrowseNode = nextNode;
        // updated listHandle->currentBrowseNode will equal null if we've gone past the end of list.
    }
    else // currentBrowseNode is NULL
    {
        if(listHandle->nullCurrentBrowseNextNode != NULL) {

            // set current browse to the "browse next node for when current browse is null"
            listHandle->currentBrowseNode = listHandle->nullCurrentBrowseNextNode;
            // set "nullCurrent Next/Prev" nodes to null (currentBrowseNode is now non-null).
            listHandle->nullCurrentBrowseNextNode = NULL;
            listHandle->nullCurrentBrowsePrevNode = NULL;
        }
        // else all browse nodes remain in same state - currentBrowseNode is null.
        // (nullCurrentBrowsePrevNode could be non-null - i.e. past end of list).
    }

    // Update the output to caller.
    *foundNodeHandlePtr = listHandle->currentBrowseNode;

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_Prev(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE * foundNodeHandlePtr )
{
    if( listHandle == NULL)
    {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    if( listHandle->currentBrowseNode !=  NULL )
    {
        RFD_LLIST_NODE_HANDLE prevNode;

        prevNode = listHandle->currentBrowseNode->prev;

        if(prevNode == NULL) {
            // this update to browse postion will result in NULL currentBrowseNode
            // (i.e. past beginning of list). Update nullCurrentBrowsePrevNode appropriately before
            // writing currentBrowseNode.
            listHandle->nullCurrentBrowseNextNode = listHandle->currentBrowseNode;
            listHandle->nullCurrentBrowsePrevNode = NULL;
        }
        else {
            listHandle->nullCurrentBrowsePrevNode = NULL;
            listHandle->nullCurrentBrowseNextNode = NULL;
        }

        // finally, update currentBrowseNode
        listHandle->currentBrowseNode = prevNode;
        // updated listHandle->currentBrowseNode will equal null if we've gone past the beginning of list.
    }
    else // currentBrowseNode is NULL
    {
        if(listHandle->nullCurrentBrowsePrevNode != NULL) {

            // set current browse to the "browse prev node for when current browse is null"
            listHandle->currentBrowseNode = listHandle->nullCurrentBrowsePrevNode;
            // set "nullCurrent Next/Prev" nodes to null.
            listHandle->nullCurrentBrowseNextNode = NULL;
            listHandle->nullCurrentBrowsePrevNode = NULL;
        }
        // else all browse nodes remain in same state - currentBrowseNode is null.
        // (nullCurrentBrowseNextNode could be non-null - i.e. past beginning of list).
    }

    // Update the output to caller.
    *foundNodeHandlePtr = listHandle->currentBrowseNode;

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_CurrentBrowseNode(
                RFD_LLIST_HANDLE listHandle,
                RFD_LLIST_NODE_HANDLE * foundNodeHandlePtr )
{
    if( listHandle == NULL )
    {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    *foundNodeHandlePtr = listHandle->currentBrowseNode;

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
RFD_LLIST_USER_DATA_HANDLE RFD_LLIST_GetUserData(
    RFD_LLIST_NODE_HANDLE referenceNode,
    RFD_STATUS * statusPtr
    )
{
    if( referenceNode == NULL ) {
        *statusPtr = RFD_STATUS_ERROR_INVALID_VALUE;
        return NULL;
    }

    *statusPtr = RFD_STATUS_OK;
    return referenceNode->data;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_SetUserData(
    RFD_LLIST_NODE_HANDLE referenceNode,
    RFD_LLIST_USER_DATA_HANDLE userDataHandle
    )
{
    if( referenceNode == NULL )
    {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    referenceNode->data = userDataHandle;

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_RemoveNode(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE referenceNode
    )
{
    RFD_STATUS status;

    // remove the node from the list, without deleting/deallocating the node.
    status = _RemoveNodeNoDelete(
                    listHandle,
                    referenceNode );

    if(status != RFD_STATUS_OK) {
        return status;
    }

    // finally, delete the node.
    _DeleteNode( referenceNode );

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_InsertAfterNode(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE referenceNode,
    RFD_LLIST_USER_DATA_HANDLE data )
{
    RFD_LLIST_NODE_STRUCT * newNode;

    if( referenceNode == NULL )
    {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    // create a new node
    newNode = _CreateNode();

    if(newNode == NULL)
    {
        // error
        return RFD_STATUS_ERROR_MEM_ALLOC;
    }

    // Assign data to the new node.
    newNode->data = data;

    // update backward and forward links to nodes involved.
    newNode->prev = referenceNode;
    newNode->next = referenceNode->next;

    if(referenceNode->next != NULL) {
        referenceNode->next->prev = newNode;
    }
    referenceNode->next = newNode;

    // update tailNode to newNode if reference was current tailNode
    if( listHandle->tailNode == referenceNode )
    {
        listHandle->tailNode = newNode;
        listHandle->tailNode->next = NULL;
    }

    // update Browse info after inserting a node.
    // Ensure that the inserted node prev/next pointers have been
    // updated before call.
    _updateBrowseInfoOnInsertNode( listHandle, referenceNode );

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_InsertBeforeNode(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_NODE_HANDLE referenceNode,
    RFD_LLIST_USER_DATA_HANDLE data )
{
    RFD_LLIST_NODE_STRUCT * newNode;

    if( referenceNode == NULL )
    {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    // create a new node
    newNode = _CreateNode();

    if(newNode == NULL)
    {
        // error
        return RFD_STATUS_ERROR_MEM_ALLOC;
    }

    // Assign data to the new node.
    newNode->data = data;

    // update backward and forward links to nodes involved.
    newNode->next = referenceNode;
    newNode->prev = referenceNode->prev;

    if(referenceNode->prev != NULL) {
        referenceNode->prev->next = newNode;
    }
    referenceNode->prev = newNode;

    // update headNode to newNode if reference was current headNode
    if( listHandle->headNode == referenceNode )
    {
        listHandle->headNode = newNode;
        listHandle->headNode->prev = NULL;
    }

    // update Browse info after inserting a node.
    // Ensure that the inserted node prev/next pointers have been
    // updated before call.
    _updateBrowseInfoOnInsertNode( listHandle, referenceNode );

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
// See definition of RFD_LLIST_SORT_COMPARE_CALLBACK_FCN to see how the callback
// compare function determines how the list is sorted (from hi to lo or lo to hi).
RFD_STATUS RFD_LLIST_Sort(
    RFD_LLIST_HANDLE listHandle,
    RFD_LLIST_SORT_COMPARE_CALLBACK_FCN sortCompareCallbackFcn,
    void * sortCompareCallbackArg )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_STRUCT * currentNode, * maxNode;
    int cmpResult;
    size_t numNodes = 0, i;

    if(sortCompareCallbackFcn == NULL)
    {
        return RFD_STATUS_ERROR_INVALID_VALUE;
    }

    status = RFD_LLIST_GetNumNodes( listHandle, &numNodes );

    if(status != RFD_STATUS_OK) {
        return status;
    }

    if(numNodes <= 1) {
        // nothing to sort, only 0 or 1 nodes in list.
        return RFD_STATUS_OK;
    }

    // The total number of calls to the user compare callback is numNodes factorial (numNodes!).

    // On each iteration over the shrinking subset of the list,
    // we move the "maximum score" Node to the overall Tail of list.
    // Therefore, at end of all iterations, the the Head will have the maximum score Node,
    // and the tail will have the minimum score Node.

    // Outer loop
    while(numNodes-- != 0) {

        currentNode = listHandle->headNode;
        maxNode = currentNode;

        i = numNodes;

        // Inner loop
        while(i-- != 0) {

            currentNode = currentNode->next;

            cmpResult = sortCompareCallbackFcn(
                            currentNode->data,
                            maxNode->data,
                            sortCompareCallbackArg );

            if(cmpResult > 0) {
                maxNode = currentNode;
            }
        }

        if(maxNode != listHandle->tailNode) {

            // if maxNode is the head node, then before moving maxNode to the tail,
            // set head as maxNode->next.
            if(maxNode == listHandle->headNode) {
                listHandle->headNode = maxNode->next;
                // can postpone till end of sort: listHandle->headNode->prev = NULL;
            }

            // first remove maxNode from list (update the link around the maxNode being moved).

            if( maxNode->prev != NULL ) {
                maxNode->prev->next = maxNode->next;
            }

            if( maxNode->next != NULL ) {
                maxNode->next->prev = maxNode->prev;
            }

            // then insert maxNode at the tail
            listHandle->tailNode->next = maxNode;
            maxNode->prev = listHandle->tailNode;
            listHandle->tailNode = maxNode;
            // can postpone till end of sort: listHandle->tailNode->next = NULL;
        }
        // else maxNode is already the tail node, so don't try to move it.
    }

    // Note: leaving browse info the same after sort.
    // It sounds unusual for a use case for browsing after a sort
    // without a re-init first, but it's possible.

    // Ensure prev of Head and next of Tail are properly set to NULL.
    // Note: In postponing this update until the end of sorting, ensure that
    // prev of Head and next of Tail values are not used during the sort.
    listHandle->headNode->prev = NULL;
    listHandle->tailNode->next = NULL;

    return RFD_STATUS_OK;
}

//////////////////////////////////////////////////////////
// Find first or next (inner) node of a 2D List
//
// A 2D List consists of one Outer List where each node (actually each user data of each node)
// of the Outer List is a List (an Inner List).
// Allow general case where some Inner Lists can be empty (contain no nodes).

RFD_STATUS _2DFirstNext(
                RFD_LLIST_HANDLE hOuterList,
                RFD_LLIST_NODE_HANDLE * hFoundOuterListNodePtr,
                RFD_LLIST_NODE_HANDLE * hFoundInnerListNodePtr,
                BOOL isFindFirst )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hOuterListNode = NULL, hInnerListNode = NULL;
    RFD_LLIST_HANDLE hInnerList = NULL;
    BOOL isInnerListFindFirst = FALSE;

    // Initialize output - indicate no node found.
    *hFoundOuterListNodePtr = NULL;
    *hFoundInnerListNodePtr = NULL;

    do { // once

        if(isFindFirst) {
            status = RFD_LLIST_First(hOuterList, &hOuterListNode);
            isInnerListFindFirst = TRUE;
        }
        else { // Next
            // With _2DNext(), for outer list, pickup browse from same node as last time (i.e. the current browse node)
            // if this node still exists (i.e. was not removed during browsing).
            status = RFD_LLIST_CurrentBrowseNode(hOuterList, &hOuterListNode);

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            if(hOuterListNode == NULL) {
                // The current browse node is null. This can happen if the "previous" current browse node
                // was was Removed by LLIST user. In this case there
                // may still be nodes to be browsable by _Next(); call _Next() here.
                //
                // Otherwise, if the current browse node was null because all nodes were already browsed
                // then _Next() will return NULL node as desired.
                status = RFD_LLIST_Next(hOuterList, &hOuterListNode);

                // Anytime we change outer list nodes with _First() or _Next(), we
                // must restart browing the inner list with a browsw _First().
                isInnerListFindFirst = TRUE;
            }
            else {
                // Still on the same Outer List Node, containing the same inner list that
                // was browsed on the previous iteration (browse in progress, not the first step).
                // So continue with Find Next (not Find First) for the inner list.
                isInnerListFindFirst = FALSE;
            }
        }

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // iterate the nodes of the outer list looking for a inner list node.
        while(hOuterListNode != NULL) {

            // The user data of an outer list node is an inner list
            hInnerList = RFD_LLIST_GetUserData(hOuterListNode, &status);
            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            if(hInnerList != NULL) {

                if(isInnerListFindFirst) {
                    status = RFD_LLIST_First(hInnerList, &hInnerListNode);
                    isInnerListFindFirst = FALSE;
                }
                else {
                    status = RFD_LLIST_Next(hInnerList, &hInnerListNode);
                }

                if(status != RFD_STATUS_OK) {
                    // error
                    break;
                }

                if(hInnerListNode != NULL) {
                    // found an inner node
                    // update outputs then break from loop.

                    // inner node - node of inner list (an innermost item of overall 2D list)
                    *hFoundInnerListNodePtr = hInnerListNode;
                    // outer node (containing an inner list)
                    *hFoundOuterListNodePtr = hOuterListNode;

                    break;
                }
            }

            // Step to next node in Outer List.
            // Allways call RFD_LLIST_Next() here, regardless of isFindFirst = true/false.
            status = RFD_LLIST_Next(hOuterList, &hOuterListNode);

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            // Anytime we change outer list nodes with _First() or _Next(), we
            // must restart browing the inner list with a browsw _First().
            isInnerListFindFirst = TRUE;

        } // while hOuterListNode != NULL

    } while(FALSE);

    return status;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_2DFirst(
                RFD_LLIST_HANDLE hOuterList,
                RFD_LLIST_NODE_HANDLE * hFoundOuterListNodePtr,
                RFD_LLIST_NODE_HANDLE * hFoundInnerListNodePtr )
{

    BOOL isFindFirst;
    RFD_STATUS status;

    // Find First
    isFindFirst = TRUE;

    status = _2DFirstNext(
                    hOuterList,
                    hFoundOuterListNodePtr,
                    hFoundInnerListNodePtr,
                    isFindFirst );

    return status;
}

//////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_2DNext(
                RFD_LLIST_HANDLE hOuterList,
                RFD_LLIST_NODE_HANDLE * hFoundOuterListNodePtr,
                RFD_LLIST_NODE_HANDLE * hFoundInnerListNodePtr )
{

    BOOL isFindFirst;
    RFD_STATUS status;

    // Find Next (First is false)
    isFindFirst = FALSE;

    status = _2DFirstNext(
                    hOuterList,
                    hFoundOuterListNodePtr,
                    hFoundInnerListNodePtr,
                    isFindFirst );

    return status;
}

/////////////////////////////////////////////////////////
RFD_STATUS RFD_LLIST_2DGetNumNodes(
                RFD_LLIST_HANDLE hOuterList,
                size_t * numNodesPtr )
{
    RFD_STATUS status = RFD_STATUS_OK;
    RFD_LLIST_NODE_HANDLE hOuterListNode = NULL;
    RFD_LLIST_HANDLE hInnerList = NULL;
    size_t numNodesTotal = 0, numNodesSingleInnerList = 0;

    // Initialize output
    *numNodesPtr = 0;

    do { // once

        status = RFD_LLIST_First(hOuterList, &hOuterListNode);

        if(status != RFD_STATUS_OK) {
            // error
            break;
        }

        // iterate the nodes of the outer list looking for a inner list node.
        while(hOuterListNode != NULL) {

            // The user data of an outer list node is an inner list
            hInnerList = RFD_LLIST_GetUserData(hOuterListNode, &status);
            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

            if(hInnerList != NULL) {

                status = RFD_LLIST_GetNumNodes(hInnerList, &numNodesSingleInnerList);

                if(status != RFD_STATUS_OK) {
                    // error
                    break;
                }

                numNodesTotal += numNodesSingleInnerList;
            }

            // Step to next node in Outer List.
            status = RFD_LLIST_Next(hOuterList, &hOuterListNode);

            if(status != RFD_STATUS_OK) {
                // error
                break;
            }

        } // while hOuterListNode != NULL

    } while(FALSE);

    // update output
    *numNodesPtr = numNodesTotal;

    return status;
}
