////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_gz.c
///
/// @brief	rfd gz class.
///
/// @remarks	This module is an adaptation of minigzip.c from the ZLIB library.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
// The following Comments are copied from the original minigzip.c source file
// on which this file is based.

/* minigzip.c -- simulate gzip using the zlib compression library
 * Copyright (C) 1995-2005 Jean-loup Gailly.
 * For conditions of distribution and use, see copyright notice in zlib.h
 */

/*
 * minigzip is a minimal implementation of the gzip utility. This is
 * only an example of using zlib and isn't meant to replace the
 * full-featured gzip. No attempt is made to deal with file systems
 * limiting names to 14 or 8+3 characters, etc... Error checking is
 * very limited. So use minigzip only for testing; use gzip for the
 * real thing. On MSDOS, use only on file names without extension
 * or in pipe mode.
 */

/* @(#) $Id$ */
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"
#include "rfd_gzio.h"
#include <stdio.h>
#include "zlib.h"

#define RFD_GZ_IO_BUFLEN      16384

#define MAX_NAME_LEN 1024

#ifdef MAXSEG_64K
#  define local static
   /* Needed for systems with limitation on stack size. */
#else
#  define local
#endif

char *prog;

/* ===========================================================================
 * Compress input to output.
 */

RFD_STATUS rfd_gz_compress(in, out)
    RFD_FILE   *in;
    gzFile out;
{
	UCHAR * buf;
	int buf_size;
    int len;
	RFD_STATUS status;

	// Allocate Buffer
	buf_size = RFD_GZ_IO_BUFLEN * sizeof(UCHAR);

	buf = (UCHAR *) RFD_MALLOC(buf_size);
	if(buf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

    for (;;) {
        len = (int)RFD_FREAD(buf, 1, sizeof(buf), in);
        if (RFD_FERROR(in)) {
			RFD_DPrint(TEXT("Error, during file read\n"));
			status = RFD_STATUS_ERROR_FILE_READ;
			break;
        }
		if (len == 0) {
			status = RFD_STATUS_OK;
			break;
		}

		if (rfd_gzwrite(out, buf, (unsigned)len) != len) {
			RFD_DPrint(TEXT("Error, during rfd_gzwrite()\n"));
			status = RFD_STATUS_ERROR_GZ_WRITE;
			break;
		}
    }

	// Free Buffer
	RFD_FREE(buf);

	return status;
}
/* ===========================================================================
 * Uncompress input to output.
 */
RFD_STATUS rfd_gz_uncompress(in, out, exitRequestFlagPtr)
    gzFile in;
    RFD_FILE   *out;
	UCHAR * exitRequestFlagPtr;
{
	UCHAR * buf;
	int buf_size;
    int len;
	RFD_STATUS status;

	// Allocate Buffer
	buf_size = RFD_GZ_IO_BUFLEN * sizeof(UCHAR);

	buf = (UCHAR *) RFD_MALLOC(buf_size);
	if(buf == NULL) {
		return RFD_STATUS_ERROR_MEM_ALLOC;
	}

    for (;;) {
        len = rfd_gzread(in, buf, buf_size);
		if (len < 0) {
			RFD_DPrint(TEXT("Error, during rfd_gzread()\n"));
			status = RFD_STATUS_ERROR_GZ_READ;
			break;
		}

		if (len == 0) {
			status = RFD_STATUS_OK;
			break;
		}

        if ((int)RFD_FWRITE(buf, 1, (unsigned)len, out) != len) {
			status = RFD_STATUS_ERROR_FILE_WRITE;
			break;
        }

		// Check for exit request on each loop.
		if(*exitRequestFlagPtr) {
			status = RFD_STATUS_EARLY_EXIT_ON_REQUEST;
			break;
		}
    }

	// Free Buffer
	RFD_FREE(buf);

	return status;
}

/* ===========================================================================
 * Compress the given file
 */
RFD_STATUS rfd_gz_file_compress(const TCHAR * outfile, const TCHAR * infile,
								int compression_level,
								RFD_GZ_COMPRESSION_STRATEGY compression_strategy)
{
    RFD_FILE  *in;
    gzFile out;
	TCHAR outmode[GZ_OPEN_MAX_MODE_STRING_LEN];
	TCHAR strategy_string[2];
	TCHAR level_string[2];
	int strategy_string_len = 2;
	int level_string_len = 2;
	RFD_STATUS status;

	// Construct the compression strategy string
	switch(compression_strategy) {

		case RFD_GZ_COMPRESSION_STRATEGY_Z_DEFAULT:
			// empty string for default
			RFD_SPRINTF_S(strategy_string, strategy_string_len, TEXT(""));
			break;

		case RFD_GZ_COMPRESSION_STRATEGY_Z_FILTERED:
			RFD_SPRINTF_S(strategy_string, strategy_string_len, TEXT("f"));
			break;

		case RFD_GZ_COMPRESSION_STRATEGY_Z_HUFFMAN_ONLY:
			RFD_SPRINTF_S(strategy_string, strategy_string_len, TEXT("h"));
			break;

		case RFD_GZ_COMPRESSION_STRATEGY_Z_RLE:
			RFD_SPRINTF_S(strategy_string, strategy_string_len, TEXT("R"));
			break;

		default:
			// error, invalid strategy
		 return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
	}

	// Construct the compression level string

	if(compression_level >= RFD_GZ_MIN_COMPRESSION_LEVEL &&
		compression_level <= RFD_GZ_MAX_COMPRESSION_LEVEL) {

		RFD_SPRINTF_S(level_string, level_string_len, TEXT("%d"), compression_level);
	}
	else if(compression_level == RFD_GZ_DEFAULT_COMPRESSION_LEVEL_FLAG) {
			// empty string for default
			RFD_SPRINTF_S(level_string, level_string_len, TEXT(""));
	}
	else {
		// error, invalid level
		 return RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE;
	}

	// Construct the overall mode string = "wb" + level_string + strategy_string
	// e.g. wb6f
	// -> wb indicates compression request (gz opens dest file with "wb" mode)
	//    6 indicates level 6
	//    f indicates strategy Z_FILTERED
	RFD_SPRINTF_S(outmode, GZ_OPEN_MAX_MODE_STRING_LEN, TEXT("%s%s%s"),
				  TEXT("wb"),
				  level_string,
				  strategy_string );

	// File Open of the source file
	if(RFD_FOPEN_S(&in, infile, TEXT("rb"))) {
		RFD_DPrint(TEXT("Error, can't open %s\n"), infile);
		return RFD_STATUS_ERROR_FILE_OPEN;
    }

	// GZ Open (also opens the destination file)
	out = rfd_gzopen(outfile, outmode);
    if (out == NULL) {
		RFD_DPrint(TEXT("Error, can't open %s\n"), outfile);
		RFD_FCLOSE(in);
		return RFD_STATUS_ERROR_GZ_OPEN_OUTPUT;
    }

	// Decompress Operation
    status = rfd_gz_compress(in, out);

	// Close the source File.
	RFD_FCLOSE(in);

	// Close GZ (and the destination file)
	if(rfd_gzclose(out) != Z_OK) {
		RFD_DPrint(TEXT("Error, failed rfd_gzclose\n"));
		status = RFD_STATUS_ERROR_GZ_CLOSE;
	}

	return status;
}

/* ===========================================================================
 * Uncompress the input file into the specified output file.
 */

RFD_STATUS rfd_gz_file_uncompress(const TCHAR * outfile, const TCHAR * infile, UCHAR * exitRequestFlagPtr, BOOL doFileSync)
{
    RFD_FILE  *out;
    gzFile in;
	RFD_STATUS status;

	if(infile == NULL || outfile == NULL || exitRequestFlagPtr == NULL) {
		return RFD_STATUS_ERROR_INVALID_VALUE;
	}

	// GZ Open (also opens the input file)
    in = rfd_gzopen(infile, TEXT("rb"));
    if (in == NULL) {
		RFD_DPrint(TEXT("Error, can't rfd_gzopen %s\n"), infile);
		return(RFD_STATUS_ERROR_GZ_OPEN_INPUT);
    }

	// File Open of the destination file
	if(RFD_FOPEN_S(&out, outfile, TEXT("wb")) || out == NULL) {
		RFD_DPrint(TEXT("Error, can't open %s\n"), outfile);
		rfd_gzclose(in);
		return(RFD_STATUS_ERROR_FILE_OPEN);
    }

	// Decompress Operation
    status = rfd_gz_uncompress(in, out, exitRequestFlagPtr);

	if(doFileSync) {
		// Must fflush upper layer buffers before file sync.
		RFD_FFLUSH(out);
		// Do File Sync
		RFD_FileSyncByHandle(out);
	}

	// Close the destination File.
	RFD_FCLOSE(out);

	// Note: for status == RFD_STATUS_EARLY_EXIT_ON_REQUEST,
	// the calling function is responsible for deleting the unfinished output file.

	// Close GZ (and the source file)
	if(rfd_gzclose(in) != Z_OK) {
		RFD_DPrint(TEXT("Error, failed rfd_gzclose()\n"));
		status = RFD_STATUS_ERROR_GZ_CLOSE;
	}

	return status;
}
