////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_codec_xcc.h
///
/// @brief	Declares the rfd codec extended-ecc-code class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef RFD_CODEC_XCC_H
#define RFD_CODEC_XCC_H

#include "rfd_config.h"
#include "rfd.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/////////////////////////////////////////////////////

typedef enum {
	XCC_ROW_INFO_SOURCE_TYPE_BLOCK = 0,
	XCC_ROW_INFO_CHECK_TYPE_BLOCK = 1
} XCC_ROW_INFO_BLOCK_TYPE_ENUM;


#define XCC_ROW_INFO_IS_PIVOT_TRUE 1
#define XCC_ROW_INFO_IS_PIVOT_FALSE	0
#define XCC_ROW_INFO_INDEX_SHIFT 1
#define XCC_ROW_INFO_IS_PIVOT_MASK 0x0001

///////////////////////////////////////////////////////////////////////////////
// rowInfo[] format //
//////////////////////
// Each rowInfo[] array entry (rowInfo[i]) provides information about
// a corresponding equation row number (the row indexed by i in rowInfo[i]).
//
// The information provided:
// 1. The column element up to which the row has been progressively cleared
//    (all elements except for the pivot position element are cleared,
//     if that row implements a pivot).
// 2. An indication that the row implements a Pivot or does not.
//
// The format is:
//    Bits[15:1] = progressionNo = number of column elements positions
//                 that have been cleared for this row.
//    Bit[0]     = isPivot = 1 if row implements a Pivot, else 0.
//
// A Source-type row always implements a Pivot (isPivot = 1)
// and has the progressionNo = dimInfo->nEquVars.
//
// A Check-type row that has been fully cleared and does not implement
// a pivot (is not linearly independent) will be identified with:
// isPivot = 0, and progressionNo = dimInfo->nEquVars.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// Set RowInfo to indicate that this equation row has been 'cleared'
// up to columnNo number of columns, and set the pivot implemented by this
// row status based on isPivot TRUE/FALSE.
// 'Cleared' means that all elements of the row are zero up to
// the columnNo column (columns 0 to columNo-1 are zero)
// excluding the Pivot column if this row implements a Pivot.
///////////////////////////////////////////////////////////////////////////////
#define XCC_DEC_SET_ROW_INFO(rowInfo, columnNo, isPivot) \
			rowInfo = (columnNo << XCC_ROW_INFO_INDEX_SHIFT) | isPivot

///////////////////////////////////////////////////////////////////////////////
// Set RowInfo to indicate that this equation row has been 'cleared'
// up to columnNo number of columns. This macro version preserves the existing
// isPivot state.
///////////////////////////////////////////////////////////////////////////////
#define XCC_DEC_SET_ROW_INFO_COLUMN_NUM_ONLY(rowInfo, columnNo) \
			rowInfo = (columnNo << XCC_ROW_INFO_INDEX_SHIFT) | \
					  (rowInfo & XCC_ROW_INFO_IS_PIVOT_MASK)

///////////////////////////////////////////////////////////////////////////////
// Set RowInfo to indicate that this equation row implements a Pivot
///////////////////////////////////////////////////////////////////////////////
#define XCC_DEC_SET_ROW_INFO_PIVOT_TRUE(rowInfo) \
			rowInfo |= XCC_ROW_INFO_IS_PIVOT_MASK

///////////////////////////////////////////////////////////////////////////////
// Set RowInfo to indicate that this equation row does Not implement a Pivot
///////////////////////////////////////////////////////////////////////////////
#define XCC_DEC_SET_ROW_INFO_PIVOT_FALSE(rowInfo) \
			rowInfo &= (~XCC_ROW_INFO_IS_PIVOT_MASK)

///////////////////////////////////////////////////////////////////////////////
// Get the next column/position to which the row has been 'cleared'.
// 'Cleared' means that all elements of the row are zero up to
// that position, excluding the Pivot column if this row implements a Pivot.
///////////////////////////////////////////////////////////////////////////////
#define XCC_DEC_ROW_INFO_GET_NEXT_POSITION(rowInfo) \
			(rowInfo >> XCC_ROW_INFO_INDEX_SHIFT)

///////////////////////////////////////////////////////////////////////////////
// Get the 'Is Pivot' status. Pivot status is true if this row
// implements a pivot postion, else false.
///////////////////////////////////////////////////////////////////////////////
#define XCC_DEC_ROW_INFO_IS_PIVOT_ROW(rowInfo) \
			(rowInfo & XCC_ROW_INFO_IS_PIVOT_MASK)



///////////////////////////////////////////////////////////////////////////////
// pivotInfo[] format //
////////////////////////
// Each pivotInfo[] array entry (pivotInfo[i]) provides information about
// a corresponding pivot position (the pivot indexed by i in pivotInfo[i]).
// There are dimInfo->nEquVars number of pivotInfo[] entries
// (pivot postions).
//
// The information provided:
// 1. If a pivot postion, i, is not yet implemented by any equation rows,
//    then pivotInfo[i] = -1.
// 2. If a pivot postion, i, is implemented by an equation rows, j,
//    then pivotInfo[i] = j.
//
// The format is:
//    Bits[15:0] =  -1 If a the pivot postion is not yet implemented
//                  by any equation rows
//               >= 0 = j If the pivot postion is implemented by
//                  equation row index j.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// Set PivotInfo to indicate the Block at row 'rowNo' implements
// this pivot position.
///////////////////////////////////////////////////////////////////////////////
#define XCC_DEC_SET_PIVOT_INFO_PIVOT_TRUE(pivotInfo, rowNo) \
			pivotInfo = rowNo

///////////////////////////////////////////////////////////////////////////////
// Set PivotInfo to indicate there is no Block
// currently implemented for this Pivot Position.
///////////////////////////////////////////////////////////////////////////////
#define XCC_DEC_SET_PIVOT_INFO_PIVOT_FALSE(pivotInfo) \
			pivotInfo = (-1)

///////////////////////////////////////////////////////////////////////////////
// Get the row that implements the pivot postion fron the PivotInfo
///////////////////////////////////////////////////////////////////////////////
#define XCC_DEC_PIVOT_INFO_GET_ROW(pivotInfo) pivotInfo

///////////////////////////////////////////////////////////////////////////////
// get True/False indication as to if this pivot postion has an
// implemented pivot row.
///////////////////////////////////////////////////////////////////////////////
#define XCC_DEC_PIVOT_INFO_IS_PIVOT(pivotInfo) \
			((pivotInfo >= 0) ? 1:0)

/////////////////////////////////////////////////////

typedef struct
{
	const TCHAR * srcBlksInbufFileName;
	const TCHAR * chkBlksInbufFileName;
	const TCHAR * transformationInfoFileName;
	const TCHAR * coeffTransformFileName;	// Coefficient Transformation Matrix file name
	const TCHAR * blockTransformFileName;	// Block Transformation Matrix file name
	const TCHAR * decodeOutputFileName;		// final decoded output file name

} RFD_DECODE_XCC_FILE_NAME_LIST;

typedef struct
{
	UINT16 blkLenBytes;
	UINT32 fileLenBytes;
	UINT16 fileLenBlocks;	//UINT16 equLenElements;

} RFD_XCC_CODEC_CONFIG_INFO;

typedef struct
{
	/* the collector thread signals the processor thread if the total number of src and enc blocks
	 * collected is greater than or equal to this threshold. The src blocks included in this count
	 * are only src blocks corresponding to pivot positions less than this threshold.
	 */
	UINT16 blksCollectedEventThresh;

	/* The decoder increments this count each time a "chunk" (set of blocks) is decoded.
	 * This count is uses as an index into decodeBlockThresholdInfo->blockThresholdNumeratorArray
	 */
	UINT16 decodeChunkCount;

	/* saves the latest number of src type blocks collected in the file buffer */
	UINT16 nSrcBlksInBuf;

	/* saves the latest number of check type blocks collected in the file buffer */
	UINT16 nChkBlksInBuf;

	/* nSrcBlksCollectedTotal:
	 * saves the total number of src type blocks that have been collected.
	 * This should match the number of elements that are set (TRUE)
	 * in the srcBlksCollectedStatus array
	 */
	UINT16 nSrcBlksCollectedTotal;

	/* nChkBlksCollectedTotal:
	 * saves the total number of enc type blocks that have been collected.
	 * This should match the number of entries into the encBlksCollectedList.
	 */
	UINT16 nChkBlksCollectedTotal;

	/* nSrcBlocksInSolution:
	 * saves the total number of source type blocks that have been used as part of
	 * the decoder solution. This is a decoder statistics type variable.
	 * nSrcBlocksInSolution could differ from nSrcBlksCollectedTotal
	 * if the processor/decoder thread completes decoding while the collector thread has
	 * received additional source blocks, which were therefore not part of the solution,
	 * but that are reflected in nSrcBlksCollectedTotal.
	 */
	UINT16 nSrcBlocksInSolution;

	/* nChkBlksEliminated
	 * Number of encoded (check) blocks that the decoder has removed from the system
	 * due to linear dependence.
	 */
	UINT16 nChkBlksEliminated;

	/* nTransformRows
	 * Number of blocks that the decoder has in the transformation matrix
	 * (the incremental solution matrix). These are blocks that are moved (and
	 * transformed) from the collection buffers (nSrcBlksInBuf and
	 * nChkBlksInBuf) and into the solution minus blocks that have been
	 * removed due to linear dependence.
	 */
	UINT16 nTransformRows;

	UINT16 nextPivotNo;

} RFD_XCC_DECODE_STATE_INFO;


typedef INT16	RFDR_ECC_PIVOT_INDEX;
typedef UCHAR	RFDR_XCC_SRC_BLKS_COLLECTED_STATUS;

typedef UINT32	RFD_XCC_ENC_BLK_INDEX;
typedef UINT16	RFD_XCC_SRC_BLK_INDEX;

///////////////////////////////////////////////////////////////////////////////
typedef struct
{
	/* if >= 0 this row is the pivot index corresponding to the value.
	 * if < 0, it's not yet a pivot index and the status of the row is given by the value
	 * -1 = reduced to the current pivot level, -2 = not yet reduced (e.g. just collected).
	 */
	INT16 pivotIndex;

	/* src block = true, enc block = false
	 * note: may not need this if source-type blocks are inserted as pivots
	 * as soon as they are read from the collection buffer
	 * (but still might be useful to avoid checking element values if it is known to be a src block)
	 */
	UCHAR srcBlockStatus;

} RFDR_SOLUTION_ROW_INFO;

///////////////////////////////////////////////////////////////////////////////
typedef struct {
	DWORD* coeffBuf;
	DWORD* blockBuf;
	INT16* rowInfo;
	INT16* pivotInfo;
	void * rowScaleInfo;
} XCC_DEC_DYNAMIC_BUFS;

///////////////////////////////////////////////////////////////////////////////
// A super Block is the 'main' block size of the RFD transported file.
// The decoder splits the superBlock into sub-blocks, simply called blocks,
// Each block of the superBlock is decoded separately to limit memory usage.
// After each block is decoded, the decoder re-combines each processed block.
typedef struct {

	UINT16 superBlkLenBytes;
	UINT16 superBlkLenDWords;		// not sure if this is needed.
	UINT16 lastSuperBlkLenBytes;	// length of the last 'non-zero valued' superBlock of the solution.
	UINT16 blkOffsetBytes;			// offset of the current block into the superBlock.
	BOOL   isFirstBlock;			// is first block of the superBlock.
	BOOL   isLastBlock;				// is last block of the superBlock.
	UINT16 nBlocks;					// number of blocks in the superBlock.
	UINT32 maxBlkLenBytes;		// maximum size of a block within the superBlock to comply with decoder memory usage limits.

} XCC_DEC_SUPER_BLOCK_INFO;

///////////////////////////////////////////////////////////////////////////////
typedef struct {
	UINT16 nEquVars;			// equLenElements
	UINT16 coeffRowLenDWords;	// equLenDWords
	UINT32 blkLenBytes;
	UINT32 blkLenDWords;		// blkLenDWords, rounded-up length.
	UINT16 nRowsTotal;			// nRows , total of existing transformed sys of equations and new src and enc rows/blocks.
	XCC_DEC_SUPER_BLOCK_INFO superBlkInfo;

} XCC_DEC_DIMENSION_INFO;


///////////////////////////////////////////////////////////////////////////////////////////////
typedef void (*FcnPtrConstructSrcTypeCoeffRow)(DWORD * coeffRow, UINT16 index, UINT16 coeffRowLenDWords);
typedef void (*FcnPtrConstructChkTypeCoeffRow)(DWORD * coeffRow, RFD_XCC_ENC_BLK_INDEX index, UINT16 nCoeffs);

///////////////////////////////////////////////////////////////////////////////////////////////
extern const unsigned char rfdEcc1Coeffs[];
extern const unsigned char rfdEcc2Coeffs[];
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_DecodeXcc1Code(const TCHAR pathBuf[],
							  const RFD_DECODE_XCC_FILE_NAME_LIST * decodeXccFileNameList,
							  RFD_XCC_CODEC_CONFIG_INFO * codecConfigInfo,
							  RFD_XCC_DECODE_STATE_INFO * decoderStateInfo,
							  FcnPtrConstructSrcTypeCoeffRow ConstructSrcTypeCoeffRowFunction,
							  FcnPtrConstructChkTypeCoeffRow ConstructChkTypeCoeffRowFunction,
							  UINT32 maxCheckBlockIndex,
							  UINT16 maxExtraBlocksForSolution,
							  UCHAR * exitRequestFlagPtr);

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_DecodeXcc2Code(const TCHAR pathBuf[],
							  const RFD_DECODE_XCC_FILE_NAME_LIST * decodeXccFileNameList,
							  RFD_XCC_CODEC_CONFIG_INFO * codecConfigInfo,
							  RFD_XCC_DECODE_STATE_INFO * decoderStateInfo,
							  FcnPtrConstructSrcTypeCoeffRow ConstructSrcTypeCoeffRowFunction,
							  FcnPtrConstructChkTypeCoeffRow ConstructChkTypeCoeffRowFunction,
							  UINT32 maxCheckBlockIndex,
							  UINT16 maxExtraBlocksForSolution,
							  UCHAR * exitRequestFlagPtr );

///////////////////////////////////////////////////////////////////////////////////////////////
void RFD_ConstructSrcTypeCoeffRowEcc1(DWORD * coeffRow, UINT16 index, UINT16 coeffRowLenDWords);
///////////////////////////////////////////////////////////////////////////////////////////////
void RFD_ConstructChkTypeCoeffRowEcc1(DWORD * coeffRow, RFD_XCC_ENC_BLK_INDEX index, UINT16 nCoeffs);
///////////////////////////////////////////////////////////////////////////////////////////////
void RFD_ConstructSrcTypeCoeffRowEcc2(DWORD * coeffRow, UINT16 index, UINT16 coeffRowLenDWords);
///////////////////////////////////////////////////////////////////////////////////////////////
void RFD_ConstructChkTypeCoeffRowEcc2(DWORD * coeffRow, RFD_XCC_ENC_BLK_INDEX index, UINT16 nCoeffs);

#ifdef __cplusplus
}
#endif

#endif // RFD_RECEIVER_H