////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_codec_utils_xcc1.c
///
/// @brief	rfd codec utils extended-ecc1-code class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd_codec_xcc.h"
#include "rfd_gf4_tables.h"
#include "rfd.h"

///////////////////////////////////////////////////////////////////////////////////////////////
#ifndef XCC_BITS_PER_ELEMENT
	#define XCC_BITS_PER_ELEMENT 1
	#define RFD_XCC1_CODE
	#define RFD_CONSTRUCT_SRC_TYPE_COEFF_ROW   RFD_ConstructSrcTypeCoeffRowEcc1
	#define RFD_CONSTRUCT_CHK_TYPE_COEFF_ROW   RFD_ConstructChkTypeCoeffRowEcc1
	#define RFD_XCC_COEFF_LUT rfdEcc1Coeffs
#endif

// XCC_IDENTITY_ELEM_DWORD_ALIGNED
//   e.g. for ecc1 = 0x80000000
//   e.g. for ecc2 = 0x40000000
#define XCC_IDENTITY_ELEM_DWORD_ALIGNED (((DWORD) 1) << (RFD_BITS_PER_DWORD - XCC_BITS_PER_ELEMENT))

///////////////////////////////////////////////////////////////////////////////
void RFD_CONSTRUCT_SRC_TYPE_COEFF_ROW(DWORD * coeffRow, UINT16 index, UINT16 coeffRowLenDWords)
{
	int blkIndexDWordOffset;
	int blkIndexBitOffset;

	// clear the coeff row
	RFD_MEMSET( coeffRow, 0, coeffRowLenDWords * sizeof(DWORD));

	blkIndexDWordOffset = (index * XCC_BITS_PER_ELEMENT) / RFD_BITS_PER_DWORD;
	blkIndexBitOffset =   (index * XCC_BITS_PER_ELEMENT) % RFD_BITS_PER_DWORD;

	if(blkIndexDWordOffset < coeffRowLenDWords) {
		// Note, this clears all other elements in this DWORD to zero.
		coeffRow[blkIndexDWordOffset] = XCC_IDENTITY_ELEM_DWORD_ALIGNED >> blkIndexBitOffset;
	}
	// else unexpected error (bad index parameter passed in).
	// The caller is responsible for this type of error
	// (probably shouldn't even do this error check here).

	return;
}

#if 1
// This version of RFD_CONSTRUCT_CHK_TYPE_COEFF_ROW() uses DWORD data sized access
// to the coefficient table and therefore requires the table to be DWORD
// aligned.
// On some processors (ARM) data address alignement must match instruction access size.
// DWORD acces should be faster that byte access.

///////////////////////////////////////////////////////////////////////////////
void RFD_CONSTRUCT_CHK_TYPE_COEFF_ROW(DWORD * coeffRow, RFD_XCC_ENC_BLK_INDEX index, UINT16 nCoeffs)
{
	INT32 indexDWordOffset;
	int indexBitOffset, indexBitOffsetInverse;
	const DWORD * pSrc;
	DWORD * pDst;
	DWORD coeff1, coeff2;
	UINT16 i, nDWords;

	// Note: This function requires the coefficient lookup table,
	// RFD_XCC_COEFF_LUT, be DWord aligned.
	// Therefore DWORD sized reads are used to access it.

	nDWords = ((nCoeffs * XCC_BITS_PER_ELEMENT) / RFD_BITS_PER_DWORD) +
		      (((nCoeffs * XCC_BITS_PER_ELEMENT) % RFD_BITS_PER_DWORD) ? 1:0);

	// Calculate byte and bit offsets into the coefficient matrix
	// from which the coefficients are read.
	indexDWordOffset = (index * XCC_BITS_PER_ELEMENT) / RFD_BITS_PER_DWORD;
	indexBitOffset =   (index * XCC_BITS_PER_ELEMENT) % RFD_BITS_PER_DWORD;
	indexBitOffsetInverse = RFD_BITS_PER_DWORD - indexBitOffset;

	// Initialize pointers to destination and src (lookup table).
	pDst = coeffRow;
	pSrc = (DWORD *) RFD_XCC_COEFF_LUT + indexDWordOffset;

	// copy coefficients from the lookup table to the destination row.
	if(indexBitOffset != 0) {
		// LUT start element is not byte aligned.
		coeff1 = *pSrc++;
#ifdef RFD_PLATFORM_LITTLE_ENDIAN
		coeff1 = (coeff1 >> 24) +
				((coeff1 >>  8) & 0x0000ff00) +
				((coeff1 <<  8) & 0x00ff0000) +
				 (coeff1 << 24);
#endif
		for(i=0;i<nDWords;i++) {
			coeff2 = *pSrc++;
#ifdef RFD_PLATFORM_LITTLE_ENDIAN
			coeff2 = (coeff2 >> 24) +
					((coeff2 >>  8) & 0x0000ff00) +
					((coeff2 <<  8) & 0x00ff0000) +
					 (coeff2 << 24);
#endif
			*pDst++ = (coeff1 << indexBitOffset) + (coeff2 >> indexBitOffsetInverse);
			coeff1 = coeff2;
		}
	}
	else {
		// LUT start element is byte aligned,
		// do simpler/faster copy without bit shifting.
		for(i=0;i<nDWords;i++) {
#ifdef RFD_PLATFORM_LITTLE_ENDIAN
			coeff1 = *pSrc++;
			coeff1 = (coeff1 >> 24) +
					((coeff1 >>  8) & 0x0000ff00) +
					((coeff1 <<  8) & 0x00ff0000) +
					 (coeff1 << 24);
			*pDst++ = coeff1;
#else
			*pDst++ = *pSrc++;
#endif
		}
	}

	return;
}

#else
// This version of RFD_CONSTRUCT_CHK_TYPE_COEFF_ROW() uses Byte data sized access
// to the coefficient table and therefore does not need the table to be DWORD
// aligned (i.e. alignment constraints on some processor platforms e.g. ARM).
///////////////////////////////////////////////////////////////////////////////
void RFD_CONSTRUCT_CHK_TYPE_COEFF_ROW(DWORD * coeffRow, RFD_XCC_ENC_BLK_INDEX index, UINT16 nCoeffs)
{
	INT32 indexByteOffset;
	int indexBitOffset, indexBitOffsetInverse;
	const UCHAR * pSrc;
	UCHAR * pDst;
	UCHAR coeff1, coeff2;
	UINT16 i, nBytes, nDWords;
	DWORD dWordCoeff;
	DWORD * pDWordCoeff;

	// Note: This function assumes the coefficient lookup table,
	// RFD_XCC_COEFF_LUT, is not guaranteed to be DWord aligned.
	// Therefore only byte sized reads are used to access it.

	nBytes = ((nCoeffs * XCC_BITS_PER_ELEMENT) / RFD_BITS_PER_BYTE) +
		     (((nCoeffs * XCC_BITS_PER_ELEMENT) % RFD_BITS_PER_BYTE) ? 1:0);

	nDWords = (nBytes / sizeof(DWORD)) +
		      ((nBytes % sizeof(DWORD)) ? 1:0);

	// Clear the last coeff row DWord incase row length in bytes
	// is not a multiple of DWords (just so the remaining unused bytes
	// have consistent content)
	coeffRow[nDWords-1] = 0;

	// Calculate byte and bit offsets into the coefficient matrix
	// from which the coefficients are read.
	indexByteOffset = (index * XCC_BITS_PER_ELEMENT) / RFD_BITS_PER_BYTE;
	indexBitOffset =   (index * XCC_BITS_PER_ELEMENT) % RFD_BITS_PER_BYTE;
	indexBitOffsetInverse = RFD_BITS_PER_BYTE - indexBitOffset;

	// Initialize pointers to destination and src (lookup table).
	pDst = (UCHAR *) coeffRow;
	pSrc = RFD_XCC_COEFF_LUT + indexByteOffset;

	// copy coefficients from the lookup table to the destination row.
	if(indexBitOffset != 0) {
		// LUT start element is not byte aligned.
		coeff1 = *pSrc++;
		for(i=0;i<nBytes;i++) {
			coeff2 = *pSrc++;
			*pDst++ = (coeff1 << indexBitOffset) + (coeff2 >> indexBitOffsetInverse);
			coeff1 = coeff2;
		}
	}
	else {
		// LUT start element is byte aligned,
		// do simpler/faster copy without bit shifting.
		for(i=0;i<nBytes;i++) {
			*pDst++ = *pSrc++;
		}
	}

#ifdef RFD_PLATFORM_LITTLE_ENDIAN
	pDWordCoeff = coeffRow;
	for(i=0;i<nDWords;i++) {
		dWordCoeff = *pDWordCoeff;
		*pDWordCoeff++ =  (dWordCoeff >> 24) +
						 ((dWordCoeff >>  8) & 0x0000ff00) +
						 ((dWordCoeff <<  8) & 0x00ff0000) +
						  (dWordCoeff << 24);
	}
#endif

	return;
}
#endif

