////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd_baudot.c
///
/// @brief	rfd baudot class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"
#include "rfd_baudot.h"
#include "rfd_bitstream_io.h"

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ConvertBaudotBitStreamToString( TCHAR outStringBuf[], int outStringBufMaxNumChars,
									    RFD_BITSTREAM_INFO * streamInfo, int baudotStringBufMaxNumSymbols )
{
	BAUDOT_COL_NUM_ENUM currColumnNo;
	BOOL isEscapeMode;
	BOOL isLineFeedInsertRequested, foundBaudotEndCode;
	UINT32 dWord;
	TCHAR * outStringPtr;
	TCHAR * previousOutStringPtr;
	int inputBaudotSymbolCount, outputCharCount;

	currColumnNo = BAUDOT_COL_NUM_LTR_UC;
	isEscapeMode = FALSE;
	isLineFeedInsertRequested = FALSE;
	foundBaudotEndCode = FALSE;
	outputCharCount = 0;
	inputBaudotSymbolCount = 0;
	outStringPtr = outStringBuf;
	previousOutStringPtr = outStringBuf;

	while(!foundBaudotEndCode) {

		// Check for output string overflow.
		// A character is not necessarily written on each loop,
		// so compare pointer with previous before incrementing the count.
		if(outStringPtr != previousOutStringPtr) {
			previousOutStringPtr = outStringPtr;
			if(outputCharCount < outStringBufMaxNumChars) {
				outputCharCount++;
			}
			else {
				return RFD_STATUS_ERROR_OUTPUT_BUFFER_OVERFLOW;
			}
		}

		// Check for Line Feed request.
		if(isLineFeedInsertRequested) {
			// signal to add line feed following carriage return.
			isLineFeedInsertRequested = FALSE;
			*outStringPtr++ = LINE_FEED;
			// Execute 'continue' so that the output string overflow check
			// will be done on each character write.
			continue;
		}

		// Check for input number of symbols limit (do this after the line feed request
		// that can do a continue)
		if(inputBaudotSymbolCount < baudotStringBufMaxNumSymbols) {
			inputBaudotSymbolCount++;
		}
		else {
			return RFD_STATUS_ERROR_INPUT_BUFFER_OVERFLOW;
		}

		// Read the next Baudot symbol
		if(RFD_BitstreamGetU32(streamInfo, RFD_BAUDOT_SYMBOL_NUM_BITS, &dWord)) {
			return RFD_STATUS_ERROR_BITSTREAM_READ_ERROR;
		}

		if(isEscapeMode) {
			isEscapeMode = FALSE;
			if(dWord < numValidRowsForBaudotColumnEscape[currColumnNo]) {
				*outStringPtr++ = baudotColumnEscape[currColumnNo][dWord];
			}
			else {
				return RFD_STATUS_ERROR_BAUDOT_PARSE;
			}
		}
		else {
			// regular mode, not escape
			switch(dWord) {

				case RFD_BAUDOT_CODE_END:
					*outStringPtr++ = CHAR_NULL;
					foundBaudotEndCode = TRUE;
					break;

				case RFD_BAUDOT_CODE_ESC:
					isEscapeMode = TRUE;
					break;

				case RFD_BAUDOT_CODE_UC_LC:
					if(currColumnNo == BAUDOT_COL_NUM_LTR_UC) {
						currColumnNo = BAUDOT_COL_NUM_LTR_LC;
					}
					else if(currColumnNo == BAUDOT_COL_NUM_LTR_LC) {
						currColumnNo = BAUDOT_COL_NUM_LTR_UC;
					}
					else { //(currColumnNo = BAUDOT_COL_NUM_FIG)
						// RFD_BAUDOT_CODE_UC_LC is not a control code for the fig column.
						if(dWord < numValidRowsForBaudotColumnNonEscape[currColumnNo]) {
							*outStringPtr++ = baudotColumnNonEscape[currColumnNo][dWord];
						}
						else {
							return RFD_STATUS_ERROR_BAUDOT_PARSE;
						}
					}
					break;

				case RFD_BAUDOT_CODE_FIG_LTR:
					if(currColumnNo == BAUDOT_COL_NUM_LTR_UC ||
					   currColumnNo == BAUDOT_COL_NUM_LTR_LC ) {
						currColumnNo = BAUDOT_COL_NUM_FIG;
					}
					else { //(currColumnNo == BAUDOT_COL_NUM_FIG) {
						currColumnNo = BAUDOT_COL_NUM_LTR_UC;
					}
					break;

				case RFD_BAUDOT_CODE_CR:
					if(dWord < numValidRowsForBaudotColumnNonEscape[currColumnNo]) {
						*outStringPtr++ = baudotColumnNonEscape[currColumnNo][dWord];
						isLineFeedInsertRequested = TRUE;
					}
					else {
						return RFD_STATUS_ERROR_BAUDOT_PARSE;
					}
					break;

				default:
					// Not a control or special code.
					if(dWord < numValidRowsForBaudotColumnNonEscape[currColumnNo]) {
						*outStringPtr++ = baudotColumnNonEscape[currColumnNo][dWord];
					}
					else {
						return RFD_STATUS_ERROR_BAUDOT_PARSE;
					}
			}
		}
	}

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ConvertStringToBaudotBitStream( TCHAR inStringBuf[], int inStringBufMaxNumChars,
										       RFD_BITSTREAM_INFO * streamInfo, int baudotStringBufMaxNumSymbols )
{
	TCHAR inChar;
	BAUDOT_COL_NUM_ENUM currColumnNo;
	BOOL isEscapeMode;
	BOOL isLineFeedInsertRequested, foundEndofString, foundBaudotSymbol;
	UINT32 controlSymbol;
	TCHAR * inStringPtr;
	int outputBaudotSymbolCount, inputCharCount;
	unsigned int iRow;
	BAUDOT_COL_NUM_ENUM eCol;

	currColumnNo = BAUDOT_COL_NUM_LTR_UC;
	isEscapeMode = FALSE;
	isLineFeedInsertRequested = FALSE;
	foundEndofString = FALSE;
	inputCharCount = 0;
	outputBaudotSymbolCount = 0;
	inStringPtr = inStringBuf;

	while(!foundEndofString) {

		// Check for input string overflow and read next character
		if(inputCharCount < inStringBufMaxNumChars) {
			inputCharCount++;
			inChar = *inStringPtr++;
		}
		else {
			return RFD_STATUS_ERROR_INPUT_BUFFER_OVERFLOW;
		}

		// Check for End of String character
		if(inChar == CHAR_NULL) {

			controlSymbol = RFD_BAUDOT_CODE_END;
			if(RFD_BitstreamPutU32(streamInfo, RFD_BAUDOT_SYMBOL_NUM_BITS, controlSymbol)) {
				return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
			}
			// This is the last output char, so don't error check for the next char
			// string overflow.

			foundEndofString = TRUE;
			continue;
		}

		if(inChar == LINE_FEED) {
			// Carriage Return + Line Feed is represented
			// by baudot as a single baudot 'cr' symbol.
			// So skip Line Feed character by continuing to the next char.
			// (assume this line feed is following a CR character).
			continue;
		}

		foundBaudotSymbol = FALSE;

		for(eCol=0;eCol<BAUDOT_NUM_COLUMNS;eCol++) {
			for(iRow=0;iRow<numValidRowsForBaudotColumnNonEscape[eCol];iRow++) {

				if(inChar == baudotColumnNonEscape[eCol][iRow] ||
				   inChar == baudotColumnEscape[eCol][iRow]) {

					foundBaudotSymbol = TRUE;

					if(currColumnNo != eCol) {

						// Current column state is different from the column of the symbol to be output.
						// Determine the control symbol to change to column in which the baudot data symbol resides.
						// Need either:
						// Single RFD_BAUDOT_CODE_FIG_LTR command,
						// or RFD_BAUDOT_CODE_FIG_LTR command followed by RFD_BAUDOT_CODE_UC_LC

						if( currColumnNo == BAUDOT_COL_NUM_FIG || eCol == BAUDOT_COL_NUM_FIG) {

							// write baudot control symbol to bitstream to toggle from or to FIG/LTR state
							controlSymbol = RFD_BAUDOT_CODE_FIG_LTR;

							if(RFD_BitstreamPutU32(streamInfo, RFD_BAUDOT_SYMBOL_NUM_BITS, controlSymbol)) {
								return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
							}
							if(++outputBaudotSymbolCount >= baudotStringBufMaxNumSymbols) {
								return RFD_STATUS_ERROR_OUTPUT_BUFFER_OVERFLOW;
							}

							// toggle the currColumnNo to it's new column state given the control action above.
							if( currColumnNo == BAUDOT_COL_NUM_FIG) {
								currColumnNo = BAUDOT_COL_NUM_LTR_UC;
							}
							else { //currColumnNo == BAUDOT_COL_NUM_LTR_UC
								currColumnNo = BAUDOT_COL_NUM_FIG;
							}
						}

						if(currColumnNo != eCol) {

							// currCol and desiredCol still do not match; they both must
							// be in one of the LTR columns but different case (UC and LC).

							// write baudot control symbol to bitstream to switch to LTR LC state
							controlSymbol = RFD_BAUDOT_CODE_UC_LC;

							if(RFD_BitstreamPutU32(streamInfo, RFD_BAUDOT_SYMBOL_NUM_BITS, controlSymbol)) {
								return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
							}
							if(++outputBaudotSymbolCount >= baudotStringBufMaxNumSymbols) {
								return RFD_STATUS_ERROR_OUTPUT_BUFFER_OVERFLOW;
							}
						}
						// else iCol is FIG or LTR-UC in which case no switch to LTR LC control is needed.


						currColumnNo = eCol;
					}

					if(inChar == baudotColumnEscape[eCol][iRow]) {
						// Character is one of the escape mode characters, w
						// rite escape baudot control symbol
						controlSymbol = RFD_BAUDOT_CODE_ESC;
						if(RFD_BitstreamPutU32(streamInfo, RFD_BAUDOT_SYMBOL_NUM_BITS, controlSymbol)) {
							return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
						}
						if(++outputBaudotSymbolCount >= baudotStringBufMaxNumSymbols) {
							return RFD_STATUS_ERROR_OUTPUT_BUFFER_OVERFLOW;
						}
					}

					// else current column state is same as column of symbol to be output.

					// write baudot data symbol to bitstream.
					if(RFD_BitstreamPutU32(streamInfo, RFD_BAUDOT_SYMBOL_NUM_BITS, iRow)) {
						return RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR;
					}
					if(++outputBaudotSymbolCount >= baudotStringBufMaxNumSymbols) {
						return RFD_STATUS_ERROR_OUTPUT_BUFFER_OVERFLOW;
					}

					break;
				}
			}

			if(foundBaudotSymbol) {
				break;
			}
		}

		if(!foundBaudotSymbol) {
			// Note, alternative to error treatment would be to just skip
			// this unsupported character.
			return RFD_STATUS_ERROR_BAUDOT_UNSUPPORTED_CHAR;
		}
	}

	return RFD_STATUS_OK;
}

///////////////////////////////////////////////////////////////////////////////
// RFD_BaudotTest() :
// Call function from code to baudot generation and parsing.
// characters are randomly selected to build an original test string.
// the original test string is converted to baudot bitstream and then
// from bitstream back to string.
// The original and reconstructed strings are compared.
// The original and reconstructed strings are also writent to files :
// RFD_BAUDOT_TEST_ORIG_STRING_FILE_NAME and RFD_BAUDOT_TEST_RECONSTRUCTED_STRING_FILE_NAME.
//


#ifdef RFD_BAUDOT_TEST_ENABLE

#include <stdlib.h>  // for rand().

#define RFD_BAUDOT_TEST_STRING_LEN 3000
#define RFD_BAUDOT_TEST_BITSTREAM_BUF_LEN_BYTES (RFD_BAUDOT_TEST_STRING_LEN * 3)
#define RFD_BAUDOT_TEST_ORIG_STRING_FILE_NAME   TEXT("rfd_baudot_test-orig_string.txt")
#define RFD_BAUDOT_TEST_RECONSTRUCTED_STRING_FILE_NAME  TEXT("rfd_baudot_test-reconst_string.txt")
#define TEST_CHAR_LOOKUP_TABLE testTableCp1252

RFD_STATUS RFD_BaudotTest( void )
{
	int charCount;
	int charIndex;
	unsigned int iCol, iRow;
	BOOL foundBaudotSymbol;
	BOOL isPreviousCharACarriageReturn;
	TCHAR stringBuf[RFD_BAUDOT_TEST_STRING_LEN+1];
	TCHAR reconstructedStringBuf[RFD_BAUDOT_TEST_STRING_LEN+1];
	TCHAR * stringBufPtr;
	UCHAR bitstreamBuf[RFD_BAUDOT_TEST_BITSTREAM_BUF_LEN_BYTES];
	RFD_BITSTREAM_INFO streamInfo;
	RFD_FILE * file;
	TCHAR charValue;
	RFD_STATUS status;

	stringBufPtr = stringBuf;
	charValue = CHAR_NULL;
	charCount = 0;
	isPreviousCharACarriageReturn = FALSE;

	RFD_BitstreamInit( &streamInfo, bitstreamBuf,
						RFD_BAUDOT_TEST_BITSTREAM_BUF_LEN_BYTES * RFD_BITS_PER_BYTE );

	while(charCount < RFD_BAUDOT_TEST_STRING_LEN) {

		if(isPreviousCharACarriageReturn) {
			// add line feed if last character value was a carriage return,
			// to match the behaviour of RFD_ConvertStringToBaudotBitStream()
			isPreviousCharACarriageReturn = FALSE;
			*stringBufPtr++ = LINE_FEED;
			charCount++;
			continue;
		}

		charIndex = RFD_STDLIB_RAND() % 256;

		charValue = TEST_CHAR_LOOKUP_TABLE[charIndex];

		if(charValue == CHAR_NULL || charValue == RFD_BAUDOT_CONTROL_CODE) {
			continue;
		}

		if(charValue == CARRIAGE_RETURN) {
			isPreviousCharACarriageReturn = TRUE;
		}

		foundBaudotSymbol = FALSE;
		for(iCol=0;iCol<BAUDOT_NUM_COLUMNS;iCol++) {
			for(iRow=0;iRow<numValidRowsForBaudotColumnNonEscape[iCol];iRow++) {

				if(charValue == baudotColumnNonEscape[iCol][iRow] ||
				   charValue == baudotColumnEscape[iCol][iRow]) {

					// This character is supported by BAUDOT,
					// add it to the string.
					*stringBufPtr++ = charValue;
					charCount++;
					foundBaudotSymbol = TRUE;
					break;
				}
			}
			if(foundBaudotSymbol) {
				break;
			}
		}
	}

	// Add string null termination.
	*stringBufPtr = CHAR_NULL;

	// Convert String to Baudot Bitstream.
	status = RFD_ConvertStringToBaudotBitStream( stringBuf, RFD_BAUDOT_TEST_STRING_LEN+1,
												 &streamInfo, RFD_BAUDOT_TEST_BITSTREAM_BUF_LEN_BYTES );
	if(status != RFD_STATUS_OK) {
		RFD_PRINTF(TEXT("\n Error in RFD_ConvertStringToBaudotBitStream(), status = %d"), status);
		return status;
	}

	// Initialize the Bitstream once again.
	RFD_BitstreamInit( &streamInfo, bitstreamBuf,
						RFD_BAUDOT_TEST_BITSTREAM_BUF_LEN_BYTES * RFD_BITS_PER_BYTE );

	// Convert Baudot Stream back to a string.
	status = RFD_ConvertBaudotBitStreamToString( reconstructedStringBuf, RFD_BAUDOT_TEST_STRING_LEN+1,
												 &streamInfo, RFD_BAUDOT_TEST_BITSTREAM_BUF_LEN_BYTES );
	if(status != RFD_STATUS_OK) {
		RFD_PRINTF(TEXT("\n Error in RFD_ConvertBaudotBitStreamToString(), status = %d"), status);
		return status;
	}

	////////////////////////////////////
	// Write the original string to file
	////////////////////////////////////
	if(RFD_FOPEN_S(&file, RFD_BAUDOT_TEST_ORIG_STRING_FILE_NAME,TEXT("w")))
	{
		RFD_PRINTF(TEXT("\n Error opening file: %s"), RFD_BAUDOT_TEST_ORIG_STRING_FILE_NAME);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	RFD_FPRINTF(file, TEXT("%s"),stringBuf);

	RFD_FCLOSE(file);

	////////////////////////////////////
	// Write the reconstructed string to file
	////////////////////////////////////
	if(RFD_FOPEN_S(&file, RFD_BAUDOT_TEST_RECONSTRUCTED_STRING_FILE_NAME,TEXT("w")))
	{
		RFD_PRINTF(TEXT("\n Error opening file: %s"), RFD_BAUDOT_TEST_RECONSTRUCTED_STRING_FILE_NAME);
		return RFD_STATUS_ERROR_FILE_OPEN;
	}

	RFD_FPRINTF(file, TEXT("%s"),reconstructedStringBuf);

	RFD_FCLOSE(file);


	////////////////////////////////////
	// Compare original and reconstructed strings
	////////////////////////////////////
	if(RFD_STRNCMP(stringBuf, reconstructedStringBuf, RFD_BAUDOT_TEST_STRING_LEN+1))  {
		RFD_PRINTF(TEXT("\n Error, original and reconstructed strings do not match"));
		return status;
	}
	else {
		RFD_PRINTF(TEXT("\n Success, Original and reconstructed strings match"));
	}

	return RFD_STATUS_OK;
}
#endif
