////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	rfd_common\rfd.h
///
/// @brief	Declares the rfd class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2009 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef RFD_H
#define RFD_H

#include "rfd_config.h"
#include "rfd_bitstream_io.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

///////////////////////////
// RFD SDK Version Info
///////////////////////////

#define RFD_SDK_MAJOR_VERSION_NO   1 // first 2 digits of text string
#define RFD_SDK_MINOR_VERSION_NO   9 // middle 1 digit of text string
#define RFD_SDK_VERSION_RELEASE_NO 01 // last 2 digits of text string

// RFD_SDK_VERSION_STRING_PRINTF_FORMAT :
// The version string printf format field.
// The version string format is:
// xx.y.zz
// where xx = major version, y = minor version, and zz = version release.
#define RFD_SDK_VERSION_STRING_PRINTF_FORMAT  TEXT("%2d.%1d.%02d")

// RFD_SDK_VERSION_STRING_MAX_LEN :
// Maxumum length for the version text string.
//   2 major version digits + 1 decimal pt.
// + 1 minor version digits + 1 decimal pt.
// + 2 release version digits + 1 termination char.
#define RFD_SDK_VERSION_STRING_MAX_LEN 8

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup RfdrManagerApi
///
/// @brief
/// RFD SDK Version Information Structure.
///
/// This structure is filled-out by the API function RFD_GetSdkVersionInfo().
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef struct {
	/// Major Version Number.
	UINT32 rfdMajorVersionNo;
	/// Minor Version Number.
	UINT32 rfdMinorVersionNo;
	/// Release Number.
	UINT32 rfdVersionReleaseNo;
	/// The Version Information String.
	/// The string format is: xx.y.zz ,
	/// where xx = major version, y = minor version, and zz = version release number.
	TCHAR  rfdVersionString[RFD_SDK_VERSION_STRING_MAX_LEN];
} RFD_SDK_VERSION_INFO;

/////////////////////////////////////////////////////
#define RFD_CMDLINE_STR__MAX_OPT_STRING_LEN	50

/////////////////////////////////////////////////////
#define ECC_MAX_NUM_CHK_BLOCKS (1024*1024) // 2^20
#define ECC_MAX_CHK_BLOCK_INDEX (ECC_MAX_NUM_CHK_BLOCKS - 1)

/////////////////////////////////////////////////////
#define SIMPLE_CODE_MAX_NUM_CHK_BLOCKS 1
#define SIMPLE_CODE_MAX_CHK_BLOCK_INDEX (SIMPLE_CODE_MAX_NUM_CHK_BLOCKS - 1)

/////////////////////////////////////////////////////
#define RFD_TEST__KEEP_LINK_ALIVE_CLIENT_ID_VALUE  0xffffffff

/////////////////////////////////////////////////////
// Enable inclusion of the fileSize field for each file of the file set,
// in the File Set Metadata Message.
// When this is disabled, the RFD Receiver calculates the file sizes
// based on the totalSize and numFiles fields.
/////////////////////////////////////////////////////
#define RFD_FSET_MESSAGE_INCLUDE_FILE_SIZES_ENABLE

/////////////////////////////////////////////////////
// Enable dynamic update of the Block Message
// Carousel ID via the FileSet Metadata Message.
/////////////////////////////////////////////////////
#define DYNAMIC_BLOCK_MSG_CARID_ENABLE

/////////////////////////////////////////////////////
// RFDR_ENABLE_BLK_MSG_EXT_FIELD_EXTRACTION
// = 1 to enable reading of the Block Message Extension field into
//     receive buffer.
// = 0 to disable reading (just skip). The initial services using RFD
//     do not utilize this field, so =0 is the normal setting.
/////////////////////////////////////////////////////
#define RFDR_ENABLE_BLK_MSG_EXT_FIELD_EXTRACTION 1

/////////////////////////////////////////////////////
// RFD_CALC_FILE_CRC_BUF_LEN
// Length of temporary read buffer used in
// file crc calculation.
/////////////////////////////////////////////////////
#define RFD_FILE_CRC_CALC_BUF_LEN 8192

/////////////////////////////////////////////////////
// RFD_APPEND_FILES_BUF_LEN
// Length of temporary read buffer used in
// AppendFiles function.
/////////////////////////////////////////////////////
#define RFD_APPEND_FILES_BUF_LEN 8192

/////////////////////////////////////////////////////
// RFD_COPY_FILE_BUF_LEN
// Length of temporary read buffer used in
// CopyFile function.
/////////////////////////////////////////////////////
#define RFD_COPY_FILES_BUF_LEN 8192

/////////////////////////////////////////////////////
#define RFD_FSET_NAME_MAX_NUM_BAUDOT_SYMBOLS 16  // size in baudot symbols
#define RFD_FSET_NAME_MAX_LEN	RFD_FSET_NAME_MAX_NUM_BAUDOT_SYMBOLS
#define RFD_FSET_MAX_NUM_FILES 16 //

/////////////////////////////////////////////////////
// RFD Message - Field Sizes
/////////////////////////////////////////////////////
#define RFD_MSG_PVN_FIELD_NUM_BITS 4
#define RFD_MSG_FILE_ID_FIELD_NUM_BITS 16
#define RFD_MSG_BLK_TYPE_FIELD_NUM_BITS 3
#define RFD_MSG_SRC_TYPE_BLK_INDEX_FIELD_NUM_BITS 13
#define RFD_MSG_ECC1_CHECK_TYPE_BLK_INDEX_FIELD_NUM_BITS 20
#define RFD_MSG_ECC2_CHECK_TYPE_BLK_INDEX_FIELD_NUM_BITS 20
#define RFD_MSG_BLK_EXT_FIELD_NUM_BITS 8
#define RFD_MSG_CRC_FIELD_NUM_BITS 32
#define RFD_FSET_MSG_FSET_SIZE_FIELD_NUM_BITS 32
#define RFD_FSET_MSG_CODING_ALG_FIELD_NUM_BITS 3
#define RFD_FSET_MSG_COMPRESSION_ALG_FIELD_NUM_BITS 3
#define RFD_FSET_MSG_LIFE_TIME_FIELD_NUM_BITS 14
#define RFD_FSET_MSG_NUM_FILES_FIELD_NUM_BITS 4
#define RFD_FSET_MSG_FILE_ID_FIELD_NUM_BITS 16
#define RFD_FSET_MSG_BLK_LEN_FIELD_NUM_BITS 14
#define RFD_FSET_MSG_FILE_SIZE_FIELD_NUM_BITS 24
#define RFD_FSET_MSG_FILE_CRC_FIELD_NUM_BITS 32
#define RFD_FSET_MSG_EXT_LENGTH_FIELD_NUM_BITS 7

// New DMI and DMIRFU fields added to EXT record in RFD Protocol version 1.2
#define RFD_FSET_MSG_UPDATE_FILES_DMI_FIELD_NUM_BITS 10
#define RFD_FSET_MSG_UPDATE_FILES_DMIRFU_FIELD_NUM_BITS 6

#define RFD_MSG_PRESENCE_FLAG_FIELD_NUM_BITS 1
#define RFD_MGS_FIELD_IS_PRESENT_TRUE 1
#define RFD_MGS_FIELD_IS_PRESENT_FALSE 0

#define RFD_MAX_BITSTREAM_MESSAGE_LEN_BYTES 9216 // 9 AUs (1024*9)
#define RFD_BLK_MSG_MAX_EXT_FIELD_LEN_BYTES 255
#define RFD_FSET_MSG_MAX_EXT_FIELD_LEN_BYTES 128
#define RFD_MSG_CRC_FIELD_NUM_BYTES  (RFD_MSG_CRC_FIELD_NUM_BITS/RFD_BITS_PER_BYTE)

/////////////////////////////////////////////////////
// Baudot Symbol related definitions
/////////////////////////////////////////////////////
#define RFD_BAUDOT_SYMBOL_NUM_BITS 5
#define RFD_BAUDOT_SYM_END 0x00
#define RFD_BAUDOT_SYM_ESC 0x1b

/////////////////////////////////////////////////////

#define RFD_MAX_BLK_LEN_DWORDS 2048 // 8192/4
#define RFD_MIN_BLK_LEN_DWORDS 1

#define RFD_MAX_BLK_LEN_BYTES  (RFD_MAX_BLK_LEN_DWORDS * 4)
#define RFD_MIN_BLK_LEN_BYTES  (RFD_MIN_BLK_LEN_DWORDS * 4)

#define RFD_MAX_FILE_LEN_BLOCKS 1024 // equivalent to equation dimension
#define RFD_MIN_FILE_LEN_BLOCKS 1

typedef UINT32 RFD_TEST_MESSAGE_CLIENT_ID_HEADER;
typedef UINT32 RFD_TEST_MESSAGE_LENGTH_HEADER;

// 0 is an unused/invalid DMI value
#define RFD_DMI_UNUSED_VALUE 0
// Default DMI Value, to set File Set Metadata blockMsgDmi field
// when extension record is not present.
#define RFD_FSET_MDATA_BLK_MSG_DMI_DEFAULT_VALUE    RFD_DMI_UNUSED_VALUE
#define RFD_FSET_MDATA_BLK_MSG_DMI_MAX_VALUE        ((1 << RFD_FSET_MSG_UPDATE_FILES_DMI_FIELD_NUM_BITS)-1)

#define RFD_ACCESS_UNIT_FILE_NAME_EXT TEXT(".au")

typedef enum {
	RFD_STATUS_OK = 0,
	RFD_STATUS_OK_DECODE_COMPLETE,
	RFD_STATUS_OK_MESSAGE_RECEIVED,			// Message Queue module
	RFD_STATUS_OK_MESSAGE_TIMEOUT,			// Message Queue module
	RFD_STATUS_OK_MAX_SEMAPHORE_COUNT,		// Message Queue module
	RFD_STATUS_OK_BUF_FULL,					// Message Queue module
	RFD_STATUS_OK_MSG_OVERSIZED,			// Message Queue module
	RFD_STATUS_ERROR_MESSAGE_QUEUE_EMPTY,	// Message Queue module
	RFD_STATUS_ERROR_MESSAGE_QUEUE_CREATE,	// Message Queue module
	RFD_STATUS_EARLY_EXIT_ON_REQUEST,
	RFD_STATUS_UNSUPPORTED_BLOCK_TYPE,
	RFD_STATUS_UNSUPPORTED_CODE_TYPE,
	RFD_STATUS_UNSUPPORTED_COMPRESSION_TYPE,
	RFD_STATUS_UNSUPPORTED_FEATURE,
	RFD_STATUS_ERROR_GENERAL,
	RFD_STATUS_ERROR_BUFFER_SIZE,
	RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE,
	RFD_STATUS_ERROR_INVALID_FSET_NUM_FILES,
	RFD_STATUS_ERROR_INVALID_CODE_TYPE,
	RFD_STATUS_ERROR_INVALID_COMP_TYPE,
	RFD_STATUS_ERROR_INVALID_BLOCK_TYPE,
	RFD_STATUS_ERROR_INVALID_BLOCK_INDEX,
	RFD_STATUS_ERROR_INVALID_BLOCK_LENGTH,
	RFD_STATUS_ERROR_INVALID_FSET_BAUDOT_NAME,
	RFD_STATUS_ERROR_INVALID_FILE_SIZE,
	RFD_STATUS_ERROR_INVALID_MESSAGE_LENGTH,
	RFD_STATUS_ERROR_INVALID_MESSAGE_TYPE,
	RFD_STATUS_ERROR_MEM_ALLOC,
	RFD_STATUS_ERROR_FILE_OPEN,
	RFD_STATUS_ERROR_FILE_FIND,
	RFD_STATUS_ERROR_FILE_READ,
	RFD_STATUS_ERROR_FILE_WRITE,
	RFD_STATUS_ERROR_FILE_SEEK,
	RFD_STATUS_ERROR_FILE_SIZE_CHECK,
	RFD_STATUS_ERROR_FILE_RENAME,
	RFD_STATUS_ERROR_FILE_DELETE,
	RFD_STATUS_ERROR_FILE_LEN,
	RFD_STATUS_ERROR_TRANSACTION_IN_PROGRESS,
	RFD_STATUS_ERROR_INVALID_STORAGE,
	RFD_STATUS_ERROR_DIRECTORY_CREATE,
	RFD_STATUS_ERROR_DECODER_STATE,
	RFD_STATUS_ERROR_DECODER_BLK_LEN,
	RFD_STATUS_ERROR_DECODER_EQU_LEN,
	RFD_STATUS_ERROR_DECODER_NUM_INPUT_BLKS,
	RFD_STATUS_ERROR_DECODER_INSUFFICIENT_MEMORY,
	RFD_STATUS_ERROR_DECODER_GENERAL,
	RFD_STATUS_ERROR_NUM_COLLECTED_BLKS,
	RFD_STATUS_ERROR_PIVOT_NUM,
	RFD_STATUS_ERROR_INVALID_VALUE,
	RFD_STATUS_ERROR_OBJECT_ABANDONED,
	RFD_STATUS_OBJECT_TIMEOUT,
	RFD_STATUS_ERROR_MUTEX_RELEASE,
	RFD_STATUS_ERROR_CREATE_THREAD,
	RFD_STATUS_ERROR_SET_THREAD_PRIORITY,
	RFD_STATUS_ERROR_CREATE_MUTEX,
	RFD_STATUS_ERROR_BITSTREAM_INIT_ERROR,
	RFD_STATUS_ERROR_BITSTREAM_READ_ERROR,
	RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR,
	RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR,
	RFD_STATUS_ERROR_GZ_OPEN_INPUT,
	RFD_STATUS_ERROR_GZ_OPEN_OUTPUT,
	RFD_STATUS_ERROR_GZ_CLOSE,
	RFD_STATUS_ERROR_GZ_READ,
	RFD_STATUS_ERROR_GZ_WRITE,
	RFD_STATUS_ERROR_GZ_FILE_SIZE_ERROR,
	RFD_STATUS_ERROR_CRC,
	RFD_STATUS_ERROR_OUTPUT_BUFFER_OVERFLOW,
	RFD_STATUS_ERROR_INPUT_BUFFER_OVERFLOW,
	RFD_STATUS_ERROR_BAUDOT_PARSE,
	RFD_STATUS_ERROR_BAUDOT_UNSUPPORTED_CHAR,
	RFD_STATUS_MSG_PROTOCOL_VERSION_NUM_OUT_OF_RANGE,
	RFD_STATUS_ERROR_THREAD_EXIT,
    RFD_STATUS_FILE_SET_COMPLETE,
    RFD_STATUS_BLK_MSG_FILE_ID_NOT_MATCHED,
    RFD_STATUS_BLK_MSG_DMI_SOURCE_NOT_MATCHED,
    RFD_STATUS_CAROUSEL_ID_NOT_MATCHED,
    RFD_STATUS_ERROR_INVALID_STATE,
    RFD_STATUS_ERROR_INVALID_MODE,
    RFD_STATUS_ERROR_FILE_SYNC,
    RFD_STATUS_ERROR_LLIST_FIND

} RFD_STATUS;

typedef enum {
	RFD_ERR_LEVEL_FATAL,
	RFD_ERR_LEVEL_MEDIUM,
	RFD_ERR_LEVEL_LOW,
} RFD_ERROR_LEVEL_ENUM;

typedef enum {
	RFD_CODE_TYPE_SIMPLE_BLK = 0,
	RFD_CODE_TYPE_ECC1 = 1,
	RFD_CODE_TYPE_ECC2 = 2,
	RFD_CODE_TYPE_MAX_NUM = 3 // maximum supported (future RFD versions may add additional code types and extend this number).

} RFD_CODE_TYPE_ENUM;

typedef enum {
	RFD_COMP_TYPE_NONE = 0,
	RFD_COMP_TYPE_GZIP = 1,
	RFD_COMP_TYPE_MAX_NUM = 2 // maximum supported (future RFD versions may add additional compression types and extend this number).

} RFD_COMP_TYPE_ENUM;

typedef enum {
	MSG_TYPE_FSET_MDATA,
	MSG_TYPE_BLOCK,
	MSG_TYPE_UNKNOWN,
	MSG_TYPE_INVALID // to indicate or initialize for invalid message type

} RFD_MSG_TYPE;

typedef enum {
	BLK_TYPE_SOURCE			= 0,			// source (message) block
	BLK_TYPE_SIMPLE_PARITY	= 1,
	BLK_TYPE_ECC1_CHECK		= 2,
	BLK_TYPE_ECC2_CHECK		= 3,
	BLK_TYPE_RESERVED1		= 4,
	BLK_TYPE_RESERVED2		= 5,
	BLK_TYPE_RESERVED3		= 6,
	BLK_TYPE_RESERVED4		= 7,
	NUM_BLK_TYPES			= 8

} RFD_BLOCK_TYPE;

typedef enum {
	BLK_CATEGORY_SOURCE		= 0, // source block for any code
	BLK_CATEGORY_CHECK		= 1	 // check/parity for any code
} RFD_BLOCK_CATEGORY;

typedef struct {
	UINT16	id;
	UINT32	crc;
	INT32	size;
	UINT16	blkLenBytes;

} RFD_FILE_INFO_STRUCT;


typedef struct {
	TCHAR name[RFD_FSET_NAME_MAX_LEN];		// name of the file set
	INT32 totalSize;	// total size of the files
	UCHAR blockMsgCarouselId;
    UINT16 blockMsgDmi; // Block messages (Update Files) DMI, from the extension record.
                        // This field is introduced in RFD Bitstream Protocol version 1.2
	RFD_CODE_TYPE_ENUM codeType;		// Block Coding Algorithm for these files
	RFD_COMP_TYPE_ENUM compType;		// Compression Algorithm for these files
	INT32 lifeTime;			// lifetime of the file set.
	UCHAR numFiles;		// Number of files in the file set.
	RFD_FILE_INFO_STRUCT fileInfo[RFD_FSET_MAX_NUM_FILES];	// info specific to each file.
	UCHAR			extensionLenBytes;
	UCHAR			extension[RFD_FSET_MSG_MAX_EXT_FIELD_LEN_BYTES];

} RFD_FSET_MDATA_STRUCT;

typedef struct {
	UINT16			fileId;			// file unique id
	RFD_BLOCK_TYPE	blkType;		// block type
	UINT32			blkIndex;		// block index
	UINT16			blkLenBytes;	// block length. Not part of the ota broadcast message.
									// Can be derived by parser using known size of overall message.
	UCHAR			block[RFD_MAX_BLK_LEN_BYTES];
	UCHAR			extensionLenBytes;
	UCHAR			extension[RFD_BLK_MSG_MAX_EXT_FIELD_LEN_BYTES];

} RFD_BLOCK_MSG_STRUCT;

typedef struct {
	UCHAR protocolVersionNo;		  // Protocol Version Number
	UCHAR nBitsCarouselIdField;		  // number of bits for the Carousel Id field.
	UCHAR nBitsFileSetUpdateCarField; /* Number of bits for the (Dynamic Update-Files/Block) CAR field of the
									   * FSET Metadata message */
	UCHAR fileSetMdataMsgCarouselId;  // Carousel ID for the File Set Metadata message.
	UINT16 maxBlockLenBytes;		  // Maximum length of a Block.
	INT32 maxFileLenBytes;			  // Maximum length of an individual file of a file set.
	int blockMsgCarouselId;           /* Carousel ID for the Block type message.
									   * For DYNAMIC_BLOCK_CARID_ENABLE, this file can be initialized to -1 to
									   * indicate value not yet rececived. */
} RFD_MESSAGE_CONFIG_INFO;

typedef union {
	RFD_FSET_MDATA_STRUCT fileSetInfoMsg;
	RFD_BLOCK_MSG_STRUCT blockMsg;

} RFD_MSG;

typedef struct {
	RFD_MSG_TYPE messageType;
	RFD_MSG message; // union
	UCHAR bitstreamBuf[RFD_MAX_BITSTREAM_MESSAGE_LEN_BYTES];
	UINT32 bitstreamBufLenBytes;
	UINT32 messageBitstreamLenBytes;
	RFD_MESSAGE_CONFIG_INFO configInfo; // num bits in carousel id field, initialized by setup thread.
} RFD_MESSAGE_INFO;

#define RFD_GZ_MAX_COMPRESSION_LEVEL 9
#define RFD_GZ_MIN_COMPRESSION_LEVEL 1
#define RFD_GZ_DEFAULT_COMPRESSION_LEVEL_FLAG -1

typedef enum {
	RFD_GZ_COMPRESSION_STRATEGY_Z_DEFAULT,
	RFD_GZ_COMPRESSION_STRATEGY_Z_FILTERED,
	RFD_GZ_COMPRESSION_STRATEGY_Z_HUFFMAN_ONLY,
	RFD_GZ_COMPRESSION_STRATEGY_Z_RLE

} RFD_GZ_COMPRESSION_STRATEGY;

typedef struct {
	int  nWaitEventEntries;
	RFD_EVENT_HANDLE * waitEventArray;
} RFD_MULTI_WAIT_EVENT_STRUCT;

#ifdef RFD_ALT_SINGLE_THREAD_MODE_ENABLE
typedef struct {
	int  initialCount;
    int maximumCount;
	int count;
} RFD_EMUL_SEMAPHORE_STRUCT;
#endif

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @ingroup MfmAPI
///
/// @brief
/// RFD SXI Time Indication structure.
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef struct {

    /// @brief Minutes past the hour (0-59)
    UCHAR minute;

    /// @brief Hour of the day (0-23)
    UCHAR hour;

    /// @brief Day of the month (1-31)
    UCHAR day;

    /// @brief Month of the year (1-12). Note: this is different than the Standard C Library
    /// tm structure definition for tm->tm_mon, which is a zero relative month (0 to 11).
    UCHAR month;

    /// @brief Years since 2000. Note: this is different than the Standard C Library tm structure
    /// definition for tm->year, which is years since 1900.
    UCHAR yearSince2k;

} RFD_SXI_TIME_INFO_STRUCT;

///////////////////////////////////////////////////////////////////////////////////////////////
int RFD_FindConsoleInputString(int argStartNo, int argEndNo, TCHAR * argv[], const TCHAR findString[]);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_DPrint( const TCHAR * format_string, ...);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_FIND_FILE_HANDLE  RFD_FindFirstFile(const TCHAR pathBuf[],
										RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr,
										BOOL * isError);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_FindNextFile(RFD_FIND_FILE_HANDLE findFileHandle,
									   RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr,
									   BOOL * isError);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_FindClose(RFD_FIND_FILE_HANDLE findFileHandle);
///////////////////////////////////////////////////////////////////////////////////////////////
TCHAR * RFD_FindDataGetFileNamePtr( RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_FindDataIsItemADirectory( RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_FindDataIsItemReadOnly( RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr);
///////////////////////////////////////////////////////////////////////////////////////////////
DWORD RFD_FindDataGetFileSize( RFD_FIND_FILE_DATA_INFO * findFileDataInfoPtr);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_IsFile(const TCHAR fileName[]);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_IsDirectory(const TCHAR dirName[]);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_GetFileSize( const TCHAR fileName[], UINT32 * pFileSize);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_FileSyncByHandle(RFD_FILE *hFile);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_DirectorySync(const TCHAR dirName[]);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_DeleteDirectoryNonRecursive(const TCHAR* sPath);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_DeleteDirectoryContentsNonRecursive(const TCHAR* sPath, BOOL includeSubDirRemoval);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_DeleteDirectoryRecursive(const TCHAR* sPath, BOOL deleteDirContentsOnly );
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_CopyDirectoryFilesSkipSingleFile(const TCHAR dstPathBuf[], const TCHAR srcPathBuf[], 
				const TCHAR skipFileName[], BOOL doFileSync);
///////////////////////////////////////////////////////////////////////////////////////////////
int RFD_GetNumFilesInDirectory(const TCHAR* sPath, BOOL includeDirs);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_IsReservedFileSystemFileName(const TCHAR* str);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ReadFile(const TCHAR pathBuf[], void * pData, INT32 DataSize, 
				INT32 * pActualReadDataSize, BOOL CheckFileSize);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CreateFile(TCHAR pathBuf[], BOOL doFileSync);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_WriteFile(const TCHAR pathBuf[], void * pData, INT32 DataSize, BOOL doFileSync);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_AppendDataToFile(TCHAR pathBuf[], void * pData, INT32 DataSize, BOOL doFileSync);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_AppendFiles(TCHAR srcFileNameBuf[], TCHAR dstFileNameBuf[], BOOL doFileSync);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CopyFile(const TCHAR dstFileNameBuf[], const TCHAR srcFileNameBuf[], BOOL doFileSync);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_GenerateMessage(RFD_MESSAGE_INFO * messageInfo, UINT32 * messageLenBitsPtr);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ParseMessage(RFD_MESSAGE_INFO * messageInfo, BOOL isBlockMsgParsingEnabled);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ParseMessageExt(
	        RFD_MSG_TYPE * messageType /* output */,
	        RFD_FSET_MDATA_STRUCT * fileSetInfoMsg /* output */,
	        RFD_BLOCK_MSG_STRUCT * blockMsg /* output */,
	        UCHAR  bitstreamBuf[],
	        UINT32 messageBitstreamLenBytes,
	        const RFD_MESSAGE_CONFIG_INFO * configInfo);
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_PeekMessageHeader(UCHAR bitstreamBuf[],
								 UINT32 messageBitstreamLenBytes,
								 UCHAR nBitsCarouselIdField,
								 UCHAR * protocolVersionNo /* output */,
								 UCHAR * carouselId /* output */);
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_PeekBlockMessageFileId(UCHAR bitstreamBuf[],
								     UINT32 messageBitstreamLenBytes,
	                                 RFD_MESSAGE_CONFIG_INFO * configInfo,
    								 UINT16 * fileId /* output */);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS rfd_gz_file_uncompress(const TCHAR * outfile, const TCHAR * infile, UCHAR * exitRequestFlagPtr, BOOL doFileSync);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS rfd_gz_file_compress(const TCHAR * outfile, const TCHAR * infile,
								int compression_level,
								RFD_GZ_COMPRESSION_STRATEGY compression_strategy);
///////////////////////////////////////////////////////////////////////////////////////////////
void RFD_ResetCrc(UINT32 * crc);
///////////////////////////////////////////////////////////////////////////////////////////////
void RFD_CalculateCrc(UINT32 * crc, UCHAR * buf, UINT32 length);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_CalculateFileCrc(const TCHAR fileNameBuf[], UINT32 *calculatedCrc);

///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ConvertBaudotBitStreamToString( TCHAR outStringBuf[], int outStringBufMaxNumChars,
									    RFD_BITSTREAM_INFO * streamInfo, int baudotStringBufMaxNumSymbols );
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ConvertStringToBaudotBitStream( TCHAR inStringBuf[], int inStringBufMaxNumChars,
									   RFD_BITSTREAM_INFO * streamInfo, int baudotStringBufMaxNumSymbols );
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_BaudotTest( void ); // for test
///////////////////////////////////////////////////////////////////////////////////////////////
void RFD_ReportError(RFD_ERROR_LEVEL_ENUM errLevel, const TCHAR errInfoString[]);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_SEMAPHORE_HANDLE RFD_CreateSemaphore(int initialCount, int maximumCount);
///////////////////////////////////////////////////////////////////////////////////////////////
BOOL RFD_CloseSemaphore(RFD_SEMAPHORE_HANDLE semaphoreHandle);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_SignalSemaphore(RFD_SEMAPHORE_HANDLE semaphoreHandle);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_WaitForSemaphore(RFD_SEMAPHORE_HANDLE semaphoreHandle, INT32 timeoutMilliseconds);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_AcquireMutex(RFD_MUTEX_HANDLE hMutex, INT32 timeoutMilliseconds);
///////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_ReleaseMutex(RFD_MUTEX_HANDLE hMutex);
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_WaitForEvent(RFD_EVENT_HANDLE hEvent, INT32 timeoutMilliseconds);
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_SetEvent(RFD_EVENT_HANDLE hEvent);
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_WaitForOneOfMultiEvents(RFD_MULTI_WAIT_EVENT_STRUCT * multiWaitEventInfo,
									   INT32 timeoutMilliseconds,
									   int * ocurredEventIndexPtr);
///////////////////////////////////////////////////////////////////////////////
RFD_MULTI_WAIT_EVENT_STRUCT * RFD_CreateMultiWaitEventObject( int nWaitEventEntries);
///////////////////////////////////////////////////////////////////////////////
void RFD_DeleteMultiWaitEventObject( RFD_MULTI_WAIT_EVENT_STRUCT * multiWaitEventObject);
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_AddEventToMultiWaitEventObject(RFD_MULTI_WAIT_EVENT_STRUCT * multiWaitEventObject,
											  RFD_EVENT_HANDLE eventObject,
											  int eventIndex);
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CalculateFileSetMetatdataFileSizes(RFD_FSET_MDATA_STRUCT * fileSetInfo);
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_CalculateFileSetMetatdataNumFiles(RFD_FSET_MDATA_STRUCT * fileSetInfo, UINT32 maxFileSizeBytes);

///////////////////////////////////////////////////////////////////////////////
RFD_EVENT_HANDLE RFD_CreateEvent(void);
///////////////////////////////////////////////////////////////////////////////
BOOL RFD_CloseEvent(RFD_EVENT_HANDLE eventHandle);
///////////////////////////////////////////////////////////////////////////////
RFD_MUTEX_HANDLE RFD_CreateMutex(void);
///////////////////////////////////////////////////////////////////////////////
BOOL RFD_CloseMutex(RFD_MUTEX_HANDLE mutexHandle);
///////////////////////////////////////////////////////////////////////////////
RFD_THREAD_HANDLE RFD_CreateThread( SIZE_T stackSize,
									const TCHAR * threadName,
									RFD_THREAD_START_ROUTINE_PTR startAddress,
									VOID * parameterPtr,
									DWORD * threadIdPtr );
///////////////////////////////////////////////////////////////////////////////
BOOL RFD_CloseThread(RFD_THREAD_HANDLE threadHandle);
///////////////////////////////////////////////////////////////////////////////
BOOL RFD_SetThreadPriority(RFD_THREAD_HANDLE threadHandle, int nPriority);
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS RFD_WaitForThreadExit(RFD_THREAD_HANDLE threadHandle, INT32 timeoutMilliseconds);
///////////////////////////////////////////////////////////////////////////////
DWORD RFD_GetLastError(VOID);
///////////////////////////////////////////////////////////////////////////////
BOOL RFD_IsMatchingFileNameExtension(TCHAR fileName[], TCHAR fileNameExtension[]);
///////////////////////////////////////////////////////////////////////////////
int RFD_StripFileNameExtension(TCHAR fileName[], TCHAR fileNameExtension[]);
///////////////////////////////////////////////////////////////////////////////
void RFD_PrintFileSetInfo( RFD_FSET_MDATA_STRUCT * fileSetInfo );
///////////////////////////////////////////////////////////////////////////////
void RFD_CopyCharacterRangeToString(
            TCHAR * dstStringPtr,
            const TCHAR * srcStringPtr,
            UINT16 srcStartCharOffset,
            UINT16 numChars );
///////////////////////////////////////////////////////////////////////////////
RFD_STATUS  RFD_DeleteFileSysCorruptedDirectoryEntries(
				const TCHAR *dirPathName, 
				UN32 *pNumCorruptedEntriesFound, 
				const TCHAR *pAncillaryDeleteFileName, 
				BOOL bDoDirSyncAfterAncillaryDelete, 
				BOOL bDoDirSyncAfterCorruptedEntryDeletes, 
				BOOL doRecursive);

#ifdef __cplusplus
}
#endif

#endif // RFD_H