////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	multifile_manager\multifile_manager.h
///
/// @brief	Declares the multifile manager class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2013 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef RFD_MULTIFILE_MANAGER_H
#define RFD_MULTIFILE_MANAGER_H

#include "rfd_config.h"
#include <time.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

// doxygen group start symbol, don't remove.
///////////////////////////////////////////////////////////////////////////////////////////////////
/// @addtogroup MfmAPI RFD Multifile Manager (MFM) API
/// @{
///////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// The default time in milliseconds for the MFM Consumer File Ready polling interval
/// (the interval that the MFM File Consumer Thread waits each time before polling the
/// set of assigned RFD Clients for New File Ready events.
////////////////////////////////////////////////////////////////////////////////////////////////////
#define MFM_DEFAULT_FILE_READY_POLL_INTERVAL_MSEC 300

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Unused value for the File Identity VersionTo and VersionFrom fields.
////////////////////////////////////////////////////////////////////////////////////////////////////
#define MFM_FILE_IDENTITY_VERSION_NUM_UNUSED 0xFFFFFFFF

// Default MFM Client Name
#define MFM_DEFAULT_CLIENT_NAME TEXT("mfm-")

// Max length of the MFM Subname string (including string termination char)
#define MFM_SUBNAME_MAX_LEN   RFD_FSET_NAME_MAX_LEN

// Require reception of 2 Update File Metadata messages to set
// Carousel Reception Status as Completed.
// The reception of a second message indicates the carousel has wrapped.
#define MFM_UPDATE_FILE_METADATA_CAROUSEL_COMPLETE_MSG_COUNT_THRESH 2

typedef UINT32 MFM_FILE_IDENTITY_MASK;

#define MFM_FILE_IDENTITY_SUBNAME       (0x00001)
#define MFM_FILE_IDENTITY_VERSION_FROM  (0x00002)
#define MFM_FILE_IDENTITY_VERSION_TO    (0x00004)

#define MFM_FILE_IDENTITY_VERSION_INFO  (MFM_FILE_IDENTITY_VERSION_FROM | MFM_FILE_IDENTITY_VERSION_TO)

#define MFM_FILE_IDENTITY_ALL      (   \
    (MFM_FILE_IDENTITY_MASK)                   \
    MFM_FILE_IDENTITY_SUBNAME |                \
    MFM_FILE_IDENTITY_VERSION_FROM |           \
    MFM_FILE_IDENTITY_VERSION_TO               \
    )

typedef struct mfm_life_expiration_criterion_info {
    time_t currentSimpleCalendarTime;
    BOOL isCurrentTimeAvailable;
} FILE_ITEM_LIFE_EXPIRATION_CRITERION_INFO;

typedef struct {
    UINT16 dmi;
    INT32 inUseCount;
} MFM_ACTIVE_DMI_INFO_STRUCT, * MFM_ACTIVE_DMI_INFO_HANDLE;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// MFM File Identity Info structure.
///
/// Within the MFM framework, the *Name* field of the RFD Update File Medatadata message is
/// decomposed into a *SubName* string, a *Version-From* number, and a *Version-To* number. These
/// parameters define the **MFM File Identity**, which uniquely identifies each RFD Update File
/// (also called an *MFM File Item)*, and which is represented by this structure.
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef struct {
    /// @brief  The **sub Name** identity derived from the RFD Metadata **Name** field.
    TCHAR subname[MFM_SUBNAME_MAX_LEN];

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The versionFrom field is relevant only in the Delta Update type MFM File Management Policy
    /// configuration (#MFM_FILE_MANAGEMENT_POLICY_DELTA_UPDATES). It is the Update File version number
    /// that this Update File, identified by this File Identity Info, is updating **from**.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    UINT32 versionFrom;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// Specifies the resulting version number (the **to** version) of an Update File identified by
    /// this File Identity Info.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    UINT32 versionTo;

} MFM_FILE_IDENTITY_STRUCT, * MFM_FILE_IDENTITY_HANDLE;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// MFM RFD Client structure.
///
/// The MFM RFD Client structure contains an RFD Client's Message Collector Handle and File
/// Consumer Handle, which are the handles needed to manage a single RFD Client.
///
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef struct {
    /// @brief  The RFD Client's Message Collector Handle (obtained using RFD_GetMessageCollectorHandle()).
    RFD_COLLECTOR_THREAD_DATA_HANDLE rfdMessageCollectorHandle;

    /// @brief  The RFD Client's File Consumer Handle (obtained using RFD_GetFileConsumerHandle()).
    RFD_CONSUMER_THREAD_DATA_HANDLE rfdFileConsumerHandle;

} MFM_RFD_CLIENT_STRUCT, * MFM_RFD_CLIENT_HANDLE;

typedef enum {
    MFM_FILE_ITEM_LIST_TYPE_ACTIVE,
    MFM_FILE_ITEM_LIST_TYPE_WAITING,
    MFM_FILE_ITEM_LIST_TYPE_STORED

} MFM_FILE_ITEM_LIST_TYPE_ENUM;

typedef struct {

    MFM_FILE_IDENTITY_STRUCT fileIdentityInfo;

    // Non-NULL when file is in the Active List
    MFM_RFD_CLIENT_HANDLE hMfmRfdClient;

    // Counts the number of times the rfd metadata message corresponding
    // to this mfm file item has been received.
    // This is used to determine when the entire rfd metadata message carousel
    // has been received (i.e. a count of >= 2 on any file item indicates
    // the carousel has wrapped).
    // For an MFM configuration enabling File Priority Sorting, knowing that the
    // entire carousel has not yet been recieved is used to hold-off Priority Sorting
    // and any corresponding File "Activations".
    // Note issues with this method due to possible missed messages upon
    // low signal strength etc. Looking for counts of >= 2 on all waiting file items
    // is another alternative, but could be problematic if the carousel message content
    // is changed e.g. one or more messages that get removed from the carousel
    // may get "stuck" at a count of 1.
    UINT32 rfdMetadataMsgRxCount;

    // Non-NULL when file is in the Waiting List
    UCHAR * rfdMetadataMsgBuf;
    UINT32 rfdMetadataMsgSize;

    // DMI identifying data channel delivering the RFD Block Messages for this
    // RFD Update file (File Item).
    UINT16 rfdBlockMsgDmi;

    // The RFD Metadata Life field defining the expiration time for this RFD Update File (File Item).
    INT32 rfdLifeTime;

} MFM_FILE_ITEM_STRUCT, * MFM_FILE_ITEM_HANDLE;

typedef struct rfd_client_management_struct {
    RFD_LLIST_HANDLE busyMfmRfdClientList;
    RFD_LLIST_HANDLE freeMfmRfdClientList;

} MFM_RFD_CLIENT_RESOURCE_INFO_STRUCT, * MFM_RFD_CLIENT_RESOURCE_INFO_HANDLE;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Enumeration of the Update Types of MFM
///
/// The Update Type is one of the policies that define how MFM operates in processing Update Files.
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef enum {

    ////////////////////////////////////////////////////////////////////////////////////////////////
    /// Specifies the Delta Update Policy. Update Files of the same MFM Subname value but with different
    /// Version Info values are all treated as separate updates that must be sequentially processed
    /// for Version-To values greater than the Version-To value of the latest stored Update File.
    /// (NOTE: This type is Not currently supported by MFM).
    ////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_FILE_MANAGEMENT_POLICY_DELTA_UPDATES,

    ////////////////////////////////////////////////////////////////////////////////////////////////
    /// Specifies the Absolute Update Policy. Only the latest available on-the-air Update File of a
    /// given MFM Subname value is processed by MFM. MFM terminates the processing of any Update File
    /// upon detection of a new on-the-air Update File of the same Subname but different Version Info
    /// value.
    ////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_FILE_MANAGEMENT_POLICY_ABSOLUTE_UPDATES

} MFM_FILE_MANAGEMENT_POLICY_UPDATE_TYPE_ENUM;


////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// MFM File Management Policy Info Structure.
///
/// Specifies the operating policies/rules that define how MFM operates in collecting and
/// processing Update Files.
///
/// An instance of this structure is passed in MFM_Create() to configure the polices of a new
/// Multifile Mangager instance.
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef struct {

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// Specifies the MFM Update Type: Absolute type or Differential type. See definition of
    /// #MFM_FILE_MANAGEMENT_POLICY_UPDATE_TYPE_ENUM.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_FILE_MANAGEMENT_POLICY_UPDATE_TYPE_ENUM updateType;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// When TRUE, MFM will perform priority sorting of Update Files (i.e., when there are more
    /// Update Files to process than available RFD Clients, the order of processing is prioritized).
    /// The application must provide priority sorting callbacks to facilitate the sorting, in the
    /// *appCallbackInfo* parameter of MFM_Create().
    ///
    /// When FALSE, there is no priority ordering of MFM Update Files.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    BOOL isFilePrioritySortingEnabled;

} MFM_FILE_MANAGEMENT_POLICY_INFO_STRUCT;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Function Prototype for the "Is the Subname In Service" callback function.
///
/// An application implemented function of this prototype indicates whether File Items with the specified
/// Subname are In-Service (should be processed by MFM because the application wants these files)
/// or Out-of-Service (MFM should bypass processing of these files because the service
/// currently does not want them). If the application indicates the Subname is In-Service, then
/// it also provides the File Identity VersionTo number of the File Item with this Subname
/// that it already has in storage.
///
/// The application may change the In Service or Out of Service
/// status of File Items over time (e.g. some data services may allow the application to
/// use a subset of all broadcast File Items based on location or some other criteria).
///
/// @param   fileIdentitySubname            The MFM File Identity Subname identifying the File Item
///                                         for which the In-Service/Out-of-Service status is requested.
/// @param   appStoredFileIdentifyVersionToPtr (output) A pointer to an MFM File Identity
///                                         VersionTo number.
///                                         For a File Item (subname) identified as In-Service,
///                                         the application implemented callback outputs the MFM
///                                         File Identity VersionTo value of the File Item
///                                         the application has currently stored for this same Subname.
///                                         If the application currently does not have a File Item stored
///                                         for this Subname (e.g. for a Subname "expansion" type data service),
///                                         or this File Item is indicated as Out-of-Service, then the
///                                         application outputs the value MFM_FILE_IDENTITY_VERSION_NUM_UNUSED.
/// @param   mfmIsSubnameInServiceArg       The argument provided by the application to
///                                         associate with this callback function. The
///                                         application callback function will typically recast
///                                         this variable from a void data type to some
///                                         application specific data type per application design.
/// @return
/// TRUE if MFM File Item with the fileIdentitySubname subname is In-Service (MFM should process it).<BR>
/// FALSE if MFM File Item with the fileIdentitySubname subname is Out-Of-Service (MFM should NOT process it).
////////////////////////////////////////////////////////////////////////////////////////////////////
typedef BOOL (*MFM_APP_IS_SUBNAME_IN_SERVICE_CALLBACK_FCN) (
            TCHAR * subname,
            UINT32 * appStoredFileIdentifyVersionToPtr,
            void * mfmIsSubnameInServiceArg );

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Function Prototype for the File Identity Parser callback function.
///
/// An application implemented function of this prototype parses an input RFD Metadata Name field
/// to produce the corresponding MFM File Identity Info.
///
/// @param   rfdMetadataName                The RFD Update File Metadata Name field to be
///                                         parsed.
/// @param   fileIdentityInfoPtr            (output) A pointer to an MFM File Identity
///                                         structure. This application implemented callback
///                                         function must parses an input RFD Metadata Name field
///                                         to produce the corresponding MFM File Identity Info.
///                                         The function writes this File Identity Info into the
///                                         structure pointed to by fileIdentityInfoPtr.
/// @param   fileIdentityParseCallbackArg   The argument provided by the application to
///                                         associate with this callback function. The
///                                         application callback function will typically recast
///                                         this variable from a void data type to some
///                                         application specific data type per application design.
/// @return
/// TRUE if the MFM File Identity Info was successfully produced and output in *fileIdentityInfoPtr.<BR>
/// FALSE is parsing was Not successful, for example, if the Metadata Name did not match a general
/// format expected for supported data service.
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef BOOL (*MFM_FILE_IDENTITY_PARSE_CALLBACK_FCN) (
            const TCHAR rfdMetadataName[],
            MFM_FILE_IDENTITY_STRUCT * fileIdentityInfoPtr,
            void * fileIdentityParseCallbackArg );

// Priority Sort Callback Functions

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief  Function Prototype for the Priority Sort *Begin* callback function.
///
/// @param   filePrioritySortCallbackArg The argument provided by the application to associate
///                                      with this callback function. The application callback
///                                      function will typically recast this variable from a
///                                      void data type to some application specific data type
///                                      per application design.
/// @return
/// TRUE if application is ready to do the sort. FALSE if application not ready to do sort. For
/// example, for a distance based priority measurment, the application returns FALSE is current
/// GPS coordinates not yet available.
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef BOOL (*MFM_FILE_PRIORITY_SORT_BEGIN_CALLBACK_FCN) (
                void * filePrioritySortCallbackArg );

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief  Function Prototype for the Priority Sort *End* callback function.
///
/// @param   filePrioritySortCallbackArg The argument provided by the application to associate
///                                      with this callback function. The application
///                                      callback function will typically recast this
///                                      variable from a void data type to some application
///                                      specific data type per application design.
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef void (*MFM_FILE_PRIORITY_SORT_END_CALLBACK_FCN) (
                void * filePrioritySortCallbackArg );

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Function Prototype for the Priority Sort *Compare* callback function.
///
/// The Priority Sort *Compare* callback function facilitates the sorting of the processing order
/// of Update Files pending MFM/RFD processing. It compares the relative MFM priority of two
/// Update Files based on the parsed Subname strings of the two Update Files.
///
/// @param   subNameA                    The MFM Subname string for a 1st Update File labled as "A".
/// @param   subNameB                    The MFM Subname string for a 2nd Update File labled as "B".
/// @param   filePrioritySortCallbackArg The argument provided by the application to associate
///                                      with this callback function. The application
///                                      callback function will typically recast this
///                                      variable from a void data type to some application
///                                      specific data type per application design.
/// @return
/// Negative value: if priority(subNameA) is less than priority(subNameB).<BR>
/// 0             : if priority(subNameA) is equal priority(subNameB).<BR>
/// Positive value: if priority(subNameA) is greater than priority(subNameB).<BR>
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef int (*MFM_FILE_PRIORITY_COMPARE_CALLBACK_FCN) (
                const TCHAR * subNameA,
                const TCHAR * subNameB,
                void * filePrioritySortCallbackArg );

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// MFM File Priority Sort Callbacks Structure.
///
/// Specifies the application provided Priority Sorting functions that MFM calls-back to facilitate
/// priority sorting of Update Files.
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef struct {

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The callback that MFM calls at each *begining* of priority sorting of the Update Files pending
    /// processing.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_FILE_PRIORITY_SORT_BEGIN_CALLBACK_FCN filePrioritySortBeginFcn;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The callback that MFM calls at each *end* of priority sorting of the Update Files pending
    /// processing.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_FILE_PRIORITY_SORT_END_CALLBACK_FCN filePrioritySortEndFcn;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The callback that MFM calls during priority sorting that compares the relative priorities of
    /// two Update Files.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_FILE_PRIORITY_COMPARE_CALLBACK_FCN filePriorityCompareFcn;

} MFM_FILE_PRIORITY_SORT_CALLBACK_FUNCTIONS_STRUCT;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Function prototype of the application implemented MFM File-Transfer Callback.
///
/// The callback occurs when each new RFD Update File, associated with a corresponding Multifile
/// Manager instance, becomes available (is reconstructed by RFD). The application uses this
/// function to complete ownership transfer of the new Update File to the
/// application. The application provides this callback function to MFM on creation of the
/// Mulitfile Manager instance. This function along with a corresponding
/// *appRfdFileTransferCallbackArg* callback argument is provided under the parameter
/// *appCallbackInfo* of MFM_Create().
///
/// After return from this File-Transfer callback function, MFM will delete the storedFileNamePath
/// named Update File.
/// The typical operation of application is one of the following, depending of the nature of the
/// supported data service:
///
/// a) The application reads the Update File in place, and processes the Update File contents
///    per the supported data service requirements (e.g. update an application owned database
///    from contents of the Update File). The application must not modify the Update File in
///    this use case (only read).
///    If a power loss (or system abort) occurs during this Update File processing (or anytime during
///    the call to the MFM File Transfer callback), then on the next power-up, MFM/RFD will
///    retry the file transfer by another call to the MFM File Transfer callback.
///    The application may have already processed some or possibly all of the same Update File
///    content on the previous cycle, therefore, the application processing should be designed to
///    handle this case.
///
/// b) The application moves (renames) the Update File to a different application owned directory
///    for later processing by the application, in the context of another application owned
///    thread or process. The application is responsible for managing power loss resilience
///    once the Update file is moved (renamed). If power loss (or system abort) can occur during
///    the file move (rename), the developer should ensure that the platform file system is roubust
///    enoungh such that the file is not lost and not corrupted (it's either at the original
///    storedFileNamePath named location, or at the new moved-to location). An alternative
///    to moving/renaming the file would be to copy the file, but this results in double the
///    storage requirement for the file and generally requires more processing time.
///    As in case (a) above, if power loss occurs during this Update File processing (or anytime during
///    the call to the MFM File Transfer callback), then on the next power-up, MFM/RFD will
///    retry the file transfer by another call to the MFM File Transfer callback. The application
///    should be aware in this case that the Update File move (rename) may have already occurred
///    (e.g. application should check for presence of destination file).
///
/// While each MFM instance supports the concurrent processing of multiple RFD Update Files
/// (i.e. when the MFM instance is assigned multiple RFD Clients), an MFM instance will only
/// make one call to this MFM File Transfer callback (i.e. serialized file transfers to the
/// application).
///
/// @param   storedFileNamePath                 Full path and name string identifying location of
///                                             the new Update File being transferred.
/// @param   rfdMetadataFileInfo                Pointer to a RFD Metadata File Info structure
///                                             describing the new file.
/// @param   appRfdFileTransferCallbackArg      The argument provided by the application to
///                                             associate with this callback function. The
///                                             application passes this argument in parameter
///                                             *appRfdFileTransferCallbackArg* of the function
///                                             MFM_Create(). The application callback function
///                                             will typically recast this variable from a void
///                                             data type to some application specific data type
///                                             per application design.
/// @return
/// - TRUE to indicate the Update File was successfully transferred to application control.
/// - FALSE to indicate the Update File was not successfully transferred. In this case, MFM will
/// reset RFD collection and processing of this Update File to attempt reprocessing (MFM calls
/// RFD_ConsumerResetFileCollectionRequest()).
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef BOOL (*MFM_APP_RFD_FILE_TRANSFER_CALLBACK_FCN)(
    const TCHAR storedFileNamePath[],
    RFD_CONSUMER_FILE_INFO_STRUCT * rfdMetadataFileInfo,
    void * appRfdFileTransferCallbackArg );

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Function prototype of the application implemented MFM Get Time Callback.
///
/// This callback provides the current time, formated as a structure of type
/// RFD_SXI_TIME_INFO_STRUCT. MFM uses this time, along with the Update File Metadata *Life*
/// parameter to determine expiration status of Update Files. Without MFM access to the time
/// information, it is possible for some or all of the RFD Clients owned by an MFM instance to be
/// indefinately assigned to expired Update Files, thus indefinately blocking the processing of
/// other Update Files that are not expired.
///
/// The application callback should base this time on the time provided by the SXM Satellite
/// Radio Chipset. For SXM SXI based receivers, this is the time information provided by
/// the *TimeInd* SXI indication message. The application must *not* derive this time from a time
/// source that is subject to being set incorrectly or subject to not being set at all (e.g. time
/// info that might be set incorrectly by the user).
///
/// The SXM Satellite time source may sometimes be unavailable due to no or low signal contition
/// at the time of system powerup. In this case, the application callback should return -1 to
/// indicate that the UTC time is unavailable.
///
/// The application provides this callback function to MFM on creation of the Mulitfile Manager
/// instance. This function along with a corresponding *appGetTimeCallbackArg* callback argument
/// is provided under the parameter *appCallbackInfo* of MFM_Create().
///
/// @param  sxiTimeInfoPtr          (output) Pointer to the current time info structure. The
///                                 callback function populates the caller allocated structure
///                                 with the current time info for output to the caller.
/// @param  appGetTimeCallbackArg   The argument provided by the application to associate with
///                                 this callback function. The application passes this argument
///                                 in parameter *appCallbackInfo* of the function MFM_Create().
///                                 The application callback function will typically recast this
///                                 variable from a void data type to some application specific
///                                 data type per application design.
///
/// @return
/// - TRUE The current time is available and output in *sxiTimeInfo.
/// - FALSE The current time is not available, and not output to *sxiTimeInfo.
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef BOOL (*MFM_APP_GET_TIME_CALLBACK_FCN)(
    RFD_SXI_TIME_INFO_STRUCT * sxiTimeInfoPtr,
    void * appGetTimeCallbackArg );

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// Function prototype of the application implemented callback function that receives the in-use
/// and out- of-use status of RFD Block Message DMIs.
///
/// This callback is relevant for Data Services that utilize the RFD Update File Metadata Message
/// DMI field (e.g. Non-Nav Traffic Maps). The field indicates the DMI on which RFD Block
/// Messages are delivered for a corresponding Update File.
///
/// MFM invokes this callback to report when a new DMI goes into use or goes out of use. A DMI
/// goes into use when MFM begins processing an Update File that specifies a new Block Message
/// DMI (i.e. a DMI that is not specified for any of the other Update Files being actively
/// processed by MFM/RFD). A DMI goes out of use when MFM/RFD stops actively processing an Update
/// File that specifies a Block Message DMI that is *not* specified for any other Update File
/// being Actively processed (i.e. when the last Update File that specified the DMI value stops
/// being processed by MFM/RFD).
///
/// The application may use the DMI in-use/out-of-use status to specify/filter the set of DMIs
/// that are received from from the SXM Chipset (e.g. for the SXM SXI interface, using the
/// DataServiceFilterCmd command).
///
/// @param  rfdBlkMsgDmi                        The RFD Block Message DMI.
/// @param  isDmiInUse                          TRUE if the RFD Block Message DMI is in use by
///                                             MFM. FALSE if the RFD Block Message DMI is out
///                                             of use by MFM.
/// @param  appRfdBlkMsgDmiStatusCallbackArg    The argument provided by the application to
///                                             associate with this callback function. The
///                                             application passes this argument in parameter
///                                             *appCallbackInfo* of the function MFM_Create(). The
///                                             application callback function will typically recast
///                                             this variable from a void data type to some
///                                             application specific data type per application design.
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef void (*MFM_APP_RFD_BLK_MSG_DMI_STATUS_CALLBACK_FCN)(
    UINT16 rfdBlkMsgDmi,
    BOOL isRfdBlkDmiInUse,
    void * appRfdBlkMsgDmiStatusCallbackArg );

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// MFM Callback Info Structure.
///
/// Specifies all the application provided functions that MFM calls-back. Also specifies the
/// application provided argument values that is provided back to the application in each call-back.
///
/// An instance of this structure is input in MFM_Create() to configure the application callback
/// functions and arguments for a new Multifile Mangager instance.
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef struct {

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief  The application provided callback functions to facilitate Update-File Priority Sorting.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_FILE_PRIORITY_SORT_CALLBACK_FUNCTIONS_STRUCT mfmFilePriorityCallbackFcns;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The application provided argument that's passed in each MFM call of the Priority Sorting
    /// callbacks defined in *mfmFilePriorityCallbackFcns*.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    void * mfmFilePriorityCallbackArg;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The application provided callback function that parses the RFD Metadata Name field into the
    /// MFM File Identity Info.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_FILE_IDENTITY_PARSE_CALLBACK_FCN mfmFileIdentityParseCallbackFcn;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The application provided argument that's passed in each MFM call of the File Identity Parsing
    /// callback function *mfmFileIdentityParseCallbackFcn*.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    void * mfmFileIdentityParseCallbackArg;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The application provided callback function that transferes a newly processed RFD Update File
    /// from MFM/RFD control to application control (hand over of the Update File).
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_APP_RFD_FILE_TRANSFER_CALLBACK_FCN appRfdFileTransferCallbackFcn;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The application provided argument that's passed in each MFM call of the Update File Transfer
    /// callback function *appRfdFileTransferCallbackFcn*.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    void * appRfdFileTransferCallbackArg;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The application provided callback function that provides the current time.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_APP_GET_TIME_CALLBACK_FCN appGetTimeCallbackFcn;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The application provided argument that's passed in each MFM call of the Get Time function,
    /// *appGetTimeCallbackFcn*.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    void * appGetTimeCallbackArg;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The application implemented callback function that receives the in-use and out- of-use status
    /// of RFD Block Message DMIs.
    ///
    /// The application can set to NULL if it does not want to or need to use this callback.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_APP_RFD_BLK_MSG_DMI_STATUS_CALLBACK_FCN appBlkMsgDmiStatusCallbackFcn;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The application provided argument that's passed in each MFM call of the Block Message DMI Status
    /// function, *appBlkMsgDmiStatusCallbackFcn*.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    void * appBlkMsgDmiStatusCallbackArg;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The application implemented callback function that provides Subname In-Service/Out-of-Service
    /// status of a specified Subname.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    MFM_APP_IS_SUBNAME_IN_SERVICE_CALLBACK_FCN appIsSubnameInServiceCallbackFcn;

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief
    /// The application provided argument that's passed in each MFM call of the Is Subname In Service
    /// function, *appIsSubnameInServiceCallbackFcn*.
    ////////////////////////////////////////////////////////////////////////////////////////////////////

    void * appIsSubnameInServiceCallbackArg;

} MFM_APP_CALLBACK_INFO_STRUCT;

typedef struct {

    // activeFileItemList:
    // File Items in this list are in the active state, being processed by
    // an assigned RFD Client for each File Item.
    RFD_LLIST_HANDLE activeFileItemList;

    // waitingSubnameTitledFileItemListList:
    // File Items in this container are in waiting state, waiting for
    // RFD Clients to become available to move to active state.
    // This is a list of Subname organized File Item Lists.
    // Each File Item List contains File Items of the same Subname.
    RFD_LLIST_HANDLE waitingSubnameTitledFileItemListList;

    // Tracks if the Waiting List (waitingSubnameTitledFileItemListList) is
    // is currently sorted or not.
    BOOL isWaitingListCurrentlySorted;

} MFM_FILE_MANAGEMENT_LISTS_STRUCT, * MFM_FILE_MANAGEMENT_LISTS_HANDLE;

typedef struct {
    MFM_FILE_CONSUMER_HANDLE hMfmFileConsumer;
	RFD_MSG_QUEUE_HANDLE hMfmFileTransferMessageQueue;

} MFM_FILE_CONSUMER_INFO_STRUCT;

typedef struct {
    // rfdDefaultMessageConfigInfo: external reference, not allocated/freed by MFM.
    const RFD_MESSAGE_CONFIG_INFO * rfdDefaultMessageConfigInfo;

    // rfdUpdateFileMetadataInfo: reference allocated/freed by MFM
    RFD_FSET_MDATA_STRUCT * rfdUpdateFileMetadataInfo;

    BOOL isRfdUpdateFileMetadataCarouselComplete;

} MFM_RFD_MESSAGE_PROCESS_INFO_STRUCT;

typedef struct {
    // Last SXI time that File Item Expirations were checked/processed.
    // This allows Expiration Checks only to be performed if SXI time has changed
    // (avoid unneccessary processing).
    RFD_SXI_TIME_INFO_STRUCT lastExpProcessSxiTime;

    // Flag indicates an existing File Item Life field was updated (that may now be expired)
    // and that the next Expiration Processing should occur immediately instead of
    // on the next Current SXI Time change.
    BOOL isLifeUpdatedSinceLastExpProcessTime;

} FILE_ITEM_RFD_FILE_EXPIRATION_PROCESS_INFO_STRUCT;

// the main manager structure
typedef struct multifile_manager_struct {

    // lists that manage MFM file items (active, waiting and stored lists)
    MFM_FILE_MANAGEMENT_LISTS_STRUCT fileManagementLists;

    // lists that manage the RFD clients (busy and free client lists)
    MFM_RFD_CLIENT_RESOURCE_INFO_STRUCT rfdClientResourceInfo;

    // List containing the Block Message DMIs for File Items that are currently being processed
    // (are Active File Items).
    RFD_LLIST_HANDLE activeDmiList;

    // info related to RFD message processing
    MFM_RFD_MESSAGE_PROCESS_INFO_STRUCT mfmRfdMessageProcessInfo;

    // callbacks to the application
    MFM_APP_CALLBACK_INFO_STRUCT appCallbackInfo;

    // MFM file management policy settings
    MFM_FILE_MANAGEMENT_POLICY_INFO_STRUCT fileManagementPolicyInfo;

    // info for managing and communicating with the MFM file consumer (thread)
    MFM_FILE_CONSUMER_INFO_STRUCT fileConsumerInfo;

    // a client Id that the application can assign to a MFM instance on creation.
    UINT16 mfmClientId;
    // a client name string that the application can assign to a MFM instance on creation.
    TCHAR * mfmClientName;

    FILE_ITEM_RFD_FILE_EXPIRATION_PROCESS_INFO_STRUCT fileItemRfdLifeExpirationProcessInfo;

    // MFM mutex to support concurrent MFM access from the app.
    RFD_MUTEX_HANDLE hMutex;

} MULTIFILE_MANAGER_STRUCT, * MULTIFILE_MANAGER_HANDLE;

// Structure posted from MFM consumer to MFM manager
typedef struct {
    // The rfd consumer handle upon which the file was received by the consumer thread
    RFD_CONSUMER_THREAD_DATA_HANDLE rfdFileConsumerHandle;
    // (the info stucture is stored here, not the pointer to structure).
    RFD_CONSUMER_FILE_INFO_STRUCT rfdConsumerFileInfo;

} MFM_RFD_FILE_TRANSFER_INFO_POST_STRUCT, * MFM_RFD_FILE_TRANSFER_INFO_POST_HANDLE;

typedef RFD_STATUS (*MFM_FILE_ITEM_EXPIRATION_CRITERION_FCN) (
                        MULTIFILE_MANAGER_HANDLE hManager,
                        MFM_FILE_ITEM_HANDLE hFileItem,
                        BOOL * isExpiredPtr,
                        void * expirationCriterionArg );

///////////////////////////////////////////////////////////////////////////////

RFD_STATUS MFM_CreateExt(
                MULTIFILE_MANAGER_HANDLE * mfmHandlePtr,
                UINT16 mfmClientId,
                const TCHAR * mfmClientName,
                MFM_RFD_CLIENT_STRUCT mfmRfdClientArray[],
                UINT16 numRfdClients,
                MFM_APP_CALLBACK_INFO_STRUCT * appCallbackInfo,
                MFM_FILE_MANAGEMENT_POLICY_INFO_STRUCT * fileManagementPolicyInfo,
                INT32 fileReadyPollIntervalMillisec,
                BOOL isAltSingleThreadMode );

RFD_STATUS MFM_Create(
                MULTIFILE_MANAGER_HANDLE * mfmHandlePtr,
                UINT16 mfmClientId,
                const TCHAR * mfmClientName,
                MFM_RFD_CLIENT_STRUCT mfmRfdClientArray[],
                UINT16 numRfdClients,
                MFM_APP_CALLBACK_INFO_STRUCT * appCallbackInfo,
                MFM_FILE_MANAGEMENT_POLICY_INFO_STRUCT * fileManagementPolicyInfo );

void MFM_Delete( MULTIFILE_MANAGER_HANDLE hMultifileManager );

RFD_STATUS MFM_PutRfdMessage(
    MULTIFILE_MANAGER_HANDLE mfmHandle,
    UCHAR * messageBuf,
    UINT32 messageSize,
    UINT16 blkMessageSourceDmi );

RFD_STATUS MFM_UpdateFilePriorityCallbackArg(
                MULTIFILE_MANAGER_HANDLE mfmHandle,
                void * newArg,
                void ** oldArgPtr );

RFD_STATUS MFM_ErrorCheck( MULTIFILE_MANAGER_HANDLE mfmHandle, INT32 timeOutMilliSeconds );

void MFM_UTIL_PrintFileItemIdentity(
        MFM_FILE_IDENTITY_HANDLE hFileIdentity,
        const TCHAR * headerName1,
        const TCHAR * headerName2,
        const TCHAR * headerName3 );

int MFM_UTIL_CompareFileIdentity(
               MFM_FILE_IDENTITY_HANDLE hFileIdentityA,
               MFM_FILE_IDENTITY_HANDLE hFileIdentityB,
               MFM_FILE_IDENTITY_MASK compareMask );

void MFM_UTIL_UpdateFileIdentity(
           MFM_FILE_IDENTITY_HANDLE hFileIdentityDst,
           MFM_FILE_IDENTITY_HANDLE hFileIdentitySrc,
           MFM_FILE_IDENTITY_MASK updateMask );

void MFM_UTIL_SetFileIdentity(
           MFM_FILE_IDENTITY_HANDLE hFileIdentity,
           const TCHAR * subname,
           UINT32 versionFrom,
           UINT32 versionTo );

RFD_STATUS MFM_Run(
            MULTIFILE_MANAGER_HANDLE mfmHandle,
            BOOL *isThreadExitRequestedPtr );

// doxygen group end symbol, don't remove.
/////////////////////////////////////////////////////////////////////////////////////////////
/// @}  doxygen group end symbol, don't remove.
////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef __cplusplus
}
#endif

#endif // RFD_MULTIFILE_MANAGER_H