////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	multifile_manager\mfm_util_fixed_field_identity_parser.c
///
/// @brief	The fixed field file identity parser mfm utilitiy class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2013 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd.h"
#include "rfd_receiver.h"
#include "rfd_msg_collector.h"
#include "rfd_file_consumer.h"
#include "rfd_linked_list.h"
#include "mfm_consumer.h"
#include "multifile_manager.h"
#include "mfm_util_fixed_field_identity_parser.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// An instance of a Fixed Field File Identity Parser Config structure configured for the Non-
/// Navigation Traffic Maps Data Service.
////////////////////////////////////////////////////////////////////////////////////////////////////

const MFM_FF_FILE_IDENTITY_PARSER_CONFIG_STRUCT gNonNavTrafficMfmFixedFieldFileIdentityParserConfig = {
    NULL,     // commonNameString - none defined, set to NULL
    { 0,0 },  // commonNameRange  - none defined, set range to {0,0}
    { 0,11 }, // subnameRange
    { 0,0 },  // versionFromRange - none defined, set range to {0,0}
    { 11,3 }, // versionToRange   - versionTo serves as the single version field.
    14        // fixedNameLen subName_field + versionTo_field = 11+3 = 14
    //'-'       // defaultFillChar (used for regenerating original Name from file identity).

};

////////////////////////////////////////////////////////////////////////////////////////////////////
// Public Functions
////////////////////////////////////////////////////////////////////////////////////////////////////

// doxygen group start symbol, don't remove.
///////////////////////////////////////////////////////////////////////////////////////////////////
/// @addtogroup MfmAPI RFD Multifile Manager (MFM) API
/// @{
///////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// A generalized *RFD Metadata Name* to *MFM File Identity* parser, supporting RFD Metadata Name
/// formats that are constrained to *fixed length* fields.
///
/// For Data Services specifying RFD Metadata Name fields comprised of fixed length MFM File
/// Identity fields, the MFM application may utilize this parser function as the *MFM File
/// Identity Parser* callback that is configured on call to MFM_Create(). The specific File
/// Identity field locations and lengths within the RFD Metadata Name string are specified by the
/// nameParserConfig structure which is passed by reference to this parser function. The
/// application configures a parser config structure of type
/// MFM_FF_FILE_IDENTITY_PARSER_CONFIG_STRUCT to specify the MFM File Identity fields for a
/// particular Data Service. This structure is passed by reference to this parser function as
/// parameter nameParserConfig.
///
/// In calling MFM_Create(), the application passes pointers to the configured
/// MFM_FF_FILE_IDENTITY_PARSER_CONFIG_STRUCT structure and the
/// MFM_UTIL_FixedFieldFileIdentityParser() function in parameters appCallbackInfo-
/// >mfmFileIdentityParseCallbackArg and appCallbackInfo->mfmFileIdentityParseCallbackFcn
/// respectively.
///
/// @param  rfdMetadataName     The RFD Metadata Name string.
/// @param  fileIdentityInfo    Pointer to the MFM File Identity Info structure that is populated
///                             by this function.
/// @param  nameParserConfig    Pointer to the Parser Config structure that specifies the format
///                             of the RFD Metadata Name string that is being parsed.
///
/// @return
/// - TRUE if the MFM File Identity Info was successfully produced and output in *fileIdentityInfo.
/// - FALSE is parsing was Not successful.
////////////////////////////////////////////////////////////////////////////////////////////////////

BOOL MFM_UTIL_FixedFieldFileIdentityParser (
            const TCHAR rfdMetadataName[],
            MFM_FILE_IDENTITY_STRUCT * fileIdentityInfo,
            const MFM_FF_FILE_IDENTITY_PARSER_CONFIG_STRUCT * nameParserConfig )
{
    BOOL isParseSuccessful = FALSE;
    UINT16 startCharOffset, numChars;
    int compareResult;
    TCHAR * strPtr = NULL;

    do { // once

        ////////////////////////////////////////////////////////////////////////
        // Check that string length matches expected length.
        // Skip check if fixedNameLen == 0, signalling that variable name length
        // is permitted
        ////////////////////////////////////////////////////////////////////////

        if(nameParserConfig->fixedNameLen != 0) {

            size_t nameLen;

            nameLen = RFD_STRNLEN( rfdMetadataName, MFM_SUBNAME_MAX_LEN);

            if(nameLen != nameParserConfig->fixedNameLen) {
                // break out to report mismatch.
                break;
            }
        }

        ////////////////////////////////////////////////////////////////////////
        // Check that the CommonName field matches expected CommonName,
        // if this feature is enabled.
        ////////////////////////////////////////////////////////////////////////

        if(nameParserConfig->commonNameString != NULL &&
            nameParserConfig->commonNameRange.numChars != 0) {

            startCharOffset = nameParserConfig->commonNameRange.startOffset;
            numChars = nameParserConfig->commonNameRange.numChars;

            compareResult = RFD_STRNCMP(
                                &rfdMetadataName[startCharOffset],
                                nameParserConfig->commonNameString,
                                numChars);

            if(compareResult != 0) {
                // range of chars doesn't match expected common name string,
                // break out to report mismatch.
                break;
            }
        }
        // else this feature is not used, continue parsing other fields.

        ////////////////////////////////////////////////////////////////////////
        // Extract the VersionFrom field
        ////////////////////////////////////////////////////////////////////////

        startCharOffset = nameParserConfig->versionFromRange.startOffset;
        numChars = nameParserConfig->versionFromRange.numChars;

        if(numChars != 0) {

            // use fileIdentityInfo->subname as temp buffer
            strPtr = fileIdentityInfo->subname;

            // copy the defined range of chars to temp string buffer.
            RFD_CopyCharacterRangeToString( strPtr, rfdMetadataName, startCharOffset, numChars );

            // do string to integer conversion
            fileIdentityInfo->versionFrom = RFD_ATOI( strPtr );
        }
        else {
            // feature not used, set to default 0.
            fileIdentityInfo->versionFrom = 0;
        }

       ////////////////////////////////////////////////////////////////////////
        // Extract the VersionTo field
        ////////////////////////////////////////////////////////////////////////

        startCharOffset = nameParserConfig->versionToRange.startOffset;
        numChars = nameParserConfig->versionToRange.numChars;

        if(numChars != 0) {

            // use fileIdentityInfo->subname as temp buffer
            strPtr = fileIdentityInfo->subname;

            // copy the defined range of chars to temp string buffer.
            RFD_CopyCharacterRangeToString( strPtr, rfdMetadataName, startCharOffset, numChars );

            // do string to integer conversion
            fileIdentityInfo->versionTo = RFD_ATOI( strPtr );
        }
        else {
            // feature not used, set to default 0.
            fileIdentityInfo->versionTo = 0;
        }

        ////////////////////////////////////////////////////////////////////////
        // Extract the Subname field
        // The fileIdentityInfo->subname buffer is used as a temporary buffer
        // for parsing of other fields, so update this field last.
        ////////////////////////////////////////////////////////////////////////

        startCharOffset = nameParserConfig->subnameRange.startOffset;
        numChars = nameParserConfig->subnameRange.numChars;

        strPtr = fileIdentityInfo->subname;

        // copy the defined range of chars to Subname string.
        RFD_CopyCharacterRangeToString( strPtr, rfdMetadataName, startCharOffset, numChars );

        ////////////////////////////////////////////////////////////////////////
        // Parsing is successful if we make it here.
        ////////////////////////////////////////////////////////////////////////
        isParseSuccessful = TRUE;

    } while(FALSE);

    return isParseSuccessful;
}

// doxygen group end symbol, don't remove.
/////////////////////////////////////////////////////////////////////////////////////////////
/// @}  doxygen group end symbol, don't remove.
////////////////////////////////////////////////////////////////////////////////////////////////
