////////////////////////////////////////////////////////////////////////////////////////////////////
/// @file	receiver_app\mfm_consumer.c
///
/// @brief	multifile manager file consumer class.
///
/// @remarks	Sirius XM Reliable File Delivery (RFD) SDK
///
/// @remarks	Copyright (c) 2013 Sirius XM Radio, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////

#include "rfd_receiver.h"
#include "rfd_file_consumer.h"
#include "rfd_linked_list.h"
#include "mfm_consumer.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_FileConsumerOpenLL(
    MFM_FILE_CONSUMER_HANDLE hConsumer,
    RFD_STATUS * statusPtr,
    BOOL *isThreadExitRequestedPtr
        )
{
	DWORD rfdClientIdNo;
	DWORD rfdClientIndex;
    RFD_STATUS status = RFD_STATUS_OK;
	BOOL isThreadExitRequested;
	RFD_EVENT_HANDLE errorExitEvent;
    RFD_LLIST_NODE_HANDLE hListNode;
    RFD_LLIST_HANDLE rfdFileConsumerHandleList;
	RFD_CONSUMER_THREAD_DATA_HANDLE rfdFileConsumerHandle;

    rfdFileConsumerHandleList = hConsumer->rfdFileConsumerHandleList;

	errorExitEvent = hConsumer->errorExitEvent;

    isThreadExitRequested = FALSE;

	//////////////////////////////////////////
    // For each RFD Client this Consumer Thread is monitoring:
	/////////////////////////////////////////

    status = RFD_LLIST_First(
                    rfdFileConsumerHandleList,
                    &hListNode );

    while(hListNode != NULL && status == RFD_STATUS_OK) {

        rfdFileConsumerHandle = RFD_LLIST_GetUserData(hListNode, &status);
        if(status != RFD_STATUS_OK) {
            break;
        }

       	rfdClientIdNo = rfdFileConsumerHandle->clientInfo->clientIdNo;
	    rfdClientIndex = rfdFileConsumerHandle->clientInfo->clientIndex;

        //////////////////////////////////////////
	    // Initialize Storage for received RFD files.
	    // i.e. dir created if it doesn't exist, and
	    // scan dir to cleanup any incomplete files and
	    // prepare to retransfer any unfinished file transfers.
	    /////////////////////////////////////////

        RFD_DPrint( TEXT("MfmConsumerHandle 0x%x, Initializing Consumer Storage,  Client Index %d, Client Id %d\n"),
            hConsumer, rfdClientIndex, rfdClientIdNo);

	    status = RFD_ConsumerInitializeStorage(rfdFileConsumerHandle);
        if(status != RFD_STATUS_OK) {
            RFD_DPrint(
                TEXT("MfmConsumerHandle 0x%x, error RFD_ConsumerInitializeStorage(), status %d,  Client Index %d, Client Id %d\n"),
                hConsumer, status, rfdClientIndex, rfdClientIdNo);
            break;
        }

        status = RFD_LLIST_Next(
                        rfdFileConsumerHandleList,
                        &hListNode );

        if(status != RFD_STATUS_OK) {
            break;
        }
    }

	if(status != RFD_STATUS_OK) {
		// Unrecoverable error, signal error to manager and exit thread.
		RFDR_SignalErrorExit(errorExitEvent);
        isThreadExitRequested = TRUE;
	    RFD_DPrint( TEXT("MFM Consumer Error, MfmConsumerHandle 0x%x, status %d\n"), hConsumer);

	}

    RFD_DPrint( TEXT("MfmConsumerHandle 0x%x, the specified New File Ready Polling Interval is %d msec\n"),
        hConsumer, hConsumer->fileReadyPollIntervalMillisec );

	RFD_DPrint( TEXT("MFM Consumer Started, MfmConsumerHandle 0x%x\n"), hConsumer);

    *isThreadExitRequestedPtr = isThreadExitRequested;
    *statusPtr = status;

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_FileConsumerCloseLL(
    MFM_FILE_CONSUMER_HANDLE hConsumer,
    RFD_STATUS * statusPtr
        )
{
	/////////////////////////
	// Shutdown - exit thread. Cleanup and return.
	/////////////////////////

	RFD_DPrint( TEXT("MFM Consumer: exiting thread, MfmConsumerHandle 0x%x\n"), hConsumer);

    *statusPtr = RFD_STATUS_OK;

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief
/// The Consumer File-Transfer Callback function.
///
/// This function is passed in the call to RFD_ConsumerTransferNewFileExt() to receive
/// the newly available file.
///
/// @param  newFileName   Full path and name string identifying location of the new file being
///                       transferred.
/// @param  fileInfoPtr   Pointer to a file info structure describing the new file.
/// @param  callbackArg   The argument provided by the application to associate with this callback
///                       function. This is the same argument that is passed in parameter
///                       \e consumerFileTransferCallbackArg of the function
///                       RFD_ConsumerTransferNewFileExt().
////////////////////////////////////////////////////////////////////////////////////////////////////

static void _ConsumerFileTransferCallback(
        const TCHAR newFileName[],
        RFD_CONSUMER_FILE_INFO_STRUCT * newFileInfo,
        void *callbackArg
            )
{
	DWORD rfdClientIdNo;
    MFM_CONSUMER_FILE_TRANSFER_CALLBACK_ARG_STRUCT * mfmConsumerCallbackInfo;
    MFM_MANAGER_FILE_TRANSFER_CALLBACK_FCN mfmManagerFileTransferCallbackFcn;
    void * mfmManagerFileTransferCallbackArg;
    RFD_CONSUMER_THREAD_DATA_HANDLE rfdFileConsumerHandle;

    if(callbackArg == NULL || newFileName == NULL || newFileInfo == NULL) {
        // error
        return;
    }

    // Recast the callback arg to the appropriate data type.
    mfmConsumerCallbackInfo = (MFM_CONSUMER_FILE_TRANSFER_CALLBACK_ARG_STRUCT *) callbackArg;

    mfmManagerFileTransferCallbackFcn =
        mfmConsumerCallbackInfo->hConsumer->mfmManagerFileTransferCallbackFcn;

    mfmManagerFileTransferCallbackArg =
        mfmConsumerCallbackInfo->hConsumer->mfmManagerFileTransferCallbackArg;

	rfdFileConsumerHandle = mfmConsumerCallbackInfo->rfdFileConsumerHandle;

	rfdClientIdNo = rfdFileConsumerHandle->clientInfo->clientIdNo;

	RFD_DPrint( TEXT("\n"));
	RFD_DPrint( TEXT("MFM +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n"));
	RFD_DPrint( TEXT("MFM +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n"));
	RFD_DPrint( TEXT("MFM Consumer:  New File Received, RFDR Client Id %d\n"), rfdClientIdNo);
	RFD_DPrint( TEXT("...MFM         File Local Name       : %s\n"), newFileName);
	RFD_DPrint( TEXT("...MFM         File Size             : %d\n"), newFileInfo->fileSize);
	RFD_DPrint( TEXT("...MFM         File LifeTime         : %d\n"), newFileInfo->lifeTime);
	RFD_DPrint( TEXT("...MFM         File RFD Metadata Name: %s\n"), newFileInfo->name);
	RFD_DPrint( TEXT("MFM +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n"));
	RFD_DPrint( TEXT("MFM +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n"));
	RFD_DPrint( TEXT("\n"));

    ////////////////////////////////////////////////////////////////////////////////////
    // Call the MFM Manager File Transfer Callback
    ////////////////////////////////////////////////////////////////////////////////////
    mfmManagerFileTransferCallbackFcn(
            rfdFileConsumerHandle,
            newFileName,
            newFileInfo,
            mfmManagerFileTransferCallbackArg );

    mfmConsumerCallbackInfo->isFileProcessed = TRUE;

    return;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
RFD_STATUS MFM_FileConsumerRunLL(
    MFM_FILE_CONSUMER_HANDLE hConsumer,
    RFD_STATUS * statusPtr,
    BOOL *isThreadExitRequestedPtr
        )
{
	DWORD rfdClientIdNo;
	DWORD rfdClientIndex;
	RFD_STATUS status = RFD_STATUS_OK;
	BOOL isThreadExitRequested;
	RFD_EVENT_HANDLE errorExitEvent;
    RFD_LLIST_NODE_HANDLE hListNode;
    RFD_LLIST_HANDLE rfdFileConsumerHandleList;
	RFD_CONSUMER_THREAD_DATA_HANDLE rfdFileConsumerHandle;

    rfdFileConsumerHandleList = hConsumer->rfdFileConsumerHandleList;

	errorExitEvent = hConsumer->errorExitEvent;

	isThreadExitRequested = FALSE;

	do { // once

		//////////////////////////////////////////
		// Check for Shutdown request event
        // - Implement the RFDR_FILE_SET_READY_TIMEOUT_MSEC wait time here
        //   instead of on the possible multiple calls to RFD_ConsumerWaitForNewFileEvent().
		/////////////////////////////////////////

		status = RFD_WaitForEvent( hConsumer->exitRequestEvent,    // handle to the shutdown event
								   hConsumer->fileReadyPollIntervalMillisec );
		if(status == RFD_STATUS_OK) {
			RFD_DPrint( TEXT("MFM Consumer: Shutdown Request Event Occurred, MfmConsumerHandle 0x%x\n"), hConsumer);
			isThreadExitRequested = TRUE;
			break;
		}

	    //////////////////////////////////////////
        // For each RFD Client this Consumer Thread is monitoring:
	    /////////////////////////////////////////

        // Start list iteration
        status = RFD_LLIST_First(
                        rfdFileConsumerHandleList,
                        &hListNode );

        while(hListNode != NULL && status == RFD_STATUS_OK) {

            rfdFileConsumerHandle = RFD_LLIST_GetUserData(hListNode, &status);
            if(status != RFD_STATUS_OK) {
                break;
            }

       	    rfdClientIdNo = rfdFileConsumerHandle->clientInfo->clientIdNo;
	        rfdClientIndex = rfdFileConsumerHandle->clientInfo->clientIndex;

		    //////////////////////////////////////////
		    // Check for File Set Ready event
		    /////////////////////////////////////////

		    status = RFD_ConsumerWaitForNewFileEvent( rfdFileConsumerHandle,    // handle to the file set ready event
													  0);  // 0 timeout since we're polling multiple clients.
		    if(status == RFD_STATUS_OK) {

                MFM_CONSUMER_FILE_TRANSFER_CALLBACK_ARG_STRUCT fileTransferCallbackInfo;
                fileTransferCallbackInfo.hConsumer = hConsumer;
                fileTransferCallbackInfo.rfdFileConsumerHandle = rfdFileConsumerHandle;

		        //////////////////////////////////////////
		        // Take ownership of newly available file by call to RFD_ConsumerTransferNewFileExt(). The
		        // passed callback function _ConsumerFileTransferCallback() will actaully receive the newly
		        // available file.
		        /////////////////////////////////////////

                fileTransferCallbackInfo.isFileProcessed = FALSE;

			    status = RFD_ConsumerTransferNewFileExt(
                                                rfdFileConsumerHandle,
											    _ConsumerFileTransferCallback,
											    (void*) &fileTransferCallbackInfo);
			    if(status == RFD_STATUS_OK) {

		            /////////////////////////////////////////
		            // _ConsumerFileTransferCallback() will update isFileProcessed
		            // status if callback is successfully called.
		            /////////////////////////////////////////
                    if(fileTransferCallbackInfo.isFileProcessed == TRUE) {
                        RFD_DPrint( TEXT("MfmConsumerHandle 0x%x, RFDR Client Id %d: A New File was successfully transferred\n"),
                            hConsumer, rfdClientIdNo);
                    }
                    else {
                        RFD_DPrint( TEXT("MfmConsumerHandle 0x%x, RFDR Client Id %d: A New File was signalled ready, but was not actually available\n"),
                            hConsumer, rfdClientIdNo);
                    }
                }
			    else {
#if 1
				    // Unrecoverable error, signal error to manager and exit processing loop to exit thread.
				    RFD_DPrint( TEXT("MfmConsumerHandle 0x%x, RFDR Client Id %d: error in RFD_ConsumerTransferNewFileExt(), status = %d\n"),
                        hConsumer, rfdClientIdNo, status);
				    RFDR_SignalErrorExit(errorExitEvent);
				    isThreadExitRequested = TRUE;
				    break;
#else
                    // By design, an error should never occur here. But if some anomalous/unexpected error does occur,
                    // it is best handled by stopping, closing, then reopening and restarting the RFD Receiver.
                    // For implementations in which this error recovery action is not possible, an alternate action
                    // is to request reset of the current file collection process:
                    RFD_ConsumerResetFileCollectionRequest(rfdFileConsumerHandle);
#endif
			    }
		    }
		    else if(status != RFD_STATUS_OBJECT_TIMEOUT) {
			    // Unrecoverable error, break from loop
			    break;
		    }
		    // else timeout.

            // Get Next node from List
            status = RFD_LLIST_Next(
                            rfdFileConsumerHandleList,
                            &hListNode );

            if(status != RFD_STATUS_OK) {
                // completed full iteration of list
                break;
            }

            // Check for exit request (with 0 timeout) after each individual list entry.
            status = RFD_WaitForEvent(
                            hConsumer->exitRequestEvent,    // handle to the shutdown event
						    0);

		    if(status == RFD_STATUS_OK) {
			    RFD_DPrint( TEXT("MFM Consumer: Shutdown Request Event Occurred, MfmConsumerHandle 0x%x\n"), hConsumer);
			    isThreadExitRequested = TRUE;
			    break;
		    }
		    else if(status == RFD_STATUS_OBJECT_TIMEOUT) {
                // timeout, that's normal. Set status back to OK to avoid error exit.
                status = RFD_STATUS_OK;
            }

        } // list iterate loop

	    if(status != RFD_STATUS_OK) {
		    // Unrecoverable error, signal error to manager and exit thread.
		    RFDR_SignalErrorExit(errorExitEvent);
            isThreadExitRequested = TRUE;
	        RFD_DPrint( TEXT("MFM Consumer Error, MfmConsumerHandle 0x%x, status %d\n"), hConsumer, status);
	    }

	} while (FALSE);

    *isThreadExitRequestedPtr = isThreadExitRequested;

    if(status == RFD_STATUS_OBJECT_TIMEOUT) {
        *statusPtr = RFD_STATUS_OK;
    }
    else {
        *statusPtr = status;
    }

    return RFD_STATUS_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief  Entry point of the Application Consumer thread.
////////////////////////////////////////////////////////////////////////////////////////////////////
DWORD RFD_PLATFORM_API MFM_FileConsumer( LPVOID lpParam )
{
	MFM_FILE_CONSUMER_HANDLE hConsumer;
    RFD_STATUS status;
    BOOL isThreadExitRequested;

	hConsumer = (MFM_FILE_CONSUMER_HANDLE) lpParam;

    ////////////////////////////////
    // Open (create and initialize)
    ////////////////////////////////
    MFM_FileConsumerOpenLL(
        hConsumer,
        &status,
        &isThreadExitRequested );

    ////////////////////////////////
    // Run loop
    ////////////////////////////////
    while(!isThreadExitRequested) {

        MFM_FileConsumerRunLL(
            hConsumer,
            &status,
            &isThreadExitRequested );
    }
    // Run loop exit

    ////////////////////////////////
    // Close (cleanup)
    ////////////////////////////////
    MFM_FileConsumerCloseLL(
        hConsumer,
        &status );

    // Thread exit
    return 0;
}

