/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *       This module will contain all the OS sem API implementations.
 * The implementations in this module are 100% POSIX compliant making
 * it suitable for any POSIX.1b (POSIX 1003.1b-1993) compliant OS.
 *
 ******************************************************************************/

// POSIX and standard includes
#define _GNU_SOURCE
#include <pthread.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#define OSAL_TRACE_ENABLE 0
#include "osal_trace.h"

#include "standard.h"
#include "osal.h"

#include "os_version.h"
#include "os.h"
#include "os_sem_.h"
#include "_os_sem_.h"

/*******************************************************************************
 *
 *   OS_eSemCreate
 *
 * Note: Consideration for the number of resources 'un32Resources' is not
 * required as it is handled in the OSAL common code.
 *
 ******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eSemCreate(OSAL_OBJECT_HDL *phSemObj,
    const char *pacName, UN32 un32InitialValue, UN32 un32Resources,
    UN32 un32Options)
{
    int status;
    OS_SEM_STRUCT *psSem = (OS_SEM_STRUCT *) OS_INVALID_OBJECT_HDL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    TRACE_START();

    // Verify initial value
    if ((un32Options & OSAL_SEM_OPTION_MUTEX) == OSAL_SEM_OPTION_MUTEX)
    {
        // only mutex supports priority inheritance
        if ((un32InitialValue > un32Resources) || (un32Resources != 1))
        {
            TRACE_END();
            return OSAL_ERROR_INVALID_INPUT;
        }
    } else if (un32InitialValue > un32Resources)
    {
        // Error! initial value exceeds number of resources
        TRACE_END();
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Allocate memory for the semaphore object
    psSem = (OS_SEM_STRUCT *) OSALC_pvMemoryAllocate(
        OSAL_NAME_PREFIX"Sem Object", sizeof(OS_SEM_STRUCT), FALSE);
    if (psSem == NULL)
    {
        // Error!
        TRACE_END();
        return OSAL_ERROR_OUT_OF_MEMORY;
    }

    do
    {
        pthread_mutexattr_t tMutexAttr;

        // Set the initial value of the semaphore.
        // The initial value of the semaphore is a positive value
        // (i.e. greater than zero) indicates an unlocked semaphore,
        // and a value of 0 (zero) indicates a locked semaphore.
        psSem->un32Resources = un32Resources; // Total being managed
        psSem->un32Available = un32InitialValue; // Number available
        psSem->un32Waiters = 0; // Number of waiting threads
        psSem->bDestroy = FALSE; // Object exists
        psSem->un32Options = un32Options; // save options

        // Initialize the mutex attr
        pthread_mutexattr_init(&tMutexAttr);

        // Set the type explicitly to NORMAL since the DEFAULT behavior
        // is undefined.
        pthread_mutexattr_settype(&tMutexAttr, PTHREAD_MUTEX_NORMAL);

#if (defined __QNX__) || (defined __INTEGRITY)
        if ((un32Options & OSAL_SEM_OPTION_MUTEX) == OSAL_SEM_OPTION_MUTEX)
        {
            // Don't rely on default behavior. The priority inversion is
            // a sort of critical issue for Real-time OS and this option
            // has to be set explicitly to achieve desired behavior.
            pthread_mutexattr_setprotocol(&tMutexAttr, PTHREAD_PRIO_INHERIT);
        }
#endif

        // Initialize mutex used for condition variable.
        status = pthread_mutex_init(&psSem->mutex, &tMutexAttr);

        pthread_mutexattr_destroy(&tMutexAttr);

        if (status != 0)
        {
            // Error!
            print_err(status, "Error! pthread_mutex_init()");
            OSALC_vMemoryFree(psSem);
            break;
        }

        // only use the condvar when not a priority inheriting mutex...
        if ((un32Options & OSAL_SEM_OPTION_MUTEX) != OSAL_SEM_OPTION_MUTEX)
        {
            // Initialize the attribute object for creating condition variables.
#ifndef ANDROID
            pthread_condattr_init(&psSem->cond_attr);
#else
            // Bionic does not have definition for pthread_condattr_init()
            // so just directly putting 0 there.
            psSem->cond_attr = 0;
#endif

            /*      This is not supported in Android.
             Using pthread_cond_timedwait_monotonic_np() instead
             */
#if !(defined ANDROID) && !(defined __INTEGRITY)
            // Set the clock attribute of a condition-variable attribute object.
            status = pthread_condattr_setclock(&psSem->cond_attr, OS_SEM_CLOCK);
            if (status != 0)
            {
                // Error!
                print_err(status, "Error! pthread_condattr_setclock()");
                OSALC_vMemoryFree(psSem);
                break;
            }
#endif
            // Initialize condition variable
            status = pthread_cond_init(&psSem->cond, &psSem->cond_attr);
            if (status != 0)
            {
                // Error!
                print_err(status, "Error! pthread_cond_init()");
                OSALC_vMemoryFree(psSem);
                break;
            }
        }

        if ((un32Options & OSAL_SEM_OPTION_MUTEX) == OSAL_SEM_OPTION_MUTEX)
        {
            if (un32InitialValue == 0)
            {
                // Initially locked.
                status = pthread_mutex_lock(&psSem->mutex);
                if (status != 0)
                {
                    // Error!
                    print_err(status, "Error! pthread_mutex_lock()");
                    OSALC_vMemoryFree(psSem);
                    break;
                }

                psSem->un32Available = 0;
            }

        }

        // Return object handle to caller
        *phSemObj = (OSAL_OBJECT_HDL) psSem;

        // All is well
        eReturnCode = OSAL_SUCCESS;

    } while (FALSE);

    TRACE_END();
    return eReturnCode;
}

/*******************************************************************************
 *
 *   OS_eSemDelete
 *
 ******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eSemDelete(OSAL_OBJECT_HDL hSemObj)
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    TRACE_START();
    do
    {
        int status;
        OS_SEM_STRUCT *psSem = (OS_SEM_STRUCT *) hSemObj;

        // Check input for validity
        if (hSemObj == OSAL_INVALID_OBJECT_HDL)
        {
            // Error!
            eReturnCode = OSAL_ERROR_INVALID_HANDLE;
            break;
        }

        // only signal cond if not a priority inheriting mutex
        if ((psSem->un32Options & OSAL_SEM_OPTION_MUTEX)
                        != OSAL_SEM_OPTION_MUTEX)
        {
            status = pthread_mutex_lock(&psSem->mutex);
            if (status == 0)
            {
                psSem->bDestroy = TRUE;

                status = pthread_cond_broadcast(&psSem->cond);
                if (status != 0)
                {
                    // Error!
                    print_err(status, "Error! pthread_cond_broadcast()");
                }
            }

            status = pthread_mutex_unlock(&psSem->mutex);
            if (status != 0)
            {
                // Error!
                print_err(status, "Error! pthread_mutex_unlock()");
            }

            // Destroy condition variable
            status = pthread_cond_destroy(&psSem->cond);
            if (status != 0)
            {
                // Error!
                print_err(status, "Error! pthread_cond_destroy()");
                break;
            }

#ifndef ANDROID
            // Destroy a condition-variable attribute object
            status = pthread_condattr_destroy(&psSem->cond_attr);
            if (status != 0)
            {
                // Error!
                print_err(status, "Error! pthread_condattr_destroy()");
                break;
            }
#else
            psSem->cond_attr = 0;
#endif
        }
        else //(OSAL_SEM_OPTION_MUTEX)
        {
            // If it is a mutex just trying to unlock it.
            // If it was locked at the moment of deletion, this call
            // will prevent deadlock condition. Otherwise it will return
            // error code, which we do not check anyway
            pthread_mutex_unlock(&psSem->mutex);

            // Now we are locking the mutex, to make sure by the time
            // it is locked we are the only owners of it
            pthread_mutex_lock(&psSem->mutex);

            // ... and unlocking it back. We are finally ready to
            // destroy it
            pthread_mutex_unlock(&psSem->mutex);
        }

        // Destroy mutex
        status = pthread_mutex_destroy(&psSem->mutex);
        if (status != 0)
        {
            // Error!
            print_err(status, "Error! pthread_mutex_destroy()");
            printf(
                "Mutex:\n\tOptions : %x\n\tResources : %d\n\tAvailable : %d\n",
                psSem->un32Options, psSem->un32Resources, psSem->un32Available);
            break;
        }

        // Initialize semaphore attributes
        psSem->un32Resources = 0;
        psSem->un32Available = 0;
        psSem->un32Options = OSAL_SEM_OPTION_NONE;

        // Release memory for object
        OSALC_vMemoryFree(psSem);

        // All is well
        eReturnCode = OSAL_SUCCESS;

    } while (FALSE);

    TRACE_END();
    return eReturnCode;
}

/*******************************************************************************
 *
 *   OS_eSemGive
 *
 ******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eSemGive(OSAL_OBJECT_HDL hSemObj)
{
    int status;
    OS_SEM_STRUCT *psSem = (OS_SEM_STRUCT *) hSemObj;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    TRACE_START();
    // Check input for validity
    if (hSemObj == OSAL_INVALID_OBJECT_HDL)
    {
        TRACE_END();
        return OSAL_ERROR_INVALID_HANDLE;
    }

    if ((psSem->un32Options & OSAL_SEM_OPTION_MUTEX) != OSAL_SEM_OPTION_MUTEX)
    {
        // The calling thread must lock the mutex before modifying the data
        // it protects (the predicate). So as you would expect we must
        // take the mutex before we can modify the resources available.
        status = pthread_mutex_lock(&psSem->mutex);

        if (status == 0)
        {
            // Increment resources available as long as it doesn't exceed
            // the specified total number of resources being managed.
            if (psSem->un32Available < psSem->un32Resources)
            {
                psSem->un32Available++;

                // We need to signal any waiting threads.
                // Don't rely on 0 resources available condition to do
                // signaling, as the scheduler could allow threads calling
                // OS_eSemGive() to run several times before the waiting
                // threads are scheduled to run.
                if (psSem->un32Waiters > 0)
                {
                    // The pthread_cond_signal() function unblocks the highest
                    // priority thread that's waiting on the condition variable.
                    // If more than one thread at the highest priority is
                    // waiting, pthread_cond_signal() unblocks the one that
                    // has been waiting the longest.
                    status = pthread_cond_signal(&psSem->cond);
                    if (status != 0)
                    {
                        // Error!
                        print_err(status, "Error! pthread_cond_signal()");
                    } else
                    {
                        // All is well.
                        eReturnCode = OSAL_SUCCESS;
                    }
                } else
                {
                    // Don't signal. Nothing else to do.
                    eReturnCode = OSAL_SUCCESS;
                }
            } else
            {
                // Drop out. Nothing else to do.
            }

            // We must now unlock the mutex we have locked.
            status = pthread_mutex_unlock(&psSem->mutex);
            if (status != 0)
            {
                // Error!
                print_err(status, "Error! pthread_mutex_unlock()");
            }
        } else
        {
            // Error!
            print_err(status, "Error! pthread_mutex_lock()");
        }
    } else
    {
        // We must now unlock the mutex we have locked.
        status = pthread_mutex_unlock(&psSem->mutex);
        if (status == 0)
        {
            eReturnCode = OSAL_SUCCESS;
            // Give the resource
            psSem->un32Available = 1;
        } else
        {
            // Error!
            print_err(status, "Error! pthread_mutex_unlock()");
            // Hmm, I guess we didn't really give the resource
            psSem->un32Available = 0;
        }
    }

    TRACE_END();
    return eReturnCode;
}

/*******************************************************************************
 *
 *   OS_eSemTake
 *
 ******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eSemTake(OSAL_OBJECT_HDL hSemObj, N32 n32Timeout)
{
    int status;
    OS_SEM_STRUCT *psSem = (OS_SEM_STRUCT *) hSemObj;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    struct timespec timeout;

    TRACE_START();
    // Check input for validity
    if (hSemObj == OSAL_INVALID_OBJECT_HDL)
    {
        TRACE_END();
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Check if we need to compute a timeout
    if (n32Timeout > 0)
    {
        UN32 sec;
        UN64 nsec;

        // Translating milliseconds into seconds and nanoseconds
        // as needed for timespec_t
        sec = n32Timeout / MSEC_PER_SEC;
        nsec = ((N64)n32Timeout - ((N64)sec * MSEC_PER_SEC)) * NSEC_PER_MSEC;

        // Determine absolute time to wait from now. Do this using a
        // pointer to a timespec structure that specifies the maximum
        // time to wait for a condition variable, expressed
        // as an absolute time.
        status = clock_gettime(OS_SEM_CLOCK, &timeout);
        if (status != 0)
        {
            // Error!
            print_err(status, "Error! clock_gettime()"); TRACE_END();
            return OSAL_ERROR;
        }

        // Add to the current absolute time the maximum amount
        // of time to wait for the condition variable.

        timeout.tv_sec += sec;
        timeout.tv_nsec += nsec;
        while (timeout.tv_nsec >= NSEC_PER_MSEC * MSEC_PER_SEC)
        {
            timeout.tv_nsec -= NSEC_PER_MSEC * MSEC_PER_SEC;
            timeout.tv_sec += 1;
        }
    }

    if ((psSem->un32Options & OSAL_SEM_OPTION_MUTEX) != OSAL_SEM_OPTION_MUTEX)
    {
        // The calling thread must lock the mutex before modifying the data
        // it protects (the predicate). So as you would expect we must
        // take the mutex before we can modify the resources available.
        status = pthread_mutex_lock(&psSem->mutex);
        if (status == 0)
        {
            do
            {
                // Check if there is a resource ready, if so, take it
                // and then we are done.
                if (psSem->un32Available > 0)
                {
                    // Take the resource
                    psSem->un32Available--;
                    eReturnCode = OSAL_SUCCESS;
                    break;
                } else if (n32Timeout == 0) // No resource available, however;
                // don't wait for resource
                {
                    // Sorry, no resource available
                    eReturnCode = OSAL_SEM_NOT_AVAILABLE;
                    break;
                } else // Wait abstime for resource
                {
                    // Resource is not available, and we want to wait for it.

                    psSem->un32Waiters++;

                    // Wait for the condition variable with/without a time
                    // limit. Both the pthread_cond_wait() and
                    // pthread_cond_timedwait() functions block the calling
                    // thread on the condition variable
                    // cond, and unlocks the associated mutex. Upon return from
                    // the function, the mutex is again locked and owned by the
                    // calling thread.
                    if (n32Timeout > 0)
                    {
                        // Wait for a specific timeout. Unlock mutex first.
#ifndef ANDROID
                        status = pthread_cond_timedwait
#else
                        status = pthread_cond_timedwait_monotonic_np
#endif
                            (&psSem->cond, &psSem->mutex, &timeout);
                    } else
                    {
                        // Wait for resource forever...
                        // Unlock mutex first.
                        status = pthread_cond_wait(&psSem->cond, &psSem->mutex);
                    }

                    psSem->un32Waiters--;

                    // Mutex is locked again, but only if an
                    // error doesn't occur

                    // Check if destroy flag is set, if so bail.
                    if (psSem->bDestroy == TRUE)
                    {
                        break;
                    }

                    // Check the condition variable wait result.
                    if (status == ETIMEDOUT) // Timeout?
                    {
                        // A timeout occurred
                        eReturnCode = OSAL_TIMEOUT;
                        break;
                    } else if (status != 0) // Error?
                    {
                        // Error!
                        if (n32Timeout > 0)
                        {
                            print_err(
                                status, "Error! pthread_cond_timedwait()");
                        } else
                        {
                            print_err(
                                status, "Error! pthread_cond_wait()");
                        }
                        break;
                    }
                }

            } while (TRUE);

            // Since the mutex is locked and owned by the calling thread
            // we must now try to unlock it, but only with we still
            // have it locked.
#ifdef __QNX__
            status = pthread_mutex_trylock(&psSem->mutex);
            if ((status == EBUSY) || (status == EOK) || (status == EAGAIN))
#endif
            {
                status = pthread_mutex_unlock(&psSem->mutex);
                if (status != 0)
                {
                    // Error!
                    print_err(status, "Error! pthread_mutex_unlock()");
                }
            }
        } else
        {
            // Error!
            print_err(status, "Error! pthread_mutex_lock()");
        }
    } else
    {
        // priority inheriting mutex...
        // Lock the mutex, according to the timeout

        if (n32Timeout == 0)
        {
            // don't wait
            status = pthread_mutex_trylock(&psSem->mutex);
            if (status == 0)
            {
                eReturnCode = OSAL_SUCCESS;
            }
            else if (status == EBUSY)
            {
                // Sorry, its already locked,
                // and you don't want to wait for it.
                eReturnCode = OSAL_SEM_NOT_AVAILABLE;
            }
            else
            {
                // Error!
                print_err(status, "Error! pthread_mutex_trylock()");
            }
        }
        else
        {
            if (n32Timeout > 0)
            {
                // wait for a specific amount of time
#if defined __QNX__
                status = pthread_mutex_timedlock_monotonic(&psSem->mutex,
                    &timeout);
#elif (defined LINUX) || (defined __INTEGRITY)
                status =
                    pthread_mutex_timedlock(&psSem->mutex, &timeout);
#elif defined(ANDROID)
                status =
                    pthread_mutex_lock_timeout_np(&psSem->mutex, n32Timeout);
#else
#error "Do not know the function to call instead of pthread_mutex_timedlock for this platform"
#endif
                if (status == 0)
                {
                    eReturnCode = OSAL_SUCCESS;
                }
                else if (status == ETIMEDOUT)
                {
                    // Sorry, its still locked,
                    // and you don't want to wait for it anymore.
                    eReturnCode = OSAL_TIMEOUT;
                }
                else
                {
                    // Error!
                    print_err(status, "Error! pthread_mutex_lock()");
                }
            }
            else
            {
                // wait forever
                status = pthread_mutex_lock(&psSem->mutex);
                if (status == 0)
                {
                    eReturnCode = OSAL_SUCCESS;
                }
                else
                {
                    // Error!
                    print_err(status, "Error! pthread_mutex_lock()");
                }
            }
        }

        if (eReturnCode == OSAL_SUCCESS)
        {
            // Take the resource
            psSem->un32Available = 0;
        }
    }

    TRACE_END();
    return eReturnCode;
}
