/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
* DESCRIPTION
*
* This module will contain all the OS specific mem implementations.
* The implementations in this module are 100% POSIX compliant making
* it suitable for any POSIX.1b (POSIX 1003.1b-1993) complint OS.
*
*******************************************************************************/

// POSIX and standard includes
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <sys/mman.h>
#ifndef __INTEGRITY
#include <sys/resource.h>
#endif

#define OSAL_TRACE_ENABLE 0
#include "osal_trace.h"

#include "standard.h"

#include "os.h"
#include "os_version.h"
#include "os_general.h"

#include "os_mem_.h"
#include "_os_mem_.h"

// Avoid using OSAL printf function
#ifdef printf
#undef printf
#endif

// Extra code for double memory free detection
#ifndef OSAL_MEM_FREE_CHECK
#define OSAL_MEM_FREE_CHECK 0
#endif

#ifndef OSAL_MEMORY_TRACKER
#define OSAL_MEMORY_TRACKER 0
#endif

#if OSAL_MEMORY_TRACKER == 1

#define MEMORY_TRACKER_SIZE 100000

static struct
{
    void *pvAddress;
    char *pacName;
} gapMemoryTracker[MEMORY_TRACKER_SIZE] = {0};

/*******************************************************************************
*
*   OSM_vUpdateAllocation
*
*******************************************************************************/
static void OSM_vUpdateAllocation(
    void *pvAddr,
    BOOLEAN bAdd,
    char *pacName
        )
{
    UN32 un32Index = 0;

    if (bAdd == TRUE)
    {
        // Adding allocated unit
        while(gapMemoryTracker[un32Index].pvAddress != NULL && un32Index < MEMORY_TRACKER_SIZE)
        {
            un32Index++;
        }
        if (un32Index < MEMORY_TRACKER_SIZE)
        {
            size_t tSize = 0;
            gapMemoryTracker[un32Index].pvAddress = pvAddr;
            if (pacName != NULL)
            {
                tSize = strlen(pacName);
            }
            if (tSize > 0)
            {
                gapMemoryTracker[un32Index].pacName = malloc(tSize+1);
                if(gapMemoryTracker[un32Index].pacName != NULL)
                {
                    strncpy(gapMemoryTracker[un32Index].pacName, pacName, tSize+1);
                }
            }
        }
    }
    else
    {
        // Removing allocated unit
        while(gapMemoryTracker[un32Index].pvAddress != pvAddr && un32Index < MEMORY_TRACKER_SIZE)
        {
            un32Index++;
        }
        if (un32Index < MEMORY_TRACKER_SIZE)
        {
            gapMemoryTracker[un32Index].pvAddress = NULL;
            if (gapMemoryTracker[un32Index].pacName != NULL)
            {
                free(gapMemoryTracker[un32Index].pacName);
                gapMemoryTracker[un32Index].pacName = NULL;
            }
        }
    }
}

/*******************************************************************************
*
*   OSM_vLogAllocation
*
*******************************************************************************/
void OSM_vLogAllocation( void )
{
    UN8 un8Counter = 1;
    UN32 un32Index;
    UN8 acName[14];
    FILE *psFile;

    strcpy(acName, "os_mem000.log");

    acName[6] = un8Counter / 100 + '0';
    acName[7] = (un8Counter % 100) / 10 + '0';
    acName[8] = un8Counter % 10 + '0';

    psFile = fopen(acName, "w");

    for( un32Index = 0; un32Index < MEMORY_TRACKER_SIZE; un32Index++ )
    {
        if(gapMemoryTracker[un32Index].pvAddress != NULL)
        {
            fprintf(psFile, "%08x - %s\n",
                gapMemoryTracker[un32Index].pvAddress,
                gapMemoryTracker[un32Index].pacName);
        }
    }
    fclose(psFile);
    un8Counter++;

    return;
}
#endif

/*******************************************************************************
*
*   OS_bMemRead8
*
*******************************************************************************/
BOOLEAN OS_bMemRead8 (
    const void *pvMemoryAddress,
    UN8 *pun8Value
        )
{
    void *pvPtr = NULL;

    TRACE_START();
    // Check input
    if(pun8Value == NULL)
    {
        // Error! No destination pointer provided
        TRACE_END();
        return FALSE;
    }

    // Map physical memory requested
    pvPtr = OS_pvMapPhysicalMemory(
        (unsigned int)pvMemoryAddress, sizeof(UN8), PROT_READ);
    if(pvPtr != MAP_FAILED)
    {
        // Copy value into local variable to return to user
        *pun8Value = *((UN8 *)pvPtr);

        // Unmap physical memory
        OS_vUnMapPhysicalMemory(pvPtr, sizeof(UN8));

        TRACE_END();
        return TRUE;
    }

    TRACE_END();
    return FALSE;
}

/*******************************************************************************
*
*   OS_bMemWrite8
*
*******************************************************************************/
BOOLEAN OS_bMemWrite8 (
     void *pvMemoryAddress,
     UN8 un8Value
         )
{
    void *pvPtr = NULL;

    TRACE_START();
    // Map physical memory requested
    pvPtr = OS_pvMapPhysicalMemory(
        (unsigned int)pvMemoryAddress, sizeof(UN8), PROT_WRITE);
    if(pvPtr != MAP_FAILED)
    {
        // Copy value from caller to physical memory
        *((UN8 *)pvPtr) = un8Value;

        // Unmap physical memory
        OS_vUnMapPhysicalMemory(pvPtr, sizeof(UN8));

        TRACE_END();
        return TRUE;
    }

    TRACE_END();
    return FALSE;
}

/*******************************************************************************
*
*   OS_bMemRead16
*
*******************************************************************************/
BOOLEAN OS_bMemRead16 (
     const void *pvMemoryAddress,
     UN16 *pun16Value
         )
{
    void *pvPtr = NULL;

    TRACE_START();
    // Check input
    if(pun16Value == NULL)
    {
        // Error! No destination pointer provided
        TRACE_END();
        return FALSE;
    }

    // Map physical memory requested
    pvPtr = OS_pvMapPhysicalMemory(
        (unsigned int)pvMemoryAddress, sizeof(UN16), PROT_READ);
    if(pvPtr != MAP_FAILED)
    {
        // Copy value into local variable to return to user
        *pun16Value = *((UN16 *)pvPtr);

        // Unmap physical memory
        OS_vUnMapPhysicalMemory(pvPtr, sizeof(UN16));

        TRACE_END();
        return TRUE;
    }

    TRACE_END();
    return FALSE;
}

/*******************************************************************************
*
*   OS_bMemWrite16
*
*******************************************************************************/
BOOLEAN OS_bMemWrite16 (
      void *pvMemoryAddress,
      UN16 un16Value
          )
{
    void *pvPtr = NULL;

    TRACE_START();
    // Map physical memory requested
    pvPtr = OS_pvMapPhysicalMemory(
        (unsigned int)pvMemoryAddress, sizeof(UN16), PROT_WRITE);
    if(pvPtr != MAP_FAILED)
    {
        // Copy value from caller to physical memory
        *((UN16 *)pvPtr) = un16Value;

        // Unmap physical memory
        OS_vUnMapPhysicalMemory(pvPtr, sizeof(UN16));

        TRACE_END();
        return TRUE;
    }

    TRACE_END();
    return FALSE;
}

/*******************************************************************************
*
*   OS_bMemRead32
*
*******************************************************************************/
BOOLEAN OS_bMemRead32 (
     const void *pvMemoryAddress,
     UN32 *pun32Value
         )
{
    void *pvPtr = NULL;

    TRACE_START();
    // Check input
    if(pun32Value == NULL)
    {
        // Error! No destination pointer provided
        TRACE_END();
        return FALSE;
    }

    // Map physical memory requested
    pvPtr = OS_pvMapPhysicalMemory(
        (unsigned int)pvMemoryAddress, sizeof(UN32), PROT_READ);
    if(pvPtr != MAP_FAILED)
    {
        // Copy value into local variable to return to user
        *pun32Value = *((UN32 *)pvPtr);

        // Unmap physical memory
        OS_vUnMapPhysicalMemory(pvPtr, sizeof(UN32));

        TRACE_END();
        return TRUE;
    }

    TRACE_END();
    return FALSE;
}

/*******************************************************************************
*
*   OS_bMemWrite32
*
*******************************************************************************/
BOOLEAN OS_bMemWrite32 (
      void *pvMemoryAddress,
      UN32 un32Value
          )
{
    void *pvPtr = NULL;

    TRACE_START();
    // Map physical memory requested
    pvPtr = OS_pvMapPhysicalMemory(
        (unsigned int)pvMemoryAddress, sizeof(UN32), PROT_WRITE);
    if(pvPtr != MAP_FAILED)
    {
        // Copy value from caller to physical memory
        *((UN32 *)pvPtr) = un32Value;

        // Unmap physical memory
        OS_vUnMapPhysicalMemory(pvPtr, sizeof(UN32));

        TRACE_END();
        return TRUE;
    }

    TRACE_END();
    return FALSE;
}

/*******************************************************************************
*
*   OS_bPortRead8
*
*******************************************************************************/
BOOLEAN OS_bPortRead8 (
     const void *pvPortAddress,
     UN8 *pun8Value
         )
{
    BOOLEAN bRetval;
    TRACE_START();
    // Just map into 'mem' function since it does the same thing
    bRetval = OS_bMemRead8(pvPortAddress, pun8Value);

    TRACE_END();
    return bRetval;
}

/*******************************************************************************
*
*   OS_bPortWrite8
*
*******************************************************************************/
BOOLEAN OS_bPortWrite8 (
      void *pvPortAddress,
      UN8 un8Value
          )
{
    BOOLEAN bRetval;
    TRACE_START();
    // Just map into 'mem' function since it does the same thing
    bRetval = OS_bMemWrite8(pvPortAddress, un8Value);

    TRACE_END();
    return bRetval;
}

/*******************************************************************************
*
*   OS_bPortRead16
*
*******************************************************************************/
BOOLEAN OS_bPortRead16 (
      const void *pvPortAddress,
      UN16 *pun16Value
          )
{
    BOOLEAN bRetval;
    TRACE_START();
    // Just map into 'mem' function since it does the same thing
    bRetval = OS_bMemRead16(pvPortAddress, pun16Value);

    TRACE_END();
    return bRetval;
}

/*******************************************************************************
*
*   OS_bPortWrite16
*
*******************************************************************************/
BOOLEAN OS_bPortWrite16 (
       void *pvPortAddress,
       UN16 un16Value
           )
{
    BOOLEAN bRetval;
    TRACE_START();
    // Just map into 'mem' function since it does the same thing
    bRetval = OS_bMemWrite16(pvPortAddress, un16Value);

    TRACE_END();
    return bRetval;
}

/*******************************************************************************
*
*   OS_bPortRead32
*
*******************************************************************************/
BOOLEAN OS_bPortRead32 (
      const void *pvPortAddress,
      UN32 *pun32Value
          )
{
    BOOLEAN bRetval;
    TRACE_START();
    // Just map into 'mem' function since it does the same thing
    bRetval = OS_bMemRead32(pvPortAddress, pun32Value);

    TRACE_END();
    return bRetval;
}

/*******************************************************************************
*
*   OS_bPortWrite32
*
*******************************************************************************/
BOOLEAN OS_bPortWrite32 (
       void *pvPortAddress,
       UN32 un32Value
           )
{
    BOOLEAN bRetval;
    TRACE_START();
    // Just map into 'mem' function since it does the same thing
    bRetval = OS_bMemWrite32(pvPortAddress, un32Value);

    TRACE_END();
    return bRetval;
}

/*******************************************************************************
*
*   OS_bMemCpy
*
*******************************************************************************/
BOOLEAN OS_bMemCpy (
      void *pvDst,
      const void *pvSrc,
      size_t tSize
          )
{
    TRACE_START();
    // Copy bytes from one buffer to another.
    // Always returns a pointer to the destination buffer
    // (that is, the value of pvDst).
    memcpy (pvDst, pvSrc, tSize );
    TRACE_END();
    return TRUE;
}

/*******************************************************************************
*
*   OS_bMemSet
*
*******************************************************************************/
BOOLEAN OS_bMemSet (
      void *pvDst,
      UN8 un8Value,
      size_t tSize
          )
{
    TRACE_START();
    // Set memory to a given value
    // Always returns a pointer to the destination buffer
    // (that is, the value of pvDst).
    memset(pvDst, un8Value, tSize);
    TRACE_END();
    return TRUE;
}

/*******************************************************************************
*
*   OS_pvMemoryAllocate
*
*******************************************************************************/
void *OS_pvMemoryAllocate(
      const char *pacName,
      size_t tSize,
      BOOLEAN bZeroInitialize,
      size_t *ptActualSizeAllocated
          )
{
    void *pvPtr = NULL;
    size_t *ptSize = NULL;
    size_t tInitializeSize = 0;
    size_t tRequestedSize = tSize;

    TRACE_START();

    // Allocate a little bit more than they asked for so that I can
    // track the size of this allocation for the eventual
    // free which will occur
    *ptActualSizeAllocated = tSize + sizeof(size_t);

#if OSAL_MEMORY_CHECK != OSAL_MEMORY_CHECK_NONE
    // One element for safety buffer at the end
    *ptActualSizeAllocated += sizeof(size_t);
    ALIGN_SIZE(*ptActualSizeAllocated);
#endif

#if OSAL_MEMORY_CHECK == OSAL_MEMORY_CHECK_FULL
    // One more element for safety buffer at the beginning
    *ptActualSizeAllocated += sizeof(size_t);
#endif

    // It is acceptable to use the ANSI-C call malloc()
    // on this platform.
    pvPtr = malloc(*ptActualSizeAllocated);
    if(pvPtr != NULL)
    {
#if OSAL_MEMORY_TRACKER == 1
        OSM_vUpdateAllocation(pvPtr, TRUE, pacName);
#endif

        ptSize = (size_t*)pvPtr;

#if OSAL_MEMORY_CHECK == OSAL_MEMORY_CHECK_FULL
        // Putting safety pattern into the beginning of the buffer
        *ptSize++ = DBG_SAFETY_PATTERN;
#endif

        // Populate size of the block
        *ptSize++ = *ptActualSizeAllocated;

#if OSAL_MEMORY_CHECK != OSAL_MEMORY_CHECK_NONE
        // Calculating the end of the buffer
        // Putting safety pattern into the end
        *(size_t *)((size_t)pvPtr +
            *ptActualSizeAllocated - sizeof(size_t)) =
                            DBG_SAFETY_PATTERN;
#endif

        // Adjust pvPtr to user memory location
        pvPtr = ptSize;

        // Calculate how many bytes are to be cleared
        if(bZeroInitialize == TRUE)
        {
            tInitializeSize = tRequestedSize;
        }
#if OSAL_MEM_FREE_CHECK != 0
        else
        {
            // Remove 'memory freed' pattern
            tInitializeSize = (sizeof(size_t) > tRequestedSize) ? tRequestedSize : sizeof(size_t);
        }
#endif

        if(tInitializeSize > 0)
        {
            // Zero initialize memory block
            OS.bMemSet(pvPtr, 0, tInitializeSize);
        }
    }
    else
    {
        // TODO: Breakpoint here. Possible heap empty/fragmented.
        printf("[ERROR] Called malloc() of %u bytes and NULL returned.\n",
            *ptActualSizeAllocated);
    }

    TRACE_END();
    return pvPtr;
}

/*******************************************************************************
*
*   OS_vMemoryFree
*
*******************************************************************************/
void OS_vMemoryFree(
      void *pvMemoryAddress,
      size_t *ptNumBytesFreed
          )
{
    size_t *ptAddr = pvMemoryAddress;
#if OSAL_MEM_FREE_CHECK != 0
    size_t tFreedPatternSize;
    const size_t tFreedPattern = DBG_SAFETY_PATTERN_FREED;
#endif

#if OSAL_MEMORY_CHECK != OSAL_MEMORY_CHECK_NONE
    size_t *pBufferStart = ptAddr - 1;
    size_t *pBufferEnd;

#if OSAL_MEMORY_CHECK == OSAL_MEMORY_CHECK_FULL
    // Minus one more element for safety pattern
    pBufferStart--;
#endif
#endif

    TRACE_START();

    // Extract real memory pointer and obtain its size
    *ptNumBytesFreed = *(--ptAddr);

#if OSAL_MEMORY_CHECK != OSAL_MEMORY_CHECK_NONE
    // Calculating the end of the buffer
    pBufferEnd = (size_t*)((unsigned int)pBufferStart + *ptNumBytesFreed);
    pBufferEnd--;
#endif

#if OSAL_MEMORY_CHECK == OSAL_MEMORY_CHECK_FULL
    // Checking if memory overflow happened in the beginning of the block
    if(*pBufferStart != DBG_SAFETY_PATTERN )
    {
        // Memory is corrupted! Doing "Segmentation Fault" to have backtrace
        // from core dump
        printf("Buffer at %x (%u bytes) overrun at beginning\n",
            (unsigned int)pvMemoryAddress, *ptNumBytesFreed);
        *(char*)0 = 1;
    }

    // Setting ptAddr to the beginning of allocated buffer
    ptAddr = pBufferStart;
#endif

#if OSAL_MEMORY_CHECK != OSAL_MEMORY_CHECK_NONE
    // Checking if memory overflow happened on the end of the block
    if(*pBufferEnd != DBG_SAFETY_PATTERN )
    {
       // Memory is corrupted! Doing "Segmentation Fault" to have backtrace
       // from core dump
       printf("Buffer at %x (%u bytes) overrun at end\n",
           (unsigned int)pvMemoryAddress, *ptNumBytesFreed);
       *(char*)0 = 1;
    }
#endif

#if OSAL_MEM_FREE_CHECK != 0
    // Make sure that we are not going to corrupt memory outside the real data blocks.
    tFreedPatternSize = (sizeof(size_t) > *ptNumBytesFreed) ?
                    *ptNumBytesFreed : sizeof(size_t);
    if (memcmp(pvMemoryAddress, &tFreedPattern, tFreedPatternSize) != 0)
    {
        memcpy(pvMemoryAddress, &tFreedPattern, tFreedPatternSize);
    }
    else
    {
        // TODO: Breakpoint here. Possible double-free call.
        printf("[ERROR] Called free() for already deallocated pointer (%p)\n",
               ptAddr);
    }
#endif

    // It is acceptable to use the ANSI-C call free()
    // on this platform.
    free(ptAddr);
#if OSAL_MEMORY_TRACKER == 1
    OSM_vUpdateAllocation(ptAddr, FALSE, NULL);
#endif

    TRACE_END();
    return;
}

/*******************************************************************************
 *
 *   OS_un32TotalSystemBytes
 *
 *******************************************************************************/
UN32 OS_un32TotalSystemBytes( void )
{
    UN32 un32TotalSystemBytes;
#ifndef __INTEGRITY
    struct rlimit tRlim;
    int iRetval;
#endif

    TRACE_START();

#ifndef __INTEGRITY
    // Getting available process memory size
    iRetval = getrlimit(RLIMIT_AS, &tRlim);
    if(iRetval == EOK)
    {
        un32TotalSystemBytes = (UN32)tRlim.rlim_cur;
    }
    else
#endif
    {
        //Something went wrong. Setting default value of 4M for heap
        un32TotalSystemBytes = (1024*1024)*4;
    }

    TRACE_END();
    return un32TotalSystemBytes;
}

/*******************************************************************************
*
*   OS_pvMapPhysicalMemory
*
*******************************************************************************/
static void *OS_pvMapPhysicalMemory (
    unsigned int uiOffset,
    size_t tSize,
    int iProtection
        )
{
    void *pvPtr = MAP_FAILED;

   TRACE_START();

#ifndef __INTEGRITY
    // Opening device only if it is not opened yet
    if(giMemFD < 0)
    {
        // Opening "memory" device for physical memory mapping
        giMemFD = open("/dev/mem", O_RDWR | O_SYNC);
        if(giMemFD < 0)
        {
            print_errno("Error! /dev/mem could not be opened.\n");
            TRACE_END();
            return MAP_FAILED;
        }
    }

    // The mmap() function maps a region within the object beginning at
    // pvMemoryAddress and continuing for tSize into the caller's address
    // space and returns the location.
    pvPtr = mmap(
        NULL,
        tSize, // Number of bytes to map
        iProtection, // Access capabilities (read/write/both)
        MAP_PRIVATE, // Map physical memory to this process only
        giMemFD, // Memory file descriptor
        uiOffset // Physical address to map
            );
    if(pvPtr == MAP_FAILED)
    {
        // Error!
        print_errno("Error! mmap()");
        close(giMemFD);
        giMemFD = -1;
    }

#endif
    TRACE_END();
    return pvPtr;
}

/*******************************************************************************
*
*   OS_vUnMapPhysicalMemory
*
*******************************************************************************/
static void OS_vUnMapPhysicalMemory (
    void *pvPtr,
    size_t tSize
        )
{
    int iRetval;

    TRACE_START();

#ifndef __INTEGRITY
    // The munmap() function removes any mappings for pages in the address
    // range starting at pvPtr and continuing for tSize bytes, rounded up to the
    // next multiple of the page size. Subsequent references to these pages
    // cause a SIGSEGV signal to be set on the process.
    iRetval = munmap(pvPtr, tSize);
    if (iRetval == -1)
    {
        // Error!
        print_errno("Error! munmap()");
    }

    if( giMemFD >= 0 )
    {
      //Closing memory device
      close(giMemFD);
      giMemFD = -1;
    }
#endif

    TRACE_END();
    return;
}
