/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*DESCRIPTION
*
*       This module will contain all the OS-Specific implementation of the
*       general OSAL APIs.
*
*******************************************************************************/

// POSIX and standard includes
#define _GNU_SOURCE
#include <string.h>
#include <pthread.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <termios.h>
#ifndef __INTEGRITY
#include <sys/resource.h>
#endif

#if OSAL_REMOTE_CONSOLE == 1
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#endif /* OSAL_REMOTE_CONSOLE == 1 */

#define OSAL_TRACE_ENABLE 0
#include "osal_trace.h"

#include "standard.h"
#include "osal.h"
#include "os_version.h"
#include "os_task_.h"
#include "os.h"
#include "os_stdio.h"
#include "os_general.h"
#include "_os_general.h"

/*******************************************************************************
*
*   OS_bStart
*
*******************************************************************************/
BOOLEAN OS_bStart (
    OSAL_START_HANDLER tStartHandler,
    const void *pvStartHandlerArg
        )
{
    BOOLEAN bOk;
    int iRetval;
    pthread_t tThreadId;
    struct sigaction sSignalSuspend, sSignalResume;
    pthread_mutexattr_t tMutexAttr;  // Mutex attribute variable
#if (defined ANDROID) || (defined LINUX) || (defined __INTEGRITY)
    pthread_mutex_t  deadlock;
#endif

#if (defined __QNX__) || (defined __INTEGRITY)
    struct sched_param sSchedParams;
    int iPolicy;
#endif

    TRACE_START();

    // Initialize general module control structure
    bOk = OSAL.bMemSet(&GsCtrl, 0, sizeof(GsCtrl));
    if(bOk == FALSE)
    {
        // Error!
        printf("Error! OSAL.bMemSet (GsCtrl) failed.\n");
        TRACE_END();
        return FALSE;
    }

    // Initialize CPU stats control structure
    bOk = OSAL.bMemSet(&GsCPUCtrl, 0, sizeof(GsCPUCtrl));
    if(bOk == FALSE)
    {
        // Error!
        printf("Error! OSAL.bMemSet (GsCPUCtrl) failed.\n");
        TRACE_END();
        return FALSE;
    }

    // Initialize local structures
    bOk = OSAL.bMemSet(&sSignalSuspend, 0, sizeof(sSignalSuspend));
    if(bOk == FALSE)
    {
        // Error!
        printf("Error! OSAL.bMemSet (sSignalSuspend) failed.\n");
        TRACE_END();
        return FALSE;
    }

    bOk = OSAL.bMemSet(&sSignalResume, 0, sizeof(sSignalResume));
    if(bOk == FALSE)
    {
        // Error!
        printf("Error! OSAL.bMemSet (sSignalResume) failed.\n");
        TRACE_END();
        return FALSE;
    }

    /*
     * Define global mutex for critical section implementation
     */
    // Initialize attributes set
    iRetval = pthread_mutexattr_init(&tMutexAttr);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_mutexattr_init()");
        TRACE_END();
        return FALSE;
    }

    // Set global critical section mutex as a recursive mutex
#if (defined __QNX__) || (defined __INTEGRITY)
    iRetval = pthread_mutexattr_settype(&tMutexAttr, PTHREAD_MUTEX_RECURSIVE);
#else
    iRetval = pthread_mutexattr_settype(&tMutexAttr, PTHREAD_MUTEX_RECURSIVE_NP);
#endif
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_mutexattr_settype()");
        pthread_mutexattr_destroy(&tMutexAttr);
        TRACE_END();
        return FALSE;
    }

    // create the mutex with the attributes set
    iRetval = pthread_mutex_init(&GsCtrl.tCriticalSectionMutex, &tMutexAttr);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_mutex_init()");
        pthread_mutexattr_destroy(&tMutexAttr);
        TRACE_END();
        return FALSE;
    }

    //After initializing the mutex, the thread attribute can be destroyed
    iRetval = pthread_mutexattr_destroy(&tMutexAttr);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_mutexattr_destroy()");
        TRACE_END();
        return FALSE;
    }

#if OSAL_DEBUG == 1
    // Set-up and configure stdio...
    vConfigureStdio();
#endif

    // Verify compile time priority levels for threads
    // has been configured properly.
#ifdef ANDROID
    if((OS_TASK_LOWEST_PRIORITY > 19 ) || (OS_TASK_HIGHEST_PRIORITY < -20))
    {
        printf("Error! Invalid priority levels specified (lowest=%d, highest=%d).",
            OS_TASK_LOWEST_PRIORITY,
            OS_TASK_HIGHEST_PRIORITY);
        TRACE_END();
        return FALSE;
    }
#else

// Bosch ID#3: Disable the check, because Bosch defined priorities are below 0
#if 0
    if((OS_TASK_LOWEST_PRIORITY < 0) || (OS_TASK_HIGHEST_PRIORITY < 0))
    {
        printf("Error! Invalid priority levels specified (min=%d, max=%d).",
            OS_TASK_LOWEST_PRIORITY,
            OS_TASK_HIGHEST_PRIORITY);
        TRACE_END();
        return FALSE;
    }
#endif	
#endif

    if(OS_TASK_NUMBER_OF_PRIORITIES < 5)
    {
        printf("Error! Incorrect priority levels (lowest=%d, highest=%d). Not enough!",
            OS_TASK_LOWEST_PRIORITY,
            OS_TASK_HIGHEST_PRIORITY);
        TRACE_END();
        return FALSE;
    }

    // Determine this thread's id.
    tThreadId = pthread_self();

#if defined (__QNX__) || defined (__INTEGRITY)
    // Obtain this thread's current scheduling policy and parameters
    iRetval = pthread_getschedparam(tThreadId, &iPolicy, &sSchedParams);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_getschedparam()");
        TRACE_END();
        return FALSE;
    }

    // Explicitly configure the priority as lowest.
    sSchedParams.sched_priority = OS_TASK_LOWEST_PRIORITY;

    // Explicitly configure policy.
    iPolicy = OS_SCHEDULING_POLICY;

    // Configure this thread's priority (lowest) and also scheduling
    // algorithm as defined by OS_SCHEDULING_POLICY.
    iRetval = pthread_setschedparam(tThreadId, iPolicy, &sSchedParams);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_setschedparam()");
        TRACE_END();
        return FALSE;
    }
#endif

    // The pthread_detach() function detaches the thread with the given ID
    // from its process. When a detached thread terminates, all system
    // resources allocated to that thread are immediately reclaimed.
    //print_err(iRetval, "Error! pthread_detach()");
#ifndef ANDROID
    iRetval = pthread_detach(tThreadId);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_detach()");
        TRACE_END();
        return FALSE;
    }
#endif
    // Create a thread specific key which will be used to reference
    // each thread's specific data. This is done only one time for
    // all threads.
    iRetval = pthread_key_create(
        &GsCtrl.tThreadSpecificObjectKey,
        OS_vThreadSpecificKeyDestructor
            );
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_key_create()");
        TRACE_END();
        return FALSE;
    }

    /* Install signal handlers for suspend and resume signals
     * The handles installed are the same for all threads created within
     * the same process.
     */

    // Initialize suspend signal handler
    sSignalSuspend.sa_flags = 0;
    sSignalSuspend.sa_handler = OS_vSuspendSignalHandler;

    iRetval = sigemptyset(&sSignalSuspend.sa_mask); // mask no signals
    if(iRetval == -1)
    {
        // Error!
        print_errno("Error! sigemptyset()");
        TRACE_END();
        return FALSE;
    }

    // The sigaddset() function adds OS_TASK_RESUME_SIGNAL to the set pointed
    // to by &sSignalSuspend.sa_mask.
    iRetval = sigaddset(&sSignalSuspend.sa_mask, OS_TASK_RESUME_SIGNAL);
    if(iRetval == -1)
    {
        // Error!
        print_errno("Error! sigaddset()");
        TRACE_END();
        return FALSE;
    }

    // Initialize resume signal handler
    sSignalResume.sa_flags = 0;
    sSignalResume.sa_handler = OS_vResumeSignalHandler;

    iRetval = sigemptyset(&sSignalResume.sa_mask); // mask no signals
    if(iRetval == -1)
    {
        // Error!
        print_errno("Error! sigemptyset()");
        TRACE_END();
        return FALSE;
    }

    // Install suspend signal handler
    iRetval = sigaction(OS_TASK_SUSPEND_SIGNAL, &sSignalSuspend, NULL);
    if(iRetval == -1)
    {
        // Error!
        print_errno("Error! sigaction()");
        TRACE_END();
        return FALSE;
    }

    // Install resume signal handler
    iRetval = sigaction(OS_TASK_RESUME_SIGNAL, &sSignalResume, NULL);
    if(iRetval == -1)
    {
        // Error!
        print_errno("Error! sigaction()");
        TRACE_END();
        return FALSE;
    }

    // Run start handler if provided by caller (OSAL)
    if(tStartHandler != NULL)
    {
        bOk = tStartHandler(pvStartHandlerArg);
        if(bOk == FALSE)
        {
            // Error!
            printf("Error! Start handler failed.\n");
            TRACE_END();
            return FALSE;
        }
    }

#if (defined ANDROID) || (defined LINUX) || (defined __INTEGRITY)

#ifdef __INTEGRITY
    // Tasks have absolute priority, NOT relative to processes.
#else
    // Setting priority to minimum
    setpriority( PRIO_PROCESS, OS_GetTid(), OS_TASK_LOWEST_PRIORITY );
#endif

    iRetval = pthread_mutexattr_init(&tMutexAttr);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_mutexattr_init()");
        TRACE_END();
        return FALSE;
    }

    // Set deadlock mutex as non-recursive
    iRetval = pthread_mutexattr_settype(&tMutexAttr, PTHREAD_MUTEX_NORMAL);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_mutexattr_settype()");
        pthread_mutexattr_destroy(&tMutexAttr);
        TRACE_END();
        return FALSE;
    }

    iRetval = pthread_mutex_init(&deadlock, &tMutexAttr);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_mutex_init()");
        pthread_mutexattr_destroy(&tMutexAttr);
        TRACE_END();
        return FALSE;
    }

    iRetval = pthread_mutexattr_destroy(&tMutexAttr);
    if(iRetval != EOK)
    {
        // Error!
        print_err(iRetval, "Error! pthread_mutexattr_destroy()");
        TRACE_END();
        return FALSE;
    }

    // Locking mutex...
    pthread_mutex_lock(&deadlock);
    // ... and forcing deadlock for this thread
    pthread_mutex_lock(&deadlock);
#else
    // This will wait for all threads to shutdown and then shutdown itself.
    // Since this thread is detached, all system resources allocated to the
    // thread are immediately reclaimed.
    pthread_exit(NULL);
#endif

    TRACE_END();
    return FALSE;
}

/*******************************************************************************
*
*   OS_vEnterCriticalSection
*
*******************************************************************************/
void OS_vEnterCriticalSection ( void )
{
   int iRetval;

   TRACE_START();

   // Locking mutex
   iRetval = pthread_mutex_lock( &GsCtrl.tCriticalSectionMutex );

   if(iRetval != EOK)
   {
       // Error!
       print_err(iRetval, "Error! pthread_mutex_lock()");
   }

   TRACE_END();
   return;
}

/*******************************************************************************
*
*   OS_vExitCriticalSection
*
*******************************************************************************/
void OS_vExitCriticalSection ( void )
{
   int iRetval;

   TRACE_START();

   // Unlocking mutex.
   iRetval = pthread_mutex_unlock( &GsCtrl.tCriticalSectionMutex );

   if(iRetval != EOK)
   {
       // Error!
       print_err(iRetval, "Error! pthread_mutex_unlock()");
   }

   TRACE_END();
   return;
}

/*******************************************************************************
*
*   OS_vSuspendSignalHandler
*
*******************************************************************************/
static void OS_vSuspendSignalHandler(
#ifdef __QNX__
    int iSignal,
    siginfo_t *ptSignalInfo,
    void *pvArg
#else
    int iSignal
#endif
        )
{
    sigset_t sSignalSetMask;
    int iRetval;

    TRACE_START();
    // Block all signals except OS_TASK_RESUME_SIGNAL...

    // Initialize a signal set to contain all signals
    iRetval = sigfillset(&sSignalSetMask);
    if(iRetval != EOK)
    {
        // Error! Cannot set signal set
        print_errno("Error! sigfillset()");
        TRACE_END();
        return;
    }

    // The sigdelset() function deletes OS_TASK_RESUME_SIGNAL from the
    // set pointed to by &sSignalSet.
    iRetval = sigdelset(&sSignalSetMask, OS_TASK_RESUME_SIGNAL);
    if(iRetval != EOK)
    {
        // Error! Cannot delete signal set
        print_errno("Error! sigdelset()");
        TRACE_END();
        return;
    }

    printf(OSAL_NAME_PREFIX"Thread suspending...\n");

    // The sigsuspend() function replaces the thread's signal mask with the
    // set of signals pointed to by &sSignalSetMask and then suspends the thread
    // until delivery of a signal whose action is either to execute a
    // signal-catching function (then return), or to terminate the thread.
    sigsuspend(&sSignalSetMask);
    // sigsuspend() always returns -1, if interrupted by signal arrival

    // Resume...
    TRACE_END();
    return;
}

/*******************************************************************************
*
*   OS_vResumeSignalHandler
*
*******************************************************************************/
static void OS_vResumeSignalHandler(
#ifdef __QNX__
    int iSignal,
    siginfo_t *ptSignalInfo,
    void *pvArg
#else
    int iSignal
#endif
        )
{
    TRACE_START();
    // This function is called when the OS_TASK_RESUME_SIGNAL is caught.
    printf(OSAL_NAME_PREFIX"Thread resumed.\n");

    TRACE_END();
    return;
}


/*******************************************************************************
*
*   OS_vUpdateCPUStats
*
*******************************************************************************/
void OS_vUpdateCPUStats(void)
{
    BOOLEAN bSuccess = FALSE;

    TRACE_START()

    if (GsCPUCtrl.bCPUStatInitialized == FALSE)
    {
        bSuccess = bGetCPUUsage(&GsCPUCtrl.sCPUInitialStat);
        if (bSuccess != TRUE)
        {
            // Error!
            print_err(bSuccess, "Error! bGetCPUUsage()");
        }
        else
        {
            GsCPUCtrl.bCPUStatInitialized = TRUE;
        }
    }
    else
    {
        OS_CPU_STAT_STRUCT sCPUCurUsage;

        OSAL.bMemSet(&sCPUCurUsage, 0, sizeof(sCPUCurUsage));
        bSuccess = bGetCPUUsage(&sCPUCurUsage);
        if (bSuccess != TRUE)
        {
            // Error!
            print_err(bSuccess, "Error! bGetCPUUsage()");
        }
        else
        {
            GsCPUCtrl.un8CPUCurrent = un8CalcCPUUsage(&sCPUCurUsage,
                &GsCPUCtrl.sCPUInitialStat
                    );
            if (GsCPUCtrl.un8CPUMax < GsCPUCtrl.un8CPUCurrent)
            {
                GsCPUCtrl.un8CPUMax = GsCPUCtrl.un8CPUCurrent;
            }

            if (GsCPUCtrl.un8CPUMin > GsCPUCtrl.un8CPUCurrent)
            {
                GsCPUCtrl.un8CPUMin = GsCPUCtrl.un8CPUCurrent;
            }
            memcpy(&GsCPUCtrl.sCPUInitialStat, &sCPUCurUsage,
                sizeof(GsCPUCtrl.sCPUInitialStat)
                    );
        }
    }
    TRACE_END()

    return;
}


/*******************************************************************************
*
*   OS_eGetUtilizationCPU
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OS_eGetUtilizationCPU(
    UN8 *pun8Current,
    UN8 *pun8Max,
    UN8 *pun8Min
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    TRACE_START();

    if (GsCPUCtrl.bCPUStatInitialized == TRUE)
    {
        OS_CPU_STAT_STRUCT sCPUCurUsage;
        BOOLEAN bSuccess = FALSE;

        OSAL.bMemSet(&sCPUCurUsage, 0, sizeof(sCPUCurUsage));

        bSuccess = bGetCPUUsage(&sCPUCurUsage);
        if (bSuccess != TRUE)
        {
            // Error!
            print_err(bSuccess, "Error! bGetCPUUsage()");
            eReturnCode = OSAL_ERROR;
        }
        else
        {
            if (pun8Current != NULL)
            {
                *pun8Current = GsCPUCtrl.un8CPUCurrent;
            }
            if (pun8Max != NULL)
            {
                *pun8Max = GsCPUCtrl.un8CPUMax;
            }
            if (pun8Min != NULL)
            {
                *pun8Min = GsCPUCtrl.un8CPUMin;
            }
            eReturnCode = OSAL_SUCCESS;
        }
    }
    else
    {
        eReturnCode = OSAL_ERROR;
    }

    TRACE_END();

    return eReturnCode;
}

/*******************************************************************************
*
*   OS_vResetUtilizationCPU
*
*******************************************************************************/
void OS_vResetUtilizationCPU(void)
{
    TRACE_START();
    OSAL.bMemSet(&GsCPUCtrl, 0, sizeof(GsCPUCtrl));
    TRACE_END();
    return;
}

/*****************************************************************************
*
*   bGetCPUUsage
*
*****************************************************************************/
static BOOLEAN bGetCPUUsage(OS_CPU_STAT_STRUCT *psResult)
{
    BOOLEAN bResult = FALSE;

#if !(defined __QNX__) && !(defined ANDROID) && !(defined __INTEGRITY)
    FILE *fpstat = NULL;
    FILE *fstat = NULL;

    do
    {
        pid_t pid = 0;
        char pid_s[20];
        char stat_filepath[30] = "/proc/";
        int i = 0;
        long unsigned int cpu_time[10];

        if (psResult == NULL)
        {
            break;
        }

        memset(cpu_time, 0, sizeof(cpu_time));
        memset(psResult, 0, sizeof(*psResult));

        pid = getpid();
        //convert  pid to string
        snprintf(pid_s, sizeof(pid_s), "%d", pid);
        strncat(stat_filepath, pid_s, sizeof(stat_filepath) - strlen(stat_filepath) - 1);
        strncat(stat_filepath, "/stat", sizeof(stat_filepath) - strlen(stat_filepath) - 1);

        //open /proc/pid/stat
        fpstat = fopen(stat_filepath, "r");
        if(fpstat == NULL)
        {
            printf("FOPEN ERROR pid stat %s:\n", stat_filepath);
            break;
        }

        //open /proc/stat
        fstat = fopen("/proc/stat", "r");
        if(fstat == NULL)
        {
            printf("FOPEN ERROR\n");
            break;
        }

        i = fscanf(fpstat->pvHdl, "%*d %*s %*c %*d %*d %*d %*d %*d %*u %*u %*u %*u %*u %lu %lu %ld %ld",
            &psResult->utime_ticks,
            &psResult->stime_ticks,
            &psResult->cutime_ticks,
            &psResult->cstime_ticks
                );

        //read values from /proc/pid/stat
        if(i == EOF)
        {
            break;
        }

        //read+calc cpu total time from /proc/stat, on linux 2.6.35-23 x86_64
        //the cpu row has 10values could differ on different architectures :/
        i = fscanf(fstat->pvHdl, "%*s %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu",
            &cpu_time[0], &cpu_time[1], &cpu_time[2], &cpu_time[3], &cpu_time[4], &cpu_time[5],
            &cpu_time[6], &cpu_time[7], &cpu_time[8], &cpu_time[9]
                );
        if(i == EOF)
        {
            break;
        }

        for(i=0; i < 10;i++)
        {
            psResult->cpu_total_time += cpu_time[i];
        }

        bResult = TRUE;
    } while (FALSE);

    if (fstat != NULL)
    {
        fclose(fstat);
        fstat = NULL;
    }

    if (fpstat != NULL)
    {
        fclose(fpstat);
        fpstat = NULL;
    }

#else
    bResult = TRUE;
#endif

    return bResult;
}

/*****************************************************************************
*
*   un8CalcCPUUsage
*
*****************************************************************************/
static UN8 un8CalcCPUUsage(OS_CPU_STAT_STRUCT *psCurUsage, OS_CPU_STAT_STRUCT *psLastUsage)
{
    UN8 un8Result = 0;
    do
    {
        long int iCPUTime = 0;
        float fUserUsage = 0;
        float fSysUsage = 0;

        if ((psCurUsage == NULL) || (psLastUsage == NULL))
        {
            break;
        }

        iCPUTime = psCurUsage->cpu_total_time - psLastUsage->cpu_total_time;
        if (iCPUTime == 0)
        {
            break;
        }

        fUserUsage = ((psCurUsage->utime_ticks + psCurUsage->cutime_ticks)
            - (psLastUsage->utime_ticks + psLastUsage->cutime_ticks)) * 100 / iCPUTime;
        fSysUsage = ((psCurUsage->stime_ticks + psCurUsage->cstime_ticks)
            - (psLastUsage->stime_ticks + psLastUsage->cstime_ticks)) * 100 / iCPUTime;

        un8Result = (UN8)(fUserUsage + fSysUsage + 0.5);
    } while (FALSE);

    return un8Result;
}


#if OSAL_DEBUG == 1
/*****************************************************************************
*
*   vConfigureStdio
*
*****************************************************************************/
static void vConfigureStdio( void )
{
    int iFileId;

    TRACE_START();
    // Configure stdio...

    stdin->psIntf = &GsFSIntf;
    stdout->psIntf = &GsFSIntf;
    stderr->psIntf = &GsFSIntf;

#if defined ANDROID
#if OSAL_REMOTE_CONSOLE == 1
    stdin->pvHdl  = FAKE_STDIN;
    stdout->pvHdl = FAKE_STDOUT;
    stderr->pvHdl = FAKE_STDERR;
#else
    stdin->pvHdl = GsNativeIntf._Stdin;
    stdout->pvHdl = GsNativeIntf._Stdout;
    stderr->pvHdl = GsNativeIntf._Stderr;
#endif
#elif (defined __QNX__) || (defined __INTEGRITY)
#if OSAL_REMOTE_CONSOLE == 1
    stdin->pvHdl = FAKE_STDIN;
#else
    stdin->pvHdl = GsNativeIntf._Stdin;
#endif
    stdout->pvHdl = GsNativeIntf._Stdout;
    stderr->pvHdl = GsNativeIntf._Stderr;
#elif defined LINUX
#if OSAL_REMOTE_CONSOLE == 1
    stdin->pvHdl = FAKE_STDIN;
#else
    stdin->pvHdl = *GsNativeIntf._Stdin;
#endif
    stdout->pvHdl = *GsNativeIntf._Stdout;
    stderr->pvHdl = *GsNativeIntf._Stderr;
#else
#error "This function is defined for QNX / LINUX / ANDROID operation system"
#endif

    // Return the file descriptor for a stream
#if OSAL_REMOTE_CONSOLE == 1
    iFileId = iStartRemoteConsole();
    gConsoleSockfd = iFileId;
#else
    iFileId = OS_Fileno(stdin->pvHdl);
#endif
    if(iFileId == -1)
    {
        print_errno("Error! fileno()");
        TRACE_END();
        return;
    }

    // Configure parameters
#if OSAL_REMOTE_CONSOLE == 1
#else
    vConfigureStdioParameters(iFileId);
#endif

    TRACE_END();
    return;
}

#if OSAL_REMOTE_CONSOLE == 1
#else
/*****************************************************************************
*
*   vConfigureStdioParameters
*
*****************************************************************************/
static void vConfigureStdioParameters( int iFileId )
{
    struct termios termios_s;
    int iErr;

    TRACE_START();

    memset(&termios_s, 0, sizeof(termios_s));

    // Get attributes
    iErr = tcgetattr( iFileId, &termios_s);
    if(iErr != 0)
    {
        print_errno("Error! tcgetattr()");
        TRACE_END();
        return;
    }

    // Configure attributes...

    // The cfmakeraw() function sets the terminal attributes as follows:
    // termios_p->c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|
    //  IGNCR|ICRNL|IXON);
    // termios_p->c_oflag &= ~OPOST;
    // termios_p->c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
    // termios_p->c_cflag &= ~(CSIZE|PARENB);
    // termios_p->c_cflag |= CS8;
#if (defined __INTEGRITY)
    termios_s.c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL|IXON);
    termios_s.c_oflag &= ~OPOST;
    termios_s.c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
    termios_s.c_cflag &= ~(CSIZE|PARENB);
    termios_s.c_cflag |= CS8;
#elif !(defined __QNX__)
    cfmakeraw(&termios_s);
#else
    iErr = cfmakeraw(&termios_s);
    if(iErr != 0)
    {
        return;
    }
#endif

    // Input Modes
    termios_s.c_iflag = termios_s.c_iflag;

    // Output Modes
    termios_s.c_oflag = termios_s.c_oflag;

    // Control Modes
    termios_s.c_cflag = termios_s.c_cflag;

    // Local Modes
    termios_s.c_lflag = termios_s.c_lflag;

    // Control Characters
    termios_s.c_cc[VMIN]  =  1; /* At least one character */
    termios_s.c_cc[VTIME] =  0; /* No time component */

    // Set attributes
    iErr = tcsetattr( iFileId, TCSANOW, &termios_s);
    if(iErr != 0)
    {
        print_errno("Error! tcsetattr()");
        TRACE_END();
        return;
    }

    TRACE_END();
    return;
}
#endif

#if OSAL_REMOTE_CONSOLE == 1

/*****************************************************************************
*
*       iStartRemoteConsole
*
*****************************************************************************/
static int iStartRemoteConsole( void )
{
    int sockfd, port;
    struct sockaddr_in serv_addr;

    port = 5353;

    do
    {
        sockfd = socket(AF_INET, SOCK_STREAM, 0);
        if (sockfd < 0)
        {
            printf("REM_CONSOLE: ERROR opening socket\n");
            sleep(1);
        }
        else
        {
            bzero((char *) &serv_addr, sizeof(serv_addr));
            serv_addr.sin_family = AF_INET;
            serv_addr.sin_addr.s_addr = INADDR_ANY;
            serv_addr.sin_port = htons(port);
            if (bind(sockfd, (struct sockaddr *)
                &serv_addr, sizeof(serv_addr)) < 0)
            {
                printf("REM_CONSOLE: ERROR on binding port %d\n", port);
                close(sockfd);
                sockfd = -1;
                port++;
            }
            else
            {
                listen(sockfd, 5);
                printf("REM_CONSOLE: You may now connect to "
                    "remote console via %s:%d\n", "localhost", port);
                break;
            }
        }
    }
    while (TRUE);

    return sockfd;
}

#endif

#endif    //OSAL_DEBUG
