/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio Inc.                      */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio Inc.              */
/*                        Proprietary & Confidential                          */
/******************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"

#include "osal_core.h"
#include "osal_time.h"
#include "os_intf.h"

#if OSAL_DEBUG == 1
#include "osal_debug.h"
#define TIMER_LOGGING
#endif

#include "_osal_timer.h"

/*******************************************************************************
*
*   OSAL_eTimerCreate
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimerCreate (
    OSAL_OBJECT_HDL *phTimerObj,
    const char *pacName,
    OSAL_TIMER_HANDLER tTimerHandler,
    const void *pvTimerHandlerArgument
    )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_STRUCT *psObj = NULL;
    OSAL_OBJECT_HDL hTimerObj;
    OSAL_TIMER_INFO_STRUCT *psTimer = NULL;

    // Check input
    if(phTimerObj == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Initialize return object handle for error cases
    *phTimerObj = OSAL_INVALID_OBJECT_HDL;

    eReturnCode = OSALC_eCreateObject(
        OSAL_OBJECT_TYPE_TIMER,
        0,
        pacName,
        &hTimerObj,
        &psObj,
        (OSAL_OBJECT_INFO_UNION **)&psTimer
            );
    if((eReturnCode == OSAL_SUCCESS) && (psTimer != NULL))
    {
        // populate the object info
        psTimer->tTimerHandler = tTimerHandler;
        psTimer->pvTimerHandlerArgument = pvTimerHandlerArgument;
        psTimer->un32InitialOffset = 0;
        psTimer->un32PeriodicRate = 0;
        psTimer->hTimerObj = hTimerObj;
        psTimer->hActiveEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psTimer->bActive = FALSE;
        psTimer->un32ExpirationDelta = 0;
        psTimer->bAbsolute = FALSE; // we'll find out later

        // Populate the object
        *phTimerObj = hTimerObj;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTimerDelete
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimerDelete (
    OSAL_OBJECT_HDL hTimerObj
    )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_TIMER_INFO_STRUCT *psTimer, *psNextTimer;
    OSAL_LINKED_LIST_ENTRY hNextEntry;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTimerObj, OSAL_OBJECT_TYPE_TIMER);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Lock-up timer task's active list data and
    // associated linked list
    OSALT_vLockTimerData();

    // Extract timer info
    psTimer = &psObj->puInfo->sTimer;

    // Before we remove a timer from the active list we need to
    // first adjust the delta time of the next timer by adding
    // the delta time this timer had. Effectively closing
    // the gap. Then we can safely remove it from the list

    // Get the next timer entry
    hNextEntry = OSAL.hLinkedListNext(psTimer->hActiveEntry,
        (void*)((void *)&psNextTimer));
    if(hNextEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // Close the gap
        psNextTimer->un32ExpirationDelta +=
            psTimer->un32ExpirationDelta;
    }

    // Remove this timer from the active list
    eReturnCode = OSAL.eLinkedListRemove(psTimer->hActiveEntry);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Mark timer as inactive
        OSALT_vMarkInactivate(psTimer);
    }

    // Unlock active timer data
    OSALT_vUnlockTimerData();

    // Remove and destroy the object
    eReturnCode = OSALC_eRemoveObject((OSAL_OBJECT_HDL)psObj);
#if OSAL_OBJECT_TRACKING == 1
    if(eReturnCode == OSAL_SUCCESS)
    {
        {
            OSAL_OBJECT_HDL hTimerList;

            // Grab the timer linked list
            hTimerList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_TIMER);
            if(hTimerList != OSAL_INVALID_OBJECT_HDL)
            {
                UN32 un32Timers;

                // Get the number of timers in the list
                eReturnCode = OSAL.eLinkedListItems(
                    hTimerList, &un32Timers);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // If this is the last timer then go ahead
                    // and simply remove the timer task
                    if(un32Timers == 0)
                    {
                        // If this is the last timer to be deleted,
                        // then simply delete the task itself
                        OSALT_vTaskUninstall();
                    }
                }
            }
        }
    }
#endif /* OSAL_OBJECT_TRACKING == 1 */

    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OSAL_eTimerSetHandler
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimerSetHandler (
    OSAL_OBJECT_HDL hTimerObj,
    OSAL_TIMER_HANDLER tTimerHandler
    )
{
    OSAL_TIMER_INFO_STRUCT *psTimer;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTimerObj, OSAL_OBJECT_TYPE_TIMER);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract timer info
    psTimer = &psObj->puInfo->sTimer;

    // Lock-up timer task's active list data and
    // associated linked list
    OSALT_vLockTimerData();

    // If this timer is already active ignore the request
    if(psTimer->bActive == FALSE)
    {
        // Modify timer's info
        psTimer->tTimerHandler = tTimerHandler;

        // Indicate success
        eReturnCode = OSAL_SUCCESS;
    }
    else
    {
        // Indicate timer is active, cannot perform action
        eReturnCode = OSAL_ERROR_TIMER_ACTIVE;
    }

    // Unlock active timer data
    OSALT_vUnlockTimerData();

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTimerSetHandlerArg
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimerSetHandlerArg (
    OSAL_OBJECT_HDL hTimerObj,
    const void *pvTimerHandlerArgument
    )
{
    OSAL_TIMER_INFO_STRUCT *psTimer;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTimerObj, OSAL_OBJECT_TYPE_TIMER);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract timer info
    psTimer = &psObj->puInfo->sTimer;

    // Lock-up timer task's active list data and
    // associated linked list
    OSALT_vLockTimerData();

    // If this timer is already active ignore the request
    if(psTimer->bActive == FALSE)
    {
        // Modify timer's info
        psTimer->pvTimerHandlerArgument = pvTimerHandlerArgument;

        // Indicate success
        eReturnCode = OSAL_SUCCESS;
    }
    else
    {
        // Indicate timer is active, cannot perform action
        eReturnCode = OSAL_ERROR_TIMER_ACTIVE;
    }

    // Unlock active timer data
    OSALT_vUnlockTimerData();

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTimerStartRelative
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimerStartRelative (
    OSAL_OBJECT_HDL hTimerObj,
    UN32 un32InitialOffset,
    UN32 un32PeriodicRate
    )
{
    OSAL_TIMER_INFO_STRUCT *psTimer;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTimerObj, OSAL_OBJECT_TYPE_TIMER);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract timer info
    psTimer = &psObj->puInfo->sTimer;

    // Lock-up timer task's active list data and
    // associated linked list
    OSALT_vLockTimerData();

    // If this timer is already active, stop the timer first before
    // proceeding with the new timer parameters
    if(psTimer->bActive == TRUE)
    {
        // Stop timer, but do not remove from the active list
        OSALT_eRemoveFromActiveList(psTimer, FALSE);
    }

    // Copy in user provided data and overflow info...

    // Check input parameters from user
    // The initial offset must be less than
    // what the maximum number a N32 can hold (due to internal
    // implementation). So to handle larger timeouts
    // we need to set an overflow flag.
    if(un32InitialOffset > N32_MAX)
    {
        // Calculate overflow amount (this will be used once
        // the maximum time has elapsed.
        psTimer->un32InitialOffset = un32InitialOffset - N32_MAX;

        // Set flag
        psTimer->bInitialOffsetOverflow = TRUE;

        // Adjust user initial offset to the maximum
        un32InitialOffset = N32_MAX;
    }
    else
    {
        // Not an overflow value!

        // Copy user initial offset as is
        psTimer->un32InitialOffset = un32InitialOffset;

        // Clear flag
        psTimer->bInitialOffsetOverflow = FALSE;
    }

    // Copy user provided periodic rate and initialize overflow flag
    psTimer->un32PeriodicRate = un32PeriodicRate;
    psTimer->ePeriodicRateOverflowHdlrMode =
        OSAL_TIMER_OVERFLOW_HANDLE_MAX_TIME;

    // Calculate the number of remaining milliseconds
    // till this timer is to expire. At this point this is the
    // initial offset time. Once the timer expires this will be the
    // periodic rate (if there is one).
    psTimer->un32ExpirationDelta = un32InitialOffset;

    // This is a relative timer
    psTimer->bAbsolute = FALSE;

    // Add this timer to the active list
    eReturnCode = OSALT_eAddToActiveList(psTimer);

    // Unlock active timer data
    OSALT_vUnlockTimerData();

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTimerStartAbsolute
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimerStartAbsolute (
    OSAL_OBJECT_HDL hTimerObj,
    UN8 un8Second,
    UN8 un8Minute,
    UN8 un8Hour,
    UN16 un16Year,
    UN8 un8Day,
    UN8 un8Month
    )
{
    OSAL_TIMER_INFO_STRUCT *psTimer;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_STRUCT *psObj;
    UN32 un32LocalTime;

    // Determine the number of seconds we have been active since epoch.
    // But if we can't we probably don't support time...so bail!
    eReturnCode = OSAL.eTimeGetLocal(&un32LocalTime);
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Time error.
        return eReturnCode;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTimerObj, OSAL_OBJECT_TYPE_TIMER);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract timer info
    psTimer = &psObj->puInfo->sTimer;

    // Lock-up timer task's active list data and
    // associated linked list
    OSALT_vLockTimerData();

    // If this timer is already active, stop the timer first before
    // proceeding with the new timer parameters
    if(psTimer->bActive == TRUE)
    {
        // Stop timer, but do not remove from the active list
        OSALT_eRemoveFromActiveList(psTimer, FALSE);
    }

    // This is an absolute timer
    psTimer->bAbsolute = TRUE;

    // Fill in user parameters for this absolute timer
    psTimer->sAbsoluteTime.un8Second = un8Second;
    psTimer->sAbsoluteTime.un8Minute = un8Minute;
    psTimer->sAbsoluteTime.un8Hour = un8Hour;
    psTimer->sAbsoluteTime.un16Year = un16Year;
    psTimer->sAbsoluteTime.un8Day = un8Day;
    psTimer->sAbsoluteTime.un8Month = un8Month;

    // Compute timer and add it to the list
    eReturnCode = OSALT_eComputeAbsoluteTimer( psTimer,
        un32LocalTime);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Add this timer to the active list
        eReturnCode = OSALT_eAddToActiveList(psTimer);
    }

    // Unlock active timer data
    OSALT_vUnlockTimerData();

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTimerStop
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimerStop (
    OSAL_OBJECT_HDL hTimerObj
    )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTimerObj, OSAL_OBJECT_TYPE_TIMER);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Lock-up timer task's active list data and
    // associated linked list
    OSALT_vLockTimerData();

    // Remove timer from active list
    eReturnCode = OSALT_eRemoveFromActiveList(
        (OSAL_TIMER_INFO_STRUCT *)psObj->puInfo, TRUE);

    // Unlock active timer data
    OSALT_vUnlockTimerData();

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTimerRemaining
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimerRemaining (
    OSAL_OBJECT_HDL hTimerObj,
    UN32 *pun32TimeRemaining
    )
{
    OSAL_TIMER_INFO_STRUCT *psTimer;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OSALT_DELTA_ACCUMULATOR_STRUCT sDeltaAccumulator;

    // Verify input, and initialize if provided
    if(pun32TimeRemaining != NULL)
    {
        // Initialize time remaining
        *pun32TimeRemaining = 0;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTimerObj, OSAL_OBJECT_TYPE_TIMER);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract timer info
    psTimer = &psObj->puInfo->sTimer;

    // Lock-up timer task's active list data and
    // associated linked list
    OSALT_vLockTimerData();

    // If this timer is not active ignore the request
    if(psTimer->bActive == TRUE)
    {
        // Timer is active...
        eReturnCode = OSAL_SUCCESS;

        // Check if return pointer for value was provided, if not no
        // need to go through this calculation.
        if(pun32TimeRemaining != NULL)
        {
            UN32 un32MsecElapsed;

            // Extract The current number of elapsed msec
            // since the last time a timer was put in the
            // front of the list. This is the amount of time
            // that has elapsed since the timer's delta was set.
            un32MsecElapsed = OSALT_un32GetMsecElapsed(FALSE);

            // has more time elapsed than this timer's expiration delta?
            if (un32MsecElapsed < psTimer->un32ExpirationDelta)
            {
                // Initialize the delta accumulator struct, this will
                // allow accumulating delta time's to arrive at an absolute
                // time when the selected timer will expire.
                sDeltaAccumulator.un32TimerNumber = 0;
                sDeltaAccumulator.un32TimeRemaining = 0 - un32MsecElapsed;
                sDeltaAccumulator.hThisEntry = psTimer->hActiveEntry;

                // Iterate the list adding the delta for each timer
                // individually until the timer we desire is reached.
                eReturnCode = OSAL.eLinkedListIterate(
                    gsTimerTaskControl.hActiveTimerList,
                    OSALT_bAccumulateDelta, &sDeltaAccumulator);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Pass accumulated time back to the caller
                    *pun32TimeRemaining = sDeltaAccumulator.un32TimeRemaining;
                }
            }
            else
            {
                // timer has been pending longer than it was set for
                // we will report that the time remaining is 0 since it is
                // ready to go off
                *pun32TimeRemaining = 0;
                eReturnCode = OSAL_SUCCESS;
            }
        }
    }
    else
    {
        // This timer is not active
        eReturnCode = OSAL_TIMER_NOT_ACTIVE;
    }

    // Unlock active timer data
    OSALT_vUnlockTimerData();

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_hTimerGetHandleByName
*
*******************************************************************************/
OSAL_OBJECT_HDL OSAL_hTimerGetHandleByName (
    const char *pacName
    )
{
    OSAL_OBJECT_HDL hObjectHdl = OSAL_INVALID_OBJECT_HDL;
    OSAL_OBJECT_STRUCT *psObj;

    // Call OS specific API
    //hObjectHdl = OS.hTimerGetHandleByName(pacName);

    psObj = OSALC_psFindObjectFromName(OSAL_OBJECT_TYPE_TIMER, pacName);
    if(psObj != NULL)
    {
        hObjectHdl = (OSAL_OBJECT_HDL)psObj;
    }

    return hObjectHdl;
}

/*******************************************************************************
*
*   OSAL_eTimerGetInfo
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimerGetInfo (
    OSAL_OBJECT_HDL hTimerObj,
    OSAL_TIMER_INFO_STRUCT *psTimerInfo
    )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    OSAL_OBJECT_STRUCT *psObj;

    // Check provided structure pointer
    if(psTimerInfo == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTimerObj, OSAL_OBJECT_TYPE_TIMER);
    if(psObj != NULL)
    {
        // Lock-up timer task's active list data and
        // associated linked list
        OSALT_vLockTimerData();

        // Copy object info I have into object info provided to
        // OS-specific API. That API can then modify it with what
        // it knows, or do nothing.
        *psTimerInfo = psObj->puInfo->sTimer;


        // Unlock active timer data
        OSALT_vUnlockTimerData();

        // Report success
        eReturnCode = OSAL_SUCCESS;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTimerList
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimerList ( void )
{
#if (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_HDL hTimerList;
    UN32 un32Timers;
    OSAL_STATISTICS_STRUCT sStatistics;

    // Iterate the list of timers and list each entry

    // Grab the timer linked list
    hTimerList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_TIMER);
    if(hTimerList == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the number of timers in the list
    eReturnCode = OSAL.eLinkedListItems(hTimerList, &un32Timers);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the object's statistics
    OSALC_vGetObjectStatistics(OSAL_OBJECT_TYPE_TIMER, &sStatistics);

    // Check that one or more elements exist in the list
    if(un32Timers > 0)
    {
        // Print title
        printf("===================\n");
        printf("| OSAL Timer List |\n");
        printf("===================\n");

        // Initialize number of timers
        un32Timers = 0;

        // Lock-up timer task's active list data and
        // associated linked list
        OSALT_vLockTimerData();

        // Iterate the list dumping each element individually
        eReturnCode = OSAL.eLinkedListIterate(hTimerList,
            OSALT_bPrintTimerObject, &un32Timers);

        // Unlock active timer data
        OSALT_vUnlockTimerData();
    }
    else
    {
        // Indicate there are no objects
        eReturnCode = OSAL_NO_OBJECTS;
    }

    if(eReturnCode == OSAL_NO_OBJECTS)
    {
        printf("Object list is empty!\n");
        return eReturnCode;
    }
    else if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }
    else
    {
        // Print summary
        printf("\n%u Timer(s) currently allocated by the system.\n",
            un32Timers);

        // Print statistics
        printf("Current = %u, Minimum = %u, Maximum = %u\n",
            sStatistics.un32Current,
            sStatistics.un32Minimum,
            sStatistics.un32Maximum );

    }

    return OSAL_SUCCESS;
#else

    return OSAL_ERROR_UNSUPPORTED_API;

#endif /*(OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */
}

/*****************************************************************************
*
*      OSALT_bTaskInstall
*
*****************************************************************************/
BOOLEAN OSALT_bTaskInstall( OSAL_TASK_PRIORITY_ENUM ePriority )
{
    static BOOLEAN bFirstTime = TRUE;
    BOOLEAN bRetVal = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check that there is not already an instance running
    if(bFirstTime == FALSE)
    {
        // Sorry, ya only get one of these!
        return OSAL_ERROR;
    }
    else
    {
        // Ok, this is the one you get...no more!
        bFirstTime = FALSE;
    }

    // Check if task is already installed, if so...bail
    if(gsTimerTaskControl.hTask != OSAL_INVALID_OBJECT_HDL)
    {
        return FALSE;
    }

    // Initialize all task memory to zero
    OSAL.bMemSet( (void *)&gsTimerTaskControl, 0,
        sizeof(OSALT_TASK_CONTROL_STRUCT) );

    // Create install complete semaphore
    eReturnCode = OSAL.eSemCreate(&gsTimerTaskControl.hInstallSemaphore,
        "Timer Install", 0, 1, OSAL_SEM_OPTION_NONE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Create the task
        eReturnCode = OSAL.eTaskCreate(
            &gsTimerTaskControl.hTask,
            "Timer Task",
            OSALT_n32Task,
            (void*)&gsTimerTaskControl,
            OSALT_TASK_STACK_SIZE,
            ePriority,
            OSALT_TASK_OPTIONS);

        if(eReturnCode == OSAL_SUCCESS)
        {
            // Wait for task to install
            eReturnCode = OSAL.eSemTake(gsTimerTaskControl.hInstallSemaphore,
                OSAL_OBJ_TIMEOUT_INFINITE);
            if(eReturnCode == OSAL_SUCCESS)
            {
                bRetVal = TRUE;
            }
        }

        // Destroy the semaphore (no longer needed)
        if(gsTimerTaskControl.hInstallSemaphore != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eSemDelete(gsTimerTaskControl.hInstallSemaphore);
            gsTimerTaskControl.hInstallSemaphore = OSAL_INVALID_OBJECT_HDL;
        }
    }

    return bRetVal;
}

/*****************************************************************************
*
*       OSALT_vTaskUninstall
*
*       This function is used to remove the Task from the system.
*
*       Inputs:
*               None
*
*       Outputs:
*               None
*
*		Notes:
*				This function runs in the context of the caller.
*
*****************************************************************************/
void OSALT_vTaskUninstall( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if(gsTimerTaskControl.hTask != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eTaskDelete(gsTimerTaskControl.hTask);
        if(eReturnCode == OSAL_SUCCESS)
        {
            gsTimerTaskControl.hTask = OSAL_INVALID_OBJECT_HDL;
        }
    }

    return;
}

/*****************************************************************************
*
*      OSALT_un32GetMsecElapsed
*
*****************************************************************************/
static UN32 OSALT_un32GetMsecElapsed( BOOLEAN bReset )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32Accumulated = 0;
    UN32 un32Sec;
    UN16 un16MSec;

    // Extract The current number of elapsed msec
    // since the last time a timer was put in the
    // front of the list. This is the amount of time
    // that has elapsed since the timer's delta was set.
    eReturnCode = OS.eGetTime(&un32Sec, &un16MSec);
    if(eReturnCode == OSAL_SUCCESS)
    {
        static UN32 un32RefSec = 0;
        static UN16 un16RefMSec = 0;
        UN32 un32SecElapsed;
        UN16 un16MSecElasped;

        /**************************************************/
        OSAL.eEnterTaskSafeSection();

        // Get and set reference time in an atomic operation
        // access to the static variables must be exclusive.
        un32SecElapsed = un32RefSec;
        un16MSecElasped = un16RefMSec;

        // Reset reference time?
        if(bReset == TRUE)
        {
            un32RefSec = un32Sec;
            un16RefMSec = un16MSec;
        }

        OSAL.eExitTaskSafeSection();
        /**************************************************/

        // Compute accumulated elapsed time in milli-seconds
        un32Accumulated = (((un32Sec * 1000) + un16MSec) -
            ((un32SecElapsed * 1000) + un16MSecElasped));
    }

    return un32Accumulated;
}

/*******************************************************************************
*
*   OSALT_vHandleTimer
*
*******************************************************************************/
static void OSALT_vHandleTimer ( OSAL_TIMER_INFO_STRUCT *psTimer )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;

    // Check if we have a valid timer
    if(psTimer == NULL)
    {
        // Nevermind...
        return;
    }

    // This timer has expired, so the expiration delta is now zero
    psTimer->un32ExpirationDelta = 0;

    // Check if this timer expired because the overflow limit was reached
    // If so, we need to load in the overflow offset and re-insert the timer.
    if(psTimer->bInitialOffsetOverflow == TRUE)
    {
        // Remove this timer from the active list
        eReturnCode = OSALT_eRemoveFromActiveList(psTimer, FALSE);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Nevermind...
            return;
        }

        // Clear offset overflow flag (no longer needed and we don't
        // want to come back here again)
        psTimer->bInitialOffsetOverflow = FALSE;

        // Calculate the number of remaining milliseconds
        // till this timer is to expire. At this point this is the
        // initial offset time. Once the timer expires this will be the
        // periodic rate (if there is one).
        psTimer->un32ExpirationDelta = psTimer->un32InitialOffset;

        // Re-add it to the active list
        OSALT_eAddToActiveList(psTimer);

        // Do nothing, just return since timer did not really expire
        return;
    }

    // Now check if this timer is periodic, if so re-activeate it
    // otherwise our work is done.
    if(psTimer->un32PeriodicRate > 0)
    {
        // It's periodic

        // Remove this timer from the active list
        eReturnCode = OSALT_eRemoveFromActiveList(psTimer, FALSE);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Nevermind...
            return;
        }

        // The periodic rate must be less than
        // what the maximum number a N32 can hold (due to internal
        // implementation). So to handle larger timeouts
        // we need to initialize an overflow value.
        if(psTimer->un32PeriodicRate > N32_MAX)
        {
            // Check if this timer is expiring
            // after the first part of the rate (the part less
            // than N32_MAX in which it really has not expired yet
            // and must be re-inserted, or if it is the overflow part
            // which means it has really expired and all can run as normal.
            if( (psTimer->ePeriodicRateOverflowHdlrMode ==
                OSAL_TIMER_OVERFLOW_HANDLE_MAX_TIME) ||
                (psTimer->ePeriodicRateOverflowHdlrMode ==
                OSAL_TIMER_OVERFLOW_HANDLE_EXPIRATION) )
            {
                // Set the periodic rate timer to expire once the maximum
                // number of msec which can be represented expires.
                psTimer->un32ExpirationDelta = N32_MAX;
                psTimer->ePeriodicRateOverflowHdlrMode =
                    OSAL_TIMER_OVERFLOW_HANDLE_REMAINDER;

                // Now just simply allow this timer to be added into the
                // list with the maximum time.

                // Re-add it to the active list
                OSALT_eAddToActiveList(psTimer);

                if(psTimer->ePeriodicRateOverflowHdlrMode ==
                    OSAL_TIMER_OVERFLOW_HANDLE_MAX_TIME)
                {
                    // Do nothing, just return since timer did not really expire
                    return;
                }

                // Fall through and allow timer to run expiration handler
            }
            else
            {
                // We have expired after the maximum number of msec
                // which can be represented. Thus we need to calculate
                // the overflow and simply re-insert the timer with this
                // new value which is the remaining timeout.

                // This is the new expiration delta
                psTimer->un32ExpirationDelta =
                    psTimer->un32PeriodicRate - N32_MAX;
                psTimer->ePeriodicRateOverflowHdlrMode =
                    OSAL_TIMER_OVERFLOW_HANDLE_EXPIRATION;

                // Re-add it to the active list (with remaining time)
                OSALT_eAddToActiveList(psTimer);

                // Do nothing, just return since timer did not really expire
                return;
            }
        }
        else // Not an overflow timer, use rate directly
        {
            // Calculate the number of remaining milliseconds
            // till this timer is to expire. At this point forward this is the
            // periodic rate time.
            psTimer->un32ExpirationDelta = psTimer->un32PeriodicRate;

            // Re-add timer to the active list
            eReturnCode = OSALT_eAddToActiveList(psTimer);
            if(eReturnCode != OSAL_SUCCESS)
            {
                // something is wrong...
                return;
            }
        }
    }
    else
    {
        // It was a one-shot...we're done

        // Remove this timer from the active list
        eReturnCode = OSALT_eRemoveFromActiveList(psTimer, TRUE);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Nevermind...
            return;
        }
    }

    // Dereference timer object
    psObj = (OSAL_OBJECT_STRUCT *)psTimer->hTimerObj;

    // Expired timer debug
    printf("TIMER: Timer '%s' expired.\n",
#if OSAL_OBJECT_TRACKING == 1
        psObj->pacName
#else
        "Unknown"
#endif
        );

    // Call this timer's handler
    if(psTimer->tTimerHandler != NULL)
    {
        // Unlock active timer data, allow callback to manipulate timers
        OSALT_vUnlockTimerData();

        psTimer->tTimerHandler(
            psTimer->hTimerObj,
            (void *)psTimer->pvTimerHandlerArgument);

        // Re-lock active timer data
        OSALT_vLockTimerData();
    }

    return;
}

/*******************************************************************************
*
*   OSALT_n32Task
*
*******************************************************************************/
static N32 OSALT_n32Task ( void *pvArg )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bTaskInitialized, bTimerExpired;
    N32 n32Timeout = OSAL_OBJ_TIMEOUT_INFINITE;
    OSAL_TIMER_INFO_STRUCT *psActiveTimer;

    // Initialize this task!
    bTaskInitialized = OSALT_bTaskInitialize();
    if( bTaskInitialized == TRUE )
    {
        // Indicate task is installed by releasing the semaphore
        eReturnCode = OSAL.eSemGive(gsTimerTaskControl.hInstallSemaphore);
        if(eReturnCode == OSAL_SUCCESS)
        {
            OSAL_LINKED_LIST_ENTRY hActiveTimer;

#ifdef TIMER_LOGGING
            UN32 un32Time1Sec;
            UN16 un16Time1Msec;
#endif

            // Forever
            for(;;)
            {
                // Initialize timer pointer to NULL (no timer)
                psActiveTimer = NULL;

                // Timer pend debug
                printf("TIMER: Pending for %d msec...\n",
                    n32Timeout);

#ifdef TIMER_LOGGING
                OSAL.vTimeUp(&un32Time1Sec,  &un16Time1Msec);
#endif

                // Based on the timeout variable, pend on the semaphore
                // until either 1) a post is made (new timer has been
                // added to the front of the list) meaning we may need
                // to wait a shorter time now or 2) The pend times out
                // indicating the next timer(s) in the list needs to
                // be handled.
                eReturnCode = OSAL.eSemTake(gsTimerTaskControl.hTimerSem,
                    n32Timeout);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // A new timer has been placed at the front of the list
                    // Which means this one will/may expire sooner than we
                    // were already waiting.
                    bTimerExpired = FALSE;

#ifdef TIMER_LOGGING
                    {
                        UN32 un32DeltaMsec;
                        UN32 un32Time2Sec;
                        UN16 un16Time2Msec;

                        OSAL.vTimeUp(&un32Time2Sec,  &un16Time2Msec);
                        un32DeltaMsec = (((un32Time2Sec * 1000) + un16Time2Msec) -
                            ((un32Time1Sec * 1000) + un16Time1Msec));

                        // Delta time could cross msec boundaries, so only
                        // report this occurrence if the difference is more
                        // than one msec
                        if (un32DeltaMsec > (((UN32)n32Timeout) + 1))
                        {
                            fprintf(stderr, "TIMER: %u,  %d\n",
                                un32DeltaMsec, n32Timeout);
                        }
                    }
#endif
                }
                else if(eReturnCode == OSAL_SEM_NOT_AVAILABLE)
                {
                    // We were told to just check the active list
                    // again for any timers which may be ready to
                    // expire. So we should check the list.
                    bTimerExpired = FALSE;
                }
                else if(eReturnCode == OSAL_TIMEOUT)
                {
                    // A timeout occured, this means the first timer
                    // (or more) in the list is ready to blow.
                    // Pop it!
                    bTimerExpired = TRUE;
                }
                else
                {
                    // Something is wrong, or task is being shutdown
                    printf("Error! Task cannot pend on semaphore: %s\n",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                    OSAL.eTaskDelay(1); // Allow other tasks to run
                    break; // exit task
                }

                // Output some debug info
                printf("TIMER: Event Occured - %s (%s)\n",
                    bTimerExpired ? "Timer Expired" : "Check List",
                    OSAL.pacGetReturnCodeName(eReturnCode));

                // Lock-up timer task's active list data and
                // associated linked list
                OSALT_vLockTimerData();

                // Search the list (well not really) to return
                // the first entry in the list of active timers
                hActiveTimer = OSAL.hLinkedListFirst (
                    gsTimerTaskControl.hActiveTimerList,
                    (void **)&psActiveTimer );

                // Check if we have a valid timer ptr
                if ((hActiveTimer == OSAL_INVALID_LINKED_LIST_ENTRY) ||
                    (psActiveTimer == NULL))
                {
                    // No active timers in the list,
                    // go back and wait indefinitely for someone to create one
                    n32Timeout = OSAL_OBJ_TIMEOUT_INFINITE;

                    // Unlock active timer data
                    OSALT_vUnlockTimerData();

                    continue;
                }

                // We have a timer in the list, set our pend timer
                // to it's time to expire.
                n32Timeout =
                    (N32)psActiveTimer->un32ExpirationDelta;
                // Note! This cast is why we cannot have timers bigger
                // than an N32 without considering an overflow mechanism!

                // Is this timer ready to blow?
                if( (n32Timeout == OSAL_OBJ_TIMEOUT_NONE) ||
                    (bTimerExpired == TRUE) )
                {
                    // This timer has expired, handle it
                    OSALT_vHandleTimer(psActiveTimer);

                    // After a timer has expired always set the next pend time
                    // to zero to force it to pull the next timer off the list
                    n32Timeout = OSAL_OBJ_TIMEOUT_NONE;
                }

                // Unlock active timer data
                OSALT_vUnlockTimerData();

            } // Forever

        } // if(eReturnCode == OSAL_SUCCESS)

    } // if( bTaskInitialized == TRUE )

    // Unallocate task resources
    OSALT_vTaskUninitialize();

    return OSAL_TASK_REPORT_NO_ERROR;
}

/*****************************************************************************
*
*       OSALT_bTaskInitialize
*
*		This is the function used to Initialize the Timer Task
*
*       Inputs:
*               None
*
*       Outputs:
*               BOOLEAN		bSuccess
*								- TRUE if successful
*								- otherwise FALSE
*****************************************************************************/
static BOOLEAN OSALT_bTaskInitialize( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

#if OSAL_DEBUG == 1

    OSAL_RETURN_CODE_ENUM eDebugResult;

    // Install the OSAL Timer debug handler
    eDebugResult = OSAL.eDebugRegister(
        "OSAL Timer Task", OSALT_vDebugHandler, NULL );
    if(eDebugResult != OSAL_SUCCESS)
    {
        return FALSE;
    }

#endif /* OSAL_DEBUG == 1 */

    // Create an active timer linked-list
    // Do not select the "protect list option" since I will go ahead
    // and protect the entire structure and access myself
    eReturnCode = OSAL.eLinkedListCreate(
        &gsTimerTaskControl.hActiveTimerList,
        "Active Timers",
        NULL,
        OSAL_LL_OPTION_LINEAR);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Create a timer pend semaphore
    eReturnCode = OSAL.eSemCreate( &gsTimerTaskControl.hTimerSem,
        OSAL_NAME_PREFIX"Timer",
        0,
        1,
        OSAL_SEM_OPTION_NONE);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Create a timer data access semaphore
    eReturnCode = OSAL.eSemCreate( &gsTimerTaskControl.hTimerDataSem,
        OSAL_NAME_PREFIX"Timer Data",
        1,
        1,
        OSAL_SEM_OPTION_NONE);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Absolute timers need to be notified when anyone updates time.
    eReturnCode = OSAL.eTimeSetRegisterNotification(
        &gsTimerTaskControl.hNotificationHandle,
        OSAL_TIME_UPDATE_MASK_ALL,
        OSALT_vReProcessAbsoluteTimers,
        NULL
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Create a timer for reporting to the task monitor.
    // This is actually the only task that can do this because timer's always
    // execute from the context of the 'timer task' therefore reporting
    // from the appropriate task. Other tasks CANNOT do this.
    // This is a good way to verify the timer task is operating properly
    eReturnCode = OSAL.eTimerCreate(
        &gsTimerTaskControl.hReportTimer,
        OSAL_NAME_PREFIX"Report Timer",
        OSALT_vReportHandler,
        NULL );
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Register Task with OSAL
    eReturnCode = OSAL.eTaskRegister(
        OSALT_REPORTING_INTERVAL,
        OSALT_vSleepHandler,
        OSALT_vWakeupHandler,
        OSALT_vShutdownHandler,
        NULL );
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Now start the report timer as a relative one
    eReturnCode = OSAL.eTimerStartRelative(
        gsTimerTaskControl.hReportTimer,
        0, // Report immediately
        (OSALT_REPORTING_INTERVAL) * (1000 / 3) // Report at appropriate rate
        );
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*       OSALT_vTaskUninitialize
*
*		This function will un-initialize any components that
*		have been created at start up.
*
*       Inputs:
*               None
*
*       Outputs:
*               None
*
*****************************************************************************/
static void OSALT_vTaskUninitialize( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Unregister Task with OSAL
    OSAL.eTaskUnregister();

    // Unregister time handler
    if(gsTimerTaskControl.hNotificationHandle !=
        OSAL_TIME_NOTIFICATION_INVALID_OBJECT)
    {
        eReturnCode = OSAL.eTimeSetUnRegisterNotification(
            gsTimerTaskControl.hNotificationHandle);
        if(eReturnCode == OSAL_SUCCESS)
        {
            gsTimerTaskControl.hNotificationHandle =
                OSAL_TIME_NOTIFICATION_INVALID_OBJECT;
        }
    }

    // Delete timer data semaphore
    if(gsTimerTaskControl.hTimerDataSem != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(gsTimerTaskControl.hTimerDataSem);
        if(eReturnCode == OSAL_SUCCESS)
        {
            gsTimerTaskControl.hTimerDataSem = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Delete timer semaphore
    if(gsTimerTaskControl.hTimerSem != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(gsTimerTaskControl.hTimerSem);
        if(eReturnCode == OSAL_SUCCESS)
        {
            gsTimerTaskControl.hTimerSem = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy the active timer linked list
    if(gsTimerTaskControl.hActiveTimerList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            gsTimerTaskControl.hActiveTimerList,
            (OSAL_LL_RELEASE_HANDLER)OSALT_vMarkInactivate);
        if(eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListDelete(
                gsTimerTaskControl.hActiveTimerList);
            if(eReturnCode == OSAL_SUCCESS)
            {
                gsTimerTaskControl.hActiveTimerList =
                    OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    // Delete the semaphore (no longer needed) - force post
    if(gsTimerTaskControl.hInstallSemaphore != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(gsTimerTaskControl.hInstallSemaphore);
        if(eReturnCode == OSAL_SUCCESS)
        {
            gsTimerTaskControl.hInstallSemaphore = OSAL_INVALID_OBJECT_HDL;
        }
    }

#if OSAL_DEBUG == 1
    // Unregister ourselves from the debug task handler
    OSAL.eDebugUnregister();
#endif

#if OSAL_OBJECT_TRACKING == 1
    {
        OSAL_OBJECT_HDL hTimerList;

        // Delete all elements in the timer object linked list
        // Grab the timer linked list
        hTimerList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_TIMER);
        if(hTimerList != OSAL_INVALID_OBJECT_HDL)
        {
            // Remove timer elements from the linked list
            OSAL.eLinkedListRemoveAll(hTimerList, NULL );
        }
    }
#endif /* OSAL_OBJECT_TRACKING == 1 */

    // Invalidate task handle, to indicate task no longer exists
    gsTimerTaskControl.hTask = OSAL_INVALID_OBJECT_HDL;

    return;
}

/*****************************************************************************
*
*       OSALT_vMarkInactivate
*
*		This is the function used to remove a timer from the active list
*
*       Inputs:
*               void *pvData
*
*       Outputs:
*
*				None
*****************************************************************************/
static void OSALT_vMarkInactivate(void *pvData)
{
    OSAL_TIMER_INFO_STRUCT *psTimer = (OSAL_TIMER_INFO_STRUCT *)pvData;

    // We need to just simply mark the timer accordingly
    psTimer->hActiveEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psTimer->bActive = FALSE;
    psTimer->un32ExpirationDelta = 0;

    return;
}

/*******************************************************************************
*
*   OSALT_eRemoveFromActiveList
*
*******************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALT_eRemoveFromActiveList (
    OSAL_TIMER_INFO_STRUCT *psTimer,
    BOOLEAN bModifyList )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hNextEntry, hPreviousTimer;
    OSAL_TIMER_INFO_STRUCT *psNextTimer;
    UN32 un32MsecElapsed;

    // Remove timer from active list and mark as inactive

    // First check if timer is ACTIVE, if not, we cannot stop it
    if(psTimer->hActiveEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // This timer is not active
        return OSAL_TIMER_NOT_ACTIVE;
    }

    // Post timer event if this timer is to be removed from the beginning of
    // the list (or it is the first in the list) If it is, the previous
    // entry will be NULL
    hPreviousTimer = OSAL.hLinkedListPrev(psTimer->hActiveEntry, NULL);
    if(hPreviousTimer == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // Get the number of msec elapsed since the first timer
        // was added to the list. Then also set the new number
        // of msec elapsed to zero.
        un32MsecElapsed = OSALT_un32GetMsecElapsed(TRUE);

        // Adjust this timer's expiration delta
        if(un32MsecElapsed > psTimer->un32ExpirationDelta)
        {
            // Set this timer's expiration delta to zero.
            // This one was ready to (or should have expired).
            psTimer->un32ExpirationDelta = 0;
        }
        else
        {
            // Set new expiration delta prior to comparison.
            psTimer->un32ExpirationDelta -= un32MsecElapsed;
        }

        // Post a timer event, to indicate the first timer has been removed
        // This will indicate to the timer task a new timer is at the front
        // of the list. This will force the timer task to run and
        // re-pend with the new delta time for a new timer.
        // Timer was will not run until the lock on the active list
        // has been removed however, so this is safe to do here
        OSAL.eSemGive(gsTimerTaskControl.hTimerSem);
    }

    // Before we remove a timer from the active list we need to
    // first adjust the delta time of the next timer by adding
    // the delta time this timer had. Effectively closing
    // the gap. Then we can safely remove it from the list.

    // Get the next timer entry
    hNextEntry = OSAL.hLinkedListNext(psTimer->hActiveEntry,
        (void*)((void *)&psNextTimer));
    if(hNextEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // Close the gap
        psNextTimer->un32ExpirationDelta +=
            psTimer->un32ExpirationDelta;
    }

    // Remove this timer from the active list (only if
    // modify list flag is set).
    if(bModifyList == TRUE)
    {
        eReturnCode = OSAL.eLinkedListRemove(psTimer->hActiveEntry);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Mark timer as inactive
            OSALT_vMarkInactivate(psTimer);
        }
    }
    else
    {
        // All is well
        eReturnCode = OSAL_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       OSALT_eInsertTimer
*
*****************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALT_eInsertTimer(
    OSAL_OBJECT_HDL hTimerList,
    OSAL_TIMER_INFO_STRUCT *psTimer
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

    do
    {
        OSAL_LINKED_LIST_ENTRY hEntry;
        OSAL_TIMER_INFO_STRUCT *psListTimer;

        if (psTimer->hActiveEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // this timer in list already, lets remove it to add again
            eReturnCode = OSAL.eLinkedListRemove(psTimer->hActiveEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                // error, skip here
                break;
            }

            psTimer->hActiveEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }

        hEntry = OSAL.hLinkedListFirst(hTimerList, (void**)&psListTimer);
        if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // there are no timers in the list, just adding our timer
            eReturnCode = OSAL.eLinkedListAdd(hTimerList,
                &psTimer->hActiveEntry, psTimer);
        }
        else
        {
            UN32 un32MsecElapsed;

            // This is the first one, so go ahead an lob off
            // any time which has expired already. leaving the new
            // relative number with the true msec till this timer will
            // expire.

            // The 'true' delta of the first timer is actually it's delta
            // minus the number of msec elapsed since it was added.
            // So before we start comparing timers we need to compensate for
            // this.
            un32MsecElapsed = OSALT_un32GetMsecElapsed(FALSE);

            // Subtract the number of msec elapsed from the current (first)
            // timer's delta. This is the new delta. We do a (possibly
            // unnecessary) check to make sure the number of elapsed msec
            // is not larger than the first timer's. If we find this is
            // so we simply make it expire now. This means a number of
            // msec have elapsed past this timer's expiration, so we need to
            // catch up.
            if(un32MsecElapsed > psListTimer->un32ExpirationDelta)
            {
                // Set new expiration delta prior to comparison.
                // This one is ready to (or should have expired).
                psListTimer->un32ExpirationDelta = 0;
            }
            else
            {
                // Set new expiration delta prior to comparison.
                psListTimer->un32ExpirationDelta -= un32MsecElapsed;
            }

            while (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                if( psTimer->un32ExpirationDelta <=
                    psListTimer->un32ExpirationDelta )
                {
                    // New timer expires before or at the same time as
                    // list timer which is the current one.

                    // Lets insert our new timer before this one
                    psTimer->hActiveEntry = hEntry;
                    eReturnCode = OSAL.eLinkedListAddBeforeEntry(hTimerList,
                        &psTimer->hActiveEntry, psTimer);
                    if (eReturnCode == OSAL_SUCCESS)
                    {
                        // Compute new delta of list timer to our newly 
                        // inserted timer since list timer expires after
                        // new timer
                        psListTimer->un32ExpirationDelta -= 
                            psTimer->un32ExpirationDelta;
                    }

                    // insert completed, exit here
                    break;
                }
                else
                {
                    // List timer expires before new timer, lets decrease
                    // our new timer delta and move to next
                    psTimer->un32ExpirationDelta -= 
                        psListTimer->un32ExpirationDelta;
                    hEntry = OSAL.hLinkedListNext(hEntry, 
                        (void**)&psListTimer);
                }
            }

            if (eReturnCode != OSAL_SUCCESS)
            {
                // error happened in cycle
                break;
            }

            if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                // if we are here it means that timer has not been inserted
                // to list since its delta is larger than delta of timers in
                // the list. So our timer should be the last.

                // receiving the last entry
                hEntry = OSAL.hLinkedListLast(hTimerList, NULL);
                if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
                {
                    // unexpected error
                    eReturnCode = OSAL_ERROR;
                    break;
                }

                psTimer->hActiveEntry = hEntry;

                eReturnCode = OSAL.eLinkedListAddAfterEntry(hTimerList, 
                    &psTimer->hActiveEntry, psTimer);
            }
        }

    } while (FALSE);

    return eReturnCode;
}

/*******************************************************************************
*
*   OSALT_eAddToActiveList
*
*******************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALT_eAddToActiveList (
    OSAL_TIMER_INFO_STRUCT *psTimer )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hPreviousTimer;

    // Mark this timer as active.
    psTimer->bActive = TRUE;

    // Add this timer to the active list
    eReturnCode = OSALT_eInsertTimer(gsTimerTaskControl.hActiveTimerList,
        psTimer);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_ERROR_CANNOT_ADD_OBJECT;
    }

    // Post timer event if this timer was put at the beginning of
    // the list (or it is the first in the list)

    // Check if this timer is the first in the list
    // If it is, the previous entry will be NULL
    hPreviousTimer = OSAL.hLinkedListPrev(psTimer->hActiveEntry, NULL);
    if(hPreviousTimer == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // Since a timer was placed at the beginning of the list
        // we need to reset the number of ticks accumulated
        // since the last time a timer was added to the front
        // of the list
        OSALT_un32GetMsecElapsed(TRUE);

        // Post a timer event, to indicate to the timer task
        // a new timer has been added to the front of the list. This
        // will force the timer task to run and re-pend with the
        // new delta time for this new timer.
        eReturnCode = OSAL.eSemGive(gsTimerTaskControl.hTimerSem);
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSALT_vLockTimerData
*
*******************************************************************************/
static void OSALT_vLockTimerData( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if(gsTimerTaskControl.hTimerDataSem != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemTake(
            gsTimerTaskControl.hTimerDataSem,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if(eReturnCode != OSAL_SUCCESS)
        {
            printf("FATAL ERROR! Timer Data Semaphore 'take' failed.\n");
            OSAL.eTaskSuspend(OSAL.hTaskGetHandle());
            // we're dead!
        }
    }

    return;
}

/*******************************************************************************
*
*   OSALT_vUnlockTimerData
*
*******************************************************************************/
static void OSALT_vUnlockTimerData( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if(gsTimerTaskControl.hTimerDataSem != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemGive(gsTimerTaskControl.hTimerDataSem);
        if(eReturnCode != OSAL_SUCCESS)
        {
            printf("FATAL ERROR! Timer Data Semaphore 'give' failed.\n");
            OSAL.eTaskSuspend(OSAL.hTaskGetHandle());
            // we're dead!
        }
    }

    return;
}

/*******************************************************************************
*
*   OSALT_bAccumulateDelta
*
*******************************************************************************/
static BOOLEAN OSALT_bAccumulateDelta ( void *pvData, void *pvArg )
{
    OSAL_TIMER_INFO_STRUCT *psTimer = (OSAL_TIMER_INFO_STRUCT *)pvData;
    OSALT_DELTA_ACCUMULATOR_STRUCT *psDeltaAccumulator =
        (OSALT_DELTA_ACCUMULATOR_STRUCT *)pvArg;

    // Check if argument is valid
    if( (psDeltaAccumulator == NULL) || (psTimer == NULL) )
    {
        return FALSE;
    }

    // Add this timer's delta to the accumulated time
    psDeltaAccumulator->un32TimeRemaining +=
        psTimer->un32ExpirationDelta;

    // Check if we are done
    if(psDeltaAccumulator->hThisEntry == psTimer->hActiveEntry)
    {
        // Check if the initial overflow flag is set, if so this means
        // we need to also wait some overflow amount in addition to
        // the current expiration delta.
        if(psTimer->bInitialOffsetOverflow == TRUE)
        {
            // Factor in remaining overflow amount
            psDeltaAccumulator->un32TimeRemaining +=
                psTimer->un32InitialOffset;
        }
        // Check if periodic rate is in overflow, if so we need to
        // account for that.
        else if(psTimer->un32PeriodicRate > N32_MAX)
        {
            // An overflow amount is in effect. Have we already serviced
            // the maximum value? If not we need to add not only this
            // remaining time, but also any time represented by the
            // overflow amount
            if(psTimer->ePeriodicRateOverflowHdlrMode ==
                OSAL_TIMER_OVERFLOW_HANDLE_REMAINDER)
            {
                // Factor in overflow amount
                psDeltaAccumulator->un32TimeRemaining +=
                    psTimer->un32PeriodicRate - N32_MAX;
            }
        }

        // No need to go any further
        return FALSE;
    }

    // Keep going
    return TRUE;
}

/*******************************************************************************
*
*   OSALT_bReProcessAbsoluteTimer
*
*******************************************************************************/
static BOOLEAN OSALT_bReProcessAbsoluteTimer ( void *pvData, void *pvArg )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_TIMER_INFO_STRUCT *psTimer = (OSAL_TIMER_INFO_STRUCT *)pvData;
    UN32 *pun32CurrentTime = (UN32 *)pvArg;

    // Check if this is an absolute timer, if not simply continue on
    if(psTimer->bAbsolute == TRUE)
    {
        // This is an absolute timer.

        // Remove this timer from the active list (for re-insertion)
        eReturnCode = OSALT_eRemoveFromActiveList(psTimer, FALSE);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Re-compute timer
            eReturnCode = OSALT_eComputeAbsoluteTimer(
                psTimer,
                *pun32CurrentTime);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Add this timer to the active list
                OSALT_eAddToActiveList(psTimer);
            }
            else if (eReturnCode == OSAL_ERROR_INVALID_INPUT)
            {
                // This timer probably occurred in the past, now
                // that the time was changed. So we need to simply
                // set it to expire immediately
                struct tm tTime, tTimeBuf;
                TIME_T tCurrentTime = (TIME_T)(*pun32CurrentTime);

                tTime = *OSAL.localtime_r(&tCurrentTime, &tTimeBuf);

                psTimer->sAbsoluteTime.un8Second = tTime.tm_sec;
                psTimer->sAbsoluteTime.un8Minute = tTime.tm_min;
                psTimer->sAbsoluteTime.un8Hour = tTime.tm_hour;
                psTimer->sAbsoluteTime.un8Day = tTime.tm_mday;
                psTimer->sAbsoluteTime.un8Month = tTime.tm_mon + 1;
                psTimer->sAbsoluteTime.un16Year = tTime.tm_year + 1900;

                OSALT_eComputeAbsoluteTimer( psTimer,
                    *pun32CurrentTime);
            }
            else
            {
                // Something is wrong with this timer
                // Remove it for good.
                OSALT_eRemoveFromActiveList(psTimer, TRUE);
            }
        }
    }

    // Continue iterating
    return TRUE;
}

/*******************************************************************************
*
*   OSALT_eComputeAbsoluteTimer
*
*******************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALT_eComputeAbsoluteTimer (
    OSAL_TIMER_INFO_STRUCT *psTimer,
    UN32 un32CurrentTime )
{
    UN32 un32InitialOffset;
    struct tm tTime;
    TIME_T tTod;

    // Check input parameters from user
    if( (psTimer->sAbsoluteTime.un8Second >= 60) ||
        (psTimer->sAbsoluteTime.un8Minute >= 60) ||
        (psTimer->sAbsoluteTime.un8Hour >= 24) ||
        // Absolute time functions are based on seconds since
        // January 1, 1970 (epoch). Thus a UN32 representation
        // of this time will roll-over in 136.19 years.
        // (1970 + 136 = 2106)
        (psTimer->sAbsoluteTime.un16Year < 1970) ||
        (psTimer->sAbsoluteTime.un16Year >= 2106) ||
        (psTimer->sAbsoluteTime.un8Day == 0) ||
        (psTimer->sAbsoluteTime.un8Day > 31) ||
        (psTimer->sAbsoluteTime.un8Month == 0) ||
        (psTimer->sAbsoluteTime.un8Month > 12)
        )
    {
        // Time is provided out of range
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Now find out how many seconds from January 1, 1970 the absolute
    // time they provided is.

    // seconds after the minute (from 0)
    tTime.tm_sec = psTimer->sAbsoluteTime.un8Second;

    // minutes after the hour (from 0)
    tTime.tm_min = psTimer->sAbsoluteTime.un8Minute;

    // hour of the day (from 0)
    tTime.tm_hour = psTimer->sAbsoluteTime.un8Hour;

    // day of the month (from 1)
    tTime.tm_mday = psTimer->sAbsoluteTime.un8Day;

    // month of the year (from 0)
    tTime.tm_mon = psTimer->sAbsoluteTime.un8Month - 1;

    // years since 1900 (from 0)
    tTime.tm_year = psTimer->sAbsoluteTime.un16Year - 1900;

    // filled in by OSAL.mktime, days since sunday (from 0)
    tTime.tm_wday = 0;

    // filled in by OSAL.mktime, day of the year (from 0)
    tTime.tm_yday = 0;

    // No dst in effect for this time
    tTime.tm_isdst = 0;

    // Call API to determine seconds since epoch (in calendar time)
    tTod = OSAL.mktime(&tTime);

    // Check that the time the user provided is at least equal to or
    // greater than the current time.
    if((TIME_T)un32CurrentTime > tTod)
    {
        // Sorry, I cannot set an alarm in the past
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Subtract off the number of seconds that have already
    // expired since epoch.
    tTod -= (TIME_T)un32CurrentTime;

    // Now we need to make sure that the timer is not any further into the
    // future than 49.7 days which is the maximum number of msec a UN32
    // can hold.
    if(tTod > (TIME_T)UN32_MAX / 1000)
    {
        // Sorry, this timer cannot be set this far in advance!
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Convert the number of seconds till the timer expires to msec
    un32InitialOffset = (UN32)tTod * 1000;

    // The initial offset must be less than
    // what the maximum number a N32 can hold (due to internal
    // implementation). So to handle larger timeouts
    // we need to set an overflow flag.
    if(un32InitialOffset > N32_MAX)
    {
        // Calculate overflow amount (this will be used once
        // the maximum time has elapsed.
        psTimer->un32InitialOffset = un32InitialOffset - N32_MAX;

        // Set flag
        psTimer->bInitialOffsetOverflow = TRUE;

        // Adjust user initial offset to the maximum
        un32InitialOffset = N32_MAX;
    }
    else
    {
        // Not an overflow value!

        // Copy user initial offset as is
        psTimer->un32InitialOffset = un32InitialOffset;

        // Clear flag
        psTimer->bInitialOffsetOverflow = FALSE;
    }

    // Absolute timers are not periodic, they are one-shot
    psTimer->un32PeriodicRate = 0;
    psTimer->ePeriodicRateOverflowHdlrMode =
        OSAL_TIMER_OVERFLOW_HANDLE_MAX_TIME;

    // Calculate the number of remaining milliseconds
    // till this timer is to expire. At this point this is the
    // initial offset time for absolute timers
    psTimer->un32ExpirationDelta = un32InitialOffset;

    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OSALT_vReProcessAbsoluteTimers
*
*******************************************************************************/
static void OSALT_vReProcessAbsoluteTimers(
    OSAL_TIME_UPDATE_MASK tUpdateMask,
    void *pvArg
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32LocalTime;

    eReturnCode = OSAL.eTimeGetLocal(&un32LocalTime);
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Time error.
        return;
    }

    // Any time a new time is set, we need to scan the list of
    // timers for any which are 'absolute' and re-insert them into the
    // timer list. This can be done by simply removing it from the list
    // and then re-starting it. This will force a new relative time to
    // be computed.
    // Lock-up timer data and associated linked list
    OSALT_vLockTimerData();

    // Iterate the list of active timers, any timer which is
    // absolute must be re-processed. Note that the time provided
    // is always local time. That means it has GMT and DST applied.
    OSAL.eLinkedListIterate(
        gsTimerTaskControl.hActiveTimerList,
        OSALT_bReProcessAbsoluteTimer, (void *)&un32LocalTime );

    // Unlock timer data
    OSALT_vUnlockTimerData();

    return;
}

/*****************************************************************************
*
*       OSALT_vSleepHandler
*
*****************************************************************************/
static void OSALT_vSleepHandler( void *pvArg )
{
    // Nothing to do

    return;
}

/*****************************************************************************
*
*       OSALT_vWakeupHandler
*
*****************************************************************************/
static void OSALT_vWakeupHandler( void *pvArg )
{
    // Nothing to do

    return;
}

/*****************************************************************************
*
*       OSALT_vShutdownHandler
*
*****************************************************************************/
static void OSALT_vShutdownHandler( void *pvArg )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Delete timer semaphore which will force the timer task to
    // run and return (i.e. uninstall itself).
    if(gsTimerTaskControl.hTimerSem != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(gsTimerTaskControl.hTimerSem);
        if(eReturnCode == OSAL_SUCCESS)
        {
            gsTimerTaskControl.hTimerSem = OSAL_INVALID_OBJECT_HDL;
        }
    }

    return;
}

/*****************************************************************************
*
*       OSALT_vReportHandler
*
*****************************************************************************/
static void OSALT_vReportHandler( OSAL_OBJECT_HDL hTimer, void *pvArg )
{
    // Check-in with the task monitor
    OSAL.eTaskReport(OSAL_TASK_REPORT_NO_ERROR);

    return;
}

#if OSAL_DEBUG == 1

/*****************************************************************************
*
*       OSALT_vDebugHandler
*
*****************************************************************************/
static void OSALT_vDebugHandler( char *pcCmdLine, const void *pvArg )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char *acTokens = " \n";

    // Make sure NULL pointer is not passed
    if (pcCmdLine != NULL)
    {
        // Tokenize the command line characters
        pcCmdLine = strtok( pcCmdLine, acTokens );
        if (pcCmdLine == NULL)
        {
            return;
        }

        if (strcmp(pcCmdLine, "timers") == 0)
        {
            eReturnCode = OSAL.eTimerList();
            if( (eReturnCode != OSAL_SUCCESS) &&
                (eReturnCode != OSAL_NO_OBJECTS) )
            {
                printf("Error! Could not get object list: %s\n",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else if (strcmp(pcCmdLine, "active") == 0)
        {
            OSALT_vActiveTimerList();
        }
        else if (strcmp(pcCmdLine, "create") == 0)
        {
            int iValue;
            OSAL_OBJECT_HDL hTimer;
            char *pacName;
            OSAL_RETURN_CODE_ENUM eReturnCode;
            UN32 un32Sec;
            UN16 un16Msec;

            // Get next token...
            pacName = strtok( NULL, acTokens );

            // Get next token...
            pcCmdLine = strtok( NULL, acTokens );

            iValue = 0;
            if(NULL != pcCmdLine)
            {
                iValue = atoi(pcCmdLine);
            }

            // Get time since power-up
            OSAL.vTimeUp(&un32Sec, &un16Msec);

            // Create a timer
            eReturnCode = OSAL.eTimerCreate(
                &hTimer, pacName,
                OSALT_vTimerCallback, (void *)iValue);
            if(eReturnCode == OSAL_SUCCESS)
            {
                printf("Created timer %s (iValue = %d), hdl = 0x%p\n"
                    "\tat %u:%u\n",
                    pacName, iValue, hTimer, un32Sec, un16Msec);
            }
            else
            {
                printf("Error! OSAL.eTimerCreate() returned %s\n",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else if (strcmp(pcCmdLine, "delete") == 0)
        {
            char *pacTimerName;
            OSAL_OBJECT_HDL hTimer;

            // Get next token...
            pacTimerName = strtok( NULL, acTokens );

            // Find the timer this belongs to
            hTimer = OSAL.hTimerGetHandleByName(pacTimerName);
            if(hTimer != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eTimerDelete(hTimer);

                if(eReturnCode != OSAL_SUCCESS)
                {
                    printf("Unable to delete timer! %s\n",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }
                else
                {
                    printf("Destroyed timer, hdl = 0x%p\n", hTimer);
                }
            }
            else
            {
                printf("Error! Cannot find timer named %s\n",
                    pacTimerName);
            }
        }
        else if (strcmp(pcCmdLine, "rel") == 0)
        {
            char *pacTimerName;
            OSAL_OBJECT_HDL hTimer;
            UN32 un32InitialOffset, un32PeriodicRate;
            OSAL_RETURN_CODE_ENUM eReturnCode;
            UN32 un32Sec;
            UN16 un16Msec;

            // Get next token...
            pacTimerName = strtok( NULL, acTokens );

            // Find the timer this belongs to
            hTimer = OSAL.hTimerGetHandleByName(pacTimerName);
            if(hTimer != OSAL_INVALID_OBJECT_HDL)
            {
                // Get next tokens...
                pcCmdLine = strtok( NULL, acTokens );
                if (pcCmdLine != NULL)
                {
                    un32InitialOffset = (UN32)atoi(pcCmdLine);

                    pcCmdLine = strtok( NULL, acTokens );
                    if (pcCmdLine != NULL)
                    {
                        un32PeriodicRate = (UN32)atoi(pcCmdLine);

                        // Get time since power-up
                        OSAL.vTimeUp(&un32Sec, &un16Msec);

                        eReturnCode = OSAL.eTimerStartRelative(
                            hTimer,
                            un32InitialOffset,
                            un32PeriodicRate );

                        if(eReturnCode != OSAL_SUCCESS)
                        {
                            printf("Unable to start timer! %s\n",
                                OSAL.pacGetReturnCodeName(eReturnCode));
                        }
                        else
                        {
                            printf("started timer, io = %u, "
                                "pr = %u, hdl = 0x%p\n"
                                "\tat %u:%u\n",
                                un32InitialOffset, un32PeriodicRate, hTimer,
                                un32Sec, un16Msec);
                        }
                    }
                }
            }
            else
            {
                printf("Error! Cannot find timer named %s\n",
                    pacTimerName);
            }
        }
        else if (strcmp(pcCmdLine, "abs") == 0)
        {
            char *pacTimerName, *pcMonth, *pcDay, *pcYear,
                *pcHour, *pcMinute, *pcSecond;
            OSAL_OBJECT_HDL hTimer;
            UN8 un8Second, un8Minute, un8Hour, un8Day, un8Month;
            UN16 un16Year;
            char *acTokens = " /:\n";

            // Get next token...
            pacTimerName = strtok( NULL, acTokens );

            // Find the timer this belongs to
            hTimer = OSAL.hTimerGetHandleByName(pacTimerName);
            if(hTimer != OSAL_INVALID_OBJECT_HDL)
            {
                // Get next tokens...
                pcMonth = strtok( NULL, acTokens );
                pcDay = strtok( NULL, acTokens );
                pcYear = strtok( NULL, acTokens );
                pcHour = strtok( NULL, acTokens );
                pcMinute = strtok( NULL, acTokens );
                pcSecond = strtok( NULL, acTokens );

                if ((pcMonth == NULL) || (pcDay == NULL) ||
                    (pcYear == NULL) || (pcHour == NULL) ||
                    (pcMinute == NULL) || (pcSecond == NULL))
                {
                    printf("\nInvalid parameters!\n");
                    return;
                }

                un8Month = (UN32)atoi(pcMonth);
                un8Day = (UN32)atoi(pcDay);
                un16Year = (UN32)atoi(pcYear);
                un8Hour = (UN32)atoi(pcHour);
                un8Minute = (UN32)atoi(pcMinute);
                un8Second = (UN32)atoi(pcSecond);

                eReturnCode = OSAL.eTimerStartAbsolute(
                    hTimer,
                    un8Second,
                    un8Minute,
                    un8Hour,
                    un16Year,
                    un8Day,
                    un8Month );

                if(eReturnCode != OSAL_SUCCESS)
                {
                    printf("Unable to start timer! %s\n",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }
                else
                {
                    printf("started timer for %u/%u/%u at %02u:%02u:%02u,"
                        " hdl = 0x%p\n",
                        un8Month, un8Day, un16Year,
                        un8Hour, un8Minute, un8Second,
                        hTimer);
                }
            }
            else
            {
                printf("Error! Cannot find timer named %s\n",
                    pacTimerName);
            }
        }
        else if (strcmp(pcCmdLine, "stop") == 0)
        {
            char *pacTimerName;
            OSAL_OBJECT_HDL hTimer;

            // Get next token...
            pacTimerName = strtok( NULL, acTokens );

            // Find the timer this belongs to
            hTimer = OSAL.hTimerGetHandleByName(pacTimerName);
            if(hTimer != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eTimerStop(hTimer);

                if(eReturnCode != OSAL_SUCCESS)
                {
                    printf("Unable to stop timer! %s\n",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }
                else
                {
                    printf("Stopped timer, hdl = 0x%p\n", hTimer);
                }
            }
            else
            {
                printf("Error! Cannot find timer named %s\n",
                    pacTimerName);
            }
        }
        else if (strcmp(pcCmdLine, "rem") == 0)
        {
            char *pacTimerName;
            OSAL_OBJECT_HDL hTimer;
            UN32 un32TimeRemaining = 0;

            // Get next token...
            pacTimerName = strtok( NULL, acTokens );

            // Find the timer this belongs to
            hTimer = OSAL.hTimerGetHandleByName(pacTimerName);
            if(hTimer != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eTimerRemaining(hTimer,
                    &un32TimeRemaining);

                if(eReturnCode != OSAL_SUCCESS)
                {
                    printf("Unable to get remaining time for timer! %s\n",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }
                else
                {
                    printf("Time remaining is %u msec, hdl = 0x%p\n",
                        un32TimeRemaining, hTimer);
                }
            }
            else
            {
                printf("Error! Cannot find timer named %s\n",
                    pacTimerName);
            }
        }
        else if (strcmp(pcCmdLine, "time") == 0)
        {
            UN32 un32CurrentTime;
            UN32 un32DST_Minutes = 0;
            N32 n32GMT_Minutes = 0;

            OSAL.eTimeGetGMToffset(&n32GMT_Minutes);
            OSAL.eTimeGetDSTadjustment(&un32DST_Minutes);

            /* Get the number of seconds since 'epoch' from the system */
            eReturnCode = OSAL.eTimeGet(&un32CurrentTime);
            if((eReturnCode == OSAL_SUCCESS) || (eReturnCode == OSAL_ERROR_INVALID_TIME))
            {
                TIME_T tTod;
                struct tm tmbuf;
                char cBuf[OSAL_ASCBUFSIZE];

                printf("%u seconds have elapsed since epoch\n",
                    un32CurrentTime );

                tTod = un32CurrentTime;
                printf("%s (LOCAL)\n", OSAL.ctime_r(&tTod, cBuf));
                printf("%s (GMT)\n",
                    OSAL.asctime_r(
                        OSAL.gmtime_r(&tTod, &tmbuf), cBuf));
                printf("[GMT:%d, DST:%u] minutes\n",
                    n32GMT_Minutes,
                    un32DST_Minutes
                        );

                printf("Result: %s\n",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else if (strcmp(pcCmdLine, "set") == 0)
        {
            struct tm tTime;
            TIME_T tTod;

            OSAL_vParseTime(pcCmdLine, &tTod, &tTime);

            /* Set the number of seconds since 'epoch' in the system */
            eReturnCode = OSAL.eTimeSet((UN32)tTod);
            if(eReturnCode == OSAL_SUCCESS)
            {
                UN32 un32DST_Minutes = 0;
                N32 n32GMT_Minutes = 0;
                char cBuf[OSAL_ASCBUFSIZE];
                OSAL.eTimeGetGMToffset(&n32GMT_Minutes);
                OSAL.eTimeGetDSTadjustment(&un32DST_Minutes);
                printf("Local time has been set to %s [GMT:%d, DST:%u]\n",
                    OSAL.ctime_r(&tTod, cBuf),
                    n32GMT_Minutes,
                    un32DST_Minutes
                        );
            }
            else
            {
                printf("Error! Unable to set time\n");
            }
        }
        else if (strcmp(pcCmdLine, "delay") == 0)
        {
            struct osald_delay {
                TIME_T tTod;
                UN32 un32Time;
                UN32 un32Seconds;
                UN16 un16Msecs;
            };
            N32 n32Msec, n32MsecElapsed;
            struct osald_delay sStart, sEnd;
            char cBuf[OSAL_ASCBUFSIZE];

            // Get next token...
            pcCmdLine = strtok( NULL, acTokens );
            if (pcCmdLine == NULL)
            {
                return;
            }
            n32Msec = atoi(pcCmdLine);

            printf("Delay for %d msec...\n", n32Msec);

            /* Get the 'uptime' from the system */
            OSAL.vTimeUp(&sStart.un32Seconds, &sStart.un16Msecs);

            /* Get the number of seconds since 'epoch' from the system */
            OSAL.eTimeGetLocal(&sStart.un32Time);

            /* Get time from the system */
            sStart.tTod = sStart.un32Time;

            // Delay
            OSAL.eTaskDelay(n32Msec);

            /* Get the 'uptime' from the system */
            OSAL.vTimeUp(&sEnd.un32Seconds, &sEnd.un16Msecs);

            /* Get the number of seconds since 'epoch' from the system */
            OSAL.eTimeGetLocal(&sEnd.un32Time);

            /* Get time from the system */
            sEnd.tTod = sEnd.un32Time;

            printf("START\n");
            printf("%u seconds have elapsed since epoch\n",
                sStart.un32Time );
            printf("System uptime: ");
            OSALD_vPrintTime(sStart.un32Seconds, sStart.un16Msecs);
            printf("%s\n", OSAL.ctime_r(&sStart.tTod, cBuf));

            printf("END\n");
            printf("%u seconds have elapsed since epoch\n",
                sEnd.un32Time );
            printf("System uptime: ");
            OSALD_vPrintTime(sEnd.un32Seconds, sEnd.un16Msecs);
            printf("%s\n\n", OSAL.ctime_r(&sEnd.tTod, cBuf));

            n32MsecElapsed =
                (((sEnd.un32Seconds * 1000) + sEnd.un16Msecs) -
                ((sStart.un32Seconds * 1000) + sStart.un16Msecs));

            printf("Delay requested %d msec, measured is %d msec (delta = %d).\n",
                n32Msec, n32MsecElapsed, abs(n32Msec - n32MsecElapsed));
        }
        else if (strcmp(pcCmdLine, "e") == 0)
        {
            OSAL.vControlOutputThisTask(TRUE);
            printf("Task output enabled\n");
        }
        else if (strcmp(pcCmdLine, "d") == 0)
        {
            printf("Task output disabled\n");
            OSAL.vControlOutputThisTask(FALSE);
        }
        else if ((strcmp(pcCmdLine, "help") == 0) ||
            (strcmp(pcCmdLine, "HELP") == 0))
        {
            printf("\nAvailable commands:\n");
            printf("\ttimers                -\tdisplay all timers\n");
            printf("\tactive                -\tdisplay all active timers\n");
            printf("\tcreate name [arg]     -\tcreate a timer\n");
            printf("\tdelete name           -\tdelete a timer\n");
            printf("\trel name i_msec p_msec-\tstart relative timer\n");
            printf("\tabs                   -\tstart absolute timer\n"
                "\t\tname mm/dd/yyyy hh:mm:ss\n");
            printf("\tstop name             -\tstop timer\n");
            printf("\trem name              -\ttime remaining\n");
            printf("\ttime                  -\tdisplay current time\n");
            printf("\tset                   -\tset current time\n"
                "\t\tmm/dd/yyyy hh:mm:ss\n");
            printf("\tdelay [msec]          -\tdelay in msec\n");
            printf("\te                     -\tenable output\n");
            printf("\td                     -\tdisable output\n");
            printf("\thelp                  -\tdisplay this message\n");
            printf("\texit                  -\texit this menu\n");
        }
        else if (strcmp(pcCmdLine, "exit") == 0)
        {
            // need to signal sw to Normal Mode!!
            OSAL.vDebugExitHandler();
        }
        else
        {
            printf("\nInvalid command.  Type 'help' for a list of commands.\n");
        }
    }

    return;
}

/*******************************************************************************
*
*   OSALT_vTimerCallback
*
*******************************************************************************/
static void OSALT_vTimerCallback ( OSAL_OBJECT_HDL hTimer, void *pvArg )
{
    static UN32 un32Count = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    int iValue = (int)pvArg;
    UN32 un32Sec, un32Time;
    UN16 un16Msec;
    char cBuf[OSAL_ASCBUFSIZE];

    // Get time since power-up
    OSAL.vTimeUp(&un32Sec, &un16Msec);

    /* Get the number of seconds since 'epoch' from the system */
    eReturnCode = OSAL.eTimeGet(&un32Time);
    if((eReturnCode == OSAL_SUCCESS) || (eReturnCode == OSAL_ERROR_INVALID_TIME))
    {
        TIME_T tTod = (TIME_T)un32Time;

        printf("%s\n", OSAL.ctime_r(&tTod, cBuf));
        printf("Result: %s\n",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    // Indicate timer fired
    printf("OSALT_vTimerCallback() Fired!\t"
        "iValue=%d\tun32Count=%u\tun32Sec=%u\tun16Msec=%u\tun32Time=%u\n",
        iValue, un32Count, un32Sec, un16Msec, un32Time);

    // Increment and check counter
    if(un32Count++ >= 100)
    {
        // Do something with this if needed.
        un32Count = 0;
    }

    return;
}

/*******************************************************************************
*
*   OSALT_vActiveTimerList
*
*******************************************************************************/
static void OSALT_vActiveTimerList ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_HDL hTimerList;
    UN32 un32Timers, un32MsecElapsed;
    OSALT_DELTA_ACCUMULATOR_STRUCT sDeltaAccumulator;

    // Iterate the list of timers and list each entry

    // Grab the timer linked list
    hTimerList = gsTimerTaskControl.hActiveTimerList;

    // Get the number of timers in the list
    eReturnCode = OSAL.eLinkedListItems(hTimerList, &un32Timers);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return;
    }

    // Check that one or more elements exist in the list
    if(un32Timers > 0)
    {
        // Print title
        printf("==========================\n");
        printf("| OSAL Active Timer List |\n");
        printf("==========================\n");
    }

    // Extract The current number of elapsed msec
    // since the first timer was added to the list

    // Get the number of msec elapsed since the first timer
    // was added to the list.
    un32MsecElapsed = OSALT_un32GetMsecElapsed(FALSE);

    // Initialize timer numbers
    sDeltaAccumulator.un32TimerNumber = 0;

    // Initialize the msec elapsed (to compensate)
    sDeltaAccumulator.un32TimeRemaining = 0 - un32MsecElapsed;

    // Initialize the delta accumulator struct
    // We set this to NULL because we want to list all the active timers.
    sDeltaAccumulator.hThisEntry = NULL;

    // Lock-up timer task's active list data and
    // associated linked list
    OSALT_vLockTimerData();

    // Iterate the list adding the delta for each timer
    // individually until the timer we desire is reached.
    eReturnCode = OSAL.eLinkedListIterate(
        gsTimerTaskControl.hActiveTimerList,
        OSALT_bPrintActiveTimer, &sDeltaAccumulator);

    // Unlock active timer data
    OSALT_vUnlockTimerData();

    if(eReturnCode == OSAL_NO_OBJECTS)
    {
        printf("Object list is empty!\n");
        return;
    }
    else if(eReturnCode != OSAL_SUCCESS)
    {
        return;
    }
    else
    {
        // Print summary
        printf("\n%u Timer(s) currently active in the system.\n",
            un32Timers);
    }

    return;
}

/*******************************************************************************
*
*   OSALT_bPrintActiveTimer
*
*******************************************************************************/
static BOOLEAN OSALT_bPrintActiveTimer ( void *pvData, void *pvArg )
{
    BOOLEAN bOk;
    OSAL_TIMER_INFO_STRUCT *psTimer = (OSAL_TIMER_INFO_STRUCT *)pvData;
    OSALT_DELTA_ACCUMULATOR_STRUCT *psDeltaAccumulator =
        (OSALT_DELTA_ACCUMULATOR_STRUCT *)pvArg;
    UN32 un32TimeRemaining;

    // Accumulate time
    psDeltaAccumulator->un32TimeRemaining +=
        psTimer->un32ExpirationDelta;

    bOk = OSALT_bPrintTimer(pvData, &psDeltaAccumulator->un32TimerNumber);

    // Get the current time remaining after adding this timer's delta
    un32TimeRemaining = psDeltaAccumulator->un32TimeRemaining;

    // Check if the initial overflow flag is set, if so this means
    // we need to also wait some overflow amount in addition to
    // the current expiration delta.
    if(psTimer->bInitialOffsetOverflow == TRUE)
    {
        // Factor in remaining overflow amount
        un32TimeRemaining +=
            psTimer->un32InitialOffset;
    }
    // Check if periodic rate is in overflow, if so we need to
    // account for that.
    else if(psTimer->un32PeriodicRate > N32_MAX)
    {
        // An overflow amount is in effect. Have we already serviced
        // the maximum value? If not we need to add not only this
        // remaining time, but also any time represented by the
        // overflow amount
        if(psTimer->ePeriodicRateOverflowHdlrMode ==
            OSAL_TIMER_OVERFLOW_HANDLE_REMAINDER)
        {
            // Factor in overflow amount
            un32TimeRemaining +=
                psTimer->un32PeriodicRate - N32_MAX;
        }
    }

    if(psTimer->bAbsolute == TRUE)
    {
        printf("\tABSOLUTE TIMER:\n");
        printf("\tExpires: %u/%u/%u %u:%u:%u\n",
            psTimer->sAbsoluteTime.un8Month,
            psTimer->sAbsoluteTime.un8Day,
            psTimer->sAbsoluteTime.un16Year,
            psTimer->sAbsoluteTime.un8Hour,
            psTimer->sAbsoluteTime.un8Minute,
            psTimer->sAbsoluteTime.un8Second
            );
    }
    else
    {
        printf("\tRELATIVE TIMER:\n");
    }

    printf("\tDelta (msec) = %u\n",
        psTimer->un32ExpirationDelta);
    printf("\tTime to expire (msec) = %u\n",
        un32TimeRemaining);

    return bOk;
}

/*******************************************************************************
*
*   OSALT_bPrintTimer
*
*******************************************************************************/
static BOOLEAN OSALT_bPrintTimer ( void *pvData, void *pvArg )
{
    OSAL_TIMER_INFO_STRUCT *psTimer = (OSAL_TIMER_INFO_STRUCT *)pvData;
    UN32 *pun32TimerNumber = (UN32 *)pvArg;
    OSAL_OBJECT_STRUCT *psCreatorObj, *psObj = (OSAL_OBJECT_STRUCT *)psTimer->hTimerObj;
    const char *pacCreatorObjName = "Unknown";

    // Prepare timer number
    (*pun32TimerNumber)++;

    // Extract the creator task's name
    psCreatorObj = (OSAL_OBJECT_STRUCT *)psObj->hCreatorObj;
    if(psCreatorObj != NULL)
    {
        // Extract name of creator
#if OSAL_OBJECT_TRACKING == 1
        pacCreatorObjName = psCreatorObj->pacName;
#else
        pacCreatorObjName = "Unknown";
#endif
    }

    puts("");
    printf("Timer #%u\n", *pun32TimerNumber);
    printf("Name = '%s' (hdl = 0x%p)\n",
#if OSAL_OBJECT_TRACKING == 1
        psObj->pacName,
#else
        "Unknown",
#endif
        psTimer->hTimerObj
        );
    printf("\tCreator = %s\n", pacCreatorObjName);
    printf("\tTimer Handler = 0x%08X, Arg = 0x%p\n",
        psTimer->tTimerHandler, psTimer->pvTimerHandlerArgument);
    printf("\tActive = %s\n",
        psTimer->bActive ? "True" : "False");
    printf("\tInitial Offset (msec) = %u, Overflow = %s\n",
        psTimer->un32InitialOffset,
        psTimer->bInitialOffsetOverflow ? "True" : "False");
    printf("\tPeriodic Rate (msec) = %u, Overflow = %s\n",
        psTimer->un32PeriodicRate,
        psTimer->un32PeriodicRate > N32_MAX ? "True" : "False");

    // Keep iterating
    return TRUE;
}

#if OSAL_OBJECT_TRACKING == 1
/*******************************************************************************
*
*   OSALT_bPrintTimerObject
*
*******************************************************************************/
static BOOLEAN OSALT_bPrintTimerObject ( void *pvData, void *pvArg )
{
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)pvData;
    BOOLEAN bOk = FALSE;
    OSAL_TIMER_INFO_STRUCT sTimer;

    // Check if argument is valid
    if( (pvArg == NULL) || (psObj == NULL) )
    {
        return FALSE;
    }

    // Point to this object's info

    // Copy object info I have into object info provided to
    // OS-specific API. That API can then modify it with what
    // it knows, or do nothing.
    sTimer = psObj->puInfo->sTimer;

    bOk = OSALT_bPrintTimer(&sTimer, pvArg);

    return bOk;
}

#endif /* (OSAL_OBJECT_TRACKING == 1) */
#endif /* (OSAL_DEBUG == 1) */
