/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio Inc.                      */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio Inc.              */
/*                        Proprietary & Confidential                          */
/******************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"

#include "osal_core.h"
#include "os_intf.h"

#if OSAL_DEBUG == 1
#include "osal_debug.h"
#define TIMER_LOGGING
#endif

#include "_osal_time.h"

/*****************************************************************************
*
*      OSALT_bTimeInitialize
*
*****************************************************************************/
BOOLEAN OSALT_bTimeInitialize( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Create a time global data access mutex
    eReturnCode = OSAL.eSemCreate(
        &gsTimeControl.sCurrent.hMutex,
        OSAL_NAME_PREFIX"Time Data",
        0, // Available
        1, // Resources
        OSAL_SEM_OPTION_NONE);
    if(OSAL_SUCCESS != eReturnCode)
    {
        OSALT_vTimeUninitialize();
        return FALSE;
    }

    // Create a time-update list mutex
    eReturnCode = OSAL.eSemCreate(
        &gsTimeControl.hListMutex,
        OSAL_NAME_PREFIX"Time-Update List",
        0, // Available
        1, // Resources
        OSAL_SEM_OPTION_NONE);
    if(OSAL_SUCCESS != eReturnCode)
    {
        OSALT_vTimeUninitialize();
        return FALSE;
    }

    // Create a list of notification callbacks
    // Do not select the "protect list option" since I will go ahead
    // and protect the entire structre and access myself
    eReturnCode = OSAL.eLinkedListCreate(
        &gsTimeControl.hList,
        "TimeSet Notification",
        NULL,
        OSAL_LL_OPTION_CIRCULAR|
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS);
    if(OSAL_SUCCESS != eReturnCode)
    {
        OSALT_vTimeUninitialize();
        return FALSE;
    }

    // Everything initialized, let'em rip
    OSAL.eSemGive(gsTimeControl.hListMutex);
    OSAL.eSemGive(gsTimeControl.sCurrent.hMutex);

    return TRUE;
}

/*****************************************************************************
*
*      OSALT_vTimeUninitialize
*
*****************************************************************************/
void OSALT_vTimeUninitialize( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Take the time list control
    eReturnCode = OSAL.eSemTake(
        gsTimeControl.hListMutex, OSAL_OBJ_TIMEOUT_INFINITE);
    if(OSAL_SUCCESS == eReturnCode)
    {
        // Destroy the notification callback list
        if(OSAL_INVALID_OBJECT_HDL != gsTimeControl.hList)
        {
            // Remove all entries from the list and destroy each content entry
            // since these are pre-allocated LL entries, we can't use
            // eRemoveAll and free the entry in the RemoveAll callback because
            // that would end up freeing the LL portion of the memory and then
            // eRemoveAll gets confused. So do it the old fashioned way.

            OSAL_LINKED_LIST_ENTRY hEntry;
            OSALT_UPDATE_HANDLER_ENTRY_STRUCT *psEntry;

            do
            {
                hEntry = OSAL.hLinkedListFirst(
                    gsTimeControl.hList, (void**)&psEntry);
                if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
                {
                    OSAL.eLinkedListRemove(hEntry);
                    OSAL.vLinkedListMemoryFree(psEntry);
                }

            } while (OSAL_INVALID_LINKED_LIST_ENTRY != hEntry);

            // Destroy the list itself
            eReturnCode = OSAL.eLinkedListDelete(gsTimeControl.hList);
            if(eReturnCode == OSAL_SUCCESS)
            {
                gsTimeControl.hList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Delete time global data mutex
        if(OSAL_INVALID_OBJECT_HDL != gsTimeControl.hListMutex)
        {
            eReturnCode = OSAL.eSemDelete(gsTimeControl.hListMutex);
            if(OSAL_SUCCESS == eReturnCode)
            {
                gsTimeControl.hListMutex = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    // Delete time global data mutex
    if(OSAL_INVALID_OBJECT_HDL != gsTimeControl.sCurrent.hMutex)
    {
        eReturnCode = OSAL.eSemDelete(gsTimeControl.sCurrent.hMutex);
        if(OSAL_SUCCESS == eReturnCode)
        {
            gsTimeControl.sCurrent.hMutex = OSAL_INVALID_OBJECT_HDL;
        }
    }

    return;
}

/*******************************************************************************
*
*   OSAL_eTimeSet
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimeSet (
    UN32 un32Seconds
    )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    UN32 un32UpTimeSec;

    // Set the current notion of time.
    // Do not effect our notion of 'uptime' time, instead just set a bias
    // variable and eTimeGet() will apply it when required. This is because
    // the OS interface does not (and should) not provide us with a means
    // to modify the uptime directly.

    // Grab current 'uptime'
    // This will help us determine how many seconds the OS believes
    // have elapsed since epoch.
    eReturnCode = OS.eGetTime(&un32UpTimeSec, NULL);
    if(OSAL_SUCCESS == eReturnCode)
    {
        // We were provided the current number of
        // seconds since January 1, 1970.

        // Make sure we are able to handle this
        if(un32UpTimeSec <= un32Seconds)
        {
            OSALT_TIME_BIAS_STRUCT sNewBias;

            // Copy existing bias into new bias
            bGetTimeBias(&sNewBias);

            // Compute bias
            // Now subtract from the new time the OS notion of time
            // This will be added whenever the time it retrieved
            sNewBias.un32UpTimeSeconds = un32Seconds - un32UpTimeSec;

            // Set new bias
            vSetTimeBias(&sNewBias);

            // Success
            eReturnCode = OSAL_SUCCESS;
        }
        else
        {
            // Error! Cannot use this bias
            eReturnCode = OSAL_ERROR_INVALID_TIME;
        }
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTimeSetGMToffset
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimeSetGMToffset (
    N32 n32Minutes
        )
{
    OSALT_TIME_BIAS_STRUCT sNewBias = {0,0,0};

    // Check input. Must be +/- 12 hours
    if((n32Minutes > (12*60)) || (n32Minutes < -(12*60)))
    {
        // Out of range
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Copy existing bias into new bias
    bGetTimeBias(&sNewBias);

    // Apply new bias
    sNewBias.n32GMT_Seconds = n32Minutes * 60; // minutes->seconds

    // Set new bias
    vSetTimeBias(&sNewBias);

    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OSAL_eTimeSetDSTadjustment
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimeSetDSTadjustment (
    UN32 un32Minutes
        )
{
    OSALT_TIME_BIAS_STRUCT sNewBias = {0,0,0};

    // Check input. Must be <= +1 hours
    if(un32Minutes > 60)
    {
        // Out of range
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Copy existing bias into new bias
    bGetTimeBias(&sNewBias);

    // Apply new bias
    sNewBias.un32DST_Seconds = un32Minutes * 60; // minutes->seconds

    // Set new bias
    vSetTimeBias(&sNewBias);

    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OSAL_eTimeGet
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimeGet (
    UN32 *pun32Seconds
    )
{
    BOOLEAN bTimeValid = FALSE;
    UN32 un32UpTimeSec;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Get the current notion of time.

    // Check input
    if(NULL == pun32Seconds)
    {
        // Use our own, maybe they just want to know if it's valid
        pun32Seconds = &un32UpTimeSec;
    }

    // Grab the os-time bias and current uptime

    // This will help us determine how many seconds the OS believes
    // have elapsed since epoch.
    eReturnCode = OS.eGetTime(&un32UpTimeSec, NULL);
    if(OSAL_SUCCESS == eReturnCode)
    {
        OSALT_TIME_BIAS_STRUCT sBias = {0, 0, 0};

        // Grab current 'uptime' bias's
        bTimeValid = bGetTimeBias(&sBias);

        // Now add the time bias's to the 'uptime'
        *pun32Seconds = un32UpTimeSec + sBias.un32UpTimeSeconds;
    }

    // We may adjust the return value if time has actually never
    // been set. If this is the case return a different code.
    if(FALSE == bTimeValid)
    {
        eReturnCode = OSAL_ERROR_INVALID_TIME;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTimeGetLocal
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimeGetLocal (
    UN32 *pun32Seconds
    )
{
    UN32 un32Seconds;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check input
    if(NULL == pun32Seconds)
    {
        // Use our own, maybe they just want to know if it's valid
        pun32Seconds = &un32Seconds;
    }

    // Get the current notion of local time.
    eReturnCode = OSAL.eTimeGet(pun32Seconds);
    if(OSAL_SUCCESS == eReturnCode)
    {
        OSALT_TIME_BIAS_STRUCT sBias = {0, 0, 0};

        // Grab current 'uptime' bias's
        bGetTimeBias(&sBias);

        // Now add the local time bias's
        *pun32Seconds += sBias.n32GMT_Seconds + sBias.un32DST_Seconds;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTimeGetGMToffset
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimeGetGMToffset (
    N32 *pn32Minutes
    )
{
    OSALT_TIME_BIAS_STRUCT sBias = {0, 0, 0};

    // Get the current notion of GMT.

    // Check input
    if(NULL == pn32Minutes)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Grab current bias's
    bGetTimeBias(&sBias);

    // Give them the GMT offset
    *pn32Minutes = sBias.n32GMT_Seconds / 60; // seconds->minutes

    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OSAL_eTimeGetDSTadjustment
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimeGetDSTadjustment (
    UN32 *pun32Minutes
    )
{
    OSALT_TIME_BIAS_STRUCT sBias = {0, 0, 0};

    // Get the current notion of DST.

    // Check input
    if(NULL == pun32Minutes)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Grab current bias's
    bGetTimeBias(&sBias);

    // Give them the DST offset
    *pun32Minutes = sBias.un32DST_Seconds / 60; // seconds->minutes

    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OSAL_vTimeUp
*
*******************************************************************************/
void OSAL_vTimeUp (
    UN32 *pun32Seconds,
    UN16 *pun16Msecs
    )
{
    // Grab the os-time current uptime.
    OS.eGetTime(pun32Seconds, pun16Msecs);

    return;
}

/*******************************************************************************
*
*   OSAL_eTimeSetRegisterNotification
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimeSetRegisterNotification (
    OSAL_TIME_NOTIFICATION_OBJECT *phNotificationHandle,
    OSAL_TIME_UPDATE_MASK tUpdateMask,
    OSAL_TIME_UPDATE_HANDLER vHandler,
    void *pvHandlerArg
    )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSALT_UPDATE_HANDLER_ENTRY_STRUCT *psEntry;

    // Caller must provide a handle pointer, a handler and must have asked
    // for something and that something must be something valid to ask for.

    if(phNotificationHandle == NULL)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Initialize return object handle for error cases
    *phNotificationHandle = OSAL_TIME_NOTIFICATION_INVALID_OBJECT;

    if(vHandler == NULL)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    if((tUpdateMask & ~OSAL_TIME_UPDATE_MASK_ALL) !=
        OSAL_TIME_UPDATE_MASK_NONE)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Allocate memory and populate entry
    psEntry = (OSALT_UPDATE_HANDLER_ENTRY_STRUCT *)
        OSAL.pvLinkedListMemoryAllocate(
            OSAL_NAME_PREFIX"Time Data",
                sizeof(OSALT_UPDATE_HANDLER_ENTRY_STRUCT), TRUE);

    if (psEntry != NULL)
    {
        // Populate this entry
        psEntry->tUpdateMask = tUpdateMask;
        psEntry->vHandler = vHandler;
        psEntry->pvHandlerArg = pvHandlerArg;

        // Take control of the time global data
        eReturnCode = OSAL.eSemTake(gsTimeControl.hListMutex,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if(OSAL_SUCCESS == eReturnCode)
        {
            // Add this entry
            eReturnCode = OSAL.eLinkedListAdd(
                  gsTimeControl.hList,
                  &psEntry->hThisEntry,
                  (void *)psEntry
                      );
            if (eReturnCode == OSAL_SUCCESS)
            {
                *phNotificationHandle =
                    (OSAL_TIME_NOTIFICATION_OBJECT)psEntry;
            }
            else
            {
                // Error!
                OSAL.vLinkedListMemoryFree(psEntry);
            }

            // Give up time global data
            OSAL.eSemGive(gsTimeControl.hListMutex);
        }
        else
        {
            // Error!
            OSAL.vLinkedListMemoryFree(psEntry);
        }
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTimeSetUnRegisterNotification
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTimeSetUnRegisterNotification (
    OSAL_TIME_NOTIFICATION_OBJECT hNotificationHandle
    )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_INPUT;

    // All we need to do is lock the time global data
    // and remove this entry.
    if(hNotificationHandle != OSAL_TIME_NOTIFICATION_INVALID_OBJECT)
    {
        eReturnCode = OSAL.eSemTake(gsTimeControl.hListMutex,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if(OSAL_SUCCESS == eReturnCode)
        {
            OSALT_UPDATE_HANDLER_ENTRY_STRUCT *psEntry =
                (OSALT_UPDATE_HANDLER_ENTRY_STRUCT *)hNotificationHandle;

            eReturnCode = OSAL.eLinkedListRemove(psEntry->hThisEntry);
            if(OSAL_SUCCESS == eReturnCode)
            {
                OSAL.vLinkedListMemoryFree(psEntry);
            }

            // Give up control
            OSAL.eSemGive(gsTimeControl.hListMutex);
        }
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_vParseTime
*
*******************************************************************************/
void OSAL_vParseTime(char *pcCmdLine, TIME_T *ptTod, struct tm *ptTime)
{
    const char *acTokens = " /:\n";
    UN32 un32DST_Minutes = 0;
    N32 n32GMT_Minutes = 0;
    BOOLEAN bOk = FALSE;

    OSAL.eTimeGetGMToffset(&n32GMT_Minutes);
    OSAL.eTimeGetDSTadjustment(&un32DST_Minutes);

    do
    {
        // Checking input
        if (ptTime == NULL)
        {
            break;
        }
        
        // Get next tokens...
        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine == NULL)
        {
            break;
        }
        ptTime->tm_mon = atoi(pcCmdLine) - 1;

        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine == NULL)
        {
            break;
        }
        ptTime->tm_mday = atoi(pcCmdLine);

        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine == NULL)
        {
            break;
        }
        ptTime->tm_year = atoi(pcCmdLine) - 1900;

        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine == NULL)
        {
            break;
        }
        ptTime->tm_hour = atoi(pcCmdLine);

        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine == NULL)
        {
            break;
        }
        ptTime->tm_min = atoi(pcCmdLine);

        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine == NULL)
        {
            break;
        }
        ptTime->tm_sec = atoi(pcCmdLine);

        // filled in by OSAL.mktime, days since sunday (from 0)
        ptTime->tm_wday = 0;

        // filled in by OSAL.mktime, day of the year (from 0)
        ptTime->tm_yday = 0;

        // DST in effect for this time?
        ptTime->tm_isdst = un32DST_Minutes * 60; // seconds
        
        // Success
        bOk = TRUE;
    } while (FALSE);

    if ((bOk == TRUE) && (ptTod != NULL))
    {
        // Call API to determine seconds since epoch (in calendar time)
        *ptTod = OSAL.mktime(ptTime);

        // Time provided is going to be local-time, so factor that in
        *ptTod -= (n32GMT_Minutes * 60);
    }
    
    return;
}

/*******************************************************************************
*
*   bGetTimeBias
*
*******************************************************************************/
static BOOLEAN bGetTimeBias( OSALT_TIME_BIAS_STRUCT *psBias )
{
    BOOLEAN bTimeValid = FALSE;

    if(gsTimeControl.sCurrent.hMutex != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Take over time global data
        eReturnCode = OSAL.eSemTake(
            gsTimeControl.sCurrent.hMutex,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if(OSAL_SUCCESS == eReturnCode)
        {
            // Grab bias and flag
            *psBias = gsTimeControl.sCurrent.sBias;
            bTimeValid = gsTimeControl.sCurrent.bTimeValid;

            // Give up control
            OSAL.eSemGive(gsTimeControl.sCurrent.hMutex);
        }
    }

    return bTimeValid;
}

/*******************************************************************************
*
*   vSetTimeBias
*
*******************************************************************************/
static void vSetTimeBias( OSALT_TIME_BIAS_STRUCT const *psBias )
{
    OSAL_TIME_UPDATE_MASK tUpdateMask = OSAL_TIME_UPDATE_MASK_NONE;

    if(gsTimeControl.sCurrent.hMutex != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Take over time global data
        eReturnCode = OSAL.eSemTake(
            gsTimeControl.sCurrent.hMutex,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if(OSAL_SUCCESS == eReturnCode)
        {
            // Compare new bias values against what we have.
            if(gsTimeControl.sCurrent.sBias.un32UpTimeSeconds !=
                    psBias->un32UpTimeSeconds)
            {
                tUpdateMask |=OSAL_TIME_UPDATE_MASK_TIME;
            }
            if(gsTimeControl.sCurrent.sBias.n32GMT_Seconds !=
                    psBias->n32GMT_Seconds)
            {
                tUpdateMask |=OSAL_TIME_UPDATE_MASK_GMT;
            }
            if(gsTimeControl.sCurrent.sBias.un32DST_Seconds !=
                    psBias->un32DST_Seconds)
            {
                tUpdateMask |=OSAL_TIME_UPDATE_MASK_DST;
            }

            // Check if new bias values are actually different than
            // what we already have.
            if(OSAL_TIME_UPDATE_MASK_NONE != tUpdateMask)
            {
                // Set new time bias.
                gsTimeControl.sCurrent.sBias = *psBias;

                // Indicate time has been set only if actual
                // time has been updated.
                if(OSAL_TIME_UPDATE_MASK_TIME & tUpdateMask)
                {
                    gsTimeControl.sCurrent.bTimeValid = TRUE;
                }
            }

            // Give up control
            OSAL.eSemGive(gsTimeControl.sCurrent.hMutex);
        }
    }

    // Anything updated?
    if(OSAL_TIME_UPDATE_MASK_NONE != tUpdateMask)
    {
        // Do we have a list?
        if(gsTimeControl.hListMutex != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Take over time-update list
            eReturnCode = OSAL.eSemTake(
                gsTimeControl.hListMutex,
                OSAL_OBJ_TIMEOUT_INFINITE);
            if(OSAL_SUCCESS == eReturnCode)
            {
                OSALT_CALLBACK_SHIM_STRUCT sShim;

                // Populate mask
                sShim.tUpdateMask = tUpdateMask;

                // Make callbacks into any time update handlers
                // Iterate the list of time handlers.
                OSAL.eLinkedListIterate(
                    gsTimeControl.hList,
                    bSetTimeCallbackShim, &sShim);

                // Give up control
                OSAL.eSemGive(gsTimeControl.hListMutex);
            }
        }
    }

    return;
}

/*******************************************************************************
*
*   bSetTimeCallbackShim
*
*******************************************************************************/
static BOOLEAN bSetTimeCallbackShim (
    void *pvData, void *pvArg
        )
{
    OSALT_UPDATE_HANDLER_ENTRY_STRUCT *psEntry =
        (OSALT_UPDATE_HANDLER_ENTRY_STRUCT *)pvData;
    OSALT_CALLBACK_SHIM_STRUCT *psShim =
        (OSALT_CALLBACK_SHIM_STRUCT *)pvArg;
    OSAL_TIME_UPDATE_MASK tUpdateMask;

    // Mask off anything they are not interested in
    tUpdateMask = psShim->tUpdateMask & psEntry->tUpdateMask;

    // Anything left?
    if(OSAL_TIME_UPDATE_MASK_NONE != tUpdateMask)
    {
        // Call registered handler
        psEntry->vHandler(
            tUpdateMask, psEntry->pvHandlerArg);
    }

    // Always continue
    return TRUE;
}

// Time routines
//
// Copyright (C) 2002 Michael Ringgaard. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. Neither the name of the project nor the names of its contributors
//    may be used to endorse or promote products derived from this software
//    without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//

/*******************************************************************************
*
*   OSAL_gmtime_r
*
*******************************************************************************/
struct tm *OSAL_gmtime_r(const TIME_T *ptTime, struct tm *psTime)
{
    TIME_T tTime = *ptTime;
    unsigned long dayclock, dayno;
    int year = EPOCH_YR;

    dayclock = (unsigned long) tTime % SECS_DAY;
    dayno = (unsigned long) tTime / SECS_DAY;

    psTime->tm_sec = dayclock % 60;
    psTime->tm_min = (dayclock % 3600) / 60;
    psTime->tm_hour = dayclock / 3600;
    psTime->tm_wday = (dayno + 4) % 7; // Day 0 was a thursday
    while (dayno >= (unsigned long) YEARSIZE(year))
    {
        dayno -= YEARSIZE(year);
        year++;
    }
    psTime->tm_year = year - YEAR0;
    psTime->tm_yday = dayno;
    psTime->tm_mon = 0;
    while (dayno >= (unsigned long) gytab[LEAPYEAR(year)][psTime->tm_mon])
    {
        dayno -= gytab[LEAPYEAR(year)][psTime->tm_mon];
        psTime->tm_mon++;
    }
    psTime->tm_mday = dayno + 1;
    psTime->tm_isdst = 0;

    return psTime;
}

/*******************************************************************************
*
*   OSAL_localtime_r
*
*******************************************************************************/
struct tm *OSAL_localtime_r(const TIME_T *ptTime, struct tm *psTime)
{
    TIME_T t = 0;
    N64 n64Time;
    OSALT_TIME_BIAS_STRUCT sBias = {0, 0, 0};

    bGetTimeBias(&sBias);
    n64Time = (N64)(*ptTime)
        + (N64)sBias.n32GMT_Seconds
        + (N64)sBias.un32DST_Seconds;

    if (n64Time > 0)
    {
        t = (TIME_T)n64Time;
    }

    return OSAL_gmtime_r(&t, psTime);
}

/*******************************************************************************
*
*   OSAL_mktime
*
*******************************************************************************/
TIME_T OSAL_mktime(struct tm *psTime)
{
    long day, year;
    int tm_year;
    int yday, month;
    UN64 seconds;
    int overflow;
    long dst;

    psTime->tm_min += psTime->tm_sec / 60;
    psTime->tm_sec %= 60;
    if (psTime->tm_sec < 0)
    {
        psTime->tm_sec += 60;
        psTime->tm_min--;
    }
    psTime->tm_hour += psTime->tm_min / 60;
    psTime->tm_min = psTime->tm_min % 60;
    if (psTime->tm_min < 0)
    {
        psTime->tm_min += 60;
        psTime->tm_hour--;
    }
    day = psTime->tm_hour / 24;
    psTime->tm_hour= psTime->tm_hour % 24;
    if (psTime->tm_hour < 0)
    {
        psTime->tm_hour += 24;
        day--;
    }
    psTime->tm_year += psTime->tm_mon / 12;
    psTime->tm_mon %= 12;
    if (psTime->tm_mon < 0)
    {
        psTime->tm_mon += 12;
        psTime->tm_year--;
    }
    day += (psTime->tm_mday - 1);
    while (day < 0)
    {
        if(--psTime->tm_mon < 0)
        {
            psTime->tm_year--;
            psTime->tm_mon = 11;
        }
        day += gytab[LEAPYEAR(YEAR0 + psTime->tm_year)][psTime->tm_mon];
    }
    while (day >= gytab[LEAPYEAR(YEAR0 + psTime->tm_year)][psTime->tm_mon])
    {
        day -= gytab[LEAPYEAR(YEAR0 + psTime->tm_year)][psTime->tm_mon];
        if (++(psTime->tm_mon) == 12)
        {
            psTime->tm_mon = 0;
            psTime->tm_year++;
        }
    }
    psTime->tm_mday = day + 1;
    year = EPOCH_YR;
    if (psTime->tm_year < year - YEAR0) return (TIME_T) -1;
    overflow = 0;

    // Assume that when day becomes negative, there will certainly
    // be overflow on seconds.
    // The check for overflow needs not to be done for leapyears
    // divisible by 400.
    // The code only works when year (1970) is not a leapyear.
    tm_year = psTime->tm_year + YEAR0;

    day = (tm_year - year) * 365;
    day += (tm_year - year) / 4 + ((tm_year % 4) && tm_year % 4 < year % 4);
    day -= (tm_year - year) / 100 + ((tm_year % 100) && tm_year % 100 < year % 100);
    day += (tm_year - year) / 400 + ((tm_year % 400) && tm_year % 400 < year % 400);

    yday = month = 0;
    while (month < psTime->tm_mon)
    {
        yday += gytab[LEAPYEAR(tm_year)][month];
        month++;
    }
    yday += (psTime->tm_mday - 1);
    if (day + yday < 0) overflow++;
    day += yday;

    psTime->tm_yday = yday;
    psTime->tm_wday = (day + 4) % 7;               // Day 0 was thursday (4)

    seconds = ((psTime->tm_hour * 60L) + psTime->tm_min) * 60L + psTime->tm_sec;
    seconds += day * SECS_DAY;

    if (psTime->tm_isdst > 0)
        dst = psTime->tm_isdst;
    else
        dst = 0;

    if (dst > seconds) overflow++;        // dst is always non-negative
    seconds -= dst;

    if (overflow) return (TIME_T)-1;

    if ((TIME_T)seconds != seconds) return (TIME_T) -1;

    return (TIME_T)seconds;
}

/*******************************************************************************
*
*   OSAL_difftime
*
*******************************************************************************/
double OSAL_difftime(TIME_T time1, TIME_T time0)
{
    return (double) (time1 - time0);
}

/*******************************************************************************
*
*   OSAL_strftime
*
*   Replacement for standard strftime() function, which alters TZ environment
*   variable and thus causes issues in Android
*
*******************************************************************************/
static int OSAL_strftime( char *pacBuf, size_t sMaxLen, const struct tm *psTime )
{
    static char *acWeekday[] =
    { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
    static char *acMonth[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
        "Aug", "Sep", "Oct", "Nov", "Dec" };
    int iReturn = 0;

    // Checking for input parameters' validity
    if( pacBuf != NULL && sMaxLen > 0 && psTime != NULL )
    {
        iReturn = snprintf( pacBuf, sMaxLen,
            "%s %s %d %02d:%02d:%02d %04d",
            acWeekday[psTime->tm_wday],
            acMonth[psTime->tm_mon],
            psTime->tm_mday,
            psTime->tm_hour,
            psTime->tm_min,
            psTime->tm_sec,
            1900 + psTime->tm_year );
    }
    return iReturn;
}

/*******************************************************************************
*
*   OSAL_asctime_r
*
*******************************************************************************/
char *OSAL_asctime_r( const struct tm *psTime, char *pacBuf )
{
    char *ascbuf = pacBuf;
    OSAL_strftime( ascbuf, OSAL_ASCBUFSIZE, psTime );
    return ascbuf;
}

/*******************************************************************************
*
*   OSAL_ctime_r
*
*******************************************************************************/
char *OSAL_ctime_r(const TIME_T *ptTime, char *pacBuf)
{
    struct tm tmbuf;
    return OSAL_asctime_r(OSAL_localtime_r(ptTime, &tmbuf), pacBuf);
}
