/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio Inc.                      */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio Inc.              */
/*                        Proprietary & Confidential                          */
/******************************************************************************/
/*******************************************************************************
*
*DESCRIPTION
*
*       This module will contain all the OSAL task APIs.
*
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"

#include "osal_debughandler.h"
#include "osal_monitor.h"
#include "osal_core.h"

#include "os_intf.h"

#include "_osal_task.h"

/*******************************************************************************
*
*   OSAL_eTaskCreate
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTaskCreate (
    OSAL_OBJECT_HDL *phTaskObj,
    const char *pacName,
    OSAL_TASK_HANDLER tTaskHandler,
    void *pvTaskHandlerArgument,
    UN32 un32StackSize,
    OSAL_TASK_PRIORITY_ENUM ePriority,
    UN32 un32Options
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_OBJECT_HDL hTaskList;
    OSAL_OBJECT_HDL hTaskObj;
    OS_OBJECT_HDL hOS;
    OSAL_TASK_INFO_STRUCT *psTask;

    // Caller must provide a valid task handler function
    if((tTaskHandler == NULL) || (phTaskObj == NULL))
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Initialize return object handle for error cases
    *phTaskObj = OSAL_INVALID_OBJECT_HDL;

    // Verify they have provided valid options
    if( (un32Options & ~OSAL_TASK_OPTION_ALL) != OSAL_TASK_OPTION_NONE)
    {
        return OSAL_ERROR_UNSUPPORTED_OPTION;
    }

    // Make sure they have provided a stack size of some length
    if(un32StackSize == 0)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Grab the task object's linked list
    hTaskList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_TASK);
    if(hTaskList == OSAL_INVALID_OBJECT_HDL)
    {
        // Error! There is no task object mgmt list.
        return OSAL_ERROR;
    }

    // If the provided priority is either HIGHEST or LOWEST
    // look for any other task which may have already registered
    // this priority. If one is found then this task cannot be
    // created at this priority.
    if( (ePriority == OSAL_TASK_PRIORITY_LOWEST) ||
        (ePriority == OSAL_TASK_PRIORITY_HIGHEST) )
    {
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;

        // Look through active task list for any other task
        // which already has this priority.
        eReturnCode =
            OSAL.eLinkedListLinearSearch(
                hTaskList, &hEntry,
                (OSAL_LL_COMPARE_HANDLER)n16CompareTaskPriority,
                (void *)(size_t)ePriority
                    );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // One was found with the same priority as this
            // one which is to be created. This is not allowed
            return OSAL_ERROR_INVALID_PRIORITY;
        }
    }

    // Create the object...
    eReturnCode = OSALC_eCreateObject(
        OSAL_OBJECT_TYPE_TASK,
        0,
        pacName,
        &hTaskObj,
        &psObj,
        (OSAL_OBJECT_INFO_UNION **)&psTask
            );

    if((eReturnCode == OSAL_SUCCESS) && (psTask != NULL))
    {
        // populate the object info
        psTask->tTaskHandler = tTaskHandler;
        psTask->pvTaskHandlerArgument = pvTaskHandlerArgument;
        psTask->un32StackSize = un32StackSize;
        psTask->ePriority = ePriority;
        psTask->un32Options = un32Options;
        psTask->un8StackPercentUsed = 0;
        psTask->un32StackBytesUsed = 0;
        psTask->un32StackBytesFree = 0;

        // Initialize Task Monitor (user provided) registration info
        psTask->un16ReportingInterval = 0;
        psTask->tSleepHandler = NULL;
        psTask->tWakeupHandler = NULL;
        psTask->tShutdownHandler = NULL;
        psTask->pvHandlerArgument = NULL;

        // Initialize Task Monitor info
        psTask->un16RemainingClicksForReport = UN16_MAX;
        psTask->hMonitorEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

#if OSAL_DEBUG == 1
        // Initialize Debug Stuff
        psTask->sDebug.bOutputEnabled = FALSE;
#if OSAL_OBJECT_TRACKING == 1
        psTask->sDebug.pacName =  psObj->pacName;
#else
        psTask->sDebug.pacName = "DEBUG";
#endif
        psTask->sDebug.pvFxnCallback = DEBUG_vDebugHandler;
        psTask->sDebug.pvArg = FALSE;
#endif

        // Now we're actually ready to create the object

        // Put 'parent' task handle into OS-specific task handle
        // Once the OS-Specific API runs successfully it will be replaced
        // with the OS-specific handle
        hOS = (OS_OBJECT_HDL)hTaskObj;

        // Assign pacName to something which is static if we are performing
        // object tracking. Otherwise we don't have a static name to use.
        // It will be up to the OS-specific call to create it's own instance
        // of that name if the OS itself does not already do so.
#if OSAL_OBJECT_TRACKING == 1
        pacName =  psObj->pacName;
#endif

        // Call OS-Specific API
        eReturnCode = OS.eTaskCreate((OSAL_OBJECT_HDL*)&hOS, pacName, OSALT_n32TaskShell,
            psObj, un32StackSize, ePriority, un32Options);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Remove and destroy the object
            OSALC_eRemoveObject(hTaskObj);

            // Return problem with creating the task
            return eReturnCode;
        }

        // Populate the returned OS-Specific object handle
        psObj->hOS = hOS;

        // Populate the object
        *phTaskObj = hTaskObj;

        eReturnCode = OSAL_SUCCESS;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTaskDelete
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTaskDelete (
    OSAL_OBJECT_HDL hTaskObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    OSAL_TASK_INFO_STRUCT *psTask;
    OSAL_OBJECT_STRUCT *psObj;

    // Note: This function must call the task's registered
    // delete or shutdown function. Actual clean-up and
    // removal of task resources is done when the user's task
    // returns (please see OSAL_vTaskShell).

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTaskObj, OSAL_OBJECT_TYPE_TASK);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Point to this object's info
    psTask = &psObj->puInfo->sTask;

    // Call this task's shutdown function (if defined)
    if(psTask->tShutdownHandler != NULL)
    {
        psTask->tShutdownHandler(
            psTask->pvHandlerArgument);
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTaskSuspend
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTaskSuspend (
    OSAL_OBJECT_HDL hTaskObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTaskObj, OSAL_OBJECT_TYPE_TASK);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Call OS-Specific API
    eReturnCode = OS.eTaskSuspend((OSAL_OBJECT_HDL)psObj->hOS);

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTaskResume
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTaskResume (
    OSAL_OBJECT_HDL hTaskObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTaskObj, OSAL_OBJECT_TYPE_TASK);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Call OS-Specific API
    eReturnCode = OS.eTaskResume((OSAL_OBJECT_HDL)psObj->hOS);

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTaskDelay
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTaskDelay (
    UN32 un32Milliseconds
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

    // If milliseconds to wait is 0 then no need to continue
    if(un32Milliseconds != 0)
    {
        // Call OS-Specific API
        eReturnCode = OS.eTaskDelay( un32Milliseconds );
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_vTaskYield
*
*******************************************************************************/
void OSAL_vTaskYield ( void )
{
    // Call OS-Specific API
    OS.vTaskYield();

    return;
}

/*******************************************************************************
*
*   OSAL_eTaskChangePriority
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTaskChangePriority (
    OSAL_OBJECT_HDL hTaskObj,
    OSAL_TASK_PRIORITY_ENUM eNewPriority
        )
{
    OSAL_OBJECT_HDL hTaskList;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_STRUCT *psObj;
    BOOLEAN bLocked;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTaskObj, OSAL_OBJECT_TYPE_TASK);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Grab the task object's linked list
    hTaskList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_TASK);
    if(hTaskList == OSAL_INVALID_OBJECT_HDL)
    {
        // Error! There is no task object mgmt list.
        return OSAL_ERROR;
    }

    // If the provided priority is either HIGHEST or LOWEST
    // look for any other task which may have already registered
    // this priority. If one is found then this task cannot be
    // created at this priority.
    if( (eNewPriority == OSAL_TASK_PRIORITY_LOWEST) ||
        (eNewPriority == OSAL_TASK_PRIORITY_HIGHEST) )
    {
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;

        // Look through active task list for any other task
        // which already has this priority.
        // The task list is always kept in priority order.
        eReturnCode =
            OSAL.eLinkedListLinearSearch(
                hTaskList, &hEntry,
                (OSAL_LL_COMPARE_HANDLER)n16CompareTaskPriority,
                (void *)(size_t)eNewPriority
                    );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // One was found with the same priority as this
            // one which is to be created. This is not allowed
            return OSAL_ERROR_INVALID_PRIORITY;
        }
    }

    // Update task priority as an atomic operation. This is required because
    // another task may also want to change its priority and to avoid
    // a potential race condition, we do it atomically.
    bLocked = OSALC_bLockCore();
    if(bLocked == TRUE)
    {
        // Call OS-Specific API
        eReturnCode = OS.eTaskChangePriority((OSAL_OBJECT_HDL)psObj->hOS, eNewPriority);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Update current priority
            OSAL_TASK_INFO_STRUCT *psTask;

            // Point to this object's info
            psTask = &psObj->puInfo->sTask;

            // Change this task's priority
            psTask->ePriority = eNewPriority;
        }

        // Re-enable core.
        OSALC_vUnlockCore();
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTaskRegister
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTaskRegister (
    UN16 un16ReportingInterval,
    OSAL_SLEEP_HANDLER tSleepHandler,
    OSAL_WAKEUP_HANDLER tWakeupHandler,
    OSAL_SHUTDOWN_HANDLER tShutdownHandler,
    void *pvHandlerArgument
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check inputs
    if(un16ReportingInterval > OSALT_MAXIMUM_REPORTING_INTERVAL)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Just call task monitor function now
    eReturnCode = OSALM_eTaskRegister(
                                      un16ReportingInterval,
                                      tSleepHandler,
                                      tWakeupHandler,
                                      tShutdownHandler,
                                      pvHandlerArgument
                                      );

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTaskUnregister
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTaskUnregister ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Just call task monitor function now
    eReturnCode = OSALM_eTaskUnregister();

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eTaskReport
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTaskReport (
    N32 n32Status
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Just call task monitor function now
    eReturnCode = OSALM_eTaskReport(n32Status);

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_hTaskGetHandle
*
*******************************************************************************/
OSAL_OBJECT_HDL OSAL_hTaskGetHandle ( void )
{
    OSAL_OBJECT_HDL hObjectHdl;
    OSAL_OBJECT_STRUCT *psObj;

    // Call OS-Specific API
    hObjectHdl = OS.hTaskGetHandle();

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hObjectHdl, OSAL_OBJECT_TYPE_TASK);
    if(psObj != NULL)
    {
        return hObjectHdl;
    }


    return OSAL_INVALID_OBJECT_HDL;
}

/*******************************************************************************
*
*   OSAL_hTaskGetHandleByName
*
*******************************************************************************/
OSAL_OBJECT_HDL OSAL_hTaskGetHandleByName (
    const char *pacName
        )
{
    OSAL_OBJECT_HDL hObjectHdl = OSAL_INVALID_OBJECT_HDL;
    OSAL_OBJECT_STRUCT *psObj;

    psObj = OSALC_psFindObjectFromName(OSAL_OBJECT_TYPE_TASK, pacName);
    if(psObj != NULL)
    {
        hObjectHdl = (OSAL_OBJECT_HDL)psObj;
    }

    return hObjectHdl;
}

/*******************************************************************************
*
*   OSAL_eTaskGetInfo
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTaskGetInfo  (
    OSAL_OBJECT_HDL hTaskObj,
    OSAL_TASK_INFO_STRUCT *psTaskInfo
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;

    // Check provided structure pointer
    if(psTaskInfo == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTaskObj, OSAL_OBJECT_TYPE_TASK);
    if(psObj != NULL)
    {
        // Copy task info I have into task info provided to
        // OS-specific API. That API can then modify it with what
        // it knows, or do nothing.
        OSAL.eEnterTaskSafeSection();
        *psTaskInfo = psObj->puInfo->sTask;
        OSAL.eExitTaskSafeSection();

        // Call OS specific API
        if(psObj->hOS != OS_INVALID_OBJECT_HDL)
        {
            eReturnCode = OS.eTaskGetInfo((OSAL_OBJECT_HDL)psObj->hOS, psTaskInfo);
        }
        else
        {
            eReturnCode = OSAL_SUCCESS;
        }
    }
    else
    {
        eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_bTaskGetId
*
*******************************************************************************/
BOOLEAN OSAL_bTaskGetId ( OSAL_TASK_ID *ptTaskId )
{
    BOOLEAN bOk = FALSE;

    if (ptTaskId != NULL)
    {
        // Call OS-Specific API
        bOk = OS.bTaskGetId(ptTaskId);
    }

    return bOk;
}

/*******************************************************************************
*
*   OSAL_eTaskList
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eTaskList ( void )
{

#if (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_HDL hTaskList;
    UN32 un32Tasks = 0, un32TaskNumber = 0;
    OSAL_STATISTICS_STRUCT sStatistics;

    // Iterate the list of tasks and list each entry

    // Grab the task linked list
    hTaskList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_TASK);
    if(hTaskList == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the number of tasks in the list
    eReturnCode = OSAL.eLinkedListItems(hTaskList, &un32Tasks);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the object's statistics
    OSALC_vGetObjectStatistics(OSAL_OBJECT_TYPE_TASK, &sStatistics);

    // Check that one or more elements exist in the list
    if(un32Tasks > 0)
    {
        // Print title
        puts("==================");
        puts("| OSAL Task List |");
        puts("==================\n");
    }

    // Iterate the list dumping each element individually
    eReturnCode = OSAL.eLinkedListIterate(hTaskList,
        OSALT_bPrintTask, &un32TaskNumber);
    if(eReturnCode == OSAL_NO_OBJECTS)
    {
        printf("Object list is empty!\n");
        return eReturnCode;
    }
    else if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }
    else
    {
        // Print summary
        printf("%u Task(s) currently allocated by the system.\n",
                    un32Tasks);

        // Print statistics
        printf("Current = %u, Minimum = %u, Maximum = %u\n",
               sStatistics.un32Current,
               sStatistics.un32Minimum,
               sStatistics.un32Maximum );

    }

    // Call OS-Specific API
    eReturnCode = OS.eTaskList();

    return eReturnCode;

#else

    return OSAL_ERROR_UNSUPPORTED_API;

#endif /*(OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */
}

#if OSAL_DEBUG == 1

/*******************************************************************************
*
*   OSALT_bPrintTask
*
*******************************************************************************/
BOOLEAN OSALT_bPrintTask ( void *pvData, void *pvArg )
{
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)pvData,
        *psCreatorObj;
    UN32 *pun32TaskNumber = (UN32 *)pvArg;
    const char *pacCreatorObjName = "Unknown";

    // Check if argument is valid
    if(pun32TaskNumber == NULL)
    {
        return FALSE;
    }

    // Prepare task number
    (*pun32TaskNumber)++;

    // Check if object is valid
    if(psObj != NULL)
    {
        // Point to this object's info
        OSAL_TASK_INFO_STRUCT sTaskInfo;

        // Copy object info I have into object info provided to
        // OS-specific API. That API can then modify it with what
        // it knows, or do nothing.
        sTaskInfo = psObj->puInfo->sTask;

        // Call OS-Specific API to acquire any dynamic data
        OS.eTaskGetInfo((OSAL_OBJECT_HDL)psObj->hOS, &sTaskInfo);

        // Extract the creator task's name
        psCreatorObj = (OSAL_OBJECT_STRUCT *)psObj->hCreatorObj;
        if(psCreatorObj != NULL)
        {
            // Extract name of creator
#if OSAL_OBJECT_TRACKING == 1
            pacCreatorObjName = psCreatorObj->pacName;
#else
            pacCreatorObjName = "Unknown";
#endif
        }

        // extract information we are interested, and print
        printf("Task #%u\n", *pun32TaskNumber);
        printf("Name = '%s' (hdl = 0x%p)\n",
#if OSAL_OBJECT_TRACKING == 1
               psObj->pacName,
#else
               "Unknown",
#endif
               psObj
                   );
        printf("\tCreator = %s\n", pacCreatorObjName);
        printf("\tPriority = %s, Options = %#010x\n",
               OSALC_pacGetPriorityName(sTaskInfo.ePriority),
               sTaskInfo.un32Options);
        printf("\tStack Size = %u, %u bytes used, %u bytes free "
               "(%u%% used)\n",
               sTaskInfo.un32StackSize, sTaskInfo.un32StackBytesUsed,
               sTaskInfo.un32StackBytesFree, sTaskInfo.un8StackPercentUsed);
        printf("\tLast reported status - %s (%d)\n",
               sTaskInfo.n32ReportedStatus == OSAL_TASK_REPORT_NO_ERROR ?
                   "OK" : "ERROR!", sTaskInfo.n32ReportedStatus);
        printf("\tTask Handler = 0x%08X, Arg = 0x%p\n",
               sTaskInfo.tTaskHandler, sTaskInfo.pvTaskHandlerArgument);
        printf("\tTask reporting interval = %u seconds\n",
               sTaskInfo.un16ReportingInterval);
        puts("");
    }

    // Keep iterating
    return TRUE;
}

/*****************************************************************************
*
*       OSALT_vTaskSetDebug
*
*****************************************************************************/
void OSALT_vTaskSetDebug (
    OSAL_OBJECT_HDL hTask,
    OSAL_TASK_DEBUG_STRUCT *psDebug
        )
{
    OSAL_OBJECT_STRUCT *psObj;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTask, OSAL_OBJECT_TYPE_TASK);
    if((psObj != NULL) && (psDebug != NULL))
    {
        BOOLEAN bLocked;

        bLocked = OSALC_bLockCore();
        if(bLocked == TRUE)
        {
            // Set debug flag
            psObj->puInfo->sTask.sDebug = *psDebug;

            // Re-enable core.
            OSALC_vUnlockCore();
        }
    }

    return;
}
#endif /* OSAL_DEBUG == 1 */

/*******************************************************************************
*
*   OSALT_bTaskDebugIsEnabled
*
*******************************************************************************/
BOOLEAN OSALT_bTaskDebugIsEnabled ( void )
{
    BOOLEAN bEnabled = TRUE;

#if OSAL_DEBUG == 1
    OSAL_OBJECT_HDL hTask;
    OSAL_OBJECT_STRUCT *psObj;

    // Call OS-Specific API
    hTask = OS.hTaskGetHandle();

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTask, OSAL_OBJECT_TYPE_TASK);
    if(psObj != NULL)
    {
            // Copy out task debug info
            bEnabled = psObj->puInfo->sTask.sDebug.bOutputEnabled;
    }
#endif

    return bEnabled;
}

/*******************************************************************************
*
*      OSALT_n32TaskShell
*
*******************************************************************************/
static N32 OSALT_n32TaskShell( void *pvArg )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)pvArg;
    N32 n32TaskError = !0;

    // Verify input is valid
    if(psObj != NULL)
    {
#if OSAL_DEBUG == 1
        // Register default DEBUG handler. If the application
        // registers another, thats fine as it will get replaced.
        DEBUG_eRegisterDebughandler(
#if OSAL_OBJECT_TRACKING == 1
            psObj->pacName,
#else
            "Unknown",
#endif
            DEBUG_vDebugHandler, NULL
                );
#endif /* OSAL_DEBUG == 1 */

      // Setting task options
#if OSAL_DEBUG == 1
      if( psObj->puInfo->sTask.un32Options & OSAL_TASK_OPTION_DEBUG_OUTPUT )
      {
         OSAL.vControlOutputThisTask(TRUE);
      }
#endif /* OSAL_DEBUG == 1 */

      // Task Enter
        n32TaskError = psObj->puInfo->sTask.tTaskHandler(
                psObj->puInfo->sTask.pvTaskHandlerArgument);

        // Task Exit, now clean up
        if(n32TaskError != OSAL_TASK_REPORT_NO_ERROR)
        {
#if OSAL_DEBUG == 1
            const char *pacTaskName = "Unknown";


#if OSAL_OBJECT_TRACKING == 1
            // Extract this task's name
            pacTaskName = psObj->pacName;
#endif

            // Print some debug
            printf("OSAL Error: Task '%s' returned non-zero result (%d).\n",
                   pacTaskName, n32TaskError);
#endif //OSAL_DEBUG == 1
        }

        // Check if this task has been removed from the task
        // monitor. If not it should be. So we'll do it for them
        if(psObj->puInfo->sTask.hMonitorEntry !=
           OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Remove this task from monitoring by the task monitor
            eReturnCode = OSAL.eTaskUnregister();
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Not sure what to do...
                printf("OSALT Error! Cannot un-register task.\n");
            }
        }

#if OSAL_DEBUG == 1
        // Unregister task from debug handler (regardless if it has already
        // been removed).
        DEBUG_eUnRegisterDebughandler();
#endif /* OSAL_DEBUG == 1 */

        // Remove object from the list being monitored
        eReturnCode = OSAL.eLinkedListRemove(psObj->hEntry);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Invalidate this entry
            psObj->hEntry =
                OSAL_INVALID_LINKED_LIST_ENTRY;

            // Call OS specific API
            eReturnCode = OS.eTaskDelete((OSAL_OBJECT_HDL)psObj->hOS);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Invalidate OS-Handle
                psObj->hOS = OS_INVALID_OBJECT_HDL;

                // Remove and destroy the object
                eReturnCode = OSALC_eRemoveObject((OSAL_OBJECT_HDL)psObj);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Invalidate object structure
                    psObj = NULL;
                }
            }
        }
    }

    // Task has exited
    return n32TaskError;
}

/*****************************************************************************
*
*       n16CompareTaskPriority
*
*       This is the function used to compare a task objects against
*       the provided priority.
*
*       Inputs:
*               pvObj1, pvObj2
*
*       Outputs:
*               0   - Objects have the same value (equal, error)
*               > 0 - Object1 is greater than (after) Object2
*               < 0 - Object1 is less than (before) Object2
*
*****************************************************************************/
static N16 n16CompareTaskPriority( void *pvObj1, void *pvObj2)
{
    N16 n16Result;
    OSAL_OBJECT_STRUCT *psObj =
        (OSAL_OBJECT_STRUCT *)pvObj1;
    OSAL_TASK_PRIORITY_ENUM ePriority =
        (OSAL_TASK_PRIORITY_ENUM)(size_t)pvObj2;

    n16Result = (N16)psObj->puInfo->sTask.ePriority -
        (N16)ePriority;
    if(n16Result != 0)
    {
        // Always move forward
        n16Result = -1;
    }

    return n16Result;
}
