/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio Inc.                      */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio Inc.              */
/*                        Proprietary & Confidential                          */
/******************************************************************************/
/*******************************************************************************
*
*  DESCRIPTION
*
*       This module will contain all the OSAL Red-Black tree
*       functionality.
*
*******************************************************************************/

#include <string.h>
#include <stdio.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"
#include "osal_core.h"
#include "os_intf.h"

#include "_osal_rb_tree.h"
#include "osal_rb_tree.h"

/*******************************************************************************
*
*   PUBLIC FUNCTIONS
*
*******************************************************************************/

/*******************************************************************************
*
*   OSAL_eRBTreeCreate
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eRBTreeCreate(
    OSAL_RB_TREE_HDL *phRBTree,
    const char *pacName,
    OSAL_OBJECT_HDL hLL,
    OSAL_LL_COMPARE_HANDLER n16CompareFunction
        )
{
    OSAL_RB_TREE_INFO_STRUCT *psRBTree = NULL;

    // Check input
    if (phRBTree == NULL)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    *phRBTree = OSAL_INVALID_RB_TREE_HDL;

    // Allocate memory for the object
    psRBTree = (OSAL_RB_TREE_INFO_STRUCT*)
        OSALC_pvMemoryAllocate(
        pacName, sizeof(OSAL_RB_TREE_INFO_STRUCT),
        FALSE
        );
    if(psRBTree == NULL)
    {
        return OSAL_ERROR_OUT_OF_MEMORY;
    }

    // Populate RB tree info from caller provided info
    strncpy(psRBTree->acName, pacName, OSAL_MAX_OBJECT_NAME_LENGTH);
    psRBTree->acName[OSAL_MAX_OBJECT_NAME_LENGTH] = '\0';

    if (n16CompareFunction == NULL)
    {
        psRBTree->n16CompareFunction = n16RBTreeCompareObjects;
    }
    else
    {
        psRBTree->n16CompareFunction = n16CompareFunction;
    }
    psRBTree->un32Count = 0;
    psRBTree->psRoot = OSAL_RB_TREE_NODE_NULL;
    psRBTree->hLL = hLL;

    // Populate created tree
    *phRBTree = (OSAL_RB_TREE_HDL)psRBTree;

    // We finished creating the RB tree successfully
    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*   OSAL_eRBTreeDelete
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eRBTreeDelete (
    OSAL_RB_TREE_HDL hRBTree
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_RB_TREE_INFO_STRUCT *psRBTree =
        (OSAL_RB_TREE_INFO_STRUCT*)hRBTree;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    if (hRBTree == OSAL_INVALID_RB_TREE_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // You can only destroy a RB tree if all elements have
    // been removed first.
    if (psRBTree->psRoot != OSAL_RB_TREE_NODE_NULL)
    {
        return OSAL_ERROR_LIST_NOT_EMPTY;
    }

    // Return object allocated for the RB tree object
    // Remove and destroy the object
    OSALC_vMemoryFree(psRBTree);

    // We finished deleting successfully
    eReturnCode = OSAL_SUCCESS;

    return eReturnCode;
}

/*******************************************************************************
*
*   OSAL_eRBTreeAdd
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eRBTreeAdd (
    OSAL_RB_TREE_HDL hRBTree,
    OSAL_RB_TREE_ENTRY *phEntry,
    void *pvData
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_INPUT;
    OSAL_RB_TREE_NODE_STRUCT *psNewNode = NULL;
    OSAL_RB_TREE_NODE_STRUCT *psNode = NULL;
    OSAL_RB_TREE_INFO_STRUCT *psRBTree =
        (OSAL_RB_TREE_INFO_STRUCT*)hRBTree;

    do
    {
        // Check input
        if (psRBTree == NULL)
        {
            eReturnCode = OSAL_ERROR_INVALID_INPUT;
            break;
        }

        psNewNode = (OSAL_RB_TREE_NODE_STRUCT*)
            OSALC_pvMemoryAllocate(OSAL_NAME_PREFIX"rbtNode",
            sizeof(*psNewNode), TRUE
                );

        // Verify we have a valid pointer to work with
        if(psNewNode == NULL)
        {
            eReturnCode = OSAL_ERROR_OUT_OF_MEMORY;
            break;
        }

        // Fill in node data
        psNewNode->hLL = psRBTree->hLL;
        psNewNode->psRBTree = psRBTree;
        psNewNode->pvData = pvData;
        psNewNode->eColor = OSAL_RB_TREE_NODE_RED;
        psNewNode->psParent = OSAL_RB_TREE_NODE_NULL;
        psNewNode->psLeft = OSAL_RB_TREE_NODE_NULL;
        psNewNode->psRight = OSAL_RB_TREE_NODE_NULL;

        psNode = psNewNode;

        eReturnCode = eRBTreeInsertNode(psRBTree, &psNode);
        if (eReturnCode == OSAL_SUCCESS)
        {
            (psRBTree->un32Count)++;
        }
        else
        {
            // Destroy just created node.
            // the psNode contains reference to existing one
            // which will be returned to caller together
            // with NO_UNIQUE return code.
            vRBTreeDestroyNode(psNewNode);
        }

        // Populate added of existing node
        if (((eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE) ||
            (eReturnCode == OSAL_SUCCESS)) && (phEntry != NULL))
        {
            *phEntry = (OSAL_RB_TREE_ENTRY)psNode;
        }
    } while (FALSE);

    return eReturnCode;
}

/*******************************************************************************
 *
 *    OSAL_eRBTreeReplace
 *
 *******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eRBTreeReplace (
    OSAL_RB_TREE_HDL hRBTree,
    OSAL_RB_TREE_ENTRY hEntry,
    void *pvData
        )
{
    OSAL_RB_TREE_INFO_STRUCT *psRBTree =
        (OSAL_RB_TREE_INFO_STRUCT*)hRBTree;
    OSAL_RB_TREE_NODE_STRUCT *psNode =
        (OSAL_RB_TREE_NODE_STRUCT*)hEntry;
    OSAL_RB_TREE_NODE_STRUCT *psInsertedNode;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        // Check input
        if ((hRBTree == OSAL_INVALID_RB_TREE_HDL) || (hEntry == OSAL_INVALID_RB_TREE_ENTRY))
        {
            eReturnCode = OSAL_ERROR_INVALID_INPUT;
            break;
        }

        // Request removing the node from the tree w/o releasing
        // memory for it for future use.
        vRBTreeDetachNode(psRBTree, psNode);

        // Assign new data
        psNode->eColor = OSAL_RB_TREE_NODE_RED;
        psNode->pvData = pvData;
        psNode->psParent = OSAL_RB_TREE_NODE_NULL;
        psNode->psLeft = OSAL_RB_TREE_NODE_NULL;
        psNode->psRight = OSAL_RB_TREE_NODE_NULL;

        // Try to re-add node
        psInsertedNode = psNode;
        eReturnCode = eRBTreeInsertNode(psRBTree, &psInsertedNode);
        if (eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
        {
            // We couldn't add this entry - just delete the node
            // and report any error code
            vRBTreeDestroyNode(psNode);
        }
        else
        {
            psRBTree->un32Count++;
        }

    } while (FALSE);

    return eReturnCode;
}

/*******************************************************************************
 *
 *    OSAL_eRBTreeRemove
 *
 *******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eRBTreeRemove(
    OSAL_RB_TREE_ENTRY hEntry
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psNode =
        (OSAL_RB_TREE_NODE_STRUCT*)hEntry;
    OSAL_RB_TREE_INFO_STRUCT *psRBTree = NULL;

    // Verify a valid handle was provided
    if(hEntry == OSAL_INVALID_RB_TREE_ENTRY)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract the RB tree handle
    psRBTree = psNode->psRBTree;

    // Remove the object
    vRBTreeDetachNode(psRBTree, psNode);
    vRBTreeDestroyNode(psNode);

    return OSAL_SUCCESS;
}

/*****************************************************************************
*
*    OSAL_eRBTreeSearch
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eRBTreeSearch(
    OSAL_RB_TREE_HDL hRBTree,
    OSAL_RB_TREE_ENTRY *phEntry,
    OSAL_LL_COMPARE_HANDLER n16CompareFunction,
    void *pvData
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_RB_TREE_NODE_STRUCT *psNode = NULL;
    OSAL_LL_COMPARE_HANDLER n16Function = NULL;
    OSAL_RB_TREE_NODE_STRUCT *psCurrent = NULL;
    OSAL_RB_TREE_INFO_STRUCT *psRBTree =
        (OSAL_RB_TREE_INFO_STRUCT*)hRBTree;
    N16 n16Compare = 0;

    // Note: It is possible the caller wishes to search for an object
    // which has value of zero. So testing pvData for NULL is not desired.

    // Check input
    if(phEntry == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Check if they have provided an over-ride compare function
    // If they did not provide one, just use the compare associated
    // with the RB tree (if provided when the object was created)
    if (n16CompareFunction == NULL)
    {
        n16Function = psRBTree->n16CompareFunction;
    }
    else
    {
        n16Function = n16CompareFunction;
    }

    psCurrent = psRBTree->psRoot;

    while (psCurrent != OSAL_RB_TREE_NODE_NULL)
    {
        n16Compare = n16Function(psCurrent->pvData, pvData);
        if (n16Compare == 0)
        {
            psNode = psCurrent;
            break;
        }
        else if (n16Compare > 0)
        {
            psCurrent = psCurrent->psLeft;
        }
        else
        {
            psCurrent = psCurrent->psRight;
        }
    }

    // Check if we found something
    if(psNode == NULL)
    {
        eReturnCode = OSAL_OBJECT_NOT_FOUND;
    }
    else
    {
        eReturnCode = OSAL_SUCCESS;
    }

    // Return the entry (as a handle)
    *phEntry = (OSAL_RB_TREE_ENTRY)psNode;

    return eReturnCode;
}

/*******************************************************************************
*
*    OSAL_eRBTreeRemoveAll
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eRBTreeRemoveAll(
    OSAL_RB_TREE_HDL hRBTree,
    OSAL_LL_RELEASE_HANDLER tvReleaseFunction
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psCurrent = NULL;
    OSAL_RB_TREE_NODE_STRUCT *psParent = NULL;
    OSAL_RB_TREE_INFO_STRUCT *psRBTree =
        (OSAL_RB_TREE_INFO_STRUCT*)hRBTree;

    // Check input
    if (hRBTree == OSAL_INVALID_RB_TREE_HDL)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Get Root node as current one
    psCurrent = psRBTree->psRoot;

    while (psCurrent != OSAL_RB_TREE_NODE_NULL)
    {
        psParent = psCurrent->psParent;
        if(psCurrent->psLeft != OSAL_RB_TREE_NODE_NULL)
        {
            psCurrent = psCurrent->psLeft;
        }
        else if(psCurrent->psRight != OSAL_RB_TREE_NODE_NULL)
        {
            psCurrent = psCurrent->psRight;
        }
        else
        {
            if (tvReleaseFunction != NULL)
            {
                tvReleaseFunction(psCurrent->pvData);
            }
            vRBTreeDestroyNode(psCurrent);
            --psRBTree->un32Count;

            if(psParent != OSAL_RB_TREE_NODE_NULL)
            {
                if(psParent->psLeft == psCurrent)
                {
                    psParent->psLeft = OSAL_RB_TREE_NODE_NULL;
                }
                else
                {
                    psParent->psRight = OSAL_RB_TREE_NODE_NULL;
                }
            }
            psCurrent = psParent;
        }
    }

    // Make root invalid
    psRBTree->psRoot = OSAL_RB_TREE_NODE_NULL;

    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*    OSAL_eRBTreeIterate
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eRBTreeIterate(
    OSAL_RB_TREE_HDL hRBTree,
    OSAL_LL_ITERATOR_HANDLER tbIterator,
    void *pvArg
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_HANDLE;
    OSAL_RB_TREE_NODE_STRUCT *psCurrent = NULL;
    BOOLEAN bContinue = FALSE;
    OSAL_RB_TREE_INFO_STRUCT *psRBTree =
        (OSAL_RB_TREE_INFO_STRUCT*)hRBTree;

    do
    {
        // Check input
        if ((hRBTree == OSAL_INVALID_RB_TREE_HDL) || (tbIterator == NULL))
        {
            eReturnCode = OSAL_ERROR_INVALID_INPUT;
            break;
        }

        //Getting 1st node
        psCurrent = psRBTreeFirst(psRBTree, NULL);
        while (psCurrent != OSAL_RB_TREE_NODE_NULL)
        {
            bContinue = tbIterator(psCurrent->pvData, pvArg);
            if (bContinue == FALSE)
            {
                break;
            }
            psCurrent = psRBTreeNext(psCurrent, NULL);
        }

        eReturnCode = OSAL_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*******************************************************************************
*
*    OSAL_pvRBTreeThis
*
*******************************************************************************/
void *OSAL_pvRBTreeThis(
    OSAL_RB_TREE_ENTRY hEntry
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psNode =
        (OSAL_RB_TREE_NODE_STRUCT*)hEntry;

    // Verify a valid handle was provided
    if(hEntry == OSAL_INVALID_RB_TREE_ENTRY)
    {
        return NULL;
    }

    return psNode->pvData;
}


/*******************************************************************************
*
*    OSAL_eRBTreeItems
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eRBTreeItems (
    OSAL_RB_TREE_HDL hRBTree,
    UN32* pun32Items
        )
{
    OSAL_RB_TREE_INFO_STRUCT *psRBTree =
        (OSAL_RB_TREE_INFO_STRUCT*)hRBTree;

    // Verify a valid pointer was provided
    if ((pun32Items == NULL) || (hRBTree == OSAL_INVALID_RB_TREE_HDL))
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Retrieve the number of items in the tree
    *pun32Items = psRBTree->un32Count;

    return OSAL_SUCCESS;
}

/*******************************************************************************
*
*    OSAL_hRBTreeFirst
*
*******************************************************************************/
OSAL_RB_TREE_ENTRY OSAL_hRBTreeFirst(
    OSAL_RB_TREE_HDL hRBTree,
    void **ppvData
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psNode = NULL;
    OSAL_RB_TREE_INFO_STRUCT *psRBTree =
        (OSAL_RB_TREE_INFO_STRUCT*)hRBTree;

    psNode = psRBTreeFirst(psRBTree, ppvData);

    if (psNode == OSAL_RB_TREE_NODE_NULL)
    {
        psNode = NULL;
    }

    return (OSAL_RB_TREE_ENTRY)psNode;
}

/*******************************************************************************
*
*    OSAL_hRBTreeLast
*
*******************************************************************************/
OSAL_RB_TREE_ENTRY OSAL_hRBTreeLast(
    OSAL_RB_TREE_HDL hRBTree,
    void **ppvData
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psNode = NULL;
    OSAL_RB_TREE_INFO_STRUCT *psRBTree =
        (OSAL_RB_TREE_INFO_STRUCT*)hRBTree;

    psNode = psRBTreeLast(psRBTree, ppvData);

    if (psNode == OSAL_RB_TREE_NODE_NULL)
    {
        psNode = NULL;
    }

    return (OSAL_RB_TREE_ENTRY)psNode;
}

/*******************************************************************************
*
*    OSAL_hRBTreeNext
*
*******************************************************************************/
OSAL_RB_TREE_ENTRY OSAL_hRBTreeNext(
    OSAL_RB_TREE_ENTRY hEntry,
    void **ppvData
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psNode =
        (OSAL_RB_TREE_NODE_STRUCT*)hEntry;
    OSAL_RB_TREE_NODE_STRUCT *psResult = NULL;

    psResult = psRBTreeNext(psNode, ppvData);

    if (psResult == OSAL_RB_TREE_NODE_NULL)
    {
        psResult = NULL;
    }

    return (OSAL_RB_TREE_ENTRY)psResult;
}

/*******************************************************************************
*
*    OSAL_hRBTreePrev
*
*******************************************************************************/
OSAL_RB_TREE_ENTRY OSAL_hRBTreePrev(
    OSAL_RB_TREE_ENTRY hEntry,
    void **ppvData
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psNode =
        (OSAL_RB_TREE_NODE_STRUCT*)hEntry;
    OSAL_RB_TREE_NODE_STRUCT *psResult = NULL;

    psResult = psRBTreePrev(psNode, ppvData);

    if (psResult == OSAL_RB_TREE_NODE_NULL)
    {
        psResult = NULL;
    }

    return (OSAL_RB_TREE_ENTRY)psResult;
}

/*******************************************************************************
*
*    PRIVATE FUNCTIONS
*
*******************************************************************************/
/*******************************************************************************
*
*    n16RBTreeCompareObjects
*
*    Gets compares two object in the tree by their values
*
*******************************************************************************/
static N16 n16RBTreeCompareObjects(
    void *pvData1,
    void *pvData2
        )
{
    if (pvData1 == pvData2)
    {
        return 0;
    }
    else if (pvData1 < pvData2)
    {
        return -1;
    }
    else
    {
        return 1;
    }
}

/*******************************************************************************
*
*    psRBTreeFirst
*
*    Gets the first item of the tree. The first item in the tree means
*    items which represents most left tree leaf.
*
*******************************************************************************/
static OSAL_RB_TREE_NODE_STRUCT *psRBTreeFirst(
    OSAL_RB_TREE_INFO_STRUCT *psTree,
    void **ppvData
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psResult = NULL;

    psResult = psTree->psRoot;
    while (psResult->psLeft != OSAL_RB_TREE_NODE_NULL)
    {
        psResult = psResult->psLeft;
    }

    // Populate node data if requested
    if (ppvData != NULL)
    {
        if (psResult != OSAL_RB_TREE_NODE_NULL)
        {
            *ppvData = psResult->pvData;
        }
        else
        {
            *ppvData = NULL;
        }
    }

    return psResult;
}

/*******************************************************************************
*
*    psRBTreeLast
*
*    Gets the last item of the tree. The last item in the tree means
*    items which represents most right tree leaf.
*
*******************************************************************************/
static OSAL_RB_TREE_NODE_STRUCT *psRBTreeLast(
    OSAL_RB_TREE_INFO_STRUCT *psRBTree,
    void **ppvData
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psResult = NULL;

    psResult = psRBTree->psRoot;
    while (psResult->psRight != OSAL_RB_TREE_NODE_NULL)
    {
        psResult = psResult->psRight;
    }

    // Populate node data if requested
    if (ppvData != NULL)
    {
        if (psResult != OSAL_RB_TREE_NODE_NULL)
        {
            *ppvData = psResult->pvData;
        }
        else
        {
            *ppvData = NULL;
        }
    }

    return psResult;
}

/*******************************************************************************
*
*    psRBTreeNext
*
*******************************************************************************/
static OSAL_RB_TREE_NODE_STRUCT *psRBTreeNext(
    OSAL_RB_TREE_NODE_STRUCT *psNode,
    void ** ppvData
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psParent = NULL;

    if (psNode->psRight != OSAL_RB_TREE_NODE_NULL)
    {
        psNode = psNode->psRight;
        while(psNode->psLeft != OSAL_RB_TREE_NODE_NULL)
        {
            psNode = psNode->psLeft;
        }
    }
    else
    {
        psParent = psNode->psParent;
        while ((psParent != OSAL_RB_TREE_NODE_NULL) &&
               (psNode == psParent->psRight))
        {
            psNode = psParent;
            psParent = psParent->psParent;
        }
        psNode = psParent;
    }

    // Populate node data if requested
    if (ppvData != NULL)
    {
        if ((psNode != NULL) && (psNode != OSAL_RB_TREE_NODE_NULL))
        {
            *ppvData = psNode->pvData;
        }
        else
        {
            *ppvData = NULL;
        }
    }

    return psNode;
}

/*******************************************************************************
*
*    psRBTreePrev
*
*******************************************************************************/
static OSAL_RB_TREE_NODE_STRUCT *psRBTreePrev(
    OSAL_RB_TREE_NODE_STRUCT *psNode,
    void ** ppvData
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psParent = NULL;

    if (psNode->psLeft != OSAL_RB_TREE_NODE_NULL)
    {
        psNode = psNode->psLeft;
        while(psNode->psRight != OSAL_RB_TREE_NODE_NULL)
        {
            psNode = psNode->psRight;
        }
    }
    else
    {
        psParent = psNode->psParent;
        while ((psParent != OSAL_RB_TREE_NODE_NULL) &&
               (psNode == psParent->psLeft))
        {
            psNode = psParent;
            psParent = psNode->psParent;
        }
        psNode = psParent;
    }

    // Populate node data if requested
    if (ppvData != NULL)
    {
        if ((psNode != NULL) && (psNode != OSAL_RB_TREE_NODE_NULL))
        {
            *ppvData = psNode->pvData;
        }
        else
        {
            *ppvData = NULL;
        }
    }

    return psNode;
}

/*******************************************************************************
 *
 *    eRBTreeInsertNode
 *
 * Note: In case the inserting node is not unique, it provides the
 * conflicling node in ppsNode argument.
 *
 *******************************************************************************/
static OSAL_RETURN_CODE_ENUM eRBTreeInsertNode(
    OSAL_RB_TREE_INFO_STRUCT *psRBTree,
    OSAL_RB_TREE_NODE_STRUCT **ppsNode
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psCurrent = NULL;
    OSAL_RB_TREE_NODE_STRUCT *psParent = OSAL_RB_TREE_NODE_NULL;
    N16 n16Result = 0;

    /* find where node belongs */
    psCurrent = psRBTree->psRoot;
    while (psCurrent != OSAL_RB_TREE_NODE_NULL)
    {
        n16Result = psRBTree->n16CompareFunction(
                                psCurrent->pvData,
                                (*ppsNode)->pvData
                                    );
        if (n16Result == 0)
        {
            *ppsNode = psCurrent;
            return OSAL_ERROR_LIST_ITEM_NOT_UNIQUE;
        }
        psParent = psCurrent;

        psCurrent = (n16Result > 0) ? psCurrent->psLeft : psCurrent->psRight;
    }

    (*ppsNode)->psParent = psParent;

    /* insert node in tree */
    if (psParent != OSAL_RB_TREE_NODE_NULL)
    {
        if (n16Result > 0)
        {
            psParent->psLeft = (*ppsNode);
        }
        else
        {
            psParent->psRight = (*ppsNode);
        }
    }
    else
    {
        psRBTree->psRoot = (*ppsNode);
    }

    vRBTreeInsertFixup(psRBTree, (*ppsNode));

    return OSAL_SUCCESS;
}

/*******************************************************************************
 *
 *    vRBTreeInsertFixup
 *
 *    Maintain the tree balance after node insertion
 *
 *******************************************************************************/
static void vRBTreeInsertFixup(
    OSAL_RB_TREE_INFO_STRUCT *psTree,
    OSAL_RB_TREE_NODE_STRUCT *psNodeX
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psNodeY = NULL;
    OSAL_RB_TREE_NODE_STRUCT *psRoot;

    if ((psTree == NULL) || (psNodeX == NULL))
    {
        return;
    }

    /* check Red-Black properties */
    psRoot = psTree->psRoot;
    while ((psNodeX != psRoot) &&
           (psNodeX->psParent->eColor == OSAL_RB_TREE_NODE_RED))
    {
        /* we have a violation */
        if (psNodeX->psParent == psNodeX->psParent->psParent->psLeft)
        {
            psNodeY = psNodeX->psParent->psParent->psRight;
            if (psNodeY->eColor == OSAL_RB_TREE_NODE_RED)
            {
                /* uncle is RED */
                psNodeX->psParent->eColor = OSAL_RB_TREE_NODE_BLACK;
                psNodeY->eColor = OSAL_RB_TREE_NODE_BLACK;
                psNodeX->psParent->psParent->eColor = OSAL_RB_TREE_NODE_RED;
                psNodeX = psNodeX->psParent->psParent;
            }
            else
            {
                /* uncle is BLACK */
                if (psNodeX == psNodeX->psParent->psRight)
                {
                    /* make x a left child */
                    psNodeX = psNodeX->psParent;
                    vRBTreeRotateLeft(psTree, psNodeX);
                }

                /* re-color and rotate */
                psNodeX->psParent->eColor = OSAL_RB_TREE_NODE_BLACK;
                psNodeX->psParent->psParent->eColor = OSAL_RB_TREE_NODE_RED;
                vRBTreeRotateRight(psTree, psNodeX->psParent->psParent);
            }
        }
        else
        {
            /* mirror image of above code */
            psNodeY = psNodeX->psParent->psParent->psLeft;
            if (psNodeY->eColor == OSAL_RB_TREE_NODE_RED)
            {
                /* uncle is RED */
                psNodeX->psParent->eColor = OSAL_RB_TREE_NODE_BLACK;
                psNodeY->eColor = OSAL_RB_TREE_NODE_BLACK;
                psNodeX->psParent->psParent->eColor = OSAL_RB_TREE_NODE_RED;
                psNodeX = psNodeX->psParent->psParent;
            }
            else
            {
                /* uncle is BLACK */
                if (psNodeX == psNodeX->psParent->psLeft)
                {
                    psNodeX = psNodeX->psParent;
                    vRBTreeRotateRight(psTree, psNodeX);
                }
                psNodeX->psParent->eColor = OSAL_RB_TREE_NODE_BLACK;
                psNodeX->psParent->psParent->eColor = OSAL_RB_TREE_NODE_RED;
                vRBTreeRotateLeft(psTree, psNodeX->psParent->psParent);
            }
        }
    }

    psRoot = psTree->psRoot;
    psRoot->eColor = OSAL_RB_TREE_NODE_BLACK;

    return;
}

/*******************************************************************************
 *
 *    vRBTreeRotateLeft
 *
 *******************************************************************************/
static void vRBTreeRotateLeft(
    OSAL_RB_TREE_INFO_STRUCT *psTree,
    OSAL_RB_TREE_NODE_STRUCT *psNodeX
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psNodeY = NULL;

    psNodeY = psNodeX->psRight;

    /* establish x->right link */
    psNodeX->psRight = psNodeY->psLeft;
    if (psNodeY->psLeft != OSAL_RB_TREE_NODE_NULL)
    {
        psNodeY->psLeft->psParent = psNodeX;
    }

    psNodeY->psParent = psNodeX->psParent;

    if (psNodeX->psParent != OSAL_RB_TREE_NODE_NULL)
    {
        if (psNodeX == psNodeX->psParent->psLeft)
        {
            psNodeX->psParent->psLeft = psNodeY;
        }
        else
        {
            psNodeX->psParent->psRight = psNodeY;
        }
    }
    else
    {
        psTree->psRoot = psNodeY;
    }

    /* link x and y */
    psNodeY->psLeft = psNodeX;
    psNodeX->psParent = psNodeY;

    return;
}

/*******************************************************************************
 *
 *    vRBTreeRotateRight
 *
 *******************************************************************************/
static void vRBTreeRotateRight(
    OSAL_RB_TREE_INFO_STRUCT *psTree,
    OSAL_RB_TREE_NODE_STRUCT *psNodeX
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psNodeY = NULL;

    psNodeY = psNodeX->psLeft;

    /* establish x->left link */
    psNodeX->psLeft = psNodeY->psRight;
    if (psNodeY->psRight != OSAL_RB_TREE_NODE_NULL)
    {
        psNodeY->psRight->psParent = psNodeX;
    }

    /* establish y->parent link */
    psNodeY->psParent = psNodeX->psParent;

    if (psNodeX->psParent != OSAL_RB_TREE_NODE_NULL)
    {
        if (psNodeX == psNodeX->psParent->psRight)
        {
            psNodeX->psParent->psRight = psNodeY;
        }
        else
        {
            psNodeX->psParent->psLeft = psNodeY;
        }
    }
    else
    {
        psTree->psRoot = psNodeY;
    }

    /* link x and y */
    psNodeY->psRight = psNodeX;
    psNodeX->psParent = psNodeY;

    return;
}

/*******************************************************************************
 *
 *    vRBTreeDetachNode
 *
 *******************************************************************************/
static void vRBTreeDetachNode(
    OSAL_RB_TREE_INFO_STRUCT *psTree,
    OSAL_RB_TREE_NODE_STRUCT *psNodeZ
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psNodeX = NULL;
    OSAL_RB_TREE_NODE_STRUCT *psNodeY = NULL;

    if ((psNodeZ == NULL) || (psNodeZ == OSAL_RB_TREE_NODE_NULL) ||
        (psTree == NULL))
    {
        return;
    }

    if ((psNodeZ->psLeft == OSAL_RB_TREE_NODE_NULL) ||
        (psNodeZ->psRight == OSAL_RB_TREE_NODE_NULL))
    {
        psNodeY = psNodeZ;
    }
    else
    {
        /* find tree successor with a NULL node as a child */
        psNodeY = psNodeZ->psRight;
        while (psNodeY->psLeft != OSAL_RB_TREE_NODE_NULL)
        {
            psNodeY = psNodeY->psLeft;
        }
    }

    /* x is y's only child */
    if (psNodeY->psLeft != OSAL_RB_TREE_NODE_NULL)
    {
        psNodeX = psNodeY->psLeft;
    }
    else
    {
        psNodeX = psNodeY->psRight;
    }

    /* remove y from the parent chain */
    psNodeX->psParent = psNodeY->psParent;
    if (psNodeY->psParent != OSAL_RB_TREE_NODE_NULL)
    {
        if (psNodeY == psNodeY->psParent->psLeft)
        {
            psNodeY->psParent->psLeft = psNodeX;
        }
        else
        {
            psNodeY->psParent->psRight = psNodeX;
        }
    }
    else
    {
        psTree->psRoot = psNodeX;
    }

    if (psNodeY != psNodeZ)
    {
        OSAL_RB_TREE_NODE_TYPE_ENUM eColor = psNodeY->eColor;

        // Substitute node Z by node Y to
        // keep passed object invalid
        psNodeY->psParent = psNodeZ->psParent;
        psNodeY->psRight = psNodeZ->psRight;
        psNodeY->psLeft = psNodeZ->psLeft;
        psNodeY->eColor = psNodeZ->eColor;
        psNodeZ->eColor = eColor;   // keep here the color of deleted node

        // Correct references from parent
        if (psNodeZ->psParent != OSAL_RB_TREE_NODE_NULL)
        {
            if (psNodeZ->psParent->psRight == psNodeZ)
            {
                psNodeY->psParent->psRight = psNodeY;
            }
            else
            {
                psNodeY->psParent->psLeft = psNodeY;
            }
        }

        // Correct references in children
        if (psNodeZ->psLeft->psParent == psNodeZ)
        {
            psNodeY->psLeft->psParent = psNodeY;
        }
        if (psNodeZ->psRight->psParent == psNodeZ)
        {
            psNodeY->psRight->psParent = psNodeY;
        }

        // Correct top
        if (psNodeZ == psTree->psRoot)
        {
            psTree->psRoot = psNodeY;
        }
    }

    if (psNodeZ->eColor == OSAL_RB_TREE_NODE_BLACK)
    {
        vRBTreeDeleteFixup(psTree, psNodeX);
    }
    --psTree->un32Count;

    return;
}

/*******************************************************************************
 *
 *    vRBTreeDestroyNode
 *
 *******************************************************************************/
static void vRBTreeDestroyNode(
    OSAL_RB_TREE_NODE_STRUCT *psNode
        )
{
    if ((psNode != NULL) && (psNode != OSAL_RB_TREE_NODE_NULL))
    {
        OSALC_vMemoryFree(psNode);
    }

    return;
}

/*******************************************************************************
 *
 *    vRBTreeDeleteFixup
 *
 *******************************************************************************/
static void vRBTreeDeleteFixup (
    OSAL_RB_TREE_INFO_STRUCT *psTree,
    OSAL_RB_TREE_NODE_STRUCT *psNodeX
        )
{
    OSAL_RB_TREE_NODE_STRUCT *psNodeW = NULL;

    if ((psTree == NULL) || (psNodeX == NULL))
    {
        return;
    }

    while ((psNodeX != psTree->psRoot) &&
           (psNodeX->eColor == OSAL_RB_TREE_NODE_BLACK))
    {
        if (psNodeX == psNodeX->psParent->psLeft)
        {
            psNodeW = psNodeX->psParent->psRight;
            if (psNodeW->eColor == OSAL_RB_TREE_NODE_RED)
            {
                psNodeW->eColor = OSAL_RB_TREE_NODE_BLACK;
                psNodeX->psParent->eColor = OSAL_RB_TREE_NODE_RED;
                vRBTreeRotateLeft(psTree, psNodeX->psParent);
                psNodeW = psNodeX->psParent->psRight;
            }
            if ((psNodeW->psLeft->eColor == OSAL_RB_TREE_NODE_BLACK) &&
                (psNodeW->psRight->eColor == OSAL_RB_TREE_NODE_BLACK))
            {
                psNodeW->eColor = OSAL_RB_TREE_NODE_RED;
                psNodeX = psNodeX->psParent;
            }
            else
            {
                if (psNodeW->psRight->eColor == OSAL_RB_TREE_NODE_BLACK)
                {
                    psNodeW->psLeft->eColor = OSAL_RB_TREE_NODE_BLACK;
                    psNodeW->eColor = OSAL_RB_TREE_NODE_RED;
                    vRBTreeRotateRight(psTree, psNodeW);
                    psNodeW = psNodeX->psParent->psRight;
                }
                psNodeW->eColor = psNodeX->psParent->eColor;
                psNodeX->psParent->eColor = OSAL_RB_TREE_NODE_BLACK;
                psNodeW->psRight->eColor = OSAL_RB_TREE_NODE_BLACK;
                vRBTreeRotateLeft(psTree, psNodeX->psParent);
                psNodeX = psTree->psRoot;
            }
        }
        else
        {
            psNodeW = psNodeX->psParent->psLeft;
            if (psNodeW->eColor == OSAL_RB_TREE_NODE_RED) {
                psNodeW->eColor = OSAL_RB_TREE_NODE_BLACK;
                psNodeX->psParent->eColor = OSAL_RB_TREE_NODE_RED;
                vRBTreeRotateRight(psTree, psNodeX->psParent);
                psNodeW = psNodeX->psParent->psLeft;
            }
            if ((psNodeW->psRight->eColor == OSAL_RB_TREE_NODE_BLACK) &&
                (psNodeW->psLeft->eColor == OSAL_RB_TREE_NODE_BLACK))
            {
                psNodeW->eColor = OSAL_RB_TREE_NODE_RED;
                psNodeX = psNodeX->psParent;
            }
            else
            {
                if (psNodeW->psLeft->eColor == OSAL_RB_TREE_NODE_BLACK)
                {
                    psNodeW->psRight->eColor = OSAL_RB_TREE_NODE_BLACK;
                    psNodeW->eColor = OSAL_RB_TREE_NODE_RED;
                    vRBTreeRotateLeft(psTree, psNodeW);
                    psNodeW = psNodeX->psParent->psLeft;
                }
                psNodeW->eColor = psNodeX->psParent->eColor;
                psNodeX->psParent->eColor = OSAL_RB_TREE_NODE_BLACK;
                psNodeW->psLeft->eColor = OSAL_RB_TREE_NODE_BLACK;
                vRBTreeRotateRight(psTree, psNodeX->psParent);
                psNodeX = psTree->psRoot;
            }
        }
    }

    psNodeX->eColor = OSAL_RB_TREE_NODE_BLACK;

    return;
}

#ifdef SUPPORT_CUNIT
#include <osal_rb_tree.cunit>
#endif
