/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                         Proprietary & Confidential                         */
/******************************************************************************/
/*******************************************************************************
*
*  DESCRIPTION
*
*       This module will contain all the OSAL monitor task.
*
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"
#include "os_intf.h"

#include "osal_core.h"
#include "osal_general.h"
#include "osal_task.h"
#include "osal_debughandler.h"
#include "_osal_monitor.h"

/*****************************************************************************
*
*      OSALM_bTaskInstall
*
*****************************************************************************/
BOOLEAN OSALM_bTaskInstall(
    OSAL_TASK_PRIORITY_ENUM ePriority,
    const OSAL_MONITOR_HANDLER_STRUCT *psMonitorHandlers
        )
{
    static BOOLEAN bFirstTime = TRUE;
    BOOLEAN bRetVal = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check that there is not already an instance running
    if(bFirstTime == FALSE)
    {
        return FALSE;
    }
    else
    {
        bFirstTime = FALSE;
    }

    // Initialize all task memory to zero
    OSAL.bMemSet( (void *)&gsMonitorTaskControl, 0,
                 sizeof(OSALM_TASK_CONTROL_STRUCT) );

    // Load in handlers
    gsMonitorTaskControl.psMonitorHandlers = psMonitorHandlers;

    // Create install complete semaphore
    eReturnCode = OSAL.eSemCreate(&gsMonitorTaskControl.hInstallSemaphore,
        "Monitor Install", 0, 1, OSAL_SEM_OPTION_NONE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Create the task
        eReturnCode = OSAL.eTaskCreate(
                &gsMonitorTaskControl.hMyTaskObj,
                "Monitor Task",
                OSALM_n32Task,
                (void*)&gsMonitorTaskControl,
                OSALM_TASK_STACK_SIZE,
                ePriority,
                OSALM_TASK_OPTIONS);

        if(eReturnCode == OSAL_SUCCESS)
        {
            // Wait for task to install
            eReturnCode = OSAL.eSemTake(gsMonitorTaskControl.hInstallSemaphore,
                          OSAL_OBJ_TIMEOUT_INFINITE);
            if(eReturnCode == OSAL_SUCCESS)
            {
                bRetVal = TRUE;
            }
        }

        // Destroy the semaphore (no longer needed)
        if(gsMonitorTaskControl.hInstallSemaphore != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eSemDelete(gsMonitorTaskControl.hInstallSemaphore);
            gsMonitorTaskControl.hInstallSemaphore = OSAL_INVALID_OBJECT_HDL;
        }
    }

    return bRetVal;
}

/*****************************************************************************
*
*       OSALM_vTaskUninstall
*
*       This function is used to remove the Task from the system.
*
*       Inputs:
*               None
*
*       Outputs:
*               None
*
*        Notes:
*                This function runs in the context of the caller.
*
*****************************************************************************/
void OSALM_vTaskUninstall( void )
{
    if(gsMonitorTaskControl.hMyTaskObj != OSAL_INVALID_OBJECT_HDL)
    {
        // Do not check return code, because we can do nothing with it anyway
        OSAL.eTaskDelete(gsMonitorTaskControl.hMyTaskObj);
    }

    return;
}

/*******************************************************************************
*
*   OSALM_n32Task
*
*******************************************************************************/
static N32 OSALM_n32Task ( void *pvArg )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bTaskInitialized;
#if OSAL_DEBUG == 1
    static BOOLEAN bDebugRegister = TRUE;
#endif

    // Initialize this task!
     bTaskInitialized = OSALM_bTaskInitialize();
    if( bTaskInitialized == TRUE )
    {
        // Indicate task is installed by releasing the semaphore
        eReturnCode = OSAL.eSemGive(gsMonitorTaskControl.hInstallSemaphore);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Start watchdog
            gsMonitorTaskControl.psMonitorHandlers->vWatchdogStartHandler(
                gsMonitorTaskControl.psMonitorHandlers->pvWatchdogHandlerArg);

            // Forever
            for(;;)
            {
                // Check if watchdog needs to be kicked!
                if(gsMonitorTaskControl.eErrorCode ==
                    OSAL_TASK_MONITOR_OK)
                {
                    // Kick watchdog
                    gsMonitorTaskControl.psMonitorHandlers->vWatchdogKickHandler(
                        gsMonitorTaskControl.psMonitorHandlers->pvWatchdogHandlerArg);
                }
                else // Violation!
                {
                    // A violation occurred...do not kick watchdog
                    // Instead run the violation handler....
                    OSALM_vTaskMonitorViolationHandler();
                }

                OS.vUpdateCPUState();

                // We use a task delay because this is the
                // best method which relies on minimal OS-resources
                eReturnCode = OSAL.eTaskDelay(
                    OSALM_REPORTING_RESOLUTION_MSEC / 2 );
                if(eReturnCode != OSAL_SUCCESS)
                {
                    printf("Error! Task cannot be delayed: %s\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                    break;
                }

                // Signal Heart Beat Handler
                // We do this in the task loop to indicate that the
                // monitor task is running properly without the need
                // for Timers or other OS resources.
                gsMonitorTaskControl.psMonitorHandlers->vHeartBeatHandler(
                    gsMonitorTaskControl.psMonitorHandlers->pvHeartBeatHandlerArg);
                printf("<--- Monitor Task Heartbeat --->\n");

                // Decrement the "time to report" for each task being
                // monitored by this task. Also do any task checking
                // which is required. If anything is not right
                // then the iteration will stop and the error type
                // will be populated accordingly. When the forever loop
                // continues the task monitor structure will be checked
                OSAL.eLinkedListIterate(
                    gsMonitorTaskControl.hTaskList,
                    OSALM_bProcessTasks, NULL);
                if( gsMonitorTaskControl.eErrorCode !=
                        OSAL_TASK_MONITOR_OK )
                {
                    // Handle any violation immediately!
                    continue;
                }

                // Perform any "system-level" checks here
                gsMonitorTaskControl.eErrorCode =
                    OSALM_eCheckSystem();
                if( gsMonitorTaskControl.eErrorCode !=
                        OSAL_TASK_MONITOR_OK )
                {
                    // Handle any violation immediately!
                    continue;
                }

#if OSAL_DEBUG == 1
                // Check if we can register with the debug handler
                // if so, go ahead and do so.
                if(bDebugRegister == TRUE)
                {
                    eReturnCode =
                        OSAL.eDebugRegister( "Task Monitor",
                                             DEBUG_vDebugHandler, NULL );
                    if(eReturnCode == OSAL_SUCCESS)
                    {
                        // We are registered now
                        bDebugRegister = FALSE;
                    }
                }
#endif /* OSAL_DEBUG == 1 */

            } // Forever (this task should never be deleted)

            // Stop watchdog
            gsMonitorTaskControl.psMonitorHandlers->vWatchdogStopHandler(
                gsMonitorTaskControl.psMonitorHandlers->pvWatchdogHandlerArg);

        } // if(eReturnCode == OSAL_SUCCESS)

    } // if( bTaskInitialized == TRUE )

    // Unallocate task resources
    OSALM_vTaskUninitialize();

    return OSAL_TASK_REPORT_NO_ERROR;
}

/*****************************************************************************
*
*       OSALM_bTaskInitialize
*
*        This is the function used to Initialize the Monitor Task
*
*       Inputs:
*               None
*
*       Outputs:
*               BOOLEAN        bSuccess
*                                - TRUE if successful
*                                - otherwise FALSE        *
*****************************************************************************/
static BOOLEAN OSALM_bTaskInitialize( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32TotalSystemBytes = 0;

    // Initialize task monitor info
    gsMonitorTaskControl.eErrorCode = OSAL_TASK_MONITOR_OK;
    gsMonitorTaskControl.hTaskObj = OSAL_INVALID_OBJECT_HDL;
    gsMonitorTaskControl.bCpuUtilizationWarning = FALSE;

    // Determine the amount of 'heap' or number of bytes available
    // for dynamic memory allocation. This needs to be computed only once.
    OSAL.bMemoryUsage(NULL, NULL, NULL, NULL,
                      NULL, NULL, NULL, NULL,
                      &un32TotalSystemBytes );

    gsMonitorTaskControl.un32ActualBytesThreshold = (UN32)
        (((UN64)un32TotalSystemBytes * OSALM_MAX_MEMORY_ALLOCATION) / 100);

    // Create a linked list of tasks to be monitored
    eReturnCode = OSAL.eLinkedListCreate(
        &gsMonitorTaskControl.hTaskList,
        "Task Monitor",
        OSALM_n16CompareTask,
        OSAL_LL_OPTION_LINEAR |
        OSAL_LL_OPTION_PROTECT |
        OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*       OSALM_vTaskUninitialize
*
*        This function will un-initialize any components that
*        have been created at start up.
*
*       Inputs:
*               None
*
*       Outputs:
*               None
*
*****************************************************************************/
static void OSALM_vTaskUninitialize( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Delete the semaphore (no longer needed) - force post
    if(gsMonitorTaskControl.hInstallSemaphore != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(gsMonitorTaskControl.hInstallSemaphore);
        if(eReturnCode == OSAL_SUCCESS)
        {
            gsMonitorTaskControl.hInstallSemaphore = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Delete all elements in the task monitor linked list and then finally
    // delete the linked list itself.
    if(gsMonitorTaskControl.hTaskList != OSAL_INVALID_OBJECT_HDL)
    {
        // Since these are pre-allocated LL entries, we can't use
        // eLinkedListRemoveAll and free the entry in the RemoveAll callback
        // because that would end up freeing the LL portion of the memory and
        // then eLinkedListRemoveAll gets confused. So do it the old
        // fashioned way.
        OSAL_LINKED_LIST_ENTRY hEntry;
        OSALM_TASK_REGISTRATION_STRUCT *psTaskRegistration;

        do
        {
            hEntry = OSAL.hLinkedListFirst(
                gsMonitorTaskControl.hTaskList, (void**)&psTaskRegistration);
            if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                OSAL.eLinkedListRemove(hEntry);
                OSAL.vLinkedListMemoryFree(psTaskRegistration);
            }
        } while (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY);

        // Destroy the linked list itself
        eReturnCode = OSAL.eLinkedListDelete(
            gsMonitorTaskControl.hTaskList);
        if(eReturnCode == OSAL_SUCCESS)
        {
            gsMonitorTaskControl.hTaskList =
                OSAL_INVALID_OBJECT_HDL;
        }
    }

#if OSAL_DEBUG == 1
    // Unregister us from the debug handler
    OSAL.eDebugUnregister();
#endif

    return;
}

/*******************************************************************************
*
*   OSALM_eTaskRegister
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSALM_eTaskRegister (
    UN16 un16ReportingInterval,
    OSAL_SLEEP_HANDLER tSleepHandler,
    OSAL_WAKEUP_HANDLER tWakeupHandler,
    OSAL_SHUTDOWN_HANDLER tShutdownHandler,
    void *pvHandlerArgument
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_OUT_OF_MEMORY;
    OSAL_OBJECT_HDL hTaskObj;
    OSAL_TASK_INFO_STRUCT *psTaskInfo;
    OSAL_OBJECT_STRUCT *psObj;
    OSALM_TASK_REGISTRATION_STRUCT *psTaskRegistration;

    // Obtain calling task's handle, this function is always called
    // from the context of the task which is doing the registration.
    hTaskObj = OSAL.hTaskGetHandle();
    if (hTaskObj == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTaskObj, OSAL_OBJECT_TYPE_TASK);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Point to this object's info
    psTaskInfo = &psObj->puInfo->sTask;

    // Truncate reporting interval if larger than maximum allowed
    if(un16ReportingInterval > OSAL_TASK_REPORT_MAXIMUM_INTERVAL_SEC)
    {
        un16ReportingInterval = OSAL_TASK_REPORT_MAXIMUM_INTERVAL_SEC;
    }

    // Populate task registration info
    psTaskInfo->un16ReportingInterval = un16ReportingInterval;
    psTaskInfo->tSleepHandler = tSleepHandler;
    psTaskInfo->tWakeupHandler = tWakeupHandler;
    psTaskInfo->tShutdownHandler = tShutdownHandler;
    psTaskInfo->pvHandlerArgument = pvHandlerArgument;
    // If task is not going to report (interval == 0) then
    // leave the remaining clicks to report the maximum (default)
    if(un16ReportingInterval == 0)
    {
        psTaskInfo->un16RemainingClicksForReport =
            UN16_MAX;
    }
    else
    {
        psTaskInfo->un16RemainingClicksForReport =
            un16ReportingInterval * 2;
    }

    psTaskInfo->hMonitorEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Allocate a task monitor entry
    psTaskRegistration = (OSALM_TASK_REGISTRATION_STRUCT*)
                            OSAL.pvLinkedListMemoryAllocate(
#if OSAL_OBJECT_TRACKING == 1
            psTaskInfo->sDebug.pacName,
#else
            "Task",
#endif
            sizeof(OSALM_TASK_REGISTRATION_STRUCT),
            TRUE
                );
    if(psTaskRegistration != NULL)
    {
    // Add this task to the monitor list
        psTaskRegistration->hTaskObj = hTaskObj;
        eReturnCode = OSAL.eLinkedListAdd(
                          gsMonitorTaskControl.hTaskList,
                          &psTaskInfo->hMonitorEntry,
                          psTaskRegistration);

        // If this is not successful, simply return an error
        if(eReturnCode != OSAL_SUCCESS)
        {
            OSAL.vLinkedListMemoryFree(psTaskRegistration);
            return OSAL_ERROR_CANNOT_ADD_OBJECT;
        }
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSALM_eTaskUnregister
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSALM_eTaskUnregister ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_HDL hTaskObj;
    OSAL_TASK_INFO_STRUCT *psTaskInfo;
    OSAL_OBJECT_STRUCT *psObj;
    OSALM_TASK_REGISTRATION_STRUCT *psTaskRegistration;

    // Call OSAL API to get current task handle
    hTaskObj = OSAL.hTaskGetHandle();
    if(hTaskObj == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTaskObj, OSAL_OBJECT_TYPE_TASK);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Point to this object's info
    psTaskInfo = &psObj->puInfo->sTask;

    // Retrieve the pointer to memory to be freed
    psTaskRegistration = (OSALM_TASK_REGISTRATION_STRUCT *)
                            OSAL.pvLinkedListThis(psTaskInfo->hMonitorEntry);

    // Remove entry from monitor task list
    eReturnCode = OSAL.eLinkedListRemove(psTaskInfo->hMonitorEntry);

    if(psTaskRegistration != NULL)
    {
        // Free allocated memory
        OSAL.vLinkedListMemoryFree(psTaskRegistration);
    }

    // If this is not successful, simply return an error
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_ERROR_CANNOT_REMOVE_OBJECT;
    }

    // Invalidate entry
    psTaskInfo->un16ReportingInterval = 0;
    psTaskInfo->tSleepHandler = NULL;
    psTaskInfo->tWakeupHandler = NULL;
    psTaskInfo->tShutdownHandler = NULL;
    psTaskInfo->pvHandlerArgument = NULL;
    psTaskInfo->un16RemainingClicksForReport = UN16_MAX;
    psTaskInfo->hMonitorEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    return eReturnCode;
}

/*******************************************************************************
*
*   OSALM_eTaskReport
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSALM_eTaskReport (
    N32 n32Status
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_HDL hTaskObj;
    OSAL_TASK_INFO_STRUCT *psTaskInfo;
    OSAL_OBJECT_STRUCT *psObj;
    OSALM_TASK_REGISTRATION_STRUCT *psTaskRegistration;

    // Obtain calling task's handle
    hTaskObj = OSAL.hTaskGetHandle();
    if (hTaskObj == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hTaskObj, OSAL_OBJECT_TYPE_TASK);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Point to this object's info
    psTaskInfo = &psObj->puInfo->sTask;

    /*
    First remove the entry from the monitoring list. Note this
    is a pretty cheap operation since the linked list uses pre-allocated
    entries and thus will not free/re-allocate memory for the re-insertion.
    We do it this strange way (instead of just using replace entry) to
    better handle (task safe) updating the report from a thread. this
    process removes the entry, modifies it, and re-inserts it, which due
    to list locks is a thread safe operation. Doing it this way shouldn't
    incur a CPU penalty because these lists use pre-allocated entries
    (so no re-alloc occurs).
    */

    psTaskRegistration = (OSALM_TASK_REGISTRATION_STRUCT *)
                            OSAL.pvLinkedListThis(psTaskInfo->hMonitorEntry);
    eReturnCode = OSAL.eLinkedListRemove(psTaskInfo->hMonitorEntry);

    // If this is not successful, simply return an error
    if(eReturnCode != OSAL_SUCCESS)
    {
        printf ("OSAL Task Monitor: Cannot remove entry (%s)\n",
                OSAL.pacGetReturnCodeName(eReturnCode));
        return OSAL_ERROR_CANNOT_REMOVE_OBJECT;
    }

    // Record task status
    psTaskInfo->n32ReportedStatus = n32Status;
    if(psTaskInfo->n32ReportedStatus != 0)
    {
        // Output some debug
        printf("OSAL Task Monitor: Task '%s' has reported a non-zero "
               "status of %d.\n",
#if OSAL_OBJECT_TRACKING == 1
               psObj->pacName,
#else
               "Unknown",
#endif
               psTaskInfo->n32ReportedStatus
                   );
    }

    // Reset task time to report, if this task is registered to
    // report at any non-zero interval
    if(psTaskInfo->un16ReportingInterval != 0)
    {
        psTaskInfo->un16RemainingClicksForReport =
            psTaskInfo->un16ReportingInterval * 2;
    }

    // Re-insert this task info to the monitor list (SORTED)
    psTaskInfo->hMonitorEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    eReturnCode = OSAL.eLinkedListAdd(
        gsMonitorTaskControl.hTaskList,
        &psTaskInfo->hMonitorEntry,
        psTaskRegistration
            );

    // If this is not successful, simply return an error
    if(eReturnCode != OSAL_SUCCESS)
    {
        OSAL.vLinkedListMemoryFree(psTaskRegistration);
        printf ("OSAL Task Monitor: Cannot replace entry (%s)\n",
                OSAL.pacGetReturnCodeName(eReturnCode));
        return OSAL_ERROR_CANNOT_ADD_OBJECT;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   OSALM_vReportError
*
*******************************************************************************/
void OSALM_vReportError (
    OSAL_RETURN_CODE_ENUM eErrorCode
        )
{
    // Toggle warning
    gsMonitorTaskControl.eSystemErrorCode = eErrorCode;

    return;
}

/*****************************************************************************
*
*       OSALM_n16CompareTask
*
*        This is the function used to compare two objects for inserting
*       them into a sorted linked list.
*
*       Inputs:
*               None
*
*       Outputs:
*               0   - Objects have the same value (equal, error)
*               > 0 - Object1 is greater than (after) Object2
*               < 0 - Object1 is less than (before) Object2
*
****************************************************************************/
static N16 OSALM_n16CompareTask( void *pvObj1, void *pvObj2 )
{
    N16 n16Result = N16_MIN;
    OSALM_TASK_REGISTRATION_STRUCT *psTask1 =
        (OSALM_TASK_REGISTRATION_STRUCT *)pvObj1;
    OSALM_TASK_REGISTRATION_STRUCT *psTask2 =
        (OSALM_TASK_REGISTRATION_STRUCT *)pvObj2;
    OSAL_OBJECT_STRUCT *psTaskObj1, *psTaskObj2;
    OSAL_TASK_INFO_STRUCT *psTaskInfo1, *psTaskInfo2;

    // Check pointers
    if((psTask1 != NULL) &&
        (psTask2 != NULL))
    {
        // I have two task handles, now I need their data
        // We already know the handles are valid because otherwise
        // if they were not they would never be in this list.
        // So for execution time considerations we just cast-n'go
        psTaskObj1 = (OSAL_OBJECT_STRUCT *)psTask1->hTaskObj;
        psTaskObj2 = (OSAL_OBJECT_STRUCT *)psTask2->hTaskObj;
        psTaskInfo1 = &psTaskObj1->puInfo->sTask;
        psTaskInfo2 = &psTaskObj2->puInfo->sTask;

        // Compare two tasks being monitored
        if (psTaskInfo1->un16RemainingClicksForReport <
                        psTaskInfo2->un16RemainingClicksForReport)
        {
            n16Result = -1;
        }
        else if (psTaskInfo1->un16RemainingClicksForReport >
                        psTaskInfo2->un16RemainingClicksForReport)
        {
            n16Result = 1;
        }
        else
        {
            n16Result = 0;
        }
    }

    return n16Result;
}

/*******************************************************************************
*
*   OSALM_bProcessTasks
*
*******************************************************************************/
static BOOLEAN OSALM_bProcessTasks ( void *pvData, void *pvArg )
{
    OSALM_TASK_REGISTRATION_STRUCT *psTaskRegistration =
        (OSALM_TASK_REGISTRATION_STRUCT *)pvData;
    OSAL_TASK_INFO_STRUCT *psTaskInfo;
    OSAL_OBJECT_STRUCT *psObj;

    // Check if data is valid
    if(psTaskRegistration != NULL)
    {
        if(psTaskRegistration->hTaskObj != OSAL_INVALID_OBJECT_HDL)
        {
            // Extract object structure from handle provided and
            // verify a valid handle was provided
                psObj = OSALC_psGetObjectFromHandle(
                    psTaskRegistration->hTaskObj, OSAL_OBJECT_TYPE_TASK);
            if(psObj != NULL)
            {
                // Point to this object's info
                psTaskInfo = &psObj->puInfo->sTask;

                // Decrement 'remaining clicks to report' as along as this
                // task is reporting with a non-zero report interval
                if(psTaskInfo->un16ReportingInterval != 0)
                {
                    psTaskInfo->un16RemainingClicksForReport--;
                }

                // Indicate which task we are processing
                gsMonitorTaskControl.hTaskObj = psTaskRegistration->hTaskObj ;

                // Check Task...
                gsMonitorTaskControl.eErrorCode =
                        OSALM_eCheckTask(psTaskRegistration->hTaskObj);
                if(gsMonitorTaskControl.eErrorCode !=
                   OSAL_TASK_MONITOR_OK )
                {
                    // Stop iteration (violation detected!)
                    return FALSE;
                }
            }
        }
    }

    // Continue the iteration (ok)
    return TRUE;
}

/*******************************************************************************
*
*   OSALM_eCheckTask
*
*******************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALM_eCheckTask (
    OSAL_OBJECT_HDL hTaskObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eTaskGetInfo(hTaskObj, &gsMonitorTaskControl.sTaskInfo);
    if(eReturnCode == OSAL_SUCCESS)
    {
#if OSAL_OBJECT_TRACKING == 1
        OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)hTaskObj;
#endif

        if(gsMonitorTaskControl.sTaskInfo.un16RemainingClicksForReport == 0)
        {
            // A violation occurred - report failure
            return OSAL_ERROR_TASK_MONITOR_REPORT_TIMEOUT;
        }

        // Check task's stack
        if(gsMonitorTaskControl.sTaskInfo.un8StackPercentUsed >
           OSALM_MAX_STACK_USAGE)
        {
            // A violation occurred - report failure
            return OSAL_ERROR_TASK_MONITOR_STACK_EXCEEDED;
        }

        printf("Task Monitor: '%s' checked ok.\n"
               "\tStack Usage = %u%% (max %u%% allowed)\n",
#if OSAL_OBJECT_TRACKING == 1
               psObj->pacName,
#else
               "Unknown",
#endif
               gsMonitorTaskControl.sTaskInfo.un8StackPercentUsed,
               OSALM_MAX_STACK_USAGE );
    }

    // All is well
    return OSAL_TASK_MONITOR_OK;
}

/*******************************************************************************
*
*   OSALM_eCheckSystem
*
*******************************************************************************/
static OSAL_RETURN_CODE_ENUM OSALM_eCheckSystem ( void )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check current level of memory allocation
    bOk = OSAL.bMemoryUsage(NULL, NULL,
                            &gsMonitorTaskControl.un32ActualBytesUsed,
                            NULL, NULL, NULL, NULL,
                            NULL, NULL );

    if(bOk == TRUE)
    {
        if(gsMonitorTaskControl.un32ActualBytesUsed >
           gsMonitorTaskControl.un32ActualBytesThreshold)
        {
            // A violation occurred - report failure
            return OSAL_ERROR_TASK_MONITOR_MEMORY_ALLOCATION_EXCEEDED;
        }
    }

    // Check current level of CPU utilization
    eReturnCode = OSAL.eGetUtilizationCPU(
        NULL,
        &gsMonitorTaskControl.un8MaxUtilization,
        NULL
            );
    if(eReturnCode == OSAL_SUCCESS)
    {
        if(gsMonitorTaskControl.un8MaxUtilization >
           OSALM_MAX_CPU_UTILIZATION)
        {
            // Current CPU utilization exceeds limits!

            // Check if this warning has not already been raised
            if(gsMonitorTaskControl.bCpuUtilizationWarning == FALSE)
            {
                // Toggle warning flag
                gsMonitorTaskControl.bCpuUtilizationWarning = TRUE;

                // A violation occurred - report failure
                return OSAL_ERROR_TASK_MONITOR_CPU_UTILIZATION_EXCEEDED;
            }

            // Otherwise...
            // This violation is a warning, and has already been
            // raised. It will be ignored now, until the user
            // clears the violation (reset's max utilization) and
            // another sample confirms the utilization has gone below
            // the preset limits.
        }
        else
        {
            // The max CPU utilization is no longer in violation (or has
            // remained not in violation, so just make sure the violation
            // warning toggle is set appropriately.
            gsMonitorTaskControl.bCpuUtilizationWarning = FALSE;

        }
    }

    if (gsMonitorTaskControl.eSystemErrorCode != OSAL_TASK_MONITOR_OK)
    {
        OSAL_RETURN_CODE_ENUM eErrorCode;

        // Save error code
        eErrorCode = gsMonitorTaskControl.eSystemErrorCode;

        // Warning handled
        gsMonitorTaskControl.eSystemErrorCode = OSAL_TASK_MONITOR_OK;
        return eErrorCode;
    }

    printf("Task Monitor: System checked ok.\n"
           "\tCPU Utilization = %u%% (max %u%% allowed)\n"
           "\tMemory = %u bytes (max %u bytes allowed)\n",
           gsMonitorTaskControl.un8MaxUtilization, OSALM_MAX_CPU_UTILIZATION,
           gsMonitorTaskControl.un32ActualBytesUsed,
           gsMonitorTaskControl.un32ActualBytesThreshold);

    // All is well
    return OSAL_TASK_MONITOR_OK;
}

/*******************************************************************************
*
*   OSALM_vTaskMonitorViolationHandler
*
*******************************************************************************/
static void OSALM_vTaskMonitorViolationHandler ( void )
{
    // Call application callback to report violation
    gsMonitorTaskControl.psMonitorHandlers->vMonitorErrorHandler(
        gsMonitorTaskControl.eErrorCode,
        &gsMonitorTaskControl.sTaskInfo,
        gsMonitorTaskControl.psMonitorHandlers->pvMonitorErrorHandlerArg
            );

    // Clear this error now that it has been reported
    gsMonitorTaskControl.eErrorCode = OSAL_TASK_MONITOR_OK;

    return;
}
