/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio Inc.                      */
/*                            All Rights Reserved                             */
/*           Licensed Materials - Property of Sirius XM Radio Inc.            */
/*                         Proprietary & Confidential                         */
/******************************************************************************/
/*******************************************************************************
*
*  DESCRIPTION
*
*       This module will contain all the OSAL mem APIs.
*
*******************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"
#include "osal_core.h"

#include "os_intf.h"

#include "_osal_mem.h"

/*******************************************************************************
*
*   OSAL_bMemRead8
*
*******************************************************************************/
BOOLEAN OSAL_bMemRead8 (
    const void *pvMemoryAddress,
    UN8 *pun8Value
        )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bMemRead8(pvMemoryAddress, pun8Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bMemWrite8
*
*******************************************************************************/
BOOLEAN OSAL_bMemWrite8 (
     void *pvMemoryAddress,
     UN8 un8Value
         )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bMemWrite8(pvMemoryAddress, un8Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bMemRead16
*
*******************************************************************************/
BOOLEAN OSAL_bMemRead16 (
     const void *pvMemoryAddress,
     UN16 *pun16Value
         )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bMemRead16(pvMemoryAddress, pun16Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bMemWrite16
*
*******************************************************************************/
BOOLEAN OSAL_bMemWrite16 (
      void *pvMemoryAddress,
      UN16 un16Value
          )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bMemWrite16(pvMemoryAddress, un16Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bMemRead32
*
*******************************************************************************/
BOOLEAN OSAL_bMemRead32 (
     const void *pvMemoryAddress,
     UN32 *pun32Value
         )
{
    BOOLEAN bRetVal = FALSE;

	// Call platform specific API
    bRetVal = OS.bMemRead32(pvMemoryAddress, pun32Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bMemWrite32
*
*******************************************************************************/
BOOLEAN OSAL_bMemWrite32 (
      void *pvMemoryAddress,
      UN32 un32Value
          )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bMemWrite32(pvMemoryAddress, un32Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bPortRead8
*
*******************************************************************************/
BOOLEAN OSAL_bPortRead8 (
     const void *pvPortAddress,
     UN8 *pun8Value
         )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bPortRead8(pvPortAddress, pun8Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bPortWrite8
*
*******************************************************************************/
BOOLEAN OSAL_bPortWrite8 (
      void *pvPortAddress,
      UN8 un8Value
          )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bPortWrite8(pvPortAddress, un8Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bPortRead16
*
*******************************************************************************/
BOOLEAN OSAL_bPortRead16 (
      const void *pvPortAddress,
      UN16 *pun16Value
          )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bPortRead16(pvPortAddress, pun16Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bPortWrite16
*
*******************************************************************************/
BOOLEAN OSAL_bPortWrite16 (
       void *pvPortAddress,
       UN16 un16Value
           )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bPortWrite16(pvPortAddress, un16Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bPortRead32
*
*******************************************************************************/
BOOLEAN OSAL_bPortRead32 (
      const void *pvPortAddress,
      UN32 *pun32Value
          )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bPortRead32(pvPortAddress, pun32Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bPortWrite32
*
*******************************************************************************/
BOOLEAN OSAL_bPortWrite32 (
       void *pvPortAddress,
       UN32 un32Value
           )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bPortWrite32(pvPortAddress, un32Value);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bMemCpy
*
*******************************************************************************/
BOOLEAN OSAL_bMemCpy (
      void *pvDst,
      const void *pvSrc,
      size_t tSize
          )
{
    BOOLEAN bRetVal;
    UN8 *pun8DstStart = (UN8*)pvDst,
        *pun8DstEnd = pun8DstStart + tSize,
        *pun8SrcStart = (UN8*)pvSrc,
        *pun8SrcEnd = pun8SrcStart + tSize;

    // Verify that regions to be copied do not overlap. If they do
    // this fuction must return failure.
    // (note that since we added tSize earlier to compute the end variables
    //  that we need to be careful with how we use the equality conditions)
    if(((pun8SrcEnd > pun8DstStart) && (pun8SrcEnd <= pun8DstEnd)) ||
       ((pun8SrcStart >= pun8DstStart) && (pun8SrcStart < pun8DstEnd)))
    {
        // Error! Regions overlap.
        return FALSE;
    }

    // Call platform specific API
    bRetVal = OS.bMemCpy(pvDst, pvSrc, tSize);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_bMemSet
*
*******************************************************************************/
BOOLEAN OSAL_bMemSet (
      void *pvDst,
      UN8 un8Value,
      size_t tSize
          )
{
    BOOLEAN bRetVal = FALSE;

    // Call platform specific API
    bRetVal = OS.bMemSet(pvDst, un8Value, tSize);

    return bRetVal;
}

/*******************************************************************************
*
*   OSAL_pvMemoryAllocate
*
*******************************************************************************/
void *OSAL_pvMemoryAllocate(
      const char *pacName,
      size_t tSize,
      BOOLEAN bZeroInitialize
          )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj = NULL;
    OSAL_OBJECT_HDL hMemory;
    OSAL_MEMORY_INFO_STRUCT *psMemory = NULL;
    void *pvMemory = NULL;

    // Memory blocks allocated by the OSAL are tightly controlled and monitored
    // Each memory block contains the block itself as well as an object
    // header. The block is then added to the memory block object list
    // providing a means to track memory allocations.

    // Each memory allocation must also have along with it an object structure
    // to accomplish this a pointer will be added to the top of the memory
    // allocated which contains the address of the associated object.
    // The object in turn contains the address of the memory block allocated.

    // Caller must ask for "some" bytes
    if(tSize == 0)
    {
        return NULL;
    }

    eReturnCode =
        OSALC_eCreateObject(
            OSAL_OBJECT_TYPE_MEMORY,
            tSize,
            pacName,
            &hMemory,
            &psObj,
            (OSAL_OBJECT_INFO_UNION **)&psMemory
                );
    if(eReturnCode == OSAL_SUCCESS)
    {
        // User space is last part of object + info memory
        pvMemory = psObj->pvUserMemory;
        psMemory->tSize = tSize;

        // Perform zero initialization if requested
        if(bZeroInitialize == TRUE)
        {
            // Zero initialize user data
            OS.bMemSet(pvMemory, 0, tSize);
        }

        // Update memory allocation statistics (OSAL, Add)
        OSALC_vUpdateMemoryAllocation(
            OSAL_MEMORY_TYPE_OSAL, OSALC_UPDATE_OPERATION_ADD, tSize );
    }

    return pvMemory;
}

/*******************************************************************************
*
*   OSAL_vMemoryFree
*
*******************************************************************************/
void OSAL_vMemoryFree(
      void *pvMemoryAddress
          )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    size_t tSize;

    // If the pointer provided is NULL...bail
    if(pvMemoryAddress == NULL)
    {
        return;
    }

    // The memory address provided is the 'user space' pointer.
    // We need to determine what the object handle is.
    psObj =(OSAL_OBJECT_STRUCT *)
                ((char *)pvMemoryAddress -
                    sizeof(OSAL_MEMORY_INFO_STRUCT) -
                    sizeof(OSAL_OBJECT_STRUCT)
                        );

    // Now that we have extracted the handle, we need to verify it is ok

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle((OSAL_OBJECT_HDL)psObj, OSAL_OBJECT_TYPE_MEMORY);
    if(psObj == NULL)
    {
        return;
    }

    // Extract size info from object
    tSize = psObj->puInfo->sMemory.tSize;

    // Remove and destroy the object
    eReturnCode = OSALC_eRemoveObject((OSAL_OBJECT_HDL)psObj);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Update memory allocation statistics (OSAL, Subtract)
        OSALC_vUpdateMemoryAllocation(
            OSAL_MEMORY_TYPE_OSAL, OSALC_UPDATE_OPERATION_SUB, tSize );
    }

    return;
}

/*******************************************************************************
*
*   OSAL_eMemoryList
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eMemoryList ( const char *pacFilterText )
{
#if (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSALM_STATISTICS_STRUCT sStatisticsCurrent, sStatisticsComputed;
    OSAL_OBJECT_HDL hMemoryBlockList;
    BOOLEAN bOk;
    UN32 un32Blocks;
    OSAL_STATISTICS_STRUCT sStatistics;
    OSALM_ITERATOR_DATA_STRUCT sIterator;

    // Iterate the list of memory blocks and list each entry

    // Grab the memory block linked list
    hMemoryBlockList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_MEMORY);
    if(hMemoryBlockList == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the number of blocks in the list
    eReturnCode = OSAL.eLinkedListItems(hMemoryBlockList, &un32Blocks);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the object's statistics
    OSALC_vGetObjectStatistics(OSAL_OBJECT_TYPE_MEMORY, &sStatistics);

    // Check that one or more elements exist in the list
    if(un32Blocks > 0)
    {
        // Print title
        printf("==========================\n");
        printf("| OSAL Memory Block List |\n");
        printf("==========================\n");

        // Print header row
        printf("| %-26s| %-26s| %-10s| %-8s |\n",
               "Creator Object",
               "Name",
               "Pointer",
               "Size(B)"
               );

        printf("================================================================================\n");
    }

    // Initialize statistics
    OSAL.bMemSet( &sStatisticsComputed, 0, sizeof(OSALM_STATISTICS_STRUCT) );

    // Prepare iterator data structure
    sIterator.psStatistics = &sStatisticsComputed;
    sIterator.pcFilter = pacFilterText;

    // Iterate the list dumping each element individually
    eReturnCode = OSAL.eLinkedListIterate(hMemoryBlockList,
        bPrintMemoryBlock, &sIterator);

    if(eReturnCode == OSAL_NO_OBJECTS)
    {
        printf("Object list is empty!\n");
        return eReturnCode;
    }
    else if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }
    else
    {
        // Print footer row
        printf("================================================================================\n");

        // Grab OSAL memory usage info
        bOk = OSAL.bMemoryUsage(
                    &sStatisticsCurrent.un32CurrentAllocatedBlocks,
                    &sStatisticsCurrent.un32CurrentActualAllocatedBlocks,
                    &sStatisticsCurrent.un32CurrentUserBytes,
                    &sStatisticsCurrent.un32CurrentActualBytes,
                    &sStatisticsCurrent.un32MaxAllocatedBlocks,
                    &sStatisticsCurrent.un32MaxActualAllocatedBlocks,
                    &sStatisticsCurrent.un32MaxUserBytes,
                    &sStatisticsCurrent.un32MaxActualBytes,
                    &sStatisticsCurrent.un32TotalSystemBytes );

        if(bOk == FALSE)
        {
            printf("Error! Unable to get memory usage from OSAL.\n");
            return OSAL_ERROR;
        }

        if(pacFilterText == NULL)
        {
            // Compare computed statistics to the OSAL reported statistics
            if( sStatisticsComputed.un32CurrentAllocatedBlocks !=
                sStatisticsCurrent.un32CurrentAllocatedBlocks )
            {
                printf("Error! Number of blocks reported from OSAL (%u) does not"
                     " match computed number of blocks (%u).\n",
                     sStatisticsCurrent.un32CurrentAllocatedBlocks,
                     sStatisticsComputed.un32CurrentAllocatedBlocks);
                eReturnCode = OSAL_ERROR_CORRUPTED_OBJECT_LIST;
            }
            if( sStatisticsComputed.un32CurrentUserBytes !=
                sStatisticsCurrent.un32CurrentUserBytes )
            {
                printf("Error! Number of bytes reported from OSAL (%u) does not"
                     " match computed number of bytes (%u).\n",
                     sStatisticsCurrent.un32CurrentUserBytes,
                     sStatisticsComputed.un32CurrentUserBytes);
                eReturnCode = OSAL_ERROR_CORRUPTED_OBJECT_LIST;
            }
            printf("\tMemory Usage:\n\n");
        }
        else
        {
            // If we had filtered blocks, need to separately print computed
            // statistics
            printf("\tMemory Usage:\n\n");

            printf("\tCurrent filtered Memory Usage:\n");
            printf("\t\t%u blocks\n",
                sStatisticsComputed.un32CurrentAllocatedBlocks );
            printf("\t\t%u user bytes\n",
                sStatisticsComputed.un32CurrentUserBytes);

        }
        // Dump stats
        printf("\tCurrent total Memory Usage:\n");
        printf("\t\t%u blocks\n",
                    sStatisticsCurrent.un32CurrentAllocatedBlocks );
        printf("\t\t%u actual blocks\n",
                    sStatisticsCurrent.un32CurrentActualAllocatedBlocks);
        printf("\t\t%u user bytes\n",
                    sStatisticsCurrent.un32CurrentUserBytes);
        printf("\t\t%u actual bytes\n",
                    sStatisticsCurrent.un32CurrentActualBytes);

        printf("\tMaximum Memory Usage:\n");
        printf("\t\t%u blocks\n",
                    sStatisticsCurrent.un32MaxAllocatedBlocks );
        printf("\t\t%u actual blocks\n",
                    sStatisticsCurrent.un32MaxActualAllocatedBlocks);
        printf("\t\t%u user bytes\n",
                    sStatisticsCurrent.un32MaxUserBytes);
        printf("\t\t%u actual bytes\n",
                    sStatisticsCurrent.un32MaxActualBytes);

        printf("\tTotal System Memory: %u\n",
                    sStatisticsCurrent.un32TotalSystemBytes );

        printf("\tOS-Overhead:  %.1f %%\n",
               sStatisticsCurrent.un32CurrentUserBytes != 0 ?
                   (((double)(sStatisticsCurrent.un32CurrentActualBytes
                    - sStatisticsCurrent.un32CurrentUserBytes))
                        / sStatisticsCurrent.un32CurrentUserBytes)
                            * 100  : 100);

    }

    return eReturnCode;
#else

    return OSAL_ERROR_UNSUPPORTED_API;

#endif /*(OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */
}

/*******************************************************************************
*
*   OSALM_vPrintMemoryBlock
*
*******************************************************************************/
void OSALM_vPrintMemoryBlock(void *pvObj)
{
#if OSAL_OBJECT_TRACKING == 1
    OSALM_ITERATOR_DATA_STRUCT sIterator;
    OSALM_STATISTICS_STRUCT sStatistics;
    BOOLEAN bOutputEnabledThisTask;

    bOutputEnabledThisTask =
                    OSAL.bOutputEnabledThisTask();

    OSAL.vControlOutputThisTask(TRUE);

    // Initialize statistics
    OSAL.bMemSet( &sStatistics, 0, sizeof(sStatistics));

    sIterator.pcFilter = NULL;
    sIterator.psStatistics = &sStatistics;
    // The memory address provided is the 'user space' pointer.
    // We need to determine what the object handle is.
    pvObj =(OSAL_OBJECT_STRUCT *)
                ((char *)pvObj -
                    sizeof(OSAL_MEMORY_INFO_STRUCT) -
                    sizeof(OSAL_OBJECT_STRUCT)
                        );

    bPrintMemoryBlock(pvObj, &sIterator);
    OSAL.vControlOutputThisTask(bOutputEnabledThisTask);
#endif
    return;
}

#if (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)

/*******************************************************************************
*
*   bPrintMemoryBlock
*
*******************************************************************************/
static BOOLEAN bPrintMemoryBlock ( void *pvData, void *pvArg )
{
    OSALM_ITERATOR_DATA_STRUCT *psIteratorData =
             (OSALM_ITERATOR_DATA_STRUCT *)pvArg;
    OSALM_STATISTICS_STRUCT *psStatistics;
    const char *pcFilter;
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)pvData,
        *psCreatorObj;
    const char *pacCreatorObjName = "Unknown";
    BOOLEAN bPrint;

    // Check is data pointer is valid
    if(psIteratorData != NULL)
    {
       psStatistics = psIteratorData->psStatistics;
       pcFilter = psIteratorData->pcFilter;
    }
    else
    {
       // Error case.
       return FALSE;
    }

    // Check if statistics are valid
    if(psStatistics != NULL)
    {
        // Check if object is valid
        if(psObj != NULL)
        {
            // Choose a reasonable length for the object name
            // Something which can be fit in 80-char columns
            const UN8 un8ObjectNameWidth =
                OSAL_MAX_OBJECT_NAME_LENGTH > 26 ? 26 :
                    OSAL_MAX_OBJECT_NAME_LENGTH;

            // extract information we are interested
            // Point to this objects info
            OSAL_MEMORY_INFO_STRUCT *psMemory =
                &psObj->puInfo->sMemory;

            // Extract the creator task's name
            psCreatorObj = OSALC_psGetObjectFromHandle(
                psObj->hCreatorObj, OSAL_OBJECT_TYPE_TASK);

            if(psCreatorObj != NULL)
            {
                // Extract name of creator
                pacCreatorObjName = psCreatorObj->pacName;
            }

            if(pcFilter != NULL)
            {
               bPrint =  ( NULL == strstr(pacCreatorObjName, pcFilter)) ? FALSE : TRUE;
               bPrint |= ( NULL == strstr(psObj->pacName, pcFilter)) ? FALSE : TRUE;
            }
            else
            {
               bPrint = TRUE;
            }

            if(bPrint)
            {
            printf("| %-*.26s| %-*.26s| %#8.8x| %8d |\n",
                   un8ObjectNameWidth,
                   pacCreatorObjName,
                   un8ObjectNameWidth,
                   psObj->pacName,
                   psObj->pvUserMemory,
                   psMemory->tSize);

            // Accumulate statistics
            psStatistics->un32CurrentAllocatedBlocks++;
            psStatistics->un32CurrentUserBytes += psMemory->tSize;
            }
        }
    }

    // Keep iterating
    return TRUE;
}

#endif /* (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */
