/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*DESCRIPTION
*
*       This module will contain all the OSAL File System APIs.
*
*******************************************************************************/
#include <string.h>

#include "standard.h"

#include "osal_version.h"

#include "osal_core.h"
#include "os_intf.h"

#include "_osal_fs.h"

#include "osal.h"


/*******************************************************************************
*
*   OSAL_bFileSystemStart
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemStart ( void )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Call OS-Specific API
    bRetval = OS.bFileSystemStart();
    return bRetval;
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bFileSystemStop
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemStop ( void )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Call OS-Specific API
    bRetval = OS.bFileSystemStop();

    // set the file system temp dir path to NULL
    // this will cause it to free up the memory used to store the temp dir path
    OSALC_vUpdateFileSystemTempPath(NULL);
    return bRetval;
#else
    return FALSE;
#endif

}

/*******************************************************************************
*
*   OSAL_vFileSystemVolumes
*
*******************************************************************************/
void OSAL_vFileSystemVolumes ( FILE *psFile )
{
#if OSAL_FILE_SYSTEM == 1
    // Verify input is something valid
    if(psFile != NULL)
    {
        // Call OS-Specific API
        OS.vFileSystemVolumes( psFile );
    }
#endif
    return;
}

/*******************************************************************************
*
*   OSAL_bFileSystemVolumeInfo
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemVolumeInfo ( FILE *psFile, const char *pcVolName )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Verify input is something valid
    if((psFile != NULL) && (pcVolName != NULL))
    {
        // Call OS-Specific API
        bRetval = OS.bFileSystemVolumeInfo(psFile, pcVolName);
    }
    return bRetval;
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bFileSystemDirectory
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemDirectory ( FILE *psFile, const char *pcPathName )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Verify input is something valid
    if((psFile != NULL) && (pcPathName != NULL))
    {
        // Call OS-Specific API
        bRetval = OS.bFileSystemDirectory(psFile, pcPathName);
    }
    return bRetval;
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bFileSystemMakeDir
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemMakeDir ( const char *pcDirName )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Verify input is something valid
    if(pcDirName != NULL)
    {
        // Call OS-Specific API
        bRetval = OS.bFileSystemMakeDir(pcDirName);
    }
    return bRetval;
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bFileSystemRemoveDir
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemRemoveDir ( const char *pcDirName )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Verify input is something valid
    if(pcDirName != NULL)
    {
        // Call OS-Specific API
        bRetval = OS.bFileSystemRemoveDir(pcDirName);
    }
    return bRetval;
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_eFileSystemGetDirItems
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eFileSystemGetDirItems (
    OSAL_OBJECT_HDL *phDirItems, const char *pcPathName,
    OSAL_FS_FILE_QUALIFIER bFileQualifierFunc, void *pvQualiferArg)
{
#if OSAL_FILE_SYSTEM == 1
    static UN32 un32DirItemInstanceID = 0;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH];
    OSAL_OBJECT_HDL hDirItemsList;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if( (phDirItems == NULL) || (pcPathName == NULL) ||
                    (strlen(pcPathName) == 0))
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // create the linked list to hold the items
    sprintf(&acName[0], "DirItemsList:%u", un32DirItemInstanceID++);
    eReturnCode = OSAL.eLinkedListCreate(&hDirItemsList,
        &acName[0], NULL, OSAL_LL_OPTION_LINEAR);

    if (eReturnCode == OSAL_SUCCESS)
    {
        eReturnCode = OS.eFileSystemGetDirItems(&hDirItemsList, pcPathName,
            bFileQualifierFunc, pvQualiferArg);

        if (eReturnCode == OSAL_SUCCESS)
        {
            *phDirItems = hDirItemsList;
        }
        else
        {
            // either an error or no items found.  Free our list
            OSAL.eFileSystemReleaseDirItems(hDirItemsList);
        }
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif

}

/*******************************************************************************
*
*   OSAL_eFileSystemReleaseDirItems
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eFileSystemReleaseDirItems(
    OSAL_OBJECT_HDL hDirItems )
{
#if OSAL_FILE_SYSTEM == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eLinkedListRemoveAll(hDirItems, vReleaseDirItems);

    if (eReturnCode == OSAL_SUCCESS)
    {
        eReturnCode = OSAL.eLinkedListDelete(hDirItems);
    }
    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*******************************************************************************
*
*   OSAL_bFileSystemTruncateFile
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemTruncateFile ( FILE *psFile, size_t tNewSize )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Verify input is something valid
    if(psFile != NULL)
    {
        // Call OS-Specific API
        bRetval = OS.bFileSystemTruncateFile(psFile, tNewSize);
    }
    return bRetval;
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bFileSystemGetFileSize
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemGetFileSize ( FILE *psFile, size_t *ptFileSize )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Verify input is something valid
    if((psFile != NULL) && (ptFileSize != NULL))
    {
        // Call OS-Specific API
        bRetval = OS.bFileSystemGetFileSize(psFile, ptFileSize);
    }
    return bRetval;
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bFileSystemGetFileAttributes
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemGetFileAttributes ( const char *pcPath,
                                            UN8 *pun8FileAttributes )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Verify input is something valid
    if((pcPath != NULL) && (pun8FileAttributes != NULL))
    {
        // Call OS-Specific API
        bRetval = OS.bFileSystemGetFileAttributes(pcPath, pun8FileAttributes);
    }
    return bRetval;
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bFileSystemSetFileAttributes
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemSetFileAttributes ( const char *pcPath,
                                            UN8 un8FileAttributes )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Verify input is something valid
    if(pcPath != NULL)
    {
        // Call OS-Specific API
        bRetval = OS.bFileSystemSetFileAttributes(pcPath, un8FileAttributes);
    }
    return bRetval;
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bFileSystemSetTempPath
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemSetTempPath ( const char *pacTempPath )
{
#if OSAL_FILE_SYSTEM == 1
    UN8 un8RequirdAttributes, un8FileAttributes;
    BOOLEAN bOk;
    char *pacTempDirPath;
    size_t tTempPathLen;
    N32 n32Written;

    do
    {
        if ( pacTempPath != NULL )
        {
            // if the requested path isn't NULL, then we need
            // to see if the directory exists, its attributes etc.
            un8RequirdAttributes =
                (OSAL_FILE_ATTR_DIRECTORY |
                 OSAL_FILE_ATTR_READ |
                 OSAL_FILE_ATTR_WRITE);

            bOk = OSAL.bFileSystemGetFileAttributes(
                pacTempPath, &un8FileAttributes);

            if (bOk == TRUE)
            {
                // we got the file attributes. did they match what we want?
                if ((un8FileAttributes & un8RequirdAttributes) !=
                    un8RequirdAttributes)
                {
                    // supplied dir does not have the proper attributes
                    break;
                }
            }
            else
            {
                // couldn't get the attributes, so it probably doesn't exist
                break;
            }

            // the directory exists and has the proper attributes.
            // now we need to copy it.

            tTempPathLen = strlen(pacTempPath)+1; //+1 for NULL terminator

            pacTempDirPath = (char *)OSALC_pvMemoryAllocate(
                                         "OSALTempPath",
                                         tTempPathLen,
                                         FALSE);
            // allocate unsuccessful?
            if (pacTempDirPath == NULL)
            {
                break;
            }

            // copy path
            n32Written = snprintf(
                pacTempDirPath,
                tTempPathLen,
                "%s",
                pacTempPath
                    );

            if ( n32Written < 0)
            {
                // sprintf failed
                OSALC_vMemoryFree(pacTempDirPath);
                break;
            }
        }
        else
        {
            // user passed in NULL, so set file system temp path to NULL
            pacTempDirPath = NULL;
        }

        // set the new path
        OSALC_vUpdateFileSystemTempPath(pacTempDirPath);

        return TRUE;

    } while (0);
#endif
    // something went wrong
    return FALSE;
}

/*******************************************************************************
*
*   OSAL_tFileSystemGetTempPath
*
*******************************************************************************/
size_t OSAL_tFileSystemGetTempPath ( size_t tDstSize,
                                     char *pacDst
                                         )
{
#if OSAL_FILE_SYSTEM == 1
    N32 n32Written = 0;

    char *pacTempDirPath;
    BOOLEAN bLocked;

    // Lock core data for access
    bLocked = OSALC_bLockCore();
    if(bLocked == TRUE)
    {
        pacTempDirPath = OSALC_pacGetFileSystemTempPath();

        // first see if the temp path was set
        if (pacTempDirPath != NULL)
        {
            // yes it was set

            // did caller provide a destination?
            if (pacDst == NULL)
            {
                size_t tLength;

                // caller provided a NULL destination, so don't try and
                // copy - just return the size.
                tLength = strlen(pacTempDirPath);

                n32Written = tLength;
            }
            else
            {
                // copy it to the provided buffer
                n32Written = snprintf(
                              pacDst,
                              tDstSize,
                              "%s",
                              pacTempDirPath
                                  );

                // condition output. (if an error occurred in snprintf it'll
                // return a negative number, but this function returns an
                // unsigned number)
                if (n32Written < 0)
                {
                    n32Written = 0;
                }
            }
        }

        // Unlock core data
        OSALC_vUnlockCore();
    }
    return (size_t)n32Written;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_bFileSystemSyncFile
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemSyncFile ( FILE *psFile )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Verify input is something valid
    if(psFile != NULL)
    {
        // Call OS-Specific API
        bRetval = OS.bFileSystemSyncFile(psFile);
    }

    return bRetval;
#else
    return TRUE;
#endif
}

/*******************************************************************************
*
*   OSAL_bFileSystemUnconditionalFileUnlink
*   
*   This function is intended for forced file removal. 
*   As opposite to OS_iRemove(), this should remove files even in case of 
*   corrupted file system when some extra system-level checks being done by
*   remove() function call would fail.
*
*******************************************************************************/
BOOLEAN OSAL_bFileSystemUnconditionalFileUnlink ( 
    const char *pcPath
        )
{
#if OSAL_FILE_SYSTEM == 1
    BOOLEAN bRetval = FALSE;

    // Verify pathname is something valid
    if(pcPath != NULL)
    {
        BOOLEAN bFile = TRUE;
        UN8 un8Attrib = 0;

        // Checking file attributes to understand whenever it is 
        // a directory or regular file
        bRetval = OS.bFileSystemGetFileAttributes(pcPath, &un8Attrib);
        if (bRetval == TRUE)
        {
            if ((un8Attrib & OSAL_FILE_ATTR_DIRECTORY) == 
                OSAL_FILE_ATTR_DIRECTORY)
            {
                // This is directory
                bFile = FALSE;
            }
        }
        else
        {
            // Cannot retrieve attributes. Some error is there, but let's 
            // assume it was a file anyway.
        }

        if (bFile == TRUE)
        {
            bRetval = OS.bFileSystemUnconditionalFileUnlink(pcPath);
        }
        else
        {
            bRetval = OS.bFileSystemRemoveDir(pcPath);
        }
    }

    return bRetval;
#else
    return TRUE;
#endif
}

/*******************************************************************************
*
*   vReleaseDirItems
*
*******************************************************************************/
static void vReleaseDirItems ( void *pvData )
{

#if OSAL_FILE_SYSTEM == 1
    // Just free the character buffer that is stored
    OSALC_vMemoryFree(pvData);
#endif

    return;
}

